// Copyright 2012-2015 Oliver Eilhard. All rights reserved.
// Use of this source code is governed by a MIT-license.
// See http://olivere.mit-license.org/license.txt for details.

package elastic

// AvgAggregation is a single-value metrics aggregation that computes
// the average of numeric values that are extracted from the
// aggregated documents. These values can be extracted either from
// specific numeric fields in the documents, or be generated by
// a provided script.
// See: http://www.elasticsearch.org/guide/en/elasticsearch/reference/current/search-aggregations-metrics-avg-aggregation.html
type AvgAggregation struct {
	field           string
	script          string
	scriptFile      string
	lang            string
	format          string
	params          map[string]interface{}
	subAggregations map[string]Aggregation
}

func NewAvgAggregation() AvgAggregation {
	a := AvgAggregation{
		params:          make(map[string]interface{}),
		subAggregations: make(map[string]Aggregation),
	}
	return a
}

func (a AvgAggregation) Field(field string) AvgAggregation {
	a.field = field
	return a
}

func (a AvgAggregation) Script(script string) AvgAggregation {
	a.script = script
	return a
}

func (a AvgAggregation) ScriptFile(scriptFile string) AvgAggregation {
	a.scriptFile = scriptFile
	return a
}

func (a AvgAggregation) Lang(lang string) AvgAggregation {
	a.lang = lang
	return a
}

func (a AvgAggregation) Format(format string) AvgAggregation {
	a.format = format
	return a
}

func (a AvgAggregation) Param(name string, value interface{}) AvgAggregation {
	a.params[name] = value
	return a
}

func (a AvgAggregation) SubAggregation(name string, subAggregation Aggregation) AvgAggregation {
	a.subAggregations[name] = subAggregation
	return a
}

func (a AvgAggregation) Source() interface{} {
	// Example:
	//	{
	//    "aggs" : {
	//      "avg_grade" : { "avg" : { "field" : "grade" } }
	//    }
	//	}
	// This method returns only the { "avg" : { "field" : "grade" } } part.

	source := make(map[string]interface{})
	opts := make(map[string]interface{})
	source["avg"] = opts

	// ValuesSourceAggregationBuilder
	if a.field != "" {
		opts["field"] = a.field
	}
	if a.script != "" {
		opts["script"] = a.script
	}
	if a.scriptFile != "" {
		opts["script_file"] = a.scriptFile
	}
	if a.lang != "" {
		opts["lang"] = a.lang
	}
	if a.format != "" {
		opts["format"] = a.format
	}
	if len(a.params) > 0 {
		opts["params"] = a.params
	}

	// AggregationBuilder (SubAggregations)
	if len(a.subAggregations) > 0 {
		aggsMap := make(map[string]interface{})
		source["aggregations"] = aggsMap
		for name, aggregate := range a.subAggregations {
			aggsMap[name] = aggregate.Source()
		}
	}

	return source
}
