/*
 *   meioshar.c -- MEIO Manager Sharing process
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>

#include <meio_api.h>    
#include "meiomacs.h"    
#include <meiogtwy.h>

#define ExclusiveTest(mask,options)                  \
              ((mask&options)? MEIO_RS_EXCLUSIVE : MEIO_RS_SHAREABLE)




/*  sharemasks[] - array used to store sharing option masks for resource
                 objects.  MS nibble of type is used to index array.
                 Mask ANDed with sharing options will determine state.
*/
static ULONG sharemasks[] =
{   0                                                /*  unused */
    ,MY_OWNER_PORT                                   /*  ANALOG_OWNER_PORT */
    ,                MY_USER_PORT                    /*  ANALOG_USER_PORT */
    ,MY_OWNER_PORT                                   /*  DIGITAL_OWNER_PORT */
    ,                MY_USER_PORT                    /*  DIGITAL_USER_PORT */
    ,                               MY_CONNECTION    /*  ANALOG_SWITCH */
    ,                               MY_CONNECTION    /*  DIGITAL_SWITCH */
    ,MY_OWNER_PORT | MY_USER_PORT | MY_CONNECTION    /*  ANALOG_INTERNAL */
    ,MY_OWNER_PORT | MY_USER_PORT | MY_CONNECTION    /*  RESOURCESET */
    ,MY_OWNER_PORT | MY_USER_PORT                    /*  BIOSTASK */
    ,0,0,0,0,0,0                                     /*  reserved A-F */
};

/*   ScanOwner() - scan an owner list and report status */
static RESOURCE_LIST ScanOwner(PRESOURCE         RObj
                       ,PUSHORT           pShare
                       ,PUSHORT           pDiffOwners
                       ,PMEIO_CONNECTION  pConnection
                       ,HMEIO             NewOwner
                       ,ULONG             NewOptions
                       )
{
  RESOURCE_LIST     FirstOwner;
  RESOURCE_LIST     match=0;
  
  PMEIO_CONNECTION  pC1=0, pC2=0;              
  USHORT            DiffOwner=0;
  USHORT            Sharing  = (USHORT)NewOptions;  

  MW_SYSLOG_5(TRACE_MEIO_CORE,"meioshar::ScanOwner entry RObj %p pConnection %p NewOwner %lx NewOptions %lx\n",
	 RObj,pConnection,NewOwner,NewOptions);
  FirstOwner = RObj->lpOwnerList;           

  if (FirstOwner) {
    pC1 = FirstOwner->owner; 
    if (NewOwner)
      DiffOwner |= pC1->hResourceOwner != NewOwner;
  }

  while (FirstOwner) {
    if ( pC1 == pConnection) {
      match = FirstOwner;
    } else {
      Sharing   |=  pC1->ulSharingOptions;
    } 
    
    if (FirstOwner->next) {
      pC2 = (FirstOwner->next)->owner; 
      DiffOwner |=  pC1->hResourceOwner != pC2->hResourceOwner;
    }
    
    FirstOwner = FirstOwner->next;
    pC1 = pC2;
  } 
  
  *pShare = Sharing;
  *pDiffOwners = DiffOwner;
  MW_SYSLOG_4(TRACE_MEIO_CORE,"meioshar::ScanOwner exitpShare %x pDiffOwners %x match %p\n",
	 *pShare,*pDiffOwners,match);
  return(match);
  
}

/*   AddOwner() - add an owner to the ownerlist, check status, return
                new list.
*/
static RESOURCE_LIST AddOwner(PRESOURCE         RObj
			      ,PUSHORT           pState
			      ,PUSHORT           pDiffOwners
			      ,PMEIO_CONNECTION  pConnection
			      )
{
  RESOURCE_LIST FirstOwner;
  RESOURCE_LIST match;
  int           i;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioshar::AddOwner entry RObj %p pConnection %p\n",
	 RObj,pConnection);
  FirstOwner = RObj->lpOwnerList;     

  for (i=0; i<roOwnersSize; i++) {
    if (0==roOwners[i].owner)  {
      match = &roOwners[i];      
      match->next = FirstOwner;  
      match->owner = pConnection;
      RObj->lpOwnerList = match;
      RObj->usOwnerCount++;     
      break;                    
    } 
  }
  
  match=ScanOwner(RObj, pState, pDiffOwners, pConnection,0,0);
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::AddOwner exit match %p\n",
	 match);
  return match;
}

/*   RemoveOwner() - remove an owner from the ownerlist, check status,
                   return new list.
*/
static RESOURCE_LIST RemoveOwner(PRESOURCE         RObj
				 ,PUSHORT           pState
				 ,PUSHORT           pDiffOwners
				 ,PMEIO_CONNECTION  pConnection
				 )
{
  RESOURCE_LIST      FirstOwner;
  RESOURCE_LIST FAR *previous;
  
  PMEIO_CONNECTION  pC1=0, pC2;                
  USHORT            DiffOwner=0;
  USHORT            Sharing  =0;
  USHORT            RemovedOne=0;  

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioshar::RemoveOwner entry RObj %p pConnection %p\n",
	 RObj,pConnection);
  FirstOwner = RObj->lpOwnerList;  
  previous   = &(RObj->lpOwnerList);

  while (FirstOwner) {
    if (FirstOwner)  pC1 = FirstOwner->owner;
    if ( RemovedOne == 0 &&  pC1 == pConnection) {
      *previous = FirstOwner->next;    
      FirstOwner->owner = 0;        
      RObj->usOwnerCount--;   
      RemovedOne++;        
    } else {
      Sharing   |=  pC1->ulSharingOptions;
      if (FirstOwner->next) {
	pC2 = (FirstOwner->next)->owner;
	DiffOwner |=  pC1->hResourceOwner != pC2->hResourceOwner;
      }
      previous = &(FirstOwner->next); 
    } 
    
    FirstOwner = FirstOwner->next;
   }

  *pState = Sharing;
  *pDiffOwners = DiffOwner;
  
  MW_SYSLOG_4(TRACE_MEIO_CORE,"meioshar::RemoveOwner exit pState %x pDiffOwners %x RObj->lpOwnerList %p\n",
	 *pState,*pDiffOwners,RObj->lpOwnerList);
  return(RObj->lpOwnerList);
}


/*   ClassifyBlock() - local function to classify blocked error */
static ULONG ClassifyBlock(ULONG ShareException)
{
  ULONG ulRC;
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::ClassifyBlock entry ShareException %lx\n",
	 ShareException);
  if (ShareException == MY_OWNER_PORT) {
    ulRC = MEIO_NC_OWNER_BLOCKED;
  } else {
    if (ShareException == MY_USER_PORT) {
      ulRC = MEIO_NC_USER_BLOCKED;
    } else {
      ulRC = MEIO_NC_CONNECTION_BLOCKED;
    }
  }
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::ClassifyBlock exit ulRC %lx\n",ulRC);
  return(ulRC);
}

#define ExclusiveTest(mask,options)                  \
              ((mask&options)? MEIO_RS_EXCLUSIVE : MEIO_RS_SHAREABLE)

/*  RS_Unowned() - function to process sharing request for unowned object */
static ULONG RS_Unowned(RF_ARGS, ULONG ShareMask)
{
  ULONG  ulRC;
  USHORT state,diff;
  
  VOID_ARGS;       

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioshar::RF_Unowned entry usMessage %x ShareMask %lx\n",
	 usMessage,ShareMask);
  if (usMessage == mTESTmsg) {
    ulRC = MEIO_NOERROR;
  } else {
    if (usMessage == mRESERVEmsg) {
      lpResource->usState =  ExclusiveTest(ShareMask,lParam2);
      AddOwner(lpResource,&state,&diff,pConnection);
      ulRC = MEIO_NOERROR;
    } else {
      ulRC = MEIO_NC_INVALID_ATTRIBUTE;  
    } 
  }
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::RF_Unowned exit ulRC %lx\n",ulRC);
  return(ulRC);
}

/*  RS_Exclusive() - function to process sharing request for exclusive object */
static ULONG RS_Exclusive(RF_ARGS, ULONG ShareMask)
{
  ULONG         ulRC=MEIO_NOERROR;
  RESOURCE_LIST match;
  USHORT        state,diff;
  
  VOID_ARGS;   

  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioshar::RF_Exclusive entry usMessage %x ShareMask %lx\n",
	 usMessage,ShareMask);
  switch (usMessage) {

  case mTESTUPDATEmsg:   
  case mUPDATEmsg:      
    match = ScanOwner(lpResource,&state,&diff,pConnection,0,lParam2);
    
    if (match==0) {
      ulRC = MEIO_ND_NOT_CONNECTED;
    } else {
      if (usMessage==mUPDATEmsg && (state & ShareMask) == 0) {
	lpResource->usState = MEIO_RS_SHAREABLE;  
      } 
    } 
    break;

  case mTESTmsg:  
  case mRESERVEmsg:
    match = ScanOwner(lpResource,&state,&diff,
		      pConnection,
		      pConnection->hResourceOwner,lParam2);
    
    if (diff) {
      ulRC = ClassifyBlock(ShareMask & state);
    } else {
      if(usMessage==mRESERVEmsg)   
	AddOwner(lpResource,&state,&diff,pConnection);
    }
    break;
    
  case mRELEASEmsg:
    match = RemoveOwner(lpResource,&state,&diff, pConnection);
    if (lpResource->usOwnerCount==0) {
      lpResource->usState = MEIO_RS_UNOWNED;  
    } else {
      if ((state & ShareMask) == 0) {
	lpResource->usState = MEIO_RS_SHAREABLE;  
      } 
    }
    break;
    
  default: 
    ulRC = MEIO_NC_INVALID_ATTRIBUTE;
    break;
  } 
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::RF_Exclusive exit ulRC %lx\n",ulRC);
  return(ulRC);
}


/*  RS_Shareable() - function to process sharing request for shareable object */
static ULONG RS_Shareable(RF_ARGS, ULONG ShareMask)
{
  ULONG         ulRC=MEIO_NOERROR;
  RESOURCE_LIST match;
  USHORT        state,diff;
  
  VOID_ARGS;
  
  MW_SYSLOG_3(TRACE_MEIO_CORE,"meioshar::RF_Shareable entry usMessage %x ShareMask %lx\n",
	 usMessage,ShareMask);

  switch (usMessage) {

  case mTESTUPDATEmsg: 
  case mUPDATEmsg:    

    match = ScanOwner(lpResource,&state,&diff,pConnection,0,lParam2);
    if (match==0) {
      ulRC = MEIO_ND_NOT_CONNECTED;  
    } else {
      if ( state & ShareMask ) {
	if (diff==0) {
	  if (usMessage==mUPDATEmsg)  
	    lpResource->usState = MEIO_RS_EXCLUSIVE; 
	} else {
	  {
	    ulRC = ClassifyBlock(ShareMask & state); 
	  }
	}
      } 
    } 
    break;


  case mTESTmsg:
  case mRESERVEmsg:
    match = ScanOwner(lpResource,&state,&diff,
		      pConnection,
		      pConnection->hResourceOwner,lParam2);
    if ( state & ShareMask ) {
      if (diff==0) {
	if (usMessage==mRESERVEmsg)  
	  lpResource->usState = MEIO_RS_EXCLUSIVE;
      } else {
	ulRC = ClassifyBlock(ShareMask & state);
      } 
    } else {
      if (usMessage == mRESERVEmsg && diff) {
	lpResource->usState = MEIO_RS_SHARED;  
      } 
    } 
    if(usMessage==mRESERVEmsg && ulRC == MEIO_NOERROR)
      AddOwner(lpResource,&state,&diff,pConnection);
      break;

  case mRELEASEmsg:
    match = RemoveOwner(lpResource,&state,&diff, pConnection);
    if (lpResource->usOwnerCount==0) {
      lpResource->usState = MEIO_RS_UNOWNED; 
    } 
    break;

  default:  
    ulRC = MEIO_NC_INVALID_ATTRIBUTE;
    break;
  } 
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::RF_Shareable exit ulRC %lx\n",ulRC);
  return(ulRC);
}

/*   ProcessSharing() - process sharing for a single resource object,
                      do not iterate into object.
*/
ULONG FAR ProcessSharing(RF_ARGS)
{
  ULONG ulRC;
  ULONG ShareMask;
  
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meioshar::ProcessSharing entry\n");
  ShareMask = sharemasks[ lpResource->usType >> 12 ];

  switch (lpResource->usState) {
  case MEIO_RS_UNOWNED:       
    ulRC = RS_Unowned(RF_ARGLIST,ShareMask);
    break;

  case MEIO_RS_EXCLUSIVE:
    ulRC = RS_Exclusive(RF_ARGLIST,ShareMask);
    break;

  case MEIO_RS_SHAREABLE: 
    ulRC = RS_Shareable(RF_ARGLIST,ShareMask);
    break;

  case MEIO_RS_SHARED:
    ulRC = RS_Shareable(RF_ARGLIST,ShareMask);
    break;

  default:
    ulRC = MEIO_FAIL_CONNECTION_BLOCK; 
    break;
   } 
  
  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::ProcessSharing exit ulRC %lx\n",ulRC);
  return(ulRC);
}

/*   QuerySharing() - Query sharing state for a single resource object,
                    do not iterate into object.
*/
ULONG FAR QuerySharing(RF_ARGS)
{
  ULONG    ulRC = MEIO_NOERROR;
  ULONG    ShareMask;
  USHORT   state,diff;
  
  VOID_ARGS;
  
  MW_SYSLOG_1(TRACE_MEIO_CORE,"meioshar::QuerySharing entry\n");
  ShareMask = sharemasks[ lpResource->usType >> 12 ];
  ScanOwner(lpResource,&state,&diff, 0, 0, 0);
  ShareMask &= state;
  *((PULONG)lParam2) = ShareMask | *((PULONG)lParam2);

  MW_SYSLOG_2(TRACE_MEIO_CORE,"meioshar::QuerySharing exit ulRC %lx\n",ulRC);
  return(ulRC);
}
