/*
  Gnake - A Curses Game
  Copyright (C) 2002-2004 Flavio Chierichetti
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License version 2 (not
  later versions) as published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>

#include <curses.h>

#define abs(a) ( ((a) >= 0) ? (a) : ( 0 - (a) ) )

enum {NColors = 8, BufferLength = 1000, TextScrollDelay = 5, ScoreLength = 10};

enum {Black = 1, Red, Green, Yellow, Blue, Magenta, Cyan, White};

enum {NewHead = -2, HeadCrush, Empty, Wall, Eatable};

enum {Dead = 0, Dying, Alive};

enum {Halt = 0, Right, Up, Left, Down};

typedef struct {
  char Name[BufferLength];
  time_t Time;
  int Apples;
} score;

typedef struct {
  int R, C;
} coord;

typedef struct n {
  coord Coord;
  int Dir;
  struct n *next;
} node;  

typedef struct {
  int Color, Dir, NLeft, Length, Apples, Alive, CPU;
  long Steps;
  node *Head;
} snake;

void getOptions(int argc, char *argv[]);
void DrawWindow();
void Init();
void Step();
void Draw();
void End();

coord getRandEmptyCell(int mandatory);
void createApple(int i);

void Resize();

char *TheGameName = "Gnake 0.94b";

coord Dim;
int NSnakes, AliveSnakes, NApples;
int Spaces, Border, MyColor, TheirColor, Reverse, ILen, ALen, SLen, SInterval, Delay;
int Verbose;

long NState;
int MySnake, FramedSnake;

coord CoordMin, CoordMax;
WINDOW *Window, *Pad;

int **Board;
snake *Snk;
coord *Apl;

int main(int argc,char *argv[]) {
  int c, n;

  getOptions(argc, argv);

  Init();

  while ( AliveSnakes > 0 && ( c = wgetch(Pad) ) != 'q' ) {
    if ( Snk[MySnake].Alive )
      switch (c) {
      case KEY_RIGHT:
	if ( Snk[MySnake].Head->Dir != Left )
	  Snk[MySnake].Dir = Right;
	break;
      case KEY_UP:
	if ( Snk[MySnake].Head->Dir != Down )
	  Snk[MySnake].Dir = Up;
	break;
      case KEY_LEFT:
	if ( Snk[MySnake].Head->Dir != Right )
	  Snk[MySnake].Dir = Left;
	break;
      case KEY_DOWN:
	if ( Snk[MySnake].Head->Dir != Up )
	  Snk[MySnake].Dir = Down;
	break;
      case 'p':
	Snk[MySnake].Dir = Halt;
	break;
      }
    
    switch (c) {
    case '.':
      for ( n = 1 ; n < NSnakes ; n++ )
	if ( Snk[(FramedSnake + n) % NSnakes].Alive ) {
	  FramedSnake = (FramedSnake + n) % NSnakes;
	  /* MySnake = FramedSnake; */
	  break;
	}
      break;
    case ',':
      for ( n = 1 ; n < NSnakes ; n++ )
	if ( Snk[(FramedSnake + NSnakes - n) % NSnakes].Alive ) {
	  FramedSnake = (FramedSnake + NSnakes - n) % NSnakes;
	  /* MySnake = FramedSnake; */
	  break;
	}
      break;
    }
  }

  exit(0);
}

void getOptions(int argc, char *argv[]) {
  extern char *optarg;
  int c;

  NSnakes = 1;
  MySnake = 0;
  Reverse = A_REVERSE; /* Per non avere la riversione: A_NORMAL */
  MyColor = Blue;
  TheirColor = Green;

  Verbose = 0;

  NApples = 1;
  Spaces = 3;
  Border = 3;
  Dim.R = 40;
  Dim.C = 100;
  Delay = 50;
  ILen = 30;
  SLen = 0;
  SInterval = 1;
  ALen = 3;

  while ( ( c = getopt(argc, argv, "hvr:c:d:i:s:a:A:C:") ) != -1 ) {
    switch (c) {
    case 'h':
      printf("%s - Copyright (C) 2002-2004 Flavio Chierichetti\n", TheGameName);
      printf("This program is released under the GNU GPL version 2\n");
      printf("usage:\t%s [-h] [-v] [-r n] [-c n] [-d n] [-i n] [-a n] [-A n]\n", argv[0]);
      printf("\t-h\tdisplay this help and exit\n");
      printf("\t-v\tbe verbose (on stderr)\n");
      printf("\t-r n\tset n rows (%d)\n", Dim.R);
      printf("\t-c n\tset n columns (%d)\n", Dim.C);
      printf("\t-d n\tset n ms as delay (%d)\n", Delay);
      printf("\t-i n\tset n as starting length (%d)\n", ILen);
      printf("\t-s n/m\tadd n blocks every m steps (%d/%d)\n", SLen, SInterval);
      printf("\t-a n\tadd n blocks per apple (%d)\n", ALen);
      printf("\t-A n\tplay with n apples (%d)\n", NApples);
      printf("\t-C n\tcpu controls n snakes (%d)\n", NSnakes - 1);
      exit(0);
      break;
    case 'v':
      Verbose++;
      break;
    case 'r':
      sscanf(optarg, " %d", &Dim.R);
      break;
    case 'c':
      sscanf(optarg, " %d", &Dim.C);
      break;
    case 'd':
      sscanf(optarg, " %d", &Delay);
      break;
    case 'i':
      sscanf(optarg, " %d", &ILen);
      break;
    case 's':
      sscanf(optarg, " %d/%d", &SLen, &SInterval);
      break;
    case 'a':
      sscanf(optarg, " %d", &ALen);
      break;
    case 'A':
      sscanf(optarg, " %d", &NApples);
      break;
    case 'C':
      sscanf(optarg, " %d", &NSnakes);
      NSnakes++;
      break;
    case '?':
      fprintf(stderr, "%s -h for more information.\n", argv[0]);
      exit(1);
      break;
    }
  }

  if ( NSnakes > Dim.R * Dim.C ) {
    fprintf(stderr, "Too many snakes for the board.\n");
    exit(1);
  }

  if ( NSnakes + NApples > Dim.R * Dim.C ) {
    if ( Verbose )
      fprintf(stderr, "I need to decrease the number of apples to fit them in the board.\n");

    NApples = Dim.R * Dim.C - NSnakes;
  }


  if ( Verbose ) {
    fprintf(stderr, "Verbose: %d\n", Verbose);
    fprintf(stderr, "NSnakes: %d\n", NSnakes);
    fprintf(stderr, "NApples: %d\n", NApples);

    fprintf(stderr, "Rows: %d\n", Dim.R);
    fprintf(stderr, "Columns: %d\n", Dim.C);
    fprintf(stderr, "Delay: %d\n", Delay);
    fprintf(stderr, "ILen: %d\n", ILen);
    fprintf(stderr, "SLen: %d\n", SLen);
    fprintf(stderr, "SInterval: %d\n", SInterval);
    fprintf(stderr, "ALen: %d\n", ALen);
  }

}

coord changeCoordinate(coord Coordinate, int Direction) {
  switch ( Direction ) {
  case Right:
    Coordinate.C++;
    break;
  case Up:
    Coordinate.R--;
    break;
  case Left:
    Coordinate.C--;
    break;
  case Down:
    Coordinate.R++;
    break;
  }
  
  return Coordinate;
}

void CPUMove(int n) {
  int Choices, N;
  coord head;
  int d;
  int a, min, best;

  /* All possible choices */
  Choices = (1 << Right) | (1 << Up) | (1 << Left) | (1 << Down);

  /* Can't go back */
  switch ( Snk[n].Dir ) {
  case Right:
    Choices &= ~(1 << Left); break; 
  case Up:
    Choices &= ~(1 << Down); break; 
  case Left:
    Choices &= ~(1 << Right); break; 
  case Down:
    Choices &= ~(1 << Up); break; 
  }

  /* Don't want to hit walls or snakes */
  for ( d = Right ; d <= Down ; d++ )
    if ( Choices & (1 << d) ) {
      head = changeCoordinate(Snk[n].Head->Coord, d);
      
      if ( ( head.R < 0 || head.R >= Dim.R || 
	     head.C < 0 || head.C >= Dim.C || 
	     Board[ head.R ][ head.C ] == Wall  ) &&
	   (Choices & ~(1 << d)) )
	Choices &= ~(1 << d);
    }

  /* Don't want to be near a snake-head */

  /* TODO: to be written */

  /* Find the distance to the nearest apple */
  best = Dim.R + Dim.C + 1;

  for ( d = Right ; d <= Down ; d++ )
    if ( Choices & (1 << d) ) {
      head = changeCoordinate(Snk[n].Head->Coord, d);
      
      for ( a = 0 ; a < NApples ; a++ )
	if ( abs(Apl[a].R - head.R) + abs(Apl[a].C - head.C) < best )
	  best = abs(Apl[a].R - head.R) + abs(Apl[a].C - head.C);
    }

  /* Want to go to one of the nearest apple */
  for ( d = Right ; d <= Down ; d++ )
    if ( Choices & (1 << d) ) {
      min = Dim.R + Dim.C + 1;

      head = changeCoordinate(Snk[n].Head->Coord, d);
      
      for ( a = 0 ; a < NApples ; a++ )
	if ( abs(Apl[a].R - head.R) + abs(Apl[a].C - head.C) < min )
	  min = abs(Apl[a].R - head.R) + abs(Apl[a].C - head.C);

      if ( min > best &&
	   (Choices & ~(1 << d)) )
	Choices &= ~(1 << d);
    }

  /* Choose one of the best */
  N = 0;
  for ( d = Right ; d <= Down ; d++ )
    if ( Choices & (1 << d) )
      N++;
  N = rand() % N;
  for ( d = Right ; d <= Down ; d++ )
    if ( Choices & (1 << d) ) {
      if ( N == 0 )
	break;
      N--;
    }

  Snk[n].Dir = d;
}

void Step() {
  node *Last;
  int n, c;

  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive && Snk[n].CPU )
      CPUMove(n);

  NState++;

  /* Stacca la coda */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Dir != Halt &&
	 Snk[n].Alive != Dead ) {
      Last = Snk[n].Head;
      if ( Snk[n].NLeft ) {
	Snk[n].Head = (node*) malloc (sizeof(node));
	Snk[n].Head->next = Last->next;
	Last->next = Snk[n].Head;
	Snk[n].NLeft--;
	Snk[n].Length++;
      }
      else {
	Snk[n].Head = Snk[n].Head->next;
	mvwaddch(Pad, Border + Snk[n].Head->Coord.R, Border + Snk[n].Head->Coord.C, ' ');
	Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = Empty;
      }
      
      Snk[n].Head->Coord.R = Last->Coord.R;
      Snk[n].Head->Coord.C = Last->Coord.C;
      Snk[n].Head->Dir = Snk[n].Dir;

      if ( Snk[n].Head != Last ) { /* Se il serpente e' piu' lungo di 1 */
	if ( ( Last->Dir == Snk[n].Head->Dir || Last->Dir == 0 ) && 
	     ( Snk[n].Head->Dir == Right || Snk[n].Head->Dir == Left ) )
	  c = ACS_HLINE;
	else if ( ( Last->Dir == Snk[n].Head->Dir || Last->Dir == 0 ) &&
		  ( Snk[n].Head->Dir == Up || Snk[n].Head->Dir == Down ) )
	  c = ACS_VLINE;
	else if ( ( Last->Dir == Right && Snk[n].Head->Dir == Up ) ||
		  ( Last->Dir == Down && Snk[n].Head->Dir == Left ) )
	  c = ACS_LRCORNER;
	else if ( ( Last->Dir == Up && Snk[n].Head->Dir == Left ) ||
		  ( Last->Dir == Right && Snk[n].Head->Dir == Down ) )
	  c = ACS_URCORNER;
	else if ( ( Last->Dir == Left && Snk[n].Head->Dir == Down ) ||
		  ( Last->Dir == Up && Snk[n].Head->Dir == Right ) )
	  c = ACS_ULCORNER;
	else
	  c = ACS_LLCORNER;
      
	wattron(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
	mvwaddch(Pad, Border + Last->Coord.R, Border + Last->Coord.C, c);
	wattroff(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
      }
    }


  /* Posiziona la testa e controlla se ha sbattuto */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Dir != Halt &&
	 Snk[n].Alive != Dead ) {
      Snk[n].Head->Coord = changeCoordinate(Snk[n].Head->Coord, Snk[n].Dir);
      
      if ( Snk[n].Head->Coord.R < 0 || 
	   Snk[n].Head->Coord.R >= Dim.R || 
	   Snk[n].Head->Coord.C < 0 || 
	   Snk[n].Head->Coord.C >= Dim.C || 
	   Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == Wall ) {
	Snk[n].Alive = Dying;
      } else if ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] >= Eatable ) {
	if ( Verbose )
	  fprintf(stderr, "%ld: %d eats.\n", NState, n);
	  
	Snk[n].NLeft += ALen;
	Snk[n].Apples++;
	  
	Apl[Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] - Eatable].R = -1;
      }
    }

  /* Segna dove andrebbero a finire le nuove teste */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive == Alive &&
	 Snk[n].Dir != Halt ) {
      if ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == Empty ||
	   Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] >= Eatable ) {

	if ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] >= Eatable )
	  mvwaddch(Pad, Border + Snk[n].Head->Coord.R, Border + Snk[n].Head->Coord.C, ' ');

	Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = NewHead;

      } else if ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == NewHead ) {
	Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = HeadCrush;
      }
    }
	
  /* Uccidi i serpenti che hanno la nuova testa nella posizione di un altro */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive == Alive &&
	 Snk[n].Dir != Halt )
      if ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == HeadCrush )
	Snk[n].Alive = Dying;

  /* Pulisci la Board dai vari NewHead e HeadCrush */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive != Dead &&
	 Snk[n].Dir != Halt )
      if ( ( Snk[n].Head->Coord.R >= 0 &&
	     Snk[n].Head->Coord.R < Dim.R &&
	     Snk[n].Head->Coord.C >= 0 &&
	     Snk[n].Head->Coord.C < Dim.C )
	   &&
	   ( Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == HeadCrush || 
	     Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] == NewHead) )
	Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = Empty;
  
  /* Disegna la testa o uccidi il serpente */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive == Alive ) {
      if ( Snk[n].Dir != Halt )
	Snk[n].Steps++;

      wattron(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
      mvwaddch(Pad, Border + Snk[n].Head->Coord.R, Border + Snk[n].Head->Coord.C, ACS_PLUS);
      wattroff(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
      
      Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = Wall;
    } else if ( Snk[n].Alive == Dying ) {

      if ( Verbose )
	fprintf(stderr, "%ld: %d dies.\n", NState, n);

      Snk[n].Alive = Dead;

      AliveSnakes--;

      Last = Snk[n].Head;
      Snk[n].Head = Snk[n].Head->next;
      Last->next = NULL;
 
      while ( Snk[n].Head->next != NULL ) {
	mvwaddch(Pad, Border + Snk[n].Head->Coord.R, Border + Snk[n].Head->Coord.C, ' ');
	Board[Snk[n].Head->Coord.R][Snk[n].Head->Coord.C] = Empty;

	Last = Snk[n].Head;
	Snk[n].Head = Snk[n].Head->next;
	free(Last);
      }
      free(Snk[n].Head);
      Snk[n].Head = NULL;


    }

  /* Eventualmente reinserisci le mele */
  for ( n = 0 ; n < NApples ; n++ )
    if ( Apl[n].R == -1 )
      createApple(n);

  /* Aggiungi gli eventuali blocchi ``temporali'' */
  for ( n = 0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive != Dead &&
	 Snk[n].Dir != Halt &&
	 Snk[n].Steps % SInterval == 0 )
      Snk[n].NLeft += SLen;
      
  Draw();
}

void DrawWindow() {
  int n;
  coord i;


  werase(Window);

  wattron(Window, COLOR_PAIR(Blue));

  n = strlen(TheGameName) + Spaces;
  for ( i.R = 0 ; i.R < LINES ; i.R++ )
    for ( i.C = 0 ; i.C < COLS ; i.C++ )
      if ( (i.C + n - i.R % n) % n < (n - Spaces) )
	waddch(Window, TheGameName[(i.C + n - i.R % n) % n]);
      else
	waddch(Window, ' ');

  wattroff(Window, COLOR_PAIR(Blue));

  wrefresh(Window);

  /* Calcolo posizione finestra */

  if ( (Dim.R + 2 * Border) <= LINES ) {
    CoordMin.R = ( LINES - (Dim.R + 2 * Border) ) / 2;
    CoordMax.R = CoordMin.R + Dim.R + 2 * Border - 1;
  } else {
    CoordMin.R = 0;
    CoordMax.R = LINES - 1;
  }

  if ( (Dim.C + 2 * Border) <= COLS ) {
    CoordMin.C = ( COLS - (Dim.C + 2 * Border) ) / 2;
    CoordMax.C = CoordMin.C + Dim.C + 2 * Border - 1;
  } else {
    CoordMin.C = 0;
    CoordMax.C = COLS - 1;
  }
}  

void Init() {
  struct itimerval Timer;
  struct sigaction Action;
  coord i;
  int n;

  /* Inizializzazioni varie */

  atexit(End);

  srand((unsigned int) time(NULL));

  Board = (int**) calloc (Dim.R, sizeof(int*));
  for ( i.R = 0 ; i.R < Dim.R ; i.R++ ) {
    Board[i.R] = (int*) calloc (Dim.C, sizeof(int));    
    for ( i.C = 0 ; i.C < Dim.C ; i.C++ )
      Board[i.R][i.C] = Empty;
  }

  AliveSnakes = NSnakes;

  FramedSnake = MySnake;
  NState = 0;

  /* Inizializzazione curses */

  Window = initscr();
  cbreak();
  nonl();
  noecho();
  curs_set(0);

  assume_default_colors(-1, -1);
  start_color();

  init_pair(Black, COLOR_BLACK, -1);
  init_pair(Red, COLOR_RED, -1);
  init_pair(Green, COLOR_GREEN, -1);
  init_pair(Yellow, COLOR_YELLOW, -1);
  init_pair(Blue, COLOR_BLUE, -1);
  init_pair(Magenta, COLOR_MAGENTA, -1);
  init_pair(Cyan, COLOR_CYAN, -1);
  init_pair(White, COLOR_WHITE, -1);

  DrawWindow();

  Pad = newpad(Dim.R + 2 * Border, Dim.C + 2 * Border);

  timeout(10);
  keypad(Pad, TRUE);

  /* Disegno sfondo e bordi */

  wattron(Pad, COLOR_PAIR(Green));
  for ( i.R = 0 ; i.R < Border ; i.R++ ) 
    for ( i.C = i.R + 1 ; i.C < Dim.C + 2 * Border - i.R - 1; i.C++ ) {
      mvwaddch(Pad, i.R, i.C, ACS_HLINE);
      mvwaddch(Pad, 2 * Border + Dim.R - 1 - i.R, i.C, ACS_HLINE);
    }
  for ( i.C = 0 ; i.C < Border ; i.C++ ) 
    for ( i.R = i.C + 1 ; i.R < Dim.R + 2 * Border - i.C - 1; i.R++ ) {
      mvwaddch(Pad, i.R, i.C, ACS_VLINE);
      mvwaddch(Pad, i.R, 2 * Border + Dim.C - 1 - i.C, ACS_VLINE);
    }
  for ( n = 0 ; n < Border ; n++ ) {
    mvwaddch(Pad, n, n, ACS_ULCORNER);
    mvwaddch(Pad, Dim.R + 2 * Border - n - 1, n, ACS_LLCORNER);
    mvwaddch(Pad, n, Dim.C + 2 * Border - 1 - n, ACS_URCORNER);
    mvwaddch(Pad, Dim.R + 2 * Border - n - 1, Dim.C + 2 * Border - n - 1, ACS_LRCORNER);
  }
  if ( Border > 0 ) {
    i.R = Dim.R + Border + Border / 2;
    mvwaddch(Pad, i.R, Border / 2 + Border % 2, ACS_RTEE);

    /*    for ( i.C = Border / 2 + Border % 2 + 1 ; i.C < Dim.C + Border + Border / 2 + Border % 2 - 1 ; i.C++ )
	  mvwaddch(Pad, i.R, i.C, ' ');*/

    mvwaddch(Pad, i.R, Dim.C + 2 * Border - (Border / 2 + Border % 2) - 1, ACS_LTEE);
  }

  wattroff(Pad, COLOR_PAIR(Green));

  /* Inizializzazione Serpenti e Mele */

  Snk = (snake*) calloc (NSnakes, sizeof(snake));
  Apl = (coord*) calloc (NApples, sizeof(coord));

  for ( n = 0 ; n < NSnakes ; n++ ) {
    Snk[n].Head = (node*) malloc (sizeof(node));
    Snk[n].Head->next = Snk[n].Head;

    Snk[n].Alive = Dead;
    Snk[n].Head->Dir = Snk[n].Dir = Halt;
    Snk[n].NLeft = ILen - 1;
    Snk[n].Length = 1;
    Snk[n].Color = (n == MySnake) ? MyColor : TheirColor;
    Snk[n].CPU = (n != MySnake);

    Snk[n].Steps = 0;
    Snk[n].Apples = 0;
  }

  for ( n = 0 ; n < NApples ; n++ )
    Apl[n].R = -1;
  
  /* Posizionamento Serpenti e Mele */

  for ( n = 0 ; n < NSnakes ; n++ ) {
    i = getRandEmptyCell(1);

    if ( Verbose )
      fprintf(stderr, "%ld: %d in (%d, %d).\n", NState, n, i.R, i.C);

    Snk[n].Alive = Alive;

    Snk[n].Head->Coord.R = i.R;
    Snk[n].Head->Coord.C = i.C;

    wattron(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
    mvwaddch(Pad, Border + i.R, Border + i.C, ACS_PLUS);
    wattroff(Pad, COLOR_PAIR(Snk[n].Color) | Reverse);
    Board[i.R][i.C] = Wall;
  }


  for ( n = 0 ; n < NApples ; n++ )
    createApple(n);

  Draw();

  /* Programmazione interrupt */

  /* SIGWINCH */

  sigemptyset(&Action.sa_mask);
  sigaddset(&Action.sa_mask, SIGALRM);
  Action.sa_flags = 0;
  Action.sa_handler = Resize;
  sigaction(SIGWINCH, &Action, NULL);

  /* SIGALRM */

  sigemptyset(&Action.sa_mask);
  sigaddset(&Action.sa_mask, SIGWINCH);
  Action.sa_flags = 0;
  Action.sa_handler = Step;
  sigaction(SIGALRM, &Action, NULL);

  /* timer */

  Timer.it_interval.tv_sec = Timer.it_value.tv_sec = 0;
  /* millisec -> microsec */
  Timer.it_interval.tv_usec = Timer.it_value.tv_usec = Delay * 1000;
  setitimer(ITIMER_REAL, &Timer, NULL);
}

void Resize() {
  FILE *p;
  int l, c;
  
  p = popen("stty size", "r");
  fscanf(p, " %d %d", &l, &c);
  pclose(p);

  wresize(Window, l, c);
  if ( resizeterm(l, c) == ERR )
    exit(1);

  DrawWindow();
}

void PrintStatus() {
  static char buffer[BufferLength];
  static int index;

  int stringLength, rowLength, rowIndex;
  int colStart, colEnd;
  int i;

  colStart = (Border / 2 + Border % 2) + 1;
  colEnd = Dim.C + 2 * Border - (Border / 2 + Border % 2) - 2;

  rowLength = colEnd - colStart + 1;
  if ( rowLength >= 1 ) {
    rowIndex = Dim.R + Border + Border / 2;

    for ( i = 0 ; i < rowLength ; i++ )
      mvwaddch(Pad, rowIndex, i + colStart, ' ');

    /*
    snprintf(buffer, BufferLength, 
	     "  Apples: %d  -  Steps: %ld  -  Length: %d (+ %d)  ",
	     Snk[FramedSnake].Apples, Snk[FramedSnake].Steps, Snk[FramedSnake].Length, Snk[FramedSnake].NLeft);
    */
    snprintf(buffer, BufferLength, 
	     "  Apples: %d  -  Steps: %ld  -  Length: %d  ",
	     Snk[FramedSnake].Apples, Snk[FramedSnake].Steps, Snk[FramedSnake].Length);

    stringLength = strlen(buffer);

    wattron(Pad, COLOR_PAIR(Green));

    if ( stringLength <= rowLength ) {
      mvwaddnstr(Pad, rowIndex, (rowLength - stringLength) / 2 + colStart, buffer, -1);
    } else {
      if ( index + rowLength > stringLength )
	index = - (stringLength - rowLength - 1);

      if ( index >= 0 ) {
	mvwaddnstr(Pad, rowIndex, colStart, buffer + index, rowLength);
      } else {
	mvwaddnstr(Pad, rowIndex, colStart, buffer - index, rowLength);
      }

      if ( (NState + 1) % TextScrollDelay == 0 )
	index++;
    }

    wattroff(Pad, COLOR_PAIR(Green));
  }
}

void Draw() {
  coord Begin;

  PrintStatus();

  if ( Snk[FramedSnake].Alive )
    Begin.R = Border + Snk[FramedSnake].Head->Coord.R - LINES / 2;
  else
    Begin.R = Border + Dim.R / 2 - LINES / 2;

  if ( Begin.R < 0 )
    Begin.R = 0;
  else if ( Begin.R > ( Dim.R + 2 * Border - LINES ) )
    Begin.R = Dim.R + 2 * Border - LINES;

  if ( Snk[FramedSnake].Alive )
    Begin.C = Border + Snk[FramedSnake].Head->Coord.C - COLS / 2;
  else
    Begin.C = Border + Dim.C / 2 - COLS / 2;
    
  if ( Begin.C < 0 )
    Begin.C = 0;
  else if ( Begin.C > ( Dim.C + 2 * Border - COLS ) )
    Begin.C = Dim.C + 2 * Border - COLS;

  prefresh(Pad, Begin.R, Begin.C, CoordMin.R, CoordMin.C, CoordMax.R, CoordMax.C);
}

coord getRandEmptyCell(int mandatory) {
  int N, n;
  coord t;

  N = Dim.C * Dim.R;

  for ( n=0 ; n < NSnakes ; n++ )
    if ( Snk[n].Alive )
      N -= Snk[n].Length;

  for ( n=0 ; n < NApples ; n++ )
    if ( Apl[n].R != -1 )
      N--;

  if ( N <= 0 ) {
    if ( mandatory ) {
      fprintf(stderr, "Board overflow.\n");
      exit(1);
    } else {
      t.R = t.C = -1;
      return t;
    }
  }

  N = rand() % N;

  t.R = t.C = 0;
  while ( 1 ) {
    if ( Board[t.R][t.C] == Empty ) {
      N--;
      if ( N < 0 )
	break;
    }
     
    t.C++;
    if ( t.C >= Dim.C ) {
      t.C = 0;
      t.R++;
    }
  }

  return t;
}


void createApple(int i) {
  int j;

  Apl[i] = getRandEmptyCell(0);

  if ( Apl[i].R != -1 ) {
    if ( Verbose ) 
      fprintf(stderr, "%ld: apple (%d, %d).\n", NState, Apl[i].R, Apl[i].C);
    
    Board[Apl[i].R][Apl[i].C] = Eatable + i;

    wattron(Pad, COLOR_PAIR(0));
    mvwaddch(Pad, Border + Apl[i].R, Border + Apl[i].C, ACS_DIAMOND);
    wattroff(Pad, COLOR_PAIR(0));
  } else {
    if ( Verbose )
      fprintf(stderr, "%ld: no more space, losing an apple.\n", NState);

    for ( j = NApples - 1 ; j > i ; j++ )
      if ( Apl[j].R != -1 )
	break;

    if ( j > i ) {
      Apl[i] = Apl[j];
      Apl[j].R = Apl[j].C = -1;

      Board[Apl[i].R][Apl[i].C] = Eatable + i;
    }

    NApples--;
  }
}

void End() {
  time_t End;

  struct sigaction Action;

  End = time(NULL);

  sigemptyset(&Action.sa_mask);
  Action.sa_flags = 0;
  Action.sa_handler = SIG_IGN;
  sigaction(SIGALRM, &Action, NULL);

  endwin();
}
