// mapedit.cpp : MAPEDIT for Insane Odyssey X
//				 aaron p. matthews, 2000
//				 This code is GPL.

// Usage:
// mapedit -file level.lvl
//
// -window = run in window mode

// Keys:

// R = Reload level, revert to saved
// S = Save level

// F1 = Map View
// F2 = Tiles View
// F3 = Icon View

// I = Insert Row that mouse is over
// K = Delete Row that mosue is over
// O = Insert Column that mouse is over
// L = Delete Column that mosue is over

// 1 = Enable/Disable Background layer
// 2 = Enable/Disable Foreground layer
// 3 = Enable/Disable Icon layer

// Left = Move view left
// Right = Move view right
// Up = Move view up
// Down = Move view down

// Left Control = Set current tile to *background* tile that mouse is over
// Left Alt = Set current tile to *foreground* tile that mouse is over

// D = Delete foreground tile that mouse is over
// F = Draw Foreground tile

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <SDL.h>

// Defines
//

#define MAXCOLS		200
#define MAXROWS		200
#define MAXLAYERS	4
#define MAXTILES 	400
#define MAXFONTS	3

#define ICONSTART 	65280		// Point of no return

#define CM		0		// Collision Map
#define AM		1		// Actor Map
#define BG		2		// Background Layer
#define FG		3		// Foreground Layer

#define MAPVIEW		0
#define TILEVIEW	1
#define ICONVIEW	2

#define TILESIZE    64

// Globals
//

short tile_orgx		= 0;
short tile_orgy		= 0;
short tileview_x	= 0;
short tileview_y	= 0;
short iconview_x	= 0;
short iconview_y	= 0;
short tiles		= 0;
short map_width		= 0;
short map_height	= 0;

short view = 0;

unsigned short map[MAXLAYERS][MAXCOLS][MAXROWS] = {0};

SDL_Surface *screen;
SDL_Surface *tile[MAXTILES];
SDL_Surface *font[MAXFONTS];
SDL_Surface *icons;
char levelname [32];
Uint8 *keys;
int mouse_x;
int mouse_y;
Uint8 button;
bool update			= true;
bool update_info		= true;
bool bg_layer			= true;
bool fg_layer			= true;
bool ob_layer			= true;

int current_tile		= 0;

// Functions
//

SDL_Surface *LoadImageTile(char *datafile, int transparent)
{
	SDL_Surface *image, *surface;

	image = SDL_LoadBMP(datafile);
	if ( image == NULL ) {
		fprintf(stderr, "Couldn't load image %s\n",
					datafile);
		return(NULL);
	}
	if ( transparent ) {
		/* Assuming 8-bit BMP image */
			SDL_SetColorKey(image, SDL_SRCCOLORKEY|SDL_RLEACCEL, (Uint8)0 );
	}
	surface = SDL_DisplayFormat(image);
	SDL_FreeSurface(image);
	return(surface);
}

void DrawChar ( short x, short y, char c, short color )
{
	SDL_Rect dst, src;
	short num;

	if ( c >= 'A' )
		num = c - 'A';
	else
		num = (c - '0') + 26;

	src.x = (num%20)*32;
	src.y = (num/20)*32;
	src.w = 32;
	src.h = 32;

	dst.x = x;
	dst.y = y;
	dst.w = 32;
	dst.h = 32;

	SDL_BlitSurface(font[color], &src, screen, &dst);
}

void Print_Number ( short x, short y, short num, short color )
{
	char letter;

	letter = '0' + (num/100);
	DrawChar ( x, y, letter, color );
	letter = '0' + ((num%100)/10);
	DrawChar ( x+28, y, letter, color );
	letter = '0' + ((num%100)%10);
	DrawChar ( x+56, y, letter, color );
}

void DrawTile ( short x, short y, short layer )
{
	SDL_Rect dst, src;
	unsigned short num = map[layer][x+tile_orgx][y+tile_orgy];

	if ( num >= MAXTILES )
		return;
	// bounds...

	src.x = (num%20)*TILESIZE;
	src.y = (num/20)*TILESIZE;
	src.w = TILESIZE;
	src.h = TILESIZE;
	dst.x = (x*TILESIZE);
	dst.y = (y*TILESIZE);
	dst.w = TILESIZE;
	dst.h = TILESIZE;

	if ( num )
		SDL_BlitSurface(tile[num], NULL, screen, &dst);
}

void DrawIcon ( short x, short y )
{
	SDL_Rect dst, src;
	unsigned short num = map[FG][x+tile_orgx][y+tile_orgy];

	if ( num < ICONSTART )
		return;
		
	num -= ICONSTART;
	// bounds...

	src.x = (num%20)*TILESIZE;
	src.y = (num/20)*TILESIZE;
	src.w = TILESIZE;
	src.h = TILESIZE;
	dst.x = (x*TILESIZE);
	dst.y = (y*TILESIZE);
	dst.w = TILESIZE;
	dst.h = TILESIZE;

	SDL_BlitSurface(icons, &src, screen, &dst);
}

void DrawCurrentTile ()
{
	SDL_Rect dst, src;
	unsigned short num = current_tile;
	
	if ( num >= MAXTILES )
	{
		num -= ICONSTART;
		
		src.x = (num%20)*TILESIZE;
		src.y = (num/20)*TILESIZE;
		src.w = TILESIZE;
		src.h = TILESIZE;
		dst.x = 800 - TILESIZE;
		dst.y = 600 - TILESIZE;
		dst.w = TILESIZE;
		dst.h = TILESIZE;
	
		SDL_BlitSurface(icons, &src, screen, &dst);
	}
	else
		{
			src.x = (num%20)*TILESIZE;
			src.y = (num/20)*TILESIZE;
			src.w = TILESIZE;
			src.h = TILESIZE;
			dst.x = 800 - TILESIZE;
			dst.y = 600 - TILESIZE;
			dst.w = TILESIZE;
			dst.h = TILESIZE;
			SDL_BlitSurface(tile[num], NULL, screen, &dst);
		}
}

void LoadTiles ()
{
	FILE *fp = fopen ( "tiles.dat", "rt" );
	char filename[32];
	char transparent[32];

	if ( fp == NULL)
	{
		fprintf(stderr, "Error reading file tiles.dat!\n" );
		exit(1);
	}

	while ( !feof ( fp ) )
	{
		fscanf ( fp, "%s %s", filename, transparent );
		tile[tiles] = LoadImageTile ( filename, transparent[0]-'0' );
		tiles ++;
	}
	
	icons = LoadImageTile ( "icons.bmp", 0 );

	fclose ( fp );
}

void FreeTiles ()
{
	while ( tiles )
	{
		SDL_FreeSurface ( tile[tiles] );
		tiles --;
	}
	
	SDL_FreeSurface ( icons );
}

void Load_Fonts ()
{
    font[0] = LoadImageTile ( "font00a.bmp", 1 );
    font[1] = LoadImageTile ( "font00b.bmp", 1 );
}

void Free_Fonts ()
{
    SDL_FreeSurface ( font[0] );
    SDL_FreeSurface ( font[1] );
}

void Draw_Level ()
{
	short x, y;

	for ( y = 0; y < (480/TILESIZE)+1; y ++ )
    		for ( x = 0; x < 640/TILESIZE; x ++ )
     		{
        		if ( bg_layer )
				DrawTile ( x, y, BG );
			if ( fg_layer )
        			DrawTile ( x, y, FG );
        		if ( ob_layer )
        			DrawIcon ( x, y );
      		}
}

void Draw_Tiles ()
{
	SDL_Rect dst;
        short x, y;
        short num = tileview_x + tileview_y*20;

        for ( y = 0; y < 480; y += TILESIZE )
        {
        	for ( x = 0; x < 640; x += TILESIZE )
        	{
        		dst.x = x;
        		dst.y = y;
        		dst.w = TILESIZE;
        		dst.h = TILESIZE;
			SDL_BlitSurface(tile[num], NULL, screen, &dst);
			num ++;
		}
		num += 10;
	}
}

void Draw_Icons ()
{
	SDL_Rect dst, src;

	src.x = iconview_x * TILESIZE;
	src.y = iconview_y * TILESIZE;
	src.w = 640;
	src.h = 512;
	dst.x = 0;
	dst.y = 0;
	dst.w = 640;
	dst.h = 512;

	SDL_BlitSurface(icons, &src, screen, &dst);
}

void Load_Level(char *name)
{
	short i;
	FILE *stream;
	/* open the level file and read the level information */
	memset ( map, 0, sizeof(map) );

	if ( (stream = fopen( name, "rb")) != NULL )
	{
		fread( &map_width, sizeof(short), 1, stream);
		fread( &map_height, sizeof(short), 1, stream);
		for (i = 0; i < map_width; i++)
			fread(&map[BG][i][0],sizeof(short),map_height,stream);
		for (i = 0; i < map_width; i++)
			fread(&map[FG][i][0],sizeof(short),map_height,stream);
		fclose(stream);
	}
	else
   	{
	    map_width = 20;
		map_height = 20;
   	}

}

void Save_Level(char *fname)
{
   register int i;
   FILE *stream;
   /* open a binary file for writing */
   if ((stream = fopen(fname,"wb")) != NULL)
   {
      fwrite(&map_width,sizeof(short),1,stream);
      fwrite(&map_height,sizeof(short),1,stream);

      for (i = 0; i < map_width; i++)
         fwrite(&map[BG][i][0],sizeof(short),map_height,stream);

      for (i = 0; i < map_width; i++)
         fwrite(&map[FG][i][0],sizeof(short),map_height,stream);

      fclose(stream);
   }
}

int CheckParm (int argc, char *argv[], char *check)
{
	int i;

        for (i = 1;i<argc;i++)
			if ( !strcmp(check,argv[i]) )
				return i;

	return 0;
}

bool MouseIn ( short x, short y, short w, short h )
{
	if ( mouse_x >= x && mouse_y >= y )
		if ( mouse_x <= x+w && mouse_y <= y+h )
			return true;
	return false;
}

void TileView ()
{
	if ( keys[SDLK_RIGHT] && tileview_x < 10)
	{
		update = true;
		tileview_x ++;
	}
	if ( keys[SDLK_LEFT] && tileview_x > 0 )
	{
		update = true;
		tileview_x --;
	}
	if ( keys[SDLK_UP] && tileview_y > 0 )
	{
		update = true;
		tileview_y --;
	}
	if ( keys[SDLK_DOWN] && tileview_y < (tiles/20)-9 )
	{
		update = true;
		tileview_y ++;
	}

	if ( MouseIn ( 0, 0, 640, 512 ) )
		if ( SDL_BUTTON ( button ) == 1 )
		{
			current_tile = tileview_y*20 + tileview_x;
			current_tile += (mouse_y/TILESIZE)*20;
			current_tile += mouse_x/TILESIZE;
			update_info = true;
			button = 0;
		}
}

void IconView ()
{
	if ( keys[SDLK_RIGHT] && iconview_x < 10)
	{
		update = true;
		iconview_x ++;
	}
	if ( keys[SDLK_LEFT] && iconview_x > 0 )
	{
		update = true;
		iconview_x --;
	}
	if ( keys[SDLK_UP] && iconview_y > 0 )
	{
		update = true;
		iconview_y --;
	}
	if ( keys[SDLK_DOWN] && iconview_y < 1 )
	{
		update = true;
		iconview_y ++;
	}
	
	
	if ( MouseIn ( 0, 0, 640, 512 ) )
		if ( SDL_BUTTON ( button ) == 1 )
		{
			current_tile = iconview_y*20 + iconview_x;
			current_tile += (mouse_y/TILESIZE)*20;
			current_tile += mouse_x/TILESIZE;
			current_tile += ICONSTART;
			update_info = true;
			button = 0;
		}
}

void MapView ()
{
	short i, j;
	
	if ( keys[SDLK_RIGHT] && tile_orgx < map_width - 10 )
	{
		update = true;
		tile_orgx ++;
	}
	if ( keys[SDLK_LEFT] && tile_orgx > 0 )
	{
		update = true;
		tile_orgx --;
	}
	if ( keys[SDLK_UP] && tile_orgy > 0 )
	{
		update = true;
		tile_orgy --;
	}
	if ( keys[SDLK_DOWN] && tile_orgy < map_height - 8 )
	{
		update = true;
		tile_orgy ++;
	}

	if ( MouseIn ( 0, 0, 640, 512 ) )
	{
		update_info = true;
		
		if ( SDL_BUTTON ( button ) == 1 )
		{
			if ( current_tile < ICONSTART )
				map[BG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy] = current_tile;
			else
				map[FG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy] = current_tile;
			update = true;
		}
		if ( SDL_BUTTON ( button ) == 3 || keys[SDLK_f] )
		{
			map[FG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy] = current_tile;
			update = true;
		}
		if ( keys[SDLK_d] )
		{
			map[FG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy] = 0;
			update = true;
		}
		if ( keys[SDLK_LCTRL] )
		{
			current_tile = map[BG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy];
			update_info = true;
		}
		if ( keys[SDLK_LALT] )
		{
			current_tile = map[FG][mouse_x/TILESIZE+tile_orgx][mouse_y/TILESIZE+tile_orgy];
			update_info = true;
		}

        if ( keys[SDLK_i] )
        {
			keys[SDLK_i] = 0;
            map_width++;
            for (j = 0; j < map_height; j++)
               for (i = map_width-1; i > (mouse_x/TILESIZE)+tile_orgx; i--)
               {
                  map[BG][i][j] = map[BG][i-1][j];
                  map[FG][i][j] = map[FG][i-1][j];
               }
			update = true;
        }
        if ( keys[SDLK_o] )
        {
			keys[SDLK_o] = 0;
            map_height++;
            for (j = map_height-1; j > (mouse_y/TILESIZE)+tile_orgy; j--)
               for (i = 0; i < map_width; i++)
               {
                  map[BG][i][j] = map[BG][i][j-1];
                  map[FG][i][j] = map[FG][i][j-1];
               }
			update = true;
        }
        if ( keys[SDLK_k] )
        {
			keys[SDLK_k] = 0;
            map_width--;
            for (j = 0; j < map_height; j++)
            {
               for (i = (mouse_x/TILESIZE)+tile_orgx; i < map_width-1; i++)
               {
                  map[BG][i][j] = map[BG][i+1][j];
                  map[FG][i][j] = map[FG][i+1][j];
               }
            }
			update = true;
        }
        if ( keys[SDLK_l] )
        {
			keys[SDLK_l] = 0;
            map_height--;
            for (j = (mouse_y/TILESIZE)+tile_orgy; j < map_height-1; j++)
            {
               for (i = 0; i < map_width; i++)
               {
                  map[BG][i][j] = map[BG][i][j+1];
                  map[FG][i][j] = map[FG][i][j+1];
               }
            }
			update = true;
        }
	}
}

void Loop ()
{
	bool done = false;
	SDL_Rect dst, dst2;

	dst.x = 0;
	dst.y = 0;
	dst.w = 640;
	dst.h = 512;

	dst2.x = 0;
	dst2.y = 513;
	dst2.w = 640;
	dst2.h = 600-513;

	while ( !done )
	{
		SDL_PollEvent(NULL);			// get input
   		keys = SDL_GetKeyState ( NULL );	// catch it
		button = SDL_GetMouseState ( &mouse_x, &mouse_y );

		if ( update_info )
		{
			SDL_FillRect ( screen, &dst2, 0 );
			DrawCurrentTile();

			Print_Number ( 64, 514, mouse_x/TILESIZE + tile_orgx, 0 );
			Print_Number ( 64, 514+32, mouse_y/TILESIZE + tile_orgy, 0 );

			Print_Number ( 256, 514, current_tile, 0 );

			SDL_UpdateRect (screen, 0, 0, 0, 0 );
			update_info = false;
		}
		if ( update )
		{
			SDL_FillRect ( screen, &dst, 0 );

			switch ( view )
			{
			case MAPVIEW:
				Draw_Level ();
				break;
			case TILEVIEW:
				Draw_Tiles ();
				break;
			case ICONVIEW:
				Draw_Icons ();
				break;
			}
			
			SDL_UpdateRect (screen, 0, 0, 0, 0 );
			update = false;
		}

		switch ( view )
		{
		case TILEVIEW:
			TileView();
			break;

		case MAPVIEW:
			MapView();
			break;
		
		case ICONVIEW:
			IconView();
			break;
		}

		if ( keys[SDLK_1] )
		{
			bg_layer ^= true;
			keys[SDLK_1] = 0;
			update = true;
		}
		if ( keys[SDLK_2] )
		{
			fg_layer ^= true;
			keys[SDLK_2] = 0;
			update = true;
		}
		if ( keys[SDLK_3] )
		{
			ob_layer ^= true;
			keys[SDLK_3] = 0;
			update = true;
		}
		if ( keys[SDLK_F1] )
		{
			view = MAPVIEW;
			update = true;
		}
		if ( keys[SDLK_F2] )
		{
			view = TILEVIEW;
			update = true;
		}
		if ( keys[SDLK_F3] )
		{
			view = ICONVIEW;
			update = true;
		}
		if ( keys[SDLK_ESCAPE] )
			done = true;
		if ( keys[SDLK_r] )
		{
			keys[SDLK_r] = 0;
			Load_Level ( levelname );
			update = true;
		}
		if ( keys[SDLK_s] )
		{
			keys[SDLK_s] = 0;
			Save_Level ( levelname );
		}
	}
}

// Main
//

int main(int argc, char *argv[])
{
	if (SDL_Init (SDL_INIT_VIDEO) < 0)
	{
		fprintf(stderr, "Initialization failed: %s\n", SDL_GetError());
		exit(1);
	}
	atexit(SDL_Quit);

	if ( argc < 3 )
	{
		fprintf(stderr, "Usage: mapedit -file level.lvl\n" );
		exit (1);
	}

 	if ( CheckParm( argc, argv, "-window" ) )
 	{
		screen = SDL_SetVideoMode(800, 600, 16, SDL_SWSURFACE );
	}
	else
		screen = SDL_SetVideoMode(800, 600, 16, SDL_SWSURFACE|SDL_FULLSCREEN );
	
	if ( screen == NULL)
	{
		fprintf(stderr, "Can't set video mode: %s\n", SDL_GetError());
		exit(1);
	}

	SDL_WM_SetCaption ( "MAPEDIT - Insane Odyssey X", NULL );

	SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);

	LoadTiles ();
	Load_Fonts ();
	
	strcat ( levelname, argv[CheckParm(argc,argv, "-file")+1] );
	Load_Level ( levelname );

	Loop ();

	FreeTiles ();
	Free_Fonts ();

	return 0;
}
