// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/ChargedFinalState.hh"
#include "Rivet/Projections/Beam.hh"

namespace Rivet {


  /// @brief e+ e- > hadrons near Upsilon resonances
  class ARGUS_1994_I375418 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_1994_I375418);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(Beam(), "Beams");
      declare(FinalState(), "FS");
      declare(ChargedFinalState(), "CFS");

      // Book histograms
      book(_c[0], "/TMP/sigma1");
      book(_c[1], "/TMP/sigma2");
      book(_c[2], "/TMP/energy");
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // get the axis, direction of incoming positron
      const ParticlePair& beams = apply<Beam>(event, "Beams").beams();
      double s = (beams.first.momentum()+beams.second.momentum()).mass2();
      _c[2]->fill(sqrt(s));
      if(apply<ChargedFinalState>(event,"CFS").particles().size()<3) vetoEvent;
      const FinalState& fs = apply<FinalState>(event, "FS");
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
	nCount[p.pid()] += 1;
	++ntotal;
      }
      // mu+mu- + photons
      if(nCount[-13]==1 and nCount[13]==1 &&
	 ntotal==2+nCount[22])
	vetoEvent;
      else if(nCount[-11]==1 and nCount[11]==1 &&
	      ntotal==2+nCount[22])
	vetoEvent;
      // everything else
      else {
	_c[0]->fill();
	double H0=0.,H2=0;
	for (unsigned int ix=0; ix<fs.particles().size();++ix) {
	  double p1 = fs.particles()[ix].p3().mod();
	  for (unsigned int iy=0; iy<fs.particles().size();++iy) {
	    double p2 = fs.particles()[iy].p3().mod();
	    double cTheta=fs.particles()[ix].p3().dot(fs.particles()[iy].p3())/p1/p2;
	    double pre=p1*p2/s;
	    H0 +=     pre;
	    H2 += 0.5*pre*(3.*sqr(cTheta)-1);
	  }
	}
	if(H2/H0<0.35) _c[1]->fill();
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      double fact = crossSection()/sumOfWeights()/nanobarn;
      double sig_h = _c[0]->val()*fact;
      double err_h = _c[0]->err()*fact;
      double energy = _c[2]->val()/ sumOfWeights();
      
      for(unsigned int ix=1;ix<4;++ix) {
	if(ix==3) {
	  sig_h = _c[1]->val()*fact;
	  err_h = _c[1]->err()*fact;
	}
	Scatter2D temphisto(refData(ix, 1, 1));
	Scatter2DPtr cross;
	book(cross, ix, 1, 1);
	double deltaE=1e30;
	unsigned int ipoint=100000000;
	for (size_t b = 0; b < temphisto.numPoints(); b++) {
	  double test = abs(temphisto.point(b).x()-energy);
	  if(test<deltaE) {
	    deltaE=test;
	    ipoint=b;
	  }
	}
	if(deltaE>0.01) ipoint=-1;
	for (size_t b = 0; b < temphisto.numPoints(); b++) {
	  const double x  = temphisto.point(b).x();
	  pair<double,double> ex = temphisto.point(b).xErrs();
	  if (b!=ipoint)
	    cross  ->addPoint(x, 0., ex, make_pair(0.,.0));
	  else
	    cross  ->addPoint(x, sig_h, ex, make_pair(err_h,err_h));
	}
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    CounterPtr _c[3];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_1994_I375418);

}
