/*
 * sgsubmon.c
 *
 * Subroutines common to sgraidmon, sgdiskmon, sgsafte.
 *
 * 03/09/07 ARCress - created
 */
/*---------------------------------------------------------------------------
Copyright (c) 2001-2008, Intel Corporation
All rights reserved.
 
Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:
 
  a.. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.
  b.. Redistributions in binary form must reproduce the above copyright notice,
      this list of conditions and the following disclaimer in the documentation
      and/or other materials provided with the distribution.
  c.. Neither the name of Intel Corporation nor the names of its contributors
      may be used to endorse or promote products derived from this software
      without specific prior written permission.
 
THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *---------------------------------------------------------------------------*/
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h> 
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <getopt.h>
#include <time.h>
#include <sys/ioctl.h>
#include <scsi/sg.h>

#define PATH_MAX   64
#include <dirent.h>

#include "sgsubmon.h"

extern char fdebug; // = 0;
extern char fdevfs; // = 0;         * =1 if DevFS is detected */       
extern char fskipreset; // = 0;     * =1 to skip SCSI reset if error */
extern char fsafte; // = 0;         * SAF-TE: is there a safte HSBP? */
extern char fses;
extern char flogopen;
extern int  ndev; // = 0;
extern int  nprocdev; // = 0;
extern FILE *fdlog; // = NULL;      * log file descriptor */
extern FILE *fdmsg; // = NULL;      * message file descriptor (stdout) */
extern DEVLISTX_TYPE devlist[MAX_DEVS];  
extern struct {
    uchar vendor[9];
    uchar model[17];
    uchar fwrev[5];
    uchar sernum[13];
    uchar devtype;
    uchar dstatus;      /* device status: DSTAT_* */
} devmatch[MAX_DEVS];
extern SLOT_LIST slot[];
extern SLOT_LIST *pslot;
extern int nslots;  // = MAX_DEVS;  * SAF-TE: number of slots */
extern int islot;  
int nsestypes;
int isestype;

// static SLOT_LIST slot[MAX_DEVS];
// static uchar  slotmode;        /* SAF-TE: 3=#insertions, 4=status */
// static int slotoff = 0xff;     /* SAF-TE: offset into data for slot info */
// static uchar inqcdb[6]     = { 0x12, 0, 0, 0, SGDATALEN, 0 };
static uchar turcdb[6]     = { 0x00, 0, 0, 0, 0,    0 };
static uchar startcdb[6]   = { 0x1B, 0, 0, 0, 0x01, 0 };
static int max_try = 12;

//////////////////////////////////////////////////////////////////////
// Local Subroutines
//////////////////////////////////////////////////////////////////////
extern void showit(char *buf);
 
void itoh(uchar * chp, int len, char *str)
{
    uchar rgascii[] = { "0123456789abcdef" };
    int i, j;
    uchar k;
    for (i = 0, j = 0; i < len; i++) {
        k = (chp[i] & 0xf0) >> 4;
        str[j++] = rgascii[k];
        k = chp[i] & 0x0f;
        str[j++] = rgascii[k];
        str[j++] = ' ';
    }
    str[j] = 0;                 /* stringify */
}

void showlog(const char * format, ...)
{
    va_list vptr;
    FILE *fd;
    if (flogopen) fd = fdlog;
    else fd = fdmsg;
    if (fd == NULL) return;
    va_start(vptr, format);
    vfprintf(fd, format, vptr);
    va_end(vptr);
}

void logit(const char *format, ...)
{
   va_list vptr;
   char logmsg[MSG_LEN];
   va_start(vptr, format);
   vsnprintf(logmsg, sizeof(logmsg), format, vptr);
   va_end(vptr);
   // strcat(logmsg,"\r\n");
   showit(logmsg);  /* to both fdlog and fdmsg */
   return;
}

#ifdef OLD
void dumpbufr(FILE * fdout, uchar * bufp, int mlen, char *hdr)
{		/* now defunct, use dump_buf or dump_log */
    int i;
    char abuf[5];
 
    if (fdout == NULL) return;
    if (hdr == NULL) hdr = "Buffer";
    fprintf(fdout, "\n%s (len=%d): ", hdr, mlen);
    for (i = 0; i < mlen; i++) {
        if (i % 16 == 0)        /* start a new line */
            fprintf(fdout, "\n%04x: ", i);
        itoh(&bufp[i], 1, abuf);
        fprintf(fdout, "%s", abuf);
    }
    fprintf(fdout, "\n");
    return;
}                               /*end dumpbufr() */    
#endif

void dump_buf(FILE * fdout, uchar * pbuf, int sz, char *tag, char fshowascii)
{
   uchar line[17];
   uchar a;
   int i, j;

   if (fdout == NULL) return;
   if (tag == NULL) tag = "Buffer";
   line[0] = 0; line[16] = 0;
   j = 0;
   if (fdout == NULL) return;
   fprintf(fdout,"%s (len=%d): ", tag,sz);
   for (i = 0; i < sz; i++) {
      if (i % 16 == 0) { j = 0; fprintf(fdout,"%s\n  %04x: ",line,i); }
      if (fshowascii) {
         a = pbuf[i];
         if (a < 0x20 || a > 0x7f) a = '.';
         line[j++] = a;
      }
      fprintf(fdout,"%02x ",pbuf[i]);
   }
   if (j < 16) {
      line[j] = 0;
      for (i = 0; i < (16-j); i++) fprintf(fdout,"   ");
   }
   fprintf(fdout,"%s\n",line);
   return;
} 

void dump_log(uchar * pbuf, int sz, char *tag, char fshowascii)
{
    FILE *fd;
    if (flogopen) fd = fdlog;
    else fd = fdmsg;
    dump_buf(fd, pbuf,sz,tag,fshowascii);
}

int get_line(FILE * fd, char *buf, int len)
{
    int i;
    int ch;
    for (i = 0; i < len; i++) {
	ch = fgetc(fd);
	if (ch == EOF)
	    return (i);
	buf[i] = (char) ch;
	if (ch == 0x0a)
	    return (++i);
    }
    return (i);
}

/*
 * IsSATA
 * Returns 1 (TRUE) if this index in devlist points to a valid SATA disk.
 */
int IsSATA(int i)
{
   if ((devlist[i].devtype == DEV_EMUL) &&   /*Emul*/
       (strncmp(devlist[i].vendor,"ATA",3) == 0) ) {
        /* It is a live SATA disk (via sg emulation) */
        /* Can only use it if it has a valid serial number though. */
        if (devlist[i].sernum[0] != 0 &&
            devlist[i].sernum[0] != ' ') {
             if (fdebug) dump_log(devlist[i].sernum,8,"ATA sernum",1);
	     return(1);
	}
   } /*endif ATA*/
   return(0);
}

/* 
 * findmatch 
 * returns offset of the match if found, or -1 if not found.  
 */
int
findmatch(char *buffer, int sbuf, char *pattern, int spattern, char figncase)
{
    int c, i, j, imatch;

    j = 0;
    imatch = 0;
    for (j = 0; j < sbuf; j++) {
        if ((sbuf - j) < spattern && imatch == 0) return(-1);
        c = buffer[j];
        if (c == pattern[imatch]) {
            imatch++;
        } else if ((figncase == 1) &&
                   ((c & 0x5f) == (pattern[imatch] & 0x5f))) {
            imatch++;
        } else if (pattern[imatch] == '?') {  /*wildcard char*/
            imatch++;
        } else {
            if (imatch > 0) {
               imatch = 0;
               if (j > 0) j--; /* try again with the first match char */
	    }
        }
        if (imatch == spattern) break;
    }
    if (imatch == spattern) {
        i = (j+1) - imatch;  /*buffer[i] is the match */
        return(i);
    } else return (-1);  /*not found*/
}				/*end findmatch */

int fillstr(char *to, uchar *from,int len)
{
        int i;
        uchar c;
        for (i = 0; i < len; i++) {
                c = from[i];
                if (c == 0) c = ' ';
                to[i] = c;
        }
        to[i] = 0;
        return(i);
}

#ifdef NOT
static uchar
cksum(const uchar *buf, register int len)
{
        register uchar csum;
        register int i;
 
        /* 8-bit 2s compliment checksum */
        csum = 0;
        for (i = 0; i < len; i++)
           csum = (csum + buf[i]) % 256;
        csum = -csum;
        return(csum);
}
#endif

#define SES_ENCL_CFG     1
#define SES_ENCL_PAGE    2
#define SES_ARRAY_PAGE   6
#define SES_ELEM_DEVICE   0x01 /*Device */
#define SES_ELEM_PWR_SUP  0x02 /*Power supply*/
#define SES_ELEM_TEMP_SNS 0x04 /*Temp sensor*/
#define SES_ELEM_P_TRANS  0x0f /*SCSI port/transceiver*/
#define SES_ELEM_T_PORT   0x14 /*SCSI target port*/
#define SES_ELEM_SUBENC   0x16 /*Simplet Sub-enclosure*/
#define SES_ELEM_ARRAY    0x17 /*Array device*/
#define SES_NUM_ELEM    5
uchar elem_types[SES_NUM_ELEM] = { SES_ELEM_ARRAY, SES_ELEM_DEVICE, 
            SES_ELEM_P_TRANS, SES_ELEM_T_PORT, SES_ELEM_SUBENC };
/* 
 * ses_decode_types
 * One of the above SES Element Types should be present, representing 
 * the number of disk slots. 
 * buf        = input: SES Enclosure Configuration buffer
 * blen       = input: length of buf
 * nsestypes  = output: number of SES enclosure types 
 * ises;      = output: index of SES enc type for slots 
 * nslots     = return value: number of slots 
 */
int ses_decode_types(uchar *buf, int blen, int *nsestypes, int *ises)
{
    int nslots, nelemt, istart, iend;
    int i,j, n;
    nslots = 1;
    nelemt = buf[10];
    *nsestypes = nelemt;  /* set global num types */
    istart = 12 + buf[11];
    iend   = istart + (nelemt * 4);
    n = 0;
    for (i = istart; i < iend; i +=4)
    {
        if (fdebug) { 
	    logit("buf[%d] = %02x\n",i,buf[i]);
            switch(buf[i]) {
		case SES_ELEM_DEVICE: 
		   logit("SES: num Device Slots = %d\n",buf[i+1]);
		   break;
		case SES_ELEM_ARRAY: 
		   logit("SES: num Array Slots = %d\n",buf[i+1]);
		   break;
		case SES_ELEM_PWR_SUP: 
		   logit("SES: num Power Supplies = %d\n",buf[i+1]);
		   break;
		case SES_ELEM_TEMP_SNS: 
		   logit("SES: num Temp Sensors = %d\n",buf[i+1]);
		   break;
		default:    /*do nothing*/ ;
            }
        }  /*endif debug*/
        for (j = 0; j < SES_NUM_ELEM; j++)
        {
            if (buf[i] == elem_types[j]) break;
        }
        if (j < SES_NUM_ELEM) {  /*this is the one for the slots*/
            nslots = buf[i+1];
            *ises = n;
        }
        n += 1 + buf[i+1];
    }
    return(nslots);
}

/*
 * ses_write_slots 
 * idev: devlist index to the disk device targeted
 * val:  1=set inserted/ok status, 2=set faulty status
 */
int ses_write_slots(int idev, int val)
{
   int ret = 0;
   uchar ctlbuf[300];
   int ctllen;
   int i, j, sgfd, k, n;
   int slotnum = 0;

   // if (fses == 0) return(-1);  /* not SES, do nothing */
   if (fdebug) printf("ses_write_slots(%d): nslots = %d\n",idev,nslots);
   /* set slot status to faulty for the indicated one */
   for (i = 0; i < ndev; i++) {
      if (((devlist[i].devtype == DEV_PROC) || (devlist[i].devtype == DEV_ENCL))
          && (devlist[i].bus == devlist[idev].bus) ) 
      {  /* a matching SES HSC device on the same bus */
	sgfd = open(devlist[i].fname, O_RDWR);
	if (sgfd < 0) {
	   if (fdebug) 
		printf("%s open errno: %d\n",devlist[i].fname,errno);
	   return(errno);
        } 
        /* find pslot and nslots for this enclosure */
        for (j=0; j<islot; j++)
           if (slot[j].bus == devlist[idev].bus) break;
        if (j >= islot) j = 0;
        pslot = &slot[j];
        nslots = devlist[i].dcap;

        /* read the existing array status buffer */
        memset(ctlbuf,0,sizeof(ctlbuf));
        ret = read_ses(sgfd, SES_ENCL_PAGE, ctlbuf, sizeof(ctlbuf)); 
        if (ret < 0) return(ret);
        ctllen = ret;
        /* set the slot status as indicated */
        n = (ctlbuf[2] << 8) + ctlbuf[3] + 4;
        k = 0;
        ctlbuf[1] &= 0x0f;  /*h.o. bits reserved*/
        for (j = 8; j < n; j += 4) {  /*element statuses*/
	    slotnum = (k - isestype); /* one-based */
            memset(&ctlbuf[j],0,4);  /*do not select, clear bits by default*/
	    if ((k > isestype) && (k <= (isestype+nslots)) 
                && (slotnum == devlist[idev].tgt))  /*slotnum == target id*/
            {
                ctlbuf[j]   = 0x80;  /*select, to set this one*/
                if (val == 2) {    /*turn on disk fault led*/
                   ctlbuf[j+1] = 0x00;  /*ok*/
                   ctlbuf[j+2] = 0x02;   /*0x02= rqst ident*/
                   ctlbuf[j+3] = 0x00;   /*0x20= rqst fault*/
                   /* ctlbuf[j+2] = 0x00; *0x08= rqst insert*/
                   /* ctlbuf[j+3] = 0x20; *0x20= rqst fault*/
                } else {  /* val==1, turn off disk fault led */
                   ctlbuf[j+1] = 0x00;  /*ok*/
                   ctlbuf[j+2] = 0x00;  /*0x08= rqst insert, 01=rqst ident*/
                   ctlbuf[j+3] = 0x00;  /*0x20= rqst fault*/
                }
		if (fdebug) 
		    logit("%d:ctlbuf[%d]: %02x %02x %02x %02x\n",k,j,
			    ctlbuf[j], ctlbuf[j+1], ctlbuf[j+2], ctlbuf[j+3]);
            }
            k++;
        }
        // ctllen = n;
        /* write the control buffer to the enclosure */
	ret = write_ses(sgfd, SES_ENCL_PAGE, ctlbuf,ctllen);
	if (ret != 0 || fdebug) {
           logit("write_ses ret=%d\n",ret);
           if (fdebug) dump_log(ctlbuf,18,"write_ses sense data",0);
	}
	close(sgfd);
      }  /*endif HSC*/
   }
   return(ret);
}  /*end ses_write_slots*/

/*
 * read_ses
 * Use the SES commands to read hot-swap controller info.
 * SES status commands utilize the ReceiveDiagnosticResults
 * SCSI command.
 */
int 
read_ses(int sgfd, int page, uchar * buf, int len)
{
    uchar cdb[6] = { 0x1C, 0x01, 0x02, 0x00, 0x50, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    uchar sesbuf[SGHDR +6 + 200];
    struct sg_header *sghp = (struct sg_header *) sesbuf;
    int max_data = 200;
    int i;
    memset(sesbuf,0,sizeof(sesbuf));
    if (len > max_data) len = max_data;
    rsplen = SGHDR + len;
    cmdlen = SGHDR + sizeof(cdb);
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12377;
    memcpy(sesbuf + SGHDR, cdb, sizeof(cdb));
    sesbuf[SGHDR + 2] = page;
    sesbuf[SGHDR + 3] = (len & 0xff00) >> 8;
    sesbuf[SGHDR + 4] = len & 0x0ff;
    ret = write(sgfd, sesbuf, cmdlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, sesbuf, rsplen);
	if (ret < 0) {
		for (i = 0; i < max_try && errno == EAGAIN; i++) {
			sleep(1);
			ret = read(sgfd, sesbuf, rsplen);
			if (ret >= 0) break;
		}
		if (ret < 0) {
			if (fdebug)  
			   printf("read_ses: read errno = %d\n",errno);
			sts = SREAD_ERR;
		}
	}
    }
    if (sghp->sense_buffer[0] != 0) {
	   /* Copy sense data into buffer */
           if (len > MAX_SENSE) len = MAX_SENSE;
           memcpy(buf, sghp->sense_buffer, len);
           if (fdebug) {
		char output[80];
		sprintf(output,"read_ses sense=%x:%x:%x\n",
				buf[2],buf[12],buf[13]);
		showit(output);
           }
	   return (SCHECK_CND);
	   }
    if (sts == 0) {  /* copy data into buffer */
	if (fdebug) {
	   logit("read_ses config[%x] (%d/%d): ",page,ret,len);
           dump_log(sesbuf,ret," ",1);
	}
	if (sesbuf[SGHDR] != page) return(SIZE_ERR);
	if (ret > (len + SGHDR)) ret = len; /*max*/
        else if (ret <= SGHDR) ret = 0;  /*min*/
        else ret -= SGHDR;
	memcpy(buf, sesbuf + SGHDR, ret);
        sts = ret;
    }
    return (sts);
}  /*end read_ses*/

/*
 * write_ses
 * Use the SES commands to set hot-swap controller info.
 * SES control commands utilize the SendDiagnostic SCSI 
 * command.
 */
int 
write_ses(int sgfd, int page, uchar * buf, int len)
{
    uchar cdb[6] = { 0x1D, 0x01, 0x02, 0x00, 0x50, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    uchar sesbuf[SGHDR +6 + 200];
    struct sg_header *sghp = (struct sg_header *) sesbuf;
    int max_data = 200;
    int i;
    memset(sesbuf,0,sizeof(sesbuf));
    if (len > max_data) len = max_data;
    rsplen = SGHDR + sizeof(cdb);
    cmdlen = SGHDR + sizeof(cdb) + len;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12377;
    memcpy(sesbuf + SGHDR, cdb, sizeof(cdb));
    sesbuf[SGHDR + 2] = page;
    sesbuf[SGHDR + 3] = (len & 0xff00) >> 8;
    sesbuf[SGHDR + 4] = len & 0x0ff;
    i = SGHDR + sizeof(cdb);
    memcpy(&sesbuf[i],buf,len);
    if (fdebug) {
        dump_log(&sesbuf[SGHDR],sizeof(cdb),"write_ses: cdb",0);
        dump_log(buf,len,"write_ses: data",1);
    }
    ret = write(sgfd, sesbuf, cmdlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, sesbuf, rsplen);
	if (ret < 0) {
		for (i = 0; i < max_try && errno == EAGAIN; i++) {
			sleep(1);
			ret = read(sgfd, sesbuf, rsplen);
			if (ret >= 0) break;
		}
		if (ret < 0) {
			if (fdebug)  
			   printf("write_ses: read errno = %d\n",errno);
			sts = SREAD_ERR;
		}
	}
    }
    if (sghp->sense_buffer[0] != 0) {
	   /* Copy sense data into buffer */
           if (len > MAX_SENSE) len = MAX_SENSE;
           memcpy(buf, sghp->sense_buffer, len);
           if (fdebug) {
		char output[80];
		sprintf(output,"write_ses sense=%x:%x:%x\n",
				buf[2],buf[12],buf[13]);
		showit(output);
           }
	   return (SCHECK_CND);
	   }
    if (sts == 0) {  /* copy data into buffer */
	if (fdebug) {
	   printf("write_ses config[%x] (%d/%d): ",page,ret,len);
	   for (i=0; i<ret; i++) {
		if ((i % 16) == 0) printf("\n");
		printf("%02x ",sesbuf[i]);
	   }
	   printf("\n");
	}
    }
    return (sts);
}  /*end write_ses*/

int check_ses(int idev, int sgfd)
{
   int ret = 0;
   uchar sdbuf[200];  /*SGDDATALEN=65 bytes*/
   char output[80];
   int i, k, n, s;
   int slotnum = 0;
   /* 
    * If the device supports SES, read the configuration
    * and enclosure status pages.
    */
    /* Read SES config page */
    if (fses == 0) {   // was (nslots == MAX_DEVS)
       ret = read_ses(sgfd, SES_ENCL_CFG, sdbuf, sizeof(sdbuf));
       if (ret >= 0) {
          sprintf(output,"[%d] is an SES device\n",idev);
          showit(output);
          devlist[idev].fses = 1;
          fses = 1;
          if (fdebug) dump_log(sdbuf,ret,"ses_buf1",1);
	  nslots = ses_decode_types(sdbuf,ret,&nsestypes,&isestype);
          if (fdebug) printf("nslots=%d nses=%d, ises=%d\n", /*++++*/
				nslots,nsestypes,isestype);
	  if (nslots > (MAX_DEVS - islot)) nslots = 6; /*sane nslots*/
	  pslot = &slot[islot];
          devlist[idev].dcap = nslots; /*save nslots for this proc*/
	}
     }
     if (nslots == MAX_DEVS) return(ret);
     /* Read SES enclosure status page */
     memset(sdbuf,0,sizeof(sdbuf));
     ret = read_ses(sgfd, SES_ENCL_PAGE, sdbuf, sizeof(sdbuf));
     if (ret >= 0) {
	    /* Show SES enclosure status page */
            if (fdebug) {
                   dump_log(sdbuf,ret,"ses_buf2",1);
                   output[0] = 0;
                   s = sdbuf[1];
                   sprintf(output,"SES overall status =");
                   if (s & 0x10) strcat(output," INVOP");
                   if (s & 0x08) strcat(output," INFO");
                   if (s & 0x04) strcat(output," NONCR");
                   if (s & 0x02) strcat(output," CRIT");
                   if (s & 0x01) strcat(output," UNREC");
		   strcat(output," .\n");
                   showit(output);
            }
	    slotnum = 0;
            n = (sdbuf[2] << 8) + sdbuf[3] + 4;
            k = 0; 
            for (i = 8; i < n; i += 4) {  /*element statuses*/
               s = sdbuf[i];   /*status*/
	       if ((k > isestype) && (k <= (isestype+nslots))) {
		      slotnum = (k - isestype) - 1;
		      pslot[slotnum].stat = s;
	              pslot[slotnum].num = slotnum;
	              pslot[slotnum].bus = devlist[idev].bus;
	              pslot[slotnum].ins = 0xFFFF; 
	       }
               if (fdebug) {
	          char tmpbuf[40];
                  output[0] = 0;
	          if ((k > isestype) && (k <= (isestype+nslots))) 
                      sprintf(output,"SES[%d] %d slot[%d] status = %02x",
				i/4,k,slotnum,s);
	          else 
                      sprintf(output,"SES[%d] other %d status =",i/4,k);
                  if (s & 0x40) strcat(output," PRDFAIL");
                  if (s & 0x10) strcat(output," SWAP");
                  switch(s & 0x0f) {
                     case 0x01: strcat(output," OK"); break;
                     case 0x02: strcat(output," CRIT"); break;
                     case 0x03: strcat(output," NONCRIT"); break;
                     case 0x04: strcat(output," UNRECOV"); break;
                     case 0x05: strcat(output," NOTINST"); break;
                     case 0x06: strcat(output," UNKNOWN"); break;
                     case 0x07: strcat(output," NOTAVAIL"); break;
                     default:   sprintf(tmpbuf," 0x%02x",s); 
			    strcat(output,tmpbuf);
			    break;
                  }
	          strcat(output,"\n");
                  showit(output);
               } /*endif fdebug*/
               k++;
            } /*for loop*/
    }
    if (ret >= 0) ret = 0;
    return(ret);
}  /*end check_ses*/

/*
 * read_safte
 * Use the SAF-TE commands to read hot-swap backplane info.
 * where
 *   iproc is the devlist index of the HSBP proc to be read
 *   mode  is:
 *     0x00 - Read Enclosure Configuration
 *     0x01 - Read Enclosure Status
 *     0x02 - Read Usage Statistics
 *     0x03 - Read Device Insertions
 *     0x04 - Read Device Slot Status
 *     0x05 - Read Global Flags
 */
int 
read_safte(int sgfd, int mode, uchar * buf, int len, int iproc)
{
    uchar cdb[10]   = { 0x3C, 0x01, 0x00, 0, 0, 0, 0, 0x00, 0x50, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    uchar saftebuf[SGHDR +10 + 200];
    struct sg_header *sghp = (struct sg_header *) saftebuf;
    uchar *pcmd;  
    int max_data = 200;
    char fvsc = 0;
    int i, j;

    memset(saftebuf,0,sizeof(saftebuf));
    if (len > max_data) len = max_data;
    rsplen = SGHDR + len;
    cmdlen = SGHDR + sizeof(cdb);
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12387;
    pcmd = &saftebuf[SGHDR]; 
    if ((devlist[iproc].devtype == DEV_ENCL) && 
        (strncmp(devlist[iproc].vendor,"ESG-SHV",7) == 0)) fvsc = 1;
    if (fvsc) {   /* VSC410 Encl type, special WriteBuffer commands */
        if (fdebug) showlog("read_safte(%d,VSC410)\n",mode);
        // return(SWRITE_ERR);
#ifdef NOT
	/* Use SES instead with the VSC 410. */
        int j, k;
        j = 0;  /* start */
        k = 7;  /* len */
        pcmd[0] = 0x3B;     /* SCSI write_buffer command */
        pcmd[1] = 0x02;     /* set write buffer mode = data */
        pcmd[2] = 0;        /* buffer id */
        pcmd[3] = 0xff & (j >> 16);  /*start*/
        pcmd[4] = 0xff & (j >> 8);
        pcmd[5] = 0xff & j; /* cdb[3,4,5] = buffer offset (start) */
        pcmd[6] = 0xff & (k >> 16);  /* len */
        pcmd[7] = 0xff & (k >> 8);
        pcmd[8] = 0xff & k;
        pcmd[9] = 0;
        cmdlen = SGHDR + 10;  // sizeof(wbcdb);
        pcmd += 10;   // sizeof(wbcdb);
        pcmd[0] = 0xC0;   /*SEP I2C slave addr*/
        pcmd[1] = 0x00;   /*Read buffer*/
        pcmd[2] = cksum(pcmd,2);
        pcmd[3] = 0x54;   /*SEMB I2C slave addr */
        pcmd[4] = 0x00;   /*SEQ*/
        pcmd[5] = mode;   /*SEP CMD*/
        pcmd[6] = cksum(&pcmd[3],3); 
        pcmd += 7;
        cmdlen += 7;
#endif
    } 
    {
        memcpy(pcmd, cdb, sizeof(cdb));
        pcmd[2] = mode;
        pcmd[7] = (len & 0xff00) >> 8;
        pcmd[8] = len & 0x0ff;
    }
    if (fdebug) {
        showlog( "read_safte/%d cdb (len=%d): ", mode,cmdlen);
        for (i = SGHDR; i < cmdlen; i++)
            showlog( "%02x ", saftebuf[i]);
        showlog( "\n");
    }
    ret = write(sgfd, saftebuf, cmdlen);
    if (ret < 0) {
	sts = SWRITE_ERR;
	if (fdebug) printf("read_safte/%d write errno = %d\n",mode,errno);
    }
    else {
	ret = read(sgfd, saftebuf, rsplen);
	if (ret < 0) {
		for (i = 0; i < max_try && errno == EAGAIN; i++) {
			sleep(1);
			ret = read(sgfd, saftebuf, rsplen);
			if (ret >= 0) break;
		}
		if (ret < 0) {
			if (fdebug) 
				printf("read_safte: read errno = %d\n",errno);
			sts = SREAD_ERR;
		}
	}
	if (ret >= 0) rsplen = ret;
    }
    if (sghp->sense_buffer[0] != 0) {
           uchar k,a,q;
	   /* Copy sense data into buffer */
           memcpy(buf, sghp->sense_buffer, MAX_SENSE);
           k = buf[2] & 0x0f;
           a = buf[12];
           q = buf[13];
	   if (fdebug) 
              showlog( 
		 "read_safte/%d sense %x/%x/%x (sts=%d len=%d,%d)\n", 
		 mode,k,a,q, sts,len,ret); 
#ifdef TEST
           if (k == 0 && a == 0 && q == 0) ;  /*keep going for VT (?)*/
	   else 
#endif
           return (SCHECK_CND);
	   }
    if (fdebug) {
        showlog( "read_safte/%d resp (len=%d): ", mode,rsplen);
        dump_log(&saftebuf[SGHDR],(rsplen-SGHDR),"read_safte",1);
    }
    if (sts == 0) {
        int istart = SGHDR;
        if (fvsc) {  /* VSC410 Encl type */
           sts = saftebuf[6];  /*SEP status*/
           istart += 7; 
        }
	j = 0;
	if (rsplen > len) rsplen = len;
        for (i = istart; i < rsplen; i++) buf[j++] = saftebuf[i];
    }
    return (sts);
}				/* end read_safte() */

/*
 * write_safte
 * Use the SAF-TE commands to write hot-swap backplane info.
 * where
 *   iproc is the devlist index of the proc
 *   idev  is the devlist index of the disk to set
 *   val   is the value to be set
 *   mode  is:
 *     10h - Write Device Slot Status
 *     11h - Set SCSI ID
 *     12h - Perform Slot Operation
 *     13h - Set Fan Speed
 *     14h - Activate Power Supply
 *     15h - Send Global Command
 */
int 
write_safte(int sgfd, int mode, int iproc, int idev, int val)
{
    uchar cdb[10] = { 0x3B, 0x01, 0x00, 0, 0, 0, 0, 0x00, 0x50, 0 };
    int ret, sts = 0;
    int rsplen, cmdlen;
    uchar saftebuf[SGHDR +10 + 200];
    struct sg_header *sghp = (struct sg_header *) saftebuf;
    int max_data = 200;
    char fvsc = 0;
    uchar *pb;
    int len;
    int i;

    if ((devlist[iproc].devtype == DEV_ENCL) &&
        (strncmp(devlist[iproc].vendor,"ESG-SHV",7) == 0)) fvsc = 1;
    if (fvsc) {  /* VSC410 Encl type */
        if (fdebug) printf("write_safte(%d,Encl)\n",mode);
        // return(SWRITE_ERR);
    }
    len = 1 + (nslots * 3);
    if (len > max_data) len = max_data;
    rsplen = SGHDR;
    cmdlen = SGHDR + sizeof(cdb) + len;
    sghp->reply_len = rsplen;
    sghp->twelve_byte = 0;
    sghp->other_flags = 0;	/* some apps assume this is done */
    sghp->sense_buffer[0] = 0;	/* only needed for old sg driver */
    sghp->pack_id = 12388;
    memcpy(saftebuf + SGHDR, cdb, sizeof(cdb));
    saftebuf[SGHDR + 7] = (len & 0xff00) >> 8;
    saftebuf[SGHDR + 8] = len & 0x0ff;

    mode = 0x10;  /* write slot status */
    pb = &saftebuf[SGHDR + 10];
    pb[0] = mode;
    pb++;
    for (i = 0; i < nslots; i++) {
	if (pslot[i].id == devlist[idev].tgt) {
	    pslot[i].wstat = val;
	    if (fdebug)
	       showlog("write_safte: setting slot %d to %02x\n",i,val);
        } else if (pslot[i].stat & 0x01) 
	      pslot[i].wstat = 0x01;  /*inserted, no error*/
        else  pslot[i].wstat = 0x80;  /*no drive*/
	pb[0] = pslot[i].wstat; /* 01h = ok, 02h = faulty, etc. */
	pb[1] = 0;  /* reserved */
	pb[2] = 0;  /* reserved */
	pb += 3;
    }
    if (fdebug) {
        showlog( "write_safte/%d cdb (len=%d): ", mode,cmdlen);
        dump_log(&saftebuf[SGHDR],(cmdlen-SGHDR)," ",1);
    }
    ret = write(sgfd, saftebuf, cmdlen);
    if (ret < 0) sts = SWRITE_ERR;
    else {
	ret = read(sgfd, saftebuf, rsplen);
	if (ret < 0) {
		for (i = 0; i < max_try && errno == EAGAIN; i++) {
			sleep(1);
			ret = read(sgfd, saftebuf, rsplen);
			if (ret >= 0) break; 
		}
		if (ret < 0) {
			if (fdebug)  
			   printf("write_safte: read errno = %d\n",errno);
			sts = SREAD_ERR;
		}
	}
    }
    if (sghp->sense_buffer[0] != 0) {
	   /* Copy sense data into buffer */
           // memcpy(buf, sghp->sense_buffer, MAX_SENSE);
	   return (SCHECK_CND);
	   }
    return (sts);
}				/* end write_safte() */

/*
 * write_slots 
 * idev: devlist index to the disk device targeted
 * val:  1=set inserted/ok status, 2=set faulty status
 */
int write_slots(int idev, int val)
{
   char buf[80];
   int i, j, sgfd;
   int ret = 0;

   if (fsafte == 0) return(-1);  /* not safte, do nothing */
   if (fdebug) printf("write_slots(%d): nslots = %d\n",idev,nslots);

   /* set slot status to faulty for the indicated one */
   for (i = 0; i < ndev; i++) {
      if (((devlist[i].devtype == DEV_PROC) || (devlist[i].devtype == DEV_ENCL))
          && (devlist[i].bus == devlist[idev].bus) ) /*ARC*/
      {  /* a SAF-TE HSBP device on the same bus */
	sgfd = open(devlist[i].fname, O_RDWR);
	if (sgfd < 0) {
	   if (fdebug) 
		printf("%s open errno: %d\n",devlist[i].fname,errno);
	   return(errno);
        } 
        /* find pslot and nslots for this proc */
        for (j=0; j<islot; j++)
           if (slot[j].bus == devlist[idev].bus) break;
        if (j >= islot) j = 0;
        pslot = &slot[j];
        nslots = devlist[i].dcap;
        /* set the slot status as indicated */
	ret = write_safte(sgfd,SAFTE_WRITE_DEV_STAT,i,idev,val);
	if (ret != 0 || fdebug) {
	   sprintf(buf,"write_safte ret=%d\n",ret);
	   showit(buf);
	}
	close(sgfd);
        // break;
      }  /*endif HSBP*/
   }
   return(ret);
}

/* 
 * sg_cmd
 * Send any SCSI Command Descriptor Block to the open
 * device handle, using the SCSI Generic interface.
 */
int
sg_cmd(int sgfd, uchar *cdb,int cdblen, uchar *rdata,int rlen)
{
        struct {
           struct sg_header hdr;
           unsigned char data[SGDATALEN];
           } sg_rq;
	uchar *sgbuf;
        int ret = 0;
	int i;
 
        /* Send sg cdb request */
	/* Assumes any write data is already in the cdb */
	/* The rdata parameter is for receive data */
        if (rlen > SGDATALEN) rlen = SGDATALEN;
        sg_rq.hdr.reply_len = SGHDR + rlen;
        sg_rq.hdr.twelve_byte = 0;
        sg_rq.hdr.other_flags = 0;
        sg_rq.hdr.pack_id     = 12397;  /* any number here */
        sg_rq.hdr.sense_buffer[0] = 0;
        memcpy(sg_rq.data,cdb,cdblen);
	sgbuf = (uchar *)&sg_rq.hdr;
	if (fdebug) {
	   showlog( "sg_cmd[%x] cdb (len=%d): ", cdb[0],cdblen);
	   for (i = 0; i < cdblen; i++) showlog( "%02x ", cdb[i]);
	   showlog( "\n");
	}
        ret = write(sgfd, sgbuf, SGHDR + cdblen);
        if (ret < 0) return (SWRITE_ERR); 
        /* Read sg response data */
        ret = read(sgfd, sgbuf, SGHDR + rlen);
	if (ret < 0) {
		for (i = 0; i < max_try && errno == EAGAIN; i++) {
			sleep(1);
        		ret = read(sgfd, sgbuf, SGHDR + rlen);
			if (ret >= 0) break;
		}
		if (ret < 0) {
			if (fdebug) 
			    showlog("sg_cmd: read ret=%d errno = %d\n",
					ret,errno);
			ret = SREAD_ERR;
		}
	}
        if (sg_rq.hdr.sense_buffer[0] != 0) {
	   /* Copy sense data into buffer */
           if (rlen > MAX_SENSE) rlen = MAX_SENSE;
           memcpy(rdata,sg_rq.hdr.sense_buffer,rlen);
           /* May want to call sg_chk_err/show_sense_err from sgerr.c */
	   ret = SCHECK_CND;
	   }
	if (ret < 0) return(ret);
        /* Success, copy response data */
	if (fdebug) {
	   showlog( "sg_cmd[%x] resp ", cdb[0]);
           dump_log(&sgbuf[SGHDR],(ret-SGHDR)," ",1);
	}
	/* Note that this subroutine does a data copy, but for 
	   small sizes like this, it is ok. */
	ret -= SGHDR;  /* length returned from read, less header */
	if (ret > rlen) ret = rlen;
        memcpy(rdata,sg_rq.data,ret);
        return (ret);;
}  /*end sg_cmd*/


static int find_lun(DEVFS_LIST * devfsnums)
{
    	char *devfs_lun = "/dev/scsi/host%d/bus%d/target%d/lun%d";
	char devfs_lun_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->lun; i < 16; i++)
	{
		sprintf(devfs_lun_buf, devfs_lun, devfsnums->host, devfsnums->bus, devfsnums->target, i);
		if(stat(devfs_lun_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode)){
				devfsnums->lun=i;
				break;
			}
		}
	}	
	if(i >= 15){
		devfsnums->lun=0;
		return -1;
	}
	return 1;
}

static int find_target(DEVFS_LIST * devfsnums)
{
    	char *devfs_target = "/dev/scsi/host%d/bus%d/target%d";
	char devfs_target_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->target; i < 16; i++)
	{
		sprintf(devfs_target_buf, devfs_target, devfsnums->host, devfsnums->bus, i);
		if(stat(devfs_target_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->target=i;
				if (find_lun(devfsnums) == 1)
					break;
		}
	}
	if(i >= 16){
		devfsnums->target=0;
		return -1;	
	}
	return 1;
}

static int find_bus(DEVFS_LIST * devfsnums)
{
    	char *devfs_bus = "/dev/scsi/host%d/bus%d";
	char devfs_bus_buf[90];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->bus ; i < 16; i++)
	{
		sprintf(devfs_bus_buf, devfs_bus, devfsnums->host,i);
		if(stat(devfs_bus_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->bus=i;
				if (find_target(devfsnums) == 1)
					break;
		}
	}	
	if(i >= 16){
		devfsnums->bus=0;
		return -1;
	}
	return 1;
}

static int find_host(DEVFS_LIST * devfsnums)
{
    	char *devfs_host = "/dev/scsi/host%d";
	char devfs_host_buf[80];
	struct stat stbuf;
	int i;
	for ( i = devfsnums->host ; i < 16; i++)
	{
		sprintf(devfs_host_buf, devfs_host, i);
		if(stat(devfs_host_buf , &stbuf) == 0) {
                	if(S_ISDIR(stbuf.st_mode))
				devfsnums->host=i;
				if (find_bus(devfsnums) == 1)
					break;
		}
	}	
	if(i >= 16)
		return -1;
	return 1;
}

static int get_first_dev_sg(DEVFS_LIST * devfsnums)
{
	if (find_host(devfsnums) == -1){
		devfsnums->all_leaves_searched=1;
		return 0; 
	}
	else
		return 1;

}

void
make_dev_name(int k, char *sg_name, char *sd_name, int fnum, 
		DEVFS_LIST *devfsnums)
{
    char buff[64];
    int big, little;
    char *fname;
    int fnumeric, i;
    char *devfs_pat = "/dev/scsi/host%d/bus%d/target%d/lun%d/";
 
    if (k == 0) {  /* first time, detect if DevFS */
        struct stat stbuf;
        /* if /dev/scsi/host0 or host1 directory exists, we are using DevFS */
        if (stat("/dev/scsi/host0",&stbuf) == 0) {
            if (S_ISDIR(stbuf.st_mode))  fdevfs = 1;
        } else if (stat("/dev/scsi/host1",&stbuf) == 0) {
            if (S_ISDIR(stbuf.st_mode))  fdevfs = 1;
        }
    }
    if (fdevfs) {  /* DevFS */
	if(get_first_dev_sg(devfsnums)){
		sprintf(buff,devfs_pat,
			devfsnums->host,devfsnums->bus,
			devfsnums->target,devfsnums->lun++); /* ~46 chars*/
	} else strcpy(buff,"/dev/scsi/end/");  /* something, but invalid */
        /*currently sizeof sg_name is 64 chars */
        strcpy(sg_name,buff);  
	strcat(sg_name,"generic");
        strcpy(sd_name,buff); 
	strcat(sd_name,"disc");
    } else {              /* normal dev names */ 
       strcpy(sg_name, "/dev/sg");
       strcpy(sd_name, "/dev/sd");
       for (i = 0; i < 2; i++)
       {
         if (i == 0)     { fname = sg_name; fnumeric = fnum; }
         else /*i == 1*/ { fname = sd_name; fnumeric = 0; }  
         if (fnumeric) {
           sprintf(buff, "%d", k);
           strcat(fname, buff);
         } else {  /*alpha*/
           if (k < 26) {
               buff[0] = 'a' + (char) k;
               buff[1] = '\0';
               strcat(fname, buff);
           } else if (k <= 255) {  /* assumes sequence goes x,y,z,aa,ab,ac */
               big = k / 26;
               little = k - (26 * big);
               big = big - 1;
    
               buff[0] = 'a' + (char) big;
               buff[1] = 'a' + (char) little;
               buff[2] = '\0';
               strcat(fname, buff);
           } else
               strcat(fname, "xxxx");
         } /*end else alpha*/
       } /*end for*/
    } /*end-else normal names*/
}

/*
 * get_dev_maj_min
 * Use sysfs (/sys/bus/scsi) to get the major+minor numbers
 * of a specified scsi device.
 * Note that there is new file naming in kernel 2.6.16.
 */
int get_dev_maj_min(uchar bus, uchar ch, uchar id, uchar lun, 
			int *pmaj, int *pmin)
{
   char sysdev_file[64];
   char mmstr[16];
   FILE *fp;
   char *sp;

   sprintf(sysdev_file,"/sys/bus/scsi/devices/%d:%d:%d:%d/block/dev", 
		bus,ch,id,lun);
   fp = fopen(sysdev_file, "r");
   /* printf("fopen(%s) fp = %p\n",sysdev_file,fp);     *DEBUG++++*/
   if (fp == NULL) return -1;
   sp = fgets(mmstr,sizeof(mmstr),fp);
   /* printf("fget() sp = %p, mmstr: %s\n",sp,mmstr);   *DEBUG++++*/
   if (sp == NULL) return -1;
   sscanf(mmstr, "%u:%u", pmaj, pmin);
   return 0;
}

/* 
 * find_mmdev
 * find a device name by its major+minor numbers
 */
int find_mmdev(int maj, int min, char *devname)
{
   int dmaj, dmin;
   int match = 0;
   char *dev_dir = "/dev";
   char device_path[ PATH_MAX + 1];
   DIR *dirp;
   struct dirent *dep;
   struct stat stats;
   int rv = -1;

   dirp = opendir (dev_dir);
   if (dirp == NULL) return(rv);
   while (1)
   {
      dep = readdir (dirp);
      if (dep == NULL) break;
      snprintf (device_path, PATH_MAX, "%s/%s", dev_dir, dep->d_name);
      device_path [PATH_MAX] = '\0';
      if (lstat(device_path, &stats)) continue;
 
      /* Skip non-block/char files. */
      if ( (!S_ISBLK(stats.st_mode)) && (!S_ISCHR(stats.st_mode)) )
          continue;
      dmaj = major(stats.st_rdev);
      dmin = minor(stats.st_rdev);
      if ((maj == dmaj) && (min == dmin)) {
          strcpy(devname,device_path);
          match = 1; 
	  rv = 0;
          break;
      }
   }  /*end while*/
   return(rv);
}

/* 
 * find_blockdev
 * find a device name by its scsi parameters
 * New method for kernel 2.6.16
 */
int find_blockdev(uchar bus, uchar ch, uchar id, uchar lun, 
		  char *devname)
{
   char sysdev_dir[64];
   char device_path[ PATH_MAX + 1];
   DIR *dirp;
   struct dirent *dep;
   int match = 0;
   int rv = -1;

   sprintf(sysdev_dir,"/sys/bus/scsi/devices/%d:%d:%d:%d", bus,ch,id,lun);
   dirp = opendir (sysdev_dir);
   if (dirp == NULL) return(rv);
   devname[0] = 0;  /*initialize to empty device name*/
   /* if here, have sysfs, so if not found, not a block device */
   while (1)
   {
      dep = readdir (dirp);
      if (dep == NULL) break;
      snprintf (device_path, PATH_MAX, "%s/%s", sysdev_dir, dep->d_name);
      device_path[PATH_MAX] = '\0';
      if (strncmp(dep->d_name,"block:",6) == 0) {
          sprintf(devname,"/dev/%s",&dep->d_name[6]);
          match = 1; 
	  rv = 0;
	  break;
      }
   }  /*end while*/
   return(rv);
}

/* 
 * sense_err
 * This routine handles SCSI sense errors, and takes appropriate action.
 * It may issue a reset for unresponsive devices, so this should only
 * do so for devices that are explicitly managed (Disk,Proc,USB).
 */
int 
sense_err(int sgfd, uchar *dbuf, int dlen, int idev, int ret, int *pretry)
{
	int k,a,q;
	int i,j;
	int retry = 0;
	char fmanaged = 0;
	char tryok = 1;

	k = 0; a = 0; q = 0;
        if ((devmatch[idev].devtype == DEV_DISK) || 
	    (devmatch[idev].devtype == DEV_EMUL))
           fmanaged = 1;  /* is only managed if Disk or USB */
        if (ret == SCHECK_CND) {
	     k = dbuf[2] & 0x1f; a = dbuf[12]; q = dbuf[13]; }
        if (fdebug) 
	    logit("sense_err(%d): start ret=%d errno=%d sense=%x:%x:%x\n", 
			idev,ret,errno,k,a,q);
	/* If we get a cmd returning SREAD_ERR/EAGAIN, it has exceeded
	   the retries and we need to do a reset. */
	if (ret == SREAD_ERR && errno == EAGAIN) {
             ret = sg_cmd(sgfd,turcdb,sizeof(turcdb), dbuf,sizeof(dbuf));
             if (fdebug) 
		logit("sense_err: read again, test_unit_ready ret=%d errno=%d\n",
			ret,errno);
	}
	if (ret == SWRITE_ERR && errno == 33) { /*ret=-3 errno=33(EDOM)*/
             ret = sg_cmd(sgfd,turcdb,sizeof(turcdb), dbuf,sizeof(dbuf));
             if (fdebug) 
		logit("sense_err: write errno33, test_unit_ready ret=%d errno=%d\n", ret,errno);
	     retry = 1;
             if (ret >= 0) tryok = 0; /*dont use retry loop below*/
	}
	if (ret < 0 && errno == 33) {  /*EDOM ?*/
		/* Dont try reset if we already know it was failed. */
		/* Also don't reset if not fmanaged. */
		if (fmanaged && (!fskipreset) &&
		    (devmatch[idev].dstatus != DSTAT_FAIL)) {
	           j = 2;  /* scsi bus reset */
	           ioctl(sgfd,SG_SCSI_RESET, &j);
	           logit("sense_err(%d): bus reset, retrying\n",idev);
		   sleep(1);  /* wait for reset to clear */
                   ret = sg_cmd(sgfd,turcdb,sizeof(turcdb),dbuf,sizeof(dbuf));
                   if (fdebug) 
                       logit("sense_err: EDOM test_unit_ready ret=%d errno=%d\n",
			     ret,errno);
	           retry = 1;  /* try again later */
		}
        }
        if (ret == SCHECK_CND) {
	     k = dbuf[2] & 0x1f; a = dbuf[12]; q = dbuf[13]; }
        if (ret >= 0) tryok = 0;

	for (i = 0; i <= NUM_RETRIES && tryok; i++)
	{
          if (fdebug) 
		logit("sense_err loop %d: ret=%d sense=%x:%x:%x\n",
			i,ret,k,a,q);
	  if (ret != SCHECK_CND) { 
             k = 0; a = 0; q = 0;
             retry = 0;  /* do not try command again*/
             tryok = 0;  /* break, exit loop */
             break;  
          } 
          if (k == 2 && a == 4 && q == 2) {
	     /* needs a start unit, so issue it. */
             ret = sg_cmd(sgfd,startcdb,sizeof(startcdb), dbuf,sizeof(dbuf));
             if (fdebug) 
		logit("sense_err loop %d start_unit ret=%d\n",i,ret);
	     retry = 1;
	  } else if (k == 6 && a == 0x29) {
	     /* reset occurred, clear with test-unit-ready */
             ret = sg_cmd(sgfd,turcdb,sizeof(turcdb), dbuf,sizeof(dbuf));
             if (fdebug) {
		if (ret == SCHECK_CND) {
	           k = dbuf[2] & 0x1f; a = dbuf[12]; q = dbuf[13]; }
		logit("sense_err loop %d clear tur, ret=%d sense=%x:%x:%x\n", 
			i,ret,k,a,q); 
		}
	     retry = 1;
             if (ret >= 0) break;
	  } else if ((k == 2 && a == 4 && q == 1) || 
		     (k == 6 && a == 4 && q == 1)) {   /* not ready, other */
	     if (i > 2) {  /* 3rd time not ready, try start unit */
               ret = sg_cmd(sgfd,startcdb,sizeof(startcdb), dbuf,sizeof(dbuf));
               logit("sense_err loop %d notrdy, start_unit ret=%d\n",i,ret);
	       retry = 1;
	     }
	  }
          /* Try TestUnitReady for every retry loop */
          ret = sg_cmd(sgfd,turcdb,sizeof(turcdb),dbuf,sizeof(dbuf));
          if (ret >= 0) break;
	  else if (ret == SCHECK_CND) {
	        k = dbuf[2] & 0x1f; a = dbuf[12]; q = dbuf[13]; }
	  if (ret < 0 && i == NUM_RETRIES) { /*tried, no luck yet, so reset*/
	     if (fdebug) logit("sense_err loop %d max tries.\n",i);
#ifdef DEVRESET
             /* Do further testing to show whether sgraidmon needs this 
              * device reset case included */
	     if (fmanaged && !fskipreset) { 
		/* Only reset if managed device type and no -s option */
	        j = 1;  /* scsi device reset */
	        ioctl(sgfd,SG_SCSI_RESET, &j);
	        logit("sense_err loop %d device reset, retrying.\n",i);
	        sleep(1);  /* wait for reset to clear */
	        /* clear unit attn with tur below */
                ret = sg_cmd(sgfd,turcdb,sizeof(turcdb),dbuf,sizeof(dbuf));
	     }
#endif
	  }
	  if (ret < 0) sleep(1);  /* wait a second each time if error */
	} /*end for NUM_RETRIES*/
	if (ret >= 0) { retry = 1;
	        k = 0; a = 0; q = 0; }
        if (fdebug) logit(
                "sense_err(%d): end ret = %d, retry = %d, sense=%x:%x:%x\n",
			idev,ret,retry,k,a,q);
	*pretry = retry;
	return(ret);
}    /* end sense_err */

/* end sgsubmon.c */
