package yqlib

import (
	"bufio"
	"container/list"
	"errors"
	"fmt"
	"io"
	"os"
	"regexp"
	"strings"

	yaml "gopkg.in/yaml.v3"
)

func readStream(filename string, leadingContentPreProcessing bool) (io.Reader, string, error) {
	var reader *bufio.Reader
	if filename == "-" {
		reader = bufio.NewReader(os.Stdin)
	} else {
		// ignore CWE-22 gosec issue - that's more targeted for http based apps that run in a public directory,
		// and ensuring that it's not possible to give a path to a file outside thar directory.
		file, err := os.Open(filename) // #nosec
		if err != nil {
			return nil, "", err
		}
		reader = bufio.NewReader(file)
	}

	if !leadingContentPreProcessing {
		return reader, "", nil
	}
	return processReadStream(reader)
}

func readString(input string, leadingContentPreProcessing bool) (io.Reader, string, error) {
	reader := bufio.NewReader(strings.NewReader(input))
	if !leadingContentPreProcessing {
		return reader, "", nil
	}
	return processReadStream(reader)
}

func writeString(writer io.Writer, txt string) error {
	_, errorWriting := writer.Write([]byte(txt))
	return errorWriting
}

func processReadStream(reader *bufio.Reader) (io.Reader, string, error) {
	var commentLineRegEx = regexp.MustCompile(`^\s*#`)
	var sb strings.Builder
	for {
		peekBytes, err := reader.Peek(3)
		if errors.Is(err, io.EOF) {
			// EOF are handled else where..
			return reader, sb.String(), nil
		} else if err != nil {
			return reader, sb.String(), err
		} else if string(peekBytes) == "---" {
			_, err := reader.ReadString('\n')
			sb.WriteString("$yqDocSeperator$\n")
			if errors.Is(err, io.EOF) {
				return reader, sb.String(), nil
			} else if err != nil {
				return reader, sb.String(), err
			}
		} else if commentLineRegEx.MatchString(string(peekBytes)) {
			line, err := reader.ReadString('\n')
			sb.WriteString(line)
			if errors.Is(err, io.EOF) {
				return reader, sb.String(), nil
			} else if err != nil {
				return reader, sb.String(), err
			}
		} else {
			return reader, sb.String(), nil
		}
	}
}

func readDocuments(reader io.Reader, filename string, fileIndex int, decoder Decoder) (*list.List, error) {
	decoder.Init(reader)
	inputList := list.New()
	var currentIndex uint

	for {
		var dataBucket yaml.Node
		errorReading := decoder.Decode(&dataBucket)

		if errors.Is(errorReading, io.EOF) {
			switch reader := reader.(type) {
			case *os.File:
				safelyCloseFile(reader)
			}
			return inputList, nil
		} else if errorReading != nil {
			return nil, fmt.Errorf("bad file '%v': %w", filename, errorReading)
		}
		candidateNode := &CandidateNode{
			Document:         currentIndex,
			Filename:         filename,
			Node:             &dataBucket,
			FileIndex:        fileIndex,
			EvaluateTogether: true,
		}

		//move document comments into candidate node
		// otherwise unwrap drops them.
		candidateNode.TrailingContent = dataBucket.FootComment
		dataBucket.FootComment = ""

		inputList.PushBack(candidateNode)

		currentIndex = currentIndex + 1
	}
}
