/*
 *	pimpd - Peter's Ident Masquerading Program Daemon v0.8
 *
 *	Copyright (c) 2000-2002 Peter Nelson <peter@cats.meow.at>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * 	Changes from 0.7 -> 0.8
 * 		Changed -f to only reply with fakeid on an error.
 * 		Added -a to always respond with the fakeid.
 *
 * 	Changes from 0.6 -> 0.7
 * 		Added -f <name> option to always respond with a specified userid.
 * 		Added -x option to respond with the IP address in hexadecimal form
 * 			if request is to be forwarded.
 *		Code cleanups.
 * 		
 *	Changes from 0.5 -> 0.6
 *		Automatically determine gateway with -g.
 *  	
 *	Changes from 0.4 -> 0.5
 *		Request forwarding code cleaned up and fixed by Doug Lim
 *		<dlim@enteract.com>.
 *					
 *	Changes from 0.3 -> 0.4
 *		Added support for 2.4's connection tracking.
 *		Restructed code a little. :)
 *
 *	Changes from 0.2 -> 0.3
 *		Added ~/.noident support. Simply touch ~/.noident to cause
 *			pimpd to withhold username.
 *		Corrected masqueraded port numbers to be what the client
 *			expects them to be.
 *
 *	Changes from 0.1 -> 0.2
 *		Change from using int to in_addr for variables storing IPs.
 *		Updated masquerade error output to be RFC compliant.
 *		Dropped or timed out connections are now logged.
 */	     

#include <stdio.h>
#include <stdlib.h>
#include <pwd.h>
#include <errno.h>
#include <syslog.h>
#include <sys/time.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <arpa/inet.h>
#include <string.h>

#define IP_CONNTRACK	"/proc/net/ip_conntrack"
#define IP_MASQUERADE	"/proc/net/ip_masquerade"
#define PROC_NET_TCP	"/proc/net/tcp" 
#define PROC_NET_ROUTE	"/proc/net/route"
#define PIMPDVERSION	"0.8"

unsigned int noident = 0;
unsigned int hexident = 0;
char *fakeid = NULL;
struct in_addr masqhost;

typedef struct ident {
	struct in_addr localhost, remotehost;
	unsigned int localport, remoteport;
} ident;

/* Retrieve default gateway from system routing table */
struct in_addr getdefaultgateway() {
	char *buf;
	struct in_addr gateway;
	FILE *fh;

	gateway.s_addr = 0;
	fh = fopen(PROC_NET_ROUTE, "r");
	buf = (char *) malloc(160);
	while(fgets(buf, 160, fh))
		sscanf(buf, "%*s 00000000 %x %*x %*x %*x 0 00000000 %*x %*x %*x", &gateway.s_addr);
	free(buf);
	fclose(fh);
	return gateway;
}

/* Return and log errors */
int errorresponse(unsigned int localport, unsigned int remoteport, char *text) {
	if(fakeid) {
		printf("%d , %d : USERID : UNIX :%s\r\n", localport, remoteport, fakeid);
    } else {
		printf("%d , %d : ERROR : %s\r\n", localport, remoteport, text);
	}
	syslog(LOG_INFO, "%d , %d : ERROR : %s", localport, remoteport, text);
	return 1;
}

/* Forward request to a masqueraded host */
int forwardlookup(struct in_addr host, unsigned int localport, unsigned int remoteport, unsigned int origport) {
	struct sockaddr_in address;
	int sock;
	char *buf, *result;

	if(hexident) {
		/* Return hex result */
		printf("%d , %d : USERID : UNIX :x%x\r\n", origport, remoteport, host.s_addr);
		return 1;
	}
	
	syslog(LOG_INFO, "Forwarding to [%s]\n", inet_ntoa(host));
				
	memset((char *) &address, 0, sizeof(address));
	address.sin_family = AF_INET;
	address.sin_port = htons(113);
	address.sin_addr = host;
	sock = socket(AF_INET, SOCK_STREAM, 0);

	if(connect(sock, (struct sockaddr *) &address, sizeof(address)) < 0) {
		errorresponse(origport, remoteport, "UNKNOWN-ERROR");
		/*
		printf("%d , %d : ERROR : UNKNOWN-ERROR\r\n", origport, remoteport);
		*/
		syslog(LOG_INFO, "Connect: %s [%s]", strerror(errno), inet_ntoa(host));
	} else {
		buf = (char *) malloc(1024);
		sprintf(buf, "%d , %d\r\n", localport, remoteport);
		if(write(sock, buf, strlen(buf)) == -1) {
			errorresponse(origport, remoteport, "UNKNOWN-ERROR");
			/*
			printf("%d , %d : ERROR : UNKNOWN-ERROR\r\n", origport, remoteport);
			*/
			syslog(LOG_INFO, "Write: %s [%s]", strerror(errno), inet_ntoa(host));
		} else {
			memset(buf, 0, 1024);
			read(sock, buf, 1024);
			result = buf;
			
			strsep(&result, ":");
			printf("%d , %d :%s", origport, remoteport, result);
			syslog(LOG_INFO, "%d , %d :%s", origport, remoteport, strsep(&result, "\r"));
		}
		free(buf);
	}
	close(sock);
	
	return 1;
}

int checktcp(struct ident *this) {
	unsigned int clp, crp, uid;
	char *buf, *noidentfile;
	struct in_addr clip, crip;
	struct passwd *pwent;
	FILE *fh, *fi;
	
	fh = fopen(PROC_NET_TCP, "r");
	buf = (char *) malloc(160);
	fgets(buf, 160, fh); /* Discard first line */
	
	while(fgets(buf, 160, fh)) {
		if(sscanf(buf, "%*d: %x:%x %x:%x %*x %*x:%*x %*x:%*x %*x %d %*s", &clip.s_addr, &clp, &crip.s_addr, &crp, &uid) == 5) {
			if(clp == this->localport && crp == this->remoteport && clip.s_addr == this->localhost.s_addr) {
				if(crip.s_addr == this->remotehost.s_addr || this->remotehost.s_addr == masqhost.s_addr) {
					/* Get details from uid */
					pwent = getpwuid(uid);
					
					/* Check for users ~/.noident */
					if(!noident) {
						noidentfile = (char *) malloc(strlen(pwent->pw_dir) + 10);
						sprintf(noidentfile, "%s/.noident", pwent->pw_dir);
						fi = fopen(noidentfile, "r");
						if(fi) {
							fclose(fi);
							return errorresponse(this->localport, this->remoteport, "HIDDEN-USER");
							/*
							printf("%d , %d : ERROR : HIDDEN-USER\r\n", this->localport, this->remoteport);
							syslog(LOG_INFO, "%d , %d : ERROR : HIDDEN-USER [%s]", this->localport, this->remoteport, pwent->pw_name);
							return 1;
							*/
						}
					}
					printf("%d , %d : USERID : UNIX :%s\r\n", this->localport, this->remoteport, pwent->pw_name);
					syslog(LOG_INFO, "%d , %d : USERID : UNIX :%s", this->localport, this->remoteport, pwent->pw_name);
					return 1;
				} else {
					return errorresponse(this->localport, this->remoteport, "UNKNOWN-ERROR");
					/*
					printf("%d , %d : ERROR : UNKNOWN-ERROR\r\n", this->localport, this->remoteport);
					syslog(LOG_INFO, "%d , %d : ERROR : UNKNOWN-ERROR", this->localport, this->remoteport);
					*/
				}
			}
		}
	}
	free(buf);
	fclose(fh);

	return 0;
}

int checkconntrack(struct ident *this) {
	unsigned int proto, timeout, out_sport, out_dport, in_sport, in_dport, use;
	char *buf, *cin_src, *cin_dst, *cout_src, *cout_dst;
	struct in_addr in_src, in_dst, out_src;
	FILE *fh;

	fh = fopen(IP_CONNTRACK, "r");
	if(fh) {
		while(!feof(fh)) {
			buf = (char *) malloc(1024);
			fgets(buf, 1024, fh);
			if(!strncasecmp("tcp", buf, 3)) {
				/* we have a TCP connection */
				if(sscanf(buf, "tcp %u %u ESTABLISHED src=%as dst=%as sport=%u dport=%u src=%as dst=%as sport=%u dport=%u [ASSURED] use=%u", &proto, &timeout, &cout_src, &cout_dst, &out_sport, &out_dport, &cin_src, &cin_dst, &in_sport, &in_dport, &use) == 11) {
					if(in_sport == this->remoteport && in_dport == this->localport) {
						inet_aton(cin_src, &in_src);
						inet_aton(cin_dst, &in_dst);
						if(in_src.s_addr == this->remotehost.s_addr && in_dst.s_addr == this->localhost.s_addr) {
							inet_aton(cout_src, &out_src);
							free(buf);
							fclose(fh);
							return forwardlookup(out_src, out_sport, out_dport, in_dport);
						}
					}
				}
			}	
			free(buf);
		}
		fclose(fh);
	}

	return 0;
}

int checkipmasquerade(struct ident *this) {
	unsigned int masqport, crp, clp;
	char *buf;
	struct in_addr masqip, crip;
	FILE *fh;
	
	fh = fopen(IP_MASQUERADE, "r");
	if(fh) {
		buf = (char *)malloc(160);
		/* ignore first line in /proc/net/ip_masquerade */
		fgets(buf, 160, fh);

		while(fgets(buf, 160, fh)) {
			if(sscanf(buf, "%*s %x:%x %x:%x %x %*s", &masqip.s_addr, &masqport, &crip.s_addr, &crp, &clp) == 5) {
				masqip.s_addr = htonl(masqip.s_addr);
				crip.s_addr = htonl(crip.s_addr);
			
				if((clp == this->localport) && (crp == this->remoteport) && (crip.s_addr == this->remotehost.s_addr)) {
					free(buf);
					fclose(fh);
					return forwardlookup(masqip, masqport, this->remoteport, this->localport);
				}
			}
		}
		free(buf);
		fclose(fh);
	}

	return 0;
}

int main(int argc, char **argv) {
	struct sockaddr_in address;
	int j;
	struct ident *this;
	char *request;
	fd_set rfds;
	struct timeval tv;
	int c, timeout = 60;
	int alwaysfake = 0;

	this = malloc(sizeof(struct ident));
	this->localport = 0;
	this->remoteport = 0;

	openlog(NULL, LOG_PID, LOG_AUTHPRIV);

	masqhost.s_addr = 0;

	while((c = getopt(argc, argv, "m:gt:af:nhvx")) != -1)
		switch(c) {
		case 'm':
			inet_aton(optarg, &masqhost);
			break;
		case 'g':
			masqhost = getdefaultgateway();
			/*
			if(masqhost.s_addr == 0) {
				printf("pimpd: Unable to automatically obtain default gateway. Please specify with -m.\n");
				exit(1);
			}
			*/
			break;
		case 't':
			timeout = atoi(optarg);
			break;
		case 'f':
			fakeid = malloc(strlen(optarg));
			strncpy(fakeid, optarg, strlen(optarg));
			break;
		case 'a':
			alwaysfake = 1;
			break;
		case 'n':
			noident = 1;
			break;
		case 'v':
			printf("pimpd-%s\n", PIMPDVERSION);
			return 0;
		case 'x':
			hexident = 1;
			break;
		case 'h':
			printf("Usage: %s [-m <host>] [-t <timeout>] [-h]\n\n"
			"	-h		Show this help.\n"
			"	-m <host>	Set the masquerading host IP.\n"
			"			This is for the hosts behind the masquerading gateway\n"
			"			to allow masqueraded operation.\n"
			"	-f <name>	Always return a USERID of <name>.\n"
			"	-a		Always use fake USERID.\n"
			"	-g		Automatically guess the masquerading host IP.\n"
			"	-n		Disable ~/.noident usage.\n"
			"	-t <timeout>	Set the connection timeout in seconds (default 60).\n"
			"	-x		Return hex value of IP address instead of forwarding.\n"
			"	-v		Return the version of pimpd.\n", argv[0]);
			return 0;
		}

	/* Get local and remote IPs from socket */
	j = sizeof(address);
	if(getsockname(0, (struct sockaddr *) &address, &j)) {
		printf("Can't work here!\n");
		return 1;
	}
	this->localhost = address.sin_addr;
	j = sizeof(address);
	if(getpeername(0, (struct sockaddr *) &address, &j)) {
		syslog(LOG_INFO, "Oddness");
		return 1;
	}
	this->remotehost = address.sin_addr;
	
	syslog(LOG_INFO, "Connection from [%s]", inet_ntoa(this->remotehost));
	
	FD_ZERO(&rfds);
	FD_SET(0, &rfds);
	tv.tv_sec = timeout;
	tv.tv_usec = 0;
	if(!select(1, &rfds, NULL, NULL, &tv)) {
		syslog(LOG_INFO, "Connection from [%s] timed out", inet_ntoa(this->remotehost));
		return 0;
	}
	request = (char *)malloc(128);
	if(!fgets(request, 128, stdin)) {
		syslog(LOG_INFO, "Connection from [%s] dropped", inet_ntoa(this->remotehost));
		return 0;
	}
	sscanf(request, "%d , %d\r\n", &this->localport, &this->remoteport);
	free(request);

	if(fakeid && alwaysfake) {
		printf("%d , %d : USERID : UNIX :%s\r\n", this->localport, this->remoteport, fakeid);
		return 0;
	}
	
	if(((this->localport < 1) || (this->localport > 65535)) || ((this->remoteport < 1) || (this->remoteport > 65535))) {
		return errorresponse(this->localport, this->remoteport, "INVALID-PORT");
		/*
		printf("%d , %d : ERROR : INVALID-PORT\r\n", this->localport, this->remoteport);
		syslog(LOG_INFO, "%d , %d : ERROR : INVALID-PORT", this->localport, this->remoteport);
		return 0;
		*/
	}
	
	if(!checktcp(this))
		if(!checkconntrack(this))
			if(!checkipmasquerade(this)) {
				return errorresponse(this->localport, this->remoteport, "UNKNOWN-ERROR");
				/*
				printf("%d , %d : ERROR : UNKNOWN-ERROR\r\n", this->localport, this->remoteport);
				syslog(LOG_INFO, "%d , %d : ERROR : UNKNOWN-ERROR", this->localport, this->remoteport);
				*/
			}	
	return 0;
}
