package vulnerability

import "strings"

const (
	UnknownSeverity Severity = iota
	NegligibleSeverity
	LowSeverity
	MediumSeverity
	HighSeverity
	CriticalSeverity
)

var matcherTypeStr = []string{
	"unknown severity",
	"negligible",
	"low",
	"medium",
	"high",
	"critical",
}

func AllSeverities() []Severity {
	return []Severity{
		NegligibleSeverity,
		LowSeverity,
		MediumSeverity,
		HighSeverity,
		CriticalSeverity,
	}
}

type Severity int

type Severities []Severity

func (f Severity) String() string {
	if int(f) >= len(matcherTypeStr) || f < 0 {
		return matcherTypeStr[0]
	}

	return matcherTypeStr[f]
}

func ParseSeverity(severity string) Severity {
	switch strings.ToLower(severity) {
	case NegligibleSeverity.String():
		return NegligibleSeverity
	case LowSeverity.String():
		return LowSeverity
	case MediumSeverity.String():
		return MediumSeverity
	case HighSeverity.String():
		return HighSeverity
	case CriticalSeverity.String():
		return CriticalSeverity
	default:
		return UnknownSeverity
	}
}

func (s Severities) Len() int {
	return len(s)
}

func (s Severities) Swap(i, j int) {
	s[i], s[j] = s[j], s[i]
}

func (s Severities) Less(i, j int) bool {
	return s[i] < s[j]
}
