/*
 * name.c - Naming table manipulation.
 *
    Copyright (C) 2001  Yao Zhang

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * yaoz@users.sourceforge.net
 */
#include <iconv.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ot.h"

NameTable *new_name_table(void)
{
    NameTable *nameTable;

    nameTable = (NameTable *)calloc(1, sizeof(NameTable));

    return nameTable;
}

void delete_name_table(NameTable *nameTable)
{
    if (nameTable->nameArray != NULL) {
        int i;

        i = nameTable->nameTableHeader.count;
        while (i-- > 0) {
            free(nameTable->nameArray[i].name);
        }
        free(nameTable->nameArray);
    }

    if (nameTable->buffer != NULL) {
        free(nameTable->buffer);
    }
    free(nameTable);
}

static int compare(const void *x, const void *y)
{
    NameArray *a, *b;

    a = (NameArray *)x;
    b = (NameArray *)y;

    if (a->nameRecord.platformID != b->nameRecord.platformID) {
        return a->nameRecord.platformID - b->nameRecord.platformID;
    }

    if (a->nameRecord.platformSpecificID != b->nameRecord.platformSpecificID) {
        return a->nameRecord.platformSpecificID - b->nameRecord.platformSpecificID;
    }

    if (a->nameRecord.languageID != b->nameRecord.languageID) {
        return a->nameRecord.languageID - b->nameRecord.languageID;
    }

    if (a->nameRecord.nameID != b->nameRecord.nameID) {
        return a->nameRecord.nameID - b->nameRecord.nameID;
    }

    return 0;
}

void read_name_table(char *file, FontDirectory *fontDirectory,
                                 NameTable *nameTable)
{
    register uint8_t *p;
    int i, n;
    TableDirectoryEntry *tableDirectoryEntry;

    /* Find the 'name' directory */
    n = fontDirectory->offsetSubtable.numTables;
    for (i = 0; i < n; i++) {
        tableDirectoryEntry = &fontDirectory->tableDirectory[i];

        if (strncmp((const char *)tableDirectoryEntry->tag, NAME_TAG, 4) == 0) {
            break;
        }
    }

    if (i >= n) {
        return;
    }

    /* Read from file to memory */
    nameTable->length = tableDirectoryEntry->length;
    nameTable->buffer = (uint8_t *)calloc(nameTable->length, sizeof(uint8_t));
    read_table(nameTable->buffer, nameTable->length,
               file, tableDirectoryEntry->offset);

    /* Name Table Header */
    p = nameTable->buffer;
    nameTable->nameTableHeader.format = GET2B(p);
    nameTable->nameTableHeader.count = GET2B(p);
    nameTable->nameTableHeader.stringOffset = GET2B(p);

    n = nameTable->nameTableHeader.count;
    nameTable->nameArray = (NameArray *)calloc(n, sizeof(NameArray));

    for (i = 0; i < n; i++) {
        NameRecordEntry *nameRecordEntry;

        nameRecordEntry = &nameTable->nameArray[i].nameRecord;
        nameRecordEntry->platformID = GET2B(p);
        nameRecordEntry->platformSpecificID = GET2B(p);
        nameRecordEntry->languageID = GET2B(p);
        nameRecordEntry->nameID = GET2B(p);
        nameRecordEntry->length = GET2B(p);
        nameRecordEntry->offset = GET2B(p);
    }

    for (i = 0; i < n; i++) {
        NameRecordEntry *nameRecordEntry;

        nameRecordEntry = &nameTable->nameArray[i].nameRecord;
        nameTable->nameArray[i].name
                = (uint8_t *)calloc(nameRecordEntry->length, 1);
        p = nameTable->buffer
          + nameTable->nameTableHeader.stringOffset
          + nameRecordEntry->offset;
        memcpy(nameTable->nameArray[i].name, p, nameRecordEntry->length);
    }
}

void print_name_table_header(const NameTableHeader *nameTableHeader)
{
    printf("  Format: %u, ", nameTableHeader->format);
    printf("Number of NameRecords: %u, ", nameTableHeader->count);
    printf("Offset to name strings: %u.\n", nameTableHeader->stringOffset);
}

void print_name_record_entry(const NameRecordEntry *nameRecordEntry)
{
    printf("  (%s,%s,%s) %s at %d+%d:\n",
           platform_string(nameRecordEntry->platformID),
           platform_specific_string(nameRecordEntry->platformID, nameRecordEntry->platformSpecificID),
           language_string(nameRecordEntry->platformID, nameRecordEntry->languageID),
           name_string(nameRecordEntry->nameID),
           nameRecordEntry->offset,
           nameRecordEntry->length);
}

void print_name_string(NameArray *nameEntry, int in_hex)
{
    int i;
    char *from;
    char *utf8;
    int utf8bytes;

    if (in_hex) {
        for (i = 0; i < nameEntry->nameRecord.length; i++) {
            printf("0x%02X ", nameEntry->name[i]);
        }
        printf("\n");
        return;
    }

    if ((nameEntry->nameRecord.platformID == 1)
     && (nameEntry->nameRecord.platformSpecificID == 0)) {
        utf8bytes = nameEntry->nameRecord.length + 1;
        utf8 = (char *)calloc(utf8bytes, 1);
        memcpy(utf8, nameEntry->name, nameEntry->nameRecord.length);
        printf("%s", utf8);
        free(utf8);
    } else if ((nameEntry->nameRecord.platformID == 3)
     && (nameEntry->nameRecord.platformSpecificID == 1)) {
        utf8bytes = (nameEntry->nameRecord.length+1)*6;
        utf8 = (char *)calloc(utf8bytes, 1);
        from = "UTF-16BE";
        if (to_utf8_string(from, (char *)nameEntry->name, nameEntry->nameRecord.length,
                                 utf8, utf8bytes) == 0) {
            printf("%s", utf8);
        }
        free(utf8);
    } else if ((nameEntry->nameRecord.platformID == 3)
     && (nameEntry->nameRecord.platformSpecificID == 3)) {
        utf8bytes = (nameEntry->nameRecord.length+1)*6;
        utf8 = (char *)calloc(utf8bytes, 1);
        from = "GBK";
        if (to_utf8_string(from, (char *)nameEntry->name, nameEntry->nameRecord.length,
                                 utf8, utf8bytes) == 0) {
            printf("%s", utf8);
        }
        free(utf8);
    } else if ((nameEntry->nameRecord.platformID == 3)
     && (nameEntry->nameRecord.platformSpecificID == 4)) {
        utf8bytes = (nameEntry->nameRecord.length+1)*6;
        utf8 = (char *)calloc(utf8bytes, 1);
        from = "BIG5";
        if (to_utf8_string(from, (char *)nameEntry->name, nameEntry->nameRecord.length,
                                 utf8, utf8bytes) == 0) {
            printf("%s", utf8);
        }
        free(utf8);
    } else {
        for (i = 0; i < nameEntry->nameRecord.length; i++) {
            printf("0x%02X ", nameEntry->name[i]);
        }
    }
    printf("\n");
}

/* Fix name table's offsets and recreate its buffer. */
void recreate_name(NameTable *nameTable)
{
    register uint8_t *p;
    int i, n;
    uint16_t offset;

    nameTable->length = 0;
    if (nameTable->buffer != NULL) {
        free(nameTable->buffer);
    }

    n = nameTable->nameTableHeader.count;

    nameTable->nameTableHeader.stringOffset = 3*2 + n*6*2;

    offset = 0;
    for (i = 0; i < n; i++) {
        NameRecordEntry *nameRecordEntry;

        nameRecordEntry = &nameTable->nameArray[i].nameRecord;
        nameRecordEntry->offset = offset;
        offset += nameRecordEntry->length;
    }

    nameTable->length = nameTable->nameTableHeader.stringOffset + offset;
    nameTable->buffer = (uint8_t *)calloc(nameTable->length, 1);

    p = nameTable->buffer;
    p += 2;					/* format: 0 */
    SET2B(p, nameTable->nameTableHeader.count);
    SET2B(p, nameTable->nameTableHeader.stringOffset);

    for (i = 0; i < n; i++) {
        NameRecordEntry *nameRecordEntry;

        nameRecordEntry = &nameTable->nameArray[i].nameRecord;
        SET2B(p, nameRecordEntry->platformID);
        SET2B(p, nameRecordEntry->platformSpecificID);
        SET2B(p, nameRecordEntry->languageID);
        SET2B(p, nameRecordEntry->nameID);
        SET2B(p, nameRecordEntry->length);
        SET2B(p, nameRecordEntry->offset);
    }

    for (i = 0; i < n; i++) {
        NameRecordEntry *nameRecordEntry;

        nameRecordEntry = &nameTable->nameArray[i].nameRecord;
        p = nameTable->buffer
          + nameTable->nameTableHeader.stringOffset
          + nameRecordEntry->offset;
        memcpy(p, nameTable->nameArray[i].name, nameRecordEntry->length);
    }
}

/* Add UTF-8 names as (Macintosh, Roman, English) encoding. */
void add_utf8_names(NameTable *nameTable)
{
    int i, n;

    n = nameTable->nameTableHeader.count;

    for (i = 0; i < n; i++) {
        if ((nameTable->nameArray[i].nameRecord.platformID == 1)
         && (nameTable->nameArray[i].nameRecord.platformSpecificID == 0)
         && (nameTable->nameArray[i].nameRecord.languageID == 0)) {
            break;
        }
    }
    if (i >= n) {
        int start;
        char *from, *to;

        start = -1;
        if (start < 0) {
            for (i = 0; i < n; i++) {
                if ((nameTable->nameArray[i].nameRecord.platformID == 3)
                 && (nameTable->nameArray[i].nameRecord.platformSpecificID == 1)) {
                    start = i;
                    from = "UTF-16BE";
                    break;
                }
            }
        }
        if (start < 0) {
            for (i = 0; i < n; i++) {
                if ((nameTable->nameArray[i].nameRecord.platformID == 3)
                 && (nameTable->nameArray[i].nameRecord.platformSpecificID == 3)) {
                    start = i;
                    from = "GBK";
                    break;
                }
            }
        }
        if (start < 0) {
            for (i = 0; i < n; i++) {
                if ((nameTable->nameArray[i].nameRecord.platformID == 3)
                 && (nameTable->nameArray[i].nameRecord.platformSpecificID == 4)) {
                    start = i;
                    from = "BIG5";
                    break;
                }
            }
        }

        to = "UTF-8";

        if (start >= 0) {
            int count;

            count = 0;
            for (i = 0; i < n; i++) {
                if ((nameTable->nameArray[start].nameRecord.platformID
                  == nameTable->nameArray[i].nameRecord.platformID)
                 && (nameTable->nameArray[start].nameRecord.platformSpecificID
                  == nameTable->nameArray[i].nameRecord.platformSpecificID)) {
                    count++;
                }
            }

            nameTable->nameArray
                    = (NameArray *)realloc(nameTable->nameArray,
                                           (n+count)*sizeof(NameArray));
            for (i = 0; i < count; i++) {
                int length;
                char *name;

                length = nameTable->nameArray[start+i].nameRecord.length*6;
                name = (char *)calloc(length, 1);

                if (to_utf8_string(from,
                       (char *)nameTable->nameArray[start+i].name,
                       nameTable->nameArray[start+i].nameRecord.length,
                       name, length) == 0) {
                    length = strlen(name);
                    name = (char *)realloc(name, length);
                } else {
                    length = 1;
                    name = (char *)realloc(name, length);
                    name[0] = '\0';
                }

                /* (Macintosh, Roman, English) */
                nameTable->nameArray[n+i].nameRecord.platformID = 1;
                nameTable->nameArray[n+i].nameRecord.platformSpecificID = 0;
                nameTable->nameArray[n+i].nameRecord.languageID = 0;
                nameTable->nameArray[n+i].nameRecord.nameID
                    = nameTable->nameArray[start+i].nameRecord.nameID;
                nameTable->nameArray[n+i].nameRecord.length = length;
                nameTable->nameArray[n+i].nameRecord.offset = 0;
                nameTable->nameArray[n+i].name = (unsigned char *)name;
            }

            nameTable->nameTableHeader.count += count;
            qsort(nameTable->nameArray,
                  nameTable->nameTableHeader.count, sizeof(NameArray),
                  compare);
            recreate_name(nameTable);
        }
    }
}

static void change_name(NameArray *na, char *utf8, int utf8bytes)
{
    char to[64];

    /* convert from UTF-8 to proper one */
    strcpy(to, "UTF-8");
    if ((na->nameRecord.platformID == 3)
     && (na->nameRecord.platformSpecificID == 1)) {
        strcpy(to, "UTF-16BE");
    } else if ((na->nameRecord.platformID == 3)
     && (na->nameRecord.platformSpecificID == 3)) {
        strcpy(to, "GBK");
    } else if ((na->nameRecord.platformID == 3)
     && (na->nameRecord.platformSpecificID == 4)) {
        strcpy(to, "BIG5");
    }

    na->name = (uint8_t *)realloc(na->name, utf8bytes*2);
    na->nameRecord.length = from_utf8_string(to, utf8, utf8bytes,
                                             (char *)na->name, utf8bytes*2);
    na->name = (uint8_t *)realloc(na->name, na->nameRecord.length);
    na->nameRecord.offset = 0;
}

void replace_names(NameTable *nameTable, char *naming_spec)
{
    FILE *fp;
    int index;
    char buffer[BUFSIZ];
    char *utf8;
    int utf8bytes;

    fp = fopen(naming_spec, "r");
    if (fp == NULL) {
        return;
    }

    index = -1;
    utf8 = NULL;
    utf8bytes = 0;
    while (fgets(buffer, BUFSIZ, fp) != NULL) {
        if ((strncmp(buffer, "  (Unicode,", 11) == 0)
         || (strncmp(buffer, "  (Macintosh,", 13) == 0)
         || (strncmp(buffer, "  (ISO [deprecated],", 20) == 0)
         || (strncmp(buffer, "  (Microsoft,", 13) == 0)
         || (strncmp(buffer, "  (Custom,", 10) == 0)
         || (strncmp(buffer, "  ((unknown platform),", 22) == 0)) {
            if (utf8 != NULL) {
                if (utf8[utf8bytes-1] == '\n') {
                    utf8[utf8bytes-1] = '\0';
                    utf8bytes--;
                }
                if ((index >= 0)
                 && (index < nameTable->nameTableHeader.count)) {
                    change_name(&nameTable->nameArray[index], utf8, utf8bytes);
                }
                free(utf8);
            }

            index++;
            utf8 = NULL;
            utf8bytes = 0;
        } else {
            int n;

            n = strlen(buffer);
            utf8 = (char *)realloc(utf8, utf8bytes+n);
            memcpy(utf8+utf8bytes, buffer, n);
            utf8bytes += n;
        }
    }
    if (utf8 != NULL) {
        if (utf8[utf8bytes-1] == '\n') {
            utf8[utf8bytes-1] = '\0';
            utf8bytes--;
        }
        if ((index >= 0)
         && (index < nameTable->nameTableHeader.count)) {
            change_name(&nameTable->nameArray[index], utf8, utf8bytes);
        }
        free(utf8);
    }

    fclose(fp);

    recreate_name(nameTable);
}
