/*
    razertool - Tool for controlling Razer Copperhead(TM) mice
    Copyright (C) 2006  Christopher Lais

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
*/

#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <usb.h>

#include "razertool.h"
#include "razerlib.h"

#ifndef USB_REQ_GET_REPORT
#define USB_REQ_GET_REPORT	0x01
#endif

#ifndef USB_REQ_SET_REPORT
#define USB_REQ_SET_REPORT	0x09
#endif

#ifndef USB_REPORT_TYPE_INPUT
#define USB_REPORT_TYPE_INPUT	0x01
#endif

#ifndef USB_REPORT_TYPE_OUTPUT
#define USB_REPORT_TYPE_OUTPUT	0x02
#endif

#ifndef USB_REPORT_TYPE_FEATURE
#define USB_REPORT_TYPE_FEATURE	0x03
#endif

#define RAZER_PROFILE_READ_LEN	0x156
#define RAZER_PROFILE_WRITE_LEN	0x15c
#define RAZER_PROFILE_WRITE_BLOCK_LEN	0x40

#define ALIGN(x,a) (((x) + ((a)-1)) & ~((a)-1))

int razerlib_debug = 0;

#define IS_DEBUG()	(razerlib_debug!=0)
#define DEBUG(fmt,...)	do{if(IS_DEBUG()) fprintf(stderr, fmt , ## __VA_ARGS__);}while(0)

const char *razer_strerror(int number)
{
	switch (number) {
	  case RAZERLIB_ERR_INTERNAL: return "Internal error (razerlib)\n";
	  case RAZERLIB_ERR_STATUS: return "Bad status (razerlib)\n";
	  case RAZERLIB_ERR_INTEGRITY: return "Data integrity error (razerlib)\n";
	  case RAZERLIB_ERR_PARSE: return "Parse error (razerlib)\n";
	}

	if (number >= USB_ERROR_BEGIN)
		return "Unknown error";

	return strerror(number);
}

int razer_2hex(const char two[2])
{
	int ret;

	if (two[0] >= '0' && two[0] <= '9')
		ret = (two[0] - '0') << 4;
	else if (two[0] >= 'a' && two[0] <= 'f')
		ret = (two[0] - 'a' + 10) << 4;
	else if (two[0] >= 'A' && two[0] <= 'F')
		ret = (two[0] - 'A' + 10) << 4;
	else return -1;

	if (two[1] >= '0' && two[1] <= '9')
		ret |= (two[1] - '0');
	else if (two[1] >= 'a' && two[1] <= 'f')
		ret |= (two[1] - 'a' + 10);
	else if (two[1] >= 'A' && two[1] <= 'F')
		ret |= (two[1] - 'A' + 10);
	else return -1;

	return ret;
}

static const int ch_to_usb[0x80] = {
/*        0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f */
/*0*/	  0,   0,   0,   0,   0,   0,   0,   0,  42,  43,  40,   0,   0,   0,   0,   0,
/*1*/	  0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,  41,   0,   0,   0,   0,
/*2*/	 44,  30,  52,  32,  33,  34,  36,  52,  38,  39,  37,  46,  54,  45,  55,  56,
/*3*/	 39,  30,  31,  32,  33,  34,  35,  36,  37,  38,  51,  51,  54,  46,  55,  56,
/*4*/	  0,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17,  18,
/*5*/	 19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  47,  49,  48,  35,  45,
/*6*/	  0,   4,   5,   6,   7,   8,   9,  10,  11,  12,  13,  14,  15,  16,  17,  18,
/*7*/	 19,  20,  21,  22,  23,  24,  25,  26,  27,  28,  29,  47,  49,  48,  53,  42,
};

static const struct { char *name; int value; } str_to_usb[] = {
	{ "1/!/Exclam", 30 },
	{ "2/@/At", 31 },
	{ "3/#/NumberSign", 32 },
	{ "4/$/Dollar", 33 },
	{ "5/%/Percent", 34 },
	{ "6/^/Circumflex", 35 },
	{ "7/&/Ampersand", 36 },
	{ "8/*/Asterisk", 37 },
	{ "9/(/LeftParen", 38 },
	{ "0/)/RightParen", 39 },
	{ "Enter", 40 },
	{ "Escape", 41 },
	{ "Backspace", 42 },
	{ "Tab", 43 },
	{ "Space", 44 },
	{ "-/_/Minus/Underscore", 45 },
	{ "=/+/Equal/Plus", 46 },
	{ "[/{/LeftBracket/LeftBrace", 47 },
	{ "]/}/RightBracket/RightBrace", 48 },
	{ "Backslash/|/Bar", 49 },
	{ "Non-US-#/Non-US-~/Non-US-NumberSign/Non-US-Tilde", 50 },
	{ ";/:/Semicolon/Colon", 51 },
	{ "'/\"/Quote/DoubleQuote", 52 },
	{ "`/~/Grave/Tidle", 53 },
	{ ",/</Comma/Less", 54 },
	{ "./>/Period/Greater", 55 },
	{ "Slash/?/Question", 56 },
	{ "CapsLock", 57 },
	{ "F1", 58 },
	{ "F2", 59 },
	{ "F3", 60 },
	{ "F4", 61 },
	{ "F5", 62 },
	{ "F6", 63 },
	{ "F7", 64 },
	{ "F8", 65 },
	{ "F9", 66 },
	{ "F10", 67 },
	{ "F11", 68 },
	{ "F12", 69 },
	{ "PrintScreen", 70 },
	{ "ScrollLock", 71 },
	{ "Pause", 62 },
	{ "Insert", 73 },
	{ "Home", 74 },
	{ "PageUp", 75 },
	{ "Delete", 76 },
	{ "End", 77 },
	{ "PageDown", 78 },
	{ "Right", 79 },
	{ "Left", 80 },
	{ "Down", 81 },
	{ "Up", 82 },
	{ "NumLock", 83 },
	{ "Clear", 83 },
	{ "KPDivide", 84 },
	{ "KP*/KPMultiply", 85 },
	{ "KP-/KPSubtract", 86 },
	{ "KP+/KPAdd", 87 },
	{ "KPEnter", 88 },
	{ "KP1", 89 },
	{ "KP2", 90 },
	{ "KP3", 91 },
	{ "KP4", 92 },
	{ "KP5", 93 },
	{ "KP6", 94 },
	{ "KP7", 95 },
	{ "KP8", 96 },
	{ "KP9", 97 },
	{ "KP0", 98 },
	{ "KP./KPDecimal", 99 },
	{ "Non-US-Backslash/Non-US-|/Non-US-Bar", 100 },
	{ "Application", 101 },
	{ "Power", 102 },
	{ "KP=/KPEqual", 103 },
	{ "F13", 104 },
	{ "F14", 105 },
	{ "F15", 106 },
	{ "F16", 107 },
	{ "F17", 108 },
	{ "F18", 109 },
	{ "F19", 110 },
	{ "F20", 111 },
	{ "F21", 112 },
	{ "F22", 113 },
	{ "F23", 114 },
	{ "F24", 115 },
	{ "Execute", 116 },
	{ "Help", 117 },
	{ "Menu", 118 },
	{ "Select", 119 },
	{ "Stop", 120 },
	{ "Again", 121 },
	{ "Undo", 122 },
	{ "Cut", 123 },
	{ "Copy", 124 },
	{ "Paste", 125 },
	{ "Find", 126 },
	{ "Mute", 127 },
	{ "VolumeUp", 128 },
	{ "VolumeDown", 129 },
	{ "LCapsLock", 130 },
	{ "LNumLock", 131 },
	{ "LScrollLock", 132 },
	{ "KP,/KPComma", 133 },
	{ "KPEqualSign", 134 },
	{ "Intl1", 135 },
	{ "Intl2", 136 },
	{ "Intl3", 137 },
	{ "Intl4", 138 },
	{ "Intl5", 139 },
	{ "Intl6", 140 },
	{ "Intl7", 141 },
	{ "Intl8", 142 },
	{ "Intl9", 143 },
	{ "Lang1", 144 },
	{ "Lang2", 145 },
	{ "Lang3", 146 },
	{ "Lang4", 147 },
	{ "Lang5", 148 },
	{ "Lang6", 149 },
	{ "Lang7", 150 },
	{ "Lang8", 151 },
	{ "Lang9", 152 },
	{ "AltErase", 153 },
	{ "SysReq", 154 },
	{ "Cancel", 155 },
	{ "Clear", 156 },
	{ "Prior", 157 },
	{ "Return", 158 },
	{ "Separator", 159 },
	{ "Out", 160 },
	{ "Oper", 161 },
	{ "Clear-Again/Again", 162 },
	{ "CrSel/Props", 163 },
	{ "ExSel", 164 },
	{ "KP00", 176 },
	{ "KP000", 177 },
	{ "ThousandsSep", 178 },
	{ "DecimalSep", 179 },
	{ "Currency", 180 },
	{ "CurrencySub", 181 },
	{ "KP(/KPLeftParen", 182 },
	{ "KP)/KPRightParen", 183 },
	{ "KP{/KPLeftBrace", 184 },
	{ "KP}/KPRightBrace", 185 },
	{ "KPTab", 186 },
	{ "KPBackspace", 187 },
	{ "KPA", 188 },
	{ "KPB", 189 },
	{ "KPC", 190 },
	{ "KPD", 191 },
	{ "KPE", 192 },
	{ "KPF", 193 },
	{ "KPXOR", 194 },
	{ "KP^/KPCircumflex", 195 },
	{ "KP%/KPPercent", 196 },
	{ "KP</KPLess", 197 },
	{ "KP>/KPGreater", 198 },
	{ "KP&/KPAnd", 199 },
	{ "KP&&/KPAndAnd", 200 },
	{ "KP|/KPOr", 201 },
	{ "KP||/KPOrOr", 202 },
	{ "KP:/KPColon", 203 },
	{ "KP#/KPNumberSign", 204 },
	{ "KPSpace", 205 },
	{ "KP@/KPAt", 206 },
	{ "KP!/KPExclam", 207 },
	{ "KPMemoryStore", 208 },
	{ "KPMemoryRecall", 209 },
	{ "KPMemoryClear", 210 },
	{ "KPMemoryAdd", 211 },
	{ "KPMemorySubtract", 212 },
	{ "KPMemoryMultiply", 213 },
	{ "KPMemoryDivide", 214 },
	{ "KP+-/KPPlusMinus", 215 },
	{ "KPClear", 216 },
	{ "KPClearEntry", 217 },
	{ "KPBinary", 218 },
	{ "KPOctal", 219 },
	{ "KPDecimal", 220 },
	{ "KPHexadecimal", 221 },
	{ "LCtrl", 0xe0 },
	{ "LShift", 0xe1 },
	{ "LAlt", 0xe2 },
	{ "LWin", 0xe3 },
	{ "RAlt", 0xe4 },
	{ "RShift", 0xe5 },
	{ "RCtrl", 0xe6 },
	{ "RWin", 0xe7 },
};

/* Translates keys to the HID equivilant */
int razer_translate_key(const char *key)
{
	int len = strlen(key);
	int i;

	if (len == 1) {
		if ((char)*key >= 0 && (unsigned char)*key <= 0x7f)
			return ch_to_usb[(int)*key];
		return 0;
	}

	if (len == 2) do {
		i = razer_2hex(key);
		if (i < 0) break;
		return i;
	} while(0);

	for (i = 0; i < (sizeof(str_to_usb)/sizeof(*str_to_usb)); i++) {
		const char *start, *end;
		if (!strcasecmp(str_to_usb[i].name, key))
			return str_to_usb[i].value;

		start = str_to_usb[i].name;
		while ((end = strchr(start, '/'))) {
			if (!strncasecmp(start, key, end - start))
				return str_to_usb[i].value;
			start = end + 1;
		}
		if (*start && !strcasecmp(start, key))
			return str_to_usb[i].value;
	}

	return 0;
}

/* FIXME: Reverse lookup tables */
void razer_untranslate_key(int usbkey, char *buf, int buflen)
{
	int i;

	if (buflen < 3)
		return;

	for (i = 0; i < (sizeof(str_to_usb)/sizeof(*str_to_usb)); i++) {
		if (str_to_usb[i].value == usbkey) {
			strncpy(buf, str_to_usb[i].name, buflen);
			buf[buflen-1] = '\0';
			return;
		}
	}

	for (i = 0x30; i < 0x7f; i++) {
		if (ch_to_usb[i] && ch_to_usb[i] == usbkey) {
			buf[0] = i;
			buf[1] = '\0';
			return;
		}
	}

	for (i = ' '; i < 0x30; i++) {
		if (ch_to_usb[i] && ch_to_usb[i] == usbkey) {
			buf[0] = i;
			buf[1] = '\0';
			return;
		}
	}

	sprintf(buf, "%02x", usbkey & 0xff);
}

static const char *razer_action[] = {
/*0*/	"INVALID",
	"BUTTON1",
	"BUTTON2",
	"BUTTON3",
	"DOUBLE_CLICK",
/*5*/	"KEY",
	"MACRO",
	"OFF",
	"UNKNOWN(08)",
	"ON_THE_FLY",
	"BUTTON4",
	"BUTTON5",
	"DPI_UP",
	"DPI_DOWN",
	"DPI_400",
	"DPI_800",
	"DPI_1600",
/*11*/	"DPI_2000",
/*12*/	"PROFILE1",
	"PROFILE2",
	"PROFILE3",
	"PROFILE4",
/*16*/	"PROFILE5",
};

const char *razer_action_string(int action)
{
	if (action < 1 || action >= RAZER_ACTION_COUNT)
		return NULL;
	return razer_action[action];
}

int razer_action_int(const char *str)
{
	int i;
	for (i = 1; i < RAZER_ACTION_COUNT; i++) {
		if (!strcasecmp(str, razer_action[i]))
			return i;
	}
	return -1;
}

static const int razer_dpi[RAZER_DPI_COUNT] = { 2000, 1600, 800, 400 };
int razer_dpi_int(int dpi)
{
	if (dpi < 1 || dpi >= RAZER_DPI_COUNT)
		return -1;
	return razer_dpi[dpi - 1];
}

static const int razer_hz[RAZER_HZ_COUNT] = { 1000, 500, 125 };
int razer_hz_int(int hz)
{
	if (hz < 1 || hz >= RAZER_HZ_COUNT)
		return -1;
	return razer_hz[hz - 1];
}

/* Takes the buffer, and the *full* length of the buffer, including the parity bytes */
static int razer_parity_valid(const unsigned char *buf, int len)
{
	int i;
	unsigned char parity2, parity1;

	if (len & 1) abort();

	parity2 = parity1 = 0;
	for (i = 0; i < (len-2); i += 2) {
		parity2 ^= buf[i];
		parity1 ^= buf[i+1];
	}

	if (parity2 == buf[i] && parity1 == buf[i+1])
		return 1;

	return 0;
}

static void razer_parity_calc(unsigned char *buf, int len)
{
	int i;
	unsigned char parity2, parity1;

	if (len & 1) abort();

	parity2 = parity1 = 0;
	for (i = 0; i < (len-2); i += 2) {
		parity2 ^= buf[i];
		parity1 ^= buf[i+1];
	}

	buf[i] = parity2;
	buf[i+1] = parity1;
}

int razer_usb_ping(struct usb_dev_handle *udev, int timeout)
{
	int status;
	char buffer[2];

	DEBUG("%s:", __FUNCTION__);

	status = usb_control_msg(
		udev,
		USB_ENDPOINT_IN | USB_TYPE_STANDARD | USB_RECIP_DEVICE,
		USB_REQ_GET_STATUS,
		0,
		0,
		buffer, 2,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	return 0;
}

static int razer_GENERIC_ping(razer_t *r, int timeout)
{
	return razer_usb_ping(r->udev, timeout);
}

/*************************************************************
 * Copperhead
 *************************************************************/
static int razer_CH_read(razer_t *r, int intrf, void *buf, int len, int timeout)
{
	int status;

	DEBUG("%s:", __FUNCTION__);

	/* The device doesn't check the recipient, so use DEVICE instead of
	   INTERFACE so we don't have to claim the interface. */
	status = usb_control_msg(
		r->udev,
		USB_ENDPOINT_IN | USB_TYPE_CLASS | USB_RECIP_DEVICE,
		USB_REQ_GET_REPORT,
		1,	/* 0 ReportType, 1 Report ID... */
		intrf,
		buf, len,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	return 0;
}

static int razer_CH_read_profile_number(razer_t *r, int *number, int timeout)
{
	int status;
	unsigned char prof_ch;

	DEBUG("%s:", __FUNCTION__);

	status = razer_CH_read(r, 0, &prof_ch, 1, timeout);

	DEBUG("%s:profile:%d\n", __FUNCTION__, prof_ch);

	if (status < 0)
		return status;

	*number = prof_ch;

	return 0;
}

/* 69 02  (6.17) */
static int razer_CH_read_firmware_version(razer_t *r, int *version, int timeout)
{
	int status;
	unsigned char ver[2];

	DEBUG("%s:", __FUNCTION__);

	status = razer_CH_read(r, 1, ver, 2, timeout);

	DEBUG("%s:version:0x%02x 0x%02x\n", __FUNCTION__, ver[0], ver[1]);

	if (status < 0)
		return status;

	*version = (ver[1] << 8) | (ver[0] << 0);

	return 0;
}

/* Driver does not use. */
#if 0
/* intrf 1: 04 01 00 03 */
static int razer_CH_read_unknown_4(razer_t *r, int intrf, unsigned char unk[4], int timeout)
{
	int status;

	DEBUG("%s:", __FUNCTION__);

	status = razer_CH_read(r, intrf, unk, 4, timeout);

	DEBUG("%s:data:0x%02x 0x%02x 0x%02x 0x%02x\n", __FUNCTION__, unk[0], unk[1], unk[2], unk[3]);

	if (status < 0)
		return status;

	return 0;
}
#endif

static int razer_CH_read_profile(razer_t *r, razer_profile_t *profile, int timeout)
{
	int status;
	unsigned char buf[0x156];
	int i;

	DEBUG("%s:", __FUNCTION__);

	status = razer_CH_read(r, 0, buf, 0x156, timeout);

	if (IS_DEBUG()) {
		DEBUG("%s:data:", __FUNCTION__);
		for (i = 0; i < 0x156; i++)
			DEBUG(" %02x", buf[i]);
		DEBUG("; %d\n", status);
	}

	if (status < 0)
		return status;

	if (!razer_parity_valid(buf, 0x156)) {
		DEBUG("%s:Parity invalid %02x %02x", __FUNCTION__, buf[0x156-2], buf[0x156-1]);
		return -RAZERLIB_ERR_INTEGRITY;
	}

	profile->number = buf[4];
	profile->dpi = buf[6];
	profile->hz = buf[7];

	for (i = 0; i < r->buttons; i++) {
		int offset = 8 + 0x30*i;
		profile->button[i].number = buf[offset];
		profile->button[i].action = buf[offset + 1];
		profile->button[i].macro_len = buf[offset + 3];
		profile->button[i].macro_max_len = RAZER_CH_MACRO_MAX_LEN;
		memcpy(profile->button[i].macro, (razer_macro_char_t*)&buf[offset + 4], 2*profile->button[i].macro_max_len);

		/* We just assume they're in the proper order for now */
		if (profile->button[i].number != (i+1)) {
			DEBUG("%s:Wrong button for %d: %d\n", __FUNCTION__, i+1, profile->button[i].number);
			return -RAZERLIB_ERR_INTEGRITY;
		}
	}

	profile->flags = RAZER_PROFILE_HAS_NUMBER | \
		RAZER_PROFILE_HAS_DPI | \
		RAZER_PROFILE_HAS_HZ | \
		RAZER_PROFILE_HAS_BUTTON;

	return 0;
}

#if 0
/* Note that unfortunately Linux (or something) resets the device in the
   short period that resetting it actually resets the entire device, thus
   bypassing firmware mode by booting normally.  So, this doesn't actually
   work with razer's firmware. */
static int razer_CH_firmware_mode(razer_t *r)
{
	unsigned char buffer[8];
	razer_CH_read(r, 0, buffer, 8, 1);
	usb_reset(r->udev);
	free(r);
	return 0;
}
#endif

static int razer_CH_write(razer_t *r, int intrf, int id, void *buf, int len, int timeout)
{
	int status;

	DEBUG("%s:", __FUNCTION__);

	/* The device doesn't check the recipient, so use DEVICE instead of
	   INTERFACE so we don't have to claim the interface. */
	status = usb_control_msg(
		r->udev,
		USB_ENDPOINT_OUT | USB_TYPE_CLASS | USB_RECIP_DEVICE,
		USB_REQ_SET_REPORT,
		id,	/* 0 ReportType... */
		intrf,
		buf, len,
		timeout
	);

	DEBUG("%d\n", status);

	if (status < 0)
		return status;

	return 0;
}

/* Writes the profile buffer and/or chooses the profile for reading */
static int razer_CH_set_profile(razer_t *r, int profile, int timeout)
{
	unsigned char prof[1] = { profile };
	if (profile < 1 || profile > r->profiles)
		return -EINVAL;

	DEBUG("%s:", __FUNCTION__);

	/* The device does not advertise interface 3 */
	/* Never-the-less, it is required to be 3. */
	return razer_CH_write(r, 3, 2, prof, 1, timeout);
}

/* Selects the currently active profile */
static int razer_CH_select_profile(razer_t *r, int profile, int timeout)
{
	unsigned char prof[1] = { profile };
	if (profile < 1 || profile > r->profiles)
		return -EINVAL;

	DEBUG("%s:", __FUNCTION__);

	return razer_CH_write(r, 1, 2, prof, 1, timeout);
}

static int razer_CH_write_profile(razer_t *r, razer_profile_t *prof, int timeout)
{
	unsigned char buf[ALIGN(RAZER_PROFILE_WRITE_LEN, RAZER_PROFILE_WRITE_BLOCK_LEN)];
	int i;
	int status;

	memset(buf, 0, sizeof(buf));

	/* Length */
	buf[0] = RAZER_PROFILE_WRITE_LEN & 0xff;
	buf[1] = RAZER_PROFILE_WRITE_LEN >> 8;

	/* Unknown. */
	buf[2] = 0x02;
	buf[3] = 0x00;
	buf[4] = prof->number;
	buf[5] = 0x00;
	buf[6] = 0x00;
	buf[7] = 0x00;
	buf[8] = 0x00;
	buf[9] = 0x00;
	buf[10] = prof->number;
	buf[11] = 0x00;

	/* rate/dpi */
	buf[12] = prof->dpi;
	buf[13] = prof->hz;

	/* buttons */
	for (i = 0; i < r->buttons; i++) {
		int offset = 14 + 0x30*i;
		buf[offset] = prof->button[i].number;
		buf[offset+1] = prof->button[i].action;
		buf[offset+3] = prof->button[i].macro_len;
		memcpy(&buf[offset+4], (unsigned char*)prof->button[i].macro, 2*prof->button[i].macro_max_len);
	}


	/* parity */
	razer_parity_calc(buf, RAZER_PROFILE_WRITE_LEN);

	DEBUG("%s:upload:", __FUNCTION__);

	for (i = 0; i < RAZER_PROFILE_WRITE_LEN; i += RAZER_PROFILE_WRITE_BLOCK_LEN) {
		status = razer_CH_write(r, 0, 1 + i / RAZER_PROFILE_WRITE_BLOCK_LEN, &buf[i], RAZER_PROFILE_WRITE_BLOCK_LEN, timeout);
		if (status < 0)
			return status;
	}

	DEBUG("%s:write/set:", __FUNCTION__);

	status = razer_CH_set_profile(r, prof->number, timeout);

	return status;
}

static razer_t *razer_CH_setup(razer_t *r)
{
	r->name = "Copperhead";

	r->profiles = 5;
	r->buttons = 7;

	r->ping = razer_GENERIC_ping;

	r->read_profile_number = razer_CH_read_profile_number;
	r->read_firmware_version = razer_CH_read_firmware_version;

	r->read_profile = razer_CH_read_profile;
	r->set_profile = razer_CH_set_profile;
	r->select_profile = razer_CH_select_profile;
	r->write_profile = razer_CH_write_profile;

#if 0
	r->firmware_mode = razer_CH_firmware_mode;
#else
	r->firmware_mode = NULL;
#endif

	return r;
}

/*************************************************************
 * Habu
 *************************************************************/
static int razer_HABU_read_firmware_version(razer_t *r, int *version, int timeout)
{
	int status;
	unsigned char ver[2];

	DEBUG("%s:", __FUNCTION__);

	status = razer_CH_read(r, 4, ver, 2, timeout);

	DEBUG("%s:version:0x%02x 0x%02x\n", __FUNCTION__, ver[0], ver[1]);

	if (status < 0)
		return status;

	*version = (ver[1] << 8) | (ver[0] << 0);

	return 0;
}

static int razer_HABU_firmware_mode(razer_t *r)
{
	/* device stops responding - even to ack the write, so use a small timeout. */
	razer_CH_write(r, 0, 2, NULL, 0, 1);
	usb_reset(r->udev);

	/* usb_reset invalidates udev, so don't close it. */
	free(r);
	return 0;
}

static razer_t *razer_HABU_setup(razer_t *r)
{
	r->name = "Habu";

	r->profiles = 5;
	r->buttons = 7;

	r->ping = razer_GENERIC_ping;

	r->read_profile_number = razer_CH_read_profile_number;
	r->read_firmware_version = razer_HABU_read_firmware_version;

	r->read_profile = razer_CH_read_profile;
	r->set_profile = razer_CH_set_profile;
	r->select_profile = razer_CH_select_profile;
	r->write_profile = razer_CH_write_profile;

	r->firmware_mode = razer_HABU_firmware_mode;

	return r;
}

/*************************************************************
 * Generic/other
 *************************************************************/
static razer_t *razer_UNKNOWN_setup(razer_t *r)
{
	r->name = "UNKNOWN DEVICE";

	r->ping = razer_GENERIC_ping;

	return r;
}

/**************************************************************/
static struct razer_ops {
	unsigned int idVendor, idVendorMask;
	unsigned int idProduct, idProductMask;
	razer_t *(*setup)(razer_t *r);
} razer_ops[] = {
	{ USB_VENDOR_RAZER, 0xffff, USB_PRODUCT_RAZER_COPPERHEAD, 0xffff, razer_CH_setup },
	{ USB_VENDOR_RAZER, 0xffff, USB_PRODUCT_RAZER_HABU, 0xffff, razer_HABU_setup },
	{ 0, 0, 0, 0, razer_UNKNOWN_setup },
};

struct usb_device *razer_find(struct usb_device *prev)
{
	struct usb_bus *busses;
	struct usb_bus *bus;
	int i;
	const char *debug;

	/* FIXME: wrong place for this. */
	debug = getenv("RAZERLIB_DEBUG");
	if (debug) { razerlib_debug = atoi(debug); }

	if (!prev) {
		usb_find_busses();
		usb_find_devices();
	}

	busses = usb_get_busses();

	for (bus = busses; bus; bus = bus->next) {
		struct usb_device *dev;

		for (dev = bus->devices; dev; dev = dev->next) {
			DEBUG("DEVICE: %04x:%04x\n", dev->descriptor.idVendor, dev->descriptor.idProduct);

			if (prev) {
				if (prev == dev)
					prev = NULL;
				continue;
			}

			for (i = 0; razer_ops[i].idVendorMask || razer_ops[i].idProductMask; i++) {
				if (razer_ops[i].idVendor != (dev->descriptor.idVendor & razer_ops[i].idVendorMask))
					continue;
				if (razer_ops[i].idProduct != (dev->descriptor.idProduct & razer_ops[i].idProductMask))
					continue;

				return dev;
			}
		}
	}

	return NULL;
}

razer_t *razer_open(struct usb_device *dev)
{
	struct usb_dev_handle *udev;
	razer_t *r;
	int i;
	struct razer_ops *op = NULL;

	for (i = 0; razer_ops[i].setup; i++) {
		if (razer_ops[i].idVendor != (dev->descriptor.idVendor & razer_ops[i].idVendorMask))
			continue;
		if (razer_ops[i].idProduct != (dev->descriptor.idProduct & razer_ops[i].idProductMask))
			continue;
		op = &razer_ops[i];
		break;
	}

	if (!op)
		return NULL;

	udev = usb_open(dev);
	if (!udev)
		return NULL;

	r = malloc(sizeof(*r));
	memset(r, 0, sizeof(*r));

	r->udev = udev;

	return op->setup(r);
}

void razer_close(razer_t *r)
{
	usb_close(r->udev);
	free(r);
}
