/*
 * render.c: Rendering-related functions for rotoscope.
 *
 * 
 * Copyright 2006, James Foster
 *
 * This file is part of rotoscope.
 *
 * Rotoscope is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Rotoscope is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with rotoscope; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <gdk-pixbuf/gdk-pixbuf.h>
#include <glib.h>
#include <math.h>
#include <stdlib.h>

#include "graph.h"
#include "render.h"

RotoPixel** create_rotomap( unsigned int width, unsigned int height )
{
	unsigned int i;
	unsigned int j;
	RotoPixel** rotomap;
	
	/* If 0 width or height is requested, return NULL */
	if( 0 == width || 0 == height )
	{
		return NULL;
	}
	
	/* Allocate 'width' number of RotoPixel pointers */
	rotomap = malloc( sizeof(RotoPixel*) * width );
	if( NULL == rotomap )
	{
		return NULL;
	}

	/* For each RotoPixel pointer, allocate 'height' number of RotoPixels */
	for( i = 0; i < width; i++ )
	{
		rotomap[i] = malloc( sizeof(RotoPixel) * height );
		
		/* If the RotoPixels couldn't be allocated, free everything and return NULL */
		if( NULL == rotomap[i] )
		{
			for( i--; i >= 0; i-- )
			{
				free( rotomap[i] );
			}
			free( rotomap );
			return NULL;
		}
		
		/* Set each RotoPixel in the column to the UNUSED state */
		for( j = 0; j < height; j++ )
		{
			rotomap[i][j] = UNUSED;
		}
	}

	return rotomap;
}

void free_rotomap( RotoPixel** rotomap, unsigned int width )
{
	unsigned int i;
	
	for( i = 0; i < width; i++ )
	{
		free( rotomap[i] );
	}
	free( rotomap );
}

void free_g_list( GList* list )
{
	void* node;
	while( NULL != list )
	{
		node = list->data;
		list = g_list_remove( list, node );
		free( node );
	}
}

GList* get_region_pixels( RotoPixel** rotomap, unsigned int width, unsigned int height, unsigned int x, unsigned int y )
{
	GList* visit_list = NULL;
	GList* region_pixels = NULL;
	Vertex* v;
	Vertex* w;
	
	/* Put the initial vertex onto the visit_list */
	v = create_vertex( x, y );
	if( NULL == v )
	{
		/* Out of memory */
		return NULL;
	}
	visit_list = g_list_prepend( visit_list, v );
	
	/* Until no more pixels need to be visited, keep visiting pixels and if they are in the UNUSED state,
	 * add their neighbours to the visit_list */
	while( NULL != visit_list )
	{
		/* Pick a pixel off the visit list */
		v = visit_list->data;
		visit_list = g_list_remove( visit_list, v );
		
		/* If the pixel is UNUSED and within boundaries, add it to the region and add its neighbours to the
		 * visit_list.
		 * Otherwise, free it. */
		if( v->x < width && v->y < height && UNUSED == rotomap[v->x][v->y] )
		{
			/* Mark the pixel as visited and add it to the region_pixels list */
			rotomap[v->x][v->y] = REGION;
			region_pixels = g_list_prepend( region_pixels, v );

			/* Add surrounding pixels to the visit list */
			/* (x+1, y) */
			w = create_vertex( v->x + 1, v->y );
			if( NULL == w )
			{
				/* Out of memory: Free the allocated memory, then return NULL */
				free_g_list( visit_list );
				free_g_list( region_pixels );
				return NULL;
			}
			visit_list = g_list_prepend( visit_list, w );

			/* (x-1, y) */
			w = create_vertex( v->x - 1, v->y );
			if( NULL == w )
			{
				/* Out of memory: Free the allocated memory, then return NULL */
				free_g_list( visit_list );
				free_g_list( region_pixels );
				return NULL;
			}
			visit_list = g_list_prepend( visit_list, w );

			/* (x, y+1) */
			w = create_vertex( v->x, v->y + 1 );
			if( NULL == w )
			{
				/* Out of memory: Free the allocated memory, then return NULL */
				free_g_list( visit_list );
				free_g_list( region_pixels );
				return NULL;
			}
			visit_list = g_list_prepend( visit_list, w );

			/* (x, y-1) */
			w = create_vertex( v->x, v->y - 1 );
			if( NULL == w )
			{
				/* Out of memory: Free the allocated memory, then return NULL */
				free_g_list( visit_list );
				free_g_list( region_pixels );
				return NULL;
			}
			visit_list = g_list_prepend( visit_list, w );
		}
		else
		{
			free( v );
		}
	}

	return region_pixels;
}

void free_regions( GList* regions )
{
	GList* pixels;
	while( NULL != regions )
	{
		pixels = regions->data;
		regions = g_list_remove( regions, pixels );
		free_g_list( pixels );
	}
}

void swap( unsigned int* x, unsigned int* y )
{
	unsigned int temp = *x;
	*x = *y;
	*y = temp;
}

float calc_slope( Vertex a, Vertex b )
{
	return ( (float) b.y - (float) a.y) / ( (float) b.x - (float) a.x );
}

/*
** WARNING: This uses a poor implementation of Bresenham's algorithm (uses floats)
** but I'm not concerned about speed yet, and this entire method of rendering will
** be replaced by something better for the next release of rotoscope.
*/
void plot_edge( RotoPixel** rotomap, Edge* e )
{
	Vertex a;
	Vertex b;
	unsigned int x;
	unsigned int y;
	float slope;
	float error = 0;
	gboolean swap_x_y = FALSE;
		
	a = e->a;
	b = e->b;

	if( fabs(calc_slope(a,b)) > 1.0 )
	{
		swap_x_y = TRUE;
		swap( &a.x, &a.y );
		swap( &b.x, &b.y );
	}

	/* If b is to the left of a, swap a and b */
	if( b.x < a.x )
	{
		swap( &a.x, &b.x );
		swap( &a.y, &b.y );
	}
	
	slope = calc_slope( a, b );
	x = a.x;
	y = a.y;
	for( x = a.x; x <= b.x; x++ )
	{
		if( swap_x_y )
		{
			rotomap[y][x] = EDGE;
		}
		else
		{
			rotomap[x][y] = EDGE;
		}
		error += fabs( slope );
		if( error >= 0.5 )
		{
			if( slope < 0 )
			{
				y--;
			}
			else
			{
				y++;
			}
			error -= 1.0;
		}
	}
}

void plot_edges( RotoPixel** rotomap, GList* edges )
{
	while( NULL != edges )
	{
		plot_edge( rotomap, edges->data );
		edges = g_list_next( edges );
	}
}

guchar* get_pixel_ptr( guchar* base, int rowstride, int num_channels, int x, int y )
{
	return base + x * num_channels + y * rowstride;
}

Color get_pixel( GdkPixbuf* src, unsigned int x, unsigned int y )
{
	Color pixel_color;
	int num_channels;
	int rowstride;
	guchar* pixels_base;
	guchar* pixel;
	
	num_channels = gdk_pixbuf_get_n_channels( src );
	rowstride = gdk_pixbuf_get_rowstride( src );
	pixels_base = gdk_pixbuf_get_pixels( src );
	g_assert( GDK_COLORSPACE_RGB == gdk_pixbuf_get_colorspace(src) );
	g_assert( 8 == gdk_pixbuf_get_bits_per_sample(src) );
	pixel = get_pixel_ptr( pixels_base, rowstride, num_channels, x, y );
	pixel_color.r = pixel[0];
	pixel_color.g = pixel[1];
	pixel_color.b = pixel[2];
	return pixel_color;
}

void put_pixel( GdkPixbuf* dest, unsigned int x, unsigned int y, Color color )
{
	int num_channels;
	int rowstride;
	guchar* pixels_base;
	guchar* pixel;

	num_channels = gdk_pixbuf_get_n_channels( dest );
	rowstride = gdk_pixbuf_get_rowstride( dest );
	pixels_base = gdk_pixbuf_get_pixels( dest );
	g_assert( GDK_COLORSPACE_RGB == gdk_pixbuf_get_colorspace(dest) );
	g_assert( 8 == gdk_pixbuf_get_bits_per_sample(dest) );
	pixel = get_pixel_ptr( pixels_base, rowstride, num_channels, x, y );
	pixel[0] = color.r;
	pixel[1] = color.g;
	pixel[2] = color.b;
}

Color calc_average_color( GList* pixels, GdkPixbuf* src )
{
	unsigned int num_samples = 0;
	float cumulative_red = 0;
	float cumulative_green = 0;
	float cumulative_blue = 0;
	Color average_color;
	Vertex* coords;
	Color pixel;
	
	/* For each pixel in the region, add its value to the cumulative color values */
	while( NULL != pixels )
	{
		/* Get a pointer to the right pixel */
		coords = pixels->data;
		pixel = get_pixel( src, coords->x, coords->y );

		/* Contribute the pixel toward the average color */
		cumulative_red += pixel.r;
		cumulative_green += pixel.g;
		cumulative_blue += pixel.b;
		num_samples++;
		
		/* Move to the next pixel */
		pixels = g_list_next( pixels );
	}

	/* Calculate the average color */
	if( num_samples > 0 )
	{
		average_color.r = cumulative_red / num_samples;
		average_color.g = cumulative_green / num_samples;
		average_color.b = cumulative_blue / num_samples;
	}

	return average_color;
}

void color_pixels( GList* pixels, GdkPixbuf* dest, Color color )
{
	Vertex* coords;
	
	/* For each pixel in the region, add its value to the cumulative color values */
	while( NULL != pixels )
	{
		/* Get a pointer to the right pixel */
		coords = pixels->data;

		/* Colour the pixel */
		put_pixel( dest, coords->x, coords->y, color );
		
		/* Move to the next pixel */
		pixels = g_list_next( pixels );
	}
}

void render_region( RotoPixel** rotomap, unsigned int width, unsigned int height,
		unsigned int x, unsigned int y,
		GdkPixbuf* src, GdkPixbuf* dest )
{
	GList* pixels;
	Color average_color;
	
	/* Find the region's pixels */
	pixels = get_region_pixels( rotomap, width, height, x, y );
	
	/* Calculate the average color of those pixels */
	average_color = calc_average_color( pixels, src );

	/* Color the region's pixels in the resulting pixbuf with the average color */
	color_pixels( pixels, dest, average_color );

	/* Free the list of the region's pixels */
	free_g_list( pixels );
}

void render( GdkPixbuf* src, GList* edges, GdkPixbuf* dest )
{
	RotoPixel** rotomap;
	int width;
	int height;
	unsigned int i;
	unsigned int j;
	Color edge_pixel_color;
	
	width = gdk_pixbuf_get_width( src );
	height = gdk_pixbuf_get_height( src );
	
	rotomap = create_rotomap( width, height );
	if( NULL == rotomap )
	{
		/* Out of memory */
		/* TODO: Handle this better! */
		return;
	}
	
	plot_edges( rotomap, edges );
	
	/* For each pixel, ensure it is rendered as a region or an edge */
	for( i = 0; i < width; i++ )
	{
		for( j = 0; j < height; j++ )
		{
			if( UNUSED == rotomap[i][j] )
			{
				render_region( rotomap, width, height, i, j, src, dest );
			}
			if( EDGE == rotomap[i][j] )
			{
				edge_pixel_color = get_pixel( src, i, j );
				put_pixel( dest, i, j, edge_pixel_color );
			}
		}
	}
}

