/*
 * Copyright 2023 Hans-Kristian Arntzen for Valve Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#define VKD3D_DBG_CHANNEL VKD3D_DBG_CHANNEL_API
#include "d3d12_crosstest.h"

enum pso_index
{
    PSO_SINGLE_SAMPLE = 0,
    PSO_MULTI_SAMPLE_PER_PIXEL = 1,
    PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED = 2,
    PSO_MULTI_SAMPLE_PER_SAMPLE = 3,
    PSO_COUNT
};

struct quad
{
    float top_left[2];
    float bottom_right[2];
    float mul;
    float add;
    enum pso_index pso_index;
    unsigned int instances;
};

struct test
{
    unsigned int width;
    unsigned int height;
    unsigned int samples;
    const struct quad *quads;
    unsigned int num_quads;
};

static const struct quad quads_simple[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 1.0f, 0.0f }, { 4.0f, 4.0f }, 3.0f, +9.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 2.0f, 2.0f }, { 3.0f, 3.0f }, 2.0f, -2.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 3.0f, 0.0f }, { 4.0f, 1.0f }, 4.0f, -1.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 1.0f, 0.0f }, { 4.0f, 2.0f }, 3.0f, +9.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 0.0f, 0.0f }, { 3.0f, 3.0f }, 2.0f, -2.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 0.0f, 1.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 1.0f, 2.0f }, { 3.0f, 4.0f }, 3.0f, +7.0f, PSO_SINGLE_SAMPLE, 1 },
    { { 0.0f, 3.0f }, { 2.0f, 4.0f }, 1.0f, -2.0f, PSO_SINGLE_SAMPLE, 1 },
};

static const struct quad quads_pixel[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 1.5f, 0.5f }, { 4.0f, 4.0f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 2.0f, 2.0f }, { 3.5f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 3.5f, 0.0f }, { 4.0f, 1.5f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 1.0f, 0.5f }, { 4.0f, 2.5f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 0.5f, 0.0f }, { 3.0f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 0.0f, 1.5f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 1.5f, 2.0f }, { 3.5f, 4.0f }, 3.0f, +7.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
    { { 0.5f, 3.0f }, { 2.0f, 4.0f }, 1.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 1 },
};

static const struct quad quads_pixel_unrolled[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 1.5f, 0.5f }, { 4.0f, 4.0f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 2.0f, 2.0f }, { 3.5f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 3.5f, 0.0f }, { 4.0f, 1.5f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 1.0f, 0.5f }, { 4.0f, 2.5f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 0.5f, 0.0f }, { 3.0f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 0.0f, 1.5f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 1.5f, 2.0f }, { 3.5f, 4.0f }, 3.0f, +7.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
    { { 0.5f, 3.0f }, { 2.0f, 4.0f }, 1.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 1 },
};

static const struct quad quads_sample[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 1.5f, 0.5f }, { 4.0f, 4.0f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 2.0f, 2.0f }, { 3.5f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 3.5f, 0.0f }, { 4.0f, 1.5f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 1.0f, 0.5f }, { 4.0f, 2.5f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 0.5f, 0.0f }, { 3.0f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 0.0f, 1.5f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 1.5f, 2.0f }, { 3.5f, 4.0f }, 3.0f, +7.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 0.5f, 3.0f }, { 2.0f, 4.0f }, 1.0f, -2.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
};

static const struct quad quads_simple_instanced[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, 1.0f, PSO_SINGLE_SAMPLE, 10 },
};

static const struct quad quads_pixel_instanced[] =
{
    { { 0.5f, 0.0f }, { 4.0f, 3.5f }, 4.0f, 1.0f, PSO_MULTI_SAMPLE_PER_PIXEL, 10 },
};

static const struct quad quads_pixel_unrolled_instanced[] =
{
    { { 0.0f, 0.5f }, { 3.5f, 4.0f }, 4.0f, 1.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 10 },
};

static const struct quad quads_sample_instanced[] =
{
    { { 0.5f, 0.5f }, { 3.5f, 3.5f }, 4.0f, 1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 10 },
};

static const struct quad quads_mixed[] =
{
    { { 0.0f, 0.0f }, { 4.0f, 4.0f }, 4.0f, -1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 1 },
    { { 1.5f, 0.5f }, { 4.0f, 4.0f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 2 },
    { { 2.0f, 2.0f }, { 3.5f, 3.5f }, 2.0f, -2.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 3 },
    { { 3.5f, 0.0f }, { 4.0f, 1.5f }, 2.0f, -1.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 4 },
    { { 1.0f, 0.5f }, { 4.0f, 2.5f }, 3.0f, +9.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 5 },
    { { 0.5f, 0.0f }, { 3.0f, 3.5f }, 1.0f, -2.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 6 },
    { { 0.0f, 1.5f }, { 4.0f, 4.0f }, 2.0f, -1.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 4 },
    { { 1.5f, 2.0f }, { 3.5f, 4.0f }, 2.0f, +7.0f, PSO_MULTI_SAMPLE_PER_PIXEL_UNROLLED, 2 },
    { { 0.5f, 3.0f }, { 2.0f, 4.0f }, 1.0f, -2.0f, PSO_MULTI_SAMPLE_PER_SAMPLE, 4 },
};

static void validate_readback(struct resource_readback *rb, const struct test *test, unsigned int slice)
{
    float sample_offset_x = 0.5f;
    float sample_offset_y = 0.5f;
    const struct quad *quad;
    unsigned int x, y, i, j;

    static const float sample_offsets[4][2] = {
        { 0.375f, 0.125f },
        { 0.875f, 0.375f },
        { 0.125f, 0.625f },
        { 0.625f, 0.875f },
    };

    if (test->samples > 1)
    {
        sample_offset_x = sample_offsets[slice][0];
        sample_offset_y = sample_offsets[slice][1];
    }

    /* Replay the quads, one pixel at a time. */
    for (y = 0; y < test->height; y++)
    {
        for (x = 0; x < test->width; x++)
        {
            float reference = 0.0f;
            float pixel;

            for (i = 0; i < test->num_quads; i++)
            {
                float sx = (float)x + sample_offset_x;
                float sy = (float)y + sample_offset_y;
                quad = &test->quads[i];

                /* Pedantic top-left raster rules. */
                if (sx >= quad->top_left[0] && sy >= quad->top_left[1] &&
                    sx < quad->bottom_right[0] && sy < quad->bottom_right[1])
                {
                    for (j = 0; j < quad->instances; j++)
                        reference = reference * quad->mul + quad->add;
                }
            }

            pixel = get_readback_float(rb, x, y);
            ok(pixel == reference, "Pixel %u, %u: Got %f, expected %f.\n", x, y, pixel, reference);
        }
    }
}

static ID3D12PipelineState *create_pso(ID3D12Device *device, ID3D12RootSignature *rs, enum pso_index pso_index, bool use_dxil)
{
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pso_desc;
    ID3D12PipelineState *pso = NULL;
    HRESULT hr;

#if 0
    RasterizerOrderedTexture2D<float> ROV : register(u0);
    RasterizerOrderedTexture2DArray<float> ROVArray : register(u0);
    cbuffer Params : register(b0)
    {
        float2 top_left;
        float2 bottom_right;
        float A, B;
        float2 inv_res;
    };

    struct VSOut
    {
        float4 pos : SV_Position;
    };

    VSOut vs_main(uint vid : SV_VertexID)
    {
        VSOut vout;
        vout.pos.x = float(vid & 1);
        vout.pos.y = 0.5 * float(vid & 2);
        vout.pos.xy = lerp(top_left, bottom_right, vout.pos.xy) * inv_res;
        vout.pos.xy *= 2.0;
        vout.pos.xy -= 1.0;
        vout.pos.y = -vout.pos.y;
        vout.pos.z = 0.0;
        vout.pos.w = 1.0;
        return vout;
    }

    void ps_main_single(VSOut vout)
    {
        uint2 coord = uint2(vout.pos.xy);
        float v = ROV[coord];
        v = v * A + B;
        ROV[coord] = v;
    }

    void ps_main_pixel(VSOut vout, uint cov : SV_Coverage)
    {
        uint2 coord = uint2(vout.pos.xy);
        for (uint i = 0; i < 4; i++)
        {
            if (cov & (1u << i))
            {
                float v = ROVArray[int3(coord, i)];
                v = v * A + B;
                ROVArray[int3(coord, i)] = v;
            }
        }
    }

    void ps_main_pixel_unrolled(VSOut vout, uint cov : SV_Coverage)
    {
        uint2 coord = uint2(vout.pos.xy);
        [unroll]
        for (uint i = 0; i < 4; i++)
        {
            if (cov & (1u << i))
            {
                float v = ROVArray[int3(coord, i)];
                v = v * A + B;
                ROVArray[int3(coord, i)] = v;
            }
        }
    }

    void ps_main_sample(VSOut vout, uint samp : SV_SampleIndex)
    {
        uint3 coord = uint3(vout.pos.xy, samp);
        float v = ROVArray[coord];
        v = v * A + B;
        ROVArray[coord] = v;
    }
#endif

    static const DWORD vs_code_dxbc[] =
    {
        0x43425844, 0xbb2e0271, 0x7f1cee5d, 0x8d9fcdac, 0xfdef5f5b, 0x00000001, 0x00000234, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x65747265, 0x00444978,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x00000198, 0x00010050,
        0x00000066, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x04000060, 0x00101012,
        0x00000000, 0x00000006, 0x04000067, 0x001020f2, 0x00000000, 0x00000001, 0x02000068, 0x00000002,
        0x0a000000, 0x00100032, 0x00000000, 0x80208046, 0x00000041, 0x00000000, 0x00000000, 0x00208ae6,
        0x00000000, 0x00000000, 0x0a000001, 0x001000c2, 0x00000000, 0x00101006, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000001, 0x00000002, 0x05000056, 0x00100032, 0x00000001, 0x00100ae6,
        0x00000000, 0x07000038, 0x00100042, 0x00000001, 0x0010001a, 0x00000001, 0x00004001, 0x3f000000,
        0x0a000032, 0x00100032, 0x00000000, 0x00100086, 0x00000001, 0x00100046, 0x00000000, 0x00208046,
        0x00000000, 0x00000000, 0x08000038, 0x00100032, 0x00000000, 0x00100046, 0x00000000, 0x00208ae6,
        0x00000000, 0x00000001, 0x0f000032, 0x00100032, 0x00000000, 0x00100046, 0x00000000, 0x00004002,
        0x40000000, 0x40000000, 0x00000000, 0x00000000, 0x00004002, 0xbf800000, 0xbf800000, 0x00000000,
        0x00000000, 0x06000036, 0x00102022, 0x00000000, 0x8010001a, 0x00000041, 0x00000000, 0x05000036,
        0x00102012, 0x00000000, 0x0010000a, 0x00000000, 0x08000036, 0x001020c2, 0x00000000, 0x00004002,
        0x00000000, 0x00000000, 0x00000000, 0x3f800000, 0x0100003e,
    };

    static const BYTE vs_code_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xf3, 0xd2, 0x8e, 0xa7, 0xb5, 0x16, 0x20, 0x2c, 0x30, 0xef, 0x8a, 0x6c, 0xdf, 0x18, 0xe5, 0xda, 0x01, 0x00, 0x00, 0x00, 0xc0, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0xc0, 0x00, 0x00, 0x00, 0x60, 0x01, 0x00, 0x00, 0x7c, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x56, 0x65, 0x72, 0x74, 0x65,
        0x78, 0x49, 0x44, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x00,
        0x50, 0x53, 0x56, 0x30, 0x98, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x01, 0x01, 0x00, 0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00,
        0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x41, 0x01, 0x01, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x14, 0xe4, 0x9e, 0xfa, 0xee, 0xe2, 0x64, 0xa6, 0x57, 0x6b, 0xf7, 0xa0, 0x08, 0x71, 0x14, 0x44, 0x58, 0x49, 0x4c,
        0x3c, 0x06, 0x00, 0x00, 0x60, 0x00, 0x01, 0x00, 0x8f, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x24, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde,
        0x21, 0x0c, 0x00, 0x00, 0x86, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39,
        0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88,
        0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06,
        0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x84, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x6d, 0x30, 0x86, 0xff,
        0xff, 0xff, 0xff, 0x1f, 0x00, 0x09, 0xa8, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00,
        0x20, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c,
        0x10, 0x5c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x18, 0x63, 0x90, 0x41, 0x65, 0x06, 0xa0, 0x18, 0x60, 0x8c, 0x41, 0x0e, 0xa1, 0xa3, 0x86, 0xcb, 0x9f,
        0xb0, 0x87, 0x90, 0x7c, 0x6e, 0xa3, 0x8a, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x11, 0x71, 0xce, 0x39, 0x44, 0xee, 0x19, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0x43, 0xa0, 0x19, 0x16, 0x02, 0x05, 0xab,
        0x10, 0x6a, 0xb0, 0x41, 0x6d, 0x8e, 0x20, 0x28, 0x06, 0x1b, 0x64, 0x8c, 0x47, 0x70, 0x20, 0x60, 0x18, 0x41, 0x38, 0xce, 0x48, 0x81, 0x08, 0x60, 0x24, 0x54, 0xcc, 0x79, 0xce, 0x24, 0x9a, 0x03,
        0x03, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d,
        0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e,
        0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10,
        0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78,
        0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x05, 0x10, 0x00,
        0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x0b, 0x04, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90,
        0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x05, 0x54, 0x06, 0xe5, 0x41, 0xa4, 0x14, 0x4a, 0x60, 0x04, 0xa0, 0x24, 0xca, 0xa0, 0x10, 0xa8, 0xce, 0x00, 0x90,
        0x1d, 0x6b, 0x08, 0x88, 0x01, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x53, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1,
        0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73,
        0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xa1, 0x98, 0x20, 0x0c, 0xc6, 0x06, 0x61, 0x20, 0x26, 0x08, 0xc3, 0xb1, 0x41, 0x18, 0x0c, 0x0a, 0x76, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08,
        0x55, 0x44, 0x60, 0x82, 0x30, 0x20, 0x1b, 0x10, 0x42, 0x59, 0x08, 0x62, 0x60, 0x80, 0x0d, 0x41, 0xb3, 0x81, 0x00, 0x00, 0x07, 0x98, 0x20, 0x58, 0xd2, 0x86, 0x00, 0x9a, 0x20, 0x08, 0x00, 0x0f,
        0xbb, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x56, 0x65, 0x72, 0x74, 0x65, 0x78, 0x49, 0x44, 0x13, 0x04, 0x62, 0x99, 0x20, 0x10, 0xcc, 0x86, 0x80, 0x98, 0x20, 0x10, 0xcd,
        0x04, 0x61, 0x48, 0x36, 0x08, 0xd9, 0xb0, 0x61, 0x21, 0xa8, 0xca, 0xba, 0xb0, 0xc1, 0x22, 0x30, 0x6d, 0x43, 0xb0, 0x71, 0x99, 0xb2, 0xfa, 0x82, 0x7a, 0x9b, 0x4b, 0xa3, 0x4b, 0x7b, 0x73, 0x9b,
        0x20, 0x10, 0xce, 0x04, 0x81, 0x78, 0x26, 0x08, 0x04, 0x34, 0x41, 0x18, 0x94, 0x0d, 0x42, 0x16, 0x06, 0x1b, 0x16, 0xa2, 0xf3, 0xbe, 0x0b, 0x0c, 0x06, 0x30, 0x20, 0x30, 0x31, 0xd8, 0x10, 0x8c,
        0xc1, 0x86, 0x81, 0x23, 0x03, 0x60, 0x43, 0x21, 0x4d, 0x65, 0xf0, 0x00, 0x55, 0xd8, 0xd8, 0xec, 0xda, 0x5c, 0xd2, 0xc8, 0xca, 0xdc, 0xe8, 0xa6, 0x04, 0x41, 0x15, 0x32, 0x3c, 0x17, 0xbb, 0x32,
        0xb9, 0xb9, 0xb4, 0x37, 0xb7, 0x29, 0x01, 0xd1, 0x84, 0x0c, 0xcf, 0xc5, 0x2e, 0x8c, 0xcd, 0xae, 0x4c, 0x6e, 0x4a, 0x60, 0xd4, 0x21, 0xc3, 0x73, 0x99, 0x43, 0x0b, 0x23, 0x2b, 0x93, 0x6b, 0x7a,
        0x23, 0x2b, 0x63, 0x9b, 0x12, 0x20, 0x65, 0xc8, 0xf0, 0x5c, 0xe4, 0xca, 0xe6, 0xde, 0xea, 0xe4, 0xc6, 0xca, 0xe6, 0xa6, 0x04, 0x4e, 0x1d, 0x32, 0x3c, 0x17, 0xbb, 0xb4, 0xb2, 0xbb, 0x24, 0xb2,
        0x29, 0xba, 0x30, 0xba, 0xb2, 0x29, 0x01, 0x54, 0x87, 0x0c, 0xcf, 0xa5, 0xcc, 0x8d, 0x4e, 0x2e, 0x0f, 0xea, 0x2d, 0xcd, 0x8d, 0x6e, 0x6e, 0x4a, 0x50, 0x06, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00,
        0x4c, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6,
        0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8,
        0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11,
        0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89,
        0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37,
        0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81,
        0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c,
        0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc,
        0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20,
        0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x15, 0x00, 0x00, 0x00, 0x36, 0xb0, 0x0d, 0x97,
        0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x04, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34,
        0xc3, 0x42, 0x58, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x8f, 0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x4d, 0x4e, 0x44, 0xa0, 0xd4, 0xf4,
        0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x01, 0x1c, 0x95, 0xf4, 0x33, 0x00, 0xd2, 0x00, 0x00, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x41, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00,
        0x0e, 0x00, 0x00, 0x00, 0x34, 0x4a, 0xa1, 0xec, 0x4a, 0xae, 0x10, 0x66, 0x00, 0x8a, 0x80, 0x48, 0x11, 0xd0, 0x1b, 0x01, 0xa0, 0x33, 0x46, 0x00, 0x82, 0x20, 0x88, 0xff, 0xc2, 0x18, 0x01, 0x08,
        0x82, 0x20, 0xfc, 0x8d, 0x11, 0x80, 0x20, 0x08, 0x82, 0xa0, 0x30, 0x02, 0x30, 0x46, 0x00, 0x82, 0x20, 0x88, 0x7f, 0x63, 0x04, 0x20, 0x08, 0x82, 0x20, 0x18, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x00, 0x75, 0x0c, 0xb2, 0x6d, 0xc7, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x14, 0x61, 0xc0, 0x70, 0x5c, 0xb5, 0x54, 0xe0, 0xe9, 0x05, 0xa7, 0x6a, 0x68, 0xf4, 0x82, 0x53, 0x16, 0x2c,
        0xf2, 0x19, 0x31, 0x38, 0x00, 0x10, 0x04, 0x83, 0xa6, 0x0c, 0xa8, 0x43, 0x0c, 0x46, 0x13, 0x02, 0x61, 0x34, 0x41, 0x18, 0x46, 0x13, 0x06, 0x60, 0x34, 0x81, 0x08, 0x8c, 0x10, 0xe2, 0x63, 0x84,
        0x10, 0x1f, 0x13, 0x16, 0xf9, 0x58, 0x22, 0xc8, 0xc7, 0x04, 0x03, 0x3e, 0x26, 0x18, 0xf0, 0x19, 0x31, 0x38, 0x00, 0x10, 0x04, 0x83, 0x06, 0x0e, 0x3e, 0xc9, 0x0d, 0x46, 0x13, 0x02, 0x61, 0x34,
        0x41, 0x18, 0x4c, 0xb0, 0xe4, 0x63, 0x81, 0x21, 0x1f, 0x1b, 0x30, 0xf9, 0x58, 0x70, 0xc8, 0xc7, 0x86, 0x0f, 0x3e, 0x26, 0x80, 0x01, 0x7c, 0xec, 0x0b, 0xe2, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08,
        0x06, 0xc8, 0x1e, 0xa8, 0xc1, 0x1d, 0xdc, 0x01, 0x1c, 0x0c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xec, 0x81, 0x1a, 0xdc, 0xc1, 0x1d, 0xc4, 0x41, 0x30, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06,
        0xc8, 0x1e, 0xa8, 0xc1, 0x1d, 0xdc, 0x01, 0x19, 0x7c, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0xec, 0x81, 0x1a, 0xdc, 0xc1, 0x1d, 0xb8, 0x81, 0x87, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const DWORD ps_code_single_dxbc[] =
    {
        0x43425844, 0xd5ca610a, 0xd65892f5, 0x1f64d850, 0x8e7e93ce, 0x00000001, 0x00000154, 0x00000004,
        0x00000030, 0x00000064, 0x00000074, 0x00000144, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000000c8, 0x00000050,
        0x00000032, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x0402189c, 0x0011e000,
        0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x02000068, 0x00000002,
        0x0500001c, 0x001000f2, 0x00000000, 0x00101546, 0x00000000, 0x890000a3, 0x800000c2, 0x00155543,
        0x00100012, 0x00000001, 0x00100fc6, 0x00000000, 0x0011ee46, 0x00000000, 0x0b000032, 0x00100012,
        0x00000001, 0x0010000a, 0x00000001, 0x0020800a, 0x00000000, 0x00000001, 0x0020801a, 0x00000000,
        0x00000001, 0x070000a4, 0x0011e0f2, 0x00000000, 0x00100e46, 0x00000000, 0x00100006, 0x00000001,
        0x0100003e, 0x30494653, 0x00000008, 0x00001000, 0x00000000,
    };

    static const BYTE ps_code_single_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xd9, 0x52, 0x6e, 0x4d, 0xe2, 0x81, 0xc6, 0x9e, 0xff, 0xcb, 0xb9, 0x55, 0x79, 0x7f, 0xc0, 0xb8, 0x01, 0x00, 0x00, 0x00, 0x9c, 0x07, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00, 0x48, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7d, 0x44, 0x1c, 0xb2, 0x7a, 0x90, 0x08, 0xca,
        0x2e, 0xbf, 0x55, 0x2a, 0x92, 0xfb, 0xcf, 0x17, 0x44, 0x58, 0x49, 0x4c, 0x4c, 0x06, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0x93, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0x34, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0x8a, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xa4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x39, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x74, 0x23, 0x00, 0x25, 0x00, 0x14, 0x66, 0x00, 0xe6, 0x08, 0xc0, 0x60, 0x8e, 0x00, 0x29, 0xc6, 0x20, 0x84, 0x14,
        0x42, 0xe6, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x51, 0x31, 0xc6, 0x18, 0x84, 0xca, 0x3d, 0xc3, 0xe5, 0x4f, 0xd8, 0x43, 0x48, 0x7e, 0x08,
        0x34, 0xc3, 0x42, 0xa0, 0x20, 0x15, 0x05, 0x11, 0x45, 0x08, 0x21, 0x84, 0x10, 0x5a, 0x65, 0x01, 0x44, 0x11, 0x42, 0xc6, 0x18, 0xa3, 0x50, 0x3b, 0x6a, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0xe7,
        0x36, 0xaa, 0x58, 0x89, 0xc9, 0x2f, 0x6e, 0x1b, 0x11, 0x63, 0x8c, 0x51, 0x88, 0x47, 0x14, 0x21, 0x38, 0x47, 0x10, 0x14, 0x43, 0x91, 0x42, 0x88, 0xa4, 0x39, 0x10, 0x30, 0xd3, 0x19, 0x8c, 0x03,
        0x3b, 0x84, 0xc3, 0x3c, 0xcc, 0x83, 0x1b, 0xc8, 0x42, 0x38, 0xcc, 0x03, 0x3d, 0x94, 0x83, 0x3c, 0xa4, 0x83, 0x3e, 0x94, 0x83, 0x3c, 0xbc, 0x82, 0x3c, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x90,
        0x03, 0x2d, 0x94, 0x03, 0x3e, 0xd0, 0x43, 0x3d, 0xc8, 0x43, 0x39, 0xc8, 0x01, 0x29, 0xf0, 0x81, 0x39, 0xb0, 0xc3, 0x3b, 0x84, 0x03, 0x3d, 0xf8, 0x01, 0x0a, 0x06, 0xd9, 0x61, 0x04, 0x61, 0x38,
        0x23, 0x05, 0x22, 0x80, 0x91, 0x50, 0x81, 0xf1, 0x18, 0x98, 0x72, 0x0a, 0x90, 0x39, 0x02, 0x50, 0x00, 0x00, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79,
        0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0,
        0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07,
        0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x06, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x16, 0x20, 0x00, 0x04, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x81, 0x80, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60,
        0xc8, 0x33, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x16, 0x08, 0x00, 0x0e, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x22, 0x25, 0x30, 0x02, 0x50, 0x0c, 0x45, 0x50, 0x12, 0x05, 0x54, 0x06, 0x54, 0x4a, 0xa2, 0x0c, 0x0a, 0x61, 0x04, 0x80, 0xe4, 0x08, 0x40, 0x0d, 0x10, 0x2f, 0x40, 0x40, 0x10,
        0xd2, 0x33, 0x00, 0xb4, 0xc7, 0x12, 0x08, 0x80, 0xee, 0x0c, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b,
        0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81,
        0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x04, 0xc2, 0x98, 0x20, 0x10, 0xc7, 0x06, 0x61, 0x20, 0x26, 0x08, 0x04, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06,
        0xc4, 0x20, 0x26, 0x08, 0x97, 0x45, 0x60, 0x82, 0x40, 0x24, 0x13, 0x04, 0x29, 0x9a, 0x20, 0x48, 0xd2, 0x04, 0x81, 0x50, 0x36, 0x08, 0xc4, 0xb3, 0x61, 0x21, 0x94, 0x85, 0x20, 0x06, 0xa6, 0x69,
        0x1c, 0x68, 0x43, 0x10, 0x4d, 0x10, 0x34, 0x6a, 0x82, 0x40, 0x2c, 0x1b, 0x10, 0x62, 0x5a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x00, 0x24, 0x0b, 0x98, 0x20, 0x6c, 0xd5, 0x86, 0x00,
        0x9b, 0x20, 0x08, 0x00, 0x0f, 0xb8, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0x37, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84, 0xa2, 0x99, 0x20, 0x14, 0xce, 0x86,
        0x80, 0x98, 0x20, 0x14, 0xcf, 0x04, 0xa1, 0x80, 0x26, 0x08, 0x04, 0xb3, 0x41, 0x10, 0x03, 0x31, 0xd8, 0xb0, 0x10, 0x5c, 0xe7, 0x7d, 0x60, 0x30, 0x80, 0x01, 0x11, 0x06, 0x63, 0xb0, 0x21, 0x20,
        0x83, 0x0d, 0x43, 0x19, 0x00, 0xc0, 0x04, 0x81, 0x9b, 0x36, 0x08, 0xc4, 0x19, 0x6c, 0x28, 0xb4, 0xcd, 0x0c, 0x2e, 0x34, 0xa8, 0xc2, 0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37,
        0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61, 0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e,
        0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7, 0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0xb0, 0xea, 0x90,
        0xe1, 0xb9, 0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d, 0x09, 0xb0, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e, 0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53,
        0x02, 0x34, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x49, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07,
        0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43,
        0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76,
        0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8,
        0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68,
        0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71,
        0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4,
        0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43,
        0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x8c, 0xc8, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x72, 0x10, 0x87, 0x73, 0x70, 0x03, 0x7b,
        0x08, 0x07, 0x79, 0x60, 0x87, 0x70, 0xc8, 0x87, 0x77, 0xa8, 0x07, 0x7a, 0x00, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x19, 0x00, 0x00, 0x00, 0x46, 0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10,
        0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x05, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3, 0x42, 0x58, 0xc0,
        0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x09, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0x44, 0xe9, 0x00, 0x83, 0x5f, 0xdc,
        0xb6, 0x0d, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0x04, 0x3b, 0x39, 0x11, 0xe1, 0x17, 0xb7, 0x6d, 0x00, 0xc7, 0x23, 0xfd, 0x0c, 0x80, 0x34, 0x00, 0x61, 0x20, 0x00, 0x00,
        0x25, 0x00, 0x00, 0x00, 0x13, 0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x44, 0x66, 0x00, 0x4a, 0xae, 0x10, 0x8a, 0x30, 0xa0, 0xec, 0xca, 0x30, 0x80, 0x4a, 0x09, 0x14,
        0x41, 0x79, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x30, 0x6d, 0xc8, 0x90, 0x65, 0xcf, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x4c, 0x5c, 0x32, 0x68, 0x1a, 0x34, 0x62, 0x90, 0x00, 0x20,
        0x08, 0x06, 0x46, 0x18, 0x24, 0xdb, 0x26, 0x2d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x60, 0x88, 0x81, 0xc2, 0x71, 0x06, 0x73, 0x02, 0x99, 0x13, 0xc8, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0xc1,
        0x42, 0x06, 0x8c, 0xf1, 0x08, 0xc1, 0xf3, 0x3c, 0xcf, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x01, 0x44, 0x06, 0xcd, 0x21, 0x06, 0xa3, 0x09, 0x01, 0x60, 0xc1, 0x20, 0x9f, 0xd1,
        0x84, 0x21, 0x30, 0x21, 0x80, 0xcf, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x34, 0x6a, 0x10, 0x35, 0x09, 0x62, 0x05, 0x41, 0x10, 0x38, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const DWORD ps_code_pixel_dxbc[] =
    {
        0x43425844, 0x3456ecbd, 0x09fb256a, 0x4fcefd5e, 0x409362dd, 0x00000001, 0x00000214, 0x00000004,
        0x00000030, 0x00000064, 0x00000074, 0x00000204, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x00000188, 0x00000050,
        0x00000062, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x0402409c, 0x0011e000,
        0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x0200005f, 0x00023001,
        0x02000068, 0x00000002, 0x0500001c, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x05000036,
        0x00100012, 0x00000001, 0x00004001, 0x00000000, 0x01000030, 0x07000050, 0x00100022, 0x00000001,
        0x0010000a, 0x00000001, 0x00004001, 0x00000004, 0x03040003, 0x0010001a, 0x00000001, 0x07000029,
        0x00100022, 0x00000001, 0x00004001, 0x00000001, 0x0010000a, 0x00000001, 0x06000001, 0x00100022,
        0x00000001, 0x0010001a, 0x00000001, 0x0002300a, 0x0304001f, 0x0010001a, 0x00000001, 0x05000036,
        0x001000c2, 0x00000000, 0x00100006, 0x00000001, 0x890000a3, 0x80000202, 0x00155543, 0x00100022,
        0x00000001, 0x00100f46, 0x00000000, 0x0011ee16, 0x00000000, 0x0b000032, 0x00100022, 0x00000001,
        0x0010001a, 0x00000001, 0x0020800a, 0x00000000, 0x00000001, 0x0020801a, 0x00000000, 0x00000001,
        0x070000a4, 0x0011e0f2, 0x00000000, 0x00100e46, 0x00000000, 0x00100556, 0x00000001, 0x01000015,
        0x0700001e, 0x00100012, 0x00000001, 0x0010000a, 0x00000001, 0x00004001, 0x00000001, 0x01000016,
        0x0100003e, 0x30494653, 0x00000008, 0x00001000, 0x00000000,
    };

    static const BYTE ps_code_pixel_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xd4, 0x87, 0x95, 0xe3, 0x8f, 0x7d, 0x82, 0x93, 0x10, 0x53, 0xfe, 0x69, 0x4c, 0x5a, 0xdb, 0x45, 0x01, 0x00, 0x00, 0x00, 0x34, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00, 0x48, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x59, 0x04, 0xd1, 0x79, 0x6e, 0x44, 0x04, 0xe2,
        0xdc, 0x46, 0x8c, 0xb8, 0x36, 0x18, 0xa0, 0xfe, 0x44, 0x58, 0x49, 0x4c, 0xe4, 0x06, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xb9, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xcc, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xb0, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x08, 0xc3, 0x40, 0xc4, 0x0c, 0xc0, 0x1c, 0x01,
        0x52, 0x0c, 0x63, 0x18, 0x86, 0x63, 0x20, 0xe4, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x51, 0x61, 0x18, 0x86, 0x31, 0xd0, 0x71, 0xcf, 0x70,
        0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0x8a, 0xa2, 0x0c, 0xcc, 0x30, 0x0c, 0xc3, 0x30, 0x0c, 0xd4, 0x94, 0x05, 0x18, 0x98, 0x61, 0x18, 0x0c, 0xc3, 0x30, 0x0e,
        0x7a, 0x8e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0xb9, 0x8d, 0x2a, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0x84, 0x61, 0x18, 0x46, 0x21, 0xa2, 0x81, 0x19, 0x48, 0x9a, 0x23, 0x08, 0x8a, 0xc1,
        0x0c, 0xc7, 0x30, 0x50, 0x54, 0x0d, 0x04, 0xcc, 0xc4, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb2, 0x10, 0x0e, 0xf3, 0x40, 0x0f, 0xe5, 0x20, 0x0f, 0xe9, 0xa0, 0x0f, 0xe5,
        0x20, 0x0f, 0xaf, 0x20, 0x0f, 0xe4, 0x50, 0x0e, 0xf2, 0x50, 0x0e, 0xe4, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0xa1, 0x20, 0x0f, 0xf2, 0x10, 0x0e,
        0xf9, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x30, 0x08, 0x1b, 0x46, 0x10, 0x8c, 0x33, 0x52, 0x20, 0x02, 0x18, 0x09, 0x15, 0x9a, 0x66, 0x18, 0x1a, 0x6d, 0x29,
        0x30, 0xcc, 0x11, 0x80, 0xc2, 0x14, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86,
        0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x90, 0xa7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x4a, 0x60, 0x04, 0xa0, 0x18, 0xca, 0xa1, 0x24, 0x0a, 0xa8, 0x0c, 0xe8, 0x28, 0x89, 0x32, 0x28, 0x84, 0x11, 0x00, 0xa2, 0x46, 0x00, 0x6a, 0x80, 0xbc, 0x02, 0x04, 0x04,
        0x21, 0x6e, 0x06, 0x80, 0xba, 0xb1, 0x04, 0x02, 0xa0, 0x6c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a,
        0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x26, 0x08, 0x43, 0xb2, 0x41, 0x18, 0x0c, 0x0a,
        0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0xd9, 0x45, 0x60, 0x82, 0x30, 0x28, 0x13, 0x04, 0x4a, 0x9a, 0x20, 0x50, 0xd3, 0x04, 0x61, 0x58, 0x36, 0x08, 0xc4, 0xb3, 0x61, 0x21, 0x94, 0x85,
        0x20, 0x06, 0xa6, 0x69, 0x1c, 0x68, 0x43, 0x10, 0x4d, 0x10, 0xb8, 0x6a, 0x82, 0x30, 0x30, 0x1b, 0x10, 0x62, 0x5a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x00, 0x24, 0x0b, 0x98, 0x20,
        0x74, 0xd6, 0x86, 0x00, 0x9b, 0x20, 0x08, 0x00, 0x1b, 0xb8, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0xb7, 0x2f, 0xb8, 0x34, 0xbc, 0x32, 0x36, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69,
        0x6f, 0x6e, 0x13, 0x84, 0xc3, 0x99, 0x20, 0x1c, 0xcf, 0x86, 0x80, 0x98, 0x20, 0x1c, 0xd0, 0x04, 0xe1, 0x88, 0x26, 0x08, 0x43, 0xb3, 0x41, 0x10, 0x03, 0x31, 0xd8, 0xb0, 0x10, 0x5c, 0xe7, 0x7d,
        0x60, 0x30, 0x80, 0x01, 0x11, 0x06, 0x63, 0xb0, 0x21, 0x20, 0x83, 0x0d, 0x43, 0x19, 0x00, 0xc0, 0x04, 0xc1, 0xa3, 0x36, 0x08, 0xc4, 0x19, 0x6c, 0x28, 0xb4, 0xcd, 0x0c, 0x2e, 0x34, 0xa8, 0xc2,
        0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61,
        0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7,
        0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0xb0, 0xea, 0x90, 0xe1, 0xb9, 0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d, 0x09, 0xb0, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e,
        0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53, 0x02, 0x34, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x51, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4,
        0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e, 0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90,
        0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x30, 0x83, 0x81, 0xc8, 0x01, 0x1f, 0xdc, 0x40, 0x1c, 0xe4, 0xa1, 0x1c, 0xc2, 0x61, 0x1d, 0xdc, 0x40, 0x1c, 0xe4, 0x01, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00,
        0x1e, 0x00, 0x00, 0x00, 0x56, 0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x01, 0xd2, 0x70, 0xf9, 0xce,
        0xe3, 0x0b, 0x4e, 0x45, 0x44, 0x80, 0x41, 0xf8, 0xc8, 0x6d, 0x9b, 0x81, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x26, 0x30, 0x0d, 0x97, 0xef, 0x3c,
        0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0xd5, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17, 0xb7, 0x6d, 0x04, 0xd6, 0x70,
        0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xc1, 0x4e, 0x4e, 0x44, 0xf8, 0xc5, 0x6d, 0x1b, 0x40, 0xf3, 0x48, 0x3f, 0x03, 0x20, 0xcd, 0xff, 0x20, 0x17, 0xb1, 0x00, 0x61, 0x20, 0x00, 0x00,
        0x33, 0x00, 0x00, 0x00, 0x13, 0x04, 0x45, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x34, 0xcc, 0x00, 0x14, 0x42, 0xc9, 0x95, 0x6d, 0x40, 0x11, 0x06, 0x94, 0x5d, 0x19, 0x06, 0xd0,
        0x51, 0x02, 0x45, 0x50, 0x1e, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x71, 0xc8, 0xa0, 0x69, 0xd0, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x54, 0x5d, 0x32, 0x6c, 0x5b, 0x34,
        0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x88, 0x18, 0x2c, 0x1c, 0x37, 0x31, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x8c, 0x01, 0xd3, 0x75, 0x46, 0x33, 0x62, 0x50, 0x00, 0x20, 0x08, 0x06, 0x44,
        0x19, 0x2c, 0x37, 0x0c, 0x73, 0xc3, 0x30, 0xb3, 0x04, 0xc1, 0x40, 0xc5, 0x20, 0x0a, 0xc0, 0x35, 0xd4, 0x18, 0x04, 0x57, 0x41, 0xa1, 0xc3, 0x0d, 0x01, 0x19, 0x80, 0xc1, 0x2c, 0xc3, 0x20, 0x04,
        0x23, 0x06, 0x0d, 0x00, 0x82, 0x60, 0xd0, 0xa8, 0x41, 0xb4, 0x54, 0x46, 0x41, 0x54, 0x55, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0xa9, 0x81, 0xc4, 0xa0, 0xc1, 0x68,
        0x42, 0x00, 0x58, 0x30, 0xc8, 0x67, 0x34, 0x61, 0x08, 0x4c, 0x08, 0xe0, 0x33, 0x62, 0xe0, 0x00, 0x20, 0x08, 0x06, 0x0f, 0x1c, 0x58, 0x52, 0xc3, 0x2c, 0x41, 0x10, 0x04, 0xd3, 0x2c, 0xc1, 0x60,
        0x4b, 0x1b, 0xc0, 0x60, 0xb8, 0x21, 0xe0, 0xc0, 0x60, 0x96, 0x81, 0x08, 0x02, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const DWORD ps_code_pixel_unrolled_dxbc[] =
    {
        0x43425844, 0xfda29fc6, 0x8a13866e, 0xb7151e52, 0x62299bc3, 0x00000001, 0x0000036c, 0x00000004,
        0x00000030, 0x00000064, 0x00000074, 0x0000035c, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008,
        0x00000020, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x505f5653, 0x7469736f,
        0x006e6f69, 0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853, 0x000002e0, 0x00000050,
        0x000000b8, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002, 0x0402409c, 0x0011e000,
        0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001, 0x0200005f, 0x00023001,
        0x02000068, 0x00000002, 0x0500001c, 0x00100032, 0x00000000, 0x00101046, 0x00000000, 0x09000001,
        0x001000f2, 0x00000001, 0x00023006, 0x00004002, 0x00000001, 0x00000002, 0x00000004, 0x00000008,
        0x0304001f, 0x0010000a, 0x00000001, 0x05000036, 0x00100042, 0x00000000, 0x00004001, 0x00000000,
        0x890000a3, 0x80000202, 0x00155543, 0x00100012, 0x00000001, 0x00100a46, 0x00000000, 0x0011ee46,
        0x00000000, 0x0b000032, 0x00100012, 0x00000001, 0x0010000a, 0x00000001, 0x0020800a, 0x00000000,
        0x00000001, 0x0020801a, 0x00000000, 0x00000001, 0x070000a4, 0x0011e0f2, 0x00000000, 0x00100a46,
        0x00000000, 0x00100006, 0x00000001, 0x01000015, 0x0304001f, 0x0010001a, 0x00000001, 0x05000036,
        0x00100082, 0x00000000, 0x00004001, 0x00000001, 0x890000a3, 0x80000202, 0x00155543, 0x00100012,
        0x00000001, 0x00100f46, 0x00000000, 0x0011ee46, 0x00000000, 0x0b000032, 0x00100012, 0x00000001,
        0x0010000a, 0x00000001, 0x0020800a, 0x00000000, 0x00000001, 0x0020801a, 0x00000000, 0x00000001,
        0x070000a4, 0x0011e0f2, 0x00000000, 0x00100f46, 0x00000000, 0x00100006, 0x00000001, 0x01000015,
        0x0304001f, 0x0010002a, 0x00000001, 0x08000036, 0x001000c2, 0x00000000, 0x00004002, 0x00000000,
        0x00000000, 0x00000002, 0x00000002, 0x890000a3, 0x80000202, 0x00155543, 0x00100012, 0x00000001,
        0x00100f46, 0x00000000, 0x0011ee46, 0x00000000, 0x0b000032, 0x00100012, 0x00000001, 0x0010000a,
        0x00000001, 0x0020800a, 0x00000000, 0x00000001, 0x0020801a, 0x00000000, 0x00000001, 0x070000a4,
        0x0011e0f2, 0x00000000, 0x00100e46, 0x00000000, 0x00100006, 0x00000001, 0x01000015, 0x0304001f,
        0x0010003a, 0x00000001, 0x08000036, 0x001000c2, 0x00000000, 0x00004002, 0x00000000, 0x00000000,
        0x00000003, 0x00000003, 0x890000a3, 0x80000202, 0x00155543, 0x00100012, 0x00000001, 0x00100f46,
        0x00000000, 0x0011ee46, 0x00000000, 0x0b000032, 0x00100012, 0x00000001, 0x0010000a, 0x00000001,
        0x0020800a, 0x00000000, 0x00000001, 0x0020801a, 0x00000000, 0x00000001, 0x070000a4, 0x0011e0f2,
        0x00000000, 0x00100e46, 0x00000000, 0x00100006, 0x00000001, 0x01000015, 0x0100003e, 0x30494653,
        0x00000008, 0x00001000, 0x00000000,
    };

    static const BYTE ps_code_pixel_unrolled_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0xb9, 0x07, 0x35, 0x08, 0xb9, 0x78, 0xac, 0x94, 0xe3, 0xc8, 0x73, 0x50, 0x46, 0xec, 0xd8, 0xaa, 0x01, 0x00, 0x00, 0x00, 0x2c, 0x09, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0x84, 0x00, 0x00, 0x00, 0x94, 0x00, 0x00, 0x00, 0x2c, 0x01, 0x00, 0x00, 0x48, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x34, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x28, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0x90, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x27, 0xf7, 0xa6, 0x41, 0x9d, 0xf5, 0x7a, 0x1d,
        0x12, 0xbd, 0xc6, 0x7e, 0xdd, 0xfe, 0x7c, 0x40, 0x44, 0x58, 0x49, 0x4c, 0xdc, 0x07, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xf7, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00,
        0x10, 0x00, 0x00, 0x00, 0xc4, 0x07, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xee, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00,
        0x07, 0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x18, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42,
        0xc4, 0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x62, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x11, 0x23, 0xc4, 0x50,
        0x41, 0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x31, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d,
        0x86, 0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42,
        0x20, 0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3e, 0x00, 0x00, 0x00, 0x32, 0x22, 0x88, 0x09, 0x20, 0x64, 0x85, 0x04, 0x13, 0x23, 0xa4, 0x84, 0x04, 0x13, 0x23, 0xe3,
        0x84, 0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8c, 0x8c, 0x0b, 0x84, 0xc4, 0x4c, 0x10, 0x80, 0xc1, 0x08, 0x40, 0x09, 0x00, 0x0a, 0xe6, 0x08, 0xc0, 0xa0, 0x08, 0xc3, 0x40, 0xc4, 0x0c, 0xc0, 0x1c, 0x01,
        0x52, 0x0c, 0x63, 0x18, 0x86, 0x63, 0x20, 0xe4, 0xa6, 0xe1, 0xf2, 0x27, 0xec, 0x21, 0x24, 0x7f, 0x25, 0xa4, 0x95, 0x98, 0xfc, 0xe2, 0xb6, 0x51, 0x61, 0x18, 0x86, 0x31, 0xd0, 0x71, 0xcf, 0x70,
        0xf9, 0x13, 0xf6, 0x10, 0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0x8a, 0xa2, 0x0c, 0xcc, 0x30, 0x0c, 0xc3, 0x30, 0x0c, 0xd4, 0x94, 0x05, 0x18, 0x98, 0x61, 0x18, 0x0c, 0xc3, 0x30, 0x0e,
        0x7a, 0x8e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e, 0x42, 0xf2, 0xb9, 0x8d, 0x2a, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0x84, 0x61, 0x18, 0x46, 0x21, 0xa2, 0x81, 0x19, 0x48, 0x9a, 0x23, 0x08, 0x8a, 0xc1,
        0x0c, 0xc7, 0x30, 0x50, 0x54, 0x0d, 0x04, 0xcc, 0xc4, 0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb2, 0x10, 0x0e, 0xf3, 0x40, 0x0f, 0xe5, 0x20, 0x0f, 0xe9, 0xa0, 0x0f, 0xe5,
        0x20, 0x0f, 0xaf, 0x20, 0x0f, 0xe4, 0x50, 0x0e, 0xf2, 0x50, 0x0e, 0xe4, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0xa1, 0x20, 0x0f, 0xf2, 0x10, 0x0e,
        0xf9, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e, 0xf4, 0xe0, 0x07, 0x28, 0x30, 0x08, 0x1b, 0x46, 0x10, 0x8c, 0x33, 0x52, 0x20, 0x02, 0x18, 0x09, 0x15, 0x9a, 0x66, 0x18, 0x1a, 0x6d, 0x29,
        0x30, 0xcc, 0x11, 0x80, 0xc2, 0x14, 0x00, 0x00, 0x13, 0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e,
        0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20,
        0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74,
        0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07,
        0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86,
        0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79, 0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0xc0, 0x90, 0xa7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x0f, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47,
        0xc6, 0x04, 0x43, 0x1a, 0x4a, 0x60, 0x04, 0xa0, 0x18, 0xca, 0xa1, 0x24, 0x0a, 0xa8, 0x0c, 0xe8, 0x28, 0x89, 0x32, 0x28, 0x84, 0x11, 0x00, 0xa2, 0x46, 0x00, 0x6a, 0x80, 0xbc, 0x02, 0x04, 0x04,
        0x21, 0x6e, 0x06, 0x80, 0xba, 0xb1, 0x04, 0x02, 0xa0, 0x6c, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x56, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44,
        0x35, 0x18, 0x63, 0x0b, 0x73, 0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a,
        0xfa, 0x9a, 0xb9, 0x81, 0x79, 0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x26, 0x08, 0x43, 0xb2, 0x41, 0x18, 0x0c, 0x0a,
        0x70, 0x73, 0x1b, 0x06, 0xc4, 0x20, 0x26, 0x08, 0xd9, 0x45, 0x60, 0x82, 0x30, 0x28, 0x13, 0x04, 0x4a, 0x9a, 0x20, 0x50, 0xd3, 0x04, 0x61, 0x58, 0x36, 0x08, 0xc4, 0xb3, 0x61, 0x21, 0x94, 0x85,
        0x20, 0x06, 0xa6, 0x69, 0x1c, 0x68, 0x43, 0x10, 0x4d, 0x10, 0xb8, 0x6a, 0x82, 0x30, 0x30, 0x1b, 0x10, 0x62, 0x5a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x00, 0x24, 0x0b, 0x98, 0x20,
        0x74, 0xd6, 0x86, 0x00, 0x9b, 0x20, 0x08, 0x00, 0x1b, 0xb8, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0xb7, 0x2f, 0xb8, 0x34, 0xbc, 0x32, 0x36, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69,
        0x6f, 0x6e, 0x13, 0x84, 0xc3, 0x99, 0x20, 0x1c, 0xcf, 0x86, 0x80, 0x98, 0x20, 0x1c, 0xd0, 0x04, 0xe1, 0x88, 0x26, 0x08, 0x43, 0xb3, 0x41, 0x10, 0x03, 0x31, 0xd8, 0xb0, 0x10, 0x5c, 0xe7, 0x7d,
        0x60, 0x30, 0x80, 0x01, 0x11, 0x06, 0x63, 0xb0, 0x21, 0x20, 0x83, 0x0d, 0x43, 0x19, 0x00, 0xc0, 0x04, 0xc1, 0xa3, 0x36, 0x08, 0xc4, 0x19, 0x6c, 0x28, 0xb4, 0xcd, 0x0c, 0x2e, 0x34, 0xa8, 0xc2,
        0xc6, 0x66, 0xd7, 0xe6, 0x92, 0x46, 0x56, 0xe6, 0x46, 0x37, 0x25, 0x08, 0xaa, 0x90, 0xe1, 0xb9, 0xd8, 0x95, 0xc9, 0xcd, 0xa5, 0xbd, 0xb9, 0x4d, 0x09, 0x88, 0x26, 0x64, 0x78, 0x2e, 0x76, 0x61,
        0x6c, 0x76, 0x65, 0x72, 0x53, 0x02, 0xa3, 0x0e, 0x19, 0x9e, 0xcb, 0x1c, 0x5a, 0x18, 0x59, 0x99, 0x5c, 0xd3, 0x1b, 0x59, 0x19, 0xdb, 0x94, 0x00, 0x29, 0x43, 0x86, 0xe7, 0x22, 0x57, 0x36, 0xf7,
        0x56, 0x27, 0x37, 0x56, 0x36, 0x37, 0x25, 0xb0, 0xea, 0x90, 0xe1, 0xb9, 0xd8, 0xa5, 0x95, 0xdd, 0x25, 0x91, 0x4d, 0xd1, 0x85, 0xd1, 0x95, 0x4d, 0x09, 0xb0, 0x3a, 0x64, 0x78, 0x2e, 0x65, 0x6e,
        0x74, 0x72, 0x79, 0x50, 0x6f, 0x69, 0x6e, 0x74, 0x73, 0x53, 0x02, 0x34, 0x00, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x55, 0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66,
        0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00, 0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e,
        0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43, 0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b,
        0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e, 0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0,
        0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b, 0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83,
        0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80, 0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76,
        0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c, 0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30,
        0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4, 0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43,
        0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83, 0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4,
        0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xca, 0xc3, 0x0e, 0xec, 0x60, 0x0f, 0xed, 0xe0, 0x06, 0xec, 0xf0, 0x0e, 0xef, 0x00, 0x0f, 0x33, 0x22, 0x88, 0x1c, 0xf0,
        0xc1, 0x0d, 0xc8, 0x41, 0x1c, 0xce, 0xc1, 0x0d, 0xec, 0x21, 0x1c, 0xe4, 0x81, 0x1d, 0xc2, 0x21, 0x1f, 0xde, 0xa1, 0x1e, 0xe8, 0x61, 0x06, 0x13, 0x91, 0x03, 0x3e, 0xb8, 0x81, 0x38, 0xc8, 0x43,
        0x39, 0x84, 0xc3, 0x3a, 0xb8, 0x81, 0x38, 0xc8, 0x03, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x00, 0x56, 0xb0, 0x0d, 0x97, 0xef, 0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44,
        0xa5, 0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x01, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x4e, 0x45, 0x44, 0x80, 0x41, 0xf8, 0xc8, 0x6d, 0x9b, 0x81, 0x34, 0x5c, 0xbe, 0xf3,
        0xf8, 0x42, 0x44, 0x00, 0x13, 0x11, 0x02, 0xcd, 0xb0, 0x10, 0x26, 0x30, 0x0d, 0x97, 0xef, 0x3c, 0xfe, 0xe2, 0x00, 0x83, 0xd8, 0x3c, 0xd4, 0xe4, 0x17, 0xb7, 0x6d, 0x03, 0xd5, 0x70, 0xf9, 0xce,
        0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0x51, 0x3a, 0xc0, 0xe0, 0x17, 0xb7, 0x6d, 0x04, 0xd6, 0x70, 0xf9, 0xce, 0xe3, 0x4f, 0xc4, 0x35, 0x51, 0x11, 0xc1, 0x4e, 0x4e, 0x44, 0xf8, 0xc5, 0x6d, 0x1b,
        0x40, 0xf3, 0x48, 0x3f, 0x03, 0x20, 0xcd, 0xff, 0x20, 0x17, 0xb1, 0x00, 0x61, 0x20, 0x00, 0x00, 0x6d, 0x00, 0x00, 0x00, 0x13, 0x04, 0x49, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00,
        0x34, 0xcc, 0x00, 0x14, 0x61, 0x40, 0xd9, 0x95, 0x61, 0x40, 0x21, 0x14, 0x41, 0xc9, 0x95, 0x6d, 0x40, 0x41, 0xd0, 0x51, 0x1e, 0x25, 0x50, 0x04, 0x00, 0x00, 0x00, 0x00, 0x23, 0x06, 0x09, 0x00,
        0x82, 0x60, 0x50, 0x79, 0x86, 0xc0, 0x71, 0xd2, 0x88, 0x41, 0x02, 0x80, 0x20, 0x18, 0x54, 0xdf, 0x21, 0x74, 0xdd, 0x34, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x08, 0x19, 0x28, 0x9e, 0x57, 0x39,
        0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x94, 0xc1, 0xf2, 0x7d, 0xc5, 0x33, 0x62, 0x50, 0x00, 0x20, 0x08, 0x06, 0xc4, 0x19, 0x24, 0x37, 0x0c, 0x73, 0xc3, 0x30, 0x35, 0x90, 0x81, 0x0e, 0x37,
        0x04, 0x64, 0x00, 0x06, 0xb3, 0x0c, 0x42, 0x10, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x41, 0xa3, 0x06, 0x54, 0x52, 0x11, 0x43, 0x19, 0x54, 0x55, 0x35, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x07, 0x00,
        0x82, 0x60, 0x20, 0xa9, 0x41, 0xa5, 0xa0, 0xc1, 0x68, 0x42, 0x00, 0x58, 0x30, 0xc8, 0x67, 0x34, 0x61, 0x08, 0x4c, 0x08, 0xe0, 0x33, 0x62, 0xe0, 0x00, 0x20, 0x08, 0x06, 0x0f, 0x1c, 0x64, 0xd0,
        0xa2, 0xb0, 0x41, 0x10, 0x04, 0xc1, 0x34, 0x4b, 0x20, 0x14, 0x73, 0xe9, 0x70, 0x43, 0xd0, 0x06, 0x60, 0x30, 0xcb, 0x40, 0x0c, 0xc1, 0x88, 0x41, 0x03, 0x80, 0x20, 0x18, 0x34, 0x73, 0xd0, 0x49,
        0x5e, 0xc3, 0xbc, 0x81, 0xe7, 0x79, 0xa3, 0x09, 0x01, 0x30, 0x62, 0x70, 0x00, 0x20, 0x08, 0x06, 0xd2, 0x1c, 0x78, 0x53, 0x1c, 0x8c, 0x26, 0x04, 0x80, 0x05, 0x83, 0x7c, 0x46, 0x13, 0x86, 0xc0,
        0x84, 0x00, 0x3e, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xe4, 0x81, 0x18, 0x64, 0xd4, 0x64, 0x07, 0x41, 0x10, 0x04, 0xdc, 0x2c, 0x01, 0x51, 0x55, 0x18, 0xe8, 0x70, 0x43, 0x60, 0x07, 0x60,
        0x30, 0xcb, 0x60, 0x14, 0xc1, 0x88, 0x41, 0x03, 0x80, 0x20, 0x18, 0x34, 0x7c, 0x60, 0x06, 0xdb, 0x19, 0x58, 0x95, 0x18, 0x9c, 0xc1, 0x19, 0x9c, 0xc1, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x1c, 0x00,
        0x08, 0x82, 0x81, 0xc4, 0x07, 0x67, 0xc0, 0xe9, 0xc1, 0x68, 0x42, 0x00, 0x58, 0x30, 0xc8, 0x67, 0x34, 0x61, 0x08, 0x4c, 0x08, 0xe0, 0x33, 0x62, 0xe0, 0x00, 0x20, 0x08, 0x06, 0x8f, 0x28, 0xac,
        0x81, 0x18, 0x74, 0x5c, 0x1a, 0x04, 0x41, 0x10, 0x94, 0xc1, 0x2c, 0x81, 0x51, 0x9e, 0x19, 0xe8, 0x70, 0x43, 0xf0, 0x07, 0x60, 0x30, 0xcb, 0x80, 0x1c, 0xc1, 0x88, 0x41, 0x03, 0x80, 0x20, 0x18,
        0x34, 0xa5, 0xf0, 0x06, 0x64, 0x00, 0x07, 0x9f, 0xb7, 0x07, 0x70, 0x00, 0x07, 0x70, 0x30, 0x9a, 0x10, 0x00, 0x23, 0x06, 0x07, 0x00, 0x82, 0x60, 0x20, 0x95, 0x02, 0x1c, 0x94, 0xc1, 0x28, 0x8c,
        0x26, 0x04, 0x80, 0x05, 0x83, 0x7c, 0x46, 0x13, 0x86, 0xc0, 0x84, 0x00, 0x3e, 0x23, 0x06, 0x0e, 0x00, 0x82, 0x60, 0xf0, 0xac, 0x02, 0x1d, 0xac, 0x81, 0x19, 0x94, 0x81, 0x28, 0x04, 0x41, 0x10,
        0xb8, 0xc1, 0x2c, 0x01, 0x82, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const DWORD ps_code_sample_dxbc[] =
    {
        0x43425844, 0x1f7ea829, 0xe3bb1989, 0xaf75ec02, 0x4c88b50d, 0x00000001, 0x000001a0, 0x00000004,
        0x00000030, 0x0000008c, 0x0000009c, 0x00000190, 0x4e475349, 0x00000054, 0x00000002, 0x00000008,
        0x00000038, 0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000030f, 0x00000044, 0x00000000,
        0x0000000a, 0x00000001, 0x00000001, 0x00000101, 0x505f5653, 0x7469736f, 0x006e6f69, 0x535f5653,
        0x6c706d61, 0x646e4965, 0xab007865, 0x4e47534f, 0x00000008, 0x00000000, 0x00000008, 0x58454853,
        0x000000ec, 0x00000050, 0x0000003b, 0x0100086a, 0x04000059, 0x00208e46, 0x00000000, 0x00000002,
        0x0402409c, 0x0011e000, 0x00000000, 0x00005555, 0x04002064, 0x00101032, 0x00000000, 0x00000001,
        0x04000863, 0x00101012, 0x00000001, 0x0000000a, 0x02000068, 0x00000002, 0x0500001c, 0x00100032,
        0x00000000, 0x00101046, 0x00000000, 0x05000036, 0x001000c2, 0x00000000, 0x00101006, 0x00000001,
        0x890000a3, 0x80000202, 0x00155543, 0x00100012, 0x00000001, 0x00100f46, 0x00000000, 0x0011ee46,
        0x00000000, 0x0b000032, 0x00100012, 0x00000001, 0x0010000a, 0x00000001, 0x0020800a, 0x00000000,
        0x00000001, 0x0020801a, 0x00000000, 0x00000001, 0x070000a4, 0x0011e0f2, 0x00000000, 0x00100e46,
        0x00000000, 0x00100006, 0x00000001, 0x0100003e, 0x30494653, 0x00000008, 0x00001000, 0x00000000,
    };

    static const BYTE ps_code_sample_dxil[] =
    {
        0x44, 0x58, 0x42, 0x43, 0x96, 0xc6, 0xd5, 0x04, 0x8a, 0xd2, 0x30, 0x71, 0xa4, 0xce, 0x89, 0x91, 0x45, 0x3e, 0xe7, 0x5e, 0x01, 0x00, 0x00, 0x00, 0x67, 0x08, 0x00, 0x00, 0x06, 0x00, 0x00, 0x00,
        0x38, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00, 0xb3, 0x00, 0x00, 0x00, 0xc3, 0x00, 0x00, 0x00, 0x6b, 0x01, 0x00, 0x00, 0x87, 0x01, 0x00, 0x00, 0x53, 0x46, 0x49, 0x30, 0x08, 0x00, 0x00, 0x00,
        0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x49, 0x53, 0x47, 0x31, 0x63, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x48, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x03, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x54, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74,
        0x69, 0x6f, 0x6e, 0x00, 0x53, 0x56, 0x5f, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65, 0x49, 0x6e, 0x64, 0x65, 0x78, 0x00, 0x4f, 0x53, 0x47, 0x31, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,
        0x00, 0x00, 0x00, 0x50, 0x53, 0x56, 0x30, 0xa0, 0x00, 0x00, 0x00, 0x30, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02,
        0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0d, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x44, 0x03, 0x03, 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0x0c, 0x01, 0x01, 0x00, 0x00, 0x48, 0x41, 0x53, 0x48, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xac, 0x8d, 0x94, 0xa7, 0x52, 0x33, 0x91, 0xc8, 0x83,
        0x92, 0xc6, 0x37, 0xe2, 0xfd, 0x51, 0x80, 0x44, 0x58, 0x49, 0x4c, 0xd8, 0x06, 0x00, 0x00, 0x60, 0x00, 0x00, 0x00, 0xb6, 0x01, 0x00, 0x00, 0x44, 0x58, 0x49, 0x4c, 0x00, 0x01, 0x00, 0x00, 0x10,
        0x00, 0x00, 0x00, 0xc0, 0x06, 0x00, 0x00, 0x42, 0x43, 0xc0, 0xde, 0x21, 0x0c, 0x00, 0x00, 0xad, 0x01, 0x00, 0x00, 0x0b, 0x82, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x13, 0x00, 0x00, 0x00, 0x07,
        0x81, 0x23, 0x91, 0x41, 0xc8, 0x04, 0x49, 0x06, 0x10, 0x32, 0x39, 0x92, 0x01, 0x84, 0x0c, 0x25, 0x05, 0x08, 0x19, 0x1e, 0x04, 0x8b, 0x62, 0x80, 0x14, 0x45, 0x02, 0x42, 0x92, 0x0b, 0x42, 0xa4,
        0x10, 0x32, 0x14, 0x38, 0x08, 0x18, 0x4b, 0x0a, 0x32, 0x52, 0x88, 0x48, 0x90, 0x14, 0x20, 0x43, 0x46, 0x88, 0xa5, 0x00, 0x19, 0x32, 0x42, 0xe4, 0x48, 0x0e, 0x90, 0x91, 0x22, 0xc4, 0x50, 0x41,
        0x51, 0x81, 0x8c, 0xe1, 0x83, 0xe5, 0x8a, 0x04, 0x29, 0x46, 0x06, 0x51, 0x18, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00, 0x1b, 0x8c, 0xe0, 0xff, 0xff, 0xff, 0xff, 0x07, 0x40, 0x02, 0xa8, 0x0d, 0x86,
        0xf0, 0xff, 0xff, 0xff, 0xff, 0x03, 0x20, 0x01, 0xd5, 0x06, 0x62, 0xf8, 0xff, 0xff, 0xff, 0xff, 0x01, 0x90, 0x00, 0x49, 0x18, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x13, 0x82, 0x60, 0x42, 0x20,
        0x4c, 0x08, 0x06, 0x00, 0x00, 0x00, 0x00, 0x89, 0x20, 0x00, 0x00, 0x3c, 0x00, 0x00, 0x00, 0x32, 0x22, 0x48, 0x09, 0x20, 0x64, 0x85, 0x04, 0x93, 0x22, 0xa4, 0x84, 0x04, 0x93, 0x22, 0xe3, 0x84,
        0xa1, 0x90, 0x14, 0x12, 0x4c, 0x8a, 0x8c, 0x0b, 0x84, 0xa4, 0x4c, 0x10, 0x7c, 0x23, 0x00, 0x25, 0x00, 0x14, 0xe6, 0x08, 0xc0, 0xa0, 0x08, 0x63, 0x10, 0x99, 0x01, 0x98, 0x23, 0x40, 0x8a, 0x61,
        0xc6, 0x18, 0x67, 0x10, 0xba, 0x69, 0xb8, 0xfc, 0x09, 0x7b, 0x08, 0xc9, 0x5f, 0x09, 0x69, 0x25, 0x26, 0xbf, 0xb8, 0x6d, 0x54, 0x18, 0x63, 0xcc, 0xa0, 0x73, 0xcf, 0x70, 0xf9, 0x13, 0xf6, 0x10,
        0x92, 0x1f, 0x02, 0xcd, 0xb0, 0x10, 0x28, 0x58, 0x45, 0x51, 0x83, 0x8d, 0x31, 0xc6, 0x18, 0x83, 0x5a, 0x59, 0xc0, 0x60, 0x63, 0x0c, 0x63, 0x8c, 0x39, 0xf4, 0x8e, 0x1a, 0x2e, 0x7f, 0xc2, 0x1e,
        0x42, 0xf2, 0xb9, 0x8d, 0x2a, 0x56, 0x62, 0xf2, 0x8b, 0xdb, 0x46, 0x84, 0x31, 0xc6, 0x14, 0x22, 0x0e, 0x36, 0x48, 0xce, 0x11, 0x04, 0xc5, 0x60, 0xe3, 0x8c, 0x41, 0xa9, 0x0e, 0x04, 0xcc, 0xc4,
        0x06, 0xe3, 0xc0, 0x0e, 0xe1, 0x30, 0x0f, 0xf3, 0xe0, 0x06, 0xb2, 0x10, 0x0e, 0xf3, 0x40, 0x0f, 0xe5, 0x20, 0x0f, 0xe9, 0xa0, 0x0f, 0xe5, 0x20, 0x0f, 0xaf, 0x20, 0x0f, 0xe4, 0x50, 0x0e, 0xf2,
        0x50, 0x0e, 0xe4, 0x40, 0x0b, 0xe5, 0x80, 0x0f, 0xf4, 0x50, 0x0f, 0xf2, 0x50, 0x0e, 0x72, 0x40, 0x0a, 0xa1, 0x20, 0x0f, 0xf2, 0x10, 0x0e, 0xf9, 0xc0, 0x07, 0xe6, 0xc0, 0x0e, 0xef, 0x10, 0x0e,
        0xf4, 0xe0, 0x07, 0x28, 0x30, 0x84, 0x87, 0x11, 0x04, 0xe3, 0x8c, 0x14, 0x88, 0x00, 0x46, 0x42, 0x85, 0xd6, 0xc6, 0x68, 0xda, 0x29, 0x30, 0xe6, 0x08, 0x40, 0x01, 0x00, 0x00, 0x00, 0x00, 0x13,
        0x14, 0x72, 0xc0, 0x87, 0x74, 0x60, 0x87, 0x36, 0x68, 0x87, 0x79, 0x68, 0x03, 0x72, 0xc0, 0x87, 0x0d, 0xaf, 0x50, 0x0e, 0x6d, 0xd0, 0x0e, 0x7a, 0x50, 0x0e, 0x6d, 0x00, 0x0f, 0x7a, 0x30, 0x07,
        0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x78, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x71, 0x60, 0x07, 0x7a, 0x30,
        0x07, 0x72, 0xd0, 0x06, 0xe9, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x73, 0x20, 0x07, 0x6d, 0x90, 0x0e, 0x76, 0x40, 0x07, 0x7a, 0x60, 0x07, 0x74, 0xd0, 0x06, 0xe6, 0x10, 0x07, 0x76, 0xa0, 0x07, 0x73,
        0x20, 0x07, 0x6d, 0x60, 0x0e, 0x73, 0x20, 0x07, 0x7a, 0x30, 0x07, 0x72, 0xd0, 0x06, 0xe6, 0x60, 0x07, 0x74, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x6d, 0xe0, 0x0e, 0x78, 0xa0, 0x07, 0x71, 0x60, 0x07,
        0x7a, 0x30, 0x07, 0x72, 0xa0, 0x07, 0x76, 0x40, 0x07, 0x43, 0x9e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x86, 0x3c, 0x04, 0x10, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x0c, 0x79, 0x10, 0x20, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0xf2, 0x34, 0x40, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xe4, 0x79,
        0x80, 0x00, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0xc8, 0x23, 0x01, 0x01, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x90, 0xa7, 0x02, 0x02, 0x40, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x00, 0x00, 0x80, 0x2c, 0x10, 0x10, 0x00, 0x00, 0x00, 0x32, 0x1e, 0x98, 0x14, 0x19, 0x11, 0x4c, 0x90, 0x8c, 0x09, 0x26, 0x47, 0xc6, 0x04, 0x43, 0x1a, 0x25, 0x30, 0x02, 0x50, 0x0c,
        0xe5, 0x50, 0x12, 0x05, 0x54, 0x06, 0x35, 0x40, 0xa7, 0x24, 0xca, 0xa0, 0x10, 0x46, 0x00, 0x4a, 0xa1, 0x30, 0x4a, 0xa0, 0x06, 0x88, 0x8e, 0x00, 0xd4, 0x00, 0xf9, 0x02, 0x04, 0x04, 0x21, 0x3e,
        0x03, 0x40, 0x7d, 0x2c, 0x81, 0x00, 0x28, 0xcf, 0x00, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x62, 0x00, 0x00, 0x00, 0x1a, 0x03, 0x4c, 0x90, 0x46, 0x02, 0x13, 0x44, 0x35, 0x18, 0x63, 0x0b, 0x73,
        0x3b, 0x03, 0xb1, 0x2b, 0x93, 0x9b, 0x4b, 0x7b, 0x73, 0x03, 0x99, 0x71, 0xb9, 0x01, 0x41, 0xa1, 0x0b, 0x3b, 0x9b, 0x7b, 0x91, 0x2a, 0x62, 0x2a, 0x0a, 0x9a, 0x2a, 0xfa, 0x9a, 0xb9, 0x81, 0x79,
        0x31, 0x4b, 0x73, 0x0b, 0x63, 0x4b, 0xd9, 0x10, 0x04, 0x13, 0x84, 0xe1, 0x98, 0x20, 0x0c, 0xc8, 0x06, 0x61, 0x20, 0x26, 0x08, 0x43, 0xb2, 0x41, 0x18, 0x0c, 0x0a, 0x70, 0x73, 0x1b, 0x06, 0xc4,
        0x20, 0x26, 0x08, 0x19, 0x47, 0x60, 0x82, 0x30, 0x28, 0x13, 0x04, 0xea, 0x9a, 0x20, 0x50, 0xd8, 0x04, 0x61, 0x58, 0x36, 0x08, 0xc4, 0xb3, 0x61, 0x21, 0x94, 0x85, 0x20, 0x06, 0xa6, 0x69, 0x1c,
        0x68, 0x43, 0x10, 0x4d, 0x10, 0x38, 0x6d, 0x82, 0x30, 0x30, 0x1b, 0x10, 0x62, 0x5a, 0x08, 0x62, 0xa0, 0x80, 0x0d, 0x41, 0xb5, 0x81, 0x00, 0x24, 0x0b, 0x98, 0x20, 0x74, 0xdb, 0x86, 0x00, 0x9b,
        0x20, 0x08, 0x00, 0x1d, 0xb8, 0xb9, 0xaf, 0xb6, 0xb0, 0x34, 0xb7, 0xaf, 0xb9, 0xb0, 0x36, 0x38, 0xb6, 0x32, 0x2e, 0x53, 0x56, 0x5f, 0x50, 0x6f, 0x73, 0x69, 0x74, 0x69, 0x6f, 0x6e, 0x13, 0x84,
        0xe3, 0x99, 0x20, 0x1c, 0xd0, 0x86, 0x80, 0x98, 0x20, 0x1c, 0xd1, 0x04, 0xe1, 0x90, 0x26, 0x08, 0x43, 0xb3, 0x41, 0x10, 0x03, 0x31, 0xd8, 0xb0, 0x10, 0x5c, 0xe7, 0x7d, 0x60, 0x30, 0x80, 0x01,
        0x11, 0x06, 0x63, 0x40, 0x67, 0xca, 0xea, 0x6b, 0x2a, 0xac, 0x0d, 0x8e, 0xad, 0x2c, 0xc9, 0x8d, 0xac, 0x0c, 0x6f, 0x82, 0x70, 0x4c, 0x13, 0x84, 0x83, 0x9a, 0x20, 0x1c, 0xd5, 0x04, 0x61, 0x70,
        0x26, 0x08, 0x87, 0xb5, 0x61, 0x19, 0xca, 0xc0, 0x0c, 0xce, 0xe0, 0x43, 0x83, 0x01, 0x0d, 0xd2, 0x40, 0x0d, 0x80, 0x0d, 0x02, 0x19, 0xac, 0xc1, 0x86, 0x81, 0x0d, 0x00, 0x60, 0x82, 0xe0, 0x65,
        0x1b, 0x04, 0xc2, 0x0d, 0x36, 0x14, 0xda, 0xd6, 0x06, 0xd7, 0x1b, 0x54, 0x61, 0x63, 0xb3, 0x6b, 0x73, 0x49, 0x23, 0x2b, 0x73, 0xa3, 0x9b, 0x12, 0x04, 0x55, 0xc8, 0xf0, 0x5c, 0xec, 0xca, 0xe4,
        0xe6, 0xd2, 0xde, 0xdc, 0xa6, 0x04, 0x44, 0x13, 0x32, 0x3c, 0x17, 0xbb, 0x30, 0x36, 0xbb, 0x32, 0xb9, 0x29, 0x81, 0x51, 0x87, 0x0c, 0xcf, 0x65, 0x0e, 0x2d, 0x8c, 0xac, 0x4c, 0xae, 0xe9, 0x8d,
        0xac, 0x8c, 0x6d, 0x4a, 0x80, 0x94, 0x21, 0xc3, 0x73, 0x91, 0x2b, 0x9b, 0x7b, 0xab, 0x93, 0x1b, 0x2b, 0x9b, 0x9b, 0x12, 0x58, 0x75, 0xc8, 0xf0, 0x5c, 0xec, 0xd2, 0xca, 0xee, 0x92, 0xc8, 0xa6,
        0xe8, 0xc2, 0xe8, 0xca, 0xa6, 0x04, 0x58, 0x1d, 0x32, 0x3c, 0x97, 0x32, 0x37, 0x3a, 0xb9, 0x3c, 0xa8, 0xb7, 0x34, 0x37, 0xba, 0xb9, 0x29, 0xc1, 0x1b, 0x00, 0x00, 0x79, 0x18, 0x00, 0x00, 0x4c,
        0x00, 0x00, 0x00, 0x33, 0x08, 0x80, 0x1c, 0xc4, 0xe1, 0x1c, 0x66, 0x14, 0x01, 0x3d, 0x88, 0x43, 0x38, 0x84, 0xc3, 0x8c, 0x42, 0x80, 0x07, 0x79, 0x78, 0x07, 0x73, 0x98, 0x71, 0x0c, 0xe6, 0x00,
        0x0f, 0xed, 0x10, 0x0e, 0xf4, 0x80, 0x0e, 0x33, 0x0c, 0x42, 0x1e, 0xc2, 0xc1, 0x1d, 0xce, 0xa1, 0x1c, 0x66, 0x30, 0x05, 0x3d, 0x88, 0x43, 0x38, 0x84, 0x83, 0x1b, 0xcc, 0x03, 0x3d, 0xc8, 0x43,
        0x3d, 0x8c, 0x03, 0x3d, 0xcc, 0x78, 0x8c, 0x74, 0x70, 0x07, 0x7b, 0x08, 0x07, 0x79, 0x48, 0x87, 0x70, 0x70, 0x07, 0x7a, 0x70, 0x03, 0x76, 0x78, 0x87, 0x70, 0x20, 0x87, 0x19, 0xcc, 0x11, 0x0e,
        0xec, 0x90, 0x0e, 0xe1, 0x30, 0x0f, 0x6e, 0x30, 0x0f, 0xe3, 0xf0, 0x0e, 0xf0, 0x50, 0x0e, 0x33, 0x10, 0xc4, 0x1d, 0xde, 0x21, 0x1c, 0xd8, 0x21, 0x1d, 0xc2, 0x61, 0x1e, 0x66, 0x30, 0x89, 0x3b,
        0xbc, 0x83, 0x3b, 0xd0, 0x43, 0x39, 0xb4, 0x03, 0x3c, 0xbc, 0x83, 0x3c, 0x84, 0x03, 0x3b, 0xcc, 0xf0, 0x14, 0x76, 0x60, 0x07, 0x7b, 0x68, 0x07, 0x37, 0x68, 0x87, 0x72, 0x68, 0x07, 0x37, 0x80,
        0x87, 0x70, 0x90, 0x87, 0x70, 0x60, 0x07, 0x76, 0x28, 0x07, 0x76, 0xf8, 0x05, 0x76, 0x78, 0x87, 0x77, 0x80, 0x87, 0x5f, 0x08, 0x87, 0x71, 0x18, 0x87, 0x72, 0x98, 0x87, 0x79, 0x98, 0x81, 0x2c,
        0xee, 0xf0, 0x0e, 0xee, 0xe0, 0x0e, 0xf5, 0xc0, 0x0e, 0xec, 0x30, 0x03, 0x62, 0xc8, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xcc, 0xa1, 0x1c, 0xe4, 0xa1, 0x1c, 0xdc, 0x61, 0x1c, 0xca, 0x21, 0x1c, 0xc4,
        0x81, 0x1d, 0xca, 0x61, 0x06, 0xd6, 0x90, 0x43, 0x39, 0xc8, 0x43, 0x39, 0x98, 0x43, 0x39, 0xc8, 0x43, 0x39, 0xb8, 0xc3, 0x38, 0x94, 0x43, 0x38, 0x88, 0x03, 0x3b, 0x94, 0xc3, 0x2f, 0xbc, 0x83,
        0x3c, 0xfc, 0x82, 0x3b, 0xd4, 0x03, 0x3b, 0xb0, 0xc3, 0x0c, 0xc4, 0x21, 0x07, 0x7c, 0x70, 0x03, 0x7a, 0x28, 0x87, 0x76, 0x80, 0x87, 0x19, 0xd1, 0x43, 0x0e, 0xf8, 0xe0, 0x06, 0xe4, 0x20, 0x0e,
        0xe7, 0xe0, 0x06, 0xf6, 0x10, 0x0e, 0xf2, 0xc0, 0x0e, 0xe1, 0x90, 0x0f, 0xef, 0x50, 0x0f, 0xf4, 0x00, 0x00, 0x00, 0x71, 0x20, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x00, 0x56, 0xb0, 0x0d, 0x97, 0xef,
        0x3c, 0xbe, 0x10, 0x50, 0x45, 0x41, 0x44, 0xa5, 0x03, 0x0c, 0x25, 0x61, 0x00, 0x02, 0xe6, 0x17, 0xb7, 0x6d, 0x06, 0xd2, 0x70, 0xf9, 0xce, 0xe3, 0x0b, 0x11, 0x01, 0x4c, 0x44, 0x08, 0x34, 0xc3,
        0x42, 0x98, 0xc0, 0x34, 0x5c, 0xbe, 0xf3, 0xf8, 0x8b, 0x03, 0x0c, 0x62, 0xf3, 0x50, 0x93, 0x5f, 0xdc, 0xb6, 0x05, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x2f, 0x01, 0xcc, 0xb3, 0x10, 0x62, 0x33, 0x10,
        0x97, 0x8f, 0xdc, 0xb6, 0x0d, 0x54, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd7, 0x44, 0x45, 0x44, 0xe9, 0x00, 0x83, 0x5f, 0xdc, 0xb6, 0x11, 0x58, 0xc3, 0xe5, 0x3b, 0x8f, 0x3f, 0x11, 0xd7, 0x44,
        0x45, 0x04, 0x3b, 0x39, 0x11, 0xe1, 0x17, 0xb7, 0x6d, 0x00, 0xce, 0x23, 0xfd, 0x0c, 0x80, 0x34, 0xbf, 0x04, 0x30, 0xcf, 0x42, 0x00, 0x00, 0x61, 0x20, 0x00, 0x00, 0x29, 0x00, 0x00, 0x00, 0x13,
        0x04, 0x41, 0x2c, 0x10, 0x00, 0x00, 0x00, 0x05, 0x00, 0x00, 0x00, 0x34, 0x66, 0x00, 0x4a, 0xae, 0x10, 0x8a, 0x36, 0xa0, 0x08, 0x03, 0xca, 0xae, 0x0c, 0x03, 0xe8, 0x14, 0x41, 0x79, 0x00, 0x23,
        0x06, 0x09, 0x00, 0x82, 0x60, 0x50, 0x81, 0x01, 0x12, 0x79, 0xde, 0x33, 0x62, 0x90, 0x00, 0x20, 0x08, 0x06, 0x55, 0x18, 0x24, 0xc3, 0xf7, 0x41, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x98,
        0x41, 0x02, 0x06, 0x60, 0x60, 0x2d, 0x23, 0x06, 0x09, 0x00, 0x82, 0x60, 0x80, 0x9c, 0x81, 0x12, 0x06, 0x61, 0x40, 0x31, 0x23, 0x06, 0x05, 0x00, 0x82, 0x60, 0x40, 0xa4, 0x81, 0x72, 0xc3, 0x98,
        0x1b, 0xc6, 0x8c, 0x18, 0x34, 0x00, 0x08, 0x82, 0x41, 0xa3, 0x06, 0xcc, 0x01, 0x09, 0xc1, 0x00, 0x41, 0xd0, 0x68, 0x42, 0x00, 0x8c, 0x18, 0x1c, 0x00, 0x08, 0x82, 0x81, 0xa4, 0x06, 0x0d, 0x82,
        0x06, 0xa3, 0x09, 0x01, 0x60, 0xc1, 0x20, 0x9f, 0xd1, 0x84, 0x21, 0x30, 0x21, 0x80, 0xcf, 0x88, 0x81, 0x03, 0x80, 0x20, 0x18, 0x3c, 0x70, 0x10, 0x39, 0x09, 0xa2, 0x04, 0x41, 0x10, 0x3c, 0x08,
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    };

    static const D3D12_SHADER_BYTECODE ps_code_dxbc[] = {
        { ps_code_single_dxbc, sizeof(ps_code_single_dxbc) },
        { ps_code_pixel_dxbc, sizeof(ps_code_pixel_dxbc) },
        { ps_code_pixel_unrolled_dxbc, sizeof(ps_code_pixel_unrolled_dxbc) },
        { ps_code_sample_dxbc, sizeof(ps_code_sample_dxbc) },
    };

    static const D3D12_SHADER_BYTECODE ps_code_dxil[] = {
        { ps_code_single_dxil, sizeof(ps_code_single_dxil) },
        { ps_code_pixel_dxil, sizeof(ps_code_pixel_dxil) },
        { ps_code_pixel_unrolled_dxil, sizeof(ps_code_pixel_unrolled_dxil) },
        { ps_code_sample_dxil, sizeof(ps_code_sample_dxil) },
    };

    /* RT format must be specified when using MSAA, or validation complains. We'll render to NULL RTV either way. */
    if (use_dxil)
    {
        init_pipeline_state_desc_shaders(&pso_desc, rs, DXGI_FORMAT_R32_FLOAT, NULL,
            vs_code_dxil, sizeof(vs_code_dxil),
            ps_code_dxil[pso_index].pShaderBytecode, ps_code_dxil[pso_index].BytecodeLength);
    }
    else
    {
        init_pipeline_state_desc_shaders(&pso_desc, rs, DXGI_FORMAT_R32_FLOAT, NULL,
            vs_code_dxbc, sizeof(vs_code_dxbc),
            ps_code_dxbc[pso_index].pShaderBytecode, ps_code_dxbc[pso_index].BytecodeLength);
    }

    pso_desc.RasterizerState.CullMode = D3D12_CULL_MODE_NONE;
    pso_desc.SampleDesc.Count = pso_index != PSO_SINGLE_SAMPLE ? 4 : 1;
    pso_desc.DepthStencilState.DepthEnable = FALSE;
    pso_desc.DepthStencilState.StencilEnable = FALSE;

    hr = ID3D12Device_CreateGraphicsPipelineState(device, &pso_desc, &IID_ID3D12PipelineState, (void **)&pso);
    ok(SUCCEEDED(hr), "Failed to create PSO: hr = #%x\n", hr);
    return pso;
}

static void test_rasterizer_ordered_views(bool use_dxil)
{
    D3D12_FEATURE_DATA_D3D12_OPTIONS options;
    D3D12_UNORDERED_ACCESS_VIEW_DESC uav_desc;
    struct resource_readback rb;
    struct test_context_desc desc;
    struct test_context context;
    ID3D12PipelineState *psos[PSO_COUNT];
    ID3D12DescriptorHeap *heap;
    D3D12_VIEWPORT vp;
    D3D12_RECT sci;
    ID3D12Resource *output;
    unsigned int i, j;
    D3D12_ROOT_SIGNATURE_DESC rs_desc;
    D3D12_ROOT_PARAMETER params[2];
    D3D12_DESCRIPTOR_RANGE range;

    static const struct test tests[] =
    {
        { 4, 4, 1, quads_simple, ARRAY_SIZE(quads_simple) },
        { 4, 4, 4, quads_pixel, ARRAY_SIZE(quads_pixel) },
        { 4, 4, 4, quads_pixel_unrolled, ARRAY_SIZE(quads_pixel_unrolled) },
        { 4, 4, 4, quads_sample, ARRAY_SIZE(quads_sample) },
        { 4, 4, 1, quads_simple_instanced, ARRAY_SIZE(quads_simple_instanced) },
        { 4, 4, 4, quads_pixel_instanced, ARRAY_SIZE(quads_pixel_instanced) },
        { 4, 4, 4, quads_pixel_unrolled_instanced, ARRAY_SIZE(quads_pixel_unrolled_instanced) },
        { 4, 4, 4, quads_sample_instanced, ARRAY_SIZE(quads_sample_instanced) },
        { 4, 4, 4, quads_mixed, ARRAY_SIZE(quads_mixed) },
    };

    memset(&desc, 0, sizeof(desc));
    desc.no_pipeline = true;
    desc.no_render_target = true;
    desc.no_root_signature = true;
    if (!init_test_context(&context, &desc))
        return;

    memset(&rs_desc, 0, sizeof(rs_desc));
    memset(params, 0, sizeof(params));
    memset(&range, 0, sizeof(range));

    rs_desc.NumParameters = ARRAY_SIZE(params);
    rs_desc.pParameters = params;
    params[0].ShaderVisibility = D3D12_SHADER_VISIBILITY_PIXEL;
    params[0].ParameterType = D3D12_ROOT_PARAMETER_TYPE_DESCRIPTOR_TABLE;
    params[0].DescriptorTable.NumDescriptorRanges = 1;
    params[0].DescriptorTable.pDescriptorRanges = &range;
    params[1].ShaderVisibility = D3D12_SHADER_VISIBILITY_ALL;
    params[1].ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    params[1].Constants.Num32BitValues = 8;
    range.RangeType = D3D12_DESCRIPTOR_RANGE_TYPE_UAV;
    range.NumDescriptors = 1;

    if (FAILED(ID3D12Device_CheckFeatureSupport(context.device, D3D12_FEATURE_D3D12_OPTIONS, &options, sizeof(options))) ||
        !options.ROVsSupported)
    {
        skip("ROVs are not supported by implementation.\n");
        destroy_test_context(&context);
        return;
    }

    create_root_signature(context.device, &rs_desc, &context.root_signature);

    for (i = 0; i < PSO_COUNT; i++)
        psos[i] = create_pso(context.device, context.root_signature, i, use_dxil);

    heap = create_gpu_descriptor_heap(context.device, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, 1);

    for (i = 0; i < ARRAY_SIZE(tests); i++)
    {
        vkd3d_test_set_context("Test %u", i);
        output = create_default_texture2d(context.device, tests[i].width, tests[i].height, tests[i].samples, 1,
            DXGI_FORMAT_R32_FLOAT, D3D12_RESOURCE_FLAG_ALLOW_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_UNORDERED_ACCESS);

        memset(&uav_desc, 0, sizeof(uav_desc));
        uav_desc.Format = DXGI_FORMAT_R32_FLOAT;
        uav_desc.ViewDimension = tests[i].samples > 1 ? D3D12_UAV_DIMENSION_TEXTURE2DARRAY : D3D12_UAV_DIMENSION_TEXTURE2D;
        if (tests[i].samples > 1)
            uav_desc.Texture2DArray.ArraySize = tests[i].samples;

        ID3D12Device_CreateUnorderedAccessView(context.device, output, NULL, &uav_desc, ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap));

        ID3D12GraphicsCommandList_SetDescriptorHeaps(context.list, 1, &heap);
        ID3D12GraphicsCommandList_SetGraphicsRootSignature(context.list, context.root_signature);

        set_viewport(&vp, 0, 0, tests[i].width, tests[i].height, 0.0f, 1.0f);
        set_rect(&sci, 0, 0, tests[i].width, tests[i].height);
        ID3D12GraphicsCommandList_RSSetViewports(context.list, 1, &vp);
        ID3D12GraphicsCommandList_RSSetScissorRects(context.list, 1, &sci);
        ID3D12GraphicsCommandList_IASetPrimitiveTopology(context.list, D3D_PRIMITIVE_TOPOLOGY_TRIANGLESTRIP);

        for (j = 0; j < tests[i].num_quads; j++)
        {
            const struct quad *quad = &tests[i].quads[j];
            const float constants[8] = {
                quad->top_left[0], quad->top_left[1],
                quad->bottom_right[0], quad->bottom_right[1],
                quad->mul, quad->add,
                1.0f / tests[i].width, 1.0f / tests[i].height,
            };
            ID3D12GraphicsCommandList_SetGraphicsRootDescriptorTable(context.list, 0, ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap));
            ID3D12GraphicsCommandList_SetGraphicsRoot32BitConstants(context.list, 1, ARRAY_SIZE(constants), constants, 0);
            ID3D12GraphicsCommandList_SetPipelineState(context.list, psos[quad->pso_index]);
            ID3D12GraphicsCommandList_DrawInstanced(context.list, 4, quad->instances, 0, 0);
        }

        transition_resource_state(context.list, output, D3D12_RESOURCE_STATE_UNORDERED_ACCESS, D3D12_RESOURCE_STATE_COPY_SOURCE);

        for (j = 0; j < tests[i].samples; j++)
        {
            get_texture_readback_with_command_list(output, j, &rb, context.queue, context.list);
            validate_readback(&rb, &tests[i], j);
            release_resource_readback(&rb);
            reset_command_list(context.list, context.allocator);
        }

        ID3D12Resource_Release(output);
    }
    vkd3d_test_set_context(NULL);

    ID3D12DescriptorHeap_Release(heap);
    for (i = 0; i < ARRAY_SIZE(psos); i++)
        ID3D12PipelineState_Release(psos[i]);
    destroy_test_context(&context);
}

void test_rasterizer_ordered_views_dxbc(void)
{
    test_rasterizer_ordered_views(false);
}

void test_rasterizer_ordered_views_dxil(void)
{
    test_rasterizer_ordered_views(true);
}