/*  $Id: wrong_qual.cpp 380396 2012-11-13 15:47:57Z kornbluh $
* ===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
* Author:  Colleen Bollin, NCBI
*
* File Description:
*   Unit tests for the validator -- specifically, of wrong qualifiers
*   on imported features.
*
* ===========================================================================
*/

#include <ncbi_pch.hpp>

#include "unit_test_validator.hpp"

#include <corelib/test_boost.hpp>

#include <objects/general/Object_id.hpp>
#include <objects/seq/Seq_data.hpp>
#include <objects/seqfeat/Gb_qual.hpp>
#include <objects/seqfeat/Imp_feat.hpp>
#include <objmgr/scope.hpp>
#include <objtools/data_loaders/genbank/gbloader.hpp>
#include <objtools/validator/validator.hpp>

USING_NCBI_SCOPE;
USING_SCOPE(objects);

using namespace validator;

static bool s_IsIllegal (string key) 
{
    if (NStr::Equal(key, "allele") || NStr::Equal(key, "virion") 
        || NStr::Equal(key, "mutation")
        || NStr::Equal(key, "Import")) {
        return true;
    } else {
        return false;
    }
}


static void s_TestOneWrongQual(const string qual, const string val, const string feat, 
                          CSeq_entry_Handle seh, CRef<CSeq_feat> misc_feat,
                          CValidator& validator, unsigned int options, CScope *scope) 
{
    misc_feat->AddQualifier (qual, val);;
    string msg = "Wrong qualifier ";
    msg.append(qual);
    msg.append(" for feature ");
    msg.append(feat);
    vector< CExpectedError *> expected_errors;
    bool is_illegal = s_IsIllegal (feat);
    if (is_illegal) {
        expected_errors.push_back(new CExpectedError("good", eDiag_Error, "UnknownImpFeatKey",
                                  "Feature key " + feat + " is no longer legal"));
    }
    expected_errors.push_back(new CExpectedError("good", eDiag_Warning, "WrongQualOnImpFeat", msg));
    if (NStr::Equal(qual, "rpt_unit_range") && NStr::Equal(feat, "polyA_site")) {
        expected_errors.push_back(new CExpectedError("good", eDiag_Warning, "RptUnitRangeProblem", 
                                  "/rpt_unit_range is not within sequence length"));
    }
    if (NStr::Equal(qual, "old_locus_tag")) {
        expected_errors.push_back(new CExpectedError("good", eDiag_Error, "LocusTagProblem", 
                                  "old_locus_tag without inherited locus_tag"));
    }
    CConstRef<CValidError> eval = validator.Validate(*misc_feat, scope, options);
    CheckErrors (*eval, expected_errors);
    misc_feat->SetQual().pop_back();
    CLEAR_ERRORS
}


typedef struct keyqualval {
  const char* key;
  const char* qual;
  const char* val;
} TKeyQualVal;

static const TKeyQualVal s_KeyQualVal[] = {
    // begin automatically generated section
    { "allele", "compare", "AY123456.1" },
    { "allele", "mitochondrion", "foo" },
    { "allele", "mobile_element", "integron" },
    { "allele", "metagenomic", "foo" },
    { "allele", "kinetoplast", "foo" },
    { "allele", "chromoplast", "foo" },
    { "allele", "specific_host", "foo" },
    { "allele", "sub_strain", "foo" },
    { "allele", "tag_peptide", "foo" },
    { "allele", "isolation_source", "foo" },
    { "allele", "collected_by", "foo" },
    { "allele", "rpt_family", "foo" },
    { "allele", "rpt_type", "flanking" },
    { "allele", "insertion_seq", "foo" },
    { "allele", "transl_table", "foo" },
    { "allele", "rearranged", "foo" },
    { "allele", "mod_base", "foo" },
    { "allele", "rpt_unit", "foo" },
    { "allele", "anticodon", "foo" },
    { "allele", "function", "foo" },
    { "allele", "number", "foo" },
    { "allele", "identified_by", "foo" },
    { "allele", "collection_date", "foo" },
    { "allele", "direction", "foo" },
    { "allele", "rpt_unit_range", "1..3" },
    { "allele", "serotype", "foo" },
    { "allele", "satellite", "foo" },
    { "allele", "organism", "foo" },
    { "allele", "transcript_id", "foo" },
    { "allele", "serovar", "foo" },
    { "allele", "variety", "foo" },
    { "allele", "country", "foo" },
    { "allele", "rpt_unit_seq", "foo" },
    { "allele", "lab_host", "foo" },
    { "allele", "macronuclear", "foo" },
    { "allele", "cyanelle", "foo" },
    { "allele", "bio_material", "foo" },
    { "allele", "chloroplast", "foo" },
    { "allele", "plasmid", "foo" },
    { "allele", "mating_type", "foo" },
    { "allele", "cell_type", "foo" },
    { "allele", "EC_number", "1.2.3.4" },
    { "allele", "mol_type", "foo" },
    { "allele", "operon", "foo" },
    { "allele", "cultivar", "foo" },
    { "allele", "artificial_location", "foo" },
    { "allele", "segment", "foo" },
    { "allele", "cons_splice", "foo" },
    { "allele", "environmental_sample", "foo" },
    { "allele", "PCR_conditions", "foo" },
    { "allele", "transposon", "foo" },
    { "allele", "haplogroup", "foo" },
    { "allele", "ribosomal_slippage", "foo" },
    { "allele", "codon_start", "foo" },
    { "allele", "clone", "foo" },
    { "allele", "gdb_xref", "foo" },
    { "allele", "translation", "foo" },
    { "allele", "transl_except", "foo" },
    { "allele", "bound_moiety", "foo" },
    { "allele", "sub_clone", "foo" },
    { "allele", "cell_line", "foo" },
    { "allele", "transgenic", "foo" },
    { "allele", "germline", "foo" },
    { "allele", "protein_id", "foo" },
    { "allele", "codon", "foo" },
    { "allele", "clone_lib", "foo" },
    { "allele", "PCR_primers", "foo" },
    { "allele", "sequenced_mol", "foo" },
    { "allele", "strain", "foo" },
    { "allele", "proviral", "foo" },
    { "allele", "lat_lon", "foo" },
    { "allele", "culture_collection", "foo" },
    { "allele", "haplotype", "foo" },
    { "allele", "estimated_length", "foo" },
    { "allele", "tissue_lib", "foo" },
    { "allele", "focus", "foo" },
    { "allele", "dev_stage", "foo" },
    { "allele", "specimen_voucher", "foo" },
    { "allele", "pseudo", "foo" },
    { "allele", "isolate", "foo" },
    { "allele", "chromosome", "foo" },
    { "allele", "allele", "foo" },
    { "allele", "ncRNA_class", "foo" },
    { "allele", "pop_variant", "foo" },
    { "allele", "tissue_type", "foo" },
    { "allele", "trans_splicing", "foo" },
    { "allele", "organelle", "foo" },
    { "allele", "sex", "foo" },
    { "allele", "virion", "foo" },
    { "allele", "sub_species", "foo" },
    { "allele", "ecotype", "foo" },
    { "attenuator", "compare", "AY123456.1" },
    { "attenuator", "mitochondrion", "foo" },
    { "attenuator", "mobile_element", "integron" },
    { "attenuator", "metagenomic", "foo" },
    { "attenuator", "standard_name", "foo" },
    { "attenuator", "kinetoplast", "foo" },
    { "attenuator", "chromoplast", "foo" },
    { "attenuator", "specific_host", "foo" },
    { "attenuator", "sub_strain", "foo" },
    { "attenuator", "tag_peptide", "foo" },
    { "attenuator", "isolation_source", "foo" },
    { "attenuator", "collected_by", "foo" },
    { "attenuator", "rpt_family", "foo" },
    { "attenuator", "rpt_type", "flanking" },
    { "attenuator", "insertion_seq", "foo" },
    { "attenuator", "transl_table", "foo" },
    { "attenuator", "rearranged", "foo" },
    { "attenuator", "mod_base", "foo" },
    { "attenuator", "rpt_unit", "foo" },
    { "attenuator", "anticodon", "foo" },
    { "attenuator", "function", "foo" },
    { "attenuator", "number", "foo" },
    { "attenuator", "identified_by", "foo" },
    { "attenuator", "collection_date", "foo" },
    { "attenuator", "direction", "foo" },
    { "attenuator", "rpt_unit_range", "1..3" },
    { "attenuator", "serotype", "foo" },
    { "attenuator", "satellite", "foo" },
    { "attenuator", "organism", "foo" },
    { "attenuator", "transcript_id", "foo" },
    { "attenuator", "serovar", "foo" },
    { "attenuator", "variety", "foo" },
    { "attenuator", "country", "foo" },
    { "attenuator", "rpt_unit_seq", "foo" },
    { "attenuator", "lab_host", "foo" },
    { "attenuator", "macronuclear", "foo" },
    { "attenuator", "cyanelle", "foo" },
    { "attenuator", "bio_material", "foo" },
    { "attenuator", "chloroplast", "foo" },
    { "attenuator", "plasmid", "foo" },
    { "attenuator", "mating_type", "foo" },
    { "attenuator", "cell_type", "foo" },
    { "attenuator", "EC_number", "1.2.3.4" },
    { "attenuator", "mol_type", "foo" },
    { "attenuator", "cultivar", "foo" },
    { "attenuator", "artificial_location", "foo" },
    { "attenuator", "segment", "foo" },
    { "attenuator", "cons_splice", "foo" },
    { "attenuator", "environmental_sample", "foo" },
    { "attenuator", "PCR_conditions", "foo" },
    { "attenuator", "frequency", "foo" },
    { "attenuator", "transposon", "foo" },
    { "attenuator", "haplogroup", "foo" },
    { "attenuator", "ribosomal_slippage", "foo" },
    { "attenuator", "codon_start", "foo" },
    { "attenuator", "clone", "foo" },
    { "attenuator", "gdb_xref", "foo" },
    { "attenuator", "translation", "foo" },
    { "attenuator", "transl_except", "foo" },
    { "attenuator", "bound_moiety", "foo" },
    { "attenuator", "sub_clone", "foo" },
    { "attenuator", "cell_line", "foo" },
    { "attenuator", "transgenic", "foo" },
    { "attenuator", "germline", "foo" },
    { "attenuator", "protein_id", "foo" },
    { "attenuator", "codon", "foo" },
    { "attenuator", "clone_lib", "foo" },
    { "attenuator", "PCR_primers", "foo" },
    { "attenuator", "sequenced_mol", "foo" },
    { "attenuator", "replace", "aaaaattttt" },
    { "attenuator", "strain", "foo" },
    { "attenuator", "proviral", "foo" },
    { "attenuator", "lat_lon", "foo" },
    { "attenuator", "culture_collection", "foo" },
    { "attenuator", "haplotype", "foo" },
    { "attenuator", "estimated_length", "foo" },
    { "attenuator", "tissue_lib", "foo" },
    { "attenuator", "focus", "foo" },
    { "attenuator", "dev_stage", "foo" },
    { "attenuator", "specimen_voucher", "foo" },
    { "attenuator", "pseudo", "foo" },
    { "attenuator", "isolate", "foo" },
    { "attenuator", "chromosome", "foo" },
    { "attenuator", "ncRNA_class", "foo" },
    { "attenuator", "pop_variant", "foo" },
    { "attenuator", "tissue_type", "foo" },
    { "attenuator", "trans_splicing", "foo" },
    { "attenuator", "organelle", "foo" },
    { "attenuator", "sex", "foo" },
    { "attenuator", "virion", "foo" },
    { "attenuator", "sub_species", "foo" },
    { "attenuator", "ecotype", "foo" },
    { "attenuator", "product", "foo" },
    { "C_region", "compare", "AY123456.1" },
    { "C_region", "mitochondrion", "foo" },
    { "C_region", "mobile_element", "integron" },
    { "C_region", "metagenomic", "foo" },
    { "C_region", "kinetoplast", "foo" },
    { "C_region", "chromoplast", "foo" },
    { "C_region", "specific_host", "foo" },
    { "C_region", "sub_strain", "foo" },
    { "C_region", "tag_peptide", "foo" },
    { "C_region", "isolation_source", "foo" },
    { "C_region", "collected_by", "foo" },
    { "C_region", "rpt_family", "foo" },
    { "C_region", "rpt_type", "flanking" },
    { "C_region", "insertion_seq", "foo" },
    { "C_region", "transl_table", "foo" },
    { "C_region", "rearranged", "foo" },
    { "C_region", "mod_base", "foo" },
    { "C_region", "rpt_unit", "foo" },
    { "C_region", "anticodon", "foo" },
    { "C_region", "function", "foo" },
    { "C_region", "number", "foo" },
    { "C_region", "identified_by", "foo" },
    { "C_region", "collection_date", "foo" },
    { "C_region", "direction", "foo" },
    { "C_region", "rpt_unit_range", "1..3" },
    { "C_region", "serotype", "foo" },
    { "C_region", "satellite", "foo" },
    { "C_region", "organism", "foo" },
    { "C_region", "transcript_id", "foo" },
    { "C_region", "serovar", "foo" },
    { "C_region", "variety", "foo" },
    { "C_region", "country", "foo" },
    { "C_region", "rpt_unit_seq", "foo" },
    { "C_region", "lab_host", "foo" },
    { "C_region", "macronuclear", "foo" },
    { "C_region", "cyanelle", "foo" },
    { "C_region", "bio_material", "foo" },
    { "C_region", "chloroplast", "foo" },
    { "C_region", "plasmid", "foo" },
    { "C_region", "mating_type", "foo" },
    { "C_region", "cell_type", "foo" },
    { "C_region", "EC_number", "1.2.3.4" },
    { "C_region", "mol_type", "foo" },
    { "C_region", "operon", "foo" },
    { "C_region", "cultivar", "foo" },
    { "C_region", "artificial_location", "foo" },
    { "C_region", "segment", "foo" },
    { "C_region", "cons_splice", "foo" },
    { "C_region", "environmental_sample", "foo" },
    { "C_region", "PCR_conditions", "foo" },
    { "C_region", "frequency", "foo" },
    { "C_region", "transposon", "foo" },
    { "C_region", "haplogroup", "foo" },
    { "C_region", "ribosomal_slippage", "foo" },
    { "C_region", "codon_start", "foo" },
    { "C_region", "clone", "foo" },
    { "C_region", "gdb_xref", "foo" },
    { "C_region", "translation", "foo" },
    { "C_region", "transl_except", "foo" },
    { "C_region", "bound_moiety", "foo" },
    { "C_region", "sub_clone", "foo" },
    { "C_region", "cell_line", "foo" },
    { "C_region", "transgenic", "foo" },
    { "C_region", "germline", "foo" },
    { "C_region", "protein_id", "foo" },
    { "C_region", "codon", "foo" },
    { "C_region", "clone_lib", "foo" },
    { "C_region", "PCR_primers", "foo" },
    { "C_region", "sequenced_mol", "foo" },
    { "C_region", "replace", "aaaaattttt" },
    { "C_region", "strain", "foo" },
    { "C_region", "proviral", "foo" },
    { "C_region", "lat_lon", "foo" },
    { "C_region", "culture_collection", "foo" },
    { "C_region", "haplotype", "foo" },
    { "C_region", "estimated_length", "foo" },
    { "C_region", "tissue_lib", "foo" },
    { "C_region", "focus", "foo" },
    { "C_region", "dev_stage", "foo" },
    { "C_region", "specimen_voucher", "foo" },
    { "C_region", "isolate", "foo" },
    { "C_region", "chromosome", "foo" },
    { "C_region", "ncRNA_class", "foo" },
    { "C_region", "pop_variant", "foo" },
    { "C_region", "tissue_type", "foo" },
    { "C_region", "trans_splicing", "foo" },
    { "C_region", "organelle", "foo" },
    { "C_region", "sex", "foo" },
    { "C_region", "virion", "foo" },
    { "C_region", "sub_species", "foo" },
    { "C_region", "phenotype", "foo" },
    { "C_region", "ecotype", "foo" },
    { "CAAT_signal", "compare", "AY123456.1" },
    { "CAAT_signal", "mitochondrion", "foo" },
    { "CAAT_signal", "mobile_element", "integron" },
    { "CAAT_signal", "metagenomic", "foo" },
    { "CAAT_signal", "standard_name", "foo" },
    { "CAAT_signal", "kinetoplast", "foo" },
    { "CAAT_signal", "chromoplast", "foo" },
    { "CAAT_signal", "specific_host", "foo" },
    { "CAAT_signal", "sub_strain", "foo" },
    { "CAAT_signal", "tag_peptide", "foo" },
    { "CAAT_signal", "isolation_source", "foo" },
    { "CAAT_signal", "collected_by", "foo" },
    { "CAAT_signal", "rpt_family", "foo" },
    { "CAAT_signal", "rpt_type", "flanking" },
    { "CAAT_signal", "insertion_seq", "foo" },
    { "CAAT_signal", "transl_table", "foo" },
    { "CAAT_signal", "rearranged", "foo" },
    { "CAAT_signal", "mod_base", "foo" },
    { "CAAT_signal", "rpt_unit", "foo" },
    { "CAAT_signal", "anticodon", "foo" },
    { "CAAT_signal", "function", "foo" },
    { "CAAT_signal", "number", "foo" },
    { "CAAT_signal", "identified_by", "foo" },
    { "CAAT_signal", "collection_date", "foo" },
    { "CAAT_signal", "direction", "foo" },
    { "CAAT_signal", "rpt_unit_range", "1..3" },
    { "CAAT_signal", "serotype", "foo" },
    { "CAAT_signal", "satellite", "foo" },
    { "CAAT_signal", "organism", "foo" },
    { "CAAT_signal", "transcript_id", "foo" },
    { "CAAT_signal", "serovar", "foo" },
    { "CAAT_signal", "variety", "foo" },
    { "CAAT_signal", "country", "foo" },
    { "CAAT_signal", "rpt_unit_seq", "foo" },
    { "CAAT_signal", "lab_host", "foo" },
    { "CAAT_signal", "macronuclear", "foo" },
    { "CAAT_signal", "cyanelle", "foo" },
    { "CAAT_signal", "bio_material", "foo" },
    { "CAAT_signal", "chloroplast", "foo" },
    { "CAAT_signal", "plasmid", "foo" },
    { "CAAT_signal", "mating_type", "foo" },
    { "CAAT_signal", "cell_type", "foo" },
    { "CAAT_signal", "EC_number", "1.2.3.4" },
    { "CAAT_signal", "mol_type", "foo" },
    { "CAAT_signal", "operon", "foo" },
    { "CAAT_signal", "cultivar", "foo" },
    { "CAAT_signal", "artificial_location", "foo" },
    { "CAAT_signal", "segment", "foo" },
    { "CAAT_signal", "cons_splice", "foo" },
    { "CAAT_signal", "environmental_sample", "foo" },
    { "CAAT_signal", "PCR_conditions", "foo" },
    { "CAAT_signal", "frequency", "foo" },
    { "CAAT_signal", "transposon", "foo" },
    { "CAAT_signal", "haplogroup", "foo" },
    { "CAAT_signal", "ribosomal_slippage", "foo" },
    { "CAAT_signal", "codon_start", "foo" },
    { "CAAT_signal", "clone", "foo" },
    { "CAAT_signal", "gdb_xref", "foo" },
    { "CAAT_signal", "translation", "foo" },
    { "CAAT_signal", "transl_except", "foo" },
    { "CAAT_signal", "bound_moiety", "foo" },
    { "CAAT_signal", "sub_clone", "foo" },
    { "CAAT_signal", "cell_line", "foo" },
    { "CAAT_signal", "transgenic", "foo" },
    { "CAAT_signal", "germline", "foo" },
    { "CAAT_signal", "protein_id", "foo" },
    { "CAAT_signal", "codon", "foo" },
    { "CAAT_signal", "clone_lib", "foo" },
    { "CAAT_signal", "PCR_primers", "foo" },
    { "CAAT_signal", "sequenced_mol", "foo" },
    { "CAAT_signal", "replace", "aaaaattttt" },
    { "CAAT_signal", "strain", "foo" },
    { "CAAT_signal", "proviral", "foo" },
    { "CAAT_signal", "lat_lon", "foo" },
    { "CAAT_signal", "culture_collection", "foo" },
    { "CAAT_signal", "haplotype", "foo" },
    { "CAAT_signal", "estimated_length", "foo" },
    { "CAAT_signal", "tissue_lib", "foo" },
    { "CAAT_signal", "focus", "foo" },
    { "CAAT_signal", "dev_stage", "foo" },
    { "CAAT_signal", "specimen_voucher", "foo" },
    { "CAAT_signal", "pseudo", "foo" },
    { "CAAT_signal", "isolate", "foo" },
    { "CAAT_signal", "chromosome", "foo" },
    { "CAAT_signal", "ncRNA_class", "foo" },
    { "CAAT_signal", "pop_variant", "foo" },
    { "CAAT_signal", "tissue_type", "foo" },
    { "CAAT_signal", "trans_splicing", "foo" },
    { "CAAT_signal", "organelle", "foo" },
    { "CAAT_signal", "sex", "foo" },
    { "CAAT_signal", "virion", "foo" },
    { "CAAT_signal", "sub_species", "foo" },
    { "CAAT_signal", "phenotype", "foo" },
    { "CAAT_signal", "ecotype", "foo" },
    { "CAAT_signal", "product", "foo" },
    { "conflict", "mitochondrion", "foo" },
    { "conflict", "mobile_element", "integron" },
    { "conflict", "metagenomic", "foo" },
    { "conflict", "standard_name", "foo" },
    { "conflict", "kinetoplast", "foo" },
    { "conflict", "chromoplast", "foo" },
    { "conflict", "specific_host", "foo" },
    { "conflict", "sub_strain", "foo" },
    { "conflict", "tag_peptide", "foo" },
    { "conflict", "isolation_source", "foo" },
    { "conflict", "collected_by", "foo" },
    { "conflict", "rpt_family", "foo" },
    { "conflict", "rpt_type", "flanking" },
    { "conflict", "insertion_seq", "foo" },
    { "conflict", "transl_table", "foo" },
    { "conflict", "rearranged", "foo" },
    { "conflict", "mod_base", "foo" },
    { "conflict", "rpt_unit", "foo" },
    { "conflict", "anticodon", "foo" },
    { "conflict", "function", "foo" },
    { "conflict", "number", "foo" },
    { "conflict", "identified_by", "foo" },
    { "conflict", "collection_date", "foo" },
    { "conflict", "direction", "foo" },
    { "conflict", "rpt_unit_range", "1..3" },
    { "conflict", "serotype", "foo" },
    { "conflict", "satellite", "foo" },
    { "conflict", "organism", "foo" },
    { "conflict", "transcript_id", "foo" },
    { "conflict", "serovar", "foo" },
    { "conflict", "variety", "foo" },
    { "conflict", "country", "foo" },
    { "conflict", "rpt_unit_seq", "foo" },
    { "conflict", "lab_host", "foo" },
    { "conflict", "macronuclear", "foo" },
    { "conflict", "cyanelle", "foo" },
    { "conflict", "bio_material", "foo" },
    { "conflict", "chloroplast", "foo" },
    { "conflict", "plasmid", "foo" },
    { "conflict", "mating_type", "foo" },
    { "conflict", "cell_type", "foo" },
    { "conflict", "EC_number", "1.2.3.4" },
    { "conflict", "mol_type", "foo" },
    { "conflict", "operon", "foo" },
    { "conflict", "cultivar", "foo" },
    { "conflict", "artificial_location", "foo" },
    { "conflict", "segment", "foo" },
    { "conflict", "cons_splice", "foo" },
    { "conflict", "environmental_sample", "foo" },
    { "conflict", "PCR_conditions", "foo" },
    { "conflict", "frequency", "foo" },
    { "conflict", "transposon", "foo" },
    { "conflict", "haplogroup", "foo" },
    { "conflict", "ribosomal_slippage", "foo" },
    { "conflict", "codon_start", "foo" },
    { "conflict", "clone", "foo" },
    { "conflict", "gdb_xref", "foo" },
    { "conflict", "translation", "foo" },
    { "conflict", "transl_except", "foo" },
    { "conflict", "bound_moiety", "foo" },
    { "conflict", "sub_clone", "foo" },
    { "conflict", "cell_line", "foo" },
    { "conflict", "transgenic", "foo" },
    { "conflict", "germline", "foo" },
    { "conflict", "protein_id", "foo" },
    { "conflict", "codon", "foo" },
    { "conflict", "clone_lib", "foo" },
    { "conflict", "PCR_primers", "foo" },
    { "conflict", "sequenced_mol", "foo" },
    { "conflict", "strain", "foo" },
    { "conflict", "proviral", "foo" },
    { "conflict", "lat_lon", "foo" },
    { "conflict", "culture_collection", "foo" },
    { "conflict", "haplotype", "foo" },
    { "conflict", "estimated_length", "foo" },
    { "conflict", "tissue_lib", "foo" },
    { "conflict", "focus", "foo" },
    { "conflict", "dev_stage", "foo" },
    { "conflict", "partial", "foo" },
    { "conflict", "specimen_voucher", "foo" },
    { "conflict", "pseudo", "foo" },
    { "conflict", "isolate", "foo" },
    { "conflict", "chromosome", "foo" },
    { "conflict", "ncRNA_class", "foo" },
    { "conflict", "pop_variant", "foo" },
    { "conflict", "tissue_type", "foo" },
    { "conflict", "trans_splicing", "foo" },
    { "conflict", "organelle", "foo" },
    { "conflict", "sex", "foo" },
    { "conflict", "virion", "foo" },
    { "conflict", "sub_species", "foo" },
    { "conflict", "phenotype", "foo" },
    { "conflict", "ecotype", "foo" },
    { "conflict", "product", "foo" },
    { "D-loop", "compare", "AY123456.1" },
    { "D-loop", "mitochondrion", "foo" },
    { "D-loop", "mobile_element", "integron" },
    { "D-loop", "metagenomic", "foo" },
    { "D-loop", "standard_name", "foo" },
    { "D-loop", "kinetoplast", "foo" },
    { "D-loop", "chromoplast", "foo" },
    { "D-loop", "specific_host", "foo" },
    { "D-loop", "sub_strain", "foo" },
    { "D-loop", "tag_peptide", "foo" },
    { "D-loop", "isolation_source", "foo" },
    { "D-loop", "collected_by", "foo" },
    { "D-loop", "rpt_family", "foo" },
    { "D-loop", "rpt_type", "flanking" },
    { "D-loop", "insertion_seq", "foo" },
    { "D-loop", "transl_table", "foo" },
    { "D-loop", "rearranged", "foo" },
    { "D-loop", "mod_base", "foo" },
    { "D-loop", "rpt_unit", "foo" },
    { "D-loop", "anticodon", "foo" },
    { "D-loop", "function", "foo" },
    { "D-loop", "number", "foo" },
    { "D-loop", "identified_by", "foo" },
    { "D-loop", "collection_date", "foo" },
    { "D-loop", "direction", "foo" },
    { "D-loop", "rpt_unit_range", "1..3" },
    { "D-loop", "serotype", "foo" },
    { "D-loop", "satellite", "foo" },
    { "D-loop", "organism", "foo" },
    { "D-loop", "transcript_id", "foo" },
    { "D-loop", "serovar", "foo" },
    { "D-loop", "variety", "foo" },
    { "D-loop", "country", "foo" },
    { "D-loop", "rpt_unit_seq", "foo" },
    { "D-loop", "lab_host", "foo" },
    { "D-loop", "macronuclear", "foo" },
    { "D-loop", "cyanelle", "foo" },
    { "D-loop", "bio_material", "foo" },
    { "D-loop", "chloroplast", "foo" },
    { "D-loop", "plasmid", "foo" },
    { "D-loop", "mating_type", "foo" },
    { "D-loop", "cell_type", "foo" },
    { "D-loop", "EC_number", "1.2.3.4" },
    { "D-loop", "mol_type", "foo" },
    { "D-loop", "operon", "foo" },
    { "D-loop", "cultivar", "foo" },
    { "D-loop", "artificial_location", "foo" },
    { "D-loop", "segment", "foo" },
    { "D-loop", "cons_splice", "foo" },
    { "D-loop", "environmental_sample", "foo" },
    { "D-loop", "PCR_conditions", "foo" },
    { "D-loop", "frequency", "foo" },
    { "D-loop", "transposon", "foo" },
    { "D-loop", "haplogroup", "foo" },
    { "D-loop", "ribosomal_slippage", "foo" },
    { "D-loop", "codon_start", "foo" },
    { "D-loop", "clone", "foo" },
    { "D-loop", "gdb_xref", "foo" },
    { "D-loop", "translation", "foo" },
    { "D-loop", "transl_except", "foo" },
    { "D-loop", "bound_moiety", "foo" },
    { "D-loop", "sub_clone", "foo" },
    { "D-loop", "cell_line", "foo" },
    { "D-loop", "transgenic", "foo" },
    { "D-loop", "germline", "foo" },
    { "D-loop", "protein_id", "foo" },
    { "D-loop", "codon", "foo" },
    { "D-loop", "clone_lib", "foo" },
    { "D-loop", "PCR_primers", "foo" },
    { "D-loop", "sequenced_mol", "foo" },
    { "D-loop", "replace", "aaaaattttt" },
    { "D-loop", "strain", "foo" },
    { "D-loop", "proviral", "foo" },
    { "D-loop", "lat_lon", "foo" },
    { "D-loop", "culture_collection", "foo" },
    { "D-loop", "haplotype", "foo" },
    { "D-loop", "estimated_length", "foo" },
    { "D-loop", "tissue_lib", "foo" },
    { "D-loop", "focus", "foo" },
    { "D-loop", "dev_stage", "foo" },
    { "D-loop", "specimen_voucher", "foo" },
    { "D-loop", "pseudo", "foo" },
    { "D-loop", "isolate", "foo" },
    { "D-loop", "chromosome", "foo" },
    { "D-loop", "ncRNA_class", "foo" },
    { "D-loop", "pop_variant", "foo" },
    { "D-loop", "tissue_type", "foo" },
    { "D-loop", "trans_splicing", "foo" },
    { "D-loop", "organelle", "foo" },
    { "D-loop", "sex", "foo" },
    { "D-loop", "virion", "foo" },
    { "D-loop", "sub_species", "foo" },
    { "D-loop", "phenotype", "foo" },
    { "D-loop", "ecotype", "foo" },
    { "D-loop", "product", "foo" },
    { "D_segment", "compare", "AY123456.1" },
    { "D_segment", "mitochondrion", "foo" },
    { "D_segment", "mobile_element", "integron" },
    { "D_segment", "metagenomic", "foo" },
    { "D_segment", "kinetoplast", "foo" },
    { "D_segment", "chromoplast", "foo" },
    { "D_segment", "specific_host", "foo" },
    { "D_segment", "sub_strain", "foo" },
    { "D_segment", "tag_peptide", "foo" },
    { "D_segment", "isolation_source", "foo" },
    { "D_segment", "collected_by", "foo" },
    { "D_segment", "rpt_family", "foo" },
    { "D_segment", "rpt_type", "flanking" },
    { "D_segment", "insertion_seq", "foo" },
    { "D_segment", "transl_table", "foo" },
    { "D_segment", "rearranged", "foo" },
    { "D_segment", "mod_base", "foo" },
    { "D_segment", "rpt_unit", "foo" },
    { "D_segment", "anticodon", "foo" },
    { "D_segment", "function", "foo" },
    { "D_segment", "number", "foo" },
    { "D_segment", "identified_by", "foo" },
    { "D_segment", "collection_date", "foo" },
    { "D_segment", "direction", "foo" },
    { "D_segment", "rpt_unit_range", "1..3" },
    { "D_segment", "serotype", "foo" },
    { "D_segment", "satellite", "foo" },
    { "D_segment", "organism", "foo" },
    { "D_segment", "transcript_id", "foo" },
    { "D_segment", "serovar", "foo" },
    { "D_segment", "variety", "foo" },
    { "D_segment", "country", "foo" },
    { "D_segment", "rpt_unit_seq", "foo" },
    { "D_segment", "lab_host", "foo" },
    { "D_segment", "macronuclear", "foo" },
    { "D_segment", "cyanelle", "foo" },
    { "D_segment", "bio_material", "foo" },
    { "D_segment", "chloroplast", "foo" },
    { "D_segment", "plasmid", "foo" },
    { "D_segment", "mating_type", "foo" },
    { "D_segment", "cell_type", "foo" },
    { "D_segment", "EC_number", "1.2.3.4" },
    { "D_segment", "mol_type", "foo" },
    { "D_segment", "operon", "foo" },
    { "D_segment", "cultivar", "foo" },
    { "D_segment", "artificial_location", "foo" },
    { "D_segment", "segment", "foo" },
    { "D_segment", "cons_splice", "foo" },
    { "D_segment", "environmental_sample", "foo" },
    { "D_segment", "PCR_conditions", "foo" },
    { "D_segment", "frequency", "foo" },
    { "D_segment", "transposon", "foo" },
    { "D_segment", "haplogroup", "foo" },
    { "D_segment", "ribosomal_slippage", "foo" },
    { "D_segment", "codon_start", "foo" },
    { "D_segment", "clone", "foo" },
    { "D_segment", "gdb_xref", "foo" },
    { "D_segment", "translation", "foo" },
    { "D_segment", "transl_except", "foo" },
    { "D_segment", "bound_moiety", "foo" },
    { "D_segment", "sub_clone", "foo" },
    { "D_segment", "cell_line", "foo" },
    { "D_segment", "transgenic", "foo" },
    { "D_segment", "germline", "foo" },
    { "D_segment", "protein_id", "foo" },
    { "D_segment", "codon", "foo" },
    { "D_segment", "clone_lib", "foo" },
    { "D_segment", "PCR_primers", "foo" },
    { "D_segment", "sequenced_mol", "foo" },
    { "D_segment", "replace", "aaaaattttt" },
    { "D_segment", "strain", "foo" },
    { "D_segment", "proviral", "foo" },
    { "D_segment", "lat_lon", "foo" },
    { "D_segment", "culture_collection", "foo" },
    { "D_segment", "haplotype", "foo" },
    { "D_segment", "estimated_length", "foo" },
    { "D_segment", "tissue_lib", "foo" },
    { "D_segment", "focus", "foo" },
    { "D_segment", "dev_stage", "foo" },
    { "D_segment", "specimen_voucher", "foo" },
    { "D_segment", "isolate", "foo" },
    { "D_segment", "chromosome", "foo" },
    { "D_segment", "ncRNA_class", "foo" },
    { "D_segment", "pop_variant", "foo" },
    { "D_segment", "tissue_type", "foo" },
    { "D_segment", "trans_splicing", "foo" },
    { "D_segment", "organelle", "foo" },
    { "D_segment", "sex", "foo" },
    { "D_segment", "virion", "foo" },
    { "D_segment", "sub_species", "foo" },
    { "D_segment", "phenotype", "foo" },
    { "D_segment", "ecotype", "foo" },
    { "enhancer", "compare", "AY123456.1" },
    { "enhancer", "mitochondrion", "foo" },
    { "enhancer", "mobile_element", "integron" },
    { "enhancer", "metagenomic", "foo" },
    { "enhancer", "kinetoplast", "foo" },
    { "enhancer", "chromoplast", "foo" },
    { "enhancer", "specific_host", "foo" },
    { "enhancer", "sub_strain", "foo" },
    { "enhancer", "tag_peptide", "foo" },
    { "enhancer", "isolation_source", "foo" },
    { "enhancer", "collected_by", "foo" },
    { "enhancer", "rpt_family", "foo" },
    { "enhancer", "rpt_type", "flanking" },
    { "enhancer", "insertion_seq", "foo" },
    { "enhancer", "transl_table", "foo" },
    { "enhancer", "rearranged", "foo" },
    { "enhancer", "mod_base", "foo" },
    { "enhancer", "rpt_unit", "foo" },
    { "enhancer", "anticodon", "foo" },
    { "enhancer", "function", "foo" },
    { "enhancer", "number", "foo" },
    { "enhancer", "identified_by", "foo" },
    { "enhancer", "collection_date", "foo" },
    { "enhancer", "direction", "foo" },
    { "enhancer", "rpt_unit_range", "1..3" },
    { "enhancer", "serotype", "foo" },
    { "enhancer", "satellite", "foo" },
    { "enhancer", "organism", "foo" },
    { "enhancer", "transcript_id", "foo" },
    { "enhancer", "serovar", "foo" },
    { "enhancer", "variety", "foo" },
    { "enhancer", "country", "foo" },
    { "enhancer", "rpt_unit_seq", "foo" },
    { "enhancer", "lab_host", "foo" },
    { "enhancer", "macronuclear", "foo" },
    { "enhancer", "cyanelle", "foo" },
    { "enhancer", "bio_material", "foo" },
    { "enhancer", "chloroplast", "foo" },
    { "enhancer", "plasmid", "foo" },
    { "enhancer", "mating_type", "foo" },
    { "enhancer", "cell_type", "foo" },
    { "enhancer", "EC_number", "1.2.3.4" },
    { "enhancer", "mol_type", "foo" },
    { "enhancer", "operon", "foo" },
    { "enhancer", "cultivar", "foo" },
    { "enhancer", "artificial_location", "foo" },
    { "enhancer", "segment", "foo" },
    { "enhancer", "cons_splice", "foo" },
    { "enhancer", "environmental_sample", "foo" },
    { "enhancer", "PCR_conditions", "foo" },
    { "enhancer", "frequency", "foo" },
    { "enhancer", "transposon", "foo" },
    { "enhancer", "haplogroup", "foo" },
    { "enhancer", "ribosomal_slippage", "foo" },
    { "enhancer", "codon_start", "foo" },
    { "enhancer", "clone", "foo" },
    { "enhancer", "gdb_xref", "foo" },
    { "enhancer", "translation", "foo" },
    { "enhancer", "transl_except", "foo" },
    { "enhancer", "sub_clone", "foo" },
    { "enhancer", "cell_line", "foo" },
    { "enhancer", "transgenic", "foo" },
    { "enhancer", "germline", "foo" },
    { "enhancer", "protein_id", "foo" },
    { "enhancer", "codon", "foo" },
    { "enhancer", "clone_lib", "foo" },
    { "enhancer", "PCR_primers", "foo" },
    { "enhancer", "sequenced_mol", "foo" },
    { "enhancer", "replace", "aaaaattttt" },
    { "enhancer", "strain", "foo" },
    { "enhancer", "proviral", "foo" },
    { "enhancer", "lat_lon", "foo" },
    { "enhancer", "culture_collection", "foo" },
    { "enhancer", "haplotype", "foo" },
    { "enhancer", "estimated_length", "foo" },
    { "enhancer", "tissue_lib", "foo" },
    { "enhancer", "focus", "foo" },
    { "enhancer", "dev_stage", "foo" },
    { "enhancer", "specimen_voucher", "foo" },
    { "enhancer", "pseudo", "foo" },
    { "enhancer", "isolate", "foo" },
    { "enhancer", "chromosome", "foo" },
    { "enhancer", "ncRNA_class", "foo" },
    { "enhancer", "pop_variant", "foo" },
    { "enhancer", "tissue_type", "foo" },
    { "enhancer", "trans_splicing", "foo" },
    { "enhancer", "organelle", "foo" },
    { "enhancer", "sex", "foo" },
    { "enhancer", "virion", "foo" },
    { "enhancer", "sub_species", "foo" },
    { "enhancer", "phenotype", "foo" },
    { "enhancer", "ecotype", "foo" },
    { "enhancer", "product", "foo" },
    { "exon", "compare", "AY123456.1" },
    { "exon", "mitochondrion", "foo" },
    { "exon", "mobile_element", "integron" },
    { "exon", "metagenomic", "foo" },
    { "exon", "kinetoplast", "foo" },
    { "exon", "chromoplast", "foo" },
    { "exon", "specific_host", "foo" },
    { "exon", "sub_strain", "foo" },
    { "exon", "tag_peptide", "foo" },
    { "exon", "isolation_source", "foo" },
    { "exon", "collected_by", "foo" },
    { "exon", "rpt_family", "foo" },
    { "exon", "rpt_type", "flanking" },
    { "exon", "insertion_seq", "foo" },
    { "exon", "transl_table", "foo" },
    { "exon", "rearranged", "foo" },
    { "exon", "mod_base", "foo" },
    { "exon", "rpt_unit", "foo" },
    { "exon", "anticodon", "foo" },
    { "exon", "identified_by", "foo" },
    { "exon", "collection_date", "foo" },
    { "exon", "direction", "foo" },
    { "exon", "rpt_unit_range", "1..3" },
    { "exon", "serotype", "foo" },
    { "exon", "satellite", "foo" },
    { "exon", "organism", "foo" },
    { "exon", "transcript_id", "foo" },
    { "exon", "serovar", "foo" },
    { "exon", "variety", "foo" },
    { "exon", "country", "foo" },
    { "exon", "rpt_unit_seq", "foo" },
    { "exon", "lab_host", "foo" },
    { "exon", "macronuclear", "foo" },
    { "exon", "cyanelle", "foo" },
    { "exon", "bio_material", "foo" },
    { "exon", "chloroplast", "foo" },
    { "exon", "plasmid", "foo" },
    { "exon", "mating_type", "foo" },
    { "exon", "cell_type", "foo" },
    { "exon", "mol_type", "foo" },
    { "exon", "operon", "foo" },
    { "exon", "cultivar", "foo" },
    { "exon", "artificial_location", "foo" },
    { "exon", "segment", "foo" },
    { "exon", "cons_splice", "foo" },
    { "exon", "environmental_sample", "foo" },
    { "exon", "PCR_conditions", "foo" },
    { "exon", "frequency", "foo" },
    { "exon", "transposon", "foo" },
    { "exon", "haplogroup", "foo" },
    { "exon", "ribosomal_slippage", "foo" },
    { "exon", "codon_start", "foo" },
    { "exon", "clone", "foo" },
    { "exon", "gdb_xref", "foo" },
    { "exon", "translation", "foo" },
    { "exon", "transl_except", "foo" },
    { "exon", "bound_moiety", "foo" },
    { "exon", "sub_clone", "foo" },
    { "exon", "cell_line", "foo" },
    { "exon", "transgenic", "foo" },
    { "exon", "germline", "foo" },
    { "exon", "protein_id", "foo" },
    { "exon", "codon", "foo" },
    { "exon", "clone_lib", "foo" },
    { "exon", "PCR_primers", "foo" },
    { "exon", "sequenced_mol", "foo" },
    { "exon", "replace", "aaaaattttt" },
    { "exon", "strain", "foo" },
    { "exon", "proviral", "foo" },
    { "exon", "lat_lon", "foo" },
    { "exon", "culture_collection", "foo" },
    { "exon", "haplotype", "foo" },
    { "exon", "estimated_length", "foo" },
    { "exon", "tissue_lib", "foo" },
    { "exon", "focus", "foo" },
    { "exon", "dev_stage", "foo" },
    { "exon", "specimen_voucher", "foo" },
    { "exon", "isolate", "foo" },
    { "exon", "chromosome", "foo" },
    { "exon", "ncRNA_class", "foo" },
    { "exon", "pop_variant", "foo" },
    { "exon", "tissue_type", "foo" },
    { "exon", "organelle", "foo" },
    { "exon", "sex", "foo" },
    { "exon", "virion", "foo" },
    { "exon", "sub_species", "foo" },
    { "exon", "phenotype", "foo" },
    { "exon", "ecotype", "foo" },
    { "GC_signal", "compare", "AY123456.1" },
    { "GC_signal", "mitochondrion", "foo" },
    { "GC_signal", "mobile_element", "integron" },
    { "GC_signal", "metagenomic", "foo" },
    { "GC_signal", "standard_name", "foo" },
    { "GC_signal", "kinetoplast", "foo" },
    { "GC_signal", "chromoplast", "foo" },
    { "GC_signal", "specific_host", "foo" },
    { "GC_signal", "sub_strain", "foo" },
    { "GC_signal", "tag_peptide", "foo" },
    { "GC_signal", "isolation_source", "foo" },
    { "GC_signal", "collected_by", "foo" },
    { "GC_signal", "rpt_family", "foo" },
    { "GC_signal", "rpt_type", "flanking" },
    { "GC_signal", "insertion_seq", "foo" },
    { "GC_signal", "transl_table", "foo" },
    { "GC_signal", "rearranged", "foo" },
    { "GC_signal", "mod_base", "foo" },
    { "GC_signal", "rpt_unit", "foo" },
    { "GC_signal", "anticodon", "foo" },
    { "GC_signal", "function", "foo" },
    { "GC_signal", "number", "foo" },
    { "GC_signal", "identified_by", "foo" },
    { "GC_signal", "collection_date", "foo" },
    { "GC_signal", "direction", "foo" },
    { "GC_signal", "rpt_unit_range", "1..3" },
    { "GC_signal", "serotype", "foo" },
    { "GC_signal", "satellite", "foo" },
    { "GC_signal", "organism", "foo" },
    { "GC_signal", "transcript_id", "foo" },
    { "GC_signal", "serovar", "foo" },
    { "GC_signal", "variety", "foo" },
    { "GC_signal", "country", "foo" },
    { "GC_signal", "rpt_unit_seq", "foo" },
    { "GC_signal", "lab_host", "foo" },
    { "GC_signal", "macronuclear", "foo" },
    { "GC_signal", "cyanelle", "foo" },
    { "GC_signal", "bio_material", "foo" },
    { "GC_signal", "chloroplast", "foo" },
    { "GC_signal", "plasmid", "foo" },
    { "GC_signal", "mating_type", "foo" },
    { "GC_signal", "cell_type", "foo" },
    { "GC_signal", "EC_number", "1.2.3.4" },
    { "GC_signal", "mol_type", "foo" },
    { "GC_signal", "operon", "foo" },
    { "GC_signal", "cultivar", "foo" },
    { "GC_signal", "artificial_location", "foo" },
    { "GC_signal", "segment", "foo" },
    { "GC_signal", "cons_splice", "foo" },
    { "GC_signal", "environmental_sample", "foo" },
    { "GC_signal", "PCR_conditions", "foo" },
    { "GC_signal", "frequency", "foo" },
    { "GC_signal", "transposon", "foo" },
    { "GC_signal", "haplogroup", "foo" },
    { "GC_signal", "ribosomal_slippage", "foo" },
    { "GC_signal", "codon_start", "foo" },
    { "GC_signal", "clone", "foo" },
    { "GC_signal", "gdb_xref", "foo" },
    { "GC_signal", "translation", "foo" },
    { "GC_signal", "transl_except", "foo" },
    { "GC_signal", "bound_moiety", "foo" },
    { "GC_signal", "sub_clone", "foo" },
    { "GC_signal", "cell_line", "foo" },
    { "GC_signal", "transgenic", "foo" },
    { "GC_signal", "germline", "foo" },
    { "GC_signal", "protein_id", "foo" },
    { "GC_signal", "codon", "foo" },
    { "GC_signal", "clone_lib", "foo" },
    { "GC_signal", "PCR_primers", "foo" },
    { "GC_signal", "sequenced_mol", "foo" },
    { "GC_signal", "replace", "aaaaattttt" },
    { "GC_signal", "strain", "foo" },
    { "GC_signal", "proviral", "foo" },
    { "GC_signal", "lat_lon", "foo" },
    { "GC_signal", "culture_collection", "foo" },
    { "GC_signal", "haplotype", "foo" },
    { "GC_signal", "estimated_length", "foo" },
    { "GC_signal", "tissue_lib", "foo" },
    { "GC_signal", "focus", "foo" },
    { "GC_signal", "dev_stage", "foo" },
    { "GC_signal", "specimen_voucher", "foo" },
    { "GC_signal", "pseudo", "foo" },
    { "GC_signal", "isolate", "foo" },
    { "GC_signal", "chromosome", "foo" },
    { "GC_signal", "ncRNA_class", "foo" },
    { "GC_signal", "pop_variant", "foo" },
    { "GC_signal", "tissue_type", "foo" },
    { "GC_signal", "trans_splicing", "foo" },
    { "GC_signal", "organelle", "foo" },
    { "GC_signal", "sex", "foo" },
    { "GC_signal", "virion", "foo" },
    { "GC_signal", "sub_species", "foo" },
    { "GC_signal", "phenotype", "foo" },
    { "GC_signal", "ecotype", "foo" },
    { "GC_signal", "product", "foo" },
    { "iDNA", "compare", "AY123456.1" },
    { "iDNA", "mitochondrion", "foo" },
    { "iDNA", "mobile_element", "integron" },
    { "iDNA", "metagenomic", "foo" },
    { "iDNA", "kinetoplast", "foo" },
    { "iDNA", "chromoplast", "foo" },
    { "iDNA", "specific_host", "foo" },
    { "iDNA", "sub_strain", "foo" },
    { "iDNA", "tag_peptide", "foo" },
    { "iDNA", "isolation_source", "foo" },
    { "iDNA", "collected_by", "foo" },
    { "iDNA", "rpt_family", "foo" },
    { "iDNA", "rpt_type", "flanking" },
    { "iDNA", "insertion_seq", "foo" },
    { "iDNA", "transl_table", "foo" },
    { "iDNA", "rearranged", "foo" },
    { "iDNA", "mod_base", "foo" },
    { "iDNA", "rpt_unit", "foo" },
    { "iDNA", "anticodon", "foo" },
    { "iDNA", "identified_by", "foo" },
    { "iDNA", "collection_date", "foo" },
    { "iDNA", "direction", "foo" },
    { "iDNA", "rpt_unit_range", "1..3" },
    { "iDNA", "serotype", "foo" },
    { "iDNA", "satellite", "foo" },
    { "iDNA", "organism", "foo" },
    { "iDNA", "transcript_id", "foo" },
    { "iDNA", "serovar", "foo" },
    { "iDNA", "variety", "foo" },
    { "iDNA", "country", "foo" },
    { "iDNA", "rpt_unit_seq", "foo" },
    { "iDNA", "lab_host", "foo" },
    { "iDNA", "macronuclear", "foo" },
    { "iDNA", "cyanelle", "foo" },
    { "iDNA", "bio_material", "foo" },
    { "iDNA", "chloroplast", "foo" },
    { "iDNA", "plasmid", "foo" },
    { "iDNA", "mating_type", "foo" },
    { "iDNA", "cell_type", "foo" },
    { "iDNA", "EC_number", "1.2.3.4" },
    { "iDNA", "mol_type", "foo" },
    { "iDNA", "operon", "foo" },
    { "iDNA", "cultivar", "foo" },
    { "iDNA", "artificial_location", "foo" },
    { "iDNA", "segment", "foo" },
    { "iDNA", "cons_splice", "foo" },
    { "iDNA", "environmental_sample", "foo" },
    { "iDNA", "PCR_conditions", "foo" },
    { "iDNA", "frequency", "foo" },
    { "iDNA", "transposon", "foo" },
    { "iDNA", "haplogroup", "foo" },
    { "iDNA", "ribosomal_slippage", "foo" },
    { "iDNA", "codon_start", "foo" },
    { "iDNA", "clone", "foo" },
    { "iDNA", "gdb_xref", "foo" },
    { "iDNA", "translation", "foo" },
    { "iDNA", "transl_except", "foo" },
    { "iDNA", "bound_moiety", "foo" },
    { "iDNA", "sub_clone", "foo" },
    { "iDNA", "cell_line", "foo" },
    { "iDNA", "transgenic", "foo" },
    { "iDNA", "germline", "foo" },
    { "iDNA", "protein_id", "foo" },
    { "iDNA", "codon", "foo" },
    { "iDNA", "clone_lib", "foo" },
    { "iDNA", "PCR_primers", "foo" },
    { "iDNA", "sequenced_mol", "foo" },
    { "iDNA", "replace", "aaaaattttt" },
    { "iDNA", "strain", "foo" },
    { "iDNA", "proviral", "foo" },
    { "iDNA", "lat_lon", "foo" },
    { "iDNA", "culture_collection", "foo" },
    { "iDNA", "haplotype", "foo" },
    { "iDNA", "estimated_length", "foo" },
    { "iDNA", "tissue_lib", "foo" },
    { "iDNA", "focus", "foo" },
    { "iDNA", "dev_stage", "foo" },
    { "iDNA", "specimen_voucher", "foo" },
    { "iDNA", "pseudo", "foo" },
    { "iDNA", "isolate", "foo" },
    { "iDNA", "chromosome", "foo" },
    { "iDNA", "ncRNA_class", "foo" },
    { "iDNA", "pop_variant", "foo" },
    { "iDNA", "tissue_type", "foo" },
    { "iDNA", "trans_splicing", "foo" },
    { "iDNA", "organelle", "foo" },
    { "iDNA", "sex", "foo" },
    { "iDNA", "virion", "foo" },
    { "iDNA", "sub_species", "foo" },
    { "iDNA", "phenotype", "foo" },
    { "iDNA", "ecotype", "foo" },
    { "iDNA", "product", "foo" },
    { "intron", "compare", "AY123456.1" },
    { "intron", "mitochondrion", "foo" },
    { "intron", "mobile_element", "integron" },
    { "intron", "metagenomic", "foo" },
    { "intron", "kinetoplast", "foo" },
    { "intron", "chromoplast", "foo" },
    { "intron", "specific_host", "foo" },
    { "intron", "sub_strain", "foo" },
    { "intron", "tag_peptide", "foo" },
    { "intron", "isolation_source", "foo" },
    { "intron", "collected_by", "foo" },
    { "intron", "rpt_family", "foo" },
    { "intron", "rpt_type", "flanking" },
    { "intron", "insertion_seq", "foo" },
    { "intron", "transl_table", "foo" },
    { "intron", "rearranged", "foo" },
    { "intron", "mod_base", "foo" },
    { "intron", "rpt_unit", "foo" },
    { "intron", "anticodon", "foo" },
    { "intron", "identified_by", "foo" },
    { "intron", "collection_date", "foo" },
    { "intron", "direction", "foo" },
    { "intron", "rpt_unit_range", "1..3" },
    { "intron", "serotype", "foo" },
    { "intron", "satellite", "foo" },
    { "intron", "organism", "foo" },
    { "intron", "transcript_id", "foo" },
    { "intron", "serovar", "foo" },
    { "intron", "variety", "foo" },
    { "intron", "country", "foo" },
    { "intron", "rpt_unit_seq", "foo" },
    { "intron", "lab_host", "foo" },
    { "intron", "macronuclear", "foo" },
    { "intron", "cyanelle", "foo" },
    { "intron", "bio_material", "foo" },
    { "intron", "chloroplast", "foo" },
    { "intron", "plasmid", "foo" },
    { "intron", "mating_type", "foo" },
    { "intron", "cell_type", "foo" },
    { "intron", "EC_number", "1.2.3.4" },
    { "intron", "mol_type", "foo" },
    { "intron", "operon", "foo" },
    { "intron", "cultivar", "foo" },
    { "intron", "artificial_location", "foo" },
    { "intron", "segment", "foo" },
    { "intron", "environmental_sample", "foo" },
    { "intron", "PCR_conditions", "foo" },
    { "intron", "frequency", "foo" },
    { "intron", "transposon", "foo" },
    { "intron", "haplogroup", "foo" },
    { "intron", "ribosomal_slippage", "foo" },
    { "intron", "codon_start", "foo" },
    { "intron", "clone", "foo" },
    { "intron", "gdb_xref", "foo" },
    { "intron", "translation", "foo" },
    { "intron", "transl_except", "foo" },
    { "intron", "bound_moiety", "foo" },
    { "intron", "sub_clone", "foo" },
    { "intron", "cell_line", "foo" },
    { "intron", "transgenic", "foo" },
    { "intron", "germline", "foo" },
    { "intron", "protein_id", "foo" },
    { "intron", "codon", "foo" },
    { "intron", "clone_lib", "foo" },
    { "intron", "PCR_primers", "foo" },
    { "intron", "sequenced_mol", "foo" },
    { "intron", "replace", "aaaaattttt" },
    { "intron", "strain", "foo" },
    { "intron", "proviral", "foo" },
    { "intron", "lat_lon", "foo" },
    { "intron", "culture_collection", "foo" },
    { "intron", "haplotype", "foo" },
    { "intron", "estimated_length", "foo" },
    { "intron", "tissue_lib", "foo" },
    { "intron", "focus", "foo" },
    { "intron", "dev_stage", "foo" },
    { "intron", "specimen_voucher", "foo" },
    { "intron", "isolate", "foo" },
    { "intron", "chromosome", "foo" },
    { "intron", "ncRNA_class", "foo" },
    { "intron", "pop_variant", "foo" },
    { "intron", "tissue_type", "foo" },
    { "intron", "organelle", "foo" },
    { "intron", "sex", "foo" },
    { "intron", "virion", "foo" },
    { "intron", "sub_species", "foo" },
    { "intron", "phenotype", "foo" },
    { "intron", "ecotype", "foo" },
    { "intron", "product", "foo" },
    { "J_segment", "compare", "AY123456.1" },
    { "J_segment", "mitochondrion", "foo" },
    { "J_segment", "mobile_element", "integron" },
    { "J_segment", "metagenomic", "foo" },
    { "J_segment", "kinetoplast", "foo" },
    { "J_segment", "chromoplast", "foo" },
    { "J_segment", "specific_host", "foo" },
    { "J_segment", "sub_strain", "foo" },
    { "J_segment", "tag_peptide", "foo" },
    { "J_segment", "isolation_source", "foo" },
    { "J_segment", "collected_by", "foo" },
    { "J_segment", "rpt_family", "foo" },
    { "J_segment", "rpt_type", "flanking" },
    { "J_segment", "insertion_seq", "foo" },
    { "J_segment", "transl_table", "foo" },
    { "J_segment", "rearranged", "foo" },
    { "J_segment", "mod_base", "foo" },
    { "J_segment", "rpt_unit", "foo" },
    { "J_segment", "anticodon", "foo" },
    { "J_segment", "function", "foo" },
    { "J_segment", "number", "foo" },
    { "J_segment", "identified_by", "foo" },
    { "J_segment", "collection_date", "foo" },
    { "J_segment", "direction", "foo" },
    { "J_segment", "rpt_unit_range", "1..3" },
    { "J_segment", "serotype", "foo" },
    { "J_segment", "satellite", "foo" },
    { "J_segment", "organism", "foo" },
    { "J_segment", "transcript_id", "foo" },
    { "J_segment", "serovar", "foo" },
    { "J_segment", "variety", "foo" },
    { "J_segment", "country", "foo" },
    { "J_segment", "rpt_unit_seq", "foo" },
    { "J_segment", "lab_host", "foo" },
    { "J_segment", "macronuclear", "foo" },
    { "J_segment", "cyanelle", "foo" },
    { "J_segment", "bio_material", "foo" },
    { "J_segment", "chloroplast", "foo" },
    { "J_segment", "plasmid", "foo" },
    { "J_segment", "mating_type", "foo" },
    { "J_segment", "cell_type", "foo" },
    { "J_segment", "EC_number", "1.2.3.4" },
    { "J_segment", "mol_type", "foo" },
    { "J_segment", "operon", "foo" },
    { "J_segment", "cultivar", "foo" },
    { "J_segment", "artificial_location", "foo" },
    { "J_segment", "segment", "foo" },
    { "J_segment", "cons_splice", "foo" },
    { "J_segment", "environmental_sample", "foo" },
    { "J_segment", "PCR_conditions", "foo" },
    { "J_segment", "frequency", "foo" },
    { "J_segment", "transposon", "foo" },
    { "J_segment", "haplogroup", "foo" },
    { "J_segment", "ribosomal_slippage", "foo" },
    { "J_segment", "codon_start", "foo" },
    { "J_segment", "clone", "foo" },
    { "J_segment", "gdb_xref", "foo" },
    { "J_segment", "translation", "foo" },
    { "J_segment", "transl_except", "foo" },
    { "J_segment", "bound_moiety", "foo" },
    { "J_segment", "sub_clone", "foo" },
    { "J_segment", "cell_line", "foo" },
    { "J_segment", "transgenic", "foo" },
    { "J_segment", "germline", "foo" },
    { "J_segment", "protein_id", "foo" },
    { "J_segment", "codon", "foo" },
    { "J_segment", "clone_lib", "foo" },
    { "J_segment", "PCR_primers", "foo" },
    { "J_segment", "sequenced_mol", "foo" },
    { "J_segment", "replace", "aaaaattttt" },
    { "J_segment", "strain", "foo" },
    { "J_segment", "proviral", "foo" },
    { "J_segment", "lat_lon", "foo" },
    { "J_segment", "culture_collection", "foo" },
    { "J_segment", "haplotype", "foo" },
    { "J_segment", "estimated_length", "foo" },
    { "J_segment", "tissue_lib", "foo" },
    { "J_segment", "focus", "foo" },
    { "J_segment", "dev_stage", "foo" },
    { "J_segment", "specimen_voucher", "foo" },
    { "J_segment", "isolate", "foo" },
    { "J_segment", "chromosome", "foo" },
    { "J_segment", "ncRNA_class", "foo" },
    { "J_segment", "pop_variant", "foo" },
    { "J_segment", "tissue_type", "foo" },
    { "J_segment", "trans_splicing", "foo" },
    { "J_segment", "organelle", "foo" },
    { "J_segment", "sex", "foo" },
    { "J_segment", "virion", "foo" },
    { "J_segment", "sub_species", "foo" },
    { "J_segment", "phenotype", "foo" },
    { "J_segment", "ecotype", "foo" },
    { "LTR", "compare", "AY123456.1" },
    { "LTR", "mitochondrion", "foo" },
    { "LTR", "mobile_element", "integron" },
    { "LTR", "metagenomic", "foo" },
    { "LTR", "kinetoplast", "foo" },
    { "LTR", "chromoplast", "foo" },
    { "LTR", "specific_host", "foo" },
    { "LTR", "sub_strain", "foo" },
    { "LTR", "tag_peptide", "foo" },
    { "LTR", "isolation_source", "foo" },
    { "LTR", "collected_by", "foo" },
    { "LTR", "rpt_family", "foo" },
    { "LTR", "rpt_type", "flanking" },
    { "LTR", "insertion_seq", "foo" },
    { "LTR", "transl_table", "foo" },
    { "LTR", "rearranged", "foo" },
    { "LTR", "mod_base", "foo" },
    { "LTR", "rpt_unit", "foo" },
    { "LTR", "anticodon", "foo" },
    { "LTR", "number", "foo" },
    { "LTR", "identified_by", "foo" },
    { "LTR", "collection_date", "foo" },
    { "LTR", "direction", "foo" },
    { "LTR", "rpt_unit_range", "1..3" },
    { "LTR", "serotype", "foo" },
    { "LTR", "satellite", "foo" },
    { "LTR", "organism", "foo" },
    { "LTR", "transcript_id", "foo" },
    { "LTR", "serovar", "foo" },
    { "LTR", "variety", "foo" },
    { "LTR", "country", "foo" },
    { "LTR", "rpt_unit_seq", "foo" },
    { "LTR", "lab_host", "foo" },
    { "LTR", "macronuclear", "foo" },
    { "LTR", "cyanelle", "foo" },
    { "LTR", "bio_material", "foo" },
    { "LTR", "chloroplast", "foo" },
    { "LTR", "plasmid", "foo" },
    { "LTR", "mating_type", "foo" },
    { "LTR", "cell_type", "foo" },
    { "LTR", "EC_number", "1.2.3.4" },
    { "LTR", "mol_type", "foo" },
    { "LTR", "operon", "foo" },
    { "LTR", "cultivar", "foo" },
    { "LTR", "artificial_location", "foo" },
    { "LTR", "segment", "foo" },
    { "LTR", "cons_splice", "foo" },
    { "LTR", "environmental_sample", "foo" },
    { "LTR", "PCR_conditions", "foo" },
    { "LTR", "frequency", "foo" },
    { "LTR", "transposon", "foo" },
    { "LTR", "haplogroup", "foo" },
    { "LTR", "ribosomal_slippage", "foo" },
    { "LTR", "codon_start", "foo" },
    { "LTR", "clone", "foo" },
    { "LTR", "gdb_xref", "foo" },
    { "LTR", "translation", "foo" },
    { "LTR", "transl_except", "foo" },
    { "LTR", "bound_moiety", "foo" },
    { "LTR", "sub_clone", "foo" },
    { "LTR", "cell_line", "foo" },
    { "LTR", "transgenic", "foo" },
    { "LTR", "germline", "foo" },
    { "LTR", "protein_id", "foo" },
    { "LTR", "codon", "foo" },
    { "LTR", "clone_lib", "foo" },
    { "LTR", "PCR_primers", "foo" },
    { "LTR", "sequenced_mol", "foo" },
    { "LTR", "replace", "aaaaattttt" },
    { "LTR", "strain", "foo" },
    { "LTR", "proviral", "foo" },
    { "LTR", "lat_lon", "foo" },
    { "LTR", "culture_collection", "foo" },
    { "LTR", "haplotype", "foo" },
    { "LTR", "estimated_length", "foo" },
    { "LTR", "tissue_lib", "foo" },
    { "LTR", "focus", "foo" },
    { "LTR", "dev_stage", "foo" },
    { "LTR", "specimen_voucher", "foo" },
    { "LTR", "pseudo", "foo" },
    { "LTR", "isolate", "foo" },
    { "LTR", "chromosome", "foo" },
    { "LTR", "ncRNA_class", "foo" },
    { "LTR", "pop_variant", "foo" },
    { "LTR", "tissue_type", "foo" },
    { "LTR", "trans_splicing", "foo" },
    { "LTR", "organelle", "foo" },
    { "LTR", "sex", "foo" },
    { "LTR", "virion", "foo" },
    { "LTR", "sub_species", "foo" },
    { "LTR", "phenotype", "foo" },
    { "LTR", "ecotype", "foo" },
    { "LTR", "product", "foo" },
    { "misc_binding", "compare", "AY123456.1" },
    { "misc_binding", "mitochondrion", "foo" },
    { "misc_binding", "mobile_element", "integron" },
    { "misc_binding", "metagenomic", "foo" },
    { "misc_binding", "standard_name", "foo" },
    { "misc_binding", "kinetoplast", "foo" },
    { "misc_binding", "chromoplast", "foo" },
    { "misc_binding", "specific_host", "foo" },
    { "misc_binding", "sub_strain", "foo" },
    { "misc_binding", "tag_peptide", "foo" },
    { "misc_binding", "isolation_source", "foo" },
    { "misc_binding", "collected_by", "foo" },
    { "misc_binding", "rpt_family", "foo" },
    { "misc_binding", "rpt_type", "flanking" },
    { "misc_binding", "insertion_seq", "foo" },
    { "misc_binding", "transl_table", "foo" },
    { "misc_binding", "rearranged", "foo" },
    { "misc_binding", "mod_base", "foo" },
    { "misc_binding", "rpt_unit", "foo" },
    { "misc_binding", "anticodon", "foo" },
    { "misc_binding", "number", "foo" },
    { "misc_binding", "identified_by", "foo" },
    { "misc_binding", "collection_date", "foo" },
    { "misc_binding", "direction", "foo" },
    { "misc_binding", "rpt_unit_range", "1..3" },
    { "misc_binding", "serotype", "foo" },
    { "misc_binding", "satellite", "foo" },
    { "misc_binding", "organism", "foo" },
    { "misc_binding", "transcript_id", "foo" },
    { "misc_binding", "serovar", "foo" },
    { "misc_binding", "variety", "foo" },
    { "misc_binding", "country", "foo" },
    { "misc_binding", "rpt_unit_seq", "foo" },
    { "misc_binding", "lab_host", "foo" },
    { "misc_binding", "macronuclear", "foo" },
    { "misc_binding", "cyanelle", "foo" },
    { "misc_binding", "bio_material", "foo" },
    { "misc_binding", "chloroplast", "foo" },
    { "misc_binding", "plasmid", "foo" },
    { "misc_binding", "mating_type", "foo" },
    { "misc_binding", "cell_type", "foo" },
    { "misc_binding", "EC_number", "1.2.3.4" },
    { "misc_binding", "mol_type", "foo" },
    { "misc_binding", "operon", "foo" },
    { "misc_binding", "cultivar", "foo" },
    { "misc_binding", "artificial_location", "foo" },
    { "misc_binding", "segment", "foo" },
    { "misc_binding", "cons_splice", "foo" },
    { "misc_binding", "environmental_sample", "foo" },
    { "misc_binding", "PCR_conditions", "foo" },
    { "misc_binding", "frequency", "foo" },
    { "misc_binding", "transposon", "foo" },
    { "misc_binding", "haplogroup", "foo" },
    { "misc_binding", "ribosomal_slippage", "foo" },
    { "misc_binding", "codon_start", "foo" },
    { "misc_binding", "clone", "foo" },
    { "misc_binding", "gdb_xref", "foo" },
    { "misc_binding", "translation", "foo" },
    { "misc_binding", "transl_except", "foo" },
    { "misc_binding", "sub_clone", "foo" },
    { "misc_binding", "cell_line", "foo" },
    { "misc_binding", "transgenic", "foo" },
    { "misc_binding", "germline", "foo" },
    { "misc_binding", "protein_id", "foo" },
    { "misc_binding", "codon", "foo" },
    { "misc_binding", "clone_lib", "foo" },
    { "misc_binding", "PCR_primers", "foo" },
    { "misc_binding", "sequenced_mol", "foo" },
    { "misc_binding", "replace", "aaaaattttt" },
    { "misc_binding", "strain", "foo" },
    { "misc_binding", "proviral", "foo" },
    { "misc_binding", "lat_lon", "foo" },
    { "misc_binding", "culture_collection", "foo" },
    { "misc_binding", "haplotype", "foo" },
    { "misc_binding", "estimated_length", "foo" },
    { "misc_binding", "tissue_lib", "foo" },
    { "misc_binding", "focus", "foo" },
    { "misc_binding", "dev_stage", "foo" },
    { "misc_binding", "specimen_voucher", "foo" },
    { "misc_binding", "pseudo", "foo" },
    { "misc_binding", "isolate", "foo" },
    { "misc_binding", "chromosome", "foo" },
    { "misc_binding", "ncRNA_class", "foo" },
    { "misc_binding", "pop_variant", "foo" },
    { "misc_binding", "tissue_type", "foo" },
    { "misc_binding", "trans_splicing", "foo" },
    { "misc_binding", "organelle", "foo" },
    { "misc_binding", "sex", "foo" },
    { "misc_binding", "virion", "foo" },
    { "misc_binding", "sub_species", "foo" },
    { "misc_binding", "phenotype", "foo" },
    { "misc_binding", "ecotype", "foo" },
    { "misc_binding", "product", "foo" },
    { "misc_difference", "mitochondrion", "foo" },
    { "misc_difference", "mobile_element", "integron" },
    { "misc_difference", "metagenomic", "foo" },
    { "misc_difference", "kinetoplast", "foo" },
    { "misc_difference", "chromoplast", "foo" },
    { "misc_difference", "specific_host", "foo" },
    { "misc_difference", "sub_strain", "foo" },
    { "misc_difference", "tag_peptide", "foo" },
    { "misc_difference", "isolation_source", "foo" },
    { "misc_difference", "collected_by", "foo" },
    { "misc_difference", "rpt_family", "foo" },
    { "misc_difference", "rpt_type", "flanking" },
    { "misc_difference", "insertion_seq", "foo" },
    { "misc_difference", "transl_table", "foo" },
    { "misc_difference", "rearranged", "foo" },
    { "misc_difference", "mod_base", "foo" },
    { "misc_difference", "rpt_unit", "foo" },
    { "misc_difference", "anticodon", "foo" },
    { "misc_difference", "function", "foo" },
    { "misc_difference", "number", "foo" },
    { "misc_difference", "identified_by", "foo" },
    { "misc_difference", "collection_date", "foo" },
    { "misc_difference", "direction", "foo" },
    { "misc_difference", "rpt_unit_range", "1..3" },
    { "misc_difference", "serotype", "foo" },
    { "misc_difference", "satellite", "foo" },
    { "misc_difference", "organism", "foo" },
    { "misc_difference", "transcript_id", "foo" },
    { "misc_difference", "serovar", "foo" },
    { "misc_difference", "variety", "foo" },
    { "misc_difference", "country", "foo" },
    { "misc_difference", "rpt_unit_seq", "foo" },
    { "misc_difference", "lab_host", "foo" },
    { "misc_difference", "macronuclear", "foo" },
    { "misc_difference", "cyanelle", "foo" },
    { "misc_difference", "bio_material", "foo" },
    { "misc_difference", "chloroplast", "foo" },
    { "misc_difference", "plasmid", "foo" },
    { "misc_difference", "mating_type", "foo" },
    { "misc_difference", "cell_type", "foo" },
    { "misc_difference", "EC_number", "1.2.3.4" },
    { "misc_difference", "mol_type", "foo" },
    { "misc_difference", "operon", "foo" },
    { "misc_difference", "cultivar", "foo" },
    { "misc_difference", "artificial_location", "foo" },
    { "misc_difference", "segment", "foo" },
    { "misc_difference", "cons_splice", "foo" },
    { "misc_difference", "environmental_sample", "foo" },
    { "misc_difference", "PCR_conditions", "foo" },
    { "misc_difference", "frequency", "foo" },
    { "misc_difference", "transposon", "foo" },
    { "misc_difference", "haplogroup", "foo" },
    { "misc_difference", "ribosomal_slippage", "foo" },
    { "misc_difference", "codon_start", "foo" },
    { "misc_difference", "gdb_xref", "foo" },
    { "misc_difference", "translation", "foo" },
    { "misc_difference", "transl_except", "foo" },
    { "misc_difference", "bound_moiety", "foo" },
    { "misc_difference", "sub_clone", "foo" },
    { "misc_difference", "cell_line", "foo" },
    { "misc_difference", "transgenic", "foo" },
    { "misc_difference", "germline", "foo" },
    { "misc_difference", "protein_id", "foo" },
    { "misc_difference", "codon", "foo" },
    { "misc_difference", "clone_lib", "foo" },
    { "misc_difference", "PCR_primers", "foo" },
    { "misc_difference", "sequenced_mol", "foo" },
    { "misc_difference", "strain", "foo" },
    { "misc_difference", "proviral", "foo" },
    { "misc_difference", "lat_lon", "foo" },
    { "misc_difference", "culture_collection", "foo" },
    { "misc_difference", "haplotype", "foo" },
    { "misc_difference", "estimated_length", "foo" },
    { "misc_difference", "tissue_lib", "foo" },
    { "misc_difference", "focus", "foo" },
    { "misc_difference", "dev_stage", "foo" },
    { "misc_difference", "specimen_voucher", "foo" },
    { "misc_difference", "pseudo", "foo" },
    { "misc_difference", "isolate", "foo" },
    { "misc_difference", "chromosome", "foo" },
    { "misc_difference", "ncRNA_class", "foo" },
    { "misc_difference", "pop_variant", "foo" },
    { "misc_difference", "tissue_type", "foo" },
    { "misc_difference", "trans_splicing", "foo" },
    { "misc_difference", "organelle", "foo" },
    { "misc_difference", "sex", "foo" },
    { "misc_difference", "virion", "foo" },
    { "misc_difference", "sub_species", "foo" },
    { "misc_difference", "ecotype", "foo" },
    { "misc_difference", "product", "foo" },
    { "misc_feature", "compare", "AY123456.1" },
    { "misc_feature", "mitochondrion", "foo" },
    { "misc_feature", "mobile_element", "integron" },
    { "misc_feature", "metagenomic", "foo" },
    { "misc_feature", "kinetoplast", "foo" },
    { "misc_feature", "chromoplast", "foo" },
    { "misc_feature", "specific_host", "foo" },
    { "misc_feature", "sub_strain", "foo" },
    { "misc_feature", "tag_peptide", "foo" },
    { "misc_feature", "isolation_source", "foo" },
    { "misc_feature", "collected_by", "foo" },
    { "misc_feature", "rpt_family", "foo" },
    { "misc_feature", "rpt_type", "flanking" },
    { "misc_feature", "insertion_seq", "foo" },
    { "misc_feature", "transl_table", "foo" },
    { "misc_feature", "rearranged", "foo" },
    { "misc_feature", "mod_base", "foo" },
    { "misc_feature", "rpt_unit", "foo" },
    { "misc_feature", "anticodon", "foo" },
    { "misc_feature", "identified_by", "foo" },
    { "misc_feature", "collection_date", "foo" },
    { "misc_feature", "direction", "foo" },
    { "misc_feature", "rpt_unit_range", "1..3" },
    { "misc_feature", "serotype", "foo" },
    { "misc_feature", "satellite", "foo" },
    { "misc_feature", "organism", "foo" },
    { "misc_feature", "transcript_id", "foo" },
    { "misc_feature", "serovar", "foo" },
    { "misc_feature", "variety", "foo" },
    { "misc_feature", "country", "foo" },
    { "misc_feature", "rpt_unit_seq", "foo" },
    { "misc_feature", "lab_host", "foo" },
    { "misc_feature", "macronuclear", "foo" },
    { "misc_feature", "cyanelle", "foo" },
    { "misc_feature", "bio_material", "foo" },
    { "misc_feature", "chloroplast", "foo" },
    { "misc_feature", "plasmid", "foo" },
    { "misc_feature", "mating_type", "foo" },
    { "misc_feature", "cell_type", "foo" },
    { "misc_feature", "EC_number", "1.2.3.4" },
    { "misc_feature", "mol_type", "foo" },
    { "misc_feature", "operon", "foo" },
    { "misc_feature", "cultivar", "foo" },
    { "misc_feature", "artificial_location", "foo" },
    { "misc_feature", "segment", "foo" },
    { "misc_feature", "cons_splice", "foo" },
    { "misc_feature", "environmental_sample", "foo" },
    { "misc_feature", "PCR_conditions", "foo" },
    { "misc_feature", "frequency", "foo" },
    { "misc_feature", "transposon", "foo" },
    { "misc_feature", "haplogroup", "foo" },
    { "misc_feature", "ribosomal_slippage", "foo" },
    { "misc_feature", "codon_start", "foo" },
    { "misc_feature", "clone", "foo" },
    { "misc_feature", "gdb_xref", "foo" },
    { "misc_feature", "translation", "foo" },
    { "misc_feature", "transl_except", "foo" },
    { "misc_feature", "bound_moiety", "foo" },
    { "misc_feature", "sub_clone", "foo" },
    { "misc_feature", "cell_line", "foo" },
    { "misc_feature", "transgenic", "foo" },
    { "misc_feature", "germline", "foo" },
    { "misc_feature", "protein_id", "foo" },
    { "misc_feature", "codon", "foo" },
    { "misc_feature", "clone_lib", "foo" },
    { "misc_feature", "PCR_primers", "foo" },
    { "misc_feature", "sequenced_mol", "foo" },
    { "misc_feature", "replace", "aaaaattttt" },
    { "misc_feature", "strain", "foo" },
    { "misc_feature", "proviral", "foo" },
    { "misc_feature", "lat_lon", "foo" },
    { "misc_feature", "culture_collection", "foo" },
    { "misc_feature", "haplotype", "foo" },
    { "misc_feature", "estimated_length", "foo" },
    { "misc_feature", "tissue_lib", "foo" },
    { "misc_feature", "focus", "foo" },
    { "misc_feature", "dev_stage", "foo" },
    { "misc_feature", "specimen_voucher", "foo" },
    { "misc_feature", "isolate", "foo" },
    { "misc_feature", "chromosome", "foo" },
    { "misc_feature", "ncRNA_class", "foo" },
    { "misc_feature", "pop_variant", "foo" },
    { "misc_feature", "tissue_type", "foo" },
    { "misc_feature", "trans_splicing", "foo" },
    { "misc_feature", "organelle", "foo" },
    { "misc_feature", "sex", "foo" },
    { "misc_feature", "virion", "foo" },
    { "misc_feature", "sub_species", "foo" },
    { "misc_feature", "ecotype", "foo" },
    { "misc_recomb", "compare", "AY123456.1" },
    { "misc_recomb", "mitochondrion", "foo" },
    { "misc_recomb", "mobile_element", "integron" },
    { "misc_recomb", "metagenomic", "foo" },
    { "misc_recomb", "kinetoplast", "foo" },
    { "misc_recomb", "chromoplast", "foo" },
    { "misc_recomb", "specific_host", "foo" },
    { "misc_recomb", "sub_strain", "foo" },
    { "misc_recomb", "tag_peptide", "foo" },
    { "misc_recomb", "isolation_source", "foo" },
    { "misc_recomb", "collected_by", "foo" },
    { "misc_recomb", "rpt_family", "foo" },
    { "misc_recomb", "rpt_type", "flanking" },
    { "misc_recomb", "insertion_seq", "foo" },
    { "misc_recomb", "transl_table", "foo" },
    { "misc_recomb", "rearranged", "foo" },
    { "misc_recomb", "mod_base", "foo" },
    { "misc_recomb", "rpt_unit", "foo" },
    { "misc_recomb", "anticodon", "foo" },
    { "misc_recomb", "function", "foo" },
    { "misc_recomb", "number", "foo" },
    { "misc_recomb", "identified_by", "foo" },
    { "misc_recomb", "collection_date", "foo" },
    { "misc_recomb", "direction", "foo" },
    { "misc_recomb", "rpt_unit_range", "1..3" },
    { "misc_recomb", "serotype", "foo" },
    { "misc_recomb", "satellite", "foo" },
    { "misc_recomb", "organism", "foo" },
    { "misc_recomb", "transcript_id", "foo" },
    { "misc_recomb", "serovar", "foo" },
    { "misc_recomb", "variety", "foo" },
    { "misc_recomb", "country", "foo" },
    { "misc_recomb", "rpt_unit_seq", "foo" },
    { "misc_recomb", "lab_host", "foo" },
    { "misc_recomb", "macronuclear", "foo" },
    { "misc_recomb", "cyanelle", "foo" },
    { "misc_recomb", "bio_material", "foo" },
    { "misc_recomb", "chloroplast", "foo" },
    { "misc_recomb", "plasmid", "foo" },
    { "misc_recomb", "mating_type", "foo" },
    { "misc_recomb", "cell_type", "foo" },
    { "misc_recomb", "EC_number", "1.2.3.4" },
    { "misc_recomb", "mol_type", "foo" },
    { "misc_recomb", "operon", "foo" },
    { "misc_recomb", "cultivar", "foo" },
    { "misc_recomb", "artificial_location", "foo" },
    { "misc_recomb", "segment", "foo" },
    { "misc_recomb", "cons_splice", "foo" },
    { "misc_recomb", "environmental_sample", "foo" },
    { "misc_recomb", "PCR_conditions", "foo" },
    { "misc_recomb", "frequency", "foo" },
    { "misc_recomb", "transposon", "foo" },
    { "misc_recomb", "haplogroup", "foo" },
    { "misc_recomb", "ribosomal_slippage", "foo" },
    { "misc_recomb", "codon_start", "foo" },
    { "misc_recomb", "clone", "foo" },
    { "misc_recomb", "gdb_xref", "foo" },
    { "misc_recomb", "translation", "foo" },
    { "misc_recomb", "transl_except", "foo" },
    { "misc_recomb", "bound_moiety", "foo" },
    { "misc_recomb", "sub_clone", "foo" },
    { "misc_recomb", "cell_line", "foo" },
    { "misc_recomb", "transgenic", "foo" },
    { "misc_recomb", "germline", "foo" },
    { "misc_recomb", "protein_id", "foo" },
    { "misc_recomb", "codon", "foo" },
    { "misc_recomb", "clone_lib", "foo" },
    { "misc_recomb", "PCR_primers", "foo" },
    { "misc_recomb", "sequenced_mol", "foo" },
    { "misc_recomb", "replace", "aaaaattttt" },
    { "misc_recomb", "strain", "foo" },
    { "misc_recomb", "proviral", "foo" },
    { "misc_recomb", "lat_lon", "foo" },
    { "misc_recomb", "culture_collection", "foo" },
    { "misc_recomb", "haplotype", "foo" },
    { "misc_recomb", "estimated_length", "foo" },
    { "misc_recomb", "tissue_lib", "foo" },
    { "misc_recomb", "focus", "foo" },
    { "misc_recomb", "dev_stage", "foo" },
    { "misc_recomb", "specimen_voucher", "foo" },
    { "misc_recomb", "pseudo", "foo" },
    { "misc_recomb", "isolate", "foo" },
    { "misc_recomb", "chromosome", "foo" },
    { "misc_recomb", "ncRNA_class", "foo" },
    { "misc_recomb", "pop_variant", "foo" },
    { "misc_recomb", "tissue_type", "foo" },
    { "misc_recomb", "trans_splicing", "foo" },
    { "misc_recomb", "organelle", "foo" },
    { "misc_recomb", "sex", "foo" },
    { "misc_recomb", "virion", "foo" },
    { "misc_recomb", "sub_species", "foo" },
    { "misc_recomb", "phenotype", "foo" },
    { "misc_recomb", "ecotype", "foo" },
    { "misc_recomb", "product", "foo" },
    { "misc_signal", "compare", "AY123456.1" },
    { "misc_signal", "mitochondrion", "foo" },
    { "misc_signal", "mobile_element", "integron" },
    { "misc_signal", "metagenomic", "foo" },
    { "misc_signal", "kinetoplast", "foo" },
    { "misc_signal", "chromoplast", "foo" },
    { "misc_signal", "specific_host", "foo" },
    { "misc_signal", "sub_strain", "foo" },
    { "misc_signal", "tag_peptide", "foo" },
    { "misc_signal", "isolation_source", "foo" },
    { "misc_signal", "collected_by", "foo" },
    { "misc_signal", "rpt_family", "foo" },
    { "misc_signal", "rpt_type", "flanking" },
    { "misc_signal", "insertion_seq", "foo" },
    { "misc_signal", "transl_table", "foo" },
    { "misc_signal", "rearranged", "foo" },
    { "misc_signal", "mod_base", "foo" },
    { "misc_signal", "rpt_unit", "foo" },
    { "misc_signal", "anticodon", "foo" },
    { "misc_signal", "number", "foo" },
    { "misc_signal", "identified_by", "foo" },
    { "misc_signal", "collection_date", "foo" },
    { "misc_signal", "direction", "foo" },
    { "misc_signal", "rpt_unit_range", "1..3" },
    { "misc_signal", "serotype", "foo" },
    { "misc_signal", "satellite", "foo" },
    { "misc_signal", "organism", "foo" },
    { "misc_signal", "transcript_id", "foo" },
    { "misc_signal", "serovar", "foo" },
    { "misc_signal", "variety", "foo" },
    { "misc_signal", "country", "foo" },
    { "misc_signal", "rpt_unit_seq", "foo" },
    { "misc_signal", "lab_host", "foo" },
    { "misc_signal", "macronuclear", "foo" },
    { "misc_signal", "cyanelle", "foo" },
    { "misc_signal", "bio_material", "foo" },
    { "misc_signal", "chloroplast", "foo" },
    { "misc_signal", "plasmid", "foo" },
    { "misc_signal", "mating_type", "foo" },
    { "misc_signal", "cell_type", "foo" },
    { "misc_signal", "EC_number", "1.2.3.4" },
    { "misc_signal", "mol_type", "foo" },
    { "misc_signal", "cultivar", "foo" },
    { "misc_signal", "artificial_location", "foo" },
    { "misc_signal", "segment", "foo" },
    { "misc_signal", "cons_splice", "foo" },
    { "misc_signal", "environmental_sample", "foo" },
    { "misc_signal", "PCR_conditions", "foo" },
    { "misc_signal", "frequency", "foo" },
    { "misc_signal", "transposon", "foo" },
    { "misc_signal", "haplogroup", "foo" },
    { "misc_signal", "ribosomal_slippage", "foo" },
    { "misc_signal", "codon_start", "foo" },
    { "misc_signal", "clone", "foo" },
    { "misc_signal", "gdb_xref", "foo" },
    { "misc_signal", "translation", "foo" },
    { "misc_signal", "transl_except", "foo" },
    { "misc_signal", "bound_moiety", "foo" },
    { "misc_signal", "sub_clone", "foo" },
    { "misc_signal", "cell_line", "foo" },
    { "misc_signal", "transgenic", "foo" },
    { "misc_signal", "germline", "foo" },
    { "misc_signal", "protein_id", "foo" },
    { "misc_signal", "codon", "foo" },
    { "misc_signal", "clone_lib", "foo" },
    { "misc_signal", "PCR_primers", "foo" },
    { "misc_signal", "sequenced_mol", "foo" },
    { "misc_signal", "replace", "aaaaattttt" },
    { "misc_signal", "strain", "foo" },
    { "misc_signal", "proviral", "foo" },
    { "misc_signal", "lat_lon", "foo" },
    { "misc_signal", "culture_collection", "foo" },
    { "misc_signal", "haplotype", "foo" },
    { "misc_signal", "estimated_length", "foo" },
    { "misc_signal", "tissue_lib", "foo" },
    { "misc_signal", "focus", "foo" },
    { "misc_signal", "dev_stage", "foo" },
    { "misc_signal", "specimen_voucher", "foo" },
    { "misc_signal", "pseudo", "foo" },
    { "misc_signal", "isolate", "foo" },
    { "misc_signal", "chromosome", "foo" },
    { "misc_signal", "ncRNA_class", "foo" },
    { "misc_signal", "pop_variant", "foo" },
    { "misc_signal", "tissue_type", "foo" },
    { "misc_signal", "trans_splicing", "foo" },
    { "misc_signal", "organelle", "foo" },
    { "misc_signal", "sex", "foo" },
    { "misc_signal", "virion", "foo" },
    { "misc_signal", "sub_species", "foo" },
    { "misc_signal", "ecotype", "foo" },
    { "misc_signal", "product", "foo" },
    { "misc_structure", "compare", "AY123456.1" },
    { "misc_structure", "mitochondrion", "foo" },
    { "misc_structure", "mobile_element", "integron" },
    { "misc_structure", "metagenomic", "foo" },
    { "misc_structure", "kinetoplast", "foo" },
    { "misc_structure", "chromoplast", "foo" },
    { "misc_structure", "specific_host", "foo" },
    { "misc_structure", "sub_strain", "foo" },
    { "misc_structure", "tag_peptide", "foo" },
    { "misc_structure", "isolation_source", "foo" },
    { "misc_structure", "collected_by", "foo" },
    { "misc_structure", "rpt_family", "foo" },
    { "misc_structure", "rpt_type", "flanking" },
    { "misc_structure", "insertion_seq", "foo" },
    { "misc_structure", "transl_table", "foo" },
    { "misc_structure", "rearranged", "foo" },
    { "misc_structure", "mod_base", "foo" },
    { "misc_structure", "rpt_unit", "foo" },
    { "misc_structure", "anticodon", "foo" },
    { "misc_structure", "number", "foo" },
    { "misc_structure", "identified_by", "foo" },
    { "misc_structure", "collection_date", "foo" },
    { "misc_structure", "direction", "foo" },
    { "misc_structure", "rpt_unit_range", "1..3" },
    { "misc_structure", "serotype", "foo" },
    { "misc_structure", "satellite", "foo" },
    { "misc_structure", "organism", "foo" },
    { "misc_structure", "transcript_id", "foo" },
    { "misc_structure", "serovar", "foo" },
    { "misc_structure", "variety", "foo" },
    { "misc_structure", "country", "foo" },
    { "misc_structure", "rpt_unit_seq", "foo" },
    { "misc_structure", "lab_host", "foo" },
    { "misc_structure", "macronuclear", "foo" },
    { "misc_structure", "cyanelle", "foo" },
    { "misc_structure", "bio_material", "foo" },
    { "misc_structure", "chloroplast", "foo" },
    { "misc_structure", "plasmid", "foo" },
    { "misc_structure", "mating_type", "foo" },
    { "misc_structure", "cell_type", "foo" },
    { "misc_structure", "EC_number", "1.2.3.4" },
    { "misc_structure", "mol_type", "foo" },
    { "misc_structure", "operon", "foo" },
    { "misc_structure", "cultivar", "foo" },
    { "misc_structure", "artificial_location", "foo" },
    { "misc_structure", "segment", "foo" },
    { "misc_structure", "cons_splice", "foo" },
    { "misc_structure", "environmental_sample", "foo" },
    { "misc_structure", "PCR_conditions", "foo" },
    { "misc_structure", "frequency", "foo" },
    { "misc_structure", "transposon", "foo" },
    { "misc_structure", "haplogroup", "foo" },
    { "misc_structure", "ribosomal_slippage", "foo" },
    { "misc_structure", "codon_start", "foo" },
    { "misc_structure", "clone", "foo" },
    { "misc_structure", "gdb_xref", "foo" },
    { "misc_structure", "translation", "foo" },
    { "misc_structure", "transl_except", "foo" },
    { "misc_structure", "bound_moiety", "foo" },
    { "misc_structure", "sub_clone", "foo" },
    { "misc_structure", "cell_line", "foo" },
    { "misc_structure", "transgenic", "foo" },
    { "misc_structure", "germline", "foo" },
    { "misc_structure", "protein_id", "foo" },
    { "misc_structure", "codon", "foo" },
    { "misc_structure", "clone_lib", "foo" },
    { "misc_structure", "PCR_primers", "foo" },
    { "misc_structure", "sequenced_mol", "foo" },
    { "misc_structure", "replace", "aaaaattttt" },
    { "misc_structure", "strain", "foo" },
    { "misc_structure", "proviral", "foo" },
    { "misc_structure", "lat_lon", "foo" },
    { "misc_structure", "culture_collection", "foo" },
    { "misc_structure", "haplotype", "foo" },
    { "misc_structure", "estimated_length", "foo" },
    { "misc_structure", "tissue_lib", "foo" },
    { "misc_structure", "focus", "foo" },
    { "misc_structure", "dev_stage", "foo" },
    { "misc_structure", "specimen_voucher", "foo" },
    { "misc_structure", "pseudo", "foo" },
    { "misc_structure", "isolate", "foo" },
    { "misc_structure", "chromosome", "foo" },
    { "misc_structure", "ncRNA_class", "foo" },
    { "misc_structure", "pop_variant", "foo" },
    { "misc_structure", "tissue_type", "foo" },
    { "misc_structure", "trans_splicing", "foo" },
    { "misc_structure", "organelle", "foo" },
    { "misc_structure", "sex", "foo" },
    { "misc_structure", "virion", "foo" },
    { "misc_structure", "sub_species", "foo" },
    { "misc_structure", "phenotype", "foo" },
    { "misc_structure", "ecotype", "foo" },
    { "misc_structure", "product", "foo" },
    { "modified_base", "compare", "AY123456.1" },
    { "modified_base", "mitochondrion", "foo" },
    { "modified_base", "mobile_element", "integron" },
    { "modified_base", "metagenomic", "foo" },
    { "modified_base", "standard_name", "foo" },
    { "modified_base", "kinetoplast", "foo" },
    { "modified_base", "chromoplast", "foo" },
    { "modified_base", "specific_host", "foo" },
    { "modified_base", "sub_strain", "foo" },
    { "modified_base", "tag_peptide", "foo" },
    { "modified_base", "isolation_source", "foo" },
    { "modified_base", "collected_by", "foo" },
    { "modified_base", "rpt_family", "foo" },
    { "modified_base", "rpt_type", "flanking" },
    { "modified_base", "insertion_seq", "foo" },
    { "modified_base", "transl_table", "foo" },
    { "modified_base", "rearranged", "foo" },
    { "modified_base", "rpt_unit", "foo" },
    { "modified_base", "anticodon", "foo" },
    { "modified_base", "function", "foo" },
    { "modified_base", "number", "foo" },
    { "modified_base", "identified_by", "foo" },
    { "modified_base", "collection_date", "foo" },
    { "modified_base", "direction", "foo" },
    { "modified_base", "rpt_unit_range", "1..3" },
    { "modified_base", "serotype", "foo" },
    { "modified_base", "satellite", "foo" },
    { "modified_base", "organism", "foo" },
    { "modified_base", "transcript_id", "foo" },
    { "modified_base", "serovar", "foo" },
    { "modified_base", "variety", "foo" },
    { "modified_base", "country", "foo" },
    { "modified_base", "rpt_unit_seq", "foo" },
    { "modified_base", "lab_host", "foo" },
    { "modified_base", "macronuclear", "foo" },
    { "modified_base", "cyanelle", "foo" },
    { "modified_base", "bio_material", "foo" },
    { "modified_base", "chloroplast", "foo" },
    { "modified_base", "plasmid", "foo" },
    { "modified_base", "mating_type", "foo" },
    { "modified_base", "cell_type", "foo" },
    { "modified_base", "EC_number", "1.2.3.4" },
    { "modified_base", "mol_type", "foo" },
    { "modified_base", "operon", "foo" },
    { "modified_base", "cultivar", "foo" },
    { "modified_base", "artificial_location", "foo" },
    { "modified_base", "segment", "foo" },
    { "modified_base", "cons_splice", "foo" },
    { "modified_base", "environmental_sample", "foo" },
    { "modified_base", "PCR_conditions", "foo" },
    { "modified_base", "transposon", "foo" },
    { "modified_base", "haplogroup", "foo" },
    { "modified_base", "ribosomal_slippage", "foo" },
    { "modified_base", "codon_start", "foo" },
    { "modified_base", "clone", "foo" },
    { "modified_base", "gdb_xref", "foo" },
    { "modified_base", "translation", "foo" },
    { "modified_base", "transl_except", "foo" },
    { "modified_base", "bound_moiety", "foo" },
    { "modified_base", "sub_clone", "foo" },
    { "modified_base", "cell_line", "foo" },
    { "modified_base", "transgenic", "foo" },
    { "modified_base", "germline", "foo" },
    { "modified_base", "protein_id", "foo" },
    { "modified_base", "codon", "foo" },
    { "modified_base", "clone_lib", "foo" },
    { "modified_base", "PCR_primers", "foo" },
    { "modified_base", "sequenced_mol", "foo" },
    { "modified_base", "replace", "aaaaattttt" },
    { "modified_base", "strain", "foo" },
    { "modified_base", "proviral", "foo" },
    { "modified_base", "lat_lon", "foo" },
    { "modified_base", "culture_collection", "foo" },
    { "modified_base", "haplotype", "foo" },
    { "modified_base", "estimated_length", "foo" },
    { "modified_base", "tissue_lib", "foo" },
    { "modified_base", "focus", "foo" },
    { "modified_base", "dev_stage", "foo" },
    { "modified_base", "partial", "foo" },
    { "modified_base", "specimen_voucher", "foo" },
    { "modified_base", "pseudo", "foo" },
    { "modified_base", "isolate", "foo" },
    { "modified_base", "chromosome", "foo" },
    { "modified_base", "ncRNA_class", "foo" },
    { "modified_base", "pop_variant", "foo" },
    { "modified_base", "tissue_type", "foo" },
    { "modified_base", "trans_splicing", "foo" },
    { "modified_base", "organelle", "foo" },
    { "modified_base", "sex", "foo" },
    { "modified_base", "virion", "foo" },
    { "modified_base", "sub_species", "foo" },
    { "modified_base", "phenotype", "foo" },
    { "modified_base", "ecotype", "foo" },
    { "modified_base", "product", "foo" },
    { "mutation", "compare", "AY123456.1" },
    { "mutation", "mitochondrion", "foo" },
    { "mutation", "mobile_element", "integron" },
    { "mutation", "metagenomic", "foo" },
    { "mutation", "kinetoplast", "foo" },
    { "mutation", "chromoplast", "foo" },
    { "mutation", "specific_host", "foo" },
    { "mutation", "sub_strain", "foo" },
    { "mutation", "tag_peptide", "foo" },
    { "mutation", "isolation_source", "foo" },
    { "mutation", "collected_by", "foo" },
    { "mutation", "rpt_family", "foo" },
    { "mutation", "rpt_type", "flanking" },
    { "mutation", "insertion_seq", "foo" },
    { "mutation", "transl_table", "foo" },
    { "mutation", "rearranged", "foo" },
    { "mutation", "mod_base", "foo" },
    { "mutation", "rpt_unit", "foo" },
    { "mutation", "anticodon", "foo" },
    { "mutation", "function", "foo" },
    { "mutation", "number", "foo" },
    { "mutation", "identified_by", "foo" },
    { "mutation", "collection_date", "foo" },
    { "mutation", "direction", "foo" },
    { "mutation", "rpt_unit_range", "1..3" },
    { "mutation", "serotype", "foo" },
    { "mutation", "satellite", "foo" },
    { "mutation", "organism", "foo" },
    { "mutation", "transcript_id", "foo" },
    { "mutation", "serovar", "foo" },
    { "mutation", "variety", "foo" },
    { "mutation", "country", "foo" },
    { "mutation", "rpt_unit_seq", "foo" },
    { "mutation", "lab_host", "foo" },
    { "mutation", "macronuclear", "foo" },
    { "mutation", "cyanelle", "foo" },
    { "mutation", "bio_material", "foo" },
    { "mutation", "chloroplast", "foo" },
    { "mutation", "plasmid", "foo" },
    { "mutation", "mating_type", "foo" },
    { "mutation", "cell_type", "foo" },
    { "mutation", "EC_number", "1.2.3.4" },
    { "mutation", "mol_type", "foo" },
    { "mutation", "operon", "foo" },
    { "mutation", "cultivar", "foo" },
    { "mutation", "artificial_location", "foo" },
    { "mutation", "segment", "foo" },
    { "mutation", "cons_splice", "foo" },
    { "mutation", "environmental_sample", "foo" },
    { "mutation", "PCR_conditions", "foo" },
    { "mutation", "transposon", "foo" },
    { "mutation", "haplogroup", "foo" },
    { "mutation", "ribosomal_slippage", "foo" },
    { "mutation", "codon_start", "foo" },
    { "mutation", "clone", "foo" },
    { "mutation", "gdb_xref", "foo" },
    { "mutation", "translation", "foo" },
    { "mutation", "transl_except", "foo" },
    { "mutation", "bound_moiety", "foo" },
    { "mutation", "sub_clone", "foo" },
    { "mutation", "cell_line", "foo" },
    { "mutation", "transgenic", "foo" },
    { "mutation", "germline", "foo" },
    { "mutation", "protein_id", "foo" },
    { "mutation", "codon", "foo" },
    { "mutation", "clone_lib", "foo" },
    { "mutation", "PCR_primers", "foo" },
    { "mutation", "sequenced_mol", "foo" },
    { "mutation", "strain", "foo" },
    { "mutation", "proviral", "foo" },
    { "mutation", "lat_lon", "foo" },
    { "mutation", "culture_collection", "foo" },
    { "mutation", "haplotype", "foo" },
    { "mutation", "estimated_length", "foo" },
    { "mutation", "tissue_lib", "foo" },
    { "mutation", "focus", "foo" },
    { "mutation", "dev_stage", "foo" },
    { "mutation", "partial", "foo" },
    { "mutation", "specimen_voucher", "foo" },
    { "mutation", "pseudo", "foo" },
    { "mutation", "isolate", "foo" },
    { "mutation", "chromosome", "foo" },
    { "mutation", "allele", "foo" },
    { "mutation", "ncRNA_class", "foo" },
    { "mutation", "pop_variant", "foo" },
    { "mutation", "tissue_type", "foo" },
    { "mutation", "trans_splicing", "foo" },
    { "mutation", "organelle", "foo" },
    { "mutation", "sex", "foo" },
    { "mutation", "virion", "foo" },
    { "mutation", "sub_species", "foo" },
    { "mutation", "ecotype", "foo" },
    { "N_region", "compare", "AY123456.1" },
    { "N_region", "mitochondrion", "foo" },
    { "N_region", "mobile_element", "integron" },
    { "N_region", "metagenomic", "foo" },
    { "N_region", "kinetoplast", "foo" },
    { "N_region", "chromoplast", "foo" },
    { "N_region", "specific_host", "foo" },
    { "N_region", "sub_strain", "foo" },
    { "N_region", "tag_peptide", "foo" },
    { "N_region", "isolation_source", "foo" },
    { "N_region", "collected_by", "foo" },
    { "N_region", "rpt_family", "foo" },
    { "N_region", "rpt_type", "flanking" },
    { "N_region", "insertion_seq", "foo" },
    { "N_region", "transl_table", "foo" },
    { "N_region", "rearranged", "foo" },
    { "N_region", "mod_base", "foo" },
    { "N_region", "rpt_unit", "foo" },
    { "N_region", "anticodon", "foo" },
    { "N_region", "function", "foo" },
    { "N_region", "number", "foo" },
    { "N_region", "identified_by", "foo" },
    { "N_region", "collection_date", "foo" },
    { "N_region", "direction", "foo" },
    { "N_region", "rpt_unit_range", "1..3" },
    { "N_region", "serotype", "foo" },
    { "N_region", "satellite", "foo" },
    { "N_region", "organism", "foo" },
    { "N_region", "transcript_id", "foo" },
    { "N_region", "serovar", "foo" },
    { "N_region", "variety", "foo" },
    { "N_region", "country", "foo" },
    { "N_region", "rpt_unit_seq", "foo" },
    { "N_region", "lab_host", "foo" },
    { "N_region", "macronuclear", "foo" },
    { "N_region", "cyanelle", "foo" },
    { "N_region", "bio_material", "foo" },
    { "N_region", "chloroplast", "foo" },
    { "N_region", "plasmid", "foo" },
    { "N_region", "mating_type", "foo" },
    { "N_region", "cell_type", "foo" },
    { "N_region", "EC_number", "1.2.3.4" },
    { "N_region", "mol_type", "foo" },
    { "N_region", "operon", "foo" },
    { "N_region", "cultivar", "foo" },
    { "N_region", "artificial_location", "foo" },
    { "N_region", "segment", "foo" },
    { "N_region", "cons_splice", "foo" },
    { "N_region", "environmental_sample", "foo" },
    { "N_region", "PCR_conditions", "foo" },
    { "N_region", "frequency", "foo" },
    { "N_region", "transposon", "foo" },
    { "N_region", "haplogroup", "foo" },
    { "N_region", "ribosomal_slippage", "foo" },
    { "N_region", "codon_start", "foo" },
    { "N_region", "clone", "foo" },
    { "N_region", "gdb_xref", "foo" },
    { "N_region", "translation", "foo" },
    { "N_region", "transl_except", "foo" },
    { "N_region", "bound_moiety", "foo" },
    { "N_region", "sub_clone", "foo" },
    { "N_region", "cell_line", "foo" },
    { "N_region", "transgenic", "foo" },
    { "N_region", "germline", "foo" },
    { "N_region", "protein_id", "foo" },
    { "N_region", "codon", "foo" },
    { "N_region", "clone_lib", "foo" },
    { "N_region", "PCR_primers", "foo" },
    { "N_region", "sequenced_mol", "foo" },
    { "N_region", "replace", "aaaaattttt" },
    { "N_region", "strain", "foo" },
    { "N_region", "proviral", "foo" },
    { "N_region", "lat_lon", "foo" },
    { "N_region", "culture_collection", "foo" },
    { "N_region", "haplotype", "foo" },
    { "N_region", "estimated_length", "foo" },
    { "N_region", "tissue_lib", "foo" },
    { "N_region", "focus", "foo" },
    { "N_region", "dev_stage", "foo" },
    { "N_region", "partial", "foo" },
    { "N_region", "specimen_voucher", "foo" },
    { "N_region", "isolate", "foo" },
    { "N_region", "chromosome", "foo" },
    { "N_region", "ncRNA_class", "foo" },
    { "N_region", "pop_variant", "foo" },
    { "N_region", "tissue_type", "foo" },
    { "N_region", "trans_splicing", "foo" },
    { "N_region", "organelle", "foo" },
    { "N_region", "sex", "foo" },
    { "N_region", "virion", "foo" },
    { "N_region", "sub_species", "foo" },
    { "N_region", "phenotype", "foo" },
    { "N_region", "ecotype", "foo" },
    { "old_sequence", "mitochondrion", "foo" },
    { "old_sequence", "mobile_element", "integron" },
    { "old_sequence", "metagenomic", "foo" },
    { "old_sequence", "standard_name", "foo" },
    { "old_sequence", "kinetoplast", "foo" },
    { "old_sequence", "chromoplast", "foo" },
    { "old_sequence", "specific_host", "foo" },
    { "old_sequence", "sub_strain", "foo" },
    { "old_sequence", "tag_peptide", "foo" },
    { "old_sequence", "isolation_source", "foo" },
    { "old_sequence", "collected_by", "foo" },
    { "old_sequence", "rpt_family", "foo" },
    { "old_sequence", "rpt_type", "flanking" },
    { "old_sequence", "insertion_seq", "foo" },
    { "old_sequence", "transl_table", "foo" },
    { "old_sequence", "rearranged", "foo" },
    { "old_sequence", "mod_base", "foo" },
    { "old_sequence", "rpt_unit", "foo" },
    { "old_sequence", "anticodon", "foo" },
    { "old_sequence", "function", "foo" },
    { "old_sequence", "number", "foo" },
    { "old_sequence", "identified_by", "foo" },
    { "old_sequence", "collection_date", "foo" },
    { "old_sequence", "direction", "foo" },
    { "old_sequence", "rpt_unit_range", "1..3" },
    { "old_sequence", "serotype", "foo" },
    { "old_sequence", "satellite", "foo" },
    { "old_sequence", "organism", "foo" },
    { "old_sequence", "transcript_id", "foo" },
    { "old_sequence", "serovar", "foo" },
    { "old_sequence", "variety", "foo" },
    { "old_sequence", "country", "foo" },
    { "old_sequence", "rpt_unit_seq", "foo" },
    { "old_sequence", "lab_host", "foo" },
    { "old_sequence", "macronuclear", "foo" },
    { "old_sequence", "cyanelle", "foo" },
    { "old_sequence", "bio_material", "foo" },
    { "old_sequence", "chloroplast", "foo" },
    { "old_sequence", "plasmid", "foo" },
    { "old_sequence", "mating_type", "foo" },
    { "old_sequence", "cell_type", "foo" },
    { "old_sequence", "EC_number", "1.2.3.4" },
    { "old_sequence", "mol_type", "foo" },
    { "old_sequence", "operon", "foo" },
    { "old_sequence", "cultivar", "foo" },
    { "old_sequence", "artificial_location", "foo" },
    { "old_sequence", "segment", "foo" },
    { "old_sequence", "cons_splice", "foo" },
    { "old_sequence", "environmental_sample", "foo" },
    { "old_sequence", "PCR_conditions", "foo" },
    { "old_sequence", "frequency", "foo" },
    { "old_sequence", "transposon", "foo" },
    { "old_sequence", "haplogroup", "foo" },
    { "old_sequence", "ribosomal_slippage", "foo" },
    { "old_sequence", "codon_start", "foo" },
    { "old_sequence", "clone", "foo" },
    { "old_sequence", "gdb_xref", "foo" },
    { "old_sequence", "translation", "foo" },
    { "old_sequence", "transl_except", "foo" },
    { "old_sequence", "bound_moiety", "foo" },
    { "old_sequence", "sub_clone", "foo" },
    { "old_sequence", "cell_line", "foo" },
    { "old_sequence", "transgenic", "foo" },
    { "old_sequence", "germline", "foo" },
    { "old_sequence", "protein_id", "foo" },
    { "old_sequence", "codon", "foo" },
    { "old_sequence", "clone_lib", "foo" },
    { "old_sequence", "PCR_primers", "foo" },
    { "old_sequence", "sequenced_mol", "foo" },
    { "old_sequence", "strain", "foo" },
    { "old_sequence", "proviral", "foo" },
    { "old_sequence", "lat_lon", "foo" },
    { "old_sequence", "culture_collection", "foo" },
    { "old_sequence", "haplotype", "foo" },
    { "old_sequence", "estimated_length", "foo" },
    { "old_sequence", "tissue_lib", "foo" },
    { "old_sequence", "focus", "foo" },
    { "old_sequence", "dev_stage", "foo" },
    { "old_sequence", "specimen_voucher", "foo" },
    { "old_sequence", "pseudo", "foo" },
    { "old_sequence", "isolate", "foo" },
    { "old_sequence", "chromosome", "foo" },
    { "old_sequence", "ncRNA_class", "foo" },
    { "old_sequence", "pop_variant", "foo" },
    { "old_sequence", "tissue_type", "foo" },
    { "old_sequence", "trans_splicing", "foo" },
    { "old_sequence", "organelle", "foo" },
    { "old_sequence", "sex", "foo" },
    { "old_sequence", "virion", "foo" },
    { "old_sequence", "sub_species", "foo" },
    { "old_sequence", "phenotype", "foo" },
    { "old_sequence", "ecotype", "foo" },
    { "old_sequence", "product", "foo" },
    { "operon", "compare", "AY123456.1" },
    { "operon", "mitochondrion", "foo" },
    { "operon", "mobile_element", "integron" },
    { "operon", "gene", "foo" },
    { "operon", "metagenomic", "foo" },
    { "operon", "kinetoplast", "foo" },
    { "operon", "chromoplast", "foo" },
    { "operon", "specific_host", "foo" },
    { "operon", "sub_strain", "foo" },
    { "operon", "tag_peptide", "foo" },
    { "operon", "isolation_source", "foo" },
    { "operon", "collected_by", "foo" },
    { "operon", "rpt_family", "foo" },
    { "operon", "rpt_type", "flanking" },
    { "operon", "insertion_seq", "foo" },
    { "operon", "transl_table", "foo" },
    { "operon", "rearranged", "foo" },
    { "operon", "mod_base", "foo" },
    { "operon", "rpt_unit", "foo" },
    { "operon", "anticodon", "foo" },
    { "operon", "locus_tag", "foo" },
    { "operon", "number", "foo" },
    { "operon", "identified_by", "foo" },
    { "operon", "collection_date", "foo" },
    { "operon", "direction", "foo" },
    { "operon", "rpt_unit_range", "1..3" },
    { "operon", "serotype", "foo" },
    { "operon", "satellite", "foo" },
    { "operon", "organism", "foo" },
    { "operon", "transcript_id", "foo" },
    { "operon", "serovar", "foo" },
    { "operon", "variety", "foo" },
    { "operon", "country", "foo" },
    { "operon", "rpt_unit_seq", "foo" },
    { "operon", "lab_host", "foo" },
    { "operon", "macronuclear", "foo" },
    { "operon", "cyanelle", "foo" },
    { "operon", "bio_material", "foo" },
    { "operon", "chloroplast", "foo" },
    { "operon", "plasmid", "foo" },
    { "operon", "mating_type", "foo" },
    { "operon", "cell_type", "foo" },
    { "operon", "EC_number", "1.2.3.4" },
    { "operon", "mol_type", "foo" },
    { "operon", "cultivar", "foo" },
    { "operon", "gene_synonym", "foo" },
    { "operon", "artificial_location", "foo" },
    { "operon", "segment", "foo" },
    { "operon", "cons_splice", "foo" },
    { "operon", "environmental_sample", "foo" },
    { "operon", "PCR_conditions", "foo" },
    { "operon", "frequency", "foo" },
    { "operon", "transposon", "foo" },
    { "operon", "haplogroup", "foo" },
    { "operon", "ribosomal_slippage", "foo" },
    { "operon", "codon_start", "foo" },
    { "operon", "clone", "foo" },
    { "operon", "gdb_xref", "foo" },
    { "operon", "translation", "foo" },
    { "operon", "transl_except", "foo" },
    { "operon", "bound_moiety", "foo" },
    { "operon", "sub_clone", "foo" },
    { "operon", "cell_line", "foo" },
    { "operon", "transgenic", "foo" },
    { "operon", "germline", "foo" },
    { "operon", "protein_id", "foo" },
    { "operon", "codon", "foo" },
    { "operon", "clone_lib", "foo" },
    { "operon", "PCR_primers", "foo" },
    { "operon", "sequenced_mol", "foo" },
    { "operon", "replace", "aaaaattttt" },
    { "operon", "strain", "foo" },
    { "operon", "proviral", "foo" },
    { "operon", "lat_lon", "foo" },
    { "operon", "culture_collection", "foo" },
    { "operon", "haplotype", "foo" },
    { "operon", "estimated_length", "foo" },
    { "operon", "tissue_lib", "foo" },
    { "operon", "focus", "foo" },
    { "operon", "dev_stage", "foo" },
    { "operon", "specimen_voucher", "foo" },
    { "operon", "isolate", "foo" },
    { "operon", "chromosome", "foo" },
    { "operon", "ncRNA_class", "foo" },
    { "operon", "pop_variant", "foo" },
    { "operon", "tissue_type", "foo" },
    { "operon", "trans_splicing", "foo" },
    { "operon", "organelle", "foo" },
    { "operon", "sex", "foo" },
    { "operon", "old_locus_tag", "foo" },
    { "operon", "virion", "foo" },
    { "operon", "sub_species", "foo" },
    { "operon", "ecotype", "foo" },
    { "operon", "product", "foo" },
    { "oriT", "compare", "AY123456.1" },
    { "oriT", "mitochondrion", "foo" },
    { "oriT", "mobile_element", "integron" },
    { "oriT", "metagenomic", "foo" },
    { "oriT", "kinetoplast", "foo" },
    { "oriT", "chromoplast", "foo" },
    { "oriT", "specific_host", "foo" },
    { "oriT", "sub_strain", "foo" },
    { "oriT", "tag_peptide", "foo" },
    { "oriT", "isolation_source", "foo" },
    { "oriT", "collected_by", "foo" },
    { "oriT", "insertion_seq", "foo" },
    { "oriT", "transl_table", "foo" },
    { "oriT", "rearranged", "foo" },
    { "oriT", "mod_base", "foo" },
    { "oriT", "anticodon", "foo" },
    { "oriT", "function", "foo" },
    { "oriT", "number", "foo" },
    { "oriT", "identified_by", "foo" },
    { "oriT", "collection_date", "foo" },
    { "oriT", "serotype", "foo" },
    { "oriT", "satellite", "foo" },
    { "oriT", "organism", "foo" },
    { "oriT", "transcript_id", "foo" },
    { "oriT", "serovar", "foo" },
    { "oriT", "variety", "foo" },
    { "oriT", "country", "foo" },
    { "oriT", "lab_host", "foo" },
    { "oriT", "macronuclear", "foo" },
    { "oriT", "cyanelle", "foo" },
    { "oriT", "bio_material", "foo" },
    { "oriT", "chloroplast", "foo" },
    { "oriT", "plasmid", "foo" },
    { "oriT", "mating_type", "foo" },
    { "oriT", "cell_type", "foo" },
    { "oriT", "EC_number", "1.2.3.4" },
    { "oriT", "mol_type", "foo" },
    { "oriT", "operon", "foo" },
    { "oriT", "cultivar", "foo" },
    { "oriT", "artificial_location", "foo" },
    { "oriT", "segment", "foo" },
    { "oriT", "cons_splice", "foo" },
    { "oriT", "environmental_sample", "foo" },
    { "oriT", "PCR_conditions", "foo" },
    { "oriT", "frequency", "foo" },
    { "oriT", "transposon", "foo" },
    { "oriT", "haplogroup", "foo" },
    { "oriT", "ribosomal_slippage", "foo" },
    { "oriT", "codon_start", "foo" },
    { "oriT", "clone", "foo" },
    { "oriT", "gdb_xref", "foo" },
    { "oriT", "translation", "foo" },
    { "oriT", "transl_except", "foo" },
    { "oriT", "sub_clone", "foo" },
    { "oriT", "cell_line", "foo" },
    { "oriT", "transgenic", "foo" },
    { "oriT", "germline", "foo" },
    { "oriT", "protein_id", "foo" },
    { "oriT", "codon", "foo" },
    { "oriT", "clone_lib", "foo" },
    { "oriT", "PCR_primers", "foo" },
    { "oriT", "sequenced_mol", "foo" },
    { "oriT", "replace", "aaaaattttt" },
    { "oriT", "strain", "foo" },
    { "oriT", "proviral", "foo" },
    { "oriT", "lat_lon", "foo" },
    { "oriT", "culture_collection", "foo" },
    { "oriT", "haplotype", "foo" },
    { "oriT", "estimated_length", "foo" },
    { "oriT", "tissue_lib", "foo" },
    { "oriT", "focus", "foo" },
    { "oriT", "dev_stage", "foo" },
    { "oriT", "specimen_voucher", "foo" },
    { "oriT", "pseudo", "foo" },
    { "oriT", "isolate", "foo" },
    { "oriT", "chromosome", "foo" },
    { "oriT", "ncRNA_class", "foo" },
    { "oriT", "pop_variant", "foo" },
    { "oriT", "tissue_type", "foo" },
    { "oriT", "trans_splicing", "foo" },
    { "oriT", "organelle", "foo" },
    { "oriT", "sex", "foo" },
    { "oriT", "virion", "foo" },
    { "oriT", "sub_species", "foo" },
    { "oriT", "phenotype", "foo" },
    { "oriT", "ecotype", "foo" },
    { "oriT", "product", "foo" },
    { "polyA_signal", "compare", "AY123456.1" },
    { "polyA_signal", "mitochondrion", "foo" },
    { "polyA_signal", "mobile_element", "integron" },
    { "polyA_signal", "metagenomic", "foo" },
    { "polyA_signal", "standard_name", "foo" },
    { "polyA_signal", "kinetoplast", "foo" },
    { "polyA_signal", "chromoplast", "foo" },
    { "polyA_signal", "specific_host", "foo" },
    { "polyA_signal", "sub_strain", "foo" },
    { "polyA_signal", "tag_peptide", "foo" },
    { "polyA_signal", "isolation_source", "foo" },
    { "polyA_signal", "collected_by", "foo" },
    { "polyA_signal", "rpt_family", "foo" },
    { "polyA_signal", "rpt_type", "flanking" },
    { "polyA_signal", "insertion_seq", "foo" },
    { "polyA_signal", "transl_table", "foo" },
    { "polyA_signal", "rearranged", "foo" },
    { "polyA_signal", "mod_base", "foo" },
    { "polyA_signal", "rpt_unit", "foo" },
    { "polyA_signal", "anticodon", "foo" },
    { "polyA_signal", "function", "foo" },
    { "polyA_signal", "number", "foo" },
    { "polyA_signal", "identified_by", "foo" },
    { "polyA_signal", "collection_date", "foo" },
    { "polyA_signal", "direction", "foo" },
    { "polyA_signal", "rpt_unit_range", "1..3" },
    { "polyA_signal", "serotype", "foo" },
    { "polyA_signal", "satellite", "foo" },
    { "polyA_signal", "organism", "foo" },
    { "polyA_signal", "transcript_id", "foo" },
    { "polyA_signal", "serovar", "foo" },
    { "polyA_signal", "variety", "foo" },
    { "polyA_signal", "country", "foo" },
    { "polyA_signal", "rpt_unit_seq", "foo" },
    { "polyA_signal", "lab_host", "foo" },
    { "polyA_signal", "macronuclear", "foo" },
    { "polyA_signal", "cyanelle", "foo" },
    { "polyA_signal", "bio_material", "foo" },
    { "polyA_signal", "chloroplast", "foo" },
    { "polyA_signal", "plasmid", "foo" },
    { "polyA_signal", "mating_type", "foo" },
    { "polyA_signal", "cell_type", "foo" },
    { "polyA_signal", "EC_number", "1.2.3.4" },
    { "polyA_signal", "mol_type", "foo" },
    { "polyA_signal", "operon", "foo" },
    { "polyA_signal", "cultivar", "foo" },
    { "polyA_signal", "artificial_location", "foo" },
    { "polyA_signal", "segment", "foo" },
    { "polyA_signal", "cons_splice", "foo" },
    { "polyA_signal", "environmental_sample", "foo" },
    { "polyA_signal", "PCR_conditions", "foo" },
    { "polyA_signal", "frequency", "foo" },
    { "polyA_signal", "transposon", "foo" },
    { "polyA_signal", "haplogroup", "foo" },
    { "polyA_signal", "ribosomal_slippage", "foo" },
    { "polyA_signal", "codon_start", "foo" },
    { "polyA_signal", "clone", "foo" },
    { "polyA_signal", "gdb_xref", "foo" },
    { "polyA_signal", "translation", "foo" },
    { "polyA_signal", "transl_except", "foo" },
    { "polyA_signal", "bound_moiety", "foo" },
    { "polyA_signal", "sub_clone", "foo" },
    { "polyA_signal", "cell_line", "foo" },
    { "polyA_signal", "transgenic", "foo" },
    { "polyA_signal", "germline", "foo" },
    { "polyA_signal", "protein_id", "foo" },
    { "polyA_signal", "codon", "foo" },
    { "polyA_signal", "clone_lib", "foo" },
    { "polyA_signal", "PCR_primers", "foo" },
    { "polyA_signal", "sequenced_mol", "foo" },
    { "polyA_signal", "replace", "aaaaattttt" },
    { "polyA_signal", "strain", "foo" },
    { "polyA_signal", "proviral", "foo" },
    { "polyA_signal", "lat_lon", "foo" },
    { "polyA_signal", "culture_collection", "foo" },
    { "polyA_signal", "haplotype", "foo" },
    { "polyA_signal", "estimated_length", "foo" },
    { "polyA_signal", "tissue_lib", "foo" },
    { "polyA_signal", "focus", "foo" },
    { "polyA_signal", "dev_stage", "foo" },
    { "polyA_signal", "specimen_voucher", "foo" },
    { "polyA_signal", "pseudo", "foo" },
    { "polyA_signal", "isolate", "foo" },
    { "polyA_signal", "chromosome", "foo" },
    { "polyA_signal", "ncRNA_class", "foo" },
    { "polyA_signal", "pop_variant", "foo" },
    { "polyA_signal", "tissue_type", "foo" },
    { "polyA_signal", "trans_splicing", "foo" },
    { "polyA_signal", "organelle", "foo" },
    { "polyA_signal", "sex", "foo" },
    { "polyA_signal", "virion", "foo" },
    { "polyA_signal", "sub_species", "foo" },
    { "polyA_signal", "phenotype", "foo" },
    { "polyA_signal", "ecotype", "foo" },
    { "polyA_signal", "product", "foo" },
    { "polyA_site", "compare", "AY123456.1" },
    { "polyA_site", "mitochondrion", "foo" },
    { "polyA_site", "mobile_element", "integron" },
    { "polyA_site", "metagenomic", "foo" },
    { "polyA_site", "standard_name", "foo" },
    { "polyA_site", "kinetoplast", "foo" },
    { "polyA_site", "chromoplast", "foo" },
    { "polyA_site", "specific_host", "foo" },
    { "polyA_site", "sub_strain", "foo" },
    { "polyA_site", "tag_peptide", "foo" },
    { "polyA_site", "isolation_source", "foo" },
    { "polyA_site", "collected_by", "foo" },
    { "polyA_site", "rpt_family", "foo" },
    { "polyA_site", "rpt_type", "flanking" },
    { "polyA_site", "insertion_seq", "foo" },
    { "polyA_site", "transl_table", "foo" },
    { "polyA_site", "rearranged", "foo" },
    { "polyA_site", "mod_base", "foo" },
    { "polyA_site", "rpt_unit", "foo" },
    { "polyA_site", "anticodon", "foo" },
    { "polyA_site", "function", "foo" },
    { "polyA_site", "number", "foo" },
    { "polyA_site", "identified_by", "foo" },
    { "polyA_site", "collection_date", "foo" },
    { "polyA_site", "direction", "foo" },
    { "polyA_site", "rpt_unit_range", "5..5" },
    { "polyA_site", "serotype", "foo" },
    { "polyA_site", "satellite", "foo" },
    { "polyA_site", "organism", "foo" },
    { "polyA_site", "transcript_id", "foo" },
    { "polyA_site", "serovar", "foo" },
    { "polyA_site", "variety", "foo" },
    { "polyA_site", "country", "foo" },
    { "polyA_site", "rpt_unit_seq", "foo" },
    { "polyA_site", "lab_host", "foo" },
    { "polyA_site", "macronuclear", "foo" },
    { "polyA_site", "cyanelle", "foo" },
    { "polyA_site", "bio_material", "foo" },
    { "polyA_site", "chloroplast", "foo" },
    { "polyA_site", "plasmid", "foo" },
    { "polyA_site", "mating_type", "foo" },
    { "polyA_site", "cell_type", "foo" },
    { "polyA_site", "EC_number", "1.2.3.4" },
    { "polyA_site", "mol_type", "foo" },
    { "polyA_site", "operon", "foo" },
    { "polyA_site", "cultivar", "foo" },
    { "polyA_site", "artificial_location", "foo" },
    { "polyA_site", "segment", "foo" },
    { "polyA_site", "cons_splice", "foo" },
    { "polyA_site", "environmental_sample", "foo" },
    { "polyA_site", "PCR_conditions", "foo" },
    { "polyA_site", "frequency", "foo" },
    { "polyA_site", "transposon", "foo" },
    { "polyA_site", "haplogroup", "foo" },
    { "polyA_site", "ribosomal_slippage", "foo" },
    { "polyA_site", "codon_start", "foo" },
    { "polyA_site", "clone", "foo" },
    { "polyA_site", "gdb_xref", "foo" },
    { "polyA_site", "translation", "foo" },
    { "polyA_site", "transl_except", "foo" },
    { "polyA_site", "bound_moiety", "foo" },
    { "polyA_site", "sub_clone", "foo" },
    { "polyA_site", "cell_line", "foo" },
    { "polyA_site", "transgenic", "foo" },
    { "polyA_site", "germline", "foo" },
    { "polyA_site", "protein_id", "foo" },
    { "polyA_site", "codon", "foo" },
    { "polyA_site", "clone_lib", "foo" },
    { "polyA_site", "PCR_primers", "foo" },
    { "polyA_site", "sequenced_mol", "foo" },
    { "polyA_site", "replace", "aaaaattttt" },
    { "polyA_site", "strain", "foo" },
    { "polyA_site", "proviral", "foo" },
    { "polyA_site", "lat_lon", "foo" },
    { "polyA_site", "culture_collection", "foo" },
    { "polyA_site", "haplotype", "foo" },
    { "polyA_site", "estimated_length", "foo" },
    { "polyA_site", "tissue_lib", "foo" },
    { "polyA_site", "focus", "foo" },
    { "polyA_site", "dev_stage", "foo" },
    { "polyA_site", "partial", "foo" },
    { "polyA_site", "specimen_voucher", "foo" },
    { "polyA_site", "pseudo", "foo" },
    { "polyA_site", "isolate", "foo" },
    { "polyA_site", "chromosome", "foo" },
    { "polyA_site", "ncRNA_class", "foo" },
    { "polyA_site", "pop_variant", "foo" },
    { "polyA_site", "tissue_type", "foo" },
    { "polyA_site", "trans_splicing", "foo" },
    { "polyA_site", "organelle", "foo" },
    { "polyA_site", "sex", "foo" },
    { "polyA_site", "virion", "foo" },
    { "polyA_site", "sub_species", "foo" },
    { "polyA_site", "phenotype", "foo" },
    { "polyA_site", "ecotype", "foo" },
    { "polyA_site", "product", "foo" },
    { "prim_transcript", "compare", "AY123456.1" },
    { "prim_transcript", "mitochondrion", "foo" },
    { "prim_transcript", "mobile_element", "integron" },
    { "prim_transcript", "metagenomic", "foo" },
    { "prim_transcript", "kinetoplast", "foo" },
    { "prim_transcript", "chromoplast", "foo" },
    { "prim_transcript", "specific_host", "foo" },
    { "prim_transcript", "sub_strain", "foo" },
    { "prim_transcript", "tag_peptide", "foo" },
    { "prim_transcript", "isolation_source", "foo" },
    { "prim_transcript", "collected_by", "foo" },
    { "prim_transcript", "rpt_family", "foo" },
    { "prim_transcript", "rpt_type", "flanking" },
    { "prim_transcript", "insertion_seq", "foo" },
    { "prim_transcript", "transl_table", "foo" },
    { "prim_transcript", "rearranged", "foo" },
    { "prim_transcript", "mod_base", "foo" },
    { "prim_transcript", "rpt_unit", "foo" },
    { "prim_transcript", "anticodon", "foo" },
    { "prim_transcript", "number", "foo" },
    { "prim_transcript", "identified_by", "foo" },
    { "prim_transcript", "collection_date", "foo" },
    { "prim_transcript", "direction", "foo" },
    { "prim_transcript", "rpt_unit_range", "1..3" },
    { "prim_transcript", "serotype", "foo" },
    { "prim_transcript", "satellite", "foo" },
    { "prim_transcript", "organism", "foo" },
    { "prim_transcript", "transcript_id", "foo" },
    { "prim_transcript", "serovar", "foo" },
    { "prim_transcript", "variety", "foo" },
    { "prim_transcript", "country", "foo" },
    { "prim_transcript", "rpt_unit_seq", "foo" },
    { "prim_transcript", "lab_host", "foo" },
    { "prim_transcript", "macronuclear", "foo" },
    { "prim_transcript", "cyanelle", "foo" },
    { "prim_transcript", "bio_material", "foo" },
    { "prim_transcript", "chloroplast", "foo" },
    { "prim_transcript", "plasmid", "foo" },
    { "prim_transcript", "mating_type", "foo" },
    { "prim_transcript", "cell_type", "foo" },
    { "prim_transcript", "EC_number", "1.2.3.4" },
    { "prim_transcript", "mol_type", "foo" },
    { "prim_transcript", "cultivar", "foo" },
    { "prim_transcript", "artificial_location", "foo" },
    { "prim_transcript", "segment", "foo" },
    { "prim_transcript", "cons_splice", "foo" },
    { "prim_transcript", "environmental_sample", "foo" },
    { "prim_transcript", "PCR_conditions", "foo" },
    { "prim_transcript", "frequency", "foo" },
    { "prim_transcript", "transposon", "foo" },
    { "prim_transcript", "haplogroup", "foo" },
    { "prim_transcript", "ribosomal_slippage", "foo" },
    { "prim_transcript", "codon_start", "foo" },
    { "prim_transcript", "clone", "foo" },
    { "prim_transcript", "gdb_xref", "foo" },
    { "prim_transcript", "translation", "foo" },
    { "prim_transcript", "transl_except", "foo" },
    { "prim_transcript", "bound_moiety", "foo" },
    { "prim_transcript", "sub_clone", "foo" },
    { "prim_transcript", "cell_line", "foo" },
    { "prim_transcript", "transgenic", "foo" },
    { "prim_transcript", "germline", "foo" },
    { "prim_transcript", "protein_id", "foo" },
    { "prim_transcript", "codon", "foo" },
    { "prim_transcript", "clone_lib", "foo" },
    { "prim_transcript", "PCR_primers", "foo" },
    { "prim_transcript", "sequenced_mol", "foo" },
    { "prim_transcript", "replace", "aaaaattttt" },
    { "prim_transcript", "strain", "foo" },
    { "prim_transcript", "proviral", "foo" },
    { "prim_transcript", "lat_lon", "foo" },
    { "prim_transcript", "culture_collection", "foo" },
    { "prim_transcript", "haplotype", "foo" },
    { "prim_transcript", "estimated_length", "foo" },
    { "prim_transcript", "tissue_lib", "foo" },
    { "prim_transcript", "focus", "foo" },
    { "prim_transcript", "dev_stage", "foo" },
    { "prim_transcript", "specimen_voucher", "foo" },
    { "prim_transcript", "pseudo", "foo" },
    { "prim_transcript", "isolate", "foo" },
    { "prim_transcript", "chromosome", "foo" },
    { "prim_transcript", "ncRNA_class", "foo" },
    { "prim_transcript", "pop_variant", "foo" },
    { "prim_transcript", "tissue_type", "foo" },
    { "prim_transcript", "trans_splicing", "foo" },
    { "prim_transcript", "organelle", "foo" },
    { "prim_transcript", "sex", "foo" },
    { "prim_transcript", "virion", "foo" },
    { "prim_transcript", "sub_species", "foo" },
    { "prim_transcript", "phenotype", "foo" },
    { "prim_transcript", "ecotype", "foo" },
    { "prim_transcript", "product", "foo" },
    { "primer_bind", "compare", "AY123456.1" },
    { "primer_bind", "mitochondrion", "foo" },
    { "primer_bind", "mobile_element", "integron" },
    { "primer_bind", "metagenomic", "foo" },
    { "primer_bind", "kinetoplast", "foo" },
    { "primer_bind", "chromoplast", "foo" },
    { "primer_bind", "specific_host", "foo" },
    { "primer_bind", "sub_strain", "foo" },
    { "primer_bind", "tag_peptide", "foo" },
    { "primer_bind", "isolation_source", "foo" },
    { "primer_bind", "collected_by", "foo" },
    { "primer_bind", "rpt_family", "foo" },
    { "primer_bind", "rpt_type", "flanking" },
    { "primer_bind", "insertion_seq", "foo" },
    { "primer_bind", "transl_table", "foo" },
    { "primer_bind", "rearranged", "foo" },
    { "primer_bind", "mod_base", "foo" },
    { "primer_bind", "rpt_unit", "foo" },
    { "primer_bind", "anticodon", "foo" },
    { "primer_bind", "function", "foo" },
    { "primer_bind", "number", "foo" },
    { "primer_bind", "identified_by", "foo" },
    { "primer_bind", "collection_date", "foo" },
    { "primer_bind", "direction", "foo" },
    { "primer_bind", "rpt_unit_range", "1..3" },
    { "primer_bind", "serotype", "foo" },
    { "primer_bind", "satellite", "foo" },
    { "primer_bind", "organism", "foo" },
    { "primer_bind", "transcript_id", "foo" },
    { "primer_bind", "serovar", "foo" },
    { "primer_bind", "variety", "foo" },
    { "primer_bind", "country", "foo" },
    { "primer_bind", "rpt_unit_seq", "foo" },
    { "primer_bind", "lab_host", "foo" },
    { "primer_bind", "macronuclear", "foo" },
    { "primer_bind", "cyanelle", "foo" },
    { "primer_bind", "bio_material", "foo" },
    { "primer_bind", "chloroplast", "foo" },
    { "primer_bind", "plasmid", "foo" },
    { "primer_bind", "mating_type", "foo" },
    { "primer_bind", "cell_type", "foo" },
    { "primer_bind", "EC_number", "1.2.3.4" },
    { "primer_bind", "mol_type", "foo" },
    { "primer_bind", "operon", "foo" },
    { "primer_bind", "cultivar", "foo" },
    { "primer_bind", "artificial_location", "foo" },
    { "primer_bind", "segment", "foo" },
    { "primer_bind", "cons_splice", "foo" },
    { "primer_bind", "environmental_sample", "foo" },
    { "primer_bind", "frequency", "foo" },
    { "primer_bind", "transposon", "foo" },
    { "primer_bind", "haplogroup", "foo" },
    { "primer_bind", "ribosomal_slippage", "foo" },
    { "primer_bind", "codon_start", "foo" },
    { "primer_bind", "clone", "foo" },
    { "primer_bind", "gdb_xref", "foo" },
    { "primer_bind", "translation", "foo" },
    { "primer_bind", "transl_except", "foo" },
    { "primer_bind", "bound_moiety", "foo" },
    { "primer_bind", "sub_clone", "foo" },
    { "primer_bind", "cell_line", "foo" },
    { "primer_bind", "transgenic", "foo" },
    { "primer_bind", "germline", "foo" },
    { "primer_bind", "protein_id", "foo" },
    { "primer_bind", "codon", "foo" },
    { "primer_bind", "clone_lib", "foo" },
    { "primer_bind", "PCR_primers", "foo" },
    { "primer_bind", "sequenced_mol", "foo" },
    { "primer_bind", "replace", "aaaaattttt" },
    { "primer_bind", "strain", "foo" },
    { "primer_bind", "proviral", "foo" },
    { "primer_bind", "lat_lon", "foo" },
    { "primer_bind", "culture_collection", "foo" },
    { "primer_bind", "haplotype", "foo" },
    { "primer_bind", "estimated_length", "foo" },
    { "primer_bind", "tissue_lib", "foo" },
    { "primer_bind", "focus", "foo" },
    { "primer_bind", "dev_stage", "foo" },
    { "primer_bind", "specimen_voucher", "foo" },
    { "primer_bind", "pseudo", "foo" },
    { "primer_bind", "isolate", "foo" },
    { "primer_bind", "chromosome", "foo" },
    { "primer_bind", "ncRNA_class", "foo" },
    { "primer_bind", "pop_variant", "foo" },
    { "primer_bind", "tissue_type", "foo" },
    { "primer_bind", "trans_splicing", "foo" },
    { "primer_bind", "organelle", "foo" },
    { "primer_bind", "sex", "foo" },
    { "primer_bind", "virion", "foo" },
    { "primer_bind", "sub_species", "foo" },
    { "primer_bind", "phenotype", "foo" },
    { "primer_bind", "ecotype", "foo" },
    { "primer_bind", "product", "foo" },
    { "promoter", "compare", "AY123456.1" },
    { "promoter", "mitochondrion", "foo" },
    { "promoter", "mobile_element", "integron" },
    { "promoter", "metagenomic", "foo" },
    { "promoter", "kinetoplast", "foo" },
    { "promoter", "chromoplast", "foo" },
    { "promoter", "specific_host", "foo" },
    { "promoter", "sub_strain", "foo" },
    { "promoter", "tag_peptide", "foo" },
    { "promoter", "isolation_source", "foo" },
    { "promoter", "collected_by", "foo" },
    { "promoter", "rpt_family", "foo" },
    { "promoter", "rpt_type", "flanking" },
    { "promoter", "insertion_seq", "foo" },
    { "promoter", "transl_table", "foo" },
    { "promoter", "rearranged", "foo" },
    { "promoter", "mod_base", "foo" },
    { "promoter", "rpt_unit", "foo" },
    { "promoter", "anticodon", "foo" },
    { "promoter", "number", "foo" },
    { "promoter", "identified_by", "foo" },
    { "promoter", "collection_date", "foo" },
    { "promoter", "direction", "foo" },
    { "promoter", "rpt_unit_range", "1..3" },
    { "promoter", "serotype", "foo" },
    { "promoter", "satellite", "foo" },
    { "promoter", "organism", "foo" },
    { "promoter", "transcript_id", "foo" },
    { "promoter", "serovar", "foo" },
    { "promoter", "variety", "foo" },
    { "promoter", "country", "foo" },
    { "promoter", "rpt_unit_seq", "foo" },
    { "promoter", "lab_host", "foo" },
    { "promoter", "macronuclear", "foo" },
    { "promoter", "cyanelle", "foo" },
    { "promoter", "bio_material", "foo" },
    { "promoter", "chloroplast", "foo" },
    { "promoter", "plasmid", "foo" },
    { "promoter", "mating_type", "foo" },
    { "promoter", "cell_type", "foo" },
    { "promoter", "EC_number", "1.2.3.4" },
    { "promoter", "mol_type", "foo" },
    { "promoter", "cultivar", "foo" },
    { "promoter", "artificial_location", "foo" },
    { "promoter", "segment", "foo" },
    { "promoter", "cons_splice", "foo" },
    { "promoter", "environmental_sample", "foo" },
    { "promoter", "PCR_conditions", "foo" },
    { "promoter", "frequency", "foo" },
    { "promoter", "transposon", "foo" },
    { "promoter", "haplogroup", "foo" },
    { "promoter", "ribosomal_slippage", "foo" },
    { "promoter", "codon_start", "foo" },
    { "promoter", "clone", "foo" },
    { "promoter", "gdb_xref", "foo" },
    { "promoter", "translation", "foo" },
    { "promoter", "transl_except", "foo" },
    { "promoter", "sub_clone", "foo" },
    { "promoter", "cell_line", "foo" },
    { "promoter", "transgenic", "foo" },
    { "promoter", "germline", "foo" },
    { "promoter", "protein_id", "foo" },
    { "promoter", "codon", "foo" },
    { "promoter", "clone_lib", "foo" },
    { "promoter", "PCR_primers", "foo" },
    { "promoter", "sequenced_mol", "foo" },
    { "promoter", "replace", "aaaaattttt" },
    { "promoter", "strain", "foo" },
    { "promoter", "proviral", "foo" },
    { "promoter", "lat_lon", "foo" },
    { "promoter", "culture_collection", "foo" },
    { "promoter", "haplotype", "foo" },
    { "promoter", "estimated_length", "foo" },
    { "promoter", "tissue_lib", "foo" },
    { "promoter", "focus", "foo" },
    { "promoter", "dev_stage", "foo" },
    { "promoter", "specimen_voucher", "foo" },
    { "promoter", "isolate", "foo" },
    { "promoter", "chromosome", "foo" },
    { "promoter", "ncRNA_class", "foo" },
    { "promoter", "pop_variant", "foo" },
    { "promoter", "tissue_type", "foo" },
    { "promoter", "trans_splicing", "foo" },
    { "promoter", "organelle", "foo" },
    { "promoter", "sex", "foo" },
    { "promoter", "virion", "foo" },
    { "promoter", "sub_species", "foo" },
    { "promoter", "ecotype", "foo" },
    { "promoter", "product", "foo" },
    { "protein_bind", "compare", "AY123456.1" },
    { "protein_bind", "mitochondrion", "foo" },
    { "protein_bind", "mobile_element", "integron" },
    { "protein_bind", "metagenomic", "foo" },
    { "protein_bind", "kinetoplast", "foo" },
    { "protein_bind", "chromoplast", "foo" },
    { "protein_bind", "specific_host", "foo" },
    { "protein_bind", "sub_strain", "foo" },
    { "protein_bind", "tag_peptide", "foo" },
    { "protein_bind", "isolation_source", "foo" },
    { "protein_bind", "collected_by", "foo" },
    { "protein_bind", "rpt_family", "foo" },
    { "protein_bind", "rpt_type", "flanking" },
    { "protein_bind", "insertion_seq", "foo" },
    { "protein_bind", "transl_table", "foo" },
    { "protein_bind", "rearranged", "foo" },
    { "protein_bind", "mod_base", "foo" },
    { "protein_bind", "rpt_unit", "foo" },
    { "protein_bind", "anticodon", "foo" },
    { "protein_bind", "number", "foo" },
    { "protein_bind", "identified_by", "foo" },
    { "protein_bind", "collection_date", "foo" },
    { "protein_bind", "direction", "foo" },
    { "protein_bind", "rpt_unit_range", "1..3" },
    { "protein_bind", "serotype", "foo" },
    { "protein_bind", "satellite", "foo" },
    { "protein_bind", "organism", "foo" },
    { "protein_bind", "transcript_id", "foo" },
    { "protein_bind", "serovar", "foo" },
    { "protein_bind", "variety", "foo" },
    { "protein_bind", "country", "foo" },
    { "protein_bind", "rpt_unit_seq", "foo" },
    { "protein_bind", "lab_host", "foo" },
    { "protein_bind", "macronuclear", "foo" },
    { "protein_bind", "cyanelle", "foo" },
    { "protein_bind", "bio_material", "foo" },
    { "protein_bind", "chloroplast", "foo" },
    { "protein_bind", "plasmid", "foo" },
    { "protein_bind", "mating_type", "foo" },
    { "protein_bind", "cell_type", "foo" },
    { "protein_bind", "EC_number", "1.2.3.4" },
    { "protein_bind", "mol_type", "foo" },
    { "protein_bind", "cultivar", "foo" },
    { "protein_bind", "artificial_location", "foo" },
    { "protein_bind", "segment", "foo" },
    { "protein_bind", "cons_splice", "foo" },
    { "protein_bind", "environmental_sample", "foo" },
    { "protein_bind", "PCR_conditions", "foo" },
    { "protein_bind", "frequency", "foo" },
    { "protein_bind", "transposon", "foo" },
    { "protein_bind", "haplogroup", "foo" },
    { "protein_bind", "ribosomal_slippage", "foo" },
    { "protein_bind", "codon_start", "foo" },
    { "protein_bind", "clone", "foo" },
    { "protein_bind", "gdb_xref", "foo" },
    { "protein_bind", "translation", "foo" },
    { "protein_bind", "transl_except", "foo" },
    { "protein_bind", "sub_clone", "foo" },
    { "protein_bind", "cell_line", "foo" },
    { "protein_bind", "transgenic", "foo" },
    { "protein_bind", "germline", "foo" },
    { "protein_bind", "protein_id", "foo" },
    { "protein_bind", "codon", "foo" },
    { "protein_bind", "clone_lib", "foo" },
    { "protein_bind", "PCR_primers", "foo" },
    { "protein_bind", "sequenced_mol", "foo" },
    { "protein_bind", "replace", "aaaaattttt" },
    { "protein_bind", "strain", "foo" },
    { "protein_bind", "proviral", "foo" },
    { "protein_bind", "lat_lon", "foo" },
    { "protein_bind", "culture_collection", "foo" },
    { "protein_bind", "haplotype", "foo" },
    { "protein_bind", "estimated_length", "foo" },
    { "protein_bind", "tissue_lib", "foo" },
    { "protein_bind", "focus", "foo" },
    { "protein_bind", "dev_stage", "foo" },
    { "protein_bind", "specimen_voucher", "foo" },
    { "protein_bind", "pseudo", "foo" },
    { "protein_bind", "isolate", "foo" },
    { "protein_bind", "chromosome", "foo" },
    { "protein_bind", "ncRNA_class", "foo" },
    { "protein_bind", "pop_variant", "foo" },
    { "protein_bind", "tissue_type", "foo" },
    { "protein_bind", "trans_splicing", "foo" },
    { "protein_bind", "organelle", "foo" },
    { "protein_bind", "sex", "foo" },
    { "protein_bind", "virion", "foo" },
    { "protein_bind", "sub_species", "foo" },
    { "protein_bind", "phenotype", "foo" },
    { "protein_bind", "ecotype", "foo" },
    { "protein_bind", "product", "foo" },
    { "RBS", "compare", "AY123456.1" },
    { "RBS", "mitochondrion", "foo" },
    { "RBS", "mobile_element", "integron" },
    { "RBS", "metagenomic", "foo" },
    { "RBS", "kinetoplast", "foo" },
    { "RBS", "chromoplast", "foo" },
    { "RBS", "specific_host", "foo" },
    { "RBS", "sub_strain", "foo" },
    { "RBS", "tag_peptide", "foo" },
    { "RBS", "isolation_source", "foo" },
    { "RBS", "collected_by", "foo" },
    { "RBS", "rpt_family", "foo" },
    { "RBS", "rpt_type", "flanking" },
    { "RBS", "insertion_seq", "foo" },
    { "RBS", "transl_table", "foo" },
    { "RBS", "rearranged", "foo" },
    { "RBS", "mod_base", "foo" },
    { "RBS", "rpt_unit", "foo" },
    { "RBS", "anticodon", "foo" },
    { "RBS", "function", "foo" },
    { "RBS", "number", "foo" },
    { "RBS", "identified_by", "foo" },
    { "RBS", "collection_date", "foo" },
    { "RBS", "direction", "foo" },
    { "RBS", "rpt_unit_range", "1..3" },
    { "RBS", "serotype", "foo" },
    { "RBS", "satellite", "foo" },
    { "RBS", "organism", "foo" },
    { "RBS", "transcript_id", "foo" },
    { "RBS", "serovar", "foo" },
    { "RBS", "variety", "foo" },
    { "RBS", "country", "foo" },
    { "RBS", "rpt_unit_seq", "foo" },
    { "RBS", "lab_host", "foo" },
    { "RBS", "macronuclear", "foo" },
    { "RBS", "cyanelle", "foo" },
    { "RBS", "bio_material", "foo" },
    { "RBS", "chloroplast", "foo" },
    { "RBS", "plasmid", "foo" },
    { "RBS", "mating_type", "foo" },
    { "RBS", "cell_type", "foo" },
    { "RBS", "EC_number", "1.2.3.4" },
    { "RBS", "mol_type", "foo" },
    { "RBS", "operon", "foo" },
    { "RBS", "cultivar", "foo" },
    { "RBS", "artificial_location", "foo" },
    { "RBS", "segment", "foo" },
    { "RBS", "cons_splice", "foo" },
    { "RBS", "environmental_sample", "foo" },
    { "RBS", "PCR_conditions", "foo" },
    { "RBS", "frequency", "foo" },
    { "RBS", "transposon", "foo" },
    { "RBS", "haplogroup", "foo" },
    { "RBS", "ribosomal_slippage", "foo" },
    { "RBS", "codon_start", "foo" },
    { "RBS", "clone", "foo" },
    { "RBS", "gdb_xref", "foo" },
    { "RBS", "translation", "foo" },
    { "RBS", "transl_except", "foo" },
    { "RBS", "bound_moiety", "foo" },
    { "RBS", "sub_clone", "foo" },
    { "RBS", "cell_line", "foo" },
    { "RBS", "transgenic", "foo" },
    { "RBS", "germline", "foo" },
    { "RBS", "protein_id", "foo" },
    { "RBS", "codon", "foo" },
    { "RBS", "clone_lib", "foo" },
    { "RBS", "PCR_primers", "foo" },
    { "RBS", "sequenced_mol", "foo" },
    { "RBS", "replace", "aaaaattttt" },
    { "RBS", "strain", "foo" },
    { "RBS", "proviral", "foo" },
    { "RBS", "lat_lon", "foo" },
    { "RBS", "culture_collection", "foo" },
    { "RBS", "haplotype", "foo" },
    { "RBS", "estimated_length", "foo" },
    { "RBS", "tissue_lib", "foo" },
    { "RBS", "focus", "foo" },
    { "RBS", "dev_stage", "foo" },
    { "RBS", "specimen_voucher", "foo" },
    { "RBS", "pseudo", "foo" },
    { "RBS", "isolate", "foo" },
    { "RBS", "chromosome", "foo" },
    { "RBS", "ncRNA_class", "foo" },
    { "RBS", "pop_variant", "foo" },
    { "RBS", "tissue_type", "foo" },
    { "RBS", "trans_splicing", "foo" },
    { "RBS", "organelle", "foo" },
    { "RBS", "sex", "foo" },
    { "RBS", "virion", "foo" },
    { "RBS", "sub_species", "foo" },
    { "RBS", "phenotype", "foo" },
    { "RBS", "ecotype", "foo" },
    { "RBS", "product", "foo" },
    { "repeat_region", "compare", "AY123456.1" },
    { "repeat_region", "mitochondrion", "foo" },
    { "repeat_region", "metagenomic", "foo" },
    { "repeat_region", "kinetoplast", "foo" },
    { "repeat_region", "chromoplast", "foo" },
    { "repeat_region", "specific_host", "foo" },
    { "repeat_region", "sub_strain", "foo" },
    { "repeat_region", "tag_peptide", "foo" },
    { "repeat_region", "isolation_source", "foo" },
    { "repeat_region", "collected_by", "foo" },
    { "repeat_region", "transl_table", "foo" },
    { "repeat_region", "rearranged", "foo" },
    { "repeat_region", "mod_base", "foo" },
    { "repeat_region", "anticodon", "foo" },
    { "repeat_region", "number", "foo" },
    { "repeat_region", "identified_by", "foo" },
    { "repeat_region", "collection_date", "foo" },
    { "repeat_region", "direction", "foo" },
    { "repeat_region", "serotype", "foo" },
    { "repeat_region", "organism", "foo" },
    { "repeat_region", "transcript_id", "foo" },
    { "repeat_region", "serovar", "foo" },
    { "repeat_region", "variety", "foo" },
    { "repeat_region", "country", "foo" },
    { "repeat_region", "lab_host", "foo" },
    { "repeat_region", "macronuclear", "foo" },
    { "repeat_region", "cyanelle", "foo" },
    { "repeat_region", "bio_material", "foo" },
    { "repeat_region", "chloroplast", "foo" },
    { "repeat_region", "plasmid", "foo" },
    { "repeat_region", "mating_type", "foo" },
    { "repeat_region", "cell_type", "foo" },
    { "repeat_region", "EC_number", "1.2.3.4" },
    { "repeat_region", "mol_type", "foo" },
    { "repeat_region", "operon", "foo" },
    { "repeat_region", "cultivar", "foo" },
    { "repeat_region", "artificial_location", "foo" },
    { "repeat_region", "segment", "foo" },
    { "repeat_region", "cons_splice", "foo" },
    { "repeat_region", "environmental_sample", "foo" },
    { "repeat_region", "PCR_conditions", "foo" },
    { "repeat_region", "frequency", "foo" },
    { "repeat_region", "haplogroup", "foo" },
    { "repeat_region", "ribosomal_slippage", "foo" },
    { "repeat_region", "codon_start", "foo" },
    { "repeat_region", "clone", "foo" },
    { "repeat_region", "gdb_xref", "foo" },
    { "repeat_region", "translation", "foo" },
    { "repeat_region", "transl_except", "foo" },
    { "repeat_region", "bound_moiety", "foo" },
    { "repeat_region", "sub_clone", "foo" },
    { "repeat_region", "cell_line", "foo" },
    { "repeat_region", "transgenic", "foo" },
    { "repeat_region", "germline", "foo" },
    { "repeat_region", "protein_id", "foo" },
    { "repeat_region", "codon", "foo" },
    { "repeat_region", "clone_lib", "foo" },
    { "repeat_region", "PCR_primers", "foo" },
    { "repeat_region", "sequenced_mol", "foo" },
    { "repeat_region", "replace", "aaaaattttt" },
    { "repeat_region", "strain", "foo" },
    { "repeat_region", "proviral", "foo" },
    { "repeat_region", "lat_lon", "foo" },
    { "repeat_region", "culture_collection", "foo" },
    { "repeat_region", "haplotype", "foo" },
    { "repeat_region", "estimated_length", "foo" },
    { "repeat_region", "tissue_lib", "foo" },
    { "repeat_region", "focus", "foo" },
    { "repeat_region", "dev_stage", "foo" },
    { "repeat_region", "specimen_voucher", "foo" },
    { "repeat_region", "pseudo", "foo" },
    { "repeat_region", "isolate", "foo" },
    { "repeat_region", "chromosome", "foo" },
    { "repeat_region", "ncRNA_class", "foo" },
    { "repeat_region", "pop_variant", "foo" },
    { "repeat_region", "tissue_type", "foo" },
    { "repeat_region", "trans_splicing", "foo" },
    { "repeat_region", "organelle", "foo" },
    { "repeat_region", "sex", "foo" },
    { "repeat_region", "virion", "foo" },
    { "repeat_region", "sub_species", "foo" },
    { "repeat_region", "phenotype", "foo" },
    { "repeat_region", "ecotype", "foo" },
    { "repeat_region", "product", "foo" },
    { "repeat_unit", "compare", "AY123456.1" },
    { "repeat_unit", "mitochondrion", "foo" },
    { "repeat_unit", "mobile_element", "integron" },
    { "repeat_unit", "metagenomic", "foo" },
    { "repeat_unit", "standard_name", "foo" },
    { "repeat_unit", "kinetoplast", "foo" },
    { "repeat_unit", "chromoplast", "foo" },
    { "repeat_unit", "specific_host", "foo" },
    { "repeat_unit", "sub_strain", "foo" },
    { "repeat_unit", "tag_peptide", "foo" },
    { "repeat_unit", "isolation_source", "foo" },
    { "repeat_unit", "collected_by", "foo" },
    { "repeat_unit", "insertion_seq", "foo" },
    { "repeat_unit", "transl_table", "foo" },
    { "repeat_unit", "rearranged", "foo" },
    { "repeat_unit", "mod_base", "foo" },
    { "repeat_unit", "anticodon", "foo" },
    { "repeat_unit", "number", "foo" },
    { "repeat_unit", "identified_by", "foo" },
    { "repeat_unit", "collection_date", "foo" },
    { "repeat_unit", "direction", "foo" },
    { "repeat_unit", "serotype", "foo" },
    { "repeat_unit", "satellite", "foo" },
    { "repeat_unit", "organism", "foo" },
    { "repeat_unit", "transcript_id", "foo" },
    { "repeat_unit", "serovar", "foo" },
    { "repeat_unit", "variety", "foo" },
    { "repeat_unit", "country", "foo" },
    { "repeat_unit", "lab_host", "foo" },
    { "repeat_unit", "macronuclear", "foo" },
    { "repeat_unit", "cyanelle", "foo" },
    { "repeat_unit", "bio_material", "foo" },
    { "repeat_unit", "chloroplast", "foo" },
    { "repeat_unit", "plasmid", "foo" },
    { "repeat_unit", "mating_type", "foo" },
    { "repeat_unit", "cell_type", "foo" },
    { "repeat_unit", "EC_number", "1.2.3.4" },
    { "repeat_unit", "mol_type", "foo" },
    { "repeat_unit", "operon", "foo" },
    { "repeat_unit", "cultivar", "foo" },
    { "repeat_unit", "artificial_location", "foo" },
    { "repeat_unit", "segment", "foo" },
    { "repeat_unit", "cons_splice", "foo" },
    { "repeat_unit", "environmental_sample", "foo" },
    { "repeat_unit", "PCR_conditions", "foo" },
    { "repeat_unit", "frequency", "foo" },
    { "repeat_unit", "transposon", "foo" },
    { "repeat_unit", "haplogroup", "foo" },
    { "repeat_unit", "ribosomal_slippage", "foo" },
    { "repeat_unit", "codon_start", "foo" },
    { "repeat_unit", "clone", "foo" },
    { "repeat_unit", "gdb_xref", "foo" },
    { "repeat_unit", "translation", "foo" },
    { "repeat_unit", "transl_except", "foo" },
    { "repeat_unit", "bound_moiety", "foo" },
    { "repeat_unit", "sub_clone", "foo" },
    { "repeat_unit", "cell_line", "foo" },
    { "repeat_unit", "transgenic", "foo" },
    { "repeat_unit", "germline", "foo" },
    { "repeat_unit", "protein_id", "foo" },
    { "repeat_unit", "codon", "foo" },
    { "repeat_unit", "clone_lib", "foo" },
    { "repeat_unit", "PCR_primers", "foo" },
    { "repeat_unit", "sequenced_mol", "foo" },
    { "repeat_unit", "replace", "aaaaattttt" },
    { "repeat_unit", "strain", "foo" },
    { "repeat_unit", "proviral", "foo" },
    { "repeat_unit", "lat_lon", "foo" },
    { "repeat_unit", "culture_collection", "foo" },
    { "repeat_unit", "haplotype", "foo" },
    { "repeat_unit", "estimated_length", "foo" },
    { "repeat_unit", "tissue_lib", "foo" },
    { "repeat_unit", "focus", "foo" },
    { "repeat_unit", "dev_stage", "foo" },
    { "repeat_unit", "specimen_voucher", "foo" },
    { "repeat_unit", "pseudo", "foo" },
    { "repeat_unit", "isolate", "foo" },
    { "repeat_unit", "chromosome", "foo" },
    { "repeat_unit", "ncRNA_class", "foo" },
    { "repeat_unit", "pop_variant", "foo" },
    { "repeat_unit", "tissue_type", "foo" },
    { "repeat_unit", "trans_splicing", "foo" },
    { "repeat_unit", "organelle", "foo" },
    { "repeat_unit", "sex", "foo" },
    { "repeat_unit", "virion", "foo" },
    { "repeat_unit", "sub_species", "foo" },
    { "repeat_unit", "phenotype", "foo" },
    { "repeat_unit", "ecotype", "foo" },
    { "repeat_unit", "product", "foo" },
    { "rep_origin", "compare", "AY123456.1" },
    { "rep_origin", "mitochondrion", "foo" },
    { "rep_origin", "mobile_element", "integron" },
    { "rep_origin", "metagenomic", "foo" },
    { "rep_origin", "kinetoplast", "foo" },
    { "rep_origin", "chromoplast", "foo" },
    { "rep_origin", "specific_host", "foo" },
    { "rep_origin", "sub_strain", "foo" },
    { "rep_origin", "tag_peptide", "foo" },
    { "rep_origin", "isolation_source", "foo" },
    { "rep_origin", "collected_by", "foo" },
    { "rep_origin", "rpt_family", "foo" },
    { "rep_origin", "rpt_type", "flanking" },
    { "rep_origin", "insertion_seq", "foo" },
    { "rep_origin", "transl_table", "foo" },
    { "rep_origin", "rearranged", "foo" },
    { "rep_origin", "mod_base", "foo" },
    { "rep_origin", "rpt_unit", "foo" },
    { "rep_origin", "anticodon", "foo" },
    { "rep_origin", "function", "foo" },
    { "rep_origin", "number", "foo" },
    { "rep_origin", "identified_by", "foo" },
    { "rep_origin", "collection_date", "foo" },
    { "rep_origin", "rpt_unit_range", "1..3" },
    { "rep_origin", "serotype", "foo" },
    { "rep_origin", "satellite", "foo" },
    { "rep_origin", "organism", "foo" },
    { "rep_origin", "transcript_id", "foo" },
    { "rep_origin", "serovar", "foo" },
    { "rep_origin", "variety", "foo" },
    { "rep_origin", "country", "foo" },
    { "rep_origin", "rpt_unit_seq", "foo" },
    { "rep_origin", "lab_host", "foo" },
    { "rep_origin", "macronuclear", "foo" },
    { "rep_origin", "cyanelle", "foo" },
    { "rep_origin", "bio_material", "foo" },
    { "rep_origin", "chloroplast", "foo" },
    { "rep_origin", "plasmid", "foo" },
    { "rep_origin", "mating_type", "foo" },
    { "rep_origin", "cell_type", "foo" },
    { "rep_origin", "EC_number", "1.2.3.4" },
    { "rep_origin", "mol_type", "foo" },
    { "rep_origin", "operon", "foo" },
    { "rep_origin", "cultivar", "foo" },
    { "rep_origin", "artificial_location", "foo" },
    { "rep_origin", "segment", "foo" },
    { "rep_origin", "cons_splice", "foo" },
    { "rep_origin", "environmental_sample", "foo" },
    { "rep_origin", "PCR_conditions", "foo" },
    { "rep_origin", "frequency", "foo" },
    { "rep_origin", "transposon", "foo" },
    { "rep_origin", "haplogroup", "foo" },
    { "rep_origin", "ribosomal_slippage", "foo" },
    { "rep_origin", "codon_start", "foo" },
    { "rep_origin", "clone", "foo" },
    { "rep_origin", "gdb_xref", "foo" },
    { "rep_origin", "translation", "foo" },
    { "rep_origin", "transl_except", "foo" },
    { "rep_origin", "bound_moiety", "foo" },
    { "rep_origin", "sub_clone", "foo" },
    { "rep_origin", "cell_line", "foo" },
    { "rep_origin", "transgenic", "foo" },
    { "rep_origin", "germline", "foo" },
    { "rep_origin", "protein_id", "foo" },
    { "rep_origin", "codon", "foo" },
    { "rep_origin", "clone_lib", "foo" },
    { "rep_origin", "PCR_primers", "foo" },
    { "rep_origin", "sequenced_mol", "foo" },
    { "rep_origin", "replace", "aaaaattttt" },
    { "rep_origin", "strain", "foo" },
    { "rep_origin", "proviral", "foo" },
    { "rep_origin", "lat_lon", "foo" },
    { "rep_origin", "culture_collection", "foo" },
    { "rep_origin", "haplotype", "foo" },
    { "rep_origin", "estimated_length", "foo" },
    { "rep_origin", "tissue_lib", "foo" },
    { "rep_origin", "focus", "foo" },
    { "rep_origin", "dev_stage", "foo" },
    { "rep_origin", "specimen_voucher", "foo" },
    { "rep_origin", "pseudo", "foo" },
    { "rep_origin", "isolate", "foo" },
    { "rep_origin", "chromosome", "foo" },
    { "rep_origin", "ncRNA_class", "foo" },
    { "rep_origin", "pop_variant", "foo" },
    { "rep_origin", "tissue_type", "foo" },
    { "rep_origin", "trans_splicing", "foo" },
    { "rep_origin", "organelle", "foo" },
    { "rep_origin", "sex", "foo" },
    { "rep_origin", "virion", "foo" },
    { "rep_origin", "sub_species", "foo" },
    { "rep_origin", "phenotype", "foo" },
    { "rep_origin", "ecotype", "foo" },
    { "rep_origin", "product", "foo" },
    { "S_region", "compare", "AY123456.1" },
    { "S_region", "mitochondrion", "foo" },
    { "S_region", "mobile_element", "integron" },
    { "S_region", "metagenomic", "foo" },
    { "S_region", "kinetoplast", "foo" },
    { "S_region", "chromoplast", "foo" },
    { "S_region", "specific_host", "foo" },
    { "S_region", "sub_strain", "foo" },
    { "S_region", "tag_peptide", "foo" },
    { "S_region", "isolation_source", "foo" },
    { "S_region", "collected_by", "foo" },
    { "S_region", "rpt_family", "foo" },
    { "S_region", "rpt_type", "flanking" },
    { "S_region", "insertion_seq", "foo" },
    { "S_region", "transl_table", "foo" },
    { "S_region", "rearranged", "foo" },
    { "S_region", "mod_base", "foo" },
    { "S_region", "rpt_unit", "foo" },
    { "S_region", "anticodon", "foo" },
    { "S_region", "function", "foo" },
    { "S_region", "number", "foo" },
    { "S_region", "identified_by", "foo" },
    { "S_region", "collection_date", "foo" },
    { "S_region", "direction", "foo" },
    { "S_region", "rpt_unit_range", "1..3" },
    { "S_region", "serotype", "foo" },
    { "S_region", "satellite", "foo" },
    { "S_region", "organism", "foo" },
    { "S_region", "transcript_id", "foo" },
    { "S_region", "serovar", "foo" },
    { "S_region", "variety", "foo" },
    { "S_region", "country", "foo" },
    { "S_region", "rpt_unit_seq", "foo" },
    { "S_region", "lab_host", "foo" },
    { "S_region", "macronuclear", "foo" },
    { "S_region", "cyanelle", "foo" },
    { "S_region", "bio_material", "foo" },
    { "S_region", "chloroplast", "foo" },
    { "S_region", "plasmid", "foo" },
    { "S_region", "mating_type", "foo" },
    { "S_region", "cell_type", "foo" },
    { "S_region", "EC_number", "1.2.3.4" },
    { "S_region", "mol_type", "foo" },
    { "S_region", "operon", "foo" },
    { "S_region", "cultivar", "foo" },
    { "S_region", "artificial_location", "foo" },
    { "S_region", "segment", "foo" },
    { "S_region", "cons_splice", "foo" },
    { "S_region", "environmental_sample", "foo" },
    { "S_region", "PCR_conditions", "foo" },
    { "S_region", "frequency", "foo" },
    { "S_region", "transposon", "foo" },
    { "S_region", "haplogroup", "foo" },
    { "S_region", "ribosomal_slippage", "foo" },
    { "S_region", "codon_start", "foo" },
    { "S_region", "clone", "foo" },
    { "S_region", "gdb_xref", "foo" },
    { "S_region", "translation", "foo" },
    { "S_region", "transl_except", "foo" },
    { "S_region", "bound_moiety", "foo" },
    { "S_region", "sub_clone", "foo" },
    { "S_region", "cell_line", "foo" },
    { "S_region", "transgenic", "foo" },
    { "S_region", "germline", "foo" },
    { "S_region", "protein_id", "foo" },
    { "S_region", "codon", "foo" },
    { "S_region", "clone_lib", "foo" },
    { "S_region", "PCR_primers", "foo" },
    { "S_region", "sequenced_mol", "foo" },
    { "S_region", "replace", "aaaaattttt" },
    { "S_region", "strain", "foo" },
    { "S_region", "proviral", "foo" },
    { "S_region", "lat_lon", "foo" },
    { "S_region", "culture_collection", "foo" },
    { "S_region", "haplotype", "foo" },
    { "S_region", "estimated_length", "foo" },
    { "S_region", "tissue_lib", "foo" },
    { "S_region", "focus", "foo" },
    { "S_region", "dev_stage", "foo" },
    { "S_region", "specimen_voucher", "foo" },
    { "S_region", "isolate", "foo" },
    { "S_region", "chromosome", "foo" },
    { "S_region", "ncRNA_class", "foo" },
    { "S_region", "pop_variant", "foo" },
    { "S_region", "tissue_type", "foo" },
    { "S_region", "trans_splicing", "foo" },
    { "S_region", "organelle", "foo" },
    { "S_region", "sex", "foo" },
    { "S_region", "virion", "foo" },
    { "S_region", "sub_species", "foo" },
    { "S_region", "phenotype", "foo" },
    { "S_region", "ecotype", "foo" },
    { "satellite", "compare", "AY123456.1" },
    { "satellite", "mitochondrion", "foo" },
    { "satellite", "mobile_element", "integron" },
    { "satellite", "metagenomic", "foo" },
    { "satellite", "kinetoplast", "foo" },
    { "satellite", "chromoplast", "foo" },
    { "satellite", "specific_host", "foo" },
    { "satellite", "sub_strain", "foo" },
    { "satellite", "tag_peptide", "foo" },
    { "satellite", "isolation_source", "foo" },
    { "satellite", "collected_by", "foo" },
    { "satellite", "insertion_seq", "foo" },
    { "satellite", "transl_table", "foo" },
    { "satellite", "rearranged", "foo" },
    { "satellite", "mod_base", "foo" },
    { "satellite", "anticodon", "foo" },
    { "satellite", "function", "foo" },
    { "satellite", "number", "foo" },
    { "satellite", "identified_by", "foo" },
    { "satellite", "collection_date", "foo" },
    { "satellite", "direction", "foo" },
    { "satellite", "serotype", "foo" },
    { "satellite", "satellite", "foo" },
    { "satellite", "organism", "foo" },
    { "satellite", "transcript_id", "foo" },
    { "satellite", "serovar", "foo" },
    { "satellite", "variety", "foo" },
    { "satellite", "country", "foo" },
    { "satellite", "lab_host", "foo" },
    { "satellite", "macronuclear", "foo" },
    { "satellite", "cyanelle", "foo" },
    { "satellite", "bio_material", "foo" },
    { "satellite", "chloroplast", "foo" },
    { "satellite", "plasmid", "foo" },
    { "satellite", "mating_type", "foo" },
    { "satellite", "cell_type", "foo" },
    { "satellite", "EC_number", "1.2.3.4" },
    { "satellite", "mol_type", "foo" },
    { "satellite", "operon", "foo" },
    { "satellite", "cultivar", "foo" },
    { "satellite", "artificial_location", "foo" },
    { "satellite", "segment", "foo" },
    { "satellite", "cons_splice", "foo" },
    { "satellite", "environmental_sample", "foo" },
    { "satellite", "PCR_conditions", "foo" },
    { "satellite", "frequency", "foo" },
    { "satellite", "transposon", "foo" },
    { "satellite", "haplogroup", "foo" },
    { "satellite", "ribosomal_slippage", "foo" },
    { "satellite", "codon_start", "foo" },
    { "satellite", "clone", "foo" },
    { "satellite", "gdb_xref", "foo" },
    { "satellite", "translation", "foo" },
    { "satellite", "transl_except", "foo" },
    { "satellite", "bound_moiety", "foo" },
    { "satellite", "sub_clone", "foo" },
    { "satellite", "cell_line", "foo" },
    { "satellite", "transgenic", "foo" },
    { "satellite", "germline", "foo" },
    { "satellite", "protein_id", "foo" },
    { "satellite", "codon", "foo" },
    { "satellite", "clone_lib", "foo" },
    { "satellite", "PCR_primers", "foo" },
    { "satellite", "sequenced_mol", "foo" },
    { "satellite", "replace", "aaaaattttt" },
    { "satellite", "strain", "foo" },
    { "satellite", "proviral", "foo" },
    { "satellite", "lat_lon", "foo" },
    { "satellite", "culture_collection", "foo" },
    { "satellite", "haplotype", "foo" },
    { "satellite", "estimated_length", "foo" },
    { "satellite", "tissue_lib", "foo" },
    { "satellite", "focus", "foo" },
    { "satellite", "dev_stage", "foo" },
    { "satellite", "specimen_voucher", "foo" },
    { "satellite", "pseudo", "foo" },
    { "satellite", "isolate", "foo" },
    { "satellite", "chromosome", "foo" },
    { "satellite", "ncRNA_class", "foo" },
    { "satellite", "pop_variant", "foo" },
    { "satellite", "tissue_type", "foo" },
    { "satellite", "trans_splicing", "foo" },
    { "satellite", "organelle", "foo" },
    { "satellite", "sex", "foo" },
    { "satellite", "virion", "foo" },
    { "satellite", "sub_species", "foo" },
    { "satellite", "phenotype", "foo" },
    { "satellite", "ecotype", "foo" },
    { "satellite", "product", "foo" },
    { "source", "compare", "AY123456.1" },
    { "source", "mobile_element", "integron" },
    { "source", "gene", "foo" },
    { "source", "standard_name", "foo" },
    { "source", "tag_peptide", "foo" },
    { "source", "rpt_family", "foo" },
    { "source", "rpt_type", "flanking" },
    { "source", "transl_table", "foo" },
    { "source", "mod_base", "foo" },
    { "source", "rpt_unit", "foo" },
    { "source", "anticodon", "foo" },
    { "source", "locus_tag", "foo" },
    { "source", "function", "foo" },
    { "source", "number", "foo" },
    { "source", "direction", "foo" },
    { "source", "rpt_unit_range", "1..3" },
    { "source", "satellite", "foo" },
    { "source", "transcript_id", "foo" },
    { "source", "rpt_unit_seq", "foo" },
    { "source", "EC_number", "1.2.3.4" },
    { "source", "operon", "foo" },
    { "source", "experiment", "foo" },
    { "source", "gene_synonym", "foo" },
    { "source", "artificial_location", "foo" },
    { "source", "cons_splice", "foo" },
    { "source", "PCR_conditions", "foo" },
    { "source", "ribosomal_slippage", "foo" },
    { "source", "inference", "similar to DNA sequence:INSD:AY123456.1" },
    { "source", "codon_start", "foo" },
    { "source", "gdb_xref", "foo" },
    { "source", "translation", "foo" },
    { "source", "transl_except", "foo" },
    { "source", "bound_moiety", "foo" },
    { "source", "protein_id", "foo" },
    { "source", "codon", "foo" },
    { "source", "replace", "aaaaattttt" },
    { "source", "estimated_length", "foo" },
    { "source", "partial", "foo" },
    { "source", "pseudo", "foo" },
    { "source", "evidence", "foo" },
    { "source", "allele", "foo" },
    { "source", "ncRNA_class", "foo" },
    { "source", "trans_splicing", "foo" },
    { "source", "old_locus_tag", "foo" },
    { "source", "phenotype", "foo" },
    { "source", "product", "foo" },
    { "stem_loop", "compare", "AY123456.1" },
    { "stem_loop", "mitochondrion", "foo" },
    { "stem_loop", "mobile_element", "integron" },
    { "stem_loop", "metagenomic", "foo" },
    { "stem_loop", "kinetoplast", "foo" },
    { "stem_loop", "chromoplast", "foo" },
    { "stem_loop", "specific_host", "foo" },
    { "stem_loop", "sub_strain", "foo" },
    { "stem_loop", "tag_peptide", "foo" },
    { "stem_loop", "isolation_source", "foo" },
    { "stem_loop", "collected_by", "foo" },
    { "stem_loop", "rpt_family", "foo" },
    { "stem_loop", "rpt_type", "flanking" },
    { "stem_loop", "insertion_seq", "foo" },
    { "stem_loop", "transl_table", "foo" },
    { "stem_loop", "rearranged", "foo" },
    { "stem_loop", "mod_base", "foo" },
    { "stem_loop", "rpt_unit", "foo" },
    { "stem_loop", "anticodon", "foo" },
    { "stem_loop", "number", "foo" },
    { "stem_loop", "identified_by", "foo" },
    { "stem_loop", "collection_date", "foo" },
    { "stem_loop", "direction", "foo" },
    { "stem_loop", "rpt_unit_range", "1..3" },
    { "stem_loop", "serotype", "foo" },
    { "stem_loop", "satellite", "foo" },
    { "stem_loop", "organism", "foo" },
    { "stem_loop", "transcript_id", "foo" },
    { "stem_loop", "serovar", "foo" },
    { "stem_loop", "variety", "foo" },
    { "stem_loop", "country", "foo" },
    { "stem_loop", "rpt_unit_seq", "foo" },
    { "stem_loop", "lab_host", "foo" },
    { "stem_loop", "macronuclear", "foo" },
    { "stem_loop", "cyanelle", "foo" },
    { "stem_loop", "bio_material", "foo" },
    { "stem_loop", "chloroplast", "foo" },
    { "stem_loop", "plasmid", "foo" },
    { "stem_loop", "mating_type", "foo" },
    { "stem_loop", "cell_type", "foo" },
    { "stem_loop", "EC_number", "1.2.3.4" },
    { "stem_loop", "mol_type", "foo" },
    { "stem_loop", "cultivar", "foo" },
    { "stem_loop", "artificial_location", "foo" },
    { "stem_loop", "segment", "foo" },
    { "stem_loop", "cons_splice", "foo" },
    { "stem_loop", "environmental_sample", "foo" },
    { "stem_loop", "PCR_conditions", "foo" },
    { "stem_loop", "frequency", "foo" },
    { "stem_loop", "transposon", "foo" },
    { "stem_loop", "haplogroup", "foo" },
    { "stem_loop", "ribosomal_slippage", "foo" },
    { "stem_loop", "codon_start", "foo" },
    { "stem_loop", "clone", "foo" },
    { "stem_loop", "gdb_xref", "foo" },
    { "stem_loop", "translation", "foo" },
    { "stem_loop", "transl_except", "foo" },
    { "stem_loop", "bound_moiety", "foo" },
    { "stem_loop", "sub_clone", "foo" },
    { "stem_loop", "cell_line", "foo" },
    { "stem_loop", "transgenic", "foo" },
    { "stem_loop", "germline", "foo" },
    { "stem_loop", "protein_id", "foo" },
    { "stem_loop", "codon", "foo" },
    { "stem_loop", "clone_lib", "foo" },
    { "stem_loop", "PCR_primers", "foo" },
    { "stem_loop", "sequenced_mol", "foo" },
    { "stem_loop", "replace", "aaaaattttt" },
    { "stem_loop", "strain", "foo" },
    { "stem_loop", "proviral", "foo" },
    { "stem_loop", "lat_lon", "foo" },
    { "stem_loop", "culture_collection", "foo" },
    { "stem_loop", "haplotype", "foo" },
    { "stem_loop", "estimated_length", "foo" },
    { "stem_loop", "tissue_lib", "foo" },
    { "stem_loop", "focus", "foo" },
    { "stem_loop", "dev_stage", "foo" },
    { "stem_loop", "specimen_voucher", "foo" },
    { "stem_loop", "pseudo", "foo" },
    { "stem_loop", "isolate", "foo" },
    { "stem_loop", "chromosome", "foo" },
    { "stem_loop", "ncRNA_class", "foo" },
    { "stem_loop", "pop_variant", "foo" },
    { "stem_loop", "tissue_type", "foo" },
    { "stem_loop", "trans_splicing", "foo" },
    { "stem_loop", "organelle", "foo" },
    { "stem_loop", "sex", "foo" },
    { "stem_loop", "virion", "foo" },
    { "stem_loop", "sub_species", "foo" },
    { "stem_loop", "phenotype", "foo" },
    { "stem_loop", "ecotype", "foo" },
    { "stem_loop", "product", "foo" },
    { "STS", "compare", "AY123456.1" },
    { "STS", "mitochondrion", "foo" },
    { "STS", "mobile_element", "integron" },
    { "STS", "metagenomic", "foo" },
    { "STS", "kinetoplast", "foo" },
    { "STS", "chromoplast", "foo" },
    { "STS", "specific_host", "foo" },
    { "STS", "sub_strain", "foo" },
    { "STS", "tag_peptide", "foo" },
    { "STS", "isolation_source", "foo" },
    { "STS", "collected_by", "foo" },
    { "STS", "rpt_family", "foo" },
    { "STS", "rpt_type", "flanking" },
    { "STS", "insertion_seq", "foo" },
    { "STS", "transl_table", "foo" },
    { "STS", "rearranged", "foo" },
    { "STS", "mod_base", "foo" },
    { "STS", "rpt_unit", "foo" },
    { "STS", "anticodon", "foo" },
    { "STS", "function", "foo" },
    { "STS", "number", "foo" },
    { "STS", "identified_by", "foo" },
    { "STS", "collection_date", "foo" },
    { "STS", "direction", "foo" },
    { "STS", "rpt_unit_range", "1..3" },
    { "STS", "serotype", "foo" },
    { "STS", "satellite", "foo" },
    { "STS", "organism", "foo" },
    { "STS", "transcript_id", "foo" },
    { "STS", "serovar", "foo" },
    { "STS", "variety", "foo" },
    { "STS", "country", "foo" },
    { "STS", "rpt_unit_seq", "foo" },
    { "STS", "lab_host", "foo" },
    { "STS", "macronuclear", "foo" },
    { "STS", "cyanelle", "foo" },
    { "STS", "bio_material", "foo" },
    { "STS", "chloroplast", "foo" },
    { "STS", "plasmid", "foo" },
    { "STS", "mating_type", "foo" },
    { "STS", "cell_type", "foo" },
    { "STS", "EC_number", "1.2.3.4" },
    { "STS", "mol_type", "foo" },
    { "STS", "operon", "foo" },
    { "STS", "cultivar", "foo" },
    { "STS", "artificial_location", "foo" },
    { "STS", "segment", "foo" },
    { "STS", "cons_splice", "foo" },
    { "STS", "environmental_sample", "foo" },
    { "STS", "PCR_conditions", "foo" },
    { "STS", "frequency", "foo" },
    { "STS", "transposon", "foo" },
    { "STS", "haplogroup", "foo" },
    { "STS", "ribosomal_slippage", "foo" },
    { "STS", "codon_start", "foo" },
    { "STS", "clone", "foo" },
    { "STS", "gdb_xref", "foo" },
    { "STS", "translation", "foo" },
    { "STS", "transl_except", "foo" },
    { "STS", "bound_moiety", "foo" },
    { "STS", "sub_clone", "foo" },
    { "STS", "cell_line", "foo" },
    { "STS", "transgenic", "foo" },
    { "STS", "germline", "foo" },
    { "STS", "protein_id", "foo" },
    { "STS", "codon", "foo" },
    { "STS", "clone_lib", "foo" },
    { "STS", "PCR_primers", "foo" },
    { "STS", "sequenced_mol", "foo" },
    { "STS", "replace", "aaaaattttt" },
    { "STS", "strain", "foo" },
    { "STS", "proviral", "foo" },
    { "STS", "lat_lon", "foo" },
    { "STS", "culture_collection", "foo" },
    { "STS", "haplotype", "foo" },
    { "STS", "estimated_length", "foo" },
    { "STS", "tissue_lib", "foo" },
    { "STS", "focus", "foo" },
    { "STS", "dev_stage", "foo" },
    { "STS", "specimen_voucher", "foo" },
    { "STS", "pseudo", "foo" },
    { "STS", "isolate", "foo" },
    { "STS", "chromosome", "foo" },
    { "STS", "ncRNA_class", "foo" },
    { "STS", "pop_variant", "foo" },
    { "STS", "tissue_type", "foo" },
    { "STS", "trans_splicing", "foo" },
    { "STS", "organelle", "foo" },
    { "STS", "sex", "foo" },
    { "STS", "virion", "foo" },
    { "STS", "sub_species", "foo" },
    { "STS", "phenotype", "foo" },
    { "STS", "ecotype", "foo" },
    { "STS", "product", "foo" },
    { "TATA_signal", "compare", "AY123456.1" },
    { "TATA_signal", "mitochondrion", "foo" },
    { "TATA_signal", "mobile_element", "integron" },
    { "TATA_signal", "metagenomic", "foo" },
    { "TATA_signal", "standard_name", "foo" },
    { "TATA_signal", "kinetoplast", "foo" },
    { "TATA_signal", "chromoplast", "foo" },
    { "TATA_signal", "specific_host", "foo" },
    { "TATA_signal", "sub_strain", "foo" },
    { "TATA_signal", "tag_peptide", "foo" },
    { "TATA_signal", "isolation_source", "foo" },
    { "TATA_signal", "collected_by", "foo" },
    { "TATA_signal", "rpt_family", "foo" },
    { "TATA_signal", "rpt_type", "flanking" },
    { "TATA_signal", "insertion_seq", "foo" },
    { "TATA_signal", "transl_table", "foo" },
    { "TATA_signal", "rearranged", "foo" },
    { "TATA_signal", "mod_base", "foo" },
    { "TATA_signal", "rpt_unit", "foo" },
    { "TATA_signal", "anticodon", "foo" },
    { "TATA_signal", "function", "foo" },
    { "TATA_signal", "number", "foo" },
    { "TATA_signal", "identified_by", "foo" },
    { "TATA_signal", "collection_date", "foo" },
    { "TATA_signal", "direction", "foo" },
    { "TATA_signal", "rpt_unit_range", "1..3" },
    { "TATA_signal", "serotype", "foo" },
    { "TATA_signal", "satellite", "foo" },
    { "TATA_signal", "organism", "foo" },
    { "TATA_signal", "transcript_id", "foo" },
    { "TATA_signal", "serovar", "foo" },
    { "TATA_signal", "variety", "foo" },
    { "TATA_signal", "country", "foo" },
    { "TATA_signal", "rpt_unit_seq", "foo" },
    { "TATA_signal", "lab_host", "foo" },
    { "TATA_signal", "macronuclear", "foo" },
    { "TATA_signal", "cyanelle", "foo" },
    { "TATA_signal", "bio_material", "foo" },
    { "TATA_signal", "chloroplast", "foo" },
    { "TATA_signal", "plasmid", "foo" },
    { "TATA_signal", "mating_type", "foo" },
    { "TATA_signal", "cell_type", "foo" },
    { "TATA_signal", "EC_number", "1.2.3.4" },
    { "TATA_signal", "mol_type", "foo" },
    { "TATA_signal", "operon", "foo" },
    { "TATA_signal", "cultivar", "foo" },
    { "TATA_signal", "artificial_location", "foo" },
    { "TATA_signal", "segment", "foo" },
    { "TATA_signal", "cons_splice", "foo" },
    { "TATA_signal", "environmental_sample", "foo" },
    { "TATA_signal", "PCR_conditions", "foo" },
    { "TATA_signal", "frequency", "foo" },
    { "TATA_signal", "transposon", "foo" },
    { "TATA_signal", "haplogroup", "foo" },
    { "TATA_signal", "ribosomal_slippage", "foo" },
    { "TATA_signal", "codon_start", "foo" },
    { "TATA_signal", "clone", "foo" },
    { "TATA_signal", "gdb_xref", "foo" },
    { "TATA_signal", "translation", "foo" },
    { "TATA_signal", "transl_except", "foo" },
    { "TATA_signal", "bound_moiety", "foo" },
    { "TATA_signal", "sub_clone", "foo" },
    { "TATA_signal", "cell_line", "foo" },
    { "TATA_signal", "transgenic", "foo" },
    { "TATA_signal", "germline", "foo" },
    { "TATA_signal", "protein_id", "foo" },
    { "TATA_signal", "codon", "foo" },
    { "TATA_signal", "clone_lib", "foo" },
    { "TATA_signal", "PCR_primers", "foo" },
    { "TATA_signal", "sequenced_mol", "foo" },
    { "TATA_signal", "replace", "aaaaattttt" },
    { "TATA_signal", "strain", "foo" },
    { "TATA_signal", "proviral", "foo" },
    { "TATA_signal", "lat_lon", "foo" },
    { "TATA_signal", "culture_collection", "foo" },
    { "TATA_signal", "haplotype", "foo" },
    { "TATA_signal", "estimated_length", "foo" },
    { "TATA_signal", "tissue_lib", "foo" },
    { "TATA_signal", "focus", "foo" },
    { "TATA_signal", "dev_stage", "foo" },
    { "TATA_signal", "specimen_voucher", "foo" },
    { "TATA_signal", "pseudo", "foo" },
    { "TATA_signal", "isolate", "foo" },
    { "TATA_signal", "chromosome", "foo" },
    { "TATA_signal", "ncRNA_class", "foo" },
    { "TATA_signal", "pop_variant", "foo" },
    { "TATA_signal", "tissue_type", "foo" },
    { "TATA_signal", "trans_splicing", "foo" },
    { "TATA_signal", "organelle", "foo" },
    { "TATA_signal", "sex", "foo" },
    { "TATA_signal", "virion", "foo" },
    { "TATA_signal", "sub_species", "foo" },
    { "TATA_signal", "phenotype", "foo" },
    { "TATA_signal", "ecotype", "foo" },
    { "TATA_signal", "product", "foo" },
    { "terminator", "compare", "AY123456.1" },
    { "terminator", "mitochondrion", "foo" },
    { "terminator", "mobile_element", "integron" },
    { "terminator", "metagenomic", "foo" },
    { "terminator", "kinetoplast", "foo" },
    { "terminator", "chromoplast", "foo" },
    { "terminator", "specific_host", "foo" },
    { "terminator", "sub_strain", "foo" },
    { "terminator", "tag_peptide", "foo" },
    { "terminator", "isolation_source", "foo" },
    { "terminator", "collected_by", "foo" },
    { "terminator", "rpt_family", "foo" },
    { "terminator", "rpt_type", "flanking" },
    { "terminator", "insertion_seq", "foo" },
    { "terminator", "transl_table", "foo" },
    { "terminator", "rearranged", "foo" },
    { "terminator", "mod_base", "foo" },
    { "terminator", "rpt_unit", "foo" },
    { "terminator", "anticodon", "foo" },
    { "terminator", "function", "foo" },
    { "terminator", "number", "foo" },
    { "terminator", "identified_by", "foo" },
    { "terminator", "collection_date", "foo" },
    { "terminator", "direction", "foo" },
    { "terminator", "rpt_unit_range", "1..3" },
    { "terminator", "serotype", "foo" },
    { "terminator", "satellite", "foo" },
    { "terminator", "organism", "foo" },
    { "terminator", "transcript_id", "foo" },
    { "terminator", "serovar", "foo" },
    { "terminator", "variety", "foo" },
    { "terminator", "country", "foo" },
    { "terminator", "rpt_unit_seq", "foo" },
    { "terminator", "lab_host", "foo" },
    { "terminator", "macronuclear", "foo" },
    { "terminator", "cyanelle", "foo" },
    { "terminator", "bio_material", "foo" },
    { "terminator", "chloroplast", "foo" },
    { "terminator", "plasmid", "foo" },
    { "terminator", "mating_type", "foo" },
    { "terminator", "cell_type", "foo" },
    { "terminator", "EC_number", "1.2.3.4" },
    { "terminator", "mol_type", "foo" },
    { "terminator", "cultivar", "foo" },
    { "terminator", "artificial_location", "foo" },
    { "terminator", "segment", "foo" },
    { "terminator", "cons_splice", "foo" },
    { "terminator", "environmental_sample", "foo" },
    { "terminator", "PCR_conditions", "foo" },
    { "terminator", "frequency", "foo" },
    { "terminator", "transposon", "foo" },
    { "terminator", "haplogroup", "foo" },
    { "terminator", "ribosomal_slippage", "foo" },
    { "terminator", "codon_start", "foo" },
    { "terminator", "clone", "foo" },
    { "terminator", "gdb_xref", "foo" },
    { "terminator", "translation", "foo" },
    { "terminator", "transl_except", "foo" },
    { "terminator", "bound_moiety", "foo" },
    { "terminator", "sub_clone", "foo" },
    { "terminator", "cell_line", "foo" },
    { "terminator", "transgenic", "foo" },
    { "terminator", "germline", "foo" },
    { "terminator", "protein_id", "foo" },
    { "terminator", "codon", "foo" },
    { "terminator", "clone_lib", "foo" },
    { "terminator", "PCR_primers", "foo" },
    { "terminator", "sequenced_mol", "foo" },
    { "terminator", "replace", "aaaaattttt" },
    { "terminator", "strain", "foo" },
    { "terminator", "proviral", "foo" },
    { "terminator", "lat_lon", "foo" },
    { "terminator", "culture_collection", "foo" },
    { "terminator", "haplotype", "foo" },
    { "terminator", "estimated_length", "foo" },
    { "terminator", "tissue_lib", "foo" },
    { "terminator", "focus", "foo" },
    { "terminator", "dev_stage", "foo" },
    { "terminator", "specimen_voucher", "foo" },
    { "terminator", "pseudo", "foo" },
    { "terminator", "isolate", "foo" },
    { "terminator", "chromosome", "foo" },
    { "terminator", "ncRNA_class", "foo" },
    { "terminator", "pop_variant", "foo" },
    { "terminator", "tissue_type", "foo" },
    { "terminator", "trans_splicing", "foo" },
    { "terminator", "organelle", "foo" },
    { "terminator", "sex", "foo" },
    { "terminator", "virion", "foo" },
    { "terminator", "sub_species", "foo" },
    { "terminator", "phenotype", "foo" },
    { "terminator", "ecotype", "foo" },
    { "terminator", "product", "foo" },
    { "unsure", "mitochondrion", "foo" },
    { "V_region", "compare", "AY123456.1" },
    { "V_region", "mitochondrion", "foo" },
    { "V_region", "mobile_element", "integron" },
    { "V_region", "metagenomic", "foo" },
    { "V_region", "kinetoplast", "foo" },
    { "V_region", "chromoplast", "foo" },
    { "V_region", "specific_host", "foo" },
    { "V_region", "sub_strain", "foo" },
    { "V_region", "tag_peptide", "foo" },
    { "V_region", "isolation_source", "foo" },
    { "V_region", "collected_by", "foo" },
    { "V_region", "rpt_family", "foo" },
    { "V_region", "rpt_type", "flanking" },
    { "V_region", "insertion_seq", "foo" },
    { "V_region", "transl_table", "foo" },
    { "V_region", "rearranged", "foo" },
    { "V_region", "mod_base", "foo" },
    { "V_region", "rpt_unit", "foo" },
    { "V_region", "anticodon", "foo" },
    { "V_region", "function", "foo" },
    { "V_region", "number", "foo" },
    { "V_region", "identified_by", "foo" },
    { "V_region", "collection_date", "foo" },
    { "V_region", "direction", "foo" },
    { "V_region", "rpt_unit_range", "1..3" },
    { "V_region", "serotype", "foo" },
    { "V_region", "satellite", "foo" },
    { "V_region", "organism", "foo" },
    { "V_region", "transcript_id", "foo" },
    { "V_region", "serovar", "foo" },
    { "V_region", "variety", "foo" },
    { "V_region", "country", "foo" },
    { "V_region", "rpt_unit_seq", "foo" },
    { "V_region", "lab_host", "foo" },
    { "V_region", "macronuclear", "foo" },
    { "V_region", "cyanelle", "foo" },
    { "V_region", "bio_material", "foo" },
    { "V_region", "chloroplast", "foo" },
    { "V_region", "plasmid", "foo" },
    { "V_region", "mating_type", "foo" },
    { "V_region", "cell_type", "foo" },
    { "V_region", "EC_number", "1.2.3.4" },
    { "V_region", "mol_type", "foo" },
    { "V_region", "operon", "foo" },
    { "V_region", "cultivar", "foo" },
    { "V_region", "artificial_location", "foo" },
    { "V_region", "segment", "foo" },
    { "V_region", "cons_splice", "foo" },
    { "V_region", "environmental_sample", "foo" },
    { "V_region", "PCR_conditions", "foo" },
    { "V_region", "frequency", "foo" },
    { "V_region", "transposon", "foo" },
    { "V_region", "haplogroup", "foo" },
    { "V_region", "ribosomal_slippage", "foo" },
    { "V_region", "codon_start", "foo" },
    { "V_region", "clone", "foo" },
    { "V_region", "gdb_xref", "foo" },
    { "V_region", "translation", "foo" },
    { "V_region", "transl_except", "foo" },
    { "V_region", "bound_moiety", "foo" },
    { "V_region", "sub_clone", "foo" },
    { "V_region", "cell_line", "foo" },
    { "V_region", "transgenic", "foo" },
    { "V_region", "germline", "foo" },
    { "V_region", "protein_id", "foo" },
    { "V_region", "codon", "foo" },
    { "V_region", "clone_lib", "foo" },
    { "V_region", "PCR_primers", "foo" },
    { "V_region", "sequenced_mol", "foo" },
    { "V_region", "replace", "aaaaattttt" },
    { "V_region", "strain", "foo" },
    { "V_region", "proviral", "foo" },
    { "V_region", "lat_lon", "foo" },
    { "V_region", "culture_collection", "foo" },
    { "V_region", "haplotype", "foo" },
    { "V_region", "estimated_length", "foo" },
    { "V_region", "tissue_lib", "foo" },
    { "V_region", "focus", "foo" },
    { "V_region", "dev_stage", "foo" },
    { "V_region", "specimen_voucher", "foo" },
    { "V_region", "isolate", "foo" },
    { "V_region", "chromosome", "foo" },
    { "V_region", "ncRNA_class", "foo" },
    { "V_region", "pop_variant", "foo" },
    { "V_region", "tissue_type", "foo" },
    { "V_region", "trans_splicing", "foo" },
    { "V_region", "organelle", "foo" },
    { "V_region", "sex", "foo" },
    { "V_region", "virion", "foo" },
    { "V_region", "sub_species", "foo" },
    { "V_region", "phenotype", "foo" },
    { "V_region", "ecotype", "foo" },
    { "V_segment", "compare", "AY123456.1" },
    { "V_segment", "mitochondrion", "foo" },
    { "V_segment", "mobile_element", "integron" },
    { "V_segment", "metagenomic", "foo" },
    { "V_segment", "kinetoplast", "foo" },
    { "V_segment", "chromoplast", "foo" },
    { "V_segment", "specific_host", "foo" },
    { "V_segment", "sub_strain", "foo" },
    { "V_segment", "tag_peptide", "foo" },
    { "V_segment", "isolation_source", "foo" },
    { "V_segment", "collected_by", "foo" },
    { "V_segment", "rpt_family", "foo" },
    { "V_segment", "rpt_type", "flanking" },
    { "V_segment", "insertion_seq", "foo" },
    { "V_segment", "transl_table", "foo" },
    { "V_segment", "rearranged", "foo" },
    { "V_segment", "mod_base", "foo" },
    { "V_segment", "rpt_unit", "foo" },
    { "V_segment", "anticodon", "foo" },
    { "V_segment", "function", "foo" },
    { "V_segment", "number", "foo" },
    { "V_segment", "identified_by", "foo" },
    { "V_segment", "collection_date", "foo" },
    { "V_segment", "direction", "foo" },
    { "V_segment", "rpt_unit_range", "1..3" },
    { "V_segment", "serotype", "foo" },
    { "V_segment", "satellite", "foo" },
    { "V_segment", "organism", "foo" },
    { "V_segment", "transcript_id", "foo" },
    { "V_segment", "serovar", "foo" },
    { "V_segment", "variety", "foo" },
    { "V_segment", "country", "foo" },
    { "V_segment", "rpt_unit_seq", "foo" },
    { "V_segment", "lab_host", "foo" },
    { "V_segment", "macronuclear", "foo" },
    { "V_segment", "cyanelle", "foo" },
    { "V_segment", "bio_material", "foo" },
    { "V_segment", "chloroplast", "foo" },
    { "V_segment", "plasmid", "foo" },
    { "V_segment", "mating_type", "foo" },
    { "V_segment", "cell_type", "foo" },
    { "V_segment", "EC_number", "1.2.3.4" },
    { "V_segment", "mol_type", "foo" },
    { "V_segment", "operon", "foo" },
    { "V_segment", "cultivar", "foo" },
    { "V_segment", "artificial_location", "foo" },
    { "V_segment", "segment", "foo" },
    { "V_segment", "cons_splice", "foo" },
    { "V_segment", "environmental_sample", "foo" },
    { "V_segment", "PCR_conditions", "foo" },
    { "V_segment", "frequency", "foo" },
    { "V_segment", "transposon", "foo" },
    { "V_segment", "haplogroup", "foo" },
    { "V_segment", "ribosomal_slippage", "foo" },
    { "V_segment", "codon_start", "foo" },
    { "V_segment", "clone", "foo" },
    { "V_segment", "gdb_xref", "foo" },
    { "V_segment", "translation", "foo" },
    { "V_segment", "transl_except", "foo" },
    { "V_segment", "bound_moiety", "foo" },
    { "V_segment", "sub_clone", "foo" },
    { "V_segment", "cell_line", "foo" },
    { "V_segment", "transgenic", "foo" },
    { "V_segment", "germline", "foo" },
    { "V_segment", "protein_id", "foo" },
    { "V_segment", "codon", "foo" },
    { "V_segment", "clone_lib", "foo" },
    { "V_segment", "PCR_primers", "foo" },
    { "V_segment", "sequenced_mol", "foo" },
    { "V_segment", "replace", "aaaaattttt" },
    { "V_segment", "strain", "foo" },
    { "V_segment", "proviral", "foo" },
    { "V_segment", "lat_lon", "foo" },
    { "V_segment", "culture_collection", "foo" },
    { "V_segment", "haplotype", "foo" },
    { "V_segment", "estimated_length", "foo" },
    { "V_segment", "tissue_lib", "foo" },
    { "V_segment", "focus", "foo" },
    { "V_segment", "dev_stage", "foo" },
    { "V_segment", "specimen_voucher", "foo" },
    { "V_segment", "isolate", "foo" },
    { "V_segment", "chromosome", "foo" },
    { "V_segment", "ncRNA_class", "foo" },
    { "V_segment", "pop_variant", "foo" },
    { "V_segment", "tissue_type", "foo" },
    { "V_segment", "trans_splicing", "foo" },
    { "V_segment", "organelle", "foo" },
    { "V_segment", "sex", "foo" },
    { "V_segment", "virion", "foo" },
    { "V_segment", "sub_species", "foo" },
    { "V_segment", "phenotype", "foo" },
    { "V_segment", "ecotype", "foo" },
    { "variation", "mitochondrion", "foo" },
    { "variation", "mobile_element", "integron" },
    { "variation", "metagenomic", "foo" },
    { "variation", "kinetoplast", "foo" },
    { "variation", "chromoplast", "foo" },
    { "variation", "specific_host", "foo" },
    { "variation", "sub_strain", "foo" },
    { "variation", "tag_peptide", "foo" },
    { "variation", "isolation_source", "foo" },
    { "variation", "collected_by", "foo" },
    { "variation", "rpt_family", "foo" },
    { "variation", "rpt_type", "flanking" },
    { "variation", "insertion_seq", "foo" },
    { "variation", "transl_table", "foo" },
    { "variation", "rearranged", "foo" },
    { "variation", "mod_base", "foo" },
    { "variation", "rpt_unit", "foo" },
    { "variation", "anticodon", "foo" },
    { "variation", "function", "foo" },
    { "variation", "number", "foo" },
    { "variation", "identified_by", "foo" },
    { "variation", "collection_date", "foo" },
    { "variation", "direction", "foo" },
    { "variation", "rpt_unit_range", "1..3" },
    { "variation", "serotype", "foo" },
    { "variation", "satellite", "foo" },
    { "variation", "organism", "foo" },
    { "variation", "transcript_id", "foo" },
    { "variation", "serovar", "foo" },
    { "variation", "variety", "foo" },
    { "variation", "country", "foo" },
    { "variation", "rpt_unit_seq", "foo" },
    { "variation", "lab_host", "foo" },
    { "variation", "macronuclear", "foo" },
    { "variation", "cyanelle", "foo" },
    { "variation", "bio_material", "foo" },
    { "variation", "chloroplast", "foo" },
    { "variation", "plasmid", "foo" },
    { "variation", "mating_type", "foo" },
    { "variation", "cell_type", "foo" },
    { "variation", "EC_number", "1.2.3.4" },
    { "variation", "mol_type", "foo" },
    { "variation", "operon", "foo" },
    { "variation", "cultivar", "foo" },
    { "variation", "artificial_location", "foo" },
    { "variation", "segment", "foo" },
    { "variation", "cons_splice", "foo" },
    { "variation", "environmental_sample", "foo" },
    { "variation", "PCR_conditions", "foo" },
    { "variation", "transposon", "foo" },
    { "variation", "haplogroup", "foo" },
    { "variation", "ribosomal_slippage", "foo" },
    { "variation", "codon_start", "foo" },
    { "variation", "clone", "foo" },
    { "variation", "gdb_xref", "foo" },
    { "variation", "translation", "foo" },
    { "variation", "transl_except", "foo" },
    { "variation", "bound_moiety", "foo" },
    { "variation", "sub_clone", "foo" },
    { "variation", "cell_line", "foo" },
    { "variation", "transgenic", "foo" },
    { "variation", "germline", "foo" },
    { "variation", "protein_id", "foo" },
    { "variation", "codon", "foo" },
    { "variation", "clone_lib", "foo" },
    { "variation", "PCR_primers", "foo" },
    { "variation", "sequenced_mol", "foo" },
    { "variation", "strain", "foo" },
    { "variation", "proviral", "foo" },
    { "variation", "lat_lon", "foo" },
    { "variation", "culture_collection", "foo" },
    { "variation", "haplotype", "foo" },
    { "variation", "estimated_length", "foo" },
    { "variation", "tissue_lib", "foo" },
    { "variation", "focus", "foo" },
    { "variation", "dev_stage", "foo" },
    { "variation", "specimen_voucher", "foo" },
    { "variation", "pseudo", "foo" },
    { "variation", "isolate", "foo" },
    { "variation", "chromosome", "foo" },
    { "variation", "ncRNA_class", "foo" },
    { "variation", "pop_variant", "foo" },
    { "variation", "tissue_type", "foo" },
    { "variation", "trans_splicing", "foo" },
    { "variation", "organelle", "foo" },
    { "variation", "sex", "foo" },
    { "variation", "virion", "foo" },
    { "variation", "sub_species", "foo" },
    { "variation", "ecotype", "foo" },
    { "3'clip", "compare", "AY123456.1" },
    { "3'clip", "mitochondrion", "foo" },
    { "3'clip", "mobile_element", "integron" },
    { "3'clip", "metagenomic", "foo" },
    { "3'clip", "kinetoplast", "foo" },
    { "3'clip", "chromoplast", "foo" },
    { "3'clip", "specific_host", "foo" },
    { "3'clip", "sub_strain", "foo" },
    { "3'clip", "tag_peptide", "foo" },
    { "3'clip", "isolation_source", "foo" },
    { "3'clip", "collected_by", "foo" },
    { "3'clip", "rpt_family", "foo" },
    { "3'clip", "rpt_type", "flanking" },
    { "3'clip", "insertion_seq", "foo" },
    { "3'clip", "transl_table", "foo" },
    { "3'clip", "rearranged", "foo" },
    { "3'clip", "mod_base", "foo" },
    { "3'clip", "rpt_unit", "foo" },
    { "3'clip", "anticodon", "foo" },
    { "3'clip", "number", "foo" },
    { "3'clip", "identified_by", "foo" },
    { "3'clip", "collection_date", "foo" },
    { "3'clip", "direction", "foo" },
    { "3'clip", "rpt_unit_range", "1..3" },
    { "3'clip", "serotype", "foo" },
    { "3'clip", "satellite", "foo" },
    { "3'clip", "organism", "foo" },
    { "3'clip", "transcript_id", "foo" },
    { "3'clip", "serovar", "foo" },
    { "3'clip", "variety", "foo" },
    { "3'clip", "country", "foo" },
    { "3'clip", "rpt_unit_seq", "foo" },
    { "3'clip", "lab_host", "foo" },
    { "3'clip", "macronuclear", "foo" },
    { "3'clip", "cyanelle", "foo" },
    { "3'clip", "bio_material", "foo" },
    { "3'clip", "chloroplast", "foo" },
    { "3'clip", "plasmid", "foo" },
    { "3'clip", "mating_type", "foo" },
    { "3'clip", "cell_type", "foo" },
    { "3'clip", "EC_number", "1.2.3.4" },
    { "3'clip", "mol_type", "foo" },
    { "3'clip", "operon", "foo" },
    { "3'clip", "cultivar", "foo" },
    { "3'clip", "artificial_location", "foo" },
    { "3'clip", "segment", "foo" },
    { "3'clip", "cons_splice", "foo" },
    { "3'clip", "environmental_sample", "foo" },
    { "3'clip", "PCR_conditions", "foo" },
    { "3'clip", "frequency", "foo" },
    { "3'clip", "transposon", "foo" },
    { "3'clip", "haplogroup", "foo" },
    { "3'clip", "ribosomal_slippage", "foo" },
    { "3'clip", "codon_start", "foo" },
    { "3'clip", "clone", "foo" },
    { "3'clip", "gdb_xref", "foo" },
    { "3'clip", "translation", "foo" },
    { "3'clip", "transl_except", "foo" },
    { "3'clip", "bound_moiety", "foo" },
    { "3'clip", "sub_clone", "foo" },
    { "3'clip", "cell_line", "foo" },
    { "3'clip", "transgenic", "foo" },
    { "3'clip", "germline", "foo" },
    { "3'clip", "protein_id", "foo" },
    { "3'clip", "codon", "foo" },
    { "3'clip", "clone_lib", "foo" },
    { "3'clip", "PCR_primers", "foo" },
    { "3'clip", "sequenced_mol", "foo" },
    { "3'clip", "replace", "aaaaattttt" },
    { "3'clip", "strain", "foo" },
    { "3'clip", "proviral", "foo" },
    { "3'clip", "lat_lon", "foo" },
    { "3'clip", "culture_collection", "foo" },
    { "3'clip", "haplotype", "foo" },
    { "3'clip", "estimated_length", "foo" },
    { "3'clip", "tissue_lib", "foo" },
    { "3'clip", "focus", "foo" },
    { "3'clip", "dev_stage", "foo" },
    { "3'clip", "specimen_voucher", "foo" },
    { "3'clip", "pseudo", "foo" },
    { "3'clip", "isolate", "foo" },
    { "3'clip", "chromosome", "foo" },
    { "3'clip", "ncRNA_class", "foo" },
    { "3'clip", "pop_variant", "foo" },
    { "3'clip", "tissue_type", "foo" },
    { "3'clip", "organelle", "foo" },
    { "3'clip", "sex", "foo" },
    { "3'clip", "virion", "foo" },
    { "3'clip", "sub_species", "foo" },
    { "3'clip", "phenotype", "foo" },
    { "3'clip", "ecotype", "foo" },
    { "3'clip", "product", "foo" },
    { "3'UTR", "compare", "AY123456.1" },
    { "3'UTR", "mitochondrion", "foo" },
    { "3'UTR", "mobile_element", "integron" },
    { "3'UTR", "metagenomic", "foo" },
    { "3'UTR", "kinetoplast", "foo" },
    { "3'UTR", "chromoplast", "foo" },
    { "3'UTR", "specific_host", "foo" },
    { "3'UTR", "sub_strain", "foo" },
    { "3'UTR", "tag_peptide", "foo" },
    { "3'UTR", "isolation_source", "foo" },
    { "3'UTR", "collected_by", "foo" },
    { "3'UTR", "rpt_family", "foo" },
    { "3'UTR", "rpt_type", "flanking" },
    { "3'UTR", "insertion_seq", "foo" },
    { "3'UTR", "transl_table", "foo" },
    { "3'UTR", "rearranged", "foo" },
    { "3'UTR", "mod_base", "foo" },
    { "3'UTR", "rpt_unit", "foo" },
    { "3'UTR", "anticodon", "foo" },
    { "3'UTR", "number", "foo" },
    { "3'UTR", "identified_by", "foo" },
    { "3'UTR", "collection_date", "foo" },
    { "3'UTR", "direction", "foo" },
    { "3'UTR", "rpt_unit_range", "1..3" },
    { "3'UTR", "serotype", "foo" },
    { "3'UTR", "satellite", "foo" },
    { "3'UTR", "organism", "foo" },
    { "3'UTR", "transcript_id", "foo" },
    { "3'UTR", "serovar", "foo" },
    { "3'UTR", "variety", "foo" },
    { "3'UTR", "country", "foo" },
    { "3'UTR", "rpt_unit_seq", "foo" },
    { "3'UTR", "lab_host", "foo" },
    { "3'UTR", "macronuclear", "foo" },
    { "3'UTR", "cyanelle", "foo" },
    { "3'UTR", "bio_material", "foo" },
    { "3'UTR", "chloroplast", "foo" },
    { "3'UTR", "plasmid", "foo" },
    { "3'UTR", "mating_type", "foo" },
    { "3'UTR", "cell_type", "foo" },
    { "3'UTR", "EC_number", "1.2.3.4" },
    { "3'UTR", "mol_type", "foo" },
    { "3'UTR", "operon", "foo" },
    { "3'UTR", "cultivar", "foo" },
    { "3'UTR", "artificial_location", "foo" },
    { "3'UTR", "segment", "foo" },
    { "3'UTR", "cons_splice", "foo" },
    { "3'UTR", "environmental_sample", "foo" },
    { "3'UTR", "PCR_conditions", "foo" },
    { "3'UTR", "frequency", "foo" },
    { "3'UTR", "transposon", "foo" },
    { "3'UTR", "haplogroup", "foo" },
    { "3'UTR", "ribosomal_slippage", "foo" },
    { "3'UTR", "codon_start", "foo" },
    { "3'UTR", "clone", "foo" },
    { "3'UTR", "gdb_xref", "foo" },
    { "3'UTR", "translation", "foo" },
    { "3'UTR", "transl_except", "foo" },
    { "3'UTR", "bound_moiety", "foo" },
    { "3'UTR", "sub_clone", "foo" },
    { "3'UTR", "cell_line", "foo" },
    { "3'UTR", "transgenic", "foo" },
    { "3'UTR", "germline", "foo" },
    { "3'UTR", "protein_id", "foo" },
    { "3'UTR", "codon", "foo" },
    { "3'UTR", "clone_lib", "foo" },
    { "3'UTR", "PCR_primers", "foo" },
    { "3'UTR", "sequenced_mol", "foo" },
    { "3'UTR", "replace", "aaaaattttt" },
    { "3'UTR", "strain", "foo" },
    { "3'UTR", "proviral", "foo" },
    { "3'UTR", "lat_lon", "foo" },
    { "3'UTR", "culture_collection", "foo" },
    { "3'UTR", "haplotype", "foo" },
    { "3'UTR", "estimated_length", "foo" },
    { "3'UTR", "tissue_lib", "foo" },
    { "3'UTR", "focus", "foo" },
    { "3'UTR", "dev_stage", "foo" },
    { "3'UTR", "specimen_voucher", "foo" },
    { "3'UTR", "pseudo", "foo" },
    { "3'UTR", "isolate", "foo" },
    { "3'UTR", "chromosome", "foo" },
    { "3'UTR", "ncRNA_class", "foo" },
    { "3'UTR", "pop_variant", "foo" },
    { "3'UTR", "tissue_type", "foo" },
    { "3'UTR", "organelle", "foo" },
    { "3'UTR", "sex", "foo" },
    { "3'UTR", "virion", "foo" },
    { "3'UTR", "sub_species", "foo" },
    { "3'UTR", "phenotype", "foo" },
    { "3'UTR", "ecotype", "foo" },
    { "3'UTR", "product", "foo" },
    { "5'clip", "compare", "AY123456.1" },
    { "5'clip", "mitochondrion", "foo" },
    { "5'clip", "mobile_element", "integron" },
    { "5'clip", "metagenomic", "foo" },
    { "5'clip", "kinetoplast", "foo" },
    { "5'clip", "chromoplast", "foo" },
    { "5'clip", "specific_host", "foo" },
    { "5'clip", "sub_strain", "foo" },
    { "5'clip", "tag_peptide", "foo" },
    { "5'clip", "isolation_source", "foo" },
    { "5'clip", "collected_by", "foo" },
    { "5'clip", "rpt_family", "foo" },
    { "5'clip", "rpt_type", "flanking" },
    { "5'clip", "insertion_seq", "foo" },
    { "5'clip", "transl_table", "foo" },
    { "5'clip", "rearranged", "foo" },
    { "5'clip", "mod_base", "foo" },
    { "5'clip", "rpt_unit", "foo" },
    { "5'clip", "anticodon", "foo" },
    { "5'clip", "number", "foo" },
    { "5'clip", "identified_by", "foo" },
    { "5'clip", "collection_date", "foo" },
    { "5'clip", "direction", "foo" },
    { "5'clip", "rpt_unit_range", "1..3" },
    { "5'clip", "serotype", "foo" },
    { "5'clip", "satellite", "foo" },
    { "5'clip", "organism", "foo" },
    { "5'clip", "transcript_id", "foo" },
    { "5'clip", "serovar", "foo" },
    { "5'clip", "variety", "foo" },
    { "5'clip", "country", "foo" },
    { "5'clip", "rpt_unit_seq", "foo" },
    { "5'clip", "lab_host", "foo" },
    { "5'clip", "macronuclear", "foo" },
    { "5'clip", "cyanelle", "foo" },
    { "5'clip", "bio_material", "foo" },
    { "5'clip", "chloroplast", "foo" },
    { "5'clip", "plasmid", "foo" },
    { "5'clip", "mating_type", "foo" },
    { "5'clip", "cell_type", "foo" },
    { "5'clip", "EC_number", "1.2.3.4" },
    { "5'clip", "mol_type", "foo" },
    { "5'clip", "operon", "foo" },
    { "5'clip", "cultivar", "foo" },
    { "5'clip", "artificial_location", "foo" },
    { "5'clip", "segment", "foo" },
    { "5'clip", "cons_splice", "foo" },
    { "5'clip", "environmental_sample", "foo" },
    { "5'clip", "PCR_conditions", "foo" },
    { "5'clip", "frequency", "foo" },
    { "5'clip", "transposon", "foo" },
    { "5'clip", "haplogroup", "foo" },
    { "5'clip", "ribosomal_slippage", "foo" },
    { "5'clip", "codon_start", "foo" },
    { "5'clip", "clone", "foo" },
    { "5'clip", "gdb_xref", "foo" },
    { "5'clip", "translation", "foo" },
    { "5'clip", "transl_except", "foo" },
    { "5'clip", "bound_moiety", "foo" },
    { "5'clip", "sub_clone", "foo" },
    { "5'clip", "cell_line", "foo" },
    { "5'clip", "transgenic", "foo" },
    { "5'clip", "germline", "foo" },
    { "5'clip", "protein_id", "foo" },
    { "5'clip", "codon", "foo" },
    { "5'clip", "clone_lib", "foo" },
    { "5'clip", "PCR_primers", "foo" },
    { "5'clip", "sequenced_mol", "foo" },
    { "5'clip", "replace", "aaaaattttt" },
    { "5'clip", "strain", "foo" },
    { "5'clip", "proviral", "foo" },
    { "5'clip", "lat_lon", "foo" },
    { "5'clip", "culture_collection", "foo" },
    { "5'clip", "haplotype", "foo" },
    { "5'clip", "estimated_length", "foo" },
    { "5'clip", "tissue_lib", "foo" },
    { "5'clip", "focus", "foo" },
    { "5'clip", "dev_stage", "foo" },
    { "5'clip", "specimen_voucher", "foo" },
    { "5'clip", "pseudo", "foo" },
    { "5'clip", "isolate", "foo" },
    { "5'clip", "chromosome", "foo" },
    { "5'clip", "ncRNA_class", "foo" },
    { "5'clip", "pop_variant", "foo" },
    { "5'clip", "tissue_type", "foo" },
    { "5'clip", "organelle", "foo" },
    { "5'clip", "sex", "foo" },
    { "5'clip", "virion", "foo" },
    { "5'clip", "sub_species", "foo" },
    { "5'clip", "phenotype", "foo" },
    { "5'clip", "ecotype", "foo" },
    { "5'clip", "product", "foo" },
    { "5'UTR", "compare", "AY123456.1" },
    { "5'UTR", "mitochondrion", "foo" },
    { "5'UTR", "mobile_element", "integron" },
    { "5'UTR", "metagenomic", "foo" },
    { "5'UTR", "kinetoplast", "foo" },
    { "5'UTR", "chromoplast", "foo" },
    { "5'UTR", "specific_host", "foo" },
    { "5'UTR", "sub_strain", "foo" },
    { "5'UTR", "tag_peptide", "foo" },
    { "5'UTR", "isolation_source", "foo" },
    { "5'UTR", "collected_by", "foo" },
    { "5'UTR", "rpt_family", "foo" },
    { "5'UTR", "rpt_type", "flanking" },
    { "5'UTR", "insertion_seq", "foo" },
    { "5'UTR", "transl_table", "foo" },
    { "5'UTR", "rearranged", "foo" },
    { "5'UTR", "mod_base", "foo" },
    { "5'UTR", "rpt_unit", "foo" },
    { "5'UTR", "anticodon", "foo" },
    { "5'UTR", "number", "foo" },
    { "5'UTR", "identified_by", "foo" },
    { "5'UTR", "collection_date", "foo" },
    { "5'UTR", "direction", "foo" },
    { "5'UTR", "rpt_unit_range", "1..3" },
    { "5'UTR", "serotype", "foo" },
    { "5'UTR", "satellite", "foo" },
    { "5'UTR", "organism", "foo" },
    { "5'UTR", "transcript_id", "foo" },
    { "5'UTR", "serovar", "foo" },
    { "5'UTR", "variety", "foo" },
    { "5'UTR", "country", "foo" },
    { "5'UTR", "rpt_unit_seq", "foo" },
    { "5'UTR", "lab_host", "foo" },
    { "5'UTR", "macronuclear", "foo" },
    { "5'UTR", "cyanelle", "foo" },
    { "5'UTR", "bio_material", "foo" },
    { "5'UTR", "chloroplast", "foo" },
    { "5'UTR", "plasmid", "foo" },
    { "5'UTR", "mating_type", "foo" },
    { "5'UTR", "cell_type", "foo" },
    { "5'UTR", "EC_number", "1.2.3.4" },
    { "5'UTR", "mol_type", "foo" },
    { "5'UTR", "operon", "foo" },
    { "5'UTR", "cultivar", "foo" },
    { "5'UTR", "artificial_location", "foo" },
    { "5'UTR", "segment", "foo" },
    { "5'UTR", "cons_splice", "foo" },
    { "5'UTR", "environmental_sample", "foo" },
    { "5'UTR", "PCR_conditions", "foo" },
    { "5'UTR", "frequency", "foo" },
    { "5'UTR", "transposon", "foo" },
    { "5'UTR", "haplogroup", "foo" },
    { "5'UTR", "ribosomal_slippage", "foo" },
    { "5'UTR", "codon_start", "foo" },
    { "5'UTR", "clone", "foo" },
    { "5'UTR", "gdb_xref", "foo" },
    { "5'UTR", "translation", "foo" },
    { "5'UTR", "transl_except", "foo" },
    { "5'UTR", "bound_moiety", "foo" },
    { "5'UTR", "sub_clone", "foo" },
    { "5'UTR", "cell_line", "foo" },
    { "5'UTR", "transgenic", "foo" },
    { "5'UTR", "germline", "foo" },
    { "5'UTR", "protein_id", "foo" },
    { "5'UTR", "codon", "foo" },
    { "5'UTR", "clone_lib", "foo" },
    { "5'UTR", "PCR_primers", "foo" },
    { "5'UTR", "sequenced_mol", "foo" },
    { "5'UTR", "replace", "aaaaattttt" },
    { "5'UTR", "strain", "foo" },
    { "5'UTR", "proviral", "foo" },
    { "5'UTR", "lat_lon", "foo" },
    { "5'UTR", "culture_collection", "foo" },
    { "5'UTR", "haplotype", "foo" },
    { "5'UTR", "estimated_length", "foo" },
    { "5'UTR", "tissue_lib", "foo" },
    { "5'UTR", "focus", "foo" },
    { "5'UTR", "dev_stage", "foo" },
    { "5'UTR", "specimen_voucher", "foo" },
    { "5'UTR", "pseudo", "foo" },
    { "5'UTR", "isolate", "foo" },
    { "5'UTR", "chromosome", "foo" },
    { "5'UTR", "ncRNA_class", "foo" },
    { "5'UTR", "pop_variant", "foo" },
    { "5'UTR", "tissue_type", "foo" },
    { "5'UTR", "organelle", "foo" },
    { "5'UTR", "sex", "foo" },
    { "5'UTR", "virion", "foo" },
    { "5'UTR", "sub_species", "foo" },
    { "5'UTR", "phenotype", "foo" },
    { "5'UTR", "ecotype", "foo" },
    { "5'UTR", "product", "foo" },
    { "-10_signal", "compare", "AY123456.1" },
    { "-10_signal", "mitochondrion", "foo" },
    { "-10_signal", "mobile_element", "integron" },
    { "-10_signal", "metagenomic", "foo" },
    { "-10_signal", "kinetoplast", "foo" },
    { "-10_signal", "chromoplast", "foo" },
    { "-10_signal", "specific_host", "foo" },
    { "-10_signal", "sub_strain", "foo" },
    { "-10_signal", "tag_peptide", "foo" },
    { "-10_signal", "isolation_source", "foo" },
    { "-10_signal", "collected_by", "foo" },
    { "-10_signal", "rpt_family", "foo" },
    { "-10_signal", "rpt_type", "flanking" },
    { "-10_signal", "insertion_seq", "foo" },
    { "-10_signal", "transl_table", "foo" },
    { "-10_signal", "rearranged", "foo" },
    { "-10_signal", "mod_base", "foo" },
    { "-10_signal", "rpt_unit", "foo" },
    { "-10_signal", "anticodon", "foo" },
    { "-10_signal", "function", "foo" },
    { "-10_signal", "number", "foo" },
    { "-10_signal", "identified_by", "foo" },
    { "-10_signal", "collection_date", "foo" },
    { "-10_signal", "direction", "foo" },
    { "-10_signal", "rpt_unit_range", "1..3" },
    { "-10_signal", "serotype", "foo" },
    { "-10_signal", "satellite", "foo" },
    { "-10_signal", "organism", "foo" },
    { "-10_signal", "transcript_id", "foo" },
    { "-10_signal", "serovar", "foo" },
    { "-10_signal", "variety", "foo" },
    { "-10_signal", "country", "foo" },
    { "-10_signal", "rpt_unit_seq", "foo" },
    { "-10_signal", "lab_host", "foo" },
    { "-10_signal", "macronuclear", "foo" },
    { "-10_signal", "cyanelle", "foo" },
    { "-10_signal", "bio_material", "foo" },
    { "-10_signal", "chloroplast", "foo" },
    { "-10_signal", "plasmid", "foo" },
    { "-10_signal", "mating_type", "foo" },
    { "-10_signal", "cell_type", "foo" },
    { "-10_signal", "EC_number", "1.2.3.4" },
    { "-10_signal", "mol_type", "foo" },
    { "-10_signal", "cultivar", "foo" },
    { "-10_signal", "artificial_location", "foo" },
    { "-10_signal", "segment", "foo" },
    { "-10_signal", "cons_splice", "foo" },
    { "-10_signal", "environmental_sample", "foo" },
    { "-10_signal", "PCR_conditions", "foo" },
    { "-10_signal", "frequency", "foo" },
    { "-10_signal", "transposon", "foo" },
    { "-10_signal", "haplogroup", "foo" },
    { "-10_signal", "ribosomal_slippage", "foo" },
    { "-10_signal", "codon_start", "foo" },
    { "-10_signal", "clone", "foo" },
    { "-10_signal", "gdb_xref", "foo" },
    { "-10_signal", "translation", "foo" },
    { "-10_signal", "transl_except", "foo" },
    { "-10_signal", "bound_moiety", "foo" },
    { "-10_signal", "sub_clone", "foo" },
    { "-10_signal", "cell_line", "foo" },
    { "-10_signal", "transgenic", "foo" },
    { "-10_signal", "germline", "foo" },
    { "-10_signal", "protein_id", "foo" },
    { "-10_signal", "codon", "foo" },
    { "-10_signal", "clone_lib", "foo" },
    { "-10_signal", "PCR_primers", "foo" },
    { "-10_signal", "sequenced_mol", "foo" },
    { "-10_signal", "replace", "aaaaattttt" },
    { "-10_signal", "strain", "foo" },
    { "-10_signal", "proviral", "foo" },
    { "-10_signal", "lat_lon", "foo" },
    { "-10_signal", "culture_collection", "foo" },
    { "-10_signal", "haplotype", "foo" },
    { "-10_signal", "estimated_length", "foo" },
    { "-10_signal", "tissue_lib", "foo" },
    { "-10_signal", "focus", "foo" },
    { "-10_signal", "dev_stage", "foo" },
    { "-10_signal", "specimen_voucher", "foo" },
    { "-10_signal", "pseudo", "foo" },
    { "-10_signal", "isolate", "foo" },
    { "-10_signal", "chromosome", "foo" },
    { "-10_signal", "ncRNA_class", "foo" },
    { "-10_signal", "pop_variant", "foo" },
    { "-10_signal", "tissue_type", "foo" },
    { "-10_signal", "trans_splicing", "foo" },
    { "-10_signal", "organelle", "foo" },
    { "-10_signal", "sex", "foo" },
    { "-10_signal", "virion", "foo" },
    { "-10_signal", "sub_species", "foo" },
    { "-10_signal", "phenotype", "foo" },
    { "-10_signal", "ecotype", "foo" },
    { "-10_signal", "product", "foo" },
    { "-35_signal", "compare", "AY123456.1" },
    { "-35_signal", "mitochondrion", "foo" },
    { "-35_signal", "mobile_element", "integron" },
    { "-35_signal", "metagenomic", "foo" },
    { "-35_signal", "kinetoplast", "foo" },
    { "-35_signal", "chromoplast", "foo" },
    { "-35_signal", "specific_host", "foo" },
    { "-35_signal", "sub_strain", "foo" },
    { "-35_signal", "tag_peptide", "foo" },
    { "-35_signal", "isolation_source", "foo" },
    { "-35_signal", "collected_by", "foo" },
    { "-35_signal", "rpt_family", "foo" },
    { "-35_signal", "rpt_type", "flanking" },
    { "-35_signal", "insertion_seq", "foo" },
    { "-35_signal", "transl_table", "foo" },
    { "-35_signal", "rearranged", "foo" },
    { "-35_signal", "mod_base", "foo" },
    { "-35_signal", "rpt_unit", "foo" },
    { "-35_signal", "anticodon", "foo" },
    { "-35_signal", "function", "foo" },
    { "-35_signal", "number", "foo" },
    { "-35_signal", "identified_by", "foo" },
    { "-35_signal", "collection_date", "foo" },
    { "-35_signal", "direction", "foo" },
    { "-35_signal", "rpt_unit_range", "1..3" },
    { "-35_signal", "serotype", "foo" },
    { "-35_signal", "satellite", "foo" },
    { "-35_signal", "organism", "foo" },
    { "-35_signal", "transcript_id", "foo" },
    { "-35_signal", "serovar", "foo" },
    { "-35_signal", "variety", "foo" },
    { "-35_signal", "country", "foo" },
    { "-35_signal", "rpt_unit_seq", "foo" },
    { "-35_signal", "lab_host", "foo" },
    { "-35_signal", "macronuclear", "foo" },
    { "-35_signal", "cyanelle", "foo" },
    { "-35_signal", "bio_material", "foo" },
    { "-35_signal", "chloroplast", "foo" },
    { "-35_signal", "plasmid", "foo" },
    { "-35_signal", "mating_type", "foo" },
    { "-35_signal", "cell_type", "foo" },
    { "-35_signal", "EC_number", "1.2.3.4" },
    { "-35_signal", "mol_type", "foo" },
    { "-35_signal", "cultivar", "foo" },
    { "-35_signal", "artificial_location", "foo" },
    { "-35_signal", "segment", "foo" },
    { "-35_signal", "cons_splice", "foo" },
    { "-35_signal", "environmental_sample", "foo" },
    { "-35_signal", "PCR_conditions", "foo" },
    { "-35_signal", "frequency", "foo" },
    { "-35_signal", "transposon", "foo" },
    { "-35_signal", "haplogroup", "foo" },
    { "-35_signal", "ribosomal_slippage", "foo" },
    { "-35_signal", "codon_start", "foo" },
    { "-35_signal", "clone", "foo" },
    { "-35_signal", "gdb_xref", "foo" },
    { "-35_signal", "translation", "foo" },
    { "-35_signal", "transl_except", "foo" },
    { "-35_signal", "bound_moiety", "foo" },
    { "-35_signal", "sub_clone", "foo" },
    { "-35_signal", "cell_line", "foo" },
    { "-35_signal", "transgenic", "foo" },
    { "-35_signal", "germline", "foo" },
    { "-35_signal", "protein_id", "foo" },
    { "-35_signal", "codon", "foo" },
    { "-35_signal", "clone_lib", "foo" },
    { "-35_signal", "PCR_primers", "foo" },
    { "-35_signal", "sequenced_mol", "foo" },
    { "-35_signal", "replace", "aaaaattttt" },
    { "-35_signal", "strain", "foo" },
    { "-35_signal", "proviral", "foo" },
    { "-35_signal", "lat_lon", "foo" },
    { "-35_signal", "culture_collection", "foo" },
    { "-35_signal", "haplotype", "foo" },
    { "-35_signal", "estimated_length", "foo" },
    { "-35_signal", "tissue_lib", "foo" },
    { "-35_signal", "focus", "foo" },
    { "-35_signal", "dev_stage", "foo" },
    { "-35_signal", "specimen_voucher", "foo" },
    { "-35_signal", "pseudo", "foo" },
    { "-35_signal", "isolate", "foo" },
    { "-35_signal", "chromosome", "foo" },
    { "-35_signal", "ncRNA_class", "foo" },
    { "-35_signal", "pop_variant", "foo" },
    { "-35_signal", "tissue_type", "foo" },
    { "-35_signal", "trans_splicing", "foo" },
    { "-35_signal", "organelle", "foo" },
    { "-35_signal", "sex", "foo" },
    { "-35_signal", "virion", "foo" },
    { "-35_signal", "sub_species", "foo" },
    { "-35_signal", "phenotype", "foo" },
    { "-35_signal", "ecotype", "foo" },
    { "-35_signal", "product", "foo" }
//end automatically generated section
};

static const int k_NumKeyQualVals = sizeof (s_KeyQualVal) / sizeof (TKeyQualVal);

BOOST_AUTO_TEST_CASE(Test_FEAT_WrongQualOnImpFeat)
{
    CRef<CSeq_entry> entry = BuildGoodSeq();

    STANDARD_SETUP_WITH_DATABASE

    string last_key = "";
    CRef<CSeq_feat> imp;

    for (int i = 0; i < k_NumKeyQualVals; i++) {
        if (!NStr::Equal(s_KeyQualVal[i].key, last_key)) {
            scope.RemoveTopLevelSeqEntry(seh);
            entry = BuildGoodSeq();
            imp = AddGoodImpFeat(entry, s_KeyQualVal[i].key);
            seh = scope.AddTopLevelSeqEntry(*entry);
            last_key = s_KeyQualVal[i].key;
        }
        s_TestOneWrongQual (s_KeyQualVal[i].qual, s_KeyQualVal[i].val, s_KeyQualVal[i].key, 
                                seh, imp, validator, options, &scope);
    }

}    
