/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* File: gkb.c
 * Purpose: GNOME Keyboard switcher
 *
 * Copyright (C) 1998-2000 Free Software Foundation
 * Authors: Szabolcs Ban  <shooby@gnome.hu>
 *          Chema Celorio <chema@celorio.com>
 *
 * Thanks for aid of George Lebl <jirka@5z.com> and solidarity
 * Balazs Nagy <js@lsc.hu>, Charles Levert <charles@comm.polymtl.ca>
 * and Emese Kovacs <emese@gnome.hu> for her docs and ideas.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>

#include <X11/keysym.h>
#include <X11/Xmd.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <gdk/gdkx.h>
#include <sys/stat.h>

#include <panel-applet.h>

#include <gtk/gtk.h>
#include <libbonobo.h>
#include <libgnomeui/libgnomeui.h>
#include <libgnome/libgnome.h>
#include <panel-applet-gconf.h>
#include "gkb.h"

/* Accessible name and description */                                           
void                                                                            
add_atk_namedesc(GtkWidget *widget, const gchar *name, const gchar *desc)       
{
    AtkObject *atk_widget;                                                      
    atk_widget = gtk_widget_get_accessible(widget);                             
    if (name)                                                                   
    {
      atk_object_set_name(atk_widget, name);                                    
    }
    if (desc)                                                                   
    {
      atk_object_set_description(atk_widget, desc);                             
    }
}


/**
 * gkb_count_sizes:
 * @gkb:
 * 
 * Calculates applet, flag, label sizes, mode mode
 */
static gint
gkb_count_sizes (GKB * gkb)
{
  gboolean small_panel = FALSE;
  gboolean label_in_vbox = TRUE;	/* If FALSE label is in hbox */
  gint panel_width = 0;		/* Zero means we have not determined it */
  gint panel_height = 0;
  gint applet_width = 0;
  gint applet_height = 0;
  gint flag_width = 0;
  gint flag_height = 0;
  gint label_height = 0;
  gint label_width = 0;
  gboolean horizontal = TRUE;

  gint size;

  size = panel_applet_get_size (PANEL_APPLET (gkb->applet));

  /* Determine if this pannel requires different handling because it is very small */
  switch (gkb->orient)
    {
    case PANEL_APPLET_ORIENT_UP:
    case PANEL_APPLET_ORIENT_DOWN:
      panel_height = size;
      if (size < GKB_SMALL_PANEL_SIZE)
	small_panel = TRUE;
      horizontal = TRUE;
      break;
    case PANEL_APPLET_ORIENT_RIGHT:
    case PANEL_APPLET_ORIENT_LEFT:
      panel_width = size;
      if (size < GKB_SMALL_PANEL_SIZE)
	small_panel = TRUE;
      horizontal = FALSE;
      break;
    default:
      g_assert_not_reached ();
    }

  flag_height =
    (gint) panel_height / ((gkb->is_small && !small_panel) ? 2 : 1);
  flag_width = (gint) panel_width / ((gkb->is_small && !small_panel) ? 2 : 1);

  /* This are the cases in which we change the label to be side by side
   * v.s. beeing top-bottom */
  if (gkb->mode == GKB_FLAG_AND_LABEL)
    {
      if (gkb->orient == PANEL_APPLET_ORIENT_UP && small_panel)
	label_in_vbox = FALSE;
      if (gkb->orient == PANEL_APPLET_ORIENT_UP && !gkb->is_small)
	label_in_vbox = FALSE;
      if (gkb->orient == PANEL_APPLET_ORIENT_DOWN && small_panel)
	label_in_vbox = FALSE;
      if (gkb->orient == PANEL_APPLET_ORIENT_DOWN && !gkb->is_small)
	label_in_vbox = FALSE;
      if (gkb->orient == PANEL_APPLET_ORIENT_RIGHT && !small_panel)
	if (gkb->is_small)
	  label_in_vbox = FALSE;
      if (gkb->orient == PANEL_APPLET_ORIENT_LEFT && !small_panel)
	if (gkb->is_small)
	  label_in_vbox = FALSE;
    }

  /* Flag has either the with or the height set */
  if (flag_width == 0)
    flag_width = (gint) flag_height *1.5;
  else
    flag_height = (gint) flag_width / 1.5;

  label_width = flag_width;
  label_height = flag_height;

  if (gkb->mode != GKB_LABEL)
    {
      applet_width += flag_width;
      applet_height += flag_height;
    }

  if (gkb->mode != GKB_FLAG)
    {
      if (label_in_vbox)
	applet_height += label_height;
      else
	applet_width += label_width;
    }

  gkb->w = flag_width - 1;
  /* FIXME the applet is just a little bigger than panel, so I add the -2 here*/
  gkb->h = flag_height - 1;
  if (horizontal) {
    gtk_widget_set_size_request (gkb->label_frame1, -1, gkb->h);
    gtk_widget_set_size_request (gkb->label_frame2, -1, gkb->h);
    gtk_widget_set_size_request (gkb->darea_frame,  -1, gkb->h);
  }
  else {
    gtk_widget_set_size_request (gkb->label_frame1, gkb->w, -1);
    gtk_widget_set_size_request (gkb->label_frame2, gkb->w, -1);
    gtk_widget_set_size_request (gkb->darea_frame,  gkb->w, -1);
  }

  return label_in_vbox;
}

/**
 * gkb_update:
 * @gkb: the GKB. global, so redundant
 * @set_command: do the command, or no
 * 
 * The size or orientation has been changed, update the widget
 **/
void
gkb_update (GKB *gkb, gboolean set_command)
{
  GkbKeymap *keymap;
  GList *list;
  gboolean label_in_vbox;

  g_return_if_fail (gkb->maps);

  /* When the size is changed, we don't need to change the actual
   * keymap, so when the set_commadn is false, it means that we
   * have changed size. In other words, we can't change size &
   * keymap at the same time */
  if (set_command) {
    if (gkb->update != NULL)
      (gkb->update) (gkb, TRUE);
    gkb_system_set_keymap (gkb);
  }

  /* When a size request is made, we need to redraw the pixbufs with the new
   * size.  Updates all the pixmaps with the new param.
   **/
  if (gkb->applet == NULL)
    return;

  label_in_vbox = gkb_count_sizes (gkb);

  /* Hide or show the flag */
  if (gkb->mode == GKB_LABEL)
    gtk_widget_hide (gkb->darea_frame);
  else
    gtk_widget_show_all (gkb->darea_frame);

  /* Hide or show the labels */
  if (gkb->mode == GKB_FLAG) {
    gtk_widget_hide (gkb->label_frame1);
    gtk_widget_hide (gkb->label_frame2);
  } else if (label_in_vbox) {
    gtk_widget_hide (gkb->label_frame2);
    gtk_widget_show_all (gkb->label_frame1);
  } else {
    gtk_widget_hide (gkb->label_frame1);
    gtk_widget_show_all (gkb->label_frame2);
  }

  list = gkb->maps;
  for (; list != NULL; list = list->next) {
    gchar *name;
    gchar *real_name;
    keymap = (GkbKeymap *) list->data;
    name = g_strdup_printf ("gkb/%s", keymap->flag);
    real_name = gnome_unconditional_pixmap_file (name);
    if (g_file_exists (real_name)) {
      if (keymap->pixbuf) 
	g_object_unref (keymap->pixbuf);
      keymap->pixbuf = gdk_pixbuf_new_from_file (real_name,NULL);
    } else {
      g_free (real_name);
      real_name = gnome_unconditional_pixmap_file ("gkb/gkb-foot.png");
      if (keymap->pixbuf)
	g_object_unref (keymap->pixbuf);
      keymap->pixbuf = gdk_pixbuf_new_from_file (real_name,NULL);
    }
    g_free (name);
    g_free (real_name);
  }

  if (gkb->applet == NULL)
    return;

  g_return_if_fail (gkb->image != NULL);
  g_return_if_fail (gkb->keymap != NULL);

  if (gkb->mode != GKB_LABEL) {
    GdkPixbuf *tmp;

    g_return_if_fail (gkb != NULL);
    g_return_if_fail (gkb->keymap != NULL);
    g_return_if_fail (gkb->keymap->pixbuf != NULL);

    tmp = gdk_pixbuf_scale_simple (gkb->keymap->pixbuf, gkb->w, 
				   gkb->h, GDK_INTERP_HYPER); 
    if (tmp) {     
      gtk_image_set_from_pixbuf (GTK_IMAGE (gkb->image), tmp);
      g_object_unref (tmp);
    }
  }
  if (gkb->mode != GKB_FLAG) {
    gtk_label_set_text (GTK_LABEL (gkb->label1), gkb->keymap->label);
    gtk_label_set_text (GTK_LABEL (gkb->label2), gkb->keymap->label);
  }

  gtk_tooltips_set_tip (gkb->tooltips, gkb->applet, gkb->keymap->name, NULL);
  add_atk_namedesc (gkb->applet, gkb->keymap->name,
		    _("Press the hotkey to switch between layouts. The hotkey can be set through Properties dialog"));
}

void
gkb_save_session (GKB *gkb)
{
  GkbKeymap *actdata;
  GList *list;
  gchar str[256];
  int i = 0;
  GConfChangeSet *cs = gconf_change_set_new ();

  gconf_change_set_set_int    (cs, GKB_GCONF_ROOT "/num",   gkb->n);
  gconf_change_set_set_bool   (cs, GKB_GCONF_ROOT "/small", gkb->is_small);
  gconf_change_set_set_string (cs, GKB_GCONF_ROOT "/key",   gkb->key);
  gconf_change_set_set_string (cs, GKB_GCONF_ROOT "/mode",
			       gkb_util_get_text_from_mode (gkb->mode));

  i = 0;
  for (list = gkb->maps; list ; list = list->next, i++) {
    actdata = list->data;
    if (actdata == NULL)
      continue;
    g_snprintf (str, sizeof (str), GKB_GCONF_ROOT "/name_%d", i);
    gconf_change_set_set_string (cs, str, actdata->name);
    g_snprintf (str, sizeof (str), GKB_GCONF_ROOT "/country_%d", i);
    gconf_change_set_set_string (cs, str, actdata->country);
    g_snprintf (str, sizeof (str),GKB_GCONF_ROOT  "/lang_%d", i);
    gconf_change_set_set_string (cs, str, actdata->lang);
    g_snprintf (str, sizeof (str),GKB_GCONF_ROOT  "/label_%d", i);
    gconf_change_set_set_string (cs, str, actdata->label);
    g_snprintf (str, sizeof (str), GKB_GCONF_ROOT "/flag_%d", i);
    gconf_change_set_set_string (cs, str, actdata->flag);
    g_snprintf (str, sizeof (str), GKB_GCONF_ROOT "/command_%d", i);
    gconf_change_set_set_string (cs, str, actdata->command);
  }

  gconf_client_commit_change_set (gconf_client_get_default (),
				  cs, FALSE, NULL);

  gconf_change_set_unref (cs);
}

static gchar *
gkb_load_pref(GKB *gkb, const gchar * key, const gchar * defaultv)
{
 gchar * value;

 value = gconf_applet_get_string (key);
 
 if (value == NULL) {
  value = g_strdup (defaultv);
 }
 return value;
}

gchar *
gkb_default_map_file (void)
{
    return g_build_path (G_DIR_SEPARATOR_S,
			 g_get_home_dir (),
			 ".gkb_default.xmm",
			 NULL);
}

GkbKeymap *
loadprop (GKB *gkb, int i)
{
  GkbKeymap *actdata;
  gchar *buf;
  gchar *default_cmd;
  gchar *default_map_file = gkb_default_map_file ();

  actdata = g_new0 (GkbKeymap, 1);


  buf = g_strdup_printf ("name_%d",i);
  actdata->name = gkb_load_pref (gkb, buf, (i?"US 105 key keyboard":"Gnome Keyboard Default"));
  g_free (buf);

  buf = g_strdup_printf ("label_%d", i);
  actdata->label = gkb_load_pref (gkb, buf, (i?"us":"default"));
  g_free (buf);
  
  buf = g_strdup_printf ("country_%d", i);
  actdata->country = gkb_load_pref (gkb, buf, (i?"United States":"Gnome Keyboard"));
  g_free (buf);
  
  buf = g_strdup_printf ("lang_%d", i);
  actdata->lang = gkb_load_pref (gkb, buf, (i?"English":"International"));
  g_free (buf);
  
  buf = g_strdup_printf ("flag_%d", i);
  actdata->flag = gkb_load_pref (gkb, buf, (i?"us.png":"gkb.png"));
  g_free (buf);
  
  buf = g_strdup_printf ("command_%d", i);
  default_cmd = g_strdup_printf ("xmodmap %s", default_map_file);

  actdata->command = gkb_load_pref (gkb, buf, (i?"gkb_xmmap us":default_cmd));
  g_free (default_cmd);
  g_free(buf);
  g_free (default_map_file);

  actdata->pixbuf = NULL;
  if (gkb->applet != NULL)
  gkb->orient = panel_applet_get_orient (PANEL_APPLET (gkb->applet));

  return actdata;
}

static void
do_at_first_run (void) 
{
	gchar *cmd, *default_map_file = gkb_default_map_file ();

	if (!g_file_test (default_map_file, G_FILE_TEST_EXISTS)) {
		cmd = g_strdup_printf ("echo \"! This file has generated by Gnome Keyboard applet\" > %s",
				       default_map_file);
		system (cmd);
		g_free (cmd);

		cmd = g_strdup_printf ("xmodmap -pke >> %s", default_map_file);
		system (cmd);
		g_free (cmd);
	}
	g_free (default_map_file);
}

static void
load_properties (GKB * gkb)
{
  GkbKeymap *actdata;
  gchar *text;
  int i;

  gkb->maps = NULL;
	gkb->n = gconf_applet_get_int ("num");
  gkb->key = gkb_load_pref (gkb, "key", "Alt-Shift_L");
  gkb->old_key = g_strdup (gkb->key);

  convert_string_to_keysym_state (gkb->key, &gkb->keysym, &gkb->state);
  gkb->old_keysym = gkb->keysym;
  gkb->old_state = gkb->state;

  gkb->is_small = gconf_applet_get_bool ("small");
  text = gkb_load_pref (gkb, "mode", "Flag and Label");
  gkb->mode = gkb_util_get_mode_from_text (text);
  g_free (text);

  if (gkb->n == 0)
    {
      actdata = loadprop (gkb, 0);
      gkb->maps = g_list_append (gkb->maps, actdata);
      gkb->n++;

      actdata = loadprop (gkb, 1);
      gkb->maps = g_list_append (gkb->maps, actdata);
      gkb->n++;
    }
  else
    for (i = 0; i < gkb->n; i++)
      {
	actdata = loadprop (gkb, i);
	gkb->maps = g_list_append (gkb->maps, actdata);
      }
	if (gkb->applet != NULL)
		gkb->orient = panel_applet_get_orient (PANEL_APPLET (gkb->applet));
	gkb_save_session (gkb);
}

static void
config_notify (GConfClient *client,
	       guint        cnxn_id,
	       GConfEntry  *entry,
	       gpointer     user_data)
{
	/*g_warning ("ba da ding"); */
}

GKB *
gkb_new (void)
{
	GKB *gkb;

	gkb = g_new0 (GKB, 1);
	gkb->n = 0;
	  gkb->cur = 0;

	do_at_first_run ();
	load_properties (gkb);
	gnome_window_icon_set_default_from_file (GNOME_ICONDIR"/gkb.png");

	gkb->keymap = g_list_nth_data (gkb->maps, 0);

	gconf_client_notify_add (gconf_client_get_default (),
		GKB_GCONF_ROOT, config_notify, gkb, NULL, NULL);
	return gkb;
}
