/*
 * Copyright (C) 2002-2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: mediamarks.c,v 1.37 2004/12/17 00:33:00 dsalt Exp $
 *
 * mediamarks - load
 *            - save
 *            - edit
 * functions
 */

#include "globals.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <glib.h>

#include "utils.h"
#include "engine.h"
#include "mediamarks.h"
#include "menu.h"
#include "playlist.h"
#include "play_item.h"

static GtkTreeStore *mm_store;
static GtkWidget    *manage_dlg, *cat_dlg, *cat_entry, *mm_tree;
static GtkTreeIter   cat_iter;
static gboolean      is_visible, cat_edited;

static void menu_cb (GtkWidget* widget, gpointer data) {

  play_item_t *item_orig = (play_item_t *) data;

  if (item_orig) {
    play_item_t *item;
    int          pos;

    item = play_item_copy (item_orig);

    pos = playlist_add (item, -1);
    playlist_play (pos);
  }
}

static void gen_items (GtkMenuShell *submenu, GtkTreeIter *iter) {

  int pos;

  pos = 4;

  do {

    GValue       v;
    play_item_t *play_item;
    GtkWidget    *item;
    char         *id;
      
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      iter, 0, &v);
    id = g_value_peek_pointer (&v);

    if (id) {
      item = gtk_menu_item_new_with_label (id);
      g_value_unset (&v);

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				iter, 2, &v);
      play_item = g_value_peek_pointer (&v);
      
      if (play_item) {
	
	gtk_signal_connect (GTK_OBJECT (item), "activate", 
			    GTK_SIGNAL_FUNC (menu_cb), (gpointer) play_item);
	
      } else {
	
	GtkWidget   *submenu;
	GtkTreeIter  children;
	
	submenu = gtk_menu_new ();
	
	if (gtk_tree_model_iter_children (GTK_TREE_MODEL(mm_store), &children, iter))
	  gen_items (GTK_MENU_SHELL(submenu), &children);
	
	gtk_menu_item_set_submenu (GTK_MENU_ITEM (item), submenu);
	
      }
      
      gtk_menu_shell_insert (submenu, item, pos);
      gtk_widget_show(item);

      pos++;
    }

    g_value_unset (&v);
      
  } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(mm_store), iter));
}

static void update_menu (GtkMenuShell *submenu) {

  GList        *olditems;
  GtkWidget    *item;
  GtkTreeIter   iter;

  if (!submenu)
    return;

  /*
   * remove old items (if any)
   */

  olditems = gtk_container_children (GTK_CONTAINER(submenu));
  while ( (item = (GtkWidget *) g_list_nth_data (olditems, 4)) ) {
    gtk_container_remove (GTK_CONTAINER (submenu), item);
    olditems = gtk_container_children (GTK_CONTAINER (submenu));
  }

  /*
   * add current items
   */

  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (mm_store), &iter))
    gen_items (submenu, &iter);
}

/* 
 * support new xml based mediamarks file format
 */

static void get_items (xml_node_t *node, int depth, GtkTreeIter *parent) {
  /* assumption: *one* of (depth == 0) or (parent == NULL) */

  while (node) {

    if (!strcasecmp (node->name, "sub")) {

      GtkTreeIter iter;
      char       *id;

      id = xml_parser_get_property (node, "name");

      gtk_tree_store_append (mm_store, &iter, parent);
      gtk_tree_store_set (mm_store, &iter, 0, id, -1);

      get_items (node->child, depth+1, &iter);

    } else if (!strcasecmp (node->name, "entry")) {
      play_item_t *play_item = play_item_load (node->child);
      GtkTreeIter  iter;

      gtk_tree_store_append (mm_store, &iter, parent);
      gtk_tree_store_set (mm_store, &iter, 0, play_item->title, 1, 
			  play_item->mrl, 2, play_item, -1);

    } else {
      printf (_("mediamarks: error, unknown node type %s\n"), node->name);
    }

    node = node->next;
  }
}

static void load_new_file (xml_node_t *node) {

  if (!strcasecmp (node->name, "gxinemm")) {

    get_items (node->child, 0, NULL);
  } else {
    printf (_("mediamarks: ERROR, root node must be GXINEMM\n"));
  }
}

/*
 * legacy support for old mediamarks file format
 */

static char * read_line (FILE *f) {
  char  str[1024];
  char *str2;
  int   len;

  str2 = fgets (str, 1024, f);

  if (!str2)
    return NULL;

  len = strlen (str);
  if ( (len>0) && (str[len-1] == 10))
    str[len-1] = 0;

  return strdup (str);
}

static void load_old_file (char *fname) {

  FILE        *f;
  int          depth;
  GtkTreeIter  parent;
  GtkTreeIter  iter;

  gtk_tree_model_get_iter_first ( GTK_TREE_MODEL (mm_store), &parent);

  f = fopen (fname, "r");
  if (f) {
    
    depth = 0;

    while (1) {
      char *id, *str;

      id = read_line (f);
      if (!id)
	break;

      if (!strncmp (id, "<sub>", 5)) {

	gtk_tree_store_append (mm_store, &iter, depth ? &parent : NULL);
	gtk_tree_store_set (mm_store, &iter, 0, &id[5], -1);

	parent = iter;
	depth++;

      } else if (!strncmp (id, "</sub>", 6)) {
      
	/* sub-tree ends here */

	gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store), 
				    &iter, &parent);
	depth--;
	if (depth<0) {

	  printf (_("mediamarks: media marks file %s corrupted - ignoring rest of file\n"),
		  fname);
	  return;
	}
	parent = iter;

      } else { 
	
	play_item_t *play_item;
	char        *mrl;
	int          time = 0;

	mrl = read_line (f);
	if (!mrl)
	  break;

	str = read_line (f);
	if (!str)
	  break;
	sscanf (str, "%d", &time);

	play_item = play_item_new (id, mrl, time);

	gtk_tree_store_append (mm_store, &iter, depth ? &parent : NULL);
	gtk_tree_store_set (mm_store, &iter, 0, id, 1, mrl, 2, play_item, -1);
      }

      /* discard newline */
      str = read_line (f);
      if (!str)
	break;
    }
    fclose (f);
  } else {

    printf (_("mediamarks: load failed!\n"));

  }
}

static void cat_response_cb (GtkDialog* widget, int response, gpointer data)
{
  switch (response)
  {
  case GTK_RESPONSE_OK:
    cat_edited = TRUE;
  default:
    gtk_widget_hide (cat_dlg);
    gtk_main_quit();
  }
}

static gboolean do_category_edit (void)
{
  cat_edited = FALSE;
  gtk_widget_show_all (cat_dlg);
  if (!GTK_WINDOW(cat_dlg)->modal)
    gtk_window_set_modal (GTK_WINDOW(cat_dlg), TRUE);
  gtk_widget_grab_focus (cat_entry);

  gtk_main();

  return cat_edited;
}

static void edit_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = GTK_TREE_VIEW (mm_tree);
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    if (play_item) {
      logprintf ("mediamarks: got a play item\n");
      if (play_item_edit (play_item, PLAY_ITEM_MEDIAMARK, NULL))
      {
	logprintf ("mediamarks: play item title after edit : %s\n",
		   play_item->title);
	gtk_tree_store_set (mm_store, &iter,
			    0, play_item->title,
			    1, play_item->mrl, 2, play_item, -1);
      }
    } else {
      GValue       v;
      char        *name;

      memcpy (&cat_iter, &iter, sizeof (cat_iter));

      memset (&v, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				&iter, 0, &v);
      name = (char *) g_value_peek_pointer (&v);
      gtk_entry_set_text (GTK_ENTRY (cat_entry), name);
      g_value_unset (&v);

      if (do_category_edit ())
	gtk_tree_store_set (mm_store, &cat_iter, 0,
			    gtk_entry_get_text (GTK_ENTRY(cat_entry)), -1);
    }
    g_value_unset (&v);
  }
}

static void delete_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = GTK_TREE_VIEW (mm_tree);
  GtkTreeSelection  *sel;
  GtkTreeIter        iter;  

  sel = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter))
    gtk_tree_store_remove (mm_store, &iter);
}

static void new_category_cb (GtkWidget* widget, gpointer data) {

  GtkTreeView       *tree = GTK_TREE_VIEW (mm_tree);
  GtkTreeSelection  *sel;
  GtkTreeIter        iter, parent;   
  gboolean           in_root = TRUE;

  /*
   * find out where to create the new section
   */

  sel     = gtk_tree_view_get_selection (tree);

  if (gtk_tree_selection_get_selected (sel, NULL, &iter)) {
    GValue       v;
    play_item_t *play_item;

    /*
     * is this a section or a play item ?
     */

    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      &iter, 2, &v);
    play_item = g_value_peek_pointer (&v);
    if (play_item) {
      in_root = !gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store),
					     &parent, &iter);
    } else {
      GtkTreePath *path;
      path = gtk_tree_model_get_path (GTK_TREE_MODEL (mm_store), &iter);

      if (gtk_tree_view_row_expanded (tree, path)) {
	in_root = 0;
	memcpy (&parent, &iter, sizeof (iter));
      } else {
	in_root = !gtk_tree_model_iter_parent (GTK_TREE_MODEL (mm_store),
					       &parent, &iter);
      }

      gtk_tree_path_free (path);
    }
    g_value_unset (&v);
  }

  /*
   * show dialog, ask for section name
   */

  gtk_entry_set_text (GTK_ENTRY (cat_entry), _("New Category"));

  if (do_category_edit ())
  {
    /* create new category */
    gtk_tree_store_append (mm_store, &cat_iter, in_root ? NULL : &parent);
    gtk_tree_store_set (mm_store, &cat_iter, 0,
			gtk_entry_get_text (GTK_ENTRY(cat_entry)), -1);
  }
}

static void tree_changed_cb (GtkTreeModel *treemodel,
			     GtkTreePath *arg1,
			     gpointer user_data) {
  update_menu (media_menu[0]);
  update_menu (media_menu[1]);
}

static int load_mediamarks (gchar *fname)
{
  char *mmfile = read_entire_file (fname, NULL);

  if (mmfile) {
    xml_node_t *node;

    xml_parser_init (mmfile, strlen (mmfile), XML_PARSER_CASE_INSENSITIVE);

    if (xml_parser_build_tree (&node)<0) {
      load_old_file (fname);
    } else 
      load_new_file (node);

    free (mmfile);
  } else
    return 0;

  return 1;
}

void mm_import (void) {

  gchar *fname;

  fname = modal_file_dialog (_("Select media marks file to load..."), NULL);

  if (fname) {

    if (!load_mediamarks (fname) && errno != ENOENT)
      display_error (_("Media marks load failed"),
		     _("Couldn't load media marks file %s\n%s"),
		     fname, strerror (errno));
  }
}

void mm_add_show () {

  play_item_t *play_item;
  GtkTreeIter  iter;
  int          pos, pos_time, len;
  char	      *dot;
  const char  *title = xine_get_meta_info (stream, XINE_META_INFO_TITLE);

  play_item = play_item_copy (playlist_get_item (playlist_get_list_pos()));

  /* default title? try to get title from meta-info */
  dot = strrchr (play_item->mrl, '.');
  if (!dot)
    dot = play_item->mrl + strlen (play_item->mrl);
  len = strlen (play_item->title);
  if (title && dot - len > play_item->mrl &&
      !strncmp (dot - len, play_item->title, len))
  {
    free (play_item->title);
    play_item->title = strdup (title);
  }

  xine_get_pos_length (stream, &pos, &pos_time, &len);
  play_item->start_time = (len >= 0) ? pos_time : 0;

  if (play_item_edit (play_item, PLAY_ITEM_MEDIAMARK_NEW, title))
  {
    gtk_tree_store_append (mm_store, &iter, NULL);
    gtk_tree_store_set (mm_store, &iter, 0, play_item->title,
			1, play_item->mrl, 2, play_item, -1);
  }
}


void mm_manage_show () {

  if (is_visible) {
    is_visible = FALSE;
    gtk_widget_hide (manage_dlg);
    mm_save();
  } else {
    is_visible = TRUE;
    gtk_widget_show_all (manage_dlg);
  }
}

static void print_depth (FILE *f, int depth) {
  depth += 2;

  while (depth) {
    fprintf (f, " ");
    depth --;
  }
}

static void save_items (FILE *f, int depth, GtkTreeIter *iter) {

  do {
    
    GValue        v;
    play_item_t  *play_item;
    char         *id;
      
    memset (&v, 0, sizeof (GValue));
    gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
			      iter, 0, &v);
    id = g_value_peek_pointer (&v);

    if (id) {

      GValue vi;

      memset (&vi, 0, sizeof (GValue));
      gtk_tree_model_get_value (GTK_TREE_MODEL (mm_store),
				iter, 2, &vi);
      play_item = g_value_peek_pointer (&vi);
      
      if (play_item) {
	
	play_item_save (play_item, f, depth);
	
      } else {

	GtkTreeIter children;

	print_depth (f, depth);
	fprintf (f, "<SUB NAME=\"%s\">\n", id);
	
	if (gtk_tree_model_iter_children (GTK_TREE_MODEL(mm_store), &children, iter))
	  save_items (f, depth+2, &children);
	
	print_depth (f, depth);
	fprintf (f, "</SUB>\n");
	
      }
      
      g_value_unset (&vi);
    }

    g_value_unset (&v);
      
  } while (gtk_tree_model_iter_next (GTK_TREE_MODEL(mm_store), iter));
}

void mm_save (void) {

  FILE    *f;
  gchar   *fname;

  fname = g_strconcat(g_get_home_dir(), "/.gxine/mediamarks", NULL);

  f = fopen (fname, "w") ;

  if (f) {

    GtkTreeIter iter;

    fprintf (f, "<GXINEMM VERSION=\"1\">\n");

    if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (mm_store), &iter))
      save_items (f, 0, &iter);

    fprintf (f, "</GXINEMM>\n");

    if (ferror (f))
      printf (_("mediamarks: save to '%s' failed: %s\n"),
	      fname, strerror (errno));
    if (fclose (f))
      printf (_("mediamarks: save to '%s' failed: %s\n"),
	      fname, strerror (errno));
  } else
    printf (_("mediamarks: save to '%s' failed: %s\n"),
	    fname, strerror (errno));

  g_free(fname);
}

static JSBool js_mm_add_show (JSContext *cx, JSObject *obj, uintN argc, 
			      jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("mm_add_show");

  mm_add_show ();
  
  return JS_TRUE;
}

static JSBool js_mm_manage_show (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("mm_manage_show");

  mm_manage_show ();
  
  return JS_TRUE;
}

static JSBool js_import_mediamarks (JSContext *cx, JSObject *obj, uintN argc, 
				    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */

  se_log_fncall ("import_mediamarks");

  mm_import ();
  
  return JS_TRUE;
}

static void mm_response_cb (GtkDialog *dbox, int response, gpointer data)
{
  switch (response)
  {
  case GTK_RESPONSE_ACCEPT:
    mm_save ();
    break;
  default:
    is_visible = FALSE;
    gtk_widget_hide (manage_dlg);
  }
}

void mediamarks_init (void) {

  GtkWidget            *tree, *b, *scrolled_window, *hbox, *label;
  gchar                *fname;
  GtkCellRenderer      *cell;
  GtkTreeViewColumn    *column;
   
  /*
   * init tree store
   */
 
  mm_store = gtk_tree_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_POINTER);

  /* 
   * now parse the mediamarks file
   */

  fname = g_strconcat(g_get_home_dir(), "/.gxine/mediamarks", NULL);

  if (!load_mediamarks (fname)) {
    /* load sample mediamarks */

    g_free (fname);

    fname = g_strconcat (miscdir, "/mediamarks", NULL);
    load_mediamarks (fname);
  }

  g_free(fname);

  g_signal_connect (G_OBJECT (mm_store), "row-changed",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-deleted",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-has-child-toggled",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "row-inserted",
		    G_CALLBACK (tree_changed_cb), NULL);
  g_signal_connect (G_OBJECT (mm_store), "rows-reordered",
		    G_CALLBACK (tree_changed_cb), NULL);

  update_menu (media_menu[0]);
  update_menu (media_menu[1]);

  /*
   * create (for now invisible) mm_manage dialog
   */

  manage_dlg = gtk_dialog_new_with_buttons (_("Manage media marks..."), NULL, 0,
				GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
				GTK_STOCK_CLOSE, GTK_RESPONSE_DELETE_EVENT,
				NULL);
  gtk_window_set_default_size (GTK_WINDOW (manage_dlg), 400, 250);
  g_signal_connect (GTK_OBJECT(manage_dlg), "response",
		    G_CALLBACK(mm_response_cb), NULL);
  hide_on_delete (manage_dlg, &is_visible);

  /* add a nice tree view widget here */

  mm_tree = tree = gtk_tree_view_new_with_model (GTK_TREE_MODEL(mm_store));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree), TRUE);

  cell = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (_("Media mark"), cell,
						     "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree),
			       GTK_TREE_VIEW_COLUMN (column));

  column = gtk_tree_view_column_new_with_attributes ("MRL", cell,
						     "text", 1, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tree),
			       GTK_TREE_VIEW_COLUMN (column));
  
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (tree), TRUE);

  scrolled_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled_window),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add (GTK_CONTAINER (scrolled_window), tree);

  g_signal_connect (GTK_OBJECT (tree), "row-activated",
		    G_CALLBACK (edit_cb), NULL);

  /* add edit button bar */
  hbox = gtk_hbox_new (FALSE, 2);

  b = gtk_button_new_with_mnemonic (_("_New media mark"));
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(mm_add_show), NULL);

  b = gtk_button_new_with_mnemonic (_("New _category"));
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(new_category_cb), tree);

  b = gtk_button_new_with_mnemonic (_("_Edit"));
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(edit_cb), NULL);

  b = gtk_button_new_from_stock (GTK_STOCK_DELETE);
  gtk_box_pack_start (GTK_BOX(hbox), b, FALSE, FALSE, 0);
  g_signal_connect (GTK_OBJECT(b), "clicked",
		    G_CALLBACK(delete_cb), NULL);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(manage_dlg)->vbox), hbox,
		    FALSE, FALSE, 2);

  gtk_box_pack_start (GTK_BOX(GTK_DIALOG(manage_dlg)->vbox), scrolled_window,
		      TRUE, TRUE, 2);

  is_visible = FALSE;

  /*
   * create (for now invisible) category name entry dialog
   */

  cat_dlg = gtk_dialog_new_with_buttons (_("Enter category name..."), NULL, 0,
				GTK_STOCK_CANCEL, GTK_RESPONSE_DELETE_EVENT,
				GTK_STOCK_OK, GTK_RESPONSE_OK,
				NULL);
  hide_on_delete (cat_dlg, NULL);
  g_signal_connect (GTK_OBJECT (cat_dlg), "response",
		    G_CALLBACK (cat_response_cb), NULL);
  gtk_dialog_set_default_response (GTK_DIALOG(cat_dlg), GTK_RESPONSE_OK);

  /* all we have is a simple text entry */

  hbox = gtk_hbox_new (2, 2);

  label = gtk_label_new (_("Category name:"));

  gtk_box_pack_start (GTK_BOX (hbox), label,
		      FALSE, FALSE, 0);

  cat_entry = gtk_entry_new ();
/*
  gtk_signal_connect (GTK_OBJECT(cat_entry), "activate",
		      GTK_SIGNAL_FUNC (cat_close_cb),
		      cat_dlg);
*/
  gtk_box_pack_start_defaults (GTK_BOX (hbox), cat_entry);

  gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (cat_dlg)->vbox), hbox);

  /* script engine functions */

  {
    static const se_f_def_t defs[] = {
      { "mm_add_show", js_mm_add_show, 0, 0,
	SE_GROUP_DIALOGUE, NULL, NULL },
      { "mm_manage_show", js_mm_manage_show, 0, 0,
	SE_GROUP_DIALOGUE, NULL, NULL },
      { "import_mediamarks", js_import_mediamarks, 0, 0,
	SE_GROUP_FILE, NULL, NULL },
      { NULL }
    };
    se_defuns (gse, NULL, defs);
  }
}

