/*
 * Copyright (C) 2004 the xine-project
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 * $Id: player.c,v 1.23 2004/12/11 00:40:18 dsalt Exp $
 *
 * player object 
 */

#include "globals.h"

#include <pthread.h>
#include <xine.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtkradiomenuitem.h>

#include "globals.h"
#include "utils.h"
#include "playlist.h"
#include "gtkvideo.h"
#include "snapshot.h"
#include "engine.h"
#include "player.h"

xine_stream_t *stream; /* global stream */
xine_audio_port_t *audio_port;
xine_video_port_t *video_port;

/*
 * async play machanism
 *
 * always start xine engine from a seperate thread so the gui stays responsive
 */

static int             goto_pos, goto_time;
static pthread_t       play_thread;
static char           *cur_mrl = NULL;

static void *play_exec (void *queue_gen) {

  pthread_mutex_lock (&engine_lock);

  gdk_threads_enter();
  infobar_clear (bar);
  infobar_line (bar, 0, _("opening..."));
  infobar_line (bar, 1, cur_mrl);
  gdk_threads_leave();

  if (!xine_open (stream, cur_mrl)) {

    gdk_threads_enter();
    switch (xine_get_error (stream)) {
    case XINE_ERROR_NO_INPUT_PLUGIN:

      display_error (_("xine engine error"),
		     _("xine engine failed to start.\n\n"
		     "No input plugin found.\n"
		     "Maybe the file does not exist, has wrong permissions or\n"
		     "URL syntax error."));

      break;
    case XINE_ERROR_NO_DEMUX_PLUGIN:

      display_error (_("xine engine error"),
		     _("xine engine failed to start.\n\n"
		     "No demuxer found - stream format not recognised."));

      break;

    case XINE_ERROR_MALFORMED_MRL:
      display_error (_("xine engine error"),
		     "xine engine failed to start.\n\n"
		     "Malformed MRL.");

      break;

    default:
      printf (_("error: %d\n"), xine_get_error (stream));
      display_error (_("xine engine error"),
		     _("xine engine failed to start.\n\n"
		     "Unknown error."));
    }

    infobar_clear (bar);
    infobar_line (bar, 0, "gxine %s", VERSION);

    gdk_threads_leave();
    goto done;
  }

  gdk_threads_enter();
  ui_set_status (playlist_showing_logo () ? UI_STOP : UI_PLAY);
  infobar_show_metadata ();
  gdk_threads_leave();

  xine_play (stream, goto_pos, goto_time);
  goto_pos = 0;
  goto_time = 0;

  /* correct for possible stream changes */
  gdk_threads_enter();
  ui_set_status (playlist_showing_logo () ? UI_STOP : UI_PLAY);
  infobar_show_metadata ();
  gdk_threads_leave();

done:
  pthread_mutex_unlock (&engine_lock);
  pthread_exit (NULL);
  return NULL;
}

void player_launch (const char *mrl, int pos, int pos_time) {
  int err;

  goto_pos  = pos;
  goto_time = pos_time;
  free (cur_mrl);
  cur_mrl = strdup(mrl);

  /*
   * start seperate thread, so xine_open will not block ui
   */

  if ((err = pthread_create (&play_thread,
			     NULL, play_exec, NULL)) != 0) {
    printf (_("playlist: can't create new thread (%s)\n"),
	    strerror(err));
    abort();
  }

  pthread_detach (play_thread);
}


static JSBool get_time (JSContext *cx, JSObject *obj, uintN argc, 
		     jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   pos, pos_time, len;

  se_log_fncall ("get_time");

  xine_get_pos_length (stream, &pos, &pos_time, &len);

  *rval = INT_TO_JSVAL (pos_time);

  return JS_TRUE;
}

static JSBool js_get_speed (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   speed;

  se_log_fncall ("get_speed");

  speed = xine_get_param (stream, XINE_PARAM_SPEED);

  *rval = INT_TO_JSVAL (speed);

  return JS_TRUE;
}

static JSBool js_get_volume (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   volume;

  se_log_fncall ("get_volume");

  volume = xine_get_param (stream, XINE_PARAM_AUDIO_VOLUME);

  *rval = INT_TO_JSVAL (volume);

  return JS_TRUE;
}

static JSBool js_get_mute (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval)
{
  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int mute;
  se_log_fncall ("get_mute");
  mute = xine_get_param (stream, XINE_PARAM_AUDIO_MUTE);
  *rval = BOOLEAN_TO_JSVAL (mute);
  return JS_TRUE;
}

static JSBool js_get_zoom (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  /* se_t *se = (se_t *) JS_GetContextPrivate(cx); */
  int   zoom;

  se_log_fncall ("get_zoom");

  zoom = xine_get_param (stream, XINE_PARAM_VO_ZOOM_X);

  *rval = INT_TO_JSVAL (zoom);

  return JS_TRUE;
}

static JSBool controls_pause (JSContext *cx, JSObject *obj, uintN argc, 
			      jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   pause_state=-1;

  se_log_fncall ("pause");
  se_argc_check_max (1, "pause");

  if (argc==1) {
    se_arg_is_int (0, "pause");
    JS_ValueToInt32 (cx, argv[0], &pause_state);
  }

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  switch (pause_state) {
  case 0: /* pause off */
    ui_set_status (UI_PLAY);
    xine_set_param (stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
    break;
  case 1: /* pause on */
    ui_set_status (UI_PAUSE);
    xine_set_param (stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    break;
  default: /* toggle */
    if (xine_get_param (stream, XINE_PARAM_SPEED) != XINE_SPEED_PAUSE)
    {
      ui_set_status (UI_PAUSE);
      xine_set_param (stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    }
    else
    {
      ui_set_status (UI_PLAY);
      xine_set_param (stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
    }
  }

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool controls_stop (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  se_log_fncall ("stop");

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  xine_stop (stream);

  playlist_logo ();

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_speed (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   speed;

  se_log_fncall ("set_speed");

  se_argc_check (1, "set_speed");
  se_arg_is_int (0, "set_speed");

  JS_ValueToInt32 (cx, argv[0], &speed);

  if (speed<0)
    speed = 0;

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  if (speed == XINE_SPEED_NORMAL)
    ui_set_status (UI_PLAY);
  else if (speed > XINE_SPEED_NORMAL)
    ui_set_status (UI_FAST_FORWARD);
  else if (speed)
    ui_set_status (UI_PLAY_SLOW);
  else
    ui_set_status (UI_PAUSE);

  xine_set_param (stream, XINE_PARAM_SPEED, speed);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_volume (JSContext *cx, JSObject *obj, uintN argc, 
			    jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   volume;

  se_log_fncall ("set_volume");
  se_argc_check (1, "set_volume");
  se_arg_is_int (0, "set_volume");

  JS_ValueToInt32 (cx, argv[0], &volume);

  if (volume<0)
    volume = 0;

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  ui_set_control_adjustment (Control_VOLUME, volume);
  ui_xine_set_param_from_adjustment (Control_VOLUME);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_mute (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  JSBool mute;

  se_log_fncall ("set_mute");
  se_argc_check_max (1, "set_mute");

  if (argc == 1)
  {
    se_arg_is_int (0, "set_mute");
    JS_ValueToBoolean (cx, argv[0], &mute);
  }
  else
    mute = !xine_get_param (stream, XINE_PARAM_AUDIO_MUTE);

  if (engine_mutex_trylock (se))
    return JS_TRUE;
  ui_set_status (mute ? UI_AUDIO_MUTE : UI_AUDIO_UNMUTE);
  xine_set_param (stream, XINE_PARAM_AUDIO_MUTE, mute);
  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_zoom (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   zoom;

  se_log_fncall ("set_zoom");
  se_argc_check (1, "set_zoom");
  se_arg_is_int (0, "set_zoom");

  JS_ValueToInt32 (cx, argv[0], &zoom);

  if (zoom<XINE_VO_ZOOM_MIN)
    zoom = XINE_VO_ZOOM_MIN;
  if (zoom>XINE_VO_ZOOM_MAX)
    zoom = XINE_VO_ZOOM_MAX;

  se_log ("zoom is %d\n", zoom);

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  xine_set_param (stream, XINE_PARAM_VO_ZOOM_X, zoom);
  xine_set_param (stream, XINE_PARAM_VO_ZOOM_Y, zoom);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

static JSBool js_set_video_size (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   zoom;

  se_log_fncall ("set_video_size");
  se_argc_check (1, "set_video_size");
  se_arg_is_int (0, "set_video_size");

  JS_ValueToInt32 (cx, argv[0], &zoom);

  if (zoom<XINE_VO_ZOOM_MIN)
    zoom = XINE_VO_ZOOM_MIN;
  if (zoom>XINE_VO_ZOOM_MAX)
    zoom = XINE_VO_ZOOM_MAX;
  gtk_video_rescale (GTK_VIDEO(gtv), (double) zoom / 100.0);

  return JS_TRUE;
}

static JSBool js_set_sub (JSContext *cx, JSObject *obj, uintN argc, 
			  jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   channel;

  se_log_fncall ("set_sub");
  se_argc_check (1, "set_sub");
  se_arg_is_int (0, "set_sub");

  JS_ValueToInt32 (cx, argv[0], &channel);

  xine_set_param (stream, XINE_PARAM_SPU_CHANNEL, channel);

  return JS_TRUE;
}

static JSBool js_set_fullscreen (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   fs;

  se_log_fncall ("set_fullscreen");
  se_argc_check_max (1, "set_fullscreen");

  if (argc==1) {
    se_arg_is_int (0, "set_fullscreen");
    JS_ValueToInt32 (cx, argv[0], &fs);
  } else {
    fs = !gtk_video_is_fullscreen (GTK_VIDEO(gtv));
  }

  fullscreen_menu_item[0]->active = fs;
  fullscreen_menu_item[1]->active = fs;
  gtk_video_set_fullscreen (GTK_VIDEO(gtv), fs);
  if (fs)
    window_toolbar_restore ();
  else
    window_toolbar_reset ();

  return JS_TRUE;
}

static JSBool js_set_deinterlace (JSContext *cx, JSObject *obj, uintN argc, 
				 jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   di;

  se_log_fncall ("set_deinterlace");
  se_argc_check_max (1, "set_deinterlace");

  if (argc==1) {
    se_arg_is_int (0, "set_deinterlace");
    JS_ValueToInt32 (cx, argv[0], &di);
  } else {
    di = !xine_get_param (stream, XINE_PARAM_VO_DEINTERLACE);
  }

  xine_set_param (stream, XINE_PARAM_VO_DEINTERLACE, di);

  /* Hmm, should be a don't-emit-signal method for this... */
  deinterlace_menu_item[0]->active = di;
  deinterlace_menu_item[1]->active = di;

  return JS_TRUE;
}

static JSBool js_set_auto_resize (JSContext *cx, JSObject *obj, uintN argc,
				 jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int   ar;

  se_log_fncall ("set_auto_resize");
  se_argc_check_max (1, "set_auto_resize");

  if (argc==1) {
    se_arg_is_int (0, "set_auto_resize");
    JS_ValueToInt32 (cx, argv[0], &ar);
  } else {
    ar = !gtk_video_get_auto_resize (GTK_VIDEO(gtv));
  }

  gtk_video_set_auto_resize (GTK_VIDEO(gtv), ar);
  auto_resize_menu_item[0]->active = ar;
  auto_resize_menu_item[1]->active = ar;

  return JS_TRUE;
}

static JSBool js_set_aspect (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  int i, aspect, count = g_slist_length (aspect_menu_items[0]);

  se_log_fncall ("set_aspect");
  se_argc_check_max (1, "set_aspect");

  if (argc==1) {
    se_arg_is_int (0, "set_aspect");
    JS_ValueToInt32 (cx, argv[0], &aspect);
  } else {
    aspect = xine_get_param (stream, XINE_PARAM_VO_ASPECT_RATIO);
    if (aspect >= count)
      aspect = XINE_VO_ASPECT_AUTO;
    else
      aspect = (aspect + 1) % count;
  }

  xine_set_param (stream, XINE_PARAM_VO_ASPECT_RATIO, aspect);

  for (i = 0; i < 2; ++i)
  {
    GSList *menuitem = aspect_menu_items[i];
    while (--count >= 0)
    {
      /* Hmm, should be a don't-emit-signal method for this... */
      GTK_CHECK_MENU_ITEM (menuitem->data)->active = (aspect == count);
      menuitem = menuitem->next;
    }
  }

  return JS_TRUE;
}

static JSBool js_snapshot (JSContext *cx, JSObject *obj, uintN argc, 
			   jsval *argv, jsval *rval) {

  se_t     *se = (se_t *) JS_GetContextPrivate(cx);
  JSString *str;
  char     *fname = NULL;
  int	    scale = -1, blend = -1;

  se_log_fncall ("snapshot");
  se_argc_check_range (0, 3, "snapshot");

  switch (argc) /* note fall-through */
  {
  case 3:
    se_arg_is_int (2, "snapshot");
    JS_ValueToInt32 (cx, argv[0], &blend);
  case 2:
    se_arg_is_int (1, "snapshot");
    JS_ValueToInt32 (cx, argv[0], &scale);
  case 1:
    se_arg_is_string (0, "snapshot");
    str = JS_ValueToString (cx, argv[0]);
    fname = JS_GetStringBytes (str);
  }

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  make_snapshot (fname, scale, blend);

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

/**************************************************************
 *
 * player.controls class & object
 *
 * adapted from http://users.skynet.be/saw/SpiderMonkey.htm
 *
 **************************************************************/

/*
 * controls object properties
 */

#if 0
int controls_prop_cb (void *user_data, char *property, int *num) {
  /* FIXME */
  return 1;
}

#define CONTROLS_PROP_AUDIO_LANGUAGE_COUNT          0
#define CONTROLS_PROP_CURRENT_AUDIO_LANGUAGE        1
#define CONTROLS_PROP_CURRENT_AUDIO_LANGUAGE_INDEX  2
#define CONTROLS_PROP_CURRENT_ITEM                  3
#define CONTROLS_PROP_CURRENT_MARKER                4
#define CONTROLS_PROP_CURRENT_POSITION              5
#define CONTROLS_PROP_CURRENT_POSITION_STRING       6
#define CONTROLS_PROP_CURRENT_POSITION_TIMECODE     7
#define CONTROLS_PROP_IS_AVAILABLE                  8

JSBool controls_JSGetProperty (JSContext *cx, JSObject *obj, jsval id, jsval *vp) {

  se_log_fncall ("get_property");

  if (JSVAL_IS_STRING (id)) {
    JSString *str;
    char     *prop;

    se_log ("id is a string\n");

    str = JS_ValueToString (cx, id);
    prop = JS_GetStringBytes (str);

    se_log ("id = '%s'\n", JS_GetStringBytes (str));

    if (!strcmp (prop, "audioLanguageCount")) {
      *vp = INT_TO_JSVAL (1);
    } else if (!strcmp (prop, "currentPosition")) {
      *vp = INT_TO_JSVAL (0);
    }
  }
#if 0
  if (JSVAL_IS_INT(id)) 
    {
      Customer *priv = (Customer *) JS_GetPrivate(cx, obj);
      switch(JSVAL_TO_INT(id))
	{
	case name_prop:
	  
	  break;
	case age_prop:
	  *vp = INT_TO_JSVAL(priv->getCustomer()->GetAge());
	  break;
	}
    }
#endif
  return JS_TRUE;
}
#endif
  
static JSBool controls_play (JSContext *cx, JSObject *obj, uintN argc, 
			     jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);
  char *mrl = NULL;
  int  pos, pos_time;

  se_log_fncall ("play");
  se_argc_check_max (3, "play");

  pos = -1; pos_time = -1;

  switch (argc) {
  case 0:
    break;

  case 1:
    {
      JSString *str;
      se_arg_is_string (0, "play");

      str = JS_ValueToString (cx, argv[0]);

      mrl = JS_GetStringBytes (str);
      se_log ("playing '%s' from start\n", mrl);
    }
    break;

  case 2: 
    {
      se_arg_is_int (0, "play");
      se_arg_is_int (1, "play");

      JS_ValueToInt32 (cx, argv[0], &pos);
      JS_ValueToInt32 (cx, argv[1], &pos_time);
      mrl = NULL;
      se_log ("playing from %d, %d\n", pos, pos_time); 
    }
    break;

  case 3:
    {
      JSString *str;
      se_arg_is_string (0, "play");
      se_arg_is_int (1, "play");
      se_arg_is_int (2, "play");

      str = JS_ValueToString (cx, argv[0]);
      JS_ValueToInt32 (cx, argv[1], &pos);
      JS_ValueToInt32 (cx, argv[2], &pos_time);
      mrl = JS_GetStringBytes (str);
      se_log ("playing '%s' from %d, %d\n", mrl, pos, pos_time);
    }
    break;
  }

  pos *=655 ;

  if (engine_mutex_trylock (se))
    return JS_TRUE;

  if (mrl) {

    int item;

    item = playlist_add_mrl (mrl, -1);

    if (pos<0)
      pos = 0;
    if (pos_time<0)
      pos_time = 0;

    playlist_play_from (item, pos, pos_time);

  } else {

    play_item_t *item;

    item = playlist_get_item (playlist_get_list_pos());

    if (item) {
      if (xine_get_status (stream) == XINE_STATUS_STOP) {

	if (pos<0)
	  pos = 0;
	if (pos_time<0)
	  pos_time = item->start_time;
	
	playlist_play_from (playlist_get_list_pos(), pos, pos_time);
      } else {
	
	if ( (pos>=0) || (pos_time>=0) ) {
	  
	  if (pos<0)
	    pos = 0;
	  if (pos_time<0)
	    pos_time = 0;
	  xine_play (stream, pos, pos_time);
	}
      }
    } else
      printf (_("script_engine: error, no valid play item available\n"));
  } 

  if (xine_get_param (stream, XINE_PARAM_SPEED) != XINE_SPEED_NORMAL) {
    ui_set_status (UI_PLAY);
    xine_set_param (stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
  }

  pthread_mutex_unlock (&engine_lock);

  return JS_TRUE;
}

/* Crude hack to allow combined play/pause function.
 * (Useful on some keyboards with CD/DVD control keys.)
 */
static JSBool controls_play_pause (JSContext *cx, JSObject *obj, uintN argc,
                                  jsval *argv, jsval *rval) {

  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  se_log_fncall ("play_pause");
  se_argc_check_max (1, "play_pause");

  if (argc==1) {
    se_arg_is_int (0, "play_pause");
  }

  if (xine_get_status (stream) == XINE_STATUS_STOP)
    return controls_play (cx, obj, 0, 0, rval);

  return controls_pause (cx, obj, argc, argv, rval);
}

static JSBool js_is_live_stream (JSContext *cx, JSObject *obj, uintN argc,
                                 jsval *argv, jsval *rval)
{
  se_t *se = (se_t *) JS_GetContextPrivate(cx);

  *rval = JSVAL_FALSE;

  se_log_fncall ("is_live_stream");
  se_argc_check_max (0, "is_live_stream");

  if (player_live_stream ())
    *rval = JSVAL_TRUE;

  return JS_TRUE;
}

int player_live_stream (void)
{
  gint pos_stream, pos_time, length_time;

  return (xine_get_status (stream) == XINE_STATUS_PLAY) &&
	 (xine_get_pos_length (stream, &pos_stream, &pos_time, &length_time)) &&
	 length_time == 0;
}

char *player_get_cur_mrl (void) {
  return cur_mrl;
}

static xine_audio_port_t *load_audio_out_driver (void) {
 
  xine_audio_port_t      *audio_port;
 
  if (!audio_driver_id) {
 
    char **driver_ids = (char **) xine_list_audio_output_plugins (xine) ;
    char **choices;
    int i;
 
    choices = malloc (sizeof (char *) * 100);
    choices[0] = "auto"; i=0;
    while (driver_ids[i]) {
      choices[i+1] = strdup(driver_ids[i]);
      i++;
    }
    choices[i+1]=0;

    /* try to init audio with stored information */
    i = xine_config_register_enum (xine,
                                   "audio.driver", 0,
                                   choices,
                                   _("audio driver to use"),
                                   NULL, 10, NULL, CONFIG_DATA_NONE);
    audio_driver_id = choices[i];
  }
 
  if (!strcmp (audio_driver_id, "null"))
    return NULL;
 
  if (strcmp (audio_driver_id, "auto")) {
 
    audio_port = xine_open_audio_driver (xine, audio_driver_id, NULL);
    if (audio_port)
      return audio_port;
    else
      printf (_("audio driver %s failed\n"), audio_driver_id);
  }
 
  /* autoprobe */
  return xine_open_audio_driver (xine, NULL, NULL);
}


void player_init (void) {

  se_o_t            *controls_obj;
  
  cur_mrl = NULL;

  /* load drivers, create xine stream */

  audio_port = load_audio_out_driver ();
  video_port = xine_open_video_driver (xine, "none", 0, NULL);

  stream = xine_stream_new (xine, audio_port, video_port);

  /*
   * set up script engine objects and functions 
   */

  controls_obj = se_create_object (gse, NULL, "controls", NULL);

  se_defun (gse, controls_obj, "play", controls_play, 0, 0,
	    SE_GROUP_HIDDEN, NULL, NULL);

  /*
   * hook up global xine functions for convenience
   */

  {
    static const se_f_def_t defs[] = {
      { "play", controls_play, 0, 0,
	SE_GROUP_ENGINE, N_("[mrl] [, pos, time]"),
	N_("time in milliseconds") },
      { "get_time", get_time, 0, 0,
	SE_GROUP_ENGINE, NULL, NULL },
      { "pause", controls_pause, 0, 0,
  	SE_GROUP_ENGINE, N_("[bool]"), NULL },
      { "stop", controls_stop, 0, 0,
	SE_GROUP_ENGINE, NULL, NULL },
      { "play_pause", controls_play_pause, 0, 0,
	SE_GROUP_ENGINE, N_("[bool]"), NULL },
      { "snapshot", js_snapshot, 0, 0,
	SE_GROUP_ENGINE, N_("[file name]"), NULL },

      { "set_speed", js_set_speed, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },
      { "get_speed", js_get_speed, 0, 0,
	SE_GROUP_PROPERTIES, NULL, NULL },
      { "set_volume", js_set_volume, 0, 0,
	SE_GROUP_PROPERTIES, N_("int"), NULL },
      { "get_volume", js_get_volume, 0, 0,
	SE_GROUP_PROPERTIES, NULL, NULL },
      { "set_mute", js_set_mute, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },
      { "get_mute", js_get_mute, 0, 0,
	SE_GROUP_PROPERTIES, NULL, NULL },
      { "set_zoom", js_set_zoom, 0, 0,
	SE_GROUP_PROPERTIES, N_("int"), NULL },
      { "get_zoom", js_get_zoom, 0, 0,
	SE_GROUP_PROPERTIES, NULL, NULL },
      { "set_video_size", js_set_video_size, 0, 0,
	SE_GROUP_PROPERTIES, N_("int"), NULL },
      { "set_fullscreen", js_set_fullscreen, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },
      { "set_aspect", js_set_aspect, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },
      { "set_sub", js_set_sub, 0, 0,
	SE_GROUP_PROPERTIES, N_("int"), NULL },
      { "set_deinterlace", js_set_deinterlace, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },
      { "set_auto_resize", js_set_auto_resize, 0, 0,
	SE_GROUP_PROPERTIES, N_("[bool]"), NULL },

      { "is_live_stream", js_is_live_stream, 0, 0,
	SE_GROUP_PROPERTIES, NULL, NULL },
      { NULL }
    };
    se_defuns (gse, NULL, defs);
  }
}
