/*
 * Copyright (c) 2003 Benedikt Meurer (benedikt.meurer@unix-ag.uni-siegen.de)
 *               2004 Jean-Franois Wauthy (pollux@xfce.org)
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifdef	HAVE_CONFIG_H
#include <config.h>
#endif /* !HAVE_CONFIG_H */

#include <string.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>
#include <libxfcegui4/libxfcegui4.h>

#include "queuedlg.h"

#include <printing-system.h>

enum
{
  PRINTERS_ICON_COLUMN,
  PRINTERS_ALIAS_COLUMN,
  PRINTERS_NAME_COLUMN,
  PRINTERS_N_COLUMNS
};

extern void show_about_dialog_cb (GtkWidget * widget, gpointer data);

static void icon_theme_changed_cb (XfceIconTheme * icon_theme, gpointer user_data);

static void tree_view_row_activated_cb (GtkWidget * widget, GtkTreePath * path, GtkTreeViewColumn * col, gpointer data);
static void tree_selection_changed_cb (GtkTreeSelection *, gpointer);
static void set_default_cb (GtkWidget *, gpointer);
static void properties_cb (GtkWidget *, gpointer);

extern GList *printers_list;

/**************/
/* Mainwindow */
/**************/
static void
load_printer_list_in_treeview (GtkListStore * store, ManagerDlg * dlg)
{
  GtkTreeIter iter;
  GList *printer_el = NULL;
  Printer *printer = NULL;
  GtkTreeSelection *selection;  
  gchar *default_printer;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dlg->treeview));
  printer = printing_system_get_default_printer ? printing_system_get_default_printer (printers_list) : NULL;
  if (printer)
    default_printer = g_strdup (printer->name);
  else
    default_printer = g_strdup ("");

  printer_el = g_list_first (printers_list);
  while (printer_el) {
    GdkPixbuf *icon = NULL;

    printer = printer_el->data;
    if (printer->type == PRINTER_TYPE_CLASS)
      icon = xfce_icon_theme_load (dlg->icontheme, "printer-class", 48);
    if (!icon || printer->type == PRINTER_TYPE_PRINTER)
      icon = xfce_icon_theme_load (dlg->icontheme, "gnome-dev-printer", 48);

    gtk_list_store_append (store, &iter);
    gtk_list_store_set (store, &iter,
                        PRINTERS_ICON_COLUMN, icon,
                        PRINTERS_ALIAS_COLUMN, printer->alias, PRINTERS_NAME_COLUMN, printer->name, -1);

    if (strcmp (default_printer, printer->name) == 0 )
	gtk_tree_selection_select_iter (selection, &iter);

    g_object_unref (G_OBJECT (icon));

    printer_el = g_list_next (printer_el);
  }

  g_free (default_printer);
}

void
mainwin_setup (void)
{
  ManagerDlg *dlg;
  GtkTreeSelection *selection;
  GtkWidget *menubar;
  GtkWidget *menuitem_printer;
  GtkWidget *menuitem_quit;
  GtkWidget *menu_about;
  GtkWidget *menuitem_about;
  GtkWidget *menuitem_aboutmanager;
  GtkWidget *menuitem_aboutplugin;
  GtkWidget *scrollwin;
  GtkWidget *separator;
  GtkWidget *vbox;
  GtkListStore *store;
  GtkCellRenderer *icon_cell;
  GtkCellRenderer *alias_cell;
  GtkTreeViewColumn *alias_column;
  GdkPixbuf *icon = NULL;

  dlg = g_new0 (ManagerDlg, 1);

  dlg->accel_group = gtk_accel_group_new ();

  dlg->icontheme = xfce_icon_theme_get_for_screen (NULL);
  g_signal_connect (G_OBJECT (dlg->icontheme), "changed", G_CALLBACK (icon_theme_changed_cb), dlg);

  dlg->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  icon = xfce_icon_theme_load (dlg->icontheme, "xfprint", 48);
  if (icon) {
    gtk_window_set_icon (GTK_WINDOW (dlg->window), xfce_icon_theme_load (dlg->icontheme, "xfprint", 48));
    g_object_unref (icon);
  }

  gtk_window_add_accel_group (GTK_WINDOW (dlg->window), dlg->accel_group);
  gtk_window_set_default_size (GTK_WINDOW (dlg->window), 450, 250);
  gtk_window_set_title (GTK_WINDOW (dlg->window), _("Printer manager"));

  vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add (GTK_CONTAINER (dlg->window), vbox);

  menubar = gtk_menu_bar_new ();
  gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

  /* Printer menu */
  menuitem_printer = gtk_menu_item_new_with_mnemonic (_("_Printer"));
  gtk_container_add (GTK_CONTAINER (menubar), menuitem_printer);

  dlg->menu_printer = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem_printer), dlg->menu_printer);

  dlg->menuitem_properties = gtk_image_menu_item_new_with_mnemonic (_("_Properties"));
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (dlg->menuitem_properties),
                                 gtk_image_new_from_stock (GTK_STOCK_PROPERTIES, GTK_ICON_SIZE_MENU));
  g_signal_connect (G_OBJECT (dlg->menuitem_properties), "activate", G_CALLBACK (properties_cb), dlg);
  gtk_container_add (GTK_CONTAINER (dlg->menu_printer), dlg->menuitem_properties);

  dlg->menuitem_setdefault = gtk_image_menu_item_new_with_mnemonic (_("Set _default"));
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (dlg->menuitem_setdefault),
                                 gtk_image_new_from_stock (GTK_STOCK_PRINT, GTK_ICON_SIZE_MENU));
  g_signal_connect (G_OBJECT (dlg->menuitem_setdefault), "activate", G_CALLBACK (set_default_cb), dlg);
  gtk_container_add (GTK_CONTAINER (dlg->menu_printer), dlg->menuitem_setdefault);

  separator = gtk_menu_item_new ();
  gtk_widget_set_sensitive (separator, FALSE);
  gtk_container_add (GTK_CONTAINER (dlg->menu_printer), separator);

  menuitem_quit = gtk_image_menu_item_new_from_stock (GTK_STOCK_QUIT, dlg->accel_group);
  g_signal_connect (G_OBJECT (menuitem_quit), "activate", G_CALLBACK (gtk_main_quit), NULL);
  gtk_container_add (GTK_CONTAINER (dlg->menu_printer), menuitem_quit);

  /* About menu */
  menuitem_about = gtk_menu_item_new_with_mnemonic (_("_About"));
  gtk_container_add (GTK_CONTAINER (menubar), menuitem_about);

  menu_about = gtk_menu_new ();
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (menuitem_about), menu_about);

  menuitem_aboutplugin = gtk_menu_item_new_with_mnemonic (_("About printing _system ..."));
  if (printing_system_about_dialog_run)
    g_signal_connect (G_OBJECT (menuitem_aboutplugin), "activate", G_CALLBACK (printing_system_about_dialog_run), dlg);
  gtk_container_add (GTK_CONTAINER (menu_about), menuitem_aboutplugin);

  menuitem_aboutmanager = gtk_menu_item_new_with_mnemonic (_("_About xfprint-manager ..."));
  g_signal_connect (G_OBJECT (menuitem_aboutmanager), "activate", G_CALLBACK (show_about_dialog_cb), dlg);
  gtk_container_add (GTK_CONTAINER (menu_about), menuitem_aboutmanager);

  /* the printer list store */
  store = gtk_list_store_new (PRINTERS_N_COLUMNS, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);

  /* The printer list view */
  scrollwin = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollwin), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (vbox), scrollwin, TRUE, TRUE, 0);

  dlg->treeview = gtk_tree_view_new_with_model (GTK_TREE_MODEL (store));
  gtk_container_add (GTK_CONTAINER (scrollwin), dlg->treeview);

  /* load the printer list in the treeview */
  load_printer_list_in_treeview (store, dlg);

  g_signal_connect (G_OBJECT (dlg->treeview), "row-activated", G_CALLBACK (tree_view_row_activated_cb), dlg);

  /* configure the tree selection for the printer list */
  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (dlg->treeview));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed", G_CALLBACK (tree_selection_changed_cb), dlg);

  icon_cell = gtk_cell_renderer_pixbuf_new ();
  alias_cell = gtk_cell_renderer_text_new ();
  alias_column = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (alias_column, icon_cell, FALSE);
  gtk_tree_view_column_set_attributes (alias_column, icon_cell, "pixbuf", PRINTERS_ICON_COLUMN, NULL);
  g_object_set (icon_cell, "xalign", 0.0, "ypad", 0, NULL);

  gtk_tree_view_column_pack_start (alias_column, alias_cell, TRUE);
  gtk_tree_view_column_set_attributes (alias_column, alias_cell, "markup", PRINTERS_ALIAS_COLUMN, NULL);
  g_object_set (alias_cell, "ypad", 0, "yalign", 0.5, NULL);
  gtk_tree_view_column_set_title (alias_column, _("Alias name"));

  gtk_tree_view_append_column (GTK_TREE_VIEW (dlg->treeview), alias_column);

  gtk_tree_view_insert_column_with_attributes (GTK_TREE_VIEW (dlg->treeview),
                                               PRINTERS_NAME_COLUMN, _("System name"), gtk_cell_renderer_text_new (),
                                               "text", PRINTERS_NAME_COLUMN, NULL);

  /* the view now holds a reference of the store */
  g_object_unref (G_OBJECT (store));

  /* connect signals for the main window */
  g_signal_connect_swapped (G_OBJECT (dlg->window), "delete-event", G_CALLBACK (gtk_menu_item_activate), menuitem_quit);

  gtk_widget_show_all (dlg->window);

  if (printing_system_main_window_customize)
    printing_system_main_window_customize (dlg);
  else {
    gtk_widget_set_sensitive (dlg->treeview, FALSE);
    gtk_widget_set_sensitive (dlg->menuitem_properties, FALSE);
    gtk_widget_set_sensitive (dlg->menuitem_setdefault, FALSE);
    gtk_widget_set_sensitive (menuitem_aboutplugin, FALSE);
  }
}

/**********************/
/* Icontheme callback */
/**********************/
static gboolean
icon_theme_update_foreach_func (GtkTreeModel * model, GtkTreePath * path, GtkTreeIter * iter, gpointer data)
{
  ManagerDlg *dlg;
  GdkPixbuf *icon = NULL;
  gchar *name = NULL;
  Printer *printer;

  dlg = (ManagerDlg *) data;

  gtk_tree_model_get (model, iter, PRINTERS_ICON_COLUMN, &icon, PRINTERS_NAME_COLUMN, &name, -1);

  if (icon)
    g_object_unref (icon);

  printer = printer_lookup_byname (printers_list, name);

  if (printer->type == PRINTER_TYPE_CLASS)
      icon = xfce_icon_theme_load (dlg->icontheme, "printer-class", 48);
  if (!icon || printer->type == PRINTER_TYPE_PRINTER)
    icon = xfce_icon_theme_load (dlg->icontheme, "gnome-dev-printer", 48);

  gtk_list_store_set (GTK_LIST_STORE (model), iter, PRINTERS_ICON_COLUMN, icon, -1);

  g_object_unref (icon);
  g_free (name);
  return FALSE;
}

static void
icon_theme_changed_cb (XfceIconTheme * icon_theme, gpointer user_data)
{
  GtkTreeModel *model;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) user_data;

  model = gtk_tree_view_get_model (GTK_TREE_VIEW (dlg->treeview));

  if (model)
    gtk_tree_model_foreach (model, &icon_theme_update_foreach_func, dlg);
}

/**********************/
/* Treeview callbacks */
/**********************/
static void
tree_view_row_activated_cb (GtkWidget * widget, GtkTreePath * path, GtkTreeViewColumn * col, gpointer data)
{
  gchar *printer_name;
  GtkTreeModel *model;
  GtkTreeIter iter;
  GtkTreeSelection *selection;

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));

  if (selection && gtk_tree_selection_get_selected (selection, &model, &iter)) {
    gtk_tree_model_get (model, &iter, PRINTERS_NAME_COLUMN, &printer_name, -1);
    show_printer_queue (printer_name);

    g_free (printer_name);
  }
}

static void
tree_selection_changed_cb (GtkTreeSelection * select, gpointer user_data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *alias;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) user_data;

  if (gtk_tree_selection_get_selected (select, &model, &iter)) {
    gtk_tree_model_get (model, &iter, PRINTERS_ALIAS_COLUMN, &alias, -1);

    gtk_widget_set_sensitive (dlg->menuitem_properties, TRUE);

    g_free (alias);
  }
  else {
    gtk_widget_set_sensitive (dlg->menuitem_properties, FALSE);
    gtk_widget_set_sensitive (dlg->menuitem_setdefault, FALSE);
  }
}

/******************/
/* Menu callbacks */
/******************/
static void
set_default_cb (GtkWidget * menuitem, gpointer user_data)
{
  GtkTreeSelection *select;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *printer;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) user_data;

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (dlg->treeview));

  if (!gtk_tree_selection_get_selected (select, &model, &iter))
    return;

  gtk_tree_model_get (model, &iter, PRINTERS_NAME_COLUMN, &printer, -1);

  if (printing_system_set_default_printer && 
      !printing_system_set_default_printer (printer))
    xfce_err (_("Unable to set it as default printer !"));

  g_free (printer);
}

static void
properties_cb (GtkWidget * menuitem, gpointer user_data)
{
  GtkTreeSelection *select;
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *printer;
  ManagerDlg *dlg;

  dlg = (ManagerDlg *) user_data;

  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (dlg->treeview));

  if (!gtk_tree_selection_get_selected (select, &model, &iter))
    return;

  gtk_tree_model_get (model, &iter, PRINTERS_NAME_COLUMN, &printer, -1);

  if (printing_system_properties_dialog_run)
    printing_system_properties_dialog_run (dlg->window, printer);

  g_free (printer);
}
