/* $Id: xfce-exec.c 17231 2005-08-31 22:22:37Z olivier $
 *  
 * Copyright (C) 2004 Jasper Huijsmans <jasper@xfce.org>
 * startup notification added by Olivier Fourdan based on gnome-desktop
 * developed by Elliot Lee <sopwith@redhat.com> and Sid Vicious
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <ctype.h>
#include <stdio.h>
#include <string.h>

#ifdef HAVE_STDDEF_H
#include <stddef.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xatom.h>

#include <gdk/gdk.h>

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
#define SN_API_NOT_YET_FROZEN
#include <libsn/sn.h>
#include <gdk/gdkx.h>
#define STARTUP_TIMEOUT (30 /* seconds */ * 1000)
#endif

#include <libxfce4util/libxfce4util.h>

#include "xfce-exec.h"

extern char **environ;

#ifdef HAVE_LIBSTARTUP_NOTIFICATION

typedef struct
{
    GSList *contexts;
    guint timeout_id;
}
StartupTimeoutData;

static StartupTimeoutData *startup_timeout_data = NULL;

static gboolean atexit_registered = FALSE;

static void
sn_error_trap_push (SnDisplay * display, Display * xdisplay)
{
    gdk_error_trap_push ();
}

static void
sn_error_trap_pop (SnDisplay * display, Display * xdisplay)
{
    gdk_error_trap_pop ();
}

static gchar **
make_spawn_environment_for_sn_context (SnLauncherContext *
				       sn_context, char **envp)
{
    gchar **retval = NULL;
    int i, j;
    int desktop_startup_id_len;

    if (envp == NULL)
    {
	envp = environ;
    }
    for (i = 0; envp[i]; i++);

    retval = g_new (gchar *, i + 2);

    desktop_startup_id_len = strlen ("DESKTOP_STARTUP_ID");

    for (i = 0, j = 0; envp[i]; i++)
    {
	if (strncmp (envp[i], "DESKTOP_STARTUP_ID", desktop_startup_id_len) !=
	    0)
	{
	    retval[j] = g_strdup (envp[i]);
	    ++j;
	}
    }

    retval[j] =
	g_strdup_printf ("DESKTOP_STARTUP_ID=%s",
			 sn_launcher_context_get_startup_id (sn_context));
    ++j;
    retval[j] = NULL;

    return retval;
}

static gboolean
startup_timeout (void *data)
{
    StartupTimeoutData *std = data;
    GSList *tmp;
    GTimeVal now;
    int min_timeout;

    min_timeout = STARTUP_TIMEOUT;

    g_get_current_time (&now);

    tmp = std->contexts;
    while (tmp != NULL)
    {
	SnLauncherContext *sn_context = tmp->data;
	GSList *next = tmp->next;
	long tv_sec, tv_usec;
	double elapsed;

	sn_launcher_context_get_last_active_time (sn_context, &tv_sec,
						  &tv_usec);

	elapsed =
	    ((((double) now.tv_sec - tv_sec) * G_USEC_PER_SEC +
	      (now.tv_usec - tv_usec))) / 1000.0;

	if (elapsed >= STARTUP_TIMEOUT)
	{
	    std->contexts = g_slist_remove (std->contexts, sn_context);
	    sn_launcher_context_complete (sn_context);
	    sn_launcher_context_unref (sn_context);
	}
	else
	{
	    min_timeout = MIN (min_timeout, (STARTUP_TIMEOUT - elapsed));
	}

	tmp = next;
    }

    if (std->contexts == NULL)
    {
	std->timeout_id = 0;
    }
    else
    {
	std->timeout_id = g_timeout_add (min_timeout, startup_timeout, std);
    }

    return FALSE;
}

static void
add_startup_timeout (SnLauncherContext * sn_context)
{
    if (startup_timeout_data == NULL)
    {
	startup_timeout_data = g_new (StartupTimeoutData, 1);
	startup_timeout_data->contexts = NULL;
	startup_timeout_data->timeout_id = 0;
    }

    sn_launcher_context_ref (sn_context);
    startup_timeout_data->contexts =
	g_slist_prepend (startup_timeout_data->contexts, sn_context);

    if (startup_timeout_data->timeout_id == 0)
    {
	startup_timeout_data->timeout_id =
	    g_timeout_add (STARTUP_TIMEOUT, startup_timeout,
			   startup_timeout_data);
    }
}

static void
free_startup_timeout (void)
{
    StartupTimeoutData *std = startup_timeout_data;

    if (!std)
    {
	/* No startup notification used, return silently */
	return;
    }

    g_slist_foreach (std->contexts, (GFunc) sn_launcher_context_unref, NULL);
    g_slist_free (std->contexts);

    if (std->timeout_id != 0)
    {
	g_source_remove (std->timeout_id);
	std->timeout_id = 0;
    }

    g_free (std);
    startup_timeout_data = NULL;

    atexit_registered = FALSE;
}

static gint
get_active_workspace_number ()
{
    GdkDisplay *gdpy = gdk_display_get_default();
    Display *dpy = GDK_DISPLAY_XDISPLAY(gdpy);
    GdkScreen *gscreen = gdk_display_get_default_screen(gdpy);
    GdkWindow *groot = gdk_screen_get_root_window(gscreen);
    Atom _NET_CURRENT_DESKTOP = XInternAtom(dpy, "_NET_CURRENT_DESKTOP", False);
    Atom _WIN_WORKSPACE = XInternAtom(dpy, "_WIN_WORKSPACE", False);
    Atom type_ret = None;
    int format_ret = 0;
    unsigned long nitems_ret = 0;
    unsigned long bytes_after_ret = 0;
    unsigned int *prop_ret = NULL;
    gint ws_num = 0;
    
    gdk_error_trap_push();
    
    if(XGetWindowProperty(dpy, GDK_WINDOW_XWINDOW(groot), _NET_CURRENT_DESKTOP,
            0, 32, False, XA_CARDINAL, &type_ret, &format_ret, &nitems_ret,
            &bytes_after_ret, (unsigned char **)&prop_ret) != Success)
    {
        if(XGetWindowProperty(dpy, GDK_WINDOW_XWINDOW(groot), _WIN_WORKSPACE,
            0, 32, False, XA_CARDINAL, &type_ret, &format_ret, &nitems_ret,
            &bytes_after_ret, (unsigned char **)&prop_ret) != Success)
        {
            return 0;
        }
    }
    
    gdk_error_trap_pop();
    
    if(type_ret == None || format_ret == 0 || !prop_ret) {
        if(prop_ret)
            XFree(prop_ret);
        return 0;
    }
    
    ws_num = *prop_ret;
    XFree(prop_ret);
    
    return ws_num;
}

#endif /* HAVE_LIBSTARTUP_NOTIFICATION */

static gboolean
real_xfce_exec_with_envp (const char *cmd, 
		     gboolean in_terminal, 
		     gboolean use_sn,
		     GError ** error,
		     char **envp,
                     gboolean async)
{
    gchar **free_envp = NULL;
    gchar **argv = NULL;
    gboolean success;
    char *realcmd;
    gint exit_status;
    
#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    SnLauncherContext *sn_context = NULL;
    SnDisplay *sn_display = NULL;

    if (!atexit_registered)
    {
	g_atexit (free_startup_timeout);
	atexit_registered = TRUE;
    }
#endif

    if (!cmd)
        return FALSE;
    
    realcmd = g_strdup (cmd);
    
    if (realcmd[0] == '"')
    {
        int i;

        for (i = 1; realcmd[i-1] != '\0'; ++i)
        {
            if (realcmd[i] != '"')
                realcmd[i-1] = realcmd[i-1];
            else {                
                realcmd[i-1] = realcmd[i] = ' ';
                break;
            }
        }
    }
    
    if (g_path_is_absolute (realcmd) && 
            g_file_test (realcmd, G_FILE_TEST_IS_DIR))
    {
        argv = g_new (char *, 3);

        if (in_terminal)
            argv[0] = g_strdup ("xfterm4");
	else
	    argv[0] = g_strdup ("xftree4");
        
        argv[1] = realcmd;
        argv[2] = NULL;
    }
    else if (in_terminal)
    {
        argv = g_new (char *, 4);
        
        argv[0] = g_strdup ("xfterm4");
        argv[1] = g_strdup ("-e");
	argv[2] = realcmd;
        argv[3] = NULL;
    }
    else
    {
        if (!g_shell_parse_argv (realcmd, NULL, &argv, error))
        {
            g_free (realcmd);

            return FALSE;
        }
        
        g_free (realcmd);
    }    

    free_envp = NULL;

#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    if (use_sn)
    {
	sn_display = sn_display_new (gdk_display, sn_error_trap_push,
				     sn_error_trap_pop);

	if (sn_display != NULL)
	{
	    sn_context = 
		sn_launcher_context_new (sn_display,
					 DefaultScreen (gdk_display));
	    if ((sn_context != NULL)
		&& !sn_launcher_context_get_initiated (sn_context))
	    {
		gint workspace = get_active_workspace_number ();
		DBG("got workspace: %d", workspace);
		sn_launcher_context_set_workspace (sn_context, workspace);
		sn_launcher_context_set_binary_name (sn_context, argv[0]);
		sn_launcher_context_initiate (sn_context,
					      g_get_prgname () ?
					      g_get_prgname () : "unknown",
					      argv[0], CurrentTime);
		free_envp =
		    make_spawn_environment_for_sn_context (sn_context, envp);
	    }
	}
    }
#endif

    if (async)
    {
        success = g_spawn_async_with_pipes (NULL, argv, 
                                            free_envp ? free_envp : envp,
                                            G_SPAWN_SEARCH_PATH, 
                                            NULL, NULL, NULL,
                                            NULL, NULL, NULL,
                                            error);
    }
    else
    {
        success = g_spawn_sync (NULL, argv, 
                                free_envp ? free_envp : envp,
                                G_SPAWN_SEARCH_PATH, 
                                NULL, NULL,
                                NULL, NULL,
                                &exit_status, error);
    }
    
#ifdef HAVE_LIBSTARTUP_NOTIFICATION
    if (use_sn && success)
    {
	if (sn_context != NULL)
	{
	    if (!success)
	    {
		sn_launcher_context_complete (sn_context); /* end sequence */
	    }
	    else
	    {
		add_startup_timeout (sn_context);
		sn_launcher_context_unref (sn_context);
	    }
	}
	if (free_envp)
	{
	    g_strfreev (free_envp);
	    free_envp = NULL;
	}
	if (sn_display)
	{
	    sn_display_unref (sn_display);
	}
    }
#endif /* HAVE_LIBSTARTUP_NOTIFICATION */

    g_strfreev (argv);

    return success;
}

/**
 * xfce_exec
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 *
 * Use #xfce_exec_with_envp if you want to specify the environment.
 * 
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean xfce_exec (const char *cmd, 
		    gboolean in_terminal, 
		    gboolean use_sn, 
		    GError **error)
{
    return real_xfce_exec_with_envp (cmd, in_terminal, use_sn, error,
                                     environ, TRUE);
}

/**
 * xfce_exec_with_envp
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 * @envp	: the environment
 *
 * See also #xfce_exec.
 *
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean
xfce_exec_with_envp (const char *cmd, 
		     gboolean in_terminal, 
		     gboolean use_sn,
		     GError ** error,
		     char **envp)
{
    return real_xfce_exec_with_envp (cmd, in_terminal, use_sn, error, envp,
                                     TRUE);
}

/**
 * xfce_exec_sync
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 *
 * Use #xfce_exec_sync_with_envp if you want to specify the environment.
 * 
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean xfce_exec_sync (const char *cmd, 
			 gboolean in_terminal, 
			 gboolean use_sn, 
			 GError **error)
{
    return real_xfce_exec_with_envp (cmd, in_terminal, use_sn, error,
                                     environ, FALSE);
}

/**
 * xfce_exec_sync_with_envp
 * @cmd         : command line to run
 * @in_terminal : whether to run @cmd in a terminal
 * @use_sn      : whether to use startup notification
 * @error       : location for a GError or NULL
 * @envp	: the environment
 *
 * See also #xfce_exec.
 *
 * Returns: TRUE on success, FALSE on failure. 
 */
gboolean
xfce_exec_sync_with_envp (const char *cmd, 
			  gboolean in_terminal, 
			  gboolean use_sn,
			  GError ** error,
			  char **envp)
{
    return real_xfce_exec_with_envp (cmd, in_terminal, use_sn, error, envp,
                                     FALSE);
}
