/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2005 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#if WANT_HAMLIB
#include <gtk/gtk.h>
#include <math.h>
#include <stdlib.h>
#include <hamlib/rig.h>

#include "hamlib-utils.h"
#include "types.h"
#include "utils.h"
#include "support.h"
#include "string.h"
#include "data/pixmaps/s0.xpm"
#include "data/pixmaps/s1.xpm"
#include "data/pixmaps/s2.xpm"
#include "data/pixmaps/s3.xpm"
#include "data/pixmaps/s4.xpm"
#include "data/pixmaps/s5.xpm"
#include "data/pixmaps/s6.xpm"
#include "data/pixmaps/s7.xpm"
#include "data/pixmaps/s8.xpm"
#include "data/pixmaps/s9.xpm"

/* The following 2 structs are generated by mkrigstruct,
    using hamlib-1.2.4 */
struct rig_id {
	const gint modelnr;
	const gchar *modelname;
};

static const struct rig_id rig_id_list[] = {
	{1504, "WR-3100"},
	{352, "OptoScan535"},
	{304, "IC-275"},
	{336, "IC-R10"},
	{320, "IC-736"},
	{208, "TS-811"},
	{224, "TS-680S"},
	{2401, "EKD-500"},
	{1601, "TT-550"},
	{1505, "WR-3150"},
	{401, "IC-PCR1000"},
	{353, "OptoScan456"},
	{337, "IC-R71"},
	{321, "IC-737"},
	{209, "TS-850"},
	{225, "TS-140S"},
	{113, "FT-900"},
	{1, "Dummy"},
	{1602, "TT-538 Jupiter"},
	{1506, "WR-3500"},
	{802, "BC245xlt"},
	{402, "IC-PCR100"},
	{354, "IC ID-1"},
	{306, "IC-471"},
	{338, "IC-R72"},
	{226, "TM-D700"},
	{210, "TS-870S"},
	{114, "FT-920"},
	{1603, "RX-320"},
	{1507, "WR-3700"},
	{803, "BC895xlt"},
	{307, "IC-475"},
	{339, "IC-R75"},
	{323, "IC-746"},
	{355, "IC-703"},
	{227, "TM-V7"},
	{211, "TS-940S"},
	{115, "FT-890"},
	{1604, "RX-340"},
	{1204, "WJ-8888"},
	{340, "IC-R7000"},
	{356, "IC-7800"},
	{324, "IC-751"},
	{228, "TS-480"},
	{116, "FT-990"},
	{2501, "Elektor 3/04"},
	{2101, "4937 DI5 tuner module"},
	{1701, "DX-77"},
	{1605, "RX-350"},
	{501, "AR8200"},
	{341, "IC-R7100"},
	{357, "IC-756PROIII"},
	{309, "IC-706"},
	{213, "TS-950SDX"},
	{117, "FRG-100"},
	{101, "FT-847"},
	{2502, "DRT1"},
	{2102, "4702 DT5 tuner module"},
	{902, "R-8A"},
	{502, "AR8000"},
	{342, "ICR-8500"},
	{358, "IC-R20"},
	{326, "IC-756"},
	{310, "IC-706MkII"},
	{214, "TS-2000"},
	{118, "FRG-9600"},
	{1607, "TT-516 Argonaut V"},
	{903, "R-8B"},
	{503, "AR7030"},
	{343, "IC-R9000"},
	{327, "IC-756PRO"},
	{311, "IC-706MkIIG"},
	{215, "R-5000"},
	{119, "FRG-8800"},
	{103, "FT-1000D"},
	{1608, "TT-565 Orion"},
	{504, "AR5000"},
	{344, "IC-910"},
	{312, "IC-707"},
	{328, "IC-761"},
	{216, "TS-570S"},
	{104, "MARK-V FT-1000MP"},
	{120, "FT-817"},
	{2601, "SW/FM radio"},
	{2201, "DSP-10"},
	{1801, "505DSP"},
	{505, "AR3030"},
	{345, "IC-78"},
	{329, "IC-765"},
	{313, "IC-718"},
	{217, "TH-D7A"},
	{201, "TS-50S"},
	{105, "FT-747GX"},
	{121, "FT-100"},
	{1402, "TRP8000"},
	{506, "AR3000A"},
	{330, "IC-775"},
	{346, "IC-746PRO"},
	{314, "IC-725"},
	{202, "TS-440"},
	{122, "FT-857"},
	{106, "FT-757GX"},
	{331, "IC-781"},
	{347, "IC-756PROII"},
	{315, "IC-726"},
	{203, "TS-450S"},
	{123, "FT-897"},
	{107, "FT-757GXII"},
	{1004, "HF-235"},
	{316, "IC-728"},
	{220, "TH-F7E"},
	{204, "TS-570D"},
	{124, "FT-1000MP"},
	{2301, "SDR-1000"},
	{1901, "RPC rig"},
	{1501, "WR-1000"},
	{221, "K2"},
	{205, "TS-690S"},
	{125, "MARK-V Field FT-1000MP"},
	{1502, "WR-1500"},
	{606, "NRD-535D"},
	{334, "IC-821H"},
	{206, "TS-711"},
	{222, "TS-930"},
	{126, "VR-5000"},
	{110, "FT-736R"},
	{1503, "WR-1550"},
	{1103, "RA6790/GM"},
	{607, "NRD-545 DSP"},
	{351, "Omni VI Plus"},
	{303, "IC-271"},
	{335, "IC-970"},
	{319, "IC-735"},
	{223, "TH-G71"},
	{207, "TS-790"},
	{0, NULL}, /* end marker */
};

typedef struct
{
	gchar **xpmdata;
	GdkPixmap *pixmap;
	GdkBitmap *mask;
}
typeLevel;

typeLevel S[] = {
	{s0_xpm, NULL, NULL},
	{s1_xpm, NULL, NULL},
	{s2_xpm, NULL, NULL},
	{s3_xpm, NULL, NULL},
	{s4_xpm, NULL, NULL},
	{s5_xpm, NULL, NULL},
	{s6_xpm, NULL, NULL},
	{s7_xpm, NULL, NULL},
	{s8_xpm, NULL, NULL},
	{s9_xpm, NULL, NULL},
	{NULL, NULL, NULL}
};
extern preferencestype preferences;
extern statetype state;
extern GtkWidget *mainwindow;
gint svalue[10];
gint hamlibtimer = -1;
RIG *myrig;
GdkPixmap *pixmap = NULL;

/* load xpm data from s-meter pixmaps */
void
loadsmeter (GtkWidget * widget)
{
	gint i = 0;

	while (S[i].xpmdata)
		{
			S[i].pixmap = gdk_pixmap_create_from_xpm_d (widget->window,
							&S[i].mask, NULL,
							S[i].xpmdata);
			i++;
		}
}

/* copy a s-meter pixmap to pixmap depending on the value of smax */
static void
draw_smeter (gint value)
{
	GtkWidget *drawingarea;

	drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
	/* clear background */
	gdk_draw_rectangle (pixmap, drawingarea->style->white_gc, TRUE, 0, 0,
		drawingarea->allocation.width, drawingarea->allocation.height);
	gdk_draw_drawable (pixmap, drawingarea->style->fg_gc[GTK_STATE_NORMAL], 
		S[value].pixmap, 0, 0, 0, 0, 
		drawingarea->allocation.width, drawingarea->allocation.height);
	gtk_widget_queue_draw_area (drawingarea, 0, 0, 
		drawingarea->allocation.width, drawingarea->allocation.height);
	/* see expose_event further down */
}

/* stop timer if used and close rig */
void
stop_hamlib (void)
{
	if (hamlibtimer != -1)
		g_source_remove (hamlibtimer);
	hamlibtimer = -1;
	rig_close ((RIG *) myrig);
	rig_cleanup ((RIG *) myrig);
}

/* configure the hamlib port [TODO: add more string checks] */
static gint set_conf(RIG *my_rig, gchar *parms)
{
	gchar *p, *q, *n, *conf_parms;
	gint ret;

	if (! strchr(parms, '=')) return -1;

	conf_parms = g_strdup (parms);
	p = conf_parms;
	while (p && *p != '\0') 
		{
			q = strchr(p, '=');
			if (q) *q++ = '\0';
			n = strchr(q, ',');
			if (n) *n++ = '\0';
 
			ret = rig_set_conf(my_rig, rig_token_lookup(my_rig, p), q);
			if (ret != RIG_OK) return ret;
			p = n;
		}
		return RIG_OK;
}

/* open rig and report errors, start timer when polling */
gboolean
start_hamlib (gchar * radio, gchar * device, gint debugmode, gint timervalue)
{
	gint rigid, retcode;
	gchar *temp;

	rig_set_debug (debugmode);
	rigid = get_rigid (radio);
	if (rigid != -1)
	{
		myrig = rig_init (rigid);
		if (g_ascii_strcasecmp (preferences.rigconf, "?"))
		{
			retcode = set_conf(myrig, preferences.rigconf);
			if (retcode != RIG_OK) 
				{
		 			temp = g_strdup_printf (_("Hamlib config parameter error: %s"),
						rigerror (retcode));
					update_statusbar (temp);
				}
		}
		strncpy (myrig->state.rigport.pathname, device, FILPATHLEN);
		retcode = rig_open (myrig);
		if (retcode != RIG_OK)
		{
			temp = g_strdup_printf (_
						("An error occured while opening port %s: %s"),
						myrig->state.rigport.pathname,
						rigerror (retcode));
			update_statusbar (temp);
			return FALSE;
		}
		else
		{
			if (timervalue == 0)
				hamlibtimer = -1;
			else
				hamlibtimer = g_timeout_add (timervalue, 
					(GSourceFunc)poll_riginfo, NULL);
		}
	}
	else
	{
		update_statusbar (_("Hamlib error: get_rigid failed"));
		return FALSE;
	}
	return TRUE;
}

/* sort function for creating the list of rigs */
static gint sortcaps (gconstpointer a, gconstpointer b)
{
	gchar *amodel = (gchar *)a;
	gchar *bmodel = (gchar *)b;

	return (g_ascii_strcasecmp(amodel, bmodel));
}

/* Return a list with pointers of available drivers from hamlib */
GList *
riglist_get_list (void)
{
	gint i;
	GList *rigs = NULL;

	for (i = 0; rig_id_list[i].modelname != NULL; i++)
		rigs = g_list_insert_sorted
			(rigs, (gpointer)rig_id_list[i].modelname, sortcaps);
	return rigs;
}

/* lookup hamlib rigid associated with model name */
gint
get_rigid (gchar * rig)
{
	guint i;

	for (i = 0; rig_id_list[i].modelname != NULL; i++)
		if (!strcmp (rig_id_list[i].modelname, rig))
			return (rig_id_list[i].modelnr);
	return -1;
}

/* return string with mode */
gchar *
rigmode (gint mode)
{
	gchar *rigmode;

	switch (mode)
	{
	case RIG_MODE_AM:
	case RIG_MODE_AMS:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("AM");
		else
			rigmode = g_strdup ("A3E");
		break;
	case RIG_MODE_CW:
	case RIG_MODE_CWR:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("CW");
		else
			rigmode = g_strdup ("A1A");
		break;
	case RIG_MODE_USB:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("USB");
		else
			rigmode = g_strdup ("J3E");
		break;
	case RIG_MODE_LSB:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("LSB");
		else
			rigmode = g_strdup ("J3E");
		break;
	case RIG_MODE_SSB:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("SSB");
		else
			rigmode = g_strdup ("J3E");
		break;
	case RIG_MODE_RTTY:
	case RIG_MODE_RTTYR:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("RTTY");
		else
			rigmode = g_strdup ("F1B");
		break;
	case RIG_MODE_FM:
	case RIG_MODE_WFM:
		if (preferences.fcc == 0)
			rigmode = g_strdup ("FM");
		else
			rigmode = g_strdup ("F3E");
		break;
	default:
		rigmode = g_strdup ("UNKNOWN");
		break;
	}
	return (rigmode);
}

static void
hamlib_error (gint code)
{
	gchar *message;

	message = g_strdup_printf (_("Hamlib error %d: %s"), code,
					 rigerror (code));
	update_statusbar (message);
	g_free (message);
}

/* get power level from the rig */
void
get_powerlevel (void)
{
	gint retcode, status;
	value_t val;
	GtkWidget *powerhbox;

	powerhbox = lookup_widget (mainwindow, "powerhbox");
	if (GTK_WIDGET_VISIBLE (powerhbox))
	{
		retcode =	rig_get_level (myrig, RIG_VFO_CURR, RIG_LEVEL_RFPOWER, &val);
		if (retcode == RIG_OK)
		{
			status = rig_power2mW (myrig, &state.rigpower, val.f, state.rigfrequency, state.rigmode);
		}
		else if (retcode != -RIG_ENAVAIL)
			hamlib_error (retcode);
	}
}

/* get mode */
void
get_mode (void)
{
	gint retcode;
	rmode_t rmode;
	pbwidth_t width;

	retcode = rig_get_mode (myrig, RIG_VFO_CURR, &rmode, &width);
	if (retcode == RIG_OK)
	{
		state.rigmode = rmode;
	}
	else if (retcode != -RIG_ENAVAIL)
		hamlib_error (retcode);
}

/* get freq */
void
get_frequency (void)
{
	gint retcode;
	freq_t freq;
	GString *digits = g_string_new ("");
	GtkWidget *frequencylabel;

	retcode = rig_get_freq (myrig, RIG_VFO_CURR, &freq);
	if (retcode == RIG_OK)
	{
		state.rigfrequency = freq;
		if ((preferences.hamlib == 2) || (preferences.hamlib == 4))
		{
			digits = convert_frequency ();
			digits = g_string_append (digits, " MHz");
			frequencylabel =
				lookup_widget (mainwindow, "frequencylabel");
			gtk_label_set_text (GTK_LABEL (frequencylabel),
							digits->str);
			g_string_free (digits, TRUE);
		}
	}
	else if (retcode != -RIG_ENAVAIL)
		hamlib_error (retcode);
}

/* are we transmitting or receiving ? */
void
get_ptt (void)
{
	gint retcode;
	ptt_t ptt;

	retcode = rig_get_ptt (myrig, RIG_VFO_CURR, &ptt);
	if (retcode == RIG_OK)
	{
		if (ptt == RIG_PTT_OFF)
			state.tx = FALSE;
		else
			state.tx = TRUE;
	}
	else if (retcode != -RIG_ENAVAIL)
		hamlib_error (retcode);
}

/* get s-meter value */
void
get_smeter (void)
{
	gint retcode, strength, spoint, i, smax = 1;

	retcode = rig_get_strength (myrig, RIG_VFO_CURR, &strength);
	if (retcode == RIG_OK)
	{
		if (strength >= 0)
			spoint = 9;
		else if (strength < -60)
			spoint = 0;
		else
			spoint = (gint) ceil ((strength + 60) / 6);
		svalue[state.scounter] = spoint;
		state.scounter++;
		if (state.scounter == 10)
			state.scounter = 0;

		/* find maximum of s-meter during last 3 seconds */
		for (i = 0; i < 10; i++)
			if (svalue[i] > smax)
				smax = svalue[i];

		if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
			draw_smeter (smax);

		if (smax == 0)
			smax = 1;
		if ((state.rigmode == RIG_MODE_CW)
				|| (state.rigmode == RIG_MODE_RTTY))
			state.rigrst = g_strdup_printf ("5%d9", smax);
		else
			state.rigrst = g_strdup_printf ("5%d", smax);
	}
	else if (retcode != -RIG_ENAVAIL)
		hamlib_error (retcode);
}

/* used by 'click all menu' when not polling */
void
get_riginfo (void)
{
	get_frequency ();
	get_mode ();
	get_ptt ();
	if (!state.tx)
		get_smeter ();
	else if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
		draw_smeter (0);
	get_powerlevel ();
}

/* poll the rig */
gint
poll_riginfo (void)
{
	if (state.hlcounter == 0)
	{
		state.hlcounter++;
		get_frequency ();
		return 1;
	}
	if (state.hlcounter == 1)
	{
		state.hlcounter++;
		get_mode ();
		return 1;
	}
	if (state.hlcounter == 2)
	{
		state.hlcounter++;
		get_ptt ();
		return 1;
	}
	if (state.hlcounter == 3)
	{
		state.hlcounter++;
		if (!state.tx)
			get_smeter ();
		else if ((preferences.hamlib == 3)
			 || (preferences.hamlib == 4))
			draw_smeter (0);
		return 1;
	}
	if (state.hlcounter == 4)
	{
		state.hlcounter = 0;
		get_powerlevel ();
		return 1;
	}
	return 1;
}


/* set appearance of some widgets dependent on preferences.hamlib */
void
sethamlibwidgets (gint status, gboolean initsmeter)
{
	GtkWidget *mhzlabel, *mhzbutton, *bandoptionmenu, *bandentry,
		*frequencyhandlebox, *modelabel, *modebutton, *drawingarea,
		*modeoptionmenu, *modeentry, *rstlabel, *rstbutton, *smeterhandlebox,
		*powerlabel, *powerbutton, *powerhbox;

	mhzlabel = lookup_widget (mainwindow, "mhzlabel");
	mhzbutton = lookup_widget (mainwindow, "mhzbutton");
	modelabel = lookup_widget (mainwindow, "modelabel");
	modebutton = lookup_widget (mainwindow, "modebutton");
	rstlabel = lookup_widget (mainwindow, "rstlabel");
	rstbutton = lookup_widget (mainwindow, "rstbutton");
	powerlabel = lookup_widget (mainwindow, "powerlabel");
	powerbutton = lookup_widget (mainwindow, "powerbutton");
	powerhbox = lookup_widget (mainwindow, "powerhbox");
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	bandentry = lookup_widget (mainwindow, "bandentry");
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	modeentry = lookup_widget (mainwindow, "modeentry");
	frequencyhandlebox = lookup_widget (mainwindow, "frequencyhandlebox");
	smeterhandlebox = lookup_widget (mainwindow, "smeterhandlebox");
	drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");

	if (status == 0)	/* hamlib disabled */
	{
		gtk_widget_show (mhzlabel);
		gtk_widget_hide (mhzbutton);
		gtk_widget_show (modelabel);
		gtk_widget_hide (modebutton);
		gtk_widget_show (rstlabel);
		gtk_widget_hide (rstbutton);
		if (GTK_WIDGET_VISIBLE (powerhbox))
		{
			gtk_widget_show (powerlabel);
			gtk_widget_hide (powerbutton);
		}
		gtk_widget_hide (frequencyhandlebox);
		gtk_widget_hide (smeterhandlebox);
	}
	else if (status == 1)	/* hamlib enabled */
	{
		gtk_widget_hide (mhzlabel);
		gtk_widget_show (mhzbutton);
		gtk_widget_hide (modelabel);
		gtk_widget_show (modebutton);
		gtk_widget_hide (rstlabel);
		gtk_widget_show (rstbutton);
		if (GTK_WIDGET_VISIBLE (powerhbox))
		{
			gtk_widget_hide (powerlabel);
			gtk_widget_show (powerbutton);
		}
		gtk_widget_hide (frequencyhandlebox);
		gtk_widget_hide (smeterhandlebox);
	}
	else if (status == 2)	/* hamlib enabled with frequency on statusbar */
	{
		gtk_widget_hide (mhzlabel);
		gtk_widget_show (mhzbutton);
		gtk_widget_hide (modelabel);
		gtk_widget_show (modebutton);
		gtk_widget_hide (rstlabel);
		gtk_widget_show (rstbutton);
		if (GTK_WIDGET_VISIBLE (powerhbox))
		{
			gtk_widget_hide (powerlabel);
			gtk_widget_show (powerbutton);
		}
		gtk_widget_show (frequencyhandlebox);
		gtk_widget_hide (smeterhandlebox);
	}
	else if (status == 3)	/* hamlib enabled with s-meter on statusbar */
	{
		gtk_widget_hide (mhzlabel);
		gtk_widget_show (mhzbutton);
		gtk_widget_hide (modelabel);
		gtk_widget_show (modebutton);
		gtk_widget_hide (rstlabel);
		gtk_widget_show (rstbutton);
		if (GTK_WIDGET_VISIBLE (powerhbox))
		{
			gtk_widget_hide (powerlabel);
			gtk_widget_show (powerbutton);
		}
		gtk_widget_hide (frequencyhandlebox);
		gtk_widget_show (smeterhandlebox);
		if (initsmeter)
		{
			drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
			loadsmeter (drawingarea);
		}
	}
	else if (status == 4)
	{			/* hamlib enabled with frequency and s-meter on statusbar */
		gtk_widget_hide (mhzlabel);
		gtk_widget_show (mhzbutton);
		gtk_widget_hide (modelabel);
		gtk_widget_show (modebutton);
		gtk_widget_hide (rstlabel);
		gtk_widget_show (rstbutton);
		if (GTK_WIDGET_VISIBLE (powerhbox))
		{
			gtk_widget_hide (powerlabel);
			gtk_widget_show (powerbutton);
		}
		gtk_widget_show (frequencyhandlebox);
		gtk_widget_show (smeterhandlebox);
		if (initsmeter)
		{
			drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
			loadsmeter (drawingarea);
		}
	}
}

GString *convert_frequency (void)
{
	GString *digits = g_string_new ("");

	if (preferences.round == 0)
		g_string_printf (digits, "%Ld", state.rigfrequency);
	else
		g_string_printf (digits, "%Ld", (long long) rint (state.rigfrequency /
			pow (10, preferences.round)));
	g_string_insert_c (digits, (digits->len) - 6 + preferences.round, '.');
	g_strstrip (digits->str);
	g_strdelimit (digits->str, " ", '0');
	return digits;
}

/* load s-meter pixmaps when main window is displayed */
void
on_mainwindow_show (GtkWidget * widget, gpointer user_data)
{
	GtkWidget *drawingarea;

	if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
		{
			drawingarea = lookup_widget (mainwindow, "smeterdrawingarea");
			loadsmeter (drawingarea);
			draw_smeter (0);
		}
}

/* create a new backing pixmap for the s-meter whenever the window is resized */
gboolean
on_smeterdrawingarea_configure_event (GtkWidget * widget,
	GdkEventConfigure * event, gpointer user_data)
{
	if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
		{
			if (pixmap)	g_object_unref (pixmap);
			pixmap = gdk_pixmap_new (widget->window, 
				widget->allocation.width, widget->allocation.height, -1);
		}
	return FALSE;
}

/* copy the background pixmap to the drawing area for the s-meter */
gboolean
on_smeterdrawingarea_expose_event (GtkWidget * widget,
	GdkEventExpose * event, gpointer user_data)
{
	if ((preferences.hamlib == 3) || (preferences.hamlib == 4))
		gdk_draw_drawable (widget->window,
			widget->style->fg_gc[GTK_WIDGET_STATE (widget)], pixmap,
			event->area.x, event->area.y, event->area.x,
			event->area.y, event->area.width, event->area.height);
	return FALSE;
}
#endif
