/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Pan - A Newsreader for Gtk+
 * Copyright (C) 2002  Charles Kerr <charles@rebelbase.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include <config.h>

#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include <glib.h>

#include <libxml/parser.h>
#include <libxml/xmlmemory.h>

#include <pan/base/debug.h>
#include <pan/base/log.h>
#include <pan/base/pan-i18n.h>
#include <pan/base/pan-glib-extensions.h>
#include <pan/base/base-prefs.h>
#include <pan/base/util-file.h>

#include <pan/base/pan-config.h>

#define MAX_KEY_LEN 2048

/** 
 ** Internal data structures
 **/

typedef enum
{
	IS_NOTYPE,
	IS_STRING,
	IS_BOOL,
	IS_INT,
	IS_UNSIGNED_LONG
} PanConfigNodeType;

typedef struct 
{
	char              * key;
	PanConfigNodeType   type;
	union
	{
		char     * val_string;
		gboolean   val_bool;
		int        val_int;
		gulong     val_ulong;
	} value;
} PanConfigNode;


/**
 ** Managing the configuration tree
 **/

static GNode       * config_root = NULL;
static gboolean      config_is_dirty = FALSE;

static PanConfigNode *
config_create_empty_node (const char * key, int key_len)
{
	PanConfigNode * node = g_new (PanConfigNode, 1);
	node->key = g_strndup (key, key_len);
	node->type = IS_NOTYPE;
	node->value.val_int = 0;
	return node;
}

static PanConfigNode *
config_create_node (const char        * key,
                    int                 key_len,
		    PanConfigNodeType   type,
		    va_list             ap)
{
	PanConfigNode * node = g_new (PanConfigNode, 1);

	g_return_val_if_fail (node != NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);
	g_return_val_if_fail (key_len>0, NULL);

	node->key = g_strndup (key, key_len);
	node->type = type;
	
	switch (type)
	{
		case IS_STRING:
			node->value.val_string = g_strdup (va_arg (ap, char *));
			break;
		case IS_BOOL:
			node->value.val_bool = va_arg (ap, gboolean);
			break;
		case IS_INT:
			node->value.val_int = va_arg (ap, int);
			break;
		case IS_UNSIGNED_LONG:
			node->value.val_ulong = va_arg (ap, gulong);
			break;
		default:
			break;
	}

	return node;
}


static gboolean
config_node_deleter (GNode * node, gpointer data)
{
	PanConfigNode * config;

	g_return_val_if_fail (node != NULL, FALSE);
	g_return_val_if_fail (node->data != NULL, FALSE);

	config = (PanConfigNode *) node->data;

	if (config->type == IS_STRING)
		g_free (config->value.val_string);
	g_free (config->key);
	g_free (config);

	node->data = NULL;

	return FALSE;
}


static void
config_delete_node (GNode * node)
{
	g_return_if_fail (node != NULL);

	g_node_traverse (node, G_IN_ORDER, G_TRAVERSE_ALL, -1, 
	                 config_node_deleter, NULL);

	g_node_destroy (node);

	if (node == config_root)
		config_root = NULL;

	config_is_dirty = TRUE;
}

static GNode *
config_scan_children (GNode * node, const char * key, int key_len)
{
	GNode * child = NULL;

	g_return_val_if_fail (node!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);
	g_return_val_if_fail (key_len>0, NULL);

	child = g_node_first_child (node);

	while (child != NULL)
	{
		const PanConfigNode * data = (const PanConfigNode*) child->data;

		if (!strncmp (key, data->key, key_len))
			break;

		child = g_node_next_sibling (child);
	}

	return child;
}

static GNode *
config_search_node (const char * key)
{
	GNode    * node = config_root;
	const char * march;
	const char * str;
	int str_len;

	g_return_val_if_fail (config_root!=NULL, NULL);
	g_return_val_if_fail (is_nonempty_string(key), NULL);
	g_return_val_if_fail (*key=='/', NULL);

	march = key + 1;
	while (node!=NULL && get_next_token_run (march, '/', &march, &str, &str_len))
		node = config_scan_children (node, str, str_len);

	return node;
}


static void
config_set_or_add_node (const char        * key, 
                        PanConfigNodeType   type,
                        ...)
{
	GNode * node = config_root;
	va_list ap;
	const char * march = NULL;
	const char * str = NULL;
	int str_len = 0;

	/* sanity clause */
	g_return_if_fail (config_root!=NULL);
	g_return_if_fail (is_nonempty_string(key));
	g_return_if_fail (*key=='/');

	/* loop down the key */
	march = key + 1;
	va_start (ap, type);
	while (get_next_token_run (march, '/', &march, &str, &str_len))
	{
		GNode * child = config_scan_children (node, str, str_len);
		const gboolean is_bottom = str[str_len] == '\0';

		if (child == NULL)
		{
			PanConfigNode * data = is_bottom
				? config_create_node (str, str_len, type, ap)
				: config_create_empty_node (str, str_len);

			child = g_node_append (node, g_node_new(data));
		}
		else if (is_bottom)
		{
			config_node_deleter (child, NULL);
			child->data = config_create_node (str, str_len, type, ap);
		}

		node = child;
	}
	va_end (ap);

	config_is_dirty = TRUE;
}

/**
***
**/

static void
get_config_filename (char * buf, int buflen)
{
	g_snprintf (buf, buflen,
	            "%s%cconfig.xml", get_data_dir(), G_DIR_SEPARATOR);
}

/**
***  Saving PanConfig
**/

static guint last_depth = 0;	/* Not too crazy about this one */

static void 
newline_depth (GString * appendme, guint depth)
{
	guint i;

	g_string_append_c (appendme, '\n');
	for (i=0; i<depth; ++i)
		g_string_append_c (appendme, '\t');
}

static void
close_previous_level (GString * appendme, guint last_level, guint level)
{
	guint i;

	for (i=last_level; i>level; i--)
	{
		newline_depth (appendme, i-2);
		g_string_append (appendme, "</section>");
	}
}

static void
pan_g_string_append_escaped (GString * gstr, char * escapeme)
{
	char * pch = pan_str_escape (escapeme);
	g_string_append (gstr, pch);
	g_free (pch);
}

static gboolean
write_node (GNode * node, gpointer data)
{
	char            types[] = "_sbn";
	GString       * appendme = (GString *) data;
	PanConfigNode * config = node->data;
	guint           depth = g_node_depth (node);

	if (depth == 1)
		return FALSE;

	close_previous_level (appendme, last_depth, depth);

	newline_depth (appendme, depth-1);
	if (config->type == IS_NOTYPE)
	{
		g_string_append (appendme, "<section key=\"");
		pan_g_string_append_escaped (appendme, config->key);
		g_string_append (appendme, "\">");
	}
	else
	{
		g_string_append (appendme, "<value key=\"");
		pan_g_string_append_escaped (appendme, config->key);
		g_string_append_printf (appendme, "\" type=\"%c\">", types[config->type]);

		switch (config->type)
		{
			case IS_STRING:
				if (config->value.val_string != NULL)
					pan_g_string_append_escaped (appendme, config->value.val_string);
				break;
			case IS_BOOL:
				g_string_append_printf (appendme, "%s", config->value.val_int ? "true" : "false");
				break;
			case IS_INT:
				g_string_append_printf (appendme, "%d", config->value.val_int);
				break;
			case IS_UNSIGNED_LONG:
				g_string_append_printf (appendme, "%lu", config->value.val_ulong);
				break;
			case IS_NOTYPE:
				pan_warn_if_reached ();
				break;
		}

		g_string_append (appendme, "</value>");
	}

	last_depth = depth;

	return FALSE;
}

static void
pan_config_save (void)
{
	GString * output;
	char      fname[PATH_MAX];
	char    * tmp_path;
	FILE    * fp;
	gboolean  ok;

	if (config_root == NULL)
		return;

	output = g_string_sized_new (4096);

	g_string_assign (output, "<?xml version=\"1.0\" encoding=\"UTF-8\" ?>\n"
		"<!DOCTYPE config SYSTEM \"config.dtd\">\n"
		"<config>");

	last_depth = 0;
	g_node_traverse (config_root, G_PRE_ORDER, G_TRAVERSE_ALL, -1, 
	                 write_node, output);
	close_previous_level (output, last_depth, 2);

	g_string_append (output, "\n</config>");

	get_config_filename (fname, sizeof(fname));
	tmp_path = pan_file_make_temp (&fp);
	ok = fp != NULL;

	if (ok) {
		const size_t to_write = strlen (output->str);
		const size_t written  = fwrite (output->str, sizeof(char), to_write, fp);
		fclose (fp);
		ok = to_write == written;
	}

	if (ok) {
		ok = pan_file_rename (tmp_path, fname);
	}

	if (!ok) {
		log_add_va (LOG_ERROR, _("Can't write to \"%s\": %s"),
		            fname, g_strerror (errno));
	}

	g_free (tmp_path);
	g_string_free (output, TRUE);
}

void
pan_config_sync (void)
{
	if (config_is_dirty)
	{
		pan_config_save ();
		config_is_dirty = FALSE;
	}
}


/**
 ** Loading PanConfig
 **/

static void
parse_and_load_file (xmlDoc *doc, xmlNode *node, const char * path)
{

	g_return_if_fail (doc!=NULL);
	g_return_if_fail (node!=NULL);

	for (; node!=NULL; node=node->next)
	{
		if (!pan_strcmp (node->name, "section")) {

			/* section: recurse through the xml structure */

			xmlChar * key = NULL;
			char    * new_path = NULL;

			key = xmlGetProp (node, "key");
			new_path = g_strdup_printf ("%s/%s", path, key);
			xmlFree (key);

			parse_and_load_file (doc, node->xmlChildrenNode, 
				new_path);

			g_free (new_path);
		}
		else
		if (!pan_strcmp (node->name, "value")) {

			/* add a value */

			xmlChar  * key = NULL;
			xmlChar  * type = NULL;
			xmlChar  * raw = NULL;
			char     * contents = NULL;
			char     * new_key = NULL;

			key  = xmlGetProp (node, "key");
			type = xmlGetProp (node, "type");
			new_key = g_strdup_printf ("%s/%s", path, key);

			if (node->xmlChildrenNode) {
				raw = xmlNodeListGetString (doc, node->xmlChildrenNode, FALSE);
			}

			contents = raw ? pan_str_unescape ((const char *) raw) : g_strdup ("");

			switch (type[0]) {
			case 's':
				config_set_or_add_node (new_key, IS_STRING, contents);

				break;
			case 'n':
				config_set_or_add_node (new_key, IS_INT, atoi (contents));
				break;
			case 'b':
				if (isdigit((guchar)contents[0]))
					config_set_or_add_node (new_key, IS_BOOL, contents[0] == '1');
				else
					config_set_or_add_node (new_key, IS_BOOL, tolower(contents[0]) == 't');
				break;
			case 'u':
				config_set_or_add_node (new_key, IS_UNSIGNED_LONG, strtoul(contents, NULL, 10));
			}

			g_free (new_key);
			g_free (contents);
			xmlFree (raw);
			xmlFree (key);
			xmlFree (type);
		}
	}
}

static void
pan_config_load_ini (void)
{
	int qty = 0;
	const char * march;
	char * pch;
	char * file;
	char * filename;
	char * category = NULL;
	GString * line;

	/* read the ini file */
	log_add (LOG_INFO, "can't find config.xml file; trying to import for ~/.gnome/Pan");
	filename = g_build_filename (g_get_home_dir(), ".gnome", "Pan", NULL);
	g_file_get_contents (filename, &file, NULL, NULL);

	/* march through the file */
	march = file;
	line = g_string_new (NULL);
	while (get_next_token_g_str (march, '\n', &march, line))
	{
		/* clean the line */
		pan_g_string_strstrip (line);
		if (!line->len)
			continue;

		/* check for new category */
		if (*line->str == '[') {
			replace_gstr (&category, g_strdup (line->str+1));
			if ((pch = strchr (category, ']')))
				*pch = '\0';
		}

		/* check for a value */
		pch = strchr (line->str, '=');
		if (pch != NULL)
		{
			char * key = g_strndup (line->str, pch-line->str);
			char * val = g_strdup (pch+1);

			g_strstrip (key);
			g_strstrip (val);

			if (strstr(key,"font") == NULL) /* omit pre-Pango font settings */
			{
				char * path = g_strdup_printf ("/Pan/%s/%s", category, key);
				config_set_or_add_node (path, IS_STRING, val);
				g_free (path);

				++qty;
			}

			g_free (val);
			g_free (key);
		}
	}

	if (qty)
		log_add_va (LOG_INFO, _("Imported %d lines from Gnome config file"), qty);

	g_string_free (line, TRUE);
	g_free (file);
	g_free (filename);
	g_free (category);
}

static void
pan_config_load (void)
{
	xmlDocPtr    doc;
	xmlNodePtr   node;
	char         fname[PATH_MAX];

	g_return_if_fail (config_root==NULL);

	if (config_is_dirty) {
		g_warning ("Trying to load over an unsynced pan_config. Aborting");
		return ;
	}

	config_root = g_node_new (config_create_empty_node ("", 0));

	get_config_filename (fname, sizeof(fname));

	if (!pan_file_exists (fname)) {
		pan_config_load_ini ();
		return;
	}

	doc = xmlParseFile (fname);
	g_return_if_fail (doc!=NULL);

	node = xmlDocGetRootElement (doc);
	g_return_if_fail (node!=NULL);

	if (pan_strcmp ((const char *) node->name, "config"))
		g_warning (_("%s does not appear to be a valid datafile"), fname);
	else 
		parse_and_load_file (doc, node->xmlChildrenNode, "");

	xmlFreeDoc (doc);
}

/**
 ** Prefix Management
 **/

static char * prefix = NULL;

static void
compose_key (const char * key, char * buf, int buf_max)
{
	if (prefix)
		g_snprintf (buf, buf_max, "%s/%s", prefix, key);
	else
		g_strlcpy (buf, key, buf_max);
}

void
pan_config_push_prefix (const char * new_prefix)
{
	if (prefix != NULL)
		g_free (prefix);

	prefix = g_strdup (new_prefix);

	if (prefix[strlen(prefix)-1] == '/')
		prefix[strlen(prefix)-1] = '\0';
}

void
pan_config_pop_prefix (void)
{
	g_free (prefix);
	prefix = NULL;
}


/**
 ** Parsing defaults out of a key
 **/

static char *
parse_default_string (char * key)
{
	char   * val = NULL;
	char   * pch = strrchr (key, '=');

	if (pch != NULL)
	{
		val = g_strdup (pch+1);
		*pch = '\0';
	}

	return val;
}


static gboolean
parse_default_bool (char * key)
{
	gboolean   val = FALSE;
	char    * pch = strrchr (key, '=');

	if (pch != NULL)
	{
#if 1
		char * tmp = g_ascii_strup (pch+1, -1);
#else
		char * tmp = g_strdup (pch+1);
		g_strup (pch+1);
#endif

		val = tolower(tmp[0]) == 't';
		*pch = '\0';

		g_free (tmp);
	}

	return val;
}


static int
parse_default_int (char * key)
{
	int    val = 0;
	char * pch = strrchr (key, '=');

	if (pch != NULL)
	{
		val = atoi (pch+1);
		*pch = '\0';
	}

	return val;
}

static gulong
parse_default_ulong (char * key)
{
	gulong retval = 0;
	char * pch = strrchr (key, '=');

	if (pch != NULL)
	{
		retval = strtoul (pch+1, NULL, 10);
		*pch = '\0';
	}

	return retval;
}

/**
 ** GETTING 
 **/

char*
pan_config_get_string (const char * key)
{
	char            new_key[MAX_KEY_LEN];
	char          * val;
	GNode         * node;
	PanConfigNode * data;
	debug_enter ("pan_config_get_string");

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	val = parse_default_string (new_key);

	node = config_search_node (new_key);
	data = (PanConfigNode *) (node ? node->data : NULL);

	if (data != NULL && data->type == IS_STRING)
	{
		g_free (val);
		val = g_strdup (data->value.val_string);
	}

	/*g_message ("get [%s] returns string [%s]", key, val ? val : "(null)");*/
	debug_exit ("pan_config_get_string");
	return val;
}

gboolean
pan_config_get_bool (const char * key)
{
	char            new_key[MAX_KEY_LEN];
	gboolean        val;
	GNode         * node;
	PanConfigNode * data;

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	val = parse_default_bool (new_key);

	node = config_search_node (new_key);
	data = (PanConfigNode *) (node ? node->data : NULL);

	if (data != NULL) {
	       if (data->type == IS_BOOL)
		       val = data->value.val_bool;
	       else if (data->type == IS_INT)
		       val = data->value.val_int != 0;
	       else if (data->type == IS_STRING)
		       val = !pan_strcmp (data->value.val_string, "true");
	}

	/*g_message ("get [%s] returns bool [%d]", key, val);*/
	return val;
}


int
pan_config_get_int (const char * key)
{
	int             val;
	char            new_key[MAX_KEY_LEN];
	GNode         * node;
	PanConfigNode * data;
	debug_enter ("pan_config_get_int");

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	val = parse_default_int (new_key);

	node = config_search_node (new_key);
	data = (PanConfigNode *) (node ? node->data : NULL);
	
	if (data) {
		if (data->type == IS_INT)
			val = data->value.val_int;
		else if (data->type == IS_STRING)
			val = atoi (data->value.val_string);
	}

	/*g_message ("get [%s] returns int [%d]", key, val);*/
	debug_exit ("pan_config_get_int");
	return val;
}

gulong
pan_config_get_ulong (const char * key)
{
	gulong          val;
	char            new_key[MAX_KEY_LEN];
	GNode         * node;
	PanConfigNode * data;
	debug_enter ("pan_config_get_ulong");

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	val = parse_default_ulong (new_key);
	
	node = config_search_node (new_key);
	data = (PanConfigNode *) (node ? node->data : NULL);
	
	if (data) {
		if (data->type == IS_UNSIGNED_LONG)
			val = data->value.val_ulong;
		else if (data->type == IS_STRING)
			val = strtoul (data->value.val_string, NULL, 10);
	}

	debug_exit ("pan_config_get_ulong");
	return val;
}


gboolean
pan_config_has_section (const char * section)
{
	GNode         * node = NULL;
	PanConfigNode * data = NULL;

	if (config_root == NULL)
		pan_config_load ();

	node = config_search_node (section);
	data = (PanConfigNode *) (node ? node->data : NULL);

	return data && data->type == IS_NOTYPE;
}


/**
 ** SETTING
 **/

void
pan_config_set_string (const char * key, const char * value)
{
	char new_key[MAX_KEY_LEN];

	if (config_root == NULL)
		pan_config_load ();

	if (value == NULL)
		value = "";

	compose_key (key, new_key, sizeof(new_key));
	config_set_or_add_node (new_key, IS_STRING, value);
}


void
pan_config_set_bool (const char * key, gboolean value)
{
	char new_key[MAX_KEY_LEN];

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	config_set_or_add_node (new_key, IS_BOOL, value);
}


void
pan_config_set_int (const char * key, int value)
{
	char new_key[MAX_KEY_LEN];

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	config_set_or_add_node (new_key, IS_INT, value);
}

void
pan_config_set_ulong (const char * key, gulong value)
{
	char new_key[MAX_KEY_LEN];

	if (config_root == NULL)
		pan_config_load ();

	compose_key (key, new_key, sizeof(new_key));
	config_set_or_add_node (new_key, IS_UNSIGNED_LONG, value);
}

void
pan_config_clean_key (const char * key)
{
	GNode * node;

	if (config_root == NULL)
		pan_config_load ();

	node = config_search_node (key);

	if (node)
	{
		g_return_if_fail (g_node_n_children (node) == 0);

		config_delete_node (node);
	}
}


void
pan_config_clean_section (const char * section)
{
	GNode * node;

	if (config_root == NULL)
		pan_config_load ();

	node = config_search_node (section);

	if (node)
	{
		config_delete_node (node);
	}
}



