
/*
 * main.c
 *
 *   July 10, 1994 -- Lawrence Kesteloot
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "hfs.h"
#include "btree.h"

#define MAXVOLS		8
#define MAXDIR		128

struct volume_s {
	int			drive;
	char			part;
	char			fn[32];
	char			name[32];
	struct hfsVolumeInfo	volinfo;
	ushort			xtblk;		/* Extents tree block	*/
	ushort			ctblk;		/* Catalog tree block	*/
} volume[MAXVOLS];

struct hfsEntryInfo	entryinfo[MAXDIR], ei, cwd, stack[16];
int			f, numdir, stackdepth;
struct volume_s		*v;
struct hfsVolumeInfo	*vi;

/*
 * mainCd()
 *
 *   Change to a new directory and display it.  Argument is either index
 *   into current list of files, or -3 for current directory (i.e., ls),
 *   -2 for root directory, or -1 for parent directory.
 */

void mainCd (int sel)
{
	if (sel < -3 || sel >= numdir) {
		printf ("Invalid selection.\n");
		return;
	}
	if (sel == -3) { /* Stay */
		/* Do nothing */
	} else if (sel == -2) { /* Goto root */
		stackdepth = 1;
	} else if (sel == -1) { /* Go up */
		if (stackdepth == 1) {
			printf ("At top level.\n");
			return;
		}
		stackdepth--;
	} else {
		stack[stackdepth++] = entryinfo[sel];
	}
	cwd = stack[stackdepth - 1];
	numdir = 0;
	ei = cwd;
	if (hfsGetFirst (f, v->ctblk, &ei) == 0) {
		printf ("Directory empty.\n");
	} else {
		do {
			printf ("%d.\t%s", numdir + 1, ei.name);
			if (ei.info.filetype == 1) {
				printf ("/\n");
			} else {
				printf (" (%d bytes)\n", ei.info.fil.filLgLen);
			}
			entryinfo[numdir++] = ei;
		} while (hfsGetNext (f, v->ctblk, &ei));
	}
}

/*
 * mainCat()
 *
 *   Displays a file.  Argument is index into current list.  We assume
 *   that this is a MacOS-format file, so carriage-returns are converted
 *   into newlines.
 */

void mainCat (int sel)
{
	uchar			buf[2048];
	int			wrote, n, i;

	if (sel < 0 || sel >= numdir) {
		printf ("Invalid selection.\n");
		return;
	}

	wrote = 0;
	do {
		n = hfsRead (f, vi, v->xtblk, &entryinfo[sel], buf, wrote,
			sizeof (buf));
		if (n == -1) {
			printf ("mainCat: hfsRead error\n");
			break;
		}
		for (i = 0; i < n; i++) {
			if (buf[i] == '\r') {
				putchar ('\n');
			} else {
				putchar (buf[i]);
			}
		}
		wrote += n;
	} while (n > 0);
	printf ("\n");
}

/*
 * mainCp()
 *
 *   Copies a file from MacOS to UNIX.  Arguments are index into current
 *   list and filename of UNIX file to write.  This is a binary copy --
 *   no C/R translation is performed.
 */

void mainCp (int sel, char *destfn)
{
	uchar			buf[8192];
	int			outf, wrote, n;

	if (sel < 0 || sel >= numdir) {
		printf ("Invalid selection.\n");
		return;
	}

	outf = open (destfn, O_WRONLY | O_CREAT | O_TRUNC, 0666);
	if (outf == -1) {
		perror (destfn);
		return;
	}

	wrote = 0;
	do {
		printf ("%d bytes copied.\r", wrote);
		fflush (stdout);
		n = hfsRead (f, vi, v->xtblk, &entryinfo[sel], buf, wrote,
			sizeof (buf));
		if (n == -1) {
			printf ("mainCp: hfsRead error     \n");
			break;
		}
		n = write (outf, buf, n);
		if (n == -1) {
			perror (destfn);
			break;
		}
		wrote += n;
	} while (n > 0);

	printf ("Wrote %d bytes to \"%s\".\n", wrote, destfn);
	close (outf);
}

/*
 * mainExecuteLine()
 *
 *   Executes a command from the user.  Returns 0 when the program should
 *   quit, 1 otherwise.
 */

static int mainExecuteLine (char *s)
{
	int	argc;
	char	*argv[16];

	argc = 0;
	argv[argc] = strtok (s, " ");
	while (argv[argc] != NULL) {
		argv[++argc] = strtok (NULL, " ");
	}

	if (argc == 0) {
		return 1;
	}
	if (strcmp (argv[0], "exit") == 0 ||
		strcmp (argv[0], "quit") == 0) {
		return 0;
	} else if (strcmp (argv[0], "cd") == 0) {
		if (argc < 2) {
			mainCd (-2); /* Goto root */
		} else if (strcmp (argv[1], ".") == 0) {
			mainCd (-3);
		} else if (strcmp (argv[1], "..") == 0) {
			mainCd (-1); /* Up one */
		} else {
			mainCd (atoi (argv[1]) - 1);
		}
	} else if (strcmp (argv[0], "ls") == 0) {
		mainCd (-3);
	} else if (strcmp (argv[0], "cat") == 0) {
		if (argc < 2) {
			printf ("Must specify file to cat.\n");
		} else {
			mainCat (atoi (argv[1]) - 1);
		}
	} else if (strcmp (argv[0], "cp") == 0) {
		if (argc < 2) {
			printf ("Must specify file to cp.\n");
		} else if (argc < 3) {
			printf ("Must specify destination UNIX "
				"file.\n");
		} else {
			mainCp (atoi (argv[1]) - 1, argv[2]);
		}
	} else if (strcmp (argv[0], "help") == 0) {
		printf ("Commands are:\n");
		printf ("\tcd [number]\n");
		printf ("\tcd .\n");
		printf ("\tcd ..\n");
		printf ("\tcd          (top level)\n");
		printf ("\tls\n");
		printf ("\tcat [number]\n");
		printf ("\tcp [number] [file]\n");
		printf ("\texit\n");
	} else if (strcmp (argv[0], "dump") == 0) {
		/* Secret dump command */
		if (argc < 2) {
			printf ("Must specify block.\n");
		} else {
			int blk;

			blk = expr (argv[1]);
			printf ("Dumping block %d (0x%x)\n", blk, blk);
			hfsDump (f, blk);
		}
	} else if (strcmp (argv[0], "eval") == 0) {
		/* Secret calculator command */
		if (argc < 2) {
			printf ("Must specify expression.\n");
		} else {
			int val;

			val = expr (argv[1]);
			printf ("%d (0x%x)\n", val, val);
		}
	} else {
		printf ("Unknown command: %s\n", argv[0]);
	}

	return 1;
}

/*
 * usage()
 *
 *   Prints usage information for the program and quits.
 */

static void usage (void)
{
	fprintf (stderr, "usage:  hfs [-d]\n");
	fprintf (stderr, "   -d    display debugging information.\n");
	exit (1);
}

/*
 * main()
 *
 *   Main function for "hfs" program.
 */

int main (int argc, char *argv[])
{
	char			fn[32], part, s[128];
	int			drive, numvols, i, sel, ch;

	while ((ch = getopt (argc, argv, "d")) != -1) {
		switch (ch) {
			case 'd':
				debug = 1;
				break;
			default:
				usage ();
				break;
		}
	}

	printf ("\nPlease wait, probing drives...\n\n");

	numvols = 0;
	v = &volume[numvols];

	for (drive = 0; drive < 8; drive++) {
		for (part = 'a'; part < 'i'; part++) {
			sprintf (fn, "/dev/sd%d%c", drive, part);
			if (hfsGetVolumeInfo (fn, &v->volinfo)) {
				v->drive = drive;
				v->part = part;
				strcpy (v->fn, fn);
				memcpy (v->name,
					v->volinfo.drVolName,
					v->volinfo.drVN);
				v->name[v->volinfo.drVN] = '\0';
				numvols++;
				v++;
			}
		}
	}

	if (numvols == 0) {
		printf ("Did not find any HFS volumes.\n");
		exit (1);
	}

	printf ("Please choose the HFS volume to browse:\n\n");
	for (i = 0; i < numvols; i++) {
		printf ("%d.\t%s (SCSI %d)\n", i + 1, volume[i].name,
			volume[i].drive + 1);
	}
	printf ("\nVolume number, or <ENTER> to exit: ");
	if (!fgets (s, sizeof (s), stdin) || s[0] == '\n') {
		exit (0);
	}

	sel = s[0] - '1';
	if (sel < 0 || sel >= numvols) {
		printf ("Invalid selection.\n");
		exit (1);
	}

	v = &volume[sel];
	vi = &v->volinfo;

	v->xtblk = vi->drAlBlSt +
		*(ushort *)&vi->drXTExtRec[0] * (vi->drAlBlkSiz / BLKSIZ);
	v->ctblk = vi->drAlBlSt +
		*(ushort *)&vi->drCTExtRec[0] * (vi->drAlBlkSiz / BLKSIZ);

	printf ("\nType \"help\" for help.\n\n");
	f = open (v->fn, O_RDONLY);
	hfsGetRootDir (f, v->ctblk, v->name, &stack[0]);
	stackdepth = 1;
	mainCd (-2);
	do {
		printf ("%s:", stack[0].name);
		if (stackdepth == 1) {
			printf ("/");
		} else {
			for (i = 1; i < stackdepth; i++) {
				printf ("/%s", stack[i].name);
			}
		}
		printf (" > ");
		if (fgets (s, sizeof (s), stdin) == NULL) {
			printf ("\n");
			break;
		}
		s[strlen(s) - 1] = '\0'; /* Remove \n */
	} while (mainExecuteLine (s));

	close (f);

	return 0;
}
