/*	$NetBSD: lcmd.c,v 1.8 2006/12/18 20:04:55 christos Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: lcmd.c,v 1.8 2006/12/18 20:04:55 christos Exp $");
#endif /* not lint */

#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <inttypes.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "defs.h"
#include "lcmd.h"
#include "var.h"


extern struct lcmd_arg arg_alias[];
extern struct lcmd_arg arg_cursormodes[];
extern struct lcmd_arg arg_debug[];
extern struct lcmd_arg arg_echo[];
extern struct lcmd_arg arg_escape[];
extern struct lcmd_arg arg_foreground[];
extern struct lcmd_arg arg_label[];
extern struct lcmd_arg arg_def_nline[];
extern struct lcmd_arg arg_def_shell[];
extern struct lcmd_arg arg_def_smooth[];
extern struct lcmd_arg arg_close[];
extern struct lcmd_arg arg_select[];
extern struct lcmd_arg arg_smooth[];
extern struct lcmd_arg arg_source[];
extern struct lcmd_arg arg_terse[];
extern struct lcmd_arg arg_time[];
extern struct lcmd_arg arg_unalias[];
extern struct lcmd_arg arg_unset[];
extern struct lcmd_arg arg_window[];
extern struct lcmd_arg arg_write[];
struct lcmd_arg arg_null[1] = { { NULL, 0, 0 } };

struct lcmd_tab lcmd_tab[] = {
	{ "alias",		1,	l_alias,	arg_alias },
	{ "close",		2,	l_close,	arg_close },
	{ "cursormodes",	2,	l_cursormodes,	arg_cursormodes },
	{ "debug",		1,	l_debug,	arg_debug },
	{ "default_nlines",	9,	l_def_nline,	arg_def_nline },
	{ "default_shell",	10,	l_def_shell,	arg_def_shell },
	{ "default_smooth",	10,	l_def_smooth,	arg_def_smooth },
	{ "echo",		2,	l_echo,		arg_echo },
	{ "escape",		2,	l_escape,	arg_escape },
	{ "foreground",		1,	l_foreground,	arg_foreground },
	{ "iostat",		1,	l_iostat,	arg_null },
	{ "label",		2,	l_label,	arg_label },
	{ "list",		2,	l_list,		arg_null },
	{ "nlines",		1,	l_def_nline,	arg_def_nline },
	{ "select",		2,	l_select,	arg_select },
	{ "shell",		2,	l_def_shell,	arg_def_shell },
	{ "smooth",		2,	l_smooth,	arg_smooth },
	{ "source",		2,	l_source,	arg_source },
	{ "terse",		2,	l_terse,	arg_terse },
	{ "time",		2,	l_time,		arg_time },
	{ "unalias",		3,	l_unalias,	arg_unalias },
	{ "unset",		3,	l_unset,	arg_unset },
	{ "variable",		1,	l_variable,	arg_null },
	{ "window",		2,	l_window,	arg_window },
	{ "write",		2,	l_write,	arg_write },
	{ NULL,			0,	NULL,		NULL }
};

struct lcmd_tab *
lcmd_lookup(char *name)
{
	struct lcmd_tab *p;

	for (p = lcmd_tab; p->lc_name != 0; p++)
		if (strncmp(name, p->lc_name, p->lc_minlen) == 0)
			return p;
	return 0;
}

int
dosource(char *filename)
{
	if (cx_beginfile(filename) < 0)
		return -1;
	p_start();
	err_end();
	cx_end();
	return 0;
}

int
dolongcmd(char *buffer, value_t *arg, int narg)
{
	if (cx_beginbuf(buffer, arg, narg) < 0)
		return -1;
	p_start();
	err_end();
	cx_end();
	return 0;
}

char	vtobool(value_t *, char, char);

struct lcmd_arg arg_window[] = {
	{ "row",	1,	ARG_NUM },
	{ "column",	1,	ARG_NUM },
	{ "nrows",	2,	ARG_NUM },
	{ "ncols",	2,	ARG_NUM },
	{ "nlines",	2,	ARG_NUM },
	{ "label",	1,	ARG_STR },
	{ "pty",	1,	ARG_ANY },
	{ "frame",	1,	ARG_ANY },
	{ "mapnl",	1,	ARG_ANY },
	{ "keepopen",	1,	ARG_ANY },
	{ "smooth",	1,	ARG_ANY },
	{ "shell",	1,	ARG_STR|ARG_LIST },
	{ NULL,		0,	0 }
};

static winvars_t	*savedwinvars;

void
save_winvars(winvars_t *winvars)
{
	savedwinvars = winvars;
}

winvars_t *
get_winvars(void)
{
	return savedwinvars;
}

void
l_window(winvars_t *winvars, value_t *v, value_t *a)
{
	ww_t *w;
	int col, row, ncol, nrow, id, nline;
	char *label;
	int haspty, hasframe, mapnl, keepopen, smooth;
	char *shf, **sh;
	char *argv[MAX_SHELL_ARGC];
	char **pp;

	if ((id = findid(winvars)) < 0)
		return;
	row = a->v_type == V_ERR ? 1 : a->v_num;
	a++;
	col = a->v_type == V_ERR ? 0 : a->v_num;
	a++;
	nrow = a->v_type == V_ERR ? winvars->wwnrow - row : a->v_num;
	a++;
	ncol = a->v_type == V_ERR ? winvars->wwncol - col : a->v_num;
	a++;
	nline = a->v_type == V_ERR ? winvars->default_nline : a->v_num;
	a++;
	label = a->v_type == V_ERR ? 0 : a->v_str;
	if ((haspty = vtobool(++a, 1, -1)) < 0)
		return;
	if ((hasframe = vtobool(++a, 1, -1)) < 0)
		return;
	if ((mapnl = vtobool(++a, !haspty, -1)) < 0)
		return;
	if ((keepopen = vtobool(++a, 0, -1)) < 0)
		return;
	if ((smooth = vtobool(++a, winvars->default_smooth, -1)) < 0)
		return;
	if ((++a)->v_type != V_ERR) {
		for (pp = argv; a->v_type != V_ERR &&
		     pp < &argv[sizeof argv/sizeof *argv-1]; pp++, a++)
			*pp = a->v_str;
		*pp = 0;
		shf = *(sh = argv);
		if ((*sh = strrchr(shf, '/')) == NULL) {
			*sh = shf;
		} else {
			(*sh)++;
		}
	} else {
		sh = winvars->default_shell;
		shf = winvars->default_shellfile;
	}
	if ((w = openwin(winvars, id, row, col, nrow, ncol, nline, label,
	    haspty ? WWT_PTY : WWT_SOCKET, hasframe ? WWU_HASFRAME : 0, shf,
	    sh)) == NULL) {
		return;
	}
	if (mapnl)
		SET(w->ww_wflags, WWW_MAPNL);
	else
		CLR(w->ww_wflags, WWW_MAPNL);
	if (keepopen)
		SET(w->ww_uflags, WWU_KEEPOPEN);
	else
		CLR(w->ww_uflags, WWU_KEEPOPEN);
	if (!smooth)
		SET(w->ww_wflags, WWW_NOUPDATE);
	else
		CLR(w->ww_wflags, WWW_NOUPDATE);
	v->v_type = V_NUM;
	v->v_num = id + 1;
}

struct lcmd_arg arg_def_nline[] = {
	{ "nlines",	1,	ARG_NUM },
	{ NULL,		0,	0 }
};

void
l_def_nline(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_num = winvars->default_nline;
	v->v_type = V_NUM;
	if (a->v_type != V_ERR)
		winvars->default_nline = a->v_num;
}

struct lcmd_arg arg_smooth[] = {
	{ "window",	1,	ARG_NUM },
	{ "flag",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_smooth(winvars_t *winvars, value_t *v, value_t *a)
{
	ww_t *w;

	v->v_type = V_NUM;
	v->v_num = 0;
	if ((w = vtowin(a++, winvars->selwin)) == 0)
		return;
	v->v_num = ISSET(w->ww_wflags, WWW_NOUPDATE) == 0;
	if (!vtobool(a, v->v_num, v->v_num))
		SET(w->ww_wflags, WWW_NOUPDATE);
	else
		CLR(w->ww_wflags, WWW_NOUPDATE);
}

struct lcmd_arg arg_def_smooth[] = {
	{ "flag",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_def_smooth(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	v->v_num = winvars->default_smooth;
	winvars->default_smooth = vtobool(a, v->v_num, v->v_num);
}

struct lcmd_arg arg_select[] = {
	{ "window",	1,	ARG_NUM },
	{ NULL,		0,	0 }
};

void
l_select(winvars_t *winvars, value_t *v, value_t *a)
{
	ww_t *w;

	v->v_type = V_NUM;
	v->v_num = winvars->selwin ? winvars->selwin->ww_id + 1 : -1;
	if (a->v_type == V_ERR)
		return;
	if ((w = vtowin(a, NULL)) == NULL)
		return;
	setselwin(winvars, w);
}

struct lcmd_arg arg_debug[] = {
	{ "flag",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_debug(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	v->v_num = winvars->debug;
	winvars->debug = vtobool(a, winvars->debug, winvars->debug);
}

struct lcmd_arg arg_escape[] = {
	{ "escapec",	1,	ARG_STR },
	{ NULL,		0,	0 }
};

void
l_escape(winvars_t *winvars, value_t *v, value_t *a)
{
	char buf[3];

	buf[0] = winvars->escapec;
	buf[1] = winvars->literalc;
	buf[2] = 0x0;
	if ((v->v_str = strdup(buf)) == NULL) {
		error("Out of memory.");
		return;
	}
	v->v_type = V_STR;
	if (a->v_type != V_ERR)
		setescape(winvars, a->v_str);
}

struct lcmd_arg arg_label[] = {
	{ "window",	1,	ARG_NUM },
	{ "label",	1,	ARG_STR },
	{ NULL,		0,	0 }
};

/* ARGSUSED0 */
void
l_label(winvars_t *winvars, value_t *v __unused, value_t *a)
{
	ww_t *w;

	if ((w = vtowin(a, winvars->selwin)) == 0)
		return;
	if ((++a)->v_type != V_ERR && setlabel(w, a->v_str) < 0)
		error("Out of memory.");
	reframe();
}

struct lcmd_arg arg_foreground[] = {
	{ "window",	1,	ARG_NUM },
	{ "flag",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_foreground(winvars_t *winvars, value_t *v, value_t *a)
{
	ww_t *w;
	char flag;

	if ((w = vtowin(a, winvars->selwin)) == 0)
		return;
	v->v_type = V_NUM;
	v->v_num = ISFG(winvars, w);
	flag = vtobool(++a, v->v_num, v->v_num);
	if (flag == v->v_num)
		return;
	deletewin(w);
	addwin(w, flag);
	reframe();
}

struct lcmd_arg arg_terse[] = {
	{ "flag",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_terse(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	v->v_num = winvars->terse;
	setterse(winvars, vtobool(a, winvars->terse, winvars->terse));
}

struct lcmd_arg arg_source[] = {
	{ "filename",	1,	ARG_STR },
	{ NULL,		0,	0 }
};

void
l_source(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	if (a->v_type != V_ERR && dosource(a->v_str) < 0) {
		error("Can't open %s.", a->v_str);
		v->v_num = -1;
	} else
		v->v_num = 0;
}

struct lcmd_arg arg_write[] = {
	{ "window",	1,	ARG_NUM },
	{ "",		0,	ARG_ANY|ARG_LIST },
	{ NULL,		0,	0 }
};

/* ARGSUSED0 */
void
l_write(winvars_t *winvars, value_t *v __unused, value_t *a)
{
	char buf[20];
	ww_t *w;

	if ((w = vtowin(a++, winvars->selwin)) == 0)
		return;
	while (a->v_type != V_ERR) {
		if (a->v_type == V_NUM) {
			(void) snprintf(buf, sizeof(buf), "%d", a->v_num);
			(void) write(w->ww_pty, buf, strlen(buf));
		} else
			(void) write(w->ww_pty, a->v_str, strlen(a->v_str));
		if ((++a)->v_type != V_ERR)
			(void) write(w->ww_pty, " ", 1);
	}
}

struct lcmd_arg arg_close[] = {
	{ "window",	1,	ARG_ANY|ARG_LIST },
	{ NULL,		0,	0 }
};

/* ARGSUSED0 */
void
l_close(winvars_t *winvars, value_t *v __unused, value_t *a)
{
	ww_t *w;

	if (a->v_type == V_STR && strncmp(a->v_str, "all", 3) == 0)
		closewin(winvars, NULL);
	else
		for (; a->v_type != V_ERR; a++)
			if ((w = vtowin(a, NULL)) != NULL)
				closewin(winvars, w);
}

struct lcmd_arg arg_cursormodes[] = {
	{ "modes",	1,	ARG_NUM },
	{ NULL,		0,	0 }
};

void
l_cursormodes(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	v->v_num = winvars->wwcursormodes;
	if (a->v_type != V_ERR)
		wwsetcursormodes(winvars, a->v_num);
}

struct lcmd_arg arg_unset[] = {
	{ "variable",	1,	ARG_ANY },
	{ NULL,		0,	0 }
};

void
l_unset(winvars_t *winvars, value_t *v, value_t *a)
{
	v->v_type = V_NUM;
	switch (a->v_type) {
	case V_ERR:
		v->v_num = -1;
		return;
	case V_NUM:
		(void) asprintf(&a->v_str, "%d", a->v_num);
		if (a->v_str == NULL) {
			error("Out of memory.");
			v->v_num = -1;
			return;
		}
		a->v_type = V_STR;
		break;
	}
	v->v_num = var_unset(a->v_str);
}

ww_t *
vtowin(value_t *v, ww_t *w)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	switch (v->v_type) {
	case V_ERR:
		if (w != 0)
			return w;
		error("No window specified.");
		return 0;
	case V_STR:
		error("%s: No such window.", v->v_str);
		return 0;
	}
	if (v->v_num < 1 || v->v_num > MAX_NUM_WINDOWS
	    || (w = winvars->window[v->v_num - 1]) == 0) {
		error("%d: No such window.", v->v_num);
		return 0;
	}
	return w;
}

char
vtobool(value_t *v, char def, char err)
{
	switch (v->v_type) {
	case V_NUM:
		return v->v_num != 0;
	case V_STR:
		if (strncmp(v->v_str, "true", 1) == 0 ||
		    strncmp(v->v_str, "on", 2) == 0 ||
		    strncmp(v->v_str, "yes", 1) == 0)
			return 1;
		else if (strncmp(v->v_str, "false", 1) == 0 ||
		    strncmp(v->v_str, "off", 2) == 0 ||
		    strncmp(v->v_str, "no", 1) == 0)
			return 0;
		else {
			error("%s: Illegal boolean value.", v->v_str);
			return err;
		}
		/*NOTREACHED*/
	case V_ERR:
		return def;
	}
	/*NOTREACHED*/
	return (0);
}

int	printalias(void *, variable_t *);
int	printvar(void *, variable_t *);
char	*strtime(struct timeval *t);

/* ARGSUSED0 */
void
l_iostat(winvars_t *winvars, value_t *v __unused, value_t *a __unused)
{
	ww_t *w;

	if ((w = openiwin(winvars, 16, "IO Statistics")) == 0) {
		error("Can't open statistics window: %s.", wwerror(winvars));
		return;
	}
	wwprintf(w, "ttflush\twrite\terror\tzero\tchar\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnflush, winvars->wwnwr, winvars->wwnwre, winvars->wwnwrz, winvars->wwnwrc);
	wwprintf(w, "token\tuse\tbad\tsaving\ttotal\tbaud\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "/%" PRIu64 " (%.1f/%.1f)\n",
		winvars->wwntokdef, winvars->wwntokuse, winvars->wwntokbad, winvars->wwntoksave, winvars->wwntokc,
		winvars->wwntokc - winvars->wwntoksave ?
			(int) ((float) winvars->wwbaud * winvars->wwntokc /
					(winvars->wwntokc - winvars->wwntoksave)) :
			winvars->wwbaud,
		winvars->wwnwrc ? (int) ((float) winvars->wwbaud * (winvars->wwnwrc + winvars->wwntoksave) /
					winvars->wwnwrc) :
			winvars->wwbaud,
		winvars->wwntokc - winvars->wwntoksave ?
			(float) winvars->wwntokc / (winvars->wwntokc - winvars->wwntoksave) : 1.0,
		winvars->wwnwrc ? (float) (winvars->wwnwrc + winvars->wwntoksave) / winvars->wwnwrc : 1.0);
	wwprintf(w, "wwwrite\tattempt\tchar\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnwwr, winvars->wwnwwra, winvars->wwnwwrc);
	wwprintf(w, "wwupdat\tline\tmiss\tscan\tclreol\tclreos\tmiss\tline\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnupdate, winvars->wwnupdline, winvars->wwnupdmiss,
		winvars->wwnupdscan, winvars->wwnupdclreol,
		winvars->wwnupdclreos, winvars->wwnupdclreosmiss, winvars->wwnupdclreosline);
	wwprintf(w, "select\terror\tzero\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnselect, winvars->wwnselecte, winvars->wwnselectz);
	wwprintf(w, "read\terror\tzero\tchar\tack\tnack\tstat\terrorc\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnread, winvars->wwnreade, winvars->wwnreadz,
		winvars->wwnreadc, winvars->wwnreadack, winvars->wwnreadnack,
		winvars->wwnreadstat, winvars->wwnreadec);
	wwprintf(w, "ptyread\terror\tzero\tcontrol\tdata\tchar\n");
	wwprintf(w, "%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\t%" PRIu64 "\n",
		winvars->wwnwread, winvars->wwnwreade, winvars->wwnwreadz,
		winvars->wwnwreadp, winvars->wwnwreadd, winvars->wwnwreadc);
	waitnl(w);
	closeiwin(w);
}

struct lcmd_arg arg_time[] = {
	{ "who",	1,	ARG_STR },
	{ NULL,		0,	0 }
};

/* ARGSUSED0 */
void
l_time(winvars_t *winvars, value_t *v __unused, value_t *a)
{
	ww_t *w;
	struct rusage rusage;
	struct timeval timeval;

	if ((w = openiwin(winvars, 8, "Timing and Resource Usage")) == 0) {
		error("Can't open time window: %s.", wwerror(winvars));
		return;
	}

	(void) gettimeofday(&timeval, NULL);
        timersub(&timeval, &winvars->starttime, &timeval);
	(void) getrusage(a->v_type == V_STR
			&& strncmp(a->v_str, "children", 1) == 0
		? RUSAGE_CHILDREN : RUSAGE_SELF, &rusage);

	wwprintf(w, "%-15s %-15s %-15s\n",
		"time", "utime", "stime");
	wwprintf(w, "%-15s ", strtime(&timeval));
	wwprintf(w, "%-15s ", strtime(&rusage.ru_utime));
	wwprintf(w, "%-15s\n", strtime(&rusage.ru_stime));
	wwprintf(w, "%-15s %-15s %-15s %-15s\n",
		"maxrss", "ixrss", "idrss", "isrss");
	wwprintf(w, "%-15ld %-15ld %-15ld %-15ld\n",
		rusage.ru_maxrss, rusage.ru_ixrss,
		rusage.ru_idrss, rusage.ru_isrss);
	wwprintf(w, "%-7s %-7s %-7s %-7s %-7s %-7s %-7s %-7s %-7s %-7s\n",
		"minflt", "majflt", "nswap", "inblk", "oublk",
		"msgsnd", "msgrcv", "nsigs", "nvcsw", "nivcsw");
	wwprintf(w, "%-7ld %-7ld %-7ld %-7ld %-7ld %-7ld %-7ld %-7ld %-7ld %-7ld\n",
		rusage.ru_minflt, rusage.ru_majflt, rusage.ru_nswap,
		rusage.ru_inblock, rusage.ru_oublock,
		rusage.ru_msgsnd, rusage.ru_msgrcv, rusage.ru_nsignals,
		rusage.ru_nvcsw, rusage.ru_nivcsw);

	waitnl(w);
	closeiwin(w);
}

char *
strtime(struct timeval *t)
{
	char fill = 0;
	static char buf[20];
	char *p = buf;

	if (t->tv_sec > 60*60) {
		(void) snprintf(p, sizeof(buf), "%ld:", (long int)t->tv_sec / (60*60));
		while (*p++)
			;
		p--;
		t->tv_sec %= 60*60;
		fill++;
	}
	if (t->tv_sec > 60) {
		(void) snprintf(p, sizeof(buf), fill ? "%02ld:" : "%ld:", t->tv_sec / 60);
		while (*p++)
			;
		p--;
		t->tv_sec %= 60;
		fill++;
	}
	(void) snprintf(p, sizeof(buf), fill ? "%02ld.%02ld" : "%ld.%02ld",
		t->tv_sec, t->tv_usec / 10000);
	return buf;
}

/* ARGSUSED0 */
void
l_list(winvars_t *winvars, value_t *v __unused, value_t *a __unused)
{
	ww_t *w, *wp;
	int i;
	int n;

	for (n = 0, i = 0; i < MAX_NUM_WINDOWS; i++)
		if (winvars->window[i] != 0)
			n++;
	if (n == 0) {
		error("No windows.");
		return;
	}
	if ((w = openiwin(winvars, n + 2, "Windows")) == 0) {
		error("Can't open listing window: %s.", wwerror(winvars));
		return;
	}
	for (i = 0; i < MAX_NUM_WINDOWS; i++) {
		if ((wp = winvars->window[i]) == 0)
			continue;
		wwprintf(w, "%c %c %-13s %-.*s\n",
			wp == winvars->selwin ? '+' : (wp == winvars->lastselwin ? '-' : ' '),
			i + '1',
			wp->ww_state == WWS_HASPROC ? "" : "(No process)",
			winvars->wwncol - 20,
			wp->ww_label ? wp->ww_label : "(No label)");
	}
	waitnl(w);
	closeiwin(w);
}

/* ARGSUSED0 */
void
l_variable(winvars_t *winvars, value_t *v __unused, value_t *a __unused)
{
	ww_t *w;

	if ((w = openiwin(get_winvars(), winvars->wwnrow - 3, "Variables")) == 0) {
		error("Can't open variable window: %s.", wwerror(winvars));
		return;
	}
	if (var_walk(printvar, (void *)w) >= 0)
		waitnl(w);
	closeiwin(w);
}

int
printvar(void *vw, variable_t *r)
{
	ww_t *w = vw;
	if (more(w, 0) == 2)
		return -1;
	wwprintf(w, "%16s    ", r->r_name);
	switch (r->r_val.v_type) {
	case V_STR:
		wwprintf(w, "%s\n", r->r_val.v_str);
		break;
	case V_NUM:
		wwprintf(w, "%d\n", r->r_val.v_num);
		break;
	case V_ERR:
		wwprintf(w, "ERROR\n");
		break;
	}
	return 0;
}

struct lcmd_arg arg_def_shell[] = {
	{ "",	0,		ARG_ANY|ARG_LIST },
	{ NULL,	0,		0 }
};

void
l_def_shell(winvars_t *winvars, value_t *v, value_t *a)
{
	char **pp;
	value_t *vp;

	if (a->v_type == V_ERR) {
		if ((v->v_str = strdup(winvars->default_shellfile)) != 0)
			v->v_type = V_STR;
		return;
	}
	if ((v->v_str = winvars->default_shellfile) != NULL) {
		v->v_type = V_STR;
		for (pp = winvars->default_shell + 1; *pp; pp++) {
			free(*pp);
			*pp = 0;
		}
	}
	for (pp = winvars->default_shell, vp = a;
	     vp->v_type != V_ERR &&
	     pp < &winvars->default_shell[MAX_SHELL_ARGC - 1];
	     pp++, vp++) {
		if (vp->v_type == V_STR) {
			*pp = strdup(vp->v_str);
		} else {
			 char	*cp;
			 (void) asprintf(&cp, "%d", vp->v_num);
			 *pp = cp;
		}
		if (*pp == NULL) {
			/* just leave default_shell[] the way it is */
			p_memerror();
			break;
		}
	}
	if ((winvars->default_shellfile = *winvars->default_shell) != NULL) {
		*winvars->default_shell = strrchr(winvars->default_shellfile, '/');
		if (*winvars->default_shell == NULL) {
			*winvars->default_shell = winvars->default_shellfile;
		} else {
			(*winvars->default_shell)++;
		}
	}
}

struct lcmd_arg arg_alias[] = {
	{ "",	0,		ARG_STR },
	{ "",	0,		ARG_STR|ARG_LIST },
	{ NULL,	0,		0 }
};

void
l_alias(winvars_t *winvars, value_t *v, value_t *a)
{
	if (a->v_type == V_ERR) {
		ww_t *w;

		if ((w = openiwin(winvars, winvars->wwnrow - 3, "Aliases")) == 0) {
			error("Can't open alias window: %s.", wwerror(winvars));
			return;
		}
		if (alias_walk(printalias, (void *)w) >= 0)
			waitnl(w);
		closeiwin(w);
	} else {
		variable_t *ap = 0;

		if ((ap = alias_lookup(a->v_str)) != NULL) {
			if ((v->v_str = strdup(ap->r_val.v_str)) == NULL) {
				p_memerror();
				return;
			}
			v->v_type = V_STR;
		}
		if (a[1].v_type == V_STR) {
			value_t *vp;
			char *p, *q;
			char *str;
			int n;

			for (n = 0, vp = a + 1; vp->v_type != V_ERR; vp++, n++)
				for (p = vp->v_str; *p; p++, n++)
					;
			if ((str = malloc((unsigned)n)) == NULL) {
				p_memerror();
				return;
			}
			for (q = str, vp = a + 1; vp->v_type != V_ERR;
			     vp++, q[-1] = ' ')
				for (p = vp->v_str; (*q++ = *p++);)
					;
			q[-1] = 0;
			if ((ap = alias_set(a[0].v_str, NULL)) == NULL) {
				p_memerror();
				free(str);
				return;
			}
			ap->r_val.v_str = str;
		}
	}
}

int
printalias(void *vw, variable_t *a)
{
	ww_t *w = vw;
	if (more(w, 0) == 2)
		return -1;
	wwprintf(w, "%16s    %s\n", a->r_name, a->r_val.v_str);
	return 0;
}

struct lcmd_arg arg_unalias[] = {
	{ "alias",	1,	ARG_STR },
	{ NULL,		0,	0 }
};

void
l_unalias(winvars_t *winvars, value_t *v, value_t *a)
{
	if (a->v_type == ARG_STR)
		v->v_num = alias_unset(a->v_str);
	v->v_type = V_NUM;
}

struct lcmd_arg arg_echo[] = {
	{ "window",	1,	ARG_NUM },
	{ "",		0,	ARG_ANY|ARG_LIST },
	{ NULL,		0,	0 }
};

/* ARGSUSED0 */
void
l_echo(winvars_t *winvars, value_t *v __unused, value_t *a)
{
	char buf[20];
	ww_t *w;

	if ((w = vtowin(a++, winvars->selwin)) == 0)
		return;
	while (a->v_type != V_ERR) {
		if (a->v_type == V_NUM) {
			(void) snprintf(buf, sizeof(buf), "%d", a->v_num);
			(void) wwwrite(winvars, w, buf, (int)strlen(buf));
		} else
			(void) wwwrite(winvars, w, a->v_str, (int)strlen(a->v_str));
		if ((++a)->v_type != V_ERR)
			(void) wwwrite(winvars, w, " ", 1);
	}
	(void) wwwrite(winvars, w, "\r\n", 2);
}
