/*	$NetBSD: ttinit.c,v 1.9 2003/08/13 15:21:07 itojun Exp $	*/

/*
 * Copyright (c) 1983, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Edward Wang at The University of California, Berkeley.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <sys/cdefs.h>

#ifndef lint
__RCSID("$NetBSD: ttinit.c,v 1.9 2003/08/13 15:21:07 itojun Exp $");
#endif /* not lint */

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <termcap.h>
#include <unistd.h>

#define TT_EXTERN
#include "ww.h"
#include "tt.h"

char PC, *BC, *UP;
short ospeed;

	/* normal frame */
short gen_frame[16] = {
	' ', '|', '-', '+',
	'|', '|', '+', '+',
	'-', '+', '-', '+',
	'+', '+', '+', '+'
};

	/* ANSI graphics frame */
#define G (WWM_GRP << WWC_MSHIFT)
short ansi_frame[16] = {
	' ',	'x'|G,	'Q'|G,	'm'|G,
	'x'|G,	'x'|G,	'l'|G,	't'|G,
	'q'|G,	'j'|G,	'q'|G,	'v'|G,
	'k'|G,	'u'|G,	'w'|G,	'n'|G
};
struct tt_str ansi_AS = {
	"\033(0", 3
};

static struct tt_str *tcap_pc;		/* pad character (default NUL) */
static struct tt_str *tcap_cm;		/* screen relative cursor motion */
static struct tt_str *tcap_im;		/* enter insert mode */
static struct tt_str *tcap_ic;		/* insert character */
static struct tt_str *tcap_IC;		/* insert n blank characters */
static struct tt_str *tcap_ip;		/* insert pad after character inserted */
static struct tt_str *tcap_ei;		/* end insert mode */
static struct tt_str *tcap_dc;		/* delete character */
static struct tt_str *tcap_DC;		/* delete n characters */
static struct tt_str *tcap_al;		/* add new blank line */
static struct tt_str *tcap_AL;		/* add n new blank lines */
static struct tt_str *tcap_dl;		/* delete line */
static struct tt_str *tcap_DL;		/* delete n lines */
static struct tt_str *tcap_ce;		/* clear to end of line */
static struct tt_str *tcap_cd;		/* clear to end of display */
static struct tt_str *tcap_cl;		/* clear screen and home cursor */
static struct tt_str *tcap_vs;		/* make cursor very visible */
static struct tt_str *tcap_ve;		/* make cursor appear normal */
static struct tt_str *tcap_ti;		/* terminal initialisation */
static struct tt_str *tcap_te;		/* terminal ending - wind up */
static struct tt_str *tcap_so;		/* begin standout mode */
static struct tt_str *tcap_se;		/* end standout mode */
static struct tt_str *tcap_us;		/* start underscore mode */
static struct tt_str *tcap_ue;		/* end underscore mode */
static struct tt_str *tcap_le;		/* move cursor left one position */
static struct tt_str *tcap_nd;		/* non destructive space - cursor right */
static struct tt_str *tcap_up;		/* upline - cursor up */
static struct tt_str *tcap_do;		/* move cursor down one line */
static struct tt_str *tcap_bc;		/* backspace if not ^H */
static struct tt_str *tcap_nl;		/* newline character if not \n */
static struct tt_str *tcap_cr;		/* carriage return */
static struct tt_str *tcap_ho;		/* home cursor */
static struct tt_str *tcap_as;		/* start alternative character set */
static struct tt_str *tcap_ae;		/* end alternative character set */
static struct tt_str *gen_XS;
static struct tt_str *gen_XE;
static struct tt_str *tcap_sf;		/* scroll text up */
static struct tt_str *tcap_SF;		/* scroll forward n lines */
static struct tt_str *tcap_sr;		/* scroll text down */
static struct tt_str *tcap_SR;		/* scroll text down n lines */
static struct tt_str *tcap_cs;		/* change scrolling region */

/* colour support */
static struct tt_str *tcap_me;		/* turn off all attributes */
static struct tt_str *tcap_op;		/* set default colour pair to the original one */
/* XXX - MD bold */

static char gen_MI;
static char gen_MS;
static char gen_AM;
static char gen_OS;
static char gen_BS;
static char gen_DA;
static char gen_DB;
static char gen_NS;
static char gen_XN;
static int tcap_co;			/* number of columns */
static int tcap_li;			/* number of lines */
static int gen_UG;
static int gen_SG;

void	gen_clear(void);
void	gen_clreol(void);
void	gen_clreos(void);
void	gen_delchar(int);
void	gen_delline(int);
void	gen_end(void);
void	gen_inschar(char);
void	gen_insline(int);
void	gen_insspace(int);
void	gen_move(int, int);
void	gen_putc(char);
void	gen_scroll_down(int);
void	gen_scroll_up(int);
void	gen_setinsert(char);
void	gen_setmodes(int);
void	gen_setscroll(int, int);
void	gen_start(void);
void	gen_write(char *, int);

static char		tt_strings[1024];		/* string buffer */
static char *tt_strp;					/* pointer for it */


static struct tt_tab tt_tab[] = {
	{ "generic",	0, tt_generic },
	{ 0,		0, 0 }
};

int
ttinit(void)
{
	int i;
	struct tt_tab *tp;
	char *p, *q;
	char *t;
	winvars_t	*winvars;

	winvars = get_winvars();

	tt_strp = tt_strings;

	/*
	 * Set output buffer size to about 1 second of output time.
	 */
	i = MIN(winvars->wwbaud/10, 512);
	if ((tt_ob = malloc((unsigned) i)) == NULL) {
		winvars->wwerrno = WWE_NOMEM;
		return -1;
	}
	tt_obp = tt_ob;
	tt_obe = tt_ob + i;

	/*
	 * Use the standard name of the terminal (i.e. the second
	 * name in termcap).
	 */
	for (p = winvars->wwtermcap; *p && *p != '|' && *p != ':'; p++)
		;
	if (*p == '|')
		p++;
	for (q = p; *q && *q != '|' && *q != ':'; q++)
		;
	if (q != p && (t = malloc((unsigned) (q - p + 1))) != NULL) {
		winvars->wwterm = t;
		while (p < q)
			*t++ = *p++;
		*t = 0;
	}
	for (tp = tt_tab; tp->tt_name != 0; tp++)
		if (strncmp(tp->tt_name, winvars->wwterm, (unsigned)tp->tt_len) == 0)
			break;
	if (tp->tt_name == 0) {
		winvars->wwerrno = WWE_BADTERM;
		return -1;
	}
	if ((*tp->tt_func)() < 0) {
		winvars->wwerrno = WWE_CANTDO;
		return -1;
	}
	if (wwgetttysize(winvars, 0, &tt.tt_nrow, &tt.tt_ncol) < 0)
		return -1;
	tt.tt_scroll_top = 0;
	tt.tt_scroll_bot = tt.tt_nrow - 1;
	return 0;
}

void
gen_setinsert(char new)
{
	if (new) {
		if (tcap_im) {
			TTXPUTS(tcap_im);
		}
	} else if (tcap_ei) {
		TTXPUTS(tcap_ei);
	}
	tt.tt_insert = new;
}

void
gen_setmodes(int new)
{
	int diff;

	diff = new ^ tt.tt_modes;
	if (diff & WWM_REV) {
		if (new & WWM_REV) {
			if (tcap_so) {
				TTXPUTS(tcap_so);
			}
		} else if (tcap_se) {
			TTXPUTS(tcap_se);
			if (strcmp(tcap_se->ts_str, tcap_ue->ts_str) == 0 &&
				   tcap_ue && tcap_us && new & WWM_UL) {
				TTXPUTS(tcap_us);
			}
		}
	}
	if (diff & WWM_UL) {
		if (new & WWM_UL) {
			if (tcap_us) {
				TTXPUTS(tcap_us);
			}
		} else if (tcap_ue) {
			TTXPUTS(tcap_ue);
			if (strcmp(tcap_ue->ts_str, tcap_se->ts_str) == 0 &&
				   tcap_se && tcap_so && new & WWM_REV) {
				TTXPUTS(tcap_so);
			}
		}
	}
	if (diff & WWM_GRP) {
		if (new & WWM_GRP) {
			if (tcap_as) {
				TTXPUTS(tcap_as);
			}
		} else if (tcap_ae) {
			TTXPUTS(tcap_ae);
		}
	}
	if (diff & WWM_USR) {
		if (new & WWM_USR) {
			if (gen_XS) {
				TTXPUTS(gen_XS);
			}
		} else if (gen_XE) {
			TTXPUTS(gen_XE);
		}
	}
	tt.tt_modes = new;
}

void
gen_insline(int n)
{
	if (tt.tt_modes) {			/* for concept 100 */
		gen_setmodes(0);
	}
	if (tcap_AL) {
		ttpgoto(tcap_AL, 0, n, tcap_li - tt.tt_row);
	} else {
		while (--n >= 0) {
			TTTPUTS(tcap_al, tcap_li - tt.tt_row);
		}
	}
}

void
gen_delline(int n)
{
	if (tt.tt_modes) {			/* for concept 100 */
		gen_setmodes(0);
	}
	if (tcap_DL) {
		ttpgoto(tcap_DL, 0, n, tcap_li - tt.tt_row);
	} else {
		while (--n >= 0) {
			TTTPUTS(tcap_dl, tcap_li - tt.tt_row);
		}
	}
}

void
gen_putc(char c)
{
	if (tt.tt_insert) {
		gen_setinsert(0);
	}
	if (tt.tt_nmodes != tt.tt_modes) {
		gen_setmodes(tt.tt_nmodes);
	}
	/* LINTED */
	TTPUTC(c);
	if (++tt.tt_col == tcap_co) {
		if (gen_XN) {
			tt.tt_col = tt.tt_row = -10;
		} else if (gen_AM) {
			tt.tt_col = 0, tt.tt_row++;
		} else {
			tt.tt_col--;
		}
	}
}

void
gen_write(char *p, int n)
{
	if (tt.tt_insert) {
		gen_setinsert(0);
	}
	if (tt.tt_nmodes != tt.tt_modes) {
		gen_setmodes(tt.tt_nmodes);
	}
	ttwrite(p, n);
	tt.tt_col += n;
	if (tt.tt_col == tcap_co) {
		if (gen_XN) {
			tt.tt_col = tt.tt_row = -10;
		} else if (gen_AM) {
			tt.tt_col = 0, tt.tt_row++;
		} else {
			tt.tt_col--;
		}
	}
}

void
gen_move(int row, int col)
{
	if (tt.tt_row == row && tt.tt_col == col) {
		return;
	}
	if (!gen_MI && tt.tt_insert) {
		gen_setinsert(0);
	}
	if (!gen_MS && tt.tt_modes) {
		gen_setmodes(0);
	}
	if (row < tt.tt_scroll_top || row > tt.tt_scroll_bot) {
		gen_setscroll(0, tt.tt_nrow - 1);
	}
	if (tt.tt_row == row) {
		if (col == 0) {
			TTXPUTS(tcap_cr);
			goto out;
		}
		if (tt.tt_col == col - 1) {
			if (tcap_nd) {
				TTXPUTS(tcap_nd);
				goto out;
			}
		} else if (tt.tt_col == col + 1) {
			if (tcap_le) {
				TTXPUTS(tcap_le);
				goto out;
			}
		}
	}
	if (tt.tt_col == col) {
		if (tt.tt_row == row + 1) {
			if (tcap_up) {
				TTXPUTS(tcap_up);
				goto out;
			}
		} else if (tt.tt_row == row - 1) {
			TTXPUTS(tcap_do);
			goto out;
		}
	}
	if (tcap_ho && col == 0 && row == 0) {
		TTXPUTS(tcap_ho);
		goto out;
	}
	tttgoto(tcap_cm, col, row);
out:
	tt.tt_col = col;
	tt.tt_row = row;
}

void
gen_start(void)
{
	if (tcap_vs) {
		TTXPUTS(tcap_vs);
	}
	if (tcap_ti) {
		TTXPUTS(tcap_ti);
	}
	TTXPUTS(tcap_cl);
	tt.tt_col = tt.tt_row = 0;
	tt.tt_insert = 0;
	tt.tt_nmodes = tt.tt_modes = 0;
}

void
gen_end(void)
{
	if (tt.tt_insert) {
		gen_setinsert(0);
	}
	if (tcap_te) {
		TTXPUTS(tcap_te);
	}
	if (tcap_ve) {
		TTXPUTS(tcap_ve);
	}
}

void
gen_clreol(void)
{
	if (tt.tt_modes) {			/* for concept 100 */
		gen_setmodes(0);
	}
	TTTPUTS(tcap_ce, tcap_co - tt.tt_col);
}

void
gen_clreos(void)
{
	if (tt.tt_modes) {			/* for concept 100 */
		gen_setmodes(0);
	}
	TTTPUTS(tcap_cd, tcap_li - tt.tt_row);
}

void
gen_clear(void)
{
	if (tt.tt_modes) {			/* for concept 100 */
		gen_setmodes(0);
	}
	TTXPUTS(tcap_cl);
}

void
gen_inschar(char c)
{
	if (!tt.tt_insert) {
		gen_setinsert(1);
	}
	if (tt.tt_nmodes != tt.tt_modes) {
		gen_setmodes(tt.tt_nmodes);
	}
	if (tcap_ic) {
		TTTPUTS(tcap_ic, tcap_co - tt.tt_col);
	}
	/* LINTED */
	TTPUTC(c);
	if (tcap_ip) {
		TTTPUTS(tcap_ip, tcap_co - tt.tt_col);
	}
	if (++tt.tt_col == tcap_co) {
		if (gen_XN) {
			tt.tt_col = tt.tt_row = -10;
		} else if (gen_AM) {
			tt.tt_col = 0, tt.tt_row++;
		} else {
			tt.tt_col--;
		}
	}
}

void
gen_insspace(int n)
{
	if (tcap_IC) {
		ttpgoto(tcap_IC, 0, n, tcap_co - tt.tt_col);
	} else {
		while (--n >= 0) {
			TTTPUTS(tcap_ic, tcap_co - tt.tt_col);
		}
	}
}

void
gen_delchar(int n)
{
	if (tcap_DC) {
		ttpgoto(tcap_DC, 0, n, tcap_co - tt.tt_col);
	} else {
		while (--n >= 0) {
			TTTPUTS(tcap_dc, tcap_co - tt.tt_col);
		}
	}
}

void
gen_scroll_down(int n)
{
	gen_move(tt.tt_scroll_bot, 0);
	if (tcap_SF) {
		ttpgoto(tcap_SF, 0, n, n);
	} else {
		while (--n >= 0) {
			TTXPUTS(tcap_sf);
		}
	}
}

void
gen_scroll_up(int n)
{
	gen_move(tt.tt_scroll_top, 0);
	if (tcap_SR) {
		ttpgoto(tcap_SR, 0, n, n);
	} else {
		while (--n >= 0) {
			TTXPUTS(tcap_sr);
		}
	}
}

void
gen_setscroll(int top, int bot)
{
	tttgoto(tcap_cs, bot, top);
	tt.tt_scroll_top = top;
	tt.tt_scroll_bot = bot;
	tt.tt_row = tt.tt_col = -10;
}

int
tt_generic(void)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	tcap_pc = tttgetstr("pc");
	PC = tcap_pc ? *tcap_pc->ts_str : 0;
	ospeed = winvars->wwospeed;

	tcap_cm = ttxgetstr("cm");		/* may not work */
	tcap_im = ttxgetstr("im");
	tcap_ic = tttgetstr("ic");
	tcap_IC = tttgetstr("IC");
	tcap_ip = tttgetstr("ip");
	tcap_ei = ttxgetstr("ei");
	tcap_dc = tttgetstr("dc");
	tcap_DC = tttgetstr("DC");
	tcap_al = tttgetstr("al");
	tcap_AL = tttgetstr("AL");
	tcap_dl = tttgetstr("dl");
	tcap_DL = tttgetstr("DL");
	tcap_ce = tttgetstr("ce");
	tcap_cd = tttgetstr("cd");
	tcap_cl = ttxgetstr("cl");
	tcap_vs = ttxgetstr("vs");
	tcap_ve = ttxgetstr("ve");
	tcap_ti = ttxgetstr("ti");
	tcap_te = ttxgetstr("te");
	tcap_so = ttxgetstr("so");
	tcap_se = ttxgetstr("se");
	tcap_us = ttxgetstr("us");
	tcap_ue = ttxgetstr("ue");
	tcap_le = ttxgetstr("le");
	tcap_nd = ttxgetstr("nd");
	tcap_up = ttxgetstr("up");
	tcap_do = ttxgetstr("do");
	tcap_bc = ttxgetstr("bc");
	tcap_nl = ttxgetstr("nl");
	tcap_cr = ttxgetstr("cr");
	tcap_ho = ttxgetstr("ho");
	tcap_as = ttxgetstr("as");
	tcap_ae = ttxgetstr("ae");
	gen_XS = ttxgetstr("XS");
	gen_XE = ttxgetstr("XE");
	tcap_sf = ttxgetstr("sf");
	tcap_SF = ttxgetstr("SF");
	tcap_sr = ttxgetstr("sr");
	tcap_SR = ttxgetstr("SR");
	tcap_cs = ttxgetstr("cs");
	gen_MI = tgetflag("mi");
	gen_MS = tgetflag("ms");
	gen_AM = tgetflag("am");
	gen_OS = tgetflag("os");
	gen_BS = tgetflag("bs");
	gen_DA = tgetflag("da");
	gen_DB = tgetflag("db");
	gen_NS = tgetflag("ns");
	gen_XN = tgetflag("xn");
	tcap_co = tgetnum("co");
	tcap_li = tgetnum("li");
	gen_UG = tgetnum("ug");
	gen_SG = tgetnum("sg");
	tcap_me = ttxgetstr("ME");
	tcap_op = ttxgetstr("OP");
	if (tcap_cl == 0 || gen_OS || tcap_cm == 0)
		return -1;

	/*
	 * Deal with obsolete termcap fields.
	 */
	if (tcap_le == 0) {
		if (tcap_bc) {
			tcap_le = tcap_bc;
		} else if (gen_BS) {
			static struct tt_str bc = { "\b", 1 };
			tcap_bc = &bc;
		}
	}
	if (tcap_nl == NULL) {
		static struct tt_str nl = { "\n", 1 };
		tcap_nl = &nl;
	}
	if (tcap_do == NULL) {
		tcap_do = tcap_nl;
	}
	if (tcap_cr == NULL) {
		static struct tt_str cr = { "\r", 1 };
		tcap_cr = &cr;
	}
	/*
	 * Most terminal will scroll with "nl", but very few specify "sf".
	 * We shouldn't use "do" here.
	 */
	if (tcap_sf == NULL && !gen_NS) {
		tcap_sf = tcap_nl;
	}
	BC = (tcap_le) ? __UNCONST(tcap_le->ts_str) : NULL;
	UP = (tcap_up) ? __UNCONST(tcap_up->ts_str) : NULL;
	/*
	 * Fix up display attributes that we can't handle, or don't
	 * really exist.
	 */
	if (gen_SG > 0) {
		tcap_so = 0;
	}
	if (gen_UG > 0 ||
	    (tcap_us && tcap_so && ttstrcmp(tcap_us, tcap_so) == 0)) {
		tcap_us = NULL;
	}
	if (tcap_im && tcap_im->ts_n == 0) {
		free(tcap_im);
		tcap_im = NULL;
	}
	if (tcap_ei && tcap_ei->ts_n == 0) {
		free(tcap_ei);
		tcap_ei = NULL;
	}
	if (tcap_ic && tcap_ic->ts_n == 0) {
		free(tcap_ic);
		tcap_ic = NULL;
	}
	if (tcap_im) {
		tt.tt_inschar = gen_inschar;
	} else if (tcap_ic) {
		tt.tt_insspace = gen_insspace;
	}
	if (tcap_dc) {
		tt.tt_delchar = gen_delchar;
	}
	if (tcap_al) {
		tt.tt_insline = gen_insline;
	}
	if (tcap_dl) {
		tt.tt_delline = gen_delline;
	}
	if (tcap_ce) {
		tt.tt_clreol = gen_clreol;
	}
	if (tcap_cd) {
		tt.tt_clreos = gen_clreos;
	}
	if (tcap_sf) {
		tt.tt_scroll_down = gen_scroll_down;
	}
	/*
	 * Don't allow scroll_up if da or db but not cs.
	 * See comment in wwscroll.c.
	 */
	if (tcap_sr && (tcap_cs || (!gen_DA && !gen_DB))) {
		tt.tt_scroll_up = gen_scroll_up;
	}
	if (tcap_cs) {
		tt.tt_setscroll = gen_setscroll;
	}
	if (tcap_so) {
		tt.tt_availmodes |= WWM_REV;
	}
	if (tcap_us) {
		tt.tt_availmodes |= WWM_UL;
	}
	if (tcap_as) {
		tt.tt_availmodes |= WWM_GRP;
	}
	if (gen_XS) {
		tt.tt_availmodes |= WWM_USR;
	}
	tt.tt_wrap = gen_AM;
	tt.tt_retain = gen_DB;
	tt.tt_ncol = tcap_co;
	tt.tt_nrow = tcap_li;
	tt.tt_start = gen_start;
	tt.tt_end = gen_end;
	tt.tt_write = gen_write;
	tt.tt_putc = gen_putc;
	tt.tt_move = gen_move;
	tt.tt_clear = gen_clear;
	tt.tt_setmodes = gen_setmodes;
	tt.tt_frame = (tcap_as && ttstrcmp(tcap_as, &ansi_AS) == 0) ?
		ansi_frame : gen_frame;
	return 0;
}

/*
 * Buffered output package.
 * We need this because stdio fails on non-blocking writes.
 */

void
ttflush(void)
{
	char *p;
	winvars_t	*winvars;
	int n = tt_obp - tt_ob;

	if (n == 0)
		return;
	if (tt.tt_checksum)
		(*tt.tt_checksum)(tt_ob, n);
	if (tt.tt_flush) {
		(*tt.tt_flush)();
		return;
	}
	winvars = get_winvars();
	winvars->wwnflush++;
	for (p = tt_ob; p < tt_obp;) {
		winvars->wwnwr++;
		n = write(1, p, (unsigned)(tt_obp - p));
		if (n < 0) {
			winvars->wwnwre++;
			if (errno != EWOULDBLOCK) {
				/* can't deal with this */
				p = tt_obp;
			}
		} else if (n == 0) {
			/* what to do? */
			winvars->wwnwrz++;
		} else {
			winvars->wwnwrc += n;
			p += n;
		}
	}
	tt_obp = tt_ob;
}

void
ttputs(char *s)
{
	while (*s) {
		/* LINTED */
		TTPUTC(*s++);
	}
}

void
ttwrite(const char *s, int n)
{
	winvars_t	*winvars;

	winvars = get_winvars();
	switch (n) {
	case 0:
		break;
	case 1:
		/* LINTED */
		TTPUTC(*s);
		break;
	case 2:
		if (tt_obe - tt_obp < 2)
			ttflush();
		*tt_obp++ = *s++;
		*tt_obp++ = *s;
		break;
	case 3:
		if (tt_obe - tt_obp < 3)
			ttflush();
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s;
		break;
	case 4:
		if (tt_obe - tt_obp < 4)
			ttflush();
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s;
		break;
	case 5:
		if (tt_obe - tt_obp < 5)
			ttflush();
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s++;
		*tt_obp++ = *s;
		break;
	default:
		while (n > 0) {
			int m;

			while ((m = tt_obe - tt_obp) == 0)
				ttflush();
			if ((m = tt_obe - tt_obp) > n)
				m = n;
			memmove(tt_obp, s, (unsigned)m);
			tt_obp += m;
			s += m;
			n -= m;
		}
	}
}

int
tttputc(int c)
{
	/* LINTED */
	TTPUTC(c);
	return 0;
}

int
ttxputc(int c)
{
	*tt_strp++ = c;
	return 0;
}

struct tt_str *
tttgetstr(const char *str)
{
	struct tt_str *s;

	if ((str = tgetstr(str, &tt_strp)) == 0)
		return 0;
	if ((s = (struct tt_str *) malloc(sizeof *s)) == 0)
		return 0;
	s->ts_str = str;
	s->ts_n = tt_strp - s->ts_str - 1;
	return s;
}

struct tt_str *
ttxgetstr(const char *str)
{
	struct tt_str *s;
	char buf[100];
	char *bufp = buf;

	if (tgetstr(str, &bufp) == 0)
		return 0;
	if ((s = (struct tt_str *) malloc(sizeof *s)) == 0)
		return 0;
	s->ts_str = tt_strp;
	tputs(buf, 1, ttxputc);
	s->ts_n = tt_strp - s->ts_str;
	*tt_strp++ = 0;
	return s;
}

void
tttgoto(struct tt_str *s, int col, int row)
{
	const char *p = s->ts_str;

	ttputs(tgoto(p, col, row));
	for (p += s->ts_n; *--p == 0;) {
		/* LINTED */
		TTPUTC(0);
	}
}

void
ttpgoto(struct tt_str *s, int col, int row, int n)
{

	tputs(tgoto(s->ts_str, col, row), n, tttputc);
}

int
ttstrcmp(struct tt_str *a, struct tt_str *b)
{
	int n, r;

	n = a->ts_n - b->ts_n;
	if ((r = memcmp(a->ts_str, b->ts_str,
			(unsigned)((n < 0) ? a->ts_n : b->ts_n))) != 0)
		return r;
	return n;
}
