/* Example of Outcome used with error code enums
(C) 2017-2019 Niall Douglas <http://www.nedproductions.biz/> (149 commits)


Boost Software License - Version 1.0 - August 17th, 2003

Permission is hereby granted, free of charge, to any person or organization
obtaining a copy of the software and accompanying documentation covered by
this license (the "Software") to use, reproduce, display, distribute,
execute, and transmit the Software, and to prepare derivative works of the
Software, and to permit third-parties to whom the Software is furnished to
do so, all subject to the following:

The copyright notices in the Software and this entire statement, including
the above license grant, this restriction and the following disclaimer,
must be included in all copies of the Software, in whole or in part, and
all derivative works of the Software, unless such copies or derivative
works are solely in the form of machine-executable object code generated by
a source language processor.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
DEALINGS IN THE SOFTWARE.
*/

#include "../../../include/boost/outcome.hpp"

#include <iostream>

//! [declaration]
struct udt
{
  int a{0};
  explicit udt(int _a)
      : a(_a)
  {
  }
  udt() = default;
  int operator*() const { return a; }
};
enum class err
{
  success,
  failure1,
  failure2
};

// Tell the standard library that enum err is an error code enum
// by specialising the is_error_code_enum trait. See
// http://en.cppreference.com/w/cpp/error/error_code/is_error_code_enum
namespace std
{
  template <> struct is_error_code_enum<err> : std::true_type
  {
  };
}

// We also must declare a free function make_error_code. This is
// discovered via ADL by the standard library. See
// http://en.cppreference.com/w/cpp/error/errc/make_error_code
inline std::error_code make_error_code(err c)
{
  // We need to inherit from std::error_category to define
  // an error code domain with the standard library for
  // our strongly typed enum. See
  // http://en.cppreference.com/w/cpp/error/error_category
  static struct err_category : std::error_category
  {
    virtual const char *name() const noexcept override final { return "err_category"; };
    virtual std::string message(int c) const override final
    {
      switch(static_cast<err>(c))
      {
      case err::success:
        return "err::success";
      case err::failure1:
        return "err::failure1";
      case err::failure2:
        return "err::failure2";
      }
      return "unknown";
    }
  } category;
  return std::error_code(static_cast<int>(c), category);
}
//! [declaration]

using namespace BOOST_OUTCOME_V2_NAMESPACE;

int main()
{
  //! [usage]
  result<udt, err> res(err::failure1);

  // What happens here? What exception type is thrown?
  try
  {
    std::cout << *res.value() << std::endl;
  }
  catch(const std::exception &e)
  {
    std::cerr << "Exception thrown was " << e.what() << std::endl;
  }
  //! [usage]
  return 0;
}

void test()
{
  //! [usage2]
  result<udt /*, std::error_code */> res(err::failure1);

  // What happens here? What exception type is thrown?
  try
  {
    std::cout << *res.value() << std::endl;
  }
  catch(const std::exception &e)
  {
    // Prints "Exception thrown was failure1", exactly the same as before
    std::cerr << "Exception thrown was " << e.what() << std::endl;
  }
  //! [usage2]
}

//! [usage3]
result<udt> boo()
{
  return err::failure1;
}
result<udt> foo()
{
  BOOST_OUTCOME_TRY(v, (boo()));
  return udt{5};  // emplace construct udt with 5
}
//! [usage3]
