/* Copyright (C) 2007 and 2009 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

*/

#ifndef CGU_GERROR_HANDLE_H
#define CGU_GERROR_HANDLE_H

#include <glib.h>
#include <c++-gtk-utils/shared_handle.h>
#include <c++-gtk-utils/cgu_config.h>

/**
 * @addtogroup handles handles and smart pointers
 */

namespace Cgu {

/**
 * @class GerrorFree gerror_handle.h c++-gtk-utils/gerror_handle.h
 * @brief A deleter functor for use as the second (Dealloc) template
 * parameter of the SharedHandle, SharedLockHandle or ScopedHandle
 * template classes, which calls glib's g_error_free().
 * @ingroup handles
 * @details This functor enables those classes to manage GError
 * objects in memory which require to be freed with g_error_free().
 * It is used in the typedefs @ref GerrorSharedHandleAnchor
 * "GerrorSharedHandle" and @ref GerrorScopedHandleAnchor
 * "GcharScopedHandle".
 */
class GerrorFree {
public:
  void operator()(const GError* obj_p) noexcept {
    if (obj_p) {
      g_error_free(const_cast<GError*>(obj_p));
    }
  }
};


/**
 * @typedef GerrorScopedHandle.
 * @brief A handle comprising a typed instance of the ScopedHandle class for managing GError objects.
 * @anchor GerrorScopedHandleAnchor
 * @ingroup handles
 * \#include <c++-gtk-utils/gerror_handle.h>
 */
typedef ScopedHandle<GError*, GerrorFree> GerrorScopedHandle;

/**
 * @typedef GerrorSharedHandle.
 * @brief A handle comprising a typed instance of the SharedHandle class for managing GError objects.
 * @anchor GerrorSharedHandleAnchor
 * @ingroup handles
 * \#include <c++-gtk-utils/gerror_handle.h>
 */
typedef SharedHandle<GError*, GerrorFree> GerrorSharedHandle;

} // namespace Cgu

#endif
