# 18 "ast/lexer.mll"
 
open Import
open Lexing
open Misc
open Parser

type error =
  | Illegal_character of char
  | Illegal_escape of string
  | Unterminated_comment of Location.t
  | Unterminated_string
  | Unterminated_string_in_comment of Location.t * Location.t
  | Keyword_as_label of string
  | Invalid_literal of string
  | Invalid_directive of string * string option
;;

exception Error of error * Location.t;;

(* The table of keywords *)

let keyword_table =
  create_hashtable 149 [
    "and", AND;
    "as", AS;
    "assert", ASSERT;
    "begin", BEGIN;
    "class", CLASS;
    "constraint", CONSTRAINT;
    "do", DO;
    "done", DONE;
    "downto", DOWNTO;
    "else", ELSE;
    "end", END;
    "exception", EXCEPTION;
    "external", EXTERNAL;
    "false", FALSE;
    "for", FOR;
    "fun", FUN;
    "function", FUNCTION;
    "functor", FUNCTOR;
    "if", IF;
    "in", IN;
    "include", INCLUDE;
    "inherit", INHERIT;
    "initializer", INITIALIZER;
    "lazy", LAZY;
    "let", LET;
    "match", MATCH;
    "method", METHOD;
    "module", MODULE;
    "mutable", MUTABLE;
    "new", NEW;
    "nonrec", NONREC;
    "object", OBJECT;
    "of", OF;
    "open", OPEN;
    "or", OR;
(*  "parser", PARSER; *)
    "private", PRIVATE;
    "rec", REC;
    "sig", SIG;
    "struct", STRUCT;
    "then", THEN;
    "to", TO;
    "true", TRUE;
    "try", TRY;
    "type", TYPE;
    "val", VAL;
    "virtual", VIRTUAL;
    "when", WHEN;
    "while", WHILE;
    "with", WITH;

    "lor", INFIXOP3("lor"); (* Should be INFIXOP2 *)
    "lxor", INFIXOP3("lxor"); (* Should be INFIXOP2 *)
    "mod", INFIXOP3("mod");
    "land", INFIXOP3("land");
    "lsl", INFIXOP4("lsl");
    "lsr", INFIXOP4("lsr");
    "asr", INFIXOP4("asr")
]

module Buffer = struct (* Imported for compatibility with 4.04.x *)
  [@@@warning "-32"]
  let add_utf_8_uchar b u = match Uchar.to_int u with
    | u when u < 0 -> assert false
    | u when u <= 0x007F ->
      Buffer.add_char b (Char.unsafe_chr u)
    | u when u <= 0x07FF ->
      Buffer.add_char b (Char.unsafe_chr (0xC0 lor (u lsr 6)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor (u land 0x3F)));
    | u when u <= 0xFFFF ->
      Buffer.add_char b (Char.unsafe_chr (0xE0 lor (u lsr 12)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor ((u lsr 6) land 0x3F)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor (u land 0x3F)));
    | u when u <= 0x10FFFF ->
      Buffer.add_char b (Char.unsafe_chr (0xF0 lor (u lsr 18)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor ((u lsr 12) land 0x3F)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor ((u lsr 6) land 0x3F)));
      Buffer.add_char b (Char.unsafe_chr (0x80 lor (u land 0x3F)));
    | _ -> assert false
  include Buffer
end

(* To buffer string literals *)

let string_buffer = Buffer.create 256
let reset_string_buffer () = Buffer.reset string_buffer
let get_stored_string () = Buffer.contents string_buffer

let store_string_char c = Buffer.add_char string_buffer c
let store_string_utf_8_uchar u = Buffer.add_utf_8_uchar string_buffer u
let store_string s = Buffer.add_string string_buffer s
let store_lexeme lexbuf = store_string (Lexing.lexeme lexbuf)

(* To store the position of the beginning of a string and comment *)
let string_start_loc = ref Location.none;;
let comment_start_loc = ref [];;
let in_comment () = !comment_start_loc <> [];;
let is_in_string = ref false
let in_string () = !is_in_string
let print_warnings = ref true

(* Escaped chars are interpreted in strings unless they are in comments. *)
let store_escaped_char lexbuf c =
  if in_comment () then store_lexeme lexbuf else store_string_char c

let store_escaped_uchar lexbuf u =
  if in_comment () then store_lexeme lexbuf else store_string_utf_8_uchar u

let with_comment_buffer comment lexbuf =
  let start_loc = Location.curr lexbuf  in
  comment_start_loc := [start_loc];
  reset_string_buffer ();
  let end_loc = comment lexbuf in
  let s = get_stored_string () in
  reset_string_buffer ();
  let loc = { start_loc with Location.loc_end = end_loc.Location.loc_end } in
  s, loc

(* To translate escape sequences *)

let hex_digit_value d = (* assert (d in '0'..'9' 'a'..'f' 'A'..'F') *)
  let d = Char.code d in
  if d >= 97 then d - 87 else
  if d >= 65 then d - 55 else
  d - 48

let hex_num_value lexbuf ~first ~last =
  let rec loop acc i = match i > last with
  | true -> acc
  | false ->
      let value = hex_digit_value (Lexing.lexeme_char lexbuf i) in
      loop (16 * acc + value) (i + 1)
  in
  loop 0 first

let char_for_backslash = function
  | 'n' -> '\010'
  | 'r' -> '\013'
  | 'b' -> '\008'
  | 't' -> '\009'
  | c   -> c

let char_for_decimal_code lexbuf i =
  let c = 100 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           10 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
                (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  if (c < 0 || c > 255) then
    if in_comment ()
    then 'x'
    else raise (Error(Illegal_escape (Lexing.lexeme lexbuf),
                      Location.curr lexbuf))
  else Char.chr c

let char_for_octal_code lexbuf i =
  let c = 64 * (Char.code(Lexing.lexeme_char lexbuf i) - 48) +
           8 * (Char.code(Lexing.lexeme_char lexbuf (i+1)) - 48) +
               (Char.code(Lexing.lexeme_char lexbuf (i+2)) - 48) in
  Char.chr c

let char_for_hexadecimal_code lexbuf i =
  let byte = hex_num_value lexbuf ~first:i ~last:(i+1) in
  Char.chr byte

let uchar_for_uchar_escape lexbuf =
  let err e =
    raise
      (Error (Illegal_escape (Lexing.lexeme lexbuf ^ e), Location.curr lexbuf))
  in
  let len = Lexing.lexeme_end lexbuf - Lexing.lexeme_start lexbuf in
  let first = 3 (* skip opening \u{ *) in
  let last = len - 2 (* skip closing } *) in
  let digit_count = last - first + 1 in
  match digit_count > 6 with
  | true -> err ", too many digits, expected 1 to 6 hexadecimal digits"
  | false ->
      let cp = hex_num_value lexbuf ~first ~last in
      if Uchar.is_valid cp then Uchar.unsafe_of_int cp else
      err (", " ^ Printf.sprintf "%X" cp ^ " is not a Unicode scalar value")

(* recover the name from a LABEL or OPTLABEL token *)

let get_label_name lexbuf =
  let s = Lexing.lexeme lexbuf in
  let name = String.sub s 1 (String.length s - 2) in
  if Hashtbl.mem keyword_table name then
    raise (Error(Keyword_as_label name, Location.curr lexbuf));
  name
;;

(* Update the current location with file name and line number. *)

let update_loc lexbuf file line absolute chars =
  let pos = lexbuf.lex_curr_p in
  let new_file = match file with
                 | None -> pos.pos_fname
                 | Some s -> s
  in
  lexbuf.lex_curr_p <- { pos with
    pos_fname = new_file;
    pos_lnum = if absolute then line else pos.pos_lnum + line;
    pos_bol = pos.pos_cnum - chars;
  }
;;

let preprocessor = ref None

let escaped_newlines = ref false

(* Warn about Latin-1 characters used in idents *)

let warn_latin1 lexbuf =
  Location.deprecated (Location.curr lexbuf)"ISO-Latin1 characters in identifiers"

let handle_docstrings = ref true
let comment_list = ref []

let add_comment com =
  comment_list := com :: !comment_list

let add_docstring_comment ds =
  let com =
    ("*" ^ Docstrings.docstring_body ds, Docstrings.docstring_loc ds)
  in
    add_comment com

let comments () = List.rev !comment_list

(* Error report *)

open Format

let report_error ppf = function
  | Illegal_character c ->
      fprintf ppf "Illegal character (%s)" (Char.escaped c)
  | Illegal_escape s ->
      fprintf ppf "Illegal backslash escape in string or character (%s)" s
  | Unterminated_comment _ ->
      fprintf ppf "Comment not terminated"
  | Unterminated_string ->
      fprintf ppf "String literal not terminated"
  | Unterminated_string_in_comment (_, loc) ->
      fprintf ppf "This comment contains an unterminated string literal@.\
                   %aString literal begins here"
              Location.print_error loc
  | Keyword_as_label kwd ->
      fprintf ppf "`%s' is a keyword, it cannot be used as label name" kwd
  | Invalid_literal s ->
      fprintf ppf "Invalid literal %s" s
  | Invalid_directive (dir, explanation) ->
      fprintf ppf "Invalid lexer directive %S" dir;
      begin match explanation with
        | None -> ()
        | Some expl -> fprintf ppf ": %s" expl
      end

let () =
  Location.register_error_of_exn
    (function
      | Error (err, loc) ->
          Some (Location.error_of_printer loc report_error err)
      | _ ->
          None
    )


# 291 "ast/lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base =
   "\000\000\163\255\164\255\224\000\003\001\038\001\073\001\108\001\
    \143\001\187\255\178\001\215\001\195\255\091\000\252\001\031\002\
    \068\000\071\000\065\002\100\002\213\255\215\255\218\255\135\002\
    \170\002\205\002\088\000\255\000\235\002\236\255\063\003\147\003\
    \231\003\183\004\135\005\087\006\054\007\146\007\098\008\122\000\
    \254\255\001\000\005\000\255\255\006\000\007\000\065\009\095\009\
    \047\010\250\255\249\255\255\010\207\011\247\255\246\255\159\012\
    \243\012\071\013\155\013\239\013\067\014\151\014\235\014\063\015\
    \147\015\231\015\087\000\059\016\143\016\227\016\055\017\139\017\
    \108\000\193\255\235\255\203\002\230\017\106\000\107\000\011\000\
    \234\255\233\255\228\255\050\004\099\000\118\000\113\000\232\255\
    \128\000\147\000\231\255\224\000\003\001\148\000\230\255\002\005\
    \149\000\229\255\148\000\224\255\217\000\223\255\222\000\230\017\
    \222\255\013\018\059\018\098\018\133\018\171\018\206\018\210\255\
    \205\255\206\255\207\255\203\255\241\018\154\000\183\000\196\255\
    \197\255\198\255\216\000\183\255\181\255\190\255\020\019\186\255\
    \188\255\055\019\090\019\125\019\160\019\226\002\105\004\080\001\
    \038\001\024\001\022\006\243\255\244\255\012\000\245\255\139\001\
    \163\005\253\255\240\000\241\000\255\255\254\255\252\255\115\006\
    \178\019\250\000\252\000\013\000\251\255\250\255\249\255\006\008\
    \236\002\003\001\248\255\073\004\004\001\247\255\173\008\005\001\
    \246\255\017\000\174\001\245\255\246\255\247\255\018\000\251\019\
    \255\255\248\255\192\000\170\009\038\001\025\005\253\255\073\001\
    \094\001\111\001\035\005\252\255\223\003\057\005\251\255\122\010\
    \250\255\074\011\029\020\249\255\112\001\130\001\252\255\143\006\
    \254\255\255\255\113\001\129\001\253\255\025\007\025\001\035\001\
    \063\001\118\001\036\001\148\001\042\001\019\000\255\255";
  Lexing.lex_backtrk =
   "\255\255\255\255\255\255\089\000\088\000\085\000\084\000\077\000\
    \075\000\255\255\066\000\063\000\255\255\056\000\055\000\053\000\
    \051\000\047\000\044\000\080\000\255\255\255\255\255\255\035\000\
    \034\000\041\000\039\000\038\000\061\000\255\255\014\000\014\000\
    \013\000\012\000\011\000\010\000\007\000\004\000\003\000\002\000\
    \255\255\092\000\092\000\255\255\255\255\255\255\083\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\018\000\
    \018\000\016\000\015\000\018\000\015\000\015\000\014\000\016\000\
    \015\000\016\000\255\255\017\000\017\000\014\000\014\000\016\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\027\000\027\000\027\000\027\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\028\000\255\255\029\000\255\255\030\000\087\000\
    \255\255\090\000\036\000\086\000\081\000\043\000\046\000\255\255\
    \255\255\255\255\255\255\255\255\054\000\073\000\070\000\255\255\
    \255\255\255\255\071\000\255\255\255\255\255\255\064\000\255\255\
    \255\255\082\000\076\000\079\000\078\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\012\000\255\255\012\000\
    \012\000\255\255\012\000\012\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\010\000\010\000\
    \255\255\255\255\007\000\007\000\007\000\007\000\255\255\001\000\
    \007\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\003\000\
    \255\255\255\255\003\000\255\255\255\255\255\255\002\000\255\255\
    \255\255\001\000\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_default =
   "\001\000\000\000\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\077\000\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\255\255\082\000\255\255\255\255\255\255\
    \000\000\000\000\000\000\255\255\255\255\255\255\255\255\000\000\
    \255\255\255\255\000\000\255\255\255\255\255\255\000\000\255\255\
    \255\255\000\000\255\255\000\000\255\255\000\000\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\255\255\000\000\
    \000\000\000\000\255\255\000\000\000\000\000\000\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\135\000\
    \255\255\137\000\139\000\000\000\000\000\255\255\000\000\153\000\
    \255\255\000\000\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\255\255\255\255\000\000\000\000\000\000\255\255\
    \255\255\255\255\000\000\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\171\000\000\000\000\000\000\000\255\255\177\000\
    \000\000\000\000\255\255\255\255\255\255\255\255\000\000\255\255\
    \255\255\255\255\255\255\000\000\255\255\255\255\000\000\255\255\
    \000\000\255\255\255\255\000\000\255\255\198\000\000\000\255\255\
    \000\000\000\000\255\255\255\255\000\000\255\255\255\255\255\255\
    \208\000\211\000\255\255\211\000\255\255\255\255\000\000";
  Lexing.lex_trans =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\039\000\040\000\040\000\039\000\041\000\045\000\043\000\
    \043\000\040\000\044\000\044\000\045\000\078\000\140\000\154\000\
    \079\000\169\000\155\000\140\000\173\000\214\000\169\000\196\000\
    \039\000\008\000\029\000\024\000\006\000\004\000\023\000\027\000\
    \026\000\021\000\025\000\007\000\020\000\019\000\018\000\003\000\
    \031\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\017\000\016\000\015\000\014\000\010\000\036\000\
    \005\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\013\000\042\000\012\000\005\000\038\000\
    \022\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\028\000\011\000\009\000\037\000\115\000\
    \117\000\114\000\098\000\039\000\113\000\112\000\039\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\081\000\080\000\091\000\091\000\091\000\091\000\120\000\
    \087\000\119\000\039\000\118\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\088\000\088\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \089\000\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\090\000\094\000\097\000\099\000\100\000\124\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\121\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\122\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \002\000\003\000\101\000\102\000\003\000\003\000\003\000\101\000\
    \102\000\078\000\003\000\003\000\079\000\003\000\003\000\003\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \123\000\149\000\003\000\148\000\003\000\003\000\003\000\003\000\
    \003\000\158\000\255\255\157\000\003\000\255\255\255\255\003\000\
    \003\000\003\000\162\000\165\000\168\000\003\000\003\000\136\000\
    \003\000\003\000\003\000\093\000\093\000\093\000\093\000\093\000\
    \093\000\093\000\093\000\193\000\207\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\208\000\212\000\136\000\005\000\
    \135\000\209\000\005\000\005\000\005\000\213\000\000\000\000\000\
    \005\000\005\000\183\000\005\000\005\000\005\000\188\000\188\000\
    \188\000\188\000\255\255\076\000\003\000\255\255\003\000\000\000\
    \005\000\003\000\005\000\005\000\005\000\005\000\005\000\000\000\
    \183\000\183\000\006\000\185\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\137\000\006\000\006\000\000\000\006\000\006\000\
    \006\000\183\000\173\000\201\000\185\000\196\000\203\000\003\000\
    \210\000\003\000\000\000\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\201\000\201\000\107\000\203\000\202\000\
    \107\000\107\000\107\000\000\000\000\000\154\000\107\000\107\000\
    \155\000\107\000\132\000\107\000\000\000\000\000\210\000\000\000\
    \000\000\000\000\005\000\000\000\005\000\000\000\107\000\006\000\
    \107\000\131\000\107\000\107\000\107\000\000\000\000\000\000\000\
    \129\000\000\000\156\000\129\000\129\000\129\000\000\000\000\000\
    \173\000\129\000\129\000\174\000\129\000\129\000\129\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \000\000\129\000\107\000\129\000\130\000\129\000\129\000\129\000\
    \176\000\000\000\000\000\006\000\000\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\006\000\006\000\000\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\000\000\000\000\152\000\
    \107\000\000\000\107\000\000\000\006\000\129\000\006\000\006\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\006\000\199\000\255\255\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\175\000\129\000\000\000\129\000\000\000\128\000\
    \006\000\006\000\000\000\006\000\006\000\006\000\006\000\006\000\
    \255\255\000\000\000\000\000\000\000\000\006\000\000\000\000\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\006\000\127\000\
    \006\000\000\000\000\000\000\000\125\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\255\255\
    \006\000\000\000\000\000\006\000\006\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\000\000\116\000\006\000\006\000\000\000\
    \255\255\000\000\000\000\126\000\000\000\006\000\000\000\000\000\
    \000\000\006\000\006\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\110\000\000\000\000\000\110\000\110\000\110\000\
    \000\000\000\000\000\000\110\000\110\000\000\000\110\000\111\000\
    \110\000\000\000\000\000\000\000\000\000\000\000\255\255\000\000\
    \006\000\000\000\006\000\110\000\000\000\006\000\110\000\110\000\
    \110\000\110\000\200\000\000\000\000\000\107\000\000\000\000\000\
    \107\000\107\000\107\000\255\255\000\000\000\000\107\000\107\000\
    \000\000\107\000\108\000\107\000\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\006\000\000\000\006\000\107\000\110\000\
    \107\000\107\000\109\000\107\000\107\000\000\000\000\000\000\000\
    \006\000\000\000\000\000\006\000\006\000\106\000\172\000\000\000\
    \000\000\006\000\006\000\000\000\006\000\006\000\006\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\110\000\000\000\110\000\
    \000\000\006\000\107\000\006\000\006\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\105\000\000\000\105\000\105\000\105\000\
    \105\000\000\000\000\000\000\000\105\000\105\000\000\000\105\000\
    \105\000\105\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \107\000\000\000\107\000\000\000\105\000\006\000\105\000\105\000\
    \105\000\105\000\105\000\133\000\000\000\000\000\003\000\000\000\
    \000\000\003\000\003\000\003\000\000\000\000\000\104\000\103\000\
    \003\000\000\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\133\000\006\000\000\000\006\000\000\000\003\000\
    \105\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\163\000\163\000\163\000\105\000\073\000\
    \105\000\000\000\075\000\003\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\000\000\074\000\
    \000\000\003\000\075\000\003\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\055\000\074\000\
    \000\000\000\000\000\000\000\000\000\000\057\000\000\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\056\000\055\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\000\000\000\000\000\000\000\000\030\000\000\000\
    \055\000\055\000\055\000\055\000\056\000\055\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\057\000\000\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\059\000\055\000\055\000\
    \056\000\055\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\060\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\061\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\030\000\000\000\055\000\059\000\055\000\055\000\
    \056\000\055\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\060\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\061\000\058\000\058\000\032\000\189\000\
    \189\000\189\000\189\000\189\000\189\000\189\000\189\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\000\000\000\000\000\000\000\000\032\000\000\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\136\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\164\000\164\000\164\000\164\000\164\000\164\000\164\000\
    \164\000\164\000\164\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\136\000\000\000\135\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\000\000\000\000\000\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\000\000\000\000\000\000\000\000\033\000\000\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\096\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\096\000\096\000\096\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\186\000\186\000\186\000\186\000\186\000\186\000\186\000\
    \186\000\186\000\186\000\187\000\187\000\187\000\187\000\187\000\
    \187\000\187\000\187\000\187\000\187\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\096\000\096\000\096\000\096\000\096\000\
    \096\000\190\000\190\000\190\000\190\000\190\000\190\000\190\000\
    \190\000\000\000\000\000\000\000\000\000\000\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\000\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\034\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\000\000\000\000\000\000\000\000\034\000\000\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\151\000\000\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\000\000\150\000\
    \140\000\000\000\000\000\141\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \145\000\000\000\000\000\000\000\000\000\143\000\147\000\000\000\
    \146\000\000\000\000\000\000\000\000\000\000\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\144\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\000\000\000\000\000\000\000\000\035\000\000\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\151\000\000\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\205\000\150\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\000\000\000\000\204\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\142\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\000\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\046\000\
    \000\000\000\000\046\000\046\000\046\000\000\000\000\000\000\000\
    \046\000\046\000\000\000\046\000\046\000\046\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\000\000\046\000\046\000\046\000\046\000\046\000\000\000\
    \205\000\000\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\046\000\052\000\204\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\000\000\046\000\046\000\046\000\000\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\046\000\046\000\000\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\046\000\000\000\046\000\046\000\
    \046\000\046\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \046\000\048\000\000\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\000\000\046\000\000\000\
    \046\000\000\000\000\000\000\000\000\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\000\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\166\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\000\000\000\000\000\000\
    \000\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\035\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\000\000\000\000\000\000\
    \000\000\035\000\000\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\167\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\167\000\167\000\167\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\167\000\167\000\
    \167\000\167\000\167\000\167\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\000\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\046\000\000\000\000\000\046\000\046\000\046\000\
    \000\000\000\000\000\000\046\000\046\000\000\000\046\000\046\000\
    \046\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\046\000\000\000\046\000\046\000\046\000\
    \046\000\046\000\000\000\000\000\000\000\000\000\047\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\050\000\000\000\000\000\000\000\000\000\000\000\046\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\000\000\000\000\000\000\046\000\047\000\046\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\048\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\049\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\000\000\000\000\000\000\000\000\048\000\000\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\192\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\192\000\192\000\192\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\192\000\192\000\192\000\192\000\192\000\
    \192\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\000\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\051\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\054\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\000\000\000\000\000\000\000\000\051\000\000\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\194\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\194\000\194\000\194\000\194\000\194\000\
    \194\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\052\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\053\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\000\000\000\000\000\000\000\000\052\000\000\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\000\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\055\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\055\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\072\000\000\000\
    \072\000\000\000\000\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\056\000\055\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\000\000\000\000\000\000\000\000\057\000\000\000\
    \055\000\055\000\055\000\055\000\056\000\055\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\070\000\
    \070\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\055\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\055\000\000\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\063\000\000\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\064\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\062\000\000\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\064\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\068\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\000\000\000\000\000\000\000\000\063\000\000\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\068\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\055\000\000\000\000\000\000\000\066\000\000\000\
    \066\000\000\000\000\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\000\000\000\000\000\000\000\000\065\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\000\000\000\000\
    \000\000\000\000\055\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\000\000\
    \000\000\000\000\066\000\000\000\066\000\000\000\000\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\000\000\000\000\000\000\000\000\055\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\055\000\055\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\000\000\000\000\
    \000\000\000\000\069\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\055\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\070\000\
    \070\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\000\000\000\000\000\000\000\000\070\000\000\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\000\000\000\000\
    \000\000\000\000\071\000\000\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\086\000\103\000\
    \086\000\000\000\103\000\103\000\103\000\086\000\000\000\000\000\
    \103\000\103\000\000\000\103\000\103\000\103\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \103\000\000\000\103\000\103\000\103\000\103\000\103\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\105\000\000\000\
    \105\000\105\000\105\000\105\000\000\000\000\000\000\000\105\000\
    \105\000\000\000\105\000\105\000\105\000\000\000\000\000\000\000\
    \000\000\000\000\086\000\000\000\103\000\000\000\000\000\105\000\
    \086\000\105\000\105\000\105\000\105\000\105\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\086\000\084\000\000\000\000\000\
    \086\000\000\000\086\000\000\000\006\000\000\000\083\000\006\000\
    \006\000\006\000\103\000\000\000\103\000\006\000\006\000\000\000\
    \006\000\006\000\006\000\105\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \006\000\006\000\006\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\107\000\000\000\000\000\107\000\107\000\
    \107\000\105\000\000\000\105\000\107\000\107\000\000\000\107\000\
    \107\000\107\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\006\000\000\000\000\000\107\000\000\000\107\000\107\000\
    \107\000\107\000\107\000\000\000\000\000\000\000\107\000\000\000\
    \000\000\107\000\107\000\107\000\000\000\000\000\000\000\107\000\
    \107\000\000\000\107\000\107\000\107\000\000\000\000\000\006\000\
    \000\000\006\000\000\000\000\000\000\000\000\000\000\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\107\000\000\000\000\000\107\000\
    \107\000\107\000\000\000\000\000\000\000\107\000\107\000\000\000\
    \107\000\107\000\107\000\000\000\000\000\000\000\107\000\000\000\
    \107\000\000\000\000\000\107\000\000\000\107\000\255\255\107\000\
    \107\000\107\000\107\000\107\000\000\000\000\000\000\000\110\000\
    \000\000\000\000\110\000\110\000\110\000\000\000\000\000\000\000\
    \110\000\110\000\000\000\110\000\110\000\110\000\000\000\000\000\
    \000\000\107\000\000\000\107\000\000\000\000\000\000\000\000\000\
    \110\000\107\000\110\000\110\000\110\000\110\000\110\000\000\000\
    \000\000\000\000\006\000\000\000\000\000\006\000\006\000\006\000\
    \000\000\000\000\000\000\006\000\006\000\000\000\006\000\006\000\
    \006\000\000\000\000\000\000\000\000\000\000\000\000\000\107\000\
    \000\000\107\000\000\000\006\000\110\000\006\000\006\000\006\000\
    \006\000\006\000\000\000\000\000\000\000\006\000\000\000\000\000\
    \006\000\006\000\006\000\000\000\000\000\000\000\006\000\006\000\
    \000\000\006\000\006\000\006\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\110\000\000\000\110\000\000\000\006\000\006\000\
    \006\000\006\000\006\000\006\000\006\000\000\000\000\000\000\000\
    \129\000\000\000\000\000\129\000\129\000\129\000\000\000\000\000\
    \000\000\129\000\129\000\000\000\129\000\129\000\129\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\006\000\000\000\006\000\
    \000\000\129\000\006\000\129\000\129\000\129\000\129\000\129\000\
    \000\000\000\000\000\000\129\000\000\000\000\000\129\000\129\000\
    \129\000\000\000\000\000\000\000\129\000\129\000\000\000\129\000\
    \129\000\129\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \006\000\000\000\006\000\000\000\129\000\129\000\129\000\129\000\
    \129\000\129\000\129\000\000\000\000\000\000\000\107\000\000\000\
    \000\000\107\000\107\000\107\000\000\000\000\000\000\000\107\000\
    \107\000\000\000\107\000\107\000\107\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\129\000\000\000\129\000\000\000\107\000\
    \129\000\107\000\107\000\107\000\107\000\107\000\000\000\000\000\
    \000\000\107\000\000\000\000\000\107\000\107\000\107\000\000\000\
    \000\000\000\000\107\000\107\000\000\000\107\000\107\000\107\000\
    \000\000\000\000\161\000\000\000\161\000\000\000\129\000\000\000\
    \129\000\161\000\107\000\107\000\107\000\107\000\107\000\107\000\
    \107\000\000\000\160\000\160\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\107\000\000\000\107\000\000\000\000\000\107\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\183\000\000\000\000\000\
    \184\000\000\000\000\000\000\000\000\000\000\000\161\000\000\000\
    \000\000\000\000\000\000\000\000\161\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\182\000\107\000\182\000\107\000\000\000\
    \161\000\000\000\182\000\000\000\161\000\000\000\161\000\000\000\
    \000\000\000\000\159\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\181\000\181\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\182\000\
    \000\000\000\000\000\000\000\000\000\000\182\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\000\000\000\000\000\000\000\000\
    \000\000\182\000\180\000\000\000\000\000\182\000\000\000\182\000\
    \178\000\000\000\000\000\179\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\195\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_check =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\041\000\000\000\000\000\041\000\042\000\
    \044\000\045\000\042\000\044\000\045\000\079\000\141\000\155\000\
    \079\000\141\000\155\000\169\000\174\000\213\000\169\000\174\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\016\000\
    \013\000\017\000\026\000\039\000\017\000\017\000\039\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\077\000\078\000\084\000\084\000\084\000\084\000\013\000\
    \086\000\013\000\039\000\013\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\085\000\085\000\
    \085\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \088\000\088\000\088\000\088\000\088\000\088\000\088\000\088\000\
    \088\000\088\000\089\000\093\000\096\000\098\000\098\000\117\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\013\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\118\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\100\000\100\000\003\000\003\000\003\000\102\000\
    \102\000\027\000\003\000\003\000\027\000\003\000\003\000\003\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \122\000\146\000\003\000\147\000\003\000\003\000\003\000\003\000\
    \003\000\153\000\137\000\154\000\004\000\137\000\027\000\004\000\
    \004\000\004\000\161\000\164\000\167\000\004\000\004\000\136\000\
    \004\000\004\000\004\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\092\000\092\000\178\000\206\000\004\000\003\000\004\000\
    \004\000\004\000\004\000\004\000\207\000\210\000\136\000\005\000\
    \136\000\208\000\005\000\005\000\005\000\212\000\255\255\255\255\
    \005\000\005\000\183\000\005\000\005\000\005\000\180\000\180\000\
    \180\000\180\000\135\000\027\000\003\000\135\000\003\000\255\255\
    \005\000\004\000\005\000\005\000\005\000\005\000\005\000\255\255\
    \184\000\183\000\006\000\184\000\255\255\006\000\006\000\006\000\
    \255\255\255\255\135\000\006\000\006\000\255\255\006\000\006\000\
    \006\000\185\000\196\000\202\000\185\000\196\000\202\000\004\000\
    \209\000\004\000\255\255\006\000\005\000\006\000\006\000\006\000\
    \006\000\006\000\255\255\203\000\197\000\007\000\203\000\197\000\
    \007\000\007\000\007\000\255\255\255\255\143\000\007\000\007\000\
    \143\000\007\000\007\000\007\000\255\255\255\255\211\000\255\255\
    \255\255\255\255\005\000\255\255\005\000\255\255\007\000\006\000\
    \007\000\007\000\007\000\007\000\007\000\255\255\255\255\255\255\
    \008\000\255\255\143\000\008\000\008\000\008\000\255\255\255\255\
    \170\000\008\000\008\000\170\000\008\000\008\000\008\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\006\000\255\255\006\000\
    \255\255\008\000\007\000\008\000\008\000\008\000\008\000\008\000\
    \170\000\255\255\255\255\010\000\255\255\255\255\010\000\010\000\
    \010\000\255\255\255\255\255\255\010\000\010\000\255\255\010\000\
    \010\000\010\000\255\255\255\255\255\255\255\255\255\255\143\000\
    \007\000\255\255\007\000\255\255\010\000\008\000\010\000\010\000\
    \010\000\010\000\010\000\255\255\255\255\255\255\255\255\255\255\
    \011\000\255\255\255\255\011\000\011\000\011\000\197\000\027\000\
    \255\255\011\000\011\000\255\255\011\000\011\000\011\000\255\255\
    \255\255\255\255\170\000\008\000\255\255\008\000\255\255\010\000\
    \010\000\011\000\255\255\011\000\011\000\011\000\011\000\011\000\
    \137\000\255\255\255\255\255\255\255\255\014\000\255\255\255\255\
    \014\000\014\000\014\000\255\255\255\255\255\255\014\000\014\000\
    \255\255\014\000\014\000\014\000\255\255\255\255\010\000\010\000\
    \010\000\255\255\255\255\255\255\011\000\011\000\014\000\255\255\
    \014\000\014\000\014\000\014\000\014\000\255\255\255\255\208\000\
    \015\000\255\255\255\255\015\000\015\000\015\000\255\255\255\255\
    \255\255\015\000\015\000\255\255\015\000\015\000\015\000\255\255\
    \135\000\255\255\255\255\011\000\255\255\011\000\255\255\255\255\
    \255\255\015\000\014\000\015\000\015\000\015\000\015\000\015\000\
    \255\255\255\255\018\000\255\255\255\255\018\000\018\000\018\000\
    \255\255\255\255\255\255\018\000\018\000\255\255\018\000\018\000\
    \018\000\255\255\255\255\255\255\255\255\255\255\209\000\255\255\
    \014\000\255\255\014\000\018\000\255\255\015\000\018\000\018\000\
    \018\000\018\000\197\000\255\255\255\255\019\000\255\255\255\255\
    \019\000\019\000\019\000\143\000\255\255\255\255\019\000\019\000\
    \255\255\019\000\019\000\019\000\211\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\015\000\255\255\015\000\019\000\018\000\
    \019\000\019\000\019\000\019\000\019\000\255\255\255\255\255\255\
    \023\000\255\255\255\255\023\000\023\000\023\000\170\000\255\255\
    \255\255\023\000\023\000\255\255\023\000\023\000\023\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\018\000\255\255\018\000\
    \255\255\023\000\019\000\023\000\023\000\023\000\023\000\023\000\
    \255\255\255\255\255\255\024\000\255\255\024\000\024\000\024\000\
    \024\000\255\255\255\255\255\255\024\000\024\000\255\255\024\000\
    \024\000\024\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \019\000\255\255\019\000\255\255\024\000\023\000\024\000\024\000\
    \024\000\024\000\024\000\133\000\255\255\255\255\025\000\255\255\
    \255\255\025\000\025\000\025\000\255\255\255\255\025\000\025\000\
    \025\000\255\255\025\000\025\000\025\000\255\255\255\255\255\255\
    \255\255\255\255\133\000\023\000\255\255\023\000\255\255\025\000\
    \024\000\025\000\025\000\025\000\025\000\025\000\255\255\255\255\
    \255\255\255\255\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\133\000\133\000\160\000\160\000\160\000\160\000\
    \160\000\160\000\160\000\160\000\160\000\160\000\024\000\028\000\
    \024\000\255\255\075\000\025\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\255\255\075\000\
    \255\255\025\000\028\000\025\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\030\000\028\000\
    \255\255\255\255\255\255\255\255\255\255\030\000\255\255\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\255\255\255\255\255\255\255\255\030\000\255\255\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\031\000\255\255\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\255\255\255\255\
    \255\255\255\255\031\000\255\255\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\032\000\188\000\
    \188\000\188\000\188\000\188\000\188\000\188\000\188\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\255\255\255\255\255\255\255\255\032\000\255\255\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\134\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\163\000\163\000\163\000\163\000\163\000\163\000\163\000\
    \163\000\163\000\163\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\134\000\255\255\134\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\083\000\083\000\083\000\083\000\083\000\
    \083\000\134\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\255\255\255\255\255\255\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\255\255\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\033\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\255\255\255\255\255\255\255\255\033\000\255\255\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\095\000\095\000\095\000\095\000\095\000\095\000\
    \095\000\095\000\095\000\095\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\095\000\095\000\095\000\095\000\095\000\
    \095\000\181\000\181\000\181\000\181\000\181\000\181\000\181\000\
    \181\000\181\000\181\000\186\000\186\000\186\000\186\000\186\000\
    \186\000\186\000\186\000\186\000\186\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\095\000\095\000\095\000\095\000\095\000\
    \095\000\189\000\189\000\189\000\189\000\189\000\189\000\189\000\
    \189\000\255\255\255\255\255\255\255\255\255\255\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\034\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\255\255\255\255\255\255\255\255\034\000\255\255\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\144\000\255\255\144\000\144\000\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\144\000\144\000\
    \144\000\144\000\144\000\144\000\144\000\144\000\255\255\144\000\
    \138\000\255\255\255\255\138\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \138\000\255\255\255\255\255\255\255\255\138\000\138\000\255\255\
    \138\000\255\255\255\255\255\255\255\255\255\255\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\035\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\138\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\255\255\255\255\255\255\255\255\035\000\255\255\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\151\000\255\255\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\151\000\151\000\
    \151\000\151\000\151\000\151\000\151\000\151\000\199\000\151\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\199\000\199\000\199\000\199\000\199\000\199\000\
    \199\000\199\000\255\255\255\255\199\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\138\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\255\255\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\036\000\
    \255\255\255\255\036\000\036\000\036\000\255\255\255\255\255\255\
    \036\000\036\000\255\255\036\000\036\000\036\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \036\000\255\255\036\000\036\000\036\000\036\000\036\000\255\255\
    \205\000\255\255\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\205\000\205\000\205\000\205\000\
    \205\000\205\000\205\000\205\000\036\000\036\000\205\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\255\255\036\000\037\000\036\000\255\255\037\000\037\000\
    \037\000\255\255\255\255\255\255\037\000\037\000\255\255\037\000\
    \037\000\037\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\037\000\255\255\037\000\037\000\
    \037\000\037\000\037\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \037\000\037\000\255\255\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\255\255\037\000\255\255\
    \037\000\255\255\255\255\255\255\255\255\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\255\255\036\000\036\000\
    \036\000\036\000\036\000\036\000\036\000\036\000\159\000\159\000\
    \159\000\159\000\159\000\159\000\159\000\159\000\159\000\159\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\159\000\
    \159\000\159\000\159\000\159\000\159\000\255\255\255\255\255\255\
    \255\255\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\038\000\037\000\037\000\037\000\037\000\037\000\037\000\
    \037\000\037\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\255\255\255\255\255\255\
    \255\255\038\000\255\255\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\166\000\166\000\166\000\
    \166\000\166\000\166\000\166\000\166\000\166\000\166\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\166\000\166\000\
    \166\000\166\000\166\000\166\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\166\000\166\000\
    \166\000\166\000\166\000\166\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\255\255\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\255\255\038\000\038\000\038\000\038\000\038\000\038\000\
    \038\000\038\000\046\000\255\255\255\255\046\000\046\000\046\000\
    \255\255\255\255\255\255\046\000\046\000\255\255\046\000\046\000\
    \046\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\046\000\255\255\046\000\046\000\046\000\
    \046\000\046\000\255\255\255\255\255\255\255\255\047\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\255\255\255\255\255\255\255\255\255\255\046\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\255\255\255\255\255\255\046\000\047\000\046\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\179\000\179\000\179\000\179\000\179\000\179\000\
    \179\000\179\000\179\000\179\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\179\000\179\000\179\000\179\000\179\000\
    \179\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\179\000\179\000\179\000\179\000\179\000\
    \179\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\255\255\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\048\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\255\255\255\255\255\255\255\255\048\000\255\255\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\191\000\191\000\191\000\191\000\191\000\191\000\
    \191\000\191\000\191\000\191\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\191\000\191\000\191\000\191\000\191\000\
    \191\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\191\000\191\000\191\000\191\000\191\000\
    \191\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\255\255\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\051\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\255\255\255\255\255\255\255\255\051\000\255\255\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\193\000\193\000\193\000\193\000\193\000\193\000\
    \193\000\193\000\193\000\193\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\193\000\193\000\193\000\193\000\193\000\
    \193\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\193\000\193\000\193\000\193\000\193\000\
    \193\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\255\255\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\052\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\255\255\255\255\255\255\255\255\052\000\255\255\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\255\255\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\055\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\255\255\255\255\255\255\255\255\055\000\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\056\000\255\255\255\255\255\255\056\000\255\255\
    \056\000\255\255\255\255\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\255\255\255\255\
    \255\255\255\255\056\000\255\255\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\057\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\255\255\255\255\255\255\255\255\057\000\255\255\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\058\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\255\255\255\255\
    \255\255\255\255\058\000\255\255\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\059\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\255\255\255\255\255\255\255\255\059\000\255\255\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\060\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\255\255\255\255\
    \255\255\255\255\060\000\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\061\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\255\255\255\255\255\255\255\255\061\000\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\062\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\062\000\255\255\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\255\255\255\255\
    \255\255\255\255\062\000\255\255\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\063\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\063\000\255\255\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\064\000\255\255\255\255\255\255\064\000\255\255\
    \064\000\255\255\255\255\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\255\255\255\255\
    \255\255\255\255\064\000\255\255\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\065\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\255\255\255\255\255\255\255\255\065\000\255\255\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\067\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\255\255\255\255\
    \255\255\255\255\067\000\255\255\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\068\000\255\255\
    \255\255\255\255\068\000\255\255\068\000\255\255\255\255\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\255\255\255\255\255\255\255\255\068\000\255\255\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\069\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\255\255\255\255\
    \255\255\255\255\069\000\255\255\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\070\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\255\255\070\000\255\255\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\071\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\255\255\255\255\
    \255\255\255\255\071\000\255\255\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\076\000\103\000\
    \076\000\255\255\103\000\103\000\103\000\076\000\255\255\255\255\
    \103\000\103\000\255\255\103\000\103\000\103\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \103\000\255\255\103\000\103\000\103\000\103\000\103\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\105\000\255\255\
    \105\000\105\000\105\000\105\000\255\255\255\255\255\255\105\000\
    \105\000\255\255\105\000\105\000\105\000\255\255\255\255\255\255\
    \255\255\255\255\076\000\255\255\103\000\255\255\255\255\105\000\
    \076\000\105\000\105\000\105\000\105\000\105\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\076\000\076\000\255\255\255\255\
    \076\000\255\255\076\000\255\255\106\000\255\255\076\000\106\000\
    \106\000\106\000\103\000\255\255\103\000\106\000\106\000\255\255\
    \106\000\106\000\106\000\105\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\106\000\255\255\106\000\
    \106\000\106\000\106\000\106\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\107\000\255\255\255\255\107\000\107\000\
    \107\000\105\000\255\255\105\000\107\000\107\000\255\255\107\000\
    \107\000\107\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\106\000\255\255\255\255\107\000\255\255\107\000\107\000\
    \107\000\107\000\107\000\255\255\255\255\255\255\108\000\255\255\
    \255\255\108\000\108\000\108\000\255\255\255\255\255\255\108\000\
    \108\000\255\255\108\000\108\000\108\000\255\255\255\255\106\000\
    \255\255\106\000\255\255\255\255\255\255\255\255\255\255\108\000\
    \107\000\108\000\108\000\108\000\108\000\108\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\109\000\255\255\255\255\109\000\
    \109\000\109\000\255\255\255\255\255\255\109\000\109\000\255\255\
    \109\000\109\000\109\000\255\255\255\255\255\255\107\000\255\255\
    \107\000\255\255\255\255\108\000\255\255\109\000\076\000\109\000\
    \109\000\109\000\109\000\109\000\255\255\255\255\255\255\110\000\
    \255\255\255\255\110\000\110\000\110\000\255\255\255\255\255\255\
    \110\000\110\000\255\255\110\000\110\000\110\000\255\255\255\255\
    \255\255\108\000\255\255\108\000\255\255\255\255\255\255\255\255\
    \110\000\109\000\110\000\110\000\110\000\110\000\110\000\255\255\
    \255\255\255\255\116\000\255\255\255\255\116\000\116\000\116\000\
    \255\255\255\255\255\255\116\000\116\000\255\255\116\000\116\000\
    \116\000\255\255\255\255\255\255\255\255\255\255\255\255\109\000\
    \255\255\109\000\255\255\116\000\110\000\116\000\116\000\116\000\
    \116\000\116\000\255\255\255\255\255\255\126\000\255\255\255\255\
    \126\000\126\000\126\000\255\255\255\255\255\255\126\000\126\000\
    \255\255\126\000\126\000\126\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\110\000\255\255\110\000\255\255\126\000\116\000\
    \126\000\126\000\126\000\126\000\126\000\255\255\255\255\255\255\
    \129\000\255\255\255\255\129\000\129\000\129\000\255\255\255\255\
    \255\255\129\000\129\000\255\255\129\000\129\000\129\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\116\000\255\255\116\000\
    \255\255\129\000\126\000\129\000\129\000\129\000\129\000\129\000\
    \255\255\255\255\255\255\130\000\255\255\255\255\130\000\130\000\
    \130\000\255\255\255\255\255\255\130\000\130\000\255\255\130\000\
    \130\000\130\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \126\000\255\255\126\000\255\255\130\000\129\000\130\000\130\000\
    \130\000\130\000\130\000\255\255\255\255\255\255\131\000\255\255\
    \255\255\131\000\131\000\131\000\255\255\255\255\255\255\131\000\
    \131\000\255\255\131\000\131\000\131\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\129\000\255\255\129\000\255\255\131\000\
    \130\000\131\000\131\000\131\000\131\000\131\000\255\255\255\255\
    \255\255\132\000\255\255\255\255\132\000\132\000\132\000\255\255\
    \255\255\255\255\132\000\132\000\255\255\132\000\132\000\132\000\
    \255\255\255\255\152\000\255\255\152\000\255\255\130\000\255\255\
    \130\000\152\000\132\000\131\000\132\000\132\000\132\000\132\000\
    \132\000\255\255\152\000\152\000\152\000\152\000\152\000\152\000\
    \152\000\152\000\152\000\152\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\131\000\255\255\131\000\255\255\255\255\132\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\175\000\255\255\255\255\
    \175\000\255\255\255\255\255\255\255\255\255\255\152\000\255\255\
    \255\255\255\255\255\255\255\255\152\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\175\000\132\000\175\000\132\000\255\255\
    \152\000\255\255\175\000\255\255\152\000\255\255\152\000\255\255\
    \255\255\255\255\152\000\175\000\175\000\175\000\175\000\175\000\
    \175\000\175\000\175\000\175\000\175\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\194\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\194\000\194\000\194\000\175\000\
    \255\255\255\255\255\255\255\255\255\255\175\000\194\000\194\000\
    \194\000\194\000\194\000\194\000\255\255\255\255\255\255\255\255\
    \255\255\175\000\175\000\255\255\255\255\175\000\255\255\175\000\
    \175\000\255\255\255\255\175\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\194\000\194\000\
    \194\000\194\000\194\000\194\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\194\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\175\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255";
  Lexing.lex_base_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\010\000\000\000\
    \001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \002\000\004\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\020\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\032\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_default_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\012\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code =
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\029\000\029\000\000\000\029\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \001\000\000\000\000\000\015\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\007\000\000\000\000\000\000\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\004\000\004\000\004\000\004\000\
    \004\000\004\000\004\000\004\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check_code =
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\133\000\135\000\175\000\184\000\135\000\185\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \133\000\255\255\135\000\136\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\134\000\255\255\255\255\255\255\
    \133\000\133\000\133\000\133\000\133\000\133\000\133\000\133\000\
    \133\000\133\000\134\000\134\000\134\000\134\000\134\000\134\000\
    \134\000\134\000\134\000\134\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \135\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_code =
   "\255\004\255\255\005\255\255\007\255\006\255\255\007\255\255\006\
    \255\007\255\255\000\004\001\005\002\006\003\007\255\001\255\255\
    \000\001\255";
}

let rec token lexbuf =
   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 344 "ast/lexer.mll"
                 (
      if not !escaped_newlines then
        raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf));
      update_loc lexbuf None 1 false 0;
      token lexbuf )
# 1903 "ast/lexer.ml"

  | 1 ->
# 351 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        EOL )
# 1909 "ast/lexer.ml"

  | 2 ->
# 354 "ast/lexer.mll"
      ( token lexbuf )
# 1914 "ast/lexer.ml"

  | 3 ->
# 356 "ast/lexer.mll"
      ( UNDERSCORE )
# 1919 "ast/lexer.ml"

  | 4 ->
# 358 "ast/lexer.mll"
      ( TILDE )
# 1924 "ast/lexer.ml"

  | 5 ->
# 360 "ast/lexer.mll"
      ( LABEL (get_label_name lexbuf) )
# 1929 "ast/lexer.ml"

  | 6 ->
# 362 "ast/lexer.mll"
      ( warn_latin1 lexbuf; LABEL (get_label_name lexbuf) )
# 1934 "ast/lexer.ml"

  | 7 ->
# 364 "ast/lexer.mll"
      ( QUESTION )
# 1939 "ast/lexer.ml"

  | 8 ->
# 366 "ast/lexer.mll"
      ( OPTLABEL (get_label_name lexbuf) )
# 1944 "ast/lexer.ml"

  | 9 ->
# 368 "ast/lexer.mll"
      ( warn_latin1 lexbuf; OPTLABEL (get_label_name lexbuf) )
# 1949 "ast/lexer.ml"

  | 10 ->
# 370 "ast/lexer.mll"
      ( let s = Lexing.lexeme lexbuf in
        try Hashtbl.find keyword_table s
        with Not_found -> LIDENT s )
# 1956 "ast/lexer.ml"

  | 11 ->
# 374 "ast/lexer.mll"
      ( warn_latin1 lexbuf; LIDENT (Lexing.lexeme lexbuf) )
# 1961 "ast/lexer.ml"

  | 12 ->
# 376 "ast/lexer.mll"
      ( UIDENT(Lexing.lexeme lexbuf) )
# 1966 "ast/lexer.ml"

  | 13 ->
# 378 "ast/lexer.mll"
      ( warn_latin1 lexbuf; UIDENT(Lexing.lexeme lexbuf) )
# 1971 "ast/lexer.ml"

  | 14 ->
# 379 "ast/lexer.mll"
                ( INT (Lexing.lexeme lexbuf, None) )
# 1976 "ast/lexer.ml"

  | 15 ->
let
# 380 "ast/lexer.mll"
                    lit
# 1982 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 380 "ast/lexer.mll"
                                              modif
# 1987 "ast/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 381 "ast/lexer.mll"
      ( INT (lit, Some modif) )
# 1991 "ast/lexer.ml"

  | 16 ->
# 383 "ast/lexer.mll"
      ( FLOAT (Lexing.lexeme lexbuf, None) )
# 1996 "ast/lexer.ml"

  | 17 ->
let
# 384 "ast/lexer.mll"
                                            lit
# 2002 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -1)
and
# 384 "ast/lexer.mll"
                                                                      modif
# 2007 "ast/lexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_curr_pos + -1) in
# 385 "ast/lexer.mll"
      ( FLOAT (lit, Some modif) )
# 2011 "ast/lexer.ml"

  | 18 ->
# 387 "ast/lexer.mll"
      ( raise (Error(Invalid_literal (Lexing.lexeme lexbuf),
                     Location.curr lexbuf)) )
# 2017 "ast/lexer.ml"

  | 19 ->
# 390 "ast/lexer.mll"
      ( reset_string_buffer();
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        string lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), None) )
# 2029 "ast/lexer.ml"

  | 20 ->
# 399 "ast/lexer.mll"
      ( reset_string_buffer();
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        is_in_string := true;
        let string_start = lexbuf.lex_start_p in
        string_start_loc := Location.curr lexbuf;
        quoted_string delim lexbuf;
        is_in_string := false;
        lexbuf.lex_start_p <- string_start;
        STRING (get_stored_string(), Some delim) )
# 2043 "ast/lexer.ml"

  | 21 ->
# 410 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        CHAR (Lexing.lexeme_char lexbuf 1) )
# 2049 "ast/lexer.ml"

  | 22 ->
# 413 "ast/lexer.mll"
      ( CHAR(Lexing.lexeme_char lexbuf 1) )
# 2054 "ast/lexer.ml"

  | 23 ->
# 415 "ast/lexer.mll"
      ( CHAR(char_for_backslash (Lexing.lexeme_char lexbuf 2)) )
# 2059 "ast/lexer.ml"

  | 24 ->
# 417 "ast/lexer.mll"
      ( CHAR(char_for_decimal_code lexbuf 2) )
# 2064 "ast/lexer.ml"

  | 25 ->
# 419 "ast/lexer.mll"
      ( CHAR(char_for_octal_code lexbuf 3) )
# 2069 "ast/lexer.ml"

  | 26 ->
# 421 "ast/lexer.mll"
      ( CHAR(char_for_hexadecimal_code lexbuf 3) )
# 2074 "ast/lexer.ml"

  | 27 ->
# 423 "ast/lexer.mll"
      ( let l = Lexing.lexeme lexbuf in
        let esc = String.sub l 1 (String.length l - 1) in
        raise (Error(Illegal_escape esc, Location.curr lexbuf))
      )
# 2082 "ast/lexer.ml"

  | 28 ->
# 428 "ast/lexer.mll"
      ( let s, loc = with_comment_buffer comment lexbuf in
        COMMENT (s, loc) )
# 2088 "ast/lexer.ml"

  | 29 ->
# 431 "ast/lexer.mll"
      ( let s, loc = with_comment_buffer comment lexbuf in
        if !handle_docstrings then
          DOCSTRING (Docstrings.docstring s loc)
        else
          COMMENT ("*" ^ s, loc)
      )
# 2098 "ast/lexer.ml"

  | 30 ->
let
# 437 "ast/lexer.mll"
                     stars
# 2104 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 3) lexbuf.Lexing.lex_curr_pos in
# 438 "ast/lexer.mll"
      ( let s, loc =
          with_comment_buffer
            (fun lexbuf ->
               store_string ("*" ^ stars);
               comment lexbuf)
            lexbuf
        in
        COMMENT (s, loc) )
# 2115 "ast/lexer.ml"

  | 31 ->
# 447 "ast/lexer.mll"
      ( if !print_warnings then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Comment_start;
        let s, loc = with_comment_buffer comment lexbuf in
        COMMENT (s, loc) )
# 2123 "ast/lexer.ml"

  | 32 ->
let
# 451 "ast/lexer.mll"
                    stars
# 2129 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 2) (lexbuf.Lexing.lex_curr_pos + -2) in
# 452 "ast/lexer.mll"
      ( if !handle_docstrings && stars="" then
         (* (**) is an empty docstring *)
          DOCSTRING(Docstrings.docstring "" (Location.curr lexbuf))
        else
          COMMENT (stars, Location.curr lexbuf) )
# 2137 "ast/lexer.ml"

  | 33 ->
# 458 "ast/lexer.mll"
      ( let loc = Location.curr lexbuf in
        Location.prerr_warning loc Warnings.Comment_not_end;
        lexbuf.Lexing.lex_curr_pos <- lexbuf.Lexing.lex_curr_pos - 1;
        let curpos = lexbuf.lex_curr_p in
        lexbuf.lex_curr_p <- { curpos with pos_cnum = curpos.pos_cnum - 1 };
        STAR
      )
# 2148 "ast/lexer.ml"

  | 34 ->
# 466 "ast/lexer.mll"
      ( let at_beginning_of_line pos = (pos.pos_cnum = pos.pos_bol) in
        if not (at_beginning_of_line lexbuf.lex_start_p)
        then HASH
        else try directive lexbuf with Failure _ -> HASH
      )
# 2157 "ast/lexer.ml"

  | 35 ->
# 471 "ast/lexer.mll"
         ( AMPERSAND )
# 2162 "ast/lexer.ml"

  | 36 ->
# 472 "ast/lexer.mll"
         ( AMPERAMPER )
# 2167 "ast/lexer.ml"

  | 37 ->
# 473 "ast/lexer.mll"
         ( BACKQUOTE )
# 2172 "ast/lexer.ml"

  | 38 ->
# 474 "ast/lexer.mll"
         ( QUOTE )
# 2177 "ast/lexer.ml"

  | 39 ->
# 475 "ast/lexer.mll"
         ( LPAREN )
# 2182 "ast/lexer.ml"

  | 40 ->
# 476 "ast/lexer.mll"
         ( RPAREN )
# 2187 "ast/lexer.ml"

  | 41 ->
# 477 "ast/lexer.mll"
         ( STAR )
# 2192 "ast/lexer.ml"

  | 42 ->
# 478 "ast/lexer.mll"
         ( COMMA )
# 2197 "ast/lexer.ml"

  | 43 ->
# 479 "ast/lexer.mll"
         ( MINUSGREATER )
# 2202 "ast/lexer.ml"

  | 44 ->
# 480 "ast/lexer.mll"
         ( DOT )
# 2207 "ast/lexer.ml"

  | 45 ->
# 481 "ast/lexer.mll"
         ( DOTDOT )
# 2212 "ast/lexer.ml"

  | 46 ->
let
# 482 "ast/lexer.mll"
                                      s
# 2218 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf (lexbuf.Lexing.lex_start_pos + 1) lexbuf.Lexing.lex_curr_pos in
# 482 "ast/lexer.mll"
                                         ( DOTOP s )
# 2222 "ast/lexer.ml"

  | 47 ->
# 483 "ast/lexer.mll"
         ( COLON )
# 2227 "ast/lexer.ml"

  | 48 ->
# 484 "ast/lexer.mll"
         ( COLONCOLON )
# 2232 "ast/lexer.ml"

  | 49 ->
# 485 "ast/lexer.mll"
         ( COLONEQUAL )
# 2237 "ast/lexer.ml"

  | 50 ->
# 486 "ast/lexer.mll"
         ( COLONGREATER )
# 2242 "ast/lexer.ml"

  | 51 ->
# 487 "ast/lexer.mll"
         ( SEMI )
# 2247 "ast/lexer.ml"

  | 52 ->
# 488 "ast/lexer.mll"
         ( SEMISEMI )
# 2252 "ast/lexer.ml"

  | 53 ->
# 489 "ast/lexer.mll"
         ( LESS )
# 2257 "ast/lexer.ml"

  | 54 ->
# 490 "ast/lexer.mll"
         ( LESSMINUS )
# 2262 "ast/lexer.ml"

  | 55 ->
# 491 "ast/lexer.mll"
         ( EQUAL )
# 2267 "ast/lexer.ml"

  | 56 ->
# 492 "ast/lexer.mll"
         ( LBRACKET )
# 2272 "ast/lexer.ml"

  | 57 ->
# 493 "ast/lexer.mll"
         ( LBRACKETBAR )
# 2277 "ast/lexer.ml"

  | 58 ->
# 494 "ast/lexer.mll"
         ( LBRACKETLESS )
# 2282 "ast/lexer.ml"

  | 59 ->
# 495 "ast/lexer.mll"
         ( LBRACKETGREATER )
# 2287 "ast/lexer.ml"

  | 60 ->
# 496 "ast/lexer.mll"
         ( RBRACKET )
# 2292 "ast/lexer.ml"

  | 61 ->
# 497 "ast/lexer.mll"
         ( LBRACE )
# 2297 "ast/lexer.ml"

  | 62 ->
# 498 "ast/lexer.mll"
         ( LBRACELESS )
# 2302 "ast/lexer.ml"

  | 63 ->
# 499 "ast/lexer.mll"
         ( BAR )
# 2307 "ast/lexer.ml"

  | 64 ->
# 500 "ast/lexer.mll"
         ( BARBAR )
# 2312 "ast/lexer.ml"

  | 65 ->
# 501 "ast/lexer.mll"
         ( BARRBRACKET )
# 2317 "ast/lexer.ml"

  | 66 ->
# 502 "ast/lexer.mll"
         ( GREATER )
# 2322 "ast/lexer.ml"

  | 67 ->
# 503 "ast/lexer.mll"
         ( GREATERRBRACKET )
# 2327 "ast/lexer.ml"

  | 68 ->
# 504 "ast/lexer.mll"
         ( RBRACE )
# 2332 "ast/lexer.ml"

  | 69 ->
# 505 "ast/lexer.mll"
         ( GREATERRBRACE )
# 2337 "ast/lexer.ml"

  | 70 ->
# 506 "ast/lexer.mll"
         ( LBRACKETAT )
# 2342 "ast/lexer.ml"

  | 71 ->
# 507 "ast/lexer.mll"
           ( LBRACKETATAT )
# 2347 "ast/lexer.ml"

  | 72 ->
# 508 "ast/lexer.mll"
           ( LBRACKETATATAT )
# 2352 "ast/lexer.ml"

  | 73 ->
# 509 "ast/lexer.mll"
           ( LBRACKETPERCENT )
# 2357 "ast/lexer.ml"

  | 74 ->
# 510 "ast/lexer.mll"
           ( LBRACKETPERCENTPERCENT )
# 2362 "ast/lexer.ml"

  | 75 ->
# 511 "ast/lexer.mll"
         ( BANG )
# 2367 "ast/lexer.ml"

  | 76 ->
# 512 "ast/lexer.mll"
         ( INFIXOP0 "!=" )
# 2372 "ast/lexer.ml"

  | 77 ->
# 513 "ast/lexer.mll"
         ( PLUS )
# 2377 "ast/lexer.ml"

  | 78 ->
# 514 "ast/lexer.mll"
         ( PLUSDOT )
# 2382 "ast/lexer.ml"

  | 79 ->
# 515 "ast/lexer.mll"
         ( PLUSEQ )
# 2387 "ast/lexer.ml"

  | 80 ->
# 516 "ast/lexer.mll"
         ( MINUS )
# 2392 "ast/lexer.ml"

  | 81 ->
# 517 "ast/lexer.mll"
         ( MINUSDOT )
# 2397 "ast/lexer.ml"

  | 82 ->
# 520 "ast/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 2402 "ast/lexer.ml"

  | 83 ->
# 522 "ast/lexer.mll"
            ( PREFIXOP(Lexing.lexeme lexbuf) )
# 2407 "ast/lexer.ml"

  | 84 ->
# 524 "ast/lexer.mll"
            ( INFIXOP0(Lexing.lexeme lexbuf) )
# 2412 "ast/lexer.ml"

  | 85 ->
# 526 "ast/lexer.mll"
            ( INFIXOP1(Lexing.lexeme lexbuf) )
# 2417 "ast/lexer.ml"

  | 86 ->
# 528 "ast/lexer.mll"
            ( INFIXOP2(Lexing.lexeme lexbuf) )
# 2422 "ast/lexer.ml"

  | 87 ->
# 530 "ast/lexer.mll"
            ( INFIXOP4(Lexing.lexeme lexbuf) )
# 2427 "ast/lexer.ml"

  | 88 ->
# 531 "ast/lexer.mll"
            ( PERCENT )
# 2432 "ast/lexer.ml"

  | 89 ->
# 533 "ast/lexer.mll"
            ( INFIXOP3(Lexing.lexeme lexbuf) )
# 2437 "ast/lexer.ml"

  | 90 ->
# 535 "ast/lexer.mll"
            ( HASHOP(Lexing.lexeme lexbuf) )
# 2442 "ast/lexer.ml"

  | 91 ->
# 536 "ast/lexer.mll"
        ( EOF )
# 2447 "ast/lexer.ml"

  | 92 ->
# 538 "ast/lexer.mll"
      ( raise (Error(Illegal_character (Lexing.lexeme_char lexbuf 0),
                     Location.curr lexbuf))
      )
# 2454 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and directive lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 8 (-1);(* L=1 [4] <- p ;  *)
  lexbuf.Lexing.lex_mem.(4) <- lexbuf.Lexing.lex_curr_pos ;
 __ocaml_lex_directive_rec lexbuf 133
and __ocaml_lex_directive_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
let
# 543 "ast/lexer.mll"
                                num
# 2469 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_mem.(1)
and
# 544 "ast/lexer.mll"
                                            name
# 2474 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(2) lexbuf.Lexing.lex_mem.(3)
and
# 544 "ast/lexer.mll"
                                                           directive
# 2479 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_mem.(3) + 1) in
# 546 "ast/lexer.mll"
      (
        match int_of_string num with
        | exception _ ->
            (* PR#7165 *)
            let loc = Location.curr lexbuf in
            let explanation = "line number out of range" in
            let error = Invalid_directive ("#" ^ directive, Some explanation) in
            raise (Error (error, loc))
        | line_num ->
           (* Documentation says that the line number should be
              positive, but we have never guarded against this and it
              might have useful hackish uses. *)
            update_loc lexbuf (Some name) (line_num - 1) true 0;
            token lexbuf
      )
# 2497 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_directive_rec lexbuf __ocaml_lex_state

and comment lexbuf =
   __ocaml_lex_comment_rec lexbuf 138
and __ocaml_lex_comment_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 563 "ast/lexer.mll"
      ( comment_start_loc := (Location.curr lexbuf) :: !comment_start_loc;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2512 "ast/lexer.ml"

  | 1 ->
# 568 "ast/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | [_] -> comment_start_loc := []; Location.curr lexbuf
        | _ :: l -> comment_start_loc := l;
                  store_lexeme lexbuf;
                  comment lexbuf
       )
# 2523 "ast/lexer.ml"

  | 2 ->
# 576 "ast/lexer.mll"
      (
        string_start_loc := Location.curr lexbuf;
        store_string_char '\"';
        is_in_string := true;
        begin try string lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '\"';
        comment lexbuf )
# 2544 "ast/lexer.ml"

  | 3 ->
# 594 "ast/lexer.mll"
      (
        let delim = Lexing.lexeme lexbuf in
        let delim = String.sub delim 1 (String.length delim - 2) in
        string_start_loc := Location.curr lexbuf;
        store_lexeme lexbuf;
        is_in_string := true;
        begin try quoted_string delim lexbuf
        with Error (Unterminated_string, str_start) ->
          match !comment_start_loc with
          | [] -> assert false
          | loc :: _ ->
            let start = List.hd (List.rev !comment_start_loc) in
            comment_start_loc := [];
            raise (Error (Unterminated_string_in_comment (start, str_start),
                          loc))
        end;
        is_in_string := false;
        store_string_char '|';
        store_string delim;
        store_string_char '}';
        comment lexbuf )
# 2569 "ast/lexer.ml"

  | 4 ->
# 617 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2574 "ast/lexer.ml"

  | 5 ->
# 619 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false 1;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2582 "ast/lexer.ml"

  | 6 ->
# 624 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2587 "ast/lexer.ml"

  | 7 ->
# 626 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2592 "ast/lexer.ml"

  | 8 ->
# 628 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2597 "ast/lexer.ml"

  | 9 ->
# 630 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2602 "ast/lexer.ml"

  | 10 ->
# 632 "ast/lexer.mll"
      ( match !comment_start_loc with
        | [] -> assert false
        | loc :: _ ->
          let start = List.hd (List.rev !comment_start_loc) in
          comment_start_loc := [];
          raise (Error (Unterminated_comment start, loc))
      )
# 2613 "ast/lexer.ml"

  | 11 ->
# 640 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        comment lexbuf
      )
# 2621 "ast/lexer.ml"

  | 12 ->
# 645 "ast/lexer.mll"
      ( store_lexeme lexbuf; comment lexbuf )
# 2626 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_comment_rec lexbuf __ocaml_lex_state

and string lexbuf =
  lexbuf.Lexing.lex_mem <- Array.make 2 (-1); __ocaml_lex_string_rec lexbuf 170
and __ocaml_lex_string_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 649 "ast/lexer.mll"
      ( () )
# 2638 "ast/lexer.ml"

  | 1 ->
let
# 650 "ast/lexer.mll"
                                  space
# 2644 "ast/lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 651 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false (String.length space);
        if in_comment () then store_lexeme lexbuf;
        string lexbuf
      )
# 2651 "ast/lexer.ml"

  | 2 ->
# 656 "ast/lexer.mll"
      ( store_escaped_char lexbuf
                           (char_for_backslash(Lexing.lexeme_char lexbuf 1));
        string lexbuf )
# 2658 "ast/lexer.ml"

  | 3 ->
# 660 "ast/lexer.mll"
      ( store_escaped_char lexbuf (char_for_decimal_code lexbuf 1);
         string lexbuf )
# 2664 "ast/lexer.ml"

  | 4 ->
# 663 "ast/lexer.mll"
      ( store_escaped_char lexbuf (char_for_octal_code lexbuf 2);
         string lexbuf )
# 2670 "ast/lexer.ml"

  | 5 ->
# 666 "ast/lexer.mll"
      ( store_escaped_char lexbuf (char_for_hexadecimal_code lexbuf 2);
         string lexbuf )
# 2676 "ast/lexer.ml"

  | 6 ->
# 669 "ast/lexer.mll"
        ( store_escaped_uchar lexbuf (uchar_for_uchar_escape lexbuf);
          string lexbuf )
# 2682 "ast/lexer.ml"

  | 7 ->
# 672 "ast/lexer.mll"
      ( if not (in_comment ()) then begin
(*  Should be an error, but we are very lax.
          raise (Error (Illegal_escape (Lexing.lexeme lexbuf),
                        Location.curr lexbuf))
*)
          let loc = Location.curr lexbuf in
          Location.prerr_warning loc Warnings.Illegal_backslash;
        end;
        store_lexeme lexbuf;
        string lexbuf
      )
# 2697 "ast/lexer.ml"

  | 8 ->
# 684 "ast/lexer.mll"
      ( if not (in_comment ()) then
          Location.prerr_warning (Location.curr lexbuf) Warnings.Eol_in_string;
        update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        string lexbuf
      )
# 2707 "ast/lexer.ml"

  | 9 ->
# 691 "ast/lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 2713 "ast/lexer.ml"

  | 10 ->
# 694 "ast/lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        string lexbuf )
# 2719 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_string_rec lexbuf __ocaml_lex_state

and quoted_string delim lexbuf =
   __ocaml_lex_quoted_string_rec delim lexbuf 197
and __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 699 "ast/lexer.mll"
      ( update_loc lexbuf None 1 false 0;
        store_lexeme lexbuf;
        quoted_string delim lexbuf
      )
# 2734 "ast/lexer.ml"

  | 1 ->
# 704 "ast/lexer.mll"
      ( is_in_string := false;
        raise (Error (Unterminated_string, !string_start_loc)) )
# 2740 "ast/lexer.ml"

  | 2 ->
# 707 "ast/lexer.mll"
      (
        let edelim = Lexing.lexeme lexbuf in
        let edelim = String.sub edelim 1 (String.length edelim - 2) in
        if delim = edelim then ()
        else (store_lexeme lexbuf; quoted_string delim lexbuf)
      )
# 2750 "ast/lexer.ml"

  | 3 ->
# 714 "ast/lexer.mll"
      ( store_string_char(Lexing.lexeme_char lexbuf 0);
        quoted_string delim lexbuf )
# 2756 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_quoted_string_rec delim lexbuf __ocaml_lex_state

and skip_hash_bang lexbuf =
   __ocaml_lex_skip_hash_bang_rec lexbuf 206
and __ocaml_lex_skip_hash_bang_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 719 "ast/lexer.mll"
       ( update_loc lexbuf None 3 false 0 )
# 2768 "ast/lexer.ml"

  | 1 ->
# 721 "ast/lexer.mll"
       ( update_loc lexbuf None 1 false 0 )
# 2773 "ast/lexer.ml"

  | 2 ->
# 722 "ast/lexer.mll"
       ( () )
# 2778 "ast/lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf;
      __ocaml_lex_skip_hash_bang_rec lexbuf __ocaml_lex_state

;;

# 724 "ast/lexer.mll"
 

  let token_with_comments lexbuf =
    match !preprocessor with
    | None -> token lexbuf
    | Some (_init, preprocess) -> preprocess token lexbuf

  type newline_state =
    | NoLine (* There have been no blank lines yet. *)
    | NewLine
        (* There have been no blank lines, and the previous
           token was a newline. *)
    | BlankLine (* There have been blank lines. *)

  type doc_state =
    | Initial  (* There have been no docstrings yet *)
    | After of docstring list
        (* There have been docstrings, none of which were
           preceded by a blank line *)
    | Before of docstring list * docstring list * docstring list
        (* There have been docstrings, some of which were
           preceded by a blank line *)

  and docstring = Docstrings.docstring

  let token lexbuf =
    let post_pos = lexeme_end_p lexbuf in
    let attach lines docs pre_pos =
      let open Docstrings in
        match docs, lines with
        | Initial, _ -> ()
        | After a, (NoLine | NewLine) ->
            set_post_docstrings post_pos (List.rev a);
            set_pre_docstrings pre_pos a;
        | After a, BlankLine ->
            set_post_docstrings post_pos (List.rev a);
            set_pre_extra_docstrings pre_pos (List.rev a)
        | Before(a, f, b), (NoLine | NewLine) ->
            set_post_docstrings post_pos (List.rev a);
            set_post_extra_docstrings post_pos
              (List.rev_append f (List.rev b));
            set_floating_docstrings pre_pos (List.rev f);
            set_pre_extra_docstrings pre_pos (List.rev a);
            set_pre_docstrings pre_pos b
        | Before(a, f, b), BlankLine ->
            set_post_docstrings post_pos (List.rev a);
            set_post_extra_docstrings post_pos
              (List.rev_append f (List.rev b));
            set_floating_docstrings pre_pos
              (List.rev_append f (List.rev b));
            set_pre_extra_docstrings pre_pos (List.rev a)
    in
    let rec loop lines docs lexbuf =
      match token_with_comments lexbuf with
      | COMMENT (s, loc) ->
          add_comment (s, loc);
          let lines' =
            match lines with
            | NoLine -> NoLine
            | NewLine -> NoLine
            | BlankLine -> BlankLine
          in
          loop lines' docs lexbuf
      | EOL ->
          let lines' =
            match lines with
            | NoLine -> NewLine
            | NewLine -> BlankLine
            | BlankLine -> BlankLine
          in
          loop lines' docs lexbuf
      | DOCSTRING doc ->
          Docstrings.register doc;
          add_docstring_comment doc;
          let docs' =
            if Docstrings.docstring_body doc = "/*" then
              match docs with
              | Initial -> Before([], [doc], [])
              | After a -> Before (a, [doc], [])
              | Before(a, f, b) -> Before(a, doc :: b @ f, [])
            else
              match docs, lines with
              | Initial, (NoLine | NewLine) -> After [doc]
              | Initial, BlankLine -> Before([], [], [doc])
              | After a, (NoLine | NewLine) -> After (doc :: a)
              | After a, BlankLine -> Before (a, [], [doc])
              | Before(a, f, b), (NoLine | NewLine) -> Before(a, f, doc :: b)
              | Before(a, f, b), BlankLine -> Before(a, b @ f, [doc])
          in
          loop NoLine docs' lexbuf
      | tok ->
          attach lines docs (lexeme_start_p lexbuf);
          tok
    in
      loop NoLine Initial lexbuf

  let init () =
    is_in_string := false;
    comment_start_loc := [];
    comment_list := [];
    match !preprocessor with
    | None -> ()
    | Some (init, _preprocess) -> init ()

  let set_preprocessor init preprocess =
    escaped_newlines := true;
    preprocessor := Some (init, preprocess)


# 2895 "ast/lexer.ml"
