/*	$NetBSD: $	*/

/*-
 * Copyright (c) 2008 The NetBSD Foundation, Inc.
 * All rights reserved.
 *
 * This code is derived from software contributed to The NetBSD Foundation
 * by Robert Swindells
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE NETBSD FOUNDATION, INC. AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE FOUNDATION OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/cdefs.h>
__KERNEL_RCSID(0, "$NetBSD: $");

#include "bpfilter.h"
#include "rnd.h"

#include <sys/param.h>
#include <sys/types.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/device.h>
#include <sys/malloc.h>
#include <sys/socket.h>

#include <uvm/uvm_extern.h>

#if NRND > 0
#include <sys/rnd.h>
#endif

#include <net/if.h>
#include <net/if_dl.h>
#include <net/if_arp.h>
#include <net/if_media.h>
#include <net/if_ether.h>

#include <dev/ofw/openfirm.h>

#include <machine/bus.h>
#include <machine/autoconf.h>
#include <machine/pio.h>

#include <dev/mii/mii.h>
#include <dev/mii/miivar.h>

#include <powerpc/mpc5200/fecvar.h>

static int	fec_match(device_t, cfdata_t, void *);
static void	fec_attach(device_t, device_t, void *);

CFATTACH_DECL_NEW(fec, sizeof(struct fec_softc),
    fec_match, fec_attach, NULL, NULL);

int
fec_match(device_t parent, cfdata_t cf, void *aux)
{
	struct confargs *ca = aux;
	char name[32];

	if (strcmp(ca->ca_name, "ethernet") != 0)
		return 0;

	OF_getprop(ca->ca_node, "compatible", name, sizeof(name));
	if (strcmp(name, "mpc5200-fec") != 0)
		return 0;

	return 1;
}

void
fec_attach(device_t parent, device_t self, void *aux)
{
	struct confargs *ca = aux;
	struct fec_softc *sc = device_private(self);
	uint8_t myaddr[ETHER_ADDR_LEN];
	char name[32];
	struct reg {
		uint32_t addr;
		uint32_t size;
	} range, *rp = &range;
	int node, task;

	sc->sc_dev = self;
	sc->sc.regt = ca->ca_tag;

	if (OF_getprop(ca->ca_node, "reg", rp, sizeof(range)) != sizeof(range))
		return;

	bus_space_map(sc->sc_regt, rp->addr, rp->size, 0, &sc->sc_regh);

	sc->sc_tail = 0;

	aprint_normal(": irq %d,%d,%d",
		ca->ca_intr[0], ca->ca_intr[1], ca->ca_intr[2]);

	if (OF_getprop(ca->ca_node, "local-mac-address", myaddr, 6) != 6) {
		aprint_normal(": failed to get MAC address.\n");
		return;
	}

	sc->sc_bus_init = fec_init;
	sc->sc_putpacket = fec_putpacket;

	for (node = OF_child(ca->ca_node); node; OF_peer(node)) {
		memset(name, 0, sizeof(name));
		OF_getprop(node, "name", name, sizeof(name));
		if (strcmp(name, "bestcomm-rxtask") == 0) {
			OF_getprop(node, "taskid", sc->sc_rx_task,
				   sizeof(sc->sc_rx_task));
		}
		if (strcmp(name, "bestcomm-txtask") == 0) {
			OF_getprop(node, "taskid", sc->sc_tx_task,
				   sizeof(sc->sc_tx_task));
		}
	}

	/* install interrupt handler */
	intr_establish(ca->ca_intr[2], IST_EDGE, IPL_NET, fec_dmaintr, sc);
	intr_establish(ca->ca_intr[0], IST_EDGE, IPL_NET, fecintr, sc);

	/* fec_init returns 1 if something fails */
	if (fec_init(sc, myaddr)) {
		printf("mcsetup returns non zero\n");
		return;
	}
#ifdef NOTYET
	sc->sc_mediachange = fec_mediachange;
	sc->sc_mediastatus = fec_mediastatus;
	sc->sc_supmedia = fec_supmedia;
	sc->sc_nsupmedia = N_SUPMEDIA;
	sc->sc_defaultmedia = IFM_ETHER | IFM_10_T;
#endif
}
