"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexPattern = void 0;

var _lodash = _interopRequireWildcard(require("lodash"));

var _i18n = require("@kbn/i18n");

var _common = require("../../../../kibana_utils/common");

var _common2 = require("../../../common");

var _utils = require("../utils");

var _lib = require("../lib");

var _fields = require("../fields");

var _fields_fetcher = require("./_fields_fetcher");

var _format_hit = require("./format_hit");

var _flatten_hit = require("./flatten_hit");

var _field_mapping = require("../../field_mapping");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

const MAX_ATTEMPTS_TO_RESOLVE_CONFLICTS = 3;
const savedObjectType = 'index-pattern';

class IndexPattern {
  constructor(id, {
    getConfig,
    savedObjectsClient,
    apiClient,
    patternCache,
    fieldFormats,
    onNotification,
    onError,
    onUnsupportedTimePattern,
    uiSettingsValues
  }) {
    _defineProperty(this, "id", void 0);

    _defineProperty(this, "title", '');

    _defineProperty(this, "fieldFormatMap", void 0);

    _defineProperty(this, "typeMeta", void 0);

    _defineProperty(this, "fields", void 0);

    _defineProperty(this, "timeFieldName", void 0);

    _defineProperty(this, "intervalName", void 0);

    _defineProperty(this, "formatHit", void 0);

    _defineProperty(this, "formatField", void 0);

    _defineProperty(this, "flattenHit", void 0);

    _defineProperty(this, "metaFields", void 0);

    _defineProperty(this, "version", void 0);

    _defineProperty(this, "savedObjectsClient", void 0);

    _defineProperty(this, "patternCache", void 0);

    _defineProperty(this, "getConfig", void 0);

    _defineProperty(this, "sourceFilters", void 0);

    _defineProperty(this, "originalBody", {});

    _defineProperty(this, "fieldsFetcher", void 0);

    _defineProperty(this, "shortDotsEnable", false);

    _defineProperty(this, "fieldFormats", void 0);

    _defineProperty(this, "onNotification", void 0);

    _defineProperty(this, "onError", void 0);

    _defineProperty(this, "onUnsupportedTimePattern", void 0);

    _defineProperty(this, "apiClient", void 0);

    _defineProperty(this, "uiSettingsValues", void 0);

    _defineProperty(this, "mapping", (0, _field_mapping.expandShorthand)({
      title: _common2.ES_FIELD_TYPES.TEXT,
      timeFieldName: _common2.ES_FIELD_TYPES.KEYWORD,
      intervalName: _common2.ES_FIELD_TYPES.KEYWORD,
      fields: 'json',
      sourceFilters: 'json',
      fieldFormatMap: {
        type: _common2.ES_FIELD_TYPES.TEXT,
        _serialize: (map = {}) => {
          const serialized = _lodash.default.transform(map, this.serializeFieldFormatMap);

          return _lodash.default.isEmpty(serialized) ? undefined : JSON.stringify(serialized);
        },
        _deserialize: (map = '{}') => {
          return _lodash.default.mapValues(JSON.parse(map), mapping => {
            return this.deserializeFieldFormatMap(mapping);
          });
        }
      },
      type: _common2.ES_FIELD_TYPES.KEYWORD,
      typeMeta: 'json'
    }));

    this.id = id;
    this.savedObjectsClient = savedObjectsClient;
    this.patternCache = patternCache; // instead of storing config we rather store the getter only as np uiSettingsClient has circular references
    // which cause problems when being consumed from angular

    this.getConfig = getConfig;
    this.fieldFormats = fieldFormats;
    this.onNotification = onNotification;
    this.onError = onError;
    this.onUnsupportedTimePattern = onUnsupportedTimePattern;
    this.uiSettingsValues = uiSettingsValues;
    this.shortDotsEnable = uiSettingsValues.shortDotsEnable;
    this.metaFields = uiSettingsValues.metaFields;
    this.createFieldList = (0, _fields.getIndexPatternFieldListCreator)({
      fieldFormats,
      onNotification
    });
    this.fields = this.createFieldList(this, [], this.shortDotsEnable);
    this.apiClient = apiClient;
    this.fieldsFetcher = (0, _fields_fetcher.createFieldsFetcher)(this, apiClient, uiSettingsValues.metaFields);
    this.flattenHit = (0, _flatten_hit.flattenHitWrapper)(this, uiSettingsValues.metaFields);
    this.formatHit = (0, _format_hit.formatHitProvider)(this, fieldFormats.getDefaultInstance(_common2.KBN_FIELD_TYPES.STRING));
    this.formatField = this.formatHit.formatField;
  }

  serializeFieldFormatMap(flat, format, field) {
    if (format && field) {
      flat[field] = format;
    }
  }

  deserializeFieldFormatMap(mapping) {
    const FieldFormat = this.fieldFormats.getType(mapping.id);
    return FieldFormat && new FieldFormat(mapping.params, key => {
      var _this$uiSettingsValue, _this$uiSettingsValue2;

      return ((_this$uiSettingsValue = this.uiSettingsValues[key]) === null || _this$uiSettingsValue === void 0 ? void 0 : _this$uiSettingsValue.userValue) || ((_this$uiSettingsValue2 = this.uiSettingsValues[key]) === null || _this$uiSettingsValue2 === void 0 ? void 0 : _this$uiSettingsValue2.value);
    });
  }

  initFields(input) {
    const newValue = input || this.fields;
    this.fields = this.createFieldList(this, newValue, this.shortDotsEnable);
  }

  isFieldRefreshRequired() {
    if (!this.fields) {
      return true;
    }

    return this.fields.every(field => {
      // See https://github.com/elastic/kibana/pull/8421
      const hasFieldCaps = 'aggregatable' in field && 'searchable' in field; // See https://github.com/elastic/kibana/pull/11969

      const hasDocValuesFlag = ('readFromDocValues' in field);
      return !hasFieldCaps || !hasDocValuesFlag;
    });
  }

  async indexFields(forceFieldRefresh = false) {
    if (!this.id) {
      return;
    }

    if (forceFieldRefresh || this.isFieldRefreshRequired()) {
      await this.refreshFields();
    }

    this.initFields();
  }

  initFromSpec(spec) {
    // create fieldFormatMap from field list
    const fieldFormatMap = {};

    if (_lodash.default.isArray(spec.fields)) {
      spec.fields.forEach(field => {
        if (field.format) {
          fieldFormatMap[field.name] = { ...field.format
          };
        }
      });
    }

    this.version = spec.version;
    this.title = spec.title || '';
    this.timeFieldName = spec.timeFieldName;
    this.sourceFilters = spec.sourceFilters; // ignoring this because the same thing happens elsewhere but via _.assign
    // @ts-expect-error

    this.fields = spec.fields || [];
    this.typeMeta = spec.typeMeta;
    this.fieldFormatMap = _lodash.default.mapValues(fieldFormatMap, mapping => {
      return this.deserializeFieldFormatMap(mapping);
    });
    this.initFields();
    return this;
  }

  async updateFromElasticSearch(response, forceFieldRefresh = false) {
    if (!response.found) {
      throw new _common.SavedObjectNotFound(savedObjectType, this.id, 'management/kibana/indexPatterns');
    }

    _lodash.default.forOwn(this.mapping, (fieldMapping, name) => {
      if (!fieldMapping._deserialize || !name) {
        return;
      }

      response[name] = fieldMapping._deserialize(response[name]);
    }); // give index pattern all of the values


    _lodash.default.assign(this, response);

    if (!this.title && this.id) {
      this.title = this.id;
    }

    this.version = response.version;

    if (this.isUnsupportedTimePattern()) {
      this.onUnsupportedTimePattern({
        id: this.id,
        title: this.title,
        index: this.getIndex()
      });
    }

    return this.indexFields(forceFieldRefresh);
  }

  getComputedFields() {
    const scriptFields = {};

    if (!this.fields) {
      return {
        storedFields: ['*'],
        scriptFields,
        docvalueFields: []
      };
    } // Date value returned in "_source" could be in any number of formats
    // Use a docvalue for each date field to ensure standardized formats when working with date fields
    // indexPattern.flattenHit will override "_source" values when the same field is also defined in "fields"


    const docvalueFields = (0, _lodash.reject)(this.fields.getByType('date'), 'scripted').map(dateField => {
      return {
        field: dateField.name,
        format: dateField.esTypes && dateField.esTypes.indexOf('date_nanos') !== -1 ? 'strict_date_time' : 'date_time'
      };
    });
    (0, _lodash.each)(this.getScriptedFields(), function (field) {
      scriptFields[field.name] = {
        script: {
          source: field.script,
          lang: field.lang
        }
      };
    });
    return {
      storedFields: ['*'],
      scriptFields,
      docvalueFields
    };
  }

  async init(forceFieldRefresh = false) {
    if (!this.id) {
      return this; // no id === no elasticsearch document
    }

    const savedObject = await this.savedObjectsClient.get(savedObjectType, this.id);
    const response = {
      version: savedObject.version,
      found: savedObject.version ? true : false,
      title: savedObject.attributes.title,
      timeFieldName: savedObject.attributes.timeFieldName,
      intervalName: savedObject.attributes.intervalName,
      fields: savedObject.attributes.fields,
      sourceFilters: savedObject.attributes.sourceFilters,
      fieldFormatMap: savedObject.attributes.fieldFormatMap,
      typeMeta: savedObject.attributes.typeMeta,
      type: savedObject.attributes.type
    }; // Do this before we attempt to update from ES since that call can potentially perform a save

    this.originalBody = this.prepBody();
    await this.updateFromElasticSearch(response, forceFieldRefresh); // Do it after to ensure we have the most up to date information

    this.originalBody = this.prepBody();
    return this;
  }

  migrate(newTitle) {
    return this.savedObjectsClient.update(savedObjectType, this.id, {
      title: newTitle,
      intervalName: null
    }, {
      version: this.version
    }).then(({
      attributes: {
        title,
        intervalName
      }
    }) => {
      this.title = title;
      this.intervalName = intervalName;
    }).then(() => this);
  }

  toSpec() {
    return {
      id: this.id,
      version: this.version,
      title: this.title,
      timeFieldName: this.timeFieldName,
      sourceFilters: this.sourceFilters,
      fields: this.fields.toSpec(),
      typeMeta: this.typeMeta
    };
  } // Get the source filtering configuration for that index.


  getSourceFiltering() {
    return {
      excludes: this.sourceFilters && this.sourceFilters.map(filter => filter.value) || []
    };
  }

  async addScriptedField(name, script, fieldType = 'string', lang) {
    const scriptedFields = this.getScriptedFields();

    const names = _lodash.default.map(scriptedFields, 'name');

    if (_lodash.default.includes(names, name)) {
      throw new _common.DuplicateField(name);
    }

    this.fields.add(new _fields.Field(this, {
      name,
      script,
      fieldType,
      scripted: true,
      lang,
      aggregatable: true,
      filterable: true,
      searchable: true
    }, false, {
      fieldFormats: this.fieldFormats,
      onNotification: this.onNotification
    }));
    await this.save();
  }

  removeScriptedField(field) {
    this.fields.remove(field);
    return this.save();
  }

  async popularizeField(fieldName, unit = 1) {
    /**
     * This function is just used by Discover and it's high likely to be removed in the near future
     * It doesn't use the save function to skip the error message that's displayed when
     * a user adds several columns in a higher frequency that the changes can be persisted to ES
     * resulting in 409 errors
     */
    if (!this.id) return;
    const field = this.fields.getByName(fieldName);

    if (!field) {
      return;
    }

    const count = Math.max((field.count || 0) + unit, 0);

    if (field.count === count) {
      return;
    }

    field.count = count;

    try {
      const res = await this.savedObjectsClient.update(savedObjectType, this.id, this.prepBody(), {
        version: this.version
      });
      this.version = res.version;
    } catch (e) {// no need for an error message here
    }
  }

  getNonScriptedFields() {
    return _lodash.default.filter(this.fields, {
      scripted: false
    });
  }

  getScriptedFields() {
    return _lodash.default.filter(this.fields, {
      scripted: true
    });
  }

  getIndex() {
    if (!this.isUnsupportedTimePattern()) {
      return this.title;
    } // Take a time-based interval index pattern title (like [foo-]YYYY.MM.DD[-bar]) and turn it
    // into the actual index (like foo-*-bar) by replacing anything not inside square brackets
    // with a *.


    const regex = /\[[^\]]*]/g; // Matches text inside brackets

    const splits = this.title.split(regex); // e.g. ['', 'YYYY.MM.DD', ''] from the above example

    const matches = this.title.match(regex) || []; // e.g. ['[foo-]', '[-bar]'] from the above example

    return splits.map((split, i) => {
      const match = i >= matches.length ? '' : matches[i].replace(/[\[\]]/g, '');
      return `${split.length ? '*' : ''}${match}`;
    }).join('');
  }

  isUnsupportedTimePattern() {
    return !!this.intervalName;
  }

  isTimeBased() {
    return !!this.timeFieldName && (!this.fields || !!this.getTimeField());
  }

  isTimeNanosBased() {
    const timeField = this.getTimeField();
    return timeField && timeField.esTypes && timeField.esTypes.indexOf('date_nanos') !== -1;
  }

  isTimeBasedWildcard() {
    return this.isTimeBased() && this.isWildcard();
  }

  getTimeField() {
    if (!this.timeFieldName || !this.fields || !this.fields.getByName) return;
    return this.fields.getByName(this.timeFieldName);
  }

  getFieldByName(name) {
    if (!this.fields || !this.fields.getByName) return;
    return this.fields.getByName(name);
  }

  getAggregationRestrictions() {
    var _this$typeMeta;

    return (_this$typeMeta = this.typeMeta) === null || _this$typeMeta === void 0 ? void 0 : _this$typeMeta.aggs;
  }

  isWildcard() {
    return _lodash.default.includes(this.title, '*');
  }

  prepBody() {
    const body = {}; // serialize json fields

    _lodash.default.forOwn(this.mapping, (fieldMapping, fieldName) => {
      if (!fieldName || this[fieldName] == null) return;
      body[fieldName] = fieldMapping._serialize ? fieldMapping._serialize(this[fieldName]) : this[fieldName];
    });

    return body;
  }

  async create(allowOverride = false) {
    const _create = async duplicateId => {
      if (duplicateId) {
        this.patternCache.clear(duplicateId);
        await this.savedObjectsClient.delete(savedObjectType, duplicateId);
      }

      const body = this.prepBody();
      const response = await this.savedObjectsClient.create(savedObjectType, body, {
        id: this.id
      });
      this.id = response.id;
      return response.id;
    };

    const potentialDuplicateByTitle = await (0, _utils.findByTitle)(this.savedObjectsClient, this.title); // If there is potentially duplicate title, just create it

    if (!potentialDuplicateByTitle) {
      return await _create();
    } // We found a duplicate but we aren't allowing override, show the warn modal


    if (!allowOverride) {
      return false;
    }

    return await _create(potentialDuplicateByTitle.id);
  }

  async save(saveAttempts = 0) {
    if (!this.id) return;
    const body = this.prepBody(); // What keys changed since they last pulled the index pattern

    const originalChangedKeys = Object.keys(body).filter(key => body[key] !== this.originalBody[key]);
    return this.savedObjectsClient.update(savedObjectType, this.id, body, {
      version: this.version
    }).then(resp => {
      this.id = resp.id;
      this.version = resp.version;
    }).catch(err => {
      if (_lodash.default.get(err, 'res.status') === 409 && saveAttempts++ < MAX_ATTEMPTS_TO_RESOLVE_CONFLICTS) {
        const samePattern = new IndexPattern(this.id, {
          getConfig: this.getConfig,
          savedObjectsClient: this.savedObjectsClient,
          apiClient: this.apiClient,
          patternCache: this.patternCache,
          fieldFormats: this.fieldFormats,
          onNotification: this.onNotification,
          onError: this.onError,
          onUnsupportedTimePattern: this.onUnsupportedTimePattern,
          uiSettingsValues: {
            shortDotsEnable: this.shortDotsEnable,
            metaFields: this.metaFields
          }
        });
        return samePattern.init().then(() => {
          // What keys changed from now and what the server returned
          const updatedBody = samePattern.prepBody(); // Build a list of changed keys from the server response
          // and ensure we ignore the key if the server response
          // is the same as the original response (since that is expected
          // if we made a change in that key)

          const serverChangedKeys = Object.keys(updatedBody).filter(key => {
            return updatedBody[key] !== body[key] && this.originalBody[key] !== updatedBody[key];
          });
          let unresolvedCollision = false;

          for (const originalKey of originalChangedKeys) {
            for (const serverKey of serverChangedKeys) {
              if (originalKey === serverKey) {
                unresolvedCollision = true;
                break;
              }
            }
          }

          if (unresolvedCollision) {
            const title = _i18n.i18n.translate('data.indexPatterns.unableWriteLabel', {
              defaultMessage: 'Unable to write index pattern! Refresh the page to get the most up to date changes for this index pattern.'
            });

            this.onNotification({
              title,
              color: 'danger'
            });
            throw err;
          } // Set the updated response on this object


          serverChangedKeys.forEach(key => {
            this[key] = samePattern[key];
          });
          this.version = samePattern.version; // Clear cache

          this.patternCache.clear(this.id); // Try the save again

          return this.save(saveAttempts);
        });
      }

      throw err;
    });
  }

  async _fetchFields() {
    const fields = await this.fieldsFetcher.fetch(this);
    const scripted = this.getScriptedFields();
    const all = fields.concat(scripted);
    await this.initFields(all);
  }

  refreshFields() {
    return this._fetchFields().then(() => this.save()).catch(err => {
      // https://github.com/elastic/kibana/issues/9224
      // This call will attempt to remap fields from the matching
      // ES index which may not actually exist. In that scenario,
      // we still want to notify the user that there is a problem
      // but we do not want to potentially make any pages unusable
      // so do not rethrow the error here
      if (err instanceof _lib.IndexPatternMissingIndices) {
        this.onNotification({
          title: err.message,
          color: 'danger',
          iconType: 'alert'
        });
        return [];
      }

      this.onError(err, {
        title: _i18n.i18n.translate('data.indexPatterns.fetchFieldErrorTitle', {
          defaultMessage: 'Error fetching fields for index pattern {title} (ID: {id})',
          values: {
            id: this.id,
            title: this.title
          }
        })
      });
    });
  }

  toJSON() {
    return this.id;
  }

  toString() {
    return '' + this.toJSON();
  }

}

exports.IndexPattern = IndexPattern;