/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import React, { Fragment } from 'react';
import { i18n } from '@kbn/i18n';
import ReactDOM from 'react-dom';
import ReactMarkdown from 'react-markdown';

function addNotFoundToPath(path, error) {
  return path + (path.indexOf('?') >= 0 ? '&' : '?') + "notFound=".concat(error.savedObjectType, "&notFoundMessage=").concat(error.message);
}
/**
 * Creates an error handler that will redirect to a url when a SavedObjectNotFound
 * error is thrown
 */


export function redirectWhenMissing(_ref) {
  var history = _ref.history,
      navigateToApp = _ref.navigateToApp,
      basePath = _ref.basePath,
      mapping = _ref.mapping,
      toastNotifications = _ref.toastNotifications,
      onBeforeRedirect = _ref.onBeforeRedirect;
  var localMappingObject;

  if (typeof mapping === 'string') {
    localMappingObject = {
      '*': mapping
    };
  } else {
    localMappingObject = mapping;
  }

  return function (error) {
    // if this error is not "404", rethrow
    // we can't check "error instanceof SavedObjectNotFound" since this class can live in a separate bundle
    // and the error will be an instance of other class with the same interface (actually the copy of SavedObjectNotFound class)
    if (!error.savedObjectType) {
      throw error;
    }

    var redirectTarget = localMappingObject[error.savedObjectType] || localMappingObject['*'] || '/';

    if (typeof redirectTarget !== 'string') {
      redirectTarget.path = addNotFoundToPath(redirectTarget.path, error);
    } else {
      redirectTarget = addNotFoundToPath(redirectTarget, error);
    }

    toastNotifications.addWarning({
      title: i18n.translate('kibana_utils.history.savedObjectIsMissingNotificationMessage', {
        defaultMessage: 'Saved object is missing'
      }),
      text: function text(element) {
        ReactDOM.render( /*#__PURE__*/React.createElement(ReactMarkdown, {
          renderers: {
            root: Fragment
          }
        }, error.message), element);
        return function () {
          return ReactDOM.unmountComponentAtNode(element);
        };
      }
    });

    if (onBeforeRedirect) {
      onBeforeRedirect(error);
    }

    if (typeof redirectTarget !== 'string') {
      if (redirectTarget.app === 'kibana') {
        // exception for kibana app because redirect won't work right otherwise
        window.location.href = basePath.prepend("/app/kibana".concat(redirectTarget.path));
      } else {
        navigateToApp(redirectTarget.app, {
          path: redirectTarget.path
        });
      }
    } else {
      history.replace(redirectTarget);
    }
  };
}