/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { createHashHistory } from 'history';
import { setStateToKbnUrl } from './kbn_url_storage';
import { unhashUrl } from './hash_unhash_url';

/**
 * Listens to history changes and optionally to global state changes and updates the nav link url of
 * a given app to point to the last visited page within the app.
 *
 * This includes the following parts:
 * * When the app is currently active, the nav link points to the configurable default url of the app.
 * * When the app is not active the last visited url is set to the nav link.
 * * When a provided observable emits a new value, the state parameter in the url of the nav link is updated
 * as long as the app is not active.
 */
export function createKbnUrlTracker(_ref) {
  var baseUrl = _ref.baseUrl,
      defaultSubUrl = _ref.defaultSubUrl,
      storageKey = _ref.storageKey,
      stateParams = _ref.stateParams,
      navLinkUpdater$ = _ref.navLinkUpdater$,
      toastNotifications = _ref.toastNotifications,
      history = _ref.history,
      getHistory = _ref.getHistory,
      storage = _ref.storage,
      _ref$shouldTrackUrlUp = _ref.shouldTrackUrlUpdate,
      shouldTrackUrlUpdate = _ref$shouldTrackUrlUp === void 0 ? function () {
    return true;
  } : _ref$shouldTrackUrlUp;
  var storageInstance = storage || sessionStorage; // local state storing previous active url to make restore possible

  var previousActiveUrl = ''; // local state storing current listeners and active url

  var activeUrl = '';
  var unsubscribeURLHistory;
  var unsubscribeGlobalState;

  function setNavLink(hash) {
    navLinkUpdater$.next(function () {
      return {
        defaultPath: hash
      };
    });
  }

  function getActiveSubUrl(url) {
    // remove baseUrl prefix (just storing the sub url part)
    return url.substr(baseUrl.length);
  }

  function unsubscribe() {
    if (unsubscribeURLHistory) {
      unsubscribeURLHistory();
      unsubscribeURLHistory = undefined;
    }

    if (unsubscribeGlobalState) {
      unsubscribeGlobalState.forEach(function (sub) {
        return sub.unsubscribe();
      });
      unsubscribeGlobalState = undefined;
    }
  }

  function setActiveUrl(newUrl) {
    var urlWithHashes = baseUrl + '#' + newUrl;
    var urlWithStates = '';

    try {
      urlWithStates = unhashUrl(urlWithHashes);
    } catch (e) {
      toastNotifications.addDanger(e.message);
    }

    previousActiveUrl = activeUrl;
    activeUrl = getActiveSubUrl(urlWithStates || urlWithHashes);
    storageInstance.setItem(storageKey, activeUrl);
  }

  function onMountApp() {
    unsubscribe();
    var historyInstance = history || getHistory && getHistory() || createHashHistory(); // track current hash when within app

    unsubscribeURLHistory = historyInstance.listen(function (location) {
      if (shouldTrackUrlUpdate(location.hash)) {
        setActiveUrl(location.hash.substr(1));
      }
    });
  }

  function onUnmountApp() {
    unsubscribe(); // propagate state updates when in other apps

    unsubscribeGlobalState = stateParams.map(function (_ref2) {
      var stateUpdate$ = _ref2.stateUpdate$,
          kbnUrlKey = _ref2.kbnUrlKey;
      return stateUpdate$.subscribe(function (state) {
        var updatedUrl = setStateToKbnUrl(kbnUrlKey, state, {
          useHash: false
        }, baseUrl + (activeUrl || defaultSubUrl));
        previousActiveUrl = activeUrl; // remove baseUrl prefix (just storing the sub url part)

        activeUrl = getActiveSubUrl(updatedUrl);
        storageInstance.setItem(storageKey, activeUrl);
        setNavLink(activeUrl);
      });
    });
  } // register listeners for unmounted app initially


  onUnmountApp(); // initialize nav link and internal state

  var storedUrl = storageInstance.getItem(storageKey);

  if (storedUrl) {
    activeUrl = storedUrl;
    previousActiveUrl = storedUrl;
    setNavLink(storedUrl);
  }

  return {
    appMounted: function appMounted() {
      onMountApp();
      setNavLink(defaultSubUrl);
    },
    appUnMounted: function appUnMounted() {
      onUnmountApp();
      setNavLink(activeUrl);
    },
    stop: function stop() {
      unsubscribe();
    },
    setActiveUrl: setActiveUrl,
    getActiveUrl: function getActiveUrl() {
      return activeUrl;
    },
    restorePreviousUrl: function restorePreviousUrl() {
      activeUrl = previousActiveUrl;
    }
  };
}