function ownKeys(object, enumerableOnly) { var keys = Object.keys(object); if (Object.getOwnPropertySymbols) { var symbols = Object.getOwnPropertySymbols(object); if (enumerableOnly) symbols = symbols.filter(function (sym) { return Object.getOwnPropertyDescriptor(object, sym).enumerable; }); keys.push.apply(keys, symbols); } return keys; }

function _objectSpread(target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i] != null ? arguments[i] : {}; if (i % 2) { ownKeys(Object(source), true).forEach(function (key) { _defineProperty(target, key, source[key]); }); } else if (Object.getOwnPropertyDescriptors) { Object.defineProperties(target, Object.getOwnPropertyDescriptors(source)); } else { ownKeys(Object(source)).forEach(function (key) { Object.defineProperty(target, key, Object.getOwnPropertyDescriptor(source, key)); }); } } return target; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import { EMPTY } from 'rxjs';
import { tap } from 'rxjs/operators';
import defaultComparator from 'fast-deep-equal';
import { distinctUntilChangedWithInitialValue } from '../../common';
import { applyDiff } from '../state_management/utils/diff_object';
/**
 * @public
 */

/**
 * Utility for syncing application state wrapped in state container
 * with some kind of storage (e.g. URL)
 *
 * Go {@link https://github.com/elastic/kibana/tree/master/src/plugins/kibana_utils/docs/state_sync | here} for a complete guide and examples.
 *
 * @example
 *
 * the simplest use case
 * ```ts
 * const stateStorage = createKbnUrlStateStorage();
 * syncState({
 *   storageKey: '_s',
 *   stateContainer,
 *   stateStorage
 * });
 * ```
 *
 * @example
 * conditionally configuring sync strategy
 * ```ts
 * const stateStorage = createKbnUrlStateStorage({useHash: config.get('state:stateContainerInSessionStorage')})
 * syncState({
 *   storageKey: '_s',
 *   stateContainer,
 *   stateStorage
 * });
 * ```
 *
 * @example
 * implementing custom sync strategy
 * ```ts
 * const localStorageStateStorage = {
 *   set: (storageKey, state) => localStorage.setItem(storageKey, JSON.stringify(state)),
 *   get: (storageKey) => localStorage.getItem(storageKey) ? JSON.parse(localStorage.getItem(storageKey)) : null
 * };
 * syncState({
 *   storageKey: '_s',
 *   stateContainer,
 *   stateStorage: localStorageStateStorage
 * });
 * ```
 *
 * @example
 * transforming state before serialising
 *  Useful for:
 *  * Migration / backward compatibility
 *  * Syncing part of state
 *  * Providing default values
 * ```ts
 * const stateToStorage = (s) => ({ tab: s.tab });
 * syncState({
 *   storageKey: '_s',
 *   stateContainer: {
 *     get: () => stateToStorage(stateContainer.get()),
 *     set: stateContainer.set(({ tab }) => ({ ...stateContainer.get(), tab }),
 *     state$: stateContainer.state$.pipe(map(stateToStorage))
 *   },
 *   stateStorage
 * });
 * ```
 *
 * @param - syncing config {@link IStateSyncConfig}
 * @returns - {@link ISyncStateRef}
 * @public
 */
export function syncState(_ref) {
  var storageKey = _ref.storageKey,
      stateStorage = _ref.stateStorage,
      stateContainer = _ref.stateContainer;
  var subscriptions = [];

  var updateState = function updateState() {
    var newState = stateStorage.get(storageKey);
    var oldState = stateContainer.get();

    if (newState) {
      // apply only real differences to new state
      var mergedState = _objectSpread({}, oldState); // merges into 'mergedState' all differences from newState,
      // but leaves references if they are deeply the same


      var diff = applyDiff(mergedState, newState);

      if (diff.keys.length > 0) {
        stateContainer.set(mergedState);
      }
    } else if (oldState !== newState) {
      // empty new state case
      stateContainer.set(newState);
    }
  };

  var updateStorage = function updateStorage() {
    var newStorageState = stateContainer.get();
    var oldStorageState = stateStorage.get(storageKey);

    if (!defaultComparator(newStorageState, oldStorageState)) {
      stateStorage.set(storageKey, newStorageState);
    }
  };

  var onStateChange$ = stateContainer.state$.pipe(distinctUntilChangedWithInitialValue(stateContainer.get(), defaultComparator), tap(function () {
    return updateStorage();
  }));
  var onStorageChange$ = stateStorage.change$ ? stateStorage.change$(storageKey).pipe(distinctUntilChangedWithInitialValue(stateStorage.get(storageKey), defaultComparator), tap(function () {
    updateState();
  })) : EMPTY;
  return {
    stop: function stop() {
      // if stateStorage has any cancellation logic, then run it
      if (stateStorage.cancel) {
        stateStorage.cancel();
      }

      subscriptions.forEach(function (s) {
        return s.unsubscribe();
      });
      subscriptions.splice(0, subscriptions.length);
    },
    start: function start() {
      if (subscriptions.length > 0) {
        throw new Error("syncState: can't start syncing state, when syncing is in progress");
      }

      subscriptions.push(onStateChange$.subscribe(), onStorageChange$.subscribe());
    }
  };
}
/**
 * @example
 * sync multiple different sync configs
 * ```ts
 * syncStates([
 *   {
 *     storageKey: '_s1',
 *     stateStorage: stateStorage1,
 *     stateContainer: stateContainer1,
 *   },
 *   {
 *     storageKey: '_s2',
 *     stateStorage: stateStorage2,
 *     stateContainer: stateContainer2,
 *   },
 * ]);
 * ```
 * @param stateSyncConfigs - Array of {@link IStateSyncConfig} to sync
 */

export function syncStates(stateSyncConfigs) {
  var syncRefs = stateSyncConfigs.map(function (config) {
    return syncState(config);
  });
  return {
    stop: function stop() {
      syncRefs.forEach(function (s) {
        return s.stop();
      });
    },
    start: function start() {
      syncRefs.forEach(function (s) {
        return s.start();
      });
    }
  };
}