#-----------------------------------------------------------------------------
# Copyright (c) 2012 - 2022, Anaconda, Inc., and Bokeh Contributors.
# All rights reserved.
#
# The full license is in the file LICENSE.txt, distributed with this software.
#-----------------------------------------------------------------------------
""" Provide the ``Alias`` class, for aliasing other properties.

.. note::
    This class should normally be imported from ``bokeh.core.properties``
    instead of directly from this module.

"""

#-----------------------------------------------------------------------------
# Boilerplate
#-----------------------------------------------------------------------------
from __future__ import annotations

import logging # isort:skip
log = logging.getLogger(__name__)

#-----------------------------------------------------------------------------
# Imports
#-----------------------------------------------------------------------------

# Standard library imports
from typing import ClassVar, TypeVar

# Bokeh imports
from .bases import Property
from .descriptors import AliasPropertyDescriptor, PropertyDescriptor

#-----------------------------------------------------------------------------
# Globals and constants
#-----------------------------------------------------------------------------

__all__ = (
    "Alias",
)

T = TypeVar("T")

#-----------------------------------------------------------------------------
# General API
#-----------------------------------------------------------------------------

class Alias(Property[T]): # lgtm [py/missing-call-to-init]
    """
    Alias another property of a model.

    Example:

        Consider the following class definitions:

        .. code-block:: python

            from bokeh.model import Model
            from bokeh.properties import Alias, Int

            class Parent(Model):
                width = Int()

            class Child(Parent):
                plot_width = Alias("width")

    """

    name: str
    _help: str | None

    # Alias is somewhat a quasi-property
    readonly: ClassVar[bool] = False
    serialized: ClassVar[bool] = False
    _default = None

    def __init__(self, aliased_name: str, *, help: str | None = None) -> None:
        self.aliased_name = aliased_name
        self._help = help

    def make_descriptors(self, base_name: str) -> list[PropertyDescriptor[T]]:
        """ Return a list of ``AliasPropertyDescriptor`` instances to
        install on a class, in order to delegate attribute access to this
        property.

        Args:
            aliased_name (str) : the name of the property this alias is for

        Returns:
            list[AliasPropertyDescriptor]

        The descriptors returned are collected by the ``MetaHasProps``
        metaclass and added to ``HasProps`` subclasses during class creation.
        """
        return [ AliasPropertyDescriptor(base_name, self.aliased_name, self) ]

#-----------------------------------------------------------------------------
# Dev API
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Private API
#-----------------------------------------------------------------------------

#-----------------------------------------------------------------------------
# Code
#-----------------------------------------------------------------------------
