"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.migrateTenantIndices = migrateTenantIndices;
exports.setupIndexTemplate = setupIndexTemplate;

var _core = require("../../../../src/core/server/saved_objects/migrations/core");

var _build_index_map = require("../../../../src/core/server/saved_objects/migrations/core/build_index_map");

var _opensearch_dashboards_migrator = require("../../../../src/core/server/saved_objects/migrations/opensearch_dashboards/opensearch_dashboards_migrator");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
async function setupIndexTemplate(esClient, opensearchDashboardsIndex, typeRegistry, logger) {
  const mappings = (0, _core.buildActiveMappings)((0, _opensearch_dashboards_migrator.mergeTypes)(typeRegistry.getAllTypes()));

  try {
    await esClient.indices.putTemplate({
      name: 'tenant_template',
      body: {
        // Setting order to the max value to avoid being overridden by custom templates.
        order: _common.MAX_INTEGER,
        index_patterns: [opensearchDashboardsIndex + '_-*_*', opensearchDashboardsIndex + '_0*_*', opensearchDashboardsIndex + '_1*_*', opensearchDashboardsIndex + '_2*_*', opensearchDashboardsIndex + '_3*_*', opensearchDashboardsIndex + '_4*_*', opensearchDashboardsIndex + '_5*_*', opensearchDashboardsIndex + '_6*_*', opensearchDashboardsIndex + '_7*_*', opensearchDashboardsIndex + '_8*_*', opensearchDashboardsIndex + '_9*_*'],
        settings: {
          number_of_shards: 1
        },
        mappings
      }
    });
  } catch (error) {
    logger.error(error);
    throw error;
  }
}

async function migrateTenantIndices(opensearchDashboardsVersion, migrationClient, securityClient, typeRegistry, serializer, logger) {
  let tenentInfo;

  try {
    tenentInfo = await securityClient.getTenantInfoWithInternalUser();
  } catch (error) {
    logger.error(error);
    throw error;
  } // follows the same approach in opensearch_dashboards_migrator.ts to initiate DocumentMigrator here
  // see: https://tiny.amazon.com/foi0x1wt/githelaskibablobe4c1srccore


  const documentMigrator = new _core.DocumentMigrator({
    opensearchDashboardsVersion,
    typeRegistry,
    log: logger
  });

  for (const indexName of Object.keys(tenentInfo)) {
    const indexMap = (0, _build_index_map.createIndexMap)({
      opensearchDashboardsIndexName: indexName,
      indexMap: (0, _opensearch_dashboards_migrator.mergeTypes)(typeRegistry.getAllTypes()),
      registry: typeRegistry
    }); // follows the same aporach in opensearch_dashboards_mirator.ts to construct IndexMigrator
    // see: https://tiny.amazon.com/9cdcchz5/githelaskibablobe4c1srccore
    //
    // FIXME: hard code batchSize, pollInterval, and scrollDuration for now
    //        they are used to fetched from `migration.xxx` config, which is not accessible from new playform

    const indexMigrator = new _core.IndexMigrator({
      batchSize: 100,
      client: migrationClient,
      documentMigrator,
      index: indexName,
      log: logger,
      mappingProperties: indexMap[indexName].typeMappings,
      pollInterval: 1500,
      // millisec
      scrollDuration: '15m',
      serializer,
      obsoleteIndexTemplatePattern: undefined,
      convertToAliasScript: indexMap[indexName].script
    });

    try {
      await indexMigrator.migrate();
    } catch (error) {
      logger.error(error); // fail early, exit the kibana process
      // NOTE: according to https://github.com/elastic/kibana/issues/41983 ,
      //       PR https://github.com/elastic/kibana/pull/75819 , API to allow plugins
      //       to set status will be available in 7.10, for now, we fail OpenSearchDashboards
      //       process to indicate index migration error. Customer can fix their
      //       tenant indices in ES then restart OpenSearchDashboards.

      process.exit(1);
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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