;;; footnote-greek.el --- Greek lettering for footnote mode

;; Copyright (C) 1997 by Free Software Foundation, Inc.

;; Author: Steven L Baur <steve@xemacs.org>
;; Keywords: mule, news, mail
;; Version: 0.17

;; This file is part of XEmacs.

;; XEmacs is free software; you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; XEmacs is distributed in the hope that it will be useful, but
;; WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;; General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs; see the file COPYING.  If not, write to the Free
;; Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
;; 02111-1307, USA.

;;; Synched up with: Not in FSF

;;; Commentary:

;; This file provides support for Greek lettering in footnote mode.
;; It requires an Emacs or XEmacs with MULE internationalization.

;;; Change Log:

;; Apr-05-1997:	Replicate letters when footnote numbers hit the end of
;;		the alphabet.
;; Apr-18-1997:	Added regexp to the footnote style alist.

;;; Code:

(require 'footnote)

(defconst footnote-greek-lower "-F"-A
  "Greek lower case alphabet.")

(defconst footnote-greek-lower-regexp
  (concat "[" footnote-greek-lower "]")
  "Regexp for lower case Greek.")

(defconst footnote-greek-upper "-F"-A
  "Greek upper case alphabet.")

(defconst footnote-greek-upper-regexp
  (concat "[" footnote-greek-upper "]")
  "Regexp for upper case Greek.")

(defun Footnote-greek-lower (n)
  "Return a letter of Greek lower case.
Wrapping around the alphabet implies successive repetitions of letters."
  (let* ((ltr (mod (1- n) (length footnote-greek-lower)))
	 (rep (/ (1- n) (length footnote-greek-lower)))
	 (chr (char-to-string (aref footnote-greek-lower ltr)))
	 rc)
    (while (>= rep 0)
      (setq rc (concat rc chr))
      (setq rep (1- rep)))
    rc))

(defun Footnote-greek-upper (n)
  "Return a letter of Greek upper case.
Wrapping around the alphabet implies successive repetitions of letters."
  (let* ((ltr (mod (1- n) (length footnote-greek-upper)))
	 (rep (/ (1- n) (length footnote-greek-upper)))
	 (chr (char-to-string (aref footnote-greek-upper ltr)))
	 rc)
    (while (>= rep 0)
      (setq rc (concat rc chr))
      (setq rep (1- rep)))
    rc))

(setq footnote-style-alist
      (append (list `(greek-lower
		      Footnote-greek-lower
		      ,footnote-greek-lower-regexp)) footnote-style-alist))

(setq footnote-style-alist
      (append (list `(greek-upper
		      Footnote-greek-upper
		      ,footnote-greek-upper-regexp)) footnote-style-alist))

(provide 'footnote-greek)

;;; footnote-greek.el ends here
