"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("../../utils/constants");

var _helpers = require("./utils/helpers");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class MonitorService {
  constructor(esDriver) {
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const createResponse = await callAsCurrentUser('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const response = await callAsCurrentUser('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getMonitor', params);

        let monitor = _lodash.default.get(getResponse, 'monitor', null);

        const version = _lodash.default.get(getResponse, '_version', null);

        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);

        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);

        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);

        if (monitor) {
          const {
            callAsCurrentUser
          } = this.esDriver.asScoped(req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);

          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);

          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);

          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);

          if (associated_workflows) {
            monitor = { ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }

          monitor = { ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await callAsCurrentUser('alerting.getWorkflow', params);

        let workflow = _lodash.default.get(getResponse, 'workflow', null);

        const version = _lodash.default.get(getResponse, '_version', null);

        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);

        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);

        workflow.monitor_type = workflow.workflow_type;
        workflow = { ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body; // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?

        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;

        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }

        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const updateResponse = await callAsCurrentUser(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };

        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }

        const should = [];
        const mustList = [must];

        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }

        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }

        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };

        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }

        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser: alertingCallAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const getResponse = await alertingCallAsCurrentUser('alerting.getMonitors', params);

        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);

        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});

        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};

        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });

        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };

        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }

        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const esAggsResponse = await callAsCurrentUser('alerting.getMonitors', aggsParams);

        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return { ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });

        const unusedMonitors = [...monitorMap.values()].map(monitor => ({ ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));

        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]); // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination


        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }

        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitors', err);

        if ((0, _helpers.isIndexNotFoundError)(err)) {
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: []
              }
            }
          });
        }

        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const {
          callAsCurrentUser
        } = this.esDriver.asScoped(req);
        const acknowledgeResponse = await callAsCurrentUser('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const executeResponse = await callAsCurrentUser('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const {
          callAsCurrentUser
        } = await this.esDriver.asScoped(req);
        const results = await callAsCurrentUser('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - searchMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });

    this.esDriver = esDriver;
  }

}

exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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