"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultBackend = void 0;

var _performanceNow = _interopRequireDefault(require("performance-now"));

var _uuid = require("uuid");

var _query_helpers = require("../../common/helpers/notebooks/query_helpers");

var _sample_notebooks = require("../../common/helpers/notebooks/sample_notebooks");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class DefaultBackend {
  constructor() {
    _defineProperty(this, "backend", 'Default Backend');

    _defineProperty(this, "createNewNotebook", newNotebookName => {
      const noteObject = {
        dateCreated: new Date().toISOString(),
        name: newNotebookName,
        dateModified: new Date().toISOString(),
        backend: 'Default',
        paragraphs: []
      };
      return {
        object: noteObject
      };
    });

    _defineProperty(this, "indexNote", async function (client, body) {
      try {
        const response = await client.callAsCurrentUser('observability.createObject', {
          body: {
            notebook: body
          }
        });
        return response;
      } catch (error) {
        throw new Error('Index Doc Error:' + error);
      }
    });

    _defineProperty(this, "updateNote", async function (client, noteId, updateBody) {
      try {
        const response = await client.callAsCurrentUser('observability.updateObjectById', {
          objectId: noteId,
          body: {
            notebook: updateBody
          }
        });
        return response;
      } catch (error) {
        throw new Error('Update Doc Error:' + error);
      }
    });

    _defineProperty(this, "getNote", async function (client, noteId) {
      try {
        const response = await client.callAsCurrentUser('observability.getObjectById', {
          objectId: noteId
        });

        if (response.observabilityObjectList.length === 0) {
          throw 'notebook id not found';
        }

        return response.observabilityObjectList[0];
      } catch (error) {
        throw new Error('Get Doc Error:' + error);
      }
    });

    _defineProperty(this, "viewNotes", async function (client, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.getObject', {
          objectType: 'notebook'
        });
        return response.observabilityObjectList.map(notebook => ({
          path: notebook.notebook.name,
          id: notebook.objectId,
          dateCreated: notebook.notebook.dateCreated,
          dateModified: notebook.notebook.dateModified
        }));
      } catch (error) {
        if (error.body.error.type === 'index_not_found_exception') {
          return [];
        } else throw new Error('View Notebooks Error:' + error);
      }
    });

    _defineProperty(this, "fetchNote", async function (client, noteId, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, noteId);
        return {
          path: noteObject.notebook.name,
          dateCreated: noteObject.notebook.dateCreated,
          dateModified: noteObject.notebook.dateModified,
          paragraphs: noteObject.notebook.paragraphs
        };
      } catch (error) {
        throw new Error('Fetching Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addNote", async function (client, params, _wreckOptions) {
      try {
        const newNotebook = this.createNewNotebook(params.name);
        const opensearchClientResponse = await this.indexNote(client, newNotebook.object);
        return {
          status: 'OK',
          message: opensearchClientResponse,
          body: opensearchClientResponse.objectId
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "addSampleNotes", async function (client, visIds, _wreckOptions) {
      try {
        const notebooks = (0, _sample_notebooks.getSampleNotebooks)(visIds);
        const newNotebooks = [];

        for (let i = 0; i < notebooks.length; i++) {
          const notebook = notebooks[i];
          await this.indexNote(client, notebook.notebook).then(response => {
            newNotebooks.push({
              id: response.objectId,
              name: notebook.notebook.name,
              dateModified: notebook.dateModified,
              dateCreated: notebook.dateCreated
            });
          });
        }

        return {
          status: 'OK',
          message: '',
          body: newNotebooks
        };
      } catch (error) {
        throw new Error('Creating New Notebook Error:' + error);
      }
    });

    _defineProperty(this, "renameNote", async function (client, params, _wreckOptions) {
      try {
        const updateNotebook = {
          name: params.name,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          status: 'OK',
          message: opensearchClientResponse
        };
      } catch (error) {
        throw new Error('Renaming Notebook Error:' + error);
      }
    });

    _defineProperty(this, "cloneNote", async function (client, params, _wreckOptions) {
      try {
        const noteObject = await this.getNote(client, params.noteId);
        const newNotebook = this.createNewNotebook(params.name);
        const cloneNotebook = { ...newNotebook.object
        };
        cloneNotebook.paragraphs = noteObject.notebook.paragraphs;
        const opensearchClientIndexResponse = await this.indexNote(client, cloneNotebook);
        return {
          status: 'OK',
          body: { ...cloneNotebook,
            id: opensearchClientIndexResponse.objectId
          }
        };
      } catch (error) {
        throw new Error('Cloning Notebook Error:' + error);
      }
    });

    _defineProperty(this, "deleteNote", async function (client, noteList, _wreckOptions) {
      try {
        const response = await client.callAsCurrentUser('observability.deleteObjectByIdList', {
          objectIdList: noteList
        });
        return {
          status: 'OK',
          message: response
        };
      } catch (error) {
        throw new Error('Deleting Notebook Error:' + error);
      }
    });

    _defineProperty(this, "exportNote", async function (client, noteId, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, noteId);
        return {
          status: 'OK',
          body: opensearchClientGetResponse
        };
      } catch (error) {
        throw new Error('Export Notebook Error:' + error);
      }
    });

    _defineProperty(this, "importNote", async function (client, noteObj, _wreckOptions) {
      try {
        let newNoteObject = { ...noteObj
        };
        newNoteObject.id = 'note_' + (0, _uuid.v4)();
        newNoteObject.dateCreated = new Date().toISOString();
        newNoteObject.dateModified = new Date().toISOString();
        const opensearchClientIndexResponse = await this.indexNote(client, newNoteObject);
        return {
          status: 'OK',
          message: opensearchClientIndexResponse,
          body: opensearchClientIndexResponse.objectId
        };
      } catch (error) {
        throw new Error('Import Notebook Error:' + error);
      }
    });

    _defineProperty(this, "updateParagraph", function (paragraphs, paragraphId, paragraphInput, paragraphType) {
      try {
        const updatedParagraphs = [];
        paragraphs.map(paragraph => {
          const updatedParagraph = { ...paragraph
          };

          if (paragraph.id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();
            updatedParagraph.input.inputText = paragraphInput;

            if (paragraphType.length > 0) {
              updatedParagraph.input.inputType = paragraphType;
            }
          }

          updatedParagraphs.push(updatedParagraph);
        });
        return updatedParagraphs;
      } catch (error) {
        throw new Error('Update Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "createParagraph", function (paragraphInput, inputType) {
      try {
        let paragraphType = 'MARKDOWN';

        if (inputType === 'VISUALIZATION') {
          paragraphType = 'VISUALIZATION';
        }

        if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          paragraphType = 'OBSERVABILITY_VISUALIZATION';
        }

        if (paragraphInput.substring(0, 3) === '%sql' || paragraphInput.substring(0, 3) === '%ppl') {
          paragraphType = 'QUERY';
        }

        const inputObject = {
          inputType: paragraphType,
          inputText: paragraphInput
        };
        const outputObjects = [{
          outputType: paragraphType,
          result: '',
          execution_time: '0s'
        }];
        const newParagraph = {
          id: 'paragraph_' + (0, _uuid.v4)(),
          dateCreated: new Date().toISOString(),
          dateModified: new Date().toISOString(),
          input: inputObject,
          output: outputObjects
        };
        return newParagraph;
      } catch (error) {
        throw new Error('Create Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "runParagraph", async function (paragraphs, paragraphId, client) {
      try {
        const updatedParagraphs = [];
        let index = 0;

        for (index = 0; index < paragraphs.length; ++index) {
          const startTime = (0, _performanceNow.default)();
          const updatedParagraph = { ...paragraphs[index]
          };

          if (paragraphs[index].id === paragraphId) {
            updatedParagraph.dateModified = new Date().toISOString();

            if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'QUERY',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: paragraphs[index].input.inputText.substring(4, paragraphs[index].input.inputText.length),
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if (paragraphs[index].input.inputType === 'OBSERVABILITY_VISUALIZATION') {
              updatedParagraph.dateModified = new Date().toISOString();
              updatedParagraph.output = [{
                outputType: 'OBSERVABILITY_VISUALIZATION',
                result: '',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
              updatedParagraph.output = [{
                outputType: 'MARKDOWN',
                result: 'Please select an input type (%sql, %ppl, or %md)',
                execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
              }];
            }
          }

          updatedParagraphs.push(updatedParagraph);
        }

        return updatedParagraphs;
      } catch (error) {
        throw new Error('Running Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateRunFetchParagraph", async function (client, request, _wreckOptions) {
      try {
        const scopedClient = client.asScoped(request);
        const params = request.body;
        const opensearchClientGetResponse = await this.getNote(scopedClient, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput, params.paragraphType);
        const updatedOutputParagraphs = await this.runParagraph(updatedInputParagraphs, params.paragraphId, client);
        const updateNotebook = {
          paragraphs: updatedOutputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(scopedClient, params.noteId, updateNotebook);
        let resultParagraph = {};
        let index = 0;

        for (index = 0; index < updatedOutputParagraphs.length; ++index) {
          if (params.paragraphId === updatedOutputParagraphs[index].id) {
            resultParagraph = updatedOutputParagraphs[index];
          }
        }

        return resultParagraph;
      } catch (error) {
        throw new Error('Update/Run Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "updateFetchParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedInputParagraphs = this.updateParagraph(opensearchClientGetResponse.notebook.paragraphs, params.paragraphId, params.paragraphInput);
        const updateNotebook = {
          paragraphs: updatedInputParagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        let resultParagraph = {};
        updatedInputParagraphs.map(paragraph => {
          if (params.paragraphId === paragraph.id) {
            resultParagraph = paragraph;
          }
        });
        return resultParagraph;
      } catch (error) {
        throw new Error('Save Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "addFetchNewParagraph", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const paragraphs = opensearchClientGetResponse.notebook.paragraphs;
        const newParagraph = this.createParagraph(params.paragraphInput, params.inputType);
        paragraphs.splice(params.paragraphIndex, 0, newParagraph);
        const updateNotebook = {
          paragraphs: paragraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return newParagraph;
      } catch (error) {
        throw new Error('add/Fetch Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "deleteFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        const updatedparagraphs = [];

        if (params.paragraphId !== undefined) {
          opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
            if (paragraph.id !== params.paragraphId) {
              updatedparagraphs.push(paragraph);
            }
          });
        }

        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        console.log('error', error);
        throw new Error('Delete Paragraph Error:' + error);
      }
    });

    _defineProperty(this, "clearAllFetchParagraphs", async function (client, params, _wreckOptions) {
      try {
        const opensearchClientGetResponse = await this.getNote(client, params.noteId);
        let updatedparagraphs = [];
        opensearchClientGetResponse.notebook.paragraphs.map((paragraph, index) => {
          let updatedParagraph = { ...paragraph
          };
          updatedParagraph.output = [];
          updatedparagraphs.push(updatedParagraph);
        });
        const updateNotebook = {
          paragraphs: updatedparagraphs,
          dateModified: new Date().toISOString()
        };
        const opensearchClientResponse = await this.updateNote(client, params.noteId, updateNotebook);
        return {
          paragraphs: updatedparagraphs
        };
      } catch (error) {
        throw new Error('Clear Paragraph Error:' + error);
      }
    });
  }

}

exports.DefaultBackend = DefaultBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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