"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AuthenticationType = void 0;

var _opensearch_security_client = require("../../backend/opensearch_security_client");

var _tenant_resolver = require("../../multitenancy/tenant_resolver");

var _errors = require("../../errors");

var _common = require("../../../common");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class AuthenticationType {
  constructor(config, sessionStorageFactory, router, esClient, coreSetup, logger) {
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.router = router;
    this.esClient = esClient;
    this.coreSetup = coreSetup;
    this.logger = logger;

    _defineProperty(this, "type", void 0);

    _defineProperty(this, "securityClient", void 0);

    _defineProperty(this, "authHandler", async (request, response, toolkit) => {
      var _this$config$multiten;

      // skip auth for APIs that do not require auth
      if (this.authNotRequired(request)) {
        return toolkit.authenticated();
      }

      const authState = {}; // if browser request, auth logic is:
      //   1. check if request includes auth header or paramter(e.g. jwt in url params) is present, if so, authenticate with auth header.
      //   2. if auth header not present, check if auth cookie is present, if no cookie, send to authentication workflow
      //   3. verify whether auth cookie is valid, if not valid, send to authentication workflow
      //   4. if cookie is valid, pass to route handlers

      const authHeaders = {};
      let cookie;
      let authInfo; // if this is an REST API call, suppose the request includes necessary auth header
      // see https://www.elastic.co/guide/en/opensearch-dashboards/master/using-api.html

      if (this.requestIncludesAuthInfo(request)) {
        try {
          const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
          Object.assign(authHeaders, additonalAuthHeader);
          authInfo = await this.securityClient.authinfo(request, additonalAuthHeader);
          cookie = this.getCookie(request, authInfo); // set tenant from cookie if exist

          const browserCookie = await this.sessionStorageFactory.asScoped(request).get();

          if (browserCookie && (0, _tenant_resolver.isValidTenant)(browserCookie.tenant)) {
            cookie.tenant = browserCookie.tenant;
          }

          this.sessionStorageFactory.asScoped(request).set(cookie);
        } catch (error) {
          return response.unauthorized({
            body: error.message
          });
        }
      } else {
        // no auth header in request, try cookie
        try {
          cookie = await this.sessionStorageFactory.asScoped(request).get();
        } catch (error) {
          this.logger.error(`Error parsing cookie: ${error.message}`);
          cookie = undefined;
        }

        if (!cookie || !(await this.isValidCookie(cookie, request))) {
          // clear cookie
          this.sessionStorageFactory.asScoped(request).clear(); // for assets, we can still pass it to resource handler as notHandled.
          // marking it as authenticated may result in login pop up when auth challenge
          // is enabled.

          if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
            return toolkit.notHandled();
          } // send to auth workflow


          return this.handleUnauthedRequest(request, response, toolkit);
        } // extend session expiration time


        if (this.config.session.keepalive) {
          cookie.expiryTime = Date.now() + this.config.session.ttl;
          this.sessionStorageFactory.asScoped(request).set(cookie);
        } // cookie is valid
        // build auth header


        const authHeadersFromCookie = this.buildAuthHeaderFromCookie(cookie, request);
        Object.assign(authHeaders, authHeadersFromCookie);
        const additonalAuthHeader = await this.getAdditionalAuthHeader(request);
        Object.assign(authHeaders, additonalAuthHeader);
      } // resolve tenant if necessary


      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        try {
          const tenant = await this.resolveTenant(request, cookie, authHeaders, authInfo); // return 401 if no tenant available

          if (!(0, _tenant_resolver.isValidTenant)(tenant)) {
            return response.badRequest({
              body: 'No available tenant for current user, please reach out to your system administrator'
            });
          }

          authState.selectedTenant = tenant; // set tenant in header

          if (this.config.multitenancy.enabled && this.config.multitenancy.enable_aggregation_view) {
            // Store all saved objects in a single kibana index.
            Object.assign(authHeaders, {
              securitytenant: _common.GLOBAL_TENANT_SYMBOL
            });
          } else {
            Object.assign(authHeaders, {
              securitytenant: tenant
            });
          } // set tenant to cookie


          if (tenant !== cookie.tenant) {
            cookie.tenant = tenant;
            this.sessionStorageFactory.asScoped(request).set(cookie);
          }
        } catch (error) {
          this.logger.error(`Failed to resolve user tenant: ${error}`);

          if (error instanceof _errors.UnauthenticatedError) {
            if (request.url.pathname && request.url.pathname.startsWith('/bundles/')) {
              return toolkit.notHandled();
            }

            this.sessionStorageFactory.asScoped(request).clear();
            return this.handleUnauthedRequest(request, response, toolkit);
          }

          throw error;
        }
      }

      if (!authInfo) {
        authInfo = await this.securityClient.authinfo(request, authHeaders);
      }

      authState.authInfo = authInfo;
      return toolkit.authenticated({
        requestHeaders: authHeaders,
        state: authState
      });
    });

    this.securityClient = new _opensearch_security_client.SecurityClient(esClient);
    this.type = '';
    this.config = config;
  }

  authNotRequired(request) {
    const pathname = request.url.pathname;

    if (!pathname) {
      return false;
    } // allow requests to ignored routes


    if (AuthenticationType.ROUTES_TO_IGNORE.includes(pathname)) {
      return true;
    } // allow requests to routes that doesn't require authentication


    if (this.config.auth.unauthenticated_routes.indexOf(pathname) > -1) {
      // TODO: use opensearch-dashboards server user
      return true;
    }

    return false;
  }

  async resolveTenant(request, cookie, authHeader, authInfo) {
    if (!authInfo) {
      try {
        authInfo = await this.securityClient.authinfo(request, authHeader);
      } catch (error) {
        throw new _errors.UnauthenticatedError(error);
      }
    }

    const dashboardsInfo = await this.securityClient.dashboardsinfo(request, authHeader);
    return (0, _tenant_resolver.resolveTenant)({
      request,
      username: authInfo.user_name,
      roles: authInfo.roles,
      availabeTenants: authInfo.tenants,
      config: this.config,
      cookie,
      multitenancyEnabled: dashboardsInfo.multitenancy_enabled,
      privateTenantEnabled: dashboardsInfo.private_tenant_enabled,
      defaultTenant: dashboardsInfo.default_tenant
    });
  }

  isPageRequest(request) {
    const path = request.url.pathname || '/';
    return path.startsWith('/app/') || path === '/' || path.startsWith('/goto/');
  } // abstract functions for concrete auth types to implement


}

exports.AuthenticationType = AuthenticationType;

_defineProperty(AuthenticationType, "ROUTES_TO_IGNORE", ['/api/core/capabilities', // FIXME: need to figureout how to bypass this API call
'/app/login']);

_defineProperty(AuthenticationType, "REST_API_CALL_HEADER", 'osd-xsrf');
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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