<elements>
  <summary>A binding to the 2D Cairo Graphics rendering API.</summary>
  <remarks>
    <para>
         The Mono.Cairo namespace is a binding to the
	 http://CairoGraphics.org API.
      </para>
    <para>
         Before Mono 1.2.5, this API only exposed the Cairo 1.0 API,
	 starting with Mono 1.2.5, this release exposes both the Cairo
	 1.0 and 1.2 APIs regardless of the underlying Cairo
	 implementation available on the underlying system.
      </para>
    <para>
         If you are planning on using the 1.2 API, you should first
	 check using the <see cref="M:Cairo.CairoAPI.Version" /> whether
	 you are running on a system that supports the 1.2 API before
	 making any 1.2 calls or make it a requirement that the system
	 must have a 1.2 Cairo installed.    Cairo 1.2 APIs have been
	 flagged in the documentation.
      </para>
    <para>
	Cairo draws into surfaces, there are many to choose from
	(in-memory image buffers, PDF surfaces, Postscript surfaces,
	hardware accelerated surfaces, Xlib surfaces and a handful
	more).  Drawing operations are performed on the Cairo surfaces, for example:
      </para>
    <example>
      <code lang="C#">
//
// To compile use: mcs test.cs -r:Mono.Cairo
//
// This generates a file test.png that contains the text
// "Hello, World" rendered with a serif font in blue, on a
// transparent background.
//
using Cairo;

class X {
	static void Main ()
	{
		//
		// Creates an Image-based surface with with data stored in
		// ARGB32 format.  
		//
		ImageSurface surface = new ImageSurface (Format.ARGB32, 240, 80);

		//
		// Create a context, "using" is used here to ensure that the
		// context is Disposed once we are done
		//
		using (Context ctx = new Cairo.Context (surface)){

			// Select a font to draw with
			ctx.SelectFontFace ("serif", FontSlant.Normal, FontWeight.Bold);
			ctx.SetFontSize (32.0);
			
			// Select a color (blue)
			ctx.SetSourceRGB (0, 0, 1);
			
			// Draw
			ctx.MoveTo (10, 50);
			ctx.ShowText ("Hello, World");
			
			surface.WriteToPng ("test.png");
		}
	}
}
	</code>
    </example>
    <para>
	To compile code that uses Mono.Cairo all you have to do is
	reference the Mono.Cairo assembly by passing the -r:Mono.Cairo command
	line option to the compiler or referencing the assembly from
	MonoDevelop.  The code is contained in the "Cairo" namespace.
      </para>
  </remarks>
  <class name="Antialias" fullname="Cairo.Antialias" assembly="Mono.Cairo">
    <summary>Specifies the type of antialiasing to do when rendering text or shapes.</summary>
    <remarks></remarks>
  </class>
  <class name="CairoAPI" fullname="Cairo.CairoAPI" assembly="Mono.Cairo">
    <summary>Provides information about the CairoAPI.</summary>
    <remarks />
  </class>
  <class name="Color" fullname="Cairo.Color" assembly="Mono.Cairo">
    <summary>Color objects holding red, green, blue, and alpha values.</summary>
    <remarks>
      <example>
        <code lang="C#">
static readonly double  M_PI = 3.14159265358979323846;

Cairo.Graphics gr;
/* initialize gr */

/* draw an arc with a color */
gr.Color = new Color (1,0.2,0.2, 0.6);
gr.Arc (0.04, 0.53, 0.02, 0, 2*M_PI);
gr.Arc (0.27, 0.65, 0.02, 0, 2*M_PI);
gr.Fill ();

  </code>
      </example>
    </remarks>
  </class>
  <class name="Content" fullname="Cairo.Content" assembly="Mono.Cairo">
    <summary>Used to describe the content that a surface will contain, whether color information, alpha information (translucence vs. opacity), or both.</summary>
    <remarks></remarks>
  </class>
  <class name="Context" fullname="Cairo.Context" assembly="Mono.Cairo">
    <summary>This is the main object when drawing with Cairo. </summary>
    <remarks>
      <para></para>
      <para>
        <example>
          <code lang="C#">
// This example sets up a surface on a GTK window and draws an arc to it. 
// More examples are available in the samples directory in mcs/classes/Mono.Cairo
//
// Compile like this:
//    mcs sample.cs -pkg:gtk-sharp-2.0 -r:Mono.Cairo
//
using System;
using System.Reflection;
using System.Runtime.InteropServices;
using Cairo;
using Gtk;

public class GtkCairo
{
        static DrawingArea a;

        static void Main ()
        {
                Application.Init ();
                Gtk.Window w = new Gtk.Window ("Mono.Cairo Circles demo");

                a = new CairoGraphic ();

                Box box = new HBox (true, 0);
                box.Add (a);
                w.Add (box);
                w.Resize (500,500);
                w.ShowAll ();

                Application.Run ();
        }


}

public class CairoGraphic : DrawingArea
{
        static readonly double  M_PI = 3.14159265358979323846;

        static void draw (Cairo.Context gr, int width, int height)
        {
                double xc = 0.5;
                double yc = 0.5;
                double radius = 0.4;
                double angle1 = 45.0  * (M_PI/180.0);  /* angles are specified */
                double angle2 = 180.0 * (M_PI/180.0);  /* in radians           */

                gr.Scale (width, height);
                gr.LineWidth = 0.04;

                gr.Arc (xc, yc, radius, angle1, angle2);
                gr.Stroke ();

                /* draw helping lines */
                gr.Color = new Color(1, 0.2, 0.2, 0.6);
                gr.Arc (xc, yc, 0.05, 0, 2*M_PI);
                gr.Fill ();
                gr.LineWidth = 0.03;
                gr.Arc (xc, yc, radius, angle1, angle1);
                gr.LineTo (new PointD(xc, yc));
                gr.Arc (xc, yc, radius, angle2, angle2);
                gr.LineTo (new PointD(xc, yc));
                gr.Stroke ();

        }

        protected override bool OnExposeEvent (Gdk.EventExpose args)
        {
                Gdk.Window win = args.Window;                

		#if OLD_SYSTEMS
		//
		// For old versions of Gtk# (before 2.8), you need the helper class
		// available in gtk-sharp/sample/GtkCairo.cs
		//
                Cairo.Context g = Gdk.Graphics.CreateDrawable (win);
                #else
		//
		// Starting with Gtk 2.8 Gtk has direct support for
		// Cairo, as its built on top of it, on older
		// versions, a helper routine is used
		//
                Context g = Gdk.CairoHelper.Create (args.Window);
		#endif

                int x, y, w, h, d;
                win.GetGeometry(out x, out y, out w, out h, out d);

                draw (g, w, h);
                return true;
        }
}
  </code>
        </example>
      </para>
    </remarks>
  </class>
  <class name="DirectFBSurface" fullname="Cairo.DirectFBSurface" assembly="Mono.Cairo">
    <summary>A DirectFBSurface</summary>
    <remarks />
  </class>
  <class name="Distance" fullname="Cairo.Distance" assembly="Mono.Cairo">
    <summary>Defines a distance between two points.</summary>
    <remarks />
  </class>
  <class name="Extend" fullname="Cairo.Extend" assembly="Mono.Cairo">
    <summary>Defines Pattern Extend types.</summary>
    <remarks>
      <example>
        <code lang="C#">
image = new ImageSurface ("data/e.png");

pattern = new SurfacePattern (image);
pattern.Extend = Cairo.Extend.Repeat;
               
matrix = new Matrix ();
matrix.InitScale (w * 5.0, h * 5.0);

pattern.Matrix = matrix;

gr.Pattern = pattern;
  </code>
      </example>
    </remarks>
  </class>
  <class name="FillRule" fullname="Cairo.FillRule" assembly="Mono.Cairo">
    <summary>Defines a FillRule.</summary>
    <remarks>A FillRule is used to select how paths are filled. For both fill rules, whether or not a point is included in the fill is determined by taking a ray from that point to infinity and looking at intersections with the path. The ray can be in any direction, as long as it doesn't pass through the end point of a segment or have a tricky intersection such as intersecting tangent to the path. (Note that filling is not actually implemented in this way. This is just a description of the rule that is applied.)</remarks>
  </class>
  <class name="Filter" fullname="Cairo.Filter" assembly="Mono.Cairo">
    <summary>Defines Filter types on Graphics and Pattern objects.</summary>
    <remarks />
  </class>
  <class name="FontExtents" fullname="Cairo.FontExtents" assembly="Mono.Cairo">
    <summary>Stores metric information for a font. Values are given in the current user-space coordinate system.</summary>
    <remarks>
      <para>Because font metrics are in user-space coordinates, they
  are mostly, but not entirely, independent of the current
  transformation matrix.</para>
      <para>  If you call <see cref="M:Cairo.Context.Scale (double,&#xA;  double)" /> with parameters (2.0, 2.0), the text will be drawn
  twice as big, but the reported text extents will not be
  doubled. They will change slightly due to hinting (so you can not
  assume that metrics are independent of the transformation matrix),
  but otherwise will remain unchanged.</para>
    </remarks>
  </class>
  <class name="FontFace" fullname="Cairo.FontFace" assembly="Mono.Cairo">
    <summary>A FontFace represents a font at a particular weight, slant, and other characteristics but no size or transformation.</summary>
    <remarks />
  </class>
  <class name="FontOptions" fullname="Cairo.FontOptions" assembly="Mono.Cairo">
    <summary>The FontOptions Class defines how a font should be rendered</summary>
    <remarks></remarks>
  </class>
  <class name="FontSlant" fullname="Cairo.FontSlant" assembly="Mono.Cairo">
    <summary>Defines the possible slants on fonts.</summary>
    <remarks />
  </class>
  <class name="FontType" fullname="Cairo.FontType" assembly="Mono.Cairo">
    <summary>Describes the type of a given font face or scaled font.</summary>
    <remarks></remarks>
  </class>
  <class name="FontWeight" fullname="Cairo.FontWeight" assembly="Mono.Cairo">
    <summary>Defines the weight of the font (normal, bold).</summary>
    <remarks />
  </class>
  <class name="Format" fullname="Cairo.Format" assembly="Mono.Cairo">
    <summary>Used to identify the memory format of image data.</summary>
    <remarks />
  </class>
  <class name="GlitzSurface" fullname="Cairo.GlitzSurface" assembly="Mono.Cairo">
    <summary>Glitz OpenGL surface support</summary>
    <remarks></remarks>
  </class>
  <class name="Glyph" fullname="Cairo.Glyph" assembly="Mono.Cairo">
    <summary>This structure holds information about a single glyph when drawing or measuring text. 
</summary>
    <remarks>
      <para>A font is (in simple terms) a collection of shapes used to draw text. A glyph is one of these shapes. There can be multiple glyphs for a single character (alternates to be used in different contexts, for example), or a glyph can be a ligature of multiple characters. Cairo doesn't expose any way of converting input text into glyphs, so in order to use the Cairo interfaces that take arrays of glyphs, you must directly access the appropriate underlying font system.</para>
      <para>    Note that the offsets given by @x and @y are not cumulative. When drawing or measuring text, each glyph is individually positioned with respect to the overall origin.</para>
    </remarks>
  </class>
  <class name="Gradient" fullname="Cairo.Gradient" assembly="Mono.Cairo">
    <summary>Base class for gradients.</summary>
    <remarks></remarks>
  </class>
  <class name="Graphics" fullname="Cairo.Graphics" assembly="Mono.Cairo">
    <summary>Obsolete version of Cairo.Context, use it instead.</summary>
    <remarks></remarks>
  </class>
  <class name="HintMetrics" fullname="Cairo.HintMetrics" assembly="Mono.Cairo">
    <summary>Specifies whether to hint font metrics.</summary>
    <remarks>Hinting font metrics means quantizing them so that they are integer values in device space. Doing this improves the consistency of letter and line spacing, however it also means that text will be laid out differently at different zoom factors.</remarks>
  </class>
  <class name="HintStyle" fullname="Cairo.HintStyle" assembly="Mono.Cairo">
    <summary>Specifies the type of hinting to do on font outlines.</summary>
    <remarks>Hinting is the process of fitting outlines to the pixel grid in order to improve the appearance of the result. Since hinting outlines involves distorting them, it also reduces the faithfulness to the original outline shapes. Not all of the outline hinting styles are supported by all font backends.</remarks>
  </class>
  <class name="ImageSurface" fullname="Cairo.ImageSurface" assembly="Mono.Cairo">
    <summary>An in-memory image surface</summary>
    <remarks></remarks>
  </class>
  <class name="LinearGradient" fullname="Cairo.LinearGradient" assembly="Mono.Cairo">
    <summary>Extends <see cref="T:Cairo.Gradient" /> by providing a linear gradient along the line defined by (x0, y0) and (x1, y1).  </summary>
    <remarks>
      <para>Before using the gradient pattern, a number of color stops should be defined using AddColorStop() or AddColorStopRgb().
</para>
      <para>
        <example>
          <code lang="C#">
Pattern pat;

pat = new LinearGradient (0.0, 0.0,  0.0, 1.0);
pat.AddColorStop (1, new Color (0, 0, 0, 1) );
pat.AddColorStop (0, new Color (1, 1, 1, 1) );

gr.Rectangle ( new PointD (0, 0), 1, 1);
gr.Pattern =  pat;
gr.Fill ();

pat.Destroy ();
  </code>
        </example>
      </para>
    </remarks>
  </class>
  <class name="LineCap" fullname="Cairo.LineCap" assembly="Mono.Cairo">
    <summary>Defines the various line ending styles.</summary>
    <remarks />
  </class>
  <class name="LineJoin" fullname="Cairo.LineJoin" assembly="Mono.Cairo">
    <summary>Defines the various line join styles</summary>
    <remarks />
  </class>
  <class name="Matrix" fullname="Cairo.Matrix" assembly="Mono.Cairo">
    <summary>
      <para>Defines the Matrix object for transformations and translations.</para>
    </summary>
    <remarks>
      <para>
2D affine transform matrices are 3x3 of the form:
</para>
      <example>
        <code>
[ xx yx tx ]
[ xy yy ty ]
[  0  0   1 ]
  </code>
      </example>
      <para>
As you can see, the bottom row is always [0 0 1]. The Cairo.Matrix implementation deals with the top 6 values only.
</para>
      <example>
        <code lang="C#">
Cairo.Graphics g;
Matrix matrix;

// Assume we initialize the Graphics obj here...

// Get the CTM (Current Transformation Matrix)
Matrix m = g.Matrix;

// Make some changes
m.Translate (translateX, translateY)
m.Scale (scaleX, scaleY);

// set the CTM to the modified matrix
g.Matrix = m;

// do some drawing operations here... 

// Perhaps we now want to compute a few device-to-user point translations...
// If we use Graphics.
// first, get the CTM.
m = g.Matrix;

// invert it
Cairo.Status cs = m.Invert();

if (cs == Cairo.Status.Success) {
	// TIP: Transform as many points as you like...
	m.TransformPoint (ref point1X, ref point1Y);
	m.TransformPoint (ref point2x, ref point2Y);
	// ...
} 
 </code>
      </example>
    </remarks>
  </class>
  <class name="NativeMethods" fullname="Cairo.NativeMethods" assembly="Mono.Cairo">
    <summary>Contains references to the native cairo methods</summary>
    <remarks />
  </class>
  <class name="Operator" fullname="Cairo.Operator" assembly="Mono.Cairo">
    <summary>Contains the Cairo Operators</summary>
    <remarks />
  </class>
  <class name="Path" fullname="Cairo.Path" assembly="Mono.Cairo">
    <summary>The Cairo Path Class</summary>
    <remarks />
  </class>
  <class name="Pattern" fullname="Cairo.Pattern" assembly="Mono.Cairo">
    <summary>Patterns allow you to tile or extend images and objects and allow you to draw things like gradients.</summary>
    <remarks>
      <example>
        <code lang="C#">
/* example assumes the Graphics object has already been set up  and width and height are initialized*/

        static void draw (Cairo.Graphics gr, int width, int height)
        {
                int w, h;
                ImageSurface image;
                Matrix matrix;
                Pattern pattern;

                gr.Scale (width, height);
                gr.LineWidth = 0.04;

                image = new ImageSurface ("data/e.png");
                w = image.Width;
                h = image.Height;

                pattern = new Pattern (image);
                pattern.Extend = Cairo.Extend.Repeat;

                gr.Translate (0.5, 0.5);
                gr.Rotate (Math.PI / 4);
                gr.Scale (1 / Math.Sqrt (2), 1 / Math.Sqrt (2));
                gr.Translate (- 0.5, - 0.5);

                matrix = new Matrix ();
                matrix.InitScale (w * 5.0, h * 5.0);

                pattern.Matrix = matrix;

                gr.Pattern = pattern;

                gr.Rectangle ( new PointD (0, 0),
                               1.0, 1.0);
                gr.Fill ();

                pattern.Destroy ();
                image.Destroy();
        }
  </code>
      </example>
    </remarks>
  </class>
  <class name="PatternType" fullname="Cairo.PatternType" assembly="Mono.Cairo">
    <summary>Describes the type of a given pattern.</summary>
    <remarks></remarks>
  </class>
  <class name="PdfSurface" fullname="Cairo.PdfSurface" assembly="Mono.Cairo">
    <summary>The PDF surface is used to render cairo graphics to Adobe PDF files.</summary>
    <remarks />
  </class>
  <class name="Point" fullname="Cairo.Point" assembly="Mono.Cairo">
    <summary>Class defining a point representd by two int values.</summary>
    <remarks />
  </class>
  <class name="PointD" fullname="Cairo.PointD" assembly="Mono.Cairo">
    <summary>Class defining a point represented by two double values for the x and y coordinates.</summary>
    <remarks>
      <example>
        <code lang="C#">
gr.Rectangle ( new PointD (0, 0), 1, 1);
  </code>
      </example>
    </remarks>
  </class>
  <class name="PSSurface" fullname="Cairo.PSSurface" assembly="Mono.Cairo">
    <summary>The PostScript surface is used to render cairo graphics to Adobe PostScript files.</summary>
    <remarks />
  </class>
  <class name="RadialGradient" fullname="Cairo.RadialGradient" assembly="Mono.Cairo">
    <summary>Extends  <see cref="T:Cairo.Gradient" /> to create a new radial gradient between the two circles defined by (cx0, cxy0, radius0) and (cx1, cy1, radius1).  
</summary>
    <remarks>
      <para>Before using the gradient pattern, a number of color stops
  should be defined using AddColorStop() or AddColorStopRgb().</para>
      <para>
        <example>
          <code lang="C#">
pat = new RadialGradient (0.45, 0.4, 0.1, 0.4,  0.4, 0.5);

pat.AddColorStop (0, new Color (1, 1, 1, 1) );
pat.AddColorStop (1, new Color (0, 0, 0, 1) );
gr.Pattern =  pat;
gr.Arc (0.5, 0.5, 0.3, 0, 2 * Math.PI);
gr.Fill ();

pat.Destroy ();
  </code>
        </example>
      </para>
    </remarks>
  </class>
  <class name="Rectangle" fullname="Cairo.Rectangle" assembly="Mono.Cairo">
    <summary>A representation of a Rectangle.</summary>
    <remarks></remarks>
  </class>
  <class name="ScaledFont" fullname="Cairo.ScaledFont" assembly="Mono.Cairo">
    <summary>To be added.</summary>
    <remarks>To be added.</remarks>
  </class>
  <class name="SolidPattern" fullname="Cairo.SolidPattern" assembly="Mono.Cairo">
    <summary>Extends Pattern to create a solid pattern consisting of a solid color and an optional alpha value. </summary>
    <remarks></remarks>
  </class>
  <class name="Status" fullname="Cairo.Status" assembly="Mono.Cairo">
    <summary>Reports the internal status of the Cairo system. When an error occurs, all further calls to acquire the status will report the error.</summary>
    <remarks />
  </class>
  <class name="SubpixelOrder" fullname="Cairo.SubpixelOrder" assembly="Mono.Cairo">
    <summary>The subpixel order specifies the order of color elements within each pixel on the display device when rendering with an antialiasing mode of Cairo.Antialias.Subpixel.</summary>
    <remarks></remarks>
  </class>
  <class name="Surface" fullname="Cairo.Surface" assembly="Mono.Cairo">
    <summary>Base class for surfaces</summary>
    <remarks>A <see cref="T:Cairo.Surface" /> represents an image, either as the destination of a drawing operation or as source when drawing onto another surface. There are different subtypes of Surface for different drawing backends; for example, <see cref="T:Cairo.ImageSurface" /> creates a bitmap image in memory.</remarks>
  </class>
  <class name="SurfacePattern" fullname="Cairo.SurfacePattern" assembly="Mono.Cairo">
    <summary>To be added.</summary>
    <remarks>To be added.</remarks>
  </class>
  <class name="SurfaceType" fullname="Cairo.SurfaceType" assembly="Mono.Cairo">
    <summary>Describes the type of a given surface.</summary>
    <remarks></remarks>
  </class>
  <class name="SvgSurface" fullname="Cairo.SvgSurface" assembly="Mono.Cairo">
    <summary>To be added.</summary>
    <remarks>To be added.</remarks>
  </class>
  <class name="SvgVersion" fullname="Cairo.SvgVersion" assembly="Mono.Cairo">
    <summary>Describes the version number of the SVG specification that a generated SVG file will conform to.</summary>
    <remarks></remarks>
  </class>
  <class name="TextExtents" fullname="Cairo.TextExtents" assembly="Mono.Cairo">
    <summary>Stores the extents of a single glyph or a string of glyphs in user-space coordinates.</summary>
    <remarks>Because text extents are in user-space coordinates, they are mostly, but not entirely, independent of the current transformation matrix. If you call cairo_scale(cr, 2.0, 2.0), text will be drawn twice as big, but the reported text extents will not be doubled. They will change slightly due to hinting (so you can't assume that metrics are independent of the transformation matrix), but otherwise will remain unchanged.</remarks>
  </class>
  <class name="Win32Surface" fullname="Cairo.Win32Surface" assembly="Mono.Cairo">
    <summary>Microsoft Windows surface support</summary>
    <remarks></remarks>
  </class>
  <class name="XcbSurface" fullname="Cairo.XcbSurface" assembly="Mono.Cairo">
    <summary>To be added.</summary>
    <remarks>To be added.</remarks>
  </class>
  <class name="XlibSurface" fullname="Cairo.XlibSurface" assembly="Mono.Cairo">
    <summary>X Window System rendering using XLib</summary>
    <remarks></remarks>
  </class>
</elements>