# -*- coding: utf-8 -*-

"""Data containing all current emoji
   Extracted from https://unicode.org/Public/emoji/latest/emoji-test.txt
   and https://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-variation-sequences.txt
   See utils/get_codes_from_unicode_emoji_data_files.py

   +----------------+-------------+------------------+-------------------+
   | Emoji Version  |    Date     | Unicode Version  | Data File Comment |
   +----------------+-------------+------------------+-------------------+
   | N/A            | 2010-10-11  | Unicode 6.0      | E0.6              |
   | N/A            | 2014-06-16  | Unicode 7.0      | E0.7              |
   | Emoji 1.0      | 2015-06-09  | Unicode 8.0      | E1.0              |
   | Emoji 2.0      | 2015-11-12  | Unicode 8.0      | E2.0              |
   | Emoji 3.0      | 2016-06-03  | Unicode 9.0      | E3.0              |
   | Emoji 4.0      | 2016-11-22  | Unicode 9.0      | E4.0              |
   | Emoji 5.0      | 2017-06-20  | Unicode 10.0     | E5.0              |
   | Emoji 11.0     | 2018-05-21  | Unicode 11.0     | E11.0             |
   | Emoji 12.0     | 2019-03-05  | Unicode 12.0     | E12.0             |
   | Emoji 12.1     | 2019-10-21  | Unicode 12.1     | E12.1             |
   | Emoji 13.0     | 2020-03-10  | Unicode 13.0     | E13.0             |
   | Emoji 13.1     | 2020-09-15  | Unicode 13.0     | E13.1             |
   | Emoji 14.0     | 2021-09-14  | Unicode 14.0     | E14.0             |

                          http://www.unicode.org/reports/tr51/#Versioning

"""

__all__ = [
    'EMOJI_DATA', 'STATUS', 'LANGUAGES'
]

component = 1
fully_qualified = 2
minimally_qualified = 3
unqualified = 4

STATUS = {
    "component" : component,
    "fully_qualified" : fully_qualified,
    "minimally_qualified" : minimally_qualified,
    "unqualified" : unqualified
}

LANGUAGES = ['en', 'es', 'pt', 'it', 'fr', 'de']

EMOJI_DATA = {
    u'\U0001F947': { # 🥇
        'en' : ':1st_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':goldmedaille:',
        'es': ':medalla_de_oro:',
        'fr': u':médaille_d’or:',
        'pt': ':medalha_de_ouro:',
        'it': u':medaglia_d’oro:'
    },
    u'\U0001F948': { # 🥈
        'en' : ':2nd_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':silbermedaille:',
        'es': ':medalla_de_plata:',
        'fr': u':médaille_d’argent:',
        'pt': ':medalha_de_prata:',
        'it': u':medaglia_d’argento:'
    },
    u'\U0001F949': { # 🥉
        'en' : ':3rd_place_medal:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':bronzemedaille:',
        'es': ':medalla_de_bronce:',
        'fr': u':médaille_de_bronze:',
        'pt': ':medalha_de_bronze:',
        'it': ':medaglia_di_bronzo:'
    },
    u'\U0001F18E': { # 🆎
        'en' : ':AB_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ab:'],
        'de': u':großbuchstaben_ab_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_ab:',
        'fr': ':groupe_sanguin_ab:',
        'pt': u':botão_ab_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_ab:'
    },
    u'\U0001F3E7': { # 🏧
        'en' : ':ATM_sign:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':atm:'],
        'de': ':symbol_geldautomat:',
        'es': u':señal_de_cajero_automático:',
        'fr': ':distributeur_de_billets:',
        'pt': u':símbolo_de_caixa_automático:',
        'it': ':simbolo_dello_sportello_bancomat:'
    },
    u'\U0001F170\U0000FE0F': { # 🅰️
        'en' : ':A_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':a:'],
        'variant': True,
        'de': u':großbuchstabe_a_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_a:',
        'fr': ':groupe_sanguin_a:',
        'pt': u':botão_a_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_a:'
    },
    u'\U0001F170': { # 🅰
        'en' : ':A_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':a:'],
        'variant': True,
        'de': u':großbuchstabe_a_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_a:',
        'fr': ':groupe_sanguin_a:',
        'pt': u':botão_a_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_a:'
    },
    u'\U0001F1E6\U0001F1EB': { # 🇦🇫
        'en' : ':Afghanistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Afghanistan:'],
        'de': ':flagge_afghanistan:',
        'es': u':bandera_afganistán:',
        'fr': ':drapeau_afghanistan:',
        'pt': u':bandeira_afeganistão:',
        'it': ':bandiera_afghanistan:'
    },
    u'\U0001F1E6\U0001F1F1': { # 🇦🇱
        'en' : ':Albania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Albania:'],
        'de': ':flagge_albanien:',
        'es': ':bandera_albania:',
        'fr': ':drapeau_albanie:',
        'pt': u':bandeira_albânia:',
        'it': ':bandiera_albania:'
    },
    u'\U0001F1E9\U0001F1FF': { # 🇩🇿
        'en' : ':Algeria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Algeria:'],
        'de': ':flagge_algerien:',
        'es': ':bandera_argelia:',
        'fr': u':drapeau_algérie:',
        'pt': u':bandeira_argélia:',
        'it': ':bandiera_algeria:'
    },
    u'\U0001F1E6\U0001F1F8': { # 🇦🇸
        'en' : ':American_Samoa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_American_Samoa:'],
        'de': ':flagge_amerikanisch-samoa:',
        'es': ':bandera_samoa_americana:',
        'fr': u':drapeau_samoa_américaines:',
        'pt': ':bandeira_samoa_americana:',
        'it': ':bandiera_samoa_americane:'
    },
    u'\U0001F1E6\U0001F1E9': { # 🇦🇩
        'en' : ':Andorra:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Andorra:'],
        'de': ':flagge_andorra:',
        'es': ':bandera_andorra:',
        'fr': ':drapeau_andorre:',
        'pt': ':bandeira_andorra:',
        'it': ':bandiera_andorra:'
    },
    u'\U0001F1E6\U0001F1F4': { # 🇦🇴
        'en' : ':Angola:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Angola:'],
        'de': ':flagge_angola:',
        'es': ':bandera_angola:',
        'fr': ':drapeau_angola:',
        'pt': ':bandeira_angola:',
        'it': ':bandiera_angola:'
    },
    u'\U0001F1E6\U0001F1EE': { # 🇦🇮
        'en' : ':Anguilla:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Anguilla:'],
        'de': ':flagge_anguilla:',
        'es': ':bandera_anguila:',
        'fr': ':drapeau_anguilla:',
        'pt': ':bandeira_anguila:',
        'it': ':bandiera_anguilla:'
    },
    u'\U0001F1E6\U0001F1F6': { # 🇦🇶
        'en' : ':Antarctica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Antarctica:'],
        'de': ':flagge_antarktis:',
        'es': u':bandera_antártida:',
        'fr': ':drapeau_antarctique:',
        'pt': u':bandeira_antártida:',
        'it': ':bandiera_antartide:'
    },
    u'\U0001F1E6\U0001F1EC': { # 🇦🇬
        'en' : ':Antigua_&_Barbuda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Antigua_&_Barbuda:'],
        'de': ':flagge_antigua_und_barbuda:',
        'es': ':bandera_antigua_y_barbuda:',
        'fr': ':drapeau_antigua-et-barbuda:',
        'pt': u':bandeira_antígua_e_barbuda:',
        'it': ':bandiera_antigua_e_barbuda:'
    },
    u'\U00002652': { # ♒
        'en' : ':Aquarius:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':aquarius:'],
        'variant': True,
        'de': ':wassermann_(sternzeichen):',
        'es': ':acuario:',
        'fr': ':verseau:',
        'pt': u':signo_de_aquário:',
        'it': u':segno_zodiacale_dell’acquario:'
    },
    u'\U0001F1E6\U0001F1F7': { # 🇦🇷
        'en' : ':Argentina:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Argentina:'],
        'de': ':flagge_argentinien:',
        'es': ':bandera_argentina:',
        'fr': ':drapeau_argentine:',
        'pt': ':bandeira_argentina:',
        'it': ':bandiera_argentina:'
    },
    u'\U00002648': { # ♈
        'en' : ':Aries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':aries:'],
        'variant': True,
        'de': ':widder_(sternzeichen):',
        'es': ':aries:',
        'fr': u':bélier_zodiaque:',
        'pt': u':signo_de_áries:',
        'it': u':segno_zodiacale_dell’ariete:'
    },
    u'\U0001F1E6\U0001F1F2': { # 🇦🇲
        'en' : ':Armenia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Armenia:'],
        'de': ':flagge_armenien:',
        'es': ':bandera_armenia:',
        'fr': u':drapeau_arménie:',
        'pt': u':bandeira_armênia:',
        'it': ':bandiera_armenia:'
    },
    u'\U0001F1E6\U0001F1FC': { # 🇦🇼
        'en' : ':Aruba:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Aruba:'],
        'de': ':flagge_aruba:',
        'es': ':bandera_aruba:',
        'fr': ':drapeau_aruba:',
        'pt': ':bandeira_aruba:',
        'it': ':bandiera_aruba:'
    },
    u'\U0001F1E6\U0001F1E8': { # 🇦🇨
        'en' : ':Ascension_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ascension_Island:'],
        'de': ':flagge_ascension:',
        'es': u':bandera_isla_de_la_ascensión:',
        'fr': u':drapeau_île_de_l’ascension:',
        'pt': u':bandeira_ilha_de_ascensão:',
        'it': ':bandiera_isola_ascensione:'
    },
    u'\U0001F1E6\U0001F1FA': { # 🇦🇺
        'en' : ':Australia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Australia:'],
        'de': ':flagge_australien:',
        'es': ':bandera_australia:',
        'fr': ':drapeau_australie:',
        'pt': u':bandeira_austrália:',
        'it': ':bandiera_australia:'
    },
    u'\U0001F1E6\U0001F1F9': { # 🇦🇹
        'en' : ':Austria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Austria:'],
        'de': u':flagge_österreich:',
        'es': ':bandera_austria:',
        'fr': ':drapeau_autriche:',
        'pt': u':bandeira_áustria:',
        'it': ':bandiera_austria:'
    },
    u'\U0001F1E6\U0001F1FF': { # 🇦🇿
        'en' : ':Azerbaijan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Azerbaijan:'],
        'de': ':flagge_aserbaidschan:',
        'es': u':bandera_azerbaiyán:',
        'fr': u':drapeau_azerbaïdjan:',
        'pt': u':bandeira_azerbaijão:',
        'it': ':bandiera_azerbaigian:'
    },
    u'\U0001F519': { # 🔙
        'en' : ':BACK_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':back:'],
        'de': ':back-pfeil:',
        'es': ':flecha_back:',
        'fr': u':flèche_retour:',
        'pt': ':seta_back:',
        'it': ':freccia_back:'
    },
    u'\U0001F171\U0000FE0F': { # 🅱️
        'en' : ':B_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':b:'],
        'variant': True,
        'de': u':großbuchstabe_b_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_b:',
        'fr': ':groupe_sanguin_b:',
        'pt': u':botão_b_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_b:'
    },
    u'\U0001F171': { # 🅱
        'en' : ':B_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':b:'],
        'variant': True,
        'de': u':großbuchstabe_b_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_b:',
        'fr': ':groupe_sanguin_b:',
        'pt': u':botão_b_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_b:'
    },
    u'\U0001F1E7\U0001F1F8': { # 🇧🇸
        'en' : ':Bahamas:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bahamas:'],
        'de': ':flagge_bahamas:',
        'es': ':bandera_bahamas:',
        'fr': ':drapeau_bahamas:',
        'pt': ':bandeira_bahamas:',
        'it': ':bandiera_bahamas:'
    },
    u'\U0001F1E7\U0001F1ED': { # 🇧🇭
        'en' : ':Bahrain:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bahrain:'],
        'de': ':flagge_bahrain:',
        'es': u':bandera_baréin:',
        'fr': u':drapeau_bahreïn:',
        'pt': ':bandeira_bahrein:',
        'it': ':bandiera_bahrein:'
    },
    u'\U0001F1E7\U0001F1E9': { # 🇧🇩
        'en' : ':Bangladesh:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bangladesh:'],
        'de': ':flagge_bangladesch:',
        'es': u':bandera_bangladés:',
        'fr': ':drapeau_bangladesh:',
        'pt': ':bandeira_bangladesh:',
        'it': ':bandiera_bangladesh:'
    },
    u'\U0001F1E7\U0001F1E7': { # 🇧🇧
        'en' : ':Barbados:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Barbados:'],
        'de': ':flagge_barbados:',
        'es': ':bandera_barbados:',
        'fr': ':drapeau_barbade:',
        'pt': ':bandeira_barbados:',
        'it': ':bandiera_barbados:'
    },
    u'\U0001F1E7\U0001F1FE': { # 🇧🇾
        'en' : ':Belarus:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Belarus:'],
        'de': ':flagge_belarus:',
        'es': ':bandera_bielorrusia:',
        'fr': u':drapeau_biélorussie:',
        'pt': u':bandeira_bielorrússia:',
        'it': ':bandiera_bielorussia:'
    },
    u'\U0001F1E7\U0001F1EA': { # 🇧🇪
        'en' : ':Belgium:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Belgium:'],
        'de': ':flagge_belgien:',
        'es': u':bandera_bélgica:',
        'fr': ':drapeau_belgique:',
        'pt': u':bandeira_bélgica:',
        'it': ':bandiera_belgio:'
    },
    u'\U0001F1E7\U0001F1FF': { # 🇧🇿
        'en' : ':Belize:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Belize:'],
        'de': ':flagge_belize:',
        'es': ':bandera_belice:',
        'fr': ':drapeau_belize:',
        'pt': ':bandeira_belize:',
        'it': ':bandiera_belize:'
    },
    u'\U0001F1E7\U0001F1EF': { # 🇧🇯
        'en' : ':Benin:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Benin:'],
        'de': ':flagge_benin:',
        'es': u':bandera_benín:',
        'fr': u':drapeau_bénin:',
        'pt': ':bandeira_benin:',
        'it': ':bandiera_benin:'
    },
    u'\U0001F1E7\U0001F1F2': { # 🇧🇲
        'en' : ':Bermuda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bermuda:'],
        'de': ':flagge_bermuda:',
        'es': ':bandera_bermudas:',
        'fr': ':drapeau_bermudes:',
        'pt': ':bandeira_bermudas:',
        'it': ':bandiera_bermuda:'
    },
    u'\U0001F1E7\U0001F1F9': { # 🇧🇹
        'en' : ':Bhutan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bhutan:'],
        'de': ':flagge_bhutan:',
        'es': u':bandera_bután:',
        'fr': ':drapeau_bhoutan:',
        'pt': u':bandeira_butão:',
        'it': ':bandiera_bhutan:'
    },
    u'\U0001F1E7\U0001F1F4': { # 🇧🇴
        'en' : ':Bolivia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bolivia:'],
        'de': ':flagge_bolivien:',
        'es': ':bandera_bolivia:',
        'fr': ':drapeau_bolivie:',
        'pt': u':bandeira_bolívia:',
        'it': ':bandiera_bolivia:'
    },
    u'\U0001F1E7\U0001F1E6': { # 🇧🇦
        'en' : ':Bosnia_&_Herzegovina:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bosnia_&_Herzegovina:'],
        'de': ':flagge_bosnien_und_herzegowina:',
        'es': ':bandera_bosnia_y_herzegovina:',
        'fr': u':drapeau_bosnie-herzégovine:',
        'pt': u':bandeira_bósnia_e_herzegovina:',
        'it': ':bandiera_bosnia_ed_erzegovina:'
    },
    u'\U0001F1E7\U0001F1FC': { # 🇧🇼
        'en' : ':Botswana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Botswana:'],
        'de': ':flagge_botsuana:',
        'es': ':bandera_botsuana:',
        'fr': ':drapeau_botswana:',
        'pt': ':bandeira_botsuana:',
        'it': ':bandiera_botswana:'
    },
    u'\U0001F1E7\U0001F1FB': { # 🇧🇻
        'en' : ':Bouvet_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bouvet_Island:'],
        'de': ':flagge_bouvetinsel:',
        'es': ':bandera_isla_bouvet:',
        'fr': u':drapeau_île_bouvet:',
        'pt': ':bandeira_ilha_bouvet:',
        'it': ':bandiera_isola_bouvet:'
    },
    u'\U0001F1E7\U0001F1F7': { # 🇧🇷
        'en' : ':Brazil:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Brazil:'],
        'de': ':flagge_brasilien:',
        'es': ':bandera_brasil:',
        'fr': u':drapeau_brésil:',
        'pt': ':bandeira_brasil:',
        'it': ':bandiera_brasile:'
    },
    u'\U0001F1EE\U0001F1F4': { # 🇮🇴
        'en' : ':British_Indian_Ocean_Territory:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_British_Indian_Ocean_Territory:'],
        'de': ':flagge_britisches_territorium_im_indischen_ozean:',
        'es': u':bandera_territorio_británico_del_océano_índico:',
        'fr': u':drapeau_territoire_britannique_de_l’océan_indien:',
        'pt': u':bandeira_território_britânico_do_oceano_índico:',
        'it': u':bandiera_territorio_britannico_dell’oceano_indiano:'
    },
    u'\U0001F1FB\U0001F1EC': { # 🇻🇬
        'en' : ':British_Virgin_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_British_Virgin_Islands:'],
        'de': ':flagge_britische_jungferninseln:',
        'es': u':bandera_islas_vírgenes_británicas:',
        'fr': u':drapeau_îles_vierges_britanniques:',
        'pt': u':bandeira_ilhas_virgens_britânicas:',
        'it': ':bandiera_isole_vergini_britanniche:'
    },
    u'\U0001F1E7\U0001F1F3': { # 🇧🇳
        'en' : ':Brunei:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Brunei:'],
        'de': ':flagge_brunei_darussalam:',
        'es': u':bandera_brunéi:',
        'fr': u':drapeau_brunéi_darussalam:',
        'pt': ':bandeira_brunei:',
        'it': ':bandiera_brunei:'
    },
    u'\U0001F1E7\U0001F1EC': { # 🇧🇬
        'en' : ':Bulgaria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Bulgaria:'],
        'de': ':flagge_bulgarien:',
        'es': ':bandera_bulgaria:',
        'fr': ':drapeau_bulgarie:',
        'pt': u':bandeira_bulgária:',
        'it': ':bandiera_bulgaria:'
    },
    u'\U0001F1E7\U0001F1EB': { # 🇧🇫
        'en' : ':Burkina_Faso:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Burkina_Faso:'],
        'de': ':flagge_burkina_faso:',
        'es': ':bandera_burkina_faso:',
        'fr': ':drapeau_burkina_faso:',
        'pt': ':bandeira_burquina_faso:',
        'it': ':bandiera_burkina_faso:'
    },
    u'\U0001F1E7\U0001F1EE': { # 🇧🇮
        'en' : ':Burundi:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Burundi:'],
        'de': ':flagge_burundi:',
        'es': ':bandera_burundi:',
        'fr': ':drapeau_burundi:',
        'pt': ':bandeira_burundi:',
        'it': ':bandiera_burundi:'
    },
    u'\U0001F191': { # 🆑
        'en' : ':CL_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cl:'],
        'de': u':großbuchstaben_cl_in_rotem_quadrat:',
        'es': ':borrar:',
        'fr': ':bouton_effacer:',
        'pt': u':botão_cl:',
        'it': ':pulsante_cl:'
    },
    u'\U0001F192': { # 🆒
        'en' : ':COOL_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cool:'],
        'de': ':wort_cool_in_blauem_quadrat:',
        'es': u':botón_cool:',
        'fr': ':bouton_cool:',
        'pt': u':botão_cool:',
        'it': ':pulsante_cool:'
    },
    u'\U0001F1F0\U0001F1ED': { # 🇰🇭
        'en' : ':Cambodia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cambodia:'],
        'de': ':flagge_kambodscha:',
        'es': ':bandera_camboya:',
        'fr': ':drapeau_cambodge:',
        'pt': ':bandeira_camboja:',
        'it': ':bandiera_cambogia:'
    },
    u'\U0001F1E8\U0001F1F2': { # 🇨🇲
        'en' : ':Cameroon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cameroon:'],
        'de': ':flagge_kamerun:',
        'es': u':bandera_camerún:',
        'fr': ':drapeau_cameroun:',
        'pt': u':bandeira_camarões:',
        'it': ':bandiera_camerun:'
    },
    u'\U0001F1E8\U0001F1E6': { # 🇨🇦
        'en' : ':Canada:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Canada:'],
        'de': ':flagge_kanada:',
        'es': u':bandera_canadá:',
        'fr': ':drapeau_canada:',
        'pt': u':bandeira_canadá:',
        'it': ':bandiera_canada:'
    },
    u'\U0001F1EE\U0001F1E8': { # 🇮🇨
        'en' : ':Canary_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Canary_Islands:'],
        'de': ':flagge_kanarische_inseln:',
        'es': ':bandera_canarias:',
        'fr': u':drapeau_îles_canaries:',
        'pt': u':bandeira_ilhas_canárias:',
        'it': ':bandiera_isole_canarie:'
    },
    u'\U0000264B': { # ♋
        'en' : ':Cancer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cancer:'],
        'variant': True,
        'de': ':krebs_(sternzeichen):',
        'es': u':cáncer:',
        'fr': ':cancer:',
        'pt': u':signo_de_câncer:',
        'it': ':segno_zodiacale_del_cancro:'
    },
    u'\U0001F1E8\U0001F1FB': { # 🇨🇻
        'en' : ':Cape_Verde:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cape_Verde:'],
        'de': ':flagge_cabo_verde:',
        'es': ':bandera_cabo_verde:',
        'fr': ':drapeau_cap-vert:',
        'pt': ':bandeira_cabo_verde:',
        'it': ':bandiera_capo_verde:'
    },
    u'\U00002651': { # ♑
        'en' : ':Capricorn:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':capricorn:'],
        'variant': True,
        'de': ':steinbock_(sternzeichen):',
        'es': ':capricornio:',
        'fr': ':capricorne:',
        'pt': u':signo_de_capricórnio:',
        'it': ':segno_zodiacale_del_capricorno:'
    },
    u'\U0001F1E7\U0001F1F6': { # 🇧🇶
        'en' : ':Caribbean_Netherlands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Caribbean_Netherlands:'],
        'de': ':flagge_bonaire,_sint_eustatius_und_saba:',
        'es': u':bandera_caribe_neerlandés:',
        'fr': u':drapeau_pays-bas_caribéens:',
        'pt': u':bandeira_países_baixos_caribenhos:',
        'it': ':bandiera_caraibi_olandesi:'
    },
    u'\U0001F1F0\U0001F1FE': { # 🇰🇾
        'en' : ':Cayman_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cayman_Islands:'],
        'de': ':flagge_kaimaninseln:',
        'es': u':bandera_islas_caimán:',
        'fr': u':drapeau_îles_caïmans:',
        'pt': ':bandeira_ilhas_cayman:',
        'it': ':bandiera_isole_cayman:'
    },
    u'\U0001F1E8\U0001F1EB': { # 🇨🇫
        'en' : ':Central_African_Republic:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Central_African_Republic:'],
        'de': ':flagge_zentralafrikanische_republik:',
        'es': u':bandera_república_centroafricana:',
        'fr': u':drapeau_république_centrafricaine:',
        'pt': u':bandeira_república_centro-africana:',
        'it': ':bandiera_repubblica_centrafricana:'
    },
    u'\U0001F1EA\U0001F1E6': { # 🇪🇦
        'en' : ':Ceuta_&_Melilla:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ceuta_&_Melilla:'],
        'de': ':flagge_ceuta_und_melilla:',
        'es': ':bandera_ceuta_y_melilla:',
        'fr': ':drapeau_ceuta_et_melilla:',
        'pt': ':bandeira_ceuta_e_melilla:',
        'it': ':bandiera_ceuta_e_melilla:'
    },
    u'\U0001F1F9\U0001F1E9': { # 🇹🇩
        'en' : ':Chad:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Chad:'],
        'de': ':flagge_tschad:',
        'es': ':bandera_chad:',
        'fr': ':drapeau_tchad:',
        'pt': ':bandeira_chade:',
        'it': ':bandiera_ciad:'
    },
    u'\U0001F1E8\U0001F1F1': { # 🇨🇱
        'en' : ':Chile:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Chile:'],
        'de': ':flagge_chile:',
        'es': ':bandera_chile:',
        'fr': ':drapeau_chili:',
        'pt': ':bandeira_chile:',
        'it': ':bandiera_cile:'
    },
    u'\U0001F1E8\U0001F1F3': { # 🇨🇳
        'en' : ':China:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_China:'],
        'de': ':flagge_china:',
        'es': ':bandera_china:',
        'fr': ':drapeau_chine:',
        'pt': ':bandeira_china:',
        'it': ':bandiera_cina:'
    },
    u'\U0001F1E8\U0001F1FD': { # 🇨🇽
        'en' : ':Christmas_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Christmas_Island:'],
        'de': ':flagge_weihnachtsinsel:',
        'es': ':bandera_isla_de_navidad:',
        'fr': u':drapeau_île_christmas:',
        'pt': ':bandeira_ilha_christmas:',
        'it': ':bandiera_isola_christmas:'
    },
    u'\U0001F384': { # 🎄
        'en' : ':Christmas_tree:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':christmas_tree:'],
        'de': ':weihnachtsbaum:',
        'es': u':árbol_de_navidad:',
        'fr': u':sapin_de_noël:',
        'pt': u':árvore_de_natal:',
        'it': ':albero_di_natale:'
    },
    u'\U0001F1E8\U0001F1F5': { # 🇨🇵
        'en' : ':Clipperton_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Clipperton_Island:'],
        'de': ':flagge_clipperton-insel:',
        'es': ':bandera_isla_clipperton:',
        'fr': u':drapeau_île_clipperton:',
        'pt': ':bandeira_ilha_de_clipperton:',
        'it': ':bandiera_isola_di_clipperton:'
    },
    u'\U0001F1E8\U0001F1E8': { # 🇨🇨
        'en' : ':Cocos_(Keeling)_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cocos__Islands:'],
        'de': ':flagge_kokosinseln:',
        'es': ':bandera_islas_cocos:',
        'fr': u':drapeau_îles_cocos:',
        'pt': ':bandeira_ilhas_cocos_(keeling):',
        'it': ':bandiera_isole_cocos_(keeling):'
    },
    u'\U0001F1E8\U0001F1F4': { # 🇨🇴
        'en' : ':Colombia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Colombia:'],
        'de': ':flagge_kolumbien:',
        'es': ':bandera_colombia:',
        'fr': ':drapeau_colombie:',
        'pt': u':bandeira_colômbia:',
        'it': ':bandiera_colombia:'
    },
    u'\U0001F1F0\U0001F1F2': { # 🇰🇲
        'en' : ':Comoros:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Comoros:'],
        'de': ':flagge_komoren:',
        'es': ':bandera_comoras:',
        'fr': ':drapeau_comores:',
        'pt': ':bandeira_comores:',
        'it': ':bandiera_comore:'
    },
    u'\U0001F1E8\U0001F1EC': { # 🇨🇬
        'en' : ':Congo_-_Brazzaville:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Congo____Brazzaville:'],
        'de': ':flagge_kongo-brazzaville:',
        'es': ':bandera_congo:',
        'fr': ':drapeau_congo-brazzaville:',
        'pt': u':bandeira_república_do_congo:',
        'it': ':bandiera_congo-brazzaville:'
    },
    u'\U0001F1E8\U0001F1E9': { # 🇨🇩
        'en' : ':Congo_-_Kinshasa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Congo____Kinshasa:'],
        'de': ':flagge_kongo-kinshasa:',
        'es': u':bandera_república_democrática_del_congo:',
        'fr': ':drapeau_congo-kinshasa:',
        'pt': u':bandeira_congo_–_kinshasa:',
        'it': u':bandiera_congo_–_kinshasa:'
    },
    u'\U0001F1E8\U0001F1F0': { # 🇨🇰
        'en' : ':Cook_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cook_Islands:'],
        'de': ':flagge_cookinseln:',
        'es': ':bandera_islas_cook:',
        'fr': u':drapeau_îles_cook:',
        'pt': ':bandeira_ilhas_cook:',
        'it': ':bandiera_isole_cook:'
    },
    u'\U0001F1E8\U0001F1F7': { # 🇨🇷
        'en' : ':Costa_Rica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Costa_Rica:'],
        'de': ':flagge_costa_rica:',
        'es': ':bandera_costa_rica:',
        'fr': ':drapeau_costa_rica:',
        'pt': ':bandeira_costa_rica:',
        'it': ':bandiera_costa_rica:'
    },
    u'\U0001F1ED\U0001F1F7': { # 🇭🇷
        'en' : ':Croatia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Croatia:'],
        'de': ':flagge_kroatien:',
        'es': ':bandera_croacia:',
        'fr': ':drapeau_croatie:',
        'pt': u':bandeira_croácia:',
        'it': ':bandiera_croazia:'
    },
    u'\U0001F1E8\U0001F1FA': { # 🇨🇺
        'en' : ':Cuba:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cuba:'],
        'de': ':flagge_kuba:',
        'es': ':bandera_cuba:',
        'fr': ':drapeau_cuba:',
        'pt': ':bandeira_cuba:',
        'it': ':bandiera_cuba:'
    },
    u'\U0001F1E8\U0001F1FC': { # 🇨🇼
        'en' : u':Curaçao:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Curaçao:'],
        'de': u':flagge_curaçao:',
        'es': ':bandera_curazao:',
        'fr': u':drapeau_curaçao:',
        'pt': u':bandeira_curaçao:',
        'it': u':bandiera_curaçao:'
    },
    u'\U0001F1E8\U0001F1FE': { # 🇨🇾
        'en' : ':Cyprus:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Cyprus:'],
        'de': ':flagge_zypern:',
        'es': ':bandera_chipre:',
        'fr': ':drapeau_chypre:',
        'pt': ':bandeira_chipre:',
        'it': ':bandiera_cipro:'
    },
    u'\U0001F1E8\U0001F1FF': { # 🇨🇿
        'en' : ':Czechia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Czech_Republic:'],
        'de': ':flagge_tschechien:',
        'es': ':bandera_chequia:',
        'fr': u':drapeau_tchéquie:',
        'pt': u':bandeira_tchéquia:',
        'it': ':bandiera_cechia:'
    },
    u'\U0001F1E8\U0001F1EE': { # 🇨🇮
        'en' : u':Côte_d’Ivoire:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Côte_d’Ivoire:'],
        'de': u':flagge_côte_d’ivoire:',
        'es': u':bandera_côte_d’ivoire:',
        'fr': u':drapeau_côte_d’ivoire:',
        'pt': ':bandeira_costa_do_marfim:',
        'it': u':bandiera_costa_d’avorio:'
    },
    u'\U0001F1E9\U0001F1F0': { # 🇩🇰
        'en' : ':Denmark:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Denmark:'],
        'de': u':flagge_dänemark:',
        'es': ':bandera_dinamarca:',
        'fr': ':drapeau_danemark:',
        'pt': ':bandeira_dinamarca:',
        'it': ':bandiera_danimarca:'
    },
    u'\U0001F1E9\U0001F1EC': { # 🇩🇬
        'en' : ':Diego_Garcia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Diego_Garcia:'],
        'de': ':flagge_diego_garcia:',
        'es': u':bandera_diego_garcía:',
        'fr': ':drapeau_diego_garcia:',
        'pt': ':bandeira_diego_garcia:',
        'it': ':bandiera_diego_garcia:'
    },
    u'\U0001F1E9\U0001F1EF': { # 🇩🇯
        'en' : ':Djibouti:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Djibouti:'],
        'de': ':flagge_dschibuti:',
        'es': ':bandera_yibuti:',
        'fr': ':drapeau_djibouti:',
        'pt': ':bandeira_djibuti:',
        'it': ':bandiera_gibuti:'
    },
    u'\U0001F1E9\U0001F1F2': { # 🇩🇲
        'en' : ':Dominica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Dominica:'],
        'de': ':flagge_dominica:',
        'es': ':bandera_dominica:',
        'fr': ':drapeau_dominique:',
        'pt': ':bandeira_dominica:',
        'it': ':bandiera_dominica:'
    },
    u'\U0001F1E9\U0001F1F4': { # 🇩🇴
        'en' : ':Dominican_Republic:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Dominican_Republic:'],
        'de': ':flagge_dominikanische_republik:',
        'es': u':bandera_república_dominicana:',
        'fr': u':drapeau_république_dominicaine:',
        'pt': u':bandeira_república_dominicana:',
        'it': ':bandiera_repubblica_dominicana:'
    },
    u'\U0001F51A': { # 🔚
        'en' : ':END_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':end:'],
        'de': ':end-pfeil:',
        'es': ':flecha_end:',
        'fr': u':flèche_fin:',
        'pt': ':seta_end:',
        'it': ':freccia_end:'
    },
    u'\U0001F1EA\U0001F1E8': { # 🇪🇨
        'en' : ':Ecuador:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ecuador:'],
        'de': ':flagge_ecuador:',
        'es': ':bandera_ecuador:',
        'fr': u':drapeau_équateur:',
        'pt': ':bandeira_equador:',
        'it': ':bandiera_ecuador:'
    },
    u'\U0001F1EA\U0001F1EC': { # 🇪🇬
        'en' : ':Egypt:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Egypt:'],
        'de': u':flagge_ägypten:',
        'es': ':bandera_egipto:',
        'fr': u':drapeau_égypte:',
        'pt': ':bandeira_egito:',
        'it': ':bandiera_egitto:'
    },
    u'\U0001F1F8\U0001F1FB': { # 🇸🇻
        'en' : ':El_Salvador:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_El_Salvador:'],
        'de': ':flagge_el_salvador:',
        'es': ':bandera_el_salvador:',
        'fr': ':drapeau_salvador:',
        'pt': ':bandeira_el_salvador:',
        'it': ':bandiera_el_salvador:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0065\U000E006E\U000E0067\U000E007F': { # 🏴󠁧󠁢󠁥󠁮󠁧󠁿
        'en' : ':England:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':england:'],
        'de': ':flagge_england:',
        'es': ':bandera_inglaterra:',
        'fr': ':drapeau_angleterre:',
        'pt': ':bandeira_inglaterra:',
        'it': ':bandiera_inghilterra:'
    },
    u'\U0001F1EC\U0001F1F6': { # 🇬🇶
        'en' : ':Equatorial_Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Equatorial_Guinea:'],
        'de': u':flagge_äquatorialguinea:',
        'es': ':bandera_guinea_ecuatorial:',
        'fr': u':drapeau_guinée_équatoriale:',
        'pt': u':bandeira_guiné_equatorial:',
        'it': ':bandiera_guinea_equatoriale:'
    },
    u'\U0001F1EA\U0001F1F7': { # 🇪🇷
        'en' : ':Eritrea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Eritrea:'],
        'de': ':flagge_eritrea:',
        'es': ':bandera_eritrea:',
        'fr': u':drapeau_érythrée:',
        'pt': ':bandeira_eritreia:',
        'it': ':bandiera_eritrea:'
    },
    u'\U0001F1EA\U0001F1EA': { # 🇪🇪
        'en' : ':Estonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Estonia:'],
        'de': ':flagge_estland:',
        'es': ':bandera_estonia:',
        'fr': ':drapeau_estonie:',
        'pt': u':bandeira_estônia:',
        'it': ':bandiera_estonia:'
    },
    u'\U0001F1F8\U0001F1FF': { # 🇸🇿
        'en' : ':Eswatini:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Swaziland:'],
        'de': ':flagge_eswatini:',
        'es': ':bandera_esuatini:',
        'fr': ':drapeau_eswatini:',
        'pt': u':bandeira_essuatíni:',
        'it': ':bandiera_swaziland:'
    },
    u'\U0001F1EA\U0001F1F9': { # 🇪🇹
        'en' : ':Ethiopia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ethiopia:'],
        'de': u':flagge_äthiopien:',
        'es': u':bandera_etiopía:',
        'fr': u':drapeau_éthiopie:',
        'pt': u':bandeira_etiópia:',
        'it': ':bandiera_etiopia:'
    },
    u'\U0001F1EA\U0001F1FA': { # 🇪🇺
        'en' : ':European_Union:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_European_Union:'],
        'de': u':flagge_europäische_union:',
        'es': u':bandera_unión_europea:',
        'fr': u':drapeau_union_européenne:',
        'pt': u':bandeira_união_europeia:',
        'it': ':bandiera_unione_europea:'
    },
    u'\U0001F193': { # 🆓
        'en' : ':FREE_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':free:'],
        'de': ':wort_free_in_blauem_quadrat:',
        'es': u':botón_free:',
        'fr': ':bouton_gratuit:',
        'pt': u':botão_free:',
        'it': ':pulsante_free:'
    },
    u'\U0001F1EB\U0001F1F0': { # 🇫🇰
        'en' : ':Falkland_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Falkland_Islands:'],
        'de': ':flagge_falklandinseln:',
        'es': ':bandera_islas_malvinas:',
        'fr': u':drapeau_îles_malouines:',
        'pt': ':bandeira_ilhas_malvinas:',
        'it': ':bandiera_isole_falkland:'
    },
    u'\U0001F1EB\U0001F1F4': { # 🇫🇴
        'en' : ':Faroe_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Faroe_Islands:'],
        'de': u':flagge_färöer:',
        'es': ':bandera_islas_feroe:',
        'fr': u':drapeau_îles_féroé:',
        'pt': ':bandeira_ilhas_faroe:',
        'it': u':bandiera_isole_fær_øer:'
    },
    u'\U0001F1EB\U0001F1EF': { # 🇫🇯
        'en' : ':Fiji:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Fiji:'],
        'de': ':flagge_fidschi:',
        'es': ':bandera_fiyi:',
        'fr': ':drapeau_fidji:',
        'pt': ':bandeira_fiji:',
        'it': ':bandiera_figi:'
    },
    u'\U0001F1EB\U0001F1EE': { # 🇫🇮
        'en' : ':Finland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Finland:'],
        'de': ':flagge_finnland:',
        'es': ':bandera_finlandia:',
        'fr': ':drapeau_finlande:',
        'pt': u':bandeira_finlândia:',
        'it': ':bandiera_finlandia:'
    },
    u'\U0001F1EB\U0001F1F7': { # 🇫🇷
        'en' : ':France:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_France:'],
        'de': ':flagge_frankreich:',
        'es': ':bandera_francia:',
        'fr': ':drapeau_france:',
        'pt': u':bandeira_frança:',
        'it': ':bandiera_francia:'
    },
    u'\U0001F1EC\U0001F1EB': { # 🇬🇫
        'en' : ':French_Guiana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_French_Guiana:'],
        'de': u':flagge_französisch-guayana:',
        'es': ':bandera_guayana_francesa:',
        'fr': u':drapeau_guyane_française:',
        'pt': ':bandeira_guiana_francesa:',
        'it': ':bandiera_guyana_francese:'
    },
    u'\U0001F1F5\U0001F1EB': { # 🇵🇫
        'en' : ':French_Polynesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_French_Polynesia:'],
        'de': u':flagge_französisch-polynesien:',
        'es': ':bandera_polinesia_francesa:',
        'fr': u':drapeau_polynésie_française:',
        'pt': u':bandeira_polinésia_francesa:',
        'it': ':bandiera_polinesia_francese:'
    },
    u'\U0001F1F9\U0001F1EB': { # 🇹🇫
        'en' : ':French_Southern_Territories:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_French_Southern_Territories:'],
        'de': u':flagge_französische_süd-_und_antarktisgebiete:',
        'es': ':bandera_territorios_australes_franceses:',
        'fr': u':drapeau_terres_australes_françaises:',
        'pt': u':bandeira_territórios_franceses_do_sul:',
        'it': ':bandiera_terre_australi_francesi:'
    },
    u'\U0001F1EC\U0001F1E6': { # 🇬🇦
        'en' : ':Gabon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Gabon:'],
        'de': ':flagge_gabun:',
        'es': u':bandera_gabón:',
        'fr': ':drapeau_gabon:',
        'pt': u':bandeira_gabão:',
        'it': ':bandiera_gabon:'
    },
    u'\U0001F1EC\U0001F1F2': { # 🇬🇲
        'en' : ':Gambia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Gambia:'],
        'de': ':flagge_gambia:',
        'es': ':bandera_gambia:',
        'fr': ':drapeau_gambie:',
        'pt': u':bandeira_gâmbia:',
        'it': ':bandiera_gambia:'
    },
    u'\U0000264A': { # ♊
        'en' : ':Gemini:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':gemini:'],
        'variant': True,
        'de': ':zwillinge_(sternzeichen):',
        'es': u':géminis:',
        'fr': u':gémeaux:',
        'pt': u':signo_de_gêmeos:',
        'it': ':segno_zodiacale_dei_gemelli:'
    },
    u'\U0001F1EC\U0001F1EA': { # 🇬🇪
        'en' : ':Georgia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Georgia:'],
        'de': ':flagge_georgien:',
        'es': ':bandera_georgia:',
        'fr': u':drapeau_géorgie:',
        'pt': u':bandeira_geórgia:',
        'it': ':bandiera_georgia:'
    },
    u'\U0001F1E9\U0001F1EA': { # 🇩🇪
        'en' : ':Germany:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_Germany:'],
        'de': ':flagge_deutschland:',
        'es': ':bandera_alemania:',
        'fr': ':drapeau_allemagne:',
        'pt': ':bandeira_alemanha:',
        'it': ':bandiera_germania:'
    },
    u'\U0001F1EC\U0001F1ED': { # 🇬🇭
        'en' : ':Ghana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ghana:'],
        'de': ':flagge_ghana:',
        'es': ':bandera_ghana:',
        'fr': ':drapeau_ghana:',
        'pt': ':bandeira_gana:',
        'it': ':bandiera_ghana:'
    },
    u'\U0001F1EC\U0001F1EE': { # 🇬🇮
        'en' : ':Gibraltar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Gibraltar:'],
        'de': ':flagge_gibraltar:',
        'es': ':bandera_gibraltar:',
        'fr': ':drapeau_gibraltar:',
        'pt': ':bandeira_gibraltar:',
        'it': ':bandiera_gibilterra:'
    },
    u'\U0001F1EC\U0001F1F7': { # 🇬🇷
        'en' : ':Greece:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Greece:'],
        'de': ':flagge_griechenland:',
        'es': ':bandera_grecia:',
        'fr': u':drapeau_grèce:',
        'pt': u':bandeira_grécia:',
        'it': ':bandiera_grecia:'
    },
    u'\U0001F1EC\U0001F1F1': { # 🇬🇱
        'en' : ':Greenland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Greenland:'],
        'de': u':flagge_grönland:',
        'es': ':bandera_groenlandia:',
        'fr': ':drapeau_groenland:',
        'pt': u':bandeira_groenlândia:',
        'it': ':bandiera_groenlandia:'
    },
    u'\U0001F1EC\U0001F1E9': { # 🇬🇩
        'en' : ':Grenada:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Grenada:'],
        'de': ':flagge_grenada:',
        'es': ':bandera_granada:',
        'fr': ':drapeau_grenade:',
        'pt': ':bandeira_granada:',
        'it': ':bandiera_grenada:'
    },
    u'\U0001F1EC\U0001F1F5': { # 🇬🇵
        'en' : ':Guadeloupe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guadeloupe:'],
        'de': ':flagge_guadeloupe:',
        'es': ':bandera_guadalupe:',
        'fr': ':drapeau_guadeloupe:',
        'pt': ':bandeira_guadalupe:',
        'it': ':bandiera_guadalupa:'
    },
    u'\U0001F1EC\U0001F1FA': { # 🇬🇺
        'en' : ':Guam:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guam:'],
        'de': ':flagge_guam:',
        'es': ':bandera_guam:',
        'fr': ':drapeau_guam:',
        'pt': ':bandeira_guam:',
        'it': ':bandiera_guam:'
    },
    u'\U0001F1EC\U0001F1F9': { # 🇬🇹
        'en' : ':Guatemala:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guatemala:'],
        'de': ':flagge_guatemala:',
        'es': ':bandera_guatemala:',
        'fr': ':drapeau_guatemala:',
        'pt': ':bandeira_guatemala:',
        'it': ':bandiera_guatemala:'
    },
    u'\U0001F1EC\U0001F1EC': { # 🇬🇬
        'en' : ':Guernsey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guernsey:'],
        'de': ':flagge_guernsey:',
        'es': ':bandera_guernsey:',
        'fr': ':drapeau_guernesey:',
        'pt': ':bandeira_guernsey:',
        'it': ':bandiera_guernsey:'
    },
    u'\U0001F1EC\U0001F1F3': { # 🇬🇳
        'en' : ':Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guinea:'],
        'de': ':flagge_guinea:',
        'es': ':bandera_guinea:',
        'fr': u':drapeau_guinée:',
        'pt': u':bandeira_guiné:',
        'it': ':bandiera_guinea:'
    },
    u'\U0001F1EC\U0001F1FC': { # 🇬🇼
        'en' : ':Guinea-Bissau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guinea__Bissau:'],
        'de': ':flagge_guinea-bissau:',
        'es': u':bandera_guinea-bisáu:',
        'fr': u':drapeau_guinée-bissau:',
        'pt': u':bandeira_guiné-bissau:',
        'it': ':bandiera_guinea-bissau:'
    },
    u'\U0001F1EC\U0001F1FE': { # 🇬🇾
        'en' : ':Guyana:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Guyana:'],
        'de': ':flagge_guyana:',
        'es': ':bandera_guyana:',
        'fr': ':drapeau_guyana:',
        'pt': ':bandeira_guiana:',
        'it': ':bandiera_guyana:'
    },
    u'\U0001F1ED\U0001F1F9': { # 🇭🇹
        'en' : ':Haiti:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Haiti:'],
        'de': ':flagge_haiti:',
        'es': u':bandera_haití:',
        'fr': u':drapeau_haïti:',
        'pt': ':bandeira_haiti:',
        'it': ':bandiera_haiti:'
    },
    u'\U0001F1ED\U0001F1F2': { # 🇭🇲
        'en' : ':Heard_&_McDonald_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Heard_&_McDonald_Islands:'],
        'de': ':flagge_heard_und_mcdonaldinseln:',
        'es': ':bandera_islas_heard_y_mcdonald:',
        'fr': u':drapeau_îles_heard_et_mcdonald:',
        'pt': ':bandeira_ilhas_heard_e_mcdonald:',
        'it': ':bandiera_isole_heard_e_mcdonald:'
    },
    u'\U0001F1ED\U0001F1F3': { # 🇭🇳
        'en' : ':Honduras:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Honduras:'],
        'de': ':flagge_honduras:',
        'es': ':bandera_honduras:',
        'fr': ':drapeau_honduras:',
        'pt': ':bandeira_honduras:',
        'it': ':bandiera_honduras:'
    },
    u'\U0001F1ED\U0001F1F0': { # 🇭🇰
        'en' : ':Hong_Kong_SAR_China:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Hong_Kong:'],
        'de': ':flagge_sonderverwaltungsregion_hongkong:',
        'es': ':bandera_rae_de_hong_kong_(china):',
        'fr': ':drapeau_r.a.s._chinoise_de_hong_kong:',
        'pt': ':bandeira_hong_kong_rae_da_china:',
        'it': ':bandiera_ras_di_hong_kong:'
    },
    u'\U0001F1ED\U0001F1FA': { # 🇭🇺
        'en' : ':Hungary:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Hungary:'],
        'de': ':flagge_ungarn:',
        'es': u':bandera_hungría:',
        'fr': ':drapeau_hongrie:',
        'pt': ':bandeira_hungria:',
        'it': ':bandiera_ungheria:'
    },
    u'\U0001F194': { # 🆔
        'en' : ':ID_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':id:'],
        'de': u':großbuchstaben_id_in_lila_quadrat:',
        'es': u':símbolo_de_identificación:',
        'fr': ':bouton_identifiant:',
        'pt': u':botão_id:',
        'it': ':pulsante_id:'
    },
    u'\U0001F1EE\U0001F1F8': { # 🇮🇸
        'en' : ':Iceland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Iceland:'],
        'de': ':flagge_island:',
        'es': ':bandera_islandia:',
        'fr': ':drapeau_islande:',
        'pt': u':bandeira_islândia:',
        'it': ':bandiera_islanda:'
    },
    u'\U0001F1EE\U0001F1F3': { # 🇮🇳
        'en' : ':India:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_India:'],
        'de': ':flagge_indien:',
        'es': ':bandera_india:',
        'fr': ':drapeau_inde:',
        'pt': u':bandeira_índia:',
        'it': ':bandiera_india:'
    },
    u'\U0001F1EE\U0001F1E9': { # 🇮🇩
        'en' : ':Indonesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Indonesia:'],
        'de': ':flagge_indonesien:',
        'es': ':bandera_indonesia:',
        'fr': u':drapeau_indonésie:',
        'pt': u':bandeira_indonésia:',
        'it': ':bandiera_indonesia:'
    },
    u'\U0001F1EE\U0001F1F7': { # 🇮🇷
        'en' : ':Iran:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Iran:'],
        'de': ':flagge_iran:',
        'es': u':bandera_irán:',
        'fr': ':drapeau_iran:',
        'pt': u':bandeira_irã:',
        'it': ':bandiera_iran:'
    },
    u'\U0001F1EE\U0001F1F6': { # 🇮🇶
        'en' : ':Iraq:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Iraq:'],
        'de': ':flagge_irak:',
        'es': ':bandera_irak:',
        'fr': ':drapeau_irak:',
        'pt': ':bandeira_iraque:',
        'it': ':bandiera_iraq:'
    },
    u'\U0001F1EE\U0001F1EA': { # 🇮🇪
        'en' : ':Ireland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ireland:'],
        'de': ':flagge_irland:',
        'es': ':bandera_irlanda:',
        'fr': ':drapeau_irlande:',
        'pt': ':bandeira_irlanda:',
        'it': ':bandiera_irlanda:'
    },
    u'\U0001F1EE\U0001F1F2': { # 🇮🇲
        'en' : ':Isle_of_Man:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Isle_of_Man:'],
        'de': ':flagge_isle_of_man:',
        'es': ':bandera_isla_de_man:',
        'fr': u':drapeau_île_de_man:',
        'pt': ':bandeira_ilha_de_man:',
        'it': ':bandiera_isola_di_man:'
    },
    u'\U0001F1EE\U0001F1F1': { # 🇮🇱
        'en' : ':Israel:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Israel:'],
        'de': ':flagge_israel:',
        'es': ':bandera_israel:',
        'fr': u':drapeau_israël:',
        'pt': ':bandeira_israel:',
        'it': ':bandiera_israele:'
    },
    u'\U0001F1EE\U0001F1F9': { # 🇮🇹
        'en' : ':Italy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_Italy:'],
        'de': ':flagge_italien:',
        'es': ':bandera_italia:',
        'fr': ':drapeau_italie:',
        'pt': u':bandeira_itália:',
        'it': ':bandiera_italia:'
    },
    u'\U0001F1EF\U0001F1F2': { # 🇯🇲
        'en' : ':Jamaica:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Jamaica:'],
        'de': ':flagge_jamaika:',
        'es': ':bandera_jamaica:',
        'fr': u':drapeau_jamaïque:',
        'pt': ':bandeira_jamaica:',
        'it': ':bandiera_giamaica:'
    },
    u'\U0001F1EF\U0001F1F5': { # 🇯🇵
        'en' : ':Japan:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_Japan:'],
        'de': ':flagge_japan:',
        'es': u':bandera_japón:',
        'fr': ':drapeau_japon:',
        'pt': u':bandeira_japão:',
        'it': ':bandiera_giappone:'
    },
    u'\U0001F251': { # 🉑
        'en' : ':Japanese_acceptable_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':accept:'],
        'de': u':schriftzeichen_für_akzeptieren:',
        'es': u':ideograma_japonés_para_aceptable:',
        'fr': ':bouton_accepter_en_japonais:',
        'pt': u':botão_japonês_de_“aceitável”:',
        'it': u':ideogramma_giapponese_di_“accettabile”:'
    },
    u'\U0001F238': { # 🈸
        'en' : ':Japanese_application_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u7533:'],
        'de': u':schriftzeichen_für_anwenden:',
        'es': u':ideograma_japonés_para_aplicación:',
        'fr': ':bouton_application_en_japonais:',
        'pt': u':botão_japonês_de_“aplicação”:',
        'it': u':ideogramma_giapponese_di_“candidatura”:'
    },
    u'\U0001F250': { # 🉐
        'en' : ':Japanese_bargain_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ideograph_advantage:'],
        'de': u':schriftzeichen_für_schnäppchen:',
        'es': u':ideograma_japonés_para_ganga:',
        'fr': ':bouton_bonne_affaire_en_japonais:',
        'pt': u':botão_japonês_de_“barganha”:',
        'it': u':ideogramma_giapponese_di_“occasione”:'
    },
    u'\U0001F3EF': { # 🏯
        'en' : ':Japanese_castle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':japanese_castle:'],
        'de': ':japanisches_schloss:',
        'es': u':castillo_japonés:',
        'fr': u':château_japonais:',
        'pt': u':castelo_japonês:',
        'it': ':castello_giapponese:'
    },
    u'\U00003297\U0000FE0F': { # ㊗️
        'en' : ':Japanese_congratulations_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':congratulations:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratulation:',
        'es': u':ideograma_japonés_para_enhorabuena:',
        'fr': u':bouton_félicitations_en_japonais:',
        'pt': u':botão_japonês_de_“parabéns”:',
        'it': u':ideogramma_giapponese_di_“congratulazioni”:'
    },
    u'\U00003297': { # ㊗
        'en' : ':Japanese_congratulations_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':congratulations:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratulation:',
        'es': u':ideograma_japonés_para_enhorabuena:',
        'fr': u':bouton_félicitations_en_japonais:',
        'pt': u':botão_japonês_de_“parabéns”:',
        'it': u':ideogramma_giapponese_di_“congratulazioni”:'
    },
    u'\U0001F239': { # 🈹
        'en' : ':Japanese_discount_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u5272:'],
        'de': u':schriftzeichen_für_rabatt:',
        'es': u':ideograma_japonés_para_descuento:',
        'fr': u':bouton_réduction_en_japonais:',
        'pt': u':botão_japonês_de_“desconto”:',
        'it': u':ideogramma_giapponese_di_“sconto”:'
    },
    u'\U0001F38E': { # 🎎
        'en' : ':Japanese_dolls:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dolls:'],
        'de': ':japanische_puppen:',
        'es': u':muñecas_japonesas:',
        'fr': u':poupées_japonaises:',
        'pt': ':bonecas_japonesas:',
        'it': ':bambole_giapponesi:'
    },
    u'\U0001F21A': { # 🈚
        'en' : ':Japanese_free_of_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u7121:'],
        'variant': True,
        'de': u':schriftzeichen_für_gratis:',
        'es': u':ideograma_japonés_para_gratis:',
        'fr': ':bouton_gratuit_en_japonais:',
        'pt': u':botão_japonês_de_“gratuito”:',
        'it': u':ideogramma_giapponese_di_“gratis”:'
    },
    u'\U0001F201': { # 🈁
        'en' : ':Japanese_here_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':koko:'],
        'de': ':schriftzeichen_koko:',
        'es': u':ideograma_japonés_para_aquí:',
        'fr': ':bouton_ici_en_japonais:',
        'pt': u':botão_japonês_de_“aqui”:',
        'it': u':ideogramma_giapponese_per_“qui”:'
    },
    u'\U0001F237\U0000FE0F': { # 🈷️
        'en' : ':Japanese_monthly_amount_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u6708:'],
        'variant': True,
        'de': u':schriftzeichen_für_monatsbetrag:',
        'es': u':ideograma_japonés_para_cantidad_mensual:',
        'fr': ':bouton_montant_mensuel_en_japonais:',
        'pt': u':botão_japonês_de_“quantidade_mensal”:',
        'it': u':ideogramma_giapponese_di_“importo_mensile”:'
    },
    u'\U0001F237': { # 🈷
        'en' : ':Japanese_monthly_amount_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':u6708:'],
        'variant': True,
        'de': u':schriftzeichen_für_monatsbetrag:',
        'es': u':ideograma_japonés_para_cantidad_mensual:',
        'fr': ':bouton_montant_mensuel_en_japonais:',
        'pt': u':botão_japonês_de_“quantidade_mensal”:',
        'it': u':ideogramma_giapponese_di_“importo_mensile”:'
    },
    u'\U0001F235': { # 🈵
        'en' : ':Japanese_no_vacancy_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u6e80:'],
        'de': u':schriftzeichen_für_kein_zimmer_frei:',
        'es': u':ideograma_japonés_para_completo:',
        'fr': ':bouton_complet_en_japonais:',
        'pt': u':botão_japonês_de_“sem_vagas”:',
        'it': u':ideogramma_giapponese_di_“nessun_posto_libero”:'
    },
    u'\U0001F236': { # 🈶
        'en' : ':Japanese_not_free_of_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u6709:'],
        'de': u':schriftzeichen_für_nicht_gratis:',
        'es': u':ideograma_japonés_para_de_pago:',
        'fr': ':bouton_pas_gratuit_en_japonais:',
        'pt': u':botão_japonês_de_“não_gratuito”:',
        'it': u':ideogramma_giapponese_di_“a_pagamento”:'
    },
    u'\U0001F23A': { # 🈺
        'en' : ':Japanese_open_for_business_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u55b6:'],
        'de': u':schriftzeichen_für_geöffnet:',
        'es': u':ideograma_japonés_para_abierto:',
        'fr': ':bouton_ouvert_pour_affaires_en_japonais:',
        'pt': u':botão_japonês_de_“aberto_para_negócios”:',
        'it': u':ideogramma_giapponese_di_“aperto_al_pubblico”:'
    },
    u'\U0001F234': { # 🈴
        'en' : ':Japanese_passing_grade_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u5408:'],
        'de': u':schriftzeichen_für_note_zum_bestehen:',
        'es': u':ideograma_japonés_para_aprobado:',
        'fr': u':bouton_note_pour_réussir_en_japonais:',
        'pt': u':botão_japonês_de_“nota_mínima”:',
        'it': u':ideogramma_giapponese_di_“voto_di_sufficienza”:'
    },
    u'\U0001F3E3': { # 🏣
        'en' : ':Japanese_post_office:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':post_office:'],
        'de': u':japanisches_postgebäude:',
        'es': ':oficina_de_correos_japonesa:',
        'fr': ':bureau_de_poste_japonais:',
        'pt': u':correio_japonês:',
        'it': ':ufficio_postale_giapponese:'
    },
    u'\U0001F232': { # 🈲
        'en' : ':Japanese_prohibited_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u7981:'],
        'de': u':schriftzeichen_für_verbieten:',
        'es': u':ideograma_japonés_para_prohibido:',
        'fr': ':bouton_interdit_en_japonais:',
        'pt': u':botão_japonês_de_“proibido”:',
        'it': u':ideogramma_giapponese_di_“proibito”:'
    },
    u'\U0001F22F': { # 🈯
        'en' : ':Japanese_reserved_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u6307:'],
        'variant': True,
        'de': u':schriftzeichen_für_reserviert:',
        'es': u':ideograma_japonés_para_reservado:',
        'fr': u':bouton_réservé_en_japonais:',
        'pt': u':botão_japonês_de_“reservado”:',
        'it': u':ideogramma_giapponese_di_“riservato”:'
    },
    u'\U00003299\U0000FE0F': { # ㊙️
        'en' : ':Japanese_secret_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':secret:'],
        'variant': True,
        'de': u':schriftzeichen_für_geheimnis:',
        'es': u':ideograma_japonés_para_secreto:',
        'fr': ':bouton_secret_en_japonais:',
        'pt': u':botão_japonês_de_“segredo”:',
        'it': u':ideogramma_giapponese_di_“segreto”:'
    },
    u'\U00003299': { # ㊙
        'en' : ':Japanese_secret_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':secret:'],
        'variant': True,
        'de': u':schriftzeichen_für_geheimnis:',
        'es': u':ideograma_japonés_para_secreto:',
        'fr': ':bouton_secret_en_japonais:',
        'pt': u':botão_japonês_de_“segredo”:',
        'it': u':ideogramma_giapponese_di_“segreto”:'
    },
    u'\U0001F202\U0000FE0F': { # 🈂️
        'en' : ':Japanese_service_charge_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sa:'],
        'variant': True,
        'de': ':schriftzeichen_sa:',
        'es': u':ideograma_japonés_para_de_cortesía:',
        'fr': ':bouton_frais_de_service_en_japonais:',
        'pt': u':botão_japonês_de_“taxa_de_serviço”:',
        'it': u':ideogramma_giapponese_di_“costo_del_servizio”:'
    },
    u'\U0001F202': { # 🈂
        'en' : ':Japanese_service_charge_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':sa:'],
        'variant': True,
        'de': ':schriftzeichen_sa:',
        'es': u':ideograma_japonés_para_de_cortesía:',
        'fr': ':bouton_frais_de_service_en_japonais:',
        'pt': u':botão_japonês_de_“taxa_de_serviço”:',
        'it': u':ideogramma_giapponese_di_“costo_del_servizio”:'
    },
    u'\U0001F530': { # 🔰
        'en' : ':Japanese_symbol_for_beginner:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':beginner:'],
        'de': u':japanisches_anfänger-zeichen:',
        'es': u':símbolo_japonés_para_principiante:',
        'fr': u':symbole_japonais_de_débutant:',
        'pt': u':símbolo_japonês_de_principiante:',
        'it': u':ideogramma_giapponese_di_“principiante”:'
    },
    u'\U0001F233': { # 🈳
        'en' : ':Japanese_vacancy_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':u7a7a:'],
        'de': u':schriftzeichen_für_zimmer_frei:',
        'es': u':ideograma_japonés_para_vacante:',
        'fr': ':bouton_chambres_disponibles_en_japonais:',
        'pt': u':botão_japonês_de_“vago”:',
        'it': u':ideogramma_giapponese_di_“posto_libero”:'
    },
    u'\U0001F1EF\U0001F1EA': { # 🇯🇪
        'en' : ':Jersey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Jersey:'],
        'de': ':flagge_jersey:',
        'es': ':bandera_jersey:',
        'fr': ':drapeau_jersey:',
        'pt': ':bandeira_jersey:',
        'it': ':bandiera_jersey:'
    },
    u'\U0001F1EF\U0001F1F4': { # 🇯🇴
        'en' : ':Jordan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Jordan:'],
        'de': ':flagge_jordanien:',
        'es': ':bandera_jordania:',
        'fr': ':drapeau_jordanie:',
        'pt': u':bandeira_jordânia:',
        'it': ':bandiera_giordania:'
    },
    u'\U0001F1F0\U0001F1FF': { # 🇰🇿
        'en' : ':Kazakhstan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kazakhstan:'],
        'de': ':flagge_kasachstan:',
        'es': u':bandera_kazajistán:',
        'fr': ':drapeau_kazakhstan:',
        'pt': u':bandeira_cazaquistão:',
        'it': ':bandiera_kazakistan:'
    },
    u'\U0001F1F0\U0001F1EA': { # 🇰🇪
        'en' : ':Kenya:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kenya:'],
        'de': ':flagge_kenia:',
        'es': ':bandera_kenia:',
        'fr': ':drapeau_kenya:',
        'pt': u':bandeira_quênia:',
        'it': ':bandiera_kenya:'
    },
    u'\U0001F1F0\U0001F1EE': { # 🇰🇮
        'en' : ':Kiribati:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kiribati:'],
        'de': ':flagge_kiribati:',
        'es': ':bandera_kiribati:',
        'fr': ':drapeau_kiribati:',
        'pt': ':bandeira_quiribati:',
        'it': ':bandiera_kiribati:'
    },
    u'\U0001F1FD\U0001F1F0': { # 🇽🇰
        'en' : ':Kosovo:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kosovo:'],
        'de': ':flagge_kosovo:',
        'es': ':bandera_kosovo:',
        'fr': ':drapeau_kosovo:',
        'pt': ':bandeira_kosovo:',
        'it': ':bandiera_kosovo:'
    },
    u'\U0001F1F0\U0001F1FC': { # 🇰🇼
        'en' : ':Kuwait:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kuwait:'],
        'de': ':flagge_kuwait:',
        'es': ':bandera_kuwait:',
        'fr': u':drapeau_koweït:',
        'pt': ':bandeira_kuwait:',
        'it': ':bandiera_kuwait:'
    },
    u'\U0001F1F0\U0001F1EC': { # 🇰🇬
        'en' : ':Kyrgyzstan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Kyrgyzstan:'],
        'de': ':flagge_kirgisistan:',
        'es': u':bandera_kirguistán:',
        'fr': ':drapeau_kirghizistan:',
        'pt': u':bandeira_quirguistão:',
        'it': ':bandiera_kirghizistan:'
    },
    u'\U0001F1F1\U0001F1E6': { # 🇱🇦
        'en' : ':Laos:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Laos:'],
        'de': ':flagge_laos:',
        'es': ':bandera_laos:',
        'fr': ':drapeau_laos:',
        'pt': ':bandeira_laos:',
        'it': ':bandiera_laos:'
    },
    u'\U0001F1F1\U0001F1FB': { # 🇱🇻
        'en' : ':Latvia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Latvia:'],
        'de': ':flagge_lettland:',
        'es': ':bandera_letonia:',
        'fr': ':drapeau_lettonie:',
        'pt': u':bandeira_letônia:',
        'it': ':bandiera_lettonia:'
    },
    u'\U0001F1F1\U0001F1E7': { # 🇱🇧
        'en' : ':Lebanon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Lebanon:'],
        'de': ':flagge_libanon:',
        'es': u':bandera_líbano:',
        'fr': ':drapeau_liban:',
        'pt': u':bandeira_líbano:',
        'it': ':bandiera_libano:'
    },
    u'\U0000264C': { # ♌
        'en' : ':Leo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':leo:'],
        'variant': True,
        'de': u':löwe_(sternzeichen):',
        'es': ':leo:',
        'fr': ':lion:',
        'pt': u':signo_de_leão:',
        'it': ':segno_zodiacale_del_leone:'
    },
    u'\U0001F1F1\U0001F1F8': { # 🇱🇸
        'en' : ':Lesotho:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Lesotho:'],
        'de': ':flagge_lesotho:',
        'es': ':bandera_lesoto:',
        'fr': ':drapeau_lesotho:',
        'pt': ':bandeira_lesoto:',
        'it': ':bandiera_lesotho:'
    },
    u'\U0001F1F1\U0001F1F7': { # 🇱🇷
        'en' : ':Liberia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Liberia:'],
        'de': ':flagge_liberia:',
        'es': ':bandera_liberia:',
        'fr': u':drapeau_libéria:',
        'pt': u':bandeira_libéria:',
        'it': ':bandiera_liberia:'
    },
    u'\U0000264E': { # ♎
        'en' : ':Libra:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':libra:'],
        'variant': True,
        'de': ':waage_(sternzeichen):',
        'es': ':libra:',
        'fr': ':balance:',
        'pt': ':signo_de_libra:',
        'it': ':segno_zodiacale_della_bilancia:'
    },
    u'\U0001F1F1\U0001F1FE': { # 🇱🇾
        'en' : ':Libya:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Libya:'],
        'de': ':flagge_libyen:',
        'es': ':bandera_libia:',
        'fr': ':drapeau_libye:',
        'pt': u':bandeira_líbia:',
        'it': ':bandiera_libia:'
    },
    u'\U0001F1F1\U0001F1EE': { # 🇱🇮
        'en' : ':Liechtenstein:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Liechtenstein:'],
        'de': ':flagge_liechtenstein:',
        'es': ':bandera_liechtenstein:',
        'fr': ':drapeau_liechtenstein:',
        'pt': ':bandeira_liechtenstein:',
        'it': ':bandiera_liechtenstein:'
    },
    u'\U0001F1F1\U0001F1F9': { # 🇱🇹
        'en' : ':Lithuania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Lithuania:'],
        'de': ':flagge_litauen:',
        'es': ':bandera_lituania:',
        'fr': ':drapeau_lituanie:',
        'pt': u':bandeira_lituânia:',
        'it': ':bandiera_lituania:'
    },
    u'\U0001F1F1\U0001F1FA': { # 🇱🇺
        'en' : ':Luxembourg:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Luxembourg:'],
        'de': ':flagge_luxemburg:',
        'es': ':bandera_luxemburgo:',
        'fr': ':drapeau_luxembourg:',
        'pt': ':bandeira_luxemburgo:',
        'it': ':bandiera_lussemburgo:'
    },
    u'\U0001F1F2\U0001F1F4': { # 🇲🇴
        'en' : ':Macao_SAR_China:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Macau:'],
        'de': ':flagge_sonderverwaltungsregion_macau:',
        'es': ':bandera_rae_de_macao_(china):',
        'fr': ':drapeau_r.a.s._chinoise_de_macao:',
        'pt': ':bandeira_macau_rae_da_china:',
        'it': ':bandiera_ras_di_macao:'
    },
    u'\U0001F1F2\U0001F1EC': { # 🇲🇬
        'en' : ':Madagascar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Madagascar:'],
        'de': ':flagge_madagaskar:',
        'es': ':bandera_madagascar:',
        'fr': ':drapeau_madagascar:',
        'pt': ':bandeira_madagascar:',
        'it': ':bandiera_madagascar:'
    },
    u'\U0001F1F2\U0001F1FC': { # 🇲🇼
        'en' : ':Malawi:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Malawi:'],
        'de': ':flagge_malawi:',
        'es': ':bandera_malaui:',
        'fr': ':drapeau_malawi:',
        'pt': ':bandeira_malaui:',
        'it': ':bandiera_malawi:'
    },
    u'\U0001F1F2\U0001F1FE': { # 🇲🇾
        'en' : ':Malaysia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Malaysia:'],
        'de': ':flagge_malaysia:',
        'es': ':bandera_malasia:',
        'fr': ':drapeau_malaisie:',
        'pt': u':bandeira_malásia:',
        'it': ':bandiera_malaysia:'
    },
    u'\U0001F1F2\U0001F1FB': { # 🇲🇻
        'en' : ':Maldives:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Maldives:'],
        'de': ':flagge_malediven:',
        'es': ':bandera_maldivas:',
        'fr': ':drapeau_maldives:',
        'pt': ':bandeira_maldivas:',
        'it': ':bandiera_maldive:'
    },
    u'\U0001F1F2\U0001F1F1': { # 🇲🇱
        'en' : ':Mali:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mali:'],
        'de': ':flagge_mali:',
        'es': ':bandera_mali:',
        'fr': ':drapeau_mali:',
        'pt': ':bandeira_mali:',
        'it': ':bandiera_mali:'
    },
    u'\U0001F1F2\U0001F1F9': { # 🇲🇹
        'en' : ':Malta:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Malta:'],
        'de': ':flagge_malta:',
        'es': ':bandera_malta:',
        'fr': ':drapeau_malte:',
        'pt': ':bandeira_malta:',
        'it': ':bandiera_malta:'
    },
    u'\U0001F1F2\U0001F1ED': { # 🇲🇭
        'en' : ':Marshall_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Marshall_Islands:'],
        'de': ':flagge_marshallinseln:',
        'es': ':bandera_islas_marshall:',
        'fr': u':drapeau_îles_marshall:',
        'pt': ':bandeira_ilhas_marshall:',
        'it': ':bandiera_isole_marshall:'
    },
    u'\U0001F1F2\U0001F1F6': { # 🇲🇶
        'en' : ':Martinique:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Martinique:'],
        'de': ':flagge_martinique:',
        'es': ':bandera_martinica:',
        'fr': ':drapeau_martinique:',
        'pt': ':bandeira_martinica:',
        'it': ':bandiera_martinica:'
    },
    u'\U0001F1F2\U0001F1F7': { # 🇲🇷
        'en' : ':Mauritania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mauritania:'],
        'de': ':flagge_mauretanien:',
        'es': ':bandera_mauritania:',
        'fr': ':drapeau_mauritanie:',
        'pt': u':bandeira_mauritânia:',
        'it': ':bandiera_mauritania:'
    },
    u'\U0001F1F2\U0001F1FA': { # 🇲🇺
        'en' : ':Mauritius:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mauritius:'],
        'de': ':flagge_mauritius:',
        'es': ':bandera_mauricio:',
        'fr': ':drapeau_maurice:',
        'pt': u':bandeira_maurício:',
        'it': ':bandiera_mauritius:'
    },
    u'\U0001F1FE\U0001F1F9': { # 🇾🇹
        'en' : ':Mayotte:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mayotte:'],
        'de': ':flagge_mayotte:',
        'es': ':bandera_mayotte:',
        'fr': ':drapeau_mayotte:',
        'pt': ':bandeira_mayotte:',
        'it': ':bandiera_mayotte:'
    },
    u'\U0001F1F2\U0001F1FD': { # 🇲🇽
        'en' : ':Mexico:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mexico:'],
        'de': ':flagge_mexiko:',
        'es': u':bandera_méxico:',
        'fr': ':drapeau_mexique:',
        'pt': u':bandeira_méxico:',
        'it': ':bandiera_messico:'
    },
    u'\U0001F1EB\U0001F1F2': { # 🇫🇲
        'en' : ':Micronesia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Micronesia:'],
        'de': ':flagge_mikronesien:',
        'es': ':bandera_micronesia:',
        'fr': u':drapeau_états_fédérés_de_micronésie:',
        'pt': u':bandeira_micronésia:',
        'it': ':bandiera_micronesia:'
    },
    u'\U0001F1F2\U0001F1E9': { # 🇲🇩
        'en' : ':Moldova:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Moldova:'],
        'de': ':flagge_republik_moldau:',
        'es': ':bandera_moldavia:',
        'fr': ':drapeau_moldavie:',
        'pt': ':bandeira_moldova:',
        'it': ':bandiera_moldavia:'
    },
    u'\U0001F1F2\U0001F1E8': { # 🇲🇨
        'en' : ':Monaco:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Monaco:'],
        'de': ':flagge_monaco:',
        'es': u':bandera_mónaco:',
        'fr': ':drapeau_monaco:',
        'pt': u':bandeira_mônaco:',
        'it': ':bandiera_monaco:'
    },
    u'\U0001F1F2\U0001F1F3': { # 🇲🇳
        'en' : ':Mongolia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mongolia:'],
        'de': ':flagge_mongolei:',
        'es': ':bandera_mongolia:',
        'fr': ':drapeau_mongolie:',
        'pt': u':bandeira_mongólia:',
        'it': ':bandiera_mongolia:'
    },
    u'\U0001F1F2\U0001F1EA': { # 🇲🇪
        'en' : ':Montenegro:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Montenegro:'],
        'de': ':flagge_montenegro:',
        'es': ':bandera_montenegro:',
        'fr': u':drapeau_monténégro:',
        'pt': ':bandeira_montenegro:',
        'it': ':bandiera_montenegro:'
    },
    u'\U0001F1F2\U0001F1F8': { # 🇲🇸
        'en' : ':Montserrat:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Montserrat:'],
        'de': ':flagge_montserrat:',
        'es': ':bandera_montserrat:',
        'fr': ':drapeau_montserrat:',
        'pt': ':bandeira_montserrat:',
        'it': ':bandiera_montserrat:'
    },
    u'\U0001F1F2\U0001F1E6': { # 🇲🇦
        'en' : ':Morocco:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Morocco:'],
        'de': ':flagge_marokko:',
        'es': ':bandera_marruecos:',
        'fr': ':drapeau_maroc:',
        'pt': ':bandeira_marrocos:',
        'it': ':bandiera_marocco:'
    },
    u'\U0001F1F2\U0001F1FF': { # 🇲🇿
        'en' : ':Mozambique:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Mozambique:'],
        'de': ':flagge_mosambik:',
        'es': ':bandera_mozambique:',
        'fr': ':drapeau_mozambique:',
        'pt': u':bandeira_moçambique:',
        'it': ':bandiera_mozambico:'
    },
    u'\U0001F936': { # 🤶
        'en' : ':Mrs._Claus:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':mrs_claus:'],
        'de': ':weihnachtsfrau:',
        'es': u':mamá_noel:',
        'fr': u':mère_noël:',
        'pt': u':mamãe_noel:',
        'it': ':mamma_natale:'
    },
    u'\U0001F936\U0001F3FF': { # 🤶🏿
        'en' : ':Mrs._Claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':weihnachtsfrau_dunkle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_oscuro:',
        'fr': u':mère_noël_peau_foncée:',
        'pt': u':mamãe_noel_pele_escura:',
        'it': ':mamma_natale_carnagione_scura:'
    },
    u'\U0001F936\U0001F3FB': { # 🤶🏻
        'en' : ':Mrs._Claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':weihnachtsfrau_helle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_claro:',
        'fr': u':mère_noël_peau_claire:',
        'pt': u':mamãe_noel_pele_clara:',
        'it': ':mamma_natale_carnagione_chiara:'
    },
    u'\U0001F936\U0001F3FE': { # 🤶🏾
        'en' : ':Mrs._Claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':weihnachtsfrau_mitteldunkle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_oscuro_medio:',
        'fr': u':mère_noël_peau_mate:',
        'pt': u':mamãe_noel_pele_morena_escura:',
        'it': ':mamma_natale_carnagione_abbastanza_scura:'
    },
    u'\U0001F936\U0001F3FC': { # 🤶🏼
        'en' : ':Mrs._Claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':weihnachtsfrau_mittelhelle_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_claro_medio:',
        'fr': u':mère_noël_peau_moyennement_claire:',
        'pt': u':mamãe_noel_pele_morena_clara:',
        'it': ':mamma_natale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F936\U0001F3FD': { # 🤶🏽
        'en' : ':Mrs._Claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':weihnachtsfrau_mittlere_hautfarbe:',
        'es': u':mamá_noel_tono_de_piel_medio:',
        'fr': u':mère_noël_peau_légèrement_mate:',
        'pt': u':mamãe_noel_pele_morena:',
        'it': ':mamma_natale_carnagione_olivastra:'
    },
    u'\U0001F1F2\U0001F1F2': { # 🇲🇲
        'en' : ':Myanmar_(Burma):',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Myanmar:'],
        'de': ':flagge_myanmar:',
        'es': ':bandera_myanmar_(birmania):',
        'fr': ':drapeau_myanmar_(birmanie):',
        'pt': u':bandeira_mianmar_(birmânia):',
        'it': ':bandiera_myanmar_(birmania):'
    },
    u'\U0001F195': { # 🆕
        'en' : ':NEW_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':new:'],
        'de': ':wort_new_in_blauem_quadrat:',
        'es': u':botón_new:',
        'fr': ':bouton_nouveau:',
        'pt': u':botão_new:',
        'it': ':pulsante_new:'
    },
    u'\U0001F196': { # 🆖
        'en' : ':NG_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ng:'],
        'de': u':großbuchstaben_ng_in_blauem_quadrat:',
        'es': u':botón_ng:',
        'fr': ':bouton_pas_bien:',
        'pt': u':botão_ng:',
        'it': ':pulsante_ng:'
    },
    u'\U0001F1F3\U0001F1E6': { # 🇳🇦
        'en' : ':Namibia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Namibia:'],
        'de': ':flagge_namibia:',
        'es': ':bandera_namibia:',
        'fr': ':drapeau_namibie:',
        'pt': u':bandeira_namíbia:',
        'it': ':bandiera_namibia:'
    },
    u'\U0001F1F3\U0001F1F7': { # 🇳🇷
        'en' : ':Nauru:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Nauru:'],
        'de': ':flagge_nauru:',
        'es': ':bandera_nauru:',
        'fr': ':drapeau_nauru:',
        'pt': ':bandeira_nauru:',
        'it': ':bandiera_nauru:'
    },
    u'\U0001F1F3\U0001F1F5': { # 🇳🇵
        'en' : ':Nepal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Nepal:'],
        'de': ':flagge_nepal:',
        'es': ':bandera_nepal:',
        'fr': u':drapeau_népal:',
        'pt': ':bandeira_nepal:',
        'it': ':bandiera_nepal:'
    },
    u'\U0001F1F3\U0001F1F1': { # 🇳🇱
        'en' : ':Netherlands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Netherlands:'],
        'de': ':flagge_niederlande:',
        'es': u':bandera_países_bajos:',
        'fr': ':drapeau_pays-bas:',
        'pt': u':bandeira_países_baixos:',
        'it': ':bandiera_paesi_bassi:'
    },
    u'\U0001F1F3\U0001F1E8': { # 🇳🇨
        'en' : ':New_Caledonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_New_Caledonia:'],
        'de': ':flagge_neukaledonien:',
        'es': ':bandera_nueva_caledonia:',
        'fr': u':drapeau_nouvelle-calédonie:',
        'pt': u':bandeira_nova_caledônia:',
        'it': ':bandiera_nuova_caledonia:'
    },
    u'\U0001F1F3\U0001F1FF': { # 🇳🇿
        'en' : ':New_Zealand:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_New_Zealand:'],
        'de': ':flagge_neuseeland:',
        'es': ':bandera_nueva_zelanda:',
        'fr': u':drapeau_nouvelle-zélande:',
        'pt': u':bandeira_nova_zelândia:',
        'it': ':bandiera_nuova_zelanda:'
    },
    u'\U0001F1F3\U0001F1EE': { # 🇳🇮
        'en' : ':Nicaragua:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Nicaragua:'],
        'de': ':flagge_nicaragua:',
        'es': ':bandera_nicaragua:',
        'fr': ':drapeau_nicaragua:',
        'pt': u':bandeira_nicarágua:',
        'it': ':bandiera_nicaragua:'
    },
    u'\U0001F1F3\U0001F1EA': { # 🇳🇪
        'en' : ':Niger:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Niger:'],
        'de': ':flagge_niger:',
        'es': u':bandera_níger:',
        'fr': ':drapeau_niger:',
        'pt': u':bandeira_níger:',
        'it': ':bandiera_niger:'
    },
    u'\U0001F1F3\U0001F1EC': { # 🇳🇬
        'en' : ':Nigeria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Nigeria:'],
        'de': ':flagge_nigeria:',
        'es': ':bandera_nigeria:',
        'fr': u':drapeau_nigéria:',
        'pt': u':bandeira_nigéria:',
        'it': ':bandiera_nigeria:'
    },
    u'\U0001F1F3\U0001F1FA': { # 🇳🇺
        'en' : ':Niue:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Niue:'],
        'de': ':flagge_niue:',
        'es': ':bandera_niue:',
        'fr': ':drapeau_niue:',
        'pt': ':bandeira_niue:',
        'it': ':bandiera_niue:'
    },
    u'\U0001F1F3\U0001F1EB': { # 🇳🇫
        'en' : ':Norfolk_Island:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Norfolk_Island:'],
        'de': ':flagge_norfolkinsel:',
        'es': ':bandera_isla_norfolk:',
        'fr': u':drapeau_île_norfolk:',
        'pt': ':bandeira_ilha_norfolk:',
        'it': ':bandiera_isola_norfolk:'
    },
    u'\U0001F1F0\U0001F1F5': { # 🇰🇵
        'en' : ':North_Korea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_North_Korea:'],
        'de': ':flagge_nordkorea:',
        'es': ':bandera_corea_del_norte:',
        'fr': u':drapeau_corée_du_nord:',
        'pt': ':bandeira_coreia_do_norte:',
        'it': ':bandiera_corea_del_nord:'
    },
    u'\U0001F1F2\U0001F1F0': { # 🇲🇰
        'en' : ':North_Macedonia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Macedonia:'],
        'de': ':flagge_nordmazedonien:',
        'es': ':bandera_macedonia_del_norte:',
        'fr': u':drapeau_macédoine_du_nord:',
        'pt': u':bandeira_macedônia_do_norte:',
        'it': ':bandiera_macedonia_del_nord:'
    },
    u'\U0001F1F2\U0001F1F5': { # 🇲🇵
        'en' : ':Northern_Mariana_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Northern_Mariana_Islands:'],
        'de': u':flagge_nördliche_marianen:',
        'es': ':bandera_islas_marianas_del_norte:',
        'fr': u':drapeau_îles_mariannes_du_nord:',
        'pt': ':bandeira_ilhas_marianas_do_norte:',
        'it': ':bandiera_isole_marianne_settentrionali:'
    },
    u'\U0001F1F3\U0001F1F4': { # 🇳🇴
        'en' : ':Norway:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Norway:'],
        'de': ':flagge_norwegen:',
        'es': ':bandera_noruega:',
        'fr': u':drapeau_norvège:',
        'pt': ':bandeira_noruega:',
        'it': ':bandiera_norvegia:'
    },
    u'\U0001F197': { # 🆗
        'en' : ':OK_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ok:'],
        'de': u':großbuchstaben_ok_in_blauem_quadrat:',
        'es': u':botón_ok:',
        'fr': ':bouton_ok:',
        'pt': u':botão_ok:',
        'it': ':pulsante_ok:'
    },
    u'\U0001F44C': { # 👌
        'en' : ':OK_hand:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ok_hand:'],
        'de': ':ok-zeichen:',
        'es': u':señal_de_aprobación_con_la_mano:',
        'fr': ':ok:',
        'pt': ':sinal_de_ok:',
        'it': ':mano_che_fa_ok:'
    },
    u'\U0001F44C\U0001F3FF': { # 👌🏿
        'en' : ':OK_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ok-zeichen_dunkle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro:',
        'fr': u':ok_peau_foncée:',
        'pt': ':sinal_de_ok_pele_escura:',
        'it': ':mano_che_fa_ok_carnagione_scura:'
    },
    u'\U0001F44C\U0001F3FB': { # 👌🏻
        'en' : ':OK_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ok-zeichen_helle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_claro:',
        'fr': ':ok_peau_claire:',
        'pt': ':sinal_de_ok_pele_clara:',
        'it': ':mano_che_fa_ok_carnagione_chiara:'
    },
    u'\U0001F44C\U0001F3FE': { # 👌🏾
        'en' : ':OK_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ok-zeichen_mitteldunkle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_oscuro_medio:',
        'fr': ':ok_peau_mate:',
        'pt': ':sinal_de_ok_pele_morena_escura:',
        'it': ':mano_che_fa_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F44C\U0001F3FC': { # 👌🏼
        'en' : ':OK_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ok-zeichen_mittelhelle_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_claro_medio:',
        'fr': ':ok_peau_moyennement_claire:',
        'pt': ':sinal_de_ok_pele_morena_clara:',
        'it': ':mano_che_fa_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44C\U0001F3FD': { # 👌🏽
        'en' : ':OK_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ok-zeichen_mittlere_hautfarbe:',
        'es': u':señal_de_aprobación_con_la_mano_tono_de_piel_medio:',
        'fr': u':ok_peau_légèrement_mate:',
        'pt': ':sinal_de_ok_pele_morena:',
        'it': ':mano_che_fa_ok_carnagione_olivastra:'
    },
    u'\U0001F51B': { # 🔛
        'en' : ':ON!_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':on:'],
        'de': ':on!-pfeil:',
        'es': ':flecha_on!:',
        'fr': u':flèche_activé:',
        'pt': ':seta_on!:',
        'it': ':freccia_on:'
    },
    u'\U0001F17E\U0000FE0F': { # 🅾️
        'en' : ':O_button_(blood_type):',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':o2:'],
        'variant': True,
        'de': u':großbuchstabe_o_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_tipo_o:',
        'fr': ':groupe_sanguin_o:',
        'pt': u':botão_o_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_0:'
    },
    u'\U0001F17E': { # 🅾
        'en' : ':O_button_(blood_type):',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':o2:'],
        'variant': True,
        'de': u':großbuchstabe_o_in_rotem_quadrat:',
        'es': u':grupo_sanguíneo_tipo_o:',
        'fr': ':groupe_sanguin_o:',
        'pt': u':botão_o_(tipo_sanguíneo):',
        'it': ':gruppo_sanguigno_0:'
    },
    u'\U0001F1F4\U0001F1F2': { # 🇴🇲
        'en' : ':Oman:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Oman:'],
        'de': ':flagge_oman:',
        'es': u':bandera_omán:',
        'fr': ':drapeau_oman:',
        'pt': u':bandeira_omã:',
        'it': ':bandiera_oman:'
    },
    u'\U000026CE': { # ⛎
        'en' : ':Ophiuchus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ophiuchus:'],
        'de': u':schlangenträger:',
        'es': ':ofiuco:',
        'fr': ':serpentaire:',
        'pt': u':signo_de_ofiúco:',
        'it': u':segno_zodiacale_dell’ofiuco:'
    },
    u'\U0001F17F\U0000FE0F': { # 🅿️
        'en' : ':P_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':parking:'],
        'variant': True,
        'de': u':großbuchstabe_p_in_blauem_quadrat:',
        'es': ':aparcamiento:',
        'fr': ':bouton_p:',
        'pt': u':botão_p:',
        'it': ':pulsante_p:'
    },
    u'\U0001F17F': { # 🅿
        'en' : ':P_button:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':parking:'],
        'variant': True,
        'de': u':großbuchstabe_p_in_blauem_quadrat:',
        'es': ':aparcamiento:',
        'fr': ':bouton_p:',
        'pt': u':botão_p:',
        'it': ':pulsante_p:'
    },
    u'\U0001F1F5\U0001F1F0': { # 🇵🇰
        'en' : ':Pakistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Pakistan:'],
        'de': ':flagge_pakistan:',
        'es': u':bandera_pakistán:',
        'fr': ':drapeau_pakistan:',
        'pt': u':bandeira_paquistão:',
        'it': ':bandiera_pakistan:'
    },
    u'\U0001F1F5\U0001F1FC': { # 🇵🇼
        'en' : ':Palau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Palau:'],
        'de': ':flagge_palau:',
        'es': ':bandera_palaos:',
        'fr': ':drapeau_palaos:',
        'pt': ':bandeira_palau:',
        'it': ':bandiera_palau:'
    },
    u'\U0001F1F5\U0001F1F8': { # 🇵🇸
        'en' : ':Palestinian_Territories:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Palestinian_Territories:'],
        'de': u':flagge_palästinensische_autonomiegebiete:',
        'es': ':bandera_territorios_palestinos:',
        'fr': ':drapeau_territoires_palestiniens:',
        'pt': u':bandeira_territórios_palestinos:',
        'it': ':bandiera_territori_palestinesi:'
    },
    u'\U0001F1F5\U0001F1E6': { # 🇵🇦
        'en' : ':Panama:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Panama:'],
        'de': ':flagge_panama:',
        'es': u':bandera_panamá:',
        'fr': ':drapeau_panama:',
        'pt': u':bandeira_panamá:',
        'it': u':bandiera_panamá:'
    },
    u'\U0001F1F5\U0001F1EC': { # 🇵🇬
        'en' : ':Papua_New_Guinea:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Papua_New_Guinea:'],
        'de': ':flagge_papua-neuguinea:',
        'es': u':bandera_papúa_nueva_guinea:',
        'fr': u':drapeau_papouasie-nouvelle-guinée:',
        'pt': u':bandeira_papua-nova_guiné:',
        'it': ':bandiera_papua_nuova_guinea:'
    },
    u'\U0001F1F5\U0001F1FE': { # 🇵🇾
        'en' : ':Paraguay:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Paraguay:'],
        'de': ':flagge_paraguay:',
        'es': ':bandera_paraguay:',
        'fr': ':drapeau_paraguay:',
        'pt': ':bandeira_paraguai:',
        'it': ':bandiera_paraguay:'
    },
    u'\U0001F1F5\U0001F1EA': { # 🇵🇪
        'en' : ':Peru:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Peru:'],
        'de': ':flagge_peru:',
        'es': u':bandera_perú:',
        'fr': u':drapeau_pérou:',
        'pt': ':bandeira_peru:',
        'it': u':bandiera_perù:'
    },
    u'\U0001F1F5\U0001F1ED': { # 🇵🇭
        'en' : ':Philippines:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Philippines:'],
        'de': ':flagge_philippinen:',
        'es': ':bandera_filipinas:',
        'fr': ':drapeau_philippines:',
        'pt': ':bandeira_filipinas:',
        'it': ':bandiera_filippine:'
    },
    u'\U00002653': { # ♓
        'en' : ':Pisces:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pisces:'],
        'variant': True,
        'de': ':fische_(sternzeichen):',
        'es': ':piscis:',
        'fr': ':poissons:',
        'pt': ':signo_de_peixes:',
        'it': ':segno_zodiacale_dei_pesci:'
    },
    u'\U0001F1F5\U0001F1F3': { # 🇵🇳
        'en' : ':Pitcairn_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Pitcairn_Islands:'],
        'de': ':flagge_pitcairninseln:',
        'es': ':bandera_islas_pitcairn:',
        'fr': u':drapeau_îles_pitcairn:',
        'pt': ':bandeira_ilhas_pitcairn:',
        'it': ':bandiera_isole_pitcairn:'
    },
    u'\U0001F1F5\U0001F1F1': { # 🇵🇱
        'en' : ':Poland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Poland:'],
        'de': ':flagge_polen:',
        'es': ':bandera_polonia:',
        'fr': ':drapeau_pologne:',
        'pt': u':bandeira_polônia:',
        'it': ':bandiera_polonia:'
    },
    u'\U0001F1F5\U0001F1F9': { # 🇵🇹
        'en' : ':Portugal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Portugal:'],
        'de': ':flagge_portugal:',
        'es': ':bandera_portugal:',
        'fr': ':drapeau_portugal:',
        'pt': ':bandeira_portugal:',
        'it': ':bandiera_portogallo:'
    },
    u'\U0001F1F5\U0001F1F7': { # 🇵🇷
        'en' : ':Puerto_Rico:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Puerto_Rico:'],
        'de': ':flagge_puerto_rico:',
        'es': ':bandera_puerto_rico:',
        'fr': ':drapeau_porto_rico:',
        'pt': ':bandeira_porto_rico:',
        'it': ':bandiera_portorico:'
    },
    u'\U0001F1F6\U0001F1E6': { # 🇶🇦
        'en' : ':Qatar:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Qatar:'],
        'de': ':flagge_katar:',
        'es': ':bandera_catar:',
        'fr': ':drapeau_qatar:',
        'pt': ':bandeira_catar:',
        'it': ':bandiera_qatar:'
    },
    u'\U0001F1F7\U0001F1F4': { # 🇷🇴
        'en' : ':Romania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Romania:'],
        'de': u':flagge_rumänien:',
        'es': u':bandera_rumanía:',
        'fr': ':drapeau_roumanie:',
        'pt': u':bandeira_romênia:',
        'it': ':bandiera_romania:'
    },
    u'\U0001F1F7\U0001F1FA': { # 🇷🇺
        'en' : ':Russia:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_Russia:'],
        'de': ':flagge_russland:',
        'es': ':bandera_rusia:',
        'fr': ':drapeau_russie:',
        'pt': u':bandeira_rússia:',
        'it': ':bandiera_russia:'
    },
    u'\U0001F1F7\U0001F1FC': { # 🇷🇼
        'en' : ':Rwanda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Rwanda:'],
        'de': ':flagge_ruanda:',
        'es': ':bandera_ruanda:',
        'fr': ':drapeau_rwanda:',
        'pt': ':bandeira_ruanda:',
        'it': ':bandiera_ruanda:'
    },
    u'\U0001F1F7\U0001F1EA': { # 🇷🇪
        'en' : u':Réunion:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_Réunion:'],
        'de': u':flagge_réunion:',
        'es': u':bandera_reunión:',
        'fr': u':drapeau_la_réunion:',
        'pt': u':bandeira_reunião:',
        'it': ':bandiera_riunione:'
    },
    u'\U0001F51C': { # 🔜
        'en' : ':SOON_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':soon:'],
        'de': ':soon-pfeil:',
        'es': ':flecha_soon:',
        'fr': u':flèche_bientôt:',
        'pt': ':seta_soon:',
        'it': ':freccia_soon:'
    },
    u'\U0001F198': { # 🆘
        'en' : ':SOS_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sos:'],
        'de': ':sos-zeichen:',
        'es': u':símbolo_de_socorro:',
        'fr': ':bouton_sos:',
        'pt': u':botão_sos:',
        'it': ':pulsante_sos:'
    },
    u'\U00002650': { # ♐
        'en' : ':Sagittarius:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sagittarius:'],
        'variant': True,
        'de': u':schütze_(sternzeichen):',
        'es': ':sagitario:',
        'fr': ':sagittaire:',
        'pt': u':signo_de_sagitário:',
        'it': ':segno_zodiacale_del_saggitario:'
    },
    u'\U0001F1FC\U0001F1F8': { # 🇼🇸
        'en' : ':Samoa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Samoa:'],
        'de': ':flagge_samoa:',
        'es': ':bandera_samoa:',
        'fr': ':drapeau_samoa:',
        'pt': ':bandeira_samoa:',
        'it': ':bandiera_samoa:'
    },
    u'\U0001F1F8\U0001F1F2': { # 🇸🇲
        'en' : ':San_Marino:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_San_Marino:'],
        'de': ':flagge_san_marino:',
        'es': ':bandera_san_marino:',
        'fr': ':drapeau_saint-marin:',
        'pt': ':bandeira_san_marino:',
        'it': ':bandiera_san_marino:'
    },
    u'\U0001F385': { # 🎅
        'en' : ':Santa_Claus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':santa:'],
        'de': ':weihnachtsmann:',
        'es': u':papá_noel:',
        'fr': u':père_noël:',
        'pt': ':papai_noel:',
        'it': ':babbo_natale:'
    },
    u'\U0001F385\U0001F3FF': { # 🎅🏿
        'en' : ':Santa_Claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':weihnachtsmann_dunkle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_oscuro:',
        'fr': u':père_noël_peau_foncée:',
        'pt': ':papai_noel_pele_escura:',
        'it': ':babbo_natale_carnagione_scura:'
    },
    u'\U0001F385\U0001F3FB': { # 🎅🏻
        'en' : ':Santa_Claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':weihnachtsmann_helle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_claro:',
        'fr': u':père_noël_peau_claire:',
        'pt': ':papai_noel_pele_clara:',
        'it': ':babbo_natale_carnagione_chiara:'
    },
    u'\U0001F385\U0001F3FE': { # 🎅🏾
        'en' : ':Santa_Claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':weihnachtsmann_mitteldunkle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_oscuro_medio:',
        'fr': u':père_noël_peau_mate:',
        'pt': ':papai_noel_pele_morena_escura:',
        'it': ':babbo_natale_carnagione_abbastanza_scura:'
    },
    u'\U0001F385\U0001F3FC': { # 🎅🏼
        'en' : ':Santa_Claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':weihnachtsmann_mittelhelle_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_claro_medio:',
        'fr': u':père_noël_peau_moyennement_claire:',
        'pt': ':papai_noel_pele_morena_clara:',
        'it': ':babbo_natale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F385\U0001F3FD': { # 🎅🏽
        'en' : ':Santa_Claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':weihnachtsmann_mittlere_hautfarbe:',
        'es': u':papá_noel_tono_de_piel_medio:',
        'fr': u':père_noël_peau_légèrement_mate:',
        'pt': ':papai_noel_pele_morena:',
        'it': ':babbo_natale_carnagione_olivastra:'
    },
    u'\U0001F1F8\U0001F1E6': { # 🇸🇦
        'en' : ':Saudi_Arabia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Saudi_Arabia:'],
        'de': ':flagge_saudi-arabien:',
        'es': u':bandera_arabia_saudí:',
        'fr': ':drapeau_arabie_saoudite:',
        'pt': u':bandeira_arábia_saudita:',
        'it': ':bandiera_arabia_saudita:'
    },
    u'\U0000264F': { # ♏
        'en' : ':Scorpio:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':scorpius:'],
        'variant': True,
        'de': ':skorpion_(sternzeichen):',
        'es': ':escorpio:',
        'fr': ':scorpion_zodiaque:',
        'pt': u':signo_de_escorpião:',
        'it': ':segno_zodiacale_dello_scorpione:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0073\U000E0063\U000E0074\U000E007F': { # 🏴󠁧󠁢󠁳󠁣󠁴󠁿
        'en' : ':Scotland:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':scotland:'],
        'de': ':flagge_schottland:',
        'es': ':bandera_escocia:',
        'fr': u':drapeau_écosse:',
        'pt': u':bandeira_escócia:',
        'it': ':bandiera_scozia:'
    },
    u'\U0001F1F8\U0001F1F3': { # 🇸🇳
        'en' : ':Senegal:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Senegal:'],
        'de': ':flagge_senegal:',
        'es': ':bandera_senegal:',
        'fr': u':drapeau_sénégal:',
        'pt': ':bandeira_senegal:',
        'it': ':bandiera_senegal:'
    },
    u'\U0001F1F7\U0001F1F8': { # 🇷🇸
        'en' : ':Serbia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Serbia:'],
        'de': ':flagge_serbien:',
        'es': ':bandera_serbia:',
        'fr': ':drapeau_serbie:',
        'pt': u':bandeira_sérvia:',
        'it': ':bandiera_serbia:'
    },
    u'\U0001F1F8\U0001F1E8': { # 🇸🇨
        'en' : ':Seychelles:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Seychelles:'],
        'de': ':flagge_seychellen:',
        'es': ':bandera_seychelles:',
        'fr': ':drapeau_seychelles:',
        'pt': ':bandeira_seicheles:',
        'it': ':bandiera_seychelles:'
    },
    u'\U0001F1F8\U0001F1F1': { # 🇸🇱
        'en' : ':Sierra_Leone:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Sierra_Leone:'],
        'de': ':flagge_sierra_leone:',
        'es': ':bandera_sierra_leona:',
        'fr': ':drapeau_sierra_leone:',
        'pt': ':bandeira_serra_leoa:',
        'it': ':bandiera_sierra_leone:'
    },
    u'\U0001F1F8\U0001F1EC': { # 🇸🇬
        'en' : ':Singapore:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Singapore:'],
        'de': ':flagge_singapur:',
        'es': ':bandera_singapur:',
        'fr': ':drapeau_singapour:',
        'pt': ':bandeira_singapura:',
        'it': ':bandiera_singapore:'
    },
    u'\U0001F1F8\U0001F1FD': { # 🇸🇽
        'en' : ':Sint_Maarten:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Sint_Maarten:'],
        'de': ':flagge_sint_maarten:',
        'es': ':bandera_sint_maarten:',
        'fr': u':drapeau_saint-martin_(partie_néerlandaise):',
        'pt': ':bandeira_sint_maarten:',
        'it': ':bandiera_sint_maarten:'
    },
    u'\U0001F1F8\U0001F1F0': { # 🇸🇰
        'en' : ':Slovakia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Slovakia:'],
        'de': ':flagge_slowakei:',
        'es': ':bandera_eslovaquia:',
        'fr': ':drapeau_slovaquie:',
        'pt': u':bandeira_eslováquia:',
        'it': ':bandiera_slovacchia:'
    },
    u'\U0001F1F8\U0001F1EE': { # 🇸🇮
        'en' : ':Slovenia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Slovenia:'],
        'de': ':flagge_slowenien:',
        'es': ':bandera_eslovenia:',
        'fr': u':drapeau_slovénie:',
        'pt': u':bandeira_eslovênia:',
        'it': ':bandiera_slovenia:'
    },
    u'\U0001F1F8\U0001F1E7': { # 🇸🇧
        'en' : ':Solomon_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Solomon_Islands:'],
        'de': ':flagge_salomonen:',
        'es': u':bandera_islas_salomón:',
        'fr': u':drapeau_îles_salomon:',
        'pt': u':bandeira_ilhas_salomão:',
        'it': ':bandiera_isole_salomone:'
    },
    u'\U0001F1F8\U0001F1F4': { # 🇸🇴
        'en' : ':Somalia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Somalia:'],
        'de': ':flagge_somalia:',
        'es': ':bandera_somalia:',
        'fr': ':drapeau_somalie:',
        'pt': u':bandeira_somália:',
        'it': ':bandiera_somalia:'
    },
    u'\U0001F1FF\U0001F1E6': { # 🇿🇦
        'en' : ':South_Africa:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_South_Africa:'],
        'de': u':flagge_südafrika:',
        'es': u':bandera_sudáfrica:',
        'fr': ':drapeau_afrique_du_sud:',
        'pt': u':bandeira_áfrica_do_sul:',
        'it': ':bandiera_sudafrica:'
    },
    u'\U0001F1EC\U0001F1F8': { # 🇬🇸
        'en' : ':South_Georgia_&_South_Sandwich_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_South_Georgia_&_South_Sandwich_Islands:'],
        'de': u':flagge_südgeorgien_und_die_südlichen_sandwichinseln:',
        'es': ':bandera_islas_georgia_del_sur_y_sandwich_del_sur:',
        'fr': u':drapeau_géorgie_du_sud_et_îles_sandwich_du_sud:',
        'pt': u':bandeira_ilhas_geórgia_do_sul_e_sandwich_do_sul:',
        'it': ':bandiera_georgia_del_sud_e_sandwich_australi:'
    },
    u'\U0001F1F0\U0001F1F7': { # 🇰🇷
        'en' : ':South_Korea:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_South_Korea:'],
        'de': u':flagge_südkorea:',
        'es': ':bandera_corea_del_sur:',
        'fr': u':drapeau_corée_du_sud:',
        'pt': ':bandeira_coreia_do_sul:',
        'it': ':bandiera_corea_del_sud:'
    },
    u'\U0001F1F8\U0001F1F8': { # 🇸🇸
        'en' : ':South_Sudan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_South_Sudan:'],
        'de': u':flagge_südsudan:',
        'es': u':bandera_sudán_del_sur:',
        'fr': ':drapeau_soudan_du_sud:',
        'pt': u':bandeira_sudão_do_sul:',
        'it': ':bandiera_sud_sudan:'
    },
    u'\U0001F1EA\U0001F1F8': { # 🇪🇸
        'en' : ':Spain:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_Spain:'],
        'de': ':flagge_spanien:',
        'es': u':bandera_españa:',
        'fr': ':drapeau_espagne:',
        'pt': ':bandeira_espanha:',
        'it': ':bandiera_spagna:'
    },
    u'\U0001F1F1\U0001F1F0': { # 🇱🇰
        'en' : ':Sri_Lanka:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Sri_Lanka:'],
        'de': ':flagge_sri_lanka:',
        'es': ':bandera_sri_lanka:',
        'fr': ':drapeau_sri_lanka:',
        'pt': ':bandeira_sri_lanka:',
        'it': ':bandiera_sri_lanka:'
    },
    u'\U0001F1E7\U0001F1F1': { # 🇧🇱
        'en' : u':St._Barthélemy:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_St._Barthélemy:'],
        'de': u':flagge_st._barthélemy:',
        'es': u':bandera_san_bartolomé:',
        'fr': u':drapeau_saint-barthélemy:',
        'pt': u':bandeira_são_bartolomeu:',
        'it': u':bandiera_saint-barthélemy:'
    },
    u'\U0001F1F8\U0001F1ED': { # 🇸🇭
        'en' : ':St._Helena:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Helena:'],
        'de': ':flagge_st._helena:',
        'es': ':bandera_santa_elena:',
        'fr': u':drapeau_sainte-hélène:',
        'pt': ':bandeira_santa_helena:',
        'it': u':bandiera_sant’elena:'
    },
    u'\U0001F1F0\U0001F1F3': { # 🇰🇳
        'en' : ':St._Kitts_&_Nevis:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Kitts_&_Nevis:'],
        'de': ':flagge_st._kitts_und_nevis:',
        'es': u':bandera_san_cristóbal_y_nieves:',
        'fr': u':drapeau_saint-christophe-et-niévès:',
        'pt': u':bandeira_são_cristóvão_e_névis:',
        'it': ':bandiera_saint_kitts_e_nevis:'
    },
    u'\U0001F1F1\U0001F1E8': { # 🇱🇨
        'en' : ':St._Lucia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Lucia:'],
        'de': ':flagge_st._lucia:',
        'es': u':bandera_santa_lucía:',
        'fr': ':drapeau_sainte-lucie:',
        'pt': u':bandeira_santa_lúcia:',
        'it': ':bandiera_saint_lucia:'
    },
    u'\U0001F1F2\U0001F1EB': { # 🇲🇫
        'en' : ':St._Martin:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Martin:'],
        'de': ':flagge_st._martin:',
        'es': u':bandera_san_martín:',
        'fr': ':drapeau_saint-martin:',
        'pt': u':bandeira_são_martinho:',
        'it': ':bandiera_saint_martin:'
    },
    u'\U0001F1F5\U0001F1F2': { # 🇵🇲
        'en' : ':St._Pierre_&_Miquelon:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Pierre_&_Miquelon:'],
        'de': ':flagge_st._pierre_und_miquelon:',
        'es': u':bandera_san_pedro_y_miquelón:',
        'fr': ':drapeau_saint-pierre-et-miquelon:',
        'pt': u':bandeira_são_pedro_e_miquelão:',
        'it': ':bandiera_saint-pierre_e_miquelon:'
    },
    u'\U0001F1FB\U0001F1E8': { # 🇻🇨
        'en' : ':St._Vincent_&_Grenadines:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_St._Vincent_&_Grenadines:'],
        'de': ':flagge_st._vincent_und_die_grenadinen:',
        'es': ':bandera_san_vicente_y_las_granadinas:',
        'fr': ':drapeau_saint-vincent-et-les-grenadines:',
        'pt': u':bandeira_são_vicente_e_granadinas:',
        'it': ':bandiera_saint_vincent_e_grenadine:'
    },
    u'\U0001F5FD': { # 🗽
        'en' : ':Statue_of_Liberty:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':statue_of_liberty:'],
        'de': ':freiheitsstatue:',
        'es': ':estatua_de_la_libertad:',
        'fr': u':statue_de_la_liberté:',
        'pt': u':estátua_da_liberdade:',
        'it': u':statua_della_libertà:'
    },
    u'\U0001F1F8\U0001F1E9': { # 🇸🇩
        'en' : ':Sudan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Sudan:'],
        'de': ':flagge_sudan:',
        'es': u':bandera_sudán:',
        'fr': ':drapeau_soudan:',
        'pt': u':bandeira_sudão:',
        'it': ':bandiera_sudan:'
    },
    u'\U0001F1F8\U0001F1F7': { # 🇸🇷
        'en' : ':Suriname:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Suriname:'],
        'de': ':flagge_suriname:',
        'es': ':bandera_surinam:',
        'fr': ':drapeau_suriname:',
        'pt': ':bandeira_suriname:',
        'it': ':bandiera_suriname:'
    },
    u'\U0001F1F8\U0001F1EF': { # 🇸🇯
        'en' : ':Svalbard_&_Jan_Mayen:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Svalbard_&_Jan_Mayen:'],
        'de': ':flagge_spitzbergen_und_jan_mayen:',
        'es': ':bandera_svalbard_y_jan_mayen:',
        'fr': ':drapeau_svalbard_et_jan_mayen:',
        'pt': ':bandeira_svalbard_e_jan_mayen:',
        'it': ':bandiera_svalbard_e_jan_mayen:'
    },
    u'\U0001F1F8\U0001F1EA': { # 🇸🇪
        'en' : ':Sweden:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Sweden:'],
        'de': ':flagge_schweden:',
        'es': ':bandera_suecia:',
        'fr': u':drapeau_suède:',
        'pt': u':bandeira_suécia:',
        'it': ':bandiera_svezia:'
    },
    u'\U0001F1E8\U0001F1ED': { # 🇨🇭
        'en' : ':Switzerland:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Switzerland:'],
        'de': ':flagge_schweiz:',
        'es': ':bandera_suiza:',
        'fr': ':drapeau_suisse:',
        'pt': u':bandeira_suíça:',
        'it': ':bandiera_svizzera:'
    },
    u'\U0001F1F8\U0001F1FE': { # 🇸🇾
        'en' : ':Syria:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Syria:'],
        'de': ':flagge_syrien:',
        'es': ':bandera_siria:',
        'fr': ':drapeau_syrie:',
        'pt': u':bandeira_síria:',
        'it': ':bandiera_siria:'
    },
    u'\U0001F1F8\U0001F1F9': { # 🇸🇹
        'en' : u':São_Tomé_&_Príncipe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [u':flag_for_São_Tomé_&_Príncipe:'],
        'de': u':flagge_são_tomé_und_príncipe:',
        'es': u':bandera_santo_tomé_y_príncipe:',
        'fr': u':drapeau_sao_tomé-et-principe:',
        'pt': u':bandeira_são_tomé_e_príncipe:',
        'it': u':bandiera_são_tomé_e_príncipe:'
    },
    u'\U0001F996': { # 🦖
        'en' : ':T-Rex:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':t-rex:'],
        'de': ':t-rex:',
        'es': ':t-rex:',
        'fr': ':t-rex:',
        'pt': ':tiranossauro_rex:',
        'it': ':t-rex:'
    },
    u'\U0001F51D': { # 🔝
        'en' : ':TOP_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':top:'],
        'de': ':top-pfeil:',
        'es': ':flecha_top:',
        'fr': u':flèche_en_haut:',
        'pt': ':seta_top:',
        'it': ':freccia_top:'
    },
    u'\U0001F1F9\U0001F1FC': { # 🇹🇼
        'en' : ':Taiwan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Taiwan:'],
        'de': ':flagge_taiwan:',
        'es': u':bandera_taiwán:',
        'fr': u':drapeau_taïwan:',
        'pt': ':bandeira_taiwan:',
        'it': ':bandiera_taiwan:'
    },
    u'\U0001F1F9\U0001F1EF': { # 🇹🇯
        'en' : ':Tajikistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tajikistan:'],
        'de': ':flagge_tadschikistan:',
        'es': u':bandera_tayikistán:',
        'fr': ':drapeau_tadjikistan:',
        'pt': u':bandeira_tadjiquistão:',
        'it': ':bandiera_tagikistan:'
    },
    u'\U0001F1F9\U0001F1FF': { # 🇹🇿
        'en' : ':Tanzania:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tanzania:'],
        'de': ':flagge_tansania:',
        'es': ':bandera_tanzania:',
        'fr': ':drapeau_tanzanie:',
        'pt': u':bandeira_tanzânia:',
        'it': ':bandiera_tanzania:'
    },
    u'\U00002649': { # ♉
        'en' : ':Taurus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':taurus:'],
        'variant': True,
        'de': ':stier_(sternzeichen):',
        'es': ':tauro:',
        'fr': ':taureau:',
        'pt': ':signo_de_touro:',
        'it': ':segno_zodiacale_del_toro:'
    },
    u'\U0001F1F9\U0001F1ED': { # 🇹🇭
        'en' : ':Thailand:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Thailand:'],
        'de': ':flagge_thailand:',
        'es': ':bandera_tailandia:',
        'fr': u':drapeau_thaïlande:',
        'pt': u':bandeira_tailândia:',
        'it': ':bandiera_thailandia:'
    },
    u'\U0001F1F9\U0001F1F1': { # 🇹🇱
        'en' : ':Timor-Leste:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Timor__Leste:'],
        'de': ':flagge_timor-leste:',
        'es': ':bandera_timor-leste:',
        'fr': ':drapeau_timor_oriental:',
        'pt': ':bandeira_timor-leste:',
        'it': ':bandiera_timor_est:'
    },
    u'\U0001F1F9\U0001F1EC': { # 🇹🇬
        'en' : ':Togo:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Togo:'],
        'de': ':flagge_togo:',
        'es': ':bandera_togo:',
        'fr': ':drapeau_togo:',
        'pt': ':bandeira_togo:',
        'it': ':bandiera_togo:'
    },
    u'\U0001F1F9\U0001F1F0': { # 🇹🇰
        'en' : ':Tokelau:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tokelau:'],
        'de': ':flagge_tokelau:',
        'es': ':bandera_tokelau:',
        'fr': ':drapeau_tokelau:',
        'pt': ':bandeira_tokelau:',
        'it': ':bandiera_tokelau:'
    },
    u'\U0001F5FC': { # 🗼
        'en' : ':Tokyo_tower:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':tokyo_tower:'],
        'de': ':tokyo_tower:',
        'es': ':torre_de_tokio:',
        'fr': ':tour_de_tokyo:',
        'pt': u':torre_de_tóquio:',
        'it': ':torre_di_tokyo:'
    },
    u'\U0001F1F9\U0001F1F4': { # 🇹🇴
        'en' : ':Tonga:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tonga:'],
        'de': ':flagge_tonga:',
        'es': ':bandera_tonga:',
        'fr': ':drapeau_tonga:',
        'pt': ':bandeira_tonga:',
        'it': ':bandiera_tonga:'
    },
    u'\U0001F1F9\U0001F1F9': { # 🇹🇹
        'en' : ':Trinidad_&_Tobago:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Trinidad_&_Tobago:'],
        'de': ':flagge_trinidad_und_tobago:',
        'es': ':bandera_trinidad_y_tobago:',
        'fr': u':drapeau_trinité-et-tobago:',
        'pt': ':bandeira_trinidad_e_tobago:',
        'it': ':bandiera_trinidad_e_tobago:'
    },
    u'\U0001F1F9\U0001F1E6': { # 🇹🇦
        'en' : ':Tristan_da_Cunha:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tristan_da_Cunha:'],
        'de': ':flagge_tristan_da_cunha:',
        'es': u':bandera_tristán_de_acuña:',
        'fr': ':drapeau_tristan_da_cunha:',
        'pt': u':bandeira_tristão_da_cunha:',
        'it': ':bandiera_tristan_da_cunha:'
    },
    u'\U0001F1F9\U0001F1F3': { # 🇹🇳
        'en' : ':Tunisia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tunisia:'],
        'de': ':flagge_tunesien:',
        'es': u':bandera_túnez:',
        'fr': ':drapeau_tunisie:',
        'pt': u':bandeira_tunísia:',
        'it': ':bandiera_tunisia:'
    },
    u'\U0001F1F9\U0001F1F7': { # 🇹🇷
        'en' : ':Turkey:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Turkey:'],
        'de': u':flagge_türkei:',
        'es': u':bandera_turquía:',
        'fr': ':drapeau_turquie:',
        'pt': ':bandeira_turquia:',
        'it': ':bandiera_turchia:'
    },
    u'\U0001F1F9\U0001F1F2': { # 🇹🇲
        'en' : ':Turkmenistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Turkmenistan:'],
        'de': ':flagge_turkmenistan:',
        'es': u':bandera_turkmenistán:',
        'fr': u':drapeau_turkménistan:',
        'pt': u':bandeira_turcomenistão:',
        'it': ':bandiera_turkmenistan:'
    },
    u'\U0001F1F9\U0001F1E8': { # 🇹🇨
        'en' : ':Turks_&_Caicos_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Turks_&_Caicos_Islands:'],
        'de': ':flagge_turks-_und_caicosinseln:',
        'es': ':bandera_islas_turcas_y_caicos:',
        'fr': u':drapeau_îles_turques-et-caïques:',
        'pt': ':bandeira_ilhas_turcas_e_caicos:',
        'it': ':bandiera_isole_turks_e_caicos:'
    },
    u'\U0001F1F9\U0001F1FB': { # 🇹🇻
        'en' : ':Tuvalu:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Tuvalu:'],
        'de': ':flagge_tuvalu:',
        'es': ':bandera_tuvalu:',
        'fr': ':drapeau_tuvalu:',
        'pt': ':bandeira_tuvalu:',
        'it': ':bandiera_tuvalu:'
    },
    u'\U0001F1FA\U0001F1F2': { # 🇺🇲
        'en' : ':U.S._Outlying_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_U.S._Outlying_Islands:'],
        'de': u':flagge_amerikanische_überseeinseln:',
        'es': ':bandera_islas_menores_alejadas_de_ee._uu.:',
        'fr': u':drapeau_îles_mineures_éloignées_des_états-unis:',
        'pt': ':bandeira_ilhas_menores_distantes_dos_eua:',
        'it': ':bandiera_altre_isole_americane_del_pacifico:'
    },
    u'\U0001F1FB\U0001F1EE': { # 🇻🇮
        'en' : ':U.S._Virgin_Islands:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_U.S._Virgin_Islands:'],
        'de': ':flagge_amerikanische_jungferninseln:',
        'es': u':bandera_islas_vírgenes_de_ee._uu.:',
        'fr': u':drapeau_îles_vierges_des_états-unis:',
        'pt': ':bandeira_ilhas_virgens_americanas:',
        'it': ':bandiera_isole_vergini_americane:'
    },
    u'\U0001F199': { # 🆙
        'en' : ':UP!_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':up:'],
        'de': ':schriftzug_up!_im_blauen_quadrat:',
        'es': u':botón_up!:',
        'fr': ':bouton_vers_le_haut:',
        'pt': u':botão_up!:',
        'it': ':pulsante_up!:'
    },
    u'\U0001F1FA\U0001F1EC': { # 🇺🇬
        'en' : ':Uganda:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Uganda:'],
        'de': ':flagge_uganda:',
        'es': ':bandera_uganda:',
        'fr': ':drapeau_ouganda:',
        'pt': ':bandeira_uganda:',
        'it': ':bandiera_uganda:'
    },
    u'\U0001F1FA\U0001F1E6': { # 🇺🇦
        'en' : ':Ukraine:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Ukraine:'],
        'de': ':flagge_ukraine:',
        'es': ':bandera_ucrania:',
        'fr': ':drapeau_ukraine:',
        'pt': u':bandeira_ucrânia:',
        'it': ':bandiera_ucraina:'
    },
    u'\U0001F1E6\U0001F1EA': { # 🇦🇪
        'en' : ':United_Arab_Emirates:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_United_Arab_Emirates:'],
        'de': ':flagge_vereinigte_arabische_emirate:',
        'es': u':bandera_emiratos_árabes_unidos:',
        'fr': u':drapeau_émirats_arabes_unis:',
        'pt': u':bandeira_emirados_árabes_unidos:',
        'it': ':bandiera_emirati_arabi_uniti:'
    },
    u'\U0001F1EC\U0001F1E7': { # 🇬🇧
        'en' : ':United_Kingdom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_United_Kingdom:'],
        'de': u':flagge_vereinigtes_königreich:',
        'es': ':bandera_reino_unido:',
        'fr': ':drapeau_royaume-uni:',
        'pt': ':bandeira_reino_unido:',
        'it': ':bandiera_regno_unito:'
    },
    u'\U0001F1FA\U0001F1F3': { # 🇺🇳
        'en' : ':United_Nations:',
        'status' : fully_qualified,
        'E' : 4,
        'alias' : [':united_nations:'],
        'de': ':flagge_vereinte_nationen:',
        'es': ':bandera_naciones_unidas:',
        'fr': ':drapeau_nations_unies:',
        'pt': u':bandeira_nações_unidas:',
        'it': ':bandiera_nazioni_unite:'
    },
    u'\U0001F1FA\U0001F1F8': { # 🇺🇸
        'en' : ':United_States:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flag_for_United_States:'],
        'de': ':flagge_vereinigte_staaten:',
        'es': ':bandera_estados_unidos:',
        'fr': u':drapeau_états-unis:',
        'pt': ':bandeira_estados_unidos:',
        'it': ':bandiera_stati_uniti:'
    },
    u'\U0001F1FA\U0001F1FE': { # 🇺🇾
        'en' : ':Uruguay:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Uruguay:'],
        'de': ':flagge_uruguay:',
        'es': ':bandera_uruguay:',
        'fr': ':drapeau_uruguay:',
        'pt': ':bandeira_uruguai:',
        'it': ':bandiera_uruguay:'
    },
    u'\U0001F1FA\U0001F1FF': { # 🇺🇿
        'en' : ':Uzbekistan:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Uzbekistan:'],
        'de': ':flagge_usbekistan:',
        'es': u':bandera_uzbekistán:',
        'fr': u':drapeau_ouzbékistan:',
        'pt': u':bandeira_uzbequistão:',
        'it': ':bandiera_uzbekistan:'
    },
    u'\U0001F19A': { # 🆚
        'en' : ':VS_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':vs:'],
        'de': ':schriftzug_vs_in_orangem_quadrat:',
        'es': u':botón_vs:',
        'fr': ':bouton_vs:',
        'pt': u':botão_vs:',
        'it': ':pulsante_vs:'
    },
    u'\U0001F1FB\U0001F1FA': { # 🇻🇺
        'en' : ':Vanuatu:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Vanuatu:'],
        'de': ':flagge_vanuatu:',
        'es': ':bandera_vanuatu:',
        'fr': ':drapeau_vanuatu:',
        'pt': ':bandeira_vanuatu:',
        'it': ':bandiera_vanuatu:'
    },
    u'\U0001F1FB\U0001F1E6': { # 🇻🇦
        'en' : ':Vatican_City:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Vatican_City:'],
        'de': ':flagge_vatikanstadt:',
        'es': ':bandera_ciudad_del_vaticano:',
        'fr': u':drapeau_état_de_la_cité_du_vatican:',
        'pt': ':bandeira_cidade_do_vaticano:',
        'it': u':bandiera_città_del_vaticano:'
    },
    u'\U0001F1FB\U0001F1EA': { # 🇻🇪
        'en' : ':Venezuela:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Venezuela:'],
        'de': ':flagge_venezuela:',
        'es': ':bandera_venezuela:',
        'fr': ':drapeau_venezuela:',
        'pt': ':bandeira_venezuela:',
        'it': ':bandiera_venezuela:'
    },
    u'\U0001F1FB\U0001F1F3': { # 🇻🇳
        'en' : ':Vietnam:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Vietnam:'],
        'de': ':flagge_vietnam:',
        'es': ':bandera_vietnam:',
        'fr': ':drapeau_vietnam:',
        'pt': u':bandeira_vietnã:',
        'it': ':bandiera_vietnam:'
    },
    u'\U0000264D': { # ♍
        'en' : ':Virgo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':virgo:'],
        'variant': True,
        'de': ':jungfrau_(sternzeichen):',
        'es': ':virgo:',
        'fr': ':vierge:',
        'pt': ':signo_de_virgem:',
        'it': ':segno_zodiacale_della_vergine:'
    },
    u'\U0001F3F4\U000E0067\U000E0062\U000E0077\U000E006C\U000E0073\U000E007F': { # 🏴󠁧󠁢󠁷󠁬󠁳󠁿
        'en' : ':Wales:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':wales:'],
        'de': ':flagge_wales:',
        'es': ':bandera_gales:',
        'fr': ':drapeau_pays_de_galles:',
        'pt': u':bandeira_país_de_gales:',
        'it': ':bandiera_galles:'
    },
    u'\U0001F1FC\U0001F1EB': { # 🇼🇫
        'en' : ':Wallis_&_Futuna:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Wallis_&_Futuna:'],
        'de': ':flagge_wallis_und_futuna:',
        'es': ':bandera_wallis_y_futuna:',
        'fr': ':drapeau_wallis-et-futuna:',
        'pt': ':bandeira_wallis_e_futuna:',
        'it': ':bandiera_wallis_e_futuna:'
    },
    u'\U0001F1EA\U0001F1ED': { # 🇪🇭
        'en' : ':Western_Sahara:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Western_Sahara:'],
        'de': ':flagge_westsahara:',
        'es': u':bandera_sáhara_occidental:',
        'fr': ':drapeau_sahara_occidental:',
        'pt': ':bandeira_saara_ocidental:',
        'it': ':bandiera_sahara_occidentale:'
    },
    u'\U0001F1FE\U0001F1EA': { # 🇾🇪
        'en' : ':Yemen:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Yemen:'],
        'de': ':flagge_jemen:',
        'es': ':bandera_yemen:',
        'fr': u':drapeau_yémen:',
        'pt': u':bandeira_iêmen:',
        'it': ':bandiera_yemen:'
    },
    u'\U0001F1FF\U0001F1F2': { # 🇿🇲
        'en' : ':Zambia:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Zambia:'],
        'de': ':flagge_sambia:',
        'es': ':bandera_zambia:',
        'fr': ':drapeau_zambie:',
        'pt': u':bandeira_zâmbia:',
        'it': ':bandiera_zambia:'
    },
    u'\U0001F1FF\U0001F1FC': { # 🇿🇼
        'en' : ':Zimbabwe:',
        'status' : fully_qualified,
        'E' : 2,
        'alias' : [':flag_for_Zimbabwe:'],
        'de': ':flagge_simbabwe:',
        'es': ':bandera_zimbabue:',
        'fr': ':drapeau_zimbabwe:',
        'pt': u':bandeira_zimbábue:',
        'it': ':bandiera_zimbabwe:'
    },
    u'\U0001F9EE': { # 🧮
        'en' : ':abacus:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':abakus:',
        'es': u':ábaco:',
        'fr': ':abaque:',
        'pt': u':ábaco:',
        'it': ':abaco:'
    },
    u'\U0001FA97': { # 🪗
        'en' : ':accordion:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':akkordeon:',
        'es': u':acordeón:',
        'fr': u':accordéon:',
        'pt': u':acordeão:',
        'it': ':fisarmonica:'
    },
    u'\U0001FA79': { # 🩹
        'en' : ':adhesive_bandage:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':heftpflaster:',
        'es': ':tirita:',
        'fr': ':sparadrap:',
        'pt': ':atadura_adesiva:',
        'it': ':cerotto:'
    },
    u'\U0001F39F\U0000FE0F': { # 🎟️
        'en' : ':admission_tickets:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':tickets:'],
        'variant': True,
        'de': ':eintrittskarten:',
        'es': ':entradas:',
        'fr': u':billet_d’entrée:',
        'pt': ':ingresso_de_cinema:',
        'it': u':biglietto_d’ingresso:'
    },
    u'\U0001F39F': { # 🎟
        'en' : ':admission_tickets:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':tickets:'],
        'variant': True,
        'de': ':eintrittskarten:',
        'es': ':entradas:',
        'fr': u':billet_d’entrée:',
        'pt': ':ingresso_de_cinema:',
        'it': u':biglietto_d’ingresso:'
    },
    u'\U0001F6A1': { # 🚡
        'en' : ':aerial_tramway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bergseilbahn:',
        'es': u':teleférico:',
        'fr': u':tramway_aérien:',
        'pt': u':teleférico:',
        'it': ':funivia_aerea:'
    },
    u'\U00002708\U0000FE0F': { # ✈️
        'en' : ':airplane:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':flugzeug:',
        'es': u':avión:',
        'fr': ':avion:',
        'pt': u':avião:',
        'it': ':aeroplano:'
    },
    u'\U00002708': { # ✈
        'en' : ':airplane:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':flugzeug:',
        'es': u':avión:',
        'fr': ':avion:',
        'pt': u':avião:',
        'it': ':aeroplano:'
    },
    u'\U0001F6EC': { # 🛬
        'en' : ':airplane_arrival:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':airplane_arriving:', ':flight_arrival:'],
        'de': ':landung_eines_flugzeugs:',
        'es': u':avión_aterrizando:',
        'fr': u':avion_à_l’atterrissage:',
        'pt': u':avião_aterrissando:',
        'it': ':atterraggio:'
    },
    u'\U0001F6EB': { # 🛫
        'en' : ':airplane_departure:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':flight_departure:'],
        'de': ':abflug:',
        'es': u':avión_despegando:',
        'fr': u':avion_au_décollage:',
        'pt': u':avião_decolando:',
        'it': ':decollo:'
    },
    u'\U000023F0': { # ⏰
        'en' : ':alarm_clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':wecker:',
        'es': ':reloj_despertador:',
        'fr': u':réveil:',
        'pt': ':despertador:',
        'it': ':sveglia:'
    },
    u'\U00002697\U0000FE0F': { # ⚗️
        'en' : ':alembic:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':destillierapparat:',
        'es': ':alambique:',
        'fr': ':alambic:',
        'pt': ':alambique:',
        'it': ':alambicco:'
    },
    u'\U00002697': { # ⚗
        'en' : ':alembic:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':destillierapparat:',
        'es': ':alambique:',
        'fr': ':alambic:',
        'pt': ':alambique:',
        'it': ':alambicco:'
    },
    u'\U0001F47D': { # 👽
        'en' : ':alien:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':außerirdischer:',
        'es': u':alienígena:',
        'fr': ':alien:',
        'pt': u':alienígena:',
        'it': ':alieno:'
    },
    u'\U0001F47E': { # 👾
        'en' : ':alien_monster:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':space_invader:'],
        'de': ':computerspiel-monster:',
        'es': u':monstruo_alienígena:',
        'fr': u':monstre_de_l’espace:',
        'pt': u':monstro_alienígena:',
        'it': ':mostro_alieno:'
    },
    u'\U0001F691': { # 🚑
        'en' : ':ambulance:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':krankenwagen:',
        'es': ':ambulancia:',
        'fr': ':ambulance:',
        'pt': u':ambulância:',
        'it': ':ambulanza:'
    },
    u'\U0001F3C8': { # 🏈
        'en' : ':american_football:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':football:'],
        'de': ':football:',
        'es': u':balón_de_fútbol_americano:',
        'fr': u':football_américain:',
        'pt': ':bola_de_futebol_americano:',
        'it': ':football_americano:'
    },
    u'\U0001F3FA': { # 🏺
        'en' : ':amphora:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':amphore:',
        'es': u':ánfora:',
        'fr': ':amphore:',
        'pt': u':ânfora:',
        'it': ':anfora:'
    },
    u'\U0001FAC0': { # 🫀
        'en' : ':anatomical_heart:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':herz_(organ):',
        'es': u':corazón_humano:',
        'fr': u':cœur:',
        'pt': u':coração_humano:',
        'it': ':organo_del_cuore:'
    },
    u'\U00002693': { # ⚓
        'en' : ':anchor:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':anker:',
        'es': ':ancla:',
        'fr': ':ancre:',
        'pt': u':âncora:',
        'it': ':ancora:'
    },
    u'\U0001F4A2': { # 💢
        'en' : ':anger_symbol:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':anger:'],
        'de': u':ärger:',
        'es': u':símbolo_de_enfado:',
        'fr': u':symbole_de_colère:',
        'pt': u':símbolo_de_raiva:',
        'it': ':rabbia:'
    },
    u'\U0001F620': { # 😠
        'en' : ':angry_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':angry:'],
        'de': u':verärgertes_gesicht:',
        'es': ':cara_enfadada:',
        'fr': u':visage_en_colère:',
        'pt': ':rosto_zangado:',
        'it': ':faccina_arrabbiata:'
    },
    u'\U0001F47F': { # 👿
        'en' : ':angry_face_with_horns:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':imp:'],
        'de': u':wütendes_gesicht_mit_hörnern:',
        'es': ':cara_enfadada_con_cuernos:',
        'fr': u':visage_en_colère_avec_des_cornes:',
        'pt': ':rosto_zangado_com_chifres:',
        'it': ':faccina_arrabbiata_con_corna:'
    },
    u'\U0001F627': { # 😧
        'en' : ':anguished_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':anguished:'],
        'de': ':qualvolles_gesicht:',
        'es': ':cara_angustiada:',
        'fr': u':visage_angoissé:',
        'pt': ':rosto_angustiado:',
        'it': ':faccina_angosciata:'
    },
    u'\U0001F41C': { # 🐜
        'en' : ':ant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':ameise:',
        'es': ':hormiga:',
        'fr': ':fourmi:',
        'pt': ':formiga:',
        'it': ':formica:'
    },
    u'\U0001F4F6': { # 📶
        'en' : ':antenna_bars:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':signal_strength:'],
        'de': u':balkenförmige_signalstärkenanzeige:',
        'es': ':barras_de_cobertura:',
        'fr': u':barres_de_réseau:',
        'pt': ':barras_de_sinal:',
        'it': ':segnale_cellulare:'
    },
    u'\U0001F630': { # 😰
        'en' : ':anxious_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cold_sweat:'],
        'de': u':besorgtes_gesicht_mit_schweißtropfen:',
        'es': ':cara_con_ansiedad_y_sudor:',
        'fr': ':visage_anxieux_avec_goutte_de_sueur:',
        'pt': ':rosto_ansioso_com_gota_de_suor:',
        'it': ':faccina_sudata_in_ansia:'
    },
    u'\U0001F69B': { # 🚛
        'en' : ':articulated_lorry:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sattelzug:',
        'es': u':camión_articulado:',
        'fr': ':semi-remorque:',
        'pt': u':caminhão:',
        'it': ':autoarticolato:'
    },
    u'\U0001F9D1\U0000200D\U0001F3A8': { # 🧑‍🎨
        'en' : ':artist:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in):',
        'es': ':artista:',
        'fr': ':artiste:',
        'pt': ':artista:',
        'it': ':artista:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F3A8': { # 🧑🏿‍🎨
        'en' : ':artist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_dunkle_hautfarbe:',
        'es': ':artista_tono_de_piel_oscuro:',
        'fr': u':artiste_peau_foncée:',
        'pt': ':artista_pele_escura:',
        'it': ':artista_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F3A8': { # 🧑🏻‍🎨
        'en' : ':artist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_helle_hautfarbe:',
        'es': ':artista_tono_de_piel_claro:',
        'fr': ':artiste_peau_claire:',
        'pt': ':artista_pele_clara:',
        'it': ':artista_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F3A8': { # 🧑🏾‍🎨
        'en' : ':artist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mitteldunkle_hautfarbe:',
        'es': ':artista_tono_de_piel_oscuro_medio:',
        'fr': ':artiste_peau_mate:',
        'pt': ':artista_pele_morena_escura:',
        'it': ':artista_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F3A8': { # 🧑🏼‍🎨
        'en' : ':artist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mittelhelle_hautfarbe:',
        'es': ':artista_tono_de_piel_claro_medio:',
        'fr': ':artiste_peau_moyennement_claire:',
        'pt': ':artista_pele_morena_clara:',
        'it': ':artista_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F3A8': { # 🧑🏽‍🎨
        'en' : ':artist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':künstler(in)_mittlere_hautfarbe:',
        'es': ':artista_tono_de_piel_medio:',
        'fr': u':artiste_peau_légèrement_mate:',
        'pt': ':artista_pele_morena:',
        'it': ':artista_carnagione_olivastra:'
    },
    u'\U0001F3A8': { # 🎨
        'en' : ':artist_palette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':art:'],
        'de': ':mischpalette:',
        'es': ':paleta_de_pintor:',
        'fr': ':palette_de_peinture:',
        'pt': ':paleta_de_tintas:',
        'it': ':tavolozza:'
    },
    u'\U0001F632': { # 😲
        'en' : ':astonished_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':astonished:'],
        'de': ':erstauntes_gesicht:',
        'es': ':cara_asombrada:',
        'fr': u':visage_stupéfait:',
        'pt': ':rosto_espantado:',
        'it': ':faccina_stupita:'
    },
    u'\U0001F9D1\U0000200D\U0001F680': { # 🧑‍🚀
        'en' : ':astronaut:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in):',
        'es': ':astronauta:',
        'fr': ':astronaute:',
        'pt': ':astronauta:',
        'it': ':astronauta:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F680': { # 🧑🏿‍🚀
        'en' : ':astronaut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in)_dunkle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_oscuro:',
        'fr': u':astronaute_peau_foncée:',
        'pt': ':astronauta_pele_escura:',
        'it': ':astronauta_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F680': { # 🧑🏻‍🚀
        'en' : ':astronaut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in)_helle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_claro:',
        'fr': ':astronaute_peau_claire:',
        'pt': ':astronauta_pele_clara:',
        'it': ':astronauta_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F680': { # 🧑🏾‍🚀
        'en' : ':astronaut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in)_mitteldunkle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_oscuro_medio:',
        'fr': ':astronaute_peau_mate:',
        'pt': ':astronauta_pele_morena_escura:',
        'it': ':astronauta_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F680': { # 🧑🏼‍🚀
        'en' : ':astronaut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in)_mittelhelle_hautfarbe:',
        'es': ':astronauta_tono_de_piel_claro_medio:',
        'fr': ':astronaute_peau_moyennement_claire:',
        'pt': ':astronauta_pele_morena_clara:',
        'it': ':astronauta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F680': { # 🧑🏽‍🚀
        'en' : ':astronaut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':astronaut(in)_mittlere_hautfarbe:',
        'es': ':astronauta_tono_de_piel_medio:',
        'fr': u':astronaute_peau_légèrement_mate:',
        'pt': ':astronauta_pele_morena:',
        'it': ':astronauta_carnagione_olivastra:'
    },
    u'\U0000269B\U0000FE0F': { # ⚛️
        'en' : ':atom_symbol:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':atomzeichen:',
        'es': u':símbolo_de_átomo:',
        'fr': u':symbole_de_l’atome:',
        'pt': u':símbolo_de_átomo:',
        'it': u':simbolo_dell’atomo:'
    },
    u'\U0000269B': { # ⚛
        'en' : ':atom_symbol:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':atomzeichen:',
        'es': u':símbolo_de_átomo:',
        'fr': u':symbole_de_l’atome:',
        'pt': u':símbolo_de_átomo:',
        'it': u':simbolo_dell’atomo:'
    },
    u'\U0001F6FA': { # 🛺
        'en' : ':auto_rickshaw:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':autorikscha:',
        'es': ':mototaxi:',
        'fr': ':tuk_tuk:',
        'pt': u':automóvel_riquixá:',
        'it': u':risciò_a_motore:'
    },
    u'\U0001F697': { # 🚗
        'en' : ':automobile:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':car:', ':red_car:'],
        'de': ':auto:',
        'es': ':coche:',
        'fr': ':voiture:',
        'pt': ':carro:',
        'it': ':auto:'
    },
    u'\U0001F951': { # 🥑
        'en' : ':avocado:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':avocado:',
        'es': ':aguacate:',
        'fr': ':avocat:',
        'pt': ':abacate:',
        'it': ':avocado:'
    },
    u'\U0001FA93': { # 🪓
        'en' : ':axe:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':axt:',
        'es': ':hacha:',
        'fr': ':hache:',
        'pt': ':machado:',
        'it': ':ascia:'
    },
    u'\U0001F476': { # 👶
        'en' : ':baby:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':baby:',
        'es': u':bebé:',
        'fr': u':bébé:',
        'pt': u':bebê:',
        'it': ':neonato:'
    },
    u'\U0001F47C': { # 👼
        'en' : ':baby_angel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':angel:'],
        'de': ':putte:',
        'es': u':bebé_ángel:',
        'fr': u':bébé_ange:',
        'pt': u':bebê_anjo:',
        'it': ':angioletto:'
    },
    u'\U0001F47C\U0001F3FF': { # 👼🏿
        'en' : ':baby_angel_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':putte_dunkle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_oscuro:',
        'fr': u':bébé_ange_peau_foncée:',
        'pt': u':bebê_anjo_pele_escura:',
        'it': ':angioletto_carnagione_scura:'
    },
    u'\U0001F47C\U0001F3FB': { # 👼🏻
        'en' : ':baby_angel_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':putte_helle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_claro:',
        'fr': u':bébé_ange_peau_claire:',
        'pt': u':bebê_anjo_pele_clara:',
        'it': ':angioletto_carnagione_chiara:'
    },
    u'\U0001F47C\U0001F3FE': { # 👼🏾
        'en' : ':baby_angel_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':putte_mitteldunkle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_oscuro_medio:',
        'fr': u':bébé_ange_peau_mate:',
        'pt': u':bebê_anjo_pele_morena_escura:',
        'it': ':angioletto_carnagione_abbastanza_scura:'
    },
    u'\U0001F47C\U0001F3FC': { # 👼🏼
        'en' : ':baby_angel_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':putte_mittelhelle_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_claro_medio:',
        'fr': u':bébé_ange_peau_moyennement_claire:',
        'pt': u':bebê_anjo_pele_morena_clara:',
        'it': ':angioletto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F47C\U0001F3FD': { # 👼🏽
        'en' : ':baby_angel_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':putte_mittlere_hautfarbe:',
        'es': u':bebé_ángel_tono_de_piel_medio:',
        'fr': u':bébé_ange_peau_légèrement_mate:',
        'pt': u':bebê_anjo_pele_morena:',
        'it': ':angioletto_carnagione_olivastra:'
    },
    u'\U0001F37C': { # 🍼
        'en' : ':baby_bottle:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':babyflasche:',
        'es': u':biberón:',
        'fr': ':biberon:',
        'pt': ':mamadeira:',
        'it': ':biberon:'
    },
    u'\U0001F424': { # 🐤
        'en' : ':baby_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':küken:',
        'es': ':pollito:',
        'fr': ':poussin:',
        'pt': ':pintinho_de_perfil:',
        'it': ':pulcino:'
    },
    u'\U0001F476\U0001F3FF': { # 👶🏿
        'en' : ':baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':baby_dunkle_hautfarbe:',
        'es': u':bebé_tono_de_piel_oscuro:',
        'fr': u':bébé_peau_foncée:',
        'pt': u':bebê_pele_escura:',
        'it': ':neonato_carnagione_scura:'
    },
    u'\U0001F476\U0001F3FB': { # 👶🏻
        'en' : ':baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':baby_helle_hautfarbe:',
        'es': u':bebé_tono_de_piel_claro:',
        'fr': u':bébé_peau_claire:',
        'pt': u':bebê_pele_clara:',
        'it': ':neonato_carnagione_chiara:'
    },
    u'\U0001F476\U0001F3FE': { # 👶🏾
        'en' : ':baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':baby_mitteldunkle_hautfarbe:',
        'es': u':bebé_tono_de_piel_oscuro_medio:',
        'fr': u':bébé_peau_mate:',
        'pt': u':bebê_pele_morena_escura:',
        'it': ':neonato_carnagione_abbastanza_scura:'
    },
    u'\U0001F476\U0001F3FC': { # 👶🏼
        'en' : ':baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':baby_mittelhelle_hautfarbe:',
        'es': u':bebé_tono_de_piel_claro_medio:',
        'fr': u':bébé_peau_moyennement_claire:',
        'pt': u':bebê_pele_morena_clara:',
        'it': ':neonato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F476\U0001F3FD': { # 👶🏽
        'en' : ':baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':baby_mittlere_hautfarbe:',
        'es': u':bebé_tono_de_piel_medio:',
        'fr': u':bébé_peau_légèrement_mate:',
        'pt': u':bebê_pele_morena:',
        'it': ':neonato_carnagione_olivastra:'
    },
    u'\U0001F6BC': { # 🚼
        'en' : ':baby_symbol:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':symbol_baby:',
        'es': u':señal_de_bebé:',
        'fr': u':symbole_bébé:',
        'pt': u':símbolo_de_bebê:',
        'it': ':simbolo_con_immagine_di_bambino:'
    },
    u'\U0001F447': { # 👇
        'en' : ':backhand_index_pointing_down:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':point_down:'],
        'variant': True,
        'de': ':nach_unten_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo:',
        'fr': ':main_avec_index_pointant_vers_le_bas:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo:',
        'it': ':indice_abbassato:'
    },
    u'\U0001F447\U0001F3FF': { # 👇🏿
        'en' : ':backhand_index_pointing_down_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_unten_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_escura:',
        'it': ':indice_abbassato_carnagione_scura:'
    },
    u'\U0001F447\U0001F3FB': { # 👇🏻
        'en' : ':backhand_index_pointing_down_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_unten_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_clara:',
        'it': ':indice_abbassato_carnagione_chiara:'
    },
    u'\U0001F447\U0001F3FE': { # 👇🏾
        'en' : ':backhand_index_pointing_down_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_unten_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_escura:',
        'it': ':indice_abbassato_carnagione_abbastanza_scura:'
    },
    u'\U0001F447\U0001F3FC': { # 👇🏼
        'en' : ':backhand_index_pointing_down_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_unten_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_bas_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena_clara:',
        'it': ':indice_abbassato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F447\U0001F3FD': { # 👇🏽
        'en' : ':backhand_index_pointing_down_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_unten_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_abajo_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_vers_le_bas_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_baixo_pele_morena:',
        'it': ':indice_abbassato_carnagione_olivastra:'
    },
    u'\U0001F448': { # 👈
        'en' : ':backhand_index_pointing_left:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':point_left:'],
        'variant': True,
        'de': ':nach_links_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda:',
        'fr': u':main_avec_index_pointant_à_gauche:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda:',
        'it': ':indice_verso_sinistra:'
    },
    u'\U0001F448\U0001F3FF': { # 👈🏿
        'en' : ':backhand_index_pointing_left_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_links_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_escura:',
        'it': ':indice_verso_sinistra_carnagione_scura:'
    },
    u'\U0001F448\U0001F3FB': { # 👈🏻
        'en' : ':backhand_index_pointing_left_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_links_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_clara:',
        'it': ':indice_verso_sinistra_carnagione_chiara:'
    },
    u'\U0001F448\U0001F3FE': { # 👈🏾
        'en' : ':backhand_index_pointing_left_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_links_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_escura:',
        'it': ':indice_verso_sinistra_carnagione_abbastanza_scura:'
    },
    u'\U0001F448\U0001F3FC': { # 👈🏼
        'en' : ':backhand_index_pointing_left_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_links_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena_clara:',
        'it': ':indice_verso_sinistra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F448\U0001F3FD': { # 👈🏽
        'en' : ':backhand_index_pointing_left_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_links_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_izquierda_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_à_gauche_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_esquerda_pele_morena:',
        'it': ':indice_verso_sinistra_carnagione_olivastra:'
    },
    u'\U0001F449': { # 👉
        'en' : ':backhand_index_pointing_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':point_right:'],
        'variant': True,
        'de': ':nach_rechts_weisender_zeigefinger:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha:',
        'fr': u':main_avec_index_pointant_à_droite:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita:',
        'it': ':indice_verso_destra:'
    },
    u'\U0001F449\U0001F3FF': { # 👉🏿
        'en' : ':backhand_index_pointing_right_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_rechts_weisender_zeigefinger_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_à_droite_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_escura:',
        'it': ':indice_verso_destra_carnagione_scura:'
    },
    u'\U0001F449\U0001F3FB': { # 👉🏻
        'en' : ':backhand_index_pointing_right_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_rechts_weisender_zeigefinger_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro:',
        'fr': u':main_avec_index_pointant_à_droite_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_clara:',
        'it': ':indice_verso_destra_carnagione_chiara:'
    },
    u'\U0001F449\U0001F3FE': { # 👉🏾
        'en' : ':backhand_index_pointing_right_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_rechts_weisender_zeigefinger_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_oscuro_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_escura:',
        'it': ':indice_verso_destra_carnagione_abbastanza_scura:'
    },
    u'\U0001F449\U0001F3FC': { # 👉🏼
        'en' : ':backhand_index_pointing_right_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_rechts_weisender_zeigefinger_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_claro_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena_clara:',
        'it': ':indice_verso_destra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F449\U0001F3FD': { # 👉🏽
        'en' : ':backhand_index_pointing_right_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_rechts_weisender_zeigefinger_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_a_la_derecha_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_à_droite_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_a_direita_pele_morena:',
        'it': ':indice_verso_destra_carnagione_olivastra:'
    },
    u'\U0001F446': { # 👆
        'en' : ':backhand_index_pointing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':point_up_2:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba:',
        'fr': ':main_avec_index_pointant_vers_le_haut:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima:',
        'it': ':indice_alzato:'
    },
    u'\U0001F446\U0001F3FF': { # 👆🏿
        'en' : ':backhand_index_pointing_up_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_dunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_foncée:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_escura:',
        'it': ':indice_alzato_carnagione_scura:'
    },
    u'\U0001F446\U0001F3FB': { # 👆🏻
        'en' : ':backhand_index_pointing_up_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_helle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_clara:',
        'it': ':indice_alzato_carnagione_chiara:'
    },
    u'\U0001F446\U0001F3FE': { # 👆🏾
        'en' : ':backhand_index_pointing_up_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mitteldunkle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_escura:',
        'it': ':indice_alzato_carnagione_abbastanza_scura:'
    },
    u'\U0001F446\U0001F3FC': { # 👆🏼
        'en' : ':backhand_index_pointing_up_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mittelhelle_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': ':main_avec_index_pointant_vers_le_haut_peau_moyennement_claire:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena_clara:',
        'it': ':indice_alzato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F446\U0001F3FD': { # 👆🏽
        'en' : ':backhand_index_pointing_up_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_hinten_mittlere_hautfarbe:',
        'es': u':dorso_de_mano_con_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': u':main_avec_index_pointant_vers_le_haut_peau_légèrement_mate:',
        'pt': u':dorso_da_mão_com_dedo_indicador_apontando_para_cima_pele_morena:',
        'it': ':indice_alzato_carnagione_olivastra:'
    },
    u'\U0001F392': { # 🎒
        'en' : ':backpack:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':school_satchel:'],
        'de': ':schulranzen:',
        'es': ':mochila_escolar:',
        'fr': ':cartable:',
        'pt': ':mochila:',
        'it': ':zaino:'
    },
    u'\U0001F953': { # 🥓
        'en' : ':bacon:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':bacon:',
        'es': ':beicon:',
        'fr': ':lard:',
        'pt': ':bacon:',
        'it': ':pancetta:'
    },
    u'\U0001F9A1': { # 🦡
        'en' : ':badger:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':dachs:',
        'es': u':tejón:',
        'fr': ':blaireau:',
        'pt': ':texugo:',
        'it': ':tasso:'
    },
    u'\U0001F3F8': { # 🏸
        'en' : ':badminton:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':badminton_racquet_and_shuttlecock:'],
        'de': ':badminton:',
        'es': u':bádminton:',
        'fr': ':badminton:',
        'pt': ':badminton:',
        'it': ':badminton:'
    },
    u'\U0001F96F': { # 🥯
        'en' : ':bagel:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bagel:',
        'es': ':bagel:',
        'fr': ':bagel:',
        'pt': ':rosca:',
        'it': ':bagel:'
    },
    u'\U0001F6C4': { # 🛄
        'en' : ':baggage_claim:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':gepäckausgabe:',
        'es': ':recogida_de_equipajes:',
        'fr': ':retrait_des_bagages:',
        'pt': u':restituição_de_bagagem:',
        'it': ':simbolo_del_ritiro_bagagli:'
    },
    u'\U0001F956': { # 🥖
        'en' : ':baguette_bread:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':baguette:',
        'es': ':baguete:',
        'fr': ':baguette:',
        'pt': ':baguete:',
        'it': ':baguette:'
    },
    u'\U00002696\U0000FE0F': { # ⚖️
        'en' : ':balance_scale:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':scales:'],
        'variant': True,
        'de': ':waage:',
        'es': ':balanza:',
        'fr': u':balance_à_poids:',
        'pt': u':balança:',
        'it': ':bilancia_a_doppio_piatto:'
    },
    u'\U00002696': { # ⚖
        'en' : ':balance_scale:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [':scales:'],
        'variant': True,
        'de': ':waage:',
        'es': ':balanza:',
        'fr': u':balance_à_poids:',
        'pt': u':balança:',
        'it': ':bilancia_a_doppio_piatto:'
    },
    u'\U0001F9B2': { # 🦲
        'en' : ':bald:',
        'status' : component,
        'E' : 11,
        'de': ':glatze:',
        'es': ':sin_pelo:',
        'fr': ':chauve:',
        'pt': ':careca:',
        'it': ':calvo:'
    },
    u'\U0001FA70': { # 🩰
        'en' : ':ballet_shoes:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':ballettschuhe:',
        'es': ':zapatillas_de_ballet:',
        'fr': ':chaussons_de_danse:',
        'pt': u':sapatilha_de_balé:',
        'it': ':scarpette_da_ballerina:'
    },
    u'\U0001F388': { # 🎈
        'en' : ':balloon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':luftballon:',
        'es': ':globo:',
        'fr': ':ballon_gonflable:',
        'pt': u':balão:',
        'it': ':palloncino:'
    },
    u'\U0001F5F3\U0000FE0F': { # 🗳️
        'en' : ':ballot_box_with_ballot:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':ballot_box:'],
        'variant': True,
        'de': ':urne_mit_wahlzettel:',
        'es': ':urna_con_papeleta:',
        'fr': u':urne_électorale:',
        'pt': u':urna_eleitoral_com_cédula:',
        'it': ':urna_elettorale_con_scheda:'
    },
    u'\U0001F5F3': { # 🗳
        'en' : ':ballot_box_with_ballot:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':ballot_box:'],
        'variant': True,
        'de': ':urne_mit_wahlzettel:',
        'es': ':urna_con_papeleta:',
        'fr': u':urne_électorale:',
        'pt': u':urna_eleitoral_com_cédula:',
        'it': ':urna_elettorale_con_scheda:'
    },
    u'\U0001F34C': { # 🍌
        'en' : ':banana:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':banane:',
        'es': u':plátano:',
        'fr': ':banane:',
        'pt': ':banana:',
        'it': ':banana:'
    },
    u'\U0001FA95': { # 🪕
        'en' : ':banjo:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':banjo:',
        'es': ':banjo:',
        'fr': ':banjo:',
        'pt': ':banjo:',
        'it': ':banjo:'
    },
    u'\U0001F3E6': { # 🏦
        'en' : ':bank:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bank:',
        'es': ':banco:',
        'fr': ':banque:',
        'pt': ':banco:',
        'it': ':banca:'
    },
    u'\U0001F4CA': { # 📊
        'en' : ':bar_chart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':balkendiagramm:',
        'es': u':gráfico_de_barras:',
        'fr': u':graphiques_à_barres:',
        'pt': u':gráfico_de_barras:',
        'it': ':grafico_a_barre:'
    },
    u'\U0001F488': { # 💈
        'en' : ':barber_pole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':barber:'],
        'de': u':barbershop-säule:',
        'es': ':poste_de_barbero:',
        'fr': ':enseigne_de_barbier:',
        'pt': ':barbearia:',
        'it': ':barbiere:'
    },
    u'\U000026BE': { # ⚾
        'en' : ':baseball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':baseball:',
        'es': u':béisbol:',
        'fr': ':baseball:',
        'pt': ':bola_de_beisebol:',
        'it': ':palla_da_baseball:'
    },
    u'\U0001F9FA': { # 🧺
        'en' : ':basket:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':korb:',
        'es': ':cesta:',
        'fr': ':panier:',
        'pt': ':cesta:',
        'it': ':cesto:'
    },
    u'\U0001F3C0': { # 🏀
        'en' : ':basketball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':basketball:',
        'es': u':balón_de_baloncesto:',
        'fr': ':basket:',
        'pt': ':bola_de_basquete:',
        'it': ':palla_da_pallacanestro:'
    },
    u'\U0001F987': { # 🦇
        'en' : ':bat:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':fledermaus:',
        'es': u':murciélago:',
        'fr': ':chauve-souris:',
        'pt': ':morcego:',
        'it': ':pipistrello:'
    },
    u'\U0001F6C1': { # 🛁
        'en' : ':bathtub:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':badewanne:',
        'es': u':bañera:',
        'fr': ':baignoire:',
        'pt': ':banheira:',
        'it': ':vasca:'
    },
    u'\U0001F50B': { # 🔋
        'en' : ':battery:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':batterie:',
        'es': ':pila:',
        'fr': ':pile:',
        'pt': ':pilha:',
        'it': ':batteria:'
    },
    u'\U0001F3D6\U0000FE0F': { # 🏖️
        'en' : ':beach_with_umbrella:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':beach_umbrella:'],
        'variant': True,
        'de': ':strand_mit_sonnenschirm:',
        'es': ':playa_y_sombrilla:',
        'fr': ':plage_avec_parasol:',
        'pt': ':praia_e_guarda-sol:',
        'it': ':spiaggia_con_ombrellone:'
    },
    u'\U0001F3D6': { # 🏖
        'en' : ':beach_with_umbrella:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':beach_umbrella:'],
        'variant': True,
        'de': ':strand_mit_sonnenschirm:',
        'es': ':playa_y_sombrilla:',
        'fr': ':plage_avec_parasol:',
        'pt': ':praia_e_guarda-sol:',
        'it': ':spiaggia_con_ombrellone:'
    },
    u'\U0001F601': { # 😁
        'en' : ':beaming_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':grin:'],
        'de': ':strahlendes_gesicht_mit_lachenden_augen:',
        'es': ':cara_radiante_con_ojos_sonrientes:',
        'fr': ':visage_souriant_aux_yeux_rieurs:',
        'pt': ':rosto_contente_com_olhos_sorridentes:',
        'it': ':faccina_raggiante_con_occhi_felici:'
    },
    u'\U0001FAD8': { # 🫘
        'en' : ':beans:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':bohnen:',
        'es': ':alubias:',
        'fr': ':haricots:',
        'pt': u':feijões:',
        'it': ':fagioli:'
    },
    u'\U0001F43B': { # 🐻
        'en' : ':bear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':bär:',
        'es': ':oso:',
        'fr': ':ours:',
        'pt': ':rosto_de_urso:',
        'it': ':orso:'
    },
    u'\U0001F493': { # 💓
        'en' : ':beating_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heartbeat:'],
        'de': ':schlagendes_herz:',
        'es': u':corazón_latiendo:',
        'fr': u':cœur_battant:',
        'pt': u':coração_pulsante:',
        'it': ':cuore_che_batte:'
    },
    u'\U0001F9AB': { # 🦫
        'en' : ':beaver:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':biber:',
        'es': ':castor:',
        'fr': ':castor:',
        'pt': ':castor:',
        'it': ':castoro:'
    },
    u'\U0001F6CF\U0000FE0F': { # 🛏️
        'en' : ':bed:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':bett:',
        'es': ':cama:',
        'fr': ':lit:',
        'pt': ':cama:',
        'it': ':letto:'
    },
    u'\U0001F6CF': { # 🛏
        'en' : ':bed:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':bett:',
        'es': ':cama:',
        'fr': ':lit:',
        'pt': ':cama:',
        'it': ':letto:'
    },
    u'\U0001F37A': { # 🍺
        'en' : ':beer_mug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':beer:'],
        'de': ':bierkrug:',
        'es': ':jarra_de_cerveza:',
        'fr': ':chope:',
        'pt': ':cerveja:',
        'it': ':boccale_di_birra:'
    },
    u'\U0001FAB2': { # 🪲
        'en' : ':beetle:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':käfer:',
        'es': ':escarabajo:',
        'fr': u':scarabée:',
        'pt': ':besouro:',
        'it': ':coleottero:'
    },
    u'\U0001F514': { # 🔔
        'en' : ':bell:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':glocke:',
        'es': ':campana:',
        'fr': ':cloche:',
        'pt': ':sino:',
        'it': ':campana:'
    },
    u'\U0001FAD1': { # 🫑
        'en' : ':bell_pepper:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':paprika:',
        'es': ':pimiento:',
        'fr': ':poivron:',
        'pt': u':pimentão:',
        'it': ':peperone:'
    },
    u'\U0001F515': { # 🔕
        'en' : ':bell_with_slash:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':no_bell:'],
        'de': ':durchgestrichene_glocke:',
        'es': u':campana_con_signo_de_cancelación:',
        'fr': u':alarme_désactivée:',
        'pt': ':sino_silenciado:',
        'it': ':campana_sbarrata:'
    },
    u'\U0001F6CE\U0000FE0F': { # 🛎️
        'en' : ':bellhop_bell:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':rezeptionsklingel:',
        'es': ':timbre_de_hotel:',
        'fr': ':cloche_de_comptoir:',
        'pt': ':sineta:',
        'it': ':campanello_da_hotel:'
    },
    u'\U0001F6CE': { # 🛎
        'en' : ':bellhop_bell:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':rezeptionsklingel:',
        'es': ':timbre_de_hotel:',
        'fr': ':cloche_de_comptoir:',
        'pt': ':sineta:',
        'it': ':campanello_da_hotel:'
    },
    u'\U0001F371': { # 🍱
        'en' : ':bento_box:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bento:'],
        'de': ':bento-box:',
        'es': ':caja_de_bento:',
        'fr': u':boîte_déjeuner:',
        'pt': u':bentô:',
        'it': ':bento_box:'
    },
    u'\U0001F9C3': { # 🧃
        'en' : ':beverage_box:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':trinkpäckchen:',
        'es': ':tetrabrik:',
        'fr': ':briquette_de_jus:',
        'pt': ':suco_de_caixa:',
        'it': ':bevanda_monodose:'
    },
    u'\U0001F6B2': { # 🚲
        'en' : ':bicycle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bike:'],
        'variant': True,
        'de': ':fahrrad:',
        'es': ':bicicleta:',
        'fr': u':vélo:',
        'pt': ':bicicleta:',
        'it': ':bicicletta:'
    },
    u'\U0001F459': { # 👙
        'en' : ':bikini:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bikini:',
        'es': ':bikini:',
        'fr': ':bikini:',
        'pt': u':biquíni:',
        'it': ':bikini:'
    },
    u'\U0001F9E2': { # 🧢
        'en' : ':billed_cap:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':baseballmütze:',
        'es': ':gorra_con_visera:',
        'fr': u':casquette_américaine:',
        'pt': u':boné:',
        'it': ':cappello_con_visiera:'
    },
    u'\U00002623\U0000FE0F': { # ☣️
        'en' : ':biohazard:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':biohazard_sign:'],
        'variant': True,
        'de': u':biogefährdung:',
        'es': u':riesgo_biológico:',
        'fr': ':danger_biologique:',
        'pt': u':risco_biológico:',
        'it': ':simbolo_del_rischio_biologico:'
    },
    u'\U00002623': { # ☣
        'en' : ':biohazard:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [':biohazard_sign:'],
        'variant': True,
        'de': u':biogefährdung:',
        'es': u':riesgo_biológico:',
        'fr': ':danger_biologique:',
        'pt': u':risco_biológico:',
        'it': ':simbolo_del_rischio_biologico:'
    },
    u'\U0001F426': { # 🐦
        'en' : ':bird:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':vogel:',
        'es': u':pájaro:',
        'fr': ':oiseau:',
        'pt': u':pássaro:',
        'it': ':uccello:'
    },
    u'\U0001F382': { # 🎂
        'en' : ':birthday_cake:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':birthday:'],
        'de': ':geburtstagskuchen:',
        'es': u':tarta_de_cumpleaños:',
        'fr': u':gâteau_d’anniversaire:',
        'pt': u':bolo_de_aniversário:',
        'it': ':torta_di_compleanno:'
    },
    u'\U0001F9AC': { # 🦬
        'en' : ':bison:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':bison:',
        'es': ':bisonte:',
        'fr': ':bison:',
        'pt': u':bisão:',
        'it': ':bisonte:'
    },
    u'\U0001FAE6': { # 🫦
        'en' : ':biting_lip:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':auf_lippe_beißen:',
        'es': ':labio_mordido:',
        'fr': u':lèvres_qui_se_mordent:',
        'pt': u':mordendo_o_lábio:',
        'it': ':bocca_che_morde_il_labbro:'
    },
    u'\U0001F408\U0000200D\U00002B1B': { # 🐈‍⬛
        'en' : ':black_cat:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':schwarze_katze:',
        'es': ':gato_negro:',
        'fr': ':chat_noir:',
        'pt': ':gato_preto:',
        'it': ':gatto_nero:'
    },
    u'\U000026AB': { # ⚫
        'en' : ':black_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':schwarzer_punkt:',
        'es': u':círculo_negro:',
        'fr': ':disque_noir:',
        'pt': u':círculo_preto:',
        'it': ':cerchio_nero:'
    },
    u'\U0001F3F4': { # 🏴
        'en' : ':black_flag:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':waving_black_flag:'],
        'de': ':schwarze_flagge:',
        'es': ':bandera_negra:',
        'fr': ':drapeau_noir:',
        'pt': ':bandeira_preta:',
        'it': ':bandiera_nera:'
    },
    u'\U0001F5A4': { # 🖤
        'en' : ':black_heart:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':schwarzes_herz:',
        'es': u':corazón_negro:',
        'fr': u':cœur_noir:',
        'pt': u':coração_preto:',
        'it': ':cuore_nero:'
    },
    u'\U00002B1B': { # ⬛
        'en' : ':black_large_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':großes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_grande:',
        'fr': u':grand_carré_noir:',
        'pt': ':quadrado_preto_grande:',
        'it': ':quadrato_nero_grande:'
    },
    u'\U000025FE': { # ◾
        'en' : ':black_medium-small_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':black_medium_small_square:'],
        'variant': True,
        'de': ':mittelkleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_mediano-pequeño:',
        'fr': u':carré_petit_moyen_noir:',
        'pt': u':quadrado_preto_médio_menor:',
        'it': ':quadrato_nero_medio-piccolo:'
    },
    u'\U000025FC\U0000FE0F': { # ◼️
        'en' : ':black_medium_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':mittelgroßes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_mediano:',
        'fr': u':carré_moyen_noir:',
        'pt': u':quadrado_preto_médio:',
        'it': ':quadrato_nero_medio:'
    },
    u'\U000025FC': { # ◼
        'en' : ':black_medium_square:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': u':mittelgroßes_schwarzes_quadrat:',
        'es': ':cuadrado_negro_mediano:',
        'fr': u':carré_moyen_noir:',
        'pt': u':quadrado_preto_médio:',
        'it': ':quadrato_nero_medio:'
    },
    u'\U00002712\U0000FE0F': { # ✒️
        'en' : ':black_nib:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':schwarzer_federhalter:',
        'es': ':pluma_negra:',
        'fr': ':stylo_plume_noir:',
        'pt': ':ponta_de_caneta_tinteiro:',
        'it': ':pennino:'
    },
    u'\U00002712': { # ✒
        'en' : ':black_nib:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':schwarzer_federhalter:',
        'es': ':pluma_negra:',
        'fr': ':stylo_plume_noir:',
        'pt': ':ponta_de_caneta_tinteiro:',
        'it': ':pennino:'
    },
    u'\U000025AA\U0000FE0F': { # ▪️
        'en' : ':black_small_square:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':kleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_pequeño:',
        'fr': u':petit_carré_noir:',
        'pt': ':quadrado_preto_pequeno:',
        'it': ':quadrato_nero_piccolo:'
    },
    u'\U000025AA': { # ▪
        'en' : ':black_small_square:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':kleines_schwarzes_quadrat:',
        'es': u':cuadrado_negro_pequeño:',
        'fr': u':petit_carré_noir:',
        'pt': ':quadrado_preto_pequeno:',
        'it': ':quadrato_nero_piccolo:'
    },
    u'\U0001F532': { # 🔲
        'en' : ':black_square_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schwarze_quadratische_schaltfläche:',
        'es': u':botón_cuadrado_con_borde_negro:',
        'fr': u':carré_noir:',
        'pt': u':botão_quadrado_preto:',
        'it': ':tasto_quadrato_bianco_con_bordo_nero:'
    },
    u'\U0001F33C': { # 🌼
        'en' : ':blossom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':gelbe_blüte:',
        'es': ':flor:',
        'fr': ':bourgeon:',
        'pt': ':flor:',
        'it': ':fiore:'
    },
    u'\U0001F421': { # 🐡
        'en' : ':blowfish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kugelfisch:',
        'es': ':pez_globo:',
        'fr': ':poisson-lune:',
        'pt': ':baiacu:',
        'it': ':pesce_palla:'
    },
    u'\U0001F4D8': { # 📘
        'en' : ':blue_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':blaues_buch:',
        'es': ':libro_azul:',
        'fr': ':livre_bleu:',
        'pt': ':livro_azul:',
        'it': ':libro_blu:'
    },
    u'\U0001F535': { # 🔵
        'en' : ':blue_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':large_blue_circle:'],
        'de': ':blauer_punkt:',
        'es': u':círculo_azul_grande:',
        'fr': ':disque_bleu:',
        'pt': u':círculo_azul:',
        'it': ':cerchio_blu:'
    },
    u'\U0001F499': { # 💙
        'en' : ':blue_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':blaues_herz:',
        'es': u':corazón_azul:',
        'fr': u':cœur_bleu:',
        'pt': u':coração_azul:',
        'it': ':cuore_azzurro:'
    },
    u'\U0001F7E6': { # 🟦
        'en' : ':blue_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':blaues_quadrat:',
        'es': ':cuadrado_azul:',
        'fr': u':carré_bleu:',
        'pt': ':quadrado_azul:',
        'it': ':quadrato_blu:'
    },
    u'\U0001FAD0': { # 🫐
        'en' : ':blueberries:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':blaubeeren:',
        'es': u':arándanos:',
        'fr': ':myrtilles:',
        'pt': ':mirtilos:',
        'it': ':mirtilli:'
    },
    u'\U0001F417': { # 🐗
        'en' : ':boar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':wildschwein:',
        'es': u':jabalí:',
        'fr': ':sanglier:',
        'pt': ':javali:',
        'it': ':cinghiale:'
    },
    u'\U0001F4A3': { # 💣
        'en' : ':bomb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':bombe:',
        'es': ':bomba:',
        'fr': ':bombe:',
        'pt': ':bomba:',
        'it': ':bomba:'
    },
    u'\U0001F9B4': { # 🦴
        'en' : ':bone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':knochen:',
        'es': ':hueso:',
        'fr': ':os:',
        'pt': ':osso:',
        'it': ':osso:'
    },
    u'\U0001F516': { # 🔖
        'en' : ':bookmark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':lesezeichen:',
        'es': u':marcapáginas:',
        'fr': ':marque-page:',
        'pt': u':marcador_de_página:',
        'it': ':segnalibro:'
    },
    u'\U0001F4D1': { # 📑
        'en' : ':bookmark_tabs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':pagemarker:',
        'es': ':marcadores:',
        'fr': ':signets:',
        'pt': u':marcadores_de_página:',
        'it': ':etichette_segnalibro:'
    },
    u'\U0001F4DA': { # 📚
        'en' : ':books:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': u':bücherstapel:',
        'es': ':libros:',
        'fr': ':livres:',
        'pt': ':livros:',
        'it': ':libri:'
    },
    u'\U0001FA83': { # 🪃
        'en' : ':boomerang:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':bumerang:',
        'es': u':bumerán:',
        'fr': ':boomerang:',
        'pt': ':bumerangue:',
        'it': ':boomerang:'
    },
    u'\U0001F37E': { # 🍾
        'en' : ':bottle_with_popping_cork:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':champagne:'],
        'de': ':flasche_mit_knallendem_korken:',
        'es': ':botella_descorchada:',
        'fr': ':bouteille_de_champagne:',
        'pt': ':garrafa_de_champanhe:',
        'it': ':bottiglia_stappata:'
    },
    u'\U0001F490': { # 💐
        'en' : ':bouquet:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':blumenstrauß:',
        'es': ':ramo_de_flores:',
        'fr': ':bouquet:',
        'pt': u':buquê:',
        'it': ':mazzo_di_fiori:'
    },
    u'\U0001F3F9': { # 🏹
        'en' : ':bow_and_arrow:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pfeil_und_bogen:',
        'es': ':arco_y_flecha:',
        'fr': u':arc_et_flèche:',
        'pt': ':arco_e_flecha:',
        'it': ':arco_e_freccia:'
    },
    u'\U0001F963': { # 🥣
        'en' : ':bowl_with_spoon:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':schüssel_mit_löffel:',
        'es': ':cuenco_con_cuchara:',
        'fr': u':bol_avec_cuillère:',
        'pt': ':tigela_com_colher:',
        'it': ':ciotola_con_cucchiaio:'
    },
    u'\U0001F3B3': { # 🎳
        'en' : ':bowling:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bowling:',
        'es': ':bolos:',
        'fr': ':bowling:',
        'pt': ':boliche:',
        'it': ':bowling:'
    },
    u'\U0001F94A': { # 🥊
        'en' : ':boxing_glove:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':boxhandschuh:',
        'es': ':guante_de_boxeo:',
        'fr': ':gant_de_boxe:',
        'pt': ':luva_de_boxe:',
        'it': ':guantone_da_pugilato:'
    },
    u'\U0001F466': { # 👦
        'en' : ':boy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':junge:',
        'es': u':niño:',
        'fr': u':garçon:',
        'pt': ':menino:',
        'it': ':bambino:'
    },
    u'\U0001F466\U0001F3FF': { # 👦🏿
        'en' : ':boy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':junge_dunkle_hautfarbe:',
        'es': u':niño_tono_de_piel_oscuro:',
        'fr': u':garçon_peau_foncée:',
        'pt': ':menino_pele_escura:',
        'it': ':bambino_carnagione_scura:'
    },
    u'\U0001F466\U0001F3FB': { # 👦🏻
        'en' : ':boy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':junge_helle_hautfarbe:',
        'es': u':niño_tono_de_piel_claro:',
        'fr': u':garçon_peau_claire:',
        'pt': ':menino_pele_clara:',
        'it': ':bambino_carnagione_chiara:'
    },
    u'\U0001F466\U0001F3FE': { # 👦🏾
        'en' : ':boy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':junge_mitteldunkle_hautfarbe:',
        'es': u':niño_tono_de_piel_oscuro_medio:',
        'fr': u':garçon_peau_mate:',
        'pt': ':menino_pele_morena_escura:',
        'it': ':bambino_carnagione_abbastanza_scura:'
    },
    u'\U0001F466\U0001F3FC': { # 👦🏼
        'en' : ':boy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':junge_mittelhelle_hautfarbe:',
        'es': u':niño_tono_de_piel_claro_medio:',
        'fr': u':garçon_peau_moyennement_claire:',
        'pt': ':menino_pele_morena_clara:',
        'it': ':bambino_carnagione_abbastanza_chiara:'
    },
    u'\U0001F466\U0001F3FD': { # 👦🏽
        'en' : ':boy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':junge_mittlere_hautfarbe:',
        'es': u':niño_tono_de_piel_medio:',
        'fr': u':garçon_peau_légèrement_mate:',
        'pt': ':menino_pele_morena:',
        'it': ':bambino_carnagione_olivastra:'
    },
    u'\U0001F9E0': { # 🧠
        'en' : ':brain:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':gehirn:',
        'es': ':cerebro:',
        'fr': ':cerveau:',
        'pt': u':cérebro:',
        'it': ':cervello:'
    },
    u'\U0001F35E': { # 🍞
        'en' : ':bread:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':brot:',
        'es': ':pan_de_molde:',
        'fr': ':pain:',
        'pt': u':pão:',
        'it': ':pane_in_cassetta:'
    },
    u'\U0001F931': { # 🤱
        'en' : ':breast-feeding:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':breast_feeding:'],
        'de': ':stillen:',
        'es': ':lactancia_materna:',
        'fr': ':allaitement:',
        'pt': ':amamentando:',
        'it': ':allattare:'
    },
    u'\U0001F931\U0001F3FF': { # 🤱🏿
        'en' : ':breast-feeding_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':stillen_dunkle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_oscuro:',
        'fr': u':allaitement_peau_foncée:',
        'pt': ':amamentando_pele_escura:',
        'it': ':allattare_carnagione_scura:'
    },
    u'\U0001F931\U0001F3FB': { # 🤱🏻
        'en' : ':breast-feeding_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':stillen_helle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_claro:',
        'fr': ':allaitement_peau_claire:',
        'pt': ':amamentando_pele_clara:',
        'it': ':allattare_carnagione_chiara:'
    },
    u'\U0001F931\U0001F3FE': { # 🤱🏾
        'en' : ':breast-feeding_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':stillen_mitteldunkle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_oscuro_medio:',
        'fr': ':allaitement_peau_mate:',
        'pt': ':amamentando_pele_morena_escura:',
        'it': ':allattare_carnagione_abbastanza_scura:'
    },
    u'\U0001F931\U0001F3FC': { # 🤱🏼
        'en' : ':breast-feeding_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':stillen_mittelhelle_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_claro_medio:',
        'fr': ':allaitement_peau_moyennement_claire:',
        'pt': ':amamentando_pele_morena_clara:',
        'it': ':allattare_carnagione_abbastanza_chiara:'
    },
    u'\U0001F931\U0001F3FD': { # 🤱🏽
        'en' : ':breast-feeding_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':stillen_mittlere_hautfarbe:',
        'es': ':lactancia_materna_tono_de_piel_medio:',
        'fr': u':allaitement_peau_légèrement_mate:',
        'pt': ':amamentando_pele_morena:',
        'it': ':allattare_carnagione_olivastra:'
    },
    u'\U0001F9F1': { # 🧱
        'en' : ':brick:',
        'status' : fully_qualified,
        'E' : 11,
        'alias' : [':bricks:'],
        'de': ':ziegelstein:',
        'es': ':ladrillo:',
        'fr': ':brique:',
        'pt': ':tijolo:',
        'it': ':mattoni:'
    },
    u'\U0001F309': { # 🌉
        'en' : ':bridge_at_night:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':brücke_vor_nachthimmel:',
        'es': ':puente_de_noche:',
        'fr': ':pont_de_nuit:',
        'pt': u':ponte_à_noite:',
        'it': ':ponte_di_notte:'
    },
    u'\U0001F4BC': { # 💼
        'en' : ':briefcase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':aktentasche:',
        'es': u':maletín:',
        'fr': ':porte-documents:',
        'pt': ':maleta:',
        'it': ':valigetta_24_ore:'
    },
    u'\U0001FA72': { # 🩲
        'en' : ':briefs:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [':swim_brief:'],
        'de': ':slip:',
        'es': ':ropa_interior:',
        'fr': ':slip:',
        'pt': ':cueca:',
        'it': ':slip:'
    },
    u'\U0001F506': { # 🔆
        'en' : ':bright_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':high_brightness:'],
        'de': ':heller-taste:',
        'es': ':brillo_alto:',
        'fr': u':luminosité_élevée:',
        'pt': u':botão_de_aumentar_brilho:',
        'it': u':luminosità_elevata:'
    },
    u'\U0001F966': { # 🥦
        'en' : ':broccoli:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':brokkoli:',
        'es': u':brócoli:',
        'fr': ':brocoli:',
        'pt': u':brócolis:',
        'it': ':broccoli:'
    },
    u'\U0001F494': { # 💔
        'en' : ':broken_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':gebrochenes_herz:',
        'es': u':corazón_roto:',
        'fr': u':cœur_brisé:',
        'pt': u':coração_partido:',
        'it': ':cuore_infranto:'
    },
    u'\U0001F9F9': { # 🧹
        'en' : ':broom:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':besen:',
        'es': ':escoba:',
        'fr': ':balai:',
        'pt': ':vassoura:',
        'it': ':scopa:'
    },
    u'\U0001F7E4': { # 🟤
        'en' : ':brown_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':brauner_punkt:',
        'es': u':círculo_marrón:',
        'fr': ':disque_marron:',
        'pt': u':círculo_marrom:',
        'it': ':cerchio_marrone:'
    },
    u'\U0001F90E': { # 🤎
        'en' : ':brown_heart:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':braunes_herz:',
        'es': u':corazón_marrón:',
        'fr': u':cœur_marron:',
        'pt': u':coração_marrom:',
        'it': ':cuore_marrone:'
    },
    u'\U0001F7EB': { # 🟫
        'en' : ':brown_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':braunes_quadrat:',
        'es': u':cuadrado_marrón:',
        'fr': u':carré_marron:',
        'pt': ':quadrado_marrom:',
        'it': ':quadrato_marrone:'
    },
    u'\U0001F9CB': { # 🧋
        'en' : ':bubble_tea:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':bubble_tea:',
        'es': u':té_de_burbujas:',
        'fr': u':thé_aux_perles:',
        'pt': u':chá_perolado:',
        'it': ':bubble_tea:'
    },
    u'\U0001FAE7': { # 🫧
        'en' : ':bubbles:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':blasen:',
        'es': ':burbujas:',
        'fr': ':bulles:',
        'pt': ':bolhas:',
        'it': ':bolle:'
    },
    u'\U0001FAA3': { # 🪣
        'en' : ':bucket:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':eimer:',
        'es': ':cubo:',
        'fr': ':seau:',
        'pt': ':balde:',
        'it': ':secchio:'
    },
    u'\U0001F41B': { # 🐛
        'en' : ':bug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':raupe:',
        'es': ':insecto:',
        'fr': ':chenille:',
        'pt': ':inseto:',
        'it': ':insetto:'
    },
    u'\U0001F3D7\U0000FE0F': { # 🏗️
        'en' : ':building_construction:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kran:',
        'es': u':construcción:',
        'fr': u':construction_d’un_bâtiment:',
        'pt': u':construção:',
        'it': ':edificio_in_costruzione:'
    },
    u'\U0001F3D7': { # 🏗
        'en' : ':building_construction:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kran:',
        'es': u':construcción:',
        'fr': u':construction_d’un_bâtiment:',
        'pt': u':construção:',
        'it': ':edificio_in_costruzione:'
    },
    u'\U0001F685': { # 🚅
        'en' : ':bullet_train:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bullettrain_front:'],
        'de': ':hochgeschwindigkeitszug:',
        'es': ':tren_bala:',
        'fr': u':train_à_grande_vitesse:',
        'pt': u':trem_de_alta_velocidade_japonês:',
        'it': u':treno_alta_velocità_punta_arrotondata:'
    },
    u'\U0001F3AF': { # 🎯
        'en' : ':bullseye:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dart:'],
        'de': ':darts:',
        'es': ':diana:',
        'fr': ':dans_le_mille:',
        'pt': ':no_alvo:',
        'it': ':bersaglio:'
    },
    u'\U0001F32F': { # 🌯
        'en' : ':burrito:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':burrito:',
        'es': ':burrito:',
        'fr': ':burrito:',
        'pt': ':burrito:',
        'it': ':burrito:'
    },
    u'\U0001F68C': { # 🚌
        'en' : ':bus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bus:',
        'es': u':autobús:',
        'fr': ':bus:',
        'pt': u':ônibus:',
        'it': ':bus:'
    },
    u'\U0001F68F': { # 🚏
        'en' : ':bus_stop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':busstop:'],
        'de': ':bushaltestelle:',
        'es': u':parada_de_autobús:',
        'fr': u':arrêt_de_bus:',
        'pt': u':ponto_de_ônibus:',
        'it': u':fermata_dell’autobus:'
    },
    u'\U0001F464': { # 👤
        'en' : ':bust_in_silhouette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':silhouette_einer_büste:',
        'es': ':silueta_de_busto:',
        'fr': ':silhouette_de_buste:',
        'pt': ':silhueta_de_busto:',
        'it': ':profilo_di_persona:'
    },
    u'\U0001F465': { # 👥
        'en' : ':busts_in_silhouette:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':silhouette_mehrerer_büsten:',
        'es': ':dos_siluetas_de_bustos:',
        'fr': ':silhouettes_de_bustes:',
        'pt': ':silhueta_de_bustos:',
        'it': ':profilo_di_due_persone:'
    },
    u'\U0001F9C8': { # 🧈
        'en' : ':butter:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':butter:',
        'es': ':mantequilla:',
        'fr': ':beurre:',
        'pt': ':manteiga:',
        'it': ':burro:'
    },
    u'\U0001F98B': { # 🦋
        'en' : ':butterfly:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':schmetterling:',
        'es': ':mariposa:',
        'fr': ':papillon:',
        'pt': ':borboleta:',
        'it': ':farfalla:'
    },
    u'\U0001F335': { # 🌵
        'en' : ':cactus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kaktus:',
        'es': ':cactus:',
        'fr': ':cactus:',
        'pt': ':cacto:',
        'it': ':cactus:'
    },
    u'\U0001F4C5': { # 📅
        'en' : ':calendar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':date:'],
        'de': ':kalender:',
        'es': ':calendario:',
        'fr': ':calendrier:',
        'pt': u':calendário:',
        'it': ':calendario:'
    },
    u'\U0001F919': { # 🤙
        'en' : ':call_me_hand:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen:',
        'es': ':mano_haciendo_el_gesto_de_llamar:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts:',
        'pt': ':sinal_me_liga:',
        'it': ':mano_con_gesto_di_chiamata:'
    },
    u'\U0001F919\U0001F3FF': { # 🤙🏿
        'en' : ':call_me_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen_dunkle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_foncée:',
        'pt': ':sinal_me_liga_pele_escura:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_scura:'
    },
    u'\U0001F919\U0001F3FB': { # 🤙🏻
        'en' : ':call_me_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen_helle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_claire:',
        'pt': ':sinal_me_liga_pele_clara:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_chiara:'
    },
    u'\U0001F919\U0001F3FE': { # 🤙🏾
        'en' : ':call_me_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen_mitteldunkle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_oscuro_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_mate:',
        'pt': ':sinal_me_liga_pele_morena_escura:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_abbastanza_scura:'
    },
    u'\U0001F919\U0001F3FC': { # 🤙🏼
        'en' : ':call_me_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen_mittelhelle_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_claro_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_moyennement_claire:',
        'pt': ':sinal_me_liga_pele_morena_clara:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F919\U0001F3FD': { # 🤙🏽
        'en' : ':call_me_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ruf-mich-an-handzeichen_mittlere_hautfarbe:',
        'es': ':mano_haciendo_el_gesto_de_llamar_tono_de_piel_medio:',
        'fr': u':signe_appel_téléphonique_avec_les_doigts_peau_légèrement_mate:',
        'pt': ':sinal_me_liga_pele_morena:',
        'it': ':mano_con_gesto_di_chiamata_carnagione_olivastra:'
    },
    u'\U0001F42A': { # 🐪
        'en' : ':camel:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':dromedary_camel:'],
        'de': ':dromedar:',
        'es': ':dromedario:',
        'fr': ':dromadaire:',
        'pt': ':camelo:',
        'it': ':dromedario:'
    },
    u'\U0001F4F7': { # 📷
        'en' : ':camera:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':fotoapparat:',
        'es': u':cámara_de_fotos:',
        'fr': ':appareil_photo:',
        'pt': u':câmera:',
        'it': ':fotocamera:'
    },
    u'\U0001F4F8': { # 📸
        'en' : ':camera_with_flash:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':camera_flash:'],
        'de': ':fotoapparat_mit_blitz:',
        'es': u':cámara_con_flash:',
        'fr': ':appareil_photo_avec_flash:',
        'pt': u':câmera_com_flash:',
        'it': ':fotocamera_con_flash:'
    },
    u'\U0001F3D5\U0000FE0F': { # 🏕️
        'en' : ':camping:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':camping:',
        'es': ':camping:',
        'fr': ':camping:',
        'pt': ':acampamento:',
        'it': ':campeggio:'
    },
    u'\U0001F3D5': { # 🏕
        'en' : ':camping:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':camping:',
        'es': ':camping:',
        'fr': ':camping:',
        'pt': ':acampamento:',
        'it': ':campeggio:'
    },
    u'\U0001F56F\U0000FE0F': { # 🕯️
        'en' : ':candle:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kerze:',
        'es': ':vela:',
        'fr': ':bougie:',
        'pt': ':vela:',
        'it': ':candela:'
    },
    u'\U0001F56F': { # 🕯
        'en' : ':candle:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kerze:',
        'es': ':vela:',
        'fr': ':bougie:',
        'pt': ':vela:',
        'it': ':candela:'
    },
    u'\U0001F36C': { # 🍬
        'en' : ':candy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bonbon:',
        'es': ':caramelo:',
        'fr': ':bonbon:',
        'pt': ':bala:',
        'it': ':caramella:'
    },
    u'\U0001F96B': { # 🥫
        'en' : ':canned_food:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':konserve:',
        'es': ':comida_enlatada:',
        'fr': ':aliments_en_conserve:',
        'pt': ':comida_enlatada:',
        'it': ':cibo_in_scatola:'
    },
    u'\U0001F6F6': { # 🛶
        'en' : ':canoe:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':kanu:',
        'es': ':canoa:',
        'fr': u':canoë:',
        'pt': ':canoa:',
        'it': ':canoa:'
    },
    u'\U0001F5C3\U0000FE0F': { # 🗃️
        'en' : ':card_file_box:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':karteikasten:',
        'es': ':archivador_de_tarjetas:',
        'fr': u':boîte_à_dossiers:',
        'pt': ':caixa_de_arquivos:',
        'it': ':schedario_da_tavolo:'
    },
    u'\U0001F5C3': { # 🗃
        'en' : ':card_file_box:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':karteikasten:',
        'es': ':archivador_de_tarjetas:',
        'fr': u':boîte_à_dossiers:',
        'pt': ':caixa_de_arquivos:',
        'it': ':schedario_da_tavolo:'
    },
    u'\U0001F4C7': { # 📇
        'en' : ':card_index:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':rotationskartei:',
        'es': ':organizador_de_fichas:',
        'fr': u':carnet_d’adresses:',
        'pt': u':índice_de_cartões:',
        'it': ':schedario:'
    },
    u'\U0001F5C2\U0000FE0F': { # 🗂️
        'en' : ':card_index_dividers:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':karteibahnen:',
        'es': ':separador_de_fichas:',
        'fr': ':intercalaires:',
        'pt': ':divisores_de_pastas:',
        'it': ':divisori_per_schedario:'
    },
    u'\U0001F5C2': { # 🗂
        'en' : ':card_index_dividers:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':karteireiter:',
        'es': ':separador_de_fichas:',
        'fr': ':intercalaires:',
        'pt': ':divisores_de_pastas:',
        'it': ':divisori_per_schedario:'
    },
    u'\U0001F3A0': { # 🎠
        'en' : ':carousel_horse:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':karussellpferd:',
        'es': ':caballo_de_tiovivo:',
        'fr': u':cheval_de_manège:',
        'pt': ':carrossel:',
        'it': ':cavallo_da_giostra:'
    },
    u'\U0001F38F': { # 🎏
        'en' : ':carp_streamer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flags:'],
        'de': u':traditionelle_japanische_windsäcke:',
        'es': u':banderín_de_carpas:',
        'fr': ':koinobori:',
        'pt': ':bandeira_de_carpas:',
        'it': ':aquilone_a_forma_di_carpa:'
    },
    u'\U0001FA9A': { # 🪚
        'en' : ':carpentry_saw:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':handsäge:',
        'es': u':sierra_de_carpintería:',
        'fr': ':scie:',
        'pt': ':serrote:',
        'it': ':sega_da_falegname:'
    },
    u'\U0001F955': { # 🥕
        'en' : ':carrot:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':karotte:',
        'es': ':zanahoria:',
        'fr': ':carotte:',
        'pt': ':cenoura:',
        'it': ':carota:'
    },
    u'\U0001F3F0': { # 🏰
        'en' : ':castle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':european_castle:'],
        'de': ':schloss:',
        'es': ':castillo_europeo:',
        'fr': u':château:',
        'pt': ':castelo:',
        'it': ':castello:'
    },
    u'\U0001F408': { # 🐈
        'en' : ':cat:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':cat2:'],
        'variant': True,
        'de': ':katze:',
        'es': ':gato:',
        'fr': ':chat:',
        'pt': ':gato:',
        'it': ':gatto:'
    },
    u'\U0001F431': { # 🐱
        'en' : ':cat_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cat:'],
        'de': ':katzengesicht:',
        'es': ':cara_de_gato:',
        'fr': u':tête_de_chat:',
        'pt': ':rosto_de_gato:',
        'it': ':muso_di_gatto:'
    },
    u'\U0001F639': { # 😹
        'en' : ':cat_with_tears_of_joy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':joy_cat:'],
        'de': u':katze_mit_freudentränen:',
        'es': ':gato_llorando_de_risa:',
        'fr': ':chat_qui_pleure_de_joie:',
        'pt': u':rosto_de_gato_com_lágrimas_de_alegria:',
        'it': ':gatto_con_lacrime_di_gioia:'
    },
    u'\U0001F63C': { # 😼
        'en' : ':cat_with_wry_smile:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smirk_cat:'],
        'de': u':verwegen_lächelnde_katze:',
        'es': ':gato_haciendo_una_mueca:',
        'fr': ':chat_avec_sourire_en_coin:',
        'pt': u':rosto_de_gato_com_sorriso_irônico:',
        'it': ':gatto_con_sorriso_sarcastico:'
    },
    u'\U000026D3\U0000FE0F': { # ⛓️
        'en' : ':chains:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':ketten:',
        'es': ':cadenas:',
        'fr': u':chaînes:',
        'pt': ':correntes:',
        'it': ':catene:'
    },
    u'\U000026D3': { # ⛓
        'en' : ':chains:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':ketten:',
        'es': ':cadenas:',
        'fr': u':chaînes:',
        'pt': ':correntes:',
        'it': ':catene:'
    },
    u'\U0001FA91': { # 🪑
        'en' : ':chair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stuhl:',
        'es': ':silla:',
        'fr': ':chaise:',
        'pt': ':cadeira:',
        'it': ':sedia:'
    },
    u'\U0001F4C9': { # 📉
        'en' : ':chart_decreasing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':chart_with_downwards_trend:'],
        'de': u':abwärtstrend:',
        'es': u':gráfica_de_evolución_descendente:',
        'fr': ':graphique_en_baisse:',
        'pt': u':gráfico_caindo:',
        'it': ':grafico_con_andamento_negativo:'
    },
    u'\U0001F4C8': { # 📈
        'en' : ':chart_increasing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':chart_with_upwards_trend:'],
        'de': u':aufwärtstrend:',
        'es': u':gráfica_de_evolución_ascendente:',
        'fr': ':graphique_en_hausse:',
        'pt': u':gráfico_subindo:',
        'it': ':grafico_con_andamento_positivo:'
    },
    u'\U0001F4B9': { # 💹
        'en' : ':chart_increasing_with_yen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':chart:'],
        'de': ':steigende_kurve_mit_yen-zeichen:',
        'es': ':mercado_al_alza:',
        'fr': ':courbe_avec_yen_en_hausse:',
        'pt': u':gráfico_subindo_com_iene:',
        'it': ':grafico_ascendente:'
    },
    u'\U00002611\U0000FE0F': { # ☑️
        'en' : ':check_box_with_check:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ballot_box_with_check:'],
        'variant': True,
        'de': u':abstimmungsfeld_mit_häkchen:',
        'es': u':casilla_con_marca_de_verificación:',
        'fr': u':case_cochée:',
        'pt': u':caixa_de_seleção_marcada_com_tique:',
        'it': ':riquadro_con_spunta:'
    },
    u'\U00002611': { # ☑
        'en' : ':check_box_with_check:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':ballot_box_with_check:'],
        'variant': True,
        'de': u':abgehaktes_kästchen:',
        'es': u':casilla_con_marca_de_verificación:',
        'fr': u':case_cochée:',
        'pt': u':caixa_de_seleção_marcada_com_tique:',
        'it': ':riquadro_con_spunta:'
    },
    u'\U00002714\U0000FE0F': { # ✔️
        'en' : ':check_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heavy_check_mark:'],
        'variant': True,
        'de': u':kräftiges_häkchen:',
        'es': u':marca_de_verificación:',
        'fr': ':coche:',
        'pt': u':marca_de_seleção:',
        'it': ':segno_di_spunta_nero:'
    },
    u'\U00002714': { # ✔
        'en' : ':check_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':heavy_check_mark:'],
        'variant': True,
        'de': u':kräftiges_häkchen:',
        'es': u':marca_de_verificación:',
        'fr': ':coche:',
        'pt': u':marca_de_seleção:',
        'it': ':segno_di_spunta:'
    },
    u'\U00002705': { # ✅
        'en' : ':check_mark_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':white_check_mark:'],
        'de': u':weißes_häkchen:',
        'es': u':botón_de_marca_de_verificación:',
        'fr': u':bouton_coché:',
        'pt': u':marca_de_seleção_branca:',
        'it': ':pulsante_segno_di_spunta:'
    },
    u'\U0001F9C0': { # 🧀
        'en' : ':cheese_wedge:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':cheese:'],
        'de': u':käsestück:',
        'es': u':cuña_de_queso:',
        'fr': ':part_de_fromage:',
        'pt': ':queijo:',
        'it': ':fetta_di_formaggio:'
    },
    u'\U0001F3C1': { # 🏁
        'en' : ':chequered_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':checkered_flag:'],
        'de': ':zielflagge:',
        'es': ':bandera_de_cuadros:',
        'fr': u':drapeau_à_damier:',
        'pt': ':bandeira_quadriculada:',
        'it': ':bandiera_a_scacchi:'
    },
    u'\U0001F352': { # 🍒
        'en' : ':cherries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kirschen:',
        'es': ':cerezas:',
        'fr': ':cerises:',
        'pt': ':cereja:',
        'it': ':ciliegie:'
    },
    u'\U0001F338': { # 🌸
        'en' : ':cherry_blossom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kirschblüte:',
        'es': ':flor_de_cerezo:',
        'fr': ':fleur_de_cerisier:',
        'pt': ':flor_de_cerejeira:',
        'it': ':fiore_di_ciliegio:'
    },
    u'\U0000265F\U0000FE0F': { # ♟️
        'en' : ':chess_pawn:',
        'status' : fully_qualified,
        'E' : 11,
        'variant': True,
        'de': ':bauer_schach:',
        'es': u':peón_de_ajedrez:',
        'fr': u':pion_d’échec:',
        'pt': u':peão_de_xadrez:',
        'it': ':pedina_degli_scacchi:'
    },
    u'\U0000265F': { # ♟
        'en' : ':chess_pawn:',
        'status' : unqualified,
        'E' : 11,
        'variant': True,
        'de': ':bauer_schach:',
        'es': u':peón_de_ajedrez:',
        'fr': u':pion_d’échec:',
        'pt': u':peão_de_xadrez:',
        'it': ':pedina_degli_scacchi:'
    },
    u'\U0001F330': { # 🌰
        'en' : ':chestnut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kastanie:',
        'es': u':castaña:',
        'fr': u':châtaigne:',
        'pt': ':castanha:',
        'it': ':castagna:'
    },
    u'\U0001F414': { # 🐔
        'en' : ':chicken:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':huhn:',
        'es': ':gallina:',
        'fr': ':poule:',
        'pt': ':galinha:',
        'it': ':gallina:'
    },
    u'\U0001F9D2': { # 🧒
        'en' : ':child:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind:',
        'es': ':infante:',
        'fr': ':enfant:',
        'pt': u':criança:',
        'it': ':bimbo:'
    },
    u'\U0001F9D2\U0001F3FF': { # 🧒🏿
        'en' : ':child_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind_dunkle_hautfarbe:',
        'es': ':infante_tono_de_piel_oscuro:',
        'fr': u':enfant_peau_foncée:',
        'pt': u':criança_pele_escura:',
        'it': ':bimbo_carnagione_scura:'
    },
    u'\U0001F9D2\U0001F3FB': { # 🧒🏻
        'en' : ':child_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind_helle_hautfarbe:',
        'es': ':infante_tono_de_piel_claro:',
        'fr': ':enfant_peau_claire:',
        'pt': u':criança_pele_clara:',
        'it': ':bimbo_carnagione_chiara:'
    },
    u'\U0001F9D2\U0001F3FE': { # 🧒🏾
        'en' : ':child_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind_mitteldunkle_hautfarbe:',
        'es': ':infante_tono_de_piel_oscuro_medio:',
        'fr': ':enfant_peau_mate:',
        'pt': u':criança_pele_morena_escura:',
        'it': ':bimbo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D2\U0001F3FC': { # 🧒🏼
        'en' : ':child_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind_mittelhelle_hautfarbe:',
        'es': ':infante_tono_de_piel_claro_medio:',
        'fr': ':enfant_peau_moyennement_claire:',
        'pt': u':criança_pele_morena_clara:',
        'it': ':bimbo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D2\U0001F3FD': { # 🧒🏽
        'en' : ':child_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kind_mittlere_hautfarbe:',
        'es': ':infante_tono_de_piel_medio:',
        'fr': u':enfant_peau_légèrement_mate:',
        'pt': u':criança_pele_morena:',
        'it': ':bimbo_carnagione_olivastra:'
    },
    u'\U0001F6B8': { # 🚸
        'en' : ':children_crossing:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':kinder_überqueren_die_straße:',
        'es': u':niños_cruzando:',
        'fr': u':traversée_d’enfants:',
        'pt': u':crianças_atravessando:',
        'it': ':attraversamento_bambini:'
    },
    u'\U0001F43F\U0000FE0F': { # 🐿️
        'en' : ':chipmunk:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':streifenhörnchen:',
        'es': ':ardilla:',
        'fr': u':écureuil:',
        'pt': ':esquilo:',
        'it': ':scoiattolo:'
    },
    u'\U0001F43F': { # 🐿
        'en' : ':chipmunk:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':streifenhörnchen:',
        'es': ':ardilla:',
        'fr': u':écureuil:',
        'pt': ':esquilo:',
        'it': ':scoiattolo:'
    },
    u'\U0001F36B': { # 🍫
        'en' : ':chocolate_bar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':schokoladentafel:',
        'es': ':tableta_de_chocolate:',
        'fr': u':barre_chocolatée:',
        'pt': ':chocolate:',
        'it': ':cioccolato:'
    },
    u'\U0001F962': { # 🥢
        'en' : ':chopsticks:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':essstäbchen:',
        'es': ':palillos:',
        'fr': ':baguettes:',
        'pt': ':hashi:',
        'it': ':bacchette:'
    },
    u'\U000026EA': { # ⛪
        'en' : ':church:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':kirche:',
        'es': ':iglesia:',
        'fr': u':église:',
        'pt': ':igreja:',
        'it': ':chiesa:'
    },
    u'\U0001F6AC': { # 🚬
        'en' : ':cigarette:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smoking:'],
        'de': ':zigarette:',
        'es': ':cigarrillo:',
        'fr': ':cigarette:',
        'pt': ':cigarro:',
        'it': ':sigaretta:'
    },
    u'\U0001F3A6': { # 🎦
        'en' : ':cinema:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kinosymbol:',
        'es': ':cine:',
        'fr': u':cinéma:',
        'pt': ':cinema:',
        'it': ':simbolo_del_cinema:'
    },
    u'\U000024C2\U0000FE0F': { # Ⓜ️
        'en' : ':circled_M:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':m:'],
        'variant': True,
        'de': ':buchstabe_m_in_kreis:',
        'es': u':m_en_círculo:',
        'fr': u':m_encerclé:',
        'pt': u':círculo_com_a_letra_m:',
        'it': ':pulsante_m_cerchiata:'
    },
    u'\U000024C2': { # Ⓜ
        'en' : ':circled_M:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':m:'],
        'variant': True,
        'de': ':buchstabe_m_in_kreis:',
        'es': u':m_en_círculo:',
        'fr': u':m_encerclé:',
        'pt': u':círculo_com_a_letra_m:',
        'it': ':pulsante_m_cerchiata:'
    },
    u'\U0001F3AA': { # 🎪
        'en' : ':circus_tent:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':zirkuszelt:',
        'es': ':carpa_de_circo:',
        'fr': ':chapiteau:',
        'pt': ':circo:',
        'it': ':circo:'
    },
    u'\U0001F3D9\U0000FE0F': { # 🏙️
        'en' : ':cityscape:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':skyline:',
        'es': ':paisaje_urbano:',
        'fr': ':ville:',
        'pt': ':cidade:',
        'it': ':paesaggio_urbano:'
    },
    u'\U0001F3D9': { # 🏙
        'en' : ':cityscape:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':skyline:',
        'es': ':paisaje_urbano:',
        'fr': ':ville:',
        'pt': ':cidade:',
        'it': ':paesaggio_urbano:'
    },
    u'\U0001F306': { # 🌆
        'en' : ':cityscape_at_dusk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':city_sunset:'],
        'de': ':abendstimmung_in_der_stadt:',
        'es': ':ciudad_al_atardecer:',
        'fr': u':ville_au_crépuscule:',
        'pt': ':cidade_ao_anoitecer:',
        'it': u':città_al_tramonto:'
    },
    u'\U0001F5DC\U0000FE0F': { # 🗜️
        'en' : ':clamp:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':compression:'],
        'variant': True,
        'de': ':schraubzwinge:',
        'es': ':tornillo_de_banco:',
        'fr': ':serre-joint:',
        'pt': u':braçadeira:',
        'it': ':morsetto:'
    },
    u'\U0001F5DC': { # 🗜
        'en' : ':clamp:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':compression:'],
        'variant': True,
        'de': ':schraubzwinge:',
        'es': ':tornillo_de_banco:',
        'fr': ':serre-joint:',
        'pt': u':braçadeira:',
        'it': ':morsetto:'
    },
    u'\U0001F3AC': { # 🎬
        'en' : ':clapper_board:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clapper:'],
        'variant': True,
        'de': ':filmklappe:',
        'es': ':claqueta:',
        'fr': ':clap:',
        'pt': ':claquete:',
        'it': ':ciak:'
    },
    u'\U0001F44F': { # 👏
        'en' : ':clapping_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clap:'],
        'de': u':klatschende_hände:',
        'es': ':manos_aplaudiendo:',
        'fr': ':applaudissements:',
        'pt': u':mãos_aplaudindo:',
        'it': ':mani_che_applaudono:'
    },
    u'\U0001F44F\U0001F3FF': { # 👏🏿
        'en' : ':clapping_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_dunkle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_oscuro:',
        'fr': u':applaudissements_peau_foncée:',
        'pt': u':mãos_aplaudindo_pele_escura:',
        'it': ':mani_che_applaudono_carnagione_scura:'
    },
    u'\U0001F44F\U0001F3FB': { # 👏🏻
        'en' : ':clapping_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_helle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_claro:',
        'fr': ':applaudissements_peau_claire:',
        'pt': u':mãos_aplaudindo_pele_clara:',
        'it': ':mani_che_applaudono_carnagione_chiara:'
    },
    u'\U0001F44F\U0001F3FE': { # 👏🏾
        'en' : ':clapping_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mitteldunkle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_oscuro_medio:',
        'fr': ':applaudissements_peau_mate:',
        'pt': u':mãos_aplaudindo_pele_morena_escura:',
        'it': ':mani_che_applaudono_carnagione_abbastanza_scura:'
    },
    u'\U0001F44F\U0001F3FC': { # 👏🏼
        'en' : ':clapping_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mittelhelle_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_claro_medio:',
        'fr': ':applaudissements_peau_moyennement_claire:',
        'pt': u':mãos_aplaudindo_pele_morena_clara:',
        'it': ':mani_che_applaudono_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44F\U0001F3FD': { # 👏🏽
        'en' : ':clapping_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':klatschende_hände_mittlere_hautfarbe:',
        'es': ':manos_aplaudiendo_tono_de_piel_medio:',
        'fr': u':applaudissements_peau_légèrement_mate:',
        'pt': u':mãos_aplaudindo_pele_morena:',
        'it': ':mani_che_applaudono_carnagione_olivastra:'
    },
    u'\U0001F3DB\U0000FE0F': { # 🏛️
        'en' : ':classical_building:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':antikes_gebäude:',
        'es': u':edificio_clásico:',
        'fr': ':monument_classique:',
        'pt': u':prédio_grego:',
        'it': ':edificio_classico:'
    },
    u'\U0001F3DB': { # 🏛
        'en' : ':classical_building:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':antikes_gebäude:',
        'es': u':edificio_clásico:',
        'fr': ':monument_classique:',
        'pt': u':prédio_grego:',
        'it': ':edificio_classico:'
    },
    u'\U0001F37B': { # 🍻
        'en' : ':clinking_beer_mugs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':beers:'],
        'de': u':bierkrüge:',
        'es': ':jarras_de_cerveza_brindando:',
        'fr': u':chopes_de_bière:',
        'pt': ':canecas_de_cerveja:',
        'it': ':boccali_di_birra:'
    },
    u'\U0001F942': { # 🥂
        'en' : ':clinking_glasses:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':sektgläser:',
        'es': ':copas_brindando:',
        'fr': ':trinquer:',
        'pt': u':taças_brindando:',
        'it': ':brindisi:'
    },
    u'\U0001F4CB': { # 📋
        'en' : ':clipboard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':klemmbrett:',
        'es': ':portapapeles:',
        'fr': ':porte-bloc:',
        'pt': ':prancheta:',
        'it': ':portablocco:'
    },
    u'\U0001F503': { # 🔃
        'en' : ':clockwise_vertical_arrows:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrows_clockwise:'],
        'de': u':kreisförmige_pfeile_im_uhrzeigersinn:',
        'es': ':flechas_verticales_en_sentido_horario:',
        'fr': u':flèches_dans_le_sens_horaire:',
        'pt': u':setas_verticais_no_sentido_horário:',
        'it': ':frecce_verticali_che_ruotano_in_senso_orario:'
    },
    u'\U0001F4D5': { # 📕
        'en' : ':closed_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':geschlossenes_buch:',
        'es': ':libro_cerrado:',
        'fr': u':livre_fermé:',
        'pt': ':livro_fechado:',
        'it': ':libro_chiuso:'
    },
    u'\U0001F4EA': { # 📪
        'en' : ':closed_mailbox_with_lowered_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mailbox_closed:'],
        'variant': True,
        'de': ':geschlossener_briefkasten_ohne_post:',
        'es': u':buzón_cerrado_con_la_bandera_bajada:',
        'fr': u':boîte_aux_lettres_fermée_drapeau_baissé:',
        'pt': ':caixa_de_correio_fechada_com_bandeira_abaixada:',
        'it': ':cassetta_postale_chiusa_bandierina_abbassata:'
    },
    u'\U0001F4EB': { # 📫
        'en' : ':closed_mailbox_with_raised_flag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mailbox:'],
        'variant': True,
        'de': ':geschlossener_briefkasten_mit_post:',
        'es': u':buzón_cerrado_con_la_bandera_levantada:',
        'fr': u':boîte_aux_lettres_fermée_drapeau_levé:',
        'pt': ':caixa_de_correio_fechada_com_bandeira_levantada:',
        'it': ':cassetta_postale_chiusa_bandierina_alzata:'
    },
    u'\U0001F302': { # 🌂
        'en' : ':closed_umbrella:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':geschlossener_regenschirm:',
        'es': ':paraguas_cerrado:',
        'fr': u':parapluie_fermé:',
        'pt': ':guarda-chuva_fechado:',
        'it': ':ombrello_chiuso:'
    },
    u'\U00002601\U0000FE0F': { # ☁️
        'en' : ':cloud:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':wolke:',
        'es': ':nube:',
        'fr': ':nuage:',
        'pt': ':nuvem:',
        'it': ':nuvola:'
    },
    u'\U00002601': { # ☁
        'en' : ':cloud:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':wolke:',
        'es': ':nube:',
        'fr': ':nuage:',
        'pt': ':nuvem:',
        'it': ':nuvola:'
    },
    u'\U0001F329\U0000FE0F': { # 🌩️
        'en' : ':cloud_with_lightning:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_blitz:',
        'es': ':nube_con_rayo:',
        'fr': u':nuage_avec_éclair:',
        'pt': u':nuvem_com_trovão:',
        'it': ':fulmini:'
    },
    u'\U0001F329': { # 🌩
        'en' : ':cloud_with_lightning:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_blitz:',
        'es': ':nube_con_rayo:',
        'fr': u':nuage_avec_éclair:',
        'pt': u':nuvem_com_trovão:',
        'it': ':fulmini:'
    },
    u'\U000026C8\U0000FE0F': { # ⛈️
        'en' : ':cloud_with_lightning_and_rain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':thunder_cloud_and_rain:'],
        'variant': True,
        'de': ':wolke_mit_blitz_und_regen:',
        'es': ':nube_con_rayo_y_lluvia:',
        'fr': u':nuage_avec_éclair_et_pluie:',
        'pt': u':chuva_com_trovão:',
        'it': ':temporale:'
    },
    u'\U000026C8': { # ⛈
        'en' : ':cloud_with_lightning_and_rain:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':thunder_cloud_and_rain:'],
        'variant': True,
        'de': ':wolke_mit_blitz_und_regen:',
        'es': ':nube_con_rayo_y_lluvia:',
        'fr': u':nuage_avec_éclair_et_pluie:',
        'pt': u':chuva_com_trovão:',
        'it': ':temporale:'
    },
    u'\U0001F327\U0000FE0F': { # 🌧️
        'en' : ':cloud_with_rain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_regen:',
        'es': ':nube_con_lluvia:',
        'fr': ':nuage_avec_pluie:',
        'pt': ':nuvem_com_chuva:',
        'it': ':pioggia:'
    },
    u'\U0001F327': { # 🌧
        'en' : ':cloud_with_rain:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_regen:',
        'es': ':nube_con_lluvia:',
        'fr': ':nuage_avec_pluie:',
        'pt': ':nuvem_com_chuva:',
        'it': ':pioggia:'
    },
    u'\U0001F328\U0000FE0F': { # 🌨️
        'en' : ':cloud_with_snow:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_schnee:',
        'es': ':nube_con_nieve:',
        'fr': ':nuage_avec_neige:',
        'pt': ':nuvem_com_neve:',
        'it': ':neve:'
    },
    u'\U0001F328': { # 🌨
        'en' : ':cloud_with_snow:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':wolke_mit_schnee:',
        'es': ':nube_con_nieve:',
        'fr': ':nuage_avec_neige:',
        'pt': ':nuvem_com_neve:',
        'it': ':neve:'
    },
    u'\U0001F921': { # 🤡
        'en' : ':clown_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':clown-gesicht:',
        'es': ':cara_de_payaso:',
        'fr': ':visage_de_clown:',
        'pt': u':rosto_de_palhaço:',
        'it': ':faccina_pagliaccio:'
    },
    u'\U00002663\U0000FE0F': { # ♣️
        'en' : ':club_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clubs:'],
        'variant': True,
        'de': ':kreuz:',
        'es': u':palo_de_tréboles:',
        'fr': u':trèfle_cartes:',
        'pt': ':naipe_de_paus:',
        'it': ':fiori:'
    },
    u'\U00002663': { # ♣
        'en' : ':club_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':clubs:'],
        'variant': True,
        'de': ':kreuz:',
        'es': u':palo_de_tréboles:',
        'fr': u':trèfle_cartes:',
        'pt': ':naipe_de_paus:',
        'it': ':fiori:'
    },
    u'\U0001F45D': { # 👝
        'en' : ':clutch_bag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pouch:'],
        'de': ':clutch:',
        'es': ':bolso_de_mano:',
        'fr': ':pochette:',
        'pt': ':bolsa_pequena:',
        'it': ':pochette:'
    },
    u'\U0001F9E5': { # 🧥
        'en' : ':coat:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mantel:',
        'es': ':abrigo:',
        'fr': ':manteau:',
        'pt': ':casaco:',
        'it': ':cappotto:'
    },
    u'\U0001FAB3': { # 🪳
        'en' : ':cockroach:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':kakerlake:',
        'es': ':cucaracha:',
        'fr': ':cafard:',
        'pt': ':barata:',
        'it': ':scarafaggio:'
    },
    u'\U0001F378': { # 🍸
        'en' : ':cocktail_glass:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cocktail:'],
        'variant': True,
        'de': ':cocktailglas:',
        'es': u':copa_de_cóctel:',
        'fr': ':cocktail:',
        'pt': ':coquetel:',
        'it': ':cocktail:'
    },
    u'\U0001F965': { # 🥥
        'en' : ':coconut:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':kokosnuss:',
        'es': ':coco:',
        'fr': ':noix_de_coco:',
        'pt': ':coco:',
        'it': ':cocco:'
    },
    u'\U000026B0\U0000FE0F': { # ⚰️
        'en' : ':coffin:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':sarg:',
        'es': u':ataúd:',
        'fr': ':cercueil:',
        'pt': u':caixão:',
        'it': ':bara:'
    },
    u'\U000026B0': { # ⚰
        'en' : ':coffin:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':sarg:',
        'es': u':ataúd:',
        'fr': ':cercueil:',
        'pt': u':caixão:',
        'it': ':bara:'
    },
    u'\U0001FA99': { # 🪙
        'en' : ':coin:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':münze:',
        'es': ':moneda:',
        'fr': u':pièce:',
        'pt': ':moeda:',
        'it': ':moneta:'
    },
    u'\U0001F976': { # 🥶
        'en' : ':cold_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':frierendes_gesicht:',
        'es': u':cara_con_frío:',
        'fr': ':visage_bleu_et_froid:',
        'pt': ':rosto_gelado:',
        'it': ':faccina_congelata:'
    },
    u'\U0001F4A5': { # 💥
        'en' : ':collision:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':boom:'],
        'de': u':zusammenstoß:',
        'es': u':colisión:',
        'fr': ':explosion:',
        'pt': u':colisão:',
        'it': ':collisione:'
    },
    u'\U00002604\U0000FE0F': { # ☄️
        'en' : ':comet:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':komet:',
        'es': ':meteorito:',
        'fr': u':comète:',
        'pt': ':cometa:',
        'it': ':cometa:'
    },
    u'\U00002604': { # ☄
        'en' : ':comet:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':komet:',
        'es': ':meteorito:',
        'fr': u':comète:',
        'pt': ':cometa:',
        'it': ':cometa:'
    },
    u'\U0001F9ED': { # 🧭
        'en' : ':compass:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':kompass:',
        'es': u':brújula:',
        'fr': ':boussole:',
        'pt': u':bússola:',
        'it': ':bussola:'
    },
    u'\U0001F4BD': { # 💽
        'en' : ':computer_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':minidisc:'],
        'de': ':minidisc:',
        'es': ':minidisc:',
        'fr': u':disque_d’ordinateur:',
        'pt': ':minidisc:',
        'it': ':minidisc:'
    },
    u'\U0001F5B1\U0000FE0F': { # 🖱️
        'en' : ':computer_mouse:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':three_button_mouse:'],
        'variant': True,
        'de': ':computermaus:',
        'es': u':ratón_de_ordenador:',
        'fr': u':souris_d’ordinateur:',
        'pt': ':mouse:',
        'it': ':mouse:'
    },
    u'\U0001F5B1': { # 🖱
        'en' : ':computer_mouse:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':three_button_mouse:'],
        'variant': True,
        'de': ':computermaus:',
        'es': u':ratón_de_ordenador:',
        'fr': u':souris_d’ordinateur:',
        'pt': ':mouse:',
        'it': ':mouse:'
    },
    u'\U0001F38A': { # 🎊
        'en' : ':confetti_ball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':konfettiball:',
        'es': ':bola_de_confeti:',
        'fr': ':confettis:',
        'pt': ':confete:',
        'it': ':coriandoli:'
    },
    u'\U0001F616': { # 😖
        'en' : ':confounded_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':confounded:'],
        'de': ':verwirrtes_gesicht:',
        'es': u':cara_de_frustración:',
        'fr': u':visage_déconcerté:',
        'pt': ':rosto_perplexo:',
        'it': ':faccina_frustrata:'
    },
    u'\U0001F615': { # 😕
        'en' : ':confused_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':confused:'],
        'de': ':verwundertes_gesicht:',
        'es': u':cara_de_confusión:',
        'fr': ':visage_confus:',
        'pt': ':rosto_confuso:',
        'it': ':faccina_confusa:'
    },
    u'\U0001F6A7': { # 🚧
        'en' : ':construction:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':baustellenabsperrung:',
        'es': ':obras:',
        'fr': ':travaux:',
        'pt': u':em_construção:',
        'it': ':area_cantiere:'
    },
    u'\U0001F477': { # 👷
        'en' : ':construction_worker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':bauarbeiter(in):',
        'es': u':profesional_de_la_construcción:',
        'fr': u':personnel_du_bâtiment:',
        'pt': u':trabalhador_de_construção_civil:',
        'it': ':operaio_edile:'
    },
    u'\U0001F477\U0001F3FF': { # 👷🏿
        'en' : ':construction_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bauarbeiter(in)_dunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_oscuro:',
        'fr': u':personnel_du_bâtiment_peau_foncée:',
        'pt': u':trabalhador_de_construção_civil_pele_escura:',
        'it': ':operaio_edile_carnagione_scura:'
    },
    u'\U0001F477\U0001F3FB': { # 👷🏻
        'en' : ':construction_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bauarbeiter(in)_helle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_claro:',
        'fr': u':personnel_du_bâtiment_peau_claire:',
        'pt': u':trabalhador_de_construção_civil_pele_clara:',
        'it': ':operaio_edile_carnagione_chiara:'
    },
    u'\U0001F477\U0001F3FE': { # 👷🏾
        'en' : ':construction_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bauarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_oscuro_medio:',
        'fr': u':personnel_du_bâtiment_peau_mate:',
        'pt': u':trabalhador_de_construção_civil_pele_morena_escura:',
        'it': ':operaio_edile_carnagione_abbastanza_scura:'
    },
    u'\U0001F477\U0001F3FC': { # 👷🏼
        'en' : ':construction_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bauarbeiter(in)_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_claro_medio:',
        'fr': u':personnel_du_bâtiment_peau_moyennement_claire:',
        'pt': u':trabalhador_de_construção_civil_pele_morena_clara:',
        'it': ':operaio_edile_carnagione_abbastanza_chiara:'
    },
    u'\U0001F477\U0001F3FD': { # 👷🏽
        'en' : ':construction_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bauarbeiter(in)_mittlere_hautfarbe:',
        'es': u':profesional_de_la_construcción_tono_de_piel_medio:',
        'fr': u':personnel_du_bâtiment_peau_légèrement_mate:',
        'pt': u':trabalhador_de_construção_civil_pele_morena:',
        'it': ':operaio_edile_carnagione_olivastra:'
    },
    u'\U0001F39B\U0000FE0F': { # 🎛️
        'en' : ':control_knobs:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':drehregler:',
        'es': ':ruedas_de_control:',
        'fr': u':boutons_de_réglage:',
        'pt': u':botões_giratórios:',
        'it': ':manopole_di_controllo:'
    },
    u'\U0001F39B': { # 🎛
        'en' : ':control_knobs:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':drehregler:',
        'es': ':ruedas_de_control:',
        'fr': u':boutons_de_réglage:',
        'pt': u':botões_giratórios:',
        'it': ':manopole_di_controllo:'
    },
    u'\U0001F3EA': { # 🏪
        'en' : ':convenience_store:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':minimarkt:',
        'es': ':tienda_24_horas:',
        'fr': u':supérette:',
        'pt': u':loja_de_conveniência:',
        'it': ':minimarket:'
    },
    u'\U0001F9D1\U0000200D\U0001F373': { # 🧑‍🍳
        'en' : ':cook:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin:',
        'es': ':chef:',
        'fr': ':cuisinier_(tous_genres):',
        'pt': ':chef_de_cozinha:',
        'it': ':persona_che_cucina:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F373': { # 🧑🏿‍🍳
        'en' : ':cook_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_dunkle_hautfarbe:',
        'es': ':chef_tono_de_piel_oscuro:',
        'fr': u':cuisinier_(tous_genres)_peau_foncée:',
        'pt': ':chef_de_cozinha_pele_escura:',
        'it': ':persona_che_cucina_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F373': { # 🧑🏻‍🍳
        'en' : ':cook_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_helle_hautfarbe:',
        'es': ':chef_tono_de_piel_claro:',
        'fr': ':cuisinier_(tous_genres)_peau_claire:',
        'pt': ':chef_de_cozinha_pele_clara:',
        'it': ':persona_che_cucina_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F373': { # 🧑🏾‍🍳
        'en' : ':cook_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mitteldunkle_hautfarbe:',
        'es': ':chef_tono_de_piel_oscuro_medio:',
        'fr': ':cuisinier_(tous_genres)_peau_mate:',
        'pt': ':chef_de_cozinha_pele_morena_escura:',
        'it': ':persona_che_cucina_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F373': { # 🧑🏼‍🍳
        'en' : ':cook_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mittelhelle_hautfarbe:',
        'es': ':chef_tono_de_piel_claro_medio:',
        'fr': ':cuisinier_(tous_genres)_peau_moyennement_claire:',
        'pt': ':chef_de_cozinha_pele_morena_clara:',
        'it': ':persona_che_cucina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F373': { # 🧑🏽‍🍳
        'en' : ':cook_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':koch/köchin_mittlere_hautfarbe:',
        'es': ':chef_tono_de_piel_medio:',
        'fr': u':cuisinier_(tous_genres)_peau_légèrement_mate:',
        'pt': ':chef_de_cozinha_pele_morena:',
        'it': ':persona_che_cucina_carnagione_olivastra:'
    },
    u'\U0001F35A': { # 🍚
        'en' : ':cooked_rice:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':rice:'],
        'de': u':reis_in_schüssel:',
        'es': ':arroz_cocido:',
        'fr': ':bol_de_riz:',
        'pt': ':arroz_cozido:',
        'it': ':riso_bollito:'
    },
    u'\U0001F36A': { # 🍪
        'en' : ':cookie:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':keks:',
        'es': ':galleta:',
        'fr': ':cookie:',
        'pt': ':biscoito:',
        'it': ':biscotto:'
    },
    u'\U0001F373': { # 🍳
        'en' : ':cooking:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':egg:', ':fried_egg:'],
        'de': ':spiegelei_in_bratpfanne:',
        'es': ':cocinar:',
        'fr': u':œuf_au_plat:',
        'pt': ':ovo_frito:',
        'it': ':cucinare:'
    },
    u'\U000000A9\U0000FE0F': { # ©️
        'en' : ':copyright:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':copyright:',
        'es': ':copyright:',
        'fr': ':symbole_copyright:',
        'pt': u':símbolo_de_copyright:',
        'it': ':copyright:'
    },
    u'\U000000A9': { # ©
        'en' : ':copyright:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':copyright:',
        'es': ':copyright:',
        'fr': ':symbole_copyright:',
        'pt': u':símbolo_de_copyright:',
        'it': ':copyright:'
    },
    u'\U0001FAB8': { # 🪸
        'en' : ':coral:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':koralle:',
        'es': ':coral:',
        'fr': ':corail:',
        'pt': ':coral:',
        'it': ':corallo:'
    },
    u'\U0001F6CB\U0000FE0F': { # 🛋️
        'en' : ':couch_and_lamp:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':sofa_und_lampe:',
        'es': u':sofá_y_lámpara:',
        'fr': u':canapé_et_lampe:',
        'pt': u':sofá_e_luminária:',
        'it': ':divano_e_lampada:'
    },
    u'\U0001F6CB': { # 🛋
        'en' : ':couch_and_lamp:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':sofa_und_lampe:',
        'es': u':sofá_y_lámpara:',
        'fr': u':canapé_et_lampe:',
        'pt': u':sofá_e_luminária:',
        'it': ':divano_e_lampada:'
    },
    u'\U0001F504': { # 🔄
        'en' : ':counterclockwise_arrows_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':arrows_counterclockwise:'],
        'de': ':pfeile_gegen_den_uhrzeigersinn:',
        'es': ':flechas_en_sentido_antihorario:',
        'fr': u':flèches_dans_le_sens_antihoraire:',
        'pt': u':botão_de_setas_em_sentido_anti-horário:',
        'it': ':frecce_che_ruotano_in_senso_antiorario:'
    },
    u'\U0001F491': { # 💑
        'en' : ':couple_with_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':liebespaar:',
        'es': ':pareja_enamorada:',
        'fr': u':couple_avec_cœur:',
        'pt': ':casal_apaixonado:',
        'it': ':coppia_con_cuore:'
    },
    u'\U0001F491\U0001F3FF': { # 💑🏿
        'en' : ':couple_with_heart_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FB': { # 💑🏻
        'en' : ':couple_with_heart_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': { # 👨‍❤️‍👨
        'en' : ':couple_with_heart_man_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':liebespaar_mann,_mann:',
        'es': ':pareja_enamorada_hombre_y_hombre:',
        'fr': u':couple_avec_cœur_homme_et_homme:',
        'pt': ':casal_apaixonado_homem_e_homem:',
        'it': ':coppia_con_cuore_uomo_e_uomo:'
    },
    u'\U0001F468\U0000200D\U00002764\U0000200D\U0001F468': { # 👨‍❤‍👨
        'en' : ':couple_with_heart_man_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👨🏿‍❤️‍👨🏿
        'en' : ':couple_with_heart_man_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👨🏿‍❤‍👨🏿
        'en' : ':couple_with_heart_man_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👨🏿‍❤️‍👨🏻
        'en' : ':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👨🏿‍❤‍👨🏻
        'en' : ':couple_with_heart_man_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👨🏿‍❤️‍👨🏾
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👨🏿‍❤‍👨🏾
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👨🏿‍❤️‍👨🏼
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👨🏿‍❤‍👨🏼
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👨🏿‍❤️‍👨🏽
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👨🏿‍❤‍👨🏽
        'en' : ':couple_with_heart_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👨🏻‍❤️‍👨🏻
        'en' : ':couple_with_heart_man_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👨🏻‍❤‍👨🏻
        'en' : ':couple_with_heart_man_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👨🏻‍❤️‍👨🏿
        'en' : ':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👨🏻‍❤‍👨🏿
        'en' : ':couple_with_heart_man_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👨🏻‍❤️‍👨🏾
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👨🏻‍❤‍👨🏾
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👨🏻‍❤️‍👨🏼
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👨🏻‍❤‍👨🏼
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👨🏻‍❤️‍👨🏽
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👨🏻‍❤‍👨🏽
        'en' : ':couple_with_heart_man_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👨🏾‍❤️‍👨🏾
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👨🏾‍❤‍👨🏾
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👨🏾‍❤️‍👨🏿
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👨🏾‍❤‍👨🏿
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👨🏾‍❤️‍👨🏻
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👨🏾‍❤‍👨🏻
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👨🏾‍❤️‍👨🏼
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👨🏾‍❤‍👨🏼
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👨🏾‍❤️‍👨🏽
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👨🏾‍❤‍👨🏽
        'en' : ':couple_with_heart_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👨🏼‍❤️‍👨🏼
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👨🏼‍❤‍👨🏼
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👨🏼‍❤️‍👨🏿
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👨🏼‍❤‍👨🏿
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👨🏼‍❤️‍👨🏻
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👨🏼‍❤‍👨🏻
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👨🏼‍❤️‍👨🏾
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👨🏼‍❤‍👨🏾
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👨🏼‍❤️‍👨🏽
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👨🏼‍❤‍👨🏽
        'en' : ':couple_with_heart_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👨🏽‍❤️‍👨🏽
        'en' : ':couple_with_heart_man_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👨🏽‍❤‍👨🏽
        'en' : ':couple_with_heart_man_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👨🏽‍❤️‍👨🏿
        'en' : ':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👨🏽‍❤‍👨🏿
        'en' : ':couple_with_heart_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👨🏽‍❤️‍👨🏻
        'en' : ':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👨🏽‍❤‍👨🏻
        'en' : ':couple_with_heart_man_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👨🏽‍❤️‍👨🏾
        'en' : ':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👨🏽‍❤‍👨🏾
        'en' : ':couple_with_heart_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👨🏽‍❤️‍👨🏼
        'en' : ':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👨🏽‍❤‍👨🏼
        'en' : ':couple_with_heart_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FE': { # 💑🏾
        'en' : ':couple_with_heart_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FC': { # 💑🏼
        'en' : ':couple_with_heart_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F491\U0001F3FD': { # 💑🏽
        'en' : ':couple_with_heart_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏿‍❤️‍🧑🏻
        'en' : ':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏿‍❤‍🧑🏻
        'en' : ':couple_with_heart_person_person_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏿‍❤️‍🧑🏾
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏿‍❤‍🧑🏾
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏿‍❤️‍🧑🏼
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏿‍❤‍🧑🏼
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏿‍❤️‍🧑🏽
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏿‍❤‍🧑🏽
        'en' : ':couple_with_heart_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏻‍❤️‍🧑🏿
        'en' : ':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏻‍❤‍🧑🏿
        'en' : ':couple_with_heart_person_person_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏻‍❤️‍🧑🏾
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏻‍❤‍🧑🏾
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏻‍❤️‍🧑🏼
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏻‍❤‍🧑🏼
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏻‍❤️‍🧑🏽
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏻‍❤‍🧑🏽
        'en' : ':couple_with_heart_person_person_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏾‍❤️‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏾‍❤‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏾‍❤️‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏾‍❤‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏾‍❤️‍🧑🏼
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏾‍❤‍🧑🏼
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏾‍❤️‍🧑🏽
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏾‍❤‍🧑🏽
        'en' : ':couple_with_heart_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏼‍❤️‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏼‍❤‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏼‍❤️‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏼‍❤‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏼‍❤️‍🧑🏾
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏼‍❤‍🧑🏾
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏼‍❤️‍🧑🏽
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏼‍❤‍🧑🏽
        'en' : ':couple_with_heart_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏽‍❤️‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏽‍❤‍🧑🏿
        'en' : ':couple_with_heart_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏽‍❤️‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏽‍❤‍🧑🏻
        'en' : ':couple_with_heart_person_person_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏽‍❤️‍🧑🏾
        'en' : ':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏽‍❤‍🧑🏾
        'en' : ':couple_with_heart_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏽‍❤️‍🧑🏼
        'en' : ':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏽‍❤‍🧑🏼
        'en' : ':couple_with_heart_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468': { # 👩‍❤️‍👨
        'en' : ':couple_with_heart_woman_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':liebespaar_frau,_mann:',
        'es': ':pareja_enamorada_mujer_y_hombre:',
        'fr': u':couple_avec_cœur_femme_et_homme:',
        'pt': ':casal_apaixonado_mulher_e_homem:',
        'it': ':coppia_con_cuore_donna_e_uomo:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F468': { # 👩‍❤‍👨
        'en' : ':couple_with_heart_woman_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👩🏿‍❤️‍👨🏿
        'en' : ':couple_with_heart_woman_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👩🏿‍❤‍👨🏿
        'en' : ':couple_with_heart_woman_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👩🏿‍❤️‍👨🏻
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👩🏿‍❤‍👨🏻
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👩🏿‍❤️‍👨🏾
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👩🏿‍❤‍👨🏾
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👩🏿‍❤️‍👨🏼
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👩🏿‍❤‍👨🏼
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👩🏿‍❤️‍👨🏽
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👩🏿‍❤‍👨🏽
        'en' : ':couple_with_heart_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👩🏻‍❤️‍👨🏻
        'en' : ':couple_with_heart_woman_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👩🏻‍❤‍👨🏻
        'en' : ':couple_with_heart_woman_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👩🏻‍❤️‍👨🏿
        'en' : ':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👩🏻‍❤‍👨🏿
        'en' : ':couple_with_heart_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👩🏻‍❤️‍👨🏾
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👩🏻‍❤‍👨🏾
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👩🏻‍❤️‍👨🏼
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👩🏻‍❤‍👨🏼
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👩🏻‍❤️‍👨🏽
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👩🏻‍❤‍👨🏽
        'en' : ':couple_with_heart_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👩🏾‍❤️‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👩🏾‍❤‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👩🏾‍❤️‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👩🏾‍❤‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👩🏾‍❤️‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👩🏾‍❤‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👩🏾‍❤️‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👩🏾‍❤‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👩🏾‍❤️‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👩🏾‍❤‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👩🏼‍❤️‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👩🏼‍❤‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👩🏼‍❤️‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👩🏼‍❤‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👩🏼‍❤️‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👩🏼‍❤‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👩🏼‍❤️‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👩🏼‍❤‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👩🏼‍❤️‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👩🏼‍❤‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FD': { # 👩🏽‍❤️‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FD': { # 👩🏽‍❤‍👨🏽
        'en' : ':couple_with_heart_woman_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FF': { # 👩🏽‍❤️‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FF': { # 👩🏽‍❤‍👨🏿
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FB': { # 👩🏽‍❤️‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FB': { # 👩🏽‍❤‍👨🏻
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FE': { # 👩🏽‍❤️‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FE': { # 👩🏽‍❤‍👨🏾
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F468\U0001F3FC': { # 👩🏽‍❤️‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F468\U0001F3FC': { # 👩🏽‍❤‍👨🏼
        'en' : ':couple_with_heart_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469': { # 👩‍❤️‍👩
        'en' : ':couple_with_heart_woman_woman:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':liebespaar_frau,_frau:',
        'es': ':pareja_enamorada_mujer_y_mujer:',
        'fr': u':couple_avec_cœur_femme_et_femme:',
        'pt': ':casal_apaixonado_mulher_e_mulher:',
        'it': ':coppia_con_cuore_donna_e_donna:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F469': { # 👩‍❤‍👩
        'en' : ':couple_with_heart_woman_woman:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': { # 👩🏿‍❤️‍👩🏿
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': { # 👩🏿‍❤‍👩🏿
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': { # 👩🏿‍❤️‍👩🏻
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': { # 👩🏿‍❤‍👩🏻
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': { # 👩🏿‍❤️‍👩🏾
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': { # 👩🏿‍❤‍👩🏾
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': { # 👩🏿‍❤️‍👩🏼
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': { # 👩🏿‍❤‍👩🏼
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': { # 👩🏿‍❤️‍👩🏽
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': { # 👩🏿‍❤‍👩🏽
        'en' : ':couple_with_heart_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': { # 👩🏻‍❤️‍👩🏻
        'en' : ':couple_with_heart_woman_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': { # 👩🏻‍❤‍👩🏻
        'en' : ':couple_with_heart_woman_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': { # 👩🏻‍❤️‍👩🏿
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': { # 👩🏻‍❤‍👩🏿
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': { # 👩🏻‍❤️‍👩🏾
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': { # 👩🏻‍❤‍👩🏾
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': { # 👩🏻‍❤️‍👩🏼
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': { # 👩🏻‍❤‍👩🏼
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': { # 👩🏻‍❤️‍👩🏽
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': { # 👩🏻‍❤‍👩🏽
        'en' : ':couple_with_heart_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': { # 👩🏾‍❤️‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': { # 👩🏾‍❤‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': { # 👩🏾‍❤️‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': { # 👩🏾‍❤‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': { # 👩🏾‍❤️‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': { # 👩🏾‍❤‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': { # 👩🏾‍❤️‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': { # 👩🏾‍❤‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': { # 👩🏾‍❤️‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': { # 👩🏾‍❤‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': { # 👩🏼‍❤️‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': { # 👩🏼‍❤‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': { # 👩🏼‍❤️‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': { # 👩🏼‍❤‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': { # 👩🏼‍❤️‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': { # 👩🏼‍❤‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': { # 👩🏼‍❤️‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': { # 👩🏼‍❤‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': { # 👩🏼‍❤️‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': { # 👩🏼‍❤‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FD': { # 👩🏽‍❤️‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FD': { # 👩🏽‍❤‍👩🏽
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FF': { # 👩🏽‍❤️‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FF': { # 👩🏽‍❤‍👩🏿
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FB': { # 👩🏽‍❤️‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FB': { # 👩🏽‍❤‍👩🏻
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FE': { # 👩🏽‍❤️‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FE': { # 👩🏽‍❤‍👩🏾
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F469\U0001F3FC': { # 👩🏽‍❤️‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F469\U0001F3FC': { # 👩🏽‍❤‍👩🏼
        'en' : ':couple_with_heart_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F404': { # 🐄
        'en' : ':cow:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':cow2:'],
        'de': ':kuh:',
        'es': ':vaca:',
        'fr': ':vache:',
        'pt': ':vaca:',
        'it': ':mucca:'
    },
    u'\U0001F42E': { # 🐮
        'en' : ':cow_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cow:'],
        'de': ':kuhgesicht:',
        'es': ':cara_de_vaca:',
        'fr': u':tête_de_vache:',
        'pt': ':rosto_de_vaca:',
        'it': ':muso_di_mucca:'
    },
    u'\U0001F920': { # 🤠
        'en' : ':cowboy_hat_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':gesicht_mit_cowboyhut:',
        'es': ':cara_con_sombrero_de_vaquero:',
        'fr': ':visage_avec_chapeau_de_cowboy:',
        'pt': u':rosto_com_chapéu_de_caubói:',
        'it': ':faccina_con_cappello_da_cowboy:'
    },
    u'\U0001F980': { # 🦀
        'en' : ':crab:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':krebs:',
        'es': ':cangrejo:',
        'fr': ':crabe:',
        'pt': ':caranguejo:',
        'it': ':granchio:'
    },
    u'\U0001F58D\U0000FE0F': { # 🖍️
        'en' : ':crayon:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':lower_left_crayon:'],
        'variant': True,
        'de': ':wachsmalstift:',
        'es': u':lápiz_de_cera:',
        'fr': ':crayon_pastel:',
        'pt': ':giz_de_cera:',
        'it': ':pastello_a_cera:'
    },
    u'\U0001F58D': { # 🖍
        'en' : ':crayon:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':lower_left_crayon:'],
        'variant': True,
        'de': ':wachsmalstift:',
        'es': u':lápiz_de_cera:',
        'fr': ':crayon_pastel:',
        'pt': ':giz_de_cera:',
        'it': ':pastello_a_cera:'
    },
    u'\U0001F4B3': { # 💳
        'en' : ':credit_card:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':kreditkarte:',
        'es': u':tarjeta_de_crédito:',
        'fr': ':carte_bancaire:',
        'pt': u':cartão_de_crédito:',
        'it': ':carta_di_credito:'
    },
    u'\U0001F319': { # 🌙
        'en' : ':crescent_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':mondsichel:',
        'es': ':luna:',
        'fr': ':croissant_de_lune:',
        'pt': ':lua_crescente:',
        'it': ':spicchio_di_luna:'
    },
    u'\U0001F997': { # 🦗
        'en' : ':cricket:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':grille:',
        'es': ':grillo:',
        'fr': ':criquet:',
        'pt': ':grilo:',
        'it': ':grillo:'
    },
    u'\U0001F3CF': { # 🏏
        'en' : ':cricket_game:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':cricket_bat_and_ball:'],
        'de': ':kricket:',
        'es': u':críquet:',
        'fr': ':cricket:',
        'pt': u':críquete:',
        'it': ':cricket:'
    },
    u'\U0001F40A': { # 🐊
        'en' : ':crocodile:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':krokodil:',
        'es': ':cocodrilo:',
        'fr': ':crocodile:',
        'pt': ':crocodilo:',
        'it': ':coccodrillo:'
    },
    u'\U0001F950': { # 🥐
        'en' : ':croissant:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':croissant:',
        'es': u':cruasán:',
        'fr': ':croissant:',
        'pt': ':croissant:',
        'it': ':croissant:'
    },
    u'\U0000274C': { # ❌
        'en' : ':cross_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':x:'],
        'de': ':kreuzzeichen:',
        'es': ':marca_de_cruz:',
        'fr': ':croix:',
        'pt': ':xis:',
        'it': ':croce:'
    },
    u'\U0000274E': { # ❎
        'en' : ':cross_mark_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':negative_squared_cross_mark:'],
        'de': u':angekreuztes_kästchen:',
        'es': u':botón_con_marca_de_cruz:',
        'fr': ':bouton_croix:',
        'pt': u':botão_de_xis:',
        'it': ':croce_con_quadrato:'
    },
    u'\U0001F91E': { # 🤞
        'en' : ':crossed_fingers:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern:',
        'es': ':dedos_cruzados:',
        'fr': u':doigts_croisés:',
        'pt': ':dedos_cruzados:',
        'it': ':dita_incrociate:'
    },
    u'\U0001F91E\U0001F3FF': { # 🤞🏿
        'en' : ':crossed_fingers_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern_dunkle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_oscuro:',
        'fr': u':doigts_croisés_peau_foncée:',
        'pt': ':dedos_cruzados_pele_escura:',
        'it': ':dita_incrociate_carnagione_scura:'
    },
    u'\U0001F91E\U0001F3FB': { # 🤞🏻
        'en' : ':crossed_fingers_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern_helle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_claro:',
        'fr': u':doigts_croisés_peau_claire:',
        'pt': ':dedos_cruzados_pele_clara:',
        'it': ':dita_incrociate_carnagione_chiara:'
    },
    u'\U0001F91E\U0001F3FE': { # 🤞🏾
        'en' : ':crossed_fingers_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern_mitteldunkle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_oscuro_medio:',
        'fr': u':doigts_croisés_peau_mate:',
        'pt': ':dedos_cruzados_pele_morena_escura:',
        'it': ':dita_incrociate_carnagione_abbastanza_scura:'
    },
    u'\U0001F91E\U0001F3FC': { # 🤞🏼
        'en' : ':crossed_fingers_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern_mittelhelle_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_claro_medio:',
        'fr': u':doigts_croisés_peau_moyennement_claire:',
        'pt': ':dedos_cruzados_pele_morena_clara:',
        'it': ':dita_incrociate_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91E\U0001F3FD': { # 🤞🏽
        'en' : ':crossed_fingers_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hand_mit_gekreuzten_fingern_mittlere_hautfarbe:',
        'es': ':dedos_cruzados_tono_de_piel_medio:',
        'fr': u':doigts_croisés_peau_légèrement_mate:',
        'pt': ':dedos_cruzados_pele_morena:',
        'it': ':dita_incrociate_carnagione_olivastra:'
    },
    u'\U0001F38C': { # 🎌
        'en' : ':crossed_flags:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':überkreuzte_flaggen:',
        'es': ':banderas_cruzadas:',
        'fr': u':drapeaux_croisés:',
        'pt': ':bandeiras_cruzadas:',
        'it': ':bandiere_del_giappone_incrociate:'
    },
    u'\U00002694\U0000FE0F': { # ⚔️
        'en' : ':crossed_swords:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':gekreuzte_schwerter:',
        'es': ':espadas_cruzadas:',
        'fr': u':épées_croisées:',
        'pt': ':espadas_cruzadas:',
        'it': ':spade_incrociate:'
    },
    u'\U00002694': { # ⚔
        'en' : ':crossed_swords:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':gekreuzte_schwerter:',
        'es': ':espadas_cruzadas:',
        'fr': u':épées_croisées:',
        'pt': ':espadas_cruzadas:',
        'it': ':spade_incrociate:'
    },
    u'\U0001F451': { # 👑
        'en' : ':crown:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':krone:',
        'es': ':corona:',
        'fr': ':couronne:',
        'pt': ':coroa:',
        'it': ':corona:'
    },
    u'\U0001FA7C': { # 🩼
        'en' : ':crutch:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':krücke:',
        'es': ':muleta:',
        'fr': u':béquille:',
        'pt': ':muleta:',
        'it': ':stampella:'
    },
    u'\U0001F63F': { # 😿
        'en' : ':crying_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':crying_cat_face:'],
        'de': ':weinende_katze:',
        'es': ':gato_llorando:',
        'fr': ':chat_qui_pleure:',
        'pt': ':rosto_de_gato_chorando:',
        'it': ':gatto_che_piange:'
    },
    u'\U0001F622': { # 😢
        'en' : ':crying_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cry:'],
        'de': ':weinendes_gesicht:',
        'es': ':cara_llorando:',
        'fr': ':visage_qui_pleure:',
        'pt': ':rosto_chorando:',
        'it': ':faccina_che_piange:'
    },
    u'\U0001F52E': { # 🔮
        'en' : ':crystal_ball:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kristallkugel:',
        'es': ':bola_de_cristal:',
        'fr': ':boule_de_cristal:',
        'pt': ':bola_de_cristal:',
        'it': ':sfera_di_cristallo:'
    },
    u'\U0001F952': { # 🥒
        'en' : ':cucumber:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':gurke:',
        'es': ':pepino:',
        'fr': ':concombre:',
        'pt': ':pepino:',
        'it': ':cetriolo:'
    },
    u'\U0001F964': { # 🥤
        'en' : ':cup_with_straw:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':becher_mit_strohhalm:',
        'es': ':vaso_con_pajita:',
        'fr': ':gobelet_avec_paille:',
        'pt': ':copo_com_canudo:',
        'it': ':bicchiere_con_cannuccia:'
    },
    u'\U0001F9C1': { # 🧁
        'en' : ':cupcake:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':cupcake:',
        'es': ':magdalena:',
        'fr': ':cupcake:',
        'pt': ':cupcake:',
        'it': ':cupcake:'
    },
    u'\U0001F94C': { # 🥌
        'en' : ':curling_stone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':curlingstein:',
        'es': ':piedra_de_curling:',
        'fr': ':pierre_de_curling:',
        'pt': ':pedra_de_curling:',
        'it': ':stone_da_curling:'
    },
    u'\U0001F9B1': { # 🦱
        'en' : ':curly_hair:',
        'status' : component,
        'E' : 11,
        'de': ':lockiges_haar:',
        'es': ':pelo_rizado:',
        'fr': u':cheveux_bouclés:',
        'pt': ':cabelo_cacheado:',
        'it': ':capelli_ricci:'
    },
    u'\U000027B0': { # ➰
        'en' : ':curly_loop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':schleife:',
        'es': ':bucle:',
        'fr': ':boucle:',
        'pt': u':laço_encaracolado:',
        'it': ':occhiello:'
    },
    u'\U0001F4B1': { # 💱
        'en' : ':currency_exchange:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':geldwechsel:',
        'es': ':cambio_de_divisas:',
        'fr': ':conversion_de_devise:',
        'pt': u':câmbio_de_moeda:',
        'it': ':cambio_valuta:'
    },
    u'\U0001F35B': { # 🍛
        'en' : ':curry_rice:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':curry:'],
        'de': ':reis_mit_curry:',
        'es': ':arroz_con_curry:',
        'fr': ':riz_au_curry:',
        'pt': ':arroz_com_curry:',
        'it': ':riso_al_curry:'
    },
    u'\U0001F36E': { # 🍮
        'en' : ':custard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':pudding:',
        'es': ':flan:',
        'fr': u':crème_renversée:',
        'pt': ':pudim:',
        'it': ':budino:'
    },
    u'\U0001F6C3': { # 🛃
        'en' : ':customs:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':zollkontrolle:',
        'es': ':aduana:',
        'fr': ':douane:',
        'pt': u':alfândega:',
        'it': ':simbolo_della_dogana:'
    },
    u'\U0001F969': { # 🥩
        'en' : ':cut_of_meat:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':fleischstück:',
        'es': ':corte_de_carne:',
        'fr': ':morceau_de_viande:',
        'pt': ':corte_de_carne:',
        'it': ':taglio_di_carne:'
    },
    u'\U0001F300': { # 🌀
        'en' : ':cyclone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':wirbel:',
        'es': u':ciclón:',
        'fr': ':cyclone:',
        'pt': ':ciclone:',
        'it': ':ciclone:'
    },
    u'\U0001F5E1\U0000FE0F': { # 🗡️
        'en' : ':dagger:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':dagger_knife:'],
        'variant': True,
        'de': ':dolch:',
        'es': u':puñal:',
        'fr': ':dague:',
        'pt': ':adaga:',
        'it': ':pugnale:'
    },
    u'\U0001F5E1': { # 🗡
        'en' : ':dagger:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':dagger_knife:'],
        'variant': True,
        'de': ':dolch:',
        'es': u':puñal:',
        'fr': ':dague:',
        'pt': ':adaga:',
        'it': ':pugnale:'
    },
    u'\U0001F361': { # 🍡
        'en' : ':dango:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':dango:',
        'es': ':dango:',
        'fr': ':brochette_de_bonbons:',
        'pt': ':dango:',
        'it': ':dango:'
    },
    u'\U0001F3FF': { # 🏿
        'en' : ':dark_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [':emoji_modifier_fitzpatrick_type__6:'],
        'de': ':dunkle_hautfarbe:',
        'es': ':tono_de_piel_oscuro:',
        'fr': u':peau_foncée:',
        'pt': ':pele_escura:',
        'it': ':carnagione_scura:'
    },
    u'\U0001F4A8': { # 💨
        'en' : ':dashing_away:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dash:'],
        'de': ':staubwolke:',
        'es': ':salir_corriendo:',
        'fr': u':décamper:',
        'pt': ':rapidez:',
        'it': ':nuvola_di_polvere:'
    },
    u'\U0001F9CF\U0000200D\U00002642\U0000FE0F': { # 🧏‍♂️
        'en' : ':deaf_man:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann:',
        'es': ':hombre_sordo:',
        'fr': ':homme_sourd:',
        'pt': ':homem_surdo:',
        'it': ':uomo_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0000200D\U00002642': { # 🧏‍♂
        'en' : ':deaf_man:',
        'status' : minimally_qualified,
        'E' : 12,
        'de': u':gehörloser_mann:',
        'es': ':hombre_sordo:',
        'fr': ':homme_sourd:',
        'pt': ':homem_surdo:',
        'it': ':uomo_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧏🏿‍♂️
        'en' : ':deaf_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_dunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro:',
        'fr': u':homme_sourd_peau_foncée:',
        'pt': ':homem_surdo_pele_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002642': { # 🧏🏿‍♂
        'en' : ':deaf_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧏🏻‍♂️
        'en' : ':deaf_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_helle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro:',
        'fr': ':homme_sourd_peau_claire:',
        'pt': ':homem_surdo_pele_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002642': { # 🧏🏻‍♂
        'en' : ':deaf_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧏🏾‍♂️
        'en' : ':deaf_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_sourd_peau_mate:',
        'pt': ':homem_surdo_pele_morena_escura:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002642': { # 🧏🏾‍♂
        'en' : ':deaf_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧏🏼‍♂️
        'en' : ':deaf_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_claro_medio:',
        'fr': ':homme_sourd_peau_moyennement_claire:',
        'pt': ':homem_surdo_pele_morena_clara:',
        'it': ':uomo_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002642': { # 🧏🏼‍♂
        'en' : ':deaf_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧏🏽‍♂️
        'en' : ':deaf_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörloser_mann_mittlere_hautfarbe:',
        'es': ':hombre_sordo_tono_de_piel_medio:',
        'fr': u':homme_sourd_peau_légèrement_mate:',
        'pt': ':homem_surdo_pele_morena:',
        'it': ':uomo_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002642': { # 🧏🏽‍♂
        'en' : ':deaf_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF': { # 🧏
        'en' : ':deaf_person:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person:',
        'es': ':persona_sorda:',
        'fr': ':personne_sourde:',
        'pt': ':pessoa_surda:',
        'it': ':persona_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0001F3FF': { # 🧏🏿
        'en' : ':deaf_person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_dunkle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_oscuro:',
        'fr': u':personne_sourde_peau_foncée:',
        'pt': ':pessoa_surda_pele_escura:',
        'it': ':persona_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FB': { # 🧏🏻
        'en' : ':deaf_person_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_helle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_claro:',
        'fr': ':personne_sourde_peau_claire:',
        'pt': ':pessoa_surda_pele_clara:',
        'it': ':persona_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FE': { # 🧏🏾
        'en' : ':deaf_person_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mitteldunkle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_oscuro_medio:',
        'fr': ':personne_sourde_peau_mate:',
        'pt': ':pessoa_surda_pele_morena_escura:',
        'it': ':persona_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FC': { # 🧏🏼
        'en' : ':deaf_person_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mittelhelle_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_claro_medio:',
        'fr': ':personne_sourde_peau_moyennement_claire:',
        'pt': ':pessoa_surda_pele_morena_clara:',
        'it': ':persona_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FD': { # 🧏🏽
        'en' : ':deaf_person_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_person_mittlere_hautfarbe:',
        'es': ':persona_sorda_tono_de_piel_medio:',
        'fr': u':personne_sourde_peau_légèrement_mate:',
        'pt': ':pessoa_surda_pele_morena:',
        'it': ':persona_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0000200D\U00002640\U0000FE0F': { # 🧏‍♀️
        'en' : ':deaf_woman:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau:',
        'es': ':mujer_sorda:',
        'fr': ':femme_sourde:',
        'pt': ':mulher_surda:',
        'it': ':donna_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0000200D\U00002640': { # 🧏‍♀
        'en' : ':deaf_woman:',
        'status' : minimally_qualified,
        'E' : 12,
        'de': u':gehörlose_frau:',
        'es': ':mujer_sorda:',
        'fr': ':femme_sourde:',
        'pt': ':mulher_surda:',
        'it': ':donna_con_problemi_di_udito:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002640\U0000FE0F': { # 🧏🏿‍♀️
        'en' : ':deaf_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_dunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro:',
        'fr': u':femme_sourde_peau_foncée:',
        'pt': ':mulher_surda_pele_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_scura:'
    },
    u'\U0001F9CF\U0001F3FF\U0000200D\U00002640': { # 🧏🏿‍♀
        'en' : ':deaf_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002640\U0000FE0F': { # 🧏🏻‍♀️
        'en' : ':deaf_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_helle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro:',
        'fr': ':femme_sourde_peau_claire:',
        'pt': ':mulher_surda_pele_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_chiara:'
    },
    u'\U0001F9CF\U0001F3FB\U0000200D\U00002640': { # 🧏🏻‍♀
        'en' : ':deaf_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002640\U0000FE0F': { # 🧏🏾‍♀️
        'en' : ':deaf_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mitteldunkle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_oscuro_medio:',
        'fr': ':femme_sourde_peau_mate:',
        'pt': ':mulher_surda_pele_morena_escura:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CF\U0001F3FE\U0000200D\U00002640': { # 🧏🏾‍♀
        'en' : ':deaf_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002640\U0000FE0F': { # 🧏🏼‍♀️
        'en' : ':deaf_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mittelhelle_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_claro_medio:',
        'fr': ':femme_sourde_peau_moyennement_claire:',
        'pt': ':mulher_surda_pele_morena_clara:',
        'it': ':donna_con_problemi_di_udito_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CF\U0001F3FC\U0000200D\U00002640': { # 🧏🏼‍♀
        'en' : ':deaf_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002640\U0000FE0F': { # 🧏🏽‍♀️
        'en' : ':deaf_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':gehörlose_frau_mittlere_hautfarbe:',
        'es': ':mujer_sorda_tono_de_piel_medio:',
        'fr': u':femme_sourde_peau_légèrement_mate:',
        'pt': ':mulher_surda_pele_morena:',
        'it': ':donna_con_problemi_di_udito_carnagione_olivastra:'
    },
    u'\U0001F9CF\U0001F3FD\U0000200D\U00002640': { # 🧏🏽‍♀
        'en' : ':deaf_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F333': { # 🌳
        'en' : ':deciduous_tree:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':laubbaum:',
        'es': u':árbol_de_hoja_caduca:',
        'fr': u':arbre_à_feuilles_caduques:',
        'pt': u':árvore_caidiça:',
        'it': ':albero_deciduo:'
    },
    u'\U0001F98C': { # 🦌
        'en' : ':deer:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':hirsch:',
        'es': ':ciervo:',
        'fr': ':cerf:',
        'pt': ':cervo:',
        'it': ':cervo:'
    },
    u'\U0001F69A': { # 🚚
        'en' : ':delivery_truck:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':truck:'],
        'de': ':lieferwagen:',
        'es': u':camión_de_reparto:',
        'fr': ':camion_de_livraison:',
        'pt': u':caminhão_de_entrega:',
        'it': ':camion:'
    },
    u'\U0001F3EC': { # 🏬
        'en' : ':department_store:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kaufhaus:',
        'es': ':grandes_almacenes:',
        'fr': ':grand_magasin:',
        'pt': ':loja_de_departamentos:',
        'it': ':grande_magazzino:'
    },
    u'\U0001F3DA\U0000FE0F': { # 🏚️
        'en' : ':derelict_house:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':derelict_house_building:'],
        'variant': True,
        'de': ':verfallenes_haus:',
        'es': ':casa_abandonada:',
        'fr': u':maison_abandonnée:',
        'pt': ':casa_abandonada:',
        'it': ':casa_in_rovina:'
    },
    u'\U0001F3DA': { # 🏚
        'en' : ':derelict_house:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':derelict_house_building:'],
        'variant': True,
        'de': ':verfallenes_haus:',
        'es': ':casa_abandonada:',
        'fr': u':maison_abandonnée:',
        'pt': ':casa_abandonada:',
        'it': ':casa_in_rovina:'
    },
    u'\U0001F3DC\U0000FE0F': { # 🏜️
        'en' : ':desert:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wüste:',
        'es': ':desierto:',
        'fr': u':désert:',
        'pt': ':deserto:',
        'it': ':deserto:'
    },
    u'\U0001F3DC': { # 🏜
        'en' : ':desert:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':wüste:',
        'es': ':desierto:',
        'fr': u':désert:',
        'pt': ':deserto:',
        'it': ':deserto:'
    },
    u'\U0001F3DD\U0000FE0F': { # 🏝️
        'en' : ':desert_island:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':einsame_insel:',
        'es': ':isla_desierta:',
        'fr': u':île_déserte:',
        'pt': ':ilha_deserta:',
        'it': ':isola_deserta:'
    },
    u'\U0001F3DD': { # 🏝
        'en' : ':desert_island:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':einsame_insel:',
        'es': ':isla_desierta:',
        'fr': u':île_déserte:',
        'pt': ':ilha_deserta:',
        'it': ':isola_deserta:'
    },
    u'\U0001F5A5\U0000FE0F': { # 🖥️
        'en' : ':desktop_computer:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':desktopcomputer:',
        'es': ':ordenador_de_sobremesa:',
        'fr': ':ordinateur_de_bureau:',
        'pt': ':computador_de_mesa:',
        'it': ':computer_fisso:'
    },
    u'\U0001F5A5': { # 🖥
        'en' : ':desktop_computer:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':desktopcomputer:',
        'es': ':ordenador_de_sobremesa:',
        'fr': ':ordinateur_de_bureau:',
        'pt': ':computador_de_mesa:',
        'it': ':computer_fisso:'
    },
    u'\U0001F575\U0000FE0F': { # 🕵️
        'en' : ':detective:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':sleuth_or_spy:'],
        'variant': True,
        'de': ':detektiv(in):',
        'es': ':detective:',
        'fr': u':détective:',
        'pt': ':detetive:',
        'it': ':detective:'
    },
    u'\U0001F575': { # 🕵
        'en' : ':detective:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':sleuth_or_spy:'],
        'variant': True,
        'de': ':detektiv(in):',
        'es': ':detective:',
        'fr': u':détective:',
        'pt': ':detetive:',
        'it': ':detective:'
    },
    u'\U0001F575\U0001F3FF': { # 🕵🏿
        'en' : ':detective_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':detektiv(in)_dunkle_hautfarbe:',
        'es': ':detective_tono_de_piel_oscuro:',
        'fr': u':détective_peau_foncée:',
        'pt': ':detetive_pele_escura:',
        'it': ':detective_carnagione_scura:'
    },
    u'\U0001F575\U0001F3FB': { # 🕵🏻
        'en' : ':detective_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':detektiv(in)_helle_hautfarbe:',
        'es': ':detective_tono_de_piel_claro:',
        'fr': u':détective_peau_claire:',
        'pt': ':detetive_pele_clara:',
        'it': ':detective_carnagione_chiara:'
    },
    u'\U0001F575\U0001F3FE': { # 🕵🏾
        'en' : ':detective_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':detektiv(in)_mitteldunkle_hautfarbe:',
        'es': ':detective_tono_de_piel_oscuro_medio:',
        'fr': u':détective_peau_mate:',
        'pt': ':detetive_pele_morena_escura:',
        'it': ':detective_carnagione_abbastanza_scura:'
    },
    u'\U0001F575\U0001F3FC': { # 🕵🏼
        'en' : ':detective_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':detektiv(in)_mittelhelle_hautfarbe:',
        'es': ':detective_tono_de_piel_claro_medio:',
        'fr': u':détective_peau_moyennement_claire:',
        'pt': ':detetive_pele_morena_clara:',
        'it': ':detective_carnagione_abbastanza_chiara:'
    },
    u'\U0001F575\U0001F3FD': { # 🕵🏽
        'en' : ':detective_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':detektiv(in)_mittlere_hautfarbe:',
        'es': ':detective_tono_de_piel_medio:',
        'fr': u':détective_peau_légèrement_mate:',
        'pt': ':detetive_pele_morena:',
        'it': ':detective_carnagione_olivastra:'
    },
    u'\U00002666\U0000FE0F': { # ♦️
        'en' : ':diamond_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':diamonds:'],
        'variant': True,
        'de': ':karo:',
        'es': ':palo_de_diamantes:',
        'fr': ':carreau:',
        'pt': ':naipe_de_ouros:',
        'it': ':quadri:'
    },
    u'\U00002666': { # ♦
        'en' : ':diamond_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':diamonds:'],
        'variant': True,
        'de': ':karo:',
        'es': ':palo_de_diamantes:',
        'fr': ':carreau:',
        'pt': ':naipe_de_ouros:',
        'it': ':quadri:'
    },
    u'\U0001F4A0': { # 💠
        'en' : ':diamond_with_a_dot:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':diamond_shape_with_a_dot_inside:'],
        'de': ':rautenform_mit_punkt:',
        'es': u':rombo_con_pétalo:',
        'fr': ':diamant_avec_un_point:',
        'pt': ':diamante_com_um_ponto:',
        'it': ':petalo_di_fiore:'
    },
    u'\U0001F505': { # 🔅
        'en' : ':dim_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':low_brightness:'],
        'de': ':taste_dimmen:',
        'es': ':brillo_bajo:',
        'fr': u':luminosité_faible:',
        'pt': u':botão_de_diminuir_brilho:',
        'it': u':luminosità_bassa:'
    },
    u'\U0001F61E': { # 😞
        'en' : ':disappointed_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':disappointed:'],
        'de': u':enttäuschtes_gesicht:',
        'es': ':cara_decepcionada:',
        'fr': u':visage_déçu:',
        'pt': ':rosto_desapontado:',
        'it': ':faccina_delusa:'
    },
    u'\U0001F978': { # 🥸
        'en' : ':disguised_face:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':verkleidet:',
        'es': ':cara_disfrazada:',
        'fr': u':visage_déguisé:',
        'pt': u':rosto_disfarçado:',
        'it': ':faccina_travestita:'
    },
    u'\U00002797': { # ➗
        'en' : ':divide:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heavy_division_sign:'],
        'de': ':geteilt_durch:',
        'es': u':división:',
        'fr': ':signe_diviser:',
        'pt': u':símbolo_de_divisão:',
        'it': ':divisione:'
    },
    u'\U0001F93F': { # 🤿
        'en' : ':diving_mask:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':tauchmaske:',
        'es': u':máscara_de_buceo:',
        'fr': u':masque_de_plongée:',
        'pt': u':máscara_de_mergulho:',
        'it': ':maschera_da_sub:'
    },
    u'\U0001FA94': { # 🪔
        'en' : ':diya_lamp:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':öllampe:',
        'es': u':lámpara_de_aceite:',
        'fr': ':diya:',
        'pt': u':lâmpada_de_óleo:',
        'it': ':diya:'
    },
    u'\U0001F4AB': { # 💫
        'en' : ':dizzy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':schwindlig:',
        'es': u':símbolo_de_mareo:',
        'fr': u':étourdissement:',
        'pt': ':zonzo:',
        'it': ':stella_con_scia:'
    },
    u'\U0001F9EC': { # 🧬
        'en' : ':dna:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':dna:',
        'es': ':adn:',
        'fr': ':adn:',
        'pt': ':dna:',
        'it': ':dna:'
    },
    u'\U0001F9A4': { # 🦤
        'en' : ':dodo:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':dodo:',
        'es': ':dodo:',
        'fr': ':dodo:',
        'pt': u':dodô:',
        'it': ':dodo:'
    },
    u'\U0001F415': { # 🐕
        'en' : ':dog:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':dog2:'],
        'variant': True,
        'de': ':hund:',
        'es': ':perro:',
        'fr': ':chien:',
        'pt': ':cachorro:',
        'it': ':cane:'
    },
    u'\U0001F436': { # 🐶
        'en' : ':dog_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dog:'],
        'de': ':hundegesicht:',
        'es': ':cara_de_perro:',
        'fr': u':tête_de_chien:',
        'pt': ':rosto_de_cachorro:',
        'it': ':muso_di_cane:'
    },
    u'\U0001F4B5': { # 💵
        'en' : ':dollar_banknote:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dollar:'],
        'de': ':dollar-banknote:',
        'es': u':billete_de_dólar:',
        'fr': ':billet_en_dollars:',
        'pt': u':nota_de_dólar:',
        'it': ':banconota_dollaro:'
    },
    u'\U0001F42C': { # 🐬
        'en' : ':dolphin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flipper:'],
        'de': ':delfin:',
        'es': u':delfín:',
        'fr': ':dauphin:',
        'pt': ':golfinho:',
        'it': ':delfino:'
    },
    u'\U0001F6AA': { # 🚪
        'en' : ':door:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':tür:',
        'es': ':puerta:',
        'fr': ':porte:',
        'pt': ':porta:',
        'it': ':porta:'
    },
    u'\U0001FAE5': { # 🫥
        'en' : ':dotted_line_face:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':gesicht_mit_gestrichelter_linie:',
        'es': u':cara_con_línea_de_puntos:',
        'fr': u':visage_en_pointillés:',
        'pt': ':rosto_com_linha_pontilhada:',
        'it': ':faccina_tratteggiata:'
    },
    u'\U0001F52F': { # 🔯
        'en' : ':dotted_six-pointed_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':six_pointed_star:'],
        'de': ':hexagramm_mit_punkt:',
        'es': ':estrella_de_seis_puntas:',
        'fr': u':étoile_à_6_branches:',
        'pt': ':estrela_de_seis_pontas:',
        'it': ':stella_a_sei_punte:'
    },
    u'\U000027BF': { # ➿
        'en' : ':double_curly_loop:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':loop:'],
        'de': ':doppelschleife:',
        'es': ':bucle_doble:',
        'fr': ':double_boucle:',
        'pt': ':loop_encaracolado_duas_vezes:',
        'it': ':doppio_occhiello:'
    },
    u'\U0000203C\U0000FE0F': { # ‼️
        'en' : ':double_exclamation_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bangbang:'],
        'variant': True,
        'de': ':doppeltes_ausrufezeichen:',
        'es': u':exclamación_doble:',
        'fr': u':double_point_d’exclamation:',
        'pt': u':dupla_exclamação:',
        'it': ':doppio_punto_esclamativo:'
    },
    u'\U0000203C': { # ‼
        'en' : ':double_exclamation_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':bangbang:'],
        'variant': True,
        'de': ':doppeltes_ausrufezeichen:',
        'es': u':exclamación_doble:',
        'fr': u':double_point_d’exclamation:',
        'pt': u':dupla_exclamação:',
        'it': ':doppio_punto_esclamativo:'
    },
    u'\U0001F369': { # 🍩
        'en' : ':doughnut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':donut:',
        'es': u':dónut:',
        'fr': ':doughnut:',
        'pt': ':donut:',
        'it': ':ciambella:'
    },
    u'\U0001F54A\U0000FE0F': { # 🕊️
        'en' : ':dove:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':dove_of_peace:'],
        'variant': True,
        'de': ':taube:',
        'es': ':paloma:',
        'fr': ':colombe:',
        'pt': ':pomba_branca:',
        'it': ':colomba:'
    },
    u'\U0001F54A': { # 🕊
        'en' : ':dove:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':dove_of_peace:'],
        'variant': True,
        'de': ':taube:',
        'es': ':paloma:',
        'fr': ':colombe:',
        'pt': ':pomba_branca:',
        'it': ':colomba:'
    },
    u'\U00002199\U0000FE0F': { # ↙️
        'en' : ':down-left_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_lower_left:'],
        'variant': True,
        'de': ':pfeil_nach_links_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': u':flèche_bas_gauche:',
        'pt': ':seta_para_baixo_e_para_a_esquerda:',
        'it': ':freccia_in_basso_a_sinistra:'
    },
    u'\U00002199': { # ↙
        'en' : ':down-left_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':arrow_lower_left:'],
        'variant': True,
        'de': ':pfeil_nach_links_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_izquierda:',
        'fr': u':flèche_bas_gauche:',
        'pt': ':seta_para_baixo_e_para_a_esquerda:',
        'it': ':freccia_in_basso_a_sinistra:'
    },
    u'\U00002198\U0000FE0F': { # ↘️
        'en' : ':down-right_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_lower_right:'],
        'variant': True,
        'de': ':pfeil_nach_rechts_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': u':flèche_bas_droite:',
        'pt': ':seta_para_baixo_e_para_a_direita:',
        'it': ':freccia_in_basso_a_destra:'
    },
    u'\U00002198': { # ↘
        'en' : ':down-right_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':arrow_lower_right:'],
        'variant': True,
        'de': ':pfeil_nach_rechts_unten:',
        'es': ':flecha_hacia_la_esquina_inferior_derecha:',
        'fr': u':flèche_bas_droite:',
        'pt': ':seta_para_baixo_e_para_a_direita:',
        'it': ':freccia_in_basso_a_destra:'
    },
    u'\U00002B07\U0000FE0F': { # ⬇️
        'en' : ':down_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_down:'],
        'variant': True,
        'de': ':pfeil_nach_unten:',
        'es': ':flecha_hacia_abajo:',
        'fr': u':flèche_bas:',
        'pt': ':seta_para_baixo:',
        'it': ':freccia_rivolta_verso_il_basso:'
    },
    u'\U00002B07': { # ⬇
        'en' : ':down_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':arrow_down:'],
        'variant': True,
        'de': ':pfeil_nach_unten:',
        'es': ':flecha_hacia_abajo:',
        'fr': u':flèche_bas:',
        'pt': ':seta_para_baixo:',
        'it': ':freccia_rivolta_verso_il_basso:'
    },
    u'\U0001F613': { # 😓
        'en' : ':downcast_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sweat:'],
        'de': u':bedrücktes_gesicht_mit_schweiß:',
        'es': u':cara_con_sudor_frío:',
        'fr': u':visage_démoralisé_avec_goutte_de_sueur:',
        'pt': ':rosto_cabisbaixo_com_gota_de_suor:',
        'it': ':faccina_abbattuta_sudata:'
    },
    u'\U0001F53D': { # 🔽
        'en' : ':downwards_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_down_small:'],
        'de': u':abwärts-schaltfläche:',
        'es': u':triángulo_hacia_abajo:',
        'fr': ':petit_triangle_bas:',
        'pt': u':botão_apontando_para_baixo:',
        'it': ':pulsante_a_triangolo_rivolto_verso_il_basso:'
    },
    u'\U0001F409': { # 🐉
        'en' : ':dragon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':drache:',
        'es': u':dragón:',
        'fr': ':dragon:',
        'pt': u':dragão:',
        'it': ':drago:'
    },
    u'\U0001F432': { # 🐲
        'en' : ':dragon_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':drachengesicht:',
        'es': u':cara_de_dragón:',
        'fr': u':tête_de_dragon:',
        'pt': u':rosto_de_dragão:',
        'it': ':testa_di_drago:'
    },
    u'\U0001F457': { # 👗
        'en' : ':dress:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kleid:',
        'es': ':vestido:',
        'fr': ':robe:',
        'pt': ':vestido:',
        'it': ':vestito:'
    },
    u'\U0001F924': { # 🤤
        'en' : ':drooling_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sabberndes_gesicht:',
        'es': ':cara_babeando:',
        'fr': ':visage_qui_bave:',
        'pt': ':rosto_babando:',
        'it': ':faccina_che_sbava:'
    },
    u'\U0001FA78': { # 🩸
        'en' : ':drop_of_blood:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':blutstropfen:',
        'es': ':gota_de_sangre:',
        'fr': ':goutte_de_sang:',
        'pt': ':gota_de_sangue:',
        'it': ':goccia_di_sangue:'
    },
    u'\U0001F4A7': { # 💧
        'en' : ':droplet:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':tropfen:',
        'es': ':gota:',
        'fr': u':goutte_d’eau:',
        'pt': ':gota:',
        'it': ':goccia:'
    },
    u'\U0001F941': { # 🥁
        'en' : ':drum:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':trommel:',
        'es': ':tambor:',
        'fr': ':batterie:',
        'pt': ':tambor:',
        'it': ':tamburo:'
    },
    u'\U0001F986': { # 🦆
        'en' : ':duck:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':ente:',
        'es': ':pato:',
        'fr': ':canard:',
        'pt': ':pato:',
        'it': ':anatra:'
    },
    u'\U0001F95F': { # 🥟
        'en' : ':dumpling:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':teigtasche:',
        'es': ':dumpling:',
        'fr': u':boulette_de_pâte:',
        'pt': u':bolinho_asiático:',
        'it': ':raviolo:'
    },
    u'\U0001F4C0': { # 📀
        'en' : ':dvd:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':dvd:',
        'es': ':disco_dvd:',
        'fr': ':dvd:',
        'pt': ':dvd:',
        'it': ':dvd:'
    },
    u'\U0001F4E7': { # 📧
        'en' : ':e-mail:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':email:', ':e__mail:'],
        'de': ':e-mail:',
        'es': u':correo_electrónico:',
        'fr': ':e-mail:',
        'pt': ':e-mail:',
        'it': ':e-mail:'
    },
    u'\U0001F985': { # 🦅
        'en' : ':eagle:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':adler:',
        'es': u':águila:',
        'fr': ':aigle:',
        'pt': u':águia:',
        'it': ':aquila:'
    },
    u'\U0001F442': { # 👂
        'en' : ':ear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':ohr:',
        'es': ':oreja:',
        'fr': ':oreille:',
        'pt': ':orelha:',
        'it': ':orecchio:'
    },
    u'\U0001F442\U0001F3FF': { # 👂🏿
        'en' : ':ear_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ohr_dunkle_hautfarbe:',
        'es': ':oreja_tono_de_piel_oscuro:',
        'fr': u':oreille_peau_foncée:',
        'pt': ':orelha_pele_escura:',
        'it': ':orecchio_carnagione_scura:'
    },
    u'\U0001F442\U0001F3FB': { # 👂🏻
        'en' : ':ear_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ohr_helle_hautfarbe:',
        'es': ':oreja_tono_de_piel_claro:',
        'fr': ':oreille_peau_claire:',
        'pt': ':orelha_pele_clara:',
        'it': ':orecchio_carnagione_chiara:'
    },
    u'\U0001F442\U0001F3FE': { # 👂🏾
        'en' : ':ear_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ohr_mitteldunkle_hautfarbe:',
        'es': ':oreja_tono_de_piel_oscuro_medio:',
        'fr': ':oreille_peau_mate:',
        'pt': ':orelha_pele_morena_escura:',
        'it': ':orecchio_carnagione_abbastanza_scura:'
    },
    u'\U0001F442\U0001F3FC': { # 👂🏼
        'en' : ':ear_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ohr_mittelhelle_hautfarbe:',
        'es': ':oreja_tono_de_piel_claro_medio:',
        'fr': ':oreille_peau_moyennement_claire:',
        'pt': ':orelha_pele_morena_clara:',
        'it': ':orecchio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F442\U0001F3FD': { # 👂🏽
        'en' : ':ear_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ohr_mittlere_hautfarbe:',
        'es': ':oreja_tono_de_piel_medio:',
        'fr': u':oreille_peau_légèrement_mate:',
        'pt': ':orelha_pele_morena:',
        'it': ':orecchio_carnagione_olivastra:'
    },
    u'\U0001F33D': { # 🌽
        'en' : ':ear_of_corn:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':corn:'],
        'de': ':maiskolben:',
        'es': u':espiga_de_maíz:',
        'fr': u':épi_de_maïs:',
        'pt': ':milho:',
        'it': ':pannocchia:'
    },
    u'\U0001F9BB': { # 🦻
        'en' : ':ear_with_hearing_aid:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörgerät:',
        'es': u':oreja_con_audífono:',
        'fr': u':oreille_appareillée:',
        'pt': ':ouvido_com_aparelho_auditivo:',
        'it': ':orecchio_con_apparecchio_acustico:'
    },
    u'\U0001F9BB\U0001F3FF': { # 🦻🏿
        'en' : ':ear_with_hearing_aid_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_dunkle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_oscuro:',
        'fr': u':oreille_appareillée_peau_foncée:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_escura:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_scura:'
    },
    u'\U0001F9BB\U0001F3FB': { # 🦻🏻
        'en' : ':ear_with_hearing_aid_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_helle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_claro:',
        'fr': u':oreille_appareillée_peau_claire:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_clara:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_chiara:'
    },
    u'\U0001F9BB\U0001F3FE': { # 🦻🏾
        'en' : ':ear_with_hearing_aid_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mitteldunkle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_oscuro_medio:',
        'fr': u':oreille_appareillée_peau_mate:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena_escura:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_abbastanza_scura:'
    },
    u'\U0001F9BB\U0001F3FC': { # 🦻🏼
        'en' : ':ear_with_hearing_aid_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mittelhelle_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_claro_medio:',
        'fr': u':oreille_appareillée_peau_moyennement_claire:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena_clara:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9BB\U0001F3FD': { # 🦻🏽
        'en' : ':ear_with_hearing_aid_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':ohr_mit_hörhilfe_mittlere_hautfarbe:',
        'es': u':oreja_con_audífono_tono_de_piel_medio:',
        'fr': u':oreille_appareillée_peau_légèrement_mate:',
        'pt': ':ouvido_com_aparelho_auditivo_pele_morena:',
        'it': ':orecchio_con_apparecchio_acustico_carnagione_olivastra:'
    },
    u'\U0001F95A': { # 🥚
        'en' : ':egg:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':egg2:'],
        'de': ':ei:',
        'es': ':huevo:',
        'fr': u':œuf:',
        'pt': ':ovo:',
        'it': ':uovo:'
    },
    u'\U0001F346': { # 🍆
        'en' : ':eggplant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':aubergine:',
        'es': ':berenjena:',
        'fr': ':aubergine:',
        'pt': ':berinjela:',
        'it': ':melanzana:'
    },
    u'\U00002734\U0000FE0F': { # ✴️
        'en' : ':eight-pointed_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':eight_pointed_black_star:'],
        'variant': True,
        'de': ':stern_mit_acht_zacken:',
        'es': ':estrella_de_ocho_puntas:',
        'fr': u':étoile_huit_branches:',
        'pt': ':estrela_de_oito_pontas:',
        'it': ':stella_stilizzata:'
    },
    u'\U00002734': { # ✴
        'en' : ':eight-pointed_star:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':eight_pointed_black_star:'],
        'variant': True,
        'de': ':achtstrahliger_stern:',
        'es': ':estrella_de_ocho_puntas:',
        'fr': u':étoile_huit_branches:',
        'pt': ':estrela_de_oito_pontas:',
        'it': ':stella_stilizzata:'
    },
    u'\U00002733\U0000FE0F': { # ✳️
        'en' : ':eight-spoked_asterisk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':eight_spoked_asterisk:'],
        'variant': True,
        'de': ':achtzackiger_stern:',
        'es': ':asterisco_de_ocho_puntas:',
        'fr': u':astérisque_huit_branches:',
        'pt': ':asterisco_de_oito_pontas:',
        'it': ':asterisco:'
    },
    u'\U00002733': { # ✳
        'en' : ':eight-spoked_asterisk:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':eight_spoked_asterisk:'],
        'variant': True,
        'de': ':achtzackiger_stern:',
        'es': ':asterisco_de_ocho_puntas:',
        'fr': u':astérisque_huit_branches:',
        'pt': ':asterisco_de_oito_pontas:',
        'it': ':asterisco:'
    },
    u'\U0001F563': { # 🕣
        'en' : ':eight-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock830:'],
        'variant': True,
        'de': ':8.30_uhr:',
        'es': ':ocho_y_media:',
        'fr': ':huit_heures_et_demie:',
        'pt': ':oito_e_meia:',
        'it': ':ore_otto_e_mezza:'
    },
    u'\U0001F557': { # 🕗
        'en' : u':eight_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock8:'],
        'variant': True,
        'de': ':8.00_uhr:',
        'es': ':8_en_punto:',
        'fr': ':huit_heures:',
        'pt': ':8_horas:',
        'it': ':ore_otto:'
    },
    u'\U000023CF\U0000FE0F': { # ⏏️
        'en' : ':eject_button:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':eject_symbol:'],
        'variant': True,
        'de': ':auswerfen:',
        'es': ':expulsar:',
        'fr': u':bouton_éjecter:',
        'pt': u':botão_ejetar:',
        'it': ':pulsante_di_espulsione:'
    },
    u'\U000023CF': { # ⏏
        'en' : ':eject_button:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [':eject_symbol:'],
        'variant': True,
        'de': ':auswerfen:',
        'es': ':expulsar:',
        'fr': u':bouton_éjecter:',
        'pt': u':botão_ejetar:',
        'it': ':pulsante_di_espulsione:'
    },
    u'\U0001F50C': { # 🔌
        'en' : ':electric_plug:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':netzstecker:',
        'es': u':enchufe_eléctrico:',
        'fr': u':câble_avec_fiche_électrique:',
        'pt': u':tomada_elétrica:',
        'it': ':spina_elettrica:'
    },
    u'\U0001F418': { # 🐘
        'en' : ':elephant:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':elefant:',
        'es': ':elefante:',
        'fr': u':éléphant:',
        'pt': ':elefante:',
        'it': ':elefante:'
    },
    u'\U0001F6D7': { # 🛗
        'en' : ':elevator:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':fahrstuhl:',
        'es': ':ascensor:',
        'fr': ':ascenseur:',
        'pt': ':elevador:',
        'it': ':ascensore:'
    },
    u'\U0001F566': { # 🕦
        'en' : ':eleven-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock1130:'],
        'variant': True,
        'de': ':11.30_uhr:',
        'es': ':once_y_media:',
        'fr': ':onze_heures_et_demie:',
        'pt': ':onze_e_meia:',
        'it': ':ore_undici_e_mezza:'
    },
    u'\U0001F55A': { # 🕚
        'en' : u':eleven_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock11:'],
        'variant': True,
        'de': ':11.00_uhr:',
        'es': ':11_en_punto:',
        'fr': ':onze_heures:',
        'pt': ':11_horas:',
        'it': ':ore_undici:'
    },
    u'\U0001F9DD': { # 🧝
        'en' : ':elf:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e):',
        'es': ':elfo:',
        'fr': ':elfe:',
        'pt': ':elfo:',
        'it': ':elfo:'
    },
    u'\U0001F9DD\U0001F3FF': { # 🧝🏿
        'en' : ':elf_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e)_dunkle_hautfarbe:',
        'es': ':elfo_tono_de_piel_oscuro:',
        'fr': u':elfe_peau_foncée:',
        'pt': ':elfo_pele_escura:',
        'it': ':elfo_carnagione_scura:'
    },
    u'\U0001F9DD\U0001F3FB': { # 🧝🏻
        'en' : ':elf_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e)_helle_hautfarbe:',
        'es': ':elfo_tono_de_piel_claro:',
        'fr': ':elfe_peau_claire:',
        'pt': ':elfo_pele_clara:',
        'it': ':elfo_carnagione_chiara:'
    },
    u'\U0001F9DD\U0001F3FE': { # 🧝🏾
        'en' : ':elf_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e)_mitteldunkle_hautfarbe:',
        'es': ':elfo_tono_de_piel_oscuro_medio:',
        'fr': ':elfe_peau_mate:',
        'pt': ':elfo_pele_morena_escura:',
        'it': ':elfo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DD\U0001F3FC': { # 🧝🏼
        'en' : ':elf_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e)_mittelhelle_hautfarbe:',
        'es': ':elfo_tono_de_piel_claro_medio:',
        'fr': ':elfe_peau_moyennement_claire:',
        'pt': ':elfo_pele_morena_clara:',
        'it': ':elfo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DD\U0001F3FD': { # 🧝🏽
        'en' : ':elf_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf(e)_mittlere_hautfarbe:',
        'es': ':elfo_tono_de_piel_medio:',
        'fr': u':elfe_peau_légèrement_mate:',
        'pt': ':elfo_pele_morena:',
        'it': ':elfo_carnagione_olivastra:'
    },
    u'\U0001FAB9': { # 🪹
        'en' : ':empty_nest:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':leeres_nest:',
        'es': u':nido_vacío:',
        'fr': ':nid_vide:',
        'pt': ':ninho_vazio:',
        'it': ':nido_vuoto:'
    },
    u'\U00002709\U0000FE0F': { # ✉️
        'en' : ':envelope:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':briefumschlag:',
        'es': ':sobre:',
        'fr': ':enveloppe:',
        'pt': ':envelope:',
        'it': ':busta:'
    },
    u'\U00002709': { # ✉
        'en' : ':envelope:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':briefumschlag:',
        'es': ':sobre:',
        'fr': ':enveloppe:',
        'pt': ':envelope:',
        'it': ':busta:'
    },
    u'\U0001F4E9': { # 📩
        'en' : ':envelope_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':umschlag_mit_pfeil:',
        'es': ':sobre_con_flecha:',
        'fr': u':enveloppe_avec_flèche:',
        'pt': ':envelope_com_seta:',
        'it': ':posta_in_uscita:'
    },
    u'\U0001F4B6': { # 💶
        'en' : ':euro_banknote:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':euro:'],
        'de': ':euro-banknote:',
        'es': ':billete_de_euro:',
        'fr': ':billet_en_euros:',
        'pt': ':nota_de_euro:',
        'it': ':banconota_euro:'
    },
    u'\U0001F332': { # 🌲
        'en' : ':evergreen_tree:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nadelbaum:',
        'es': u':árbol_de_hoja_perenne:',
        'fr': u':conifère:',
        'pt': u':conífera:',
        'it': ':albero_sempreverde:'
    },
    u'\U0001F411': { # 🐑
        'en' : ':ewe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sheep:'],
        'de': ':schaf:',
        'es': ':oveja:',
        'fr': ':mouton:',
        'pt': ':ovelha:',
        'it': ':pecora:'
    },
    u'\U00002049\U0000FE0F': { # ⁉️
        'en' : ':exclamation_question_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':interrobang:'],
        'variant': True,
        'de': ':ausrufe-_und_fragezeichen:',
        'es': u':exclamación_e_interrogación:',
        'fr': u':points_d’exclamation_et_d’interrogation:',
        'pt': u':exclamação_com_interrogação:',
        'it': ':punto_esclamativo_e_interrogativo:'
    },
    u'\U00002049': { # ⁉
        'en' : ':exclamation_question_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':interrobang:'],
        'variant': True,
        'de': ':ausrufe-_und_fragezeichen:',
        'es': u':exclamación_e_interrogación:',
        'fr': u':points_d’exclamation_et_d’interrogation:',
        'pt': u':exclamação_com_interrogação:',
        'it': ':punto_esclamativo_e_interrogativo:'
    },
    u'\U0001F92F': { # 🤯
        'en' : ':exploding_head:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':explodierender_kopf:',
        'es': ':cabeza_explotando:',
        'fr': u':tête_qui_explose:',
        'pt': u':cabeça_explodindo:',
        'it': ':testa_che_esplode:'
    },
    u'\U0001F611': { # 😑
        'en' : ':expressionless_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':expressionless:'],
        'de': ':ausdrucksloses_gesicht:',
        'es': u':cara_sin_expresión:',
        'fr': ':visage_sans_expression:',
        'pt': ':rosto_inexpressivo:',
        'it': ':faccina_inespressiva:'
    },
    u'\U0001F441\U0000FE0F': { # 👁️
        'en' : ':eye:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':auge:',
        'es': ':ojo:',
        'fr': u':œil:',
        'pt': ':olho:',
        'it': ':occhio:'
    },
    u'\U0001F441': { # 👁
        'en' : ':eye:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':auge:',
        'es': ':ojo:',
        'fr': u':œil:',
        'pt': ':olho:',
        'it': ':occhio:'
    },
    u'\U0001F441\U0000FE0F\U0000200D\U0001F5E8\U0000FE0F': { # 👁️‍🗨️
        'en' : ':eye_in_speech_bubble:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': u':œil_dans_une_bulle_de_bd:',
        'pt': u':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:'
    },
    u'\U0001F441\U0000200D\U0001F5E8\U0000FE0F': { # 👁‍🗨️
        'en' : ':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2
    },
    u'\U0001F441\U0000FE0F\U0000200D\U0001F5E8': { # 👁️‍🗨
        'en' : ':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2
    },
    u'\U0001F441\U0000200D\U0001F5E8': { # 👁‍🗨
        'en' : ':eye_in_speech_bubble:',
        'status' : unqualified,
        'E' : 2,
        'de': ':auge_in_sprechblase:',
        'es': ':ojo_en_bocadillo_de_texto:',
        'fr': u':œil_dans_une_bulle_de_bd:',
        'pt': u':olho_no_balão_de_diálogo:',
        'it': ':occhio_nel_fumetto:'
    },
    u'\U0001F440': { # 👀
        'en' : ':eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':augen:',
        'es': ':ojos:',
        'fr': ':yeux:',
        'pt': ':olhos:',
        'it': ':occhi:'
    },
    u'\U0001F618': { # 😘
        'en' : ':face_blowing_a_kiss:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':kissing_heart:'],
        'de': ':kuss_zuwerfendes_gesicht:',
        'es': ':cara_lanzando_un_beso:',
        'fr': ':visage_envoyant_un_bisou:',
        'pt': ':rosto_mandando_um_beijo:',
        'it': ':faccina_che_manda_un_bacio:'
    },
    u'\U0001F62E\U0000200D\U0001F4A8': { # 😮‍💨
        'en' : ':face_exhaling:',
        'status' : fully_qualified,
        'E' : 13.1,
        'de': ':gesicht_das_ausatmet:',
        'es': ':cara_exhalando:',
        'fr': ':visage_expirant:',
        'pt': ':rosto_exalando:',
        'it': ':faccina_che_espira:'
    },
    u'\U0001F979': { # 🥹
        'en' : ':face_holding_back_tears:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':gesicht_das_tränen_zurückhält:',
        'es': u':cara_aguantándose_las_lágrimas:',
        'fr': ':visage_retenant_ses_larmes:',
        'pt': u':rosto_segurando_as_lágrimas:',
        'it': ':faccina_che_trattiene_le_lacrime:'
    },
    u'\U0001F636\U0000200D\U0001F32B\U0000FE0F': { # 😶‍🌫️
        'en' : ':face_in_clouds:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F636\U0000200D\U0001F32B': { # 😶‍🌫
        'en' : ':face_in_clouds:',
        'status' : minimally_qualified,
        'E' : 13.1,
        'de': ':gesicht_in_wolken:',
        'es': ':cara_en_las_nubes:',
        'fr': ':visage_dans_les_nuages:',
        'pt': ':rosto_nas_nuvens:',
        'it': ':testa_tra_le_nuvole:'
    },
    u'\U0001F60B': { # 😋
        'en' : ':face_savoring_food:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':yum:'],
        'de': ':sich_die_lippen_leckendes_gesicht:',
        'es': ':cara_saboreando_comida:',
        'fr': ':miam:',
        'pt': ':rosto_saboreando_comida:',
        'it': ':faccina_che_si_lecca_i_baffi:'
    },
    u'\U0001F631': { # 😱
        'en' : ':face_screaming_in_fear:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':scream:'],
        'de': ':vor_angst_schreiendes_gesicht:',
        'es': ':cara_gritando_de_miedo:',
        'fr': ':visage_qui_hurle_de_peur:',
        'pt': ':rosto_gritando_de_medo:',
        'it': ':faccina_terrorizzata:'
    },
    u'\U0001F92E': { # 🤮
        'en' : ':face_vomiting:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':vomiting_face:'],
        'de': ':kotzendes_gesicht:',
        'es': ':cara_vomitando:',
        'fr': ':visage_qui_vomit:',
        'pt': ':rosto_vomitando:',
        'it': ':faccina_che_vomita:'
    },
    u'\U0001F635': { # 😵
        'en' : ':face_with_crossed-out_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dizzy_face:'],
        'de': ':benommenes_gesicht:',
        'es': ':cara_mareada:',
        'fr': u':visage_étourdi:',
        'pt': ':rosto_atordoado:',
        'it': ':faccina_frastornata:'
    },
    u'\U0001FAE4': { # 🫤
        'en' : ':face_with_diagonal_mouth:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':gesicht_mit_schrägem_mund:',
        'es': ':cara_con_boca_diagonal:',
        'fr': ':visage_avec_bouche_en_diagonale:',
        'pt': ':rosto_com_boca_diagonal:',
        'it': ':faccina_perplessa:'
    },
    u'\U0001F92D': { # 🤭
        'en' : ':face_with_hand_over_mouth:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':hand_over_mouth:'],
        'de': ':verlegen_kicherndes_gesicht:',
        'es': ':cara_con_mano_sobre_la_boca:',
        'fr': ':visage_avec_une_main_sur_la_bouche:',
        'pt': u':rosto_com_a_mão_sobre_a_boca:',
        'it': ':faccina_con_mano_sulla_bocca:'
    },
    u'\U0001F915': { # 🤕
        'en' : ':face_with_head-bandage:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':face_with_head_bandage:', ':face_with_head__bandage:'],
        'de': ':gesicht_mit_kopfverband:',
        'es': ':cara_con_la_cabeza_vendada:',
        'fr': u':visage_avec_bandage_autour_de_la_tête:',
        'pt': u':rosto_com_atadura_na_cabeça:',
        'it': ':faccina_con_la_testa_bendata:'
    },
    u'\U0001F637': { # 😷
        'en' : ':face_with_medical_mask:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mask:'],
        'de': ':gesicht_mit_atemschutzmaske:',
        'es': u':cara_con_mascarilla_médica:',
        'fr': ':visage_avec_masque:',
        'pt': u':rosto_com_máscara_médica:',
        'it': ':faccina_con_mascherina:'
    },
    u'\U0001F9D0': { # 🧐
        'en' : ':face_with_monocle:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':monocle_face:'],
        'de': ':gesicht_mit_monokel:',
        'es': u':cara_con_monóculo:',
        'fr': ':visage_avec_un_monocle:',
        'pt': u':rosto_com_monóculo:',
        'it': ':faccina_con_monocolo:'
    },
    u'\U0001FAE2': { # 🫢
        'en' : ':face_with_open_eyes_and_hand_over_mouth:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':gesicht_mit_offenen_augen_und_hand_über_dem_mund:',
        'es': ':cara_con_ojos_abiertos_y_boca_tapada:',
        'fr': ':visage_avec_yeux_ouverts_et_main_sur_la_bouche:',
        'pt': u':rosto_com_olhos_abertos_e_mão_sobre_a_boca:',
        'it': ':faccina_con_occhi_aperti_e_mano_sulla_bocca:'
    },
    u'\U0001F62E': { # 😮
        'en' : ':face_with_open_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':open_mouth:'],
        'de': ':gesicht_mit_offenem_mund:',
        'es': ':cara_con_la_boca_abierta:',
        'fr': ':visage_avec_bouche_ouverte:',
        'pt': ':rosto_com_boca_aberta:',
        'it': ':faccina_con_bocca_aperta:'
    },
    u'\U0001FAE3': { # 🫣
        'en' : ':face_with_peeking_eye:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':gesicht_mit_durch_die_finger_linsendem_auge:',
        'es': ':cara_tapada_con_ojo_espiando:',
        'fr': ':visage_qui_regarde_entre_ses_doigts:',
        'pt': ':rosto_com_olho_espiando:',
        'it': ':faccina_che_sbircia_tra_le_dita:'
    },
    u'\U0001F928': { # 🤨
        'en' : ':face_with_raised_eyebrow:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':raised_eyebrow:'],
        'de': ':gesicht_mit_hochgezogenen_augenbrauen:',
        'es': ':cara_con_ceja_alzada:',
        'fr': u':visage_avec_les_sourcils_relevés:',
        'pt': ':rosto_com_sobrancelha_levantada:',
        'it': ':faccia_con_sopracciglia_alzate:'
    },
    u'\U0001F644': { # 🙄
        'en' : ':face_with_rolling_eyes:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':roll_eyes:'],
        'de': ':augen_verdrehendes_gesicht:',
        'es': ':cara_con_ojos_en_blanco:',
        'fr': ':visage_roulant_des_yeux:',
        'pt': ':rosto_com_olhos_revirados:',
        'it': ':faccina_con_occhi_al_cielo:'
    },
    u'\U0001F635\U0000200D\U0001F4AB': { # 😵‍💫
        'en' : ':face_with_spiral_eyes:',
        'status' : fully_qualified,
        'E' : 13.1,
        'de': ':gesicht_mit_spiralen_als_augen:',
        'es': ':cara_con_ojos_de_espiral:',
        'fr': ':visage_aux_yeux_en_spirales:',
        'pt': ':rosto_com_olhos_em_espiral:',
        'it': ':faccina_con_occhi_a_spirale:'
    },
    u'\U0001F624': { # 😤
        'en' : ':face_with_steam_from_nose:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':triumph:'],
        'de': ':schnaubendes_gesicht:',
        'es': ':cara_resoplando:',
        'fr': u':visage_avec_fumée_sortant_des_narines:',
        'pt': ':rosto_soltando_vapor_pelo_nariz:',
        'it': ':faccina_che_sbuffa:'
    },
    u'\U0001F92C': { # 🤬
        'en' : ':face_with_symbols_on_mouth:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':cursing_face:'],
        'de': u':gesicht_mit_symbolen_über_dem_mund:',
        'es': u':cara_con_símbolos_en_la_boca:',
        'fr': ':visage_avec_des_symboles_dans_la_bouche:',
        'pt': u':rosto_com_símbolos_na_boca:',
        'it': ':faccina_con_simboli_sulla_bocca:'
    },
    u'\U0001F602': { # 😂
        'en' : ':face_with_tears_of_joy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':joy:'],
        'de': u':gesicht_mit_freudentränen:',
        'es': ':cara_llorando_de_risa:',
        'fr': ':visage_riant_aux_larmes:',
        'pt': ':rosto_chorando_de_rir:',
        'it': ':faccina_con_lacrime_di_gioia:'
    },
    u'\U0001F912': { # 🤒
        'en' : ':face_with_thermometer:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':gesicht_mit_fieberthermometer:',
        'es': u':cara_con_termómetro:',
        'fr': u':visage_avec_thermomètre:',
        'pt': u':rosto_com_termômetro:',
        'it': ':faccina_con_termometro:'
    },
    u'\U0001F61B': { # 😛
        'en' : ':face_with_tongue:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':stuck_out_tongue:'],
        'de': ':gesicht_mit_herausgestreckter_zunge:',
        'es': ':cara_sacando_la_lengua:',
        'fr': ':visage_qui_tire_la_langue:',
        'pt': u':rosto_mostrando_a_língua:',
        'it': ':faccina_che_mostra_la_lingua:'
    },
    u'\U0001F636': { # 😶
        'en' : ':face_without_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':no_mouth:'],
        'de': ':gesicht_ohne_mund:',
        'es': ':cara_sin_boca:',
        'fr': ':visage_sans_bouche:',
        'pt': ':rosto_sem_boca:',
        'it': ':faccina_senza_bocca:'
    },
    u'\U0001F3ED': { # 🏭
        'en' : ':factory:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':fabrik:',
        'es': u':fábrica:',
        'fr': ':usine:',
        'pt': u':fábrica:',
        'it': ':fabbrica:'
    },
    u'\U0001F9D1\U0000200D\U0001F3ED': { # 🧑‍🏭
        'en' : ':factory_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in):',
        'es': ':profesional_industrial:',
        'fr': ':ouvrier_(tous_genres):',
        'pt': u':funcionário_de_fábrica:',
        'it': ':persona_che_lavora_in_fabbrica:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F3ED': { # 🧑🏿‍🏭
        'en' : ':factory_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in)_dunkle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_oscuro:',
        'fr': u':ouvrier_(tous_genres)_peau_foncée:',
        'pt': u':funcionário_de_fábrica_pele_escura:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F3ED': { # 🧑🏻‍🏭
        'en' : ':factory_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in)_helle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_claro:',
        'fr': ':ouvrier_(tous_genres)_peau_claire:',
        'pt': u':funcionário_de_fábrica_pele_clara:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F3ED': { # 🧑🏾‍🏭
        'en' : ':factory_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in)_mitteldunkle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_oscuro_medio:',
        'fr': ':ouvrier_(tous_genres)_peau_mate:',
        'pt': u':funcionário_de_fábrica_pele_morena_escura:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F3ED': { # 🧑🏼‍🏭
        'en' : ':factory_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in)_mittelhelle_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_claro_medio:',
        'fr': ':ouvrier_(tous_genres)_peau_moyennement_claire:',
        'pt': u':funcionário_de_fábrica_pele_morena_clara:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F3ED': { # 🧑🏽‍🏭
        'en' : ':factory_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':fabrikarbeiter(in)_mittlere_hautfarbe:',
        'es': ':profesional_industrial_tono_de_piel_medio:',
        'fr': u':ouvrier_(tous_genres)_peau_légèrement_mate:',
        'pt': u':funcionário_de_fábrica_pele_morena:',
        'it': ':persona_che_lavora_in_fabbrica_carnagione_olivastra:'
    },
    u'\U0001F9DA': { # 🧚
        'en' : ':fairy:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee:',
        'es': ':hada:',
        'fr': u':personnage_féérique:',
        'pt': ':fada:',
        'it': ':fata:'
    },
    u'\U0001F9DA\U0001F3FF': { # 🧚🏿
        'en' : ':fairy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_dunkle_hautfarbe:',
        'es': ':hada_tono_de_piel_oscuro:',
        'fr': u':personnage_féérique_peau_foncée:',
        'pt': ':fada_pele_escura:',
        'it': ':fata_carnagione_scura:'
    },
    u'\U0001F9DA\U0001F3FB': { # 🧚🏻
        'en' : ':fairy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_helle_hautfarbe:',
        'es': ':hada_tono_de_piel_claro:',
        'fr': u':personnage_féérique_peau_claire:',
        'pt': ':fada_pele_clara:',
        'it': ':fata_carnagione_chiara:'
    },
    u'\U0001F9DA\U0001F3FE': { # 🧚🏾
        'en' : ':fairy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mitteldunkle_hautfarbe:',
        'es': ':hada_tono_de_piel_oscuro_medio:',
        'fr': u':personnage_féérique_peau_mate:',
        'pt': ':fada_pele_morena_escura:',
        'it': ':fata_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DA\U0001F3FC': { # 🧚🏼
        'en' : ':fairy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mittelhelle_hautfarbe:',
        'es': ':hada_tono_de_piel_claro_medio:',
        'fr': u':personnage_féérique_peau_moyennement_claire:',
        'pt': ':fada_pele_morena_clara:',
        'it': ':fata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DA\U0001F3FD': { # 🧚🏽
        'en' : ':fairy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':märchenfee_mittlere_hautfarbe:',
        'es': ':hada_tono_de_piel_medio:',
        'fr': u':personnage_féérique_peau_légèrement_mate:',
        'pt': ':fada_pele_morena:',
        'it': ':fata_carnagione_olivastra:'
    },
    u'\U0001F9C6': { # 🧆
        'en' : ':falafel:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':falafel:',
        'es': ':falafel:',
        'fr': ':falafels:',
        'pt': ':falafel:',
        'it': ':falafel:'
    },
    u'\U0001F342': { # 🍂
        'en' : ':fallen_leaf:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':laub:',
        'es': u':hojas_caídas:',
        'fr': ':feuille_morte:',
        'pt': u':folhas_caídas:',
        'it': ':foglia_caduta:'
    },
    u'\U0001F46A': { # 👪
        'en' : ':family:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':familie:',
        'es': ':familia:',
        'fr': ':famille:',
        'pt': u':família:',
        'it': ':famiglia:'
    },
    u'\U0001F468\U0000200D\U0001F466': { # 👨‍👦
        'en' : ':family_man_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':familie_mann,_junge:',
        'es': u':familia_hombre_y_niño:',
        'fr': u':famille_homme_et_garçon:',
        'pt': u':família_homem_e_menino:',
        'it': ':famiglia_uomo_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': { # 👨‍👦‍👦
        'en' : ':family_man_boy_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':familie_mann,_junge_und_junge:',
        'es': u':familia_hombre_niño_niño:',
        'fr': u':famille_homme,_garçon_et_garçon:',
        'pt': u':família_homem_menino_e_menino:',
        'it': ':famiglia_uomo_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F467': { # 👨‍👧
        'en' : ':family_man_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen:',
        'es': u':familia_hombre_y_niña:',
        'fr': ':famille_homme_et_fille:',
        'pt': u':família_homem_e_menina:',
        'it': ':famiglia_uomo_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': { # 👨‍👧‍👦
        'en' : ':family_man_girl_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen_und_junge:',
        'es': u':familia_hombre_niña_niño:',
        'fr': u':famille_homme,_fille_et_garçon:',
        'pt': u':família_homem_menina_e_menino:',
        'it': ':famiglia_uomo_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': { # 👨‍👧‍👧
        'en' : ':family_man_girl_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_mann,_mädchen_und_mädchen:',
        'es': u':familia_hombre_niña_niña:',
        'fr': ':famille_homme,_fille_et_fille:',
        'pt': u':família_homem_menina_e_menina:',
        'it': ':famiglia_uomo_bambina_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466': { # 👨‍👨‍👦
        'en' : ':family_man_man_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_mann,_mann_und_junge:',
        'es': u':familia_hombre_hombre_niño:',
        'fr': u':famille_homme,_homme_et_garçon:',
        'pt': u':família_homem_homem_e_menino:',
        'it': ':famiglia_uomo_uomo_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F466\U0000200D\U0001F466': { # 👨‍👨‍👦‍👦
        'en' : ':family_man_man_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_mann,_mann,_junge_und_junge:',
        'es': u':familia_hombre_hombre_niño_niño:',
        'fr': u':famille_homme,_homme,_garçon_et_garçon:',
        'pt': u':família_homem_homem_menino_e_menino:',
        'it': ':famiglia_uomo_uomo_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467': { # 👨‍👨‍👧
        'en' : ':family_man_man_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann_und_mädchen:',
        'es': u':familia_hombre_hombre_niña:',
        'fr': ':famille_homme,_homme_et_fille:',
        'pt': u':família_homem_homem_e_menina:',
        'it': ':famiglia_uomo_uomo_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F466': { # 👨‍👨‍👧‍👦
        'en' : ':family_man_man_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann,_mädchen_und_junge:',
        'es': u':familia_hombre_hombre_niña_niño:',
        'fr': u':famille_homme,_homme,_fille_et_garçon:',
        'pt': u':família_homem_homem_menina_e_menino:',
        'it': ':famiglia_uomo_uomo_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F468\U0000200D\U0001F467\U0000200D\U0001F467': { # 👨‍👨‍👧‍👧
        'en' : ':family_man_man_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_mann,_mädchen_und_mädchen:',
        'es': u':familia_hombre_hombre_niña_niña:',
        'fr': ':famille_homme,_homme,_fille_et_fille:',
        'pt': u':família_homem_homem_menina_e_menina:',
        'it': ':famiglia_uomo_uomo_bambina_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466': { # 👨‍👩‍👦
        'en' : ':family_man_woman_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_mann,_frau_und_junge:',
        'es': u':familia_hombre_mujer_niño:',
        'fr': u':famille_homme,_femme_et_garçon:',
        'pt': u':família_homem_mulher_e_menino:',
        'it': ':famiglia_uomo_donna_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': { # 👨‍👩‍👦‍👦
        'en' : ':family_man_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_mann,_frau,_junge_und_junge:',
        'es': u':familia_hombre_mujer_niño_niño:',
        'fr': u':famille_homme,_femme,_garçon_et_garçon:',
        'pt': u':família_homem_mulher_menino_e_menino:',
        'it': ':famiglia_uomo_donna_bambino_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467': { # 👨‍👩‍👧
        'en' : ':family_man_woman_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau_und_mädchen:',
        'es': u':familia_hombre_mujer_niña:',
        'fr': ':famille_homme,_femme_et_fille:',
        'pt': u':família_homem_mulher_e_menina:',
        'it': ':famiglia_uomo_donna_e_bambina:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': { # 👨‍👩‍👧‍👦
        'en' : ':family_man_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau,_mädchen_und_junge:',
        'es': u':familia_hombre_mujer_niña_niño:',
        'fr': u':famille_homme,_femme,_fille_et_garçon:',
        'pt': u':família_homem_mulher_menina_e_menino:',
        'it': ':famiglia_uomo_donna_bambina_e_bambino:'
    },
    u'\U0001F468\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': { # 👨‍👩‍👧‍👧
        'en' : ':family_man_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_mann,_frau,_mädchen_und_mädchen:',
        'es': u':familia_hombre_mujer_niña_niña:',
        'fr': ':famille_homme,_femme,_fille_et_fille:',
        'pt': u':família_homem_mulher_menina_e_menina:',
        'it': ':famiglia_uomo_donna_bambina_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F466': { # 👩‍👦
        'en' : ':family_woman_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':familie_frau,_junge:',
        'es': u':familia_mujer_y_niño:',
        'fr': u':famille_femme_et_garçon:',
        'pt': u':família_mulher_e_menino:',
        'it': ':famiglia_donna_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': { # 👩‍👦‍👦
        'en' : ':family_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':familie_frau,_junge_und_junge:',
        'es': u':familia_mujer_niño_niño:',
        'fr': u':famille_femme,_garçon_et_garçon:',
        'pt': u':família_mulher_menino_e_menino:',
        'it': ':famiglia_donna_bambino_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F467': { # 👩‍👧
        'en' : ':family_woman_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen:',
        'es': u':familia_mujer_y_niña:',
        'fr': ':famille_femme_et_fille:',
        'pt': u':família_mulher_e_menina:',
        'it': ':famiglia_donna_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': { # 👩‍👧‍👦
        'en' : ':family_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen_und_junge:',
        'es': u':familia_mujer_niña_niño:',
        'fr': u':famille_femme,_fille_et_garçon:',
        'pt': u':família_mulher_menina_e_menino:',
        'it': ':famiglia_donna_bambina_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': { # 👩‍👧‍👧
        'en' : ':family_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':familie_frau,_mädchen_und_mädchen:',
        'es': u':familia_mujer_niña_niña:',
        'fr': ':famille_femme,_fille_et_fille:',
        'pt': u':família_mulher_menina_e_menina:',
        'it': ':famiglia_donna_bambina_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466': { # 👩‍👩‍👦
        'en' : ':family_woman_woman_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_frau,_frau_und_junge:',
        'es': u':familia_mujer_mujer_niño:',
        'fr': u':famille_femme,_femme_et_garçon:',
        'pt': u':família_mulher_mulher_e_menino:',
        'it': ':famiglia_donna_donna_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F466\U0000200D\U0001F466': { # 👩‍👩‍👦‍👦
        'en' : ':family_woman_woman_boy_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':familie_frau,_frau,_junge_und_junge:',
        'es': u':familia_mujer_mujer_niño_niño:',
        'fr': u':famille_femme,_femme,_garçon_et_garçon:',
        'pt': u':família_mulher_mulher_menino_e_menino:',
        'it': ':famiglia_donna_donna_bambino_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467': { # 👩‍👩‍👧
        'en' : ':family_woman_woman_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau_und_mädchen:',
        'es': u':familia_mujer_mujer_niña:',
        'fr': ':famille_femme,_femme_et_fille:',
        'pt': u':família_mulher_mulher_e_menina:',
        'it': ':famiglia_donna_donna_e_bambina:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F466': { # 👩‍👩‍👧‍👦
        'en' : ':family_woman_woman_girl_boy:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau,_mädchen_und_junge:',
        'es': u':familia_mujer_mujer_niña_niño:',
        'fr': u':famille_femme,_femme,_fille_et_garçon:',
        'pt': u':família_mulher_mulher_menina_e_menino:',
        'it': ':famiglia_donna_donna_bambina_e_bambino:'
    },
    u'\U0001F469\U0000200D\U0001F469\U0000200D\U0001F467\U0000200D\U0001F467': { # 👩‍👩‍👧‍👧
        'en' : ':family_woman_woman_girl_girl:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':familie_frau,_frau,_mädchen_und_mädchen:',
        'es': u':familia_mujer_mujer_niña_niña:',
        'fr': ':famille_femme,_femme,_fille_et_fille:',
        'pt': u':família_mulher_mulher_menina_e_menina:',
        'it': ':famiglia_donna_donna_bambina_e_bambina:'
    },
    u'\U0001F9D1\U0000200D\U0001F33E': { # 🧑‍🌾
        'en' : ':farmer:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin:',
        'es': ':profesional_de_la_agricultura:',
        'fr': ':fermier_(tous_genres):',
        'pt': ':agricultor:',
        'it': ':agricoltore:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F33E': { # 🧑🏿‍🌾
        'en' : ':farmer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_dunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_oscuro:',
        'fr': u':fermier_(tous_genres)_peau_foncée:',
        'pt': ':agricultor_pele_escura:',
        'it': ':agricoltore_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F33E': { # 🧑🏻‍🌾
        'en' : ':farmer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_helle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_claro:',
        'fr': ':fermier_(tous_genres)_peau_claire:',
        'pt': ':agricultor_pele_clara:',
        'it': ':agricoltore_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F33E': { # 🧑🏾‍🌾
        'en' : ':farmer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mitteldunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_oscuro_medio:',
        'fr': ':fermier_(tous_genres)_peau_mate:',
        'pt': ':agricultor_pele_morena_escura:',
        'it': ':agricoltore_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F33E': { # 🧑🏼‍🌾
        'en' : ':farmer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mittelhelle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_claro_medio:',
        'fr': ':fermier_(tous_genres)_peau_moyennement_claire:',
        'pt': ':agricultor_pele_morena_clara:',
        'it': ':agricoltore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F33E': { # 🧑🏽‍🌾
        'en' : ':farmer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':bauer/bäuerin_mittlere_hautfarbe:',
        'es': ':profesional_de_la_agricultura_tono_de_piel_medio:',
        'fr': u':fermier_(tous_genres)_peau_légèrement_mate:',
        'pt': ':agricultor_pele_morena:',
        'it': ':agricoltore_carnagione_olivastra:'
    },
    u'\U000023E9': { # ⏩
        'en' : ':fast-forward_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fast_forward:'],
        'variant': True,
        'de': ':doppelpfeile_nach_rechts:',
        'es': u':avance_rápido:',
        'fr': ':bouton_avance_rapide:',
        'pt': u':botão_avançar:',
        'it': ':pulsante_di_avanzamento_rapido:'
    },
    u'\U000023EC': { # ⏬
        'en' : ':fast_down_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_double_down:'],
        'de': ':doppelpfeile_nach_unten:',
        'es': u':triángulo_doble_hacia_abajo:',
        'fr': u':double_flèche_vers_le_bas:',
        'pt': u':botão_de_avanço_para_baixo:',
        'it': ':doppia_freccia_in_basso:'
    },
    u'\U000023EA': { # ⏪
        'en' : ':fast_reverse_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':rewind:'],
        'variant': True,
        'de': u':zurückspulen:',
        'es': ':rebobinado:',
        'fr': ':bouton_retour_rapide:',
        'pt': u':botão_de_retroceder:',
        'it': ':pulsante_di_riavvolgimento_rapido:'
    },
    u'\U000023EB': { # ⏫
        'en' : ':fast_up_button:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_double_up:'],
        'de': ':doppelpfeile_nach_oben:',
        'es': u':triángulo_doble_hacia_arriba:',
        'fr': u':double_flèche_vers_le_haut:',
        'pt': u':botão_de_avanço_para_cima:',
        'it': ':pulsante_doppia_freccia_in_alto:'
    },
    u'\U0001F4E0': { # 📠
        'en' : ':fax_machine:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fax:'],
        'de': u':faxgerät:',
        'es': u':máquina_de_fax:',
        'fr': ':fax:',
        'pt': ':fax:',
        'it': ':fax:'
    },
    u'\U0001F628': { # 😨
        'en' : ':fearful_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fearful:'],
        'de': u':ängstliches_gesicht:',
        'es': ':cara_asustada:',
        'fr': u':visage_effrayé:',
        'pt': ':rosto_amedrontado:',
        'it': ':faccina_impaurita:'
    },
    u'\U0001FAB6': { # 🪶
        'en' : ':feather:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':feder:',
        'es': ':pluma:',
        'fr': ':plume:',
        'pt': ':pena:',
        'it': ':piuma:'
    },
    u'\U00002640\U0000FE0F': { # ♀️
        'en' : ':female_sign:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': ':frauensymbol:',
        'es': ':signo_femenino:',
        'fr': ':symbole_de_la_femme:',
        'pt': u':símbolo_de_feminino:',
        'it': ':simbolo_genere_femminile:'
    },
    u'\U00002640': { # ♀
        'en' : ':female_sign:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': ':frauensymbol:',
        'es': ':signo_femenino:',
        'fr': ':symbole_de_la_femme:',
        'pt': u':símbolo_de_feminino:',
        'it': ':simbolo_genere_femminile:'
    },
    u'\U0001F3A1': { # 🎡
        'en' : ':ferris_wheel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':riesenrad:',
        'es': ':noria_de_feria:',
        'fr': ':grande_roue:',
        'pt': ':roda_gigante:',
        'it': ':ruota_panoramica:'
    },
    u'\U000026F4\U0000FE0F': { # ⛴️
        'en' : ':ferry:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':fähre:',
        'es': ':ferri:',
        'fr': ':ferry:',
        'pt': ':balsa:',
        'it': ':traghetto:'
    },
    u'\U000026F4': { # ⛴
        'en' : ':ferry:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':fähre:',
        'es': ':ferri:',
        'fr': ':ferry:',
        'pt': ':balsa:',
        'it': ':traghetto:'
    },
    u'\U0001F3D1': { # 🏑
        'en' : ':field_hockey:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':field_hockey_stick_and_ball:'],
        'de': ':feldhockey:',
        'es': ':hockey_sobre_hierba:',
        'fr': ':hockey_sur_gazon:',
        'pt': u':hóquei_de_campo:',
        'it': ':hockey_su_prato:'
    },
    u'\U0001F5C4\U0000FE0F': { # 🗄️
        'en' : ':file_cabinet:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':aktenschrank:',
        'es': ':archivador:',
        'fr': u':meuble_à_dossiers:',
        'pt': u':gavetas_de_escritório:',
        'it': ':schedario_da_ufficio:'
    },
    u'\U0001F5C4': { # 🗄
        'en' : ':file_cabinet:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':aktenschrank:',
        'es': ':archivador:',
        'fr': u':meuble_à_dossiers:',
        'pt': u':gavetas_de_escritório:',
        'it': ':schedario_da_ufficio:'
    },
    u'\U0001F4C1': { # 📁
        'en' : ':file_folder:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':ordner:',
        'es': ':carpeta_de_archivos:',
        'fr': ':dossier:',
        'pt': ':pasta_de_arquivos:',
        'it': ':cartella_file:'
    },
    u'\U0001F39E\U0000FE0F': { # 🎞️
        'en' : ':film_frames:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':film_strip:'],
        'variant': True,
        'de': ':filmstreifen:',
        'es': u':fotograma_de_película:',
        'fr': ':pellicule:',
        'pt': ':rolo_de_filmes:',
        'it': ':pellicola_cinematografica:'
    },
    u'\U0001F39E': { # 🎞
        'en' : ':film_frames:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':film_strip:'],
        'variant': True,
        'de': ':filmstreifen:',
        'es': u':fotograma_de_película:',
        'fr': ':pellicule:',
        'pt': ':rolo_de_filmes:',
        'it': ':pellicola_cinematografica:'
    },
    u'\U0001F4FD\U0000FE0F': { # 📽️
        'en' : ':film_projector:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':filmprojektor:',
        'es': ':proyector_de_cine:',
        'fr': u':projecteur_cinématographique:',
        'pt': ':projetor_de_filmes:',
        'it': ':proiettore_cinematografico:'
    },
    u'\U0001F4FD': { # 📽
        'en' : ':film_projector:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':filmprojektor:',
        'es': ':proyector_de_cine:',
        'fr': u':projecteur_cinématographique:',
        'pt': ':projetor_de_filmes:',
        'it': ':proiettore_cinematografico:'
    },
    u'\U0001F525': { # 🔥
        'en' : ':fire:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':feuer:',
        'es': ':fuego:',
        'fr': ':feu:',
        'pt': ':fogo:',
        'it': ':fuoco:'
    },
    u'\U0001F692': { # 🚒
        'en' : ':fire_engine:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':feuerwehrauto:',
        'es': ':coche_de_bomberos:',
        'fr': ':camion_de_pompier:',
        'pt': ':carro_do_corpo_de_bombeiros:',
        'it': ':camion_dei_pompieri:'
    },
    u'\U0001F9EF': { # 🧯
        'en' : ':fire_extinguisher:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':feuerlöscher:',
        'es': ':extintor:',
        'fr': ':extincteur:',
        'pt': u':extintor_de_incêndio:',
        'it': ':estintore:'
    },
    u'\U0001F9E8': { # 🧨
        'en' : ':firecracker:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':feuerwerkskörper:',
        'es': ':petardo:',
        'fr': u':pétard:',
        'pt': ':bombinha:',
        'it': ':petardo:'
    },
    u'\U0001F9D1\U0000200D\U0001F692': { # 🧑‍🚒
        'en' : ':firefighter:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau:',
        'es': ':bombero:',
        'fr': ':pompier:',
        'pt': ':bombeiro:',
        'it': ':pompiere:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F692': { # 🧑🏿‍🚒
        'en' : ':firefighter_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau_dunkle_hautfarbe:',
        'es': ':bombero_tono_de_piel_oscuro:',
        'fr': u':pompier_peau_foncée:',
        'pt': ':bombeiro_pele_escura:',
        'it': ':pompiere_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F692': { # 🧑🏻‍🚒
        'en' : ':firefighter_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau_helle_hautfarbe:',
        'es': ':bombero_tono_de_piel_claro:',
        'fr': ':pompier_peau_claire:',
        'pt': ':bombeiro_pele_clara:',
        'it': ':pompiere_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F692': { # 🧑🏾‍🚒
        'en' : ':firefighter_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau_mitteldunkle_hautfarbe:',
        'es': ':bombero_tono_de_piel_oscuro_medio:',
        'fr': ':pompier_peau_mate:',
        'pt': ':bombeiro_pele_morena_escura:',
        'it': ':pompiere_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F692': { # 🧑🏼‍🚒
        'en' : ':firefighter_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau_mittelhelle_hautfarbe:',
        'es': ':bombero_tono_de_piel_claro_medio:',
        'fr': ':pompier_peau_moyennement_claire:',
        'pt': ':bombeiro_pele_morena_clara:',
        'it': ':pompiere_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F692': { # 🧑🏽‍🚒
        'en' : ':firefighter_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':feuerwehrmann/-frau_mittlere_hautfarbe:',
        'es': ':bombero_tono_de_piel_medio:',
        'fr': u':pompier_peau_légèrement_mate:',
        'pt': ':bombeiro_pele_morena:',
        'it': ':pompiere_carnagione_olivastra:'
    },
    u'\U0001F386': { # 🎆
        'en' : ':fireworks:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':feuerwerk:',
        'es': ':fuegos_artificiales:',
        'fr': u':feu_d’artifice:',
        'pt': u':fogos_de_artifício:',
        'it': u':fuochi_d’artificio:'
    },
    u'\U0001F313': { # 🌓
        'en' : ':first_quarter_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':zunehmender_halbmond:',
        'es': ':luna_en_cuarto_creciente:',
        'fr': ':premier_quartier_de_lune:',
        'pt': ':quarto_crescente:',
        'it': ':primo_quarto_di_luna:'
    },
    u'\U0001F31B': { # 🌛
        'en' : ':first_quarter_moon_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':first_quarter_moon_with_face:'],
        'de': ':mondsichel_mit_gesicht_links:',
        'es': ':luna_de_cuarto_creciente_con_cara:',
        'fr': ':premier_quartier_de_lune_avec_visage:',
        'pt': ':rosto_da_lua_de_quarto_crescente:',
        'it': ':faccina_primo_quarto_di_luna:'
    },
    u'\U0001F41F': { # 🐟
        'en' : ':fish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':fisch:',
        'es': ':pez:',
        'fr': ':poisson:',
        'pt': ':peixe:',
        'it': ':pesce:'
    },
    u'\U0001F365': { # 🍥
        'en' : ':fish_cake_with_swirl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fish_cake:'],
        'de': ':fischfrikadelle:',
        'es': u':pastel_de_pescado_japonés:',
        'fr': ':croquette_de_poisson:',
        'pt': ':bolinho_de_peixe:',
        'it': ':tortino_di_pesce_a_spirale:'
    },
    u'\U0001F3A3': { # 🎣
        'en' : ':fishing_pole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fishing_pole_and_fish:'],
        'de': ':angel_mit_fisch:',
        'es': u':caña_de_pescar:',
        'fr': u':pêche_à_la_ligne:',
        'pt': ':pesca:',
        'it': ':canna_da_pesca:'
    },
    u'\U0001F560': { # 🕠
        'en' : ':five-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock530:'],
        'variant': True,
        'de': ':5.30_uhr:',
        'es': ':cinco_y_media:',
        'fr': ':cinq_heures_et_demie:',
        'pt': ':cinco_e_meia:',
        'it': ':ore_cinque_e_mezza:'
    },
    u'\U0001F554': { # 🕔
        'en' : u':five_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock5:'],
        'variant': True,
        'de': ':5.00_uhr:',
        'es': ':5_en_punto:',
        'fr': ':cinq_heures:',
        'pt': ':5_horas:',
        'it': ':ore_cinque:'
    },
    u'\U000026F3': { # ⛳
        'en' : ':flag_in_hole:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':golf:'],
        'variant': True,
        'de': ':golffahne:',
        'es': u':banderín_en_hoyo:',
        'fr': ':drapeau_de_golf:',
        'pt': ':bandeira_no_buraco:',
        'it': ':bandiera_in_buca:'
    },
    u'\U0001F9A9': { # 🦩
        'en' : ':flamingo:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':flamingo:',
        'es': ':flamenco:',
        'fr': ':flamant:',
        'pt': ':flamingo:',
        'it': ':fenicottero:'
    },
    u'\U0001F526': { # 🔦
        'en' : ':flashlight:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':taschenlampe:',
        'es': ':linterna:',
        'fr': ':torche:',
        'pt': ':lanterna:',
        'it': ':torcia:'
    },
    u'\U0001F97F': { # 🥿
        'en' : ':flat_shoe:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':flacher_schuh:',
        'es': ':bailarina:',
        'fr': ':chaussure_plate:',
        'pt': ':sapatilha:',
        'it': ':ballerina:'
    },
    u'\U0001FAD3': { # 🫓
        'en' : ':flatbread:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':fladenbrot:',
        'es': ':pan_sin_levadura:',
        'fr': ':galette:',
        'pt': u':pão_sírio:',
        'it': ':focaccia:'
    },
    u'\U0000269C\U0000FE0F': { # ⚜️
        'en' : ':fleur-de-lis:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':fleur__de__lis:', ':fleur_de_lis:'],
        'variant': True,
        'de': ':lilie:',
        'es': ':flor_de_lis:',
        'fr': ':fleur_de_lys:',
        'pt': ':flor-de-lis:',
        'it': ':giglio_araldico:'
    },
    u'\U0000269C': { # ⚜
        'en' : ':fleur-de-lis:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [':fleur__de__lis:', ':fleur_de_lis:'],
        'variant': True,
        'de': ':lilie:',
        'es': ':flor_de_lis:',
        'fr': ':fleur_de_lys:',
        'pt': ':flor-de-lis:',
        'it': ':giglio_araldico:'
    },
    u'\U0001F4AA': { # 💪
        'en' : ':flexed_biceps:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':muscle:'],
        'de': ':angespannter_bizeps:',
        'es': u':bíceps_flexionado:',
        'fr': u':biceps_contracté:',
        'pt': u':bíceps:',
        'it': ':bicipite:'
    },
    u'\U0001F4AA\U0001F3FF': { # 💪🏿
        'en' : ':flexed_biceps_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':angespannter_bizeps_dunkle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_oscuro:',
        'fr': u':biceps_contracté_peau_foncée:',
        'pt': u':bíceps_pele_escura:',
        'it': ':bicipite_carnagione_scura:'
    },
    u'\U0001F4AA\U0001F3FB': { # 💪🏻
        'en' : ':flexed_biceps_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':angespannter_bizeps_helle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_claro:',
        'fr': u':biceps_contracté_peau_claire:',
        'pt': u':bíceps_pele_clara:',
        'it': ':bicipite_carnagione_chiara:'
    },
    u'\U0001F4AA\U0001F3FE': { # 💪🏾
        'en' : ':flexed_biceps_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':angespannter_bizeps_mitteldunkle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_oscuro_medio:',
        'fr': u':biceps_contracté_peau_mate:',
        'pt': u':bíceps_pele_morena_escura:',
        'it': ':bicipite_carnagione_abbastanza_scura:'
    },
    u'\U0001F4AA\U0001F3FC': { # 💪🏼
        'en' : ':flexed_biceps_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':angespannter_bizeps_mittelhelle_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_claro_medio:',
        'fr': u':biceps_contracté_peau_moyennement_claire:',
        'pt': u':bíceps_pele_morena_clara:',
        'it': ':bicipite_carnagione_abbastanza_chiara:'
    },
    u'\U0001F4AA\U0001F3FD': { # 💪🏽
        'en' : ':flexed_biceps_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':angespannter_bizeps_mittlere_hautfarbe:',
        'es': u':bíceps_flexionado_tono_de_piel_medio:',
        'fr': u':biceps_contracté_peau_légèrement_mate:',
        'pt': u':bíceps_pele_morena:',
        'it': ':bicipite_carnagione_olivastra:'
    },
    u'\U0001F4BE': { # 💾
        'en' : ':floppy_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':diskette:',
        'es': ':disquete:',
        'fr': ':disquette:',
        'pt': ':disquete:',
        'it': ':floppy_disc:'
    },
    u'\U0001F3B4': { # 🎴
        'en' : ':flower_playing_cards:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':japanische_blumenkarte:',
        'es': ':cartas_de_flores:',
        'fr': ':jeu_des_fleurs:',
        'pt': ':carta_de_flores:',
        'it': ':carta_da_gioco_hanafuda:'
    },
    u'\U0001F633': { # 😳
        'en' : ':flushed_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':flushed:'],
        'de': u':errötetes_gesicht_mit_großen_augen:',
        'es': ':cara_sonrojada:',
        'fr': ':visage_qui_rougit:',
        'pt': ':rosto_ruborizado:',
        'it': ':faccina_imbarazzata:'
    },
    u'\U0001FAB0': { # 🪰
        'en' : ':fly:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':fliege:',
        'es': ':mosca:',
        'fr': ':mouche:',
        'pt': ':mosca:',
        'it': ':mosca:'
    },
    u'\U0001F94F': { # 🥏
        'en' : ':flying_disc:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':frisbee:',
        'es': ':disco_volador:',
        'fr': ':disque_volant:',
        'pt': ':frisbee:',
        'it': ':frisbee:'
    },
    u'\U0001F6F8': { # 🛸
        'en' : ':flying_saucer:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':fliegende_untertasse:',
        'es': ':platillo_volante:',
        'fr': ':soucoupe_volante:',
        'pt': ':disco_voador:',
        'it': ':disco_volante:'
    },
    u'\U0001F32B\U0000FE0F': { # 🌫️
        'en' : ':fog:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':nebel:',
        'es': ':niebla:',
        'fr': ':brouillard:',
        'pt': ':neblina:',
        'it': ':nebbia:'
    },
    u'\U0001F32B': { # 🌫
        'en' : ':fog:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':nebel:',
        'es': ':niebla:',
        'fr': ':brouillard:',
        'pt': ':nevoeiro:',
        'it': ':nebbia:'
    },
    u'\U0001F301': { # 🌁
        'en' : ':foggy:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':neblig:',
        'es': ':bruma:',
        'fr': ':brume:',
        'pt': ':enevoado:',
        'it': ':nebbioso:'
    },
    u'\U0001F64F': { # 🙏
        'en' : ':folded_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pray:'],
        'de': u':zusammengelegte_handflächen:',
        'es': u':manos_en_oración:',
        'fr': u':mains_en_prière:',
        'pt': u':mãos_juntas:',
        'it': ':mani_giunte:'
    },
    u'\U0001F64F\U0001F3FF': { # 🙏🏿
        'en' : ':folded_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_dunkle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_oscuro:',
        'fr': u':mains_en_prière_peau_foncée:',
        'pt': u':mãos_juntas_pele_escura:',
        'it': ':mani_giunte_carnagione_scura:'
    },
    u'\U0001F64F\U0001F3FB': { # 🙏🏻
        'en' : ':folded_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_helle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_claro:',
        'fr': u':mains_en_prière_peau_claire:',
        'pt': u':mãos_juntas_pele_clara:',
        'it': ':mani_giunte_carnagione_chiara:'
    },
    u'\U0001F64F\U0001F3FE': { # 🙏🏾
        'en' : ':folded_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mitteldunkle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_oscuro_medio:',
        'fr': u':mains_en_prière_peau_mate:',
        'pt': u':mãos_juntas_pele_morena_escura:',
        'it': ':mani_giunte_carnagione_abbastanza_scura:'
    },
    u'\U0001F64F\U0001F3FC': { # 🙏🏼
        'en' : ':folded_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mittelhelle_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_claro_medio:',
        'fr': u':mains_en_prière_peau_moyennement_claire:',
        'pt': u':mãos_juntas_pele_morena_clara:',
        'it': ':mani_giunte_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64F\U0001F3FD': { # 🙏🏽
        'en' : ':folded_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':betende_hände_mittlere_hautfarbe:',
        'es': u':manos_en_oración_tono_de_piel_medio:',
        'fr': u':mains_en_prière_peau_légèrement_mate:',
        'pt': u':mãos_juntas_pele_morena:',
        'it': ':mani_giunte_carnagione_olivastra:'
    },
    u'\U0001FAD5': { # 🫕
        'en' : ':fondue:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':fondue:',
        'es': ':fondue:',
        'fr': ':fondue:',
        'pt': ':fondue:',
        'it': ':fonduta:'
    },
    u'\U0001F9B6': { # 🦶
        'en' : ':foot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß:',
        'es': ':pie:',
        'fr': ':pied:',
        'pt': u':pé:',
        'it': ':piede:'
    },
    u'\U0001F9B6\U0001F3FF': { # 🦶🏿
        'en' : ':foot_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_dunkle_hautfarbe:',
        'es': ':pie_tono_de_piel_oscuro:',
        'fr': u':pied_peau_foncée:',
        'pt': u':pé_pele_escura:',
        'it': ':piede_carnagione_scura:'
    },
    u'\U0001F9B6\U0001F3FB': { # 🦶🏻
        'en' : ':foot_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_helle_hautfarbe:',
        'es': ':pie_tono_de_piel_claro:',
        'fr': ':pied_peau_claire:',
        'pt': u':pé_pele_clara:',
        'it': ':piede_carnagione_chiara:'
    },
    u'\U0001F9B6\U0001F3FE': { # 🦶🏾
        'en' : ':foot_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mitteldunkle_hautfarbe:',
        'es': ':pie_tono_de_piel_oscuro_medio:',
        'fr': ':pied_peau_mate:',
        'pt': u':pé_pele_morena_escura:',
        'it': ':piede_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B6\U0001F3FC': { # 🦶🏼
        'en' : ':foot_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mittelhelle_hautfarbe:',
        'es': ':pie_tono_de_piel_claro_medio:',
        'fr': ':pied_peau_moyennement_claire:',
        'pt': u':pé_pele_morena_clara:',
        'it': ':piede_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B6\U0001F3FD': { # 🦶🏽
        'en' : ':foot_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':fuß_mittlere_hautfarbe:',
        'es': ':pie_tono_de_piel_medio:',
        'fr': u':pied_peau_légèrement_mate:',
        'pt': u':pé_pele_morena:',
        'it': ':piede_carnagione_olivastra:'
    },
    u'\U0001F463': { # 👣
        'en' : ':footprints:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':fußabdrücke:',
        'es': ':huellas_de_pies:',
        'fr': ':traces_de_pas:',
        'pt': ':pegadas:',
        'it': ':impronta_di_piedi:'
    },
    u'\U0001F374': { # 🍴
        'en' : ':fork_and_knife:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':messer_und_gabel:',
        'es': ':tenedor_y_cuchillo:',
        'fr': ':fourchette_et_couteau:',
        'pt': ':garfo_e_faca:',
        'it': ':forchetta_e_coltello:'
    },
    u'\U0001F37D\U0000FE0F': { # 🍽️
        'en' : ':fork_and_knife_with_plate:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':plate_with_cutlery:'],
        'variant': True,
        'de': ':teller_mit_messer_und_gabel:',
        'es': ':cuchillo_y_tenedor_con_un_plato:',
        'fr': ':fourchette_et_couteau_avec_assiette:',
        'pt': ':prato_com_talheres:',
        'it': ':piatto_e_posate:'
    },
    u'\U0001F37D': { # 🍽
        'en' : ':fork_and_knife_with_plate:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':plate_with_cutlery:'],
        'variant': True,
        'de': ':teller_mit_messer_und_gabel:',
        'es': ':cuchillo_y_tenedor_con_un_plato:',
        'fr': ':fourchette_et_couteau_avec_assiette:',
        'pt': ':prato_com_talheres:',
        'it': ':piatto_e_posate:'
    },
    u'\U0001F960': { # 🥠
        'en' : ':fortune_cookie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':glückskeks:',
        'es': ':galleta_de_la_fortuna:',
        'fr': ':biscuit_chinois:',
        'pt': ':biscoito_da_sorte:',
        'it': ':biscotto_della_fortuna:'
    },
    u'\U000026F2': { # ⛲
        'en' : ':fountain:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':springbrunnen:',
        'es': ':fuente:',
        'fr': ':fontaine:',
        'pt': ':fonte:',
        'it': ':fontana:'
    },
    u'\U0001F58B\U0000FE0F': { # 🖋️
        'en' : ':fountain_pen:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':lower_left_fountain_pen:'],
        'variant': True,
        'de': u':füllhalter:',
        'es': u':estilográfica:',
        'fr': ':stylo_plume:',
        'pt': ':caneta_tinteiro:',
        'it': ':penna_stilografica:'
    },
    u'\U0001F58B': { # 🖋
        'en' : ':fountain_pen:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':lower_left_fountain_pen:'],
        'variant': True,
        'de': u':füllhalter:',
        'es': u':estilográfica:',
        'fr': ':stylo_plume:',
        'pt': ':caneta_tinteiro:',
        'it': ':penna_stilografica:'
    },
    u'\U0001F55F': { # 🕟
        'en' : ':four-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock430:'],
        'variant': True,
        'de': ':4.30_uhr:',
        'es': ':cuatro_y_media:',
        'fr': ':quatre_heures_et_demie:',
        'pt': ':quatro_e_meia:',
        'it': ':ore_quattro_e_mezza:'
    },
    u'\U0001F340': { # 🍀
        'en' : ':four_leaf_clover:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':glücksklee:',
        'es': u':trébol_de_cuatro_hojas:',
        'fr': u':trèfle_à_quatre_feuilles:',
        'pt': ':trevo_de_quatro_folhas:',
        'it': ':quadrifoglio:'
    },
    u'\U0001F553': { # 🕓
        'en' : u':four_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock4:'],
        'variant': True,
        'de': ':4.00_uhr:',
        'es': ':4_en_punto:',
        'fr': ':quatre_heures:',
        'pt': ':4_horas:',
        'it': ':ore_quattro:'
    },
    u'\U0001F98A': { # 🦊
        'en' : ':fox:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':fox_face:'],
        'de': ':fuchs:',
        'es': ':zorro:',
        'fr': ':renard:',
        'pt': ':rosto_de_raposa:',
        'it': ':volpe:'
    },
    u'\U0001F5BC\U0000FE0F': { # 🖼️
        'en' : ':framed_picture:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':frame_with_picture:'],
        'variant': True,
        'de': ':gerahmtes_bild:',
        'es': ':cuadro_enmarcado:',
        'fr': ':cadre_avec_image:',
        'pt': ':quadro_emoldurado:',
        'it': ':quadro_incorniciato:'
    },
    u'\U0001F5BC': { # 🖼
        'en' : ':framed_picture:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':frame_with_picture:'],
        'variant': True,
        'de': ':gerahmtes_bild:',
        'es': ':cuadro_enmarcado:',
        'fr': ':cadre_avec_image:',
        'pt': ':quadro_emoldurado:',
        'it': ':quadro_incorniciato:'
    },
    u'\U0001F35F': { # 🍟
        'en' : ':french_fries:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fries:'],
        'de': ':pommes_frites:',
        'es': ':patatas_fritas:',
        'fr': ':frites:',
        'pt': ':batata_frita:',
        'it': ':patatine:'
    },
    u'\U0001F364': { # 🍤
        'en' : ':fried_shrimp:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':frittierte_garnele:',
        'es': ':gamba_frita:',
        'fr': ':beignet_de_crevette:',
        'pt': u':camarão_frito:',
        'it': ':gambero_fritto:'
    },
    u'\U0001F438': { # 🐸
        'en' : ':frog:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':frosch:',
        'es': ':rana:',
        'fr': ':grenouille:',
        'pt': ':sapo:',
        'it': ':rana:'
    },
    u'\U0001F425': { # 🐥
        'en' : ':front-facing_baby_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hatched_chick:'],
        'de': u':küken_von_vorne:',
        'es': ':pollito_de_frente:',
        'fr': ':poussin_de_face:',
        'pt': ':pintinho_de_frente:',
        'it': ':pulcino_visto_di_fronte:'
    },
    u'\U00002639\U0000FE0F': { # ☹️
        'en' : ':frowning_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':white_frowning_face:'],
        'variant': True,
        'de': u':düsteres_gesicht:',
        'es': u':cara_con_el_ceño_fruncido:',
        'fr': u':visage_mécontent:',
        'pt': ':rosto_descontente:',
        'it': ':faccina_imbronciata:'
    },
    u'\U00002639': { # ☹
        'en' : ':frowning_face:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':white_frowning_face:'],
        'variant': True,
        'de': u':düsteres_gesicht:',
        'es': u':cara_con_el_ceño_fruncido:',
        'fr': u':visage_mécontent:',
        'pt': ':rosto_descontente:',
        'it': ':faccina_imbronciata:'
    },
    u'\U0001F626': { # 😦
        'en' : ':frowning_face_with_open_mouth:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':frowning:'],
        'de': ':entsetztes_gesicht:',
        'es': u':cara_con_el_ceño_fruncido_y_la_boca_abierta:',
        'fr': u':visage_mécontent_avec_bouche_ouverte:',
        'pt': ':rosto_franzido_com_boca_aberta:',
        'it': ':faccina_imbronciata_con_bocca_aperta:'
    },
    u'\U000026FD': { # ⛽
        'en' : ':fuel_pump:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fuelpump:'],
        'variant': True,
        'de': u':tanksäule:',
        'es': ':surtidor_de_gasolina:',
        'fr': u':pompe_à_essence:',
        'pt': ':posto_de_gasolina:',
        'it': ':stazione_di_servizio:'
    },
    u'\U0001F315': { # 🌕
        'en' : ':full_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':vollmond:',
        'es': ':luna_llena:',
        'fr': ':pleine_lune:',
        'pt': ':lua_cheia:',
        'it': ':luna_piena:'
    },
    u'\U0001F31D': { # 🌝
        'en' : ':full_moon_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':full_moon_with_face:'],
        'de': ':vollmond_mit_gesicht:',
        'es': ':luna_llena_con_cara:',
        'fr': ':pleine_lune_avec_visage:',
        'pt': ':rosto_da_lua_cheia:',
        'it': ':faccina_luna_piena:'
    },
    u'\U000026B1\U0000FE0F': { # ⚱️
        'en' : ':funeral_urn:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':urne:',
        'es': ':urna_funeraria:',
        'fr': u':urne_funéraire:',
        'pt': u':urna_funerária:',
        'it': ':urna_funeraria:'
    },
    u'\U000026B1': { # ⚱
        'en' : ':funeral_urn:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':urne:',
        'es': ':urna_funeraria:',
        'fr': u':urne_funéraire:',
        'pt': u':urna_funerária:',
        'it': ':urna_funeraria:'
    },
    u'\U0001F3B2': { # 🎲
        'en' : ':game_die:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':spielwürfel:',
        'es': ':dado:',
        'fr': u':dés:',
        'pt': ':jogo_de_dado:',
        'it': ':dado:'
    },
    u'\U0001F9C4': { # 🧄
        'en' : ':garlic:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':knoblauch:',
        'es': ':ajo:',
        'fr': ':ail:',
        'pt': ':alho:',
        'it': ':aglio:'
    },
    u'\U00002699\U0000FE0F': { # ⚙️
        'en' : ':gear:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':zahnrad:',
        'es': ':engranaje:',
        'fr': u':roue_dentée:',
        'pt': ':engrenagem:',
        'it': ':ingranaggio:'
    },
    u'\U00002699': { # ⚙
        'en' : ':gear:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':zahnrad:',
        'es': ':engranaje:',
        'fr': u':roue_dentée:',
        'pt': ':engrenagem:',
        'it': ':ingranaggio:'
    },
    u'\U0001F48E': { # 💎
        'en' : ':gem_stone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':gem:'],
        'de': ':edelstein:',
        'es': ':piedra_preciosa:',
        'fr': u':pierre_précieuse:',
        'pt': ':pedra_preciosa:',
        'it': ':gemma:'
    },
    u'\U0001F9DE': { # 🧞
        'en' : ':genie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':flaschengeist:',
        'es': ':genio:',
        'fr': u':génie:',
        'pt': u':gênio:',
        'it': ':genio:'
    },
    u'\U0001F47B': { # 👻
        'en' : ':ghost:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':gespenst:',
        'es': ':fantasma:',
        'fr': u':fantôme:',
        'pt': ':fantasma:',
        'it': ':fantasma:'
    },
    u'\U0001F992': { # 🦒
        'en' : ':giraffe:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':giraffe:',
        'es': ':jirafa:',
        'fr': ':girafe:',
        'pt': ':girafa:',
        'it': ':giraffa:'
    },
    u'\U0001F467': { # 👧
        'en' : ':girl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':mädchen:',
        'es': u':niña:',
        'fr': ':fille:',
        'pt': ':menina:',
        'it': ':bambina:'
    },
    u'\U0001F467\U0001F3FF': { # 👧🏿
        'en' : ':girl_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_dunkle_hautfarbe:',
        'es': u':niña_tono_de_piel_oscuro:',
        'fr': u':fille_peau_foncée:',
        'pt': ':menina_pele_escura:',
        'it': ':bambina_carnagione_scura:'
    },
    u'\U0001F467\U0001F3FB': { # 👧🏻
        'en' : ':girl_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_helle_hautfarbe:',
        'es': u':niña_tono_de_piel_claro:',
        'fr': ':fille_peau_claire:',
        'pt': ':menina_pele_clara:',
        'it': ':bambina_carnagione_chiara:'
    },
    u'\U0001F467\U0001F3FE': { # 👧🏾
        'en' : ':girl_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mitteldunkle_hautfarbe:',
        'es': u':niña_tono_de_piel_oscuro_medio:',
        'fr': ':fille_peau_mate:',
        'pt': ':menina_pele_morena_escura:',
        'it': ':bambina_carnagione_abbastanza_scura:'
    },
    u'\U0001F467\U0001F3FC': { # 👧🏼
        'en' : ':girl_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mittelhelle_hautfarbe:',
        'es': u':niña_tono_de_piel_claro_medio:',
        'fr': ':fille_peau_moyennement_claire:',
        'pt': ':menina_pele_morena_clara:',
        'it': ':bambina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F467\U0001F3FD': { # 👧🏽
        'en' : ':girl_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':mädchen_mittlere_hautfarbe:',
        'es': u':niña_tono_de_piel_medio:',
        'fr': u':fille_peau_légèrement_mate:',
        'pt': ':menina_pele_morena:',
        'it': ':bambina_carnagione_olivastra:'
    },
    u'\U0001F95B': { # 🥛
        'en' : ':glass_of_milk:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':milk_glass:'],
        'de': ':glas_milch:',
        'es': ':vaso_de_leche:',
        'fr': ':verre_de_lait:',
        'pt': ':copo_de_leite:',
        'it': ':bicchiere_di_latte:'
    },
    u'\U0001F453': { # 👓
        'en' : ':glasses:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':eyeglasses:'],
        'variant': True,
        'de': ':brille:',
        'es': ':gafas:',
        'fr': ':lunettes_de_vue:',
        'pt': u':óculos:',
        'it': ':occhiali_da_vista:'
    },
    u'\U0001F30E': { # 🌎
        'en' : ':globe_showing_Americas:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':earth_americas:'],
        'variant': True,
        'de': ':globus_mit_amerika:',
        'es': u':globo_terráqueo_mostrando_américa:',
        'fr': u':globe_tourné_sur_les_amériques:',
        'pt': u':globo_mostrando_as_américas:',
        'it': ':america:'
    },
    u'\U0001F30F': { # 🌏
        'en' : ':globe_showing_Asia-Australia:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':earth_asia:'],
        'variant': True,
        'de': ':globus_mit_asien_und_australien:',
        'es': u':globo_terráqueo_mostrando_asia_y_australia:',
        'fr': u':globe_tourné_sur_l’asie_et_l’australie:',
        'pt': u':globo_mostrando_ásia_e_oceania:',
        'it': ':asia_e_australia:'
    },
    u'\U0001F30D': { # 🌍
        'en' : ':globe_showing_Europe-Africa:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':earth_africa:'],
        'variant': True,
        'de': ':globus_mit_europa_und_afrika:',
        'es': u':globo_terráqueo_mostrando_europa_y_áfrica:',
        'fr': u':globe_tourné_sur_l’afrique_et_l’europe:',
        'pt': u':globo_mostrando_europa_e_áfrica:',
        'it': ':europa_e_africa:'
    },
    u'\U0001F310': { # 🌐
        'en' : ':globe_with_meridians:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':globus_mit_meridianen:',
        'es': u':globo_terráqueo_con_meridianos:',
        'fr': u':globe_avec_méridiens:',
        'pt': ':globo_com_meridianos:',
        'it': ':globo_con_meridiani:'
    },
    u'\U0001F9E4': { # 🧤
        'en' : ':gloves:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':handschuhe:',
        'es': ':guantes:',
        'fr': ':gants:',
        'pt': ':luvas:',
        'it': ':guanti:'
    },
    u'\U0001F31F': { # 🌟
        'en' : ':glowing_star:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':star2:'],
        'de': ':funkelnder_stern:',
        'es': ':estrella_brillante:',
        'fr': u':étoile_brillante:',
        'pt': ':estrela_brilhante:',
        'it': ':stella_che_brilla:'
    },
    u'\U0001F945': { # 🥅
        'en' : ':goal_net:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tor:',
        'es': u':portería:',
        'fr': ':cage:',
        'pt': ':gol:',
        'it': ':porta_da_calcio:'
    },
    u'\U0001F410': { # 🐐
        'en' : ':goat:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ziege:',
        'es': ':cabra:',
        'fr': u':chèvre:',
        'pt': ':cabra:',
        'it': ':capra:'
    },
    u'\U0001F47A': { # 👺
        'en' : ':goblin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':japanese_goblin:'],
        'de': ':kobold:',
        'es': u':demonio_japonés_tengu:',
        'fr': ':monstre_japonais:',
        'pt': u':duende_japonês:',
        'it': ':goblin:'
    },
    u'\U0001F97D': { # 🥽
        'en' : ':goggles:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':schutzbrille:',
        'es': u':gafas_de_protección:',
        'fr': ':lunettes:',
        'pt': u':óculos_de_proteção:',
        'it': ':maschera:'
    },
    u'\U0001F98D': { # 🦍
        'en' : ':gorilla:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':gorilla:',
        'es': ':gorila:',
        'fr': ':gorille:',
        'pt': ':gorila:',
        'it': ':gorilla:'
    },
    u'\U0001F393': { # 🎓
        'en' : ':graduation_cap:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mortar_board:'],
        'variant': True,
        'de': ':doktorhut:',
        'es': ':birrete:',
        'fr': ':toque_universitaire:',
        'pt': u':chapéu_de_formatura:',
        'it': ':cappello_da_laureato:'
    },
    u'\U0001F347': { # 🍇
        'en' : ':grapes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':trauben:',
        'es': ':uvas:',
        'fr': ':raisin:',
        'pt': ':uvas:',
        'it': ':uva:'
    },
    u'\U0001F34F': { # 🍏
        'en' : ':green_apple:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grüner_apfel:',
        'es': ':manzana_verde:',
        'fr': ':pomme_verte:',
        'pt': u':maçã_verde:',
        'it': ':mela_verde:'
    },
    u'\U0001F4D7': { # 📗
        'en' : ':green_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grünes_buch:',
        'es': ':libro_verde:',
        'fr': ':livre_vert:',
        'pt': ':livro_verde:',
        'it': ':libro_verde:'
    },
    u'\U0001F7E2': { # 🟢
        'en' : ':green_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':grüner_punkt:',
        'es': u':círculo_verde:',
        'fr': ':disque_vert:',
        'pt': u':círculo_verde:',
        'it': ':cerchio_verde:'
    },
    u'\U0001F49A': { # 💚
        'en' : ':green_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':grünes_herz:',
        'es': u':corazón_verde:',
        'fr': u':cœur_vert:',
        'pt': u':coração_verde:',
        'it': ':cuore_verde:'
    },
    u'\U0001F957': { # 🥗
        'en' : ':green_salad:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':salat:',
        'es': ':ensalada:',
        'fr': ':salade_verte:',
        'pt': ':salada_verde:',
        'it': ':insalata_verde:'
    },
    u'\U0001F7E9': { # 🟩
        'en' : ':green_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':grünes_quadrat:',
        'es': ':cuadrado_verde:',
        'fr': u':carré_vert:',
        'pt': ':quadrado_verde:',
        'it': ':quadrato_verde:'
    },
    u'\U0001F62C': { # 😬
        'en' : ':grimacing_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':grimacing:'],
        'de': ':grimassen_schneidendes_gesicht:',
        'es': ':cara_haciendo_una_mueca:',
        'fr': u':visage_grimaçant:',
        'pt': ':rosto_expressando_desagrado:',
        'it': ':faccina_con_smorfia:'
    },
    u'\U0001F63A': { # 😺
        'en' : ':grinning_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smiley_cat:'],
        'de': ':grinsende_katze:',
        'es': ':gato_sonriendo:',
        'fr': ':chat_qui_sourit:',
        'pt': ':rosto_de_gato_sorrindo:',
        'it': ':gatto_che_sorride:'
    },
    u'\U0001F638': { # 😸
        'en' : ':grinning_cat_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smile_cat:'],
        'de': ':grinsende_katze_mit_lachenden_augen:',
        'es': ':gato_sonriendo_con_ojos_sonrientes:',
        'fr': ':chat_qui_sourit_avec_des_yeux_rieurs:',
        'pt': ':rosto_de_gato_sorrindo_com_olhos_sorridentes:',
        'it': ':gatto_che_sogghigna:'
    },
    u'\U0001F600': { # 😀
        'en' : ':grinning_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':grinning:'],
        'de': ':grinsendes_gesicht:',
        'es': ':cara_sonriendo:',
        'fr': ':visage_rieur:',
        'pt': ':rosto_risonho:',
        'it': ':faccina_con_un_gran_sorriso:'
    },
    u'\U0001F603': { # 😃
        'en' : ':grinning_face_with_big_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smiley:'],
        'de': u':grinsendes_gesicht_mit_großen_augen:',
        'es': ':cara_sonriendo_con_ojos_grandes:',
        'fr': ':visage_souriant_avec_de_grands_yeux:',
        'pt': ':rosto_risonho_com_olhos_bem_abertos:',
        'it': ':faccina_con_un_gran_sorriso_e_occhi_spalancati:'
    },
    u'\U0001F604': { # 😄
        'en' : ':grinning_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':smile:'],
        'de': ':grinsendes_gesicht_mit_lachenden_augen:',
        'es': ':cara_sonriendo_con_ojos_sonrientes:',
        'fr': u':visage_très_souriant_aux_yeux_rieurs:',
        'pt': ':rosto_risonho_com_olhos_sorridentes:',
        'it': ':faccina_con_sorriso_e_occhi_sorridenti:'
    },
    u'\U0001F605': { # 😅
        'en' : ':grinning_face_with_sweat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sweat_smile:'],
        'de': u':grinsendes_gesicht_mit_schweißtropfen:',
        'es': u':cara_sonriendo_con_sudor_frío:',
        'fr': ':visage_souriant_avec_une_goutte_de_sueur:',
        'pt': ':rosto_risonho_com_gota_de_suor:',
        'it': ':faccina_con_un_gran_sorriso_e_goccia_di_sudore:'
    },
    u'\U0001F606': { # 😆
        'en' : ':grinning_squinting_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':satisfied:', ':laughing:'],
        'de': ':grinsegesicht_mit_zugekniffenen_augen:',
        'es': ':cara_sonriendo_con_los_ojos_cerrados:',
        'fr': u':visage_souriant_avec_yeux_plissés:',
        'pt': ':rosto_risonho_com_olhos_semicerrados:',
        'it': ':sorriso_a_bocca_aperta_con_occhi_chiusi:'
    },
    u'\U0001F497': { # 💗
        'en' : ':growing_heart:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heartpulse:'],
        'de': ':wachsendes_herz:',
        'es': u':corazón_creciente:',
        'fr': u':cœur_grandissant:',
        'pt': u':coração_crescendo:',
        'it': ':cuore_che_cresce:'
    },
    u'\U0001F482': { # 💂
        'en' : ':guard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':guardsman:'],
        'de': ':wache:',
        'es': ':guardia:',
        'fr': ':garde:',
        'pt': ':guarda:',
        'it': ':guardia:'
    },
    u'\U0001F482\U0001F3FF': { # 💂🏿
        'en' : ':guard_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':wachmann/wachfrau_dunkle_hautfarbe:',
        'es': ':guardia_tono_de_piel_oscuro:',
        'fr': u':garde_peau_foncée:',
        'pt': ':guarda_pele_escura:',
        'it': ':guardia_carnagione_scura:'
    },
    u'\U0001F482\U0001F3FB': { # 💂🏻
        'en' : ':guard_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':wachmann/wachfrau_helle_hautfarbe:',
        'es': ':guardia_tono_de_piel_claro:',
        'fr': ':garde_peau_claire:',
        'pt': ':guarda_pele_clara:',
        'it': ':guardia_carnagione_chiara:'
    },
    u'\U0001F482\U0001F3FE': { # 💂🏾
        'en' : ':guard_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':wachmann/wachfrau_mitteldunkle_hautfarbe:',
        'es': ':guardia_tono_de_piel_oscuro_medio:',
        'fr': ':garde_peau_mate:',
        'pt': ':guarda_pele_morena_escura:',
        'it': ':guardia_carnagione_abbastanza_scura:'
    },
    u'\U0001F482\U0001F3FC': { # 💂🏼
        'en' : ':guard_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':wachmann/wachfrau_mittelhelle_hautfarbe:',
        'es': ':guardia_tono_de_piel_claro_medio:',
        'fr': ':garde_peau_moyennement_claire:',
        'pt': ':guarda_pele_morena_clara:',
        'it': ':guardia_carnagione_abbastanza_chiara:'
    },
    u'\U0001F482\U0001F3FD': { # 💂🏽
        'en' : ':guard_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':wachmann/wachfrau_mittlere_hautfarbe:',
        'es': ':guardia_tono_de_piel_medio:',
        'fr': u':garde_peau_légèrement_mate:',
        'pt': ':guarda_pele_morena:',
        'it': ':guardia_carnagione_olivastra:'
    },
    u'\U0001F9AE': { # 🦮
        'en' : ':guide_dog:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':blindenhund:',
        'es': u':perro_guía:',
        'fr': ':chien_guide:',
        'pt': u':cão-guia:',
        'it': ':cane_guida:'
    },
    u'\U0001F3B8': { # 🎸
        'en' : ':guitar:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':gitarre:',
        'es': ':guitarra:',
        'fr': ':guitare:',
        'pt': ':guitarra:',
        'it': ':chitarra:'
    },
    u'\U0001F354': { # 🍔
        'en' : ':hamburger:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hamburger:',
        'es': ':hamburguesa:',
        'fr': ':hamburger:',
        'pt': u':hambúrguer:',
        'it': ':hamburger:'
    },
    u'\U0001F528': { # 🔨
        'en' : ':hammer:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hammer:',
        'es': ':martillo:',
        'fr': ':marteau:',
        'pt': ':martelo:',
        'it': ':martello:'
    },
    u'\U00002692\U0000FE0F': { # ⚒️
        'en' : ':hammer_and_pick:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':hammer_und_pickel:',
        'es': ':martillo_y_pico:',
        'fr': ':marteau_et_pioche:',
        'pt': ':martelo_e_picareta:',
        'it': ':piccone_e_martello:'
    },
    u'\U00002692': { # ⚒
        'en' : ':hammer_and_pick:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':hammer_und_pickel:',
        'es': ':martillo_y_pico:',
        'fr': ':marteau_et_pioche:',
        'pt': ':martelo_e_picareta:',
        'it': ':piccone_e_martello:'
    },
    u'\U0001F6E0\U0000FE0F': { # 🛠️
        'en' : ':hammer_and_wrench:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':hammer_und_schraubenschlüssel:',
        'es': ':martillo_y_llave_inglesa:',
        'fr': u':marteau_et_clé_à_molette:',
        'pt': ':martelo_e_chave-inglesa:',
        'it': ':martello_e_chiave_inglese:'
    },
    u'\U0001F6E0': { # 🛠
        'en' : ':hammer_and_wrench:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':hammer_und_schraubenschlüssel:',
        'es': ':martillo_y_llave_inglesa:',
        'fr': u':marteau_et_clé_à_molette:',
        'pt': ':martelo_e_chave-inglesa:',
        'it': ':martello_e_chiave_inglese:'
    },
    u'\U0001FAAC': { # 🪬
        'en' : ':hamsa:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':hamsa:',
        'es': ':hamsa:',
        'fr': ':main_de_fatma:',
        'pt': u':hamsá:',
        'it': ':mano_di_fatima:'
    },
    u'\U0001F439': { # 🐹
        'en' : ':hamster:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hamster:',
        'es': u':hámster:',
        'fr': ':hamster:',
        'pt': ':rosto_de_hamster:',
        'it': ':criceto:'
    },
    u'\U0001F590\U0000FE0F': { # 🖐️
        'en' : ':hand_with_fingers_splayed:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': ':hand_mit_gespreizten_fingern:',
        'es': ':mano_abierta:',
        'fr': u':main_levée_doigts_écartés:',
        'pt': u':mão_aberta_com_os_dedos_separados:',
        'it': ':mano_aperta:'
    },
    u'\U0001F590': { # 🖐
        'en' : ':hand_with_fingers_splayed:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':raised_hand_with_fingers_splayed:'],
        'variant': True,
        'de': ':hand_mit_gespreizten_fingern:',
        'es': ':mano_abierta:',
        'fr': u':main_levée_doigts_écartés:',
        'pt': u':mão_aberta_com_os_dedos_separados:',
        'it': ':mano_aperta:'
    },
    u'\U0001F590\U0001F3FF': { # 🖐🏿
        'en' : ':hand_with_fingers_splayed_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hand_mit_gespreizten_fingern_dunkle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_oscuro:',
        'fr': u':main_levée_doigts_écartés_peau_foncée:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_escura:',
        'it': ':mano_aperta_carnagione_scura:'
    },
    u'\U0001F590\U0001F3FB': { # 🖐🏻
        'en' : ':hand_with_fingers_splayed_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hand_mit_gespreizten_fingern_helle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_claro:',
        'fr': u':main_levée_doigts_écartés_peau_claire:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_clara:',
        'it': ':mano_aperta_carnagione_chiara:'
    },
    u'\U0001F590\U0001F3FE': { # 🖐🏾
        'en' : ':hand_with_fingers_splayed_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hand_mit_gespreizten_fingern_mitteldunkle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_oscuro_medio:',
        'fr': u':main_levée_doigts_écartés_peau_mate:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena_escura:',
        'it': ':mano_aperta_carnagione_abbastanza_scura:'
    },
    u'\U0001F590\U0001F3FC': { # 🖐🏼
        'en' : ':hand_with_fingers_splayed_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hand_mit_gespreizten_fingern_mittelhelle_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_claro_medio:',
        'fr': u':main_levée_doigts_écartés_peau_moyennement_claire:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena_clara:',
        'it': ':mano_aperta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F590\U0001F3FD': { # 🖐🏽
        'en' : ':hand_with_fingers_splayed_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hand_mit_gespreizten_fingern_mittlere_hautfarbe:',
        'es': ':mano_abierta_tono_de_piel_medio:',
        'fr': u':main_levée_doigts_écartés_peau_légèrement_mate:',
        'pt': u':mão_aberta_com_os_dedos_separados_pele_morena:',
        'it': ':mano_aperta_carnagione_olivastra:'
    },
    u'\U0001FAF0': { # 🫰
        'en' : ':hand_with_index_finger_and_thumb_crossed:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':hand_mit_gekreuztem_zeigefinger_und_daumen:',
        'es': u':mano_con_dedo_índice_y_pulgar_cruzados:',
        'fr': u':main_avec_index_et_pouce_croisés:',
        'pt': u':mão_com_dedo_indicador_e_polegar_cruzados:',
        'it': ':mano_con_indice_e_pollice_incrociati:'
    },
    u'\U0001FAF0\U0001F3FF': { # 🫰🏿
        'en' : ':hand_with_index_finger_and_thumb_crossed_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FB': { # 🫰🏻
        'en' : ':hand_with_index_finger_and_thumb_crossed_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FE': { # 🫰🏾
        'en' : ':hand_with_index_finger_and_thumb_crossed_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FC': { # 🫰🏼
        'en' : ':hand_with_index_finger_and_thumb_crossed_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF0\U0001F3FD': { # 🫰🏽
        'en' : ':hand_with_index_finger_and_thumb_crossed_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F45C': { # 👜
        'en' : ':handbag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':handtasche:',
        'es': ':bolso:',
        'fr': u':sac_à_main:',
        'pt': u':bolsa_de_mão:',
        'it': ':borsa:'
    },
    u'\U0001F91D': { # 🤝
        'en' : ':handshake:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':handschlag:',
        'es': u':apretón_de_manos:',
        'fr': u':poignée_de_main:',
        'pt': u':aperto_de_mãos:',
        'it': ':stretta_di_mano:'
    },
    u'\U0001F91D\U0001F3FF': { # 🤝🏿
        'en' : ':handshake_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FB': { # 🫱🏿‍🫲🏻
        'en' : ':handshake_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FE': { # 🫱🏿‍🫲🏾
        'en' : ':handshake_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FC': { # 🫱🏿‍🫲🏼
        'en' : ':handshake_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FF\U0000200D\U0001FAF2\U0001F3FD': { # 🫱🏿‍🫲🏽
        'en' : ':handshake_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FB': { # 🤝🏻
        'en' : ':handshake_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FF': { # 🫱🏻‍🫲🏿
        'en' : ':handshake_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FE': { # 🫱🏻‍🫲🏾
        'en' : ':handshake_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FC': { # 🫱🏻‍🫲🏼
        'en' : ':handshake_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FB\U0000200D\U0001FAF2\U0001F3FD': { # 🫱🏻‍🫲🏽
        'en' : ':handshake_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FE': { # 🤝🏾
        'en' : ':handshake_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FF': { # 🫱🏾‍🫲🏿
        'en' : ':handshake_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FB': { # 🫱🏾‍🫲🏻
        'en' : ':handshake_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FC': { # 🫱🏾‍🫲🏼
        'en' : ':handshake_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FE\U0000200D\U0001FAF2\U0001F3FD': { # 🫱🏾‍🫲🏽
        'en' : ':handshake_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FC': { # 🤝🏼
        'en' : ':handshake_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FF': { # 🫱🏼‍🫲🏿
        'en' : ':handshake_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FB': { # 🫱🏼‍🫲🏻
        'en' : ':handshake_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FE': { # 🫱🏼‍🫲🏾
        'en' : ':handshake_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FC\U0000200D\U0001FAF2\U0001F3FD': { # 🫱🏼‍🫲🏽
        'en' : ':handshake_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F91D\U0001F3FD': { # 🤝🏽
        'en' : ':handshake_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FF': { # 🫱🏽‍🫲🏿
        'en' : ':handshake_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FB': { # 🫱🏽‍🫲🏻
        'en' : ':handshake_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FE': { # 🫱🏽‍🫲🏾
        'en' : ':handshake_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF1\U0001F3FD\U0000200D\U0001FAF2\U0001F3FC': { # 🫱🏽‍🫲🏼
        'en' : ':handshake_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F423': { # 🐣
        'en' : ':hatching_chick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schlüpfendes_küken:',
        'es': u':pollito_rompiendo_el_cascarón:',
        'fr': u':poussin_qui_éclôt:',
        'pt': ':pintinho_chocando:',
        'it': ':pulcino_che_nasce:'
    },
    u'\U0001F3A7': { # 🎧
        'en' : ':headphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':headphones:'],
        'variant': True,
        'de': u':kopfhörer:',
        'es': ':auricular:',
        'fr': ':casque:',
        'pt': ':fones_de_ouvido:',
        'it': ':cuffie:'
    },
    u'\U0001FAA6': { # 🪦
        'en' : ':headstone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':grabstein:',
        'es': u':lápida:',
        'fr': ':pierre_tombale:',
        'pt': u':lápide:',
        'it': ':lapide:'
    },
    u'\U0001F9D1\U0000200D\U00002695\U0000FE0F': { # 🧑‍⚕️
        'en' : ':health_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin:',
        'es': ':profesional_sanitario:',
        'fr': u':professionnel_de_la_santé_(tous_genres):',
        'pt': u':profissional_de_saúde:',
        'it': u':persona_che_lavora_nella_sanità:'
    },
    u'\U0001F9D1\U0000200D\U00002695': { # 🧑‍⚕
        'en' : ':health_worker:',
        'status' : minimally_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin:',
        'es': ':profesional_sanitario:',
        'fr': u':professionnel_de_la_santé_(tous_genres):',
        'pt': u':profissional_de_saúde:',
        'it': u':persona_che_lavora_nella_sanità:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002695\U0000FE0F': { # 🧑🏿‍⚕️
        'en' : ':health_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_dunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_foncée:',
        'pt': u':profissional_de_saúde_pele_escura:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002695': { # 🧑🏿‍⚕
        'en' : ':health_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002695\U0000FE0F': { # 🧑🏻‍⚕️
        'en' : ':health_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_helle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_claire:',
        'pt': u':profissional_de_saúde_pele_clara:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002695': { # 🧑🏻‍⚕
        'en' : ':health_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002695\U0000FE0F': { # 🧑🏾‍⚕️
        'en' : ':health_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mitteldunkle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_oscuro_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_mate:',
        'pt': u':profissional_de_saúde_pele_morena_escura:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002695': { # 🧑🏾‍⚕
        'en' : ':health_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002695\U0000FE0F': { # 🧑🏼‍⚕️
        'en' : ':health_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mittelhelle_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_claro_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_moyennement_claire:',
        'pt': u':profissional_de_saúde_pele_morena_clara:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002695': { # 🧑🏼‍⚕
        'en' : ':health_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002695\U0000FE0F': { # 🧑🏽‍⚕️
        'en' : ':health_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':arzt/ärztin_mittlere_hautfarbe:',
        'es': ':profesional_sanitario_tono_de_piel_medio:',
        'fr': u':professionnel_de_la_santé_(tous_genres)_peau_légèrement_mate:',
        'pt': u':profissional_de_saúde_pele_morena:',
        'it': u':persona_che_lavora_nella_sanità_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002695': { # 🧑🏽‍⚕
        'en' : ':health_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F649': { # 🙉
        'en' : ':hear-no-evil_monkey:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hear_no_evil:'],
        'de': ':sich_die_ohren_zuhaltendes_affengesicht:',
        'es': u':mono_con_los_oídos_tapados:',
        'fr': ':singe_ne_rien_entendre:',
        'pt': u':macaco_que_não_ouve_nada:',
        'it': ':non_sento:'
    },
    u'\U0001F49F': { # 💟
        'en' : ':heart_decoration:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':herzdekoration:',
        'es': u':adorno_de_corazón:',
        'fr': u':décoration_avec_cœur:',
        'pt': u':coração_decorativo:',
        'it': ':decorazione_con_cuore:'
    },
    u'\U00002763\U0000FE0F': { # ❣️
        'en' : ':heart_exclamation:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':heavy_heart_exclamation:', ':heavy_heart_exclamation_mark_ornament:'],
        'variant': True,
        'de': ':herz_als_ausrufezeichen:',
        'es': u':exclamación_de_corazón:',
        'fr': u':cœur_point_d’exclamation:',
        'pt': u':exclamação_de_coração:',
        'it': ':punto_esclamativo_a_cuore:'
    },
    u'\U00002763': { # ❣
        'en' : ':heart_exclamation:',
        'status' : unqualified,
        'E' : 1,
        'alias' : [':heavy_heart_exclamation:', ':heavy_heart_exclamation_mark_ornament:'],
        'variant': True,
        'de': ':herz_als_ausrufezeichen:',
        'es': u':exclamación_de_corazón:',
        'fr': u':cœur_point_d’exclamation:',
        'pt': u':exclamação_de_coração:',
        'it': ':punto_esclamativo_a_cuore:'
    },
    u'\U0001FAF6': { # 🫶
        'en' : ':heart_hands:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':hände_die_herz_bilden:',
        'es': u':manos_formando_un_corazón:',
        'fr': u':mains_qui_forment_un_cœur:',
        'pt': u':mãos_de_coração:',
        'it': ':mani_a_cuore:'
    },
    u'\U0001FAF6\U0001F3FF': { # 🫶🏿
        'en' : ':heart_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FB': { # 🫶🏻
        'en' : ':heart_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FE': { # 🫶🏾
        'en' : ':heart_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FC': { # 🫶🏼
        'en' : ':heart_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF6\U0001F3FD': { # 🫶🏽
        'en' : ':heart_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U00002764\U0000FE0F\U0000200D\U0001F525': { # ❤️‍🔥
        'en' : ':heart_on_fire:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U00002764\U0000200D\U0001F525': { # ❤‍🔥
        'en' : ':heart_on_fire:',
        'status' : unqualified,
        'E' : 13.1,
        'de': ':herz_in_flammen:',
        'es': u':corazón_en_llamas:',
        'fr': u':cœur_enflammé:',
        'pt': u':coração_em_chamas:',
        'it': ':cuore_in_fiamme:'
    },
    u'\U00002665\U0000FE0F': { # ♥️
        'en' : ':heart_suit:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hearts:'],
        'variant': True,
        'de': ':herz:',
        'es': ':palo_de_corazones:',
        'fr': u':cœur_cartes:',
        'pt': ':naipe_de_copas:',
        'it': ':cuori:'
    },
    u'\U00002665': { # ♥
        'en' : ':heart_suit:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':hearts:'],
        'variant': True,
        'de': ':herz:',
        'es': ':palo_de_corazones:',
        'fr': u':cœur_cartes:',
        'pt': ':naipe_de_copas:',
        'it': ':cuori:'
    },
    u'\U0001F498': { # 💘
        'en' : ':heart_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cupid:'],
        'de': ':herz_mit_pfeil:',
        'es': u':corazón_con_flecha:',
        'fr': u':cœur_et_flèche:',
        'pt': u':coração_com_flecha:',
        'it': ':cuore_con_freccia:'
    },
    u'\U0001F49D': { # 💝
        'en' : ':heart_with_ribbon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':gift_heart:'],
        'de': ':herz_mit_schleife:',
        'es': u':corazón_con_lazo:',
        'fr': u':cœur_avec_ruban:',
        'pt': u':coração_com_fita:',
        'it': ':cuore_con_fiocco:'
    },
    u'\U0001F4B2': { # 💲
        'en' : ':heavy_dollar_sign:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':dollarzeichen_extrafett:',
        'es': u':símbolo_de_dólar:',
        'fr': ':symbole_dollar:',
        'pt': u':cifrão:',
        'it': ':dollaro:'
    },
    u'\U0001F7F0': { # 🟰
        'en' : ':heavy_equals_sign:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':gleichheitszeichen_extrafett:',
        'es': ':signo_igual_grueso:',
        'fr': u':signe_égal_gras:',
        'pt': ':sinal_de_igual:',
        'it': ':segno_di_uguaglianza:'
    },
    u'\U0001F994': { # 🦔
        'en' : ':hedgehog:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':igel:',
        'es': ':erizo:',
        'fr': u':hérisson:',
        'pt': ':porco-espinho:',
        'it': ':riccio:'
    },
    u'\U0001F681': { # 🚁
        'en' : ':helicopter:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':hubschrauber:',
        'es': u':helicóptero:',
        'fr': u':hélicoptère:',
        'pt': u':helicóptero:',
        'it': ':elicottero:'
    },
    u'\U0001F33F': { # 🌿
        'en' : ':herb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':kräuter:',
        'es': ':hierba:',
        'fr': ':feuille:',
        'pt': ':erva:',
        'it': ':pianta:'
    },
    u'\U0001F33A': { # 🌺
        'en' : ':hibiscus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hibiskus:',
        'es': ':flor_de_hibisco:',
        'fr': ':hibiscus:',
        'pt': ':hibisco:',
        'it': ':ibisco:'
    },
    u'\U0001F460': { # 👠
        'en' : ':high-heeled_shoe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':high_heel:'],
        'de': u':stöckelschuh:',
        'es': u':zapato_de_tacón:',
        'fr': u':chaussure_à_talon_haut:',
        'pt': ':sapato_de_salto_alto:',
        'it': ':scarpa_con_il_tacco:'
    },
    u'\U0001F684': { # 🚄
        'en' : ':high-speed_train:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bullettrain_side:'],
        'de': ':hochgeschwindigkeitszug_mit_spitzer_nase:',
        'es': ':tren_de_alta_velocidad:',
        'fr': ':tgv:',
        'pt': ':trem_de_alta_velocidade:',
        'it': u':treno_alta_velocità:'
    },
    u'\U000026A1': { # ⚡
        'en' : ':high_voltage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':zap:'],
        'variant': True,
        'de': ':hochspannung:',
        'es': ':alto_voltaje:',
        'fr': ':haute_tension:',
        'pt': u':alta_tensão:',
        'it': ':alta_tensione:'
    },
    u'\U0001F97E': { # 🥾
        'en' : ':hiking_boot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':wanderstiefel:',
        'es': ':bota_de_senderismo:',
        'fr': u':chaussure_de_randonnée:',
        'pt': ':bota_de_trekking:',
        'it': ':stivale_da_trekking:'
    },
    u'\U0001F6D5': { # 🛕
        'en' : ':hindu_temple:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':hindutempel:',
        'es': u':templo_hindú:',
        'fr': ':temple_hindou:',
        'pt': ':templo_hindu:',
        'it': u':tempio_indù:'
    },
    u'\U0001F99B': { # 🦛
        'en' : ':hippopotamus:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':nilpferd:',
        'es': u':hipopótamo:',
        'fr': ':hippopotame:',
        'pt': u':hipopótamo:',
        'it': ':ippopotamo:'
    },
    u'\U0001F573\U0000FE0F': { # 🕳️
        'en' : ':hole:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':loch:',
        'es': ':agujero:',
        'fr': ':trou:',
        'pt': ':buraco:',
        'it': ':buco:'
    },
    u'\U0001F573': { # 🕳
        'en' : ':hole:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':loch:',
        'es': ':agujero:',
        'fr': ':trou:',
        'pt': ':buraco:',
        'it': ':buco:'
    },
    u'\U00002B55': { # ⭕
        'en' : ':hollow_red_circle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':o:'],
        'variant': True,
        'de': ':hohler_roter_kreis:',
        'es': u':círculo_rojo_hueco:',
        'fr': ':cercle_rouge:',
        'pt': u':círculo_grande_oco:',
        'it': ':cerchio_rosso_vuoto:'
    },
    u'\U0001F36F': { # 🍯
        'en' : ':honey_pot:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':honigtopf:',
        'es': ':tarro_de_miel:',
        'fr': ':pot_de_miel:',
        'pt': ':pote_de_mel:',
        'it': ':barattolo_di_miele:'
    },
    u'\U0001F41D': { # 🐝
        'en' : ':honeybee:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bee:'],
        'de': ':biene:',
        'es': ':abeja:',
        'fr': ':abeille:',
        'pt': ':abelha:',
        'it': ':ape:'
    },
    u'\U0001FA9D': { # 🪝
        'en' : ':hook:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':haken:',
        'es': ':gancho:',
        'fr': ':crochet:',
        'pt': ':gancho:',
        'it': ':gancio:'
    },
    u'\U0001F6A5': { # 🚥
        'en' : ':horizontal_traffic_light:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':traffic_light:'],
        'de': ':horizontale_verkehrsampel:',
        'es': u':semáforo_horizontal:',
        'fr': ':feu_tricolore_horizontal:',
        'pt': u':semáforo_horizontal:',
        'it': ':semaforo_orizzontale:'
    },
    u'\U0001F40E': { # 🐎
        'en' : ':horse:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':racehorse:'],
        'de': ':pferd:',
        'es': ':caballo:',
        'fr': ':cheval:',
        'pt': ':cavalo:',
        'it': ':cavallo:'
    },
    u'\U0001F434': { # 🐴
        'en' : ':horse_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':horse:'],
        'de': ':pferdegesicht:',
        'es': ':cara_de_caballo:',
        'fr': u':tête_de_cheval:',
        'pt': ':rosto_de_cavalo:',
        'it': ':muso_di_cavallo:'
    },
    u'\U0001F3C7': { # 🏇
        'en' : ':horse_racing:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen:',
        'es': ':carrera_de_caballos:',
        'fr': ':course_hippique:',
        'pt': ':corrida_de_cavalos:',
        'it': ':ippica:'
    },
    u'\U0001F3C7\U0001F3FF': { # 🏇🏿
        'en' : ':horse_racing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen_dunkle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_oscuro:',
        'fr': u':course_hippique_peau_foncée:',
        'pt': ':corrida_de_cavalos_pele_escura:',
        'it': ':ippica_carnagione_scura:'
    },
    u'\U0001F3C7\U0001F3FB': { # 🏇🏻
        'en' : ':horse_racing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen_helle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_claro:',
        'fr': ':course_hippique_peau_claire:',
        'pt': ':corrida_de_cavalos_pele_clara:',
        'it': ':ippica_carnagione_chiara:'
    },
    u'\U0001F3C7\U0001F3FE': { # 🏇🏾
        'en' : ':horse_racing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen_mitteldunkle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_oscuro_medio:',
        'fr': ':course_hippique_peau_mate:',
        'pt': ':corrida_de_cavalos_pele_morena_escura:',
        'it': ':ippica_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C7\U0001F3FC': { # 🏇🏼
        'en' : ':horse_racing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen_mittelhelle_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_claro_medio:',
        'fr': ':course_hippique_peau_moyennement_claire:',
        'pt': ':corrida_de_cavalos_pele_morena_clara:',
        'it': ':ippica_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C7\U0001F3FD': { # 🏇🏽
        'en' : ':horse_racing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':pferderennen_mittlere_hautfarbe:',
        'es': ':carrera_de_caballos_tono_de_piel_medio:',
        'fr': u':course_hippique_peau_légèrement_mate:',
        'pt': ':corrida_de_cavalos_pele_morena:',
        'it': ':ippica_carnagione_olivastra:'
    },
    u'\U0001F3E5': { # 🏥
        'en' : ':hospital:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':krankenhaus:',
        'es': ':hospital:',
        'fr': u':hôpital:',
        'pt': ':hospital:',
        'it': ':ospedale:'
    },
    u'\U00002615': { # ☕
        'en' : ':hot_beverage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':coffee:'],
        'variant': True,
        'de': u':heißgetränk:',
        'es': ':bebida_caliente:',
        'fr': ':boisson_chaude:',
        'pt': u':café:',
        'it': ':bevanda_calda:'
    },
    u'\U0001F32D': { # 🌭
        'en' : ':hot_dog:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':hotdog:'],
        'de': ':hotdog:',
        'es': ':perrito_caliente:',
        'fr': ':hot_dog:',
        'pt': ':cachorro-quente:',
        'it': ':hot_dog:'
    },
    u'\U0001F975': { # 🥵
        'en' : ':hot_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':schwitzendes_gesicht:',
        'es': ':cara_con_calor:',
        'fr': ':visage_rouge_et_chaud:',
        'pt': ':rosto_fervendo_de_calor:',
        'it': ':faccina_accaldata:'
    },
    u'\U0001F336\U0000FE0F': { # 🌶️
        'en' : ':hot_pepper:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':peperoni:',
        'es': ':chile_picante:',
        'fr': ':piment_rouge:',
        'pt': ':pimenta:',
        'it': ':peperoncino:'
    },
    u'\U0001F336': { # 🌶
        'en' : ':hot_pepper:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':peperoni:',
        'es': ':chile_picante:',
        'fr': ':piment_rouge:',
        'pt': ':pimenta:',
        'it': ':peperoncino:'
    },
    u'\U00002668\U0000FE0F': { # ♨️
        'en' : ':hot_springs:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hotsprings:'],
        'variant': True,
        'de': u':heiße_quellen:',
        'es': ':aguas_termales:',
        'fr': ':sources_chaudes:',
        'pt': ':chamas:',
        'it': ':sorgenti_termali:'
    },
    u'\U00002668': { # ♨
        'en' : ':hot_springs:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':hotsprings:'],
        'variant': True,
        'de': u':heiße_quellen:',
        'es': ':aguas_termales:',
        'fr': ':sources_chaudes:',
        'pt': ':chamas:',
        'it': ':sorgenti_termali:'
    },
    u'\U0001F3E8': { # 🏨
        'en' : ':hotel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hotel:',
        'es': ':hotel:',
        'fr': u':hôtel:',
        'pt': ':hotel:',
        'it': ':hotel:'
    },
    u'\U0000231B': { # ⌛
        'en' : ':hourglass_done:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hourglass:'],
        'variant': True,
        'de': ':sanduhr:',
        'es': ':reloj_de_arena_sin_tiempo:',
        'fr': ':sablier:',
        'pt': ':ampulheta:',
        'it': ':clessidra:'
    },
    u'\U000023F3': { # ⏳
        'en' : ':hourglass_not_done:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hourglass_flowing_sand:'],
        'variant': True,
        'de': ':laufende_sanduhr:',
        'es': ':reloj_de_arena_con_tiempo:',
        'fr': ':sablier_avec_sable_qui_coule:',
        'pt': ':ampulheta_contando_o_tempo:',
        'it': ':clessidra_che_scorre:'
    },
    u'\U0001F3E0': { # 🏠
        'en' : ':house:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':haus:',
        'es': ':casa:',
        'fr': ':maison:',
        'pt': ':casa:',
        'it': ':casa:'
    },
    u'\U0001F3E1': { # 🏡
        'en' : ':house_with_garden:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':haus_mit_garten:',
        'es': u':casa_con_jardín:',
        'fr': ':maison_avec_jardin:',
        'pt': ':casa_com_jardim:',
        'it': ':casa_con_giardino:'
    },
    u'\U0001F3D8\U0000FE0F': { # 🏘️
        'en' : ':houses:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':house_buildings:'],
        'variant': True,
        'de': u':wohnhäuser:',
        'es': ':casas:',
        'fr': ':maisons:',
        'pt': ':casas:',
        'it': ':case:'
    },
    u'\U0001F3D8': { # 🏘
        'en' : ':houses:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':house_buildings:'],
        'variant': True,
        'de': u':wohnhäuser:',
        'es': ':casas:',
        'fr': ':maisons:',
        'pt': ':casas:',
        'it': ':case:'
    },
    u'\U0001F4AF': { # 💯
        'en' : ':hundred_points:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':100:'],
        'de': ':100_punkte:',
        'es': ':cien_puntos:',
        'fr': ':cent_points:',
        'pt': ':cem_pontos:',
        'it': ':100_punti:'
    },
    u'\U0001F62F': { # 😯
        'en' : ':hushed_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':hushed:'],
        'de': ':verdutztes_gesicht:',
        'es': ':cara_estupefacta:',
        'fr': u':visage_ébahi:',
        'pt': ':rosto_surpreso:',
        'it': ':faccina_sorpresa:'
    },
    u'\U0001F6D6': { # 🛖
        'en' : ':hut:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':hütte:',
        'es': u':cabaña:',
        'fr': ':hutte:',
        'pt': ':cabana:',
        'it': ':capanna:'
    },
    u'\U0001F9CA': { # 🧊
        'en' : ':ice:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [':ice_cube:'],
        'de': u':eiswürfel:',
        'es': ':cubito_de_hielo:',
        'fr': u':glaçon:',
        'pt': ':cubo_de_gelo:',
        'it': ':cubetto_di_ghiaccio:'
    },
    u'\U0001F368': { # 🍨
        'en' : ':ice_cream:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':eiscreme:',
        'es': ':helado:',
        'fr': ':glace:',
        'pt': ':sorvete:',
        'it': ':coppa_di_gelato:'
    },
    u'\U0001F3D2': { # 🏒
        'en' : ':ice_hockey:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':ice_hockey_stick_and_puck:'],
        'de': ':eishockey:',
        'es': ':hockey_sobre_hielo:',
        'fr': ':hockey_sur_glace:',
        'pt': u':hóquei_no_gelo:',
        'it': ':hockey_su_ghiaccio:'
    },
    u'\U000026F8\U0000FE0F': { # ⛸️
        'en' : ':ice_skate:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':schlittschuh:',
        'es': u':patín_de_hielo:',
        'fr': u':patin_à_glace:',
        'pt': ':patins_de_gelo:',
        'it': ':pattinaggio_su_ghiaccio:'
    },
    u'\U000026F8': { # ⛸
        'en' : ':ice_skate:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':schlittschuh:',
        'es': u':patín_de_hielo:',
        'fr': u':patin_à_glace:',
        'pt': ':patins_de_gelo:',
        'it': ':pattinaggio_su_ghiaccio:'
    },
    u'\U0001FAAA': { # 🪪
        'en' : ':identification_card:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':ausweis:',
        'es': u':carné_de_identidad:',
        'fr': u':carte_d’identité:',
        'pt': u':cartão_de_identificação:',
        'it': u':carta_d’identità:'
    },
    u'\U0001F4E5': { # 📥
        'en' : ':inbox_tray:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':posteingang:',
        'es': ':bandeja_de_entrada:',
        'fr': u':boîte_de_réception:',
        'pt': ':bandeja_de_entrada:',
        'it': ':posta_ricevuta:'
    },
    u'\U0001F4E8': { # 📨
        'en' : ':incoming_envelope:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':eingehender_briefumschlag:',
        'es': ':sobre_entrante:',
        'fr': u':message_reçu:',
        'pt': ':envelope_chegando:',
        'it': ':posta_in_arrivo:'
    },
    u'\U0001FAF5': { # 🫵
        'en' : ':index_pointing_at_the_viewer:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':auf_betrachter_zeigender_zeigefinger:',
        'es': u':dedo_índice_apuntándote_a_ti:',
        'fr': u':index_pointant_vers_l’utilisateur:',
        'pt': ':indicador_apontando_para_o_visualizador:',
        'it': u':indice_verso_l’osservatore:'
    },
    u'\U0001FAF5\U0001F3FF': { # 🫵🏿
        'en' : ':index_pointing_at_the_viewer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FB': { # 🫵🏻
        'en' : ':index_pointing_at_the_viewer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FE': { # 🫵🏾
        'en' : ':index_pointing_at_the_viewer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FC': { # 🫵🏼
        'en' : ':index_pointing_at_the_viewer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF5\U0001F3FD': { # 🫵🏽
        'en' : ':index_pointing_at_the_viewer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0000261D\U0000FE0F': { # ☝️
        'en' : ':index_pointing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':point_up:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': u':dedo_índice_hacia_arriba:',
        'fr': ':index_pointant_vers_le_haut:',
        'pt': ':indicador_apontando_para_cima:',
        'it': u':indice_verso_l’alto:'
    },
    u'\U0000261D': { # ☝
        'en' : ':index_pointing_up:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':point_up:'],
        'variant': True,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne:',
        'es': u':dedo_índice_hacia_arriba:',
        'fr': ':index_pointant_vers_le_haut:',
        'pt': ':indicador_apontando_para_cima:',
        'it': u':indice_verso_l’alto:'
    },
    u'\U0000261D\U0001F3FF': { # ☝🏿
        'en' : ':index_pointing_up_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_dunkle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':index_pointant_vers_le_haut_peau_foncée:',
        'pt': ':indicador_apontando_para_cima_pele_escura:',
        'it': u':indice_verso_l’alto_carnagione_scura:'
    },
    u'\U0000261D\U0001F3FB': { # ☝🏻
        'en' : ':index_pointing_up_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_helle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_claro:',
        'fr': ':index_pointant_vers_le_haut_peau_claire:',
        'pt': ':indicador_apontando_para_cima_pele_clara:',
        'it': u':indice_verso_l’alto_carnagione_chiara:'
    },
    u'\U0000261D\U0001F3FE': { # ☝🏾
        'en' : ':index_pointing_up_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mitteldunkle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': ':index_pointant_vers_le_haut_peau_mate:',
        'pt': ':indicador_apontando_para_cima_pele_morena_escura:',
        'it': u':indice_verso_l’alto_carnagione_abbastanza_scura:'
    },
    u'\U0000261D\U0001F3FC': { # ☝🏼
        'en' : ':index_pointing_up_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mittelhelle_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': ':index_pointant_vers_le_haut_peau_moyennement_claire:',
        'pt': ':indicador_apontando_para_cima_pele_morena_clara:',
        'it': u':indice_verso_l’alto_carnagione_abbastanza_chiara:'
    },
    u'\U0000261D\U0001F3FD': { # ☝🏽
        'en' : ':index_pointing_up_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nach_oben_weisender_zeigefinger_von_vorne_mittlere_hautfarbe:',
        'es': u':dedo_índice_hacia_arriba_tono_de_piel_medio:',
        'fr': u':index_pointant_vers_le_haut_peau_légèrement_mate:',
        'pt': ':indicador_apontando_para_cima_pele_morena:',
        'it': u':indice_verso_l’alto_carnagione_olivastra:'
    },
    u'\U0000267E\U0000FE0F': { # ♾️
        'en' : ':infinity:',
        'status' : fully_qualified,
        'E' : 11,
        'variant': True,
        'de': ':unendlichkeit:',
        'es': ':infinito:',
        'fr': ':infini:',
        'pt': ':infinito:',
        'it': u':simbolo_dell’infinito:'
    },
    u'\U0000267E': { # ♾
        'en' : ':infinity:',
        'status' : unqualified,
        'E' : 11,
        'variant': True,
        'de': ':unendlichkeit:',
        'es': ':infinito:',
        'fr': ':infini:',
        'pt': ':infinito:',
        'it': u':simbolo_dell’infinito:'
    },
    u'\U00002139\U0000FE0F': { # ℹ️
        'en' : ':information:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':information_source:'],
        'variant': True,
        'de': ':buchstabe_i_in_blauem_quadrat:',
        'es': u':información:',
        'fr': u':source_d’informations:',
        'pt': u':informações:',
        'it': ':punto_informazioni:'
    },
    u'\U00002139': { # ℹ
        'en' : ':information:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':information_source:'],
        'variant': True,
        'de': ':buchstabe_i_in_blauem_quadrat:',
        'es': u':información:',
        'fr': u':source_d’informations:',
        'pt': u':informações:',
        'it': ':punto_informazioni:'
    },
    u'\U0001F524': { # 🔤
        'en' : ':input_latin_letters:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':abc:'],
        'de': ':eingabesymbol_lateinische_buchstaben:',
        'es': ':alfabeto_latino:',
        'fr': ':alphabet_latin:',
        'pt': ':letras_latinas:',
        'it': ':pulsante_con_lettere_latine:'
    },
    u'\U0001F521': { # 🔡
        'en' : ':input_latin_lowercase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':abcd:'],
        'de': ':eingabesymbol_lateinische_kleinbuchstaben:',
        'es': u':letras_latinas_minúsculas:',
        'fr': ':minuscules:',
        'pt': u':letras_latinas_minúsculas:',
        'it': ':pulsante_con_lettere_latine_minuscole:'
    },
    u'\U0001F520': { # 🔠
        'en' : ':input_latin_uppercase:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':capital_abcd:'],
        'de': u':eingabesymbol_lateinische_großbuchstaben:',
        'es': u':letras_latinas_mayúsculas:',
        'fr': ':majuscules:',
        'pt': u':letras_latinas_maiúsculas:',
        'it': ':pulsante_con_lettere_latine_maiuscole:'
    },
    u'\U0001F522': { # 🔢
        'en' : ':input_numbers:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':1234:'],
        'de': ':eingabesymbol_zahlen:',
        'es': u':números:',
        'fr': ':saisie_de_chiffres:',
        'pt': u':números:',
        'it': ':pulsante_con_numeri:'
    },
    u'\U0001F523': { # 🔣
        'en' : ':input_symbols:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':symbols:'],
        'de': ':eingabesymbol_sonderzeichen:',
        'es': u':símbolos:',
        'fr': ':saisie_de_symboles:',
        'pt': u':símbolos:',
        'it': ':pulsante_con_simboli:'
    },
    u'\U0001F383': { # 🎃
        'en' : ':jack-o-lantern:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':jack_o_lantern:'],
        'de': u':halloweenkürbis:',
        'es': ':calabaza_de_halloween:',
        'fr': ':citrouille:',
        'pt': u':abóbora_de_halloween:',
        'it': ':zucca_di_halloween:'
    },
    u'\U0001FAD9': { # 🫙
        'en' : ':jar:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':einmachglas:',
        'es': ':tarro:',
        'fr': ':bocal:',
        'pt': ':jarro:',
        'it': ':barattolo:'
    },
    u'\U0001F456': { # 👖
        'en' : ':jeans:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':jeans:',
        'es': ':vaqueros:',
        'fr': ':jean:',
        'pt': ':jeans:',
        'it': ':jeans:'
    },
    u'\U0001F0CF': { # 🃏
        'en' : ':joker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':black_joker:'],
        'de': ':jokerkarte:',
        'es': u':comodín:',
        'fr': ':carte_joker:',
        'pt': ':curinga:',
        'it': ':jolly:'
    },
    u'\U0001F579\U0000FE0F': { # 🕹️
        'en' : ':joystick:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':joystick:',
        'es': ':joystick:',
        'fr': ':manette_de_jeu:',
        'pt': ':joystick:',
        'it': ':joystick:'
    },
    u'\U0001F579': { # 🕹
        'en' : ':joystick:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':joystick:',
        'es': ':joystick:',
        'fr': ':manette_de_jeu:',
        'pt': ':joystick:',
        'it': ':joystick:'
    },
    u'\U0001F9D1\U0000200D\U00002696\U0000FE0F': { # 🧑‍⚖️
        'en' : ':judge:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in):',
        'es': ':fiscal:',
        'fr': ':juge:',
        'pt': ':juiz_no_tribunal:',
        'it': ':giudice:'
    },
    u'\U0001F9D1\U0000200D\U00002696': { # 🧑‍⚖
        'en' : ':judge:',
        'status' : minimally_qualified,
        'E' : 12.1,
        'de': ':richter(in):',
        'es': ':fiscal:',
        'fr': ':juge:',
        'pt': ':juiz_no_tribunal:',
        'it': ':giudice:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002696\U0000FE0F': { # 🧑🏿‍⚖️
        'en' : ':judge_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in)_dunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro:',
        'fr': u':juge_peau_foncée:',
        'pt': ':juiz_no_tribunal_pele_escura:',
        'it': ':giudice_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002696': { # 🧑🏿‍⚖
        'en' : ':judge_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002696\U0000FE0F': { # 🧑🏻‍⚖️
        'en' : ':judge_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in)_helle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro:',
        'fr': ':juge_peau_claire:',
        'pt': ':juiz_no_tribunal_pele_clara:',
        'it': ':giudice_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002696': { # 🧑🏻‍⚖
        'en' : ':judge_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002696\U0000FE0F': { # 🧑🏾‍⚖️
        'en' : ':judge_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in)_mitteldunkle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_oscuro_medio:',
        'fr': ':juge_peau_mate:',
        'pt': ':juiz_no_tribunal_pele_morena_escura:',
        'it': ':giudice_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002696': { # 🧑🏾‍⚖
        'en' : ':judge_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002696\U0000FE0F': { # 🧑🏼‍⚖️
        'en' : ':judge_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in)_mittelhelle_hautfarbe:',
        'es': ':fiscal_tono_de_piel_claro_medio:',
        'fr': ':juge_peau_moyennement_claire:',
        'pt': ':juiz_no_tribunal_pele_morena_clara:',
        'it': ':giudice_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002696': { # 🧑🏼‍⚖
        'en' : ':judge_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002696\U0000FE0F': { # 🧑🏽‍⚖️
        'en' : ':judge_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':richter(in)_mittlere_hautfarbe:',
        'es': ':fiscal_tono_de_piel_medio:',
        'fr': u':juge_peau_légèrement_mate:',
        'pt': ':juiz_no_tribunal_pele_morena:',
        'it': ':giudice_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002696': { # 🧑🏽‍⚖
        'en' : ':judge_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12.1
    },
    u'\U0001F54B': { # 🕋
        'en' : ':kaaba:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':kaaba:',
        'es': ':kaaba:',
        'fr': ':kaaba:',
        'pt': u':caaba_islã:',
        'it': ':kaaba:'
    },
    u'\U0001F998': { # 🦘
        'en' : ':kangaroo:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':känguru:',
        'es': ':canguro:',
        'fr': ':kangourou:',
        'pt': ':canguru:',
        'it': ':canguro:'
    },
    u'\U0001F511': { # 🔑
        'en' : ':key:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':schlüssel:',
        'es': ':llave:',
        'fr': u':clé:',
        'pt': ':chave:',
        'it': ':chiave:'
    },
    u'\U00002328\U0000FE0F': { # ⌨️
        'en' : ':keyboard:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':tastatur:',
        'es': ':teclado:',
        'fr': ':clavier:',
        'pt': ':teclado:',
        'it': ':tastiera:'
    },
    u'\U00002328': { # ⌨
        'en' : ':keyboard:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':tastatur:',
        'es': ':teclado:',
        'fr': ':clavier:',
        'pt': ':teclado:',
        'it': ':tastiera:'
    },
    u'\U00000023\U0000FE0F\U000020E3': { # #️⃣
        'en' : ':keycap_#:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':taste_#:',
        'es': ':teclas_#:',
        'fr': ':touches_#:',
        'pt': ':tecla_#:',
        'it': ':tasto_#:'
    },
    u'\U00000023\U000020E3': { # #⃣
        'en' : ':keycap_#:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':hash:']
    },
    u'\U0000002A\U0000FE0F\U000020E3': { # *️⃣
        'en' : ':keycap_*:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':taste_*:',
        'es': ':teclas_*:',
        'fr': ':touches_*:',
        'pt': ':tecla_*:',
        'it': ':tasto_*:'
    },
    u'\U0000002A\U000020E3': { # *⃣
        'en' : ':keycap_*:',
        'status' : unqualified,
        'E' : 2,
        'alias' : [':asterisk:']
    },
    u'\U00000030\U0000FE0F\U000020E3': { # 0️⃣
        'en' : ':keycap_0:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':zero:'],
        'de': ':taste_0:',
        'es': ':teclas_0:',
        'fr': ':touches_0:',
        'pt': ':tecla_0:',
        'it': ':tasto_0:'
    },
    u'\U00000030\U000020E3': { # 0⃣
        'en' : ':keycap_0:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':zero:']
    },
    u'\U00000031\U0000FE0F\U000020E3': { # 1️⃣
        'en' : ':keycap_1:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':one:'],
        'de': ':taste_1:',
        'es': ':teclas_1:',
        'fr': ':touches_1:',
        'pt': ':tecla_1:',
        'it': ':tasto_1:'
    },
    u'\U00000031\U000020E3': { # 1⃣
        'en' : ':keycap_1:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':one:']
    },
    u'\U0001F51F': { # 🔟
        'en' : ':keycap_10:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ten:', ':keycap_ten:'],
        'de': ':taste_10:',
        'es': ':teclas_10:',
        'fr': ':touches_10:',
        'pt': ':tecla_10:',
        'it': ':tasto_10:'
    },
    u'\U00000032\U0000FE0F\U000020E3': { # 2️⃣
        'en' : ':keycap_2:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':two:'],
        'de': ':taste_2:',
        'es': ':teclas_2:',
        'fr': ':touches_2:',
        'pt': ':tecla_2:',
        'it': ':tasto_2:'
    },
    u'\U00000032\U000020E3': { # 2⃣
        'en' : ':keycap_2:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':two:']
    },
    u'\U00000033\U0000FE0F\U000020E3': { # 3️⃣
        'en' : ':keycap_3:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':three:'],
        'de': ':taste_3:',
        'es': ':teclas_3:',
        'fr': ':touches_3:',
        'pt': ':tecla_3:',
        'it': ':tasto_3:'
    },
    u'\U00000033\U000020E3': { # 3⃣
        'en' : ':keycap_3:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':three:']
    },
    u'\U00000034\U0000FE0F\U000020E3': { # 4️⃣
        'en' : ':keycap_4:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':four:'],
        'de': ':taste_4:',
        'es': ':teclas_4:',
        'fr': ':touches_4:',
        'pt': ':tecla_4:',
        'it': ':tasto_4:'
    },
    u'\U00000034\U000020E3': { # 4⃣
        'en' : ':keycap_4:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':four:']
    },
    u'\U00000035\U0000FE0F\U000020E3': { # 5️⃣
        'en' : ':keycap_5:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':five:'],
        'de': ':taste_5:',
        'es': ':teclas_5:',
        'fr': ':touches_5:',
        'pt': ':tecla_5:',
        'it': ':tasto_5:'
    },
    u'\U00000035\U000020E3': { # 5⃣
        'en' : ':keycap_5:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':five:']
    },
    u'\U00000036\U0000FE0F\U000020E3': { # 6️⃣
        'en' : ':keycap_6:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':six:'],
        'de': ':taste_6:',
        'es': ':teclas_6:',
        'fr': ':touches_6:',
        'pt': ':tecla_6:',
        'it': ':tasto_6:'
    },
    u'\U00000036\U000020E3': { # 6⃣
        'en' : ':keycap_6:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':six:']
    },
    u'\U00000037\U0000FE0F\U000020E3': { # 7️⃣
        'en' : ':keycap_7:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':seven:'],
        'de': ':taste_7:',
        'es': ':teclas_7:',
        'fr': ':touches_7:',
        'pt': ':tecla_7:',
        'it': ':tasto_7:'
    },
    u'\U00000037\U000020E3': { # 7⃣
        'en' : ':keycap_7:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':seven:']
    },
    u'\U00000038\U0000FE0F\U000020E3': { # 8️⃣
        'en' : ':keycap_8:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':eight:'],
        'de': ':taste_8:',
        'es': ':teclas_8:',
        'fr': ':touches_8:',
        'pt': ':tecla_8:',
        'it': ':tasto_8:'
    },
    u'\U00000038\U000020E3': { # 8⃣
        'en' : ':keycap_8:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':eight:']
    },
    u'\U00000039\U0000FE0F\U000020E3': { # 9️⃣
        'en' : ':keycap_9:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':nine:'],
        'de': ':taste_9:',
        'es': ':teclas_9:',
        'fr': ':touches_9:',
        'pt': ':tecla_9:',
        'it': ':tasto_9:'
    },
    u'\U00000039\U000020E3': { # 9⃣
        'en' : ':keycap_9:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':nine:']
    },
    u'\U0001F6F4': { # 🛴
        'en' : ':kick_scooter:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tretroller:',
        'es': ':patinete:',
        'fr': ':trottinette:',
        'pt': ':patinete:',
        'it': ':monopattino:'
    },
    u'\U0001F458': { # 👘
        'en' : ':kimono:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':kimono:',
        'es': ':kimono:',
        'fr': ':kimono:',
        'pt': ':quimono:',
        'it': ':kimono:'
    },
    u'\U0001F48F': { # 💏
        'en' : ':kiss:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':couplekiss:'],
        'de': u':sich_küssendes_paar:',
        'es': ':beso:',
        'fr': ':bisou:',
        'pt': ':beijo:',
        'it': ':bacio_tra_coppia:'
    },
    u'\U0001F48F\U0001F3FF': { # 💏🏿
        'en' : ':kiss_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FB': { # 💏🏻
        'en' : ':kiss_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468': { # 👨‍❤️‍💋‍👨
        'en' : ':kiss_man_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_mann,_mann:',
        'es': ':beso_hombre_y_hombre:',
        'fr': ':bisou_homme_et_homme:',
        'pt': ':beijo_homem_e_homem:',
        'it': ':bacio_tra_coppia_uomo_e_uomo:'
    },
    u'\U0001F468\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468': { # 👨‍❤‍💋‍👨
        'en' : ':kiss_man_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏿‍❤️‍💋‍👨🏿
        'en' : ':kiss_man_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏿‍❤‍💋‍👨🏿
        'en' : ':kiss_man_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏿‍❤️‍💋‍👨🏻
        'en' : ':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏿‍❤‍💋‍👨🏻
        'en' : ':kiss_man_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏿‍❤️‍💋‍👨🏾
        'en' : ':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏿‍❤‍💋‍👨🏾
        'en' : ':kiss_man_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏿‍❤️‍💋‍👨🏼
        'en' : ':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏿‍❤‍💋‍👨🏼
        'en' : ':kiss_man_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏿‍❤️‍💋‍👨🏽
        'en' : ':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏿‍❤‍💋‍👨🏽
        'en' : ':kiss_man_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏻‍❤️‍💋‍👨🏻
        'en' : ':kiss_man_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏻‍❤‍💋‍👨🏻
        'en' : ':kiss_man_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏻‍❤️‍💋‍👨🏿
        'en' : ':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏻‍❤‍💋‍👨🏿
        'en' : ':kiss_man_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏻‍❤️‍💋‍👨🏾
        'en' : ':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏻‍❤‍💋‍👨🏾
        'en' : ':kiss_man_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏻‍❤️‍💋‍👨🏼
        'en' : ':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏻‍❤‍💋‍👨🏼
        'en' : ':kiss_man_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏻‍❤️‍💋‍👨🏽
        'en' : ':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏻‍❤‍💋‍👨🏽
        'en' : ':kiss_man_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏾‍❤️‍💋‍👨🏾
        'en' : ':kiss_man_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏾‍❤‍💋‍👨🏾
        'en' : ':kiss_man_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏾‍❤️‍💋‍👨🏿
        'en' : ':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏾‍❤‍💋‍👨🏿
        'en' : ':kiss_man_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏾‍❤️‍💋‍👨🏻
        'en' : ':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏾‍❤‍💋‍👨🏻
        'en' : ':kiss_man_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏾‍❤️‍💋‍👨🏼
        'en' : ':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏾‍❤‍💋‍👨🏼
        'en' : ':kiss_man_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏾‍❤️‍💋‍👨🏽
        'en' : ':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏾‍❤‍💋‍👨🏽
        'en' : ':kiss_man_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏼‍❤️‍💋‍👨🏼
        'en' : ':kiss_man_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏼‍❤‍💋‍👨🏼
        'en' : ':kiss_man_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏼‍❤️‍💋‍👨🏿
        'en' : ':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏼‍❤‍💋‍👨🏿
        'en' : ':kiss_man_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏼‍❤️‍💋‍👨🏻
        'en' : ':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏼‍❤‍💋‍👨🏻
        'en' : ':kiss_man_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏼‍❤️‍💋‍👨🏾
        'en' : ':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏼‍❤‍💋‍👨🏾
        'en' : ':kiss_man_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏼‍❤️‍💋‍👨🏽
        'en' : ':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏼‍❤‍💋‍👨🏽
        'en' : ':kiss_man_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏽‍❤️‍💋‍👨🏽
        'en' : ':kiss_man_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👨🏽‍❤‍💋‍👨🏽
        'en' : ':kiss_man_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏽‍❤️‍💋‍👨🏿
        'en' : ':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👨🏽‍❤‍💋‍👨🏿
        'en' : ':kiss_man_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏽‍❤️‍💋‍👨🏻
        'en' : ':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👨🏽‍❤‍💋‍👨🏻
        'en' : ':kiss_man_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏽‍❤️‍💋‍👨🏾
        'en' : ':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👨🏽‍❤‍💋‍👨🏾
        'en' : ':kiss_man_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏽‍❤️‍💋‍👨🏼
        'en' : ':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👨🏽‍❤‍💋‍👨🏼
        'en' : ':kiss_man_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F48B': { # 💋
        'en' : ':kiss_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':kiss:'],
        'de': ':kussabdruck:',
        'es': ':marca_de_beso:',
        'fr': u':trace_de_rouge_à_lèvres:',
        'pt': ':marca_de_beijo:',
        'it': ':impronta_della_bocca:'
    },
    u'\U0001F48F\U0001F3FE': { # 💏🏾
        'en' : ':kiss_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FC': { # 💏🏼
        'en' : ':kiss_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F48F\U0001F3FD': { # 💏🏽
        'en' : ':kiss_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏿‍❤️‍💋‍🧑🏻
        'en' : ':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏿‍❤‍💋‍🧑🏻
        'en' : ':kiss_person_person_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏿‍❤️‍💋‍🧑🏾
        'en' : ':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏿‍❤‍💋‍🧑🏾
        'en' : ':kiss_person_person_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏿‍❤️‍💋‍🧑🏼
        'en' : ':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏿‍❤‍💋‍🧑🏼
        'en' : ':kiss_person_person_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏿‍❤️‍💋‍🧑🏽
        'en' : ':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏿‍❤‍💋‍🧑🏽
        'en' : ':kiss_person_person_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏻‍❤️‍💋‍🧑🏿
        'en' : ':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏻‍❤‍💋‍🧑🏿
        'en' : ':kiss_person_person_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏻‍❤️‍💋‍🧑🏾
        'en' : ':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏻‍❤‍💋‍🧑🏾
        'en' : ':kiss_person_person_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏻‍❤️‍💋‍🧑🏼
        'en' : ':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏻‍❤‍💋‍🧑🏼
        'en' : ':kiss_person_person_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏻‍❤️‍💋‍🧑🏽
        'en' : ':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏻‍❤‍💋‍🧑🏽
        'en' : ':kiss_person_person_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏾‍❤️‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏾‍❤‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏾‍❤️‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏾‍❤‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏾‍❤️‍💋‍🧑🏼
        'en' : ':kiss_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏾‍❤‍💋‍🧑🏼
        'en' : ':kiss_person_person_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏾‍❤️‍💋‍🧑🏽
        'en' : ':kiss_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏾‍❤‍💋‍🧑🏽
        'en' : ':kiss_person_person_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏼‍❤️‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏼‍❤‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏼‍❤️‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏼‍❤‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏼‍❤️‍💋‍🧑🏾
        'en' : ':kiss_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏼‍❤‍💋‍🧑🏾
        'en' : ':kiss_person_person_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏼‍❤️‍💋‍🧑🏽
        'en' : ':kiss_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏼‍❤‍💋‍🧑🏽
        'en' : ':kiss_person_person_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏽‍❤️‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏽‍❤‍💋‍🧑🏿
        'en' : ':kiss_person_person_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏽‍❤️‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏽‍❤‍💋‍🧑🏻
        'en' : ':kiss_person_person_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏽‍❤️‍💋‍🧑🏾
        'en' : ':kiss_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏽‍❤‍💋‍🧑🏾
        'en' : ':kiss_person_person_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏽‍❤️‍💋‍🧑🏼
        'en' : ':kiss_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏽‍❤‍💋‍🧑🏼
        'en' : ':kiss_person_person_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468': { # 👩‍❤️‍💋‍👨
        'en' : ':kiss_woman_man:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_frau,_mann:',
        'es': ':beso_mujer_y_hombre:',
        'fr': ':bisou_femme_et_homme:',
        'pt': ':beijo_mulher_e_homem:',
        'it': ':bacio_tra_coppia_donna_e_uomo:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468': { # 👩‍❤‍💋‍👨
        'en' : ':kiss_woman_man:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏿‍❤️‍💋‍👨🏿
        'en' : ':kiss_woman_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏿‍❤‍💋‍👨🏿
        'en' : ':kiss_woman_man_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏿‍❤️‍💋‍👨🏻
        'en' : ':kiss_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏿‍❤‍💋‍👨🏻
        'en' : ':kiss_woman_man_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏿‍❤️‍💋‍👨🏾
        'en' : ':kiss_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏿‍❤‍💋‍👨🏾
        'en' : ':kiss_woman_man_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏿‍❤️‍💋‍👨🏼
        'en' : ':kiss_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏿‍❤‍💋‍👨🏼
        'en' : ':kiss_woman_man_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏿‍❤️‍💋‍👨🏽
        'en' : ':kiss_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏿‍❤‍💋‍👨🏽
        'en' : ':kiss_woman_man_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏻‍❤️‍💋‍👨🏻
        'en' : ':kiss_woman_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏻‍❤‍💋‍👨🏻
        'en' : ':kiss_woman_man_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏻‍❤️‍💋‍👨🏿
        'en' : ':kiss_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏻‍❤‍💋‍👨🏿
        'en' : ':kiss_woman_man_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏻‍❤️‍💋‍👨🏾
        'en' : ':kiss_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏻‍❤‍💋‍👨🏾
        'en' : ':kiss_woman_man_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏻‍❤️‍💋‍👨🏼
        'en' : ':kiss_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏻‍❤‍💋‍👨🏼
        'en' : ':kiss_woman_man_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏻‍❤️‍💋‍👨🏽
        'en' : ':kiss_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏻‍❤‍💋‍👨🏽
        'en' : ':kiss_woman_man_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏾‍❤️‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏾‍❤‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏾‍❤️‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏾‍❤‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏾‍❤️‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏾‍❤‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏾‍❤️‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏾‍❤‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏾‍❤️‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏾‍❤‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏼‍❤️‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏼‍❤‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏼‍❤️‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏼‍❤‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏼‍❤️‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏼‍❤‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏼‍❤️‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏼‍❤‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏼‍❤️‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏼‍❤‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏽‍❤️‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FD': { # 👩🏽‍❤‍💋‍👨🏽
        'en' : ':kiss_woman_man_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏽‍❤️‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FF': { # 👩🏽‍❤‍💋‍👨🏿
        'en' : ':kiss_woman_man_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏽‍❤️‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FB': { # 👩🏽‍❤‍💋‍👨🏻
        'en' : ':kiss_woman_man_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏽‍❤️‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FE': { # 👩🏽‍❤‍💋‍👨🏾
        'en' : ':kiss_woman_man_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏽‍❤️‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F468\U0001F3FC': { # 👩🏽‍❤‍💋‍👨🏼
        'en' : ':kiss_woman_man_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469': { # 👩‍❤️‍💋‍👩
        'en' : ':kiss_woman_woman:',
        'status' : fully_qualified,
        'E' : 2,
        'de': u':sich_küssendes_paar_frau,_frau:',
        'es': ':beso_mujer_y_mujer:',
        'fr': ':bisou_femme_et_femme:',
        'pt': ':beijo_mulher_e_mulher:',
        'it': ':bacio_tra_coppia_donna_e_donna:'
    },
    u'\U0001F469\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469': { # 👩‍❤‍💋‍👩
        'en' : ':kiss_woman_woman:',
        'status' : minimally_qualified,
        'E' : 2
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏿‍❤️‍💋‍👩🏿
        'en' : ':kiss_woman_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏿‍❤‍💋‍👩🏿
        'en' : ':kiss_woman_woman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏿‍❤️‍💋‍👩🏻
        'en' : ':kiss_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏿‍❤‍💋‍👩🏻
        'en' : ':kiss_woman_woman_dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏿‍❤️‍💋‍👩🏾
        'en' : ':kiss_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏿‍❤‍💋‍👩🏾
        'en' : ':kiss_woman_woman_dark_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏿‍❤️‍💋‍👩🏼
        'en' : ':kiss_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏿‍❤‍💋‍👩🏼
        'en' : ':kiss_woman_woman_dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏿‍❤️‍💋‍👩🏽
        'en' : ':kiss_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FF\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏿‍❤‍💋‍👩🏽
        'en' : ':kiss_woman_woman_dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏻‍❤️‍💋‍👩🏻
        'en' : ':kiss_woman_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏻‍❤‍💋‍👩🏻
        'en' : ':kiss_woman_woman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏻‍❤️‍💋‍👩🏿
        'en' : ':kiss_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏻‍❤‍💋‍👩🏿
        'en' : ':kiss_woman_woman_light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏻‍❤️‍💋‍👩🏾
        'en' : ':kiss_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏻‍❤‍💋‍👩🏾
        'en' : ':kiss_woman_woman_light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏻‍❤️‍💋‍👩🏼
        'en' : ':kiss_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏻‍❤‍💋‍👩🏼
        'en' : ':kiss_woman_woman_light_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏻‍❤️‍💋‍👩🏽
        'en' : ':kiss_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FB\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏻‍❤‍💋‍👩🏽
        'en' : ':kiss_woman_woman_light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏾‍❤️‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏾‍❤‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏾‍❤️‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏾‍❤‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏾‍❤️‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏾‍❤‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏾‍❤️‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏾‍❤‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏾‍❤️‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FE\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏾‍❤‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium-dark_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏼‍❤️‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏼‍❤‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏼‍❤️‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏼‍❤‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium-light_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏼‍❤️‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏼‍❤‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium-light_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏼‍❤️‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏼‍❤‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏼‍❤️‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FC\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏼‍❤‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium-light_skin_tone_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏽‍❤️‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FD': { # 👩🏽‍❤‍💋‍👩🏽
        'en' : ':kiss_woman_woman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏽‍❤️‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FF': { # 👩🏽‍❤‍💋‍👩🏿
        'en' : ':kiss_woman_woman_medium_skin_tone_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏽‍❤️‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FB': { # 👩🏽‍❤‍💋‍👩🏻
        'en' : ':kiss_woman_woman_medium_skin_tone_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏽‍❤️‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FE': { # 👩🏽‍❤‍💋‍👩🏾
        'en' : ':kiss_woman_woman_medium_skin_tone_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000FE0F\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏽‍❤️‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F469\U0001F3FD\U0000200D\U00002764\U0000200D\U0001F48B\U0000200D\U0001F469\U0001F3FC': { # 👩🏽‍❤‍💋‍👩🏼
        'en' : ':kiss_woman_woman_medium_skin_tone_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F63D': { # 😽
        'en' : ':kissing_cat:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':küssende_katze:',
        'es': ':gato_besando:',
        'fr': ':chat_qui_fait_un_bisou:',
        'pt': ':rosto_de_gato_mandando_um_beijo:',
        'it': ':gatto_che_manda_baci:'
    },
    u'\U0001F617': { # 😗
        'en' : ':kissing_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':kissing:'],
        'de': u':küssendes_gesicht:',
        'es': ':cara_besando:',
        'fr': ':visage_faisant_un_bisou:',
        'pt': ':rosto_beijando:',
        'it': ':faccina_che_bacia:'
    },
    u'\U0001F61A': { # 😚
        'en' : ':kissing_face_with_closed_eyes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':kissing_closed_eyes:'],
        'de': u':küssendes_gesicht_mit_geschlossenen_augen:',
        'es': ':cara_besando_con_los_ojos_cerrados:',
        'fr': u':visage_faisant_un_bisou_avec_les_yeux_fermés:',
        'pt': ':rosto_beijando_com_olhos_fechados:',
        'it': ':faccina_che_bacia_con_occhi_chiusi:'
    },
    u'\U0001F619': { # 😙
        'en' : ':kissing_face_with_smiling_eyes:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':kissing_smiling_eyes:'],
        'de': u':küssendes_gesicht_mit_lächelnden_augen:',
        'es': ':cara_besando_con_ojos_sonrientes:',
        'fr': ':visage_aux_yeux_rieurs_faisant_un_bisou:',
        'pt': ':rosto_beijando_com_olhos_sorridentes:',
        'it': ':faccina_che_bacia_con_occhi_sorridenti:'
    },
    u'\U0001F52A': { # 🔪
        'en' : ':kitchen_knife:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':hocho:', ':knife:'],
        'de': u':küchenmesser:',
        'es': ':cuchillo_de_cocina:',
        'fr': ':couteau_de_cuisine:',
        'pt': ':faca_de_cozinha:',
        'it': ':coltello_da_cucina:'
    },
    u'\U0001FA81': { # 🪁
        'en' : ':kite:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':drachen:',
        'es': ':cometa:',
        'fr': ':cerf-volant:',
        'pt': ':pipa:',
        'it': ':aquilone:'
    },
    u'\U0001F95D': { # 🥝
        'en' : ':kiwi_fruit:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':kiwi:',
        'es': ':kiwi:',
        'fr': ':kiwi:',
        'pt': ':kiwi:',
        'it': ':kiwi:'
    },
    u'\U0001FAA2': { # 🪢
        'en' : ':knot:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':knoten:',
        'es': ':nudo:',
        'fr': u':nœud:',
        'pt': u':nó:',
        'it': ':nodo:'
    },
    u'\U0001F428': { # 🐨
        'en' : ':koala:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':koala:',
        'es': ':koala:',
        'fr': ':koala:',
        'pt': ':coala:',
        'it': ':koala:'
    },
    u'\U0001F97C': { # 🥼
        'en' : ':lab_coat:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':laborkittel:',
        'es': ':bata_de_laboratorio:',
        'fr': ':blouse_blanche:',
        'pt': ':jaleco:',
        'it': ':camice:'
    },
    u'\U0001F3F7\U0000FE0F': { # 🏷️
        'en' : ':label:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':etikett:',
        'es': ':etiqueta:',
        'fr': u':étiquette:',
        'pt': ':etiqueta:',
        'it': ':etichetta:'
    },
    u'\U0001F3F7': { # 🏷
        'en' : ':label:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':etikett:',
        'es': ':etiqueta:',
        'fr': u':étiquette:',
        'pt': ':etiqueta:',
        'it': ':etichetta:'
    },
    u'\U0001F94D': { # 🥍
        'en' : ':lacrosse:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':lacrosse:',
        'es': ':lacrosse:',
        'fr': ':crosse:',
        'pt': ':lacrosse:',
        'it': ':lacrosse:'
    },
    u'\U0001FA9C': { # 🪜
        'en' : ':ladder:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':leiter:',
        'es': ':escalera:',
        'fr': u':échelle:',
        'pt': ':escada:',
        'it': ':scala:'
    },
    u'\U0001F41E': { # 🐞
        'en' : ':lady_beetle:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':beetle:'],
        'de': u':marienkäfer:',
        'es': ':mariquita:',
        'fr': ':coccinelle:',
        'pt': ':joaninha:',
        'it': ':coccinella:'
    },
    u'\U0001F4BB': { # 💻
        'en' : ':laptop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':computer:'],
        'variant': True,
        'de': ':laptop:',
        'es': u':ordenador_portátil:',
        'fr': ':ordinateur_portable:',
        'pt': ':laptop:',
        'it': ':computer_portatile:'
    },
    u'\U0001F537': { # 🔷
        'en' : ':large_blue_diamond:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':große_blaue_raute:',
        'es': ':rombo_azul_grande:',
        'fr': ':grand_losange_bleu:',
        'pt': ':losango_azul_grande:',
        'it': ':rombo_blu_grande:'
    },
    u'\U0001F536': { # 🔶
        'en' : ':large_orange_diamond:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':große_orangefarbene_raute:',
        'es': ':rombo_naranja_grande:',
        'fr': ':grand_losange_orange:',
        'pt': ':losango_laranja_grande:',
        'it': ':rombo_arancione_grande:'
    },
    u'\U0001F317': { # 🌗
        'en' : ':last_quarter_moon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':abnehmender_halbmond:',
        'es': ':luna_en_cuarto_menguante:',
        'fr': ':dernier_quartier_de_lune:',
        'pt': ':quarto_minguante:',
        'it': ':ultimo_quarto_di_luna:'
    },
    u'\U0001F31C': { # 🌜
        'en' : ':last_quarter_moon_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':last_quarter_moon_with_face:'],
        'variant': True,
        'de': ':mondsichel_mit_gesicht_rechts:',
        'es': ':luna_de_cuarto_menguante_con_cara:',
        'fr': ':dernier_quartier_de_lune_avec_visage:',
        'pt': ':rosto_da_lua_de_quarto_minguante:',
        'it': ':faccina_ultimo_quarto_di_luna:'
    },
    u'\U000023EE\U0000FE0F': { # ⏮️
        'en' : ':last_track_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':previous_track_button:', ':black_left__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': ':vorheriger_titel:',
        'es': ':pista_anterior:',
        'fr': u':bouton_piste_précédente:',
        'pt': u':botão_de_faixa_anterior:',
        'it': ':pulsante_traccia_precedente:'
    },
    u'\U000023EE': { # ⏮
        'en' : ':last_track_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':previous_track_button:', ':black_left__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': ':vorheriger_titel:',
        'es': ':pista_anterior:',
        'fr': u':bouton_piste_précédente:',
        'pt': u':botão_de_faixa_anterior:',
        'it': ':pulsante_traccia_precedente:'
    },
    u'\U0000271D\U0000FE0F': { # ✝️
        'en' : ':latin_cross:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':römisches_kreuz:',
        'es': ':cruz_latina:',
        'fr': ':croix_latine:',
        'pt': ':cruz_latina:',
        'it': ':croce_latina:'
    },
    u'\U0000271D': { # ✝
        'en' : ':latin_cross:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':lateinisches_kreuz:',
        'es': ':cruz_latina:',
        'fr': ':croix_latine:',
        'pt': ':cruz_latina:',
        'it': ':croce_latina:'
    },
    u'\U0001F343': { # 🍃
        'en' : ':leaf_fluttering_in_wind:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':leaves:'],
        'de': u':blätter_im_wind:',
        'es': ':hojas_revoloteando_al_viento:',
        'fr': ':feuille_virevoltante:',
        'pt': ':folha_ao_vento:',
        'it': ':foglia_al_vento:'
    },
    u'\U0001F96C': { # 🥬
        'en' : ':leafy_green:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':blattgemüse:',
        'es': ':verdura_de_hoja_verde:',
        'fr': u':légume_à_feuilles_vertes:',
        'pt': ':verdura:',
        'it': ':verdure_a_foglia:'
    },
    u'\U0001F4D2': { # 📒
        'en' : ':ledger:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':spiralblock:',
        'es': ':libro_de_contabilidad:',
        'fr': ':carnet_de_compte:',
        'pt': u':livro_contábil:',
        'it': ':quaderno_ad_anelli:'
    },
    u'\U0001F91B': { # 🤛
        'en' : ':left-facing_fist:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':fist_left:'],
        'de': ':faust_nach_links:',
        'es': u':puño_hacia_la_izquierda:',
        'fr': u':poing_à_gauche:',
        'pt': ':punho_esquerdo:',
        'it': ':pugno_a_sinistra:'
    },
    u'\U0001F91B\U0001F3FF': { # 🤛🏿
        'en' : ':left-facing_fist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':faust_nach_links_dunkle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_oscuro:',
        'fr': u':poing_à_gauche_peau_foncée:',
        'pt': ':punho_esquerdo_pele_escura:',
        'it': ':pugno_a_sinistra_carnagione_scura:'
    },
    u'\U0001F91B\U0001F3FB': { # 🤛🏻
        'en' : ':left-facing_fist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':faust_nach_links_helle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_claro:',
        'fr': u':poing_à_gauche_peau_claire:',
        'pt': ':punho_esquerdo_pele_clara:',
        'it': ':pugno_a_sinistra_carnagione_chiara:'
    },
    u'\U0001F91B\U0001F3FE': { # 🤛🏾
        'en' : ':left-facing_fist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':faust_nach_links_mitteldunkle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_oscuro_medio:',
        'fr': u':poing_à_gauche_peau_mate:',
        'pt': ':punho_esquerdo_pele_morena_escura:',
        'it': ':pugno_a_sinistra_carnagione_abbastanza_scura:'
    },
    u'\U0001F91B\U0001F3FC': { # 🤛🏼
        'en' : ':left-facing_fist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':faust_nach_links_mittelhelle_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_claro_medio:',
        'fr': u':poing_à_gauche_peau_moyennement_claire:',
        'pt': ':punho_esquerdo_pele_morena_clara:',
        'it': ':pugno_a_sinistra_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91B\U0001F3FD': { # 🤛🏽
        'en' : ':left-facing_fist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':faust_nach_links_mittlere_hautfarbe:',
        'es': u':puño_hacia_la_izquierda_tono_de_piel_medio:',
        'fr': u':poing_à_gauche_peau_légèrement_mate:',
        'pt': ':punho_esquerdo_pele_morena:',
        'it': ':pugno_a_sinistra_carnagione_olivastra:'
    },
    u'\U00002194\U0000FE0F': { # ↔️
        'en' : ':left-right_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':left_right_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_links_und_rechts:',
        'es': ':flecha_izquierda_y_derecha:',
        'fr': u':flèche_gauche_droite:',
        'pt': ':seta_para_esquerda_e_direita:',
        'it': ':freccia_sinistra-destra:'
    },
    u'\U00002194': { # ↔
        'en' : ':left-right_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':left_right_arrow:'],
        'variant': True,
        'de': ':pfeil_nach_links_und_rechts:',
        'es': ':flecha_izquierda_y_derecha:',
        'fr': u':flèche_gauche_droite:',
        'pt': ':seta_para_esquerda_e_direita:',
        'it': ':freccia_sinistra-destra:'
    },
    u'\U00002B05\U0000FE0F': { # ⬅️
        'en' : ':left_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_left:'],
        'variant': True,
        'de': ':pfeil_nach_links:',
        'es': ':flecha_hacia_la_izquierda:',
        'fr': u':flèche_gauche:',
        'pt': ':seta_para_a_esquerda:',
        'it': ':freccia_rivolta_a_sinistra:'
    },
    u'\U00002B05': { # ⬅
        'en' : ':left_arrow:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':arrow_left:'],
        'variant': True,
        'de': ':pfeil_nach_links:',
        'es': ':flecha_hacia_la_izquierda:',
        'fr': u':flèche_gauche:',
        'pt': ':seta_para_a_esquerda:',
        'it': ':freccia_rivolta_a_sinistra:'
    },
    u'\U000021AA\U0000FE0F': { # ↪️
        'en' : ':left_arrow_curving_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':arrow_right_hook:'],
        'variant': True,
        'de': ':geschwungener_pfeil_nach_rechts:',
        'es': u':flecha_izquierda_curvándose_a_la_derecha:',
        'fr': u':flèche_courbe_droite:',
        'pt': ':seta_curva_da_esquerda_para_a_direita:',
        'it': ':freccia_curva_a_destra:'
    },
    u'\U000021AA': { # ↪
        'en' : ':left_arrow_curving_right:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':arrow_right_hook:'],
        'variant': True,
        'de': ':geschwungener_pfeil_nach_rechts:',
        'es': u':flecha_izquierda_curvándose_a_la_derecha:',
        'fr': u':flèche_courbe_droite:',
        'pt': ':seta_curva_da_esquerda_para_a_direita:',
        'it': ':freccia_curva_a_destra:'
    },
    u'\U0001F6C5': { # 🛅
        'en' : ':left_luggage:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':gepäckaufbewahrung:',
        'es': ':consigna:',
        'fr': ':consigne:',
        'pt': u':depósito_de_bagagem:',
        'it': ':simbolo_del_deposito_bagagli:'
    },
    u'\U0001F5E8\U0000FE0F': { # 🗨️
        'en' : ':left_speech_bubble:',
        'status' : fully_qualified,
        'E' : 2,
        'variant': True,
        'de': ':sprechblase_links:',
        'es': u':bocadillo_de_diálogo_por_la_izquierda:',
        'fr': ':bulle_de_parole_gauche:',
        'pt': u':balão_de_diálogo_à_esquerda:',
        'it': ':nuvoletta_nera:'
    },
    u'\U0001F5E8': { # 🗨
        'en' : ':left_speech_bubble:',
        'status' : unqualified,
        'E' : 2,
        'variant': True,
        'de': ':sprechblase_links:',
        'es': u':bocadillo_de_diálogo_por_la_izquierda:',
        'fr': ':bulle_de_parole_gauche:',
        'pt': u':balão_de_diálogo_à_esquerda:',
        'it': ':nuvoletta_nera:'
    },
    u'\U0001FAF2': { # 🫲
        'en' : ':leftwards_hand:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':nach_links_weisende_hand:',
        'es': ':mano_hacia_la_izquierda:',
        'fr': ':main_vers_la_gauche:',
        'pt': u':mão_para_a_esquerda:',
        'it': ':mano_rivolta_a_sinistra:'
    },
    u'\U0001FAF2\U0001F3FF': { # 🫲🏿
        'en' : ':leftwards_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FB': { # 🫲🏻
        'en' : ':leftwards_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FE': { # 🫲🏾
        'en' : ':leftwards_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FC': { # 🫲🏼
        'en' : ':leftwards_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF2\U0001F3FD': { # 🫲🏽
        'en' : ':leftwards_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F9B5': { # 🦵
        'en' : ':leg:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein:',
        'es': ':pierna:',
        'fr': ':jambe:',
        'pt': ':perna:',
        'it': ':gamba:'
    },
    u'\U0001F9B5\U0001F3FF': { # 🦵🏿
        'en' : ':leg_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein_dunkle_hautfarbe:',
        'es': ':pierna_tono_de_piel_oscuro:',
        'fr': u':jambe_peau_foncée:',
        'pt': ':perna_pele_escura:',
        'it': ':gamba_carnagione_scura:'
    },
    u'\U0001F9B5\U0001F3FB': { # 🦵🏻
        'en' : ':leg_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein_helle_hautfarbe:',
        'es': ':pierna_tono_de_piel_claro:',
        'fr': ':jambe_peau_claire:',
        'pt': ':perna_pele_clara:',
        'it': ':gamba_carnagione_chiara:'
    },
    u'\U0001F9B5\U0001F3FE': { # 🦵🏾
        'en' : ':leg_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein_mitteldunkle_hautfarbe:',
        'es': ':pierna_tono_de_piel_oscuro_medio:',
        'fr': ':jambe_peau_mate:',
        'pt': ':perna_pele_morena_escura:',
        'it': ':gamba_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B5\U0001F3FC': { # 🦵🏼
        'en' : ':leg_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein_mittelhelle_hautfarbe:',
        'es': ':pierna_tono_de_piel_claro_medio:',
        'fr': ':jambe_peau_moyennement_claire:',
        'pt': ':perna_pele_morena_clara:',
        'it': ':gamba_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B5\U0001F3FD': { # 🦵🏽
        'en' : ':leg_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':bein_mittlere_hautfarbe:',
        'es': ':pierna_tono_de_piel_medio:',
        'fr': u':jambe_peau_légèrement_mate:',
        'pt': ':perna_pele_morena:',
        'it': ':gamba_carnagione_olivastra:'
    },
    u'\U0001F34B': { # 🍋
        'en' : ':lemon:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':zitrone:',
        'es': u':limón:',
        'fr': ':citron:',
        'pt': u':limão:',
        'it': ':limone:'
    },
    u'\U0001F406': { # 🐆
        'en' : ':leopard:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':leopard:',
        'es': ':leopardo:',
        'fr': u':léopard:',
        'pt': ':leopardo:',
        'it': ':leopardo:'
    },
    u'\U0001F39A\U0000FE0F': { # 🎚️
        'en' : ':level_slider:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':schieberegler:',
        'es': ':control_de_volumen:',
        'fr': ':curseur_de_niveau:',
        'pt': ':controle_de_volume:',
        'it': ':cursore_del_volume:'
    },
    u'\U0001F39A': { # 🎚
        'en' : ':level_slider:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':schieberegler:',
        'es': ':control_de_volumen:',
        'fr': ':curseur_de_niveau:',
        'pt': ':controle_de_volume:',
        'it': ':cursore_del_volume:'
    },
    u'\U0001F4A1': { # 💡
        'en' : ':light_bulb:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bulb:'],
        'de': u':glühbirne:',
        'es': ':bombilla:',
        'fr': ':ampoule:',
        'pt': u':lâmpada:',
        'it': ':lampadina:'
    },
    u'\U0001F688': { # 🚈
        'en' : ':light_rail:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':s-bahn:',
        'es': ':tren_ligero:',
        'fr': u':métro_léger:',
        'pt': ':trem_urbano:',
        'it': ':metrotranvia:'
    },
    u'\U0001F3FB': { # 🏻
        'en' : ':light_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [':emoji_modifier_fitzpatrick_type__1__2:'],
        'de': ':helle_hautfarbe:',
        'es': ':tono_de_piel_claro:',
        'fr': ':peau_claire:',
        'pt': ':pele_clara:',
        'it': ':carnagione_chiara:'
    },
    u'\U0001F517': { # 🔗
        'en' : ':link:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':linksymbol:',
        'es': u':eslabón:',
        'fr': u':chaînons:',
        'pt': ':link:',
        'it': ':collegamento:'
    },
    u'\U0001F587\U0000FE0F': { # 🖇️
        'en' : ':linked_paperclips:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':paperclips:'],
        'variant': True,
        'de': u':verhakte_büroklammern:',
        'es': ':clips_unidos:',
        'fr': ':trombones:',
        'pt': ':clipes_de_papel_conectados:',
        'it': ':graffette_attaccate:'
    },
    u'\U0001F587': { # 🖇
        'en' : ':linked_paperclips:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':paperclips:'],
        'variant': True,
        'de': u':verhakte_büroklammern:',
        'es': ':clips_unidos:',
        'fr': ':trombones:',
        'pt': ':clipes_de_papel_conectados:',
        'it': ':graffette_attaccate:'
    },
    u'\U0001F981': { # 🦁
        'en' : ':lion:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':lion_face:'],
        'de': u':löwe:',
        'es': u':león:',
        'fr': u':tête_de_lion:',
        'pt': u':rosto_de_leão:',
        'it': ':leone:'
    },
    u'\U0001F484': { # 💄
        'en' : ':lipstick:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':lippenstift:',
        'es': ':pintalabios:',
        'fr': u':rouge_à_lèvres:',
        'pt': ':batom:',
        'it': ':rossetto:'
    },
    u'\U0001F6AE': { # 🚮
        'en' : ':litter_in_bin_sign:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':put_litter_in_its_place:'],
        'de': ':symbol_papierkorb:',
        'es': u':señal_de_usar_papelera:',
        'fr': u':icône_poubelle:',
        'pt': u':símbolo_de_lixeira:',
        'it': ':simbolo_per_la_raccolta_dei_rifiuti:'
    },
    u'\U0001F98E': { # 🦎
        'en' : ':lizard:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':eidechse:',
        'es': ':lagarto:',
        'fr': u':lézard:',
        'pt': ':lagartixa:',
        'it': ':lucertola:'
    },
    u'\U0001F999': { # 🦙
        'en' : ':llama:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':lama:',
        'es': ':llama:',
        'fr': ':lama:',
        'pt': ':lhama:',
        'it': ':lama:'
    },
    u'\U0001F99E': { # 🦞
        'en' : ':lobster:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':hummer:',
        'es': ':bogavante:',
        'fr': ':homard:',
        'pt': ':lagosta:',
        'it': ':aragosta:'
    },
    u'\U0001F512': { # 🔒
        'en' : ':locked:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':lock:'],
        'variant': True,
        'de': ':geschlossenes_schloss:',
        'es': ':candado_cerrado:',
        'fr': u':cadenas_fermé:',
        'pt': ':cadeado:',
        'it': ':lucchetto_chiuso:'
    },
    u'\U0001F510': { # 🔐
        'en' : ':locked_with_key:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':closed_lock_with_key:'],
        'de': u':schloss_mit_schlüssel:',
        'es': ':candado_cerrado_y_llave:',
        'fr': u':cadenas_fermé_avec_clé:',
        'pt': ':cadeado_fechado_com_chave:',
        'it': ':lucchetto_chiuso_con_chiave:'
    },
    u'\U0001F50F': { # 🔏
        'en' : ':locked_with_pen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':lock_with_ink_pen:'],
        'de': u':schloss_mit_füller:',
        'es': u':candado_con_pluma_estilográfica:',
        'fr': u':cadenas_fermé_avec_stylo:',
        'pt': ':cadeado_com_caneta:',
        'it': ':lucchetto_con_penna_stilo:'
    },
    u'\U0001F682': { # 🚂
        'en' : ':locomotive:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':steam_locomotive:'],
        'de': ':dampflokomotive:',
        'es': ':locomotora_de_vapor:',
        'fr': ':locomotive:',
        'pt': ':locomotiva:',
        'it': ':locomotiva:'
    },
    u'\U0001F36D': { # 🍭
        'en' : ':lollipop:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':lutscher:',
        'es': ':piruleta:',
        'fr': ':sucette:',
        'pt': ':pirulito:',
        'it': ':lecca_lecca:'
    },
    u'\U0001FA98': { # 🪘
        'en' : ':long_drum:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':afrikanische_trommel:',
        'es': ':tamboril:',
        'fr': u':djembé:',
        'pt': ':tambor_comprido:',
        'it': ':tamburo_lungo:'
    },
    u'\U0001F9F4': { # 🧴
        'en' : ':lotion_bottle:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':creme:',
        'es': ':bote_de_crema:',
        'fr': ':bouteille_de_lotion:',
        'pt': u':frasco_de_loção:',
        'it': ':flacone:'
    },
    u'\U0001FAB7': { # 🪷
        'en' : ':lotus:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':lotusblüte:',
        'es': ':loto:',
        'fr': ':lotus:',
        'pt': u':lótus:',
        'it': ':loto:'
    },
    u'\U0001F62D': { # 😭
        'en' : ':loudly_crying_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':sob:'],
        'de': ':heulendes_gesicht:',
        'es': ':cara_llorando_fuerte:',
        'fr': u':visage_qui_pleure_à_chaudes_larmes:',
        'pt': ':rosto_chorando_aos_berros:',
        'it': ':faccina_disperata:'
    },
    u'\U0001F4E2': { # 📢
        'en' : ':loudspeaker:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':lautsprecher:',
        'es': ':altavoz_de_mano:',
        'fr': ':haut-parleur:',
        'pt': ':buzina:',
        'it': ':altoparlante:'
    },
    u'\U0001F91F': { # 🤟
        'en' : ':love-you_gesture:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':love_you_gesture:'],
        'de': ':ich-liebe-dich-geste:',
        'es': ':gesto_de_te_quiero:',
        'fr': u':signe_je_t’aime:',
        'pt': ':gesto_de_te_amo:',
        'it': ':gesto_ti_amo:'
    },
    u'\U0001F91F\U0001F3FF': { # 🤟🏿
        'en' : ':love-you_gesture_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':ich-liebe-dich-geste_dunkle_hautfarbe:',
        'es': ':gesto_de_te_quiero_tono_de_piel_oscuro:',
        'fr': u':signe_je_t’aime_peau_foncée:',
        'pt': ':gesto_de_te_amo_pele_escura:',
        'it': ':gesto_ti_amo_carnagione_scura:'
    },
    u'\U0001F91F\U0001F3FB': { # 🤟🏻
        'en' : ':love-you_gesture_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':ich-liebe-dich-geste_helle_hautfarbe:',
        'es': ':gesto_de_te_quiero_tono_de_piel_claro:',
        'fr': u':signe_je_t’aime_peau_claire:',
        'pt': ':gesto_de_te_amo_pele_clara:',
        'it': ':gesto_ti_amo_carnagione_chiara:'
    },
    u'\U0001F91F\U0001F3FE': { # 🤟🏾
        'en' : ':love-you_gesture_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':ich-liebe-dich-geste_mitteldunkle_hautfarbe:',
        'es': ':gesto_de_te_quiero_tono_de_piel_oscuro_medio:',
        'fr': u':signe_je_t’aime_peau_mate:',
        'pt': ':gesto_de_te_amo_pele_morena_escura:',
        'it': ':gesto_ti_amo_carnagione_abbastanza_scura:'
    },
    u'\U0001F91F\U0001F3FC': { # 🤟🏼
        'en' : ':love-you_gesture_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':ich-liebe-dich-geste_mittelhelle_hautfarbe:',
        'es': ':gesto_de_te_quiero_tono_de_piel_claro_medio:',
        'fr': u':signe_je_t’aime_peau_moyennement_claire:',
        'pt': ':gesto_de_te_amo_pele_morena_clara:',
        'it': ':gesto_ti_amo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F91F\U0001F3FD': { # 🤟🏽
        'en' : ':love-you_gesture_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':ich-liebe-dich-geste_mittlere_hautfarbe:',
        'es': ':gesto_de_te_quiero_tono_de_piel_medio:',
        'fr': u':signe_je_t’aime_peau_légèrement_mate:',
        'pt': ':gesto_de_te_amo_pele_morena:',
        'it': ':gesto_ti_amo_carnagione_olivastra:'
    },
    u'\U0001F3E9': { # 🏩
        'en' : ':love_hotel:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':stundenhotel:',
        'es': ':hotel_del_amor:',
        'fr': ':love_hotel:',
        'pt': ':motel:',
        'it': ':love_hotel:'
    },
    u'\U0001F48C': { # 💌
        'en' : ':love_letter:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':liebesbrief:',
        'es': ':carta_de_amor:',
        'fr': u':lettre_d’amour:',
        'pt': ':carta_de_amor:',
        'it': u':lettera_d’amore:'
    },
    u'\U0001FAAB': { # 🪫
        'en' : ':low_battery:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':schwache_batterie:',
        'es': u':batería_baja:',
        'fr': ':pile_faible:',
        'pt': ':pouca_bateria:',
        'it': ':batteria_scarica:'
    },
    u'\U0001F9F3': { # 🧳
        'en' : ':luggage:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':gepäck:',
        'es': ':equipaje:',
        'fr': ':bagage:',
        'pt': ':mala:',
        'it': ':valigia:'
    },
    u'\U0001FAC1': { # 🫁
        'en' : ':lungs:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':lunge:',
        'es': ':pulmones:',
        'fr': ':poumons:',
        'pt': u':pulmões:',
        'it': ':polmoni:'
    },
    u'\U0001F925': { # 🤥
        'en' : ':lying_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':lügendes_gesicht:',
        'es': ':cara_de_mentiroso:',
        'fr': ':visage_de_menteur:',
        'pt': ':rosto_de_mentiroso:',
        'it': ':faccina_bugiarda:'
    },
    u'\U0001F9D9': { # 🧙
        'en' : ':mage:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in):',
        'es': ':persona_maga:',
        'fr': ':mage:',
        'pt': ':mago:',
        'it': ':mago:'
    },
    u'\U0001F9D9\U0001F3FF': { # 🧙🏿
        'en' : ':mage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in)_dunkle_hautfarbe:',
        'es': ':persona_maga_tono_de_piel_oscuro:',
        'fr': u':mage_peau_foncée:',
        'pt': ':mago_pele_escura:',
        'it': ':mago_carnagione_scura:'
    },
    u'\U0001F9D9\U0001F3FB': { # 🧙🏻
        'en' : ':mage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in)_helle_hautfarbe:',
        'es': ':persona_maga_tono_de_piel_claro:',
        'fr': ':mage_peau_claire:',
        'pt': ':mago_pele_clara:',
        'it': ':mago_carnagione_chiara:'
    },
    u'\U0001F9D9\U0001F3FE': { # 🧙🏾
        'en' : ':mage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in)_mitteldunkle_hautfarbe:',
        'es': ':persona_maga_tono_de_piel_oscuro_medio:',
        'fr': ':mage_peau_mate:',
        'pt': ':mago_pele_morena_escura:',
        'it': ':mago_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D9\U0001F3FC': { # 🧙🏼
        'en' : ':mage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in)_mittelhelle_hautfarbe:',
        'es': ':persona_maga_tono_de_piel_claro_medio:',
        'fr': ':mage_peau_moyennement_claire:',
        'pt': ':mago_pele_morena_clara:',
        'it': ':mago_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D9\U0001F3FD': { # 🧙🏽
        'en' : ':mage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier(in)_mittlere_hautfarbe:',
        'es': ':persona_maga_tono_de_piel_medio:',
        'fr': u':mage_peau_légèrement_mate:',
        'pt': ':mago_pele_morena:',
        'it': ':mago_carnagione_olivastra:'
    },
    u'\U0001FA84': { # 🪄
        'en' : ':magic_wand:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':zauberstab:',
        'es': u':varita_mágica:',
        'fr': ':baguette_magique:',
        'pt': u':varinha_mágica:',
        'it': ':bacchetta_magica:'
    },
    u'\U0001F9F2': { # 🧲
        'en' : ':magnet:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':magnet:',
        'es': u':imán:',
        'fr': ':aimant:',
        'pt': u':ímã:',
        'it': ':calamita:'
    },
    u'\U0001F50D': { # 🔍
        'en' : ':magnifying_glass_tilted_left:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mag:'],
        'variant': True,
        'de': ':lupe_nach_links:',
        'es': ':lupa_orientada_hacia_la_izquierda:',
        'fr': u':loupe_orientée_à_gauche:',
        'pt': ':lupa_para_a_esquerda:',
        'it': ':lente_di_ingrandimento_rivolta_a_sinistra:'
    },
    u'\U0001F50E': { # 🔎
        'en' : ':magnifying_glass_tilted_right:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mag_right:'],
        'de': ':lupe_nach_rechts:',
        'es': ':lupa_orientada_hacia_la_derecha:',
        'fr': u':loupe_orientée_à_droite:',
        'pt': ':lupa_para_a_direita:',
        'it': ':lente_di_ingrandimento_rivolta_a_destra:'
    },
    u'\U0001F004': { # 🀄
        'en' : ':mahjong_red_dragon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mahjong:'],
        'variant': True,
        'de': ':mahjong-stein:',
        'es': u':dragón_rojo_de_mahjong:',
        'fr': ':dragon_rouge_mahjong:',
        'pt': u':dragão_vermelho_de_mahjong:',
        'it': ':tessera_mahjong:'
    },
    u'\U00002642\U0000FE0F': { # ♂️
        'en' : ':male_sign:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': u':männersymbol:',
        'es': ':signo_masculino:',
        'fr': u':symbole_de_l’homme:',
        'pt': u':símbolo_de_masculino:',
        'it': ':simbolo_genere_maschile:'
    },
    u'\U00002642': { # ♂
        'en' : ':male_sign:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': u':männersymbol:',
        'es': ':signo_masculino:',
        'fr': u':symbole_de_l’homme:',
        'pt': u':símbolo_de_masculino:',
        'it': ':simbolo_genere_maschile:'
    },
    u'\U0001F9A3': { # 🦣
        'en' : ':mammoth:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mammut:',
        'es': ':mamut:',
        'fr': ':mammouth:',
        'pt': ':mamute:',
        'it': ':mammut:'
    },
    u'\U0001F468': { # 👨
        'en' : ':man:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':mann:',
        'es': ':hombre:',
        'fr': ':homme:',
        'pt': ':homem:',
        'it': ':uomo:'
    },
    u'\U0001F468\U0000200D\U0001F3A8': { # 👨‍🎨
        'en' : ':man_artist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler:',
        'es': ':artista_hombre:',
        'fr': ':artiste_homme:',
        'pt': u':artista_plástico:',
        'it': ':artista_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3A8': { # 👨🏿‍🎨
        'en' : ':man_artist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_dunkle_hautfarbe:',
        'es': ':artista_hombre_tono_de_piel_oscuro:',
        'fr': u':artiste_homme_peau_foncée:',
        'pt': u':artista_plástico_pele_escura:',
        'it': ':artista_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3A8': { # 👨🏻‍🎨
        'en' : ':man_artist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_helle_hautfarbe:',
        'es': ':artista_hombre_tono_de_piel_claro:',
        'fr': ':artiste_homme_peau_claire:',
        'pt': u':artista_plástico_pele_clara:',
        'it': ':artista_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3A8': { # 👨🏾‍🎨
        'en' : ':man_artist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mitteldunkle_hautfarbe:',
        'es': ':artista_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':artiste_homme_peau_mate:',
        'pt': u':artista_plástico_pele_morena_escura:',
        'it': ':artista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3A8': { # 👨🏼‍🎨
        'en' : ':man_artist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mittelhelle_hautfarbe:',
        'es': ':artista_hombre_tono_de_piel_claro_medio:',
        'fr': ':artiste_homme_peau_moyennement_claire:',
        'pt': u':artista_plástico_pele_morena_clara:',
        'it': ':artista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3A8': { # 👨🏽‍🎨
        'en' : ':man_artist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':künstler_mittlere_hautfarbe:',
        'es': ':artista_hombre_tono_de_piel_medio:',
        'fr': u':artiste_homme_peau_légèrement_mate:',
        'pt': u':artista_plástico_pele_morena:',
        'it': ':artista_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F680': { # 👨‍🚀
        'en' : ':man_astronaut:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut:',
        'es': ':astronauta_hombre:',
        'fr': ':astronaute_homme:',
        'pt': ':astronauta_homem:',
        'it': ':astronauta_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F680': { # 👨🏿‍🚀
        'en' : ':man_astronaut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut_dunkle_hautfarbe:',
        'es': ':astronauta_hombre_tono_de_piel_oscuro:',
        'fr': u':astronaute_homme_peau_foncée:',
        'pt': ':astronauta_homem_pele_escura:',
        'it': ':astronauta_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F680': { # 👨🏻‍🚀
        'en' : ':man_astronaut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut_helle_hautfarbe:',
        'es': ':astronauta_hombre_tono_de_piel_claro:',
        'fr': ':astronaute_homme_peau_claire:',
        'pt': ':astronauta_homem_pele_clara:',
        'it': ':astronauta_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F680': { # 👨🏾‍🚀
        'en' : ':man_astronaut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut_mitteldunkle_hautfarbe:',
        'es': ':astronauta_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':astronaute_homme_peau_mate:',
        'pt': ':astronauta_homem_pele_morena_escura:',
        'it': ':astronauta_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F680': { # 👨🏼‍🚀
        'en' : ':man_astronaut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut_mittelhelle_hautfarbe:',
        'es': ':astronauta_hombre_tono_de_piel_claro_medio:',
        'fr': ':astronaute_homme_peau_moyennement_claire:',
        'pt': ':astronauta_homem_pele_morena_clara:',
        'it': ':astronauta_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F680': { # 👨🏽‍🚀
        'en' : ':man_astronaut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':astronaut_mittlere_hautfarbe:',
        'es': ':astronauta_hombre_tono_de_piel_medio:',
        'fr': u':astronaute_homme_peau_légèrement_mate:',
        'pt': ':astronauta_homem_pele_morena:',
        'it': ':astronauta_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9B2': { # 👨‍🦲
        'en' : ':man_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_glatze:',
        'es': ':hombre_sin_pelo:',
        'fr': ':homme_chauve:',
        'pt': ':homem_careca:',
        'it': ':uomo_calvo:'
    },
    u'\U0001F9D4\U0000200D\U00002642\U0000FE0F': { # 🧔‍♂️
        'en' : ':man_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0000200D\U00002642': { # 🧔‍♂
        'en' : ':man_beard:',
        'status' : minimally_qualified,
        'E' : 13.1,
        'de': ':mann_bart:',
        'es': ':hombre_barba:',
        'fr': ':homme_barbu:',
        'pt': ':homem_barbudo:',
        'it': ':uomo_con_la_barba:'
    },
    u'\U0001F6B4\U0000200D\U00002642\U0000FE0F': { # 🚴‍♂️
        'en' : ':man_biking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer:',
        'es': ':hombre_en_bicicleta:',
        'fr': ':cycliste_homme:',
        'pt': ':homem_ciclista:',
        'it': ':ciclista_uomo:'
    },
    u'\U0001F6B4\U0000200D\U00002642': { # 🚴‍♂
        'en' : ':man_biking:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':radfahrer:',
        'es': ':hombre_en_bicicleta:',
        'fr': ':cycliste_homme:',
        'pt': ':homem_ciclista:',
        'it': ':ciclista_uomo:'
    },
    u'\U0001F6B4\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🚴🏿‍♂️
        'en' : ':man_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer_dunkle_hautfarbe:',
        'es': ':hombre_en_bicicleta_tono_de_piel_oscuro:',
        'fr': u':cycliste_homme_peau_foncée:',
        'pt': ':homem_ciclista_pele_escura:',
        'it': ':ciclista_uomo_carnagione_scura:'
    },
    u'\U0001F6B4\U0001F3FF\U0000200D\U00002642': { # 🚴🏿‍♂
        'en' : ':man_biking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🚴🏻‍♂️
        'en' : ':man_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer_helle_hautfarbe:',
        'es': ':hombre_en_bicicleta_tono_de_piel_claro:',
        'fr': ':cycliste_homme_peau_claire:',
        'pt': ':homem_ciclista_pele_clara:',
        'it': ':ciclista_uomo_carnagione_chiara:'
    },
    u'\U0001F6B4\U0001F3FB\U0000200D\U00002642': { # 🚴🏻‍♂
        'en' : ':man_biking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🚴🏾‍♂️
        'en' : ':man_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer_mitteldunkle_hautfarbe:',
        'es': ':hombre_en_bicicleta_tono_de_piel_oscuro_medio:',
        'fr': ':cycliste_homme_peau_mate:',
        'pt': ':homem_ciclista_pele_morena_escura:',
        'it': ':ciclista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B4\U0001F3FE\U0000200D\U00002642': { # 🚴🏾‍♂
        'en' : ':man_biking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🚴🏼‍♂️
        'en' : ':man_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer_mittelhelle_hautfarbe:',
        'es': ':hombre_en_bicicleta_tono_de_piel_claro_medio:',
        'fr': ':cycliste_homme_peau_moyennement_claire:',
        'pt': ':homem_ciclista_pele_morena_clara:',
        'it': ':ciclista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B4\U0001F3FC\U0000200D\U00002642': { # 🚴🏼‍♂
        'en' : ':man_biking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B4\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🚴🏽‍♂️
        'en' : ':man_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':radfahrer_mittlere_hautfarbe:',
        'es': ':hombre_en_bicicleta_tono_de_piel_medio:',
        'fr': u':cycliste_homme_peau_légèrement_mate:',
        'pt': ':homem_ciclista_pele_morena:',
        'it': ':ciclista_uomo_carnagione_olivastra:'
    },
    u'\U0001F6B4\U0001F3FD\U0000200D\U00002642': { # 🚴🏽‍♂
        'en' : ':man_biking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F471\U0000200D\U00002642\U0000FE0F': { # 👱‍♂️
        'en' : ':man_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_blond:',
        'es': ':hombre_rubio:',
        'fr': ':homme_blond:',
        'pt': ':homem_cabelo_loiro:',
        'it': ':uomo_biondo:'
    },
    u'\U0001F471\U0000200D\U00002642': { # 👱‍♂
        'en' : ':man_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_blond:',
        'es': ':hombre_rubio:',
        'fr': ':homme_blond:',
        'pt': ':homem_cabelo_loiro:',
        'it': ':uomo_biondo:'
    },
    u'\U000026F9\U0000FE0F\U0000200D\U00002642\U0000FE0F': { # ⛹️‍♂️
        'en' : ':man_bouncing_ball:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball:',
        'es': u':hombre_botando_un_balón:',
        'fr': ':homme_avec_ballon:',
        'pt': ':homem_jogando_basquete:',
        'it': ':uomo_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9\U0000200D\U00002642\U0000FE0F': { # ⛹‍♂️
        'en' : ':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U000026F9\U0000FE0F\U0000200D\U00002642': { # ⛹️‍♂
        'en' : ':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U000026F9\U0000200D\U00002642': { # ⛹‍♂
        'en' : ':man_bouncing_ball:',
        'status' : unqualified,
        'E' : 4,
        'de': ':mann_mit_ball:',
        'es': u':hombre_botando_un_balón:',
        'fr': ':homme_avec_ballon:',
        'pt': ':homem_jogando_basquete:',
        'it': ':uomo_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # ⛹🏿‍♂️
        'en' : ':man_bouncing_ball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball_dunkle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_oscuro:',
        'fr': u':homme_avec_ballon_peau_foncée:',
        'pt': ':homem_jogando_basquete_pele_escura:',
        'it': ':uomo_che_fa_rimbalzare_una_palla_carnagione_scura:'
    },
    u'\U000026F9\U0001F3FF\U0000200D\U00002642': { # ⛹🏿‍♂
        'en' : ':man_bouncing_ball_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # ⛹🏻‍♂️
        'en' : ':man_bouncing_ball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball_helle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_claro:',
        'fr': ':homme_avec_ballon_peau_claire:',
        'pt': ':homem_jogando_basquete_pele_clara:',
        'it': ':uomo_che_fa_rimbalzare_una_palla_carnagione_chiara:'
    },
    u'\U000026F9\U0001F3FB\U0000200D\U00002642': { # ⛹🏻‍♂
        'en' : ':man_bouncing_ball_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # ⛹🏾‍♂️
        'en' : ':man_bouncing_ball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball_mitteldunkle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_oscuro_medio:',
        'fr': ':homme_avec_ballon_peau_mate:',
        'pt': ':homem_jogando_basquete_pele_morena_escura:',
        'it': ':uomo_che_fa_rimbalzare_una_palla_carnagione_abbastanza_scura:'
    },
    u'\U000026F9\U0001F3FE\U0000200D\U00002642': { # ⛹🏾‍♂
        'en' : ':man_bouncing_ball_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # ⛹🏼‍♂️
        'en' : ':man_bouncing_ball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball_mittelhelle_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_claro_medio:',
        'fr': ':homme_avec_ballon_peau_moyennement_claire:',
        'pt': ':homem_jogando_basquete_pele_morena_clara:',
        'it': ':uomo_che_fa_rimbalzare_una_palla_carnagione_abbastanza_chiara:'
    },
    u'\U000026F9\U0001F3FC\U0000200D\U00002642': { # ⛹🏼‍♂
        'en' : ':man_bouncing_ball_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U000026F9\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # ⛹🏽‍♂️
        'en' : ':man_bouncing_ball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_ball_mittlere_hautfarbe:',
        'es': u':hombre_botando_un_balón_tono_de_piel_medio:',
        'fr': u':homme_avec_ballon_peau_légèrement_mate:',
        'pt': ':homem_jogando_basquete_pele_morena:',
        'it': ':uomo_che_fa_rimbalzare_una_palla_carnagione_olivastra:'
    },
    u'\U000026F9\U0001F3FD\U0000200D\U00002642': { # ⛹🏽‍♂
        'en' : ':man_bouncing_ball_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0000200D\U00002642\U0000FE0F': { # 🙇‍♂️
        'en' : ':man_bowing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann:',
        'es': ':hombre_haciendo_una_reverencia:',
        'fr': u':homme_qui_s’incline:',
        'pt': u':homem_fazendo_reverência:',
        'it': ':uomo_che_fa_inchino_profondo:'
    },
    u'\U0001F647\U0000200D\U00002642': { # 🙇‍♂
        'en' : ':man_bowing:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann:',
        'es': ':hombre_haciendo_una_reverencia:',
        'fr': u':homme_qui_s’incline:',
        'pt': u':homem_fazendo_reverência:',
        'it': ':uomo_che_fa_inchino_profondo:'
    },
    u'\U0001F647\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙇🏿‍♂️
        'en' : ':man_bowing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_una_reverencia_tono_de_piel_oscuro:',
        'fr': u':homme_qui_s’incline_peau_foncée:',
        'pt': u':homem_fazendo_reverência_pele_escura:',
        'it': ':uomo_che_fa_inchino_profondo_carnagione_scura:'
    },
    u'\U0001F647\U0001F3FF\U0000200D\U00002642': { # 🙇🏿‍♂
        'en' : ':man_bowing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙇🏻‍♂️
        'en' : ':man_bowing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann_helle_hautfarbe:',
        'es': ':hombre_haciendo_una_reverencia_tono_de_piel_claro:',
        'fr': u':homme_qui_s’incline_peau_claire:',
        'pt': u':homem_fazendo_reverência_pele_clara:',
        'it': ':uomo_che_fa_inchino_profondo_carnagione_chiara:'
    },
    u'\U0001F647\U0001F3FB\U0000200D\U00002642': { # 🙇🏻‍♂
        'en' : ':man_bowing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙇🏾‍♂️
        'en' : ':man_bowing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_una_reverencia_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_s’incline_peau_mate:',
        'pt': u':homem_fazendo_reverência_pele_morena_escura:',
        'it': ':uomo_che_fa_inchino_profondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F647\U0001F3FE\U0000200D\U00002642': { # 🙇🏾‍♂
        'en' : ':man_bowing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙇🏼‍♂️
        'en' : ':man_bowing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_una_reverencia_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_s’incline_peau_moyennement_claire:',
        'pt': u':homem_fazendo_reverência_pele_morena_clara:',
        'it': ':uomo_che_fa_inchino_profondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F647\U0001F3FC\U0000200D\U00002642': { # 🙇🏼‍♂
        'en' : ':man_bowing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F647\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙇🏽‍♂️
        'en' : ':man_bowing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_verbeugender_mann_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_una_reverencia_tono_de_piel_medio:',
        'fr': u':homme_qui_s’incline_peau_légèrement_mate:',
        'pt': u':homem_fazendo_reverência_pele_morena:',
        'it': ':uomo_che_fa_inchino_profondo_carnagione_olivastra:'
    },
    u'\U0001F647\U0001F3FD\U0000200D\U00002642': { # 🙇🏽‍♂
        'en' : ':man_bowing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0000200D\U00002642\U0000FE0F': { # 🤸‍♂️
        'en' : ':man_cartwheeling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann:',
        'es': ':hombre_haciendo_una_voltereta_lateral:',
        'fr': ':homme_faisant_la_roue:',
        'pt': ':homem_fazendo_estrela:',
        'it': ':uomo_che_fa_la_ruota:'
    },
    u'\U0001F938\U0000200D\U00002642': { # 🤸‍♂
        'en' : ':man_cartwheeling:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann:',
        'es': ':hombre_dando_una_voltereta_lateral:',
        'fr': ':homme_faisant_la_roue:',
        'pt': ':homem_fazendo_estrela:',
        'it': ':uomo_che_fa_la_ruota:'
    },
    u'\U0001F938\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤸🏿‍♂️
        'en' : ':man_cartwheeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_una_voltereta_lateral_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_la_roue_peau_foncée:',
        'pt': ':homem_fazendo_estrela_pele_escura:',
        'it': ':uomo_che_fa_la_ruota_carnagione_scura:'
    },
    u'\U0001F938\U0001F3FF\U0000200D\U00002642': { # 🤸🏿‍♂
        'en' : ':man_cartwheeling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤸🏻‍♂️
        'en' : ':man_cartwheeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann_helle_hautfarbe:',
        'es': ':hombre_haciendo_una_voltereta_lateral_tono_de_piel_claro:',
        'fr': ':homme_faisant_la_roue_peau_claire:',
        'pt': ':homem_fazendo_estrela_pele_clara:',
        'it': ':uomo_che_fa_la_ruota_carnagione_chiara:'
    },
    u'\U0001F938\U0001F3FB\U0000200D\U00002642': { # 🤸🏻‍♂
        'en' : ':man_cartwheeling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤸🏾‍♂️
        'en' : ':man_cartwheeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_una_voltereta_lateral_tono_de_piel_oscuro_medio:',
        'fr': ':homme_faisant_la_roue_peau_mate:',
        'pt': ':homem_fazendo_estrela_pele_morena_escura:',
        'it': ':uomo_che_fa_la_ruota_carnagione_abbastanza_scura:'
    },
    u'\U0001F938\U0001F3FE\U0000200D\U00002642': { # 🤸🏾‍♂
        'en' : ':man_cartwheeling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤸🏼‍♂️
        'en' : ':man_cartwheeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_una_voltereta_lateral_tono_de_piel_claro_medio:',
        'fr': ':homme_faisant_la_roue_peau_moyennement_claire:',
        'pt': ':homem_fazendo_estrela_pele_morena_clara:',
        'it': ':uomo_che_fa_la_ruota_carnagione_abbastanza_chiara:'
    },
    u'\U0001F938\U0001F3FC\U0000200D\U00002642': { # 🤸🏼‍♂
        'en' : ':man_cartwheeling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F938\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤸🏽‍♂️
        'en' : ':man_cartwheeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':rad_schlagender_mann_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_una_voltereta_lateral_tono_de_piel_medio:',
        'fr': u':homme_faisant_la_roue_peau_légèrement_mate:',
        'pt': ':homem_fazendo_estrela_pele_morena:',
        'it': ':uomo_che_fa_la_ruota_carnagione_olivastra:'
    },
    u'\U0001F938\U0001F3FD\U0000200D\U00002642': { # 🤸🏽‍♂
        'en' : ':man_cartwheeling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9D7\U0000200D\U00002642\U0000FE0F': { # 🧗‍♂️
        'en' : ':man_climbing:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger:',
        'es': ':hombre_escalando:',
        'fr': ':homme_qui_grimpe:',
        'pt': ':homem_escalando:',
        'it': ':scalatore:'
    },
    u'\U0001F9D7\U0000200D\U00002642': { # 🧗‍♂
        'en' : ':man_climbing:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':bergsteiger:',
        'es': ':hombre_escalando:',
        'fr': ':homme_qui_grimpe:',
        'pt': ':homem_escalando:',
        'it': ':scalatore:'
    },
    u'\U0001F9D7\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧗🏿‍♂️
        'en' : ':man_climbing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger_dunkle_hautfarbe:',
        'es': ':hombre_escalando_tono_de_piel_oscuro:',
        'fr': u':homme_qui_grimpe_peau_foncée:',
        'pt': ':homem_escalando_pele_escura:',
        'it': ':scalatore_carnagione_scura:'
    },
    u'\U0001F9D7\U0001F3FF\U0000200D\U00002642': { # 🧗🏿‍♂
        'en' : ':man_climbing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧗🏻‍♂️
        'en' : ':man_climbing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger_helle_hautfarbe:',
        'es': ':hombre_escalando_tono_de_piel_claro:',
        'fr': ':homme_qui_grimpe_peau_claire:',
        'pt': ':homem_escalando_pele_clara:',
        'it': ':scalatore_carnagione_chiara:'
    },
    u'\U0001F9D7\U0001F3FB\U0000200D\U00002642': { # 🧗🏻‍♂
        'en' : ':man_climbing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧗🏾‍♂️
        'en' : ':man_climbing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger_mitteldunkle_hautfarbe:',
        'es': ':hombre_escalando_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_grimpe_peau_mate:',
        'pt': ':homem_escalando_pele_morena_escura:',
        'it': ':scalatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D7\U0001F3FE\U0000200D\U00002642': { # 🧗🏾‍♂
        'en' : ':man_climbing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧗🏼‍♂️
        'en' : ':man_climbing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger_mittelhelle_hautfarbe:',
        'es': ':hombre_escalando_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_grimpe_peau_moyennement_claire:',
        'pt': ':homem_escalando_pele_morena_clara:',
        'it': ':scalatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D7\U0001F3FC\U0000200D\U00002642': { # 🧗🏼‍♂
        'en' : ':man_climbing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D7\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧗🏽‍♂️
        'en' : ':man_climbing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger_mittlere_hautfarbe:',
        'es': ':hombre_escalando_tono_de_piel_medio:',
        'fr': u':homme_qui_grimpe_peau_légèrement_mate:',
        'pt': ':homem_escalando_pele_morena:',
        'it': ':scalatore_carnagione_olivastra:'
    },
    u'\U0001F9D7\U0001F3FD\U0000200D\U00002642': { # 🧗🏽‍♂
        'en' : ':man_climbing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F477\U0000200D\U00002642\U0000FE0F': { # 👷‍♂️
        'en' : ':man_construction_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter:',
        'es': u':profesional_de_la_construcción_hombre:',
        'fr': u':ouvrier_du_bâtiment:',
        'pt': ':pedreiro:',
        'it': ':operaio_edile_uomo:'
    },
    u'\U0001F477\U0000200D\U00002642': { # 👷‍♂
        'en' : ':man_construction_worker:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':bauarbeiter:',
        'es': u':profesional_de_la_construcción_hombre:',
        'fr': u':ouvrier_du_bâtiment:',
        'pt': ':pedreiro:',
        'it': ':operaio_edile_uomo:'
    },
    u'\U0001F477\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 👷🏿‍♂️
        'en' : ':man_construction_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter_dunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_oscuro:',
        'fr': u':ouvrier_du_bâtiment_peau_foncée:',
        'pt': ':pedreiro_pele_escura:',
        'it': ':operaio_edile_uomo_carnagione_scura:'
    },
    u'\U0001F477\U0001F3FF\U0000200D\U00002642': { # 👷🏿‍♂
        'en' : ':man_construction_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 👷🏻‍♂️
        'en' : ':man_construction_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter_helle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_claro:',
        'fr': u':ouvrier_du_bâtiment_peau_claire:',
        'pt': ':pedreiro_pele_clara:',
        'it': ':operaio_edile_uomo_carnagione_chiara:'
    },
    u'\U0001F477\U0001F3FB\U0000200D\U00002642': { # 👷🏻‍♂
        'en' : ':man_construction_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 👷🏾‍♂️
        'en' : ':man_construction_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_mate:',
        'pt': ':pedreiro_pele_morena_escura:',
        'it': ':operaio_edile_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F477\U0001F3FE\U0000200D\U00002642': { # 👷🏾‍♂
        'en' : ':man_construction_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 👷🏼‍♂️
        'en' : ':man_construction_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_claro_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_moyennement_claire:',
        'pt': ':pedreiro_pele_morena_clara:',
        'it': ':operaio_edile_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F477\U0001F3FC\U0000200D\U00002642': { # 👷🏼‍♂
        'en' : ':man_construction_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F477\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 👷🏽‍♂️
        'en' : ':man_construction_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauarbeiter_mittlere_hautfarbe:',
        'es': u':profesional_de_la_construcción_hombre_tono_de_piel_medio:',
        'fr': u':ouvrier_du_bâtiment_peau_légèrement_mate:',
        'pt': ':pedreiro_pele_morena:',
        'it': ':operaio_edile_uomo_carnagione_olivastra:'
    },
    u'\U0001F477\U0001F3FD\U0000200D\U00002642': { # 👷🏽‍♂
        'en' : ':man_construction_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F373': { # 👨‍🍳
        'en' : ':man_cook:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch:',
        'es': ':chef_hombre:',
        'fr': ':cuisinier:',
        'pt': ':cozinheiro:',
        'it': ':cuoco:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F373': { # 👨🏿‍🍳
        'en' : ':man_cook_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch_dunkle_hautfarbe:',
        'es': ':chef_hombre_tono_de_piel_oscuro:',
        'fr': u':cuisinier_peau_foncée:',
        'pt': ':cozinheiro_pele_escura:',
        'it': ':cuoco_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F373': { # 👨🏻‍🍳
        'en' : ':man_cook_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch_helle_hautfarbe:',
        'es': ':chef_hombre_tono_de_piel_claro:',
        'fr': ':cuisinier_peau_claire:',
        'pt': ':cozinheiro_pele_clara:',
        'it': ':cuoco_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F373': { # 👨🏾‍🍳
        'en' : ':man_cook_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch_mitteldunkle_hautfarbe:',
        'es': ':chef_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':cuisinier_peau_mate:',
        'pt': ':cozinheiro_pele_morena_escura:',
        'it': ':cuoco_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F373': { # 👨🏼‍🍳
        'en' : ':man_cook_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch_mittelhelle_hautfarbe:',
        'es': ':chef_hombre_tono_de_piel_claro_medio:',
        'fr': ':cuisinier_peau_moyennement_claire:',
        'pt': ':cozinheiro_pele_morena_clara:',
        'it': ':cuoco_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F373': { # 👨🏽‍🍳
        'en' : ':man_cook_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':koch_mittlere_hautfarbe:',
        'es': ':chef_hombre_tono_de_piel_medio:',
        'fr': u':cuisinier_peau_légèrement_mate:',
        'pt': ':cozinheiro_pele_morena:',
        'it': ':cuoco_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9B1': { # 👨‍🦱
        'en' : ':man_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_lockiges_haar:',
        'es': ':hombre_pelo_rizado:',
        'fr': u':homme_cheveux_bouclés:',
        'pt': ':homem_cabelo_cacheado:',
        'it': ':uomo_capelli_ricci:'
    },
    u'\U0001F57A': { # 🕺
        'en' : ':man_dancing:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann:',
        'es': ':hombre_bailando:',
        'fr': ':danseur:',
        'pt': u':homem_dançando:',
        'it': ':uomo_che_balla:'
    },
    u'\U0001F57A\U0001F3FF': { # 🕺🏿
        'en' : ':man_dancing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann_dunkle_hautfarbe:',
        'es': ':hombre_bailando_tono_de_piel_oscuro:',
        'fr': u':danseur_peau_foncée:',
        'pt': u':homem_dançando_pele_escura:',
        'it': ':uomo_che_balla_carnagione_scura:'
    },
    u'\U0001F57A\U0001F3FB': { # 🕺🏻
        'en' : ':man_dancing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann_helle_hautfarbe:',
        'es': ':hombre_bailando_tono_de_piel_claro:',
        'fr': ':danseur_peau_claire:',
        'pt': u':homem_dançando_pele_clara:',
        'it': ':uomo_che_balla_carnagione_chiara:'
    },
    u'\U0001F57A\U0001F3FE': { # 🕺🏾
        'en' : ':man_dancing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_bailando_tono_de_piel_oscuro_medio:',
        'fr': ':danseur_peau_mate:',
        'pt': u':homem_dançando_pele_morena_escura:',
        'it': ':uomo_che_balla_carnagione_abbastanza_scura:'
    },
    u'\U0001F57A\U0001F3FC': { # 🕺🏼
        'en' : ':man_dancing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_bailando_tono_de_piel_claro_medio:',
        'fr': ':danseur_peau_moyennement_claire:',
        'pt': u':homem_dançando_pele_morena_clara:',
        'it': ':uomo_che_balla_carnagione_abbastanza_chiara:'
    },
    u'\U0001F57A\U0001F3FD': { # 🕺🏽
        'en' : ':man_dancing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':tanzender_mann_mittlere_hautfarbe:',
        'es': ':hombre_bailando_tono_de_piel_medio:',
        'fr': u':danseur_peau_légèrement_mate:',
        'pt': u':homem_dançando_pele_morena:',
        'it': ':uomo_che_balla_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FF': { # 👨🏿
        'en' : ':man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mann_dunkle_hautfarbe:',
        'es': ':hombre_tono_de_piel_oscuro:',
        'fr': u':homme_peau_foncée:',
        'pt': ':homem_pele_escura:',
        'it': ':uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B2': { # 👨🏿‍🦲
        'en' : ':man_dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_dunkle_hautfarbe,_glatze:',
        'es': ':hombre_tono_de_piel_oscuro_y_sin_pelo:',
        'fr': u':homme_peau_foncée_et_chauve:',
        'pt': ':homem_pele_escura_e_careca:',
        'it': ':uomo_carnagione_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧔🏿‍♂️
        'en' : ':man_dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FF\U0000200D\U00002642': { # 🧔🏿‍♂
        'en' : ':man_dark_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 👱🏿‍♂️
        'en' : ':man_dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_dunkle_hautfarbe,_blond:',
        'es': ':hombre_rubio_tono_de_piel_oscuro:',
        'fr': u':homme_blond_peau_foncée:',
        'pt': ':homem_pele_escura_e_cabelo_loiro:',
        'it': ':uomo_biondo_carnagione_scura:'
    },
    u'\U0001F471\U0001F3FF\U0000200D\U00002642': { # 👱🏿‍♂
        'en' : ':man_dark_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B1': { # 👨🏿‍🦱
        'en' : ':man_dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_dunkle_hautfarbe,_lockiges_haar:',
        'es': ':hombre_tono_de_piel_oscuro_y_pelo_rizado:',
        'fr': u':homme_peau_foncée_et_cheveux_bouclés:',
        'pt': ':homem_pele_escura_e_cabelo_cacheado:',
        'it': ':uomo_carnagione_scura_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B0': { # 👨🏿‍🦰
        'en' : ':man_dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_dunkle_hautfarbe,_rotes_haar:',
        'es': ':hombre_tono_de_piel_oscuro_y_pelo_pelirrojo:',
        'fr': u':homme_peau_foncée_et_cheveux_roux:',
        'pt': ':homem_pele_escura_e_cabelo_vermelho:',
        'it': ':uomo_carnagione_scura_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9B3': { # 👨🏿‍🦳
        'en' : ':man_dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_dunkle_hautfarbe,_weißes_haar:',
        'es': ':hombre_tono_de_piel_oscuro_y_pelo_blanco:',
        'fr': u':homme_peau_foncée_et_cheveux_blancs:',
        'pt': ':homem_pele_escura_e_cabelo_branco:',
        'it': ':uomo_carnagione_scura_e_capelli_bianchi:'
    },
    u'\U0001F575\U0000FE0F\U0000200D\U00002642\U0000FE0F': { # 🕵️‍♂️
        'en' : ':man_detective:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv:',
        'es': ':detective_hombre:',
        'fr': u':détective_homme:',
        'pt': ':detetive_homem:',
        'it': ':investigatore:'
    },
    u'\U0001F575\U0000200D\U00002642\U0000FE0F': { # 🕵‍♂️
        'en' : ':man_detective:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F575\U0000FE0F\U0000200D\U00002642': { # 🕵️‍♂
        'en' : ':man_detective:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F575\U0000200D\U00002642': { # 🕵‍♂
        'en' : ':man_detective:',
        'status' : unqualified,
        'E' : 4,
        'de': ':detektiv:',
        'es': ':detective_hombre:',
        'fr': u':détective_homme:',
        'pt': ':detetive_homem:',
        'it': ':investigatore:'
    },
    u'\U0001F575\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🕵🏿‍♂️
        'en' : ':man_detective_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv_dunkle_hautfarbe:',
        'es': ':detective_hombre_tono_de_piel_oscuro:',
        'fr': u':détective_homme_peau_foncée:',
        'pt': ':detetive_homem_pele_escura:',
        'it': ':investigatore_carnagione_scura:'
    },
    u'\U0001F575\U0001F3FF\U0000200D\U00002642': { # 🕵🏿‍♂
        'en' : ':man_detective_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🕵🏻‍♂️
        'en' : ':man_detective_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv_helle_hautfarbe:',
        'es': ':detective_hombre_tono_de_piel_claro:',
        'fr': u':détective_homme_peau_claire:',
        'pt': ':detetive_homem_pele_clara:',
        'it': ':investigatore_carnagione_chiara:'
    },
    u'\U0001F575\U0001F3FB\U0000200D\U00002642': { # 🕵🏻‍♂
        'en' : ':man_detective_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🕵🏾‍♂️
        'en' : ':man_detective_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv_mitteldunkle_hautfarbe:',
        'es': ':detective_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':détective_homme_peau_mate:',
        'pt': ':detetive_homem_pele_morena_escura:',
        'it': ':investigatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F575\U0001F3FE\U0000200D\U00002642': { # 🕵🏾‍♂
        'en' : ':man_detective_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🕵🏼‍♂️
        'en' : ':man_detective_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv_mittelhelle_hautfarbe:',
        'es': ':detective_hombre_tono_de_piel_claro_medio:',
        'fr': u':détective_homme_peau_moyennement_claire:',
        'pt': ':detetive_homem_pele_morena_clara:',
        'it': ':investigatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F575\U0001F3FC\U0000200D\U00002642': { # 🕵🏼‍♂
        'en' : ':man_detective_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F575\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🕵🏽‍♂️
        'en' : ':man_detective_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':detektiv_mittlere_hautfarbe:',
        'es': ':detective_hombre_tono_de_piel_medio:',
        'fr': u':détective_homme_peau_légèrement_mate:',
        'pt': ':detetive_homem_pele_morena:',
        'it': ':investigatore_carnagione_olivastra:'
    },
    u'\U0001F575\U0001F3FD\U0000200D\U00002642': { # 🕵🏽‍♂
        'en' : ':man_detective_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DD\U0000200D\U00002642\U0000FE0F': { # 🧝‍♂️
        'en' : ':man_elf:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf:',
        'es': ':elfo_hombre:',
        'fr': ':elfe_homme:',
        'pt': ':elfo_homem:',
        'it': ':elfo_uomo:'
    },
    u'\U0001F9DD\U0000200D\U00002642': { # 🧝‍♂
        'en' : ':man_elf:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':elf:',
        'es': ':elfo_hombre:',
        'fr': ':elfe_homme:',
        'pt': ':elfo_homem:',
        'it': ':elfo_uomo:'
    },
    u'\U0001F9DD\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧝🏿‍♂️
        'en' : ':man_elf_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf_dunkle_hautfarbe:',
        'es': ':elfo_hombre_tono_de_piel_oscuro:',
        'fr': u':elfe_homme_peau_foncée:',
        'pt': ':elfo_homem_pele_escura:',
        'it': ':elfo_uomo_carnagione_scura:'
    },
    u'\U0001F9DD\U0001F3FF\U0000200D\U00002642': { # 🧝🏿‍♂
        'en' : ':man_elf_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧝🏻‍♂️
        'en' : ':man_elf_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf_helle_hautfarbe:',
        'es': ':elfo_hombre_tono_de_piel_claro:',
        'fr': ':elfe_homme_peau_claire:',
        'pt': ':elfo_homem_pele_clara:',
        'it': ':elfo_uomo_carnagione_chiara:'
    },
    u'\U0001F9DD\U0001F3FB\U0000200D\U00002642': { # 🧝🏻‍♂
        'en' : ':man_elf_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧝🏾‍♂️
        'en' : ':man_elf_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf_mitteldunkle_hautfarbe:',
        'es': ':elfo_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':elfe_homme_peau_mate:',
        'pt': ':elfo_homem_pele_morena_escura:',
        'it': ':elfo_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DD\U0001F3FE\U0000200D\U00002642': { # 🧝🏾‍♂
        'en' : ':man_elf_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧝🏼‍♂️
        'en' : ':man_elf_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf_mittelhelle_hautfarbe:',
        'es': ':elfo_hombre_tono_de_piel_claro_medio:',
        'fr': ':elfe_homme_peau_moyennement_claire:',
        'pt': ':elfo_homem_pele_morena_clara:',
        'it': ':elfo_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DD\U0001F3FC\U0000200D\U00002642': { # 🧝🏼‍♂
        'en' : ':man_elf_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DD\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧝🏽‍♂️
        'en' : ':man_elf_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':elf_mittlere_hautfarbe:',
        'es': ':elfo_hombre_tono_de_piel_medio:',
        'fr': u':elfe_homme_peau_légèrement_mate:',
        'pt': ':elfo_homem_pele_morena:',
        'it': ':elfo_uomo_carnagione_olivastra:'
    },
    u'\U0001F9DD\U0001F3FD\U0000200D\U00002642': { # 🧝🏽‍♂
        'en' : ':man_elf_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F926\U0000200D\U00002642\U0000FE0F': { # 🤦‍♂️
        'en' : ':man_facepalming:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann:',
        'es': ':hombre_con_la_mano_en_la_frente:',
        'fr': ':homme_avec_la_paume_sur_le_visage:',
        'pt': ':homem_decepcionado:',
        'it': ':uomo_esasperato:'
    },
    u'\U0001F926\U0000200D\U00002642': { # 🤦‍♂
        'en' : ':man_facepalming:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann:',
        'es': ':hombre_con_la_mano_en_la_frente:',
        'fr': ':homme_avec_la_paume_sur_le_visage:',
        'pt': ':homem_decepcionado:',
        'it': ':uomo_esasperato:'
    },
    u'\U0001F926\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤦🏿‍♂️
        'en' : ':man_facepalming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann_dunkle_hautfarbe:',
        'es': ':hombre_con_la_mano_en_la_frente_tono_de_piel_oscuro:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_foncée:',
        'pt': ':homem_decepcionado_pele_escura:',
        'it': ':uomo_esasperato_carnagione_scura:'
    },
    u'\U0001F926\U0001F3FF\U0000200D\U00002642': { # 🤦🏿‍♂
        'en' : ':man_facepalming_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤦🏻‍♂️
        'en' : ':man_facepalming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann_helle_hautfarbe:',
        'es': ':hombre_con_la_mano_en_la_frente_tono_de_piel_claro:',
        'fr': ':homme_avec_la_paume_sur_le_visage_peau_claire:',
        'pt': ':homem_decepcionado_pele_clara:',
        'it': ':uomo_esasperato_carnagione_chiara:'
    },
    u'\U0001F926\U0001F3FB\U0000200D\U00002642': { # 🤦🏻‍♂
        'en' : ':man_facepalming_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤦🏾‍♂️
        'en' : ':man_facepalming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_con_la_mano_en_la_frente_tono_de_piel_oscuro_medio:',
        'fr': ':homme_avec_la_paume_sur_le_visage_peau_mate:',
        'pt': ':homem_decepcionado_pele_morena_escura:',
        'it': ':uomo_esasperato_carnagione_abbastanza_scura:'
    },
    u'\U0001F926\U0001F3FE\U0000200D\U00002642': { # 🤦🏾‍♂
        'en' : ':man_facepalming_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤦🏼‍♂️
        'en' : ':man_facepalming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_con_la_mano_en_la_frente_tono_de_piel_claro_medio:',
        'fr': ':homme_avec_la_paume_sur_le_visage_peau_moyennement_claire:',
        'pt': ':homem_decepcionado_pele_morena_clara:',
        'it': ':uomo_esasperato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F926\U0001F3FC\U0000200D\U00002642': { # 🤦🏼‍♂
        'en' : ':man_facepalming_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F926\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤦🏽‍♂️
        'en' : ':man_facepalming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':sich_an_den_kopf_fassender_mann_mittlere_hautfarbe:',
        'es': ':hombre_con_la_mano_en_la_frente_tono_de_piel_medio:',
        'fr': u':homme_avec_la_paume_sur_le_visage_peau_légèrement_mate:',
        'pt': ':homem_decepcionado_pele_morena:',
        'it': ':uomo_esasperato_carnagione_olivastra:'
    },
    u'\U0001F926\U0001F3FD\U0000200D\U00002642': { # 🤦🏽‍♂
        'en' : ':man_facepalming_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3ED': { # 👨‍🏭
        'en' : ':man_factory_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter:',
        'es': ':profesional_industrial_hombre:',
        'fr': ':ouvrier:',
        'pt': u':operário:',
        'it': ':operaio:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3ED': { # 👨🏿‍🏭
        'en' : ':man_factory_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter_dunkle_hautfarbe:',
        'es': ':profesional_industrial_hombre_tono_de_piel_oscuro:',
        'fr': u':ouvrier_peau_foncée:',
        'pt': u':operário_pele_escura:',
        'it': ':operaio_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3ED': { # 👨🏻‍🏭
        'en' : ':man_factory_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter_helle_hautfarbe:',
        'es': ':profesional_industrial_hombre_tono_de_piel_claro:',
        'fr': ':ouvrier_peau_claire:',
        'pt': u':operário_pele_clara:',
        'it': ':operaio_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3ED': { # 👨🏾‍🏭
        'en' : ':man_factory_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter_mitteldunkle_hautfarbe:',
        'es': ':profesional_industrial_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':ouvrier_peau_mate:',
        'pt': u':operário_pele_morena_escura:',
        'it': ':operaio_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3ED': { # 👨🏼‍🏭
        'en' : ':man_factory_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter_mittelhelle_hautfarbe:',
        'es': ':profesional_industrial_hombre_tono_de_piel_claro_medio:',
        'fr': ':ouvrier_peau_moyennement_claire:',
        'pt': u':operário_pele_morena_clara:',
        'it': ':operaio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3ED': { # 👨🏽‍🏭
        'en' : ':man_factory_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':fabrikarbeiter_mittlere_hautfarbe:',
        'es': ':profesional_industrial_hombre_tono_de_piel_medio:',
        'fr': u':ouvrier_peau_légèrement_mate:',
        'pt': u':operário_pele_morena:',
        'it': ':operaio_carnagione_olivastra:'
    },
    u'\U0001F9DA\U0000200D\U00002642\U0000FE0F': { # 🧚‍♂️
        'en' : ':man_fairy:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee:',
        'es': ':hada_hombre:',
        'fr': u':féetaud:',
        'pt': ':homem_fada:',
        'it': ':folletto_alato:'
    },
    u'\U0001F9DA\U0000200D\U00002642': { # 🧚‍♂
        'en' : ':man_fairy:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': u':männliche_fee:',
        'es': ':hada_hombre:',
        'fr': u':féetaud:',
        'pt': ':homem_fada:',
        'it': ':folletto_alato:'
    },
    u'\U0001F9DA\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧚🏿‍♂️
        'en' : ':man_fairy_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_dunkle_hautfarbe:',
        'es': ':hada_hombre_tono_de_piel_oscuro:',
        'fr': u':féetaud_peau_foncée:',
        'pt': ':homem_fada_pele_escura:',
        'it': ':folletto_alato_carnagione_scura:'
    },
    u'\U0001F9DA\U0001F3FF\U0000200D\U00002642': { # 🧚🏿‍♂
        'en' : ':man_fairy_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧚🏻‍♂️
        'en' : ':man_fairy_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_helle_hautfarbe:',
        'es': ':hada_hombre_tono_de_piel_claro:',
        'fr': u':féetaud_peau_claire:',
        'pt': ':homem_fada_pele_clara:',
        'it': ':folletto_alato_carnagione_chiara:'
    },
    u'\U0001F9DA\U0001F3FB\U0000200D\U00002642': { # 🧚🏻‍♂
        'en' : ':man_fairy_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧚🏾‍♂️
        'en' : ':man_fairy_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mitteldunkle_hautfarbe:',
        'es': ':hada_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':féetaud_peau_mate:',
        'pt': ':homem_fada_pele_morena_escura:',
        'it': ':folletto_alato_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DA\U0001F3FE\U0000200D\U00002642': { # 🧚🏾‍♂
        'en' : ':man_fairy_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧚🏼‍♂️
        'en' : ':man_fairy_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mittelhelle_hautfarbe:',
        'es': ':hada_hombre_tono_de_piel_claro_medio:',
        'fr': u':féetaud_peau_moyennement_claire:',
        'pt': ':homem_fada_pele_morena_clara:',
        'it': ':folletto_alato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DA\U0001F3FC\U0000200D\U00002642': { # 🧚🏼‍♂
        'en' : ':man_fairy_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DA\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧚🏽‍♂️
        'en' : ':man_fairy_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männliche_fee_mittlere_hautfarbe:',
        'es': ':hada_hombre_tono_de_piel_medio:',
        'fr': u':féetaud_peau_légèrement_mate:',
        'pt': ':homem_fada_pele_morena:',
        'it': ':folletto_alato_carnagione_olivastra:'
    },
    u'\U0001F9DA\U0001F3FD\U0000200D\U00002642': { # 🧚🏽‍♂
        'en' : ':man_fairy_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F33E': { # 👨‍🌾
        'en' : ':man_farmer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer:',
        'es': ':profesional_de_la_agricultura_hombre:',
        'fr': ':fermier:',
        'pt': ':fazendeiro:',
        'it': ':contadino:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F33E': { # 👨🏿‍🌾
        'en' : ':man_farmer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer_dunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_hombre_tono_de_piel_oscuro:',
        'fr': u':fermier_peau_foncée:',
        'pt': ':fazendeiro_pele_escura:',
        'it': ':contadino_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F33E': { # 👨🏻‍🌾
        'en' : ':man_farmer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer_helle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_hombre_tono_de_piel_claro:',
        'fr': ':fermier_peau_claire:',
        'pt': ':fazendeiro_pele_clara:',
        'it': ':contadino_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F33E': { # 👨🏾‍🌾
        'en' : ':man_farmer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer_mitteldunkle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':fermier_peau_mate:',
        'pt': ':fazendeiro_pele_morena_escura:',
        'it': ':contadino_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F33E': { # 👨🏼‍🌾
        'en' : ':man_farmer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer_mittelhelle_hautfarbe:',
        'es': ':profesional_de_la_agricultura_hombre_tono_de_piel_claro_medio:',
        'fr': ':fermier_peau_moyennement_claire:',
        'pt': ':fazendeiro_pele_morena_clara:',
        'it': ':contadino_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F33E': { # 👨🏽‍🌾
        'en' : ':man_farmer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':bauer_mittlere_hautfarbe:',
        'es': ':profesional_de_la_agricultura_hombre_tono_de_piel_medio:',
        'fr': u':fermier_peau_légèrement_mate:',
        'pt': ':fazendeiro_pele_morena:',
        'it': ':contadino_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F37C': { # 👨‍🍼
        'en' : ':man_feeding_baby:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann:',
        'es': u':hombre_alimentando_a_bebé:',
        'fr': u':homme_allaitant_un_bébé:',
        'pt': u':homem_alimentando_bebê:',
        'it': ':uomo_che_allatta:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F37C': { # 👨🏿‍🍼
        'en' : ':man_feeding_baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann_dunkle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_oscuro:',
        'fr': u':homme_allaitant_un_bébé_peau_foncée:',
        'pt': u':homem_alimentando_bebê_pele_escura:',
        'it': ':uomo_che_allatta_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F37C': { # 👨🏻‍🍼
        'en' : ':man_feeding_baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann_helle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_claro:',
        'fr': u':homme_allaitant_un_bébé_peau_claire:',
        'pt': u':homem_alimentando_bebê_pele_clara:',
        'it': ':uomo_che_allatta_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F37C': { # 👨🏾‍🍼
        'en' : ':man_feeding_baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_oscuro_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_mate:',
        'pt': u':homem_alimentando_bebê_pele_morena_escura:',
        'it': ':uomo_che_allatta_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F37C': { # 👨🏼‍🍼
        'en' : ':man_feeding_baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_claro_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_moyennement_claire:',
        'pt': u':homem_alimentando_bebê_pele_morena_clara:',
        'it': ':uomo_che_allatta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F37C': { # 👨🏽‍🍼
        'en' : ':man_feeding_baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillender_mann_mittlere_hautfarbe:',
        'es': u':hombre_alimentando_a_bebé_tono_de_piel_medio:',
        'fr': u':homme_allaitant_un_bébé_peau_légèrement_mate:',
        'pt': u':homem_alimentando_bebê_pele_morena:',
        'it': ':uomo_che_allatta_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F692': { # 👨‍🚒
        'en' : ':man_firefighter:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann:',
        'es': ':bombero_hombre:',
        'fr': ':pompier_homme:',
        'pt': ':bombeiro_homem:',
        'it': ':pompiere_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F692': { # 👨🏿‍🚒
        'en' : ':man_firefighter_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann_dunkle_hautfarbe:',
        'es': ':bombero_hombre_tono_de_piel_oscuro:',
        'fr': u':pompier_homme_peau_foncée:',
        'pt': ':bombeiro_homem_pele_escura:',
        'it': ':pompiere_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F692': { # 👨🏻‍🚒
        'en' : ':man_firefighter_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann_helle_hautfarbe:',
        'es': ':bombero_hombre_tono_de_piel_claro:',
        'fr': ':pompier_homme_peau_claire:',
        'pt': ':bombeiro_homem_pele_clara:',
        'it': ':pompiere_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F692': { # 👨🏾‍🚒
        'en' : ':man_firefighter_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann_mitteldunkle_hautfarbe:',
        'es': ':bombero_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':pompier_homme_peau_mate:',
        'pt': ':bombeiro_homem_pele_morena_escura:',
        'it': ':pompiere_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F692': { # 👨🏼‍🚒
        'en' : ':man_firefighter_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann_mittelhelle_hautfarbe:',
        'es': ':bombero_hombre_tono_de_piel_claro_medio:',
        'fr': ':pompier_homme_peau_moyennement_claire:',
        'pt': ':bombeiro_homem_pele_morena_clara:',
        'it': ':pompiere_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F692': { # 👨🏽‍🚒
        'en' : ':man_firefighter_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':feuerwehrmann_mittlere_hautfarbe:',
        'es': ':bombero_hombre_tono_de_piel_medio:',
        'fr': u':pompier_homme_peau_légèrement_mate:',
        'pt': ':bombeiro_homem_pele_morena:',
        'it': ':pompiere_uomo_carnagione_olivastra:'
    },
    u'\U0001F64D\U0000200D\U00002642\U0000FE0F': { # 🙍‍♂️
        'en' : ':man_frowning:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann:',
        'es': u':hombre_frunciendo_el_ceño:',
        'fr': u':homme_fronçant_les_sourcils:',
        'pt': ':homem_franzindo_a_sobrancelha:',
        'it': ':uomo_corrucciato:'
    },
    u'\U0001F64D\U0000200D\U00002642': { # 🙍‍♂
        'en' : ':man_frowning:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':missmutiger_mann:',
        'es': u':hombre_frunciendo_el_ceño:',
        'fr': u':homme_fronçant_les_sourcils:',
        'pt': ':homem_franzindo_a_sobrancelha:',
        'it': ':uomo_corrucciato:'
    },
    u'\U0001F64D\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙍🏿‍♂️
        'en' : ':man_frowning_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann_dunkle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_oscuro:',
        'fr': u':homme_fronçant_les_sourcils_peau_foncée:',
        'pt': ':homem_franzindo_a_sobrancelha_pele_escura:',
        'it': ':uomo_corrucciato_carnagione_scura:'
    },
    u'\U0001F64D\U0001F3FF\U0000200D\U00002642': { # 🙍🏿‍♂
        'en' : ':man_frowning_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙍🏻‍♂️
        'en' : ':man_frowning_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann_helle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_claro:',
        'fr': u':homme_fronçant_les_sourcils_peau_claire:',
        'pt': ':homem_franzindo_a_sobrancelha_pele_clara:',
        'it': ':uomo_corrucciato_carnagione_chiara:'
    },
    u'\U0001F64D\U0001F3FB\U0000200D\U00002642': { # 🙍🏻‍♂
        'en' : ':man_frowning_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙍🏾‍♂️
        'en' : ':man_frowning_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann_mitteldunkle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_oscuro_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_mate:',
        'pt': ':homem_franzindo_a_sobrancelha_pele_morena_escura:',
        'it': ':uomo_corrucciato_carnagione_abbastanza_scura:'
    },
    u'\U0001F64D\U0001F3FE\U0000200D\U00002642': { # 🙍🏾‍♂
        'en' : ':man_frowning_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙍🏼‍♂️
        'en' : ':man_frowning_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann_mittelhelle_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_claro_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_moyennement_claire:',
        'pt': ':homem_franzindo_a_sobrancelha_pele_morena_clara:',
        'it': ':uomo_corrucciato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64D\U0001F3FC\U0000200D\U00002642': { # 🙍🏼‍♂
        'en' : ':man_frowning_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64D\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙍🏽‍♂️
        'en' : ':man_frowning_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':missmutiger_mann_mittlere_hautfarbe:',
        'es': u':hombre_frunciendo_el_ceño_tono_de_piel_medio:',
        'fr': u':homme_fronçant_les_sourcils_peau_légèrement_mate:',
        'pt': ':homem_franzindo_a_sobrancelha_pele_morena:',
        'it': ':uomo_corrucciato_carnagione_olivastra:'
    },
    u'\U0001F64D\U0001F3FD\U0000200D\U00002642': { # 🙍🏽‍♂
        'en' : ':man_frowning_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DE\U0000200D\U00002642\U0000FE0F': { # 🧞‍♂️
        'en' : ':man_genie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_flaschengeist:',
        'es': ':genio_hombre:',
        'fr': u':génie_homme:',
        'pt': u':homem_gênio:',
        'it': ':genio_uomo:'
    },
    u'\U0001F9DE\U0000200D\U00002642': { # 🧞‍♂
        'en' : ':man_genie:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': u':männlicher_flaschengeist:',
        'es': ':genio_hombre:',
        'fr': u':génie_homme:',
        'pt': u':homem_gênio:',
        'it': ':genio_uomo:'
    },
    u'\U0001F645\U0000200D\U00002642\U0000FE0F': { # 🙅‍♂️
        'en' : ':man_gesturing_NO:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen:',
        'es': ':hombre_haciendo_el_gesto_de_no:',
        'fr': u':homme_faisant_un_geste_d’interdiction:',
        'pt': u':homem_fazendo_gesto_de_“não”:',
        'it': ':uomo_con_gesto_di_rifiuto:'
    },
    u'\U0001F645\U0000200D\U00002642': { # 🙅‍♂
        'en' : ':man_gesturing_NO:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen:',
        'es': ':hombre_haciendo_el_gesto_de_no:',
        'fr': u':homme_faisant_un_geste_d’interdiction:',
        'pt': u':homem_fazendo_gesto_de_“não”:',
        'it': ':uomo_con_gesto_di_rifiuto:'
    },
    u'\U0001F645\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙅🏿‍♂️
        'en' : ':man_gesturing_NO_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_no_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_foncée:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_escura:',
        'it': ':uomo_con_gesto_di_rifiuto_carnagione_scura:'
    },
    u'\U0001F645\U0001F3FF\U0000200D\U00002642': { # 🙅🏿‍♂
        'en' : ':man_gesturing_NO_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙅🏻‍♂️
        'en' : ':man_gesturing_NO_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_helle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_no_tono_de_piel_claro:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_claire:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_clara:',
        'it': ':uomo_con_gesto_di_rifiuto_carnagione_chiara:'
    },
    u'\U0001F645\U0001F3FB\U0000200D\U00002642': { # 🙅🏻‍♂
        'en' : ':man_gesturing_NO_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙅🏾‍♂️
        'en' : ':man_gesturing_NO_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_no_tono_de_piel_oscuro_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_mate:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena_escura:',
        'it': ':uomo_con_gesto_di_rifiuto_carnagione_abbastanza_scura:'
    },
    u'\U0001F645\U0001F3FE\U0000200D\U00002642': { # 🙅🏾‍♂
        'en' : ':man_gesturing_NO_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙅🏼‍♂️
        'en' : ':man_gesturing_NO_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_no_tono_de_piel_claro_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_moyennement_claire:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena_clara:',
        'it': ':uomo_con_gesto_di_rifiuto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F645\U0001F3FC\U0000200D\U00002642': { # 🙅🏼‍♂
        'en' : ':man_gesturing_NO_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F645\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙅🏽‍♂️
        'en' : ':man_gesturing_NO_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_überkreuzten_armen_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_no_tono_de_piel_medio:',
        'fr': u':homme_faisant_un_geste_d’interdiction_peau_légèrement_mate:',
        'pt': u':homem_fazendo_gesto_de_“não”_pele_morena:',
        'it': ':uomo_con_gesto_di_rifiuto_carnagione_olivastra:'
    },
    u'\U0001F645\U0001F3FD\U0000200D\U00002642': { # 🙅🏽‍♂
        'en' : ':man_gesturing_NO_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0000200D\U00002642\U0000FE0F': { # 🙆‍♂️
        'en' : ':man_gesturing_OK:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo:',
        'fr': u':homme_faisant_un_geste_d’acceptation:',
        'pt': u':homem_fazendo_gesto_de_“ok”:',
        'it': ':uomo_con_gesto_ok:'
    },
    u'\U0001F646\U0000200D\U00002642': { # 🙆‍♂
        'en' : ':man_gesturing_OK:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo:',
        'fr': u':homme_faisant_un_geste_d’acceptation:',
        'pt': u':homem_fazendo_gesto_de_“ok”:',
        'it': ':uomo_con_gesto_ok:'
    },
    u'\U0001F646\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙆🏿‍♂️
        'en' : ':man_gesturing_OK_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_foncée:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_escura:',
        'it': ':uomo_con_gesto_ok_carnagione_scura:'
    },
    u'\U0001F646\U0001F3FF\U0000200D\U00002642': { # 🙆🏿‍♂
        'en' : ':man_gesturing_OK_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙆🏻‍♂️
        'en' : ':man_gesturing_OK_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_helle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_claire:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_clara:',
        'it': ':uomo_con_gesto_ok_carnagione_chiara:'
    },
    u'\U0001F646\U0001F3FB\U0000200D\U00002642': { # 🙆🏻‍♂
        'en' : ':man_gesturing_OK_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙆🏾‍♂️
        'en' : ':man_gesturing_OK_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_mate:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena_escura:',
        'it': ':uomo_con_gesto_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F646\U0001F3FE\U0000200D\U00002642': { # 🙆🏾‍♂
        'en' : ':man_gesturing_OK_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙆🏼‍♂️
        'en' : ':man_gesturing_OK_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_moyennement_claire:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena_clara:',
        'it': ':uomo_con_gesto_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F646\U0001F3FC\U0000200D\U00002642': { # 🙆🏼‍♂
        'en' : ':man_gesturing_OK_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F646\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙆🏽‍♂️
        'en' : ':man_gesturing_OK_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':mann_mit_händen_auf_dem_kopf_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_medio:',
        'fr': u':homme_faisant_un_geste_d’acceptation_peau_légèrement_mate:',
        'pt': u':homem_fazendo_gesto_de_“ok”_pele_morena:',
        'it': ':uomo_con_gesto_ok_carnagione_olivastra:'
    },
    u'\U0001F646\U0001F3FD\U0000200D\U00002642': { # 🙆🏽‍♂
        'en' : ':man_gesturing_OK_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0000200D\U00002642\U0000FE0F': { # 💇‍♂️
        'en' : ':man_getting_haircut:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden:',
        'es': u':hombre_cortándose_el_pelo:',
        'fr': ':homme_qui_se_fait_couper_les_cheveux:',
        'pt': ':homem_cortando_o_cabelo:',
        'it': ':taglio_di_capelli_per_uomo:'
    },
    u'\U0001F487\U0000200D\U00002642': { # 💇‍♂
        'en' : ':man_getting_haircut:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden:',
        'es': u':hombre_cortándose_el_pelo:',
        'fr': ':homme_qui_se_fait_couper_les_cheveux:',
        'pt': ':homem_cortando_o_cabelo:',
        'it': ':taglio_di_capelli_per_uomo:'
    },
    u'\U0001F487\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 💇🏿‍♂️
        'en' : ':man_getting_haircut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden_dunkle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_oscuro:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_foncée:',
        'pt': ':homem_cortando_o_cabelo_pele_escura:',
        'it': ':taglio_di_capelli_per_uomo_carnagione_scura:'
    },
    u'\U0001F487\U0001F3FF\U0000200D\U00002642': { # 💇🏿‍♂
        'en' : ':man_getting_haircut_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 💇🏻‍♂️
        'en' : ':man_getting_haircut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden_helle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_claro:',
        'fr': ':homme_qui_se_fait_couper_les_cheveux_peau_claire:',
        'pt': ':homem_cortando_o_cabelo_pele_clara:',
        'it': ':taglio_di_capelli_per_uomo_carnagione_chiara:'
    },
    u'\U0001F487\U0001F3FB\U0000200D\U00002642': { # 💇🏻‍♂
        'en' : ':man_getting_haircut_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 💇🏾‍♂️
        'en' : ':man_getting_haircut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden_mitteldunkle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_se_fait_couper_les_cheveux_peau_mate:',
        'pt': ':homem_cortando_o_cabelo_pele_morena_escura:',
        'it': ':taglio_di_capelli_per_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F487\U0001F3FE\U0000200D\U00002642': { # 💇🏾‍♂
        'en' : ':man_getting_haircut_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 💇🏼‍♂️
        'en' : ':man_getting_haircut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden_mittelhelle_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_se_fait_couper_les_cheveux_peau_moyennement_claire:',
        'pt': ':homem_cortando_o_cabelo_pele_morena_clara:',
        'it': ':taglio_di_capelli_per_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F487\U0001F3FC\U0000200D\U00002642': { # 💇🏼‍♂
        'en' : ':man_getting_haircut_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F487\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 💇🏽‍♂️
        'en' : ':man_getting_haircut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_beim_haareschneiden_mittlere_hautfarbe:',
        'es': u':hombre_cortándose_el_pelo_tono_de_piel_medio:',
        'fr': u':homme_qui_se_fait_couper_les_cheveux_peau_légèrement_mate:',
        'pt': ':homem_cortando_o_cabelo_pele_morena:',
        'it': ':taglio_di_capelli_per_uomo_carnagione_olivastra:'
    },
    u'\U0001F487\U0001F3FD\U0000200D\U00002642': { # 💇🏽‍♂
        'en' : ':man_getting_haircut_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0000200D\U00002642\U0000FE0F': { # 💆‍♂️
        'en' : ':man_getting_massage:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt:',
        'es': ':hombre_recibiendo_masaje:',
        'fr': ':homme_qui_se_fait_masser:',
        'pt': ':homem_recebendo_massagem_facial:',
        'it': ':uomo_che_riceve_un_massaggio:'
    },
    u'\U0001F486\U0000200D\U00002642': { # 💆‍♂
        'en' : ':man_getting_massage:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_der_eine_kopfmassage_bekommt:',
        'es': ':hombre_recibiendo_masaje:',
        'fr': ':homme_qui_se_fait_masser:',
        'pt': ':homem_recebendo_massagem_facial:',
        'it': ':uomo_che_riceve_un_massaggio:'
    },
    u'\U0001F486\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 💆🏿‍♂️
        'en' : ':man_getting_massage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt_dunkle_hautfarbe:',
        'es': ':hombre_recibiendo_masaje_tono_de_piel_oscuro:',
        'fr': u':homme_qui_se_fait_masser_peau_foncée:',
        'pt': ':homem_recebendo_massagem_facial_pele_escura:',
        'it': ':uomo_che_riceve_un_massaggio_carnagione_scura:'
    },
    u'\U0001F486\U0001F3FF\U0000200D\U00002642': { # 💆🏿‍♂
        'en' : ':man_getting_massage_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 💆🏻‍♂️
        'en' : ':man_getting_massage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt_helle_hautfarbe:',
        'es': ':hombre_recibiendo_masaje_tono_de_piel_claro:',
        'fr': ':homme_qui_se_fait_masser_peau_claire:',
        'pt': ':homem_recebendo_massagem_facial_pele_clara:',
        'it': ':uomo_che_riceve_un_massaggio_carnagione_chiara:'
    },
    u'\U0001F486\U0001F3FB\U0000200D\U00002642': { # 💆🏻‍♂
        'en' : ':man_getting_massage_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 💆🏾‍♂️
        'en' : ':man_getting_massage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt_mitteldunkle_hautfarbe:',
        'es': ':hombre_recibiendo_masaje_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_se_fait_masser_peau_mate:',
        'pt': ':homem_recebendo_massagem_facial_pele_morena_escura:',
        'it': ':uomo_che_riceve_un_massaggio_carnagione_abbastanza_scura:'
    },
    u'\U0001F486\U0001F3FE\U0000200D\U00002642': { # 💆🏾‍♂
        'en' : ':man_getting_massage_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 💆🏼‍♂️
        'en' : ':man_getting_massage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt_mittelhelle_hautfarbe:',
        'es': ':hombre_recibiendo_masaje_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_se_fait_masser_peau_moyennement_claire:',
        'pt': ':homem_recebendo_massagem_facial_pele_morena_clara:',
        'it': ':uomo_che_riceve_un_massaggio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F486\U0001F3FC\U0000200D\U00002642': { # 💆🏼‍♂
        'en' : ':man_getting_massage_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F486\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 💆🏽‍♂️
        'en' : ':man_getting_massage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann,_der_eine_kopfmassage_bekommt_mittlere_hautfarbe:',
        'es': ':hombre_recibiendo_masaje_tono_de_piel_medio:',
        'fr': u':homme_qui_se_fait_masser_peau_légèrement_mate:',
        'pt': ':homem_recebendo_massagem_facial_pele_morena:',
        'it': ':uomo_che_riceve_un_massaggio_carnagione_olivastra:'
    },
    u'\U0001F486\U0001F3FD\U0000200D\U00002642': { # 💆🏽‍♂
        'en' : ':man_getting_massage_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000FE0F\U0000200D\U00002642\U0000FE0F': { # 🏌️‍♂️
        'en' : ':man_golfing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer:',
        'es': ':hombre_jugando_al_golf:',
        'fr': ':golfeur:',
        'pt': ':homem_golfista:',
        'it': ':golfista_uomo:'
    },
    u'\U0001F3CC\U0000200D\U00002642\U0000FE0F': { # 🏌‍♂️
        'en' : ':man_golfing:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000FE0F\U0000200D\U00002642': { # 🏌️‍♂
        'en' : ':man_golfing:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CC\U0000200D\U00002642': { # 🏌‍♂
        'en' : ':man_golfing:',
        'status' : unqualified,
        'E' : 4,
        'de': ':golfer:',
        'es': ':hombre_jugando_al_golf:',
        'fr': ':golfeur:',
        'pt': ':homem_golfista:',
        'it': ':golfista_uomo:'
    },
    u'\U0001F3CC\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🏌🏿‍♂️
        'en' : ':man_golfing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer_dunkle_hautfarbe:',
        'es': ':hombre_jugando_al_golf_tono_de_piel_oscuro:',
        'fr': u':golfeur_peau_foncée:',
        'pt': ':homem_golfista_pele_escura:',
        'it': ':golfista_uomo_carnagione_scura:'
    },
    u'\U0001F3CC\U0001F3FF\U0000200D\U00002642': { # 🏌🏿‍♂
        'en' : ':man_golfing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🏌🏻‍♂️
        'en' : ':man_golfing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer_helle_hautfarbe:',
        'es': ':hombre_jugando_al_golf_tono_de_piel_claro:',
        'fr': ':golfeur_peau_claire:',
        'pt': ':homem_golfista_pele_clara:',
        'it': ':golfista_uomo_carnagione_chiara:'
    },
    u'\U0001F3CC\U0001F3FB\U0000200D\U00002642': { # 🏌🏻‍♂
        'en' : ':man_golfing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🏌🏾‍♂️
        'en' : ':man_golfing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer_mitteldunkle_hautfarbe:',
        'es': ':hombre_jugando_al_golf_tono_de_piel_oscuro_medio:',
        'fr': ':golfeur_peau_mate:',
        'pt': ':homem_golfista_pele_morena_escura:',
        'it': ':golfista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CC\U0001F3FE\U0000200D\U00002642': { # 🏌🏾‍♂
        'en' : ':man_golfing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🏌🏼‍♂️
        'en' : ':man_golfing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer_mittelhelle_hautfarbe:',
        'es': ':hombre_jugando_al_golf_tono_de_piel_claro_medio:',
        'fr': ':golfeur_peau_moyennement_claire:',
        'pt': ':homem_golfista_pele_morena_clara:',
        'it': ':golfista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CC\U0001F3FC\U0000200D\U00002642': { # 🏌🏼‍♂
        'en' : ':man_golfing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CC\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🏌🏽‍♂️
        'en' : ':man_golfing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':golfer_mittlere_hautfarbe:',
        'es': ':hombre_jugando_al_golf_tono_de_piel_medio:',
        'fr': u':golfeur_peau_légèrement_mate:',
        'pt': ':homem_golfista_pele_morena:',
        'it': ':golfista_uomo_carnagione_olivastra:'
    },
    u'\U0001F3CC\U0001F3FD\U0000200D\U00002642': { # 🏌🏽‍♂
        'en' : ':man_golfing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0000200D\U00002642\U0000FE0F': { # 💂‍♂️
        'en' : ':man_guard:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann:',
        'es': ':guardia_hombre:',
        'fr': ':garde_homme:',
        'pt': ':guarda_homem:',
        'it': ':guardia_uomo:'
    },
    u'\U0001F482\U0000200D\U00002642': { # 💂‍♂
        'en' : ':man_guard:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':wachsoldat:',
        'es': ':guardia_hombre:',
        'fr': ':garde_homme:',
        'pt': ':guarda_homem:',
        'it': ':guardia_uomo:'
    },
    u'\U0001F482\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 💂🏿‍♂️
        'en' : ':man_guard_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann_dunkle_hautfarbe:',
        'es': ':guardia_hombre_tono_de_piel_oscuro:',
        'fr': u':garde_homme_peau_foncée:',
        'pt': ':guarda_homem_pele_escura:',
        'it': ':guardia_uomo_carnagione_scura:'
    },
    u'\U0001F482\U0001F3FF\U0000200D\U00002642': { # 💂🏿‍♂
        'en' : ':man_guard_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 💂🏻‍♂️
        'en' : ':man_guard_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann_helle_hautfarbe:',
        'es': ':guardia_hombre_tono_de_piel_claro:',
        'fr': ':garde_homme_peau_claire:',
        'pt': ':guarda_homem_pele_clara:',
        'it': ':guardia_uomo_carnagione_chiara:'
    },
    u'\U0001F482\U0001F3FB\U0000200D\U00002642': { # 💂🏻‍♂
        'en' : ':man_guard_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 💂🏾‍♂️
        'en' : ':man_guard_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann_mitteldunkle_hautfarbe:',
        'es': ':guardia_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':garde_homme_peau_mate:',
        'pt': ':guarda_homem_pele_morena_escura:',
        'it': ':guardia_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F482\U0001F3FE\U0000200D\U00002642': { # 💂🏾‍♂
        'en' : ':man_guard_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 💂🏼‍♂️
        'en' : ':man_guard_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann_mittelhelle_hautfarbe:',
        'es': ':guardia_hombre_tono_de_piel_claro_medio:',
        'fr': ':garde_homme_peau_moyennement_claire:',
        'pt': ':guarda_homem_pele_morena_clara:',
        'it': ':guardia_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F482\U0001F3FC\U0000200D\U00002642': { # 💂🏼‍♂
        'en' : ':man_guard_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F482\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 💂🏽‍♂️
        'en' : ':man_guard_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wachmann_mittlere_hautfarbe:',
        'es': ':guardia_hombre_tono_de_piel_medio:',
        'fr': u':garde_homme_peau_légèrement_mate:',
        'pt': ':guarda_homem_pele_morena:',
        'it': ':guardia_uomo_carnagione_olivastra:'
    },
    u'\U0001F482\U0001F3FD\U0000200D\U00002642': { # 💂🏽‍♂
        'en' : ':man_guard_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U00002695\U0000FE0F': { # 👨‍⚕️
        'en' : ':man_health_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt:',
        'es': ':profesional_sanitario_hombre:',
        'fr': u':professionnel_de_la_santé:',
        'pt': u':homem_profissional_da_saúde:',
        'it': ':operatore_sanitario:'
    },
    u'\U0001F468\U0000200D\U00002695': { # 👨‍⚕
        'en' : ':man_health_worker:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':arzt:',
        'es': ':profesional_sanitario_hombre:',
        'fr': u':professionnel_de_la_santé:',
        'pt': u':homem_profissional_da_saúde:',
        'it': ':operatore_sanitario:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002695\U0000FE0F': { # 👨🏿‍⚕️
        'en' : ':man_health_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt_dunkle_hautfarbe:',
        'es': ':profesional_sanitario_hombre_tono_de_piel_oscuro:',
        'fr': u':professionnel_de_la_santé_peau_foncée:',
        'pt': u':homem_profissional_da_saúde_pele_escura:',
        'it': ':operatore_sanitario_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002695': { # 👨🏿‍⚕
        'en' : ':man_health_worker_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002695\U0000FE0F': { # 👨🏻‍⚕️
        'en' : ':man_health_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt_helle_hautfarbe:',
        'es': ':profesional_sanitario_hombre_tono_de_piel_claro:',
        'fr': u':professionnel_de_la_santé_peau_claire:',
        'pt': u':homem_profissional_da_saúde_pele_clara:',
        'it': ':operatore_sanitario_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002695': { # 👨🏻‍⚕
        'en' : ':man_health_worker_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002695\U0000FE0F': { # 👨🏾‍⚕️
        'en' : ':man_health_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt_mitteldunkle_hautfarbe:',
        'es': ':profesional_sanitario_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':professionnel_de_la_santé_peau_mate:',
        'pt': u':homem_profissional_da_saúde_pele_morena_escura:',
        'it': ':operatore_sanitario_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002695': { # 👨🏾‍⚕
        'en' : ':man_health_worker_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002695\U0000FE0F': { # 👨🏼‍⚕️
        'en' : ':man_health_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt_mittelhelle_hautfarbe:',
        'es': ':profesional_sanitario_hombre_tono_de_piel_claro_medio:',
        'fr': u':professionnel_de_la_santé_peau_moyennement_claire:',
        'pt': u':homem_profissional_da_saúde_pele_morena_clara:',
        'it': ':operatore_sanitario_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002695': { # 👨🏼‍⚕
        'en' : ':man_health_worker_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002695\U0000FE0F': { # 👨🏽‍⚕️
        'en' : ':man_health_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':arzt_mittlere_hautfarbe:',
        'es': ':profesional_sanitario_hombre_tono_de_piel_medio:',
        'fr': u':professionnel_de_la_santé_peau_légèrement_mate:',
        'pt': u':homem_profissional_da_saúde_pele_morena:',
        'it': ':operatore_sanitario_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002695': { # 👨🏽‍⚕
        'en' : ':man_health_worker_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9D8\U0000200D\U00002642\U0000FE0F': { # 🧘‍♂️
        'en' : ':man_in_lotus_position:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz:',
        'es': u':hombre_en_posición_de_loto:',
        'fr': ':homme_dans_la_posture_du_lotus:',
        'pt': u':homem_na_posição_de_lótus:',
        'it': ':uomo_in_posizione_del_loto:'
    },
    u'\U0001F9D8\U0000200D\U00002642': { # 🧘‍♂
        'en' : ':man_in_lotus_position:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz:',
        'es': u':hombre_en_posición_de_loto:',
        'fr': ':homme_dans_la_posture_du_lotus:',
        'pt': u':homem_na_posição_de_lótus:',
        'it': ':uomo_in_posizione_del_loto:'
    },
    u'\U0001F9D8\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧘🏿‍♂️
        'en' : ':man_in_lotus_position_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz_dunkle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_oscuro:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_foncée:',
        'pt': u':homem_na_posição_de_lótus_pele_escura:',
        'it': ':uomo_in_posizione_del_loto_carnagione_scura:'
    },
    u'\U0001F9D8\U0001F3FF\U0000200D\U00002642': { # 🧘🏿‍♂
        'en' : ':man_in_lotus_position_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧘🏻‍♂️
        'en' : ':man_in_lotus_position_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz_helle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_claro:',
        'fr': ':homme_dans_la_posture_du_lotus_peau_claire:',
        'pt': u':homem_na_posição_de_lótus_pele_clara:',
        'it': ':uomo_in_posizione_del_loto_carnagione_chiara:'
    },
    u'\U0001F9D8\U0001F3FB\U0000200D\U00002642': { # 🧘🏻‍♂
        'en' : ':man_in_lotus_position_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧘🏾‍♂️
        'en' : ':man_in_lotus_position_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_oscuro_medio:',
        'fr': ':homme_dans_la_posture_du_lotus_peau_mate:',
        'pt': u':homem_na_posição_de_lótus_pele_morena_escura:',
        'it': ':uomo_in_posizione_del_loto_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D8\U0001F3FE\U0000200D\U00002642': { # 🧘🏾‍♂
        'en' : ':man_in_lotus_position_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧘🏼‍♂️
        'en' : ':man_in_lotus_position_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz_mittelhelle_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_claro_medio:',
        'fr': ':homme_dans_la_posture_du_lotus_peau_moyennement_claire:',
        'pt': u':homem_na_posição_de_lótus_pele_morena_clara:',
        'it': ':uomo_in_posizione_del_loto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D8\U0001F3FC\U0000200D\U00002642': { # 🧘🏼‍♂
        'en' : ':man_in_lotus_position_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D8\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧘🏽‍♂️
        'en' : ':man_in_lotus_position_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_im_lotossitz_mittlere_hautfarbe:',
        'es': u':hombre_en_posición_de_loto_tono_de_piel_medio:',
        'fr': u':homme_dans_la_posture_du_lotus_peau_légèrement_mate:',
        'pt': u':homem_na_posição_de_lótus_pele_morena:',
        'it': ':uomo_in_posizione_del_loto_carnagione_olivastra:'
    },
    u'\U0001F9D8\U0001F3FD\U0000200D\U00002642': { # 🧘🏽‍♂
        'en' : ':man_in_lotus_position_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F9BD': { # 👨‍🦽
        'en' : ':man_in_manual_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl:',
        'es': ':hombre_en_silla_de_ruedas_manual:',
        'fr': ':homme_en_fauteuil_roulant_manuel:',
        'pt': ':homem_em_cadeira_de_rodas_manual:',
        'it': ':uomo_su_sedia_a_rotelle_manuale:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9BD': { # 👨🏿‍🦽
        'en' : ':man_in_manual_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl_dunkle_hautfarbe:',
        'es': ':hombre_en_silla_de_ruedas_manual_tono_de_piel_oscuro:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_foncée:',
        'pt': ':homem_em_cadeira_de_rodas_manual_pele_escura:',
        'it': ':uomo_su_sedia_a_rotelle_manuale_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9BD': { # 👨🏻‍🦽
        'en' : ':man_in_manual_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl_helle_hautfarbe:',
        'es': ':hombre_en_silla_de_ruedas_manual_tono_de_piel_claro:',
        'fr': ':homme_en_fauteuil_roulant_manuel_peau_claire:',
        'pt': ':homem_em_cadeira_de_rodas_manual_pele_clara:',
        'it': ':uomo_su_sedia_a_rotelle_manuale_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9BD': { # 👨🏾‍🦽
        'en' : ':man_in_manual_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': ':hombre_en_silla_de_ruedas_manual_tono_de_piel_oscuro_medio:',
        'fr': ':homme_en_fauteuil_roulant_manuel_peau_mate:',
        'pt': ':homem_em_cadeira_de_rodas_manual_pele_morena_escura:',
        'it': ':uomo_su_sedia_a_rotelle_manuale_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9BD': { # 👨🏼‍🦽
        'en' : ':man_in_manual_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl_mittelhelle_hautfarbe:',
        'es': ':hombre_en_silla_de_ruedas_manual_tono_de_piel_claro_medio:',
        'fr': ':homme_en_fauteuil_roulant_manuel_peau_moyennement_claire:',
        'pt': ':homem_em_cadeira_de_rodas_manual_pele_morena_clara:',
        'it': ':uomo_su_sedia_a_rotelle_manuale_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9BD': { # 👨🏽‍🦽
        'en' : ':man_in_manual_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_manuellem_rollstuhl_mittlere_hautfarbe:',
        'es': ':hombre_en_silla_de_ruedas_manual_tono_de_piel_medio:',
        'fr': u':homme_en_fauteuil_roulant_manuel_peau_légèrement_mate:',
        'pt': ':homem_em_cadeira_de_rodas_manual_pele_morena:',
        'it': ':uomo_su_sedia_a_rotelle_manuale_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F9BC': { # 👨‍🦼
        'en' : ':man_in_motorized_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica:',
        'fr': u':homme_en_fauteuil_motorisé:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9BC': { # 👨🏿‍🦼
        'en' : ':man_in_motorized_wheelchair_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl_dunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro:',
        'fr': u':homme_en_fauteuil_motorisé_peau_foncée:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada_pele_escura:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9BC': { # 👨🏻‍🦼
        'en' : ':man_in_motorized_wheelchair_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl_helle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_claro:',
        'fr': u':homme_en_fauteuil_motorisé_peau_claire:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada_pele_clara:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9BC': { # 👨🏾‍🦼
        'en' : ':man_in_motorized_wheelchair_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_oscuro_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_mate:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada_pele_morena_escura:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9BC': { # 👨🏼‍🦼
        'en' : ':man_in_motorized_wheelchair_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl_mittelhelle_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_claro_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_moyennement_claire:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada_pele_morena_clara:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9BC': { # 👨🏽‍🦼
        'en' : ':man_in_motorized_wheelchair_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_in_elektrischem_rollstuhl_mittlere_hautfarbe:',
        'es': u':hombre_en_silla_de_ruedas_eléctrica_tono_de_piel_medio:',
        'fr': u':homme_en_fauteuil_motorisé_peau_légèrement_mate:',
        'pt': ':homem_em_cadeira_de_rodas_motorizada_pele_morena:',
        'it': ':uomo_su_sedia_a_rotelle_motorizzata_carnagione_olivastra:'
    },
    u'\U0001F9D6\U0000200D\U00002642\U0000FE0F': { # 🧖‍♂️
        'en' : ':man_in_steamy_room:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna:',
        'es': ':hombre_en_una_sauna:',
        'fr': ':homme_au_hammam:',
        'pt': ':homem_na_sauna:',
        'it': ':uomo_in_sauna:'
    },
    u'\U0001F9D6\U0000200D\U00002642': { # 🧖‍♂
        'en' : ':man_in_steamy_room:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna:',
        'es': ':hombre_en_una_sauna:',
        'fr': ':homme_au_hammam:',
        'pt': ':homem_na_sauna:',
        'it': ':uomo_in_sauna:'
    },
    u'\U0001F9D6\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧖🏿‍♂️
        'en' : ':man_in_steamy_room_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna_dunkle_hautfarbe:',
        'es': ':hombre_en_una_sauna_tono_de_piel_oscuro:',
        'fr': u':homme_au_hammam_peau_foncée:',
        'pt': ':homem_na_sauna_pele_escura:',
        'it': ':uomo_in_sauna_carnagione_scura:'
    },
    u'\U0001F9D6\U0001F3FF\U0000200D\U00002642': { # 🧖🏿‍♂
        'en' : ':man_in_steamy_room_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧖🏻‍♂️
        'en' : ':man_in_steamy_room_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna_helle_hautfarbe:',
        'es': ':hombre_en_una_sauna_tono_de_piel_claro:',
        'fr': ':homme_au_hammam_peau_claire:',
        'pt': ':homem_na_sauna_pele_clara:',
        'it': ':uomo_in_sauna_carnagione_chiara:'
    },
    u'\U0001F9D6\U0001F3FB\U0000200D\U00002642': { # 🧖🏻‍♂
        'en' : ':man_in_steamy_room_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧖🏾‍♂️
        'en' : ':man_in_steamy_room_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna_mitteldunkle_hautfarbe:',
        'es': ':hombre_en_una_sauna_tono_de_piel_oscuro_medio:',
        'fr': ':homme_au_hammam_peau_mate:',
        'pt': ':homem_na_sauna_pele_morena_escura:',
        'it': ':uomo_in_sauna_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D6\U0001F3FE\U0000200D\U00002642': { # 🧖🏾‍♂
        'en' : ':man_in_steamy_room_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧖🏼‍♂️
        'en' : ':man_in_steamy_room_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna_mittelhelle_hautfarbe:',
        'es': ':hombre_en_una_sauna_tono_de_piel_claro_medio:',
        'fr': ':homme_au_hammam_peau_moyennement_claire:',
        'pt': ':homem_na_sauna_pele_morena_clara:',
        'it': ':uomo_in_sauna_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D6\U0001F3FC\U0000200D\U00002642': { # 🧖🏼‍♂
        'en' : ':man_in_steamy_room_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D6\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧖🏽‍♂️
        'en' : ':man_in_steamy_room_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_in_dampfsauna_mittlere_hautfarbe:',
        'es': ':hombre_en_una_sauna_tono_de_piel_medio:',
        'fr': u':homme_au_hammam_peau_légèrement_mate:',
        'pt': ':homem_na_sauna_pele_morena:',
        'it': ':uomo_in_sauna_carnagione_olivastra:'
    },
    u'\U0001F9D6\U0001F3FD\U0000200D\U00002642': { # 🧖🏽‍♂
        'en' : ':man_in_steamy_room_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F935\U0000200D\U00002642\U0000FE0F': { # 🤵‍♂️
        'en' : ':man_in_tuxedo:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking:',
        'es': ':hombre_con_esmoquin:',
        'fr': ':homme_en_smoking:',
        'pt': ':homem_de_smoking:',
        'it': ':uomo_in_smoking:'
    },
    u'\U0001F935\U0000200D\U00002642': { # 🤵‍♂
        'en' : ':man_in_tuxedo:',
        'status' : minimally_qualified,
        'E' : 13,
        'de': ':mann_im_smoking:',
        'es': ':hombre_con_esmoquin:',
        'fr': ':homme_en_smoking:',
        'pt': ':homem_de_smoking:',
        'it': ':uomo_in_smoking:'
    },
    u'\U0001F935\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤵🏿‍♂️
        'en' : ':man_in_tuxedo_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking_dunkle_hautfarbe:',
        'es': ':hombre_con_esmoquin_tono_de_piel_oscuro:',
        'fr': u':homme_en_smoking_peau_foncée:',
        'pt': ':homem_de_smoking_pele_escura:',
        'it': ':uomo_in_smoking_carnagione_scura:'
    },
    u'\U0001F935\U0001F3FF\U0000200D\U00002642': { # 🤵🏿‍♂
        'en' : ':man_in_tuxedo_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤵🏻‍♂️
        'en' : ':man_in_tuxedo_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking_helle_hautfarbe:',
        'es': ':hombre_con_esmoquin_tono_de_piel_claro:',
        'fr': ':homme_en_smoking_peau_claire:',
        'pt': ':homem_de_smoking_pele_clara:',
        'it': ':uomo_in_smoking_carnagione_chiara:'
    },
    u'\U0001F935\U0001F3FB\U0000200D\U00002642': { # 🤵🏻‍♂
        'en' : ':man_in_tuxedo_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤵🏾‍♂️
        'en' : ':man_in_tuxedo_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking_mitteldunkle_hautfarbe:',
        'es': ':hombre_con_esmoquin_tono_de_piel_oscuro_medio:',
        'fr': ':homme_en_smoking_peau_mate:',
        'pt': ':homem_de_smoking_pele_morena_escura:',
        'it': ':uomo_in_smoking_carnagione_abbastanza_scura:'
    },
    u'\U0001F935\U0001F3FE\U0000200D\U00002642': { # 🤵🏾‍♂
        'en' : ':man_in_tuxedo_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤵🏼‍♂️
        'en' : ':man_in_tuxedo_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking_mittelhelle_hautfarbe:',
        'es': ':hombre_con_esmoquin_tono_de_piel_claro_medio:',
        'fr': ':homme_en_smoking_peau_moyennement_claire:',
        'pt': ':homem_de_smoking_pele_morena_clara:',
        'it': ':uomo_in_smoking_carnagione_abbastanza_chiara:'
    },
    u'\U0001F935\U0001F3FC\U0000200D\U00002642': { # 🤵🏼‍♂
        'en' : ':man_in_tuxedo_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F935\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤵🏽‍♂️
        'en' : ':man_in_tuxedo_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_im_smoking_mittlere_hautfarbe:',
        'es': ':hombre_con_esmoquin_tono_de_piel_medio:',
        'fr': u':homme_en_smoking_peau_légèrement_mate:',
        'pt': ':homem_de_smoking_pele_morena:',
        'it': ':uomo_in_smoking_carnagione_olivastra:'
    },
    u'\U0001F935\U0001F3FD\U0000200D\U00002642': { # 🤵🏽‍♂
        'en' : ':man_in_tuxedo_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F468\U0000200D\U00002696\U0000FE0F': { # 👨‍⚖️
        'en' : ':man_judge:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter:',
        'es': ':fiscal_hombre:',
        'fr': ':juge_homme:',
        'pt': ':juiz:',
        'it': ':giudice_uomo:'
    },
    u'\U0001F468\U0000200D\U00002696': { # 👨‍⚖
        'en' : ':man_judge:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':richter:',
        'es': ':fiscal_hombre:',
        'fr': ':juge_homme:',
        'pt': ':juiz:',
        'it': ':giudice_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002696\U0000FE0F': { # 👨🏿‍⚖️
        'en' : ':man_judge_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter_dunkle_hautfarbe:',
        'es': ':fiscal_hombre_tono_de_piel_oscuro:',
        'fr': u':juge_homme_peau_foncée:',
        'pt': ':juiz_pele_escura:',
        'it': ':giudice_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002696': { # 👨🏿‍⚖
        'en' : ':man_judge_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002696\U0000FE0F': { # 👨🏻‍⚖️
        'en' : ':man_judge_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter_helle_hautfarbe:',
        'es': ':fiscal_hombre_tono_de_piel_claro:',
        'fr': ':juge_homme_peau_claire:',
        'pt': ':juiz_pele_clara:',
        'it': ':giudice_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002696': { # 👨🏻‍⚖
        'en' : ':man_judge_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002696\U0000FE0F': { # 👨🏾‍⚖️
        'en' : ':man_judge_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter_mitteldunkle_hautfarbe:',
        'es': ':fiscal_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':juge_homme_peau_mate:',
        'pt': ':juiz_pele_morena_escura:',
        'it': ':giudice_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002696': { # 👨🏾‍⚖
        'en' : ':man_judge_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002696\U0000FE0F': { # 👨🏼‍⚖️
        'en' : ':man_judge_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter_mittelhelle_hautfarbe:',
        'es': ':fiscal_hombre_tono_de_piel_claro_medio:',
        'fr': ':juge_homme_peau_moyennement_claire:',
        'pt': ':juiz_pele_morena_clara:',
        'it': ':giudice_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002696': { # 👨🏼‍⚖
        'en' : ':man_judge_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002696\U0000FE0F': { # 👨🏽‍⚖️
        'en' : ':man_judge_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':richter_mittlere_hautfarbe:',
        'es': ':fiscal_hombre_tono_de_piel_medio:',
        'fr': u':juge_homme_peau_légèrement_mate:',
        'pt': ':juiz_pele_morena:',
        'it': ':giudice_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002696': { # 👨🏽‍⚖
        'en' : ':man_judge_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0000200D\U00002642\U0000FE0F': { # 🤹‍♂️
        'en' : ':man_juggling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur:',
        'es': ':hombre_haciendo_malabares:',
        'fr': ':jongleur:',
        'pt': ':homem_malabarista:',
        'it': ':giocoliere_uomo:'
    },
    u'\U0001F939\U0000200D\U00002642': { # 🤹‍♂
        'en' : ':man_juggling:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':jongleur:',
        'es': ':hombre_haciendo_malabares:',
        'fr': ':jongleur:',
        'pt': ':homem_malabarista:',
        'it': ':giocoliere_uomo:'
    },
    u'\U0001F939\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤹🏿‍♂️
        'en' : ':man_juggling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_malabares_tono_de_piel_oscuro:',
        'fr': u':jongleur_peau_foncée:',
        'pt': ':homem_malabarista_pele_escura:',
        'it': ':giocoliere_uomo_carnagione_scura:'
    },
    u'\U0001F939\U0001F3FF\U0000200D\U00002642': { # 🤹🏿‍♂
        'en' : ':man_juggling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤹🏻‍♂️
        'en' : ':man_juggling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur_helle_hautfarbe:',
        'es': ':hombre_haciendo_malabares_tono_de_piel_claro:',
        'fr': ':jongleur_peau_claire:',
        'pt': ':homem_malabarista_pele_clara:',
        'it': ':giocoliere_uomo_carnagione_chiara:'
    },
    u'\U0001F939\U0001F3FB\U0000200D\U00002642': { # 🤹🏻‍♂
        'en' : ':man_juggling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤹🏾‍♂️
        'en' : ':man_juggling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_malabares_tono_de_piel_oscuro_medio:',
        'fr': ':jongleur_peau_mate:',
        'pt': ':homem_malabarista_pele_morena_escura:',
        'it': ':giocoliere_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F939\U0001F3FE\U0000200D\U00002642': { # 🤹🏾‍♂
        'en' : ':man_juggling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤹🏼‍♂️
        'en' : ':man_juggling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_malabares_tono_de_piel_claro_medio:',
        'fr': ':jongleur_peau_moyennement_claire:',
        'pt': ':homem_malabarista_pele_morena_clara:',
        'it': ':giocoliere_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F939\U0001F3FC\U0000200D\U00002642': { # 🤹🏼‍♂
        'en' : ':man_juggling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F939\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤹🏽‍♂️
        'en' : ':man_juggling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':jongleur_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_malabares_tono_de_piel_medio:',
        'fr': u':jongleur_peau_légèrement_mate:',
        'pt': ':homem_malabarista_pele_morena:',
        'it': ':giocoliere_uomo_carnagione_olivastra:'
    },
    u'\U0001F939\U0001F3FD\U0000200D\U00002642': { # 🤹🏽‍♂
        'en' : ':man_juggling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9CE\U0000200D\U00002642\U0000FE0F': { # 🧎‍♂️
        'en' : ':man_kneeling:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann:',
        'es': ':hombre_de_rodillas:',
        'fr': u':homme_à_genoux:',
        'pt': ':homem_ajoelhando:',
        'it': ':uomo_inginocchiato:'
    },
    u'\U0001F9CE\U0000200D\U00002642': { # 🧎‍♂
        'en' : ':man_kneeling:',
        'status' : minimally_qualified,
        'E' : 12,
        'de': ':kniender_mann:',
        'es': ':hombre_de_rodillas:',
        'fr': u':homme_à_genoux:',
        'pt': ':homem_ajoelhando:',
        'it': ':uomo_inginocchiato:'
    },
    u'\U0001F9CE\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧎🏿‍♂️
        'en' : ':man_kneeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann_dunkle_hautfarbe:',
        'es': ':hombre_de_rodillas_tono_de_piel_oscuro:',
        'fr': u':homme_à_genoux_peau_foncée:',
        'pt': ':homem_ajoelhando_pele_escura:',
        'it': ':uomo_inginocchiato_carnagione_scura:'
    },
    u'\U0001F9CE\U0001F3FF\U0000200D\U00002642': { # 🧎🏿‍♂
        'en' : ':man_kneeling_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧎🏻‍♂️
        'en' : ':man_kneeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann_helle_hautfarbe:',
        'es': ':hombre_de_rodillas_tono_de_piel_claro:',
        'fr': u':homme_à_genoux_peau_claire:',
        'pt': ':homem_ajoelhando_pele_clara:',
        'it': ':uomo_inginocchiato_carnagione_chiara:'
    },
    u'\U0001F9CE\U0001F3FB\U0000200D\U00002642': { # 🧎🏻‍♂
        'en' : ':man_kneeling_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧎🏾‍♂️
        'en' : ':man_kneeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_de_rodillas_tono_de_piel_oscuro_medio:',
        'fr': u':homme_à_genoux_peau_mate:',
        'pt': ':homem_ajoelhando_pele_morena_escura:',
        'it': ':uomo_inginocchiato_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CE\U0001F3FE\U0000200D\U00002642': { # 🧎🏾‍♂
        'en' : ':man_kneeling_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧎🏼‍♂️
        'en' : ':man_kneeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_de_rodillas_tono_de_piel_claro_medio:',
        'fr': u':homme_à_genoux_peau_moyennement_claire:',
        'pt': ':homem_ajoelhando_pele_morena_clara:',
        'it': ':uomo_inginocchiato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CE\U0001F3FC\U0000200D\U00002642': { # 🧎🏼‍♂
        'en' : ':man_kneeling_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CE\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧎🏽‍♂️
        'en' : ':man_kneeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':kniender_mann_mittlere_hautfarbe:',
        'es': ':hombre_de_rodillas_tono_de_piel_medio:',
        'fr': u':homme_à_genoux_peau_légèrement_mate:',
        'pt': ':homem_ajoelhando_pele_morena:',
        'it': ':uomo_inginocchiato_carnagione_olivastra:'
    },
    u'\U0001F9CE\U0001F3FD\U0000200D\U00002642': { # 🧎🏽‍♂
        'en' : ':man_kneeling_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F3CB\U0000FE0F\U0000200D\U00002642\U0000FE0F': { # 🏋️‍♂️
        'en' : ':man_lifting_weights:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber:',
        'es': ':hombre_levantando_pesas:',
        'fr': u':homme_haltérophile:',
        'pt': ':homem_levantando_peso:',
        'it': ':uomo_che_solleva_pesi:'
    },
    u'\U0001F3CB\U0000200D\U00002642\U0000FE0F': { # 🏋‍♂️
        'en' : ':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CB\U0000FE0F\U0000200D\U00002642': { # 🏋️‍♂
        'en' : ':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4
    },
    u'\U0001F3CB\U0000200D\U00002642': { # 🏋‍♂
        'en' : ':man_lifting_weights:',
        'status' : unqualified,
        'E' : 4,
        'de': ':gewichtheber:',
        'es': ':hombre_levantando_pesas:',
        'fr': u':homme_haltérophile:',
        'pt': ':homem_levantando_peso:',
        'it': ':uomo_che_solleva_pesi:'
    },
    u'\U0001F3CB\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🏋🏿‍♂️
        'en' : ':man_lifting_weights_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber_dunkle_hautfarbe:',
        'es': ':hombre_levantando_pesas_tono_de_piel_oscuro:',
        'fr': u':homme_haltérophile_peau_foncée:',
        'pt': ':homem_levantando_peso_pele_escura:',
        'it': ':uomo_che_solleva_pesi_carnagione_scura:'
    },
    u'\U0001F3CB\U0001F3FF\U0000200D\U00002642': { # 🏋🏿‍♂
        'en' : ':man_lifting_weights_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🏋🏻‍♂️
        'en' : ':man_lifting_weights_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber_helle_hautfarbe:',
        'es': ':hombre_levantando_pesas_tono_de_piel_claro:',
        'fr': u':homme_haltérophile_peau_claire:',
        'pt': ':homem_levantando_peso_pele_clara:',
        'it': ':uomo_che_solleva_pesi_carnagione_chiara:'
    },
    u'\U0001F3CB\U0001F3FB\U0000200D\U00002642': { # 🏋🏻‍♂
        'en' : ':man_lifting_weights_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🏋🏾‍♂️
        'en' : ':man_lifting_weights_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber_mitteldunkle_hautfarbe:',
        'es': ':hombre_levantando_pesas_tono_de_piel_oscuro_medio:',
        'fr': u':homme_haltérophile_peau_mate:',
        'pt': ':homem_levantando_peso_pele_morena_escura:',
        'it': ':uomo_che_solleva_pesi_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CB\U0001F3FE\U0000200D\U00002642': { # 🏋🏾‍♂
        'en' : ':man_lifting_weights_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🏋🏼‍♂️
        'en' : ':man_lifting_weights_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber_mittelhelle_hautfarbe:',
        'es': ':hombre_levantando_pesas_tono_de_piel_claro_medio:',
        'fr': u':homme_haltérophile_peau_moyennement_claire:',
        'pt': ':homem_levantando_peso_pele_morena_clara:',
        'it': ':uomo_che_solleva_pesi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CB\U0001F3FC\U0000200D\U00002642': { # 🏋🏼‍♂
        'en' : ':man_lifting_weights_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CB\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🏋🏽‍♂️
        'en' : ':man_lifting_weights_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':gewichtheber_mittlere_hautfarbe:',
        'es': ':hombre_levantando_pesas_tono_de_piel_medio:',
        'fr': u':homme_haltérophile_peau_légèrement_mate:',
        'pt': ':homem_levantando_peso_pele_morena:',
        'it': ':uomo_che_solleva_pesi_carnagione_olivastra:'
    },
    u'\U0001F3CB\U0001F3FD\U0000200D\U00002642': { # 🏋🏽‍♂
        'en' : ':man_lifting_weights_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB': { # 👨🏻
        'en' : ':man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mann_helle_hautfarbe:',
        'es': ':hombre_tono_de_piel_claro:',
        'fr': ':homme_peau_claire:',
        'pt': ':homem_pele_clara:',
        'it': ':uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B2': { # 👨🏻‍🦲
        'en' : ':man_light_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_helle_hautfarbe,_glatze:',
        'es': ':hombre_tono_de_piel_claro_y_sin_pelo:',
        'fr': ':homme_peau_claire_et_chauve:',
        'pt': ':homem_pele_clara_e_careca:',
        'it': ':uomo_carnagione_chiara_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧔🏻‍♂️
        'en' : ':man_light_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FB\U0000200D\U00002642': { # 🧔🏻‍♂
        'en' : ':man_light_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 👱🏻‍♂️
        'en' : ':man_light_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_helle_hautfarbe,_blond:',
        'es': ':hombre_rubio_tono_de_piel_claro:',
        'fr': ':homme_blond_peau_claire:',
        'pt': ':homem_pele_clara_e_cabelo_loiro:',
        'it': ':uomo_biondo_carnagione_chiara:'
    },
    u'\U0001F471\U0001F3FB\U0000200D\U00002642': { # 👱🏻‍♂
        'en' : ':man_light_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B1': { # 👨🏻‍🦱
        'en' : ':man_light_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_helle_hautfarbe,_lockiges_haar:',
        'es': ':hombre_tono_de_piel_claro_y_pelo_rizado:',
        'fr': u':homme_peau_claire_et_cheveux_bouclés:',
        'pt': ':homem_pele_clara_e_cabelo_cacheado:',
        'it': ':uomo_carnagione_chiara_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B0': { # 👨🏻‍🦰
        'en' : ':man_light_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_helle_hautfarbe,_rotes_haar:',
        'es': ':hombre_tono_de_piel_claro_y_pelo_pelirrojo:',
        'fr': ':homme_peau_claire_et_cheveux_roux:',
        'pt': ':homem_pele_clara_e_cabelo_vermelho:',
        'it': ':uomo_carnagione_chiara_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9B3': { # 👨🏻‍🦳
        'en' : ':man_light_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_helle_hautfarbe,_weißes_haar:',
        'es': ':hombre_tono_de_piel_claro_y_pelo_blanco:',
        'fr': ':homme_peau_claire_et_cheveux_blancs:',
        'pt': ':homem_pele_clara_e_cabelo_branco:',
        'it': ':uomo_carnagione_chiara_e_capelli_bianchi:'
    },
    u'\U0001F9D9\U0000200D\U00002642\U0000FE0F': { # 🧙‍♂️
        'en' : ':man_mage:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier:',
        'es': ':mago:',
        'fr': ':mage_homme:',
        'pt': ':homem_mago:',
        'it': ':mago_uomo:'
    },
    u'\U0001F9D9\U0000200D\U00002642': { # 🧙‍♂
        'en' : ':man_mage:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':magier:',
        'es': ':mago:',
        'fr': ':mage_homme:',
        'pt': ':homem_mago:',
        'it': ':mago_uomo:'
    },
    u'\U0001F9D9\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧙🏿‍♂️
        'en' : ':man_mage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier_dunkle_hautfarbe:',
        'es': ':mago_tono_de_piel_oscuro:',
        'fr': u':mage_homme_peau_foncée:',
        'pt': ':homem_mago_pele_escura:',
        'it': ':mago_uomo_carnagione_scura:'
    },
    u'\U0001F9D9\U0001F3FF\U0000200D\U00002642': { # 🧙🏿‍♂
        'en' : ':man_mage_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧙🏻‍♂️
        'en' : ':man_mage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier_helle_hautfarbe:',
        'es': ':mago_tono_de_piel_claro:',
        'fr': ':mage_homme_peau_claire:',
        'pt': ':homem_mago_pele_clara:',
        'it': ':mago_uomo_carnagione_chiara:'
    },
    u'\U0001F9D9\U0001F3FB\U0000200D\U00002642': { # 🧙🏻‍♂
        'en' : ':man_mage_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧙🏾‍♂️
        'en' : ':man_mage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier_mitteldunkle_hautfarbe:',
        'es': ':mago_tono_de_piel_oscuro_medio:',
        'fr': ':mage_homme_peau_mate:',
        'pt': ':homem_mago_pele_morena_escura:',
        'it': ':mago_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D9\U0001F3FE\U0000200D\U00002642': { # 🧙🏾‍♂
        'en' : ':man_mage_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧙🏼‍♂️
        'en' : ':man_mage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier_mittelhelle_hautfarbe:',
        'es': ':mago_tono_de_piel_claro_medio:',
        'fr': ':mage_homme_peau_moyennement_claire:',
        'pt': ':homem_mago_pele_morena_clara:',
        'it': ':mago_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D9\U0001F3FC\U0000200D\U00002642': { # 🧙🏼‍♂
        'en' : ':man_mage_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9D9\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧙🏽‍♂️
        'en' : ':man_mage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':magier_mittlere_hautfarbe:',
        'es': ':mago_tono_de_piel_medio:',
        'fr': u':mage_homme_peau_légèrement_mate:',
        'pt': ':homem_mago_pele_morena:',
        'it': ':mago_uomo_carnagione_olivastra:'
    },
    u'\U0001F9D9\U0001F3FD\U0000200D\U00002642': { # 🧙🏽‍♂
        'en' : ':man_mage_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F468\U0000200D\U0001F527': { # 👨‍🔧
        'en' : ':man_mechanic:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker:',
        'es': u':profesional_de_la_mecánica_hombre:',
        'fr': u':mécanicien:',
        'pt': u':mecânico:',
        'it': ':meccanico_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F527': { # 👨🏿‍🔧
        'en' : ':man_mechanic_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker_dunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_oscuro:',
        'fr': u':mécanicien_peau_foncée:',
        'pt': u':mecânico_pele_escura:',
        'it': ':meccanico_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F527': { # 👨🏻‍🔧
        'en' : ':man_mechanic_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker_helle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_claro:',
        'fr': u':mécanicien_peau_claire:',
        'pt': u':mecânico_pele_clara:',
        'it': ':meccanico_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F527': { # 👨🏾‍🔧
        'en' : ':man_mechanic_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':mécanicien_peau_mate:',
        'pt': u':mecânico_pele_morena_escura:',
        'it': ':meccanico_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F527': { # 👨🏼‍🔧
        'en' : ':man_mechanic_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_claro_medio:',
        'fr': u':mécanicien_peau_moyennement_claire:',
        'pt': u':mecânico_pele_morena_clara:',
        'it': ':meccanico_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F527': { # 👨🏽‍🔧
        'en' : ':man_mechanic_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mechaniker_mittlere_hautfarbe:',
        'es': u':profesional_de_la_mecánica_hombre_tono_de_piel_medio:',
        'fr': u':mécanicien_peau_légèrement_mate:',
        'pt': u':mecânico_pele_morena:',
        'it': ':meccanico_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FE': { # 👨🏾
        'en' : ':man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_tono_de_piel_oscuro_medio:',
        'fr': ':homme_peau_mate:',
        'pt': ':homem_pele_morena_escura:',
        'it': ':uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B2': { # 👨🏾‍🦲
        'en' : ':man_medium-dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mitteldunkle_hautfarbe,_glatze:',
        'es': ':hombre_tono_de_piel_oscuro_medio_y_sin_pelo:',
        'fr': ':homme_peau_mate_et_chauve:',
        'pt': ':homem_pele_morena_escura_e_careca:',
        'it': ':uomo_carnagione_abbastanza_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧔🏾‍♂️
        'en' : ':man_medium-dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FE\U0000200D\U00002642': { # 🧔🏾‍♂
        'en' : ':man_medium-dark_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 👱🏾‍♂️
        'en' : ':man_medium-dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mitteldunkle_hautfarbe,_blond:',
        'es': ':hombre_rubio_tono_de_piel_oscuro_medio:',
        'fr': ':homme_blond_peau_mate:',
        'pt': ':homem_pele_morena_escura_e_cabelo_loiro:',
        'it': ':uomo_biondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F471\U0001F3FE\U0000200D\U00002642': { # 👱🏾‍♂
        'en' : ':man_medium-dark_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B1': { # 👨🏾‍🦱
        'en' : ':man_medium-dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mitteldunkle_hautfarbe,_lockiges_haar:',
        'es': ':hombre_tono_de_piel_oscuro_medio_y_pelo_rizado:',
        'fr': u':homme_peau_mate_et_cheveux_bouclés:',
        'pt': ':homem_pele_morena_escura_e_cabelo_cacheado:',
        'it': ':uomo_carnagione_abbastanza_scura_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B0': { # 👨🏾‍🦰
        'en' : ':man_medium-dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mitteldunkle_hautfarbe,_rotes_haar:',
        'es': ':hombre_tono_de_piel_oscuro_medio_y_pelo_pelirrojo:',
        'fr': ':homme_peau_mate_et_cheveux_roux:',
        'pt': ':homem_pele_morena_escura_e_cabelo_vermelho:',
        'it': ':uomo_carnagione_abbastanza_scura_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9B3': { # 👨🏾‍🦳
        'en' : ':man_medium-dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mitteldunkle_hautfarbe,_weißes_haar:',
        'es': ':hombre_tono_de_piel_oscuro_medio_y_pelo_blanco:',
        'fr': ':homme_peau_mate_et_cheveux_blancs:',
        'pt': ':homem_pele_morena_escura_e_cabelo_branco:',
        'it': ':uomo_carnagione_abbastanza_scura_e_capelli_bianchi:'
    },
    u'\U0001F468\U0001F3FC': { # 👨🏼
        'en' : ':man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mann_mittelhelle_hautfarbe:',
        'es': ':hombre_tono_de_piel_claro_medio:',
        'fr': ':homme_peau_moyennement_claire:',
        'pt': ':homem_pele_morena_clara:',
        'it': ':uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B2': { # 👨🏼‍🦲
        'en' : ':man_medium-light_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittelhelle_hautfarbe,_glatze:',
        'es': ':hombre_tono_de_piel_claro_medio_y_sin_pelo:',
        'fr': ':homme_peau_moyennement_claire_et_chauve:',
        'pt': ':homem_pele_morena_clara_e_careca:',
        'it': ':uomo_carnagione_abbastanza_chiara_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧔🏼‍♂️
        'en' : ':man_medium-light_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FC\U0000200D\U00002642': { # 🧔🏼‍♂
        'en' : ':man_medium-light_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 👱🏼‍♂️
        'en' : ':man_medium-light_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mittelhelle_hautfarbe,_blond:',
        'es': ':hombre_rubio_tono_de_piel_claro_medio:',
        'fr': ':homme_blond_peau_moyennement_claire:',
        'pt': ':homem_pele_morena_clara_e_cabelo_loiro:',
        'it': ':uomo_biondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F471\U0001F3FC\U0000200D\U00002642': { # 👱🏼‍♂
        'en' : ':man_medium-light_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B1': { # 👨🏼‍🦱
        'en' : ':man_medium-light_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittelhelle_hautfarbe,_lockiges_haar:',
        'es': ':hombre_tono_de_piel_claro_medio_y_pelo_rizado:',
        'fr': u':homme_peau_moyennement_claire_et_cheveux_bouclés:',
        'pt': ':homem_pele_morena_clara_e_cabelo_cacheado:',
        'it': ':uomo_carnagione_abbastanza_chiara_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B0': { # 👨🏼‍🦰
        'en' : ':man_medium-light_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittelhelle_hautfarbe,_rotes_haar:',
        'es': ':hombre_tono_de_piel_claro_medio_y_pelo_pelirrojo:',
        'fr': ':homme_peau_moyennement_claire_et_cheveux_roux:',
        'pt': ':homem_pele_morena_clara_e_cabelo_vermelho:',
        'it': ':uomo_carnagione_abbastanza_chiara_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9B3': { # 👨🏼‍🦳
        'en' : ':man_medium-light_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittelhelle_hautfarbe,_weißes_haar:',
        'es': ':hombre_tono_de_piel_claro_medio_y_pelo_blanco:',
        'fr': ':homme_peau_moyennement_claire_et_cheveux_blancs:',
        'pt': ':homem_pele_morena_clara_e_cabelo_branco:',
        'it': ':uomo_carnagione_abbastanza_chiara_e_capelli_bianchi:'
    },
    u'\U0001F468\U0001F3FD': { # 👨🏽
        'en' : ':man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mann_mittlere_hautfarbe:',
        'es': ':hombre_tono_de_piel_medio:',
        'fr': u':homme_peau_légèrement_mate:',
        'pt': ':homem_pele_morena:',
        'it': ':uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B2': { # 👨🏽‍🦲
        'en' : ':man_medium_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittlere_hautfarbe,_glatze:',
        'es': ':hombre_tono_de_piel_medio_y_sin_pelo:',
        'fr': u':homme_peau_légèrement_mate_et_chauve:',
        'pt': ':homem_pele_morena_e_careca:',
        'it': ':uomo_carnagione_olivastra_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧔🏽‍♂️
        'en' : ':man_medium_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U0001F9D4\U0001F3FD\U0000200D\U00002642': { # 🧔🏽‍♂
        'en' : ':man_medium_skin_tone_beard:',
        'status' : minimally_qualified,
        'E' : 13.1
    },
    u'\U0001F471\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 👱🏽‍♂️
        'en' : ':man_medium_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mittlere_hautfarbe,_blond:',
        'es': ':hombre_rubio_tono_de_piel_medio:',
        'fr': u':homme_blond_peau_légèrement_mate:',
        'pt': ':homem_pele_morena_e_cabelo_loiro:',
        'it': ':uomo_biondo_carnagione_olivastra:'
    },
    u'\U0001F471\U0001F3FD\U0000200D\U00002642': { # 👱🏽‍♂
        'en' : ':man_medium_skin_tone_blond_hair:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B1': { # 👨🏽‍🦱
        'en' : ':man_medium_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittlere_hautfarbe,_lockiges_haar:',
        'es': ':hombre_tono_de_piel_medio_y_pelo_rizado:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_bouclés:',
        'pt': ':homem_pele_morena_e_cabelo_cacheado:',
        'it': ':uomo_carnagione_olivastra_e_capelli_ricci:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B0': { # 👨🏽‍🦰
        'en' : ':man_medium_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_mittlere_hautfarbe,_rotes_haar:',
        'es': ':hombre_tono_de_piel_medio_y_pelo_pelirrojo:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_roux:',
        'pt': ':homem_pele_morena_e_cabelo_vermelho:',
        'it': ':uomo_carnagione_olivastra_e_capelli_rossi:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9B3': { # 👨🏽‍🦳
        'en' : ':man_medium_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_mittlere_hautfarbe,_weißes_haar:',
        'es': ':hombre_tono_de_piel_medio_y_pelo_blanco:',
        'fr': u':homme_peau_légèrement_mate_et_cheveux_blancs:',
        'pt': ':homem_pele_morena_e_cabelo_branco:',
        'it': ':uomo_carnagione_olivastra_e_capelli_bianchi:'
    },
    u'\U0001F6B5\U0000200D\U00002642\U0000FE0F': { # 🚵‍♂️
        'en' : ':man_mountain_biking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker:',
        'es': u':hombre_en_bicicleta_de_montaña:',
        'fr': ':homme_en_vtt:',
        'pt': ':homem_fazendo_mountain_bike:',
        'it': ':ciclista_uomo_di_mountain_bike:'
    },
    u'\U0001F6B5\U0000200D\U00002642': { # 🚵‍♂
        'en' : ':man_mountain_biking:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mountainbiker:',
        'es': u':hombre_en_bicicleta_de_montaña:',
        'fr': ':homme_en_vtt:',
        'pt': ':homem_fazendo_mountain_bike:',
        'it': ':ciclista_uomo_di_mountain_bike:'
    },
    u'\U0001F6B5\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🚵🏿‍♂️
        'en' : ':man_mountain_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker_dunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_oscuro:',
        'fr': u':homme_en_vtt_peau_foncée:',
        'pt': ':homem_fazendo_mountain_bike_pele_escura:',
        'it': ':ciclista_uomo_di_mountain_bike_carnagione_scura:'
    },
    u'\U0001F6B5\U0001F3FF\U0000200D\U00002642': { # 🚵🏿‍♂
        'en' : ':man_mountain_biking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🚵🏻‍♂️
        'en' : ':man_mountain_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker_helle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_claro:',
        'fr': ':homme_en_vtt_peau_claire:',
        'pt': ':homem_fazendo_mountain_bike_pele_clara:',
        'it': ':ciclista_uomo_di_mountain_bike_carnagione_chiara:'
    },
    u'\U0001F6B5\U0001F3FB\U0000200D\U00002642': { # 🚵🏻‍♂
        'en' : ':man_mountain_biking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🚵🏾‍♂️
        'en' : ':man_mountain_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker_mitteldunkle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_oscuro_medio:',
        'fr': ':homme_en_vtt_peau_mate:',
        'pt': ':homem_fazendo_mountain_bike_pele_morena_escura:',
        'it': ':ciclista_uomo_di_mountain_bike_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B5\U0001F3FE\U0000200D\U00002642': { # 🚵🏾‍♂
        'en' : ':man_mountain_biking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🚵🏼‍♂️
        'en' : ':man_mountain_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker_mittelhelle_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_claro_medio:',
        'fr': ':homme_en_vtt_peau_moyennement_claire:',
        'pt': ':homem_fazendo_mountain_bike_pele_morena_clara:',
        'it': ':ciclista_uomo_di_mountain_bike_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B5\U0001F3FC\U0000200D\U00002642': { # 🚵🏼‍♂
        'en' : ':man_mountain_biking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B5\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🚵🏽‍♂️
        'en' : ':man_mountain_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mountainbiker_mittlere_hautfarbe:',
        'es': u':hombre_en_bicicleta_de_montaña_tono_de_piel_medio:',
        'fr': u':homme_en_vtt_peau_légèrement_mate:',
        'pt': ':homem_fazendo_mountain_bike_pele_morena:',
        'it': ':ciclista_uomo_di_mountain_bike_carnagione_olivastra:'
    },
    u'\U0001F6B5\U0001F3FD\U0000200D\U00002642': { # 🚵🏽‍♂
        'en' : ':man_mountain_biking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F4BC': { # 👨‍💼
        'en' : ':man_office_worker:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter:',
        'es': ':oficinista_hombre:',
        'fr': u':employé_de_bureau:',
        'pt': u':funcionário_de_escritório:',
        'it': ':impiegato:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F4BC': { # 👨🏿‍💼
        'en' : ':man_office_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_dunkle_hautfarbe:',
        'es': ':oficinista_hombre_tono_de_piel_oscuro:',
        'fr': u':employé_de_bureau_peau_foncée:',
        'pt': u':funcionário_de_escritório_pele_escura:',
        'it': ':impiegato_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F4BC': { # 👨🏻‍💼
        'en' : ':man_office_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_helle_hautfarbe:',
        'es': ':oficinista_hombre_tono_de_piel_claro:',
        'fr': u':employé_de_bureau_peau_claire:',
        'pt': u':funcionário_de_escritório_pele_clara:',
        'it': ':impiegato_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F4BC': { # 👨🏾‍💼
        'en' : ':man_office_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mitteldunkle_hautfarbe:',
        'es': ':oficinista_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':employé_de_bureau_peau_mate:',
        'pt': u':funcionário_de_escritório_pele_morena_escura:',
        'it': ':impiegato_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F4BC': { # 👨🏼‍💼
        'en' : ':man_office_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mittelhelle_hautfarbe:',
        'es': ':oficinista_hombre_tono_de_piel_claro_medio:',
        'fr': u':employé_de_bureau_peau_moyennement_claire:',
        'pt': u':funcionário_de_escritório_pele_morena_clara:',
        'it': ':impiegato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F4BC': { # 👨🏽‍💼
        'en' : ':man_office_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':büroangestellter_mittlere_hautfarbe:',
        'es': ':oficinista_hombre_tono_de_piel_medio:',
        'fr': u':employé_de_bureau_peau_légèrement_mate:',
        'pt': u':funcionário_de_escritório_pele_morena:',
        'it': ':impiegato_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U00002708\U0000FE0F': { # 👨‍✈️
        'en' : ':man_pilot:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot:',
        'es': ':piloto_hombre:',
        'fr': ':pilote_homme:',
        'pt': u':piloto_de_avião_homem:',
        'it': ':pilota_uomo:'
    },
    u'\U0001F468\U0000200D\U00002708': { # 👨‍✈
        'en' : ':man_pilot:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':pilot:',
        'es': ':piloto_hombre:',
        'fr': ':pilote_homme:',
        'pt': u':piloto_de_avião_homem:',
        'it': ':pilota_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002708\U0000FE0F': { # 👨🏿‍✈️
        'en' : ':man_pilot_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot_dunkle_hautfarbe:',
        'es': ':piloto_hombre_tono_de_piel_oscuro:',
        'fr': u':pilote_homme_peau_foncée:',
        'pt': u':piloto_de_avião_homem_pele_escura:',
        'it': ':pilota_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U00002708': { # 👨🏿‍✈
        'en' : ':man_pilot_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002708\U0000FE0F': { # 👨🏻‍✈️
        'en' : ':man_pilot_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot_helle_hautfarbe:',
        'es': ':piloto_hombre_tono_de_piel_claro:',
        'fr': ':pilote_homme_peau_claire:',
        'pt': u':piloto_de_avião_homem_pele_clara:',
        'it': ':pilota_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U00002708': { # 👨🏻‍✈
        'en' : ':man_pilot_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002708\U0000FE0F': { # 👨🏾‍✈️
        'en' : ':man_pilot_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot_mitteldunkle_hautfarbe:',
        'es': ':piloto_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':pilote_homme_peau_mate:',
        'pt': u':piloto_de_avião_homem_pele_morena_escura:',
        'it': ':pilota_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U00002708': { # 👨🏾‍✈
        'en' : ':man_pilot_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002708\U0000FE0F': { # 👨🏼‍✈️
        'en' : ':man_pilot_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot_mittelhelle_hautfarbe:',
        'es': ':piloto_hombre_tono_de_piel_claro_medio:',
        'fr': ':pilote_homme_peau_moyennement_claire:',
        'pt': u':piloto_de_avião_homem_pele_morena_clara:',
        'it': ':pilota_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U00002708': { # 👨🏼‍✈
        'en' : ':man_pilot_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002708\U0000FE0F': { # 👨🏽‍✈️
        'en' : ':man_pilot_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':pilot_mittlere_hautfarbe:',
        'es': ':piloto_hombre_tono_de_piel_medio:',
        'fr': u':pilote_homme_peau_légèrement_mate:',
        'pt': u':piloto_de_avião_homem_pele_morena:',
        'it': ':pilota_uomo_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U00002708': { # 👨🏽‍✈
        'en' : ':man_pilot_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0000200D\U00002642\U0000FE0F': { # 🤾‍♂️
        'en' : ':man_playing_handball:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler:',
        'es': ':hombre_jugando_al_balonmano:',
        'fr': ':handballeur:',
        'pt': ':jogador_de_handebol:',
        'it': ':pallamanista_uomo:'
    },
    u'\U0001F93E\U0000200D\U00002642': { # 🤾‍♂
        'en' : ':man_playing_handball:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':handballspieler:',
        'es': ':hombre_jugando_al_balonmano:',
        'fr': ':handballeur:',
        'pt': ':jogador_de_handebol:',
        'it': ':pallamanista_uomo:'
    },
    u'\U0001F93E\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤾🏿‍♂️
        'en' : ':man_playing_handball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler_dunkle_hautfarbe:',
        'es': ':hombre_jugando_al_balonmano_tono_de_piel_oscuro:',
        'fr': u':handballeur_peau_foncée:',
        'pt': ':jogador_de_handebol_pele_escura:',
        'it': ':pallamanista_uomo_carnagione_scura:'
    },
    u'\U0001F93E\U0001F3FF\U0000200D\U00002642': { # 🤾🏿‍♂
        'en' : ':man_playing_handball_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤾🏻‍♂️
        'en' : ':man_playing_handball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler_helle_hautfarbe:',
        'es': ':hombre_jugando_al_balonmano_tono_de_piel_claro:',
        'fr': ':handballeur_peau_claire:',
        'pt': ':jogador_de_handebol_pele_clara:',
        'it': ':pallamanista_uomo_carnagione_chiara:'
    },
    u'\U0001F93E\U0001F3FB\U0000200D\U00002642': { # 🤾🏻‍♂
        'en' : ':man_playing_handball_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤾🏾‍♂️
        'en' : ':man_playing_handball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler_mitteldunkle_hautfarbe:',
        'es': ':hombre_jugando_al_balonmano_tono_de_piel_oscuro_medio:',
        'fr': ':handballeur_peau_mate:',
        'pt': ':jogador_de_handebol_pele_morena_escura:',
        'it': ':pallamanista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F93E\U0001F3FE\U0000200D\U00002642': { # 🤾🏾‍♂
        'en' : ':man_playing_handball_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤾🏼‍♂️
        'en' : ':man_playing_handball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler_mittelhelle_hautfarbe:',
        'es': ':hombre_jugando_al_balonmano_tono_de_piel_claro_medio:',
        'fr': ':handballeur_peau_moyennement_claire:',
        'pt': ':jogador_de_handebol_pele_morena_clara:',
        'it': ':pallamanista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F93E\U0001F3FC\U0000200D\U00002642': { # 🤾🏼‍♂
        'en' : ':man_playing_handball_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93E\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤾🏽‍♂️
        'en' : ':man_playing_handball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':handballspieler_mittlere_hautfarbe:',
        'es': ':hombre_jugando_al_balonmano_tono_de_piel_medio:',
        'fr': u':handballeur_peau_légèrement_mate:',
        'pt': ':jogador_de_handebol_pele_morena:',
        'it': ':pallamanista_uomo_carnagione_olivastra:'
    },
    u'\U0001F93E\U0001F3FD\U0000200D\U00002642': { # 🤾🏽‍♂
        'en' : ':man_playing_handball_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0000200D\U00002642\U0000FE0F': { # 🤽‍♂️
        'en' : ':man_playing_water_polo:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler:',
        'es': ':hombre_jugando_al_waterpolo:',
        'fr': ':joueur_de_water-polo:',
        'pt': u':homem_jogando_polo_aquático:',
        'it': ':pallanuotista_uomo:'
    },
    u'\U0001F93D\U0000200D\U00002642': { # 🤽‍♂
        'en' : ':man_playing_water_polo:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':wasserballspieler:',
        'es': ':hombre_jugando_al_waterpolo:',
        'fr': ':joueur_de_water-polo:',
        'pt': u':homem_jogando_polo_aquático:',
        'it': ':pallanuotista_uomo:'
    },
    u'\U0001F93D\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤽🏿‍♂️
        'en' : ':man_playing_water_polo_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler_dunkle_hautfarbe:',
        'es': ':hombre_jugando_al_waterpolo_tono_de_piel_oscuro:',
        'fr': u':joueur_de_water-polo_peau_foncée:',
        'pt': u':homem_jogando_polo_aquático_pele_escura:',
        'it': ':pallanuotista_uomo_carnagione_scura:'
    },
    u'\U0001F93D\U0001F3FF\U0000200D\U00002642': { # 🤽🏿‍♂
        'en' : ':man_playing_water_polo_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤽🏻‍♂️
        'en' : ':man_playing_water_polo_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler_helle_hautfarbe:',
        'es': ':hombre_jugando_al_waterpolo_tono_de_piel_claro:',
        'fr': ':joueur_de_water-polo_peau_claire:',
        'pt': u':homem_jogando_polo_aquático_pele_clara:',
        'it': ':pallanuotista_uomo_carnagione_chiara:'
    },
    u'\U0001F93D\U0001F3FB\U0000200D\U00002642': { # 🤽🏻‍♂
        'en' : ':man_playing_water_polo_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤽🏾‍♂️
        'en' : ':man_playing_water_polo_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler_mitteldunkle_hautfarbe:',
        'es': ':hombre_jugando_al_waterpolo_tono_de_piel_oscuro_medio:',
        'fr': ':joueur_de_water-polo_peau_mate:',
        'pt': u':homem_jogando_polo_aquático_pele_morena_escura:',
        'it': ':pallanuotista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F93D\U0001F3FE\U0000200D\U00002642': { # 🤽🏾‍♂
        'en' : ':man_playing_water_polo_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤽🏼‍♂️
        'en' : ':man_playing_water_polo_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler_mittelhelle_hautfarbe:',
        'es': ':hombre_jugando_al_waterpolo_tono_de_piel_claro_medio:',
        'fr': ':joueur_de_water-polo_peau_moyennement_claire:',
        'pt': u':homem_jogando_polo_aquático_pele_morena_clara:',
        'it': ':pallanuotista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F93D\U0001F3FC\U0000200D\U00002642': { # 🤽🏼‍♂
        'en' : ':man_playing_water_polo_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F93D\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤽🏽‍♂️
        'en' : ':man_playing_water_polo_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wasserballspieler_mittlere_hautfarbe:',
        'es': ':hombre_jugando_al_waterpolo_tono_de_piel_medio:',
        'fr': u':joueur_de_water-polo_peau_légèrement_mate:',
        'pt': u':homem_jogando_polo_aquático_pele_morena:',
        'it': ':pallanuotista_uomo_carnagione_olivastra:'
    },
    u'\U0001F93D\U0001F3FD\U0000200D\U00002642': { # 🤽🏽‍♂
        'en' : ':man_playing_water_polo_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0000200D\U00002642\U0000FE0F': { # 👮‍♂️
        'en' : ':man_police_officer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist:',
        'es': u':agente_de_policía_hombre:',
        'fr': ':policier:',
        'pt': ':policial_homem:',
        'it': ':poliziotto_uomo:'
    },
    u'\U0001F46E\U0000200D\U00002642': { # 👮‍♂
        'en' : ':man_police_officer:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':polizist:',
        'es': u':agente_de_policía_hombre:',
        'fr': ':policier:',
        'pt': ':policial_homem:',
        'it': ':poliziotto_uomo:'
    },
    u'\U0001F46E\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 👮🏿‍♂️
        'en' : ':man_police_officer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist_dunkle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_oscuro:',
        'fr': u':policier_peau_foncée:',
        'pt': ':policial_homem_pele_escura:',
        'it': ':poliziotto_uomo_carnagione_scura:'
    },
    u'\U0001F46E\U0001F3FF\U0000200D\U00002642': { # 👮🏿‍♂
        'en' : ':man_police_officer_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 👮🏻‍♂️
        'en' : ':man_police_officer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist_helle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_claro:',
        'fr': ':policier_peau_claire:',
        'pt': ':policial_homem_pele_clara:',
        'it': ':poliziotto_uomo_carnagione_chiara:'
    },
    u'\U0001F46E\U0001F3FB\U0000200D\U00002642': { # 👮🏻‍♂
        'en' : ':man_police_officer_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 👮🏾‍♂️
        'en' : ':man_police_officer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist_mitteldunkle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':policier_peau_mate:',
        'pt': ':policial_homem_pele_morena_escura:',
        'it': ':poliziotto_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F46E\U0001F3FE\U0000200D\U00002642': { # 👮🏾‍♂
        'en' : ':man_police_officer_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 👮🏼‍♂️
        'en' : ':man_police_officer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist_mittelhelle_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_claro_medio:',
        'fr': ':policier_peau_moyennement_claire:',
        'pt': ':policial_homem_pele_morena_clara:',
        'it': ':poliziotto_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F46E\U0001F3FC\U0000200D\U00002642': { # 👮🏼‍♂
        'en' : ':man_police_officer_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F46E\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 👮🏽‍♂️
        'en' : ':man_police_officer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':polizist_mittlere_hautfarbe:',
        'es': u':agente_de_policía_hombre_tono_de_piel_medio:',
        'fr': u':policier_peau_légèrement_mate:',
        'pt': ':policial_homem_pele_morena:',
        'it': ':poliziotto_uomo_carnagione_olivastra:'
    },
    u'\U0001F46E\U0001F3FD\U0000200D\U00002642': { # 👮🏽‍♂
        'en' : ':man_police_officer_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0000200D\U00002642\U0000FE0F': { # 🙎‍♂️
        'en' : ':man_pouting:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann:',
        'es': ':hombre_haciendo_pucheros:',
        'fr': ':homme_qui_boude:',
        'pt': ':homem_fazendo_bico:',
        'it': ':uomo_imbronciato:'
    },
    u'\U0001F64E\U0000200D\U00002642': { # 🙎‍♂
        'en' : ':man_pouting:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':schmollender_mann:',
        'es': ':hombre_haciendo_pucheros:',
        'fr': ':homme_qui_boude:',
        'pt': ':homem_fazendo_bico:',
        'it': ':uomo_imbronciato:'
    },
    u'\U0001F64E\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙎🏿‍♂️
        'en' : ':man_pouting_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_pucheros_tono_de_piel_oscuro:',
        'fr': u':homme_qui_boude_peau_foncée:',
        'pt': ':homem_fazendo_bico_pele_escura:',
        'it': ':uomo_imbronciato_carnagione_scura:'
    },
    u'\U0001F64E\U0001F3FF\U0000200D\U00002642': { # 🙎🏿‍♂
        'en' : ':man_pouting_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙎🏻‍♂️
        'en' : ':man_pouting_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann_helle_hautfarbe:',
        'es': ':hombre_haciendo_pucheros_tono_de_piel_claro:',
        'fr': ':homme_qui_boude_peau_claire:',
        'pt': ':homem_fazendo_bico_pele_clara:',
        'it': ':uomo_imbronciato_carnagione_chiara:'
    },
    u'\U0001F64E\U0001F3FB\U0000200D\U00002642': { # 🙎🏻‍♂
        'en' : ':man_pouting_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙎🏾‍♂️
        'en' : ':man_pouting_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_pucheros_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_boude_peau_mate:',
        'pt': ':homem_fazendo_bico_pele_morena_escura:',
        'it': ':uomo_imbronciato_carnagione_abbastanza_scura:'
    },
    u'\U0001F64E\U0001F3FE\U0000200D\U00002642': { # 🙎🏾‍♂
        'en' : ':man_pouting_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙎🏼‍♂️
        'en' : ':man_pouting_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_pucheros_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_boude_peau_moyennement_claire:',
        'pt': ':homem_fazendo_bico_pele_morena_clara:',
        'it': ':uomo_imbronciato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64E\U0001F3FC\U0000200D\U00002642': { # 🙎🏼‍♂
        'en' : ':man_pouting_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64E\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙎🏽‍♂️
        'en' : ':man_pouting_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schmollender_mann_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_pucheros_tono_de_piel_medio:',
        'fr': u':homme_qui_boude_peau_légèrement_mate:',
        'pt': ':homem_fazendo_bico_pele_morena:',
        'it': ':uomo_imbronciato_carnagione_olivastra:'
    },
    u'\U0001F64E\U0001F3FD\U0000200D\U00002642': { # 🙎🏽‍♂
        'en' : ':man_pouting_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0000200D\U00002642\U0000FE0F': { # 🙋‍♂️
        'en' : ':man_raising_hand:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm:',
        'es': ':hombre_con_la_mano_levantada:',
        'fr': u':homme_qui_lève_la_main:',
        'pt': u':homem_levantando_a_mão:',
        'it': ':uomo_con_mano_alzata:'
    },
    u'\U0001F64B\U0000200D\U00002642': { # 🙋‍♂
        'en' : ':man_raising_hand:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm:',
        'es': ':hombre_con_la_mano_levantada:',
        'fr': u':homme_qui_lève_la_main:',
        'pt': u':homem_levantando_a_mão:',
        'it': ':uomo_con_mano_alzata:'
    },
    u'\U0001F64B\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🙋🏿‍♂️
        'en' : ':man_raising_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm_dunkle_hautfarbe:',
        'es': ':hombre_con_la_mano_levantada_tono_de_piel_oscuro:',
        'fr': u':homme_qui_lève_la_main_peau_foncée:',
        'pt': u':homem_levantando_a_mão_pele_escura:',
        'it': ':uomo_con_mano_alzata_carnagione_scura:'
    },
    u'\U0001F64B\U0001F3FF\U0000200D\U00002642': { # 🙋🏿‍♂
        'en' : ':man_raising_hand_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🙋🏻‍♂️
        'en' : ':man_raising_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm_helle_hautfarbe:',
        'es': ':hombre_con_la_mano_levantada_tono_de_piel_claro:',
        'fr': u':homme_qui_lève_la_main_peau_claire:',
        'pt': u':homem_levantando_a_mão_pele_clara:',
        'it': ':uomo_con_mano_alzata_carnagione_chiara:'
    },
    u'\U0001F64B\U0001F3FB\U0000200D\U00002642': { # 🙋🏻‍♂
        'en' : ':man_raising_hand_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🙋🏾‍♂️
        'en' : ':man_raising_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm_mitteldunkle_hautfarbe:',
        'es': ':hombre_con_la_mano_levantada_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_lève_la_main_peau_mate:',
        'pt': u':homem_levantando_a_mão_pele_morena_escura:',
        'it': ':uomo_con_mano_alzata_carnagione_abbastanza_scura:'
    },
    u'\U0001F64B\U0001F3FE\U0000200D\U00002642': { # 🙋🏾‍♂
        'en' : ':man_raising_hand_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🙋🏼‍♂️
        'en' : ':man_raising_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm_mittelhelle_hautfarbe:',
        'es': ':hombre_con_la_mano_levantada_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_lève_la_main_peau_moyennement_claire:',
        'pt': u':homem_levantando_a_mão_pele_morena_clara:',
        'it': ':uomo_con_mano_alzata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64B\U0001F3FC\U0000200D\U00002642': { # 🙋🏼‍♂
        'en' : ':man_raising_hand_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F64B\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🙋🏽‍♂️
        'en' : ':man_raising_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_erhobenem_arm_mittlere_hautfarbe:',
        'es': ':hombre_con_la_mano_levantada_tono_de_piel_medio:',
        'fr': u':homme_qui_lève_la_main_peau_légèrement_mate:',
        'pt': u':homem_levantando_a_mão_pele_morena:',
        'it': ':uomo_con_mano_alzata_carnagione_olivastra:'
    },
    u'\U0001F64B\U0001F3FD\U0000200D\U00002642': { # 🙋🏽‍♂
        'en' : ':man_raising_hand_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F9B0': { # 👨‍🦰
        'en' : ':man_red_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mann_rotes_haar:',
        'es': ':hombre_pelo_pelirrojo:',
        'fr': ':homme_cheveux_roux:',
        'pt': ':homem_cabelo_vermelho:',
        'it': ':uomo_capelli_rossi:'
    },
    u'\U0001F6A3\U0000200D\U00002642\U0000FE0F': { # 🚣‍♂️
        'en' : ':man_rowing_boat:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot:',
        'es': ':hombre_remando_en_un_bote:',
        'fr': ':rameur_dans_une_barque:',
        'pt': ':homem_remando:',
        'it': ':uomo_in_barca_a_remi:'
    },
    u'\U0001F6A3\U0000200D\U00002642': { # 🚣‍♂
        'en' : ':man_rowing_boat:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot:',
        'es': ':hombre_remando_en_un_bote:',
        'fr': ':rameur_dans_une_barque:',
        'pt': ':homem_remando:',
        'it': ':uomo_in_barca_a_remi:'
    },
    u'\U0001F6A3\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🚣🏿‍♂️
        'en' : ':man_rowing_boat_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot_dunkle_hautfarbe:',
        'es': ':hombre_remando_en_un_bote_tono_de_piel_oscuro:',
        'fr': u':rameur_dans_une_barque_peau_foncée:',
        'pt': ':homem_remando_pele_escura:',
        'it': ':uomo_in_barca_a_remi_carnagione_scura:'
    },
    u'\U0001F6A3\U0001F3FF\U0000200D\U00002642': { # 🚣🏿‍♂
        'en' : ':man_rowing_boat_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🚣🏻‍♂️
        'en' : ':man_rowing_boat_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot_helle_hautfarbe:',
        'es': ':hombre_remando_en_un_bote_tono_de_piel_claro:',
        'fr': ':rameur_dans_une_barque_peau_claire:',
        'pt': ':homem_remando_pele_clara:',
        'it': ':uomo_in_barca_a_remi_carnagione_chiara:'
    },
    u'\U0001F6A3\U0001F3FB\U0000200D\U00002642': { # 🚣🏻‍♂
        'en' : ':man_rowing_boat_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🚣🏾‍♂️
        'en' : ':man_rowing_boat_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot_mitteldunkle_hautfarbe:',
        'es': ':hombre_remando_en_un_bote_tono_de_piel_oscuro_medio:',
        'fr': ':rameur_dans_une_barque_peau_mate:',
        'pt': ':homem_remando_pele_morena_escura:',
        'it': ':uomo_in_barca_a_remi_carnagione_abbastanza_scura:'
    },
    u'\U0001F6A3\U0001F3FE\U0000200D\U00002642': { # 🚣🏾‍♂
        'en' : ':man_rowing_boat_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🚣🏼‍♂️
        'en' : ':man_rowing_boat_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot_mittelhelle_hautfarbe:',
        'es': ':hombre_remando_en_un_bote_tono_de_piel_claro_medio:',
        'fr': ':rameur_dans_une_barque_peau_moyennement_claire:',
        'pt': ':homem_remando_pele_morena_clara:',
        'it': ':uomo_in_barca_a_remi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6A3\U0001F3FC\U0000200D\U00002642': { # 🚣🏼‍♂
        'en' : ':man_rowing_boat_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6A3\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🚣🏽‍♂️
        'en' : ':man_rowing_boat_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_im_ruderboot_mittlere_hautfarbe:',
        'es': ':hombre_remando_en_un_bote_tono_de_piel_medio:',
        'fr': u':rameur_dans_une_barque_peau_légèrement_mate:',
        'pt': ':homem_remando_pele_morena:',
        'it': ':uomo_in_barca_a_remi_carnagione_olivastra:'
    },
    u'\U0001F6A3\U0001F3FD\U0000200D\U00002642': { # 🚣🏽‍♂
        'en' : ':man_rowing_boat_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0000200D\U00002642\U0000FE0F': { # 🏃‍♂️
        'en' : ':man_running:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann:',
        'es': ':hombre_corriendo:',
        'fr': ':homme_qui_court:',
        'pt': ':homem_correndo:',
        'it': ':uomo_che_corre:'
    },
    u'\U0001F3C3\U0000200D\U00002642': { # 🏃‍♂
        'en' : ':man_running:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':laufender_mann:',
        'es': ':hombre_corriendo:',
        'fr': ':homme_qui_court:',
        'pt': ':homem_correndo:',
        'it': ':uomo_che_corre:'
    },
    u'\U0001F3C3\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🏃🏿‍♂️
        'en' : ':man_running_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann_dunkle_hautfarbe:',
        'es': ':hombre_corriendo_tono_de_piel_oscuro:',
        'fr': u':homme_qui_court_peau_foncée:',
        'pt': ':homem_correndo_pele_escura:',
        'it': ':uomo_che_corre_carnagione_scura:'
    },
    u'\U0001F3C3\U0001F3FF\U0000200D\U00002642': { # 🏃🏿‍♂
        'en' : ':man_running_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🏃🏻‍♂️
        'en' : ':man_running_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann_helle_hautfarbe:',
        'es': ':hombre_corriendo_tono_de_piel_claro:',
        'fr': ':homme_qui_court_peau_claire:',
        'pt': ':homem_correndo_pele_clara:',
        'it': ':uomo_che_corre_carnagione_chiara:'
    },
    u'\U0001F3C3\U0001F3FB\U0000200D\U00002642': { # 🏃🏻‍♂
        'en' : ':man_running_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🏃🏾‍♂️
        'en' : ':man_running_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_corriendo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_court_peau_mate:',
        'pt': ':homem_correndo_pele_morena_escura:',
        'it': ':uomo_che_corre_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C3\U0001F3FE\U0000200D\U00002642': { # 🏃🏾‍♂
        'en' : ':man_running_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🏃🏼‍♂️
        'en' : ':man_running_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_corriendo_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_court_peau_moyennement_claire:',
        'pt': ':homem_correndo_pele_morena_clara:',
        'it': ':uomo_che_corre_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C3\U0001F3FC\U0000200D\U00002642': { # 🏃🏼‍♂
        'en' : ':man_running_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C3\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🏃🏽‍♂️
        'en' : ':man_running_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':laufender_mann_mittlere_hautfarbe:',
        'es': ':hombre_corriendo_tono_de_piel_medio:',
        'fr': u':homme_qui_court_peau_légèrement_mate:',
        'pt': ':homem_correndo_pele_morena:',
        'it': ':uomo_che_corre_carnagione_olivastra:'
    },
    u'\U0001F3C3\U0001F3FD\U0000200D\U00002642': { # 🏃🏽‍♂
        'en' : ':man_running_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F52C': { # 👨‍🔬
        'en' : ':man_scientist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler:',
        'es': ':profesional_de_la_ciencia_hombre:',
        'fr': ':scientifique_homme:',
        'pt': ':cientista_homem:',
        'it': ':scienziato:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F52C': { # 👨🏿‍🔬
        'en' : ':man_scientist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler_dunkle_hautfarbe:',
        'es': ':profesional_de_la_ciencia_hombre_tono_de_piel_oscuro:',
        'fr': u':scientifique_homme_peau_foncée:',
        'pt': ':cientista_homem_pele_escura:',
        'it': ':scienziato_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F52C': { # 👨🏻‍🔬
        'en' : ':man_scientist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler_helle_hautfarbe:',
        'es': ':profesional_de_la_ciencia_hombre_tono_de_piel_claro:',
        'fr': ':scientifique_homme_peau_claire:',
        'pt': ':cientista_homem_pele_clara:',
        'it': ':scienziato_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F52C': { # 👨🏾‍🔬
        'en' : ':man_scientist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler_mitteldunkle_hautfarbe:',
        'es': ':profesional_de_la_ciencia_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':scientifique_homme_peau_mate:',
        'pt': ':cientista_homem_pele_morena_escura:',
        'it': ':scienziato_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F52C': { # 👨🏼‍🔬
        'en' : ':man_scientist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler_mittelhelle_hautfarbe:',
        'es': ':profesional_de_la_ciencia_hombre_tono_de_piel_claro_medio:',
        'fr': ':scientifique_homme_peau_moyennement_claire:',
        'pt': ':cientista_homem_pele_morena_clara:',
        'it': ':scienziato_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F52C': { # 👨🏽‍🔬
        'en' : ':man_scientist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':wissenschaftler_mittlere_hautfarbe:',
        'es': ':profesional_de_la_ciencia_hombre_tono_de_piel_medio:',
        'fr': u':scientifique_homme_peau_légèrement_mate:',
        'pt': ':cientista_homem_pele_morena:',
        'it': ':scienziato_carnagione_olivastra:'
    },
    u'\U0001F937\U0000200D\U00002642\U0000FE0F': { # 🤷‍♂️
        'en' : ':man_shrugging:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann:',
        'es': ':hombre_encogido_de_hombros:',
        'fr': u':homme_qui_hausse_les_épaules:',
        'pt': ':homem_dando_de_ombros:',
        'it': ':uomo_che_scrolla_le_spalle:'
    },
    u'\U0001F937\U0000200D\U00002642': { # 🤷‍♂
        'en' : ':man_shrugging:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann:',
        'es': ':hombre_encogido_de_hombros:',
        'fr': u':homme_qui_hausse_les_épaules:',
        'pt': ':homem_dando_de_ombros:',
        'it': ':uomo_che_scrolla_le_spalle:'
    },
    u'\U0001F937\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🤷🏿‍♂️
        'en' : ':man_shrugging_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann_dunkle_hautfarbe:',
        'es': ':hombre_encogido_de_hombros_tono_de_piel_oscuro:',
        'fr': u':homme_qui_hausse_les_épaules_peau_foncée:',
        'pt': ':homem_dando_de_ombros_pele_escura:',
        'it': ':uomo_che_scrolla_le_spalle_carnagione_scura:'
    },
    u'\U0001F937\U0001F3FF\U0000200D\U00002642': { # 🤷🏿‍♂
        'en' : ':man_shrugging_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🤷🏻‍♂️
        'en' : ':man_shrugging_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann_helle_hautfarbe:',
        'es': ':hombre_encogido_de_hombros_tono_de_piel_claro:',
        'fr': u':homme_qui_hausse_les_épaules_peau_claire:',
        'pt': ':homem_dando_de_ombros_pele_clara:',
        'it': ':uomo_che_scrolla_le_spalle_carnagione_chiara:'
    },
    u'\U0001F937\U0001F3FB\U0000200D\U00002642': { # 🤷🏻‍♂
        'en' : ':man_shrugging_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🤷🏾‍♂️
        'en' : ':man_shrugging_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_encogido_de_hombros_tono_de_piel_oscuro_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_mate:',
        'pt': ':homem_dando_de_ombros_pele_morena_escura:',
        'it': ':uomo_che_scrolla_le_spalle_carnagione_abbastanza_scura:'
    },
    u'\U0001F937\U0001F3FE\U0000200D\U00002642': { # 🤷🏾‍♂
        'en' : ':man_shrugging_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🤷🏼‍♂️
        'en' : ':man_shrugging_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_encogido_de_hombros_tono_de_piel_claro_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_moyennement_claire:',
        'pt': ':homem_dando_de_ombros_pele_morena_clara:',
        'it': ':uomo_che_scrolla_le_spalle_carnagione_abbastanza_chiara:'
    },
    u'\U0001F937\U0001F3FC\U0000200D\U00002642': { # 🤷🏼‍♂
        'en' : ':man_shrugging_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F937\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🤷🏽‍♂️
        'en' : ':man_shrugging_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schulterzuckender_mann_mittlere_hautfarbe:',
        'es': ':hombre_encogido_de_hombros_tono_de_piel_medio:',
        'fr': u':homme_qui_hausse_les_épaules_peau_légèrement_mate:',
        'pt': ':homem_dando_de_ombros_pele_morena:',
        'it': ':uomo_che_scrolla_le_spalle_carnagione_olivastra:'
    },
    u'\U0001F937\U0001F3FD\U0000200D\U00002642': { # 🤷🏽‍♂
        'en' : ':man_shrugging_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3A4': { # 👨‍🎤
        'en' : ':man_singer:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger:',
        'es': ':cantante_hombre:',
        'fr': ':chanteur:',
        'pt': ':cantor_homem:',
        'it': ':cantante_uomo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3A4': { # 👨🏿‍🎤
        'en' : ':man_singer_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_dunkle_hautfarbe:',
        'es': ':cantante_hombre_tono_de_piel_oscuro:',
        'fr': u':chanteur_peau_foncée:',
        'pt': ':cantor_homem_pele_escura:',
        'it': ':cantante_uomo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3A4': { # 👨🏻‍🎤
        'en' : ':man_singer_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_helle_hautfarbe:',
        'es': ':cantante_hombre_tono_de_piel_claro:',
        'fr': ':chanteur_peau_claire:',
        'pt': ':cantor_homem_pele_clara:',
        'it': ':cantante_uomo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3A4': { # 👨🏾‍🎤
        'en' : ':man_singer_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mitteldunkle_hautfarbe:',
        'es': ':cantante_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':chanteur_peau_mate:',
        'pt': ':cantor_homem_pele_morena_escura:',
        'it': ':cantante_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3A4': { # 👨🏼‍🎤
        'en' : ':man_singer_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mittelhelle_hautfarbe:',
        'es': ':cantante_hombre_tono_de_piel_claro_medio:',
        'fr': ':chanteur_peau_moyennement_claire:',
        'pt': ':cantor_homem_pele_morena_clara:',
        'it': ':cantante_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3A4': { # 👨🏽‍🎤
        'en' : ':man_singer_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':sänger_mittlere_hautfarbe:',
        'es': ':cantante_hombre_tono_de_piel_medio:',
        'fr': u':chanteur_peau_légèrement_mate:',
        'pt': ':cantor_homem_pele_morena:',
        'it': ':cantante_uomo_carnagione_olivastra:'
    },
    u'\U0001F9CD\U0000200D\U00002642\U0000FE0F': { # 🧍‍♂️
        'en' : ':man_standing:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann:',
        'es': ':hombre_de_pie:',
        'fr': ':homme_debout:',
        'pt': u':homem_em_pé:',
        'it': ':uomo_in_piedi:'
    },
    u'\U0001F9CD\U0000200D\U00002642': { # 🧍‍♂
        'en' : ':man_standing:',
        'status' : minimally_qualified,
        'E' : 12,
        'de': ':stehender_mann:',
        'es': ':hombre_de_pie:',
        'fr': ':homme_debout:',
        'pt': u':homem_em_pé:',
        'it': ':uomo_in_piedi:'
    },
    u'\U0001F9CD\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧍🏿‍♂️
        'en' : ':man_standing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann_dunkle_hautfarbe:',
        'es': ':hombre_de_pie_tono_de_piel_oscuro:',
        'fr': u':homme_debout_peau_foncée:',
        'pt': u':homem_em_pé_pele_escura:',
        'it': ':uomo_in_piedi_carnagione_scura:'
    },
    u'\U0001F9CD\U0001F3FF\U0000200D\U00002642': { # 🧍🏿‍♂
        'en' : ':man_standing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧍🏻‍♂️
        'en' : ':man_standing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann_helle_hautfarbe:',
        'es': ':hombre_de_pie_tono_de_piel_claro:',
        'fr': ':homme_debout_peau_claire:',
        'pt': u':homem_em_pé_pele_clara:',
        'it': ':uomo_in_piedi_carnagione_chiara:'
    },
    u'\U0001F9CD\U0001F3FB\U0000200D\U00002642': { # 🧍🏻‍♂
        'en' : ':man_standing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧍🏾‍♂️
        'en' : ':man_standing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann_mitteldunkle_hautfarbe:',
        'es': ':hombre_de_pie_tono_de_piel_oscuro_medio:',
        'fr': ':homme_debout_peau_mate:',
        'pt': u':homem_em_pé_pele_morena_escura:',
        'it': ':uomo_in_piedi_carnagione_abbastanza_scura:'
    },
    u'\U0001F9CD\U0001F3FE\U0000200D\U00002642': { # 🧍🏾‍♂
        'en' : ':man_standing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧍🏼‍♂️
        'en' : ':man_standing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann_mittelhelle_hautfarbe:',
        'es': ':hombre_de_pie_tono_de_piel_claro_medio:',
        'fr': ':homme_debout_peau_moyennement_claire:',
        'pt': u':homem_em_pé_pele_morena_clara:',
        'it': ':uomo_in_piedi_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9CD\U0001F3FC\U0000200D\U00002642': { # 🧍🏼‍♂
        'en' : ':man_standing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F9CD\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧍🏽‍♂️
        'en' : ':man_standing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':stehender_mann_mittlere_hautfarbe:',
        'es': ':hombre_de_pie_tono_de_piel_medio:',
        'fr': u':homme_debout_peau_légèrement_mate:',
        'pt': u':homem_em_pé_pele_morena:',
        'it': ':uomo_in_piedi_carnagione_olivastra:'
    },
    u'\U0001F9CD\U0001F3FD\U0000200D\U00002642': { # 🧍🏽‍♂
        'en' : ':man_standing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 12
    },
    u'\U0001F468\U0000200D\U0001F393': { # 👨‍🎓
        'en' : ':man_student:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student:',
        'es': ':estudiante_hombre:',
        'fr': u':étudiant:',
        'pt': ':estudante:',
        'it': ':studente_maschio:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F393': { # 👨🏿‍🎓
        'en' : ':man_student_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student_dunkle_hautfarbe:',
        'es': ':estudiante_hombre_tono_de_piel_oscuro:',
        'fr': u':étudiant_peau_foncée:',
        'pt': ':estudante_pele_escura:',
        'it': ':studente_maschio_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F393': { # 👨🏻‍🎓
        'en' : ':man_student_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student_helle_hautfarbe:',
        'es': ':estudiante_hombre_tono_de_piel_claro:',
        'fr': u':étudiant_peau_claire:',
        'pt': ':estudante_pele_clara:',
        'it': ':studente_maschio_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F393': { # 👨🏾‍🎓
        'en' : ':man_student_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student_mitteldunkle_hautfarbe:',
        'es': ':estudiante_hombre_tono_de_piel_oscuro_medio:',
        'fr': u':étudiant_peau_mate:',
        'pt': ':estudante_pele_morena_escura:',
        'it': ':studente_maschio_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F393': { # 👨🏼‍🎓
        'en' : ':man_student_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student_mittelhelle_hautfarbe:',
        'es': ':estudiante_hombre_tono_de_piel_claro_medio:',
        'fr': u':étudiant_peau_moyennement_claire:',
        'pt': ':estudante_pele_morena_clara:',
        'it': ':studente_maschio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F393': { # 👨🏽‍🎓
        'en' : ':man_student_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':student_mittlere_hautfarbe:',
        'es': ':estudiante_hombre_tono_de_piel_medio:',
        'fr': u':étudiant_peau_légèrement_mate:',
        'pt': ':estudante_pele_morena:',
        'it': ':studente_maschio_carnagione_olivastra:'
    },
    u'\U0001F9B8\U0000200D\U00002642\U0000FE0F': { # 🦸‍♂️
        'en' : ':man_superhero:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld:',
        'es': u':superhéroe:',
        'fr': u':super-héros_homme:',
        'pt': u':homem_super-herói:',
        'it': ':supereroe_uomo:'
    },
    u'\U0001F9B8\U0000200D\U00002642': { # 🦸‍♂
        'en' : ':man_superhero:',
        'status' : minimally_qualified,
        'E' : 11,
        'de': ':superheld:',
        'es': u':superhéroe:',
        'fr': u':super-héros_homme:',
        'pt': u':homem_super-herói:',
        'it': ':supereroe_uomo:'
    },
    u'\U0001F9B8\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🦸🏿‍♂️
        'en' : ':man_superhero_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld_dunkle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_oscuro:',
        'fr': u':super-héros_homme_peau_foncée:',
        'pt': u':homem_super-herói_pele_escura:',
        'it': ':supereroe_uomo_carnagione_scura:'
    },
    u'\U0001F9B8\U0001F3FF\U0000200D\U00002642': { # 🦸🏿‍♂
        'en' : ':man_superhero_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🦸🏻‍♂️
        'en' : ':man_superhero_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld_helle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_claro:',
        'fr': u':super-héros_homme_peau_claire:',
        'pt': u':homem_super-herói_pele_clara:',
        'it': ':supereroe_uomo_carnagione_chiara:'
    },
    u'\U0001F9B8\U0001F3FB\U0000200D\U00002642': { # 🦸🏻‍♂
        'en' : ':man_superhero_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🦸🏾‍♂️
        'en' : ':man_superhero_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld_mitteldunkle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_oscuro_medio:',
        'fr': u':super-héros_homme_peau_mate:',
        'pt': u':homem_super-herói_pele_morena_escura:',
        'it': ':supereroe_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B8\U0001F3FE\U0000200D\U00002642': { # 🦸🏾‍♂
        'en' : ':man_superhero_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🦸🏼‍♂️
        'en' : ':man_superhero_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld_mittelhelle_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_claro_medio:',
        'fr': u':super-héros_homme_peau_moyennement_claire:',
        'pt': u':homem_super-herói_pele_morena_clara:',
        'it': ':supereroe_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B8\U0001F3FC\U0000200D\U00002642': { # 🦸🏼‍♂
        'en' : ':man_superhero_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B8\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🦸🏽‍♂️
        'en' : ':man_superhero_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':superheld_mittlere_hautfarbe:',
        'es': u':superhéroe_tono_de_piel_medio:',
        'fr': u':super-héros_homme_peau_légèrement_mate:',
        'pt': u':homem_super-herói_pele_morena:',
        'it': ':supereroe_uomo_carnagione_olivastra:'
    },
    u'\U0001F9B8\U0001F3FD\U0000200D\U00002642': { # 🦸🏽‍♂
        'en' : ':man_superhero_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0000200D\U00002642\U0000FE0F': { # 🦹‍♂️
        'en' : ':man_supervillain:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht:',
        'es': ':supervillano:',
        'fr': ':super-vilain_homme:',
        'pt': u':homem_supervilão:',
        'it': ':supercattivo_uomo:'
    },
    u'\U0001F9B9\U0000200D\U00002642': { # 🦹‍♂
        'en' : ':man_supervillain:',
        'status' : minimally_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht:',
        'es': ':supervillano:',
        'fr': ':super-vilain_homme:',
        'pt': u':homem_supervilão:',
        'it': ':supercattivo_uomo:'
    },
    u'\U0001F9B9\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🦹🏿‍♂️
        'en' : ':man_supervillain_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_dunkle_hautfarbe:',
        'es': ':supervillano_tono_de_piel_oscuro:',
        'fr': u':super-vilain_homme_peau_foncée:',
        'pt': u':homem_supervilão_pele_escura:',
        'it': ':supercattivo_uomo_carnagione_scura:'
    },
    u'\U0001F9B9\U0001F3FF\U0000200D\U00002642': { # 🦹🏿‍♂
        'en' : ':man_supervillain_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🦹🏻‍♂️
        'en' : ':man_supervillain_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_helle_hautfarbe:',
        'es': ':supervillano_tono_de_piel_claro:',
        'fr': ':super-vilain_homme_peau_claire:',
        'pt': u':homem_supervilão_pele_clara:',
        'it': ':supercattivo_uomo_carnagione_chiara:'
    },
    u'\U0001F9B9\U0001F3FB\U0000200D\U00002642': { # 🦹🏻‍♂
        'en' : ':man_supervillain_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🦹🏾‍♂️
        'en' : ':man_supervillain_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mitteldunkle_hautfarbe:',
        'es': ':supervillano_tono_de_piel_oscuro_medio:',
        'fr': ':super-vilain_homme_peau_mate:',
        'pt': u':homem_supervilão_pele_morena_escura:',
        'it': ':supercattivo_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9B9\U0001F3FE\U0000200D\U00002642': { # 🦹🏾‍♂
        'en' : ':man_supervillain_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🦹🏼‍♂️
        'en' : ':man_supervillain_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mittelhelle_hautfarbe:',
        'es': ':supervillano_tono_de_piel_claro_medio:',
        'fr': ':super-vilain_homme_peau_moyennement_claire:',
        'pt': u':homem_supervilão_pele_morena_clara:',
        'it': ':supercattivo_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9B9\U0001F3FC\U0000200D\U00002642': { # 🦹🏼‍♂
        'en' : ':man_supervillain_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F9B9\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🦹🏽‍♂️
        'en' : ':man_supervillain_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':männlicher_bösewicht_mittlere_hautfarbe:',
        'es': ':supervillano_tono_de_piel_medio:',
        'fr': u':super-vilain_homme_peau_légèrement_mate:',
        'pt': u':homem_supervilão_pele_morena:',
        'it': ':supercattivo_uomo_carnagione_olivastra:'
    },
    u'\U0001F9B9\U0001F3FD\U0000200D\U00002642': { # 🦹🏽‍♂
        'en' : ':man_supervillain_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 11
    },
    u'\U0001F3C4\U0000200D\U00002642\U0000FE0F': { # 🏄‍♂️
        'en' : ':man_surfing:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer:',
        'es': ':hombre_haciendo_surf:',
        'fr': ':surfeur:',
        'pt': ':homem_surfista:',
        'it': ':surfista_uomo:'
    },
    u'\U0001F3C4\U0000200D\U00002642': { # 🏄‍♂
        'en' : ':man_surfing:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':surfer:',
        'es': ':hombre_haciendo_surf:',
        'fr': ':surfeur:',
        'pt': ':homem_surfista:',
        'it': ':surfista_uomo:'
    },
    u'\U0001F3C4\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🏄🏿‍♂️
        'en' : ':man_surfing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer_dunkle_hautfarbe:',
        'es': ':hombre_haciendo_surf_tono_de_piel_oscuro:',
        'fr': u':surfeur_peau_foncée:',
        'pt': ':homem_surfista_pele_escura:',
        'it': ':surfista_uomo_carnagione_scura:'
    },
    u'\U0001F3C4\U0001F3FF\U0000200D\U00002642': { # 🏄🏿‍♂
        'en' : ':man_surfing_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🏄🏻‍♂️
        'en' : ':man_surfing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer_helle_hautfarbe:',
        'es': ':hombre_haciendo_surf_tono_de_piel_claro:',
        'fr': ':surfeur_peau_claire:',
        'pt': ':homem_surfista_pele_clara:',
        'it': ':surfista_uomo_carnagione_chiara:'
    },
    u'\U0001F3C4\U0001F3FB\U0000200D\U00002642': { # 🏄🏻‍♂
        'en' : ':man_surfing_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🏄🏾‍♂️
        'en' : ':man_surfing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer_mitteldunkle_hautfarbe:',
        'es': ':hombre_haciendo_surf_tono_de_piel_oscuro_medio:',
        'fr': ':surfeur_peau_mate:',
        'pt': ':homem_surfista_pele_morena_escura:',
        'it': ':surfista_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F3C4\U0001F3FE\U0000200D\U00002642': { # 🏄🏾‍♂
        'en' : ':man_surfing_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🏄🏼‍♂️
        'en' : ':man_surfing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer_mittelhelle_hautfarbe:',
        'es': ':hombre_haciendo_surf_tono_de_piel_claro_medio:',
        'fr': ':surfeur_peau_moyennement_claire:',
        'pt': ':homem_surfista_pele_morena_clara:',
        'it': ':surfista_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3C4\U0001F3FC\U0000200D\U00002642': { # 🏄🏼‍♂
        'en' : ':man_surfing_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3C4\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🏄🏽‍♂️
        'en' : ':man_surfing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':surfer_mittlere_hautfarbe:',
        'es': ':hombre_haciendo_surf_tono_de_piel_medio:',
        'fr': u':surfeur_peau_légèrement_mate:',
        'pt': ':homem_surfista_pele_morena:',
        'it': ':surfista_uomo_carnagione_olivastra:'
    },
    u'\U0001F3C4\U0001F3FD\U0000200D\U00002642': { # 🏄🏽‍♂
        'en' : ':man_surfing_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0000200D\U00002642\U0000FE0F': { # 🏊‍♂️
        'en' : ':man_swimming:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer:',
        'es': ':hombre_nadando:',
        'fr': ':nageur:',
        'pt': ':homem_nadando:',
        'it': ':nuotatore:'
    },
    u'\U0001F3CA\U0000200D\U00002642': { # 🏊‍♂
        'en' : ':man_swimming:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':schwimmer:',
        'es': ':hombre_nadando:',
        'fr': ':nageur:',
        'pt': ':homem_nadando:',
        'it': ':nuotatore:'
    },
    u'\U0001F3CA\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🏊🏿‍♂️
        'en' : ':man_swimming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer_dunkle_hautfarbe:',
        'es': ':hombre_nadando_tono_de_piel_oscuro:',
        'fr': u':nageur_peau_foncée:',
        'pt': ':homem_nadando_pele_escura:',
        'it': ':nuotatore_carnagione_scura:'
    },
    u'\U0001F3CA\U0001F3FF\U0000200D\U00002642': { # 🏊🏿‍♂
        'en' : ':man_swimming_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🏊🏻‍♂️
        'en' : ':man_swimming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer_helle_hautfarbe:',
        'es': ':hombre_nadando_tono_de_piel_claro:',
        'fr': ':nageur_peau_claire:',
        'pt': ':homem_nadando_pele_clara:',
        'it': ':nuotatore_carnagione_chiara:'
    },
    u'\U0001F3CA\U0001F3FB\U0000200D\U00002642': { # 🏊🏻‍♂
        'en' : ':man_swimming_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🏊🏾‍♂️
        'en' : ':man_swimming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer_mitteldunkle_hautfarbe:',
        'es': ':hombre_nadando_tono_de_piel_oscuro_medio:',
        'fr': ':nageur_peau_mate:',
        'pt': ':homem_nadando_pele_morena_escura:',
        'it': ':nuotatore_carnagione_abbastanza_scura:'
    },
    u'\U0001F3CA\U0001F3FE\U0000200D\U00002642': { # 🏊🏾‍♂
        'en' : ':man_swimming_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🏊🏼‍♂️
        'en' : ':man_swimming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer_mittelhelle_hautfarbe:',
        'es': ':hombre_nadando_tono_de_piel_claro_medio:',
        'fr': ':nageur_peau_moyennement_claire:',
        'pt': ':homem_nadando_pele_morena_clara:',
        'it': ':nuotatore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F3CA\U0001F3FC\U0000200D\U00002642': { # 🏊🏼‍♂
        'en' : ':man_swimming_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F3CA\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🏊🏽‍♂️
        'en' : ':man_swimming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':schwimmer_mittlere_hautfarbe:',
        'es': ':hombre_nadando_tono_de_piel_medio:',
        'fr': u':nageur_peau_légèrement_mate:',
        'pt': ':homem_nadando_pele_morena:',
        'it': ':nuotatore_carnagione_olivastra:'
    },
    u'\U0001F3CA\U0001F3FD\U0000200D\U00002642': { # 🏊🏽‍♂
        'en' : ':man_swimming_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F3EB': { # 👨‍🏫
        'en' : ':man_teacher:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer:',
        'es': ':docente_hombre:',
        'fr': ':enseignant:',
        'pt': ':professor:',
        'it': ':professore:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F3EB': { # 👨🏿‍🏫
        'en' : ':man_teacher_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer_dunkle_hautfarbe:',
        'es': ':docente_hombre_tono_de_piel_oscuro:',
        'fr': u':enseignant_peau_foncée:',
        'pt': ':professor_pele_escura:',
        'it': ':professore_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F3EB': { # 👨🏻‍🏫
        'en' : ':man_teacher_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer_helle_hautfarbe:',
        'es': ':docente_hombre_tono_de_piel_claro:',
        'fr': ':enseignant_peau_claire:',
        'pt': ':professor_pele_clara:',
        'it': ':professore_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F3EB': { # 👨🏾‍🏫
        'en' : ':man_teacher_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer_mitteldunkle_hautfarbe:',
        'es': ':docente_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':enseignant_peau_mate:',
        'pt': ':professor_pele_morena_escura:',
        'it': ':professore_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F3EB': { # 👨🏼‍🏫
        'en' : ':man_teacher_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer_mittelhelle_hautfarbe:',
        'es': ':docente_hombre_tono_de_piel_claro_medio:',
        'fr': ':enseignant_peau_moyennement_claire:',
        'pt': ':professor_pele_morena_clara:',
        'it': ':professore_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F3EB': { # 👨🏽‍🏫
        'en' : ':man_teacher_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':lehrer_mittlere_hautfarbe:',
        'es': ':docente_hombre_tono_de_piel_medio:',
        'fr': u':enseignant_peau_légèrement_mate:',
        'pt': ':professor_pele_morena:',
        'it': ':professore_carnagione_olivastra:'
    },
    u'\U0001F468\U0000200D\U0001F4BB': { # 👨‍💻
        'en' : ':man_technologist:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte:',
        'es': u':profesional_de_la_tecnología_hombre:',
        'fr': ':informaticien:',
        'pt': u':tecnólogo:',
        'it': ':tecnologo:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F4BB': { # 👨🏿‍💻
        'en' : ':man_technologist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte_dunkle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_oscuro:',
        'fr': u':informaticien_peau_foncée:',
        'pt': u':tecnólogo_pele_escura:',
        'it': ':tecnologo_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F4BB': { # 👨🏻‍💻
        'en' : ':man_technologist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte_helle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_claro:',
        'fr': ':informaticien_peau_claire:',
        'pt': u':tecnólogo_pele_clara:',
        'it': ':tecnologo_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F4BB': { # 👨🏾‍💻
        'en' : ':man_technologist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':informaticien_peau_mate:',
        'pt': u':tecnólogo_pele_morena_escura:',
        'it': ':tecnologo_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F4BB': { # 👨🏼‍💻
        'en' : ':man_technologist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_claro_medio:',
        'fr': ':informaticien_peau_moyennement_claire:',
        'pt': u':tecnólogo_pele_morena_clara:',
        'it': ':tecnologo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F4BB': { # 👨🏽‍💻
        'en' : ':man_technologist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':it-experte_mittlere_hautfarbe:',
        'es': u':profesional_de_la_tecnología_hombre_tono_de_piel_medio:',
        'fr': u':informaticien_peau_légèrement_mate:',
        'pt': u':tecnólogo_pele_morena:',
        'it': ':tecnologo_carnagione_olivastra:'
    },
    u'\U0001F481\U0000200D\U00002642\U0000FE0F': { # 💁‍♂️
        'en' : ':man_tipping_hand:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter:',
        'es': u':empleado_de_mostrador_de_información:',
        'fr': ':homme_paume_vers_le_haut:',
        'pt': ':homem_com_a_palma_virada_para_cima:',
        'it': ':uomo_con_suggerimento:'
    },
    u'\U0001F481\U0000200D\U00002642': { # 💁‍♂
        'en' : ':man_tipping_hand:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter:',
        'es': u':empleado_de_mostrador_de_información:',
        'fr': ':homme_paume_vers_le_haut:',
        'pt': ':homem_com_a_palma_virada_para_cima:',
        'it': ':uomo_al_punto_informazioni:'
    },
    u'\U0001F481\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 💁🏿‍♂️
        'en' : ':man_tipping_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter_dunkle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_oscuro:',
        'fr': u':homme_paume_vers_le_haut_peau_foncée:',
        'pt': ':homem_com_a_palma_virada_para_cima_pele_escura:',
        'it': ':uomo_con_suggerimento_carnagione_scura:'
    },
    u'\U0001F481\U0001F3FF\U0000200D\U00002642': { # 💁🏿‍♂
        'en' : ':man_tipping_hand_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 💁🏻‍♂️
        'en' : ':man_tipping_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter_helle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_claro:',
        'fr': ':homme_paume_vers_le_haut_peau_claire:',
        'pt': ':homem_com_a_palma_virada_para_cima_pele_clara:',
        'it': ':uomo_con_suggerimento_carnagione_chiara:'
    },
    u'\U0001F481\U0001F3FB\U0000200D\U00002642': { # 💁🏻‍♂
        'en' : ':man_tipping_hand_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 💁🏾‍♂️
        'en' : ':man_tipping_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter_mitteldunkle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_oscuro_medio:',
        'fr': ':homme_paume_vers_le_haut_peau_mate:',
        'pt': ':homem_com_a_palma_virada_para_cima_pele_morena_escura:',
        'it': ':uomo_con_suggerimento_carnagione_abbastanza_scura:'
    },
    u'\U0001F481\U0001F3FE\U0000200D\U00002642': { # 💁🏾‍♂
        'en' : ':man_tipping_hand_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 💁🏼‍♂️
        'en' : ':man_tipping_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter_mittelhelle_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_claro_medio:',
        'fr': ':homme_paume_vers_le_haut_peau_moyennement_claire:',
        'pt': ':homem_com_a_palma_virada_para_cima_pele_morena_clara:',
        'it': ':uomo_con_suggerimento_carnagione_abbastanza_chiara:'
    },
    u'\U0001F481\U0001F3FC\U0000200D\U00002642': { # 💁🏼‍♂
        'en' : ':man_tipping_hand_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F481\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 💁🏽‍♂️
        'en' : ':man_tipping_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':infoschalter-mitarbeiter_mittlere_hautfarbe:',
        'es': u':empleado_de_mostrador_de_información_tono_de_piel_medio:',
        'fr': u':homme_paume_vers_le_haut_peau_légèrement_mate:',
        'pt': ':homem_com_a_palma_virada_para_cima_pele_morena:',
        'it': ':uomo_con_suggerimento_carnagione_olivastra:'
    },
    u'\U0001F481\U0001F3FD\U0000200D\U00002642': { # 💁🏽‍♂
        'en' : ':man_tipping_hand_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F9DB\U0000200D\U00002642\U0000FE0F': { # 🧛‍♂️
        'en' : ':man_vampire:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir:',
        'es': ':vampiro_hombre:',
        'fr': ':vampire_homme:',
        'pt': ':homem_vampiro:',
        'it': ':vampiro_uomo:'
    },
    u'\U0001F9DB\U0000200D\U00002642': { # 🧛‍♂
        'en' : ':man_vampire:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': u':männlicher_vampir:',
        'es': ':vampiro_hombre:',
        'fr': ':vampire_homme:',
        'pt': ':homem_vampiro:',
        'it': ':vampiro_uomo:'
    },
    u'\U0001F9DB\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧛🏿‍♂️
        'en' : ':man_vampire_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_dunkle_hautfarbe:',
        'es': ':vampiro_hombre_tono_de_piel_oscuro:',
        'fr': u':vampire_homme_peau_foncée:',
        'pt': ':homem_vampiro_pele_escura:',
        'it': ':vampiro_uomo_carnagione_scura:'
    },
    u'\U0001F9DB\U0001F3FF\U0000200D\U00002642': { # 🧛🏿‍♂
        'en' : ':man_vampire_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧛🏻‍♂️
        'en' : ':man_vampire_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_helle_hautfarbe:',
        'es': ':vampiro_hombre_tono_de_piel_claro:',
        'fr': ':vampire_homme_peau_claire:',
        'pt': ':homem_vampiro_pele_clara:',
        'it': ':vampiro_uomo_carnagione_chiara:'
    },
    u'\U0001F9DB\U0001F3FB\U0000200D\U00002642': { # 🧛🏻‍♂
        'en' : ':man_vampire_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧛🏾‍♂️
        'en' : ':man_vampire_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mitteldunkle_hautfarbe:',
        'es': ':vampiro_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':vampire_homme_peau_mate:',
        'pt': ':homem_vampiro_pele_morena_escura:',
        'it': ':vampiro_uomo_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DB\U0001F3FE\U0000200D\U00002642': { # 🧛🏾‍♂
        'en' : ':man_vampire_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧛🏼‍♂️
        'en' : ':man_vampire_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mittelhelle_hautfarbe:',
        'es': ':vampiro_hombre_tono_de_piel_claro_medio:',
        'fr': ':vampire_homme_peau_moyennement_claire:',
        'pt': ':homem_vampiro_pele_morena_clara:',
        'it': ':vampiro_uomo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DB\U0001F3FC\U0000200D\U00002642': { # 🧛🏼‍♂
        'en' : ':man_vampire_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DB\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧛🏽‍♂️
        'en' : ':man_vampire_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_vampir_mittlere_hautfarbe:',
        'es': ':vampiro_hombre_tono_de_piel_medio:',
        'fr': u':vampire_homme_peau_légèrement_mate:',
        'pt': ':homem_vampiro_pele_morena:',
        'it': ':vampiro_uomo_carnagione_olivastra:'
    },
    u'\U0001F9DB\U0001F3FD\U0000200D\U00002642': { # 🧛🏽‍♂
        'en' : ':man_vampire_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F6B6\U0000200D\U00002642\U0000FE0F': { # 🚶‍♂️
        'en' : ':man_walking:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger:',
        'es': ':hombre_caminando:',
        'fr': ':homme_qui_marche:',
        'pt': ':homem_andando:',
        'it': ':uomo_che_cammina:'
    },
    u'\U0001F6B6\U0000200D\U00002642': { # 🚶‍♂
        'en' : ':man_walking:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': u':fußgänger:',
        'es': ':hombre_caminando:',
        'fr': ':homme_qui_marche:',
        'pt': ':homem_andando:',
        'it': ':uomo_che_cammina:'
    },
    u'\U0001F6B6\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🚶🏿‍♂️
        'en' : ':man_walking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_dunkle_hautfarbe:',
        'es': ':hombre_caminando_tono_de_piel_oscuro:',
        'fr': u':homme_qui_marche_peau_foncée:',
        'pt': ':homem_andando_pele_escura:',
        'it': ':uomo_che_cammina_carnagione_scura:'
    },
    u'\U0001F6B6\U0001F3FF\U0000200D\U00002642': { # 🚶🏿‍♂
        'en' : ':man_walking_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🚶🏻‍♂️
        'en' : ':man_walking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_helle_hautfarbe:',
        'es': ':hombre_caminando_tono_de_piel_claro:',
        'fr': ':homme_qui_marche_peau_claire:',
        'pt': ':homem_andando_pele_clara:',
        'it': ':uomo_che_cammina_carnagione_chiara:'
    },
    u'\U0001F6B6\U0001F3FB\U0000200D\U00002642': { # 🚶🏻‍♂
        'en' : ':man_walking_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🚶🏾‍♂️
        'en' : ':man_walking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mitteldunkle_hautfarbe:',
        'es': ':hombre_caminando_tono_de_piel_oscuro_medio:',
        'fr': ':homme_qui_marche_peau_mate:',
        'pt': ':homem_andando_pele_morena_escura:',
        'it': ':uomo_che_cammina_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B6\U0001F3FE\U0000200D\U00002642': { # 🚶🏾‍♂
        'en' : ':man_walking_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🚶🏼‍♂️
        'en' : ':man_walking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mittelhelle_hautfarbe:',
        'es': ':hombre_caminando_tono_de_piel_claro_medio:',
        'fr': ':homme_qui_marche_peau_moyennement_claire:',
        'pt': ':homem_andando_pele_morena_clara:',
        'it': ':uomo_che_cammina_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B6\U0001F3FC\U0000200D\U00002642': { # 🚶🏼‍♂
        'en' : ':man_walking_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F6B6\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🚶🏽‍♂️
        'en' : ':man_walking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':fußgänger_mittlere_hautfarbe:',
        'es': ':hombre_caminando_tono_de_piel_medio:',
        'fr': u':homme_qui_marche_peau_légèrement_mate:',
        'pt': ':homem_andando_pele_morena:',
        'it': ':uomo_che_cammina_carnagione_olivastra:'
    },
    u'\U0001F6B6\U0001F3FD\U0000200D\U00002642': { # 🚶🏽‍♂
        'en' : ':man_walking_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0000200D\U00002642\U0000FE0F': { # 👳‍♂️
        'en' : ':man_wearing_turban:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban:',
        'es': ':hombre_con_turbante:',
        'fr': ':homme_en_turban:',
        'pt': ':homem_com_turbante:',
        'it': ':uomo_con_turbante:'
    },
    u'\U0001F473\U0000200D\U00002642': { # 👳‍♂
        'en' : ':man_wearing_turban:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': ':mann_mit_turban:',
        'es': ':hombre_con_turbante:',
        'fr': ':homme_en_turban:',
        'pt': ':homem_com_turbante:',
        'it': ':uomo_con_turbante:'
    },
    u'\U0001F473\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 👳🏿‍♂️
        'en' : ':man_wearing_turban_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban_dunkle_hautfarbe:',
        'es': ':hombre_con_turbante_tono_de_piel_oscuro:',
        'fr': u':homme_en_turban_peau_foncée:',
        'pt': ':homem_com_turbante_pele_escura:',
        'it': ':uomo_con_turbante_carnagione_scura:'
    },
    u'\U0001F473\U0001F3FF\U0000200D\U00002642': { # 👳🏿‍♂
        'en' : ':man_wearing_turban_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 👳🏻‍♂️
        'en' : ':man_wearing_turban_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban_helle_hautfarbe:',
        'es': ':hombre_con_turbante_tono_de_piel_claro:',
        'fr': ':homme_en_turban_peau_claire:',
        'pt': ':homem_com_turbante_pele_clara:',
        'it': ':uomo_con_turbante_carnagione_chiara:'
    },
    u'\U0001F473\U0001F3FB\U0000200D\U00002642': { # 👳🏻‍♂
        'en' : ':man_wearing_turban_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 👳🏾‍♂️
        'en' : ':man_wearing_turban_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban_mitteldunkle_hautfarbe:',
        'es': ':hombre_con_turbante_tono_de_piel_oscuro_medio:',
        'fr': ':homme_en_turban_peau_mate:',
        'pt': ':homem_com_turbante_pele_morena_escura:',
        'it': ':uomo_con_turbante_carnagione_abbastanza_scura:'
    },
    u'\U0001F473\U0001F3FE\U0000200D\U00002642': { # 👳🏾‍♂
        'en' : ':man_wearing_turban_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 👳🏼‍♂️
        'en' : ':man_wearing_turban_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban_mittelhelle_hautfarbe:',
        'es': ':hombre_con_turbante_tono_de_piel_claro_medio:',
        'fr': ':homme_en_turban_peau_moyennement_claire:',
        'pt': ':homem_com_turbante_pele_morena_clara:',
        'it': ':uomo_con_turbante_carnagione_abbastanza_chiara:'
    },
    u'\U0001F473\U0001F3FC\U0000200D\U00002642': { # 👳🏼‍♂
        'en' : ':man_wearing_turban_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F473\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 👳🏽‍♂️
        'en' : ':man_wearing_turban_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 4,
        'de': ':mann_mit_turban_mittlere_hautfarbe:',
        'es': ':hombre_con_turbante_tono_de_piel_medio:',
        'fr': u':homme_en_turban_peau_légèrement_mate:',
        'pt': ':homem_com_turbante_pele_morena:',
        'it': ':uomo_con_turbante_carnagione_olivastra:'
    },
    u'\U0001F473\U0001F3FD\U0000200D\U00002642': { # 👳🏽‍♂
        'en' : ':man_wearing_turban_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 4
    },
    u'\U0001F468\U0000200D\U0001F9B3': { # 👨‍🦳
        'en' : ':man_white_hair:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mann_weißes_haar:',
        'es': ':hombre_pelo_blanco:',
        'fr': ':homme_cheveux_blancs:',
        'pt': ':homem_cabelo_branco:',
        'it': ':uomo_capelli_bianchi:'
    },
    u'\U0001F470\U0000200D\U00002642\U0000FE0F': { # 👰‍♂️
        'en' : ':man_with_veil:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier:',
        'es': ':hombre_con_velo:',
        'fr': ':homme_avec_voile:',
        'pt': u':noivo_com_véu:',
        'it': ':sposo_con_velo:'
    },
    u'\U0001F470\U0000200D\U00002642': { # 👰‍♂
        'en' : ':man_with_veil:',
        'status' : minimally_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier:',
        'es': ':hombre_con_velo:',
        'fr': ':homme_avec_voile:',
        'pt': u':noivo_com_véu:',
        'it': ':sposo_con_velo:'
    },
    u'\U0001F470\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 👰🏿‍♂️
        'en' : ':man_with_veil_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier_dunkle_hautfarbe:',
        'es': ':hombre_con_velo_tono_de_piel_oscuro:',
        'fr': u':homme_avec_voile_peau_foncée:',
        'pt': u':noivo_com_véu_pele_escura:',
        'it': ':sposo_con_velo_carnagione_scura:'
    },
    u'\U0001F470\U0001F3FF\U0000200D\U00002642': { # 👰🏿‍♂
        'en' : ':man_with_veil_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 👰🏻‍♂️
        'en' : ':man_with_veil_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier_helle_hautfarbe:',
        'es': ':hombre_con_velo_tono_de_piel_claro:',
        'fr': ':homme_avec_voile_peau_claire:',
        'pt': u':noivo_com_véu_pele_clara:',
        'it': ':sposo_con_velo_carnagione_chiara:'
    },
    u'\U0001F470\U0001F3FB\U0000200D\U00002642': { # 👰🏻‍♂
        'en' : ':man_with_veil_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 👰🏾‍♂️
        'en' : ':man_with_veil_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier_mitteldunkle_hautfarbe:',
        'es': ':hombre_con_velo_tono_de_piel_oscuro_medio:',
        'fr': ':homme_avec_voile_peau_mate:',
        'pt': u':noivo_com_véu_pele_morena_escura:',
        'it': ':sposo_con_velo_carnagione_abbastanza_scura:'
    },
    u'\U0001F470\U0001F3FE\U0000200D\U00002642': { # 👰🏾‍♂
        'en' : ':man_with_veil_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 👰🏼‍♂️
        'en' : ':man_with_veil_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier_mittelhelle_hautfarbe:',
        'es': ':hombre_con_velo_tono_de_piel_claro_medio:',
        'fr': ':homme_avec_voile_peau_moyennement_claire:',
        'pt': u':noivo_com_véu_pele_morena_clara:',
        'it': ':sposo_con_velo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F470\U0001F3FC\U0000200D\U00002642': { # 👰🏼‍♂
        'en' : ':man_with_veil_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F470\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 👰🏽‍♂️
        'en' : ':man_with_veil_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mann_mit_schleier_mittlere_hautfarbe:',
        'es': ':hombre_con_velo_tono_de_piel_medio:',
        'fr': u':homme_avec_voile_peau_légèrement_mate:',
        'pt': u':noivo_com_véu_pele_morena:',
        'it': ':sposo_con_velo_carnagione_olivastra:'
    },
    u'\U0001F470\U0001F3FD\U0000200D\U00002642': { # 👰🏽‍♂
        'en' : ':man_with_veil_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 13
    },
    u'\U0001F468\U0000200D\U0001F9AF': { # 👨‍🦯
        'en' : ':man_with_white_cane:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_langstock:',
        'es': u':hombre_con_bastón:',
        'fr': ':homme_avec_canne_blanche:',
        'pt': ':homem_com_bengala_para_cego:',
        'it': ':uomo_con_bastone_bianco_di_orientamento:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F9AF': { # 👨🏿‍🦯
        'en' : ':man_with_white_cane_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_gehstock_dunkle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_oscuro:',
        'fr': u':homme_avec_canne_blanche_peau_foncée:',
        'pt': ':homem_com_bengala_para_cego_pele_escura:',
        'it': ':uomo_con_bastone_bianco_di_orientamento_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F9AF': { # 👨🏻‍🦯
        'en' : ':man_with_white_cane_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_gehstock_helle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_claro:',
        'fr': ':homme_avec_canne_blanche_peau_claire:',
        'pt': ':homem_com_bengala_para_cego_pele_clara:',
        'it': ':uomo_con_bastone_bianco_di_orientamento_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F9AF': { # 👨🏾‍🦯
        'en' : ':man_with_white_cane_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_gehstock_mitteldunkle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_oscuro_medio:',
        'fr': ':homme_avec_canne_blanche_peau_mate:',
        'pt': ':homem_com_bengala_para_cego_pele_morena_escura:',
        'it': ':uomo_con_bastone_bianco_di_orientamento_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F9AF': { # 👨🏼‍🦯
        'en' : ':man_with_white_cane_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_gehstock_mittelhelle_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_claro_medio:',
        'fr': ':homme_avec_canne_blanche_peau_moyennement_claire:',
        'pt': ':homem_com_bengala_para_cego_pele_morena_clara:',
        'it': ':uomo_con_bastone_bianco_di_orientamento_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F9AF': { # 👨🏽‍🦯
        'en' : ':man_with_white_cane_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mann_mit_gehstock_mittlere_hautfarbe:',
        'es': u':hombre_con_bastón_tono_de_piel_medio:',
        'fr': u':homme_avec_canne_blanche_peau_légèrement_mate:',
        'pt': ':homem_com_bengala_para_cego_pele_morena:',
        'it': ':uomo_con_bastone_bianco_di_orientamento_carnagione_olivastra:'
    },
    u'\U0001F9DF\U0000200D\U00002642\U0000FE0F': { # 🧟‍♂️
        'en' : ':man_zombie:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':männlicher_zombie:',
        'es': ':zombi_hombre:',
        'fr': ':zombie_homme:',
        'pt': ':homem_zumbi:',
        'it': ':zombie_uomo:'
    },
    u'\U0001F9DF\U0000200D\U00002642': { # 🧟‍♂
        'en' : ':man_zombie:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': u':männlicher_zombie:',
        'es': ':zombi_hombre:',
        'fr': ':zombie_homme:',
        'pt': ':homem_zumbi:',
        'it': ':zombie_uomo:'
    },
    u'\U0001F96D': { # 🥭
        'en' : ':mango:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mango:',
        'es': ':mango:',
        'fr': ':mangue:',
        'pt': ':manga:',
        'it': ':mango:'
    },
    u'\U0001F570\U0000FE0F': { # 🕰️
        'en' : ':mantelpiece_clock:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kaminuhr:',
        'es': ':reloj_de_sobremesa:',
        'fr': ':pendule:',
        'pt': u':relógio_de_mesa:',
        'it': ':orologio_da_mensola:'
    },
    u'\U0001F570': { # 🕰
        'en' : ':mantelpiece_clock:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':kaminuhr:',
        'es': ':reloj_de_sobremesa:',
        'fr': ':pendule:',
        'pt': u':relógio_de_mesa:',
        'it': ':orologio_da_mensola:'
    },
    u'\U0001F9BD': { # 🦽
        'en' : ':manual_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':manueller_rollstuhl:',
        'es': ':silla_de_ruedas_manual:',
        'fr': ':fauteuil_roulant_manuel:',
        'pt': ':cadeira_de_rodas_manual:',
        'it': ':sedia_a_rotelle_manuale:'
    },
    u'\U0001F45E': { # 👞
        'en' : u':man’s_shoe:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mans_shoe:', ':shoe:'],
        'de': ':herrenschuh:',
        'es': ':zapato_de_hombre:',
        'fr': u':chaussure_d’homme:',
        'pt': ':sapato_masculino:',
        'it': ':scarpa_da_uomo:'
    },
    u'\U0001F5FE': { # 🗾
        'en' : ':map_of_Japan:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':japan:'],
        'de': ':umriss_von_japan:',
        'es': u':mapa_de_japón:',
        'fr': ':carte_du_japon:',
        'pt': u':mapa_do_japão:',
        'it': ':mappa_del_giappone:'
    },
    u'\U0001F341': { # 🍁
        'en' : ':maple_leaf:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':ahornblatt:',
        'es': ':hoja_de_arce:',
        'fr': u':feuille_d’érable:',
        'pt': ':folha_de_bordo:',
        'it': u':foglia_d’acero:'
    },
    u'\U0001F94B': { # 🥋
        'en' : ':martial_arts_uniform:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':kampfsportanzug:',
        'es': ':uniforme_de_artes_marciales:',
        'fr': u':tenue_d’arts_martiaux:',
        'pt': ':quimono_de_artes_marciais:',
        'it': ':kimono_per_arti_marziali:'
    },
    u'\U0001F9C9': { # 🧉
        'en' : ':mate:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':mate-tee:',
        'es': ':mate:',
        'fr': u':maté:',
        'pt': ':mate:',
        'it': ':mate:'
    },
    u'\U0001F356': { # 🍖
        'en' : ':meat_on_bone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':fleischhachse:',
        'es': ':carne_con_hueso:',
        'fr': ':viande_sur_un_os:',
        'pt': ':carne:',
        'it': ':carne:'
    },
    u'\U0001F9D1\U0000200D\U0001F527': { # 🧑‍🔧
        'en' : ':mechanic:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in):',
        'es': u':profesional_de_la_mecánica:',
        'fr': u':mécanicien_(tous_genres):',
        'pt': u':mecânico_trabalhando:',
        'it': ':meccanico:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F527': { # 🧑🏿‍🔧
        'en' : ':mechanic_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in)_dunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_oscuro:',
        'fr': u':mécanicien_(tous_genres)_peau_foncée:',
        'pt': u':mecânico_trabalhando_pele_escura:',
        'it': ':meccanico_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F527': { # 🧑🏻‍🔧
        'en' : ':mechanic_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in)_helle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_claro:',
        'fr': u':mécanicien_(tous_genres)_peau_claire:',
        'pt': u':mecânico_trabalhando_pele_clara:',
        'it': ':meccanico_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F527': { # 🧑🏾‍🔧
        'en' : ':mechanic_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in)_mitteldunkle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_oscuro_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_mate:',
        'pt': u':mecânico_trabalhando_pele_morena_escura:',
        'it': ':meccanico_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F527': { # 🧑🏼‍🔧
        'en' : ':mechanic_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in)_mittelhelle_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_claro_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_moyennement_claire:',
        'pt': u':mecânico_trabalhando_pele_morena_clara:',
        'it': ':meccanico_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F527': { # 🧑🏽‍🔧
        'en' : ':mechanic_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':mechaniker(in)_mittlere_hautfarbe:',
        'es': u':profesional_de_la_mecánica_tono_de_piel_medio:',
        'fr': u':mécanicien_(tous_genres)_peau_légèrement_mate:',
        'pt': u':mecânico_trabalhando_pele_morena:',
        'it': ':meccanico_carnagione_olivastra:'
    },
    u'\U0001F9BE': { # 🦾
        'en' : ':mechanical_arm:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':armprothese:',
        'es': u':brazo_mecánico:',
        'fr': u':bras_mécanique:',
        'pt': u':braço_mecânico:',
        'it': ':protesi_robotica_per_il_braccio:'
    },
    u'\U0001F9BF': { # 🦿
        'en' : ':mechanical_leg:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':beinprothese:',
        'es': u':pierna_mecánica:',
        'fr': u':jambe_mécanique:',
        'pt': u':perna_mecânica:',
        'it': ':protesi_robotica_per_la_gamba:'
    },
    u'\U00002695\U0000FE0F': { # ⚕️
        'en' : ':medical_symbol:',
        'status' : fully_qualified,
        'E' : 4,
        'variant': True,
        'de': u':äskulapstab:',
        'es': u':símbolo_de_medicina:',
        'fr': u':caducée:',
        'pt': u':símbolo_da_medicina:',
        'it': ':simbolo_della_medicina:'
    },
    u'\U00002695': { # ⚕
        'en' : ':medical_symbol:',
        'status' : unqualified,
        'E' : 4,
        'variant': True,
        'de': u':äskulapstab:',
        'es': u':símbolo_de_medicina:',
        'fr': u':caducée:',
        'pt': u':símbolo_da_medicina:',
        'it': ':simbolo_della_medicina:'
    },
    u'\U0001F3FE': { # 🏾
        'en' : ':medium-dark_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [':emoji_modifier_fitzpatrick_type__5:'],
        'de': ':mitteldunkle_hautfarbe:',
        'es': ':tono_de_piel_oscuro_medio:',
        'fr': ':peau_mate:',
        'pt': ':pele_morena_escura:',
        'it': ':carnagione_abbastanza_scura:'
    },
    u'\U0001F3FC': { # 🏼
        'en' : ':medium-light_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [':emoji_modifier_fitzpatrick_type__3:'],
        'de': ':mittelhelle_hautfarbe:',
        'es': ':tono_de_piel_claro_medio:',
        'fr': ':peau_moyennement_claire:',
        'pt': ':pele_morena_clara:',
        'it': ':carnagione_abbastanza_chiara:'
    },
    u'\U0001F3FD': { # 🏽
        'en' : ':medium_skin_tone:',
        'status' : component,
        'E' : 1,
        'alias' : [':emoji_modifier_fitzpatrick_type__4:'],
        'de': ':mittlere_hautfarbe:',
        'es': ':tono_de_piel_medio:',
        'fr': u':peau_légèrement_mate:',
        'pt': ':pele_morena:',
        'it': ':carnagione_olivastra:'
    },
    u'\U0001F4E3': { # 📣
        'en' : ':megaphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mega:'],
        'de': ':megafon:',
        'es': u':megáfono:',
        'fr': ':porte-voix:',
        'pt': ':megafone:',
        'it': ':megafono:'
    },
    u'\U0001F348': { # 🍈
        'en' : ':melon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':honigmelone:',
        'es': u':melón:',
        'fr': ':melon:',
        'pt': u':melão:',
        'it': ':melone:'
    },
    u'\U0001FAE0': { # 🫠
        'en' : ':melting_face:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':schmelzendes_gesicht:',
        'es': u':cara_derritiéndose:',
        'fr': ':visage_qui_fond:',
        'pt': ':rosto_derretendo:',
        'it': ':faccina_che_si_scioglie:'
    },
    u'\U0001F4DD': { # 📝
        'en' : ':memo:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pencil:'],
        'de': ':papier_und_bleistift:',
        'es': ':cuaderno_de_notas:',
        'fr': u':mémo:',
        'pt': ':memorando:',
        'it': ':memo:'
    },
    u'\U0001F46C': { # 👬
        'en' : ':men_holding_hands:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':two_men_holding_hands:'],
        'de': u':händchen_haltende_männer:',
        'es': ':hombres_de_la_mano:',
        'fr': ':deux_hommes_se_tenant_la_main:',
        'pt': u':dois_homens_de_mãos_dadas:',
        'it': ':due_uomini_che_si_tengono_per_mano:'
    },
    u'\U0001F46C\U0001F3FF': { # 👬🏿
        'en' : ':men_holding_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': { # 👨🏿‍🤝‍👨🏻
        'en' : ':men_holding_hands_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_helle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': { # 👨🏿‍🤝‍👨🏾
        'en' : ':men_holding_hands_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': { # 👨🏿‍🤝‍👨🏼
        'en' : ':men_holding_hands_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': { # 👨🏿‍🤝‍👨🏽
        'en' : ':men_holding_hands_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_dunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_foncée_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_escura_e_pele_morena:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_scura_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FB': { # 👬🏻
        'en' : ':men_holding_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_helle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': { # 👨🏻‍🤝‍👨🏿
        'en' : ':men_holding_hands_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_dunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': { # 👨🏻‍🤝‍👨🏾
        'en' : ':men_holding_hands_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_claire_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': { # 👨🏻‍🤝‍👨🏼
        'en' : ':men_holding_hands_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_claro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_claire_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': { # 👨🏻‍🤝‍👨🏽
        'en' : ':men_holding_hands_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_helle_hautfarbe,_mittlere_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_claire_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_clara_e_pele_morena:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FE': { # 👬🏾
        'en' : ':men_holding_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': { # 👨🏾‍🤝‍👨🏿
        'en' : ':men_holding_hands_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_dunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': { # 👨🏾‍🤝‍👨🏻
        'en' : ':men_holding_hands_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_helle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_mate_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': { # 👨🏾‍🤝‍👨🏼
        'en' : ':men_holding_hands_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_mate_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_morena_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': { # 👨🏾‍🤝‍👨🏽
        'en' : ':men_holding_hands_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mitteldunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_mate_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_escura_e_pele_morena:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FC': { # 👬🏼
        'en' : ':men_holding_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': { # 👨🏼‍🤝‍👨🏿
        'en' : ':men_holding_hands_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_dunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': { # 👨🏼‍🤝‍👨🏻
        'en' : ':men_holding_hands_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_helle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_claro:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': { # 👨🏼‍🤝‍👨🏾
        'en' : ':men_holding_hands_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro_medio:',
        'fr': ':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_morena_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FD': { # 👨🏼‍🤝‍👨🏽
        'en' : ':men_holding_hands_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittelhelle_hautfarbe,_mittlere_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_moyennement_claire_et_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_clara_e_pele_morena:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F46C\U0001F3FD': { # 👬🏽
        'en' : ':men_holding_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_olivastra:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FF': { # 👨🏽‍🤝‍👨🏿
        'en' : ':men_holding_hands_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_dunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_foncée:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_scura:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FB': { # 👨🏽‍🤝‍👨🏻
        'en' : ':men_holding_hands_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_helle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_claro:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_chiara:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FE': { # 👨🏽‍🤝‍👨🏾
        'en' : ':men_holding_hands_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_mate:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_morena_escura:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F468\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F468\U0001F3FC': { # 👨🏽‍🤝‍👨🏼
        'en' : ':men_holding_hands_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':händchenhaltende_männer_mittlere_hautfarbe,_mittelhelle_hautfarbe:',
        'es': ':hombres_de_la_mano_tono_de_piel_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_hommes_se_tenant_la_main_peau_légèrement_mate_et_peau_moyennement_claire:',
        'pt': u':dois_homens_de_mãos_dadas_pele_morena_e_pele_morena_clara:',
        'it': ':due_uomini_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F46F\U0000200D\U00002642\U0000FE0F': { # 👯‍♂️
        'en' : ':men_with_bunny_ears:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':männer_mit_hasenohren:',
        'es': ':hombres_con_orejas_de_conejo:',
        'fr': ':hommes_avec_des_oreilles_de_lapin:',
        'pt': ':homens_com_orelhas_de_coelho:',
        'it': ':uomini_con_orecchie_da_coniglio:'
    },
    u'\U0001F46F\U0000200D\U00002642': { # 👯‍♂
        'en' : ':men_with_bunny_ears:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': u':männer_mit_hasenohren:',
        'es': ':hombres_con_orejas_de_conejo:',
        'fr': ':hommes_avec_des_oreilles_de_lapin:',
        'pt': ':homens_com_orelhas_de_coelho:',
        'it': ':uomini_con_orecchie_da_coniglio:'
    },
    u'\U0001F93C\U0000200D\U00002642\U0000FE0F': { # 🤼‍♂️
        'en' : ':men_wrestling:',
        'status' : fully_qualified,
        'E' : 4,
        'de': u':ringende_männer:',
        'es': ':hombres_luchando:',
        'fr': ':lutteurs:',
        'pt': ':homens_lutando:',
        'it': ':lottatori:'
    },
    u'\U0001F93C\U0000200D\U00002642': { # 🤼‍♂
        'en' : ':men_wrestling:',
        'status' : minimally_qualified,
        'E' : 4,
        'de': u':ringende_männer:',
        'es': ':hombres_luchando:',
        'fr': ':lutteurs:',
        'pt': ':homens_lutando:',
        'it': ':lottatori:'
    },
    u'\U00002764\U0000FE0F\U0000200D\U0001FA79': { # ❤️‍🩹
        'en' : ':mending_heart:',
        'status' : fully_qualified,
        'E' : 13.1
    },
    u'\U00002764\U0000200D\U0001FA79': { # ❤‍🩹
        'en' : ':mending_heart:',
        'status' : unqualified,
        'E' : 13.1,
        'de': ':herz_mit_verband:',
        'es': u':corazón_curándose:',
        'fr': u':cœur_avec_bandages:',
        'pt': u':coração_remendado:',
        'it': ':cuore_guarito:'
    },
    u'\U0001F54E': { # 🕎
        'en' : ':menorah:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':menorah_with_nine_branches:'],
        'de': ':menora:',
        'es': u':menorá:',
        'fr': u':chandelier_à_sept_branches:',
        'pt': u':menorá:',
        'it': ':menorah:'
    },
    u'\U0001F6B9': { # 🚹
        'en' : u':men’s_room:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mens:'],
        'variant': True,
        'de': ':herren:',
        'es': ':aseo_para_hombres:',
        'fr': ':symbole_toilettes_hommes:',
        'pt': ':banheiro_masculino:',
        'it': ':simbolo_del_bagno_degli_uomini:'
    },
    u'\U0001F9DC\U0000200D\U00002640\U0000FE0F': { # 🧜‍♀️
        'en' : ':mermaid:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau:',
        'es': ':sirena:',
        'fr': u':sirène:',
        'pt': ':sereia:',
        'it': ':sirena_donna:'
    },
    u'\U0001F9DC\U0000200D\U00002640': { # 🧜‍♀
        'en' : ':mermaid:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':meerjungfrau:',
        'es': ':sirena:',
        'fr': u':sirène:',
        'pt': ':sereia:',
        'it': ':sirena_donna:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002640\U0000FE0F': { # 🧜🏿‍♀️
        'en' : ':mermaid_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau_dunkle_hautfarbe:',
        'es': ':sirena_tono_de_piel_oscuro:',
        'fr': u':sirène_peau_foncée:',
        'pt': ':sereia_pele_escura:',
        'it': ':sirena_donna_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002640': { # 🧜🏿‍♀
        'en' : ':mermaid_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002640\U0000FE0F': { # 🧜🏻‍♀️
        'en' : ':mermaid_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau_helle_hautfarbe:',
        'es': ':sirena_tono_de_piel_claro:',
        'fr': u':sirène_peau_claire:',
        'pt': ':sereia_pele_clara:',
        'it': ':sirena_donna_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002640': { # 🧜🏻‍♀
        'en' : ':mermaid_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002640\U0000FE0F': { # 🧜🏾‍♀️
        'en' : ':mermaid_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau_mitteldunkle_hautfarbe:',
        'es': ':sirena_tono_de_piel_oscuro_medio:',
        'fr': u':sirène_peau_mate:',
        'pt': ':sereia_pele_morena_escura:',
        'it': ':sirena_donna_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002640': { # 🧜🏾‍♀
        'en' : ':mermaid_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002640\U0000FE0F': { # 🧜🏼‍♀️
        'en' : ':mermaid_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau_mittelhelle_hautfarbe:',
        'es': ':sirena_tono_de_piel_claro_medio:',
        'fr': u':sirène_peau_moyennement_claire:',
        'pt': ':sereia_pele_morena_clara:',
        'it': ':sirena_donna_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002640': { # 🧜🏼‍♀
        'en' : ':mermaid_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002640\U0000FE0F': { # 🧜🏽‍♀️
        'en' : ':mermaid_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':meerjungfrau_mittlere_hautfarbe:',
        'es': ':sirena_tono_de_piel_medio:',
        'fr': u':sirène_peau_légèrement_mate:',
        'pt': ':sereia_pele_morena:',
        'it': ':sirena_donna_carnagione_olivastra:'
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002640': { # 🧜🏽‍♀
        'en' : ':mermaid_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0000200D\U00002642\U0000FE0F': { # 🧜‍♂️
        'en' : ':merman:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann:',
        'es': ':sirena_hombre:',
        'fr': ':triton:',
        'pt': ':sereio:',
        'it': ':tritone:'
    },
    u'\U0001F9DC\U0000200D\U00002642': { # 🧜‍♂
        'en' : ':merman:',
        'status' : minimally_qualified,
        'E' : 5,
        'de': ':wassermann:',
        'es': ':sirena_hombre:',
        'fr': ':triton:',
        'pt': ':sereio:',
        'it': ':tritone:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002642\U0000FE0F': { # 🧜🏿‍♂️
        'en' : ':merman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann_dunkle_hautfarbe:',
        'es': ':sirena_hombre_tono_de_piel_oscuro:',
        'fr': u':triton_peau_foncée:',
        'pt': ':sereio_pele_escura:',
        'it': ':tritone_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FF\U0000200D\U00002642': { # 🧜🏿‍♂
        'en' : ':merman_dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002642\U0000FE0F': { # 🧜🏻‍♂️
        'en' : ':merman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann_helle_hautfarbe:',
        'es': ':sirena_hombre_tono_de_piel_claro:',
        'fr': ':triton_peau_claire:',
        'pt': ':sereio_pele_clara:',
        'it': ':tritone_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FB\U0000200D\U00002642': { # 🧜🏻‍♂
        'en' : ':merman_light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002642\U0000FE0F': { # 🧜🏾‍♂️
        'en' : ':merman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann_mitteldunkle_hautfarbe:',
        'es': ':sirena_hombre_tono_de_piel_oscuro_medio:',
        'fr': ':triton_peau_mate:',
        'pt': ':sereio_pele_morena_escura:',
        'it': ':tritone_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FE\U0000200D\U00002642': { # 🧜🏾‍♂
        'en' : ':merman_medium-dark_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002642\U0000FE0F': { # 🧜🏼‍♂️
        'en' : ':merman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann_mittelhelle_hautfarbe:',
        'es': ':sirena_hombre_tono_de_piel_claro_medio:',
        'fr': ':triton_peau_moyennement_claire:',
        'pt': ':sereio_pele_morena_clara:',
        'it': ':tritone_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FC\U0000200D\U00002642': { # 🧜🏼‍♂
        'en' : ':merman_medium-light_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002642\U0000FE0F': { # 🧜🏽‍♂️
        'en' : ':merman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermann_mittlere_hautfarbe:',
        'es': ':sirena_hombre_tono_de_piel_medio:',
        'fr': u':triton_peau_légèrement_mate:',
        'pt': ':sereio_pele_morena:',
        'it': ':tritone_carnagione_olivastra:'
    },
    u'\U0001F9DC\U0001F3FD\U0000200D\U00002642': { # 🧜🏽‍♂
        'en' : ':merman_medium_skin_tone:',
        'status' : minimally_qualified,
        'E' : 5
    },
    u'\U0001F9DC': { # 🧜
        'en' : ':merperson:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch:',
        'es': ':persona_sirena:',
        'fr': u':créature_aquatique:',
        'pt': ':pessoa_sereia:',
        'it': ':sirena:'
    },
    u'\U0001F9DC\U0001F3FF': { # 🧜🏿
        'en' : ':merperson_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch_dunkle_hautfarbe:',
        'es': ':persona_sirena_tono_de_piel_oscuro:',
        'fr': u':créature_aquatique_peau_foncée:',
        'pt': ':pessoa_sereia_pele_escura:',
        'it': ':sirena_carnagione_scura:'
    },
    u'\U0001F9DC\U0001F3FB': { # 🧜🏻
        'en' : ':merperson_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch_helle_hautfarbe:',
        'es': ':persona_sirena_tono_de_piel_claro:',
        'fr': u':créature_aquatique_peau_claire:',
        'pt': ':pessoa_sereia_pele_clara:',
        'it': ':sirena_carnagione_chiara:'
    },
    u'\U0001F9DC\U0001F3FE': { # 🧜🏾
        'en' : ':merperson_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch_mitteldunkle_hautfarbe:',
        'es': ':persona_sirena_tono_de_piel_oscuro_medio:',
        'fr': u':créature_aquatique_peau_mate:',
        'pt': ':pessoa_sereia_pele_morena_escura:',
        'it': ':sirena_carnagione_abbastanza_scura:'
    },
    u'\U0001F9DC\U0001F3FC': { # 🧜🏼
        'en' : ':merperson_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch_mittelhelle_hautfarbe:',
        'es': ':persona_sirena_tono_de_piel_claro_medio:',
        'fr': u':créature_aquatique_peau_moyennement_claire:',
        'pt': ':pessoa_sereia_pele_morena_clara:',
        'it': ':sirena_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9DC\U0001F3FD': { # 🧜🏽
        'en' : ':merperson_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':wassermensch_mittlere_hautfarbe:',
        'es': ':persona_sirena_tono_de_piel_medio:',
        'fr': u':créature_aquatique_peau_légèrement_mate:',
        'pt': ':pessoa_sereia_pele_morena:',
        'it': ':sirena_carnagione_olivastra:'
    },
    u'\U0001F687': { # 🚇
        'en' : ':metro:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':u-bahn:',
        'es': ':metro:',
        'fr': u':métro:',
        'pt': u':metrô:',
        'it': ':metropolitana:'
    },
    u'\U0001F9A0': { # 🦠
        'en' : ':microbe:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mikrobe:',
        'es': ':microbio:',
        'fr': ':microbe:',
        'pt': u':micróbio:',
        'it': ':microbo:'
    },
    u'\U0001F3A4': { # 🎤
        'en' : ':microphone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':mikrofon:',
        'es': u':micrófono:',
        'fr': ':micro:',
        'pt': ':microfone:',
        'it': ':microfono:'
    },
    u'\U0001F52C': { # 🔬
        'en' : ':microscope:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mikroskop:',
        'es': ':microscopio:',
        'fr': ':microscope:',
        'pt': u':microscópio:',
        'it': ':microscopio:'
    },
    u'\U0001F595': { # 🖕
        'en' : ':middle_finger:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':fu:', ':reversed_hand_with_middle_finger_extended:'],
        'de': ':mittelfinger:',
        'es': u':dedo_corazón_hacia_arriba:',
        'fr': u':doigt_d’honneur:',
        'pt': ':dedo_do_meio:',
        'it': ':dito_medio:'
    },
    u'\U0001F595\U0001F3FF': { # 🖕🏿
        'en' : ':middle_finger_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mittelfinger_dunkle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_oscuro:',
        'fr': u':doigt_d’honneur_peau_foncée:',
        'pt': ':dedo_do_meio_pele_escura:',
        'it': ':dito_medio_carnagione_scura:'
    },
    u'\U0001F595\U0001F3FB': { # 🖕🏻
        'en' : ':middle_finger_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mittelfinger_helle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_claro:',
        'fr': u':doigt_d’honneur_peau_claire:',
        'pt': ':dedo_do_meio_pele_clara:',
        'it': ':dito_medio_carnagione_chiara:'
    },
    u'\U0001F595\U0001F3FE': { # 🖕🏾
        'en' : ':middle_finger_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mittelfinger_mitteldunkle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_oscuro_medio:',
        'fr': u':doigt_d’honneur_peau_mate:',
        'pt': ':dedo_do_meio_pele_morena_escura:',
        'it': ':dito_medio_carnagione_abbastanza_scura:'
    },
    u'\U0001F595\U0001F3FC': { # 🖕🏼
        'en' : ':middle_finger_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mittelfinger_mittelhelle_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_claro_medio:',
        'fr': u':doigt_d’honneur_peau_moyennement_claire:',
        'pt': ':dedo_do_meio_pele_morena_clara:',
        'it': ':dito_medio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F595\U0001F3FD': { # 🖕🏽
        'en' : ':middle_finger_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mittelfinger_mittlere_hautfarbe:',
        'es': u':dedo_corazón_hacia_arriba_tono_de_piel_medio:',
        'fr': u':doigt_d’honneur_peau_légèrement_mate:',
        'pt': ':dedo_do_meio_pele_morena:',
        'it': ':dito_medio_carnagione_olivastra:'
    },
    u'\U0001FA96': { # 🪖
        'en' : ':military_helmet:',
        'status' : fully_qualified,
        'E' : 13,
        'de': u':militärhelm:',
        'es': ':casco_militar:',
        'fr': ':casque_militaire:',
        'pt': ':capacete_militar:',
        'it': ':elmetto_militare:'
    },
    u'\U0001F396\U0000FE0F': { # 🎖️
        'en' : ':military_medal:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':medal_military:'],
        'variant': True,
        'de': u':militärorden:',
        'es': ':medalla_militar:',
        'fr': u':médaille_militaire:',
        'pt': ':medalha_militar:',
        'it': ':medaglia_militare:'
    },
    u'\U0001F396': { # 🎖
        'en' : ':military_medal:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':medal_military:'],
        'variant': True,
        'de': u':militärorden:',
        'es': ':medalla_militar:',
        'fr': u':médaille_militaire:',
        'pt': ':medalha_militar:',
        'it': ':medaglia_militare:'
    },
    u'\U0001F30C': { # 🌌
        'en' : ':milky_way:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':milchstraße:',
        'es': u':vía_láctea:',
        'fr': u':voie_lactée:',
        'pt': u':via_láctea:',
        'it': ':via_lattea:'
    },
    u'\U0001F690': { # 🚐
        'en' : ':minibus:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':kleinbus:',
        'es': u':minibús:',
        'fr': ':minibus:',
        'pt': ':van:',
        'it': ':pulmino:'
    },
    u'\U00002796': { # ➖
        'en' : ':minus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heavy_minus_sign:'],
        'de': ':minus:',
        'es': ':menos:',
        'fr': ':moins:',
        'pt': u':símbolo_de_subtração:',
        'it': ':sottrazione:'
    },
    u'\U0001FA9E': { # 🪞
        'en' : ':mirror:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':spiegel:',
        'es': ':espejo:',
        'fr': ':miroir:',
        'pt': ':espelho:',
        'it': ':specchio:'
    },
    u'\U0001FAA9': { # 🪩
        'en' : ':mirror_ball:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':discokugel:',
        'es': ':bola_de_espejos:',
        'fr': u':boule_à_facettes:',
        'pt': ':globo_de_espelhos:',
        'it': ':palla_da_discoteca:'
    },
    u'\U0001F5FF': { # 🗿
        'en' : ':moai:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':moyai:'],
        'de': ':statue:',
        'es': u':estatua_moái:',
        'fr': ':moai:',
        'pt': ':moai:',
        'it': ':moai:'
    },
    u'\U0001F4F1': { # 📱
        'en' : ':mobile_phone:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':iphone:'],
        'de': ':mobiltelefon:',
        'es': u':teléfono_móvil:',
        'fr': u':téléphone_portable:',
        'pt': ':telefone_celular:',
        'it': ':telefono_cellulare:'
    },
    u'\U0001F4F4': { # 📴
        'en' : ':mobile_phone_off:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':mobiltelefon_aus:',
        'es': u':teléfono_móvil_apagado:',
        'fr': u':téléphone_éteint:',
        'pt': ':telefone_celular_desligado:',
        'it': ':cellulare_spento:'
    },
    u'\U0001F4F2': { # 📲
        'en' : ':mobile_phone_with_arrow:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':calling:'],
        'de': ':mobiltelefon_mit_pfeil:',
        'es': u':móvil_con_una_flecha:',
        'fr': ':appel_entrant:',
        'pt': ':telefone_celular_com_seta:',
        'it': ':telefono_cellulare_con_freccia:'
    },
    u'\U0001F911': { # 🤑
        'en' : ':money-mouth_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':money_mouth_face:', ':money__mouth_face:'],
        'de': ':gesicht_mit_dollarzeichen:',
        'es': ':cara_con_lengua_de_dinero:',
        'fr': ':argent_dans_les_yeux_et_la_bouche:',
        'pt': u':rosto_com_cifrões:',
        'it': ':faccina_avida_di_denaro:'
    },
    u'\U0001F4B0': { # 💰
        'en' : ':money_bag:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':moneybag:'],
        'variant': True,
        'de': ':geldsack:',
        'es': ':bolsa_de_dinero:',
        'fr': u':sac_plein_d’argent:',
        'pt': ':saco_de_dinheiro:',
        'it': ':sacco_di_soldi:'
    },
    u'\U0001F4B8': { # 💸
        'en' : ':money_with_wings:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geldschein_mit_flügeln:',
        'es': ':billete_con_alas:',
        'fr': ':billet_avec_des_ailes:',
        'pt': ':dinheiro_voando:',
        'it': ':soldi_con_le_ali:'
    },
    u'\U0001F412': { # 🐒
        'en' : ':monkey:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':affe:',
        'es': ':mono:',
        'fr': ':singe:',
        'pt': ':macaco:',
        'it': ':scimmia:'
    },
    u'\U0001F435': { # 🐵
        'en' : ':monkey_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':affengesicht:',
        'es': ':cara_de_mono:',
        'fr': u':tête_de_singe:',
        'pt': ':rosto_de_macaco:',
        'it': ':muso_di_scimmia:'
    },
    u'\U0001F69D': { # 🚝
        'en' : ':monorail:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':einschienenbahn:',
        'es': u':monorraíl:',
        'fr': ':monorail:',
        'pt': ':monotrilho:',
        'it': ':monorotaia:'
    },
    u'\U0001F96E': { # 🥮
        'en' : ':moon_cake:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':mondkuchen:',
        'es': ':pastel_de_luna:',
        'fr': u':gâteau_de_lune:',
        'pt': ':bolo_lunar:',
        'it': ':torta_della_luna:'
    },
    u'\U0001F391': { # 🎑
        'en' : ':moon_viewing_ceremony:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':rice_scene:'],
        'de': ':traditionelles_mondfest:',
        'es': u':ceremonia_de_contemplación_de_la_luna:',
        'fr': u':cérémonie_de_la_lune:',
        'pt': u':contemplação_da_lua:',
        'it': ':contemplazione_della_luna:'
    },
    u'\U0001F54C': { # 🕌
        'en' : ':mosque:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':moschee:',
        'es': ':mezquita:',
        'fr': u':mosquée:',
        'pt': ':mesquita:',
        'it': ':moschea:'
    },
    u'\U0001F99F': { # 🦟
        'en' : ':mosquito:',
        'status' : fully_qualified,
        'E' : 11,
        'de': u':mücke:',
        'es': ':mosquito:',
        'fr': ':moustique:',
        'pt': ':mosquito:',
        'it': ':zanzara:'
    },
    u'\U0001F6E5\U0000FE0F': { # 🛥️
        'en' : ':motor_boat:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':motorboot:',
        'es': ':barco_a_motor:',
        'fr': u':bateau_à_moteur:',
        'pt': ':barco:',
        'it': ':barca_a_motore:'
    },
    u'\U0001F6E5': { # 🛥
        'en' : ':motor_boat:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':motorboot:',
        'es': ':barco_a_motor:',
        'fr': u':bateau_à_moteur:',
        'pt': ':barco:',
        'it': ':barca_a_motore:'
    },
    u'\U0001F6F5': { # 🛵
        'en' : ':motor_scooter:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':motorroller:',
        'es': ':scooter:',
        'fr': ':scooter:',
        'pt': ':scooter:',
        'it': ':scooter:'
    },
    u'\U0001F3CD\U0000FE0F': { # 🏍️
        'en' : ':motorcycle:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':racing_motorcycle:'],
        'variant': True,
        'de': ':motorrad:',
        'es': ':moto:',
        'fr': ':moto:',
        'pt': ':motocicleta:',
        'it': ':motocicletta:'
    },
    u'\U0001F3CD': { # 🏍
        'en' : ':motorcycle:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':racing_motorcycle:'],
        'variant': True,
        'de': ':motorrad:',
        'es': ':moto:',
        'fr': ':moto:',
        'pt': ':motocicleta:',
        'it': ':motocicletta:'
    },
    u'\U0001F9BC': { # 🦼
        'en' : ':motorized_wheelchair:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':elektrischer_rollstuhl:',
        'es': u':silla_de_ruedas_eléctrica:',
        'fr': u':fauteuil_motorisé:',
        'pt': ':cadeira_de_rodas_motorizada:',
        'it': ':sedia_a_rotelle_motorizzata:'
    },
    u'\U0001F6E3\U0000FE0F': { # 🛣️
        'en' : ':motorway:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':autobahn:',
        'es': ':autopista:',
        'fr': ':autoroute:',
        'pt': ':estrada:',
        'it': ':autostrada:'
    },
    u'\U0001F6E3': { # 🛣
        'en' : ':motorway:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':autobahn:',
        'es': ':autopista:',
        'fr': ':autoroute:',
        'pt': ':estrada:',
        'it': ':autostrada:'
    },
    u'\U0001F5FB': { # 🗻
        'en' : ':mount_fuji:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':fuji:',
        'es': ':monte_fuji:',
        'fr': ':mont_fuji:',
        'pt': ':monte_fuji:',
        'it': ':monte_fuji:'
    },
    u'\U000026F0\U0000FE0F': { # ⛰️
        'en' : ':mountain:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':berg:',
        'es': u':montaña:',
        'fr': ':montagne:',
        'pt': ':montanha:',
        'it': ':montagna:'
    },
    u'\U000026F0': { # ⛰
        'en' : ':mountain:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':berg:',
        'es': u':montaña:',
        'fr': ':montagne:',
        'pt': ':montanha:',
        'it': ':montagna:'
    },
    u'\U0001F6A0': { # 🚠
        'en' : ':mountain_cableway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bergschwebebahn:',
        'es': u':teleférico_de_montaña:',
        'fr': u':téléphérique:',
        'pt': u':cabo_suspenso_usado_em_teleféricos_nas_montanhas:',
        'it': ':funivia:'
    },
    u'\U0001F69E': { # 🚞
        'en' : ':mountain_railway:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':bergbahn:',
        'es': u':ferrocarril_de_montaña:',
        'fr': ':train_de_montagne:',
        'pt': ':estrada_de_ferro_na_montanha:',
        'it': ':ferrovia_di_montagna:'
    },
    u'\U0001F401': { # 🐁
        'en' : ':mouse:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':mouse2:'],
        'de': ':maus:',
        'es': u':ratón:',
        'fr': ':souris:',
        'pt': ':camundongo:',
        'it': ':topo:'
    },
    u'\U0001F42D': { # 🐭
        'en' : ':mouse_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':mouse:'],
        'de': u':mäusegesicht:',
        'es': u':cara_de_ratón:',
        'fr': u':tête_de_souris:',
        'pt': ':rosto_de_camundongo:',
        'it': ':muso_di_topo:'
    },
    u'\U0001FAA4': { # 🪤
        'en' : ':mouse_trap:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':mausefalle:',
        'es': ':trampa_de_ratones:',
        'fr': u':tapette_à_souris:',
        'pt': ':ratoeira:',
        'it': ':trappola_per_topi:'
    },
    u'\U0001F444': { # 👄
        'en' : ':mouth:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':lips:'],
        'de': ':mund:',
        'es': ':boca:',
        'fr': ':bouche:',
        'pt': ':boca:',
        'it': ':bocca:'
    },
    u'\U0001F3A5': { # 🎥
        'en' : ':movie_camera:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':filmkamera:',
        'es': u':cámara_de_cine:',
        'fr': u':caméra:',
        'pt': u':câmera_de_cinema:',
        'it': ':cinepresa:'
    },
    u'\U00002716\U0000FE0F': { # ✖️
        'en' : ':multiply:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':heavy_multiplication_x:'],
        'variant': True,
        'de': ':multiplikationszeichen:',
        'es': u':multiplicación:',
        'fr': ':signe_multiplier:',
        'pt': u':sinal_de_multiplicação:',
        'it': ':segno_moltiplicazione:'
    },
    u'\U00002716': { # ✖
        'en' : ':multiply:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':heavy_multiplication_x:'],
        'variant': True,
        'de': ':mal:',
        'es': u':multiplicación:',
        'fr': ':signe_multiplier:',
        'pt': u':sinal_de_multiplicação:',
        'it': ':moltiplicazione:'
    },
    u'\U0001F344': { # 🍄
        'en' : ':mushroom:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':fliegenpilz:',
        'es': u':champiñón:',
        'fr': ':champignon:',
        'pt': ':cogumelo:',
        'it': ':fungo:'
    },
    u'\U0001F3B9': { # 🎹
        'en' : ':musical_keyboard:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':klaviatur:',
        'es': ':teclado_musical:',
        'fr': ':piano:',
        'pt': ':teclado_musical:',
        'it': ':piano:'
    },
    u'\U0001F3B5': { # 🎵
        'en' : ':musical_note:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':musiknote:',
        'es': ':nota_musical:',
        'fr': ':note_de_musique:',
        'pt': ':nota_musical:',
        'it': ':nota_musicale:'
    },
    u'\U0001F3B6': { # 🎶
        'en' : ':musical_notes:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':notes:'],
        'de': ':musiknoten:',
        'es': ':notas_musicales:',
        'fr': ':notes_de_musique:',
        'pt': ':notas_musicais:',
        'it': ':note_musicali:'
    },
    u'\U0001F3BC': { # 🎼
        'en' : ':musical_score:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':notenschlüssel:',
        'es': ':pentagrama:',
        'fr': ':partition:',
        'pt': ':partitura_musical:',
        'it': ':pentagramma:'
    },
    u'\U0001F507': { # 🔇
        'en' : ':muted_speaker:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':mute:'],
        'de': ':durchgestrichener_lautsprecher:',
        'es': ':altavoz_silenciado:',
        'fr': ':muet:',
        'pt': ':alto-falante_silenciado:',
        'it': ':altoparlante_disattivato:'
    },
    u'\U0001F9D1\U0000200D\U0001F384': { # 🧑‍🎄
        'en' : ':mx_claus:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson:',
        'es': ':noel:',
        'fr': ':santa:',
        'pt': ':noel:',
        'it': ':santa_claus:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F384': { # 🧑🏿‍🎄
        'en' : ':mx_claus_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson_dunkle_hautfarbe:',
        'es': ':noel_tono_de_piel_oscuro:',
        'fr': u':santa_peau_foncée:',
        'pt': ':noel_pele_escura:',
        'it': ':santa_claus_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F384': { # 🧑🏻‍🎄
        'en' : ':mx_claus_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson_helle_hautfarbe:',
        'es': ':noel_tono_de_piel_claro:',
        'fr': ':santa_peau_claire:',
        'pt': ':noel_pele_clara:',
        'it': ':santa_claus_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F384': { # 🧑🏾‍🎄
        'en' : ':mx_claus_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson_mitteldunkle_hautfarbe:',
        'es': ':noel_tono_de_piel_oscuro_medio:',
        'fr': ':santa_peau_mate:',
        'pt': ':noel_pele_morena_escura:',
        'it': ':santa_claus_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F384': { # 🧑🏼‍🎄
        'en' : ':mx_claus_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson_mittelhelle_hautfarbe:',
        'es': ':noel_tono_de_piel_claro_medio:',
        'fr': ':santa_peau_moyennement_claire:',
        'pt': ':noel_pele_morena_clara:',
        'it': ':santa_claus_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F384': { # 🧑🏽‍🎄
        'en' : ':mx_claus_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':weihnachtsperson_mittlere_hautfarbe:',
        'es': ':noel_tono_de_piel_medio:',
        'fr': u':santa_peau_légèrement_mate:',
        'pt': ':noel_pele_morena:',
        'it': ':santa_claus_carnagione_olivastra:'
    },
    u'\U0001F485': { # 💅
        'en' : ':nail_polish:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':nail_care:'],
        'de': ':nagellack:',
        'es': u':pintarse_las_uñas:',
        'fr': u':vernis_à_ongles:',
        'pt': ':esmalte_de_unha:',
        'it': ':smalto_per_unghie:'
    },
    u'\U0001F485\U0001F3FF': { # 💅🏿
        'en' : ':nail_polish_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nagellack_dunkle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_oscuro:',
        'fr': u':vernis_à_ongles_peau_foncée:',
        'pt': ':esmalte_de_unha_pele_escura:',
        'it': ':smalto_per_unghie_carnagione_scura:'
    },
    u'\U0001F485\U0001F3FB': { # 💅🏻
        'en' : ':nail_polish_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nagellack_helle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_claro:',
        'fr': u':vernis_à_ongles_peau_claire:',
        'pt': ':esmalte_de_unha_pele_clara:',
        'it': ':smalto_per_unghie_carnagione_chiara:'
    },
    u'\U0001F485\U0001F3FE': { # 💅🏾
        'en' : ':nail_polish_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nagellack_mitteldunkle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_oscuro_medio:',
        'fr': u':vernis_à_ongles_peau_mate:',
        'pt': ':esmalte_de_unha_pele_morena_escura:',
        'it': ':smalto_per_unghie_carnagione_abbastanza_scura:'
    },
    u'\U0001F485\U0001F3FC': { # 💅🏼
        'en' : ':nail_polish_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nagellack_mittelhelle_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_claro_medio:',
        'fr': u':vernis_à_ongles_peau_moyennement_claire:',
        'pt': ':esmalte_de_unha_pele_morena_clara:',
        'it': ':smalto_per_unghie_carnagione_abbastanza_chiara:'
    },
    u'\U0001F485\U0001F3FD': { # 💅🏽
        'en' : ':nail_polish_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nagellack_mittlere_hautfarbe:',
        'es': u':pintarse_las_uñas_tono_de_piel_medio:',
        'fr': u':vernis_à_ongles_peau_légèrement_mate:',
        'pt': ':esmalte_de_unha_pele_morena:',
        'it': ':smalto_per_unghie_carnagione_olivastra:'
    },
    u'\U0001F4DB': { # 📛
        'en' : ':name_badge:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':namensschild:',
        'es': ':etiqueta_identificativa:',
        'fr': ':badge_nominatif:',
        'pt': u':crachá:',
        'it': ':tesserino_per_nome:'
    },
    u'\U0001F3DE\U0000FE0F': { # 🏞️
        'en' : ':national_park:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':nationalpark:',
        'es': ':parque_nacional:',
        'fr': ':parc_national:',
        'pt': ':parque_nacional:',
        'it': ':parco_nazionale:'
    },
    u'\U0001F3DE': { # 🏞
        'en' : ':national_park:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':nationalpark:',
        'es': ':parque_nacional:',
        'fr': ':parc_national:',
        'pt': ':parque_nacional:',
        'it': ':parco_nazionale:'
    },
    u'\U0001F922': { # 🤢
        'en' : ':nauseated_face:',
        'status' : fully_qualified,
        'E' : 3,
        'de': u':würgendes_gesicht:',
        'es': u':cara_de_náuseas:',
        'fr': u':visage_nauséeux:',
        'pt': ':rosto_nauseado:',
        'it': ':faccina_nauseata:'
    },
    u'\U0001F9FF': { # 🧿
        'en' : ':nazar_amulet:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':nazar-amulett:',
        'es': ':ojo_turco:',
        'fr': u':mauvais_œil:',
        'pt': ':olho_grego:',
        'it': ':occhio_di_allah:'
    },
    u'\U0001F454': { # 👔
        'en' : ':necktie:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':hemd_mit_krawatte:',
        'es': ':corbata:',
        'fr': ':cravate:',
        'pt': ':gravata:',
        'it': ':cravatta:'
    },
    u'\U0001F913': { # 🤓
        'en' : ':nerd_face:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':strebergesicht:',
        'es': u':cara_de_empollón:',
        'fr': ':visage_de_geek:',
        'pt': ':rosto_de_nerd:',
        'it': ':faccina_nerd:'
    },
    u'\U0001FABA': { # 🪺
        'en' : ':nest_with_eggs:',
        'status' : fully_qualified,
        'E' : 14,
        'de': ':nest_mit_eiern:',
        'es': ':nido_con_huevos:',
        'fr': u':nid_avec_œufs:',
        'pt': ':ninho_com_ovos:',
        'it': ':nido_con_uova:'
    },
    u'\U0001FA86': { # 🪆
        'en' : ':nesting_dolls:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':matroschka:',
        'es': u':muñeca_rusa:',
        'fr': u':poupées_russes:',
        'pt': ':boneca_russa:',
        'it': ':matrioska:'
    },
    u'\U0001F610': { # 😐
        'en' : ':neutral_face:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':neutrales_gesicht:',
        'es': ':cara_neutral:',
        'fr': ':visage_neutre:',
        'pt': ':rosto_neutro:',
        'it': ':faccina_neutra:'
    },
    u'\U0001F311': { # 🌑
        'en' : ':new_moon:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':neumond:',
        'es': ':luna_nueva:',
        'fr': ':nouvelle_lune:',
        'pt': ':lua_nova:',
        'it': ':luna_nuova:'
    },
    u'\U0001F31A': { # 🌚
        'en' : ':new_moon_face:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':new_moon_with_face:'],
        'de': ':neumond_mit_gesicht:',
        'es': ':luna_nueva_con_cara:',
        'fr': ':nouvelle_lune_avec_visage:',
        'pt': ':rosto_da_lua_nova:',
        'it': ':faccina_luna_nuova:'
    },
    u'\U0001F4F0': { # 📰
        'en' : ':newspaper:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':zeitung:',
        'es': u':periódico:',
        'fr': ':journal:',
        'pt': ':jornal:',
        'it': ':quotidiano:'
    },
    u'\U000023ED\U0000FE0F': { # ⏭️
        'en' : ':next_track_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':black_right__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':nächster_titel:',
        'es': ':pista_siguiente:',
        'fr': ':bouton_piste_suivante:',
        'pt': u':botão_de_próxima_faixa:',
        'it': ':traccia_successiva:'
    },
    u'\U000023ED': { # ⏭
        'en' : ':next_track_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':black_right__pointing_double_triangle_with_vertical_bar:'],
        'variant': True,
        'de': u':nächster_titel:',
        'es': ':pista_siguiente:',
        'fr': ':bouton_piste_suivante:',
        'pt': u':botão_de_próxima_faixa:',
        'it': ':traccia_successiva:'
    },
    u'\U0001F303': { # 🌃
        'en' : ':night_with_stars:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':sternenhimmel:',
        'es': ':noche_estrellada:',
        'fr': u':nuit_étoilée:',
        'pt': ':noite_estrelada:',
        'it': ':notte_stellata:'
    },
    u'\U0001F564': { # 🕤
        'en' : ':nine-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock930:'],
        'variant': True,
        'de': ':9.30_uhr:',
        'es': ':nueve_y_media:',
        'fr': ':neuf_heures_et_demie:',
        'pt': ':nove_e_meia:',
        'it': ':ore_nove_e_mezza:'
    },
    u'\U0001F558': { # 🕘
        'en' : u':nine_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock9:'],
        'variant': True,
        'de': ':9.00_uhr:',
        'es': ':9_en_punto:',
        'fr': ':neuf_heures:',
        'pt': ':9_horas:',
        'it': ':ore_nove:'
    },
    u'\U0001F977': { # 🥷
        'en' : ':ninja:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja:',
        'es': ':ninja:',
        'fr': ':ninja:',
        'pt': ':ninja:',
        'it': ':ninja:'
    },
    u'\U0001F977\U0001F3FF': { # 🥷🏿
        'en' : ':ninja_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja_dunkle_hautfarbe:',
        'es': ':ninja_tono_de_piel_oscuro:',
        'fr': u':ninja_peau_foncée:',
        'pt': ':ninja_pele_escura:',
        'it': ':ninja_carnagione_scura:'
    },
    u'\U0001F977\U0001F3FB': { # 🥷🏻
        'en' : ':ninja_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja_helle_hautfarbe:',
        'es': ':ninja_tono_de_piel_claro:',
        'fr': ':ninja_peau_claire:',
        'pt': ':ninja_pele_clara:',
        'it': ':ninja_carnagione_chiara:'
    },
    u'\U0001F977\U0001F3FE': { # 🥷🏾
        'en' : ':ninja_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja_mitteldunkle_hautfarbe:',
        'es': ':ninja_tono_de_piel_oscuro_medio:',
        'fr': ':ninja_peau_mate:',
        'pt': ':ninja_pele_morena_escura:',
        'it': ':ninja_carnagione_abbastanza_scura:'
    },
    u'\U0001F977\U0001F3FC': { # 🥷🏼
        'en' : ':ninja_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja_mittelhelle_hautfarbe:',
        'es': ':ninja_tono_de_piel_claro_medio:',
        'fr': ':ninja_peau_moyennement_claire:',
        'pt': ':ninja_pele_morena_clara:',
        'it': ':ninja_carnagione_abbastanza_chiara:'
    },
    u'\U0001F977\U0001F3FD': { # 🥷🏽
        'en' : ':ninja_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':ninja_mittlere_hautfarbe:',
        'es': ':ninja_tono_de_piel_medio:',
        'fr': u':ninja_peau_légèrement_mate:',
        'pt': ':ninja_pele_morena:',
        'it': ':ninja_carnagione_olivastra:'
    },
    u'\U0001F6B3': { # 🚳
        'en' : ':no_bicycles:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':fahrräder_verboten:',
        'es': ':bicicletas_prohibidas:',
        'fr': u':vélos_interdits:',
        'pt': ':proibido_andar_de_bicicleta:',
        'it': ':segnale_di_divieto_di_transito_delle_biciclette:'
    },
    u'\U000026D4': { # ⛔
        'en' : ':no_entry:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':zutritt_verboten:',
        'es': u':dirección_prohibida:',
        'fr': ':sens_interdit:',
        'pt': ':entrada_proibida:',
        'it': ':segnale_di_divieto_di_accesso:'
    },
    u'\U0001F6AF': { # 🚯
        'en' : ':no_littering:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':do_not_litter:'],
        'de': ':abfall_verboten:',
        'es': ':prohibido_tirar_basura:',
        'fr': u':dépôt_d’ordures_interdit:',
        'pt': u':proibido_jogar_lixo_no_chão:',
        'it': ':simbolo_del_divieto_di_gettare_rifiuti:'
    },
    u'\U0001F4F5': { # 📵
        'en' : ':no_mobile_phones:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':mobiltelefone_verboten:',
        'es': u':prohibido_el_uso_de_móviles:',
        'fr': u':téléphones_portables_interdits:',
        'pt': ':proibido_o_uso_de_telefone_celular:',
        'it': u':simbolo_che_vieta_l’uso_dei_cellulari:'
    },
    u'\U0001F51E': { # 🔞
        'en' : ':no_one_under_eighteen:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':underage:'],
        'de': u':minderjährige_verboten:',
        'es': u':prohibido_para_menos_de_18_años:',
        'fr': ':18_ans_et_plus:',
        'pt': ':proibido_para_menores_de_18_anos:',
        'it': ':simbolo_di_divieto_ai_minorenni:'
    },
    u'\U0001F6B7': { # 🚷
        'en' : ':no_pedestrians:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':fußgänger_verboten:',
        'es': ':prohibido_el_paso_de_peatones:',
        'fr': u':interdit_aux_piétons:',
        'pt': ':proibida_a_passagem_de_pedestres:',
        'it': ':simbolo_del_divieto_di_transito_pedoni:'
    },
    u'\U0001F6AD': { # 🚭
        'en' : ':no_smoking:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':rauchverbot:',
        'es': ':prohibido_fumar:',
        'fr': ':interdiction_de_fumer:',
        'pt': ':proibido_fumar:',
        'it': ':segnale_di_vietato_fumare:'
    },
    u'\U0001F6B1': { # 🚱
        'en' : ':non-potable_water:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':non__potable_water:'],
        'de': ':kein_trinkwasser:',
        'es': ':agua_no_potable:',
        'fr': ':eau_non_potable:',
        'pt': u':água_não_potável:',
        'it': ':simbolo_di_acqua_non_potabile:'
    },
    u'\U0001F443': { # 👃
        'en' : ':nose:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':nase:',
        'es': ':nariz:',
        'fr': ':nez:',
        'pt': ':nariz:',
        'it': ':naso:'
    },
    u'\U0001F443\U0001F3FF': { # 👃🏿
        'en' : ':nose_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nase_dunkle_hautfarbe:',
        'es': ':nariz_tono_de_piel_oscuro:',
        'fr': u':nez_peau_foncée:',
        'pt': ':nariz_pele_escura:',
        'it': ':naso_carnagione_scura:'
    },
    u'\U0001F443\U0001F3FB': { # 👃🏻
        'en' : ':nose_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nase_helle_hautfarbe:',
        'es': ':nariz_tono_de_piel_claro:',
        'fr': ':nez_peau_claire:',
        'pt': ':nariz_pele_clara:',
        'it': ':naso_carnagione_chiara:'
    },
    u'\U0001F443\U0001F3FE': { # 👃🏾
        'en' : ':nose_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nase_mitteldunkle_hautfarbe:',
        'es': ':nariz_tono_de_piel_oscuro_medio:',
        'fr': ':nez_peau_mate:',
        'pt': ':nariz_pele_morena_escura:',
        'it': ':naso_carnagione_abbastanza_scura:'
    },
    u'\U0001F443\U0001F3FC': { # 👃🏼
        'en' : ':nose_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nase_mittelhelle_hautfarbe:',
        'es': ':nariz_tono_de_piel_claro_medio:',
        'fr': ':nez_peau_moyennement_claire:',
        'pt': ':nariz_pele_morena_clara:',
        'it': ':naso_carnagione_abbastanza_chiara:'
    },
    u'\U0001F443\U0001F3FD': { # 👃🏽
        'en' : ':nose_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':nase_mittlere_hautfarbe:',
        'es': ':nariz_tono_de_piel_medio:',
        'fr': u':nez_peau_légèrement_mate:',
        'pt': ':nariz_pele_morena:',
        'it': ':naso_carnagione_olivastra:'
    },
    u'\U0001F4D3': { # 📓
        'en' : ':notebook:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':notizbuch:',
        'es': ':cuaderno:',
        'fr': ':carnet:',
        'pt': ':caderno:',
        'it': ':quaderno:'
    },
    u'\U0001F4D4': { # 📔
        'en' : ':notebook_with_decorative_cover:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':notizbuch_mit_dekorativem_einband:',
        'es': ':cuaderno_con_tapa_decorativa:',
        'fr': ':carnet_avec_couverture:',
        'pt': ':caderno_decorado:',
        'it': ':quaderno_con_copertina_decorata:'
    },
    u'\U0001F529': { # 🔩
        'en' : ':nut_and_bolt:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':mutter_und_schraube:',
        'es': ':tornillo_y_tuerca:',
        'fr': u':vis_et_écrou:',
        'pt': ':porca_e_parafuso:',
        'it': ':vite_e_bullone:'
    },
    u'\U0001F419': { # 🐙
        'en' : ':octopus:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':oktopus:',
        'es': ':pulpo:',
        'fr': ':pieuvre:',
        'pt': ':polvo:',
        'it': ':polpo:'
    },
    u'\U0001F362': { # 🍢
        'en' : ':oden:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':oden:',
        'es': ':brocheta:',
        'fr': ':brochette_de_poisson:',
        'pt': ':oden:',
        'it': ':oden_giapponese:'
    },
    u'\U0001F3E2': { # 🏢
        'en' : ':office_building:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':office:'],
        'de': u':bürogebäude:',
        'es': ':edificio_de_oficinas:',
        'fr': ':immeuble_de_bureaux:',
        'pt': u':edifício_comercial:',
        'it': ':edificio:'
    },
    u'\U0001F9D1\U0000200D\U0001F4BC': { # 🧑‍💼
        'en' : ':office_worker:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r):',
        'es': ':oficinista:',
        'fr': u':employé_de_bureau_(tous_genres):',
        'pt': u':trabalhador_de_escritório:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F4BC': { # 🧑🏿‍💼
        'en' : ':office_worker_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_dunkle_hautfarbe:',
        'es': ':oficinista_tono_de_piel_oscuro:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_foncée:',
        'pt': u':trabalhador_de_escritório_pele_escura:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F4BC': { # 🧑🏻‍💼
        'en' : ':office_worker_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_helle_hautfarbe:',
        'es': ':oficinista_tono_de_piel_claro:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_claire:',
        'pt': u':trabalhador_de_escritório_pele_clara:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F4BC': { # 🧑🏾‍💼
        'en' : ':office_worker_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mitteldunkle_hautfarbe:',
        'es': ':oficinista_tono_de_piel_oscuro_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_mate:',
        'pt': u':trabalhador_de_escritório_pele_morena_escura:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F4BC': { # 🧑🏼‍💼
        'en' : ':office_worker_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mittelhelle_hautfarbe:',
        'es': ':oficinista_tono_de_piel_claro_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_moyennement_claire:',
        'pt': u':trabalhador_de_escritório_pele_morena_clara:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F4BC': { # 🧑🏽‍💼
        'en' : ':office_worker_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':büroangestellte(r)_mittlere_hautfarbe:',
        'es': ':oficinista_tono_de_piel_medio:',
        'fr': u':employé_de_bureau_(tous_genres)_peau_légèrement_mate:',
        'pt': u':trabalhador_de_escritório_pele_morena:',
        'it': u':persona_che_fa_un_lavoro_d’ufficio_carnagione_olivastra:'
    },
    u'\U0001F479': { # 👹
        'en' : ':ogre:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':japanese_ogre:'],
        'de': ':ungeheuer:',
        'es': u':demonio_japonés_oni:',
        'fr': ':ogre:',
        'pt': ':ogro:',
        'it': ':orco:'
    },
    u'\U0001F6E2\U0000FE0F': { # 🛢️
        'en' : ':oil_drum:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ölfass:',
        'es': u':barril_de_petróleo:',
        'fr': u':baril_de_pétrole:',
        'pt': u':barril_de_óleo:',
        'it': ':barile_di_petrolio:'
    },
    u'\U0001F6E2': { # 🛢
        'en' : ':oil_drum:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':ölfass:',
        'es': u':barril_de_petróleo:',
        'fr': u':baril_de_pétrole:',
        'pt': u':barril_de_óleo:',
        'it': ':barile_di_petrolio:'
    },
    u'\U0001F5DD\U0000FE0F': { # 🗝️
        'en' : ':old_key:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': u':alter_schlüssel:',
        'es': ':llave_antigua:',
        'fr': u':clé_ancienne:',
        'pt': ':chave_antiga:',
        'it': ':chiave_antica:'
    },
    u'\U0001F5DD': { # 🗝
        'en' : ':old_key:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': u':alter_schlüssel:',
        'es': ':llave_antigua:',
        'fr': u':clé_ancienne:',
        'pt': ':chave_antiga:',
        'it': ':chiave_antica:'
    },
    u'\U0001F474': { # 👴
        'en' : ':old_man:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':older_man:'],
        'de': u':älterer_mann:',
        'es': ':anciano:',
        'fr': u':homme_âgé:',
        'pt': ':homem_idoso:',
        'it': ':uomo_anziano:'
    },
    u'\U0001F474\U0001F3FF': { # 👴🏿
        'en' : ':old_man_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_dunkle_hautfarbe:',
        'es': ':anciano_tono_de_piel_oscuro:',
        'fr': u':homme_âgé_peau_foncée:',
        'pt': ':homem_idoso_pele_escura:',
        'it': ':uomo_anziano_carnagione_scura:'
    },
    u'\U0001F474\U0001F3FB': { # 👴🏻
        'en' : ':old_man_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_helle_hautfarbe:',
        'es': ':anciano_tono_de_piel_claro:',
        'fr': u':homme_âgé_peau_claire:',
        'pt': ':homem_idoso_pele_clara:',
        'it': ':uomo_anziano_carnagione_chiara:'
    },
    u'\U0001F474\U0001F3FE': { # 👴🏾
        'en' : ':old_man_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mitteldunkle_hautfarbe:',
        'es': ':anciano_tono_de_piel_oscuro_medio:',
        'fr': u':homme_âgé_peau_mate:',
        'pt': ':homem_idoso_pele_morena_escura:',
        'it': ':uomo_anziano_carnagione_abbastanza_scura:'
    },
    u'\U0001F474\U0001F3FC': { # 👴🏼
        'en' : ':old_man_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mittelhelle_hautfarbe:',
        'es': ':anciano_tono_de_piel_claro_medio:',
        'fr': u':homme_âgé_peau_moyennement_claire:',
        'pt': ':homem_idoso_pele_morena_clara:',
        'it': ':uomo_anziano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F474\U0001F3FD': { # 👴🏽
        'en' : ':old_man_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':älterer_mann_mittlere_hautfarbe:',
        'es': ':anciano_tono_de_piel_medio:',
        'fr': u':homme_âgé_peau_légèrement_mate:',
        'pt': ':homem_idoso_pele_morena:',
        'it': ':uomo_anziano_carnagione_olivastra:'
    },
    u'\U0001F475': { # 👵
        'en' : ':old_woman:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':older_woman:'],
        'de': u':ältere_frau:',
        'es': ':anciana:',
        'fr': u':femme_âgée:',
        'pt': ':idosa:',
        'it': ':donna_anziana:'
    },
    u'\U0001F475\U0001F3FF': { # 👵🏿
        'en' : ':old_woman_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_dunkle_hautfarbe:',
        'es': ':anciana_tono_de_piel_oscuro:',
        'fr': u':femme_âgée_peau_foncée:',
        'pt': ':idosa_pele_escura:',
        'it': ':donna_anziana_carnagione_scura:'
    },
    u'\U0001F475\U0001F3FB': { # 👵🏻
        'en' : ':old_woman_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_helle_hautfarbe:',
        'es': ':anciana_tono_de_piel_claro:',
        'fr': u':femme_âgée_peau_claire:',
        'pt': ':idosa_pele_clara:',
        'it': ':donna_anziana_carnagione_chiara:'
    },
    u'\U0001F475\U0001F3FE': { # 👵🏾
        'en' : ':old_woman_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mitteldunkle_hautfarbe:',
        'es': ':anciana_tono_de_piel_oscuro_medio:',
        'fr': u':femme_âgée_peau_mate:',
        'pt': ':idosa_pele_morena_escura:',
        'it': ':donna_anziana_carnagione_abbastanza_scura:'
    },
    u'\U0001F475\U0001F3FC': { # 👵🏼
        'en' : ':old_woman_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mittelhelle_hautfarbe:',
        'es': ':anciana_tono_de_piel_claro_medio:',
        'fr': u':femme_âgée_peau_moyennement_claire:',
        'pt': ':idosa_pele_morena_clara:',
        'it': ':donna_anziana_carnagione_abbastanza_chiara:'
    },
    u'\U0001F475\U0001F3FD': { # 👵🏽
        'en' : ':old_woman_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':ältere_frau_mittlere_hautfarbe:',
        'es': ':anciana_tono_de_piel_medio:',
        'fr': u':femme_âgée_peau_légèrement_mate:',
        'pt': ':idosa_pele_morena:',
        'it': ':donna_anziana_carnagione_olivastra:'
    },
    u'\U0001F9D3': { # 🧓
        'en' : ':older_person:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':older_adult:'],
        'de': u':ältere_person:',
        'es': ':persona_mayor:',
        'fr': u':personne_âgée:',
        'pt': ':idoso:',
        'it': ':persona_anziana:'
    },
    u'\U0001F9D3\U0001F3FF': { # 🧓🏿
        'en' : ':older_person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_dunkle_hautfarbe:',
        'es': ':persona_adulta_madura_tono_de_piel_oscuro:',
        'fr': u':personne_âgée_peau_foncée:',
        'pt': ':idoso_pele_escura:',
        'it': ':persona_anziana_carnagione_scura:'
    },
    u'\U0001F9D3\U0001F3FB': { # 🧓🏻
        'en' : ':older_person_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_helle_hautfarbe:',
        'es': ':persona_adulta_madura_tono_de_piel_claro:',
        'fr': u':personne_âgée_peau_claire:',
        'pt': ':idoso_pele_clara:',
        'it': ':persona_anziana_carnagione_chiara:'
    },
    u'\U0001F9D3\U0001F3FE': { # 🧓🏾
        'en' : ':older_person_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mitteldunkle_hautfarbe:',
        'es': ':persona_adulta_madura_tono_de_piel_oscuro_medio:',
        'fr': u':personne_âgée_peau_mate:',
        'pt': ':idoso_pele_morena_escura:',
        'it': ':persona_anziana_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D3\U0001F3FC': { # 🧓🏼
        'en' : ':older_person_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mittelhelle_hautfarbe:',
        'es': ':persona_adulta_madura_tono_de_piel_claro_medio:',
        'fr': u':personne_âgée_peau_moyennement_claire:',
        'pt': ':idoso_pele_morena_clara:',
        'it': ':persona_anziana_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D3\U0001F3FD': { # 🧓🏽
        'en' : ':older_person_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':älterer_erwachsener_mittlere_hautfarbe:',
        'es': ':persona_adulta_madura_tono_de_piel_medio:',
        'fr': u':personne_âgée_peau_légèrement_mate:',
        'pt': ':idoso_pele_morena:',
        'it': ':persona_anziana_carnagione_olivastra:'
    },
    u'\U0001FAD2': { # 🫒
        'en' : ':olive:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':olive:',
        'es': ':aceituna:',
        'fr': ':olive:',
        'pt': ':azeitona:',
        'it': ':oliva:'
    },
    u'\U0001F549\U0000FE0F': { # 🕉️
        'en' : ':om:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':om_symbol:'],
        'variant': True,
        'de': ':om:',
        'es': ':om:',
        'fr': ':om:',
        'pt': ':om:',
        'it': ':om:'
    },
    u'\U0001F549': { # 🕉
        'en' : ':om:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':om_symbol:'],
        'variant': True,
        'de': ':om:',
        'es': ':om:',
        'fr': ':om:',
        'pt': ':om:',
        'it': ':om:'
    },
    u'\U0001F698': { # 🚘
        'en' : ':oncoming_automobile:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':auto_von_vorne:',
        'es': u':coche_próximo:',
        'fr': ':voiture_de_face:',
        'pt': ':carro_se_aproximando:',
        'it': ':auto_in_arrivo:'
    },
    u'\U0001F68D': { # 🚍
        'en' : ':oncoming_bus:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':bus_von_vorne:',
        'es': u':autobús_próximo:',
        'fr': ':bus_de_face:',
        'pt': u':ônibus_se_aproximando:',
        'it': ':bus_in_arrivo:'
    },
    u'\U0001F44A': { # 👊
        'en' : ':oncoming_fist:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':fist_oncoming:', ':punch:', ':facepunch:'],
        'de': ':geballte_faust:',
        'es': u':puño_cerrado:',
        'fr': ':poing_de_face:',
        'pt': ':soco:',
        'it': ':pugno_chiuso:'
    },
    u'\U0001F44A\U0001F3FF': { # 👊🏿
        'en' : ':oncoming_fist_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':geballte_faust_dunkle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_oscuro:',
        'fr': u':poing_de_face_peau_foncée:',
        'pt': ':soco_pele_escura:',
        'it': ':pugno_chiuso_carnagione_scura:'
    },
    u'\U0001F44A\U0001F3FB': { # 👊🏻
        'en' : ':oncoming_fist_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':geballte_faust_helle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_claro:',
        'fr': ':poing_de_face_peau_claire:',
        'pt': ':soco_pele_clara:',
        'it': ':pugno_chiuso_carnagione_chiara:'
    },
    u'\U0001F44A\U0001F3FE': { # 👊🏾
        'en' : ':oncoming_fist_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':geballte_faust_mitteldunkle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_oscuro_medio:',
        'fr': ':poing_de_face_peau_mate:',
        'pt': ':soco_pele_morena_escura:',
        'it': ':pugno_chiuso_carnagione_abbastanza_scura:'
    },
    u'\U0001F44A\U0001F3FC': { # 👊🏼
        'en' : ':oncoming_fist_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':geballte_faust_mittelhelle_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_claro_medio:',
        'fr': ':poing_de_face_peau_moyennement_claire:',
        'pt': ':soco_pele_morena_clara:',
        'it': ':pugno_chiuso_carnagione_abbastanza_chiara:'
    },
    u'\U0001F44A\U0001F3FD': { # 👊🏽
        'en' : ':oncoming_fist_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':geballte_faust_mittlere_hautfarbe:',
        'es': u':puño_cerrado_tono_de_piel_medio:',
        'fr': u':poing_de_face_peau_légèrement_mate:',
        'pt': ':soco_pele_morena:',
        'it': ':pugno_chiuso_carnagione_olivastra:'
    },
    u'\U0001F694': { # 🚔
        'en' : ':oncoming_police_car:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':polizeiwagen_von_vorne:',
        'es': u':coche_de_policía_próximo:',
        'fr': ':voiture_de_police_de_face:',
        'pt': ':viatura_policial_se_aproximando:',
        'it': ':macchina_della_polizia_in_arrivo:'
    },
    u'\U0001F696': { # 🚖
        'en' : ':oncoming_taxi:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':taxi_von_vorne:',
        'es': u':taxi_próximo:',
        'fr': ':taxi_de_face:',
        'pt': u':táxi_se_aproximando:',
        'it': ':taxi_in_arrivo:'
    },
    u'\U0001FA71': { # 🩱
        'en' : ':one-piece_swimsuit:',
        'status' : fully_qualified,
        'E' : 12,
        'alias' : [':one_piece_swimsuit:'],
        'de': ':einteiliger_badeanzug:',
        'es': u':traje_de_baño_de_una_pieza:',
        'fr': u':maillot_de_bain_une_pièce:',
        'pt': u':maiô:',
        'it': ':costume_da_bagno_intero:'
    },
    u'\U0001F55C': { # 🕜
        'en' : ':one-thirty:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':clock130:'],
        'variant': True,
        'de': ':1.30_uhr:',
        'es': ':una_y_media:',
        'fr': ':une_heure_et_demie:',
        'pt': ':uma_e_meia:',
        'it': ':ore_una_e_mezza:'
    },
    u'\U0001F550': { # 🕐
        'en' : u':one_o’clock:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':clock1:'],
        'variant': True,
        'de': ':1.00_uhr:',
        'es': ':1_en_punto:',
        'fr': ':une_heure:',
        'pt': ':1_hora:',
        'it': ':ore_una:'
    },
    u'\U0001F9C5': { # 🧅
        'en' : ':onion:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':zwiebel:',
        'es': ':cebolla:',
        'fr': ':oignon:',
        'pt': ':cebola:',
        'it': ':cipolla:'
    },
    u'\U0001F4D6': { # 📖
        'en' : ':open_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':book:'],
        'de': ':offenes_buch:',
        'es': ':libro_abierto:',
        'fr': ':livre_ouvert:',
        'pt': ':livro_aberto:',
        'it': ':libro_aperto:'
    },
    u'\U0001F4C2': { # 📂
        'en' : ':open_file_folder:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':geöffneter_ordner:',
        'es': ':carpeta_de_archivos_abierta:',
        'fr': ':dossier_ouvert:',
        'pt': ':pasta_de_arquivos_aberta:',
        'it': ':cartella_aperta:'
    },
    u'\U0001F450': { # 👐
        'en' : ':open_hands:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':offene_hände:',
        'es': ':manos_abiertas:',
        'fr': ':mains_ouvertes:',
        'pt': u':mãos_abertas:',
        'it': ':mani_aperte:'
    },
    u'\U0001F450\U0001F3FF': { # 👐🏿
        'en' : ':open_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_dunkle_hautfarbe:',
        'es': ':manos_abiertas_tono_de_piel_oscuro:',
        'fr': u':mains_ouvertes_peau_foncée:',
        'pt': u':mãos_abertas_pele_escura:',
        'it': ':mani_aperte_carnagione_scura:'
    },
    u'\U0001F450\U0001F3FB': { # 👐🏻
        'en' : ':open_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_helle_hautfarbe:',
        'es': ':manos_abiertas_tono_de_piel_claro:',
        'fr': ':mains_ouvertes_peau_claire:',
        'pt': u':mãos_abertas_pele_clara:',
        'it': ':mani_aperte_carnagione_chiara:'
    },
    u'\U0001F450\U0001F3FE': { # 👐🏾
        'en' : ':open_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mitteldunkle_hautfarbe:',
        'es': ':manos_abiertas_tono_de_piel_oscuro_medio:',
        'fr': ':mains_ouvertes_peau_mate:',
        'pt': u':mãos_abertas_pele_morena_escura:',
        'it': ':mani_aperte_carnagione_abbastanza_scura:'
    },
    u'\U0001F450\U0001F3FC': { # 👐🏼
        'en' : ':open_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mittelhelle_hautfarbe:',
        'es': ':manos_abiertas_tono_de_piel_claro_medio:',
        'fr': ':mains_ouvertes_peau_moyennement_claire:',
        'pt': u':mãos_abertas_pele_morena_clara:',
        'it': ':mani_aperte_carnagione_abbastanza_chiara:'
    },
    u'\U0001F450\U0001F3FD': { # 👐🏽
        'en' : ':open_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':offene_hände_mittlere_hautfarbe:',
        'es': ':manos_abiertas_tono_de_piel_medio:',
        'fr': u':mains_ouvertes_peau_légèrement_mate:',
        'pt': u':mãos_abertas_pele_morena:',
        'it': ':mani_aperte_carnagione_olivastra:'
    },
    u'\U0001F4ED': { # 📭
        'en' : ':open_mailbox_with_lowered_flag:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':mailbox_with_no_mail:'],
        'variant': True,
        'de': ':offener_briefkasten_ohne_post:',
        'es': u':buzón_abierto_con_la_bandera_bajada:',
        'fr': u':boîte_aux_lettres_ouverte_drapeau_baissé:',
        'pt': ':caixa_de_correio_aberta_com_bandeira_abaixada:',
        'it': ':cassetta_postale_aperta_bandierina_abbassata:'
    },
    u'\U0001F4EC': { # 📬
        'en' : ':open_mailbox_with_raised_flag:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':mailbox_with_mail:'],
        'variant': True,
        'de': ':offener_briefkasten_mit_post:',
        'es': u':buzón_abierto_con_la_bandera_levantada:',
        'fr': u':boîte_aux_lettres_ouverte_drapeau_levé:',
        'pt': ':caixa_de_correio_aberta_com_bandeira_levantada:',
        'it': ':cassetta_postale_aperta_bandierina_alzata:'
    },
    u'\U0001F4BF': { # 💿
        'en' : ':optical_disk:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':cd:'],
        'variant': True,
        'de': ':cd:',
        'es': u':disco_óptico:',
        'fr': ':cd:',
        'pt': ':cd:',
        'it': ':disco_ottico:'
    },
    u'\U0001F4D9': { # 📙
        'en' : ':orange_book:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':orangefarbenes_buch:',
        'es': ':libro_naranja:',
        'fr': ':livre_orange:',
        'pt': ':livro_laranja:',
        'it': ':libro_arancione:'
    },
    u'\U0001F7E0': { # 🟠
        'en' : ':orange_circle:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':oranger_punkt:',
        'es': u':círculo_naranja:',
        'fr': ':disque_orange:',
        'pt': u':círculo_laranja:',
        'it': ':cerchio_arancione:'
    },
    u'\U0001F9E1': { # 🧡
        'en' : ':orange_heart:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':oranges_herz:',
        'es': u':corazón_naranja:',
        'fr': u':cœur_orange:',
        'pt': u':coração_laranja:',
        'it': ':cuore_arancione:'
    },
    u'\U0001F7E7': { # 🟧
        'en' : ':orange_square:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':oranges_quadrat:',
        'es': ':cuadrado_naranja:',
        'fr': u':carré_orange:',
        'pt': ':quadrado_laranja:',
        'it': ':quadrato_arancione:'
    },
    u'\U0001F9A7': { # 🦧
        'en' : ':orangutan:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':orang-utan:',
        'es': u':orangután:',
        'fr': ':orang-outan:',
        'pt': ':orangotango:',
        'it': ':orangotango:'
    },
    u'\U00002626\U0000FE0F': { # ☦️
        'en' : ':orthodox_cross:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':orthodoxes_kreuz:',
        'es': ':cruz_ortodoxa:',
        'fr': ':croix_orthodoxe:',
        'pt': ':cruz_ortodoxa:',
        'it': ':croce_ortodossa:'
    },
    u'\U00002626': { # ☦
        'en' : ':orthodox_cross:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':orthodoxes_kreuz:',
        'es': ':cruz_ortodoxa:',
        'fr': ':croix_orthodoxe:',
        'pt': ':cruz_ortodoxa:',
        'it': ':croce_ortodossa:'
    },
    u'\U0001F9A6': { # 🦦
        'en' : ':otter:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':otter:',
        'es': ':nutria:',
        'fr': ':loutre:',
        'pt': ':lontra:',
        'it': ':lontra:'
    },
    u'\U0001F4E4': { # 📤
        'en' : ':outbox_tray:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':postausgang:',
        'es': ':bandeja_de_salida:',
        'fr': u':boîte_d’envoi:',
        'pt': u':bandeja_de_saída:',
        'it': ':posta_inviata:'
    },
    u'\U0001F989': { # 🦉
        'en' : ':owl:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':eule:',
        'es': u':búho:',
        'fr': ':chouette:',
        'pt': ':coruja:',
        'it': ':gufo:'
    },
    u'\U0001F402': { # 🐂
        'en' : ':ox:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':ochse:',
        'es': ':buey:',
        'fr': u':bœuf:',
        'pt': ':boi:',
        'it': ':bue:'
    },
    u'\U0001F9AA': { # 🦪
        'en' : ':oyster:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':auster:',
        'es': ':ostra:',
        'fr': u':huître:',
        'pt': ':ostra:',
        'it': ':ostrica:'
    },
    u'\U0001F4E6': { # 📦
        'en' : ':package:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':paket:',
        'es': ':paquete:',
        'fr': ':colis:',
        'pt': ':pacote:',
        'it': ':pacco:'
    },
    u'\U0001F4C4': { # 📄
        'en' : ':page_facing_up:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':vorderseite_eines_blattes:',
        'es': u':página_hacia_arriba:',
        'fr': ':page:',
        'pt': u':página_voltada_para_cima:',
        'it': ':pagina:'
    },
    u'\U0001F4C3': { # 📃
        'en' : ':page_with_curl:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':teilweise_eingerolltes_blatt:',
        'es': u':página_doblada:',
        'fr': u':page_enroulée:',
        'pt': u':página_dobrada:',
        'it': ':pagina_con_piega:'
    },
    u'\U0001F4DF': { # 📟
        'en' : ':pager:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':pager:',
        'es': ':busca:',
        'fr': ':bipeur:',
        'pt': ':pager:',
        'it': ':cercapersone:'
    },
    u'\U0001F58C\U0000FE0F': { # 🖌️
        'en' : ':paintbrush:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':lower_left_paintbrush:'],
        'variant': True,
        'de': ':pinsel:',
        'es': ':pincel:',
        'fr': ':pinceau:',
        'pt': ':pincel:',
        'it': ':pennello:'
    },
    u'\U0001F58C': { # 🖌
        'en' : ':paintbrush:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':lower_left_paintbrush:'],
        'variant': True,
        'de': ':pinsel:',
        'es': ':pincel:',
        'fr': ':pinceau:',
        'pt': ':pincel:',
        'it': ':pennello:'
    },
    u'\U0001FAF3': { # 🫳
        'en' : ':palm_down_hand:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':hand_mit_handfläche_nach_unten:',
        'es': ':mano_con_la_palma_hacia_abajo:',
        'fr': ':main_paume_vers_le_bas:',
        'pt': u':mão_com_a_palma_para_baixo:',
        'it': ':mano_con_il_palmo_verso_il_basso:'
    },
    u'\U0001FAF3\U0001F3FF': { # 🫳🏿
        'en' : ':palm_down_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FB': { # 🫳🏻
        'en' : ':palm_down_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FE': { # 🫳🏾
        'en' : ':palm_down_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FC': { # 🫳🏼
        'en' : ':palm_down_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF3\U0001F3FD': { # 🫳🏽
        'en' : ':palm_down_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F334': { # 🌴
        'en' : ':palm_tree:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':palme:',
        'es': ':palmera:',
        'fr': ':palmier:',
        'pt': ':palmeira:',
        'it': ':palma:'
    },
    u'\U0001FAF4': { # 🫴
        'en' : ':palm_up_hand:',
        'status' : fully_qualified,
        'E' : 14,
        'de': u':hand_mit_handfläche_nach_oben:',
        'es': ':mano_con_la_palma_hacia_arriba:',
        'fr': ':main_paume_vers_le_haut:',
        'pt': u':mão_com_a_palma_para_cima:',
        'it': u':mano_con_il_palmo_verso_l’alto:'
    },
    u'\U0001FAF4\U0001F3FF': { # 🫴🏿
        'en' : ':palm_up_hand_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FB': { # 🫴🏻
        'en' : ':palm_up_hand_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FE': { # 🫴🏾
        'en' : ':palm_up_hand_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FC': { # 🫴🏼
        'en' : ':palm_up_hand_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001FAF4\U0001F3FD': { # 🫴🏽
        'en' : ':palm_up_hand_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 14
    },
    u'\U0001F932': { # 🤲
        'en' : ':palms_up_together:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben:',
        'es': ':palmas_hacia_arriba_juntas:',
        'fr': ':paume_contre_paume_doigts_vers_le_haut:',
        'pt': ':palmas_unidas_para_cima:',
        'it': ':mani_unite_in_alto:'
    },
    u'\U0001F932\U0001F3FF': { # 🤲🏿
        'en' : ':palms_up_together_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_dunkle_hautfarbe:',
        'es': ':palmas_hacia_arriba_juntas_tono_de_piel_oscuro:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_foncée:',
        'pt': ':palmas_unidas_para_cima_pele_escura:',
        'it': ':mani_unite_in_alto_carnagione_scura:'
    },
    u'\U0001F932\U0001F3FB': { # 🤲🏻
        'en' : ':palms_up_together_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_helle_hautfarbe:',
        'es': ':palmas_hacia_arriba_juntas_tono_de_piel_claro:',
        'fr': ':paume_contre_paume_doigts_vers_le_haut_peau_claire:',
        'pt': ':palmas_unidas_para_cima_pele_clara:',
        'it': ':mani_unite_in_alto_carnagione_chiara:'
    },
    u'\U0001F932\U0001F3FE': { # 🤲🏾
        'en' : ':palms_up_together_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mitteldunkle_hautfarbe:',
        'es': ':palmas_hacia_arriba_juntas_tono_de_piel_oscuro_medio:',
        'fr': ':paume_contre_paume_doigts_vers_le_haut_peau_mate:',
        'pt': ':palmas_unidas_para_cima_pele_morena_escura:',
        'it': ':mani_unite_in_alto_carnagione_abbastanza_scura:'
    },
    u'\U0001F932\U0001F3FC': { # 🤲🏼
        'en' : ':palms_up_together_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mittelhelle_hautfarbe:',
        'es': ':palmas_hacia_arriba_juntas_tono_de_piel_claro_medio:',
        'fr': ':paume_contre_paume_doigts_vers_le_haut_peau_moyennement_claire:',
        'pt': ':palmas_unidas_para_cima_pele_morena_clara:',
        'it': ':mani_unite_in_alto_carnagione_abbastanza_chiara:'
    },
    u'\U0001F932\U0001F3FD': { # 🤲🏽
        'en' : ':palms_up_together_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': u':handflächen_nach_oben_mittlere_hautfarbe:',
        'es': ':palmas_hacia_arriba_juntas_tono_de_piel_medio:',
        'fr': u':paume_contre_paume_doigts_vers_le_haut_peau_légèrement_mate:',
        'pt': ':palmas_unidas_para_cima_pele_morena:',
        'it': ':mani_unite_in_alto_carnagione_olivastra:'
    },
    u'\U0001F95E': { # 🥞
        'en' : ':pancakes:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':pfannkuchen:',
        'es': ':tortitas:',
        'fr': ':pancakes:',
        'pt': ':panquecas:',
        'it': ':pancake:'
    },
    u'\U0001F43C': { # 🐼
        'en' : ':panda:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':panda_face:'],
        'de': ':panda:',
        'es': ':panda:',
        'fr': ':panda:',
        'pt': ':rosto_de_panda:',
        'it': ':panda:'
    },
    u'\U0001F4CE': { # 📎
        'en' : ':paperclip:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': u':büroklammer:',
        'es': ':clip:',
        'fr': ':trombone:',
        'pt': ':clipe_de_papel:',
        'it': ':graffetta:'
    },
    u'\U0001FA82': { # 🪂
        'en' : ':parachute:',
        'status' : fully_qualified,
        'E' : 12,
        'de': ':fallschirm:',
        'es': u':paracaídas:',
        'fr': ':parachute:',
        'pt': ':paraquedas:',
        'it': ':paracadute:'
    },
    u'\U0001F99C': { # 🦜
        'en' : ':parrot:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':papagei:',
        'es': ':loro:',
        'fr': ':perroquet:',
        'pt': ':papagaio:',
        'it': ':pappagallo:'
    },
    u'\U0000303D\U0000FE0F': { # 〽️
        'en' : ':part_alternation_mark:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':teilalternationszeichen:',
        'es': ':marca_de_alternancia:',
        'fr': ':alternance:',
        'pt': u':sinal_japonês_indicando_início_de_música:',
        'it': ':simbolo_di_alternanza_delle_parti:'
    },
    u'\U0000303D': { # 〽
        'en' : ':part_alternation_mark:',
        'status' : unqualified,
        'E' : 0.6,
        'variant': True,
        'de': ':teilalternationszeichen:',
        'es': ':marca_de_alternancia:',
        'fr': ':alternance:',
        'pt': u':sinal_japonês_indicando_início_de_música:',
        'it': ':simbolo_di_alternanza_delle_parti:'
    },
    u'\U0001F389': { # 🎉
        'en' : ':party_popper:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':tada:'],
        'de': ':konfettibombe:',
        'es': u':cañón_de_confeti:',
        'fr': ':cotillons:',
        'pt': ':cone_de_festa:',
        'it': ':spara_coriandoli:'
    },
    u'\U0001F973': { # 🥳
        'en' : ':partying_face:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':partygesicht:',
        'es': ':cara_de_fiesta:',
        'fr': ':visage_festif:',
        'pt': ':rosto_festivo:',
        'it': ':faccina_che_festeggia:'
    },
    u'\U0001F6F3\U0000FE0F': { # 🛳️
        'en' : ':passenger_ship:',
        'status' : fully_qualified,
        'E' : 0.7,
        'variant': True,
        'de': ':passagierschiff:',
        'es': ':barco_de_pasajeros:',
        'fr': ':paquebot:',
        'pt': ':cruzeiro:',
        'it': ':nave_da_crociera:'
    },
    u'\U0001F6F3': { # 🛳
        'en' : ':passenger_ship:',
        'status' : unqualified,
        'E' : 0.7,
        'variant': True,
        'de': ':passagierschiff:',
        'es': ':barco_de_pasajeros:',
        'fr': ':paquebot:',
        'pt': ':cruzeiro:',
        'it': ':nave_da_crociera:'
    },
    u'\U0001F6C2': { # 🛂
        'en' : ':passport_control:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':passkontrolle:',
        'es': ':control_de_pasaportes:',
        'fr': u':contrôle_des_passeports:',
        'pt': ':controle_de_passaportes:',
        'it': ':simbolo_del_controllo_passaporti:'
    },
    u'\U000023F8\U0000FE0F': { # ⏸️
        'en' : ':pause_button:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':double_vertical_bar:'],
        'variant': True,
        'de': ':pause:',
        'es': ':pausa:',
        'fr': ':bouton_pause:',
        'pt': u':botão_pausar:',
        'it': ':pulsante_pausa:'
    },
    u'\U000023F8': { # ⏸
        'en' : ':pause_button:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':double_vertical_bar:'],
        'variant': True,
        'de': ':pause:',
        'es': ':pausa:',
        'fr': ':bouton_pause:',
        'pt': u':botão_pausar:',
        'it': ':pulsante_pausa:'
    },
    u'\U0001F43E': { # 🐾
        'en' : ':paw_prints:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':feet:'],
        'de': u':tatzenabdrücke:',
        'es': u':huellas_de_pezuñas:',
        'fr': u':empreintes_d’animaux:',
        'pt': ':patas:',
        'it': ':impronte_di_zampe:'
    },
    u'\U0000262E\U0000FE0F': { # ☮️
        'en' : ':peace_symbol:',
        'status' : fully_qualified,
        'E' : 1,
        'variant': True,
        'de': ':friedenszeichen:',
        'es': u':símbolo_de_la_paz:',
        'fr': ':symbole_de_paix:',
        'pt': u':símbolo_da_paz:',
        'it': ':simbolo_della_pace:'
    },
    u'\U0000262E': { # ☮
        'en' : ':peace_symbol:',
        'status' : unqualified,
        'E' : 1,
        'variant': True,
        'de': ':friedenszeichen:',
        'es': u':símbolo_de_la_paz:',
        'fr': ':symbole_de_paix:',
        'pt': u':símbolo_da_paz:',
        'it': ':simbolo_della_pace:'
    },
    u'\U0001F351': { # 🍑
        'en' : ':peach:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':pfirsich:',
        'es': u':melocotón:',
        'fr': u':pêche:',
        'pt': u':pêssego:',
        'it': ':pesca:'
    },
    u'\U0001F99A': { # 🦚
        'en' : ':peacock:',
        'status' : fully_qualified,
        'E' : 11,
        'de': ':pfau:',
        'es': ':pavo_real:',
        'fr': ':paon:',
        'pt': u':pavão:',
        'it': ':pavone:'
    },
    u'\U0001F95C': { # 🥜
        'en' : ':peanuts:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':erdnuss:',
        'es': ':cacahuetes:',
        'fr': u':cacahuètes:',
        'pt': ':amendoim:',
        'it': ':arachidi:'
    },
    u'\U0001F350': { # 🍐
        'en' : ':pear:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':birne:',
        'es': ':pera:',
        'fr': ':poire:',
        'pt': ':pera:',
        'it': ':pera:'
    },
    u'\U0001F58A\U0000FE0F': { # 🖊️
        'en' : ':pen:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':lower_left_ballpoint_pen:'],
        'variant': True,
        'de': ':kugelschreiber:',
        'es': u':bolígrafo:',
        'fr': ':stylo:',
        'pt': ':caneta:',
        'it': ':penna_a_sfera:'
    },
    u'\U0001F58A': { # 🖊
        'en' : ':pen:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':lower_left_ballpoint_pen:'],
        'variant': True,
        'de': ':kugelschreiber:',
        'es': u':bolígrafo:',
        'fr': ':stylo:',
        'pt': ':caneta:',
        'it': ':penna_a_sfera:'
    },
    u'\U0000270F\U0000FE0F': { # ✏️
        'en' : ':pencil:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pencil2:'],
        'variant': True,
        'de': ':bleistift:',
        'es': u':lápiz:',
        'fr': ':crayon:',
        'pt': u':lápis:',
        'it': ':matita:'
    },
    u'\U0000270F': { # ✏
        'en' : ':pencil:',
        'status' : unqualified,
        'E' : 0.6,
        'alias' : [':pencil2:'],
        'variant': True,
        'de': ':bleistift:',
        'es': u':lápiz:',
        'fr': ':crayon:',
        'pt': u':lápis:',
        'it': ':matita:'
    },
    u'\U0001F427': { # 🐧
        'en' : ':penguin:',
        'status' : fully_qualified,
        'E' : 0.6,
        'de': ':pinguin:',
        'es': u':pingüino:',
        'fr': ':pingouin:',
        'pt': ':pinguim:',
        'it': ':pinguino:'
    },
    u'\U0001F614': { # 😔
        'en' : ':pensive_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':pensive:'],
        'de': ':nachdenkliches_gesicht:',
        'es': ':cara_desanimada:',
        'fr': ':visage_pensif:',
        'pt': ':rosto_deprimido:',
        'it': ':faccina_pensierosa:'
    },
    u'\U0001F9D1\U0000200D\U0001F91D\U0000200D\U0001F9D1': { # 🧑‍🤝‍🧑
        'en' : ':people_holding_hands:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen:',
        'es': ':dos_personas_de_la_mano:',
        'fr': ':deux_personnes_se_tenant_la_main:',
        'pt': u':pessoas_de_mãos_dadas:',
        'it': ':persone_che_si_tengono_per_mano:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏿‍🤝‍🧑🏿
        'en' : ':people_holding_hands_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏿‍🤝‍🧑🏻
        'en' : ':people_holding_hands_dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏿‍🤝‍🧑🏾
        'en' : ':people_holding_hands_dark_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏿‍🤝‍🧑🏼
        'en' : ':people_holding_hands_dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏿‍🤝‍🧑🏽
        'en' : ':people_holding_hands_dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_dunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_foncée_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_escura_e_pele_morena:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_scura_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏻‍🤝‍🧑🏻
        'en' : ':people_holding_hands_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏻‍🤝‍🧑🏿
        'en' : ':people_holding_hands_light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏻‍🤝‍🧑🏾
        'en' : ':people_holding_hands_light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_oscuro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_claire_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏻‍🤝‍🧑🏼
        'en' : ':people_holding_hands_light_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_claro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_claire_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏻‍🤝‍🧑🏽
        'en' : ':people_holding_hands_light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_helle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_claire_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_clara_e_pele_morena:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏾‍🤝‍🧑🏾
        'en' : ':people_holding_hands_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏾‍🤝‍🧑🏿
        'en' : ':people_holding_hands_medium-dark_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏾‍🤝‍🧑🏻
        'en' : ':people_holding_hands_medium-dark_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_mate_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏾‍🤝‍🧑🏼
        'en' : ':people_holding_hands_medium-dark_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_claro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_mate_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_morena_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏾‍🤝‍🧑🏽
        'en' : ':people_holding_hands_medium-dark_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mitteldunkle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_oscuro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_mate_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_escura_e_pele_morena:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_scura_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏼‍🤝‍🧑🏼
        'en' : ':people_holding_hands_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏼‍🤝‍🧑🏿
        'en' : ':people_holding_hands_medium-light_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏼‍🤝‍🧑🏻
        'en' : ':people_holding_hands_medium-light_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_claro:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏼‍🤝‍🧑🏾
        'en' : ':people_holding_hands_medium-light_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_oscuro_medio:',
        'fr': ':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_morena_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏼‍🤝‍🧑🏽
        'en' : ':people_holding_hands_medium-light_skin_tone_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittelhelle_hautfarbe,_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_claro_medio_y_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_moyennement_claire_et_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_clara_e_pele_morena:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_abbastanza_chiara_e_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FD': { # 🧑🏽‍🤝‍🧑🏽
        'en' : ':people_holding_hands_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FF': { # 🧑🏽‍🤝‍🧑🏿
        'en' : ':people_holding_hands_medium_skin_tone_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_dunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_foncée:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FB': { # 🧑🏽‍🤝‍🧑🏻
        'en' : ':people_holding_hands_medium_skin_tone_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_helle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_claro:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FE': { # 🧑🏽‍🤝‍🧑🏾
        'en' : ':people_holding_hands_medium_skin_tone_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_mitteldunkle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_oscuro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_mate:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_morena_escura:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F91D\U0000200D\U0001F9D1\U0001F3FC': { # 🧑🏽‍🤝‍🧑🏼
        'en' : ':people_holding_hands_medium_skin_tone_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 12,
        'de': u':sich_an_den_händen_haltende_personen_mittlere_hautfarbe,_mittelhelle_hautfarbe:',
        'es': u':dos_personas_dándose_la_mano_tono_de_piel_medio_y_tono_de_piel_claro_medio:',
        'fr': u':deux_personnes_se_tenant_la_main_peau_légèrement_mate_et_peau_moyennement_claire:',
        'pt': u':pessoas_de_mãos_dadas_pele_morena_e_pele_morena_clara:',
        'it': ':persone_che_si_tengono_per_mano_carnagione_olivastra_e_carnagione_abbastanza_chiara:'
    },
    u'\U0001FAC2': { # 🫂
        'en' : ':people_hugging:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':sich_umarmende_personen:',
        'es': u':personas_abrazándose:',
        'fr': u':étreinte:',
        'pt': u':pessoas_se_abraçando:',
        'it': ':persone_che_si_abbracciano:'
    },
    u'\U0001F46F': { # 👯
        'en' : ':people_with_bunny_ears:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':dancers:'],
        'de': ':personen_mit_hasenohren:',
        'es': ':personas_con_orejas_de_conejo:',
        'fr': ':personnes_avec_des_oreilles_de_lapin:',
        'pt': ':pessoas_com_orelhas_de_coelho:',
        'it': ':persone_con_orecchie_da_coniglio:'
    },
    u'\U0001F93C': { # 🤼
        'en' : ':people_wrestling:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':wrestling:'],
        'de': ':ringer(in):',
        'es': ':personas_luchando:',
        'fr': ':personnes_faisant_de_la_lutte:',
        'pt': ':pessoas_lutando:',
        'it': ':persone_che_fanno_la_lotta:'
    },
    u'\U0001F3AD': { # 🎭
        'en' : ':performing_arts:',
        'status' : fully_qualified,
        'E' : 0.6,
        'variant': True,
        'de': ':masken:',
        'es': u':máscaras_de_teatro:',
        'fr': ':spectacle_vivant:',
        'pt': u':máscara:',
        'it': ':maschere:'
    },
    u'\U0001F623': { # 😣
        'en' : ':persevering_face:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':persevere:'],
        'de': ':entschlossenes_gesicht:',
        'es': ':cara_desesperada:',
        'fr': u':visage_persévérant:',
        'pt': ':rosto_perseverante:',
        'it': ':faccina_perseverante:'
    },
    u'\U0001F9D1': { # 🧑
        'en' : ':person:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':adult:'],
        'de': ':person:',
        'es': ':persona_adulta:',
        'fr': ':adulte:',
        'pt': ':pessoa:',
        'it': ':persona:'
    },
    u'\U0001F9D1\U0000200D\U0001F9B2': { # 🧑‍🦲
        'en' : ':person_bald:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':erwachsener_glatze:',
        'es': ':persona_adulta_sin_pelo:',
        'fr': ':adulte_chauve:',
        'pt': ':pessoa_careca:',
        'it': ':persona_calvo:'
    },
    u'\U0001F9D4': { # 🧔
        'en' : ':person_beard:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':bearded_person:'],
        'de': ':person_bart:',
        'es': ':hombre_con_barba:',
        'fr': ':personne_barbue:',
        'pt': ':homem_barba:',
        'it': ':persona_con_la_barba:'
    },
    u'\U0001F6B4': { # 🚴
        'en' : ':person_biking:',
        'status' : fully_qualified,
        'E' : 1,
        'alias' : [':bicyclist:'],
        'de': ':radfahrer(in):',
        'es': ':persona_en_bicicleta:',
        'fr': ':cycliste:',
        'pt': ':ciclista:',
        'it': ':ciclista:'
    },
    u'\U0001F6B4\U0001F3FF': { # 🚴🏿
        'en' : ':person_biking_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':radfahrer(in)_dunkle_hautfarbe:',
        'es': ':persona_en_bicicleta_tono_de_piel_oscuro:',
        'fr': u':cycliste_peau_foncée:',
        'pt': ':ciclista_pele_escura:',
        'it': ':ciclista_carnagione_scura:'
    },
    u'\U0001F6B4\U0001F3FB': { # 🚴🏻
        'en' : ':person_biking_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':radfahrer(in)_helle_hautfarbe:',
        'es': ':persona_en_bicicleta_tono_de_piel_claro:',
        'fr': ':cycliste_peau_claire:',
        'pt': ':ciclista_pele_clara:',
        'it': ':ciclista_carnagione_chiara:'
    },
    u'\U0001F6B4\U0001F3FE': { # 🚴🏾
        'en' : ':person_biking_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':radfahrer(in)_mitteldunkle_hautfarbe:',
        'es': ':persona_en_bicicleta_tono_de_piel_oscuro_medio:',
        'fr': ':cycliste_peau_mate:',
        'pt': ':ciclista_pele_morena_escura:',
        'it': ':ciclista_carnagione_abbastanza_scura:'
    },
    u'\U0001F6B4\U0001F3FC': { # 🚴🏼
        'en' : ':person_biking_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':radfahrer(in)_mittelhelle_hautfarbe:',
        'es': ':persona_en_bicicleta_tono_de_piel_claro_medio:',
        'fr': ':cycliste_peau_moyennement_claire:',
        'pt': ':ciclista_pele_morena_clara:',
        'it': ':ciclista_carnagione_abbastanza_chiara:'
    },
    u'\U0001F6B4\U0001F3FD': { # 🚴🏽
        'en' : ':person_biking_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':radfahrer(in)_mittlere_hautfarbe:',
        'es': ':persona_en_bicicleta_tono_de_piel_medio:',
        'fr': u':cycliste_peau_légèrement_mate:',
        'pt': ':ciclista_pele_morena:',
        'it': ':ciclista_carnagione_olivastra:'
    },
    u'\U0001F471': { # 👱
        'en' : ':person_blond_hair:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':blond_haired_person:', ':person_with_blond_hair:'],
        'de': ':person_blondes_haar:',
        'es': ':persona_adulta_rubia:',
        'fr': ':personne_blonde:',
        'pt': ':pessoa_cabelo_louro:',
        'it': ':persona_bionda:'
    },
    u'\U000026F9\U0000FE0F': { # ⛹️
        'en' : ':person_bouncing_ball:',
        'status' : fully_qualified,
        'E' : 0.7,
        'alias' : [':bouncing_ball_person:', ':person_with_ball:'],
        'variant': True,
        'de': ':person_mit_ball:',
        'es': u':persona_botando_un_balón:',
        'fr': ':personne_avec_ballon:',
        'pt': ':pessoa_jogando_basquete:',
        'it': ':persona_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9': { # ⛹
        'en' : ':person_bouncing_ball:',
        'status' : unqualified,
        'E' : 0.7,
        'alias' : [':bouncing_ball_person:', ':person_with_ball:'],
        'variant': True,
        'de': ':person_mit_ball:',
        'es': u':persona_botando_un_balón:',
        'fr': ':personne_avec_ballon:',
        'pt': ':pessoa_jogando_basquete:',
        'it': ':persona_che_fa_rimbalzare_una_palla:'
    },
    u'\U000026F9\U0001F3FF': { # ⛹🏿
        'en' : ':person_bouncing_ball_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':person_mit_ball_dunkle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_oscuro:',
        'fr': u':personne_avec_ballon_peau_foncée:',
        'pt': ':pessoa_jogando_basquete_pele_escura:',
        'it': ':persona_che_fa_rimbalzare_una_palla_carnagione_scura:'
    },
    u'\U000026F9\U0001F3FB': { # ⛹🏻
        'en' : ':person_bouncing_ball_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':person_mit_ball_helle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_claro:',
        'fr': ':personne_avec_ballon_peau_claire:',
        'pt': ':pessoa_jogando_basquete_pele_clara:',
        'it': ':persona_che_fa_rimbalzare_una_palla_carnagione_chiara:'
    },
    u'\U000026F9\U0001F3FE': { # ⛹🏾
        'en' : ':person_bouncing_ball_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':person_mit_ball_mitteldunkle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_oscuro_medio:',
        'fr': ':personne_avec_ballon_peau_mate:',
        'pt': ':pessoa_jogando_basquete_pele_morena_escura:',
        'it': ':persona_che_fa_rimbalzare_una_palla_carnagione_abbastanza_scura:'
    },
    u'\U000026F9\U0001F3FC': { # ⛹🏼
        'en' : ':person_bouncing_ball_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':person_mit_ball_mittelhelle_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_claro_medio:',
        'fr': ':personne_avec_ballon_peau_moyennement_claire:',
        'pt': ':pessoa_jogando_basquete_pele_morena_clara:',
        'it': ':persona_che_fa_rimbalzare_una_palla_carnagione_abbastanza_chiara:'
    },
    u'\U000026F9\U0001F3FD': { # ⛹🏽
        'en' : ':person_bouncing_ball_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 2,
        'de': ':person_mit_ball_mittlere_hautfarbe:',
        'es': u':persona_botando_un_balón_tono_de_piel_medio:',
        'fr': u':personne_avec_ballon_peau_légèrement_mate:',
        'pt': ':pessoa_jogando_basquete_pele_morena:',
        'it': ':persona_che_fa_rimbalzare_una_palla_carnagione_olivastra:'
    },
    u'\U0001F647': { # 🙇
        'en' : ':person_bowing:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':bow:'],
        'de': ':sich_verbeugende_person:',
        'es': ':persona_haciendo_una_reverencia:',
        'fr': u':personne_qui_s’incline:',
        'pt': u':pessoa_fazendo_reverência:',
        'it': ':persona_che_fa_un_inchino_profondo:'
    },
    u'\U0001F647\U0001F3FF': { # 🙇🏿
        'en' : ':person_bowing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sich_verbeugende_person_dunkle_hautfarbe:',
        'es': ':persona_haciendo_una_reverencia_tono_de_piel_oscuro:',
        'fr': u':personne_qui_s’incline_peau_foncée:',
        'pt': u':pessoa_fazendo_reverência_pele_escura:',
        'it': ':persona_che_fa_un_inchino_profondo_carnagione_scura:'
    },
    u'\U0001F647\U0001F3FB': { # 🙇🏻
        'en' : ':person_bowing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sich_verbeugende_person_helle_hautfarbe:',
        'es': ':persona_haciendo_una_reverencia_tono_de_piel_claro:',
        'fr': u':personne_qui_s’incline_peau_claire:',
        'pt': u':pessoa_fazendo_reverência_pele_clara:',
        'it': ':persona_che_fa_un_inchino_profondo_carnagione_chiara:'
    },
    u'\U0001F647\U0001F3FE': { # 🙇🏾
        'en' : ':person_bowing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sich_verbeugende_person_mitteldunkle_hautfarbe:',
        'es': ':persona_haciendo_una_reverencia_tono_de_piel_oscuro_medio:',
        'fr': u':personne_qui_s’incline_peau_mate:',
        'pt': u':pessoa_fazendo_reverência_pele_morena_escura:',
        'it': ':persona_che_fa_un_inchino_profondo_carnagione_abbastanza_scura:'
    },
    u'\U0001F647\U0001F3FC': { # 🙇🏼
        'en' : ':person_bowing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sich_verbeugende_person_mittelhelle_hautfarbe:',
        'es': ':persona_haciendo_una_reverencia_tono_de_piel_claro_medio:',
        'fr': u':personne_qui_s’incline_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_reverência_pele_morena_clara:',
        'it': ':persona_che_fa_un_inchino_profondo_carnagione_abbastanza_chiara:'
    },
    u'\U0001F647\U0001F3FD': { # 🙇🏽
        'en' : ':person_bowing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':sich_verbeugende_person_mittlere_hautfarbe:',
        'es': ':persona_haciendo_una_reverencia_tono_de_piel_medio:',
        'fr': u':personne_qui_s’incline_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_reverência_pele_morena:',
        'it': ':persona_che_fa_un_inchino_profondo_carnagione_olivastra:'
    },
    u'\U0001F938': { # 🤸
        'en' : ':person_cartwheeling:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':cartwheeling:'],
        'de': ':rad_schlagende_person:',
        'es': ':persona_haciendo_voltereta_lateral:',
        'fr': ':personne_faisant_la_roue:',
        'pt': ':pessoa_fazendo_estrela:',
        'it': ':persona_che_fa_la_ruota:'
    },
    u'\U0001F938\U0001F3FF': { # 🤸🏿
        'en' : ':person_cartwheeling_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':rad_schlagende_person_dunkle_hautfarbe:',
        'es': ':persona_haciendo_voltereta_lateral_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_la_roue_peau_foncée:',
        'pt': ':pessoa_fazendo_estrela_pele_escura:',
        'it': ':persona_che_fa_la_ruota_carnagione_scura:'
    },
    u'\U0001F938\U0001F3FB': { # 🤸🏻
        'en' : ':person_cartwheeling_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':rad_schlagende_person_helle_hautfarbe:',
        'es': ':persona_haciendo_voltereta_lateral_tono_de_piel_claro:',
        'fr': ':personne_faisant_la_roue_peau_claire:',
        'pt': ':pessoa_fazendo_estrela_pele_clara:',
        'it': ':persona_che_fa_la_ruota_carnagione_chiara:'
    },
    u'\U0001F938\U0001F3FE': { # 🤸🏾
        'en' : ':person_cartwheeling_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':rad_schlagende_person_mitteldunkle_hautfarbe:',
        'es': ':persona_haciendo_voltereta_lateral_tono_de_piel_oscuro_medio:',
        'fr': ':personne_faisant_la_roue_peau_mate:',
        'pt': ':pessoa_fazendo_estrela_pele_morena_escura:',
        'it': ':persona_che_fa_la_ruota_carnagione_abbastanza_scura:'
    },
    u'\U0001F938\U0001F3FC': { # 🤸🏼
        'en' : ':person_cartwheeling_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':rad_schlagende_person_mittelhelle_hautfarbe:',
        'es': ':persona_haciendo_voltereta_lateral_tono_de_piel_claro_medio:',
        'fr': ':personne_faisant_la_roue_peau_moyennement_claire:',
        'pt': ':pessoa_fazendo_estrela_pele_morena_clara:',
        'it': ':persona_che_fa_la_ruota_carnagione_abbastanza_chiara:'
    },
    u'\U0001F938\U0001F3FD': { # 🤸🏽
        'en' : ':person_cartwheeling_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':rad_schlagende_person_mittlere_hautfarbe:',
        'es': ':persona_haciendo_voltereta_lateral_tono_de_piel_medio:',
        'fr': u':personne_faisant_la_roue_peau_légèrement_mate:',
        'pt': ':pessoa_fazendo_estrela_pele_morena:',
        'it': ':persona_che_fa_la_ruota_carnagione_olivastra:'
    },
    u'\U0001F9D7': { # 🧗
        'en' : ':person_climbing:',
        'status' : fully_qualified,
        'E' : 5,
        'alias' : [':climbing:'],
        'de': ':bergsteiger(in):',
        'es': ':persona_escalando:',
        'fr': ':personne_qui_grimpe:',
        'pt': ':pessoa_escalando:',
        'it': ':persona_che_scala:'
    },
    u'\U0001F9D7\U0001F3FF': { # 🧗🏿
        'en' : ':person_climbing_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger(in)_dunkle_hautfarbe:',
        'es': ':persona_escalando_tono_de_piel_oscuro:',
        'fr': u':personne_qui_grimpe_peau_foncée:',
        'pt': ':pessoa_escalando_pele_escura:',
        'it': ':persona_che_scala_carnagione_scura:'
    },
    u'\U0001F9D7\U0001F3FB': { # 🧗🏻
        'en' : ':person_climbing_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger(in)_helle_hautfarbe:',
        'es': ':persona_escalando_tono_de_piel_claro:',
        'fr': ':personne_qui_grimpe_peau_claire:',
        'pt': ':pessoa_escalando_pele_clara:',
        'it': ':persona_che_scala_carnagione_chiara:'
    },
    u'\U0001F9D7\U0001F3FE': { # 🧗🏾
        'en' : ':person_climbing_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger(in)_mitteldunkle_hautfarbe:',
        'es': ':persona_escalando_tono_de_piel_oscuro_medio:',
        'fr': ':personne_qui_grimpe_peau_mate:',
        'pt': ':pessoa_escalando_pele_morena_escura:',
        'it': ':persona_che_scala_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D7\U0001F3FC': { # 🧗🏼
        'en' : ':person_climbing_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger(in)_mittelhelle_hautfarbe:',
        'es': ':persona_escalando_tono_de_piel_claro_medio:',
        'fr': ':personne_qui_grimpe_peau_moyennement_claire:',
        'pt': ':pessoa_escalando_pele_morena_clara:',
        'it': ':persona_che_scala_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D7\U0001F3FD': { # 🧗🏽
        'en' : ':person_climbing_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':bergsteiger(in)_mittlere_hautfarbe:',
        'es': ':persona_escalando_tono_de_piel_medio:',
        'fr': u':personne_qui_grimpe_peau_légèrement_mate:',
        'pt': ':pessoa_escalando_pele_morena:',
        'it': ':persona_che_scala_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F9B1': { # 🧑‍🦱
        'en' : ':person_curly_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':erwachsener_lockiges_haar:',
        'es': ':persona_adulta_pelo_rizado:',
        'fr': u':adulte_cheveux_bouclés:',
        'pt': ':pessoa_cabelo_cacheado:',
        'it': ':persona_capelli_ricci:'
    },
    u'\U0001F9D1\U0001F3FF': { # 🧑🏿
        'en' : ':person_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':erwachsener_dunkle_hautfarbe:',
        'es': ':persona_adulta_tono_de_piel_oscuro:',
        'fr': u':adulte_peau_foncée:',
        'pt': ':pessoa_pele_escura:',
        'it': ':persona_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B2': { # 🧑🏿‍🦲
        'en' : ':person_dark_skin_tone_bald:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':erwachsener_dunkle_hautfarbe,_glatze:',
        'es': ':persona_adulta_tono_de_piel_oscuro_y_sin_pelo:',
        'fr': u':adulte_peau_foncée_et_chauve:',
        'pt': ':pessoa_pele_escura_e_careca:',
        'it': ':persona_carnagione_scura_e_calvo:'
    },
    u'\U0001F9D4\U0001F3FF': { # 🧔🏿
        'en' : ':person_dark_skin_tone_beard:',
        'status' : fully_qualified,
        'E' : 5,
        'de': ':mann_dunkle_hautfarbe,_bart:',
        'es': ':hombre_con_barba_tono_de_piel_oscuro:',
        'fr': u':homme_barbu_peau_foncée:',
        'pt': ':homem_pele_escura_e_barba:',
        'it': ':uomo_con_la_barba_carnagione_scura:'
    },
    u'\U0001F471\U0001F3FF': { # 👱🏿
        'en' : ':person_dark_skin_tone_blond_hair:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_dunkle_hautfarbe,_blondes_haar:',
        'es': ':persona_adulta_rubia_tono_de_piel_oscuro:',
        'fr': u':personne_blonde_peau_foncée:',
        'pt': ':pessoa_pele_escura_e_cabelo_louro:',
        'it': ':persona_bionda_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B1': { # 🧑🏿‍🦱
        'en' : ':person_dark_skin_tone_curly_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':erwachsener_dunkle_hautfarbe,_lockiges_haar:',
        'es': ':persona_adulta_tono_de_piel_oscuro_y_pelo_rizado:',
        'fr': u':adulte_peau_foncée_et_cheveux_bouclés:',
        'pt': ':pessoa_pele_escura_e_cabelo_cacheado:',
        'it': ':persona_carnagione_scura_e_capelli_ricci:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B0': { # 🧑🏿‍🦰
        'en' : ':person_dark_skin_tone_red_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': ':erwachsener_dunkle_hautfarbe,_rotes_haar:',
        'es': ':persona_adulta_tono_de_piel_oscuro_y_pelo_pelirrojo:',
        'fr': u':adulte_peau_foncée_et_cheveux_roux:',
        'pt': ':pessoa_pele_escura_e_cabelo_vermelho:',
        'it': ':persona_carnagione_scura_e_capelli_rossi:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F9B3': { # 🧑🏿‍🦳
        'en' : ':person_dark_skin_tone_white_hair:',
        'status' : fully_qualified,
        'E' : 12.1,
        'de': u':erwachsener_dunkle_hautfarbe,_weißes_haar:',
        'es': ':persona_adulta_tono_de_piel_oscuro_y_pelo_blanco:',
        'fr': u':adulte_peau_foncée_et_cheveux_blancs:',
        'pt': ':pessoa_pele_escura_e_cabelo_branco:',
        'it': ':persona_carnagione_scura_e_capelli_bianchi:'
    },
    u'\U0001F926': { # 🤦
        'en' : ':person_facepalming:',
        'status' : fully_qualified,
        'E' : 3,
        'alias' : [':facepalm:'],
        'de': ':sich_an_den_kopf_fassende_person:',
        'es': ':persona_con_la_mano_en_la_frente:',
        'fr': ':personne_avec_la_paume_sur_le_visage:',
        'pt': ':pessoa_decepcionada:',
        'it': ':persona_esasperata:'
    },
    u'\U0001F926\U0001F3FF': { # 🤦🏿
        'en' : ':person_facepalming_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sich_an_den_kopf_fassende_person_dunkle_hautfarbe:',
        'es': ':persona_con_la_mano_en_la_frente_tono_de_piel_oscuro:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_foncée:',
        'pt': ':pessoa_decepcionada_pele_escura:',
        'it': ':persona_esasperata_carnagione_scura:'
    },
    u'\U0001F926\U0001F3FB': { # 🤦🏻
        'en' : ':person_facepalming_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sich_an_den_kopf_fassende_person_helle_hautfarbe:',
        'es': ':persona_con_la_mano_en_la_frente_tono_de_piel_claro:',
        'fr': ':personne_avec_la_paume_sur_le_visage_peau_claire:',
        'pt': ':pessoa_decepcionada_pele_clara:',
        'it': ':persona_esasperata_carnagione_chiara:'
    },
    u'\U0001F926\U0001F3FE': { # 🤦🏾
        'en' : ':person_facepalming_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sich_an_den_kopf_fassende_person_mitteldunkle_hautfarbe:',
        'es': ':persona_con_la_mano_en_la_frente_tono_de_piel_oscuro_medio:',
        'fr': ':personne_avec_la_paume_sur_le_visage_peau_mate:',
        'pt': ':pessoa_decepcionada_pele_morena_escura:',
        'it': ':persona_esasperata_carnagione_abbastanza_scura:'
    },
    u'\U0001F926\U0001F3FC': { # 🤦🏼
        'en' : ':person_facepalming_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sich_an_den_kopf_fassende_person_mittelhelle_hautfarbe:',
        'es': ':persona_con_la_mano_en_la_frente_tono_de_piel_claro_medio:',
        'fr': ':personne_avec_la_paume_sur_le_visage_peau_moyennement_claire:',
        'pt': ':pessoa_decepcionada_pele_morena_clara:',
        'it': ':persona_esasperata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F926\U0001F3FD': { # 🤦🏽
        'en' : ':person_facepalming_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':sich_an_den_kopf_fassende_person_mittlere_hautfarbe:',
        'es': ':persona_con_la_mano_en_la_frente_tono_de_piel_medio:',
        'fr': u':personne_avec_la_paume_sur_le_visage_peau_légèrement_mate:',
        'pt': ':pessoa_decepcionada_pele_morena:',
        'it': ':persona_esasperata_carnagione_olivastra:'
    },
    u'\U0001F9D1\U0000200D\U0001F37C': { # 🧑‍🍼
        'en' : ':person_feeding_baby:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person:',
        'es': u':persona_alimentando_a_bebé:',
        'fr': u':personne_allaitant_un_bébé:',
        'pt': u':pessoa_alimentando_bebê:',
        'it': ':persona_che_allatta:'
    },
    u'\U0001F9D1\U0001F3FF\U0000200D\U0001F37C': { # 🧑🏿‍🍼
        'en' : ':person_feeding_baby_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person_dunkle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_oscuro:',
        'fr': u':personne_allaitant_un_bébé_peau_foncée:',
        'pt': u':pessoa_alimentando_bebê_pele_escura:',
        'it': ':persona_che_allatta_carnagione_scura:'
    },
    u'\U0001F9D1\U0001F3FB\U0000200D\U0001F37C': { # 🧑🏻‍🍼
        'en' : ':person_feeding_baby_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person_helle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_claro:',
        'fr': u':personne_allaitant_un_bébé_peau_claire:',
        'pt': u':pessoa_alimentando_bebê_pele_clara:',
        'it': ':persona_che_allatta_carnagione_chiara:'
    },
    u'\U0001F9D1\U0001F3FE\U0000200D\U0001F37C': { # 🧑🏾‍🍼
        'en' : ':person_feeding_baby_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person_mitteldunkle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_oscuro_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_mate:',
        'pt': u':pessoa_alimentando_bebê_pele_morena_escura:',
        'it': ':persona_che_allatta_carnagione_abbastanza_scura:'
    },
    u'\U0001F9D1\U0001F3FC\U0000200D\U0001F37C': { # 🧑🏼‍🍼
        'en' : ':person_feeding_baby_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person_mittelhelle_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_claro_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_moyennement_claire:',
        'pt': u':pessoa_alimentando_bebê_pele_morena_clara:',
        'it': ':persona_che_allatta_carnagione_abbastanza_chiara:'
    },
    u'\U0001F9D1\U0001F3FD\U0000200D\U0001F37C': { # 🧑🏽‍🍼
        'en' : ':person_feeding_baby_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 13,
        'de': ':stillende_person_mittlere_hautfarbe:',
        'es': u':persona_alimentando_a_bebé_tono_de_piel_medio:',
        'fr': u':personne_allaitant_un_bébé_peau_légèrement_mate:',
        'pt': u':pessoa_alimentando_bebê_pele_morena:',
        'it': ':persona_che_allatta_carnagione_olivastra:'
    },
    u'\U0001F93A': { # 🤺
        'en' : ':person_fencing:',
        'status' : fully_qualified,
        'E' : 3,
        'de': ':fechter(in):',
        'es': ':esgrima:',
        'fr': ':escrimeur:',
        'pt': ':esgrimista:',
        'it': ':schermidore:'
    },
    u'\U0001F64D': { # 🙍
        'en' : ':person_frowning:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':frowning_person:'],
        'de': ':missmutige_person:',
        'es': u':persona_frunciendo_el_ceño:',
        'fr': u':personne_fronçant_les_sourcils:',
        'pt': ':franzindo_a_sobrancelha:',
        'it': ':persona_corrucciata:'
    },
    u'\U0001F64D\U0001F3FF': { # 🙍🏿
        'en' : ':person_frowning_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':missmutige_person_dunkle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_oscuro:',
        'fr': u':personne_fronçant_les_sourcils_peau_foncée:',
        'pt': ':franzindo_a_sobrancelha_pele_escura:',
        'it': ':persona_corrucciata_carnagione_scura:'
    },
    u'\U0001F64D\U0001F3FB': { # 🙍🏻
        'en' : ':person_frowning_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':missmutige_person_helle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_claro:',
        'fr': u':personne_fronçant_les_sourcils_peau_claire:',
        'pt': ':franzindo_a_sobrancelha_pele_clara:',
        'it': ':persona_corrucciata_carnagione_chiara:'
    },
    u'\U0001F64D\U0001F3FE': { # 🙍🏾
        'en' : ':person_frowning_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':missmutige_person_mitteldunkle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_oscuro_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_mate:',
        'pt': ':franzindo_a_sobrancelha_pele_morena_escura:',
        'it': ':persona_corrucciata_carnagione_abbastanza_scura:'
    },
    u'\U0001F64D\U0001F3FC': { # 🙍🏼
        'en' : ':person_frowning_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':missmutige_person_mittelhelle_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_claro_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_moyennement_claire:',
        'pt': ':franzindo_a_sobrancelha_pele_morena_clara:',
        'it': ':persona_corrucciata_carnagione_abbastanza_chiara:'
    },
    u'\U0001F64D\U0001F3FD': { # 🙍🏽
        'en' : ':person_frowning_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':missmutige_person_mittlere_hautfarbe:',
        'es': u':persona_frunciendo_el_ceño_tono_de_piel_medio:',
        'fr': u':personne_fronçant_les_sourcils_peau_légèrement_mate:',
        'pt': ':franzindo_a_sobrancelha_pele_morena:',
        'it': ':persona_corrucciata_carnagione_olivastra:'
    },
    u'\U0001F645': { # 🙅
        'en' : ':person_gesturing_NO:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':no_good:'],
        'de': u':person_mit_überkreuzten_armen:',
        'es': ':persona_haciendo_el_gesto_de_no:',
        'fr': u':personne_faisant_un_geste_d’interdiction:',
        'pt': u':pessoa_fazendo_gesto_de_“não”:',
        'it': ':persona_che_fa_segno_di_no:'
    },
    u'\U0001F645\U0001F3FF': { # 🙅🏿
        'en' : ':person_gesturing_NO_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_dunkle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_no_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_foncée:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_escura:',
        'it': ':persona_che_fa_segno_di_no_carnagione_scura:'
    },
    u'\U0001F645\U0001F3FB': { # 🙅🏻
        'en' : ':person_gesturing_NO_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_helle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_no_tono_de_piel_claro:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_clara:',
        'it': ':persona_che_fa_segno_di_no_carnagione_chiara:'
    },
    u'\U0001F645\U0001F3FE': { # 🙅🏾
        'en' : ':person_gesturing_NO_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mitteldunkle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_no_tono_de_piel_oscuro_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena_escura:',
        'it': ':persona_che_fa_segno_di_no_carnagione_abbastanza_scura:'
    },
    u'\U0001F645\U0001F3FC': { # 🙅🏼
        'en' : ':person_gesturing_NO_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mittelhelle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_no_tono_de_piel_claro_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena_clara:',
        'it': ':persona_che_fa_segno_di_no_carnagione_abbastanza_chiara:'
    },
    u'\U0001F645\U0001F3FD': { # 🙅🏽
        'en' : ':person_gesturing_NO_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_überkreuzten_armen_mittlere_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_no_tono_de_piel_medio:',
        'fr': u':personne_faisant_un_geste_d’interdiction_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“não”_pele_morena:',
        'it': ':persona_che_fa_segno_di_no_carnagione_olivastra:'
    },
    u'\U0001F646': { # 🙆
        'en' : ':person_gesturing_OK:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':ok_woman:', ':ok_person:'],
        'de': u':person_mit_händen_auf_dem_kopf:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo:',
        'fr': u':personne_faisant_un_geste_d’acceptation:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”:',
        'it': ':persona_con_gesto_ok:'
    },
    u'\U0001F646\U0001F3FF': { # 🙆🏿
        'en' : ':person_gesturing_OK_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_dunkle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_foncée:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_escura:',
        'it': ':persona_con_gesto_ok_carnagione_scura:'
    },
    u'\U0001F646\U0001F3FB': { # 🙆🏻
        'en' : ':person_gesturing_OK_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_helle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_clara:',
        'it': ':persona_con_gesto_ok_carnagione_chiara:'
    },
    u'\U0001F646\U0001F3FE': { # 🙆🏾
        'en' : ':person_gesturing_OK_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mitteldunkle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_oscuro_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena_escura:',
        'it': ':persona_con_gesto_ok_carnagione_abbastanza_scura:'
    },
    u'\U0001F646\U0001F3FC': { # 🙆🏼
        'en' : ':person_gesturing_OK_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mittelhelle_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_claro_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_moyennement_claire:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena_clara:',
        'it': ':persona_con_gesto_ok_carnagione_abbastanza_chiara:'
    },
    u'\U0001F646\U0001F3FD': { # 🙆🏽
        'en' : ':person_gesturing_OK_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': u':person_mit_händen_auf_dem_kopf_mittlere_hautfarbe:',
        'es': ':persona_haciendo_el_gesto_de_de_acuerdo_tono_de_piel_medio:',
        'fr': u':personne_faisant_un_geste_d’acceptation_peau_légèrement_mate:',
        'pt': u':pessoa_fazendo_gesto_de_“ok”_pele_morena:',
        'it': ':persona_con_gesto_ok_carnagione_olivastra:'
    },
    u'\U0001F487': { # 💇
        'en' : ':person_getting_haircut:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':haircut:'],
        'de': ':person_beim_haareschneiden:',
        'es': u':persona_cortándose_el_pelo:',
        'fr': ':personne_qui_se_fait_couper_les_cheveux:',
        'pt': ':pessoa_cortando_o_cabelo:',
        'it': ':taglio_di_capelli:'
    },
    u'\U0001F487\U0001F3FF': { # 💇🏿
        'en' : ':person_getting_haircut_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_beim_haareschneiden_dunkle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_oscuro:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_foncée:',
        'pt': ':pessoa_cortando_o_cabelo_pele_escura:',
        'it': ':taglio_di_capelli_carnagione_scura:'
    },
    u'\U0001F487\U0001F3FB': { # 💇🏻
        'en' : ':person_getting_haircut_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_beim_haareschneiden_helle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_claro:',
        'fr': ':personne_qui_se_fait_couper_les_cheveux_peau_claire:',
        'pt': ':pessoa_cortando_o_cabelo_pele_clara:',
        'it': ':taglio_di_capelli_carnagione_chiara:'
    },
    u'\U0001F487\U0001F3FE': { # 💇🏾
        'en' : ':person_getting_haircut_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_beim_haareschneiden_mitteldunkle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_oscuro_medio:',
        'fr': ':personne_qui_se_fait_couper_les_cheveux_peau_mate:',
        'pt': ':pessoa_cortando_o_cabelo_pele_morena_escura:',
        'it': ':taglio_di_capelli_carnagione_abbastanza_scura:'
    },
    u'\U0001F487\U0001F3FC': { # 💇🏼
        'en' : ':person_getting_haircut_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_beim_haareschneiden_mittelhelle_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_claro_medio:',
        'fr': ':personne_qui_se_fait_couper_les_cheveux_peau_moyennement_claire:',
        'pt': ':pessoa_cortando_o_cabelo_pele_morena_clara:',
        'it': ':taglio_di_capelli_carnagione_abbastanza_chiara:'
    },
    u'\U0001F487\U0001F3FD': { # 💇🏽
        'en' : ':person_getting_haircut_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person_beim_haareschneiden_mittlere_hautfarbe:',
        'es': u':persona_cortándose_el_pelo_tono_de_piel_medio:',
        'fr': u':personne_qui_se_fait_couper_les_cheveux_peau_légèrement_mate:',
        'pt': ':pessoa_cortando_o_cabelo_pele_morena:',
        'it': ':taglio_di_capelli_carnagione_olivastra:'
    },
    u'\U0001F486': { # 💆
        'en' : ':person_getting_massage:',
        'status' : fully_qualified,
        'E' : 0.6,
        'alias' : [':massage:'],
        'de': ':person_die_eine_kopfmassage_bekommt:',
        'es': ':persona_recibiendo_masaje:',
        'fr': ':personne_qui_se_fait_masser:',
        'pt': ':pessoa_recebendo_massagem_facial:',
        'it': ':persona_che_riceve_un_massaggio:'
    },
    u'\U0001F486\U0001F3FF': { # 💆🏿
        'en' : ':person_getting_massage_dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person,_die_eine_kopfmassage_bekommt_dunkle_hautfarbe:',
        'es': ':persona_recibiendo_masaje_tono_de_piel_oscuro:',
        'fr': u':personne_qui_se_fait_masser_peau_foncée:',
        'pt': ':pessoa_recebendo_massagem_facial_pele_escura:',
        'it': ':persona_che_riceve_un_massaggio_carnagione_scura:'
    },
    u'\U0001F486\U0001F3FB': { # 💆🏻
        'en' : ':person_getting_massage_light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person,_die_eine_kopfmassage_bekommt_helle_hautfarbe:',
        'es': ':persona_recibiendo_masaje_tono_de_piel_claro:',
        'fr': ':personne_qui_se_fait_masser_peau_claire:',
        'pt': ':pessoa_recebendo_massagem_facial_pele_clara:',
        'it': ':persona_che_riceve_un_massaggio_carnagione_chiara:'
    },
    u'\U0001F486\U0001F3FE': { # 💆🏾
        'en' : ':person_getting_massage_medium-dark_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person,_die_eine_kopfmassage_bekommt_mitteldunkle_hautfarbe:',
        'es': ':persona_recibiendo_masaje_tono_de_piel_oscuro_medio:',
        'fr': ':personne_qui_se_fait_masser_peau_mate:',
        'pt': ':pessoa_recebendo_massagem_facial_pele_morena_escura:',
        'it': ':persona_che_riceve_un_massaggio_carnagione_abbastanza_scura:'
    },
    u'\U0001F486\U0001F3FC': { # 💆🏼
        'en' : ':person_getting_massage_medium-light_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person,_die_eine_kopfmassage_bekommt_mittelhelle_hautfarbe:',
        'es': ':persona_recibiendo_masaje_tono_de_piel_claro_medio:',
        'fr': ':personne_qui_se_fait_masser_peau_moyennement_claire:',
        'pt': ':pessoa_recebendo_massagem_facial_pele_morena_clara:',
        'it': ':persona_che_riceve_un_massaggio_carnagione_abbastanza_chiara:'
    },
    u'\U0001F486\U0001F3FD': { # 💆🏽
        'en' : ':person_getting_massage_medium_skin_tone:',
        'status' : fully_qualified,
        'E' : 1,
        'de': ':person,_die_eine_kopfmassage_bekommt_mittlere_hautfarbe:',
        'es': ':persona_recibiendo_masaje_tono_de_piel_medio:'