"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addClickToMetric = addClickToMetric;
exports.addRequestToMetric = addRequestToMetric;
exports.getMetrics = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _constants = require("./constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
const time2CountWin = new Map();

function addClickToMetric(element, counter = 'count') {
  // remove outdated key-value pairs
  trim();
  const timeKey = getKey(Date.now());

  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER);

  const key = `click.${element}.${counter}`;

  _lodash.default.set(rollingCounter, key, _lodash.default.get(rollingCounter, key, 0) + 1);

  if (counter === 'count') {
    const basicCounterKey = `click.${element}.total`;

    _lodash.default.set(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, _lodash.default.get(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, 0) + 1);
  }

  time2CountWin.set(timeKey, rollingCounter);
}

function addRequestToMetric(component, request, counterNameOrError) {
  const counter = typeof counterNameOrError === 'object' ? checkErrorType(counterNameOrError) : counterNameOrError; // remove outdated key-value pairs

  trim();
  const timeKey = getKey(Date.now());

  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER);

  rollingCounter[component][request][counter]++;

  if (counter === 'count') {
    _constants.GLOBAL_BASIC_COUNTER[component][request].total++;
  }

  time2CountWin.set(timeKey, rollingCounter);
}

const getMetrics = () => {
  const preTimeKey = getPreKey(Date.now());
  const rollingCounters = time2CountWin.get(preTimeKey);
  return buildMetrics(rollingCounters);
};

exports.getMetrics = getMetrics;

const checkErrorType = error => {
  if (error.statusCode && Math.floor(error.statusCode / 100) === 4) {
    return 'user_error';
  } else {
    return 'system_error';
  }
};

const trim = () => {
  if (time2CountWin.size > _constants.CAPACITY) {
    const currentKey = getKey(Date.now() - _constants.WINDOW * _constants.MILLIS_MULTIPLIER);
    time2CountWin.forEach((_value, key, map) => {
      if (key < currentKey) {
        map.delete(key);
      }
    });
  }
};

const getKey = milliseconds => {
  return Math.floor(milliseconds / _constants.MILLIS_MULTIPLIER / _constants.INTERVAL);
};

const getPreKey = milliseconds => {
  return getKey(milliseconds) - 1;
};

const isComponent = arg => {
  return _constants.COMPONENTS.includes(arg);
};

const buildMetrics = rollingCounters => {
  if (!rollingCounters) {
    rollingCounters = _constants.DEFAULT_ROLLING_COUNTER;
  }

  const basicMetrics = _lodash.default.merge(rollingCounters, _constants.GLOBAL_BASIC_COUNTER);

  const overallActionMetrics = {
    request_total: 0,
    request_count: 0,
    success_count: 0,
    failed_request_count_system_error: 0,
    failed_request_count_user_error: 0
  };
  Object.keys(basicMetrics).forEach(key => {
    if (isComponent(key)) {
      for (const counter of Object.values(basicMetrics[key])) {
        overallActionMetrics.request_count += (counter === null || counter === void 0 ? void 0 : counter.count) || 0;
        overallActionMetrics.request_total += (counter === null || counter === void 0 ? void 0 : counter.total) || 0;
        overallActionMetrics.failed_request_count_system_error += (counter === null || counter === void 0 ? void 0 : counter.system_error) || 0;
        overallActionMetrics.failed_request_count_user_error += (counter === null || counter === void 0 ? void 0 : counter.user_error) || 0;
      }
    }
  });
  overallActionMetrics.success_count = overallActionMetrics.request_count - (overallActionMetrics.failed_request_count_system_error + overallActionMetrics.failed_request_count_user_error);
  return { ...basicMetrics,
    ...overallActionMetrics
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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