"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PanelsRouter = PanelsRouter;

var _configSchema = require("@osd/config-schema");

var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");

var _custom_panels = require("../../../common/constants/custom_panels");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function PanelsRouter(router) {
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor(); // Fetch all the custom panels available

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelsList = await customPanelBackend.viewPanelList(opensearchNotebooksClient);
      return response.ok({
        body: {
          panels: panelsList
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Fetch the required panel by id

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelObject = await customPanelBackend.getPanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: panelObject
      });
    } catch (error) {
      console.error('Issue in fetching panel:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Create a new panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels`,
    validate: {
      body: _configSchema.schema.object({
        panelName: _configSchema.schema.string(),
        applicationId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newPanelId = await customPanelBackend.createNewPanel(opensearchNotebooksClient, request.body.panelName, request.body.applicationId || '');
      return response.ok({
        body: {
          message: 'Panel Created',
          newPanelId
        }
      });
    } catch (error) {
      console.error('Issue in creating new panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // update an existing panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/update`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panel: _configSchema.schema.any()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const responseBody = await customPanelBackend.updatePanel(opensearchNotebooksClient, request.body.panelId, request.body.panel);
      return response.ok({
        body: {
          message: 'Panel Updated'
        }
      });
    } catch (error) {
      console.error('Issue in updating panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // rename an existing panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/rename`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const responseBody = await customPanelBackend.renamePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Renamed'
        }
      });
    } catch (error) {
      console.error('Issue in renaming panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // clones an existing panel
  // returns new panel Id

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/clone`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        panelName: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const cloneResponse = await customPanelBackend.clonePanel(opensearchNotebooksClient, request.body.panelId, request.body.panelName);
      return response.ok({
        body: {
          message: 'Panel Cloned',
          clonePanelId: cloneResponse.clonePanelId,
          dateCreated: cloneResponse.dateCreated,
          dateModified: cloneResponse.dateModified
        }
      });
    } catch (error) {
      console.error('Issue in cloning panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // delete an existing panel

  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/{panelId}`,
    validate: {
      params: _configSchema.schema.object({
        panelId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const deleteResponse = await customPanelBackend.deletePanel(opensearchNotebooksClient, request.params.panelId);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // delete an existing panel(s)

  router.delete({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panelList/{panelIdList}`,
    validate: {
      params: _configSchema.schema.object({
        panelIdList: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const deleteResponse = await customPanelBackend.deletePanelList(opensearchNotebooksClient, request.params.panelIdList);
      return response.ok({
        body: {
          message: 'Panel Deleted'
        }
      });
    } catch (error) {
      console.error('Issue in deleting panel', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // replaces the ppl query filter in panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/filter`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        query: _configSchema.schema.string(),
        language: _configSchema.schema.string(),
        to: _configSchema.schema.string(),
        from: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelFilterResponse = await customPanelBackend.addPanelFilter(opensearchNotebooksClient, request.body.panelId, request.body.query, request.body.language, request.body.to, request.body.from);
      return response.ok({
        body: {
          message: 'Panel PPL Filter Changed'
        }
      });
    } catch (error) {
      console.error('Issue in adding query filter', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add Sample Panels

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/panels/addSamplePanels`,
    validate: {
      body: _configSchema.schema.object({
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const panelsData = await customPanelBackend.addSamplePanels(opensearchNotebooksClient, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          demoPanelsData: panelsData
        }
      });
    } catch (error) {
      console.error('Issue in fetching panel list:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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