"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearSplitCookies = clearSplitCookies;
exports.getExtraAuthStorageValue = getExtraAuthStorageValue;
exports.setExtraAuthStorage = setExtraAuthStorage;
exports.splitValueIntoCookies = splitValueIntoCookies;
exports.unsplitCookiesIntoValue = unsplitCookiesIntoValue;

var _router = require("../../../../src/core/server/http/router");

var _compression = require("../utils/compression");

var _common = require("../../common");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
function getExtraAuthStorageValue(request, options) {
  let compressedContent = '';
  let content = '';

  if (options.additionalCookies > 0) {
    compressedContent = unsplitCookiesIntoValue(request, options.cookiePrefix, options.additionalCookies);
  }

  try {
    content = (0, _compression.inflateValue)(Buffer.from(compressedContent, 'base64')).toString();
  } catch (error) {
    throw error;
  }

  return content;
}
/**
 * Compress and split up the given value into multiple cookies
 * @param request
 * @param cookie
 * @param options
 */


function setExtraAuthStorage(request, content, options) {
  const compressedAuthorizationHeaderValue = (0, _compression.deflateValue)(content);
  const compressedContent = compressedAuthorizationHeaderValue.toString('base64');
  splitValueIntoCookies(request, options.cookiePrefix, compressedContent, options.additionalCookies, options.logger);
}

function splitValueIntoCookies(request, // @todo Should be OpenSearchDashboardsRequest, I believe?
cookiePrefix, value, additionalCookies, logger) {
  /**
   * Assume that Iron adds around 50%.
   * Remember that an empty cookie is around 30 bytes
   */
  const maxLengthPerCookie = Math.floor(_common.MAX_LENGTH_OF_COOKIE_BYTES / _common.ESTIMATED_IRON_COOKIE_OVERHEAD);
  const cookiesNeeded = value.length / maxLengthPerCookie; // Assume 1 bit per character since this value is encoded
  // If the amount of additional cookies aren't enough for our logic, we try to write the value anyway
  // TODO We could also consider throwing an error, since a failed cookie leads to weird redirects.
  // But throwing would probably also lead to a weird redirect, since we'd get the token from the IdP again and again

  let splitValueAt = maxLengthPerCookie;

  if (cookiesNeeded > additionalCookies) {
    splitValueAt = Math.ceil(value.length / additionalCookies);

    if (logger) {
      logger.warn('The payload may be too large for the cookies. To be safe, we would need ' + Math.ceil(cookiesNeeded) + ' cookies in total, but we only have ' + additionalCookies + '. This can be changed with opensearch_security.openid.extra_storage.additional_cookies.');
    }
  }

  const rawRequest = (0, _router.ensureRawRequest)(request);
  const values = [];

  for (let i = 1; i <= additionalCookies; i++) {
    values.push(value.substring((i - 1) * splitValueAt, i * splitValueAt));
  }

  values.forEach(async (cookieSplitValue, index) => {
    const cookieName = cookiePrefix + (index + 1);

    if (cookieSplitValue === '') {
      // Make sure we clean up cookies that are not needed for the given value
      rawRequest.cookieAuth.h.unstate(cookieName);
    } else {
      rawRequest.cookieAuth.h.state(cookieName, cookieSplitValue);
    }
  });
}

function unsplitCookiesIntoValue(request, cookiePrefix, additionalCookies) {
  const rawRequest = (0, _router.ensureRawRequest)(request);
  let fullCookieValue = ''; // We don't want to mix and match between _states and .state.
  // If we find the first additional cookie in _states, we
  // use _states for all subsequent additional cookies

  const requestHasNewerCookieState = rawRequest._states && rawRequest._states[cookiePrefix + 1];

  for (let i = 1; i <= additionalCookies; i++) {
    const cookieName = cookiePrefix + i;

    if (requestHasNewerCookieState && rawRequest._states[cookieName] && rawRequest._states[cookieName].value) {
      fullCookieValue = fullCookieValue + rawRequest._states[cookieName].value;
    } else if (!requestHasNewerCookieState && rawRequest.state[cookieName]) {
      fullCookieValue = fullCookieValue + rawRequest.state[cookieName];
    }
  }

  return fullCookieValue;
}

function clearSplitCookies(request, options) {
  const rawRequest = (0, _router.ensureRawRequest)(request);

  for (let i = 1; i <= options.additionalCookies; i++) {
    const cookieName = options.cookiePrefix + i;

    if (rawRequest.state[cookieName]) {
      rawRequest.cookieAuth.h.unstate(cookieName);
    }
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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