/*****************************************************************************/

/*
 *      parse.y  --  Parser for STP router configuration file.
 *
 *      Copyright (C) 2000
 *        Thomas Sailer (sailer@ife.ee.ethz.ch)
 *
 *      This program is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published by
 *      the Free Software Foundation; either version 2 of the License, or
 *      (at your option) any later version.
 *
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License
 *      along with this program; if not, write to the Free Software
 *      Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/*****************************************************************************/


%{

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <netinet/in.h>

#include "stp.h"
        
%}

%union{
	int ival;
	unsigned int uval;
	u_int32_t ip4addr;
	char str[1024];
	struct {
		u_int32_t addr;
		u_int32_t mask;
	} ip4addrmask;
	struct confudpinput *confudpin;
	struct conftcpinput *conftcpin;
	struct confudpoutput *confudpout;
	struct conftcppasvoutput *conftcppasvout;
	struct conftcpoutput *conftcpout;

}

%token_table

%token        TOK_EOF
%token        TOK_INTVAL
%token        TOK_STRING
%token        TOK_STRING2

%type <ival>            number
%type <ip4addr>         ip4addr
%type <str>             string
%type <ip4addrmask>     ip4addrmask
%type <ival>            optionalport

%type <confudpin>       udpinargs
%type <confudpin>       udpinargs1
%type <confudpin>       udpinargsinit

%type <conftcpin>       tcpinargs
%type <conftcpin>       tcpinargs1
%type <conftcpin>       tcpinargsinit

%type <confudpout>      udpoutargs
%type <confudpout>      udpoutargs1
%type <confudpout>      udpoutargsinit

%type <conftcpout>      tcpoutargs
%type <conftcpout>      tcpoutargs1
%type <conftcpout>      tcpoutargsinit

%left '-' '+'
%left '*'
%left NEG

%%

/* start symbol */

input:		  /* empty */
		| input statement2
;

statement2:	  ' ' statement2
		| statement1
;

statement1:	  statement1 ' '
		| statement ';'
;

number: 	  TOK_INTVAL					{ $$ = strtol(yytext, NULL, 0); }
		| number '+' number				{ $$ = $1 + $3; }
		| number '-' number				{ $$ = $1 - $3; }
		| '-' number %prec NEG				{ $$ = - $2; }
		| number '*' number				{ $$ = $1 * $3; }
		| '(' number ')'				{ $$ = $2; }
;

ip4addr: 	  number '.' number '.' number '.' number
{
	$$ = htonl((($1 & 255) << 24) | (($3 & 255) << 16) | (($5 & 255) << 8) | ($7 & 255));
}
;

ip4addrmask:	  ip4addr
{
	$$.addr = $1;
	$$.mask = 0xffffffff;
}
		| ip4addr '/' ip4addr
{
	$$.addr = $1;
	$$.mask = $3;
}
		| ip4addr '/' number
{
	$$.addr = $1;
	$$.mask = htonl(-(1 << $3));
}
;

string:	 	  TOK_STRING	
{
	$$[0] = 0;
	if (yyleng > 2) {
		yytext[yyleng-1] = 0;
		strncpy($$, yytext+1, sizeof($$));
	}
}
		| TOK_STRING2
{
	strncpy($$, yytext, sizeof($$));
}
;

optionalport: 	  /* empty */
{
	$$ = -1;
}
		| "port" number
{
	$$ = htons($2);
}
;

udpinargsinit:
{
	$$ = calloc(1, sizeof(struct confudpinput));
	INIT_LIST_HEAD(&$$->list);
	INIT_LIST_HEAD(&$$->uifdlist);
	$$->maxhops = 255;
}
;

udpinargs: 	  udpinargsinit
{
	$$ = $1;
}
		| '{' udpinargs1 '}'
{
	$$ = $2;
}
;

udpinargs1: 	  udpinargsinit
{
	$$ = $1;
}
		| udpinargs1 "peer" ip4addrmask ';'
{
	$$ = $1;
	$$->addr = $3.addr;
	$$->mask = $3.mask;
	$$->maskbits = hweight32($3.mask);
}
		| udpinargs1 "maxhops" number ';'
{
	$$ = $1;
	$$->maxhops = $3;
}
;

tcpinargsinit:
{
	$$ = calloc(1, sizeof(struct conftcpinput));
	INIT_LIST_HEAD(&$$->list);
	INIT_LIST_HEAD(&$$->tifdlist);
	$$->maxconn = ~0;
	$$->maxhops = 255;
}
;

tcpinargs: 	  tcpinargsinit
{
	$$ = $1;
}
		| '{' tcpinargs1 '}'
{
	$$ = $2;
}
;

tcpinargs1: 	  tcpinargsinit
{
	$$ = $1;
}
		| tcpinargs1 "peer" ip4addrmask ';'
{
	$$ = $1;
	$$->addr = $3.addr;
	$$->mask = $3.mask;
	$$->maskbits = hweight32($3.mask);
}
		| tcpinargs1 "maxconn" number ';'
{	
	$$ = $1;
	$$->maxconn = $3;
}
		| tcpinargs1 "maxhops" number ';'
{
	$$ = $1;
	$$->maxhops = $3;
}
;

udpoutargsinit:
{
	$$ = calloc(1, sizeof(struct confudpoutput));
	INIT_LIST_HEAD(&$$->list);
}
;

udpoutargs: 	  udpoutargsinit
{
	$$ = $1;
}
		| '{' udpoutargs1 '}'
{
	$$ = $2;
}
;

udpoutargs1: 	  udpoutargsinit
{
	$$ = $1;
}
;

tcpoutargsinit:
{
	$$ = calloc(1, sizeof(struct conftcpoutput));
	INIT_LIST_HEAD(&$$->list);
	INIT_LIST_HEAD(&$$->tofdlist);
	$$->maxconn = ~0;
	$$->interval = 15*60;
}
;

tcpoutargs: 	  tcpoutargsinit
{
	$$ = $1;
}
		| '{' tcpoutargs1 '}'
{
	$$ = $2;
}
;

tcpoutargs1: 	  tcpoutargsinit
{
	$$ = $1;
}
		| tcpoutargs1 "maxconn" number ';'
{	
	$$ = $1;
	$$->maxconn = $3;
}
		| tcpoutargs1 "interval" number ';'
{	
	$$ = $1;
	$$->interval = $3;
}
;

statement: 	  "udp" "input" "port" number udpinargs
{
	$5->port = htons($4);
	list_add(&$5->list, &confudpinputlist);
}
		| "tcp" "input" "port" number tcpinargs
{
	$5->port = htons($4);
	list_add(&$5->list, &conftcpinputlist);
}
		| "udp" "output" ip4addr "port" number udpoutargs
{
	$6->addr = $3;
	$6->port = htons($5);
	list_add(&$6->list, &confudpoutputlist);
}
		| "tcp" "output" ip4addrmask optionalport tcpoutargs
{
	$5->passive = 1;
	$5->addr = $3.addr;
	$5->mask = $3.mask;
	$5->maskbits = hweight32($3.mask);
	$5->port = $4;
	list_add(&$5->list, &conftcpoutputlist);
}
		| "tcp" "active" "output" ip4addr "port" number tcpoutargs
{
	$7->passive = 0;
	$7->addr = $4;
	$7->mask = 0xffffffff;
	$7->maskbits = 32;
	$7->port = htons($6);
	list_add(&$7->list, &conftcpoutputlist);
}
;





%%

int find_token(void)
{
	unsigned int i, j;

	j = strlen(yytext);
	for (i = 0; i < YYNTOKENS; i++) {
		if (!yytname[i])
			continue;
		if (yytname[i][0] != '"' || yytname[i][1+j] != '"' || yytname[i][2+j] != 0)
			continue;
		if (!strncmp(yytext, &yytname[i][1], j)) {
			return yytoknum[i];
		}
	}
	return TOK_STRING2;
}
