/*
 * ataprint.c
 *
 * Copyright (C) 1999-2000 Michael Cornwell <cornwell@acm.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * You should have received a copy of the GNU General Public License
 * (for example COPYING); if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

  
#include "ataprint.h"
#include "smartctl.h"


#include "extern.h"

void ataPrintDriveInfo (struct hd_driveid drive)
{
    printf ("Device: %s  Supports ATA Version %i\n", 
             drive.model, ataVersionInfo ( drive) );
}


/* void PrintSmartOfflineStatus ( struct ata_smart_values data) 
   prints verbose value Off-line data collection status byte */

void PrintSmartOfflineStatus ( struct ata_smart_values data)
{
   printf ("Off-line data collection status: ");	
   
   switch (data.offline_data_collection_status)
   {
      case 0x0: case 0x80:
          printf ("(0x%02x)\tOffline data collection activity was\n\t\t\t\t\t",
                  data.offline_data_collection_status);
          printf("never started\n");
          break;
      case 0x01: case 0x81:
          printf ("(0x%02x)\tReserved\n",
                  data.offline_data_collection_status);
          break;
      case 0x02: case 0x82:
          printf ("(0x%02x)\tOffline data collection activity \n\t\t\t\t\t",
                  data.offline_data_collection_status);
          printf ("completed without error\n");
          break;
      case 0x03: case 0x83:
          printf ("(0x%02x)\tReserved\n",
                  data.offline_data_collection_status);
          break;
      case 0x04: case 0x84:
          printf ("(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
                  data.offline_data_collection_status);
          printf ("suspended by an interrupting command\n");
          break;
      case 0x05: case 0x85:
          printf ("(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
                  data.offline_data_collection_status);
          printf ("aborted by an interrupting command\n");
          break;
      case 0x06: case 0x86:
          printf ("(0x%02x)\tOffline data collection activity was \n\t\t\t\t\t",
                  data.offline_data_collection_status);
          printf ("aborted by the device with fatal error\n");
          break;
      default:
          if ( ((data.offline_data_collection_status >= 0x07) &&
                (data.offline_data_collection_status <= 0x3f)) ||
                ((data.offline_data_collection_status >= 0xc0) &&
                (data.offline_data_collection_status <= 0xff)) )
          {
              printf ("(0x%02x)\tVendor Specific\n",
                      data.offline_data_collection_status);
          } 
          else 
          {
              printf ("(0x%02x)\tReserved\n",
                      data.offline_data_collection_status);
          }
   }
}



void PrintSmartSelfExecStatus ( struct ata_smart_values data)
{
   printf ("Self-test execution status:      ");
   
   switch (data.self_test_exec_status >> 4)
   {
      case 0:
        printf ("(%4d)\tThe previous self-test routine completed\n\t\t\t\t\t",
                data.self_test_exec_status);
        printf ("without error or no self-test has ever \n\t\t\t\t\tbeen run\n");
        break;
       case 1:
         printf ("(%4d)\tThe self-test routine was aborted by\n\t\t\t\t\t",
                 data.self_test_exec_status);
         printf ("the host\n");
         break;
       case 2:
         printf ("(%4d)\tThe self-test routine was interrupted\n\t\t\t\t\t",
                 data.self_test_exec_status);
         printf ("by the host with a hard or soft reset\n");
         break;
       case 3:
          printf ("(%4d)\tA fatal error or unknown test error\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("occurred while the device was executing\n\t\t\t\t\t");
          printf ("its self-test routine and the device \n\t\t\t\t\t");
          printf ("was unable to complete the self-test \n\t\t\t\t\t");
          printf ("routine\n");
          break;
       case 4:
          printf ("(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("a test element that failed and the test\n\t\t\t\t\t");
          printf ("element that failed is not known\n");
          break;
       case 5:
          printf ("(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("the electrical element of the test\n\t\t\t\t\t");
          printf ("failed\n");
          break;
       case 6:
          printf ("(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("the servo (and/or seek) element of the \n\t\t\t\t\t");
          printf ("test failed\n");
          break;
       case 7:
          printf ("(%4d)\tThe previous self-test completed having\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("the read element of the test failed\n");
          break;
       case 15:
          printf ("(%4d)\tSelf-test routine in progess\n\t\t\t\t\t",
                  data.self_test_exec_status);
          printf ("%1d0%% of test remaining\n", 
                  data.self_test_exec_status & 0x0f);
          break;
       default:
          printf ("(%4d)\tReserved\n",
                  data.self_test_exec_status);
          break;
   }
	
}



void PrintSmartTotalTimeCompleteOffline ( struct ata_smart_values data)
{
   printf ("Total time to complete off-line \n");
   printf ("data collection: \t\t (%4d) Seconds\n", 
           data.total_time_to_complete_off_line);
}



void PrintSmartOfflineCollectCap ( struct ata_smart_values data)
{
   printf ("Offline data collection \n");
   printf ("Capabilities: \t\t\t (0x%02x)",
            data.offline_data_collection_capability);

   if (data.offline_data_collection_capability == 0x00)
   {
      printf ("\tOff-line data collection not supported\n");
   } 
   else 
   {
      printf( "%s\n", isSupportExecuteOfflineImmediate(data)?
              "SMART EXECUTE OFF-LINE IMMEDIATE" :
              "NO SMART EXECUTE OFF-LINE IMMEDIATE");

      printf( "\t\t\t\t\t%s\n", isSupportAutomaticTimer(data)? 
              "Automatic timer ON/OFF support":
              "NO Automatic timer ON/OFF support");
		
      printf( "\t\t\t\t\t%s\n", isSupportOfflineAbort(data)? 
              "Abort Offline Collection upon new\n\t\t\t\t\tcommand":
              "Suspend Offline Collection upon new\n\t\t\t\t\tcommand");

      printf( "\t\t\t\t\t%s\n", isSupportOfflineSurfaceScan(data)? 
              "Offline surface scan supported":
              "NO Offline surface scan supported");

      printf( "\t\t\t\t\t%s\n", isSupportSelfTest(data)? 
              "Self-test supported":
              "NO Self-test supported");
    }
}



void PrintSmartCapability ( struct ata_smart_values data)
{
   printf ("Smart Capablilities:           ");
   printf ("(0x%04x)\t", data.smart_capability);
   
   if (data.smart_capability == 0x00)
   {
       printf ("automatic saving of SMART data"); 
       printf ("\t\t\t\t\tis not implemented\n");
   } 
   else 
   {
	
      printf( "%s\n", (data.smart_capability & 0x01)? 
              "Saves SMART data before entering\n\t\t\t\t\tpower-saving mode":
              "does not save SMART data before\n\t\t\t\t\tentering power-saving mode");
		
      if ( data.smart_capability & 0x02 )
      {
          printf ("\t\t\t\t\tSupports SMART auto save timer\n");
      }
   }
}



void PrintSmartErrorLogCapability ( struct ata_smart_values data)
{

   printf ("Error logging capability:       ");
    
   if ( isSmartErrorLogCapable(data) )
   {
      printf (" (0x%02x)\tError logging supported\n",
               data.errorlog_capability);
   }
   else {
       printf (" (0x%02x)\tError logging NOT supported\n",
                data.errorlog_capability);
   }
}



void PrintSmartShortSelfTestPollingTime ( struct ata_smart_values data)
{
   if ( isSupportSelfTest(data) )
   {
      printf ("Short self-test routine \n");
      printf ("recommended polling time: \t (%4d) Minutes\n", 
               data.short_test_completion_time);

   }
   else
   {
      printf ("Short self-test routine \n");
      printf ("recommended polling time: \t        Not Supported\n");
   }
}


void PrintSmartExtendedSelfTestPollingTime ( struct ata_smart_values data)
{
   if ( isSupportSelfTest(data) )
   {
      printf ("Extended self-test routine \n");
      printf ("recommended polling time: \t (%4d) Minutes\n", 
               data.extend_test_completion_time);
   }
   else
   {
      printf ("Extended self-test routine \n");
      printf ("recommended polling time: \t        Not Supported\n");
   }
}



void PrintSmartAttribWithThres ( struct ata_smart_values data, 
                                 struct ata_smart_thresholds thresholds)
{
   int i,j;
   long long rawvalue; 
   printf ("Vendor Specific SMART Attributes with Thresholds:\n");
   printf ( "Revision Number: %i\n", data.revnumber);
   printf ("Attribute                    Flag     Value Worst Threshold Raw Value\n");
	
   for ( i = 0 ; i < 30 ; i++ )
   {
	   
      if ( (data.vendor_attributes[i].id !=0) &&
           (thresholds.thres_entries[i].id != 0))
      {
		   
          ataPrintSmartAttribName(data.vendor_attributes[i].id);
          printf(" 0x%04x   %.3i   %.3i   %.3i       ", 
                 data.vendor_attributes[i].status.all,
                 data.vendor_attributes[i].current,
                 data.vendor_attributes[i].worst,
                 thresholds.thres_entries[i].threshold);
		  rawvalue = 0;
          for (j = 0 ; j < 6 ; j++) 
          {
             rawvalue |= data.vendor_attributes[i].raw[j] << (8*j) ;
          }
          printf ("%u\n", (unsigned int)rawvalue);
      }
   }
}


void ataPrintGeneralSmartValues  ( struct ata_smart_values data)
{
   printf ("\nGeneral Smart Values: \n");

   PrintSmartOfflineStatus (data); 
   printf("\n");
	
   if (isSupportSelfTest(data))
   {
       PrintSmartSelfExecStatus (data);
       printf("\n");
   }
	
   PrintSmartTotalTimeCompleteOffline (data);
   printf("\n");
	
   PrintSmartOfflineCollectCap (data);
   printf("\n");
	
   PrintSmartCapability ( data);
   printf("\n");

   PrintSmartErrorLogCapability (data);
   printf ("\n");
	
   if (isSupportSelfTest(data))
   {
      PrintSmartShortSelfTestPollingTime (data);
      printf ("\n");

      PrintSmartExtendedSelfTestPollingTime (data);
      printf ("\n");
   }

}


void ataPrintSmartThresholds (struct ata_smart_thresholds data)
{
   int i;

   printf ("Smart Thresholds\n");
   printf ("Smart Threshold Revision Number: %i\n", data.revnumber);
	
   for ( i = 0 ; i < 30 ; i++)
   {
      if (data.thres_entries[i].id != 0)	
          printf ("Atrribute %3i threshold: %02x (%2i)\n", 
                   data.thres_entries[i].id, 
                   data.thres_entries[i].threshold, 
                   data.thres_entries[i].threshold);
	
   }
}



void ataPrintSmartErrorlog (struct ata_smart_errorlog data)
{
    int i,j;
	

    printf ("SMART Error Log:\n");
    printf ( "SMART Error Logging Version: %i\n", data.revnumber);

    if ( ! data.error_log_pointer)
    {
	printf ("No Errors Logged\n");
	return;
    }

    printf ( "Error Log Data Structure Pointer: %02x\n", 
              data.error_log_pointer);
    printf ( "ATA Error Count: %u\n", data.ata_error_count);
    printf ( "Non-Fatal Count: %u\n", data.non_fatal_count);
	
	
   for (i = 0; i < 5; i++ )
   {
       if ( ( data.errorlog_struct[i].commands[0].devicecontrolreg ||
		   data.errorlog_struct[i].commands[0].featuresreg ||
                   data.errorlog_struct[i].commands[0].sector_count ||
                   data.errorlog_struct[i].commands[0].sector_number ||
                   data.errorlog_struct[i].commands[0].cylinder_low ||
                   data.errorlog_struct[i].commands[0].cylinder_high ||
                   data.errorlog_struct[i].commands[0].drive_head ||
                   data.errorlog_struct[i].commands[0].commandreg ) != 0)
	{
            printf("\nError Log Structure %i:\n", i+1);
            printf("DCR   FR   SC   SN   CL   SH   D/H   CR   Timestamp\n");
		
            for ( j = 0; j < 6; j++)
            {
		printf ( " %02x   %02x   %02x   %02x   %02x   %02x    %02x   %02x     %u\n", 
		data.errorlog_struct[i].commands[j].devicecontrolreg,
		data.errorlog_struct[i].commands[j].featuresreg,
		data.errorlog_struct[i].commands[j].sector_count,
		data.errorlog_struct[i].commands[j].sector_number,
		data.errorlog_struct[i].commands[j].cylinder_low,
		data.errorlog_struct[i].commands[j].cylinder_high,
		data.errorlog_struct[i].commands[j].drive_head,
		data.errorlog_struct[i].commands[j].commandreg,
		( unsigned int) data.errorlog_struct[i].commands[j].timestamp / 1000); 
            }
        }

        if ( (data.errorlog_struct[i].error_struct.error_condition ||
              data.errorlog_struct[i].error_struct.state) != 0)
        {
            printf ( "Error condition: %3i\t",
                     data.errorlog_struct[i].error_struct.error_condition);
            printf ( "Error State:     %3i\n", 
                     data.errorlog_struct[i].error_struct.state);
            printf ( "Number of Hours in Drive Life: %u (life of the drive in hours)\n",  
                     data.errorlog_struct[i].error_struct.timestamp );	
        } 
   }

}
	

void ataPrintSmartSelfTestlog (struct ata_smart_selftestlog data)
{

	/* excluded from first release */

}


void ataPsuedoCheckSmart ( struct ata_smart_values data, 
                           struct ata_smart_thresholds thresholds)
{
   int i;
   int failed = 0;

   for ( i = 0 ; i < 30 ; i++ )
   {
      if ( (data.vendor_attributes[i].id !=0) &&   
           (thresholds.thres_entries[i].id != 0) &&
           ( data.vendor_attributes[i].status.flag.prefailure) &&
           ( data.vendor_attributes[i].current <
            thresholds.thres_entries[i].threshold) &&
           (thresholds.thres_entries[i].threshold != 0xFE) )
      {
          printf("Attribute ID %i Failed\n", 
                  data.vendor_attributes[i].id);
			
          failed = 1;
      } 

   }
   
   printf("%s\n", ( failed )?
     "Please save all data and call drive manufacture immediately.":
     "Check S.M.A.R.T. Passed.");
}

void ataPrintSmartAttribName ( unsigned char id )
{
   switch (id)
   {
	
      case 1:
         printf("(  1)Raw Read Error Rate    ");
	 break;
      case 2:
         printf("(  2)Throughput Performance ");
         break;
      case 3:
         printf("(  3)Spin Up Time           ");
         break;
      case 4:
         printf("(  4)Start Stop Count       ");
         break;
      case 5:
         printf("(  5)Reallocated Sector Ct  ");
	 break;
      case 6:
         printf("(  6)Read Channel Margin    ");
	 break;
      case 7:
         printf("(  7)Seek Error Rate        ");
         break;
      case 8:
         printf("(  8)Seek Time Preformance  ");
         break;
      case 9:
         printf("(  9)Power On Hours         ");
         break;
      case 10:
         printf("( 10)Spin Retry Count       ");
         break;
      case 11:
         printf("( 11)Calibration Retry Count");
         break;
      case 12:
         printf("( 12)Power Cycle Count      ");
         break;
      case 13:
         printf("( 13)Read Soft Error Rate   ");
         break;
	  case 191:
         printf("(191)Gsense Error Rate      ");
         break;
	  case 192:
         printf("(192)Power-Off Retract Count");
         break;
      case 193:
         printf("(193)Load Cycle Count       ");
         break;
      case 194:
         printf("(194)Temperature            ");
         break;
      case 195:
         printf("(195)Hardware ECC Recovered ");
         break;
      case 196:
         printf("(196)Reallocated Event Count");
         break;
      case 197:
         printf("(197)Current Pending Sector ");
         break;
      case 198:
         printf("(198)Offline Uncorrectable  ");
         break;
      case 199:
         printf("(199)UDMA CRC Error Count   ");
         break;
      default:
         printf("(%3d)Unknown Attribute      ", id);
         break;
  }
}	

/****
 Called by smartctl to access ataprint  
**/

void ataPrintMain ( int fd )
{
   struct hd_driveid drive;
   struct ata_smart_values smartval;
   struct ata_smart_thresholds smartthres;
   struct ata_smart_errorlog smarterror;
   struct ata_smart_selftestlog smartselftest;

   if ( driveinfo )
   {
      if (  ataReadHDIdentity ( fd, &drive) != 0 )
      {
         printf("Smartctl: Hard Drive Identity Failed\n");
         exit(0);	
      } 
		
      ataPrintDriveInfo(drive); 
		

      if (ataSmartSupport(drive))
      {
          printf ("Drive supports S.M.A.R.T. and is ");
	  
          if ( ataSmartStatus(fd) != 0) 
          {
              printf( "disabled\n");
              printf( "Use option -%c to enable\n", SMARTENABLE );
              exit(0);
          }
          else { 
              printf( "enabled\n");
          } 
      }
      else {
           printf("Drive does not support S.M.A.R.T.\n");
	   exit (0);
      }
  }

  if ( smartdisable )
  {
		
     if ( ataDisableSmart(fd) != 0) 
     {
	printf( "Smartctl: Smart Enable Failed\n");
	exit(-1);
     }
    
     printf("S.M.A.R.T. Disabled\n");
     exit (0);
		
   }

   if ( smartenable )
   {
      if ( ataEnableSmart(fd) != 0) 
      {
          printf( "Smartctl: Smart Enable Failed\n");
          exit(-1);
      }
	
		/* printstatus here */
   }

	
   /* for everything else read values and thresholds 
   are needed */	

   if ( ataReadSmartValues ( fd, &smartval) != 0 )
   {
      printf("Smartctl: Smart Values Read Failed\n");
      exit (-1);
   }

   if ( ataReadSmartThresholds ( fd, &smartthres) != 0 )
   {
      printf("Smartctl: Smart THresholds Read Failed\n");
      exit (-1); 
   }
	
   if ( checksmart )
   {
	/* pseudo is used because linux does not support access to
	   Task Fiule registers */

      ataPsuedoCheckSmart ( smartval , smartthres);

   }
	
   if (  generalsmartvalues )
   {
      ataPrintGeneralSmartValues( smartval ); 
   }

   if ( smartvendorattrib )
   {
      PrintSmartAttribWithThres( smartval, smartthres);
   }
	
   if ( smarterrorlog )
   {
      if ( isSmartErrorLogCapable(smartval) == 0)
      {
          printf("Device does not support Error Logging\n");
      } 
      else
      {
         if ( ataReadErrorLog ( fd, &smarterror) != 0 )
	 {
             printf("Smartctl: Smart Errorlog Read Failed\n");
         }
         else
         {
             ataPrintSmartErrorlog ( smarterror); 
         }
      }
    }
	

   if ( smartselftestlog )
   {
       if ( isSmartErrorLogCapable(smartval) == 0)
       {
           printf("Device does not support Self Test Logging\n");
       }
       else
       {
	    if (  ataReadSelfTestLog( fd, &smartselftest) != 0 )
            {
                printf("Smartctl: Smart Self Test log Read Failed\n");
            }
            else
            {
                 ataPrintSmartSelfTestlog (smartselftest); 
            }
        } 
    }

    if (  smartautoofflineenable  )
    {
        if ( !isSupportAutomaticTimer (smartval))
        {
            printf("Device does not support S.M.A.R.T. Automatic Timers\n");
            exit(-1);
        }

        if ( ataEnableAutoOffline (fd) != 0) 
        {
            printf( "Smartctl: Smart Enable Automatic Offline Failed\n");
            exit(-1);
        }
        
        printf ("S.M.A.R.T. Automatic Offline Testing Enabled every four hours\n");
    }

    if (  smartautoofflinedisable  )
    {
        if ( !isSupportAutomaticTimer (smartval))
        {
            printf("Device does not support S.M.A.R.T. Automatic Timers\n");
            exit(-1);
        }
			
        if ( ataDisableAutoOffline (fd) != 0) 
        {
            printf( "Smartctl: Smart Disable Automatic Offline Failed\n");
            exit(-1);
        }
         
        printf ("S.M.A.R.T. Automatic Offline Testing Disabled\n");

     }


     if ( smartexeoffimmediate )
     {
        if ( ataSmartOfflineTest (fd) != 0) 
	{
            printf( "Smartctl: Smart Offline Failed\n");
            exit(-1);
        }

        printf ("Drive Command Successful offline test has begun\n");
        printf ("Please wait %d seconds for test to complete\n", 
                 isOfflineTestTime(smartval) );
        printf ("Use smartctl -%c to abort test\n", SMARTSELFTESTABORT);	
	exit (0);
     }

     if ( smartshortcapselftest )
     {
         if ( ! isSupportSelfTest(smartval) )
         {
             printf (" ERROR: device does not support Self-Test function\n");
             exit(-1);
         }
		
         if ( ataSmartShortCapSelfTest (fd) != 0) 
         {
              printf( "Smartctl: Smart Short Self Test Failed\n");
              exit(-1);
         }
     
         printf ("Drive Command Successful offline test has begun\n");
         printf ("Please wait %d minutes for test to complete\n", 
                  isShortSelfTestTime (smartval) );
	 printf ("Use smartctl -%c to abort test\n", SMARTSELFTESTABORT);	
	
         /* Make sure Offline testing is last thing done */
        exit (0);
     }

     if ( smartshortselftest )
     {
        if ( ! isSupportSelfTest(smartval) )
        {
            printf (" ERROR: device does not support Self-Test function\n");
            exit(-1);
        }
		
        if ( ataSmartShortSelfTest (fd) != 0) 
        {
            printf( "Smartctl: Smart Short Self Test Failed\n");
            exit(-1);
        }

        printf ("Drive Command Successful offline test has begun\n");
        printf ("Please wait %d minutes for test to complete\n", 
                 isShortSelfTestTime (smartval) );
        printf ("Use smartctl -%c to abort test\n", SMARTSELFTESTABORT);	
		
	/* Make sure Offline testing is last thing done */
	exit (0);
     }
 
	
     if ( smartextendselftest )
     {
        if ( ! isSupportSelfTest(smartval) )
        {
           printf (" ERROR: device does not support Self-Test function\n");
           exit(-1);
        }

        if ( ataSmartExtendSelfTest (fd) != 0) 
        {
           printf( "S.M.A.R.T. Extendend Self Test Failed\n");
           exit(-1);
        }
		
        printf ("Drive Command Successful self test has begun\n");
        printf ("Please wait %d minutes for test to complete\n", 
                 isExtendedSelfTestTime(smartval) );
        printf ("Use smartctl -%c to abort test\n", SMARTSELFTESTABORT);	
	
        exit (0);
     }

	
     if ( smartextendcapselftest )
     {
         if ( ! isSupportSelfTest(smartval) )
         {
            printf (" ERROR: device does not support self test function\n");
            exit(-1);
         }

         if ( ataSmartExtendCapSelfTest (fd) != 0) 
         {
            printf( "S.M.A.R.T. Extendend Self Test Failed\n");
            exit(-1);
         }
		
            printf ("Drive Command Successful captive extended self test has begun\n");
            printf ("Please wait %d minutes for test to complete\n", 
                    isExtendedSelfTestTime(smartval) );
            printf ("Use smartctl -%c to abort test\n", SMARTSELFTESTABORT);	
            exit (0);
     }

     if ( smartselftestabort )
     {
        if ( ! isSupportSelfTest(smartval) )
        {
            printf (" ERROR: device does not support Self-Test function\n");
            exit(-1);
        }

        if ( ataSmartSelfTestAbort (fd) != 0) 
        {
            printf( "S.M.A.R.T. Self Test Abort Failed\n");
            exit(-1);
        }
		
        printf ("Drive Command Successful self test aborted\n");
    }		
	
}
