// META: title=test WebNN API pooling operations
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-pool2d
// Compute a pooling operation across all the elements within the moving window
// over the input tensor.
//
// enum MLRoundingType {
//   "floor",
//   "ceil"
// };
//
// dictionary MLPool2dOptions {
//   sequence<[EnforceRange] unsigned long> windowDimensions;
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   MLInputOperandLayout layout = "nchw";
//   MLRoundingType roundingType = "floor";
//   sequence<[EnforceRange] unsigned long> outputSizes;
// };
//
// MLOperand averagePool2d(
//     MLOperand input, optional MLPool2dOptions options = {});
// MLOperand l2Pool2d(
//     MLOperand input, optional MLPool2dOptions options = {});
// MLOperand maxPool2d(
//     MLOperand input, optional MLPool2dOptions options = {});

const poolingOperatorsTests = [
  // averagePool2d tests
  {
    'name':
        'averagePool2d float32 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor all positive default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor all negative default options',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            -83.87757873535156,  -2.0740277767181396, -7.561108589172363,
            -45.274261474609375, -16.36655616760254,  -44.908512115478516,
            -42.04186248779297,  -44.77231979370117,  -1.5066279172897339,
            -52.65203857421875,  -92.01856231689453,  -48.004093170166016,
            -61.522972106933594, -93.44403839111328,  -25.780330657958984,
            -95.51873779296875,  -10.963757514953613, -59.132747650146484,
            -32.60173797607422,  -21.4510440826416,   -87.115966796875,
            -61.326114654541016, -41.989723205566406, -87.8764877319336,
            -71.69316101074219,  -80.24160766601562,  -97.48886108398438,
            -75.89422607421875,  -45.08991622924805,  -88.27134704589844,
            -90.71282958984375,  -93.32392120361328,  -59.14753341674805,
            -45.33106231689453,  -51.32562255859375,  -31.154796600341797,
            -31.62424087524414,  -62.80168151855469,  -63.558509826660156,
            -68.96183013916016,  -43.09415054321289,  -15.803443908691406,
            -64.31092071533203,  -66.45872497558594,  -42.027252197265625,
            -26.032955169677734, -22.73752784729004,  -70.32036590576172,
            -85.28227996826172,  -92.10668182373047
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [{'input': 'averagePool2dInput'}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [-49.258975982666016, -60.52408981323242],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            43.46498107910156, 49.37273406982422, 42.7481689453125,
            50.038944244384766, 52.452327728271484, 58.46046447753906,
            32.15948486328125, 34.75465393066406, 54.00202560424805,
            49.65404510498047, 41.824440002441406, 35.84912109375,
            43.23125457763672, 37.842769622802734, 32.67961120605469,
            41.17021942138672, 42.79708480834961, 38.987247467041016
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float32 4D tensor all positive options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            52.43666076660156, 49.84208297729492, 47.26926803588867,
            46.15715408325195, 46.63268280029297, 43.616947174072266,
            44.72445297241211, 44.05451583862305
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            43.46498107910156, 42.7481689453125, 32.15948486328125,
            54.00202560424805, 49.65404510498047, 35.84912109375,
            41.17021942138672, 38.987247467041016
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [32.2001838684082, 42.971012115478516],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments':
            [{'input': 'averagePool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'global averagePool2d float32 4D tensor options.layout=nhwc and options.windowDimensions',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'},
          {'options': {'windowDimensions': [5, 5], 'layout': 'nhwc'}}
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [47.26926803588867, 44.72445297241211],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 50.038944244384766,
            58.46046447753906, 52.73374557495117, 39.1442985534668,
            43.23125457763672, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'averagePool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 21.206613540649414,
            50.038944244384766, 58.46046447753906, 51.3569221496582,
            37.24428939819336, 54.04661178588867, 78.58363342285156,
            52.73374557495117, 39.1442985534668, 57.1103515625,
            43.23125457763672, 32.67961120605469, 56.23945999145508,
            40.00800323486328, 43.85149002075195, 41.061283111572266
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.roundingType=ceil and no padding',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656
          ],
          'descriptor': {shape: [1, 2, 4, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            51.20364761352539, 40.29140853881836, 50.77684020996094,
            51.70764923095703, 50.63130187988281, 49.3919792175293,
            53.128265380859375, 51.11610412597656
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.layout=nhwc and options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 52.73374557495117, 37.16582489013672,
            39.1442985534668, 50.038944244384766, 43.23125457763672,
            58.46046447753906, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.layout=nhwc and options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 91.59549713134766,  78.15438079833984,
            65.64701080322266,  9.686111450195312,  55.14215087890625,
            51.298038482666016, 18.432437896728516, 32.193084716796875,
            49.34624099731445,  87.65037536621094,  15.648024559020996,
            87.25082397460938,  68.02723693847656,  39.49794006347656,
            20.342548370361328, 80.0996322631836,   26.727949142456055,
            10.220142364501953, 64.87446594238281,  52.602699279785156,
            46.5671501159668,   1.4128639698028564, 79.57833099365234,
            11.95406436920166,  4.33846378326416,   85.00074768066406,
            38.183837890625,    64.78374481201172,  45.25398254394531,
            88.03128814697266,  80.9718017578125,   11.333850860595703,
            67.58124542236328,  70.61659240722656,  6.0264997482299805,
            84.90442657470703,  29.7788143157959,   79.06687927246094,
            58.58993148803711,  7.3287248611450195, 2.2384984493255615,
            35.97796630859375,  14.50549030303955,  10.177306175231934,
            68.72449493408203,  1.4140757322311401, 76.45657348632812,
            78.10037994384766,  23.53263282775879
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc',
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 52.73374557495117, 37.16582489013672,
            39.1442985534668, 21.206613540649414, 57.1103515625,
            50.038944244384766, 43.23125457763672, 58.46046447753906,
            32.67961120605469, 51.3569221496582, 56.23945999145508,
            37.24428939819336, 40.00800323486328, 54.04661178588867,
            43.85149002075195, 78.58363342285156, 41.061283111572266
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 21.206613540649414,
            50.038944244384766, 58.46046447753906, 51.3569221496582,
            37.24428939819336, 54.04661178588867, 78.58363342285156,
            52.73374557495117, 39.1442985534668, 57.1103515625,
            43.23125457763672, 32.67961120605469, 56.23945999145508,
            40.00800323486328, 43.85149002075195, 41.061283111572266
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            22.975555419921875, 78.15438079833984,  9.686111450195312,
            51.298038482666016, 32.193084716796875, 87.65037536621094,
            87.25082397460938,  39.49794006347656,  80.0996322631836,
            10.220142364501953, 52.602699279785156, 1.4128639698028564,
            11.95406436920166,  85.00074768066406,  64.78374481201172,
            88.03128814697266,  11.333850860595703, 70.61659240722656,
            84.90442657470703,  79.06687927246094,  7.3287248611450195,
            35.97796630859375,  10.177306175231934, 1.4140757322311401,
            78.10037994384766,  91.59549713134766,  65.64701080322266,
            55.14215087890625,  18.432437896728516, 49.34624099731445,
            15.648024559020996, 68.02723693847656,  20.342548370361328,
            26.727949142456055, 64.87446594238281,  46.5671501159668,
            79.57833099365234,  4.33846378326416,   38.183837890625,
            45.25398254394531,  80.9718017578125,   67.58124542236328,
            6.0264997482299805, 29.7788143157959,   58.58993148803711,
            2.2384984493255615, 14.50549030303955,  68.72449493408203,
            76.45657348632812,  23.53263282775879
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            54.20252990722656, 37.16582489013672, 50.038944244384766,
            58.46046447753906, 52.73374557495117, 39.1442985534668,
            43.23125457763672, 32.67961120605469
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'averagePool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'averagePool2dInput': {
          'data': [
            70.71148681640625,  99.33489990234375,  76.41767883300781,
            39.40980911254883,  38.16328811645508,  45.971256256103516,
            65.3527603149414,   64.51607513427734,  7.725966930389404,
            41.7672004699707,   94.92633819580078,  53.475772857666016,
            95.46460723876953,  58.461795806884766, 15.831390380859375,
            78.41020202636719,  24.454092025756836, 20.630916595458984,
            32.06352233886719,  47.85192108154297,  91.60813903808594,
            72.3534927368164,   74.69429779052734,  28.860214233398438,
            71.82395935058594,  7.989691734313965,  88.16659545898438,
            58.69850540161133,  63.6061897277832,   55.88187789916992,
            52.809974670410156, 72.91474151611328,  46.957664489746094,
            22.10279655456543,  87.14309692382812,  89.6496810913086,
            63.19610595703125,  11.760882377624512, 70.68730926513672,
            57.70444107055664,  1.183821439743042,  25.26912498474121,
            95.29122924804688,  1.9658530950546265, 53.368465423583984,
            21.400854110717773, 55.86185836791992,  27.824508666992188,
            7.642839431762695,  82.34233093261719,  91.75215911865234,
            62.79155731201172,  28.11526107788086,  28.72478675842285,
            29.887035369873047, 66.4310302734375,   7.0103044509887695,
            34.33702087402344,  73.20159912109375,  7.8835601806640625,
            17.82563591003418,  33.799156188964844, 65.01251220703125,
            30.264028549194336, 75.76551818847656,  21.150800704956055,
            60.84249496459961,  98.56522369384766,  62.60990905761719,
            42.42991256713867,  53.142147064208984, 36.29545974731445,
            79.95863342285156,  79.60734558105469,  16.059114456176758,
            19.27552032470703,  53.93022918701172,  48.41620635986328,
            93.00965118408203,  62.086524963378906, 83.50532531738281,
            61.07964324951172,  75.51439666748047,  54.193782806396484,
            2.572873830795288,  59.47652053833008,  34.22541427612305,
            13.07015323638916,  12.419061660766602, 55.82337188720703,
            4.553813934326172,  63.47830581665039,  62.3555908203125,
            56.961090087890625, 34.77016067504883,  0.9611223936080933,
            35.30686950683594,  98.00790405273438
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'averagePool2d',
        'arguments': [
          {'input': 'averagePool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'averagePool2dOutput'
      }],
      'expectedOutputs': {
        'averagePool2dOutput': {
          'data': [
            42.940242767333984, 55.268165588378906, 51.6013298034668,
            50.220027923583984, 72.13362884521484, 41.542198181152344,
            48.91604232788086, 38.775962829589844, 61.21329879760742,
            49.504154205322266, 57.72294998168945, 38.6922492980957,
            50.19099807739258, 29.15436363220215, 52.98439025878906,
            43.10562515258789, 66.77796936035156, 55.2725830078125
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },

  // l2Pool2d tests
  {
    'name': 'l2Pool2d float32 4D constant tensor all positive default options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor default all positive options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor default all negative options',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            -1.1957088708877563, -9.706199645996094,  -39.54935836791992,
            -82.34971618652344,  -32.87415313720703,  -50.22603225708008,
            -31.17849349975586,  -55.817893981933594, -46.70829391479492,
            -38.68181228637695,  -63.299320220947266, -35.09224319458008,
            -80.93848419189453,  -82.8619613647461,   -40.41627502441406,
            -34.86458206176758,  -84.33639526367188,  -84.11852264404297,
            -5.525088787078857,  -99.03114318847656,  -75.505126953125,
            -91.43389129638672,  -96.71258544921875,  -16.722585678100586,
            -17.98292350769043,  -58.06570816040039,  -11.846800804138184,
            -97.90313720703125,  -38.69822692871094,  -80.19510650634766,
            -48.72047805786133,  -90.86722564697266,  -99.10758209228516,
            -79.70288848876953,  -59.3824462890625,   -9.967330932617188,
            -39.27534866333008,  -10.469644546508789, -27.565326690673828,
            -2.0468990802764893, -81.88761901855469,  -66.88040161132812,
            -85.98504638671875,  -29.674592971801758, -19.649417877197266,
            -89.39192199707031,  -61.13504409790039,  -84.16869354248047,
            -77.36112213134766,  -91.17266082763672
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [{'input': 'l2Pool2dInput'}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [298.928955078125, 326.83587646484375],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.45481872558594, 189.54539489746094, 189.85488891601562,
            160.0518341064453, 167.1435546875, 149.63897705078125,
            161.15570068359375, 190.5449981689453, 168.4636688232422,
            170.331787109375, 155.60073852539062, 174.72145080566406,
            165.07762145996094, 165.45819091796875, 161.11062622070312,
            176.6307373046875, 174.245361328125, 180.60714721679688
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            254.81358337402344, 233.14259338378906, 289.01953125,
            269.777587890625, 241.52200317382812, 212.99337768554688,
            292.6146545410156, 253.77178955078125
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            194.45481872558594, 189.85488891601562, 161.15570068359375,
            168.4636688232422, 170.331787109375, 174.72145080566406,
            176.6307373046875, 180.60714721679688
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [189.47933959960938, 207.25343322753906],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   52.027313232421875, 76.55464172363281,
            6.397815227508545,   62.71847152709961,  84.54785919189453,
            83.8726577758789,    18.622516632080078, 73.10235595703125,
            34.10626220703125,   41.52470779418945,  73.96932220458984,
            39.3339729309082,    36.1437873840332,   86.59486389160156,
            60.73781967163086,   23.09039306640625,  55.09187316894531,
            53.650146484375,     63.8924446105957,   0.00902052316814661,
            59.36124038696289,   42.78899383544922,  50.91202926635742,
            81.03960418701172,   50.339813232421875, 33.48585510253906,
            59.31963348388672,   33.67196273803711,  70.78031921386719,
            0.42822372913360596, 35.56179428100586,  80.07991790771484,
            82.53382873535156,   5.929991722106934,  7.572360038757324,
            48.89164733886719,   61.90089416503906,  15.282920837402344,
            14.084012985229492,  13.335721969604492, 90.86540985107422,
            39.06557846069336,   39.56248474121094,  97.06050109863281,
            67.77167510986328,   83.68133544921875,  69.69512176513672,
            21.79571533203125,   89.54518127441406
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments':
            [{'input': 'l2Pool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [289.01953125, 292.6146545410156],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 160.0518341064453,
            149.63897705078125, 142.6990966796875, 139.51637268066406,
            165.07762145996094, 161.11062622070312
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 90.6768569946289,
            160.0518341064453, 149.63897705078125, 65.15908813476562,
            132.56260681152344, 139.84808349609375, 26.61993408203125,
            142.6990966796875, 139.51637268066406, 72.42569732666016,
            165.07762145996094, 161.11062622070312, 96.38701629638672,
            150.1616668701172, 146.8201904296875, 90.64601135253906
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 90.6768569946289,
            160.0518341064453, 149.63897705078125, 65.15908813476562,
            132.56260681152344, 139.84808349609375, 26.61993408203125,
            142.6990966796875, 139.51637268066406, 72.42569732666016,
            165.07762145996094, 161.11062622070312, 96.38701629638672,
            150.1616668701172, 146.8201904296875, 90.64601135253906
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'l2Pool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            94.07447814941406,   76.55464172363281,   62.71847152709961,
            83.8726577758789,    73.10235595703125,   41.52470779418945,
            39.3339729309082,    86.59486389160156,   23.09039306640625,
            53.650146484375,     0.00902052316814661, 42.78899383544922,
            81.03960418701172,   33.48585510253906,   33.67196273803711,
            0.42822372913360596, 80.07991790771484,   5.929991722106934,
            48.89164733886719,   15.282920837402344,  13.335721969604492,
            39.06557846069336,   97.06050109863281,   83.68133544921875,
            21.79571533203125,   52.027313232421875,  6.397815227508545,
            84.54785919189453,   18.622516632080078,  34.10626220703125,
            73.96932220458984,   36.1437873840332,    60.73781967163086,
            55.09187316894531,   63.8924446105957,    59.36124038696289,
            50.91202926635742,   50.339813232421875,  59.31963348388672,
            70.78031921386719,   35.56179428100586,   82.53382873535156,
            7.572360038757324,   61.90089416503906,   14.084012985229492,
            90.86540985107422,   39.56248474121094,   67.77167510986328,
            69.69512176513672,   89.54518127441406
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            171.5061492919922, 164.9919891357422, 160.0518341064453,
            149.63897705078125, 142.6990966796875, 139.51637268066406,
            165.07762145996094, 161.11062622070312
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'l2Pool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'l2Pool2dInput': {
          'data': [
            6.5550384521484375, 26.254413604736328, 28.47271156311035,
            64.81202697753906,  39.65838623046875,  10.465584754943848,
            47.94060134887695,  42.208946228027344, 36.834041595458984,
            68.50249481201172,  2.0496721267700195, 49.73927688598633,
            59.97947311401367,  71.08380889892578,  0.20033331215381622,
            19.39293670654297,  70.1269302368164,   86.8837661743164,
            84.28858184814453,  9.695697784423828,  62.69126510620117,
            51.924110412597656, 5.412675857543945,  70.82118225097656,
            81.61302947998047,  29.148712158203125, 85.83409881591797,
            71.36548614501953,  44.09445571899414,  58.343570709228516,
            43.37118148803711,  54.025882720947266, 85.50556945800781,
            93.19215393066406,  10.992993354797363, 34.864158630371094,
            96.2605209350586,   44.29584503173828,  61.12482833862305,
            79.62699127197266,  4.066447734832764,  64.89644622802734,
            97.5897445678711,   11.257055282592773, 61.151283264160156,
            20.312341690063477, 39.862640380859375, 68.747314453125,
            89.61034393310547,  22.28224754333496,  41.36311721801758,
            62.9378662109375,   79.54936218261719,  55.64254379272461,
            54.47548294067383,  77.04864501953125,  56.83576965332031,
            80.57747650146484,  70.43293762207031,  85.67094421386719,
            19.527807235717773, 33.87490463256836,  14.498117446899414,
            92.85955810546875,  96.8167724609375,   28.399721145629883,
            99.917236328125,    48.76692199707031,  86.08634948730469,
            47.32324981689453,  7.223662376403809,  82.97200775146484,
            38.374778747558594, 22.10988426208496,  14.797550201416016,
            2.3872148990631104, 83.26342010498047,  46.41500473022461,
            28.659175872802734, 13.919462203979492, 55.413089752197266,
            62.68498992919922,  78.54127502441406,  31.142845153808594,
            4.806727886199951,  33.233642578125,    24.749773025512695,
            1.529007077217102,  42.976322174072266, 93.08572387695312,
            77.908935546875,    45.74395751953125,  62.868892669677734,
            60.689762115478516, 20.046878814697266, 13.203198432922363,
            33.33952713012695,  0.5279953479766846
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'l2Pool2d',
        'arguments': [
          {'input': 'l2Pool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'l2Pool2dOutput'
      }],
      'expectedOutputs': {
        'l2Pool2dOutput': {
          'data': [
            120.20333862304688, 114.0977783203125, 127.63969421386719,
            119.95613861083984, 137.89837646484375, 152.24261474609375,
            194.9647216796875, 168.20205688476562, 197.7173309326172,
            169.85887145996094, 195.1484832763672, 190.96127319335938,
            158.64576721191406, 166.2051544189453, 171.07916259765625,
            148.70985412597656, 218.7123260498047, 153.33311462402344
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  },

  // maxPool2d tests
  {
    'name': 'maxPool2d float32 4D constant tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor default options',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [{'input': 'maxPool2dInput'}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.windowDimensions',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'windowDimensions': [3, 3]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 99.28312683105469, 66.09259033203125,
            99.28312683105469, 99.28312683105469, 72.1085205078125,
            97.90348052978516, 72.1085205078125, 72.1085205078125,
            81.73119354248047, 72.44898986816406, 72.44898986816406,
            81.73119354248047, 72.8883056640625, 72.44898986816406,
            80.30484008789062, 72.8883056640625, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.padding',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {'options': {'padding': [1, 0, 0, 1]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 99.28312683105469, 99.28312683105469,
            99.28312683105469, 81.73119354248047, 72.8883056640625,
            81.73119354248047, 72.8883056640625
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'strides': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 66.09259033203125, 97.90348052978516,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            80.30484008789062, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.dilations',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'},
          {'options': {'windowDimensions': [3, 3], 'dilations': [2, 2]}}
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [89.00830078125, 72.33577728271484],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.layout=nchw',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nchw'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 2, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.layout=nhwc',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -76.72171020507812,  -45.72039031982422,
            50.217063903808594,  -61.306129455566406, -52.895477294921875,
            -4.014514446258545,  -44.642333984375,    -94.54893493652344,
            -97.86752319335938,  46.28090286254883,   81.73119354248047,
            99.28312683105469,   5.428491115570068,   -10.057873725891113,
            -29.22772789001465,  9.742474555969238,   72.44898986816406,
            -39.03501892089844,  -59.34124755859375,  75.08192443847656,
            39.19960021972656,   12.819415092468262,  -65.99439239501953,
            -33.01505661010742,  -4.204323768615723,  38.691341400146484,
            -60.54586410522461,  66.09259033203125,   55.890525817871094,
            97.90348052978516,   80.30484008789062,   -8.737770080566406,
            72.8883056640625,    -53.42162322998047,  -46.59611129760742,
            72.1085205078125,    20.50387954711914,   -40.423091888427734,
            -31.126462936401367, -35.68864440917969,  -57.294559478759766,
            -87.64779663085938,  -26.623577117919922, 38.874244689941406,
            15.935754776000977,  39.383602142333984,  -78.77953338623047,
            7.429088115692139,   72.33577728271484
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments':
            [{'input': 'maxPool2dInput'}, {'options': {'layout': 'nhwc'}}],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [99.28312683105469, 81.73119354248047],
          'descriptor': {shape: [1, 1, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, 99.28312683105469,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            81.73119354248047, 72.44898986816406
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'maxPool2d float32 4D tensor options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, -39.03501892089844,
            99.28312683105469, 72.1085205078125, 66.09259033203125,
            97.90348052978516, 72.1085205078125, 7.429088115692139,
            81.73119354248047, 72.44898986816406, -59.34124755859375,
            81.73119354248047, 72.44898986816406, 55.890525817871094,
            80.30484008789062, 72.33577728271484, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.outputSizes ignores options.roundingType=floor',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'floor',
              'outputSizes': [3, 3]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, -39.03501892089844,
            99.28312683105469, 72.1085205078125, 66.09259033203125,
            97.90348052978516, 72.1085205078125, 7.429088115692139,
            81.73119354248047, 72.44898986816406, -59.34124755859375,
            81.73119354248047, 72.44898986816406, 55.890525817871094,
            80.30484008789062, 72.33577728271484, 72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.outputSizes ignores options.roundingType=ceil',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            89.00830078125,      -45.72039031982422,  -61.306129455566406,
            -4.014514446258545,  -94.54893493652344,  46.28090286254883,
            99.28312683105469,   -10.057873725891113, 9.742474555969238,
            -39.03501892089844,  75.08192443847656,   12.819415092468262,
            -33.01505661010742,  38.691341400146484,  66.09259033203125,
            97.90348052978516,   -8.737770080566406,  -53.42162322998047,
            72.1085205078125,    -40.423091888427734, -35.68864440917969,
            -87.64779663085938,  38.874244689941406,  39.383602142333984,
            7.429088115692139,   -76.72171020507812,  50.217063903808594,
            -52.895477294921875, -44.642333984375,    -97.86752319335938,
            81.73119354248047,   5.428491115570068,   -29.22772789001465,
            72.44898986816406,   -59.34124755859375,  39.19960021972656,
            -65.99439239501953,  -4.204323768615723,  -60.54586410522461,
            55.890525817871094,  80.30484008789062,   72.8883056640625,
            -46.59611129760742,  20.50387954711914,   -31.126462936401367,
            -57.294559478759766, -26.623577117919922, 15.935754776000977,
            -78.77953338623047,  72.33577728271484
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'roundingType': 'ceil',
              'outputSizes': [2, 2]
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            99.28312683105469, 9.742474555969238, 99.28312683105469,
            72.1085205078125, 81.73119354248047, 72.44898986816406,
            81.73119354248047, 72.44898986816406
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'maxPool2d float32 4D tensor options.dilations with options.strides',
    'graph': {
      'inputs': {
        'maxPool2dInput': {
          'data': [
            34.69258117675781,   -24.706249237060547, -60.428070068359375,
            35.93883514404297,   60.896095275878906,  79.42220306396484,
            -77.85906219482422,  54.188209533691406,  -21.576934814453125,
            -49.10390853881836,  78.55176544189453,   74.28213500976562,
            -87.92769622802734,  79.82047271728516,   11.680922508239746,
            -12.154505729675293, -22.33293914794922,  33.084861755371094,
            8.640676498413086,   47.040645599365234,  95.7823486328125,
            -88.01998138427734,  -98.53630065917969,  16.158977508544922,
            21.396089553833008,  95.1323471069336,    -40.80724334716797,
            -88.70922088623047,  -40.772769927978516, 67.89842987060547,
            -50.337467193603516, -96.56610870361328,  12.508098602294922,
            -6.6358113288879395, -44.80198287963867,  80.27474212646484,
            -65.68411254882812,  -14.884790420532227, -66.54857635498047,
            20.072338104248047,  -27.954269409179688, -56.62217330932617,
            82.7479476928711,    93.30175018310547,   -34.073394775390625,
            -22.87164306640625,  73.25525665283203,   41.14021682739258,
            -19.75514793395996,  80.07701110839844,   -69.89276885986328,
            14.013250350952148,  -61.36640167236328,  51.53046417236328,
            43.53886413574219,   -89.5888671875,      51.45121765136719,
            73.9239730834961,    -80.25264739990234,  94.72747802734375,
            95.25411224365234,   58.12575149536133,   19.885723114013672,
            -70.0301284790039,   63.419517517089844,  -54.11785125732422,
            -97.22807312011719,  -60.65826416015625,  -31.04998016357422,
            -14.646553039550781, -63.73688888549805,  47.34630584716797,
            85.56405639648438,   -53.389251708984375, -70.84739685058594,
            47.355045318603516,  83.38397979736328,   7.361695289611816,
            46.85823440551758,   98.13465881347656,   -43.9396858215332,
            50.33780288696289,   37.45563507080078,   -54.52760696411133,
            -6.212307929992676,  34.41734313964844,   11.8167724609375,
            72.44517517089844,   86.3460922241211,    4.14656925201416,
            88.33040618896484,   98.29994201660156,   -66.72379302978516,
            58.0643424987793,    -51.168277740478516, -17.768583297729492,
            9.961172103881836,   -52.73843002319336
          ],
          'descriptor': {shape: [1, 7, 7, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'maxPool2d',
        'arguments': [
          {'input': 'maxPool2dInput'}, {
            'options': {
              'windowDimensions': [3, 3],
              'padding': [1, 0, 0, 1],
              'strides': [2, 2],
              'dilations': [1, 1],
              'layout': 'nhwc'
            }
          }
        ],
        'outputs': 'maxPool2dOutput'
      }],
      'expectedOutputs': {
        'maxPool2dOutput': {
          'data': [
            60.896095275878906, 79.42220306396484, 95.7823486328125,
            79.42220306396484, 78.55176544189453, 95.1323471069336,
            82.7479476928711, 93.30175018310547, 95.7823486328125,
            80.27474212646484, 43.53886413574219, 95.1323471069336,
            95.25411224365234, 94.72747802734375, 95.25411224365234,
            98.13465881347656, 63.419517517089844, 98.13465881347656
          ],
          'descriptor': {shape: [1, 3, 3, 2], dataType: 'float32'}
        }
      }
    }
  }
];

if (navigator.ml) {
  poolingOperatorsTests.forEach((test) => {
    webnn_conformance_test(buildAndExecuteGraph, getPrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
