// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System;
using System.Diagnostics;
using System.Security.Cryptography;

namespace Internal.Cryptography
{
    internal sealed partial class DesImplementation
    {
        private static UniversalCryptoTransform CreateTransformCore(
            CipherMode cipherMode,
            PaddingMode paddingMode,
            byte[] key,
            byte[]? iv,
            int blockSize,
            int feedbackSize,
            int paddingSize,
            bool encrypting)
        {
            // The algorithm pointer is a static pointer, so not having any cleanup code is correct.
            IntPtr algorithm = IntPtr.Zero;

            switch (cipherMode)
            {
                case CipherMode.CBC:
                    algorithm = Interop.Crypto.EvpDesCbc();
                    break;
                case CipherMode.ECB:
                    algorithm = Interop.Crypto.EvpDesEcb();
                    break;
                case CipherMode.CFB:

                    Debug.Assert(feedbackSize == 1, "DES with CFB should have FeedbackSize set to 1");
                    algorithm = Interop.Crypto.EvpDesCfb8();

                    break;
                default:
                    throw new NotSupportedException();
            }

            Interop.Crypto.EnsureLegacyAlgorithmsRegistered();

            BasicSymmetricCipher cipher = new OpenSslCipher(algorithm, cipherMode, blockSize, paddingSize, key, 0, iv, encrypting);
            return UniversalCryptoTransform.Create(paddingMode, cipher, encrypting);
        }
    }
}
