/* 
 *
 * GNU Paint 
 * Copyright 2000-2003, 2007  Li-Cheng (Andy) Tai
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 3
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <gtk/gtk.h>
#include <glade/glade.h>
#include "util.h"
#include "debug.h"
#include "global.h"

GtkWidget*
lookup_widget                          (GtkWidget       *widget,
                                        const gchar     *widget_name)
{
  GtkWidget *parent, *scrolledwindow;
  GladeXML *glade_xml;
  int is_drawing_area = 0;
  if (!strcmp(widget_name, DRAWING_AREA)) is_drawing_area = 1;
  
  for (;;)
  {
      if (GTK_IS_MENU (widget))
        parent = gtk_menu_get_attach_widget (GTK_MENU (widget));
      else
        parent = widget->parent;
      if (!parent)
        parent = (GtkWidget*) g_object_get_data (G_OBJECT (widget), "GladeParentKey");
      if (parent == NULL)
        break;
      widget = parent;
  }

  glade_xml = (GladeXML*) g_object_get_data (G_OBJECT (widget),
                                                 GLADE_XML);
  if (!glade_xml)
  {
     g_warning ("XML object not found: %s", widget_name);
     return NULL;
  }
  if (is_drawing_area)
  {
      scrolledwindow =   glade_xml_get_widget(glade_xml, "scroll_frame");
      g_assert(scrolledwindow);
      return (GtkWidget*) g_object_get_data(G_OBJECT(scrolledwindow),
                DRAWING_AREA);
  }
  return glade_xml_get_widget(glade_xml, widget_name);
}

static GList *pixmaps_directories = NULL;

/* Use this function to set the directory containing installed pixmaps. */
void
add_pixmap_directory                   (const gchar     *directory)
{
  pixmaps_directories = g_list_prepend (pixmaps_directories,
                                        g_strdup (directory));
}

/* This is an internally used function to find pixmap files. */
static gchar*
find_pixmap_file                       (const gchar     *filename)
{
  GList *elem;

  /* We step through each of the pixmaps directory to find it. */
  elem = pixmaps_directories;
  while (elem)
    {
      gchar *pathname = g_strdup_printf ("%s%s%s", (gchar*)elem->data,
                                         G_DIR_SEPARATOR_S, filename);
      if (g_file_test (pathname, G_FILE_TEST_EXISTS))
        return pathname;
      g_free (pathname);
      elem = elem->next;
    }
  return NULL;
}

/* This is an internally used function to create pixmaps. */
GtkWidget*
create_pixmap                          (GtkWidget       *widget,
                                        const gchar     *filename)
{
  gchar *pathname = NULL;
  GtkWidget *pixmap;

  if (!filename || !filename[0])
      return gtk_image_new ();

  pathname = find_pixmap_file (filename);

  if (!pathname)
    {
      g_warning (_("Couldn't find pixmap file: %s"), filename);
      return gtk_image_new ();
    }

  pixmap = gtk_image_new_from_file (pathname);
  g_free (pathname);
  return pixmap;
}

/* This is an internally used function to create pixmaps. */
GdkPixbuf*
create_pixbuf                          (const gchar     *filename)
{
  gchar *pathname = NULL;
  GdkPixbuf *pixbuf;
  GError *error = NULL;

  if (!filename || !filename[0])
      return NULL;

  pathname = find_pixmap_file (filename);

  if (!pathname)
    {
      g_warning (_("Couldn't find pixmap file: %s"), filename);
      return NULL;
    }

  pixbuf = gdk_pixbuf_new_from_file (pathname, &error);
  if (!pixbuf)
    {
      fprintf (stderr, "Failed to load pixbuf file: %s: %s\n",
               pathname, error->message);
      g_error_free (error);
    }
  g_free (pathname);
  return pixbuf;
}

/* This is used to set ATK action descriptions. */
void
glade_set_atk_action_description       (AtkAction       *action,
                                        const gchar     *action_name,
                                        const gchar     *description)
{
  gint n_actions, i;

  n_actions = atk_action_get_n_actions (action);
  for (i = 0; i < n_actions; i++)
    {
      if (!strcmp (atk_action_get_name (action, i), action_name))
        atk_action_set_description (action, i, description);
    }
}


inline void invalidate_window(GdkWindow *win)
{
        GdkRectangle wreckt;
        wreckt.x = 0;
        wreckt.y = 0;

        if(win)
        {
                debug("Invalidating window\n");
                gdk_drawable_get_size (GDK_DRAWABLE (win),
                               &wreckt.width,
                               &wreckt.height);
                gdk_window_invalidate_rect(win, &wreckt, FALSE);
                gdk_window_process_all_updates();
        }
}


/*
 * Convert an 8 bit color [0,255] to a 16 bit color [0,65538].
 */
guint16
color16(guint8 color)
{
    return ((guint16)color * (guint16)MAX_8_BIT_COLOR) + (guint16)color;
}

/*
 * convert a GdkColor to its red, blue, green values
 */
#define SHIFT_COLOR(NAME) \
 (color->pixel & visual->NAME##_mask) >> \
 visual->NAME##_shift << \
 (sizeof(unsigned char) * 8 - visual->NAME##_prec);

void
convert_color(const GdkColor *color, unsigned char *r, unsigned char *g, unsigned char *b)
{
    GdkColormap *cmap = gdk_rgb_get_cmap();
    GdkVisual *visual = gdk_rgb_get_visual();
    GdkColor *c;
    switch(visual->type)
    {
        case GDK_VISUAL_STATIC_COLOR:
        case GDK_VISUAL_PSEUDO_COLOR:
            c = cmap->colors + color->pixel;
            *r = c->red >> 8;
            *g = c->green >> 8;
            *b = c->blue >> 8;
            break;

        case GDK_VISUAL_TRUE_COLOR:
        case GDK_VISUAL_DIRECT_COLOR:
            *r = SHIFT_COLOR(red);
            *g = SHIFT_COLOR(green);
            *b = SHIFT_COLOR(blue);
            break;
   
       default:
            g_assert_not_reached();
            break;
    }
}



