/*
 * Copyright (c) 2001, 2018, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package nsk.jdi.BreakpointRequest.addInstanceFilter;

import nsk.share.*;
import nsk.share.jpda.*;
import nsk.share.jdi.*;

import com.sun.jdi.*;
import com.sun.jdi.event.*;
import com.sun.jdi.request.*;

import java.util.*;
import java.io.*;

/**
 * The test for the implementation of an object of the type
 * BreakpointRequest.
 *
 * The test checks that results of the method
 * <code>com.sun.jdi.BreakpointRequest.addInstanceFilter()</code>
 * complies with its spec.
 *
 * The test checks up on the following assertion:
 *    Restricts the events generated by this request to those
 *    in which the currently executing instance is
 *    the object specified.
 *
 * The test works as follows.
 * - The debugger resumes the debuggee and waits for the BreakpointEvent.
 * - The debuggee creates a special array with three Object instances,
 *   and two threads, thread1 and thread2, and invokes
 *   the methodForCommunication to be suspended and
 *   to inform the debugger with the event.
 * - Upon getting the BreakpointEvent, the debugger
 *   - sets up a BreakpointRequest
 *     within the method in the class TestClass
 *     whose array element instances #0 and #1
 *     will be calling by the thread1 and the thread2 accordinly,
 *   - invokes addInstanceFilter() on array element #0 for the thread1
 *     and #2 for the thread2,
 *     thus restricting the breakpoint event only for thread1,
 *   - resumes debuggee's main thread, and
 *   - waits for the event.
 * - Debuggee's main thread starts both threads.
 * - Upon getting the event, the debugger performs the checks required.
 */

public class instancefilter001 extends TestDebuggerType1 {

    public static void main (String argv[]) {
        System.exit(run(argv, System.out) + Consts.JCK_STATUS_BASE);
    }

    public static int run (String argv[], PrintStream out) {
        debuggeeName = "nsk.jdi.BreakpointRequest.addInstanceFilter.instancefilter001a";
        return new instancefilter001().runThis(argv, out);
    }

    private String testedClassName =
      "nsk.jdi.BreakpointRequest.addInstanceFilter.instancefilter001aTestClass";


    protected void testRun() {
        if ( !vm.canUseInstanceFilters() ) {
            display("......vm.canUseInstanceFilters == false :: test cancelled");
            vm.exit(Consts.JCK_STATUS_BASE);
            return;
        }

        EventRequest eventRequest1 = null;
        EventRequest eventRequest2 = null;

        ThreadReference thread1 = null;
        ThreadReference thread2 = null;

        String thread1Name = "thread1";
        String thread2Name = "thread2";

        String property1 = "BreakpointRequest1";
        String property2 = "BreakpointRequest2";

        String methodName = "method";
        String bpLineName = "breakpointLine";

        ReferenceType testClassReference = null;

        String         arrayName = "objTC";
        ObjectReference instance = null;
        Event newEvent = null;

        for (int i = 0; ; i++) {

            if (!shouldRunAfterBreakpoint()) {
                vm.resume();
                break;
            }

            display(":::::: case: # " + i);

            switch (i) {

                case 0:
                testClassReference =
                     (ReferenceType) vm.classesByName(testedClassName).get(0);

                thread1 = (ThreadReference) debuggeeClass.getValue(
                                            debuggeeClass.fieldByName(thread1Name));
                thread2 = (ThreadReference) debuggeeClass.getValue(
                                            debuggeeClass.fieldByName(thread2Name));

                eventRequest1 = setting2BreakpointRequest (thread1,
                                       testClassReference, methodName, bpLineName,
                                       EventRequest.SUSPEND_ALL, property1);

                instance = (ObjectReference)
                           ((ArrayReference) debuggeeClass.getValue(
                           debuggeeClass.fieldByName(arrayName)) ).getValue(0);

                ((BreakpointRequest) eventRequest1).addInstanceFilter(instance);

                eventRequest2 = setting2BreakpointRequest (thread2,
                                       testClassReference, methodName, bpLineName,
                                       EventRequest.SUSPEND_ALL, property2);

                instance = (ObjectReference)
                           ((ArrayReference) debuggeeClass.getValue(
                           debuggeeClass.fieldByName(arrayName)) ).getValue(2);

                ((BreakpointRequest) eventRequest2).addInstanceFilter(instance);

                display("......waiting for BreakpointEvent in expected thread");
                newEvent = eventHandler.waitForRequestedEvent(new EventRequest[]{eventRequest1, eventRequest2}, waitTime, true);

                if ( !(newEvent instanceof BreakpointEvent)) {
                    setFailedStatus("ERROR: new event is not BreakpointEvent");
                } else {

                    String property = (String) newEvent.request().getProperty("number");
                    display("       got new BreakpointEvent with propety 'number' == " + property);

                    if ( !property.equals(property1) ) {
                        setFailedStatus("ERROR: property is not : " + property1);
                    }
                }
                vm.resume();
                break;

                default:
                throw new Failure("** default case 2 **");
            }
        }
        return;
    }

    private BreakpointRequest setting2BreakpointRequest ( ThreadReference thread,
                                                          ReferenceType   testedClass,
                                                          String          methodName,
                                                          String          bpLine,
                                                          int             suspendPolicy,
                                                          String          property        ) {
        try {
            display("......setting up a breakpoint:");
            display("       thread: " + thread + "; class: " + testedClass + "; method: " + methodName + "; line: " + bpLine);

            int n = ( (IntegerValue) testedClass.getValue(testedClass.fieldByName(bpLine) ) ).value();
            Location loc = (Location) ((Method) testedClass.methodsByName(methodName).get(0)).allLineLocations().get(n);

            BreakpointRequest
            bpr = eventRManager.createBreakpointRequest(loc);
            bpr.putProperty("number", property);
            if (thread != null)
                bpr.addThreadFilter(thread);
            bpr.setSuspendPolicy(suspendPolicy);

            display("      a breakpoint has been set up");
            return bpr;
        } catch ( Exception e ) {
            throw new Failure("** FAILURE to set up BreakpointRequest **");
        }
    }

}
