/* test.c - Test implementation
 *
 * Copyright 2010 Petteri Hintsanen <petterih@iki.fi>
 *
 * This file is part of abx.
 *
 * abx is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * abx is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with abx.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "test.h"
#include <glib.h>
#include <math.h>

/*
 * Ongoing test structure.
 */
typedef struct {
    char isvalid;
    unsigned int ntrials;
    answer *answers;  /* correct answers array with ntrials elements */
    answer *guesses;  /* guesses array with ntrials elements */
} Test;

static Test current_test = { 0, -1, NULL, NULL };

/*
 * Initialize a new test with ntrials trials.  Close current test
 * first, if it is valid.
 *
 * Use the specified PortAudio device for audio output, or scan for
 * default audio device if outdev is -1.
 *
 * Return
 *   0 on success,
 *   1 if playback could not be initialized for sample a, 
 *   2 if playback could not be initialized for sample b,
 *   3 if the samples have differed duration, or
 *   4 if ntrials is zero.
 */
int
init_test(const char *a, const char *b, unsigned int ntrials, 
          PaDeviceIndex outdev)
{
    Test test;
    int i, res;

    /* g_debug("creating new test, a: '%s', b: '%s', ntrials: %d", */
    /*         a, b, ntrials); */

    close_test();
   
    res = init_playback(a, b, outdev);
    if (res == 2 || res == 3 || res == 4) {
	g_warning("can't initialize playback");
        test.isvalid = 0;
	return res - 1;
    }

    if (ntrials == 0) {
        g_warning("number of trials is zero");
        test.isvalid = 0;
        return 4;
    }

    test.guesses = g_malloc(ntrials * sizeof(answer));
    test.answers = g_malloc(ntrials * sizeof(answer));
    for (i = 0; i < ntrials; i++) {
        if (g_random_boolean() == TRUE)	test.answers[i] = SAMPLE_A;
        else test.answers[i] = SAMPLE_B;
        test.guesses[i] = SAMPLE_NA;
    }

    test.ntrials = ntrials;
    test.isvalid = 1;
    current_test = test;
    return 0;
}

/*
 * Close the current test.  Close playback and release all
 * test-related resources.  Mark the current test invalid.  Return
 * immediately if there is no test in progress.
 */
void
close_test(void)
{
    if (!current_test.isvalid) return;
    close_playback();
    g_free(current_test.guesses);
    g_free(current_test.answers);
    current_test.isvalid = 0;
}

/*
 * Return non-zero if the current test is valid.
 */
int
is_test_valid(void)
{
    return current_test.isvalid;
}

/*
 * Get the listener's guess for trial t.
 * 
 * Return 
 *   SAMPLE_A if the listener has guessed sample A,
 *   SAMPLE_B if the listener has guessed sample B,
 *   SAMPLE_NA if the listener has not yet guessed, or
 *   -1 on error.
 */
answer
get_guess(unsigned int t)
{
    if (!is_test_valid()
        || t < 0 || t > current_test.ntrials) return -1;
    else return current_test.guesses[t];
}

/*
 * Get the correct answer for trial t.
 *
 * Return 
 *   SAMPLE_A if sample A is the correct answer,
 *   SAMPLE_B if sample B is the correct answer, or
 *   -1 on error.
 */
answer
get_answer(unsigned int t)
{
    if (!is_test_valid()
        || t < 0 || t > current_test.ntrials) return -1;
    else return current_test.answers[t];
}

/*
 * Return the number of test trials, or -1 if no test is in progress.
 */
int
num_test_trials(void)
{
    if (!is_test_valid()) return -1;
    else return current_test.ntrials;
}

/*
 * Set the guessed value to ans for trial t.  Return 0 on success, or
 * -1 on error.
 */
int
set_guess(unsigned int t, answer ans)
{
    if (!is_test_valid()
        || t < 0 || t > current_test.ntrials) {
        return -1;
    } else {
        current_test.guesses[t] = ans;
        return 0;
    }
}

/*
 * Calculate cumulative distribution function value F(k) for binomial
 * distribution Bin(n, p).
 *
 * The algorithm is from Råde, Westergren: Mathematics Handbook, page
 * 426, 4th ed., 1998.  Studentlitteratur, Lund.
 */
static double
cdf_binomial(unsigned int k, unsigned int n, double p)
{
    double f, F;
    int x;

    if (k >= n) return 1;

    f = pow((1 - p), n);        /* f is the pf, init to f(0) */
    F = f;                      /* F is the cdf, init to F(k) */

    for (x = 1; x <= k; x++) {
        f = (p / (1 - p)) * ((1.0 * n - x + 1) / x) * f;
        F += f;
    }

    return F;
}


/*
 * Return the p-value for the test statistic under the null
 * hypothesis.
 */
double
calculate_p_value(void)
{
    int i;
    int ncorr = 0;
    for (i = 0; i < num_test_trials(); i++) {
        if (get_answer(i) == get_guess(i)) ncorr++;
    }

    if (ncorr == 0) {
        return 1.0;
    } else {
        return 1.0 - cdf_binomial(ncorr - 1, num_test_trials(), 0.5);
    }
}
